/*
 * Copyright (c) 2005 Sendmail, Inc. and its suppliers.
 *	All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 *
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: t-bdbrd-0.c,v 1.2 2005/10/19 20:33:58 ca Exp $")

#include "sm/error.h"
#include "sm/heap.h"
#include "sm/memops.h"
#include "sm/pthread.h"
#include "sm/maps.h"
#include "sm/mapc.h"
#include "sm/map.h"
#include "sm/mapclasses.h"
#include "sm/bdb.h"
#include "sm/sysexits.h"
#include "sm/test.h"

#include "sm/io.h"

static int Verbose = 0;
static uint Loops = 1;
static uint Done = 0;

#define MAPC_TYPE	"hash"
#define MAPC_NAME	"bdbrd0"
#define MAPC_FILE	"./bdbrd0.db"

static char *keys[] =
{
	"key1",
	"key2",
	"cltaddr:127.0.0.1",
	"cltaddr:127.0.0.3",
	"cltname:one.sm9.org",
	"cltresolve:NO",
	"from:@.z",
	"from:@x.y",
	"from:a@b.c",
	"from:d*@b.c",
	"from:local",
	"protectedrcpt:list1@b.c",
	"protectedrcpt:list2",
	"protectedrcpt:list3",
	"protectedrcpt:list4",
	"sm9all:127.0.0.2",
	"sm9bl:127.0.0.128",
	"sm9dnsbl:127.0.0.1",
	"sm9dnsbl:127.0.0.2",
	"sm9dnsbl:127.0.0.3",
	"to:@not2.c",
	"to:not",
	"to:not1@b.c",
	NULL
};

static void *
testrd(void *arg)
{
	sm_ret_T ret;
	uint u, l, seed;
	char *key;
	sm_str_P lhs, rhs;
	sm_map_P map;

	map = (sm_map_P) arg;
	lhs = rhs = NULL;
	lhs = sm_str_new(NULL, 256, 1024);
	SM_TEST(lhs != NULL);
	if (lhs == NULL)
		goto done;
	rhs = sm_str_new(NULL, 256, 1024);
	SM_TEST(rhs != NULL);
	if (rhs == NULL)
		goto done;

	seed = (uint) pthread_self();
	for (l = 0; l < Loops; l++)
	{
		u = (uint) rand_r(&seed) % SM_ARRAY_SIZE(keys);
		key = keys[u];
		if (key == NULL)
			continue;
		sm_str_clr(lhs);
		sm_str_clr(rhs);
		sm_str_scat(lhs, key);
		ret = sm_map_lookup(map, SMMAP_FL_NONE, lhs, rhs);
		SM_TEST(ret == SM_SUCCESS || ret == SM_MAP_NOTFOUND);
	}

  done:
	SM_STR_FREE(lhs);
	SM_STR_FREE(rhs);

	/* needs locking... */
	++Done;
	return NULL;
}

#define SM_MAX_THREADS	32

static void
testbdb(uint thrds)
{
	sm_ret_T ret;
	uint u;
	sm_maps_P maps;
	sm_map_P map;
	sm_cstr_P mtype, mname;
	pthread_t threads[SM_MAX_THREADS];

	maps = NULL;
	mtype = mname = NULL;
	SM_TEST(thrds < SM_MAX_THREADS);
	if (thrds >= SM_MAX_THREADS)
		goto error;

	ret = sm_maps_init(&maps);
	SM_TEST(maps != NULL);
	if (maps == NULL)
		return;
	SM_TEST(sm_is_success(ret));

	mtype = sm_cstr_scpyn0((const uchar *)MAPC_TYPE, strlen(MAPC_TYPE));
	SM_TEST(mtype != NULL);
	if (mtype == NULL)
		goto error;

	mname = sm_cstr_scpyn0((const uchar *)MAPC_NAME, strlen(MAPC_NAME));
	SM_TEST(mname != NULL);
	if (mname == NULL)
		goto error;

	ret = sm_bdb_class_create(maps);
	SM_TEST(sm_is_success(ret));

	map = NULL;
	ret = sm_map_open(maps, mname, mtype, 0, MAPC_FILE, SMAP_MODE_RDONLY,
			&map, SMPO_END);
	SM_TEST(sm_is_success(ret));
	if (!sm_is_success(ret))
	{
		if (Verbose > 0)
			sm_io_fprintf(smioerr, "map_open=%r\n", ret);
		goto error;
	}

	for (u = 0; u < thrds; u++)
	{
		ret = pthread_create(&(threads[u]), NULL, testrd, (void *)map);
		SM_TEST(ret == 0);
		if (ret != 0)
			goto error;
	}
	for (u = 0; u < thrds; u++)
	{
		ret = pthread_join(threads[u], NULL);
		SM_TEST(ret == 0);
		if (ret != 0)
			goto error;
	}

	ret = sm_map_close(map, 0);
	SM_TEST(sm_is_success(ret));

	ret = sm_maps_term(maps);
	SM_TEST(sm_is_success(ret));

	SM_CSTR_FREE(mtype);
	SM_CSTR_FREE(mname);
	return;

  error:
	sm_maps_term(maps);
}


int
main(int argc, char *argv[])
{
	int c;
	uint thrds;

#if SM_HEAP_CHECK
	SmHeapCheck = 0;
#endif
	thrds = 4;
	while ((c = getopt(argc, argv, "H:l:t:V")) != -1)
	{
		switch (c)
		{
		  case 'H':
#if SM_HEAP_CHECK
			SmHeapCheck = atoi(optarg);
#endif
			break;

		  case 'l':
			Loops = strtoul(optarg, 0, NULL);
			break;

		  case 't':
			thrds = strtoul(optarg, 0, NULL);
			break;

		  case 'V':
			++Verbose;
			break;

		  default:
			return EX_USAGE;
		}
	}
	sm_test_begin(argc, argv, "test read bdb map 0");
	testbdb(thrds);
#if SM_HEAP_CHECK
	if (SmHeapCheck > 0)
		sm_heap_report(smioerr, 3);
#endif
	return sm_test_end();
}
