/*
 * Copyright (c) 2002-2005 Sendmail, Inc. and its suppliers.
 *      All rights reserved.
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENSE file which can be found at the top level of
 * the sendmail distribution.
 */

#include "sm/generic.h"
SM_RCSID("@(#)$Id: edbreqdel.c,v 1.21 2005/04/01 22:06:49 ca Exp $")
#include "sm/error.h"
#include "sm/memops.h"
#include "sm/assert.h"
#include "sm/mta.h"
#include "edb-int.h"
#include "sm/edb.h"
#include "sm/pthread.h"

/*
**  EDB_TA_DEL_REQ --  add a request to remove a TA
**
**	Parameters:
**		edb_ctx - EDB context
**		ta_id - SMTPS transaction id (key for DB)
**		edb_req_hd - EDB request list (used if not NULL)
**
**	Side Effects: none on error (except if unlock fails)
**
**	Returns:
**		usual sm_error code
**
**	Last code review: 2005-03-30 19:59:38
**	Last code change: 2005-03-30 19:57:10
*/

sm_ret_T
edb_ta_rm_req(edb_ctx_P edb_ctx, sessta_id_T ta_id, edb_req_hd_P edb_req_hd)
{
	sm_ret_T ret;
	int r;
	edb_req_P edb_req;

	SM_IS_EDB_CTX(edb_ctx);
	r = pthread_mutex_lock(&(edb_ctx->edb_mutex));
	SM_LOCK_OK(r);
	if (r != 0)
	{
		/* LOG? */
		return sm_error_perm(SM_EM_EDB, r);
	}

	/* Ask for a request element without RCB?? Two types of req? */
	ret = edb_req_new(edb_ctx, EDB_RQF_SMALL, &edb_req, false);
	if (sm_is_success(ret))
	{
		SESSTA_COPY(edb_req->edb_req_id, ta_id);
		edb_req->edb_req_type = EDB_REQ_TA_DEL;
		if (edb_req_hd == NULL)
			EDBREQL_APP(&(edb_ctx->edb_reql_wr), edb_req);
		else
			EDBREQL_APP(edb_req_hd, edb_req);
	}
	r = pthread_mutex_unlock(&(edb_ctx->edb_mutex));
	SM_ASSERT(r == 0);
	if (r != 0 && sm_is_success(ret))
		ret = sm_error_perm(SM_EM_EDB, r);
	return ret;
}

/*
**  EDB_RCPT_RM_REQ -- add a request to remove a recipient
**
**	Parameters:
**		edb_ctx - EDB context
**		rcpt_id - recipient id (key for DB)
**		edb_req_hd - EDB request list (used if not NULL)
**
**	Side Effects: none on error (except if unlock fails)
**
**	Returns:
**		usual sm_error code; ENOMEM, (un)lock
**
**	Locking: locks/unlocks edb_ctx
**
**	Last code review: 2005-03-30 19:59:10
**	Last code change: 2005-03-30 19:54:51
*/

sm_ret_T
edb_rcpt_rm_req(edb_ctx_P edb_ctx, rcpt_id_T rcpt_id, edb_req_hd_P edb_req_hd)
{
	sm_ret_T ret;
	int r;
	edb_req_P edb_req;

	SM_IS_EDB_CTX(edb_ctx);
	r = pthread_mutex_lock(&(edb_ctx->edb_mutex));
	SM_LOCK_OK(r);
	if (r != 0)
	{
		/* LOG? */
		return sm_error_perm(SM_EM_EDB, r);
	}

	/* Ask for a request element without RCB?? Two types of req? */
	ret = edb_req_new(edb_ctx, EDB_RQF_SMALL, &edb_req, THR_NO_LOCK);
	if (sm_is_success(ret))
	{
		RCPT_ID_COPY(edb_req->edb_req_id, rcpt_id);
		edb_req->edb_req_type = EDB_REQ_RCPT_DEL;
		if (edb_req_hd == NULL)
			EDBREQL_APP(&(edb_ctx->edb_reql_wr), edb_req);
		else
			EDBREQL_APP(edb_req_hd, edb_req);
	}
	r = pthread_mutex_unlock(&(edb_ctx->edb_mutex));
	SM_ASSERT(r == 0);
	if (r != 0 && sm_is_success(ret))
		ret = sm_error_perm(SM_EM_EDB, r);
	return ret;
}
