# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Report.pm,v 1.4 2002/12/06 02:21:14 lidl Exp $

package ISC::Stats::Report;

use strict;
use warnings;

use Event;
use Carp;

use Time::HiRes qw(gettimeofday);

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.4 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;

    my $self = bless({}, $class);

    $self->{freq} = $args{freq};

    if ($self->{freq}) {
	$self->{tick} = Event->timer(interval => $self->{freq},
				     cb => [$self, "_tick"]);
    }

    croak "Missing 'msgbus' argument" unless ($args{msgbus});
    croak "Missing 'group' argument" unless ($args{group});
    croak "Missing 'instance' argument" unless ($args{instance});
    croak "Missing 'cb' argument" unless ($args{cb});

    $self->{cb} = $args{cb};
    my $msgbus = $args{msgbus};

    $self->{chan} = $msgbus->join(group => $args{group},
				  instance => $args{instance},
				  subtype => "meonly");

    $self->{lasttime} = 0;
    $self->{lastcb} = 0;

    return $self;
}

sub _doit {
    my ($self) = @_;

    my $now = gettimeofday;

    my $cb = $self->{cb};
    my $stats = $cb->($self);
    $self->{lastcb} = $now;

    return unless ($stats);

    my $msg = {
	stats => $stats,
	now => $now,
	interval => $self->{freq},
	lastsent => $self->{lastsent},
    };

    $self->{chan}->send(msg => $msg);
    $self->{lastsent} = $now;
}

sub _tick {
    my ($self, $ev) = @_;

    $self->_doit;
}

sub cancel {
    my ($self) = @_;

    $self->{tick}->cancel if ($self->{tick});
    $self->{tick} = undef;

    $self->{chan} = undef;
}

sub lastsent {
    my ($self) = @_;

    return $self->{lastsent};
}

sub lastcb {
    my ($self) = @_;

    return $self->{lastcb};
}

sub now {
    my ($self) = @_;

    $self->_doit;
}

1;

__END__

=head1 NAME

ISC::Stats::Report - periodically report statistics to a msgbus channel

=head1 SYNOPSIS

use ISC::Stats::Report;

my $stats = new ISC::Stats::Report(cb => \&make_msg,
				   freq => 60,  # in seconds
				   group => "groupname",
				   instance => "instancename");

sub make_msg {
    my ($stats) = @_;

    ...

    my $msg = { foo => "bar" };

    ...

    return ($msg);
}

=head1 DESCRIPTION

This package will call the function make_msg, which should return undef,
or a reference to a hash.  This reference will be transmitted to the
C<group> and C<instance> specified.

=over

=item new ARGS

Create a new stats object.

C<cb> specifies the callback function which will return a reference to
a hash, or return C<undef>.  If it returns C<undef> no message will be
transmitted.  Otherwise, a command channel message will be created
and transmitted on the specified C<channel> and C<instance>.

C<freq> specifies the frequency of message sends.  If it is 0 or undef,
no periodic transmissions will occur.

The entire message format is application specific.  Some suggestions:
an identifier (hostname, pid, etc) if there are multiple reporters,
a validity period, etc.

=item lastsent

Returns the Unix format time (same as what C<time> would return) of when
a message was last transmitted.

=item lastcb

Returns the Unix format time (same as what C<time> would return) of when
the callback was last contacted.  This is set on return from the callback.

=item cancel

Shut down all operation, and free all resources.

=item now

Trigger a stats callback and transmission now.

=back

=head1 AUTHOR

Written by Michael Graff for the Internet Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.
