# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Log.pm,v 1.3 2002/12/06 02:21:09 lidl Exp $

package ISC::Log;

use strict;
use warnings;

use Carp;
use Time::HiRes qw(gettimeofday);

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.3 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT_OK = qw();
    @EXPORT = qw(ISC_LOG_DISABLED ISC_LOG_EMERG ISC_LOG_ALERT ISC_LOG_CRIT
		 ISC_LOG_ERR ISC_LOG_WARNING	ISC_LOG_NOTICE
		 ISC_LOG_INFO ISC_LOG_DEBUG);
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

use constant ISC_LOG_DISABLED	=> -1;
use constant ISC_LOG_EMERG	=> 0;
use constant ISC_LOG_ALERT	=> 1;
use constant ISC_LOG_CRIT	=> 2;
use constant ISC_LOG_ERR	=> 3;
use constant ISC_LOG_WARNING	=> 4;
use constant ISC_LOG_NOTICE	=> 5;
use constant ISC_LOG_INFO	=> 6;
use constant ISC_LOG_DEBUG	=> 7;

our %level_names = ( &ISC_LOG_DISABLED => "DISABLED",
		     &ISC_LOG_EMERG => "EMERG",
		     &ISC_LOG_ALERT => "ALERT",
		     &ISC_LOG_CRIT => "CRIT",
		     &ISC_LOG_ERR => "ERR",
		     &ISC_LOG_WARNING => "WARNING",
		     &ISC_LOG_NOTICE => "NOTICE",
		     &ISC_LOG_INFO => "INFO",
		     &ISC_LOG_DEBUG => "DEBUG",
		     );

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;
    my $self = bless({}, $class);

    croak "Missing 'facility' argument" unless ($args{facility});

    $self->{maxlevel_msgbus} = ISC_LOG_DISABLED;
    $self->{maxlevel_file} = ISC_LOG_DISABLED;
    $self->{maxlevel_stderr} = ISC_LOG_DISABLED;
    $self->{maxlevel_syslog} = ISC_LOG_DISABLED;

    $self->{facility} = $args{facility};

    if ($args{msgbus} || $args{group} || $args{instance}
	|| $args{maxlevel_msgbus}) {
	croak "Missing 'msgbus' argument" unless ($args{msgbus});
	croak "Missing 'group' argument" unless ($args{group});
	croak "Missing 'instance' argument" unless ($args{instance});

	my $msgbus = $args{msgbus};

	$self->{chan} = $msgbus->join(group => $args{group},
				      instance => $args{instance},
				      subtype => "meonly");

	if (defined($args{maxlevel_msgbus})) {
	    $self->{maxlevel_msgbus} = $args{maxlevel_msgbus};
	} else {
	    $self->{maxlevel_msgbus} = ISC_LOG_ERR;
	}
    }

    if ($args{maxlevel_stderr}) {
	$self->{maxlevel_stderr} = $args{maxlevel_stderr};
    } else {
	$self->{maxlevel_stderr} = ISC_LOG_DISABLED;
    }

    if (!$self->{chan}
	&& !$self->{file}
	&& !$self->{syslog}
	&& $self->{maxlevel_stderr} == ISC_LOG_DISABLED) {
	croak "Must specifiy data for either msgbus, file, stderr, or syslog logging";
    }

    return $self;
}

#
# non-OOP interface
#
sub level_totext {
    my ($l) = @_;

    return undef unless (defined($l));

    return $level_names{$l};
}

sub level_name {
    my ($self, $l) = @_;

    return undef unless (defined($l));

    return $level_names{$l};
}

sub facility {
    my ($self, $f) = @_;

    $self->{facility} = $f if ($f);

    return $self->{facility};
}

sub maxlevel_msgbus {
    my ($self, $f) = @_;

    $self->{maxlevel_msgbus} = $f if ($f);

    return $self->{maxlevel_msgbus};
}

sub maxlevel_file {
    my ($self, $f) = @_;

    $self->{maxlevel_file} = $f if ($f);

    return $self->{maxlevel_file};
}

sub maxlevel_syslog {
    my ($self, $f) = @_;

    $self->{maxlevel_syslog} = $f if ($f);

    return $self->{maxlevel_syslog};
}

sub maxlevel_stderr {
    my ($self, $f) = @_;

    $self->{maxlevel_stderr} = $f if ($f);

    return $self->{maxlevel_stderr};
}

sub log {
    my ($self, $level, $msg) = @_;

    my $now = gettimeofday;

    if ($self->{maxlevel_msgbus} >= $level) {
	my $msg = {
	    facility => $self->{facility},
	    level => $level,
	    msg => $msg,
	    now => $now,
	};

	$self->{chan}->send(msg => $msg);
    }

    my $txt;
    if ($level <= $self->{maxlevel_file}
	|| $level <= $self->{maxlevel_stderr}
	|| $level <= $self->{maxlevel_syslog}) {
	$txt = $now . " " . $self->{facility} . " " . $self->level_name($level) . " $msg";
    } else {
	return;
    }

    if ($level <= $self->{maxlevel_stderr}) {
	print STDERR "$txt\n";
    }
}

1;

__END__

=head1 NAME

ISC::Log - Log messages to a file, syslog, stderr, or msgbus channel

=head1 SYNOPSIS

B<NOTE:  syslog and file support are not yet implemented.  XXXMLG>

 use ISC::Log;

 my $loghandle = new ISC::Log(
    facility => "something",
    <<config for file logging>>
    <<config for msgbus logging>>
    <<config for syslog logging>>
 );

 my $facility = $loghandle->facility;
 $loghandle->facility("newval");

 For file logging, provide C<file> to C<new>, and:

 my $maxlevel_file = $loghandle->maxlevel_file;
 $loghandle->maxlevel_file(ISC_LOG_INFO);

 For msgbus logging, provide C<msgbus>, C<group>, and C<instance>
 to C<new>, and:

 my $maxlevel_msgbus = $loghandle->maxlevel_msgbus;
 $loghandle->maxlevel_msgbus(ISC_LOG_INFO);

 For syslog logging, provide C<syslog_facility> to C<new>, and:

 my $maxlevel_syslog = $loghandle->maxlevel_syslog;
 $loghandle->maxlevel_syslog(ISC_LOG_INFO);

 For stderr logging, set C<maxlevel_stderr> either in C<new>, or with:

 my $maxlevel_stderr = $loghandle->maxlevel_stderr;
 $loghandle->maxlevel_stderr(ISC_LOG_INFO);

=head1 DESCRIPTION

This module provides an interface to log messages to a file, syslog,
a msgbus group/instance, or C<stderr>.  Different thresholds can be set for
each of these.  This would allow debugging information to be written
to a file, warning and above to go to syslog, and error and other critical
messages to go to a group/instance.

One of file, syslog, stderr, or msgbus must be selected (although more
than one is allowed) or an error is returned using C<Carp>.

The default level for C<maxlevel_file> is ISC_LOG_NOTICE.

The default level for C<maxlevel_syslog> is ISC_LOG_NOTICE.

The default level for C<maxlevel_msgbus> is ISC_LOG_ERR.

The default level for C<maxlevel_stderr> is ISC_LOG_DISABLED,
meaning no messages are sent.

=over

=item new ARGS

Create a new logging context.

C<facility> specifies the facility to use when logging.  This is not
the same as C<syslog_facility> in the syslog logging method.

For file logging:

C<file> specifies the file to log messages to.

C<maxlevel_file> sets an initial logging threshold on which messages are
logged via the file method.

For msgbus logging:

C<msgbus> is a handle to a msgbus connection.

C<group> and C<instance> are the name of the group and instance to join,
and to send messages to.

C<maxlevel_msgbus> sets an initial logging threshold on which messages are
logged via the msgbus method.

For syslog logging:

C<syslog_facility> is the syslog facility name to use for logging.

C<maxlevel_syslog> sets an initial logging threshold on which messages are
logged via the syslog method.

For stderr logging:

C<maxlevel_stderr> sets an initial logging threshold on which messages are
logged via the stderr method.

=item facility NAME

If C<NAME> is specified, changes the facility name to NAME.  Returns the
current or new level.

=item level_name LEVEL

Returns a string representing the name of the log level.  For example,
C<ISC_LOG_DEBUG> would return C<"DEBUG">.

=item maxlevel_file LEVEL

If C<LEVEL> is specified, set the file level to LEVEL.  Returns the current
or new level.

=item maxlevel_msgbus LEVEL

If C<LEVEL> is specified, set the msgbus level to LEVEL.  Returns the current
or new level.

=item maxlevel_syslog LEVEL

If C<LEVEL> is specified, set the syslog level to LEVEL.  Returns the current
or new level.

=back

=head1 AUTHOR

Written by Michael Graff for the Internet Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.
