#!/usr/bin/env perl
#
# Copyright (C) 2002, 2003  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: epp-frontend.pl,v 1.25.2.1 2003/04/08 16:35:36 lidl Exp $

use lib '@prefix@';

use strict;
use warnings;

use Carp;
use Data::Dumper;
use Event qw(loop unloop);
use POSIX qw(strftime);
use Getopt::Long;
use Time::HiRes qw(gettimeofday);
use Sys::Hostname;

use ISC::Net::LengthPrefix;
use ISC::Net::Listen;
use ISC::Stats::Report;
use ISC::Log;

use ISC::SRS::EPPConnection;
use ISC::SRS::EPPResultCode;
use ISC::SRS::EPPParser;
use ISC::SRS::EPPWriter;

use constant MAX_NAMESERVERS => 13;
use constant MAX_ADDRESSES => 13;
use constant MAX_STATUS => 10;

use constant SHUTDOWN_TIMEOUT => 30;

my $hostname = hostname;
my $version = do {
    my @r = (q$Revision: 1.25.2.1 $ =~ /\d+/g);
    sprintf "%d" . ".%d" x $#r, @r;
};
my $builddate = do {
    my @r = reverse (q$Date: 2003/04/08 16:35:36 $ =~ /\d+/g);
    $r[4]--;
    $r[5] -= 1900;
    strftime("%a %b %d %H:%M:%S UTC %Y", @r);
};

my $port = $ENV{ISC_SRS_EPPPORT_NOENC} || 5544;
my $addr = $ENV{ISC_SRS_EPPADDR_NOENC} || "0.0.0.0";
my $dostats = 1;
my $srsdb_channel = "srsdb";
my $result = GetOptions("port=s" => \$port,
			"addr=s" => \$addr,
			"dostats!" => \$dostats,
			"srschannel=s" => \$srsdb_channel);

my $msgbus = new ISC::CC::Group::Connect(connect_cb => \&_connect_cb,
					 connect_error_cb => \&_connect_err,
					 error_cb => \&_msgbus_err);

my $srs;
my $stats = {
    ident => "$hostname/$addr/$port",
    start => time,

    #
    # The following are "counter" types.  That is, they start at 0 and
    # count up, without any reset.
    #
    counters => {
	dom_add => 0,
	dom_mod => 0,
	dom_del => 0,
	dom_check => 0,
	dom_status => 0,
	dom_renew => 0,
	dom_transfer => 0,

	ns_add => 0,
	ns_mod => 0,
	ns_del => 0,
	ns_check => 0,
	ns_status => 0,

	login => 0,
	describe => 0,
	quit => 0,

	badcmd => 0,

	tcp => 0,
    },

    #
    # The following are current stats, or "gauge" types.  These report the
    # number of things in use right now.
    #
    gauges => {
	tcp => 0,
	sessions => 0,

        # the following are latency gauges. These report some measure
        # of the average completion time for each transaction type.

        dom_add => 0,
        dom_mod => 0,
        dom_del => 0,
        dom_check => 0,
        dom_status => 0,
        dom_renew => 0,
        dom_transfer => 0,

        ns_add => 0,
        ns_mod => 0,
        ns_del => 0,
        ns_check => 0,
        ns_status => 0,
    
        login => 0,
        describe => 0,
        quit => 0,
    },
};

my $listener;
my $log;

# code which maintains average transaction completion times for
# individual operations. The number of recent transaction samples
# which are considered is OPTIMER_SAMPLE_LENGTH.

use constant OPTIMER_SAMPLE_LENGTH => 1000;

my %optimer;
my %opstart;
my %optype;

sub start_op {
    my ($id, $op) = @_;

    my $now = gettimeofday;

    $opstart{$id} = $now;
    $optype{$id} = $op;
}

sub stop_op {
    my ($id) = @_;

    my $st = $opstart{$id};
    my $op = $optype{$id};

    if ($st && $op) {
        my $now = gettimeofday;
        my $elapsed = $now - $st;

        my $o = $optimer{$op};

        if ($optimer{$op}) {
            my $o = $optimer{$op};

            $o->{sum} += $elapsed;
            if (@{$o->{samples}}[$o->{i}]) {
                $o->{sum} -= @{$o->{samples}}[$o->{i}];
            } else {
              $o->{n}++;
            }
            @{$o->{samples}}[$o->{i}] = $elapsed;

            $o->{i} = ($o->{i} + 1) % OPTIMER_SAMPLE_LENGTH;
        } else {
            $optimer{$op}->{sum} = $elapsed;
            $optimer{$op}->{i} = 1;
            $optimer{$op}->{n} = 1;
            @{$optimer{$op}->{samples}}[0] = $elapsed;
        }

        # populate the appropriate gauge with a mean value from
        # the sample population in the circular buffer, in nanoseconds
        # in order to avoid gratuitous floating-point values
        $stats->{gauges}->{$op} =
            int(1000000 * $optimer{$op}->{sum} / $optimer{$op}->{n});
    }
}

# end of optimer bits


sub _msgbus_err {
    my ($msgbus, $msg) = @_;

    $log->log(ISC_LOG_ERR, "msgbus error: $msg");
    unloop();
}

sub _connect_cb {
    my ($con) = @_;

    $srs = new ISC::SRS::EPPConnection(msgbus => $con,
				       srsdb_channel => $srsdb_channel,
				       feid => $stats->{ident});
    $srs->dbtimeout(30);
    $srs->fe_start($stats->{ident});

    if ($dostats) {
	my $reporter = new ISC::Stats::Report(msgbus => $msgbus,
					  freq => 6,
					  group => "stats",
					  instance => "epp-frontend",
					  cb => \&_stats_msg);
    }

    $log = new ISC::Log(facility => "epp-frontend($stats->{ident})",

			maxlevel_stderr => ISC_LOG_DEBUG,

			msgbus => $con,
			group => "log",
			instance => "epp-frontend",
			maxlevel_msgbus => ISC_LOG_INFO,
			);

    $log->log(ISC_LOG_INFO,
	      "Connected to msgbus, local name is " . $con->myname);
    $log->log(ISC_LOG_INFO, "Setting up listeners.");

    $listener = new ISC::Net::Listen(cb => \&_listen_newsock,
				     error_cb => \&_listen_err,
				     LocalPort => $port,
				     LocalAddr => $addr,
				     ReuseAddr => 1,
				     Proto => "tcp");
}

sub _stats_msg {
    return $stats;
}

sub _listen_newsock {
    my ($l, $newsock) = @_;

    my $data = {};

    my $lp = new ISC::Net::LengthPrefix(cb => \&_msg_cb,
					error_cb => \&_lp_err,
					lenbytes => 4,
					data => $data,
					oneshot => 1,
					flush => 1,
					headercounted => 1,
					socket => $newsock);
    $data->{lp} = $lp;
    $data->{login_retry} = 0;

    $log->log(ISC_LOG_INFO, "New connection " . $lp->name);

    $stats->{counters}->{tcp}++;
    $stats->{gauges}->{tcp}++;

    my $blob = {};
    $blob->{command} = "greeting";
    my $msg = epp_writer($blob);
    $lp->send($msg);
}

sub _lp_err {
    my ($lp, $msg) = @_;

    $log->log(ISC_LOG_INFO, "Shutting down lp reader " . $lp->name);

#
# probably should dummy up a logout event if logged in
# and send it to the backend, so it can nuke its state
# XXXKJL
#
    $stats->{gauges}->{tcp}--;

    my $data = $lp->data;
    $stats->{gauges}->{sessions}-- if ($data->{registrar});

    $lp->cancel;
    $lp = undef;
    $data->{lp} = undef;
}

# temporary hack to send the client a data::dump output
sub _send {
    my ($lp, $r) = @_;

    my $msg = Data::Dumper->Dump([$r], ['REPLY']);

    $lp->send($msg);
}

sub _make_response {
    my ($code, $msg) = @_;

    my $ret = { result => { resultcode => $code,
			    result => epp_result_totext($code) } };
    $ret->{result}->{msg} = $msg if ($msg);

    return $ret;
}

my $cmdmap = {
    login => \&_msg_login,
    logout => \&_msg_logout,
    hello => \&_msg_hello,

    contact_check => \&_msg_contact_check,
    contact_add => \&_msg_contact_add,
    contact_del => \&_msg_contact_del,
    contact_mod => \&_msg_contact_mod,
    contact_info => \&_msg_contact_info,
    contact_transfer => \&_msg_contact_transfer,

    host_check => \&_msg_host_check,
    host_add => \&_msg_host_add,
    host_del => \&_msg_host_del,
    host_mod => \&_msg_host_mod,
    host_info => \&_msg_host_info,
    host_transfer => \&_msg_host_transfer,

    domain_check => \&_msg_domain_check,
    domain_add => \&_msg_domain_add,
    domain_del => \&_msg_domain_del,
    domain_mod => \&_msg_domain_mod,
    domain_info => \&_msg_domain_info,
    domain_renew => \&_msg_domain_renew,
    domain_transfer => \&_msg_domain_transfer,
};

sub _msg_cb {
    my ($lp, $list) = @_;

    my $data = $lp->data;

#    print Data::Dumper->Dump([$list], ["EPP_list"]);

    my $req;

    my $parser = new ISC::SRS::EPPParser;
    eval {
	$req = $parser->parse_string($list);
    };
    if ($@) {
	print "WARNING:  crappy input XML\n";
	my $blob = {};
	$blob->{command} = "unknown";
	$blob->{resultCode} = 2001;
	$blob->{clTRID} = $lp->data->{clTRID} if (defined($lp->data->{clTRID}));
	$blob->{svTRID} = $lp->data->{svTRID} if (defined($lp->data->{svTRID}));
	my $msg = epp_writer($blob);
	$lp->send($msg);
	$lp->oneshot_reset;
	return;
    }

    my $cmd = $req->{command};

    # if the request has a client transaction id, remember it
    if (defined($req->{clTRID})) {
	$lp->data->{clTRID} = $req->{clTRID};
	$lp->data->{svTRID} = $req->{clTRID} . "-ISC-SRV-TXN";
    } else {
	# nuke any transaction ids from previous commands
	delete $lp->data->{clTRID};
	delete $lp->data->{svTRID};
    }
    $lp->data->{current_command} = $cmd;

    if (!defined($cmd)) { $cmd = ""};		# prevent useless warnings
    my $func = $cmdmap->{$cmd};
    if (!$func) {
        my $rep = _make_response(EPP_RF_UNKCMD);
        $stats->{counters}->{badcmd}++;

	my $blob = {};
	$blob->{command} = "unknown";
	$blob->{resultCode} = 2000;
	$blob->{clTRID} = $lp->data->{clTRID} if (defined($lp->data->{clTRID}));
	$blob->{svTRID} = $lp->data->{svTRID} if (defined($lp->data->{svTRID}));
	my $msg = epp_writer($blob);
	$lp->send($msg);
        $lp->oneshot_reset;
	$log->log(ISC_LOG_NOTICE, "Unknown command \"$cmd\".");
        return;
    }

    if (defined($ENV{ISC_SRS_DEBUG})) {
        my $s = Data::Dumper->Dump([$req], ['req']);
	print "Calling $cmd with:\n$s";
	undef $s;
    } else {
	print "Calling $cmd:\n";
    }

    my ($retcode, $msg) = $func->($lp, $data, $req);
    if ($retcode) {
        my $rep = _make_response($retcode, $msg);
        _send($lp, $rep);
        $lp->oneshot_reset;
    }
}


# --- miscellaneous debris ----------------------------------------- #

# the following pattern is more restrictive than that specified in
# RFC2832; RFC2832 allows IP addresses with octets greater than 255,
# and we don't.

sub _match_epp_v4_address {
  my ($v4) = @_;

  my ($a, $b, $c, $d) = split(/\./, $v4, 4);

  return ($v4 =~ m/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/ &&
    $a < 256 &&  $b < 256 && $c < 256 && $d < 256);
}

sub _match_epp_v6_address {
  my ($v6) = @_;

  return ($v6 =~ m/^[\da-fA-F]{1,4}(:[\da-fA-F]{1,4}){7}$/ ||
      $v6 =~ m/^::$/ ||
      $v6 =~ m/^([\da-fA-F]{1,4}:){1,7}:$/ ||
      $v6 =~ m/^[\da-fA-F]{1,4}:(:[\da-fA-F]{1,4}){1,6}$/ ||
      $v6 =~ m/^([\da-fA-F]{1,4}:){2}(:[\da-fA-F]{1,4}){1,5}$/ ||
      $v6 =~ m/^([\da-fA-F]{1,4}:){3}(:[\da-fA-F]{1,4}){1,4}$/ ||
      $v6 =~ m/^([\da-fA-F]{1,4}:){4}(:[\da-fA-F]{1,4}){1,3}$/ ||
      $v6 =~ m/^([\da-fA-F]{1,4}:){5}(:[\da-fA-F]{1,4}){1,2}$/ ||
      $v6 =~ m/^([\da-fA-F]{1,4}:){6}:[\da-fA-F]{1,4}$/);
}


sub _normalise_v4_address {
  my ($v4) = @_;

  $v4 =~ s/^0+(\d)/$1/;     # remove leading zeros from the first octet
  $v4 =~ s/\.0+(\d)/.$1/g;  # remove leading zeros from successive octets

  return ($v4);
}


sub _normalise_v6_address {
  my ($v6) = @_;

  $v6 =~ uc $v6;               # upper case any alphabetics
  $v6 =~ s/^0+([\dA-F])/$1/;   # remove leading zeros from the first word
  $v6 =~ s/:0+([\dA-F])/:$1/g; # remove leading zeros from successive words

  $v6 =~ s/:0:0:/::/           # introduce a :: if there isn't one already
    unless ($v6 =~ m/::/);

  $v6 =~ s/^0+::/::/;          # remove initial zero word before a ::
  $v6 =~ s/(:0)+::/::/;        # remove other zero words before a ::
  $v6 =~ s/:(:0)+/:/;         # remove zero words following a ::

  return ($v6);
}


# parse parameters/qualifiers/whatever-you-call-them in one place
# for great justice

sub _parms {
    my ($list, $spec) = @_;

    my $count = {};
    my $results = {};

    foreach my $k (keys %$spec) {
	$count->{$k} = 0;
    }

    foreach my $line (@$list) {
	my ($k, $val) = split(/:/, $line, 2);

        $k = lc $k;
#        carp "got parm '$k' value '$val'";

	if (!$spec->{$k}) {
            carp "_parms: unexpected parameter: $k";
	    return (undef);  # unknown parameter type
	}

	my $max = $spec->{$k}[1];

	if (++$count->{$k} > $max) {
            carp "_parms: too many $k parameters (more than $max)";
	    return (undef);  # too many parameters of this type
	}

	if (($max) > 1) {
	    $results->{$k} = [] unless (exists($results->{$k}));
	    push(@{$results->{$k}}, $val);
	} else {
	    $results->{$k} = $val;
	}
    }

    foreach my $k (keys %$spec) {
        if ($spec->{$k}[0] > $count->{$k}) {
            carp "_parms: not enough $k parameters";
            return (undef);  # too few parameters of this type
        }
    }

    return ($results);
}


# --- login -------------------------------------------------------- #

sub _msg_login {  
    my ($lp, $data, $list) = @_;

    $stats->{counters}->{login}++;

    start_op($lp->name, "login");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_login_cb,
	feid => $stats->{ident},
    };

    return $srs->login(%$request);
}

sub _login_cb {
    my ($srs2, $data, $result, $app) = @_;

    if (defined($ENV{ISC_SRS_DEBUG})) {
	print Data::Dumper->Dump([$result, $app], ["result", "app"]);
    }

    my $lp = $data->{lp};
    die "lp not set" unless ($lp);

    my $resp = { result => $result };

    if ($result->{resultcode} != EPP_RS_SUCCESS) {
	$data->{session_retry}++;
	if ($data->{session_retry} == 2) {
	    $log->log(ISC_LOG_NOTICE, "Session retry limit exceeded.");
	    $lp->shutdown(30);
	}
	$resp = _make_response(EPP_RF_AUTHCLOSING);
	$lp->oneshot_reset;
	return;
    }
    $data->{session_retry} = 0;

    my $blob = {};
    $blob->{command} = "login";
    $blob->{resultCode} = $result->{resultcode};
    $blob->{clTRID} = $lp->data->{clTRID} if (defined($lp->data->{clTRID}));
    $blob->{svTRID} = $lp->data->{svTRID} if (defined($lp->data->{svTRID}));
    my $msg = epp_writer($blob);
    $lp->send($msg);

    if ($result->{resultcode} =~ /^.5..$/) {
	$lp->shutdown(30);
    } else {
	$lp->oneshot_reset;
    }

    stop_op($lp->name);

    #
    # if we got a successful return, remember it.
    #
    my $registrar = $app->{registrar};
    my $id = $registrar->{id};
    my $sid = $registrar->{sid};
    $lp->data->{registrar} = $id;
    $lp->data->{sid} = $sid;

# print "saving session state: lp->data->{registrar} = $id\n";
# print "saving session state: lp->data->{sid} = $sid\n";

    $data->{registrar} = $registrar;
    $stats->{gauges}->{sessions}++;
#    warn "Got registrar id " . $registrar->id;
}

# --- logout ------------------------------------------------------- #

sub _msg_logout {  
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{logout}++;

    start_op($lp->name, "logout");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
	feid => $stats->{ident},
    };

    return $registrar->logout(%$request);
}

sub _msg_hello {
    my ($lp, $data, $list) = @_;

    $stats->{counters}->{hello}++;
    my $registrar = $data->{registrar};

    start_op($lp->name, "hello");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $srs->hello(%$request);
}

# --- contact_check --------------------------------------------------- #

sub _msg_contact_check {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{contact_check}++;

    start_op($lp->name, "contact_check");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->contact_check(%$request);
}

# --- contact_add --------------------------------------------------- #

sub _msg_contact_add {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{contact_add}++;

    start_op($lp->name, "contact_add");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->contact_add(%$request);
}

# --- contact_del --------------------------------------------------- #

sub _msg_contact_del {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{contact_del}++;

    start_op($lp->name, "contact_del");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->contact_del(%$request);
}

# --- contact_mod --------------------------------------------------- #

sub _msg_contact_mod {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{contact_mod}++;

    start_op($lp->name, "contact_mod");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->contact_mod(%$request);
}

# --- contact_info --------------------------------------------------- #

sub _msg_contact_info {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{contact_info}++;

    start_op($lp->name, "contact_info");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->contact_info(%$request);
}

# --- contact_transfer --------------------------------------------------- #

sub _msg_contact_transfer {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{contact_transfer}++;

    start_op($lp->name, "contact_transfer");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->contact_transfer(%$request);
}

# --- domain_add --------------------------------------------------- #

sub _msg_domain_add {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{domain_add}++;

    start_op($lp->name, "domain_add");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->domain_add(%$request);
}

# --- domain_check --------------------------------------------------- #

sub _msg_domain_check {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{domain_check}++;

    start_op($lp->name, "domain_check");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->domain_check(%$request);
}

# --- domain_del --------------------------------------------------- #

sub _msg_domain_del {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{domain_del}++;

    start_op($lp->name, "domain_del");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->domain_del(%$request);
}

# --- domain_mod --------------------------------------------------- #

sub _msg_domain_mod {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{domain_mod}++;

    start_op($lp->name, "domain_mod");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->domain_mod(%$request);
}

# --- domain_info --------------------------------------------------- #

sub _msg_domain_info {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{domain_info}++;

    start_op($lp->name, "domain_info");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->domain_info(%$request);
}

# --- domain_renew --------------------------------------------------- #

sub _msg_domain_renew {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{domain_renew}++;

    start_op($lp->name, "domain_renew");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->domain_renew(%$request);
}

# --- domain_transfer --------------------------------------------------- #

sub _msg_domain_transfer {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{domain_transfer}++;

    start_op($lp->name, "domain_transfer");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->domain_transfer(%$request);
}

# --- host_check --------------------------------------------------- #

sub _msg_host_check {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{host_check}++;

    start_op($lp->name, "host_check");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->host_check(%$request);
}

# --- host_add --------------------------------------------------- #

sub _msg_host_add {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{host_add}++;

    start_op($lp->name, "host_add");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->host_add(%$request);
}

# --- host_del --------------------------------------------------- #

sub _msg_host_del {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{host_del}++;

    start_op($lp->name, "host_del");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->host_del(%$request);
}

# --- host_mod --------------------------------------------------- #

sub _msg_host_mod {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{host_mod}++;

    start_op($lp->name, "host_mod");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->host_mod(%$request);
}

# --- host_info --------------------------------------------------- #

sub _msg_host_info {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{host_info}++;

    start_op($lp->name, "host_info");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->host_info(%$request);
}

# --- host_transfer --------------------------------------------------- #

sub _msg_host_transfer {
    my ($lp, $data, $list) = @_;

    my $registrar = $data->{registrar};
    return EPP_RF_AUTHORIZATION unless ($registrar);

    $stats->{counters}->{host_transfer}++;

    start_op($lp->name, "host_transfer");

    my $request = {
	req => $list,
	data => $data,
	cb => \&_generic_cb,
    };

    return $registrar->host_transfer(%$request);
}

# --- generic callback routine ------------------------------------- #

sub _generic_cb {
    my ($registrar, $data, $result, $app) = @_;

#    print Data::Dumper->Dump([$result, $app], ["result", "app"]);

    my $lp = $data->{lp};
    die "lp not set" unless ($lp);

    my $resp = { result => $result };
    $resp->{app} = $app if ($app);

    my $command;
    if (defined($lp->data->{current_command})) {
	$command = $lp->data->{current_command};
	undef $lp->data->{current_command};
    } else {
	undef $command;
    }

    my $cltxid;
    if (defined($lp->data->{clTRID})) {
	$cltxid = $lp->data->{clTRID};
	undef $lp->data->{clTRID};
    } else {
	undef $cltxid;
    }

    my $srvxid;
    if (defined($lp->data->{svTRID})) {
	$srvxid = $lp->data->{svTRID};
	undef $lp->data->{svTRID};
    } else {
	undef $srvxid;
    }

    if ($command) {
	my $blob = {};
	if ($command =~ "hello") {
	    $blob->{command} = "greeting";
	} else {
	    $blob->{command} = $command;
	    $blob->{resultCode} = $result->{resultcode};

            if ($app && scalar($app)) {
	        foreach my $key (keys %$app) {
                    $blob->{$key} = $app->{$key};
	        }
	    }
	}
	$blob->{clTRID} = $cltxid if defined($cltxid);
	$blob->{svTRID} = $srvxid if defined($srvxid);
	my $msg = epp_writer($blob);
	$lp->send($msg);
    } else {
        print "generic_cb: unset/unknown command\n";

	_send($lp, $resp);
    }

    if ($result->{resultcode} =~ /^.5..$/) {
	$lp->shutdown(30);
    } else {
	$lp->oneshot_reset;
    }

    stop_op($lp->name);
}

# --- Event Glue --------------------------------------------------- #


$Event::DIED = sub {
    Event::verbose_exception_handler(@_);
    Event::unloop_all();
};

$SIG{PIPE} = "IGNORE";

Event::loop();
