# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Listen.pm,v 1.6 2002/12/06 02:21:12 lidl Exp $

package ISC::Net::Listen;

use strict;
use warnings;

use Carp;
use Event;
use IO::Socket::UNIX;
use IO::Socket::INET;
use IO::Handle;

our @EXPORT_OK;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.6 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

END {
}

use constant LISTEN_QUEUE => 128;

my $sock_count = 0;

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;
    my $ret = bless({}, $class);

    my $name;
    if ($args{name}) {
	$name = $args{name};
    } else {
	$name = "listen" . $sock_count++;
    }

    croak "Missing argument:  cb" if (!$args{cb});
    croak "Missing argument:  error_cb" if (!$args{error_cb});

    my $proto = $args{Proto} || "tcp";

    if ($proto eq "tcp") {
	my %largs;
	$largs{LocalPort} = $args{LocalPort} if ($args{LocalPort});
	$largs{LocalAddr} = $args{LocalAddr} if ($args{LocalAddr});
	$largs{LocalHost} = $args{LocalHost} if ($args{LocalHost});
	$largs{Listen} = $args{Listen} || LISTEN_QUEUE;
	$largs{ReuseAddr} = $args{ReuseAddr} if ($args{ReuseAddr});

	$largs{Proto} = "tcp";
	$largs{Type} = SOCK_STREAM;

	$ret->{sock} = new IO::Socket::INET(%largs);
	croak "Unable to open socket: $!" unless ($ret->{sock});
	$ret->{sock}->sockopt(SO_KEEPALIVE, 1);
    } elsif ($proto eq "local" || $proto eq "unix") {
	my %largs;

	#
	# ok, this mostly sucks.  Either I'm an idiot, or the code
	# below isn't working properly.  For now, open up a socket,
	# bind it to the right path, and give the file descriptor to
	# IO::Handle.
	#
	$largs{Peer} = $args{Peer} if ($args{Peer});
	$largs{Local} = $args{Local} if ($args{Local});
	$largs{Listen} = $args{Listen} || 1;

	$ret->{sock} = new IO::Socket::UNIX(%largs);

	croak "Unable to open socket: $!" unless ($ret->{sock});
    } else {
	croak "Unknown protocol type $proto";
    }

    $ret->{sock}->blocking(0);

    $ret->{cb} = $args{cb};
    $ret->{error_cb} = $args{error_cb};
    $ret->{data} = $args{data};
    $ret->{name} = $name;

    $ret->{reader} = Event->io(desc => "$name reader",
			       fd => $ret->{sock},
			       cb => \&_sock_accept,
			       poll => "r");
    $ret->{exception} = Event->io(desc => "$name exception",
				  fd => $ret->{sock},
				  cb => \&_sock_err,
				  poll => "e");

    $ret->{reader}->data($ret);
    $ret->{exception}->data($ret);

    return $ret;
}

sub data {
    my ($self, $data) = @_;

    if (defined($data)) {
	$self->{data} = $data;
    }

    return $self->{data};
}

sub name {
    my ($self) = @_;

    return $self->{name};
}

sub socket {
    my ($self) = @_;

    return $self->{sock};
}

sub stop {
    my ($self) = @_;

    $self->{reader}->stop if defined($self->{reader});
    $self->{exception}->stop if defined($self->{exception});
}

sub start {
    my ($self) = @_;

    $self->{reader}->start if defined($self->{reader});
    $self->{exception}->start if defined($self->{exception});
}

sub cancel {
    my ($self) = @_;

    $self->{reader}->cancel;
    $self->{exception}->cancel;

    #
    # destroy this socket fully
    #
    $self->{reader} = undef;
    $self->{exception} = undef;

    #
    # close the socket down
    #
    $self->{sock} = undef;

    $self = undef;
}

#
# new connection on our local socket
#
sub _sock_accept {
    my ($ev) = @_;

    my $self = $ev->w->data;

    my $newsock = $self->{sock}->accept();
    if (!defined($newsock)) {
	$self->{error_cb}->($self, "Accept failed: $!");
	return;
    }

    $newsock->blocking(0);
    $newsock->sockopt(SO_KEEPALIVE, 1);
    $self->{cb}->($self, $newsock);
}

1;

__END__

=head1 NAME

ISC::Net::Listen - listen for incoming TCP connections

=head1 SYNOPSIS

use ISC::Net::Listen;

my $listen = new ISC::Net::Listen(
    cb => \&callback,
    error_cb => \&error_callback,
    LocalPort => 1234,
    LocalAddr => "127.0.0.1",
    Listen => 128,
    ReuseAddr => 1,
    Proto => "tcp");

callback($listen, $newsock);

error_callback($listen, $msg);

=head1 DESCRIPTION

This package uses the C<Event> package to listen on a socket, waiting for
incoming TCP connections.  For each one accepted, the callback function
is called.  If an error occurs on the listening socket, error_callback is
called.  For errors, the socket is NOT closed.

Error handling needs some work.

=over

=item new ARGS

C<cb> specifies the new connection callback.

C<error_cb> specifies the on-error callback.

C<LocalPort>, C<LocalAddr>, C<ReuseAddr>, and C<Listen> are passed directly
to the socket code to lock down the local port and/or address, to set
the reuse-addr flag, and to define the listen queue depth.  The queue
depth defaults to 128.

C<Proto> is used to select a plain old tcp or a ssl socket.  Currently,
only "tcp" is supported.

C<name> may be set.  If it is not, a default name is used.  This can be
used by the application for identification purposes.

=item data DATA

Returns the current application-defined data.  If C<DATA> is provided, it
will set a new value (and return it).

=item name

Returns the object's name.

=item socket

Returns the socket associated with this object.

=item stop

Stop all events on this object.  They may be restarted with C<start>.

=item start

Start all events on this object, after being stopped with C<stop>.

=item cancel

Fully shut down this object.  All pending events are cancelled, and
the socket is released.

=back

=head1 AUTHOR

Written by Michael Graff for the Internet Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.
