# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Group.pm,v 1.10 2002/12/06 02:21:10 lidl Exp $

package ISC::CC::Group;

use strict;
use warnings;

use ISC::CC::Direct;
use ISC::CC::Message;
use ISC::Net::LengthPrefix;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.10 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter ISC::CC::Direct);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

use constant GROUP_STATE_READY => 1;
use constant GROUP_STATE_LNAME => 2;
use constant GROUP_STATE_CONNECTING => 3;

use constant GROUP_TIMEOUT => 120;

#
# fully close down the connections
#
sub _close {
    my ($self) = @_;

    $self->{subs} = undef;
}

sub name {
    my ($self) = @_;

    return $self->{name};
}

sub _init {
    my ($self, %args) = @_;

    $self->{data} = $args{data} if ($args{data});
    $self->{subs} = {};
    $self->{replies} = {};
    $self->{timeout} = $args{timeout} || GROUP_TIMEOUT;

    return ($self);
}

sub myname {
    my ($self) = @_;

    return $self->{myname};
}

my $_type_map = {
    send => \&_type_map_send,
    getlname => \&_type_map_getlname,
    stats => \&_type_map_stats,
};

sub get_stats {
    my ($self) = @_;

    my $wmsg = {
	type => "stats",
    };
    my $buf = ISC::CC::Message::towire($wmsg);
    eval {
	$self->{lp}->send($buf);
    };
}

sub _msg_cb {
    my ($self, $buf) = @_;

    my $wmsg;

    eval {
	$wmsg = ISC::CC::Message::fromwire($buf);
    };
    if ($@) {
	warn "Got garbage message on " . $self->name;
	return;
    }

    my $type = $wmsg->{type};
    if (!$type) {
	warn "No message type provided on " . $self->name;
	return;
    }
    if (!exists($_type_map->{$type})) {
	warn "Unknown message type '$type' on " . $self->name;
	return;
    }
    $_type_map->{$type}->($self, $wmsg);
}

sub _repl_timeout {
    my ($self, $ev) = @_;

    my $seq = $ev->w->data;

    my $repl = $self->{replies}->{$seq};
    return if (!defined($repl));

    delete($self->{replies}->{$seq});

    $repl->{timer}->cancel;
    $repl->{timer} = undef;
    my $cb = $repl->{cb};
    my $channel = $repl->{channel};
    my $data = $repl->{data};

    $channel->{data} = $data;
    $cb->($channel, undef, undef);
}

sub _type_map_send {
    my ($self, $wmsg) = @_;

    my $msg;

    eval {
	$msg = ISC::CC::Message::fromwire($wmsg->{msg});
    };
    if ($@) {
	warn "Got garbage message on " . $self->name;
	return;
    }
    delete($wmsg->{msg});

    if ($wmsg->{repl}) {
	my $repl = $self->{replies}->{$wmsg->{repl}};
	if (!$repl) {
	    my $group = $wmsg->{group};
	    my $instance = $wmsg->{instance};
	    my $cb = $self->{subs}->{$group}->{$instance}->{cb};
	    if (!$cb) {
		$cb = $self->{subs}->{$group}->{"*"}->{cb};
		$instance = "*";
	    }
	    my $channel = $self->{subs}->{$group}->{$instance}->{channel};
	    if ($cb && $channel->{promisc}) {
		$cb->($channel, $msg, $wmsg);
		return;
	    }
#	    warn "Received reply to seq " . $wmsg->{repl} . " (expired?)";
	    return;
	}
	delete($self->{replies}->{$wmsg->{repl}});

	my $cb = $repl->{cb};
	my $channel = $repl->{channel};
	my $data = $repl->{data};
	$repl->{timer}->cancel;
	$repl->{timer} = undef;
	
	$channel->{data} = $data;
	$cb->($channel, $msg, $wmsg);

	return;
    }

    my $group = $wmsg->{group};
    my $instance = $wmsg->{instance};

    my $cb = $self->{subs}->{$group}->{$instance}->{cb};
    if (!$cb) {
	$cb = $self->{subs}->{$group}->{"*"}->{cb};
	$instance = "*";
    }
    if (!$cb) {
	warn "Message to group $group instance $instance has no listeners";
	return;
    }

    my $channel = $self->{subs}->{$group}->{$instance}->{channel};
    my $data = $self->{subs}->{$group}->{$instance}->{data};
    $channel->{wmsg} = $wmsg;

    $channel->{data} = $data;
    $cb->($channel, $msg, $wmsg);
}

sub _type_map_getlname {
    my ($self, $wmsg) = @_;

    $self->{myname} = $wmsg->{lname};

    $self->{state} = GROUP_STATE_READY;

    if ($self->{connect_cb}) {
	$self->{connect_cb}->($self, $self->{lp}->socket);
    }
}

sub _type_map_stats {
    my ($self, $wmsg) = @_;

    print Data::Dumper->Dump([$wmsg->{stats}], ["msgq_stats"]);
}

sub join {
    my ($self, %args) = @_;

    die "Group not in ready state"
	unless ($self->{state} == GROUP_STATE_READY);

    my $group = $args{group};

    $self->{subs}->{$group} = {} unless (exists($self->{subs}->{$group}));

    my $instance = $args{instance} || "*";
    my $cb = $args{cb};
    my $channel = new ISC::CC::GroupSocket(parent => $self,
					   group => $group,
					   instance => $instance);
    $self->{subs}->{$group}->{$instance}->{channel} = $channel;
    if ($cb) {
	$self->{subs}->{$group}->{$instance}->{cb} = $cb;
	my $subtype = "normal";
	if ($args{subtype}) {
	    if ($args{subtype} eq "promisc") {
		$subtype = "promisc";
		$channel->{promisc} = 1;
	    } elsif ($args{subtype} eq "meonly") {
		$subtype = "meonly";
	    }
	}
	$channel->_send(type => "subscribe",
			wmsg => { subtype => $subtype });
    }

    return $channel;
}

sub part {
    my ($self, %args) = @_;

    my $group = $args{group};
    my $instance = $args{instance};

    delete($self->{subs}->{$group}->{$instance});
}

package ISC::CC::GroupSocket;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.10 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;
    my $self = bless({}, $class);

    $self->{group} = $args{group};
    $self->{instance} = $args{instance} || "*";
    $self->{parent} = $args{parent};
    $self->{data} = $args{data} || undef;

    return $self;
}

sub DESTROY {
    my ($self) = @_;

    my $parent = $self->{parent};

    if ($parent) {
	$parent->part(group => $self->group, instance => $self->instance);
    }
    $self->{group} = undef;
    $self->{instance} = undef;
    $self->{parent} = undef;
    $self->{data} = undef;
}

sub data {
    my ($self, $data) = @_;

    if (defined($data)) {
	$self->{data} = $data;
    }
    return $self->{data};
}

sub group {
    my ($self) = @_;

    return $self->{group};
}

sub instance {
    my ($self) = @_;

    return $self->{instance};
}

my $seq_counter = int(rand 0x0fffffff);

sub _send {
    my ($self, %args) = @_;

    my $type = $args{type} || "send";
    my $from = $self->{parent}->{myname};
    my $group = $args{group} || $self->{group};
    my $instance = $args{instance} || $self->{instance};
    my $to = $args{to} || "*";
    my $wmsg = $args{wmsg} || {};

    $wmsg->{group} = $group;
    $wmsg->{instance} = $instance;
    $wmsg->{type} = $type;
    $wmsg->{from} = $from;
    $wmsg->{to} = $to;
    $wmsg->{repl} = $args{repl} if ($args{repl});
    $wmsg->{seq} = $seq_counter++;

    #
    # If the caller wants a callback, set it here.  We will also set a timer
    # and call the sender's callback with a timeout if we don't get a
    # response in time.
    #
    if ($args{cb}) {
	my $timeout = $args{timeout} || $self->{parent}->{timeout};
	my $data = $args{data} || undef;
	my $repl = {
	    cb => $args{cb},
	    timer => Event->timer(interval => $timeout,
				  data => $wmsg->{seq},
				  cb => [ $self->{parent}, "_repl_timeout"]),
	    channel => $self,
	    data => $data,
	};
	$self->{parent}->{replies}->{$wmsg->{seq}} = $repl;
    }

    my $buf = ISC::CC::Message::towire($wmsg);
    $self->{parent}->{lp}->send($buf);
}

sub reply {
    my ($self, %args) = @_;

    die "missing argument:  msg" if (!$args{msg});

    my $msg = ISC::CC::Message::towire($args{msg});
    my $wmsg = { msg => $msg };

    $self->_send(wmsg => $wmsg,
		 to => $self->{wmsg}->{from},
		 repl => $self->{wmsg}->{seq},
		 %args);
}

sub send {
    my ($self, %args) = @_;

    die "missing argument:  msg" if (!$args{msg});

    my $msg = ISC::CC::Message::towire($args{msg});
    my $wmsg = { msg => $msg };

    $self->_send(wmsg => $wmsg, %args);
}

1;
