# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: Random.pm,v 1.5 2002/12/06 02:21:10 lidl Exp $

package ISC::Random;

use strict;
use warnings;

use MIME::Base64;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.5 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

our @EXPORT_OK;

my $file_map = {
    strong => "/dev/random",
    medium => "/dev/urandom",
    weak => "dummy",
};

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;

    my $self = bless({}, $class);

    $self->{strength} = $args{strength} || "medium";
    if (!exists($file_map->{$self->{strength}})) {
	die "strength should be one of (strong|medium|weak)";
    }

    if ($self->{strength} eq "weak" && $args{seed}) {
	srand($args{seed});
    }

    $self->{file} = $file_map->{$self->{strength}};

    if ($self->{file} =~ /^\//) {
	local *FH;
	open(FH, $self->{file}) || die "Cannot open file " . $self->{file};
	$self->{fh} = *FH;
    }

    return $self;
}

sub DESTROY {
    my ($self) = @_;

    if ($self->{file} =~ /^\//) {
	close($self->{fh});
	$self->{fh} = undef;
    }
}

sub _get_bytes {
    my ($self, $len) = @_;

    my $buf;

    if ($self->{strength} eq "weak") {
	$buf = '';
	while ($len) {
	    my $r = int(rand 0xffffffff);
	    my $x = pack("N", $r);
	    if ($len >= 4) {
		$buf .= $x;
		$len -= 4;
	    } else {
		$buf .= substr($x, 4 - $len);
		$len = 0;
	    }
	}
    } else {
	sysread($self->{fh}, $buf, $len);
    }

    return $buf;
}

sub base64 {
    my ($self, $len) = @_;

    return encode_base64($self->_get_bytes($len), '');
}

sub hex {
    my ($self, $len) = @_;

    return unpack("h*", $self->_get_bytes($len));
}

sub bytes {
    my ($self, $len) = @_;

    return $self->_get_bytes($len);
}

1;

__END__

=head1 NAME

ISC::Random - Get weak, medium-strength, or strong randomness

=head1 SYNOPSIS

use ISC::Random;

my $weak = new ISC::Random(strength => "weak");

my $bytes = $weak->bytes(16);

my $hexstring = $weak->hex(16);

my $base64string = $weak->base64(16);

=head1 DESCRIPTION

This package will return the specified number of random bytes, in either
raw binary format, base64 encoded, or as a hex string.  By default,
"medium" strength data is returned, with options for "weak" and "strong".

=over

=item new ARGS

Create a new random object.

C<strength> specifies the random strength desired.  If set to
"weak" the Perl C<rand> function is used internally.  If it is set to
"medium" (or not specified) F</dev/urandom> is read when bytes are needed.
If set to "strong", F</dev/random> is used.

C<seed> sets the seed used for "weak" randomness.  If not specified, it
lets Perl choose.

=item bytes COUNT

Returns C<COUNT> bytes from the random source.  This call may block.

=item hex COUNT

Returns a hex string consisting of C<COUNT> bytes of data.  This means
for 16 bytes, 32 characters are returned.  This call may block.

=item base64 COUNT

Same as C<hex> except they are base64 encoded, which may be slightly more
compact than C<hex>.

=back

=head1 AUTHOR

Written by Michael Graff for the Internet Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.
