# Copyright (C) 2002  Internet Software Consortium.
#
# Permission to use, copy, modify, and distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
# DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
# INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
# INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
# FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
# NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
# WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

# $Id: CRLF.pm,v 1.16 2002/12/06 02:21:12 lidl Exp $

package ISC::Net::CRLF;

use strict;
use warnings;

use Carp;
use Event;
use IO::Socket;

our @EXPORT_OK;

BEGIN {
    use Exporter ();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);

    $VERSION = do { my @r = (q$Revision: 1.16 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r};
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = ();
}

use constant MAX_IO_SIZE => (1024 * 128);
use constant MAX_QUEUE_LENGTH => (MAX_IO_SIZE * 4);
use constant MAX_LINES => 500;
use constant MAX_LINE_LENGTH => 1024;

use constant STATE_READ_LOOKING => 1;
use constant STATE_READ_ONESHOT => 2;
use constant STATE_DEAD_JIM     => 3;
use constant STATE_READ_ONESHOTRESET => 4;
use constant STATE_SHUTTING_DOWN => 5;

my $sock_count = 0;

sub new {
    my ($class, %args) = @_;

    $class = ref($class) || $class;
    my $self = bless({}, $class);

    my $name;
    if ($args{name}) {
	$name = $args{name};
    } else {
	$name = "CRLF" . $sock_count++;
    }

    croak "Missing argument:  cb" if (!$args{cb});
    croak "Missing argument:  error_cb" if (!$args{error_cb});
    croak "Missing argument:  socket" if (!$args{socket});

    $self->{cb} = $args{cb};
    $self->{error_cb} = $args{error_cb};
    $self->{data} = $args{data};
    $self->{name} = $name;

    if (exists($args{in_block_terminator})) {
	$self->{in_block_terminator} = $args{in_block_terminator};
    } else {
	$self->{in_block_terminator} = ".";
    }
    if (exists($args{out_block_terminator})) {
	$self->{out_block_terminator} = $args{out_block_terminator};
    } else {
	$self->{out_block_terminator} = $self->{in_block_terminator};
    }


    $self->{sock} = $args{socket};

    $self->{sock}->blocking(0);
    $self->{sock}->sockopt(SO_KEEPALIVE, 1);

    $self->{write_queue} = "";
    $self->{input_queue} = "";
    $self->{input_len} = 0;
    $self->{state} = STATE_READ_LOOKING;
    $self->{lines} = [];

    $self->{oneshot} = $args{oneshot} || 0;
    $self->{flush} = $args{flush} || 0;

    $self->{max_read_size} = $args{max_read_size} || MAX_IO_SIZE;
    $self->{max_write_size} = $args{max_write_size} || MAX_IO_SIZE;
    $self->{max_queue_length} = $args{max_queue_length} || MAX_QUEUE_LENGTH;
    $self->{max_lines} = $args{max_lines} || MAX_LINES;
    $self->{max_line_length} = $args{max_line_length} || MAX_LINE_LENGTH;

    if ($self->{max_read_size} > $self->{max_queue_length}) {
	croak "Invalid arguments:  max_read_size must be <= max_queue_length";
    }

    if ($self->{max_write_size} > $self->{max_queue_length}) {
	croak "Invalid arguments:  max_write_size must be <= max_queue_length";
    }

    $self->{reader} = Event->io(desc => "$name reader",
				fd => $self->{sock},
				cb => \&_sock_read,
				poll => "r");
    $self->{writer} = Event->io(desc => "$name writer",
				fd => $self->{sock},
				cb => \&_sock_write,
				poll => "w",
				parked => 1);
    $self->{timer} = Event->timer(cb => \&_timer_cb,
				  parked => 1);


    $self->{reader}->data($self);
    $self->{writer}->data($self);
    $self->{timer}->data($self);

    return $self;
}

sub data {
    my ($self, $data) = @_;

    if (defined($data)) {
	$self->{data} = $data;
    }

    return $self->{data};
}

sub name {
    my ($self) = @_;

    return $self->{name};
}

sub socket {
    my ($self) = @_;

    return $self->{sock};
}

sub stop {
    my ($self) = @_;

    $self->{reader}->stop if defined($self->{reader});
    $self->{writer}->stop if defined($self->{writer});
    $self->{timer}->stop if defined($self->{timer});
}

sub start {
    my ($self) = @_;

    $self->{reader}->start if defined($self->{reader});
    $self->{writer}->start if defined($self->{writer});
    $self->{timer}->start if defined($self->{timer});
}

sub cancel {
    my ($self) = @_;

    $self->{write_queue} = "";
    $self->{input_queue} = "";

    $self->{reader}->cancel;
    $self->{writer}->cancel;
    $self->{timer}->cancel;

    #
    # destroy this socket fully
    #
    $self->{reader} = undef;
    $self->{writer} = undef;
    $self->{timer} = undef;

    #
    # close the socket down
    #
    $self->{sock} = undef;

    $self->{state} = STATE_DEAD_JIM;
    $self->{data} = undef;
}

#
# queue a write on a socket.  This code will append the new data to the
# queue, then set the look-for-writable flag on the socket.  The event
# system should handle the rest.
#
sub send {
    my ($self, $data) = @_;

    croak "It's dead, jim"
	if ($self->{state} == STATE_DEAD_JIM);
    croak "Cannot send when shutting down"
	if ($self->{state} == STATE_SHUTTING_DOWN);

    my @list = ();

    if (!ref($data)) {
	push(@list, $data);
    } elsif (ref($data) eq "SCALAR") {
	push(@list, $$data);
    } elsif (ref($data) eq "ARRAY") {
	@list = @$data;
    } else {
	croak "send method takes scalar, ref to scalar, or ref to array";
    }

    push(@list, $self->{out_block_terminator})
	if (defined($self->{out_block_terminator}));

    if ($self->{max_queue_length}) {
	my $len = 0;
	foreach my $line (@list) {
	    $len += length($line) + 2;
	}
	$len += length($self->{write_queue});
	if ($len > $self->{max_queue_length}) {
	    croak "Send queue length exceeded";
	}
    }

    foreach my $line (@list) {
	$self->{write_queue} .= $line . "\r\n";
    }

    $self->{writer}->start;
}

sub _timer_cb {
    my ($event) = @_;

    my $self = $event->w->data;

    if ($self->{state} == STATE_SHUTTING_DOWN) {
	$self->{error_cb}->($self, "EOF/TIMEOUT");
	$self->stop;
    }
}

sub _sock_err {
    my ($event) = @_;

    my $self = $event->w->data;

    if ($self->{state} == STATE_SHUTTING_DOWN) {
	$self->{error_cb}->($self, "EOF/SHUTDOWN");
	$self->stop;
    }

    $self->{error_cb}->($self, "Exceptional condition");
    $self->stop;
}

#
# read data from a socket.  Reads the length and data, and will call upper
# layers when there is a complete read.
#
sub _sock_read {
    my ($event) = @_;

    my $self = $event->w->data;

    if ($self->{state} == STATE_DEAD_JIM) {
	$self->stop;
	return;
    }

    if ($self->{state} == STATE_READ_ONESHOT) {
	$self->{reader}->stop;
	return;
    }

    my $sock = $self->{sock};

    #
    # Only read data from a socket that is in LOOKING state.
    #
    if ($self->{state} == STATE_READ_ONESHOTRESET) {
	$self->{state} = STATE_READ_LOOKING;
    } else {
	my $buf;
	my $cc;
	eval {
	    $cc = $sock->sysread($buf, $self->{max_read_size});
	};
	if ($@ || !defined($cc)) {
	    if ($!{EWOULDBLOCK} || $!{EINPROGRESS}) {
		return;
	    }
	    $self->{error_cb}->($self, "SOCKET READ: $!");
	    $self->{state} = STATE_DEAD_JIM;
	    $self->stop;
	    return;
	}
	if ($cc == 0) {
	    $self->{error_cb}->($self, "SOCKET READ: EOF");
	    $self->{state} = STATE_DEAD_JIM;
	    $self->stop;
	    return;
	}
	if ($self->{state} == STATE_SHUTTING_DOWN) {
	    return;
	}
	$self->{input} .= $buf;
    }

  again:
    #
    # Pull off one CRLF terminated string.  If it is the input block
    # terminator, return all the lines we've accumulated so far.
    # Otherwise, tack it on the end of our list of lines and loop.
    #
    my $term = '[\r]*[\n]+';
    if ($self->{input} =~ /$term/) {
	my ($line, $remaining) = split(/$term/, $self->{input}, 2);
	$self->{input} = $remaining;
	if ($self->{max_line_length}
	    && length($line) > $self->{max_line_length}) {
	    $self->{error_cb}->($self, "Line length exceeded");
	    $self->stop;
	    return;
	}
	if (defined($self->{in_block_terminator})) {
	    if ($line ne $self->{in_block_terminator}) {
		push(@{$self->{lines}}, $line);
		if ($self->{max_lines}
		    && scalar(@{$self->{lines}}) > $self->{max_lines}) {
		    $self->{error_cb}->($self, "Line count exceeded");
		    $self->stop;
		    return;
		}
		goto again;
	    }
	} else {
	    push(@{$self->{lines}}, $line);
	}

	if ($self->{oneshot}) {
	    $self->{reader}->stop;
	    $self->{state} = STATE_READ_ONESHOT;
	}
	$self->{cb}->($self, $self->{lines});
	$self->{lines} = [];

	goto again if (length($self->{input})
		       && $self->{state} != STATE_READ_ONESHOT);
    } elsif ($self->{max_line_length}
	     && length($self->{input}) > $self->{max_line_length}) {
	$self->{error_cb}->($self, "Line length exceeded");
	$self->stop;
    }
}

sub oneshot_reset {
    my ($self) = @_;

    return if ($self->{state} != STATE_READ_ONESHOT);

    $self->{state} = STATE_READ_ONESHOTRESET;
    if ($self->{flush} && $self->{write_queue}) {
	return;
    }
    $self->{reader}->start;
    $self->{reader}->now;
}

#
# write available on a socket
#
sub _sock_write {
    my ($event) = @_;

    my $self = $event->w->data;
    my $sock = $self->{sock};

    if (!$self->{write_queue}) {
	$self->{writer}->stop;
	return;
    }

    my $cc;
    eval {
	$cc = $sock->syswrite($self->{write_queue});
    };
    if ($@ || !defined($cc)) {
	$self->{error_cb}->($self, "SOCKET WRITE: $!");
	$self->{state} = STATE_DEAD_JIM;
	$self->stop;
	return;
    }
    if ($cc) {
	$self->{write_queue} = substr($self->{write_queue}, $cc);
	if (length($self->{write_queue}) == 0) {
	    $self->{writer}->stop;
	}
    }

    if ($self->{state} == STATE_READ_ONESHOTRESET
	&& $self->{flush}
	&& !$self->{write_queue}) {
	$self->{reader}->start;
	$self->{reader}->now;
    }

    if ($self->{state} == STATE_SHUTTING_DOWN
	&& !$self->{write_queue}) {
	$self->{sock}->shutdown(1);  # shut down writer
	$self->{input_queue} = "";
	$self->{reader}->start; # start reader.  It will toss all input.
	$self->{reader}->now;
    }
}

sub shutdown {
    my ($self, $timeout) = @_;

    $self->{state} = STATE_SHUTTING_DOWN;
    $self->{shutdown_timeout} = $timeout;

    if (!$self->{write_queue}) {
	$self->{sock}->shutdown(1);
	$self->{input_queue} = "";
	$self->{reader}->start; # start reader.  It will toss all input.
	$self->{reader}->now;
    }

    #
    # Set the timer of death.  We may want to touch the timer if
    # writes are happening, to avoid it going off when it really isn't
    # needed yet.  But, on the other hand, we may not.  Setting it here
    # gives us a definite time to live before we will drop the other
    # end forcefully.  XXXMLG
    #
    if ($self->{shutdown_timeout}) {
	$self->{timer}->at(time + $self->{shutdown_timeout});
	$self->{timer}->start;
    }
}

1;

__END__

=head1 NAME

ISC::Net::CRLF - Read a block of .<CRLF> terminated lines

=head1 SYNOPSIS

use ISC::Net::CRLF;

my $crlf = new ISC::Net::CRLF(
    cb => \&callback,
    error_cb => \&error_callback,
    socket => $sock,
    ...);

callback($crlf, $lines);

error_callback($crlf, $errmsg);

=head1 DESCRIPTION

This package uses the C<Event> to listen on a socket, reading and gathering
<CRLF> terminated lines.  Each line is returned as a callback if the
C<in_block_terminator> option is undef.  Otherwise, if it is defined,
When a line containing only the block terminator is read, the
list of lines is sent to the application callback.

=over

=item new ARGS

Create a new CRLF object.

C<cb> is the application callback, and is required.  When a block of
input is read, this function is called for every block.  See C<oneshot>,
below, for a method to keep the application in control of when these
messages are sent.

C<error_cb> is called when there is a socket error.  Errors are fatal, and
the socket should be closed down.

The C<socket> argument should be an C<IO::Socket> or a subclass, such as
C<IO::Socket::INET>.

C<data> is application specific state.  It can be set during the C<new>
call, or using the data() method.

C<name> is the application's name for this object.  It can be retrieved
using the name() method.  If it is not set, a default name will be
generated.

C<oneshot> can be set, which gives the application control over the reading
process.  If it is set, after a callback, the application must call
oneshot_reset() to receive any more events on this object.

C<flush> is used in conjunction with <oneshot> and will only cause another
read to happen after the write queue is fully written.

C<max_lines> defines how many lines can be read before returning an error.
The default is 500.  Setting it to 0 will cause all lines to be read,
potentially consuming a lot of memory.  Use with caution.

C<max_line_length> defines the longest line we will accept before returning
an error.  The default is 1024.  Setting it to 0 will cause any length of
line to be read, potentially consuming a lot of memory.  Use with caution.

C<in_block_terminator> defines the input block terminator string to
indicate a block of data has been read.  If set to "", it will return
on blank lines.  If set to "." (the default if not specified) it will
return when a line containing only "." is read.  In all cases, the callback
function receives an array of data.  If only one line is returned due to
there being no block terminator, use element [0] of the array.

C<out_block_terminator> defines the output block terminator string to
append to all lines sent.  If C<undef> no lines are appended.  Otherwise,
this string is appended to every message sent using send().

=item data DATA

Returns the current application-defined data.  If C<DATA> is provided, it
will set a new value (and return it).

=item name

Returns the object's name.

=item socket

Returns the socket associated with this object.

=item stop

Stop all events on this object.  They may be restarted with C<start>.

=item start

Start all events on this object, after being stopped with C<stop>.

=item cancel

Fully shut down this object.  All pending events are cancelled, and
the socket is released.

=item send DATA

Queue C<DATA> for output to the socket.  C<DATA> may be a scalar,
a reference to a scalar, or a reference to an array.  For scalars or
references to scalars, the string is prepended with <CRLF> and a line
consisting of .<CRLF> is added.  For a reference to an array, each item
in the array needs to be a simple scalar, and is added to the output
stream, <CRLF> terminated, with a trailing .<CRLF>.

If this will exceed a quota, it will raise an error using C<croak>.

This will implicitly start the writer process if it is stopped.

=item oneshot_reset

If the CRLF object is created with C<oneshot> set, this function must be
called when the application wishes to continue receiving callbacks for new
data.

=item shutdown TIME

Shut down the write side of the socket.  If TIME is specified, it is a
timeout on how long we will wait for output to flush and the other end
to close the socket.

=back

=head1 AUTHOR

Written by Michael Graff for the Internet Software Consortium.

=head1 COPYRIGHT

Copyright (C) 2002 Internet Software Consortium.
