--
-- Copyright (C) 2002, 2003  Internet Software Consortium.
--
-- Permission to use, copy, modify, and distribute this software for any
-- purpose with or without fee is hereby granted, provided that the above
-- copyright notice and this permission notice appear in all copies.
--
-- THE SOFTWARE IS PROVIDED "AS IS" AND INTERNET SOFTWARE CONSORTIUM
-- DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL
-- INTERNET SOFTWARE CONSORTIUM BE LIABLE FOR ANY SPECIAL, DIRECT,
-- INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
-- FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
-- NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
-- WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

-- $Id: srs.sql,v 1.61 2003/02/06 20:59:43 lidl Exp $

--
-- the first three tables do not ever get updated during normal
-- operations.  Because they are static (outside of the
-- initial loading of the values), they do not have a replication
-- version number.
--

--
-- contact name status values
--
CREATE TABLE contact_status (
    code                 CHAR(1) NOT NULL,
    status               TEXT NOT NULL,
    description          TEXT NOT NULL,
    PRIMARY KEY (code)
);
CREATE UNIQUE INDEX cstatus_idx1 on contact_status (status);

--
-- domain name status values
--
CREATE TABLE domain_status (
    code                 CHAR(1) NOT NULL,
    status               TEXT NOT NULL,
    description          TEXT NOT NULL,
    PRIMARY KEY (code)
);
CREATE UNIQUE INDEX dstatus_idx1 on domain_status (status);

--
-- host status values
--
CREATE TABLE host_status (
    code                 CHAR(1) NOT NULL,
    status               TEXT NOT NULL,
    description          TEXT NOT NULL,
    PRIMARY KEY (code)
);
CREATE UNIQUE INDEX hstatus_idx1 on host_status (status);

--
-- information on registrars in the system
--
CREATE TABLE registrar (
    st                   INT8 NOT NULL,

    registrar_id         INT,
    name                 TEXT,
    userid               TEXT,
    password             TEXT,
    whois                TEXT,
    url                  TEXT,

    PRIMARY KEY (registrar_id)
);
CREATE UNIQUE INDEX registrar_idx1 ON registrar (userid);

--
-- The table to hold information on domain name servers
--
CREATE TABLE host (
    st                   INT8 NOT NULL,

    host_id              INT NOT NULL,
    domain_id            INT,
    fqdn                 TEXT NOT NULL,
    updated_date         DATETIME,
    updated_id           INT,
    created_date         DATETIME NOT NULL,
    created_id           INT,
    registrar_id         INT,

    PRIMARY KEY (host_id),
    UNIQUE (fqdn),
    FOREIGN KEY (created_id) REFERENCES registrar
	ON UPDATE CASCADE ON DELETE SET NULL
);
CREATE INDEX host_idx1 ON host (domain_id);
CREATE INDEX host_idx2 ON host (fqdn);

CREATE TABLE host_status_map (
    st			INT8 NOT NULL,

    host_id             INT NOT NULL,
    status              CHAR(1) NOT NULL,
    lang                CHAR(2) DEFAULT 'en',
    msg                 TEXT, -- explanatory text

    UNIQUE(host_id, status),

    FOREIGN KEY (status) REFERENCES host_status
        ON UPDATE CASCADE,
    FOREIGN KEY (host_id) REFERENCES host
        ON UPDATE CASCADE ON DELETE CASCADE
);
CREATE INDEX hstatus_map_idx1 ON host_status_map (host_id);

--
-- The table to hold information on domain contacts
-- used for contacts of registrars in RRP and
-- domain contacts in thick EPP
--
CREATE TABLE contact (
    st                   INT8 NOT NULL,

    contact_id           INT NOT NULL,  -- local identifier for this contact
    handle               TEXT NOT NULL,  -- externally visible handle
    guid                 TEXT NOT NULL, -- globally-unique identifier

    voice                TEXT,  -- voice phone number
    voice_ext            TEXT,  -- extension, if needed
    fax                  TEXT,  -- fax phone number
    email                TEXT NOT NULL,  -- email address

    clid                 INT NOT NULL,  -- sponsoring client (current registrar)

    crid                 INT NOT NULL,  -- creation client (created by)
    crdate               DATETIME NOT NULL,  -- creation date

    upid                 INT,  -- most recent change made by client id
    update               DATETIME,  -- most recent change

    trstatus             TEXT, -- transfer status (if any)
    acdate               DATETIME,  -- transfer completion date or next activity

    PRIMARY KEY (contact_id),
    FOREIGN KEY (clid) REFERENCES registrar
        ON UPDATE CASCADE,
    FOREIGN KEY (crid) REFERENCES registrar
        ON UPDATE CASCADE ON DELETE SET NULL,
    FOREIGN KEY (upid) REFERENCES registrar
        ON UPDATE CASCADE ON DELETE SET NULL
);
CREATE INDEX contact_idx1 ON contact (clid);
CREATE INDEX contact_idx2 ON contact (guid);
CREATE UNIQUE INDEX contact_idx3 ON contact (handle);

--
-- currently not used -- probably a defect in the current EPP spec
-- create the table anyway, just to verify the DDL
--
CREATE TABLE registrar_authinfo (
    st			INT8 NOT NULL,

    registrar_id        INT NOT NULL,
    authtype            TEXT NOT NULL CHECK (authtype in ('pw','ext')),
    authinfo            TEXT,

    FOREIGN KEY (registrar_id) REFERENCES registrar
       ON UPDATE CASCADE ON DELETE CASCADE,
    UNIQUE(registrar_id)
);

--
-- map registers to contact objects
--
CREATE TABLE registrar_contact_map (
    st                  INT8 NOT NULL,

    registrar_id        INT NOT NULL,
    contact_id          INT,

    PRIMARY KEY (registrar_id, contact_id),
    -- the following restraint is NOT MARKED as ON DELETE CASCADE
    -- to prevent someone from doing something stupid and
    -- destroying the database by accidentally attempting to 
    -- remove a register
    FOREIGN KEY (registrar_id) REFERENCES registrar
        ON UPDATE CASCADE,
    FOREIGN KEY (contact_id) REFERENCES contact
        ON UPDATE CASCADE ON DELETE CASCADE
);
CREATE INDEX rc_idx1 ON registrar_contact_map (registrar_id);
CREATE INDEX rc_idx2 ON registrar_contact_map (contact_id);

--
-- domain name objects
--
CREATE TABLE domain (
    st                   INT8 NOT NULL,

    domain_id            INT NOT NULL,
    fqdn                 TEXT NOT NULL,
    owner_id             INT,
    updated_date         DATETIME,
    updated_id           INT,
    created_date         DATETIME NOT NULL,
    created_id           INT,
    last_transfer_date   DATETIME,
    transfer_status      INT,
    expire_date          DATETIME,
    pending_delete_date  DATETIME,
    registrar_id         INT NOT NULL,

    PRIMARY KEY (domain_id),
    UNIQUE (fqdn),
    -- the following restraints are NOT MARKED as ON DELETE CASCADE
    -- to prevent someone from doing something stupid and
    -- makeing a huge number of domains disappear when attempting
    -- an errorous deletion
    FOREIGN KEY (owner_id) REFERENCES contact
        ON UPDATE CASCADE,
    FOREIGN KEY (updated_id) REFERENCES registrar
        ON UPDATE CASCADE ON DELETE SET NULL,
    FOREIGN KEY (created_id) REFERENCES registrar
        ON UPDATE CASCADE ON DELETE SET NULL,
    FOREIGN KEY (registrar_id) REFERENCES registrar
        ON UPDATE CASCADE
);
CREATE INDEX d_idx1 on domain (domain_id);
CREATE INDEX d_idx2 on domain (fqdn);

CREATE TABLE domain_status_map (
    st			INT8 NOT NULL,

    domain_id           INT NOT NULL,
    status              CHAR(1) NOT NULL,
    lang                CHAR(2) DEFAULT 'en',
    msg                 TEXT, -- explanatory text

    UNIQUE(domain_id, status),

    FOREIGN KEY (status) REFERENCES domain_status
        ON UPDATE CASCADE,
    FOREIGN KEY (domain_id) REFERENCES domain
        ON UPDATE CASCADE ON DELETE CASCADE
);
CREATE INDEX dstatus_map_idx1 ON domain_status_map (domain_id);

CREATE TABLE domain_authinfo (
    st			INT8 NOT NULL,

    domain_id           INT NOT NULL,
    authtype            TEXT NOT NULL CHECK (authtype in ('pw','ext')),
    authinfo            TEXT,

    FOREIGN KEY (domain_id) REFERENCES domain
        ON UPDATE CASCADE ON DELETE CASCADE,
    UNIQUE(domain_id)
);

--
-- maps nameserver hosts to domains
--
CREATE TABLE domain_host_map (
    st                  INT8 NOT NULL,

    domain_id           INT NOT NULL,
    host_id             INT NOT NULL,

    FOREIGN KEY (domain_id) REFERENCES domain
	ON UPDATE CASCADE ON DELETE CASCADE,
    -- do not let someone delete a host that still has domains
    -- dependent on it (hence no ON DELETE CASCADE)!
    FOREIGN KEY (host_id) REFERENCES host
        ON UPDATE CASCADE,
    UNIQUE (host_id, domain_id)
);
CREATE INDEX dh_map_idx1 ON domain_host_map (domain_id);
-- it is not clear we need the following index --- XXXlidl
CREATE INDEX dh_map_idx2 ON domain_host_map (host_id);

-- to get a list of nameservers for all domains:
--
-- select d.fqdn, h.fqdn, i.addr
--    from domain d, host h, domain_host_map m, host_ip_map i 
--    where d.domain_id = m.domain_id and m.host_id = h.host_id and
--    h.host_id = i.host_id;

--
-- map hosts to ip addresses
-- used for generating glue A records in DNS
--
CREATE TABLE host_ip_map (
    st                  INT8 NOT NULL,

    host_id             INT NOT NULL,
    type                CHAR(1) NOT NULL CHECK (type in ('4','6')),
    addr                TEXT NOT NULL,

    FOREIGN KEY (host_id) REFERENCES host
	ON UPDATE CASCADE ON DELETE CASCADE,
    UNIQUE (host_id, type, addr)
);
CREATE INDEX hip_map_idx1 ON host_ip_map (host_id);

--
-- map contacts to domains (used in thick EPP registry)
--
CREATE TABLE domain_contact_map (
    st                  INT8 NOT NULL,

    domain_id           INT NOT NULL,
    contact_id          INT NOT NULL,
    type                TEXT NOT NULL CHECK (type in ('admin','billing','tech')),
    FOREIGN KEY (domain_id) REFERENCES domain
	ON UPDATE CASCADE ON DELETE CASCADE,
    -- deletion of a contact while it still has dependent domains
    -- should be prevented (hence no ON DELETE CASCADE restraint)
    FOREIGN KEY (contact_id) REFERENCES contact
	ON UPDATE CASCADE,
    UNIQUE(domain_id, contact_id, type)
);
CREATE INDEX dc_map_idx1 ON domain_contact_map (domain_id);

CREATE TABLE contact_postal (
    st                  INT8 NOT NULL,

    contact_id          INT NOT NULL,
    type                TEXT NOT NULL CHECK (type in ('int','loc')),
    name                TEXT NOT NULL,
    org                 TEXT,
    street1             TEXT,
    street2             TEXT,
    street3             TEXT,
    city                TEXT NOT NULL,
    sp                  TEXT,
    pc                  TEXT,
    cc                  TEXT NOT NULL,

    FOREIGN KEY (contact_id) REFERENCES contact
        ON UPDATE CASCADE ON DELETE CASCADE,
    UNIQUE(contact_id, type)
);
CREATE INDEX cpostal_idx1 ON contact_postal (contact_id);

CREATE TABLE contact_authinfo (
    st                  INT8 NOT NULL,

    contact_id          INT NOT NULL,
    authtype            TEXT NOT NULL CHECK (authtype in ('pw','ext')),
    authinfo            TEXT,

    FOREIGN KEY (contact_id) REFERENCES contact
        ON UPDATE CASCADE ON DELETE CASCADE,
    UNIQUE(contact_id)
);

CREATE TABLE contact_status_map (
    st                  INT8 NOT NULL,

    contact_id          INT NOT NULL,
    status              CHAR(1) NOT NULL,
    lang                CHAR(2) DEFAULT 'en',
    msg                 TEXT, -- explanatory text

    UNIQUE(contact_id, status),

    FOREIGN KEY (status) REFERENCES contact_status
        ON UPDATE CASCADE,
    FOREIGN KEY (contact_id) REFERENCES contact
        ON UPDATE CASCADE ON DELETE CASCADE
);
CREATE INDEX cstatus_map_idx1 on contact_status_map (contact_id);

--
-- SOA table
--
CREATE TABLE soa (
    st                   INT8 NOT NULL,

    apex                 TEXT NOT NULL,

    origin               TEXT NOT NULL,
    contact              TEXT NOT NULL,
    serial               INT NOT NULL,
    refresh              INT NOT NULL,
    retry                INT NOT NULL,
    expire               INT NOT NULL,
    minimum              INT NOT NULL,
    ttl                  INT NOT NULL
);
CREATE INDEX soa_idx1 ON soa (apex);

--
-- Sequences
--
CREATE SEQUENCE domainid_seq MINVALUE 10000 START 10000 CYCLE;
CREATE SEQUENCE contactid_seq MINVALUE 10000 START 10000 CYCLE;
CREATE SEQUENCE hostid_seq MINVALUE 10000 START 10000 CYCLE;
CREATE SEQUENCE registrarid_seq MINVALUE 10000 START 10000 CYCLE;

