// Copyright (C) 2021-2024 Internet Systems Consortium, Inc. ("ISC")
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#ifndef HTTP_COMMAND_RESPONSE_CREATOR_FACTORY_H
#define HTTP_COMMAND_RESPONSE_CREATOR_FACTORY_H

#include <http/response_creator_factory.h>
#include <config/http_command_config.h>
#include <config/http_command_response_creator.h>

namespace isc {
namespace config {

/// @brief HTTP response creator factory for HTTP control socket.
///
/// @param emulate_agent_response if true results for normal command
/// outcomes are wrapped in Element::list.  This emulates responses
/// generated by kea-ctrl-agent.  The value is passed into the
/// HttpCommandResponseCreator when created. Defaults to true.
///
/// See the documentation of the @ref isc::http::HttpResponseCreatorFactory
/// for the details how the response factory object is used by the
/// @ref isc::http::HttpListener.
///
/// This class always returns the same instance of the
/// @ref HttpCommandResponseCreator which @ref isc::http::HttpListener and
/// @ref isc::http::HttpConnection classes use to generate HTTP response
/// messages which comply with the formats required by the Control Agent.
class HttpCommandResponseCreatorFactory : public http::HttpResponseCreatorFactory {
public:

    /// @brief Constructor.
    ///
    /// Creates sole instance of the @ref HttpCommandResponseCreator object
    /// returned by the @ref HttpCommandResponseCreatorFactory::create.
    ///
    /// @param config The HTTP control socket config.
    HttpCommandResponseCreatorFactory(HttpCommandConfigPtr config)
        : sole_creator_(new HttpCommandResponseCreator(config)) {
    }

    /// @brief Returns an instance of the @ref HttpCommandResponseCreator which
    /// is used by HTTP server to generate responses to commands.
    ///
    /// @return Pointer to the @ref HttpCommandResponseCreator object.
    virtual http::HttpResponseCreatorPtr create() const {
        return (sole_creator_);
    }

private:

    /// @brief Instance of the @ref HttpCommandResponseCreator returned.
    http::HttpResponseCreatorPtr sole_creator_;
};

} // end of namespace isc::config
} // end of namespace isc

#endif
