/* packet-pcp.c
 * Routines for Port Control Protocol packet disassembly.
 * draft-ietf-pcp-base-12.txt
 *
 * Copyright 2010-2011, Francis Dupont <fdupont@isc.org>
 *
 * $Id: packet-pcp.c 1247 2011-07-03 20:22:42Z fdupont $
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <epan/packet.h>
#include <epan/expert.h>
#include <epan/ipproto.h>

#define PNAME  "Port Control Protocol"
#define PSNAME "PCP"
#define PFNAME "pcp"

/* defines */
#define PCP_VERSION			1
#define PCP_RESPONSE			128
#define PCP_OC_MAP4			1
#define PCP_OC_MAP4_RESPONSE		PCP_OC_MAP4 | PCP_RESPONSE
#define PCP_OC_MAP6			2
#define PCP_OC_MAP6_RESPONSE		PCP_OC_MAP6 | PCP_RESPONSE
#define PCP_OC_PEER4			3
#define PCP_OC_PEER4_RESPONSE		PCP_OC_PEER4 | PCP_RESPONSE
#define PCP_OC_PEER6			4
#define PCP_OC_PEER6_RESPONSE		PCP_OC_PEER6 | PCP_RESPONSE
#define PCP_AF_IPv4			1
#define PCP_AF_IPv6			2
#define PCP_RC_SUCCESS			0
#define PCP_RC_UNSUPP_VERSION		1
#define PCP_RC_MALFORMED_REQUEST	2
#define PCP_RC_UNSUPP_OPCODE		3
#define PCP_RC_UNSUPP_OPTION		4
#define PCP_RC_MALFORMED_OPTION		5
#define PCP_RC_PROCESSING_ERROR		6
#define PCP_RC_SERVER_OVERLOADED	7
#define PCP_RC_ADDRESS_MISMATCH		8  /* XXX not defined in the spec */
#define PCP_RC_NETWORK_FAILURE		20
#define PCP_RC_NO_RESOURCES		21
#define PCP_RC_UNSUPP_PROTOCOL		22
#define PCP_RC_NOT_AUTHORIZED		23
#define PCP_RC_USER_EX_QUOTA		24
#define PCP_RC_CANT_PROVIDE_EXT_PORT	25
#define PCP_RC_EXCESSIVE_REMOTE_PEERS	26
#define PCP_RC_UNAUTH_THIRD_PARTY	51
#define PCP_PROTO_ALL			0
#define PCP_OPT_UNPROCESSED		0
#define PCP_OPT_FILTER			2
#define PCP_OPT_PREFER_FAILURE		3
#define PCP_OPT_THIRD_PARTY		4

static int proto_pcp = -1;

static int hf_version = -1;
static int hf_opcode = -1;
static int hf_result_code = -1;
static int hf_epoch = -1;
static int hf_protocol = -1;
static int hf_external_af = -1;
static int hf_client_address4 = -1;
static int hf_client_address6 = -1;
static int hf_external_address4_requested = -1;
static int hf_external_address4_assigned = -1;
static int hf_external_address6_requested = -1;
static int hf_external_address6_assigned = -1;
static int hf_lifetime_requested = -1;
static int hf_lifetime_assigned = -1;
static int hf_internal_port = -1;
static int hf_external_port_requested = -1;
static int hf_external_port_assigned = -1;
static int hf_option_code = -1;
static int hf_option_length = -1;
static int hf_remote_peer_port = -1;
static int hf_remote_peer_address4 = -1;
static int hf_remote_peer_address6 = -1;
static int hf_third_party_address4 = -1;
static int hf_third_party_address6 = -1;
static int hf_filter_prefix_len = -1;
static int hf_filter_remote_peer_port = -1;
static int hf_filter_remote_peer_address4 = -1;
static int hf_filter_remote_peer_address6 = -1;
static int hf_unprocessed = -1;

static gint ett_pcp = -1;
static gint ett_option = -1;

static const value_string opcode_vals[] = {
  { PCP_OC_MAP4,           "MAP4 Request"   },
  { PCP_OC_MAP4_RESPONSE,  "MAP4 Response"  },
  { PCP_OC_MAP6,           "MAP6 Request"   },
  { PCP_OC_MAP6_RESPONSE,  "MAP6 Response"  },
  { PCP_OC_PEER4,          "PEER4 Request"  },
  { PCP_OC_PEER4_RESPONSE, "PEER4 Response" },
  { PCP_OC_PEER6,          "PEER6 Request"  },
  { PCP_OC_PEER6_RESPONSE, "PEER6 Response" },
  { 0, NULL }
};

static const value_string result_vals[] = {
  { PCP_RC_SUCCESS,                "success"                             },
  { PCP_RC_UNSUPP_VERSION,         "unsupported version"                 },
  { PCP_RC_MALFORMED_REQUEST,      "malformed request"                   },
  { PCP_RC_UNSUPP_OPCODE,          "unsupported opcode"                  },
  { PCP_RC_UNSUPP_OPTION,          "unsupported option"                  },
  { PCP_RC_MALFORMED_OPTION,       "malformed option"                    },
  { PCP_RC_PROCESSING_ERROR,       "processing error"                    },
  { PCP_RC_SERVER_OVERLOADED,      "server overloaded"                   },
  { PCP_RC_NETWORK_FAILURE,        "network failure"                     },
  { PCP_RC_NO_RESOURCES,           "out of resources"                    },
  { PCP_RC_UNSUPP_PROTOCOL,        "unsupported protocol"                },
  { PCP_RC_NOT_AUTHORIZED,         "not authorized"                      },
  { PCP_RC_USER_EX_QUOTA,          "user exceeded quota"                 },
  { PCP_RC_CANT_PROVIDE_EXT_PORT,  "cannot provide external port"        },
  { PCP_RC_EXCESSIVE_REMOTE_PEERS, "excessive remote peers"              },
  { PCP_RC_UNAUTH_THIRD_PARTY,     "unauth third party internal address" },
  { PCP_RC_ADDRESS_MISMATCH,       "address mismatch"                    },
  { 0, NULL }
};

static const value_string protocol_vals[] = {
  { PCP_PROTO_ALL, "ALL" },
  { IP_PROTO_TCP,  "TCP" },
  { IP_PROTO_UDP,  "UDP" },
  { 0, NULL }
};

static const value_string externalaf_vals[] = {
  { PCP_AF_IPv4, "IPv4" },
  { PCP_AF_IPv6, "IPv6" },
  { 0, NULL }
};

static const value_string option_vals[] = {
  { PCP_OPT_UNPROCESSED,    "Unprocessed"    },
  { PCP_OPT_FILTER,         "Filter"         },
  { PCP_OPT_PREFER_FAILURE, "Prefer Failure" },
  { PCP_OPT_THIRD_PARTY,    "Third-Party"    },
  { 0, NULL }
};

static void dissect_pcp (tvbuff_t *tvb, packet_info *pinfo, proto_tree *tree)
{
  proto_tree *pcp_tree, *opt_tree;
  proto_item *ti, *ver_ti, *op_ti;
  gint offset, olen;
  guint8 version, opcode, extaf, opt;

  offset = 0;
  if (check_col (pinfo->cinfo, COL_PROTOCOL))
    col_set_str (pinfo->cinfo, COL_PROTOCOL, PSNAME);

  if (check_col (pinfo->cinfo, COL_INFO))
    col_clear (pinfo->cinfo, COL_INFO);

  ti = proto_tree_add_item (tree, proto_pcp, tvb, offset, -1, FALSE);
  pcp_tree = proto_item_add_subtree (ti, ett_pcp);

  ver_ti = proto_tree_add_item (pcp_tree, hf_version, tvb, offset, 1, FALSE);
  version = tvb_get_guint8 (tvb, offset);
  if (version != PCP_VERSION) {
    expert_add_info_format (pinfo, ver_ti, PI_RESPONSE_CODE,
                            PI_WARN, "Unknown version: %d", version);
    return;
  }
  offset++;

  opcode = tvb_get_guint8 (tvb, offset);
  if ((opcode & PCP_RESPONSE) == 0)
    proto_item_append_text (ti, ", Request");
  else
    proto_item_append_text (ti, ", Response");
  switch (opcode & ~PCP_RESPONSE) {
  case PCP_OC_MAP4:
    proto_item_append_text (ti, " MAP4");
    break;
  case PCP_OC_MAP6:
    proto_item_append_text (ti, " MAP6");
    break;
  case PCP_OC_PEER4:
    proto_item_append_text (ti, " PEER4");
    break;
  case PCP_OC_PEER6:
    proto_item_append_text (ti, " PEER6");
    break;
  default:
    proto_item_append_text (ti, " ??");
    break;
  }
  if (check_col (pinfo->cinfo, COL_INFO)) {
    if ((opcode & PCP_RESPONSE) == 0)
      col_add_str (pinfo->cinfo, COL_INFO, "Request");
    else
      col_add_str (pinfo->cinfo, COL_INFO, "Response");
  }
  op_ti = proto_tree_add_item (pcp_tree, hf_opcode, tvb, offset, 1, FALSE);
  offset += 2;

  if ((opcode & PCP_RESPONSE) != 0)
    proto_tree_add_item (pcp_tree, hf_result_code, tvb, offset, 1, FALSE);
  offset++;

  if ((opcode & PCP_RESPONSE) == 0)
    proto_tree_add_item (pcp_tree, hf_lifetime_requested,
                         tvb, offset, 4, FALSE);
  else
    proto_tree_add_item (pcp_tree, hf_lifetime_assigned,
                         tvb, offset, 4, FALSE);
  offset += 4;

  if ((opcode & PCP_RESPONSE) != 0)
    proto_tree_add_item (pcp_tree, hf_epoch, tvb, offset, 4, FALSE);
  offset += 4;

  switch (opcode & ~PCP_RESPONSE) {
  case PCP_OC_MAP4:
  case PCP_OC_PEER4:
    proto_tree_add_item (pcp_tree, hf_client_address4,
                         tvb, offset, 4, FALSE);
    break;
  case PCP_OC_MAP6:
  case PCP_OC_PEER6:
    proto_tree_add_item (pcp_tree, hf_client_address6,
                         tvb, offset, 16, FALSE);
    break;
  default:
    /* unknown opcode: infer address family from packet header */
    if (pinfo->src.type == AT_IPv4)
      proto_tree_add_item (pcp_tree, hf_client_address4,
                           tvb, offset, 4, FALSE);
    else if (pinfo->src.type == AT_IPv6)
      proto_tree_add_item (pcp_tree, hf_client_address6,
                           tvb, offset, 16, FALSE);
    break;
  }
  offset += 16;

  switch (opcode & ~PCP_RESPONSE) {
  case PCP_OC_MAP4:
  case PCP_OC_MAP6:
    proto_tree_add_item (pcp_tree, hf_protocol, tvb, offset, 1, FALSE);
    offset += 4;
    proto_tree_add_item (pcp_tree, hf_internal_port, tvb, offset, 2, FALSE);
    offset += 2;
    if ((opcode & PCP_RESPONSE) == 0)
      proto_tree_add_item (pcp_tree, hf_external_port_requested,
                           tvb, offset, 2, FALSE);
    else
      proto_tree_add_item (pcp_tree, hf_external_port_assigned,
                           tvb, offset, 2, FALSE);
    offset += 2;
    switch (opcode) {
    case PCP_OC_MAP4:
      proto_tree_add_item (pcp_tree, hf_external_address4_requested,
                           tvb, offset, 4, FALSE);
      offset += 4;
      break;
    case PCP_OC_MAP4_RESPONSE:
      proto_tree_add_item (pcp_tree, hf_external_address4_assigned,
                           tvb, offset, 4, FALSE);
      offset += 4;
      break;
    case PCP_OC_MAP6:
      proto_tree_add_item (pcp_tree, hf_external_address6_requested,
                           tvb, offset, 16, FALSE);
      offset += 16;
      break;
    case PCP_OC_MAP6_RESPONSE:
      proto_tree_add_item (pcp_tree, hf_external_address6_assigned,
                           tvb, offset, 16, FALSE);
      offset += 16;
      break;
    }
    break;
  case PCP_OC_PEER4:
  case PCP_OC_PEER6:
    proto_tree_add_item (pcp_tree, hf_protocol, tvb, offset, 1, FALSE);
    offset++;
    proto_tree_add_item (pcp_tree, hf_external_af, tvb, offset, 1, FALSE);
    extaf = tvb_get_guint8 (tvb, offset);
    offset += 3;
    proto_tree_add_item (pcp_tree, hf_internal_port, tvb, offset, 2, FALSE);
    offset += 2;
    if ((opcode & PCP_RESPONSE) == 0)
      proto_tree_add_item (pcp_tree, hf_external_port_requested,
                           tvb, offset, 2, FALSE);
    else
      proto_tree_add_item (pcp_tree, hf_external_port_assigned,
                           tvb, offset, 2, FALSE);
    offset += 2;
    proto_tree_add_item (pcp_tree, hf_remote_peer_port, tvb, offset, 2, FALSE);
    offset += 4;
    switch (opcode & ~PCP_RESPONSE) {
    case PCP_OC_PEER4:
      proto_tree_add_item (pcp_tree, hf_remote_peer_address4,
                           tvb, offset, 4, FALSE);
      offset += 4;
      break;
    case PCP_OC_PEER6:
      proto_tree_add_item (pcp_tree, hf_remote_peer_address6,
                           tvb, offset, 16, FALSE);
      offset += 16;
      break;
    }
    switch (extaf) {
    case PCP_AF_IPv4:
      if (opcode & PCP_RESPONSE)
        proto_tree_add_item (pcp_tree, hf_external_address4_assigned,
                             tvb, offset, 4, FALSE);
      else
        proto_tree_add_item (pcp_tree, hf_external_address4_requested,
                             tvb, offset, 4, FALSE);
      break;
    case PCP_AF_IPv6:
      if (opcode & PCP_RESPONSE)
        proto_tree_add_item (pcp_tree, hf_external_address6_assigned,
                             tvb, offset, 16, FALSE);
      else
        proto_tree_add_item (pcp_tree, hf_external_address6_requested,
                             tvb, offset, 16, FALSE);
      break;
    }
    offset += 16;
    break;
  default:
    expert_add_info_format (pinfo, op_ti, PI_RESPONSE_CODE,
                            PI_WARN, "Unknown opcode: %d", opcode);
    return;
  }
    
  while (tvb_length_remaining (tvb, offset) > 0) {
    opt = tvb_get_guint8 (tvb, offset);
    ti = proto_tree_add_item (pcp_tree, hf_option_code, tvb, offset, 1, FALSE);
    opt_tree = proto_item_add_subtree (ti, ett_option);
    offset += 2;
    olen = tvb_get_ntohs(tvb, offset);
    proto_tree_add_item (opt_tree, hf_option_length, tvb, offset, 2, FALSE);
    offset += 2;
    switch (opt) {
    case PCP_OPT_UNPROCESSED:
      proto_tree_add_item (opt_tree, hf_unprocessed, tvb, offset, olen, FALSE);
      break;
    case PCP_OPT_FILTER:
      proto_tree_add_item (opt_tree, hf_filter_prefix_len, tvb,
                           offset + 1, 1, FALSE);
      proto_tree_add_item (opt_tree, hf_remote_peer_port, tvb,
                           offset + 2, 2, FALSE);
      if (olen == 8)
        proto_tree_add_item (opt_tree, hf_remote_peer_address4, tvb,
                             offset + 4, 4, FALSE);
      else if (olen == 20)
        proto_tree_add_item (opt_tree, hf_remote_peer_address6,tvb,
                             offset + 4, 16, FALSE);
      break;
    case PCP_OPT_PREFER_FAILURE:
      break;
    case PCP_OPT_THIRD_PARTY:
      if (olen == 4)
        proto_tree_add_item (opt_tree, hf_third_party_address4, tvb,
                             offset, 4, FALSE);
      else if (olen == 16)
        proto_tree_add_item (opt_tree, hf_third_party_address6, tvb,
                             offset, 16, FALSE);
      break;
    default:
      break;
    }
    olen = (olen + 3) & ~3;
    offset += olen;
  }
}

void proto_register_pcp (void)
{
  static hf_register_info hf[] = {
    { &hf_version,
      { "Version", "pcp.version", FT_UINT8, BASE_DEC,
        NULL, 0x0, NULL, HFILL } },
    { &hf_opcode,
      { "OpCode", "pcp.opcode", FT_UINT8, BASE_DEC,
        VALS(opcode_vals), 0x0, NULL, HFILL } },
    { &hf_result_code,
      { "Result Code", "pcp.result_code", FT_UINT8, BASE_DEC,
        VALS(result_vals), 0x0, NULL, HFILL } },
    { &hf_lifetime_requested,
      { "Requested Lifetime", "pcp.lifetime", FT_UINT32, BASE_DEC,
        NULL, 0x0, "Requested Port Mapping Lifetime in Seconds", HFILL } },
    { &hf_lifetime_assigned,
      { "Lifetime", "pcp.lifetime", FT_UINT32, BASE_DEC,
        NULL, 0x0, "Port Mapping Lifetime in Seconds", HFILL } },
    { &hf_epoch,
      { "Epoch", "pcp.epoch", FT_UINT32, BASE_DEC,
        NULL, 0x0, "Seconds Since Start of Epoch", HFILL } },
    { &hf_client_address4,
      { "PCP Client's IP address", "pcp.internal_address", FT_IPv4, BASE_NONE,
        NULL, 0x0, NULL, HFILL } },
    { &hf_client_address6,
      { "PCP Client's IP address", "pcp.internal_address", FT_IPv6, BASE_NONE,
        NULL, 0x0, NULL, HFILL } },
    { &hf_protocol,
      { "Protocol", "pcp.protocol", FT_UINT8, BASE_DEC,
        VALS(protocol_vals), 0x0, NULL, HFILL } },
    { &hf_external_af,
      { "External AF", "pcp.external_af", FT_UINT8, BASE_DEC,
        VALS(externalaf_vals), 0x0, NULL, HFILL } },
    { &hf_internal_port,
      { "Internal Port", "pcp.internal_port", FT_UINT16, BASE_DEC,
        NULL, 0x0, NULL, HFILL } },
    { &hf_external_port_requested,
      { "Suggested External Port", "pcp.external_port", FT_UINT16, BASE_DEC,
        NULL, 0x0, NULL, HFILL } },
    { &hf_external_port_assigned,
      { "Assigned External Port", "pcp.external_port", FT_UINT16, BASE_DEC,
        NULL, 0x0, NULL, HFILL } },
    { &hf_remote_peer_port,
      { "Remote Peer Port", "pcp.remote_peer_port", FT_UINT16, BASE_DEC,
        NULL, 0x0, NULL, HFILL } },
    { &hf_remote_peer_address4,
      { "Remote Peer IP Address", "pcp.remote_peer_address", FT_IPv4,
        BASE_NONE, NULL, 0x0, NULL, HFILL } },
    { &hf_remote_peer_address6,
      { "Remote Peer IP Address", "pcp.remote_peer_address", FT_IPv6,
        BASE_NONE, NULL, 0x0, NULL, HFILL } },
    { &hf_external_address4_requested,
      { "Suggested External IP Address", "pcp.external_address",
        FT_IPv4, BASE_NONE, NULL, 0x0, NULL, HFILL } },
    { &hf_external_address6_requested,
      { "Suggested External IP Address", "pcp.external_address",
        FT_IPv6, BASE_NONE, NULL, 0x0, NULL, HFILL } },
    { &hf_external_address4_assigned,
      { "Assigned External IP Address", "pcp.external_address",
        FT_IPv4, BASE_NONE, NULL, 0x0, NULL, HFILL } },
    { &hf_external_address6_assigned,
      { "Assigned External IP Address", "pcp.external_address",
        FT_IPv6, BASE_NONE, NULL, 0x0, NULL, HFILL } },
    { &hf_option_code,
      { "Option Code", "pcp.option_code", FT_UINT8, BASE_DEC,
        VALS(option_vals), 0x0, NULL, HFILL } },
    { &hf_option_length,
      { "Option Length", "pcp.option_length", FT_UINT8, BASE_DEC,
        NULL, 0x0, NULL, HFILL } },
    { &hf_third_party_address4,
      { "Internal IP Address", "pcp.third_party_address", FT_IPv4,
        BASE_NONE, NULL, 0x0, NULL, HFILL } },
    { &hf_third_party_address6,
      { "Internal IP Address", "pcp.third_party_address", FT_IPv6,
        BASE_NONE, NULL, 0x0, NULL, HFILL } },
    { &hf_unprocessed,
      { "Unprocessed Options", "pcp.unprocessed", FT_BYTES, BASE_NONE,
        NULL, 0x0, NULL, HFILL } },
    { &hf_filter_prefix_len,
      { "Prefix Length", "pcp.filter_prefix_len", FT_UINT8, BASE_DEC,
        NULL, 0x0, NULL, HFILL } },
    { &hf_filter_remote_peer_port,
      { "Remote Peer Port", "pcp.filter_remote_peer_port", FT_UINT16, BASE_DEC,
        NULL, 0x0, NULL, HFILL } },
    { &hf_filter_remote_peer_address4,
      { "Remote Peer IP Address", "pcp.filter_remote_peer_address", FT_IPv4,
        BASE_NONE, NULL, 0x0, NULL, HFILL } },
    { &hf_filter_remote_peer_address6,
      { "Remote Peer IP Address", "pcp.filter_remote_peer_address", FT_IPv6,
        BASE_NONE, NULL, 0x0, NULL, HFILL } }
  };

  static gint *ett[] = {
    &ett_pcp,
    &ett_option,
  };

  proto_pcp = proto_register_protocol (PNAME, PSNAME, PFNAME);
  register_dissector (PFNAME, dissect_pcp, proto_pcp);
  
  proto_register_field_array (proto_pcp, hf, array_length (hf));
  proto_register_subtree_array (ett, array_length (ett));
}

void proto_reg_handoff_pcp (void)
{
  dissector_handle_t pcp_handle;

  pcp_handle = find_dissector (PFNAME);
  dissector_add ("nat-pmp.version", PCP_VERSION, pcp_handle);
}

/*
 * Editor modelines
 *
 * Local Variables:
 * c-basic-offset: 2
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * ex: set shiftwidth=2 tabstop=8 noexpandtab
 * :indentSize=2:tabSize=8:noTabs=false:
 */
