/* $id: upnpredirect.c,v 1.49 2009/12/22 17:20:10 nanard Exp $ */
/* MiniUPnP project
 * http://miniupnp.free.fr/ or http://miniupnp.tuxfamily.org/
 * (c) 2006-2009 Thomas Bernard 
 * This software is subject to the conditions detailed
 * in the LICENCE file provided within the distribution */

#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>

#include <stdio.h>
#include <ctype.h>
#include <unistd.h>

#include "config.h"
#include "upnpglobalvars.h"
#include "upnpredirect.h"
#include "upnpevents.h"
#include "upnpsoap.h"
#include "pcpiwf.h"
#include <sys/stat.h>

/* proto_atoi() 
 * convert the string "UDP" or "TCP" to IPPROTO_UDP and IPPROTO_TCP */
static int
proto_atoi(const char * protocol)
{
	int proto = IPPROTO_TCP;

	if (strcmp(protocol, "UDP") == 0)
		proto = IPPROTO_UDP;
	return proto;
}

static struct upnp_redirect *
add_redirect(unsigned short eport, const char *iaddr,
		      unsigned short iport, int proto,
		      const char *desc)
{
	struct upnp_redirect *p;
	size_t l = 1;
	int ret;

	if (desc)
		l += strlen(desc);
	l += sizeof(struct upnp_redirect);
	p = (struct upnp_redirect *)malloc(l);
	if (p == NULL)
		return NULL;
	memset(p, 0, l);
	l -= sizeof(struct upnp_redirect);
	ret = OpenPCPServerSocket(&p->t);
	if (ret != PCP_OK) {
		free(p);
		return NULL;
	}
	p->t.parent = p;
	p->t.type = TTYPE_UPNP;
	p->t.status = PCP_NOT_READY;
	LIST_INSERT_HEAD(&transactions, &p->t, chain);
	strncpy(p->iaddr, iaddr, INET_ADDRSTRLEN);
	p->proto = (char)proto;
	p->iport = iport;
	p->eport = eport;
	if (desc)
		memcpy(p->desc, desc, l);
	SystemUpdateID++;
	return p;
}

static struct upnp_redirect *
del_redirect_internal(unsigned short eport, int proto)
{
	struct transaction *t;
	struct upnp_redirect *p;

	LIST_FOREACH(t, &transactions, chain) {
		if (t->type != TTYPE_UPNP)
			continue;
		p = t->parent;
		if ((p->eport == eport) && (p->proto == (char)proto)) {
			(void) pcp_close(&t->pcp);
			memset(&t->pcp, 0, sizeof(t->pcp));
			memset(&t->response, 0, sizeof(t->response));
			LIST_REMOVE(t, chain);
			if (t->retrans.action != NULL)
				LIST_REMOVE(&t->retrans, chain);
			memset(&t->retrans, 0, sizeof(t->retrans));
			if (p->renew.action != NULL)
				LIST_REMOVE(&p->renew, chain);
			memset(&p->renew, 0, sizeof(p->renew));
			if (p->expire.action != NULL)
				LIST_REMOVE(&p->expire, chain);
			memset(&p->expire, 0, sizeof(p->expire));
			return p;
		}
	}
	return NULL;
}

static void
expire_redirect(void *arg)
{
	struct upnp_redirect  *r = (struct upnp_redirect *) arg;

	SystemUpdateID++;
	r = del_redirect_internal(r->eport, (int)r->proto);
	if (r != NULL)
		free(r);
	else
		syslog(LOG_ERR, "can't find expiring redirect");
}

static void
cancel_redirect(struct upnp_redirect *r)
{
	r = del_redirect_internal(r->eport, (int)r->proto);
	if (r != NULL)
		free(r);
	else
		syslog(LOG_ERR, "can't find canceling redirect");
}

static void
renew_redirect(void *arg)
{
	struct upnp_redirect *r = (struct upnp_redirect *) arg;
	pcp_request_t req;
	pcp_option_t **options = NULL;
	uint32_t clientaddr;

	memset(&req, 0, sizeof(req));
	/* opcode (map4) */
	req.opcode = PCP_OPCODE_MAP4;
	/* protocol */
	req.protocol = r->proto;
	/* lifetime */
	if (r->lifetime) {
		r->lifetime = (uint32_t) (r->expire.when.tv_sec - now.tv_sec);
		r->lifetime += 1;
		req.lifetime = r->lifetime;
	} else
		req.lifetime = 2592000U;
	/* internal address (192.0.0.2) */
	if (inet_pton(AF_INET, "192.0.0.2", &req.intaddr4) <= 0) {
		syslog(LOG_WARNING, "renew_redirect(inet_pton0)");
		return;
	}
	/* internal port */
	req.intport = r->iport;
	/* external port (unused) */
	req.extport = r->eport;
	/* client address */
	if (inet_pton(AF_INET, r->iaddr, &clientaddr) <= 0) {
		syslog(LOG_WARNING, "renew_redirect(inet_pton)");
		return;
	}	
	if (pcp_third_party4(&options, clientaddr) != PCP_OK) {
		syslog(LOG_WARNING, "renew_redirect(pcp_third_party)");
		return;
	}
	if (pcp_prefer_failure(&options) != PCP_OK) {
		syslog(LOG_WARNING, "renew_redirect(pcp_prefer_failure)");
		return;
	}
	if (pcp_makerequest(&r->t.pcp, &req, options) != PCP_OK) {
		syslog(LOG_WARNING, "renew_redirect(pcp_makerequest)");
		pcp_freeoptions(options);
		return;
	}
	pcp_freeoptions(options);
	r->t.handler = UPnPRenewHandler;
	pcp_send_request(&r->t);
}

static struct upnp_redirect *
get_redirect(unsigned short eport, int proto)
{
	struct transaction *t;
	struct upnp_redirect *p;

	LIST_FOREACH(t, &transactions, chain) {
		if (t->type != TTYPE_UPNP)
			continue;
		p = t->parent;
		if ((p->eport == eport) && (p->proto == (char)proto))
			return p;
	}
	return NULL;
}

static struct upnp_redirect *
get_redirect_internal(const char *iaddr, unsigned short iport, int proto)
{
	struct transaction *t;
	struct upnp_redirect *p;

	LIST_FOREACH(t, &transactions, chain) {
		if (t->type != TTYPE_UPNP)
			continue;
		p = t->parent;
		if ((strcmp(iaddr, p->iaddr) == 0) &&
		    (p->iport == iport) &&
		    (p->proto == (char)proto))
			return p;
	}
	return NULL;
}

struct upnp_redirect *
upnp_get_redirect(unsigned short eport, const char * protocol)
{
	return get_redirect(eport, proto_atoi(protocol));
}

struct upnp_redirect *
upnp_get_redirect_by_index(int index)
{
	struct transaction *t;
	int i = 0;

	LIST_FOREACH(t, &transactions, chain) {
		if (t->type != TTYPE_UPNP)
			continue;
		if (i == index)
			break;
		i++;
	}
	if (t == NULL)
		return NULL;
	return t->parent;
}

/* upnp_get_redirect_number()
 * TODO: improve this code */
int
upnp_get_redirect_number(void)
{
	struct transaction *t;
	int n = 0;

	LIST_FOREACH(t, &transactions, chain)
		if (t->type == TTYPE_UPNP)
			n++;
	return n;
}

struct upnp_redirect *
upnp_add_redirect(struct upnphttp *h, unsigned short * eport,
		  const char * iaddr, unsigned short iport,
		  int proto, unsigned int life,
		  const char * desc, int kind)
{
	struct upnp_redirect *r;
	pcp_request_t req;
	pcp_option_t **options = NULL;
	uint32_t clientaddr;

	r = add_redirect(*eport, iaddr, iport, proto, desc);
	if (r == NULL)
		return NULL;
	r->h = h;
	r->lifetime = life;

	/* translate */
	memset(&req, 0, sizeof(req));
	/* opcode (map4) */
	req.opcode = PCP_OPCODE_MAP4;
	/* protocol */
	req.protocol = proto;
	/* lifetime */
	if (life)
		req.lifetime = (uint32_t)life;
	else
		req.lifetime = 2592000U;
	/* internal address (192.0.0.2) */
	if (inet_pton(AF_INET, "192.0.0.2", &req.intaddr4) <= 0) {
		syslog(LOG_WARNING, "renew_redirect(inet_pton0)");
		goto bad;
	}
	/* ports */
	req.intport = iport;
	req.extport = *eport;
	/* client address */
	if (inet_pton(AF_INET, iaddr, &clientaddr) <= 0) {
		syslog(LOG_WARNING, "upnp_add_redirect(inet_pton)");
		goto bad;
	}
	if (pcp_third_party4(&options, clientaddr) != PCP_OK) {
		syslog(LOG_WARNING, "upnp_add_redirect(pcp_third_party)");
		goto bad;
	}

	if ((kind >= 0) &&
	    (pcp_prefer_failure(&options) != PCP_OK)) {
		syslog(LOG_WARNING, "upnp_add_redirect(pcp_prefer_failure)");
		goto bad;
	}

	if (pcp_makerequest(&r->t.pcp, &req, options) != PCP_OK) {
		syslog(LOG_WARNING, "upnp_add_redirect(pcp_makerequest)");
		pcp_freeoptions(options);
		goto bad;
	}
	pcp_freeoptions(options);

	if (life) {
		r->expire.when.tv_sec = now.tv_sec + life;
		r->expire.when.tv_usec = now.tv_usec + life;
		r->expire.arg = r;
	}

	if (kind < 0)
		r->t.handler = UPnPAddAnyHandler;
	else {
		r->t.handler = UPnPAddHandler;
		r->kind = kind;
	}
	pcp_send_request(&r->t);
	return r;

    bad:
	(void) pcp_close(&r->t.pcp);
	LIST_REMOVE(&r->t, chain);
	free(r);
	return NULL;
}

/* upnp_static_redirect() 
 * calls OS/fw dependant implementation of the redirection.
 * protocol should be the string "TCP" or "UDP"
 * returns: 0 on expected success
 *          -1 failed to redirect
 *          -2 already redirected
 *          -3 permission check failed
 */
int
upnp_static_redirect(unsigned short eport,
		     const char * iaddr, unsigned short iport,
		     const char * protocol, const char * desc)
{
	int proto;
	struct in_addr address;
	struct upnp_redirect *r;

	proto = proto_atoi(protocol);
	if (inet_aton(iaddr, &address) < 0) {
		syslog(LOG_ERR, "inet_aton(%s) : %m", iaddr);
		return -1;
	}

	if (!check_permissions(upnppermlist, num_upnpperm,
			       eport, address, iport)) {
		syslog(LOG_INFO,
		       "redirection permission check failed for "
		       "%hu->%s:%hu %s",
		       eport, iaddr, iport, protocol);
		return -3;
	}
	r = get_redirect(eport, proto);
	if (r != NULL) {
		syslog(LOG_INFO,
		       "port %hu protocol %s already redirected to %s:%hu",
		       eport, protocol, r->iaddr, r->iport);
		return -2;
	}
	syslog(LOG_INFO,
	       "redirecting port %hu to %s:%hu protocol %s for: %s",
	       eport, iaddr, iport, protocol, desc);
	r = upnp_add_redirect(NULL, &eport, iaddr, iport, proto, 0, desc, 0);
	if (r == NULL)
		return -1;
	return 0;
}

/* upnp_dynamic_redirect() 
 * calls OS/fw dependant implementation of the redirection.
 * protocol should be the string "TCP" or "UDP"
 */
void
upnp_dynamic_redirect(struct upnphttp *h,
		      unsigned short * eport,
		      const char * iaddr, unsigned short iport,
		      const char * protocol, unsigned int life,
		      const char * desc, int kind)
{
	int proto;
	struct in_addr address;
	struct upnp_redirect *r;

	proto = proto_atoi(protocol);
	if (inet_aton(iaddr, &address) < 0) {
		syslog(LOG_ERR, "inet_aton(%s) : %m", iaddr);
		SoapError(h, 501, "ActionFailed");
	}

	if ((*eport != 0) &&
	    !check_permissions(upnppermlist, num_upnpperm,
			       *eport, address, iport)) {
		syslog(LOG_INFO,
		       "redirection permission check failed for "
		       "%hu->%s:%hu %s",
		       *eport, iaddr, iport, protocol);
		if (kind != 0)
			SoapError(h, 606, "Action not authorized");
		else
			SoapError(h, 718, "ConflictInMappingEntry");
		return;
	}
	if (kind >= 0) {
		r = get_redirect(*eport, proto);
		if (r != NULL) {
			/* if existing redirect rule matches redirect request
			 * return success xbox 360 does not keep track of the
			 * port it redirects and will redirect another port
			 * when receiving ConflictInMappingEntry */
			if ((strcmp(iaddr, r->iaddr) == 0) &&
			    (iport == r->iport))
				goto renew;
			else {
				syslog(LOG_INFO,
				       "port %hu protocol %s already "
				       "redirected to %s:%hu",
				       *eport, protocol, r->iaddr, r->iport);
				SoapError(h, 718, "ConflictInMappingEntry");
				return;
			}
		}
	} else {
		r = get_redirect_internal(iaddr, iport, proto);
		if (r != NULL) {
			*eport = r->eport;
			goto renew;
		}
		if (*eport != 0)
			r = get_redirect(*eport, proto);
		if (r != NULL)
			*eport = 0;
	}
	syslog(LOG_INFO,
	       "redirecting port %hu to %s:%hu protocol %s life %u for: %s",
	       *eport, iaddr, iport, protocol, life, desc);
	r = upnp_add_redirect(h, eport, iaddr, iport,
			      proto, life, desc, kind);
	if (r == NULL)
		SoapError(h, 501, "ActionFailed");
	return;

    renew:
	syslog(LOG_INFO,
	       "renew redirect request as it matches existing redirect");
	if (life) {
		r->expire.when.tv_sec = now.tv_sec + life;
		r->expire.when.tv_usec = now.tv_usec + life;
	}
	if ((r->lifetime == 0) && (life != 0)) {
		r->expire.action = expire_redirect;
		r->expire.arg = r;
		LIST_INSERT_HEAD(&timeouts, &r->expire, chain);
	} else if ((r->lifetime != 0) && (life == 0)) {
		LIST_REMOVE(&r->expire, chain);
		memset(&r->expire, 0, sizeof(r->expire));
	}
	r->lifetime = life;
	if (r->renew.action != NULL) {
		LIST_REMOVE(&r->renew, chain);
		memset(&r->renew, 0, sizeof(r->renew));
	}
	renew_redirect(r);
	if (kind < 0)
		AddAnyPortMappingHandler(h, *eport);
	else
		AddPortMappingHandler(h, kind);
}

int
upnp_delete_redirect(unsigned short eport, const char * protocol, int kind)
{
	struct upnp_redirect *r;
	pcp_request_t req;
	pcp_option_t **options = NULL;
	int proto, ret;
	uint32_t clientaddr;

	SystemUpdateID++;
	syslog(LOG_INFO, 
	       "removing redirect rule port %hu %s",
	       eport, protocol);
	proto = proto_atoi(protocol);
	r = del_redirect_internal(eport, proto);

#ifdef ENABLE_EVENTS
	upnp_event_var_change_notify(EWanIPC);
#endif
	if (r == NULL)
		return -1;
	ret = OpenPCPServerSocket(&r->t);
	if (ret != PCP_OK) {
		syslog(LOG_WARNING, "delete_(init)");
		return -1;
	}
	r->t.status = PCP_NOT_READY;
	/* translate */
	memset(&req, 0, sizeof(req));
	/* opcode (map4) */
	req.opcode = PCP_OPCODE_MAP4;
	/* protocol */
	req.protocol = proto;
	/* lifetime (0) */
	/* internal address (192.0.0.2) */
	if (inet_pton(AF_INET, "192.0.0.2", &req.intaddr4) <= 0) {
		syslog(LOG_WARNING, "renew_redirect(inet_pton0)");
		(void) pcp_close(&r->t.pcp);
		return -1;
	}
	/* internal port */
	req.intport = r->iport;
	/* external port (unused) */
	req.extport = r->eport;
	/* client address */
	if (inet_pton(AF_INET, r->iaddr, &clientaddr) <= 0) {
		syslog(LOG_WARNING, "delete_(inet_pton)");
		(void) pcp_close(&r->t.pcp);
		return -1;
	}
	if (pcp_third_party4(&options, clientaddr) != PCP_OK) {
		syslog(LOG_WARNING, "delete_(pcp_third_party)");
		return -1;
	}

	if (pcp_makerequest(&r->t.pcp, &req, options) != PCP_OK) {
		syslog(LOG_WARNING, "delete_(pcp_makerequest)");
		(void) pcp_close(&r->t.pcp);
		pcp_freeoptions(options);
		return -1;
	}
	pcp_freeoptions(options);

	LIST_INSERT_HEAD(&transactions, &r->t, chain);
	r->t.handler = UPnPDelHandler;
	r->kind = kind;
	pcp_send_request(&r->t);
	return 0;
}

void
UPnPAddHandler(struct transaction *t)
{
	struct upnp_redirect *p = t->parent;
	uint32_t lt;

	if (t->status > PCP_OK)
		return;
	else if (t->status < PCP_OK) {
		syslog(LOG_WARNING, "UPnPAddHandler got error %d", t->status);
		switch (t->status) {
		case PCP_ERR_NOTAUTH:
			if (p->kind)
				SoapError(p->h, 606, "Action not authorized");
			else
				SoapError(p->h, 718, "ConflictInMappingEntry");
			break;
		case PCP_ERR_UNSUPOPTION:
		case PCP_ERR_CANTPROVIDE:
			SoapError(p->h, 718, "ConflictInMappingEntry");
			break;
		case PCP_ERR_NORESOURCES:
		case PCP_ERR_EXQUOTA:
			if (p->kind)
				SoapError(p->h, 728, "NoPortMapsAvailable");
			else
				SoapError(p->h, 501, "ActionFailed");
			break;
		default:
			SoapError(p->h, 501, "ActionFailed");
			break;
		}
		cancel_redirect(p);
		return;
	}

	if (p->lifetime != 0) {
		p->expire.action = expire_redirect;
		p->expire.arg = p;
		LIST_INSERT_HEAD(&timeouts, &p->expire, chain);
	}
#ifdef ENABLE_EVENTS
	upnp_event_var_change_notify(EWanIPC);
#endif

	/* get real lifetime */
	lt = t->response.assigned.lifetime;
	if ((p->lifetime == 0) || (lt < p->lifetime)) {
		p->t.status = PCP_NOT_READY;
		p->renew.when.tv_sec = now.tv_sec + (3 * lt) / 4;
		p->renew.when.tv_usec = now.tv_usec;
		p->renew.action = renew_redirect;
		p->renew.arg = p;
		LIST_INSERT_HEAD(&timeouts, &p->renew, chain);
	}
	if (p->h != NULL)
		AddPortMappingHandler(p->h, p->kind);
}

void
UPnPAddAnyHandler(struct transaction *t)
{
	struct upnp_redirect *p = t->parent;
	uint32_t lt;

	if (t->status > PCP_OK)
		return;
	else if (t->status < PCP_OK) {
		syslog(LOG_WARNING, "UPnPAddHandler got error %d", t->status);
		switch (t->status) {
		case PCP_ERR_NOTAUTH:
			SoapError(p->h, 606, "Action not authorized");
			break;
		case PCP_ERR_NORESOURCES:
		case PCP_ERR_EXQUOTA:
			SoapError(p->h, 728, "NoPortMapsAvailable");
			break;
		default:
			SoapError(p->h, 501, "ActionFailed");
			break;
		}
		cancel_redirect(p);
		return;
	}

	p->eport = t->response.assigned.extport;

	if (p->lifetime != 0) {
		p->expire.action = expire_redirect;
		p->expire.arg = p;
		LIST_INSERT_HEAD(&timeouts, &p->expire, chain);
	}
#ifdef ENABLE_EVENTS
	upnp_event_var_change_notify(EWanIPC);
#endif

	/* get real lifetime */
	lt = t->response.assigned.lifetime;
	if ((p->lifetime == 0) || (lt < p->lifetime)) {
		p->t.status = PCP_NOT_READY;
		p->renew.when.tv_sec = now.tv_sec + (3 * lt) / 4;
		p->renew.when.tv_usec = now.tv_usec;
		p->renew.action = renew_redirect;
		p->renew.arg = p;
		LIST_INSERT_HEAD(&timeouts, &p->renew, chain);
	}
	if (p->h != NULL)
		AddAnyPortMappingHandler(p->h, p->eport);
}

void
UPnPDelHandler(struct transaction *t)
{
	struct upnp_redirect *p = t->parent;

	if (t->status > PCP_OK)
		return;
	if (t->status < PCP_OK) {
		syslog(LOG_ERR, "UPnPDelHandler got error %d", t->status);
		if (p->kind >= 0)
			SoapError(p->h, 501, "ActionFailed");
	} else if ((p->kind >= 0) && (p->h != NULL))
		DelPortMappingHandler(p->h, p->kind);
	expire_redirect(p);
}

void
UPnPRenewHandler(struct transaction *t)
{
	struct upnp_redirect *p = t->parent;
	uint32_t lt;

	if (t->status > PCP_OK)
		return;
	else if (t->status < PCP_OK) {
		syslog(LOG_ERR, "UPnPRenewHandler got error %d", t->status);
		SystemUpdateID++;
		cancel_redirect(p);
		return;
	}

	/* get real lifetime */
	lt = t->response.assigned.lifetime;
	if ((p->lifetime == 0) || (lt < p->lifetime)) {
		p->renew.when.tv_sec = now.tv_sec + (3 * lt) / 4;
		p->renew.when.tv_usec = now.tv_usec;
		p->renew.action = renew_redirect;
		p->renew.arg = p;
		LIST_INSERT_HEAD(&timeouts, &p->renew, chain);
	} else
		memset(&p->renew, 0, sizeof(p->renew));
}
