/* $Id: natpmp.c,v 1.20 2010/05/06 13:42:47 nanard Exp $ */
/* MiniUPnP project
 * (c) 2007-2010 Thomas Bernard
 * http://miniupnp.free.fr/ or http://miniupnp.tuxfamily.org/
 * This software is subject to the conditions detailed
 * in the LICENCE file provided within the distribution */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "config.h"
#include "natpmp.h"
#include "upnpglobalvars.h"
#include "upnpredirect.h"
#include "pcpiwf.h"

#ifdef ENABLE_NATPMP

int OpenNATPMPSocket(in_addr_t addr)
{
	int snatpmp;

	snatpmp = socket(PF_INET, SOCK_DGRAM, 0/*IPPROTO_UDP*/);
	if (snatpmp < 0) {
		syslog(LOG_ERR, "socket(natpmp): %m");
		return -1;
	} else {
		struct sockaddr_in natpmp_addr;

		memset(&natpmp_addr, 0, sizeof(natpmp_addr));
		natpmp_addr.sin_family = AF_INET;
		natpmp_addr.sin_port = htons(NATPMP_PORT);
		//natpmp_addr.sin_addr.s_addr = INADDR_ANY;
		natpmp_addr.sin_addr.s_addr = addr;
		//natpmp_addr.sin_addr.s_addr = inet_addr("192.168.0.1");
		if (bind(snatpmp,
			 (struct sockaddr *)&natpmp_addr,
			 sizeof(natpmp_addr)) < 0) {
			syslog(LOG_ERR, "bind(natpmp): %m");
			(void)close(snatpmp);
			return -1;
		}
	}
	return snatpmp;
}

int OpenNATPMPSockets(int *sockets)
{
	int i, j;

	for (i = 0; i < n_lan_addr; i++) {
		sockets[i] = OpenNATPMPSocket(lan_addr[i].addr.s_addr);
		if (sockets[i] < 0) {
			for (j = 0; j < i; j++) {
				(void)close(sockets[j]);
				sockets[j] = -1;
			}
			return -1;
		}
	}
	return 0;
}

struct natpmp_request {
	struct transaction t;
	struct sockaddr_in client;
	unsigned char req[12];
	int s;
	struct timeout expire;
};

static void
natpmp_expire(void *arg)
{
	struct natpmp_request *n = (struct natpmp_request *) arg;

	(void) pcp_close(&n->t.pcp);
	LIST_REMOVE(&n->t, chain);
	if (n->t.retrans.action != NULL)
		LIST_REMOVE(&n->t.retrans, chain);
	n->t.retrans.action = NULL;
	n->t.retrans.arg = NULL;
	n->expire.action = NULL;
	n->expire.arg = NULL;
	free(n);
}

void
natpmp_handler(struct transaction *t)
{
	struct natpmp_request *n = t->parent;
	unsigned char resp[16];
	uint32_t n32;
	uint16_t n16;

	/* ready? */
	if (t->status > PCP_OK)
		return;
	/* remap recv() syscall failure */
	if (t->status == PCP_ERR_RECV)
		t->status = PCP_ERR_NETFAILURE;
	/* not protocol errors */
	if ((t->status < PCP_OK) && (t->status > PCP_ERR_PROTOBASE)) {
		syslog(LOG_WARNING,
		       "NAT-PMP failed: %s",
		       pcp_strerror(t->status));
		return;
	}
		
	memset(resp, 0, sizeof(resp));
	/* version (0) */
	/* opcode (req + 128) */
	resp[1] = 128 + n->req[1];
	/* result code */
	switch (t->status) {
	case PCP_OK:
		break;
	case PCP_ERR_NETFAILURE:
		resp[3] = 3;
		break;
	case PCP_ERR_NORESOURCES:
	case PCP_ERR_EXQUOTA:
		resp[3] = 4;
		break;
	case PCP_ERR_NOTAUTH:
	default:
		resp[3] = 2;
		break;
	}
	/* sssoe */
	n32 = htonl(now.tv_sec - startup_time);
	memcpy(resp + 4, &n32, 4);
	/* ports */
	n16 = htons(t->response.assigned.intport);
	memcpy(resp + 8, &n16, 2);
	n16 = htons(t->response.assigned.extport);
	memcpy(resp + 10, &n16, 2);
	/* lifetime */
	n32 = htonl(t->response.assigned.lifetime);
	memcpy(resp + 12, &n32, 4);

	/* send response back to the client */
	if (sendto(n->s, resp, sizeof(resp), 0,
		   (struct sockaddr *) &n->client, sizeof(n->client)) < 0)
		syslog(LOG_ERR, "sendto(natpmp): %m");
}

void
call_pcp(struct sockaddr_in *client, unsigned char *req, int s)
{
	struct transaction *t;
	struct natpmp_request *n;
	pcp_request_t pcp;
	pcp_option_t **options = NULL;
	uint32_t n32;
	uint16_t n16;
	int ret;

	/* known? */
	LIST_FOREACH(t, &transactions, chain) {
		if (t->type != TTYPE_NATPMP)
			continue;
		n = t->parent;
		if ((n->s == s) &&
		    (n->client.sin_addr.s_addr == client->sin_addr.s_addr) &&
		    (n->client.sin_port == client->sin_port) &&
		    (memcmp(n->req, req, 12) == 0)) {
			if (t->status <= PCP_OK)
				(t->handler)(t);
			return;
		}
	}
	
	/* new one */
	n = (struct natpmp_request *)malloc(sizeof(*n));
	if (n == NULL)
		return;
	memset(n, 0, sizeof(*n));
	/* fill natpmp request structure */
	ret = OpenPCPServerSocket(&n->t);
	if (ret != PCP_OK) {
		syslog(LOG_ERR, "call_pcp(init): %s", pcp_strerror(ret));
		free(n);
		return;
	}
	n->t.parent = n;
	n->t.type = TTYPE_NATPMP;
	n->t.status = PCP_NOT_READY;
	memcpy(&n->client, client, sizeof(*client));
	memcpy(n->req, req, sizeof(n->req));
	n->s = s;

	/* translate */
	memset(&pcp, 0, sizeof(pcp));
	/* opcode */
	pcp.opcode = PCP_OPCODE_MAP4;
	/* protocol */
	pcp.protocol = (req[1] == 1) ? IPPROTO_UDP : IPPROTO_TCP;
	/* lifetime */
	memcpy(&n32, req + 8, 4);
	pcp.lifetime = ntohl(n32);
	/* ports */
	memcpy(&n16, req + 4, 2);
	pcp.intport = ntohs(n16);
	memcpy(&n16, req + 6, 2);
	pcp.extport = ntohs(n16);
	/* internal address */
	if (pcp_third_party4(&options, client->sin_addr.s_addr) != PCP_OK) {
		syslog(LOG_ERR, "natpmp(call_pcp): %s", pcp_strerror(ret));
		(void) pcp_close(&n->t.pcp);
		free(n);
		return;
	}

	ret = pcp_makerequest(&n->t.pcp, &pcp, NULL);
	if (ret != PCP_OK) {
		syslog(LOG_ERR, "natpmp(call_pcp): %s", pcp_strerror(ret));
		(void) pcp_close(&n->t.pcp);
		free(n);
		return;
	}

	n->expire.when.tv_sec = now.tv_sec + 60;
	n->expire.when.tv_usec = now.tv_usec;
	n->expire.action = natpmp_expire;
	n->expire.arg = n;
	LIST_INSERT_HEAD(&timeouts, &n->expire, chain);

	LIST_INSERT_HEAD(&transactions, &n->t, chain);
	n->t.handler = natpmp_handler;
	pcp_send_request(&n->t);
}

/** read the request from the socket, process it and then send the
 * response back.
 */
void ProcessNATPMPRequest(int s)
{
	unsigned char req[32];	/* request udp packet */
	unsigned char resp[32];	/* response udp packet */
	int resplen;
	struct sockaddr_in senderaddr;
	socklen_t senderaddrlen = sizeof(senderaddr);
	int n;
	char senderaddrstr[16];
	uint32_t tm;
	uint16_t iport;		/* private port */
	uint16_t eport;		/* public port */
	uint32_t lifetime;	/* lifetime == 0 => remove port mapping */
	int proto;

	n = recvfrom(s, req, sizeof(req), MSG_PEEK,
	             (struct sockaddr *)&senderaddr, &senderaddrlen);
	if (n < 0) {
		syslog(LOG_ERR, "recvfrom(natpmp/peek): %m");
		return;
	}
	n = recvfrom(s, req, sizeof(req), 0,
	             (struct sockaddr *)&senderaddr, &senderaddrlen);
	if (n < 0) {
		syslog(LOG_ERR, "recvfrom(natpmp): %m");
		return;
	}
	if (inet_ntop(AF_INET, &senderaddr.sin_addr,
	              senderaddrstr, sizeof(senderaddrstr)) == NULL)
		syslog(LOG_ERR, "inet_ntop(natpmp): %m");
	syslog(LOG_INFO,
	       "NAT-PMP request received from %s:%hu %dbytes",
	       senderaddrstr, ntohs(senderaddr.sin_port), n);
	if ((n < 2) || ((((req[1] - 1) & ~1) == 0) && (n < 12))) {
		syslog(LOG_WARNING,
		       "discarding NAT-PMP request (too short) %dBytes",
		       n);
		return;
	}
	if ((req[1] & 128) != 0) {
		/* discarding NAT-PMP responses silently */
		return;
	}
	memset(resp, 0, sizeof(resp));
	resplen = 8;
	resp[1] = 128 + req[1];	/* response OPCODE is request OPCODE + 128 */
	/* setting response TIME STAMP */
	tm = htonl(now.tv_sec - startup_time);
	memcpy(resp + 4, &tm, 4);
	if (req[0] > 0) {
		/* invalid version */
		syslog(LOG_WARNING,
		       "unsupported NAT-PMP version : %u",
		       (unsigned)req[0]);
		resp[3] = 1;	/* unsupported version */
	} else switch(req[1]) {
	case 0:	/* Public address request */
		syslog(LOG_INFO, "NAT-PMP public address request");
		if (external_addr == 0)
			/* Network Failure */
			resp[3] = 3;
		memcpy(resp + 8, &external_addr, 4);
		resplen = 12;
		break;
	case 1:	/* UDP port mapping request */
	case 2:	/* TCP port mapping request */
		memcpy(&iport, req + 4, 2);
		memcpy(&eport, req + 6, 2);
		memcpy(&lifetime, req + 8, 4);
		proto = (req[1] == 1) ? IPPROTO_UDP : IPPROTO_TCP;
		syslog(LOG_INFO,
		       "NAT-PMP port mapping request : "
		       "%hu->%s:%hu %s lifetime=%us",
		       ntohs(eport), senderaddrstr, ntohs(iport),
		       (req[1] == 1) ? "udp" : "tcp", ntohl(lifetime));
		if (external_addr == 0) {
			/* not yet ready */
			return;
		}
		if (lifetime == 0) {
			/* remove the mapping */
			if (iport == 0)
				syslog(LOG_INFO,
				       "NAT-PMP %s %s removing port mappings",
				       proto == IPPROTO_TCP ? "TCP" : "UDP",
				       senderaddrstr);
			call_pcp(&senderaddr, req, s);
			return;
		}
		if ((iport == 0) ||
		    !check_permissions(upnppermlist, num_upnpperm,
				       eport == 0 ? iport : eport,
				       senderaddr.sin_addr, iport)) {
			resp[3] = 2;	/* Not Authorized/Refused */
			resplen = 16;
			break;
		}
		call_pcp(&senderaddr, req, s);
		return;
	default:
		resp[3] = 5;	/* Unsupported OPCODE */
	}
	n = sendto(s, resp, resplen, 0,
	           (struct sockaddr *)&senderaddr, sizeof(senderaddr));
	if (n < 0)
		syslog(LOG_ERR, "sendto(natpmp): %m");
        else if (n < resplen)
		syslog(LOG_ERR,
		       "sendto(natpmp): sent only %d bytes out of %d",
		       n, resplen);
}

/* SendNATPMPNotification()
 * should be called when the public IP address changed */
void SendNATPMPNotification(int *sockets, int n_sockets)
{
	struct sockaddr_in sockname;
	unsigned char notif[12];
	int j, n;
	uint32_t tm;

	notif[0] = 0;
	notif[1] = 128;
	notif[2] = 0;
	notif[3] = 0;
	tm = htonl(now.tv_sec - startup_time);
	memcpy(notif + 4, &tm, 4);
	memcpy(notif + 8, &external_addr, 4);
	if (external_addr == 0) {
		syslog(LOG_WARNING,
		       "%s: cannot get public IP address, stopping",
		       "SendNATPMPNotification");
		return;
	}
	memset(&sockname, 0, sizeof(struct sockaddr_in));
	sockname.sin_family = AF_INET;
	sockname.sin_port = htons(NATPMP_PORT);
	sockname.sin_addr.s_addr = inet_addr(NATPMP_NOTIF_ADDR);

	for (j = 0; j < n_sockets; j++) {
		if (sockets[j] < 0)
			continue;
		n = sendto(sockets[j], notif, 12, 0,
		           (struct sockaddr *)&sockname, sizeof(sockname));
		if (n < 0) {	
			syslog(LOG_ERR,
			       "%s: sendto(s_udp=%d): %m",
			       "SendNATPMPNotification",
			       sockets[j]);
			return;
		}
	}
}
#endif

