/* $Id: relayd.cc 1.2 Fri, 18 Jul 1997 15:55:45 -0700 wlee $
// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): eddy@isi.edu
*/

#include "relayd.hh"
#include <sys/types.h>
#include <sys/stat.h>
#include "Argv.hh"
#include "version.hh"
#include "Allocator.hh"

// globals that can be set from set_options

char *opt_config_file = NULL;
int  opt_server_port = 5055;
int  opt_waittime = 0;

int  debug = 0;
int  verbose = 0;
char *progname = NULL;
char *logfile = NULL;

FILE *relaylog = stdout;

#define MAXLINE 256

TCP server;

void death (int dummy) {
    server.close();
    exit (0);
}

main (int argc, char **argv, char **envp)
{
    signal (SIGTERM, death);
    signal (SIGINT,  death);

    set_options(argc, argv, envp);
    if (logfile) {
	relaylog = fopen (logfile, "w");
	if (relaylog == NULL) {
	    perror ("Could not open logfile");
	    exit (1);
	}
    }
    
    read_config (opt_config_file);

    // XXX: cause verbose to use a log file unless -d is set...
    if (!debug && !verbose) {
	if (go_daemonic())
	    exit (0);
    }

    // open up a tcp socket to listen on opt_server_port
    server (opt_server_port);
    server.error.setErrorlog (relaylog);
    server.error.die (progname);

    int finished = 0;
    while (!finished) {
	Socket client = server.accept (); // wait for a client request
	client.error.setErrorlog (relaylog);
	client.set_sockdst (server.get_socknew()); 
	client.error.die (progname);
	server.error.die (progname);

	struct sockaddr_in *cli = client.get_sockdst();
	ipAddr ipaddr(cli->sin_addr.s_addr);

	// XXX: use a log file
	if (verbose)
	    fprintf (relaylog,  "Client request from: %s\n", ipaddr.getName()); 
	
	if (!debug) {
	    int pid = fork();
	    if (!pid) {
		server.close();
		relay (client);
		finished = 1;
	    }
	} else {
	    relay (client);
	}
	client.close();
	if (verbose)
	    fprintf (relaylog, "Client session terminated: %s\n", ipaddr.getName());
    }
    exit (0);
}

Cache cache;

void
relay (Socket &client)
{
    Whois whois;
    Error error;
    QueryList qlist;
    
    whois.Open();
    if (!whois.is_open())
	whois.error.die ("relayd");
    client.fdopen();

    fd_set allsock, incoming;
    bzero ((char *) &allsock, sizeof (fd_set));
    bzero ((char *) &incoming, sizeof (fd_set));

    FD_SET (client.getsock(), &allsock);
    FD_SET (whois.in_fileno(), &allsock);
    int maxsock = client.getsock();

     if (whois.in_fileno() > maxsock)
  	maxsock = whois.in_fileno();

    int nsock = 0, finished = 0;
    while (!finished) {
	Query *query;

	// write all query's that have responses.  we stop
	// when we come to a query and a NULL response, or
	// all queries have been exhausted.  this will leave us
	// with the next unsatisfied query that needs to be matched
	// with a response from the IRR.
	while (qlist.head && qlist.head->reply) {
	    query = qlist.getnext(); // removes from the queue
	    if (query->cache_hit) {
	       client.fprintf("A%d\n", query->reply_size);
	       client.fwrite (query->reply, query->reply_size);
	       client.fprintf("C1,CACHE\n");
	    } else
	       client.fwrite (query->reply, query->reply_size);
	    client.error.warn (progname);
	    delete query;	// free the query
	}
	
	// Wait for either client requests, or responses to client
	// requests from the IRR.
	incoming = allsock;
	if ((nsock = select (maxsock + 1, &incoming,
			     (fd_set *) 0, (fd_set *) 0,
			     (struct timeval *)0)) <= 0) {
	    if (nsock < 0) {
		if (errno == EINTR) {
		    errno = 0;
		    sleep (1);
		    continue;
		}
		error = errno;
		error.die ("relay, select: ");
	    }
	}
	
	// process client requests.
	if (FD_ISSET (client.getsock(), &incoming)) {
	    while (client.PendingData()) {
		char line[MAXLINE];
		bzero ((char *) &line, MAXLINE);

		client.fgets ((char *) &line, MAXLINE);
		client.error.die ("realyd: reading client");
		
		if (!strncmp (line, "q", 1))
		    return;

		if (!strncmp (line, "!!", 2)) // ignore, done by whois.Open()
		    continue;

		// yak, replace [\r\n] with \0
		int i,  slen = strlen(line);
		for (i = 0; line[i] && i < slen; i++) {
		    if (line[i] == '\n' || line[i] == '\r') {
			line[i] = '\0';
			break;
		    }
		    if (!isgraph (line[i])) {
			i = -1;	// XXX HACK
			break;
		    }
		}
		if (i < 1)
		    continue;

		fprintf (relaylog, "recieved query: %s\n", line);
		Query *query = new Query((char *) &line, strlen (line));

		if (strncmp (line, "!man,", 5) == 0) {
		    CacheElement *elt = cache.find (&line[5]);
		    if (elt) {
			query->add_reply (elt->get_blob(), elt->get_blobsize());
			if (verbose)
			    fprintf (relaylog, "Cache hit: %s\n", line);
			query->cache_hit = 1;
		    }
		}

		if (!query->reply) {
		    if (verbose)
			fprintf (relaylog, "sending irr: %s\n", line);
 		    whois.Query (line);
		    whois.error.die ("relayd: ");
		}
		qlist.append (query);
	    }
	    FD_CLR (client.getsock(), &incoming);
	}

	// check for responses from the irr.
	if (FD_ISSET (whois.in_fileno(), &incoming)) {
	    while (whois.PendingData()) {
		char *ptr;
		int c = whois.TotalResponse (ptr);
		whois.error.die ("relayd: ");
		if (verbose)
		    fprintf (relaylog, "Recieved response from the IRR.\n");
		qlist.add_reply(ptr, c);
	    }
	    FD_CLR (whois.in_fileno(), &incoming);
	}
    }
}

Allocator key_allocator(1024*10,1024);

void
read_config (const char *config_file)
{
    struct stat statbuf;
    Error error;
    
    error = stat (config_file, (struct stat *) &statbuf);
    error.die ("%s: unable to read %s", progname, config_file);

    char *buffer = new char[statbuf.st_size+1];

    int fd = open (config_file, O_RDONLY);
    int size = read (fd, buffer, statbuf.st_size);
    if (size != statbuf.st_size) {
        error = errno;
        error.die (progname);
    }
    close (fd);
    buffer[statbuf.st_size] = '\0';

    // now we have the entire file, parse the buffer seperating
    // objects
    int newobj = 1;
    char *curobj = buffer;
    char keybuf[80];
    char *key;
    char type[3];
    while (*curobj != '*')	// find the first obj
	curobj++;
    char *ptr = curobj;
    int recognize = 0;

    //    if (verbose) {
	fprintf (relaylog, "Reading config file..., ");
	fflush (stdout);
	//    }

    do {
	if (newobj && *ptr == '*') {
	    newobj = 0;
	    curobj = ptr;	// redundant for the first obj
	    sscanf (ptr, "*%2s: %s\n", type, keybuf);
	    if (strcmp (type, "an") == 0) {
		if (verbose)
		    fprintf (relaylog, "Cacheing object: %s: %s\n", type, keybuf);
		recognize = 1;
	    } else {
		if (verbose) 
		    fprintf (relaylog, "unrecognized object: %s: %s\n", type, keybuf);
		recognize = 0;
		continue;
	    }
	    key = (char *) key_allocator.allocate(strlen(keybuf) + 1);
	    strcpy (key, keybuf);
	} else if (!newobj && (*ptr == '\0' || (*ptr == '\n' && *(ptr-1) == '\n'))) {
	    newobj = 1;
	    if (!recognize)
		continue;
	    *ptr = '\0';
	    cache.add (key, curobj, ptr - curobj);
	}
    } while (*(++ptr));

    //    if (verbose) {
	fprintf (relaylog, "done\n");
	fflush(stdout);
	//    }
}

void
foo ()
{
    fprintf (relaylog, "FOO\n");
}
    

void
usage ()
{
    fprintf (stderr,
	     "usage: %s [-vd] [-irr_server host] [-irr_port port] [-irr_sources sources]\n",
	     progname);
    fprintf (stderr,
	     "            [-c config_file] [-p server_port#] [-w timeout] [-help|-rusage]\n");
    exit (1);
}

int
go_daemonic()
{
    int cpid = fork();
    if (!cpid) {
	(void) setsid();
	(void) chdir("/");
	int devnull = open("/dev/null", O_RDWR, 0);
	if (!verbose && devnull != -1) {
	    (void) dup2(devnull, STDIN_FILENO);
	    (void) dup2(devnull, STDOUT_FILENO);
	    (void) dup2(devnull, STDERR_FILENO);
	    if (devnull > 2)
		(void) close(devnull);
	}
    }
    return cpid;
}

void
set_options (int argc, char **argv, char **envp)
{
   progname = argv[0];
    
   ArgvInfo argTable[] = {
     // RAToolSet common arguments
     // key, type, src, dst, help
//     {"-T", ARGV_FUNC, (char *) &start_tracing,      (char *) NULL, 
//      "Start tracing the next argument"},
//     {"-D", ARGV_FUNC, (char *) &start_debugging,    (char *) NULL, 
//      "Start debugging the next argument"},
     {"-version", ARGV_FUNC, (char *) &version,      (char *) NULL,
      "Show version"},
     {"-h", ARGV_FUNC, (char *) &Whois::ArgvHost,    (char *) NULL,
      "Host name of the RAWhoisd server"},
     {"-p", ARGV_FUNC, (char *) &Whois::ArgvPort,    (char *) NULL,
      "Port number of the RAWhoisd server"},
     {"-s", ARGV_FUNC, (char *) &Whois::ArgvSources, (char *) NULL,
      "Order of databases"},
//     {"-rusage", ARGV_CONSTANT, (char *) 1,          (char *) &opt_rusage,
//      "On termination print resource usage"},
//     {"-prompt", ARGV_STRING,  (char *) NULL,        (char *) &opt_prompt,
//      "Prompt"},
     {"-ignore_errors", ARGV_FUNC, (char *)&Whois::IgnoreErrors, (char *)NULL,
      "Ignore IRR error and warning messages"},
     {"-report_errors", ARGV_FUNC, (char *)&Whois::ReportErrors, (char *)NULL,
      "Print IRR error and warning messages"},
     
     // relayd specific arguments
     {"-irr_server", ARGV_FUNC, (char *) &Whois::ArgvHost,    (char *) NULL,
      "Same as -h"},
     {"-irr_port", ARGV_FUNC, (char *) &Whois::ArgvPort,    (char *) NULL,
      "Same as -p"},
     {"-irr_sources", ARGV_FUNC, (char *) &Whois::ArgvSources, (char *) NULL,
      "Same as -s"},
     {"-c", ARGV_STRING, (char *)NULL, (char *)&opt_config_file,
      "Configurtion file"},
     {"-d", ARGV_CONSTANT, (char *)1, (char *)&debug,
      "Debug"},
     {"-l", ARGV_STRING, (char *)NULL, (char *)&logfile,
      "Log file"},
     {"-sp", ARGV_INT, (char *)NULL, (char *)&opt_server_port,
      "relayd server port"},
     {"-v", ARGV_CONSTANT, (char *)1,  (char *)&verbose,
      "Verbose output"},
     {"-w", ARGV_INT, (char *)NULL, (char *)&opt_waittime,
      "Wait time"},
     {(char *) NULL, ARGV_END, (char *) NULL, (char *) NULL, (char *) NULL}
   };
   

   for (char **p = envp; *p != NULL; p++) {
     if (strncmp(*p, "IRR_HOST=", 9) == 0)  {
       whois.SetDefaultHost (*p + 9);
       continue;
     }
     if (strncmp(*p, "IRR_PORT=", 9) == 0)  {
       whois.SetDefaultPort (atoi(*p + 9));
       continue;
     }
     if (strncmp(*p, "IRR_SOURCES=", 12) == 0)  {
       whois.SetDefaultSources (*p + 12);
       continue;
     }
   }

   if (ParseArgv(&argc, argv, argTable, 0) != ARGV_OK) {
     cerr << endl;
     exit(1);
   }

   if (opt_config_file == NULL)
     opt_config_file = "/etc/relayd.conf";
}


