/*
 *    Hardware descriptions for HP 9000 based hardware, including
 *    system types, SCSI controllers, DMA controllers, HPPB controllers
 *    and lots more.
 * 
 *    Based on the document "PA-RISC 1.1 I/O Firmware Architecture 
 *    Reference Specification", March 7, 1999, version 0.96.  This
 *    is available at http://parisc-linux.org/documentation/
 *
 *    Copyright 1999 by Alex deVries <adevries@thepuffingroup.com>
 *    and copyright 1999 The Puffin Group Inc.
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


#include <asm/hardware.h>
#include <linux/stddef.h>
#include <linux/kernel.h>
#include <linux/init.h>

/*
 *	HP PARISC Hardware Database
 *	Access to this database is only possible during bootup
 *	so don't reference this table after starting the init process
 */
 
static struct hp_hardware hp_hardware_list[] __initdata = {
	{HPHW_NPROC,0x01,0x4,0x0,"Indigo (840, 930)"},
	{HPHW_NPROC,0x8,0x4,0x01,"Firefox(825,925)"},
	{HPHW_NPROC,0xA,0x4,0x01,"Top Gun (835,834,935,635)"},
	{HPHW_NPROC,0xB,0x4,0x01,"Technical Shogun (845, 645)"},
	{HPHW_NPROC,0xF,0x4,0x01,"Commercial Shogun (949)"},
	{HPHW_NPROC,0xC,0x4,0x01,"Cheetah (850, 950)"},
	{HPHW_NPROC,0x80,0x4,0x01,"Cheetah (950S)"},
	{HPHW_NPROC,0x81,0x4,0x01,"Jaguar (855, 955)"},
	{HPHW_NPROC,0x82,0x4,0x01,"Cougar (860, 960)"},
	{HPHW_NPROC,0x83,0x4,0x13,"Panther (865, 870, 980)"},
	{HPHW_NPROC,0x100,0x4,0x01,"Burgundy (810)"},
	{HPHW_NPROC,0x101,0x4,0x01,"SilverFox Low (822, 922)"},
	{HPHW_NPROC,0x102,0x4,0x01,"SilverFox High (832, 932)"},
	{HPHW_NPROC,0x103,0x4,0x01,"Lego, SilverLite (815, 808, 920)"},
	{HPHW_NPROC,0x104,0x4,0x03,"SilverBullet Low (842, 948)"},
	{HPHW_NPROC,0x105,0x4,0x03,"SilverBullet High (852, 958)"},
	{HPHW_NPROC,0x106,0x4,0x81,"Oboe"},
	{HPHW_NPROC,0x180,0x4,0x12,"Dragon"},
	{HPHW_NPROC,0x181,0x4,0x13,"Chimera (890, 990, 992)"},
	{HPHW_NPROC,0x182,0x4,0x91,"TNT 100 (891,T500)"},
	{HPHW_NPROC,0x183,0x4,0x91,"TNT 120 (892,T520)"},
	{HPHW_NPROC,0x184,0x4,0x91,"Jade 180 U (893,T540)"},
	{HPHW_NPROC,0x1FF,0x4,0x91,"Hitachi X Processor"},
	{HPHW_NPROC,0x200,0x4,0x81,"Cobra (720)"},
	{HPHW_NPROC,0x201,0x4,0x81,"Coral (750)"},
	{HPHW_NPROC,0x202,0x4,0x81,"King Cobra (730)"},
	{HPHW_NPROC,0x203,0x4,0x81,"Hardball (735/99)"},
	{HPHW_NPROC,0x204,0x4,0x81,"Coral II (755/99)"},
	{HPHW_NPROC,0x205,0x4,0x81,"Coral II (755/125)"},
	{HPHW_NPROC,0x205,0x4,0x91,"Snake Eagle "},
	{HPHW_NPROC,0x206,0x4,0x81,"Snake Cheetah (735/130)"},
	{HPHW_NPROC,0x280,0x4,0x81,"Nova Low (817, 827, 957, 957LX)"},
	{HPHW_NPROC,0x281,0x4,0x81,"Nova High (837, 847, 857, 967, 967LX)"},
	{HPHW_NPROC,0x282,0x4,0x81,"Nova8 (807, 917, 917LX, 927,927LX, 937, 937LX, 947,947LX)"},
	{HPHW_NPROC,0x283,0x4,0x81,"Nova64 (867, 877, 977)"},
	{HPHW_NPROC,0x284,0x4,0x81,"TNova (887, 897, 987)"},
	{HPHW_NPROC,0x285,0x4,0x81,"TNova64"},
	{HPHW_NPROC,0x286,0x4,0x91,"Hydra64 (Nova)"},
	{HPHW_NPROC,0x287,0x4,0x91,"Hydra96 (Nova)"},
	{HPHW_NPROC,0x288,0x4,0x81,"TNova96"},
	{HPHW_NPROC,0x300,0x4,0x81,"Bushmaster (710)"},
	{HPHW_NPROC,0x302,0x4,0x81,"Flounder (705)"},
	{HPHW_NPROC,0x310,0x4,0x81,"Scorpio (715/50)"},
	{HPHW_NPROC,0x311,0x4,0x81,"Scorpio Jr.(715/33)"},
	{HPHW_NPROC,0x312,0x4,0x81,"Strider-50 (715S/50)"},
	{HPHW_NPROC,0x313,0x4,0x81,"Strider-33 (715S/33)"},
	{HPHW_NPROC,0x314,0x4,0x81,"Trailways-50 (715T/50)"},
	{HPHW_NPROC,0x315,0x4,0x81,"Trailways-33 (715T/33)"},
	{HPHW_NPROC,0x316,0x4,0x81,"Scorpio Sr.(715/75)"},
	{HPHW_NPROC,0x317,0x4,0x81,"Scorpio 100 (715/100)"},
	{HPHW_NPROC,0x318,0x4,0x81,"Spectra (725/50)"},
	{HPHW_NPROC,0x319,0x4,0x81,"Spectra (725/75)"},
	{HPHW_NPROC,0x320,0x4,0x81,"Spectra (725/100)"},
	{HPHW_NPROC,0x401,0x4,0x81,"Pace (745i, 747i)"},
	{HPHW_NPROC,0x402,0x4,0x81,"Sidewinder (742i)"},
	{HPHW_NPROC,0x403,0x4,0x81,"Fast Pace"},
	{HPHW_NPROC,0x480,0x4,0x81,"Orville (E23)"},
	{HPHW_NPROC,0x481,0x4,0x81,"Wilbur (E25)"},
	{HPHW_NPROC,0x482,0x4,0x81,"WB-80 (E35)"},
	{HPHW_NPROC,0x483,0x4,0x81,"WB-96 (E45)"},
	{HPHW_NPROC,0x484,0x4,0x81,"UL Proc L-100 (811/D210,D310)"},
	{HPHW_NPROC,0x485,0x4,0x81,"UL Proc L-75 (801/D200)"},
	{HPHW_NPROC,0x501,0x4,0x81,"Merlin L2 132 (9000/778/B132L)"},
	{HPHW_NPROC,0x502,0x4,0x81,"Merlin L2 160 (9000/778/B160L)"},
	{HPHW_NPROC,0x503,0x4,0x81,"Merlin L2+ 132 (9000/778/B132L)"},
	{HPHW_NPROC,0x504,0x4,0x81,"Merlin L2+ 180 (9000/778/B180L)"},
	{HPHW_NPROC,0x505,0x4,0x81,"Raven L2 132 (9000/778/C132L)"},
	{HPHW_NPROC,0x506,0x4,0x81,"Raven L2 160 (9000/779/C160L)"},
	{HPHW_NPROC,0x507,0x4,0x81,"Raven L2 180 (9000/779/C180L)"},
	{HPHW_NPROC,0x508,0x4,0x81,"Raven L2 160 (9000/779/C160L)"},
	{HPHW_NPROC,0x509,0x4,0x81,"712/132 L2 Upgrade"},
	{HPHW_NPROC,0x50A,0x4,0x81,"712/160 L2 Upgrade"},
	{HPHW_NPROC,0x50B,0x4,0x81,"715/132 L2 Upgrade"},
	{HPHW_NPROC,0x50C,0x4,0x81,"715/160 L2 Upgrade"},
	{HPHW_NPROC,0x50D,0x4,0x81,"Rocky2 L2 120"},
	{HPHW_NPROC,0x50E,0x4,0x81,"Rocky2 L2 150"},
	{HPHW_NPROC,0x50F,0x4,0x81,"Anole L2 132 (744)"},
	{HPHW_NPROC,0x510,0x4,0x81,"Anole L2 165 (744)"},
	{HPHW_NPROC,0x511,0x4,0x81,"Kiji L2 132"},
	{HPHW_NPROC,0x512,0x4,0x81,"UL L2 132 (803/D220,D320)"},
	{HPHW_NPROC,0x513,0x4,0x81,"UL L2 160 (813/D220,D320)"},
	{HPHW_NPROC,0x514,0x4,0x81,"Merlin Jr L2 132"},
	{HPHW_NPROC,0x515,0x4,0x81,"Staccato L2 132"},
	{HPHW_NPROC,0x516,0x4,0x81,"Staccato L2 180 (A Class 180)"},
	{HPHW_NPROC,0x580,0x4,0x81,"KittyHawk DC2-100 (K100)"},
	{HPHW_NPROC,0x581,0x4,0x91,"KittyHawk DC3-120 (K210)"},
	{HPHW_NPROC,0x582,0x4,0x91,"KittyHawk DC3 100 (K400)"},
	{HPHW_NPROC,0x583,0x4,0x91,"KittyHawk DC3 120 (K410)"},
	{HPHW_NPROC,0x584,0x4,0x91,"LighteningHawk T120"},
	{HPHW_NPROC,0x585,0x4,0x91,"SkyHawk 100"},
	{HPHW_NPROC,0x586,0x4,0x91,"SkyHawk 120"},
	{HPHW_NPROC,0x587,0x4,0x81,"UL Proc 1-way T'120"},
	{HPHW_NPROC,0x588,0x4,0x91,"UL Proc 2-way T'120"},
	{HPHW_NPROC,0x589,0x4,0x81,"UL Proc 1-way T'100 (821/D250,D350)"},
	{HPHW_NPROC,0x58A,0x4,0x91,"UL Proc 2-way T'100 (831/D250,D350)"},
	{HPHW_NPROC,0x58B,0x4,0x91,"KittyHawk DC2 100 (K200)"},
	{HPHW_NPROC,0x58C,0x4,0x91,"ThunderHawk DC3- 120 1M (K220)"},
	{HPHW_NPROC,0x58D,0x4,0x91,"ThunderHawk DC3 120 1M (K420)"},
	{HPHW_NPROC,0x58E,0x4,0x81,"Raven 120 T'"},
	{HPHW_NPROC,0x58F,0x4,0x91,"Mohawk 160 U 1M DC3 (K450)"},
	{HPHW_NPROC,0x590,0x4,0x91,"Mohawk 180 U 1M DC3 (K460)"},
	{HPHW_NPROC,0x591,0x4,0x91,"Mohawk 200 U 1M DC3"},
	{HPHW_NPROC,0x592,0x4,0x81,"Raven 100 T'"},
	{HPHW_NPROC,0x593,0x4,0x91,"FireHawk 160 U"},
	{HPHW_NPROC,0x594,0x4,0x91,"FireHawk 180 U"},
	{HPHW_NPROC,0x595,0x4,0x91,"FireHawk 220 U"},
	{HPHW_NPROC,0x596,0x4,0x91,"FireHawk 240 U"},
	{HPHW_NPROC,0x597,0x4,0x91,"SPP2000 processor"},
	{HPHW_NPROC,0x598,0x4,0x81,"Raven U 230 (9000/780/C230)"},
	{HPHW_NPROC,0x599,0x4,0x81,"Raven U 240 (9000/780/C240)"},
	{HPHW_NPROC,0x59A,0x4,0x91,"Unlisted but reserved"},
	{HPHW_NPROC,0x59A,0x4,0x81,"Unlisted but reserved"},
	{HPHW_NPROC,0x59B,0x4,0x81,"Raven U 160 (9000/780/C160)"},
	{HPHW_NPROC,0x59C,0x4,0x81,"Raven U 180 (9000/780/C180)"},
	{HPHW_NPROC,0x59D,0x4,0x81,"Raven U 200 (9000/780/C200)"},
	{HPHW_NPROC,0x59E,0x4,0x91,"ThunderHawk T' 120"},
	{HPHW_NPROC,0x59F,0x4,0x91,"Raven U 180+ (9000/780)"},
	{HPHW_NPROC,0x5A0,0x4,0x81,"UL 1w T120 1MB/1MB (841/D260,D360)"},
	{HPHW_NPROC,0x5A1,0x4,0x91,"UL 2w T120 1MB/1MB (851/D260,D360)"},
	{HPHW_NPROC,0x5A2,0x4,0x81,"UL 1w U160 512K/512K (861/D270,D370)"},
	{HPHW_NPROC,0x5A3,0x4,0x91,"UL 2w U160 512K/512K (871/D270,D370)"},
	{HPHW_NPROC,0x5A4,0x4,0x91,"Mohawk 160 U 1M DC3- (K250)"},
	{HPHW_NPROC,0x5A5,0x4,0x91,"Mohawk 180 U 1M DC3- (K260)"},
	{HPHW_NPROC,0x5A6,0x4,0x91,"Mohawk 200 U 1M DC3-"},
	{HPHW_NPROC,0x5A7,0x4,0x81,"UL proc 1-way U160 1M/1M"},
	{HPHW_NPROC,0x5A8,0x4,0x91,"UL proc 2-way U160 1M/1M"},
	{HPHW_NPROC,0x5A9,0x4,0x81,"UL proc 1-way U180 1M/1M"},
	{HPHW_NPROC,0x5AA,0x4,0x91,"UL proc 2-way U180 1M/1M"},
	{HPHW_NPROC,0x5AB,0x4,0x91,"Obsolete"},
	{HPHW_NPROC,0x5AB,0x4,0x81,"Obsolete"},
	{HPHW_NPROC,0x5AC,0x4,0x91,"Obsolete"},
	{HPHW_NPROC,0x5AC,0x4,0x81,"Obsolete"},
	{HPHW_NPROC,0x5AD,0x4,0x91,"BraveHawk 180MHz DC3-"},
	{HPHW_NPROC,0x5AE,0x4,0x91,"BraveHawk 200MHz DC3- (898/K370)"},
	{HPHW_NPROC,0x5AF,0x4,0x91,"BraveHawk 220MHz DC3-"},
	{HPHW_NPROC,0x5B0,0x4,0x91,"BraveHawk 180MHz DC3"},
	{HPHW_NPROC,0x5B1,0x4,0x91,"BraveHawk 200MHz DC3 (899/K570)"},
	{HPHW_NPROC,0x5B2,0x4,0x91,"BraveHawk 220MHz DC3"},
	{HPHW_NPROC,0x5B3,0x4,0x91,"FireHawk 200"},
	{HPHW_NPROC,0x5B4,0x4,0x91,"SPP2500"},
	{HPHW_NPROC,0x5B5,0x4,0x91,"SummitHawk U+"},
	{HPHW_NPROC,0x5B6,0x4,0x91,"DragonHawk U+ 240 DC3"},
	{HPHW_NPROC,0x5B7,0x4,0x91,"DragonHawk U+ 240 DC3-"},
	{HPHW_NPROC,0x5B8,0x4,0x91,"SPP2250 240 MHz"},
	{HPHW_NPROC,0x5B9,0x4,0x81,"UL 1w U+/240 (350/550)"},
	{HPHW_NPROC,0x5BA,0x4,0x91,"UL 2w U+/240 (350/550)"},
	{HPHW_NPROC,0x5BB,0x4,0x81,"AllegroHigh W"},
	{HPHW_NPROC,0x5BC,0x4,0x91,"AllegroLow W"},
	{HPHW_NPROC,0x5BD,0x4,0x91,"Forte W 2-way"},
	{HPHW_NPROC,0x5BE,0x4,0x91,"Prelude W"},
	{HPHW_NPROC,0x5BF,0x4,0x91,"Forte W 4-way"},
	{HPHW_NPROC,0x5C0,0x4,0x91,"M2250"},
	{HPHW_NPROC,0x5C1,0x4,0x91,"M2500"},
	{HPHW_NPROC,0x5C2,0x4,0x91,"Sonata 440"},
	{HPHW_NPROC,0x5C3,0x4,0x91,"Sonata 360"},
	{HPHW_NPROC,0x5C4,0x4,0x91,"Rhapsody 440"},
	{HPHW_NPROC,0x5C5,0x4,0x91,"Rhapsody 360"},
	{HPHW_NPROC,0x5C6,0x4,0x91,"Raven W 360 (9000/780)"},
	{HPHW_NPROC,0x5C7,0x4,0x91,"Halfdome W 440"},
	{HPHW_NPROC,0x5C8,0x4,0x81,"Lego 360 processor"},
	{HPHW_NPROC,0x5C9,0x4,0x91,"Rhapsody DC- 440"},
	{HPHW_NPROC,0x5CA,0x4,0x91,"Rhapsody DC- 360"},
	{HPHW_NPROC,0x5CB,0x4,0x91,"Crescendo 440"},
	{HPHW_NPROC,0x5CC,0x4,0x91,"Prelude W 440"},
	{HPHW_NPROC,0x5CD,0x4,0x91,"SPP2600"},
	{HPHW_NPROC,0x5CE,0x4,0x91,"M2600"},
	{HPHW_NPROC,0x5CF,0x4,0x81,"Allegro W+"},
	{HPHW_NPROC,0x5D0,0x4,0x81,"Kazoo W+"},
	{HPHW_NPROC,0x5D1,0x4,0x91,"Forte W+ 2w"},
	{HPHW_NPROC,0x5D2,0x4,0x91,"Forte W+ 4w"},
	{HPHW_NPROC,0x5D3,0x4,0x91,"Prelude W+ 540"},
	{HPHW_NPROC,0x5D4,0x4,0x91,"Duet W+"},
	{HPHW_NPROC,0x5D5,0x4,0x91,"Crescendo 550"},
	{HPHW_NPROC,0x5D6,0x4,0x81,"Crescendo DC- 440"},
	{HPHW_NPROC,0x5D7,0x4,0x91,"Keystone W+"},
	{HPHW_NPROC,0x5D8,0x4,0x91,"Rhapsody wave 2 W+ DC-"},
	{HPHW_NPROC,0x5D9,0x4,0x91,"Rhapsody wave 2 W+"},
	{HPHW_NPROC,0x5DA,0x4,0x91,"Marcato W+ DC-"},
	{HPHW_NPROC,0x5DB,0x4,0x91,"Marcato W+"},
	{HPHW_NPROC,0x5DC,0x4,0x91,"Allegro W2"},
	{HPHW_NPROC,0x5DD,0x4,0x81,"Duet W2"},
	{HPHW_NPROC,0x5DE,0x4,0x81,"Piccolo W+"},
	{HPHW_NPROC,0x5DF,0x4,0x81,"Cantata W2"},
	{HPHW_NPROC,0x5E0,0x4,0x91,"Cantata DC- W2"},
	{HPHW_NPROC,0x5E1,0x4,0x91,"Crescendo DC- W2"},
	{HPHW_NPROC,0x5E2,0x4,0x91,"Crescendo 650 W2"},
	{HPHW_NPROC,0x5E3,0x4,0x91,"Crescendo 750 W2"},
	{HPHW_NPROC,0x5E4,0x4,0x91,"Keystone/Matterhorn W2 750"},
	{HPHW_NPROC,0x5E5,0x4,0x91,"PowerBar W+"},
	{HPHW_NPROC,0x5E6,0x4,0x91,"Keystone/Matterhorn W2 650"},
	{HPHW_NPROC,0x5E7,0x4,0x91,"Caribe W2 800"},
	{HPHW_NPROC,0x5E8,0x4,0x91,"Pikes Peak W2"},
	{HPHW_NPROC,0x5FF,0x4,0x91,"Hitachi W"},
	{HPHW_NPROC,0x600,0x4,0x81,"Gecko (712/60)"},
	{HPHW_NPROC,0x601,0x4,0x81,"Gecko 80 (712/80)"},
	{HPHW_NPROC,0x602,0x4,0x81,"Gecko 100 (712/100)"},
	{HPHW_NPROC,0x603,0x4,0x81,"Anole 64 (743/64)"},
	{HPHW_NPROC,0x604,0x4,0x81,"Anole 100 (743/100)"},
	{HPHW_NPROC,0x605,0x4,0x81,"Gecko 120 (712/120)"},
	{HPHW_NPROC,0x606,0x4,0x81,"Gila 80"},
	{HPHW_NPROC,0x607,0x4,0x81,"Gila 100"},
	{HPHW_NPROC,0x608,0x4,0x81,"Gila 120"},
	{HPHW_NPROC,0x609,0x4,0x81,"Scorpio-L 80"},
	{HPHW_NPROC,0x60A,0x4,0x81,"Mirage Jr (715/64)"},
	{HPHW_NPROC,0x60B,0x4,0x81,"Mirage 100"},
	{HPHW_NPROC,0x60C,0x4,0x81,"Mirage 100+"},
	{HPHW_NPROC,0x60D,0x4,0x81,"Electra 100"},
	{HPHW_NPROC,0x60E,0x4,0x81,"Electra 120"},
	{HPHW_NPROC,0x610,0x4,0x81,"Scorpio-L 100"},
	{HPHW_NPROC,0x611,0x4,0x81,"Scorpio-L 120"},
	{HPHW_NPROC,0x612,0x4,0x81,"Spectra-L 80"},
	{HPHW_NPROC,0x613,0x4,0x81,"Spectra-L 100"},
	{HPHW_NPROC,0x614,0x4,0x81,"Spectra-L 120"},
	{HPHW_NPROC,0x615,0x4,0x81,"Piranha 100"},
	{HPHW_NPROC,0x616,0x4,0x81,"Piranha 120"},
	{HPHW_NPROC,0x617,0x4,0x81,"Jason 50"},
	{HPHW_NPROC,0x618,0x4,0x81,"Jason 100"},
	{HPHW_NPROC,0x619,0x4,0x81,"Mirage 80"},
	{HPHW_NPROC,0x61A,0x4,0x81,"SAIC L-80"},
	{HPHW_NPROC,0x61B,0x4,0x81,"Rocky1 L-60"},
	{HPHW_NPROC,0x61C,0x4,0x81,"Anole T (743/T)"},
	{HPHW_NPROC,0x67E,0x4,0x81,"Hitachi Tiny 80"},
	{HPHW_NPROC,0x67F,0x4,0x81,"Hitachi Tiny 64"},
	{HPHW_NPROC,0x700,0x4,0x91,"NEC Aska Processor"},
	{HPHW_NPROC,0x880,0x4,0x91,"Orca Mako"},
	{HPHW_NPROC,0x881,0x4,0x91,"Everest Mako"},
	{HPHW_NPROC,0x882,0x4,0x91,"Rainier/Medel Mako Slow"},
	{HPHW_NPROC,0x883,0x4,0x91,"Rainier/Medel Mako Fast"},
	{HPHW_NPROC,0x884,0x4,0x91,"Mt. Hamilton"},
	{HPHW_NPROC,0x885,0x4,0x91,"Mt. Hamilton DC-"},
	{HPHW_NPROC,0x886,0x4,0x91,"Storm Peak Slow DC-"},
	{HPHW_NPROC,0x887,0x4,0x91,"Storm Peak Slow"},
	{HPHW_NPROC,0x888,0x4,0x91,"Storm Peak Fast DC-"},
	{HPHW_NPROC,0x889,0x4,0x91,"Storm Peak Fast"},
	{HPHW_NPROC,0x88A,0x4,0x91,"Crestone Peak"},
	{HPHW_A_DIRECT, 0x004, 0x0000D, 0x00, "Arrakis MUX"}, 
	{HPHW_A_DIRECT, 0x005, 0x0000D, 0x00, "Dyun Kiuh MUX"}, 
	{HPHW_A_DIRECT, 0x006, 0x0000D, 0x00, "Baat Kiuh AP/MUX (40299B)"}, 
	{HPHW_A_DIRECT, 0x007, 0x0000D, 0x00, "Dino AP"}, 
	{HPHW_A_DIRECT, 0x009, 0x0000D, 0x00, "Solaris Direct Connect MUX (J2092A)"}, 
	{HPHW_A_DIRECT, 0x00A, 0x0000D, 0x00, "Solaris RS-422/423 MUX (J2093A)"}, 
	{HPHW_A_DIRECT, 0x00B, 0x0000D, 0x00, "Solaris RS-422/423 Quadriloops MUX"}, 
	{HPHW_A_DIRECT, 0x00C, 0x0000D, 0x00, "Solaris Modem MUX (J2094A)"}, 
	{HPHW_A_DIRECT, 0x00D, 0x0000D, 0x00, "Twins Direct Connect MUX"}, 
	{HPHW_A_DIRECT, 0x00E, 0x0000D, 0x00, "Twins Modem MUX"}, 
	{HPHW_A_DIRECT, 0x00F, 0x0000D, 0x00, "Nautilus RS-485"}, 
	{HPHW_A_DIRECT, 0x010, 0x0000D, 0x00, "UltraLight CAP/MUX"}, 
	{HPHW_A_DIRECT, 0x015, 0x0000D, 0x00, "Eole CAP/MUX"}, 
	{HPHW_A_DIRECT, 0x024, 0x0000D, 0x00, "Sahp Kiuh AP/MUX"}, 
	{HPHW_A_DIRECT, 0x034, 0x0000D, 0x00, "Sahp Kiuh Low AP/MUX"}, 
	{HPHW_A_DIRECT, 0x044, 0x0000D, 0x00, "Sahp Baat Kiuh AP/MUX"}, 
	{HPHW_A_DIRECT, 0x004, 0x0000E, 0x80, "Burgundy RS-232"}, 
	{HPHW_A_DIRECT, 0x005, 0x0000E, 0x80, "Silverfox RS-232"}, 
	{HPHW_A_DIRECT, 0x006, 0x0000E, 0x80, "Lego RS-232"}, 
	{HPHW_A_DIRECT, 0x004, 0x0000F, 0x00, "Peacock Graphics"}, 
	{HPHW_A_DIRECT, 0x004, 0x00014, 0x80, "Burgundy HIL"}, 
	{HPHW_A_DIRECT, 0x005, 0x00014, 0x80, "Peacock HIL"}, 
	{HPHW_A_DIRECT, 0x004, 0x00015, 0x80, "Leonardo"}, 
	{HPHW_A_DIRECT, 0x004, 0x00016, 0x80, "HP-PB HRM"}, 
	{HPHW_A_DIRECT, 0x004, 0x00017, 0x80, "HP-PB HRC"}, 
	{HPHW_A_DIRECT, 0x004, 0x0003A, 0x80, "Skunk Centronics (28655A)"}, 
	{HPHW_A_DIRECT, 0x024, 0x0003A, 0x80, "Sahp Kiuh Centronics"}, 
	{HPHW_A_DIRECT, 0x044, 0x0003A, 0x80, "Sahp Baat Kiuh Centronics"}, 
	{HPHW_A_DIRECT, 0x004, 0x0004E, 0x80, "AT&T DataKit (AMSO)"}, 
	{HPHW_A_DIRECT, 0x004, 0x0009B, 0x80, "Test&Meas GSC HPIB"}, 
	{HPHW_A_DIRECT, 0x004, 0x000A8, 0x00, "Rocky2-120 Front Keyboard"}, 
	{HPHW_A_DIRECT, 0x005, 0x000A8, 0x00, "Rocky2-150 Front Keyboard"}, 
	{HPHW_A_DIRECT, 0x004, 0x00101, 0x80, "Hitachi Console Module"}, 
	{HPHW_A_DIRECT, 0x004, 0x00102, 0x80, "Hitachi Boot Module"}, 
	{HPHW_A_DIRECT, 0x004, 0x00203, 0x80, "MELCO HBMLA MLAIT"}, 
	{HPHW_A_DIRECT, 0x004, 0x00208, 0x80, "MELCO HBDPC"}, 
	{HPHW_A_DIRECT, 0x004, 0x00300, 0x00, "DCI TWINAX TERM IO MUX"}, 
	{HPHW_A_DMA, 0x004, 0x00039, 0x80, "Skunk SCSI (28655A)"}, 
	{HPHW_A_DMA, 0x005, 0x00039, 0x80, "KittyHawk CSY Core SCSI"}, 
	{HPHW_A_DMA, 0x014, 0x00039, 0x80, "Diablo SCSI"}, 
	{HPHW_A_DMA, 0x024, 0x00039, 0x80, "Sahp Kiuh SCSI"}, 
	{HPHW_A_DMA, 0x034, 0x00039, 0x80, "Sahp Kiuh Low SCSI"}, 
	{HPHW_A_DMA, 0x044, 0x00039, 0x80, "Sahp Baat Kiuh SCSI"}, 
	{HPHW_A_DMA, 0x004, 0x0003B, 0x80, "Wizard SCSI"}, 
	{HPHW_A_DMA, 0x005, 0x0003B, 0x80, "KittyHawk CSY Core FW-SCSI"}, 
	{HPHW_A_DMA, 0x006, 0x0003B, 0x80, "Symbios EPIC FW-SCSI"}, 
	{HPHW_A_DMA, 0x004, 0x00040, 0x80, "HP-PB Shazam HPIB (28650A)"}, 
	{HPHW_A_DMA, 0x005, 0x00040, 0x80, "Burgundy HPIB"}, 
	{HPHW_A_DMA, 0x004, 0x00041, 0x80, "HP-PB HP-FL"}, 
	{HPHW_A_DMA, 0x004, 0x00042, 0x80, "HP-PB LoQuix HPIB (28650B)"}, 
	{HPHW_A_DMA, 0x004, 0x00043, 0x80, "HP-PB Crypt LoQuix"}, 
	{HPHW_A_DMA, 0x004, 0x00044, 0x80, "HP-PB Shazam GPIO (28651A)"}, 
	{HPHW_A_DMA, 0x004, 0x00045, 0x80, "HP-PB LoQuix GPIO"}, 
	{HPHW_A_DMA, 0x004, 0x00046, 0x80, "2-Port X.25 NIO_ACC (AMSO)"}, 
	{HPHW_A_DMA, 0x004, 0x00047, 0x80, "4-Port X.25 NIO_ACC (AMSO)"}, 
	{HPHW_A_DMA, 0x004, 0x0004B, 0x80, "LGB Control"}, 
	{HPHW_A_DMA, 0x004, 0x0004C, 0x80, "Martian RTI (AMSO)"}, 
	{HPHW_A_DMA, 0x004, 0x0004D, 0x80, "ACC Mux (AMSO)"}, 
	{HPHW_A_DMA, 0x004, 0x00050, 0x80, "Lanbrusca 802.3 (36967A)"}, 
	{HPHW_A_DMA, 0x004, 0x00056, 0x80, "HP-PB LoQuix FDDI"}, 
	{HPHW_A_DMA, 0x004, 0x00057, 0x80, "HP-PB LoQuix FDDI (28670A)"}, 
	{HPHW_A_DMA, 0x004, 0x0005E, 0x00, "Gecko Add-on Token Ring"}, 
	{HPHW_A_DMA, 0x012, 0x00089, 0x80, "Barracuda Add-on FW-SCSI"}, 
	{HPHW_A_DMA, 0x013, 0x00089, 0x80, "Bluefish Add-on FW-SCSI"}, 
	{HPHW_A_DMA, 0x014, 0x00089, 0x80, "Shrike Add-on FW-SCSI"}, 
	{HPHW_A_DMA, 0x015, 0x00089, 0x80, "KittyHawk GSY Core FW-SCSI"}, 
	{HPHW_A_DMA, 0x017, 0x00089, 0x80, "Shrike Jade Add-on FW-SCSI (A3644A)"}, 
	{HPHW_A_DMA, 0x01F, 0x00089, 0x80, "SkyHawk 100/120 FW-SCSI"}, 
	{HPHW_A_DMA, 0x027, 0x00089, 0x80, "Piranha 100 FW-SCSI"}, 
	{HPHW_A_DMA, 0x032, 0x00089, 0x80, "Raven T' Core FW-SCSI"}, 
	{HPHW_A_DMA, 0x03B, 0x00089, 0x80, "Raven U/L2 Core FW-SCSI"}, 
	{HPHW_A_DMA, 0x03C, 0x00089, 0x80, "Merlin 132 Core FW-SCSI"},
	{HPHW_A_DMA, 0x03D, 0x00089, 0x80, "Merlin 160 Core FW-SCSI"},
	{HPHW_A_DMA, 0x044, 0x00089, 0x80, "Mohawk Core FW-SCSI"}, 
	{HPHW_A_DMA, 0x051, 0x00089, 0x80, "Firehawk FW-SCSI"}, 
	{HPHW_A_DMA, 0x058, 0x00089, 0x80, "FireHawk 200 FW-SCSI"}, 
	{HPHW_A_DMA, 0x05C, 0x00089, 0x80, "SummitHawk 230 Ultra-SCSI"}, 
	{HPHW_A_DMA, 0x014, 0x00091, 0x80, "Baby Hugo Add-on Net FC (A3406A)"}, 
	{HPHW_A_DMA, 0x020, 0x00091, 0x80, "Baby Jade Add-on Net FC (A3638A)"}, 
	{HPHW_A_DMA, 0x004, 0x00092, 0x80, "GSC+ YLIASTER ATM"}, 
	{HPHW_A_DMA, 0x004, 0x00095, 0x80, "Hamlyn GSC+ Network Card"}, 
	{HPHW_A_DMA, 0x004, 0x00098, 0x80, "Lo-fat Emulator"}, 
	{HPHW_A_DMA, 0x004, 0x0009A, 0x80, "GSC+ Venus ATM"}, 
	{HPHW_A_DMA, 0x005, 0x0009A, 0x80, "GSC+ Samorobrive ATM"}, 
	{HPHW_A_DMA, 0x004, 0x0009D, 0x80, "HP HSC-PCI Cards"}, 
	{HPHW_A_DMA, 0x004, 0x0009E, 0x80, "Alaxis GSC+ 155Mb ATM"}, 
	{HPHW_A_DMA, 0x005, 0x0009E, 0x80, "Alaxis GSC+ 622Mb ATM"}, 
	{HPHW_A_DMA, 0x05C, 0x0009F, 0x80, "SummitHawk 230 USB"}, 
	{HPHW_A_DMA, 0x05C, 0x000A0, 0x80, "SummitHawk 230 100BaseT"}, 
	{HPHW_A_DMA, 0x015, 0x000A7, 0x80, "Baby Hugo Add-on mass FC (A3404A)"}, 
	{HPHW_A_DMA, 0x018, 0x000A7, 0x80, "Mombasa GS Add-on mass FC (A3591)"}, 
	{HPHW_A_DMA, 0x021, 0x000A7, 0x80, "Baby Jade Add-on mass FC (A3636A)"}, 
	{HPHW_A_DMA, 0x004, 0x00201, 0x80, "MELCO HCMAP"}, 
	{HPHW_A_DMA, 0x004, 0x00202, 0x80, "MELCO HBMLA MLAMA"}, 
	{HPHW_A_DMA, 0x004, 0x00205, 0x80, "MELCO HBRFU"}, 
	{HPHW_A_DMA, 0x004, 0x00380, 0x80, "Interphase NIO-FC"}, 
	{HPHW_A_DMA, 0x004, 0x00381, 0x80, "Interphase NIO-ATM"}, 
	{HPHW_A_DMA, 0x004, 0x00382, 0x80, "Interphase NIO-100BaseTX"}, 
	{HPHW_BA, 0x004, 0x00070, 0x0, "Cobra Core BA"}, 
	{HPHW_BA, 0x005, 0x00070, 0x0, "Coral Core BA"}, 
	{HPHW_BA, 0x006, 0x00070, 0x0, "Bushmaster Core BA"}, 
	{HPHW_BA, 0x007, 0x00070, 0x0, "Scorpio Core BA"}, 
	{HPHW_BA, 0x008, 0x00070, 0x0, "Flounder Core BA"}, 
	{HPHW_BA, 0x009, 0x00070, 0x0, "Outfield Core BA"}, 
	{HPHW_BA, 0x00A, 0x00070, 0x0, "CoralII Core BA"}, 
	{HPHW_BA, 0x00B, 0x00070, 0x0, "Scorpio Jr. Core BA"}, 
	{HPHW_BA, 0x00C, 0x00070, 0x0, "Strider-50 Core BA"}, 
	{HPHW_BA, 0x00D, 0x00070, 0x0, "Strider-33 Core BA"}, 
	{HPHW_BA, 0x00E, 0x00070, 0x0, "Trailways-50 Core BA"}, 
	{HPHW_BA, 0x00F, 0x00070, 0x0, "Trailways-33 Core BA"}, 
	{HPHW_BA, 0x010, 0x00070, 0x0, "Pace Core BA"}, 
	{HPHW_BA, 0x011, 0x00070, 0x0, "Sidewinder Core BA"}, 
	{HPHW_BA, 0x019, 0x00070, 0x0, "Scorpio Sr. Core BA"}, 
	{HPHW_BA, 0x020, 0x00070, 0x0, "Scorpio 100 Core BA"}, 
	{HPHW_BA, 0x021, 0x00070, 0x0, "Spectra 50 Core BA"}, 
	{HPHW_BA, 0x022, 0x00070, 0x0, "Spectra 75 Core BA"}, 
	{HPHW_BA, 0x023, 0x00070, 0x0, "Spectra 100 Core BA"}, 
	{HPHW_BA, 0x024, 0x00070, 0x0, "Fast Pace Core BA"}, 
	{HPHW_BA, 0x026, 0x00070, 0x0, "CoralII Jaguar Core BA"}, 
	{HPHW_BA, 0x004, 0x00076, 0x0, "Cobra EISA BA"}, 
	{HPHW_BA, 0x005, 0x00076, 0x0, "Coral EISA BA"}, 
	{HPHW_BA, 0x007, 0x00076, 0x0, "Scorpio EISA BA"}, 
	{HPHW_BA, 0x00A, 0x00076, 0x0, "CoralII EISA BA"}, 
	{HPHW_BA, 0x00B, 0x00076, 0x0, "Scorpio Jr. EISA BA"}, 
	{HPHW_BA, 0x00C, 0x00076, 0x0, "Strider-50 Core EISA"}, 
	{HPHW_BA, 0x00D, 0x00076, 0x0, "Strider-33 Core EISA"}, 
	{HPHW_BA, 0x00E, 0x00076, 0x0, "Trailways-50 Core EISA"}, 
	{HPHW_BA, 0x00F, 0x00076, 0x0, "Trailways-33 Core EISA"}, 
	{HPHW_BA, 0x010, 0x00076, 0x0, "Pace Core EISA"}, 
	{HPHW_BA, 0x019, 0x00076, 0x0, "Scorpio Sr. EISA BA"}, 
	{HPHW_BA, 0x020, 0x00076, 0x0, "Scorpio 100 EISA BA"}, 
	{HPHW_BA, 0x021, 0x00076, 0x0, "Spectra 50 EISA BA"}, 
	{HPHW_BA, 0x022, 0x00076, 0x0, "Spectra 75 EISA BA"}, 
	{HPHW_BA, 0x023, 0x00076, 0x0, "Spectra 100 EISA BA"}, 
	{HPHW_BA, 0x026, 0x00076, 0x0, "CoralII Jaguar EISA BA"}, 
	{HPHW_BA, 0x010, 0x00078, 0x0, "Pace VME BA"}, 
	{HPHW_BA, 0x011, 0x00078, 0x0, "Sidewinder VME BA"}, 
	{HPHW_BA, 0x01A, 0x00078, 0x0, "Anole 64 VME BA"}, 
	{HPHW_BA, 0x01B, 0x00078, 0x0, "Anole 100 VME BA"}, 
	{HPHW_BA, 0x024, 0x00078, 0x0, "Fast Pace VME BA"}, 
	{HPHW_BA, 0x034, 0x00078, 0x0, "Anole T VME BA"}, 
	{HPHW_BA, 0x04A, 0x00078, 0x0, "Anole L2 132 VME BA"}, 
	{HPHW_BA, 0x04C, 0x00078, 0x0, "Anole L2 165 VME BA"}, 
	{HPHW_BA, 0x011, 0x00081, 0x0, "WB-96 Core BA"}, 
	{HPHW_BA, 0x012, 0x00081, 0x0, "Orville UX Core BA"}, 
	{HPHW_BA, 0x013, 0x00081, 0x0, "Wilbur UX Core BA"}, 
	{HPHW_BA, 0x014, 0x00081, 0x0, "WB-80 Core BA"}, 
	{HPHW_BA, 0x015, 0x00081, 0x0, "KittyHawk GSY Core BA"}, 
	{HPHW_BA, 0x016, 0x00081, 0x0, "Gecko Core BA"}, 
	{HPHW_BA, 0x018, 0x00081, 0x0, "Gecko Optional BA"}, 
	{HPHW_BA, 0x01A, 0x00081, 0x0, "Anole 64 Core BA"}, 
	{HPHW_BA, 0x01B, 0x00081, 0x0, "Anole 100 Core BA"}, 
	{HPHW_BA, 0x01C, 0x00081, 0x0, "Gecko 80 Core BA"}, 
	{HPHW_BA, 0x01D, 0x00081, 0x0, "Gecko 100 Core BA"}, 
	{HPHW_BA, 0x01F, 0x00081, 0x0, "SkyHawk 100/120 Core BA"}, 
	{HPHW_BA, 0x027, 0x00081, 0x0, "Piranha 100 Core BA"}, 
	{HPHW_BA, 0x028, 0x00081, 0x0, "Mirage Jr Core BA"}, 
	{HPHW_BA, 0x029, 0x00081, 0x0, "Mirage Core BA"}, 
	{HPHW_BA, 0x02A, 0x00081, 0x0, "Electra Core BA"}, 
	{HPHW_BA, 0x02B, 0x00081, 0x0, "Mirage 80 Core BA"}, 
	{HPHW_BA, 0x02C, 0x00081, 0x0, "Mirage 100+ Core BA"}, 
	{HPHW_BA, 0x02E, 0x00081, 0x0, "UL 350 Lasi Core BA"}, 
	{HPHW_BA, 0x02F, 0x00081, 0x0, "UL 550 Lasi Core BA"}, 
	{HPHW_BA, 0x032, 0x00081, 0x0, "Raven T' Core BA"}, 
	{HPHW_BA, 0x033, 0x00081, 0x0, "Anole T Core BA"}, 
	{HPHW_BA, 0x034, 0x00081, 0x0, "SAIC L-80 Core BA"}, 
	{HPHW_BA, 0x035, 0x00081, 0x0, "PCX-L2 712/132 Core BA"}, 
	{HPHW_BA, 0x036, 0x00081, 0x0, "PCX-L2 712/160 Core BA"}, 
	{HPHW_BA, 0x03B, 0x00081, 0x0, "Raven U/L2 Core BA"}, 
	{HPHW_BA, 0x03C, 0x00081, 0x0, "Merlin 132 Core BA"}, 
	{HPHW_BA, 0x03D, 0x00081, 0x0, "Merlin 160 Core BA"}, 
	{HPHW_BA, 0x03E, 0x00081, 0x0, "Merlin+ 132 Core BA"}, 
	{HPHW_BA, 0x03F, 0x00081, 0x0, "Merlin+ 180 Core BA"}, 
	{HPHW_BA, 0x044, 0x00081, 0x0, "Mohawk Core BA"}, 
	{HPHW_BA, 0x045, 0x00081, 0x0, "Rocky1 Core BA"}, 
	{HPHW_BA, 0x046, 0x00081, 0x0, "Rocky2 120 Core BA"}, 
	{HPHW_BA, 0x047, 0x00081, 0x0, "Rocky2 150 Core BA"}, 
	{HPHW_BA, 0x04B, 0x00081, 0x0, "Anole L2 132 Core BA"}, 
	{HPHW_BA, 0x04D, 0x00081, 0x0, "Anole L2 165 Core BA"}, 
	{HPHW_BA, 0x04E, 0x00081, 0x0, "Kiji L2 132 Core BA"}, 
	{HPHW_BA, 0x050, 0x00081, 0x0, "Merlin Jr 132 Core BA"}, 
	{HPHW_BA, 0x051, 0x00081, 0x0, "Firehawk Core BA"}, 
	{HPHW_BA, 0x056, 0x00081, 0x0, "Raven+ w SE FWSCSI Core BA"}, 
	{HPHW_BA, 0x057, 0x00081, 0x0, "Raven+ w Diff FWSCSI Core BA"}, 
	{HPHW_BA, 0x058, 0x00081, 0x0, "FireHawk 200 Core BA"}, 
	{HPHW_BA, 0x05C, 0x00081, 0x0, "SummitHawk 230 Core BA"}, 
	{HPHW_BA, 0x05E, 0x00081, 0x0, "Staccato 132 Core BA"}, 
	{HPHW_BA, 0x05E, 0x00081, 0x0, "Staccato 180 Core BA"}, 
	{HPHW_BA, 0x05F, 0x00081, 0x0, "Staccato 180 Lasi"}, 
	{HPHW_BA, 0x800, 0x00081, 0x0, "Hitachi Tiny 64 Core BA"}, 
	{HPHW_BA, 0x801, 0x00081, 0x0, "Hitachi Tiny 80 Core BA"}, 
	{HPHW_BA, 0x004, 0x0008B, 0x0, "Anole Optional PCMCIA BA"}, 
	{HPHW_BA, 0x004, 0x0008E, 0x0, "GSC ITR Wax BA"}, 
	{HPHW_BA, 0x00C, 0x0008E, 0x0, "Gecko Optional Wax BA"}, 
	{HPHW_BA, 0x010, 0x0008E, 0x0, "Pace Wax BA"}, 
	{HPHW_BA, 0x011, 0x0008E, 0x0, "SuperPace Wax BA"}, 
	{HPHW_BA, 0x012, 0x0008E, 0x0, "Mirage Jr Wax BA"}, 
	{HPHW_BA, 0x013, 0x0008E, 0x0, "Mirage Wax BA"}, 
	{HPHW_BA, 0x014, 0x0008E, 0x0, "Electra Wax BA"}, 
	{HPHW_BA, 0x017, 0x0008E, 0x0, "Raven Backplane Wax BA"}, 
	{HPHW_BA, 0x01E, 0x0008E, 0x0, "Raven T' Wax BA"}, 
	{HPHW_BA, 0x01F, 0x0008E, 0x0, "SkyHawk Wax BA"}, 
	{HPHW_BA, 0x023, 0x0008E, 0x0, "Rocky1 Wax BA"}, 
	{HPHW_BA, 0x02B, 0x0008E, 0x0, "Mirage 80 Wax BA"}, 
	{HPHW_BA, 0x02C, 0x0008E, 0x0, "Mirage 100+ Wax BA"}, 
	{HPHW_BA, 0x030, 0x0008E, 0x0, "UL 350 Core Wax BA"}, 
	{HPHW_BA, 0x031, 0x0008E, 0x0, "UL 550 Core Wax BA"}, 
	{HPHW_BA, 0x034, 0x0008E, 0x0, "SAIC L-80 Wax BA"}, 
	{HPHW_BA, 0x03A, 0x0008E, 0x0, "Merlin+ Wax BA"}, 
	{HPHW_BA, 0x040, 0x0008E, 0x0, "Merlin 132 Wax BA"}, 
	{HPHW_BA, 0x041, 0x0008E, 0x0, "Merlin 160 Wax BA"}, 
	{HPHW_BA, 0x043, 0x0008E, 0x0, "Merlin 132/160 Wax BA"}, 
	{HPHW_BA, 0x052, 0x0008E, 0x0, "Raven+ Hi Power Backplane w/EISA Wax BA"}, 
	{HPHW_BA, 0x054, 0x0008E, 0x0, "Raven+ Lo Power Backplane w/EISA Wax BA"}, 
	{HPHW_BA, 0x059, 0x0008E, 0x0, "FireHawk 200 Wax BA"}, 
	{HPHW_BA, 0x05A, 0x0008E, 0x0, "Raven+ L2 Backplane w/EISA Wax BA"}, 
	{HPHW_BA, 0x05D, 0x0008E, 0x0, "SummitHawk Wax BA"}, 
	{HPHW_BA, 0x800, 0x0008E, 0x0, "Hitachi Tiny 64 Wax BA"}, 
	{HPHW_BA, 0x801, 0x0008E, 0x0, "Hitachi Tiny 80 Wax BA"}, 
	{HPHW_BA, 0x011, 0x00090, 0x0, "SuperPace Wax EISA BA"}, 
	{HPHW_BA, 0x017, 0x00090, 0x0, "Raven Backplane Wax EISA BA"}, 
	{HPHW_BA, 0x01E, 0x00090, 0x0, "Raven T' Wax EISA BA"}, 
	{HPHW_BA, 0x01F, 0x00090, 0x0, "SkyHawk 100/120 Wax EISA BA"}, 
	{HPHW_BA, 0x027, 0x00090, 0x0, "Piranha 100 Wax EISA BA"}, 
	{HPHW_BA, 0x028, 0x00090, 0x0, "Mirage Jr Wax EISA BA"}, 
	{HPHW_BA, 0x029, 0x00090, 0x0, "Mirage Wax EISA BA"}, 
	{HPHW_BA, 0x02A, 0x00090, 0x0, "Electra Wax EISA BA"}, 
	{HPHW_BA, 0x02B, 0x00090, 0x0, "Mirage 80 Wax EISA BA"}, 
	{HPHW_BA, 0x02C, 0x00090, 0x0, "Mirage 100+ Wax EISA BA"}, 
	{HPHW_BA, 0x030, 0x00090, 0x0, "UL 350 Wax EISA BA"}, 
	{HPHW_BA, 0x031, 0x00090, 0x0, "UL 550 Wax EISA BA"}, 
	{HPHW_BA, 0x034, 0x00090, 0x0, "SAIC L-80 Wax EISA BA"}, 
	{HPHW_BA, 0x03A, 0x00090, 0x0, "Merlin+ Wax EISA BA"}, 
	{HPHW_BA, 0x040, 0x00090, 0x0, "Merlin 132 Wax EISA BA"}, 
	{HPHW_BA, 0x041, 0x00090, 0x0, "Merlin 160 Wax EISA BA"}, 
	{HPHW_BA, 0x043, 0x00090, 0x0, "Merlin 132/160 Wax EISA BA"}, 
	{HPHW_BA, 0x052, 0x00090, 0x0, "Raven Hi Power Backplane Wax EISA BA"}, 
	{HPHW_BA, 0x054, 0x00090, 0x0, "Raven Lo Power Backplane Wax EISA BA"}, 
	{HPHW_BA, 0x059, 0x00090, 0x0, "FireHawk 200 Wax EISA BA"}, 
	{HPHW_BA, 0x05A, 0x00090, 0x0, "Raven L2 Backplane Wax EISA BA"}, 
	{HPHW_BA, 0x05D, 0x00090, 0x0, "SummitHawk Wax EISA BA"}, 
	{HPHW_BA, 0x800, 0x00090, 0x0, "Hitachi Tiny 64 Wax EISA BA"}, 
	{HPHW_BA, 0x801, 0x00090, 0x0, "Hitachi Tiny 80 Wax EISA BA"}, 
	{HPHW_BA, 0x01A, 0x00093, 0x0, "Anole 64 TIMI BA"}, 
	{HPHW_BA, 0x01B, 0x00093, 0x0, "Anole 100 TIMI BA"}, 
	{HPHW_BA, 0x034, 0x00093, 0x0, "Anole T TIMI BA"}, 
	{HPHW_BA, 0x04A, 0x00093, 0x0, "Anole L2 132 TIMI BA"}, 
	{HPHW_BA, 0x04C, 0x00093, 0x0, "Anole L2 165 TIMI BA"}, 
	{HPHW_BA, 0x582, 0x000A5, 0x00, "Epic PCI Bridge"}, 
	{HPHW_BCPORT, 0x504, 0x00000, 0x00, "Phantom PseudoBC GSC+ Port"}, 
	{HPHW_BCPORT, 0x505, 0x00000, 0x00, "Phantom PseudoBC GSC+ Port"}, 
	{HPHW_BCPORT, 0x503, 0x0000C, 0x00, "Java BC GSC+ Port"}, 
	{HPHW_BCPORT, 0x57F, 0x0000C, 0x00, "Hitachi Ghostview GSC+ Port"}, 
	{HPHW_BCPORT, 0x501, 0x0000C, 0x00, "U2-IOA BC GSC+ Port"}, 
	{HPHW_BCPORT, 0x502, 0x0000C, 0x00, "Uturn-IOA BC GSC+ Port"}, 
	{HPHW_BCPORT, 0x780, 0x0000C, 0x00, "Astro BC Ropes Port"}, 
	{HPHW_BCPORT, 0x506, 0x0000C, 0x00, "NEC-IOS BC HSC Port"}, 
	{HPHW_BCPORT, 0x004, 0x0000C, 0x00, "Cheetah BC SMB Port"}, 
	{HPHW_BCPORT, 0x006, 0x0000C, 0x00, "Cheetah BC MID_BUS Port"}, 
	{HPHW_BCPORT, 0x005, 0x0000C, 0x00, "Condor BC MID_BUS Port"}, 
	{HPHW_BCPORT, 0x100, 0x0000C, 0x00, "Condor BC HP-PB Port"}, 
	{HPHW_BCPORT, 0x184, 0x0000C, 0x00, "Summit BC Port"}, 
	{HPHW_BCPORT, 0x101, 0x0000C, 0x00, "Summit BC HP-PB Port"}, 
	{HPHW_BCPORT, 0x102, 0x0000C, 0x00, "HP-PB Port (prefetch)"}, 
	{HPHW_BCPORT, 0x500, 0x0000C, 0x00, "Gecko BOA BC GSC+ Port"}, 
	{HPHW_BCPORT, 0x103, 0x0000C, 0x00, "Gecko BOA BC HP-PB Port"}, 
	{HPHW_BCPORT, 0x507, 0x0000C, 0x00, "Keyaki BC GSC+ Port"}, 
	{HPHW_BCPORT, 0x508, 0x0000C, 0x00, "Keyaki-DX BC GSC+ Port"}, 
	{HPHW_BCPORT, 0x584, 0x0000C, 0x10, "DEW BC Runway Port"}, 
	{HPHW_BCPORT, 0x800, 0x0000C, 0x10, "DEW BC Merced Port"}, 
	{HPHW_BCPORT, 0x801, 0x0000C, 0x10, "SMC Bus Interface Merced Bus0"}, 
	{HPHW_BCPORT, 0x802, 0x0000C, 0x10, "SMC Bus INterface Merced Bus1"}, 
	{HPHW_BCPORT, 0x803, 0x0000C, 0x10, "IKE I/O BC Merced Port"}, 
	{HPHW_BCPORT, 0x781, 0x0000C, 0x00, "IKE I/O BC Ropes Port"}, 
	{HPHW_BCPORT, 0x804, 0x0000C, 0x10, "REO I/O BC Merced Port"}, 
	{HPHW_BCPORT, 0x782, 0x0000C, 0x00, "REO I/O BC Ropes Port"}, 
	{HPHW_BCPORT, 0x784, 0x0000C, 0x00, "Pluto I/O BC Ropes Port"}, 
	{HPHW_BRIDGE, 0x680, 0x0000A, 0x00, "Dino PCI Bridge"}, 
	{HPHW_BRIDGE, 0x682, 0x0000A, 0x00, "Cujo PCI Bridge"}, 
	{HPHW_BRIDGE, 0x782, 0x0000A, 0x00, "Elroy PCI Bridge"}, 
	{HPHW_BRIDGE, 0x583, 0x000A5, 0x00, "Saga PCI Bridge"}, 
	{HPHW_BRIDGE, 0x783, 0x0000A, 0x00, "Mercury PCI Bridge"}, 
	{HPHW_BRIDGE, 0x784, 0x0000A, 0x00, "Quicksilver AGP Bridge"}, 
	{HPHW_B_DMA, 0x004, 0x00018, 0x00, "Parallel I/O"}, 
	{HPHW_B_DMA, 0x004, 0x00019, 0x00, "Parallel RDB"}, 
	{HPHW_B_DMA, 0x004, 0x00020, 0x80, "MID_BUS PSI"}, 
	{HPHW_B_DMA, 0x004, 0x0002F, 0x80, "HP-PB Transit PSI (36960A)"}, 
	{HPHW_B_DMA, 0x008, 0x00051, 0x80, "HP-PB Transit 802.3"}, 
	{HPHW_B_DMA, 0x004, 0x00052, 0x80, "Miura LAN/Console (J2146A)"}, 
	{HPHW_B_DMA, 0x008, 0x00058, 0x80, "HP-PB Transit 802.4"}, 
	{HPHW_B_DMA, 0x005, 0x00060, 0x80, "KittyHawk CSY Core LAN/Console"}, 
	{HPHW_B_DMA, 0x014, 0x00060, 0x80, "Diablo LAN/Console"}, 
	{HPHW_B_DMA, 0x054, 0x00060, 0x80, "Countach LAN/Console"}, 
	{HPHW_B_DMA, 0x004, 0x00094, 0x80, "KittyHawk GSC+ Exerciser"}, 
	{HPHW_B_DMA, 0x004, 0x00100, 0x80, "HP-PB HF Interface"}, 
	{HPHW_B_DMA, 0x000, 0x00206, 0x80, "MELCO HMPHA"}, 
	{HPHW_B_DMA, 0x005, 0x00206, 0x80, "MELCO HMPHA_10"}, 
	{HPHW_B_DMA, 0x006, 0x00206, 0x80, "MELCO HMQHA"}, 
	{HPHW_B_DMA, 0x007, 0x00206, 0x80, "MELCO HMQHA_10"}, 
	{HPHW_B_DMA, 0x004, 0x207, 0x80, "MELCO HNDWA MDWS-70"}, 
	{HPHW_CIO, 0x004, 0x00010, 0x00, "VLSI CIO"}, 
	{HPHW_CIO, 0x005, 0x00010, 0x00, "Silverfox CIO"}, 
	{HPHW_CIO, 0x006, 0x00010, 0x00, "Emerald CIO"}, 
	{HPHW_CIO, 0x008, 0x00010, 0x00, "Discrete CIO"}, 
	{HPHW_CONSOLE, 0x004, 0x0001C, 0x00, "Cheetah console"}, 
	{HPHW_CONSOLE, 0x005, 0x0001C, 0x00, "Emerald console"}, 
	{HPHW_CONSOLE, 0x01A, 0x0001F, 0x00, "Jason/Anole 64 Null Console"}, 
	{HPHW_CONSOLE, 0x01B, 0x0001F, 0x00, "Jason/Anole 100 Null Console"}, 
	{HPHW_FABRIC, 0x004, 0x000AA, 0x80, "Halfdome DNA Central Agent"}, 
	{HPHW_FABRIC, 0x007, 0x000AA, 0x80, "Caribe DNA Central Agent"}, 
	{HPHW_FABRIC, 0x004, 0x000AB, 0x00, "Halfdome TOGO Fabric Crossbar"}, 
	{HPHW_FABRIC, 0x004, 0x000AC, 0x00, "Halfdome Sakura Fabric Router"}, 
	{HPHW_FIO, 0x025, 0x0002E, 0x80, "Armyknife Optional X.25"}, 
	{HPHW_FIO, 0x004, 0x0004F, 0x0, "8-Port X.25 EISA-ACC (AMSO)"}, 
	{HPHW_FIO, 0x004, 0x00071, 0x0, "Cobra Core SCSI"}, 
	{HPHW_FIO, 0x005, 0x00071, 0x0, "Coral Core SCSI"}, 
	{HPHW_FIO, 0x006, 0x00071, 0x0, "Bushmaster Core SCSI"}, 
	{HPHW_FIO, 0x007, 0x00071, 0x0, "Scorpio Core SCSI"}, 
	{HPHW_FIO, 0x008, 0x00071, 0x0, "Flounder Core SCSI"}, 
	{HPHW_FIO, 0x009, 0x00071, 0x0, "Outfield Core SCSI"}, 
	{HPHW_FIO, 0x00A, 0x00071, 0x0, "CoralII Core SCSI"}, 
	{HPHW_FIO, 0x00B, 0x00071, 0x0, "Scorpio Jr. Core SCSI"}, 
	{HPHW_FIO, 0x00C, 0x00071, 0x0, "Strider-50 Core SCSI"}, 
	{HPHW_FIO, 0x00D, 0x00071, 0x0, "Strider-33 Core SCSI"}, 
	{HPHW_FIO, 0x00E, 0x00071, 0x0, "Trailways-50 Core SCSI"}, 
	{HPHW_FIO, 0x00F, 0x00071, 0x0, "Trailways-33 Core SCSI"}, 
	{HPHW_FIO, 0x010, 0x00071, 0x0, "Pace Core SCSI"}, 
	{HPHW_FIO, 0x011, 0x00071, 0x0, "Sidewinder Core SCSI"}, 
	{HPHW_FIO, 0x019, 0x00071, 0x0, "Scorpio Sr. Core SCSI"}, 
	{HPHW_FIO, 0x020, 0x00071, 0x0, "Scorpio 100 Core SCSI"}, 
	{HPHW_FIO, 0x021, 0x00071, 0x0, "Spectra 50 Core SCSI"}, 
	{HPHW_FIO, 0x022, 0x00071, 0x0, "Spectra 75 Core SCSI"}, 
	{HPHW_FIO, 0x023, 0x00071, 0x0, "Spectra 100 Core SCSI"}, 
	{HPHW_FIO, 0x024, 0x00071, 0x0, "Fast Pace Core SCSI"}, 
	{HPHW_FIO, 0x026, 0x00071, 0x0, "CoralII Jaguar Core SCSI"}, 
	{HPHW_FIO, 0x004, 0x00072, 0x0, "Cobra Core LAN (802.3)"}, 
	{HPHW_FIO, 0x005, 0x00072, 0x0, "Coral Core LAN (802.3)"}, 
	{HPHW_FIO, 0x006, 0x00072, 0x0, "Bushmaster Core LAN (802.3)"}, 
	{HPHW_FIO, 0x007, 0x00072, 0x0, "Scorpio Core LAN (802.3)"}, 
	{HPHW_FIO, 0x008, 0x00072, 0x0, "Flounder Core LAN (802.3)"}, 
	{HPHW_FIO, 0x009, 0x00072, 0x0, "Outfield Core LAN (802.3)"}, 
	{HPHW_FIO, 0x00A, 0x00072, 0x0, "CoralII Core LAN (802.3)"}, 
	{HPHW_FIO, 0x00B, 0x00072, 0x0, "Scorpio Jr. Core LAN (802.3)"}, 
	{HPHW_FIO, 0x00C, 0x00072, 0x0, "Strider-50 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x00D, 0x00072, 0x0, "Strider-33 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x00E, 0x00072, 0x0, "Trailways-50 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x00F, 0x00072, 0x0, "Trailways-33 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x010, 0x00072, 0x0, "Pace Core LAN (802.3)"}, 
	{HPHW_FIO, 0x011, 0x00072, 0x0, "Sidewinder Core LAN (802.3)"}, 
	{HPHW_FIO, 0x019, 0x00072, 0x0, "Scorpio Sr. Core LAN (802.3)"}, 
	{HPHW_FIO, 0x020, 0x00072, 0x0, "Scorpio 100 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x021, 0x00072, 0x0, "Spectra 50 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x022, 0x00072, 0x0, "Spectra 75 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x023, 0x00072, 0x0, "Spectra 100 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x024, 0x00072, 0x0, "Fast Pace Core LAN (802.3)"}, 
	{HPHW_FIO, 0x026, 0x00072, 0x0, "CoralII Jaguar Core LAN (802.3)"}, 
	{HPHW_FIO, 0x004, 0x00073, 0x0, "Cobra Core HIL"}, 
	{HPHW_FIO, 0x005, 0x00073, 0x0, "Coral Core HIL"}, 
	{HPHW_FIO, 0x006, 0x00073, 0x0, "Bushmaster Core HIL"}, 
	{HPHW_FIO, 0x007, 0x00073, 0x0, "Scorpio Core HIL"}, 
	{HPHW_FIO, 0x008, 0x00073, 0x0, "Flounder Core HIL"}, 
	{HPHW_FIO, 0x009, 0x00073, 0x0, "Outfield Core HIL"}, 
	{HPHW_FIO, 0x00A, 0x00073, 0x0, "CoralII Core HIL"}, 
	{HPHW_FIO, 0x00B, 0x00073, 0x0, "Scorpio Jr. Core HIL"}, 
	{HPHW_FIO, 0x00C, 0x00073, 0x0, "Strider-50 Core HIL"}, 
	{HPHW_FIO, 0x00D, 0x00073, 0x0, "Strider-33 Core HIL"}, 
	{HPHW_FIO, 0x00E, 0x00073, 0x0, "Trailways-50 Core HIL"}, 
	{HPHW_FIO, 0x00F, 0x00073, 0x0, "Trailways-33 Core HIL"}, 
	{HPHW_FIO, 0x010, 0x00073, 0x0, "Pace Core HIL"}, 
	{HPHW_FIO, 0x011, 0x00073, 0xcc, "SuperPace Wax HIL"}, 
	{HPHW_FIO, 0x012, 0x00073, 0x0, "Mirage Jr Wax HIL"}, 
	{HPHW_FIO, 0x013, 0x00073, 0x0, "Mirage 100 Wax HIL"}, 
	{HPHW_FIO, 0x014, 0x00073, 0x0, "Electra Wax HIL"}, 
	{HPHW_FIO, 0x017, 0x00073, 0x0, "Raven Backplane Wax HIL"}, 
	{HPHW_FIO, 0x019, 0x00073, 0x0, "Scorpio Sr. Core HIL"}, 
	{HPHW_FIO, 0x01E, 0x00073, 0x0, "Raven T' Wax HIL"}, 
	{HPHW_FIO, 0x01F, 0x00073, 0x0, "SkyHawk 100/120 Wax HIL"}, 
	{HPHW_FIO, 0x020, 0x00073, 0x0, "Scorpio 100 Core HIL"}, 
	{HPHW_FIO, 0x021, 0x00073, 0x0, "Spectra 50 Core HIL"}, 
	{HPHW_FIO, 0x022, 0x00073, 0x0, "Spectra 75 Core HIL"}, 
	{HPHW_FIO, 0x023, 0x00073, 0x0, "Spectra 100 Core HIL"}, 
	{HPHW_FIO, 0x024, 0x00073, 0x0, "Fast Pace Core HIL"}, 
	{HPHW_FIO, 0x026, 0x00073, 0x0, "CoralII Jaguar Core HIL"}, 
	{HPHW_FIO, 0x02B, 0x00073, 0x0, "Mirage 80 Wax HIL"}, 
	{HPHW_FIO, 0x02C, 0x00073, 0x0, "Mirage 100+ Wax HIL"}, 
	{HPHW_FIO, 0x03A, 0x00073, 0x0, "Merlin+ Wax HIL"}, 
	{HPHW_FIO, 0x040, 0x00073, 0x0, "Merlin 132 Wax HIL"}, 
	{HPHW_FIO, 0x041, 0x00073, 0x0, "Merlin 160 Wax HIL"}, 
	{HPHW_FIO, 0x043, 0x00073, 0x0, "Merlin 132/160 Wax HIL"}, 
	{HPHW_FIO, 0x052, 0x00073, 0x0, "Raven+ Hi Power Backplane w/EISA Wax HIL"}, 
	{HPHW_FIO, 0x053, 0x00073, 0x0, "Raven+ Hi Power Backplane wo/EISA Wax HIL"}, 
	{HPHW_FIO, 0x054, 0x00073, 0x0, "Raven+ Lo Power Backplane w/EISA Wax HIL"}, 
	{HPHW_FIO, 0x055, 0x00073, 0x0, "Raven+ Lo Power Backplane wo/EISA Wax HIL"}, 
	{HPHW_FIO, 0x059, 0x00073, 0x0, "FireHawk 200 Wax HIL"}, 
	{HPHW_FIO, 0x05A, 0x00073, 0x0, "Raven+ L2 Backplane w/EISA Wax HIL"}, 
	{HPHW_FIO, 0x05B, 0x00073, 0x0, "Raven+ L2 Backplane wo/EISA Wax HIL"}, 
	{HPHW_FIO, 0x05D, 0x00073, 0x0, "SummitHawk Wax HIL"}, 
	{HPHW_FIO, 0x800, 0x00073, 0x0, "Hitachi Tiny 64 Wax HIL"}, 
	{HPHW_FIO, 0x801, 0x00073, 0x0, "Hitachi Tiny 80 Wax HIL"}, 
	{HPHW_FIO, 0x004, 0x00074, 0x0, "Cobra Core Centronics"}, 
	{HPHW_FIO, 0x005, 0x00074, 0x0, "Coral Core Centronics"}, 
	{HPHW_FIO, 0x006, 0x00074, 0x0, "Bushmaster Core Centronics"}, 
	{HPHW_FIO, 0x007, 0x00074, 0x0, "Scorpio Core Centronics"}, 
	{HPHW_FIO, 0x008, 0x00074, 0x0, "Flounder Core Centronics"}, 
	{HPHW_FIO, 0x009, 0x00074, 0x0, "Outfield Core Centronics"}, 
	{HPHW_FIO, 0x00A, 0x00074, 0x0, "CoralII Core Centronics"}, 
	{HPHW_FIO, 0x00B, 0x00074, 0x0, "Scorpio Jr. Core Centronics"}, 
	{HPHW_FIO, 0x00C, 0x00074, 0x0, "Strider-50 Core Centronics"}, 
	{HPHW_FIO, 0x00D, 0x00074, 0x0, "Strider-33 Core Centronics"}, 
	{HPHW_FIO, 0x00E, 0x00074, 0x0, "Trailways-50 Core Centronics"}, 
	{HPHW_FIO, 0x00F, 0x00074, 0x0, "Trailways-33 Core Centronics"}, 
	{HPHW_FIO, 0x010, 0x00074, 0x0, "Pace Core Centronics"}, 
	{HPHW_FIO, 0x011, 0x00074, 0x0, "Sidewinder Core Centronics"}, 
	{HPHW_FIO, 0x015, 0x00074, 0x0, "KittyHawk GSY Core Centronics"}, 
	{HPHW_FIO, 0x016, 0x00074, 0x0, "Gecko Core Centronics"}, 
	{HPHW_FIO, 0x019, 0x00074, 0x0, "Scorpio Sr. Core Centronics"}, 
	{HPHW_FIO, 0x01A, 0x00074, 0x0, "Anole 64 Core Centronics"}, 
	{HPHW_FIO, 0x01B, 0x00074, 0x0, "Anole 100 Core Centronics"}, 
	{HPHW_FIO, 0x01C, 0x00074, 0x0, "Gecko 80 Core Centronics"}, 
	{HPHW_FIO, 0x01D, 0x00074, 0x0, "Gecko 100 Core Centronics"}, 
	{HPHW_FIO, 0x01F, 0x00074, 0x0, "SkyHawk 100/120 Core Centronics"}, 
	{HPHW_FIO, 0x020, 0x00074, 0x0, "Scorpio 100 Core Centronics"}, 
	{HPHW_FIO, 0x021, 0x00074, 0x0, "Spectra 50 Core Centronics"}, 
	{HPHW_FIO, 0x022, 0x00074, 0x0, "Spectra 75 Core Centronics"}, 
	{HPHW_FIO, 0x023, 0x00074, 0x0, "Spectra 100 Core Centronics"}, 
	{HPHW_FIO, 0x024, 0x00074, 0x0, "Fast Pace Core Centronics"}, 
	{HPHW_FIO, 0x026, 0x00074, 0x0, "CoralII Jaguar Core Centronics"}, 
	{HPHW_FIO, 0x027, 0x00074, 0x0, "Piranha 100 Core Centronics"}, 
	{HPHW_FIO, 0x028, 0x00074, 0x0, "Mirage Jr Core Centronics"}, 
	{HPHW_FIO, 0x029, 0x00074, 0x0, "Mirage Core Centronics"}, 
	{HPHW_FIO, 0x02A, 0x00074, 0x0, "Electra Core Centronics"}, 
	{HPHW_FIO, 0x02B, 0x00074, 0x0, "Mirage 80 Core Centronics"}, 
	{HPHW_FIO, 0x02C, 0x00074, 0x0, "Mirage 100+ Core Centronics"}, 
	{HPHW_FIO, 0x02E, 0x00074, 0x0, "UL 350 Core Centronics"}, 
	{HPHW_FIO, 0x02F, 0x00074, 0x0, "UL 550 Core Centronics"}, 
	{HPHW_FIO, 0x032, 0x00074, 0x0, "Raven T' Core Centronics"}, 
	{HPHW_FIO, 0x033, 0x00074, 0x0, "Anole T Core Centronics"}, 
	{HPHW_FIO, 0x034, 0x00074, 0x0, "SAIC L-80 Core Centronics"}, 
	{HPHW_FIO, 0x035, 0x00074, 0x0, "PCX-L2 712/132 Core Centronics"}, 
	{HPHW_FIO, 0x036, 0x00074, 0x0, "PCX-L2 712/160 Core Centronics"}, 
	{HPHW_FIO, 0x03B, 0x00074, 0x0, "Raven U/L2 Core Centronics"}, 
	{HPHW_FIO, 0x03C, 0x00074, 0x0, "Merlin 132 Core Centronics"}, 
	{HPHW_FIO, 0x03D, 0x00074, 0x0, "Merlin 160 Core Centronics"}, 
	{HPHW_FIO, 0x03E, 0x00074, 0x0, "Merlin+ 132 Core Centronics"}, 
	{HPHW_FIO, 0x03F, 0x00074, 0x0, "Merlin+ 180 Core Centronics"}, 
	{HPHW_FIO, 0x044, 0x00074, 0x0, "Mohawk Core Centronics"}, 
	{HPHW_FIO, 0x045, 0x00074, 0x0, "Rocky1 Core Centronics"}, 
	{HPHW_FIO, 0x046, 0x00074, 0x0, "Rocky2 120 Core Centronics"}, 
	{HPHW_FIO, 0x047, 0x00074, 0x0, "Rocky2 150 Core Centronics"}, 
	{HPHW_FIO, 0x04B, 0x00074, 0x0, "Anole L2 132 Core Centronics"}, 
	{HPHW_FIO, 0x04D, 0x00074, 0x0, "Anole L2 165 Core Centronics"}, 
	{HPHW_FIO, 0x050, 0x00074, 0x0, "Merlin Jr 132 Core Centronics"}, 
	{HPHW_FIO, 0x051, 0x00074, 0x0, "Firehawk Core Centronics"}, 
	{HPHW_FIO, 0x056, 0x00074, 0x0, "Raven+ w SE FWSCSI Core Centronics"}, 
	{HPHW_FIO, 0x057, 0x00074, 0x0, "Raven+ w Diff FWSCSI Core Centronics"}, 
	{HPHW_FIO, 0x058, 0x00074, 0x0, "FireHawk 200 Core Centronics"}, 
	{HPHW_FIO, 0x05C, 0x00074, 0x0, "SummitHawk 230 Core Centronics"}, 
	{HPHW_FIO, 0x800, 0x00074, 0x0, "Hitachi Tiny 64 Core Centronics"}, 
	{HPHW_FIO, 0x801, 0x00074, 0x0, "Hitachi Tiny 80 Core Centronics"}, 
	{HPHW_FIO, 0x004, 0x00075, 0x0, "Cobra Core RS-232"}, 
	{HPHW_FIO, 0x005, 0x00075, 0x0, "Coral Core RS-232"}, 
	{HPHW_FIO, 0x006, 0x00075, 0x0, "Bushmaster Core RS-232"}, 
	{HPHW_FIO, 0x007, 0x00075, 0x0, "Scorpio Core RS-232"}, 
	{HPHW_FIO, 0x008, 0x00075, 0x0, "Flounder Core RS-232"}, 
	{HPHW_FIO, 0x009, 0x00075, 0x0, "Outfield Core RS-232"}, 
	{HPHW_FIO, 0x00A, 0x00075, 0x0, "CoralII Core RS-232"}, 
	{HPHW_FIO, 0x00B, 0x00075, 0x0, "Scorpio Jr. Core RS-232"}, 
	{HPHW_FIO, 0x00C, 0x00075, 0x0, "Strider-50 Core RS-232"}, 
	{HPHW_FIO, 0x00D, 0x00075, 0x0, "Strider-33 Core RS-232"}, 
	{HPHW_FIO, 0x00E, 0x00075, 0x0, "Trailways-50 Core RS-232"}, 
	{HPHW_FIO, 0x00F, 0x00075, 0x0, "Trailways-33 Core RS-232"}, 
	{HPHW_FIO, 0x010, 0x00075, 0x0, "Pace Core RS-232"}, 
	{HPHW_FIO, 0x011, 0x00075, 0x0, "Sidewinder Core RS-232"}, 
	{HPHW_FIO, 0x019, 0x00075, 0x0, "Scorpio Sr. Core RS-232"}, 
	{HPHW_FIO, 0x020, 0x00075, 0x0, "Scorpio 100 Core RS-232"}, 
	{HPHW_FIO, 0x021, 0x00075, 0x0, "Spectra 50 Core RS-232"}, 
	{HPHW_FIO, 0x022, 0x00075, 0x0, "Spectra 75 Core RS-232"}, 
	{HPHW_FIO, 0x023, 0x00075, 0x0, "Spectra 100 Core RS-232"}, 
	{HPHW_FIO, 0x024, 0x00075, 0x0, "Fast Pace Core RS-232"}, 
	{HPHW_FIO, 0x026, 0x00075, 0x0, "CoralII Jaguar Core RS-232"}, 
	{HPHW_FIO, 0x004, 0x00077, 0x0, "Coral SGC Graphics"}, 
	{HPHW_FIO, 0x005, 0x00077, 0x0, "Hyperdrive Optional Graphics"}, 
	{HPHW_FIO, 0x006, 0x00077, 0x0, "Stinger Optional Graphics"}, 
	{HPHW_FIO, 0x007, 0x00077, 0x0, "Scorpio Builtin Graphics"}, 
	{HPHW_FIO, 0x008, 0x00077, 0x0, "Anole Hyperdrive Optional Graphics"}, 
	{HPHW_FIO, 0x009, 0x00077, 0x0, "Thunder II graphics EISA form"}, 
	{HPHW_FIO, 0x00A, 0x00077, 0x0, "Thunder II graphics GSA form"}, 
	{HPHW_FIO, 0x00B, 0x00077, 0x0, "Scorpio Jr Builtin Graphics"}, 
	{HPHW_FIO, 0x00C, 0x00077, 0x0, "Strider-50 SSC Graphics"}, 
	{HPHW_FIO, 0x00D, 0x00077, 0x0, "Strider-33 SSC Graphics"}, 
	{HPHW_FIO, 0x00E, 0x00077, 0x0, "Trailways-50 SSC Graphics"}, 
	{HPHW_FIO, 0x00F, 0x00077, 0x0, "Trailways-33 SSC Graphics"}, 
	{HPHW_FIO, 0x010, 0x00077, 0x0, "Pace SGC Graphics"}, 
	{HPHW_FIO, 0x011, 0x00077, 0x0, "Mohawk Opt. 2D Graphics (Kid)"}, 
	{HPHW_FIO, 0x012, 0x00077, 0x0, "Raven Opt. 2D Graphics (Goat)"}, 
	{HPHW_FIO, 0x016, 0x00077, 0x0, "Lego 24 SCG Graphics"}, 
	{HPHW_FIO, 0x017, 0x00077, 0x0, "Lego 24Z SCG Graphics"}, 
	{HPHW_FIO, 0x018, 0x00077, 0x0, "Lego 48Z SCG Graphics"}, 
	{HPHW_FIO, 0x019, 0x00077, 0x0, "Scorpio Sr Builtin Graphics"}, 
	{HPHW_FIO, 0x020, 0x00077, 0x0, "Scorpio 100 Builtin Graphics"}, 
	{HPHW_FIO, 0x021, 0x00077, 0x0, "Spectra 50 Builtin Graphics"}, 
	{HPHW_FIO, 0x022, 0x00077, 0x0, "Spectra 75 Builtin Graphics"}, 
	{HPHW_FIO, 0x023, 0x00077, 0x0, "Spectra 100 Builtin Graphics"}, 
	{HPHW_FIO, 0x024, 0x00077, 0x0, "Fast Pace SGC Graphics"}, 
	{HPHW_FIO, 0x006, 0x0007A, 0x0, "Bushmaster Audio"}, 
	{HPHW_FIO, 0x008, 0x0007A, 0x0, "Flounder Audio"}, 
	{HPHW_FIO, 0x004, 0x0007B, 0x0, "UL Optional Audio"}, 
	{HPHW_FIO, 0x007, 0x0007B, 0x0, "Scorpio Audio"}, 
	{HPHW_FIO, 0x00B, 0x0007B, 0x0, "Scorpio Jr. Audio"}, 
	{HPHW_FIO, 0x00C, 0x0007B, 0x0, "Strider-50 Audio"}, 
	{HPHW_FIO, 0x00D, 0x0007B, 0x0, "Strider-33 Audio"}, 
	{HPHW_FIO, 0x00E, 0x0007B, 0x0, "Trailways-50 Audio"}, 
	{HPHW_FIO, 0x00F, 0x0007B, 0x0, "Trailways-33 Audio"}, 
	{HPHW_FIO, 0x015, 0x0007B, 0x0, "KittyHawk GSY Core Audio"}, 
	{HPHW_FIO, 0x016, 0x0007B, 0x0, "Gecko Audio"}, 
	{HPHW_FIO, 0x019, 0x0007B, 0x0, "Scorpio Sr. Audio"}, 
	{HPHW_FIO, 0x01A, 0x0007B, 0x0, "Anole 64 Audio"}, 
	{HPHW_FIO, 0x01B, 0x0007B, 0x0, "Anole 100 Audio"}, 
	{HPHW_FIO, 0x01C, 0x0007B, 0x0, "Gecko 80 Audio"}, 
	{HPHW_FIO, 0x01D, 0x0007B, 0x0, "Gecko 100 Audio"}, 
	{HPHW_FIO, 0x01F, 0x0007B, 0x0, "SkyHawk 100/120 Audio"}, 
	{HPHW_FIO, 0x020, 0x0007B, 0x0, "Scorpio 100 Audio"}, 
	{HPHW_FIO, 0x021, 0x0007B, 0x0, "Spectra 50 Audio"}, 
	{HPHW_FIO, 0x022, 0x0007B, 0x0, "Spectra 75 Audio"}, 
	{HPHW_FIO, 0x023, 0x0007B, 0x0, "Spectra 100 Audio"}, 
	{HPHW_FIO, 0x028, 0x0007B, 0x0, "Mirage Jr Audio"}, 
	{HPHW_FIO, 0x029, 0x0007B, 0x0, "Mirage Audio"}, 
	{HPHW_FIO, 0x02A, 0x0007B, 0x0, "Electra Audio"}, 
	{HPHW_FIO, 0x02B, 0x0007B, 0x0, "Mirage 80 Audio"}, 
	{HPHW_FIO, 0x02C, 0x0007B, 0x0, "Mirage 100+ Audio"}, 
	{HPHW_FIO, 0x032, 0x0007B, 0x0, "Raven T' Audio"}, 
	{HPHW_FIO, 0x034, 0x0007B, 0x0, "SAIC L-80 Audio"}, 
	{HPHW_FIO, 0x035, 0x0007B, 0x0, "PCX-L2 712/132 Core Audio"}, 
	{HPHW_FIO, 0x036, 0x0007B, 0x0, "PCX-L2 712/160 Core Audio"}, 
	{HPHW_FIO, 0x03B, 0x0007B, 0x0, "Raven U/L2 Core Audio"}, 
	{HPHW_FIO, 0x03C, 0x0007B, 0x0, "Merlin 132 Core Audio"}, 
	{HPHW_FIO, 0x03D, 0x0007B, 0x0, "Merlin 160 Core Audio"}, 
	{HPHW_FIO, 0x03E, 0x0007B, 0x0, "Merlin+ 132 Core Audio"}, 
	{HPHW_FIO, 0x03F, 0x0007B, 0x0, "Merlin+ 180 Core Audio"}, 
	{HPHW_FIO, 0x044, 0x0007B, 0x0, "Mohawk Core Audio"}, 
	{HPHW_FIO, 0x046, 0x0007B, 0x0, "Rocky2 120 Core Audio"}, 
	{HPHW_FIO, 0x047, 0x0007B, 0x0, "Rocky2 150 Core Audio"}, 
	{HPHW_FIO, 0x04B, 0x0007B, 0x0, "Anole L2 132 Core Audio"}, 
	{HPHW_FIO, 0x04D, 0x0007B, 0x0, "Anole L2 165 Core Audio"}, 
	{HPHW_FIO, 0x04E, 0x0007B, 0x0, "Kiji L2 132 Core Audio"}, 
	{HPHW_FIO, 0x050, 0x0007B, 0x0, "Merlin Jr 132 Core Audio"}, 
	{HPHW_FIO, 0x051, 0x0007B, 0x0, "Firehawk Audio"}, 
	{HPHW_FIO, 0x056, 0x0007B, 0x0, "Raven+ w SE FWSCSI Core Audio"}, 
	{HPHW_FIO, 0x057, 0x0007B, 0x0, "Raven+ w Diff FWSCSI Core Audio"}, 
	{HPHW_FIO, 0x058, 0x0007B, 0x0, "FireHawk 200 Audio"}, 
	{HPHW_FIO, 0x05C, 0x0007B, 0x0, "SummitHawk 230 Core Audio"}, 
	{HPHW_FIO, 0x800, 0x0007B, 0x0, "Hitachi Tiny 64 Audio"}, 
	{HPHW_FIO, 0x801, 0x0007B, 0x0, "Hitachi Tiny 80 Audio"}, 
	{HPHW_FIO, 0x009, 0x0007C, 0x0, "Outfield FW SCSI"}, 
	{HPHW_FIO, 0x00A, 0x0007C, 0x0, "CoralII FW SCSI"}, 
	{HPHW_FIO, 0x026, 0x0007C, 0x0, "CoralII Jaguar FW SCSI"}, 
	{HPHW_FIO, 0x009, 0x0007D, 0x0, "Outfield FDDI"}, 
	{HPHW_FIO, 0x00A, 0x0007D, 0x0, "CoralII FDDI"}, 
	{HPHW_FIO, 0x026, 0x0007D, 0x0, "CoralII Jaguar FDDI"}, 
	{HPHW_FIO, 0x010, 0x0007E, 0x0, "Pace Audio"}, 
	{HPHW_FIO, 0x024, 0x0007E, 0x0, "Fast Pace Audio"}, 
	{HPHW_FIO, 0x009, 0x0007F, 0x0, "Outfield Audio"}, 
	{HPHW_FIO, 0x00A, 0x0007F, 0x0, "CoralII Audio"}, 
	{HPHW_FIO, 0x026, 0x0007F, 0x0, "CoralII Jaguar Audio"}, 
	{HPHW_FIO, 0x010, 0x00080, 0x0, "Pace Core HPIB"}, 
	{HPHW_FIO, 0x024, 0x00080, 0x0, "Fast Pace Core HPIB"}, 
	{HPHW_FIO, 0x015, 0x00082, 0x0, "KittyHawk GSY Core SCSI"},
	{HPHW_FIO, 0x016, 0x00082, 0x0, "Gecko Core SCSI"}, 
	{HPHW_FIO, 0x01A, 0x00082, 0x0, "Anole 64 Core SCSI"}, 
	{HPHW_FIO, 0x01B, 0x00082, 0x0, "Anole 100 Core SCSI"}, 
	{HPHW_FIO, 0x01C, 0x00082, 0x0, "Gecko 80 Core SCSI"}, 
	{HPHW_FIO, 0x01D, 0x00082, 0x0, "Gecko 100 Core SCSI"}, 
	{HPHW_FIO, 0x01F, 0x00082, 0x0, "SkyHawk 100/120 Core SCSI"}, 
	{HPHW_FIO, 0x027, 0x00082, 0x0, "Piranha 100 Core SCSI"}, 
	{HPHW_FIO, 0x028, 0x00082, 0x0, "Mirage Jr Core SCSI"}, 
	{HPHW_FIO, 0x029, 0x00082, 0x0, "Mirage Core SCSI"}, 
	{HPHW_FIO, 0x02A, 0x00082, 0x0, "Electra Core SCSI"}, 
	{HPHW_FIO, 0x02B, 0x00082, 0x0, "Mirage 80 Core SCSI"}, 
	{HPHW_FIO, 0x02C, 0x00082, 0x0, "Mirage 100+ Core SCSI"}, 
	{HPHW_FIO, 0x02E, 0x00082, 0x0, "UL 350 Core SCSI"}, 
	{HPHW_FIO, 0x02F, 0x00082, 0x0, "UL 550 Core SCSI"}, 
	{HPHW_FIO, 0x032, 0x00082, 0x0, "Raven T' Core SCSI"}, 
	{HPHW_FIO, 0x033, 0x00082, 0x0, "Anole T Core SCSI"}, 
	{HPHW_FIO, 0x034, 0x00082, 0x0, "SAIC L-80 Core SCSI"}, 
	{HPHW_FIO, 0x035, 0x00082, 0x0, "PCX-L2 712/132 Core SCSI"}, 
	{HPHW_FIO, 0x036, 0x00082, 0x0, "PCX-L2 712/160 Core SCSI"}, 
	{HPHW_FIO, 0x03B, 0x00082, 0x0, "Raven U/L2 Core SCSI"}, 
	{HPHW_FIO, 0x03C, 0x00082, 0x0, "Merlin 132 Core SCSI"}, 
	{HPHW_FIO, 0x03D, 0x00082, 0x0, "Merlin 160 Core SCSI"}, 
	{HPHW_FIO, 0x03E, 0x00082, 0x0, "Merlin+ 132 Core SCSI"}, 
	{HPHW_FIO, 0x03F, 0x00082, 0x0, "Merlin+ 180 Core SCSI"}, 
	{HPHW_FIO, 0x044, 0x00082, 0x0, "Mohawk Core SCSI"}, 
	{HPHW_FIO, 0x045, 0x00082, 0x0, "Rocky1 Core SCSI"}, 
	{HPHW_FIO, 0x046, 0x00082, 0x0, "Rocky2 120 Core SCSI"}, 
	{HPHW_FIO, 0x047, 0x00082, 0x0, "Rocky2 150 Core SCSI"}, 
	{HPHW_FIO, 0x04B, 0x00082, 0x0, "Anole L2 132 Core SCSI"}, 
	{HPHW_FIO, 0x04D, 0x00082, 0x0, "Anole L2 165 Core SCSI"}, 
	{HPHW_FIO, 0x04E, 0x00082, 0x0, "Kiji L2 132 Core SCSI"}, 
	{HPHW_FIO, 0x050, 0x00082, 0x0, "Merlin Jr 132 Core SCSI"}, 
	{HPHW_FIO, 0x051, 0x00082, 0x0, "Firehawk Core SCSI"}, 
	{HPHW_FIO, 0x056, 0x00082, 0x0, "Raven+ w SE FWSCSI Core SCSI"}, 
	{HPHW_FIO, 0x057, 0x00082, 0x0, "Raven+ w Diff FWSCSI Core SCSI"}, 
	{HPHW_FIO, 0x058, 0x00082, 0x0, "FireHawk 200 Core SCSI"}, 
	{HPHW_FIO, 0x05C, 0x00082, 0x0, "SummitHawk 230 Core SCSI"}, 
	{HPHW_FIO, 0x05E, 0x00082, 0x0, "Staccato 132 Core SCSI"}, 
	{HPHW_FIO, 0x05F, 0x00082, 0x0, "Staccato 180 Core SCSI"}, 
	{HPHW_FIO, 0x800, 0x00082, 0x0, "Hitachi Tiny 64 Core SCSI"}, 
	{HPHW_FIO, 0x801, 0x00082, 0x0, "Hitachi Tiny 80 Core SCSI"}, 
	{HPHW_FIO, 0x016, 0x00083, 0x0, "Gecko Core PC Floppy"}, 
	{HPHW_FIO, 0x01C, 0x00083, 0x0, "Gecko 80 Core PC Floppy"}, 
	{HPHW_FIO, 0x01D, 0x00083, 0x0, "Gecko 100 Core PC Floppy"}, 
	{HPHW_FIO, 0x051, 0x00083, 0x0, "Firehawk Core PC Floppy"}, 
	{HPHW_FIO, 0x058, 0x00083, 0x0, "FireHawk 200 Core PC Floppy"}, 
	{HPHW_FIO, 0x027, 0x00083, 0x0, "Piranha 100 Core PC Floppy"}, 
	{HPHW_FIO, 0x028, 0x00083, 0x0, "Mirage Jr Core PC Floppy"}, 
	{HPHW_FIO, 0x029, 0x00083, 0x0, "Mirage Core PC Floppy"}, 
	{HPHW_FIO, 0x02A, 0x00083, 0x0, "Electra Core PC Floppy"}, 
	{HPHW_FIO, 0x02B, 0x00083, 0x0, "Mirage 80 Core PC Floppy"}, 
	{HPHW_FIO, 0x02C, 0x00083, 0x0, "Mirage 100+ Core PC Floppy"}, 
	{HPHW_FIO, 0x02E, 0x00083, 0x0, "UL 350 Core PC Floppy"}, 
	{HPHW_FIO, 0x02F, 0x00083, 0x0, "UL 550 Core PC Floppy"}, 
	{HPHW_FIO, 0x032, 0x00083, 0x0, "Raven T' Core PC Floppy"}, 
	{HPHW_FIO, 0x034, 0x00083, 0x0, "SAIC L-80 Core PC Floppy"}, 
	{HPHW_FIO, 0x035, 0x00083, 0x0, "PCX-L2 712/132 Core Floppy"}, 
	{HPHW_FIO, 0x036, 0x00083, 0x0, "PCX-L2 712/160 Core Floppy"}, 
	{HPHW_FIO, 0x03B, 0x00083, 0x0, "Raven U/L2 Core PC Floppy"}, 
	{HPHW_FIO, 0x03C, 0x00083, 0x0, "Merlin 132 Core PC Floppy"}, 
	{HPHW_FIO, 0x03D, 0x00083, 0x0, "Merlin 160 Core PC Floppy"}, 
	{HPHW_FIO, 0x03E, 0x00083, 0x0, "Merlin+ 132 Core PC Floppy"}, 
	{HPHW_FIO, 0x03F, 0x00083, 0x0, "Merlin+ 180 Core PC Floppy"}, 
	{HPHW_FIO, 0x045, 0x00083, 0x0, "Rocky1 Core PC Floppy"}, 
	{HPHW_FIO, 0x046, 0x00083, 0x0, "Rocky2 120 Core PC Floppy"}, 
	{HPHW_FIO, 0x047, 0x00083, 0x0, "Rocky2 150 Core PC Floppy"}, 
	{HPHW_FIO, 0x04E, 0x00083, 0x0, "Kiji L2 132 Core PC Floppy"}, 
	{HPHW_FIO, 0x050, 0x00083, 0x0, "Merlin Jr 132 Core PC Floppy"}, 
	{HPHW_FIO, 0x056, 0x00083, 0x0, "Raven+ w SE FWSCSI Core PC Floppy"}, 
	{HPHW_FIO, 0x057, 0x00083, 0x0, "Raven+ w Diff FWSCSI Core PC Floppy"}, 
	{HPHW_FIO, 0x800, 0x00083, 0x0, "Hitachi Tiny 64 Core PC Floppy"}, 
	{HPHW_FIO, 0x801, 0x00083, 0x0, "Hitachi Tiny 80 Core PC Floppy"},
	{HPHW_FIO, 0x015, 0x00084, 0x0, "KittyHawk GSY Core PS/2 Port"}, 
	{HPHW_FIO, 0x016, 0x00084, 0x0, "Gecko Core PS/2 Port"}, 
	{HPHW_FIO, 0x018, 0x00084, 0x0, "Gecko Optional PS/2 Port"}, 
	{HPHW_FIO, 0x01A, 0x00084, 0x0, "Anole 64 Core PS/2 Port"}, 
	{HPHW_FIO, 0x01B, 0x00084, 0x0, "Anole 100 Core PS/2 Port"}, 
	{HPHW_FIO, 0x01C, 0x00084, 0x0, "Gecko 80 Core PS/2 Port"}, 
	{HPHW_FIO, 0x01D, 0x00084, 0x0, "Gecko 100 Core PS/2 Port"}, 
	{HPHW_FIO, 0x01F, 0x00084, 0x0, "SkyHawk 100/120 Core PS/2 Port"}, 
	{HPHW_FIO, 0x027, 0x00084, 0x0, "Piranha 100 Core PS/2 Port"}, 
	{HPHW_FIO, 0x028, 0x00084, 0x0, "Mirage Jr Core PS/2 Port"}, 
	{HPHW_FIO, 0x029, 0x00084, 0x0, "Mirage Core PS/2 Port"}, 
	{HPHW_FIO, 0x02A, 0x00084, 0x0, "Electra Core PS/2 Port"}, 
	{HPHW_FIO, 0x02B, 0x00084, 0x0, "Mirage 80 Core PS/2 Port"}, 
	{HPHW_FIO, 0x02C, 0x00084, 0x0, "Mirage 100+ Core PS/2 Port"}, 
	{HPHW_FIO, 0x02E, 0x00084, 0x0, "UL 350 Core PS/2 Port"}, 
	{HPHW_FIO, 0x02F, 0x00084, 0x0, "UL 550 Core PS/2 Port"}, 
	{HPHW_FIO, 0x032, 0x00084, 0x0, "Raven T' Core PS/2 Port"}, 
	{HPHW_FIO, 0x033, 0x00084, 0x0, "Anole T Core PS/2 Port"}, 
	{HPHW_FIO, 0x034, 0x00084, 0x0, "SAIC L-80 Core PS/2 Port"}, 
	{HPHW_FIO, 0x035, 0x00084, 0x0, "PCX-L2 712/132 Core PS/2 Port"}, 
	{HPHW_FIO, 0x036, 0x00084, 0x0, "PCX-L2 712/160 Core PS/2 Port"}, 
	{HPHW_FIO, 0x03B, 0x00084, 0x0, "Raven U/L2 Core PS/2 Port"}, 
	{HPHW_FIO, 0x03C, 0x00084, 0x0, "Merlin 132 Core PS/2 Port"}, 
	{HPHW_FIO, 0x03D, 0x00084, 0x0, "Merlin 160 Core PS/2 Port"}, 
	{HPHW_FIO, 0x03E, 0x00084, 0x0, "Merlin+ 132 Core PS/2 Port"}, 
	{HPHW_FIO, 0x03F, 0x00084, 0x0, "Merlin+ 180 Core PS/2 Port"}, 
	{HPHW_FIO, 0x044, 0x00084, 0x0, "Mohawk Core PS/2 Port"}, 
	{HPHW_FIO, 0x045, 0x00084, 0x0, "Rocky1 Core PS/2 Port"}, 
	{HPHW_FIO, 0x046, 0x00084, 0x0, "Rocky2 120 Core PS/2 Port"}, 
	{HPHW_FIO, 0x047, 0x00084, 0x0, "Rocky2 150 Core PS/2 Port"}, 
	{HPHW_FIO, 0x048, 0x00084, 0x0, "Rocky2 120 Dino PS/2 Port"}, 
	{HPHW_FIO, 0x049, 0x00084, 0x0, "Rocky2 150 Dino PS/2 Port"}, 
	{HPHW_FIO, 0x04B, 0x00084, 0x0, "Anole L2 132 Core PS/2 Port"}, 
	{HPHW_FIO, 0x04D, 0x00084, 0x0, "Anole L2 165 Core PS/2 Port"}, 
	{HPHW_FIO, 0x04E, 0x00084, 0x0, "Kiji L2 132 Core PS/2 Port"}, 
	{HPHW_FIO, 0x050, 0x00084, 0x0, "Merlin Jr 132 Core PS/2 Port"}, 
	{HPHW_FIO, 0x051, 0x00084, 0x0, "Firehawk Core PS/2 Port"}, 
	{HPHW_FIO, 0x056, 0x00084, 0x0, "Raven+ w SE FWSCSI Core PS/2 Port"}, 
	{HPHW_FIO, 0x057, 0x00084, 0x0, "Raven+ w Diff FWSCSI Core PS/2 Port"}, 
	{HPHW_FIO, 0x058, 0x00084, 0x0, "FireHawk 200 Core PS/2 Port"}, 
	{HPHW_FIO, 0x05C, 0x00084, 0x0, "SummitHawk 230 Core PS/2 Port"}, 
	{HPHW_FIO, 0x800, 0x00084, 0x0, "Hitachi Tiny 64 Core PS/2 Port"}, 
	{HPHW_FIO, 0x801, 0x00084, 0x0, "Hitachi Tiny 80 Core PS/2 Port"}, 
	{HPHW_FIO, 0x004, 0x00085, 0x0, "Solo GSC Optional Graphics"}, 
	{HPHW_FIO, 0x005, 0x00085, 0x0, "Duet GSC Optional Graphics"}, 
	{HPHW_FIO, 0x008, 0x00085, 0x0, "Anole Artist Optional Graphics"}, 
	{HPHW_FIO, 0x010, 0x00085, 0x0, "Mirage 80 GSC Builtin Graphics"}, 
	{HPHW_FIO, 0x011, 0x00085, 0x0, "Mirage 100+ GSC Builtin Graphics"}, 
	{HPHW_FIO, 0x012, 0x00085, 0x0, "Mirage Jr GSC Builtin Graphics"}, 
	{HPHW_FIO, 0x013, 0x00085, 0x0, "Mirage GSC Builtin Graphics"}, 
	{HPHW_FIO, 0x014, 0x00085, 0x0, "Electra GSC Builtin Graphics"}, 
	{HPHW_FIO, 0x016, 0x00085, 0x0, "Gecko GSC Core Graphics"}, 
	{HPHW_FIO, 0x017, 0x00085, 0x0, "Gecko GSC Optional Graphics"}, 
	{HPHW_FIO, 0x01A, 0x00085, 0x0, "Anole 64 Artist Builtin Graphics"}, 
	{HPHW_FIO, 0x01B, 0x00085, 0x0, "Anole 100 Artist Builtin Graphics"}, 
	{HPHW_FIO, 0x01C, 0x00085, 0x0, "Gecko 80 GSC Core Graphics"}, 
	{HPHW_FIO, 0x01D, 0x00085, 0x0, "Gecko 100 GSC Core Graphics"}, 
	{HPHW_FIO, 0x032, 0x00085, 0x0, "Raven T' GSC Core Graphics"}, 
	{HPHW_FIO, 0x033, 0x00085, 0x0, "Anole T Artist Builtin Graphics"}, 
	{HPHW_FIO, 0x034, 0x00085, 0x0, "SAIC L-80 GSC Core Graphics"}, 
	{HPHW_FIO, 0x035, 0x00085, 0x0, "PCX-L2 712/132 Core Graphics"}, 
	{HPHW_FIO, 0x036, 0x00085, 0x0, "PCX-L2 712/160 Core Graphics"}, 
	{HPHW_FIO, 0x03B, 0x00085, 0x0, "Raven U/L2 Core Graphics"}, 
	{HPHW_FIO, 0x03C, 0x00085, 0x0, "Merlin 132 Core Graphics"}, 
	{HPHW_FIO, 0x03D, 0x00085, 0x0, "Merlin 160 Core Graphics"}, 
	{HPHW_FIO, 0x03E, 0x00085, 0x0, "Merlin+ 132 Core Graphics"}, 
	{HPHW_FIO, 0x03F, 0x00085, 0x0, "Merlin+ 180 Core Graphics"}, 
	{HPHW_FIO, 0x045, 0x00085, 0x0, "Rocky1 Core Graphics"}, 
	{HPHW_FIO, 0x046, 0x00085, 0x0, "Rocky2 120 Core Graphics"}, 
	{HPHW_FIO, 0x047, 0x00085, 0x0, "Rocky2 150 Core Graphics"}, 
	{HPHW_FIO, 0x04B, 0x00085, 0x0, "Anole L2 132 Core Graphics"}, 
	{HPHW_FIO, 0x04D, 0x00085, 0x0, "Anole L2 165 Core Graphics"}, 
	{HPHW_FIO, 0x04E, 0x00085, 0x0, "Kiji L2 132 Core Graphics"}, 
	{HPHW_FIO, 0x050, 0x00085, 0x0, "Merlin Jr 132 Core Graphics"}, 
	{HPHW_FIO, 0x056, 0x00085, 0x0, "Raven+ w SE FWSCSI Core Graphics"}, 
	{HPHW_FIO, 0x057, 0x00085, 0x0, "Raven+ w Diff FWSCSI Core Graphics"}, 
	{HPHW_FIO, 0x800, 0x00085, 0x0, "Hitachi Tiny 64 Core Graphics"}, 
	{HPHW_FIO, 0x801, 0x00085, 0x0, "Hitachi Tiny 80 Core Graphics"}, 
	{HPHW_FIO, 0x004, 0x00086, 0x0, "GSC IBM Token Ring"}, 
	{HPHW_FIO, 0x015, 0x00087, 0x0, "Gecko Optional ISDN"}, 
	{HPHW_FIO, 0x016, 0x00087, 0x0, "Gecko Core ISDN"}, 
	{HPHW_FIO, 0x01C, 0x00087, 0x0, "Gecko 80 Core ISDN"}, 
	{HPHW_FIO, 0x01D, 0x00087, 0x0, "Gecko 100 Core ISDN"}, 
	{HPHW_FIO, 0x010, 0x00088, 0x0, "Pace VME Networking"}, 
	{HPHW_FIO, 0x011, 0x00088, 0x0, "Sidewinder VME Networking"}, 
	{HPHW_FIO, 0x01A, 0x00088, 0x0, "Anole 64 VME Networking"}, 
	{HPHW_FIO, 0x01B, 0x00088, 0x0, "Anole 100 VME Networking"}, 
	{HPHW_FIO, 0x024, 0x00088, 0x0, "Fast Pace VME Networking"}, 
	{HPHW_FIO, 0x034, 0x00088, 0x0, "Anole T VME Networking"}, 
	{HPHW_FIO, 0x04A, 0x00088, 0x0, "Anole L2 132 VME Networking"}, 
	{HPHW_FIO, 0x04C, 0x00088, 0x0, "Anole L2 165 VME Networking"}, 
	{HPHW_FIO, 0x011, 0x0008A, 0x0, "WB-96 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x012, 0x0008A, 0x0, "Orville Core LAN (802.3)"}, 
	{HPHW_FIO, 0x013, 0x0008A, 0x0, "Wilbur Core LAN (802.3)"}, 
	{HPHW_FIO, 0x014, 0x0008A, 0x0, "WB-80 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x015, 0x0008A, 0x0, "KittyHawk GSY Core LAN (802.3)"}, 
	{HPHW_FIO, 0x016, 0x0008A, 0x0, "Gecko Core LAN (802.3)"}, 
	{HPHW_FIO, 0x018, 0x0008A, 0x0, "Gecko Optional LAN (802.3)"}, 
	{HPHW_FIO, 0x01A, 0x0008A, 0x0, "Anole 64 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x01B, 0x0008A, 0x0, "Anole 100 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x01C, 0x0008A, 0x0, "Gecko 80 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x01D, 0x0008A, 0x0, "Gecko 100 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x01F, 0x0008A, 0x0, "SkyHawk 100/120 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x027, 0x0008A, 0x0, "Piranha 100 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x028, 0x0008A, 0x0, "Mirage Jr Core LAN (802.3)"}, 
	{HPHW_FIO, 0x029, 0x0008A, 0x0, "Mirage Core LAN (802.3)"}, 
	{HPHW_FIO, 0x02A, 0x0008A, 0x0, "Electra Core LAN (802.3)"}, 
	{HPHW_FIO, 0x02B, 0x0008A, 0x0, "Mirage 80 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x02C, 0x0008A, 0x0, "Mirage 100+ Core LAN (802.3)"}, 
	{HPHW_FIO, 0x02E, 0x0008A, 0x0, "UL 350 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x02F, 0x0008A, 0x0, "UL 350 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x032, 0x0008A, 0x0, "Raven T' Core LAN (802.3)"}, 
	{HPHW_FIO, 0x033, 0x0008A, 0x0, "Anole T Core LAN (802.3)"}, 
	{HPHW_FIO, 0x034, 0x0008A, 0x0, "SAIC L-80 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x035, 0x0008A, 0x0, "PCX-L2 712/132 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x036, 0x0008A, 0x0, "PCX-L2 712/160 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x03B, 0x0008A, 0x0, "Raven U/L2 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x03C, 0x0008A, 0x0, "Merlin 132 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x03D, 0x0008A, 0x0, "Merlin 160 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x044, 0x0008A, 0x0, "Mohawk Core LAN (802.3)"}, 
	{HPHW_FIO, 0x045, 0x0008A, 0x0, "Rocky1 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x046, 0x0008A, 0x0, "Rocky2 120 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x047, 0x0008A, 0x0, "Rocky2 150 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x04B, 0x0008A, 0x0, "Anole L2 132 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x04D, 0x0008A, 0x0, "Anole L2 165 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x04E, 0x0008A, 0x0, "Kiji L2 132 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x050, 0x0008A, 0x0, "Merlin Jr 132 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x058, 0x0008A, 0x0, "FireHawk 200 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x800, 0x0008A, 0x0, "Hitachi Tiny 64 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x801, 0x0008A, 0x0, "Hitachi Tiny 80 Core LAN (802.3)"}, 
	{HPHW_FIO, 0x004, 0x0008C, 0x0, "SkyHawk 100/120 Wax RS-232"}, 
	{HPHW_FIO, 0x005, 0x0008C, 0x0, "SAIC L-80 Wax RS-232"}, 
	{HPHW_FIO, 0x006, 0x0008C, 0x0, "Raven U/L2 Dino RS-232"}, 
	{HPHW_FIO, 0x007, 0x0008C, 0x0, "Dino RS-232"}, 
	{HPHW_FIO, 0x008, 0x0008C, 0x0, "Merlin 132 Dino RS-232"}, 
	{HPHW_FIO, 0x009, 0x0008C, 0x0, "Merlin 160 Dino RS-232"}, 
	{HPHW_FIO, 0x00A, 0x0008C, 0x0, "Merlin Jr 132 Dino RS-232"}, 
	{HPHW_FIO, 0x010, 0x0008C, 0x0, "Mirage 80 Wax RS-232"}, 
	{HPHW_FIO, 0x011, 0x0008C, 0x0, "Mirage 100+ Wax RS-232"}, 
	{HPHW_FIO, 0x012, 0x0008C, 0x0, "Mirage Jr Wax RS-232"}, 
	{HPHW_FIO, 0x013, 0x0008C, 0x0, "Mirage Wax RS-232"}, 
	{HPHW_FIO, 0x014, 0x0008C, 0x0, "Electra Wax RS-232"}, 
	{HPHW_FIO, 0x015, 0x0008C, 0x0, "KittyHawk GSY Core RS-232"}, 
	{HPHW_FIO, 0x016, 0x0008C, 0x0, "Gecko Core RS-232"}, 
	{HPHW_FIO, 0x017, 0x0008C, 0x0, "Raven Backplane RS-232"}, 
	{HPHW_FIO, 0x018, 0x0008C, 0x0, "Gecko Optional RS-232"}, 
	{HPHW_FIO, 0x019, 0x0008C, 0x0, "Merlin+ 180 Dino RS-232"}, 
	{HPHW_FIO, 0x01A, 0x0008C, 0x0, "Anole 64 Core RS-232"}, 
	{HPHW_FIO, 0x01B, 0x0008C, 0x0, "Anole 100 Core RS-232"}, 
	{HPHW_FIO, 0x01C, 0x0008C, 0x0, "Gecko 80 Core RS-232"}, 
	{HPHW_FIO, 0x01D, 0x0008C, 0x0, "Gecko 100 Core RS-232"}, 
	{HPHW_FIO, 0x01E, 0x0008C, 0x0, "Raven T' Wax RS-232"}, 
	{HPHW_FIO, 0x01F, 0x0008C, 0x0, "SkyHawk 100/120 Core RS-232"}, 
	{HPHW_FIO, 0x020, 0x0008C, 0x0, "Anole 64 Timi RS-232"}, 
	{HPHW_FIO, 0x021, 0x0008C, 0x0, "Anole 100 Timi RS-232"}, 
	{HPHW_FIO, 0x022, 0x0008C, 0x0, "Merlin+ 132 Dino RS-232"}, 
	{HPHW_FIO, 0x023, 0x0008C, 0x0, "Rocky1 Wax RS-232"}, 
	{HPHW_FIO, 0x025, 0x0008C, 0x0, "Armyknife Optional RS-232"}, 
	{HPHW_FIO, 0x026, 0x0008C, 0x0, "Piranha 100 Wax RS-232"}, 
	{HPHW_FIO, 0x027, 0x0008C, 0x0, "Piranha 100 Core RS-232"}, 
	{HPHW_FIO, 0x028, 0x0008C, 0x0, "Mirage Jr Core RS-232"}, 
	{HPHW_FIO, 0x029, 0x0008C, 0x0, "Mirage Core RS-232"}, 
	{HPHW_FIO, 0x02A, 0x0008C, 0x0, "Electra Core RS-232"}, 
	{HPHW_FIO, 0x02B, 0x0008C, 0x0, "Mirage 80 Core RS-232"}, 
	{HPHW_FIO, 0x02C, 0x0008C, 0x0, "Mirage 100+ Core RS-232"}, 
	{HPHW_FIO, 0x02E, 0x0008C, 0x0, "UL 350 Lasi Core RS-232"}, 
	{HPHW_FIO, 0x02F, 0x0008C, 0x0, "UL 550 Lasi Core RS-232"}, 
	{HPHW_FIO, 0x030, 0x0008C, 0x0, "UL 350 Wax Core RS-232"}, 
	{HPHW_FIO, 0x031, 0x0008C, 0x0, "UL 550 Wax Core RS-232"}, 
	{HPHW_FIO, 0x032, 0x0008C, 0x0, "Raven T' Lasi Core RS-232"}, 
	{HPHW_FIO, 0x033, 0x0008C, 0x0, "Anole T Core RS-232"}, 
	{HPHW_FIO, 0x034, 0x0008C, 0x0, "SAIC L-80 Core RS-232"}, 
	{HPHW_FIO, 0x035, 0x0008C, 0x0, "PCX-L2 712/132 Core RS-232"}, 
	{HPHW_FIO, 0x036, 0x0008C, 0x0, "PCX-L2 712/160 Core RS-232"}, 
	{HPHW_FIO, 0x03A, 0x0008C, 0x0, "Merlin+ Wax RS-232"}, 
	{HPHW_FIO, 0x03B, 0x0008C, 0x0, "Raven U/L2 Core RS-232"}, 
	{HPHW_FIO, 0x03C, 0x0008C, 0x0, "Merlin 132 Core RS-232"}, 
	{HPHW_FIO, 0x03D, 0x0008C, 0x0, "Merlin 160 Core RS-232"}, 
	{HPHW_FIO, 0x03E, 0x0008C, 0x0, "Merlin+ 132 Core RS-232"}, 
	{HPHW_FIO, 0x03F, 0x0008C, 0x0, "Merlin+ 180 Core RS-232"}, 
	{HPHW_FIO, 0x040, 0x0008C, 0x0, "Merlin 132 Wax RS-232"}, 
	{HPHW_FIO, 0x041, 0x0008C, 0x0, "Merlin 160 Wax RS-232"}, 
	{HPHW_FIO, 0x043, 0x0008C, 0x0, "Merlin 132/160 Wax RS-232"}, 
	{HPHW_FIO, 0x044, 0x0008C, 0x0, "Mohawk Core RS-232"}, 
	{HPHW_FIO, 0x045, 0x0008C, 0x0, "Rocky1 Core RS-232"}, 
	{HPHW_FIO, 0x046, 0x0008C, 0x0, "Rocky2 120 Core RS-232"}, 
	{HPHW_FIO, 0x047, 0x0008C, 0x0, "Rocky2 150 Core RS-232"}, 
	{HPHW_FIO, 0x048, 0x0008C, 0x0, "Rocky2 120 Dino RS-232"}, 
	{HPHW_FIO, 0x049, 0x0008C, 0x0, "Rocky2 150 Dino RS-232"}, 
	{HPHW_FIO, 0x04A, 0x0008C, 0x0, "Anole L2 132 TIMI RS-232"}, 
	{HPHW_FIO, 0x04B, 0x0008C, 0x0, "Anole L2 l32 Core RS-232"}, 
	{HPHW_FIO, 0x04C, 0x0008D, 0x0, "Anole L2 165 TIMI RS-232"}, 
	{HPHW_FIO, 0x04D, 0x0008C, 0x0, "Anole L2 165 Core RS-232"}, 
	{HPHW_FIO, 0x04E, 0x0008C, 0x0, "Kiji L2 132 Core RS-232"}, 
	{HPHW_FIO, 0x04F, 0x0008C, 0x0, "Kiji L2 132 Dino RS-232"}, 
	{HPHW_FIO, 0x050, 0x0008C, 0x0, "Merlin Jr 132 Core RS-232"}, 
	{HPHW_FIO, 0x051, 0x0008C, 0x0, "Firehawk Core RS-232"}, 
	{HPHW_FIO, 0x052, 0x0008C, 0x0, "Raven+ Hi Power Backplane w EISA RS-232"}, 
	{HPHW_FIO, 0x053, 0x0008C, 0x0, "Raven+ Hi Power Backplane w/o EISA RS-232"}, 
	{HPHW_FIO, 0x054, 0x0008C, 0x0, "Raven+ Lo Power Backplane w EISA RS-232"}, 
	{HPHW_FIO, 0x055, 0x0008C, 0x0, "Raven+ Lo Power Backplane w/o EISA RS-232"}, 
	{HPHW_FIO, 0x056, 0x0008C, 0x0, "Raven+ w SE FWSCSI Core RS-232"}, 
	{HPHW_FIO, 0x057, 0x0008C, 0x0, "Raven+ w Diff FWSCSI Core RS-232"}, 
	{HPHW_FIO, 0x058, 0x0008C, 0x0, "FireHawk 200 Core RS-232"}, 
	{HPHW_FIO, 0x059, 0x0008C, 0x0, "FireHawk 200 Wax RS-232"}, 
	{HPHW_FIO, 0x05A, 0x0008C, 0x0, "Raven+ L2 Backplane w EISA RS-232"}, 
	{HPHW_FIO, 0x05B, 0x0008C, 0x0, "Raven+ L2 Backplane w/o EISA RS-232"}, 
	{HPHW_FIO, 0x05D, 0x0008C, 0x0, "SummitHawk Dino RS-232"}, 
	{HPHW_FIO, 0x05E, 0x0008C, 0x0, "Staccato 132 Core LAN RS-232"}, 
	{HPHW_FIO, 0x05F, 0x0008C, 0x0, "Staccato 180 Core LAN RS-232"}, 
	{HPHW_FIO, 0x800, 0x0008C, 0x0, "Hitachi Tiny 64 Core RS-232"}, 
	{HPHW_FIO, 0x801, 0x0008C, 0x0, "Hitachi Tiny 80 Core RS-232"}, 
	{HPHW_FIO, 0x015, 0x0008D, 0x0, "Gecko Optional RJ-16"}, 
	{HPHW_FIO, 0x016, 0x0008D, 0x0, "Gecko Core RJ-16"}, 
	{HPHW_FIO, 0x01C, 0x0008D, 0x0, "Gecko 80 Core RJ-16"}, 
	{HPHW_FIO, 0x01D, 0x0008D, 0x0, "Gecko 100 Core RJ-16"}, 
	{HPHW_FIO, 0x004, 0x0008F, 0x0, "Anole Boot Rom"}, 
	{HPHW_FIO, 0x005, 0x0008F, 0x0, "Rocky1 Boot Rom"}, 
	{HPHW_FIO, 0x006, 0x0008F, 0x0, "Rocky2 120 Boot Rom"}, 
	{HPHW_FIO, 0x007, 0x0008F, 0x0, "Rocky2 150 Boot Rom"}, 
	{HPHW_FIO, 0x01B, 0x0008F, 0x0, "Anole 100 Boot Rom"}, 
	{HPHW_FIO, 0x006, 0x00096, 0x0, "Raven U/L2 Dino PS/2 Port"}, 
	{HPHW_FIO, 0x007, 0x00096, 0x0, "Dino PS/2 Port"}, 
	{HPHW_FIO, 0x008, 0x00096, 0x0, "Merlin 132 Dino PS/2 Port"}, 
	{HPHW_FIO, 0x009, 0x00096, 0x0, "Merlin 160 Dino PS/2 Port"}, 
	{HPHW_FIO, 0x00A, 0x00096, 0x0, "Merlin Jr 132 Dino PS/2 Port"}, 
	{HPHW_FIO, 0x019, 0x00096, 0x0, "Merlin+ 180 Dino PS/2 Port"}, 
	{HPHW_FIO, 0x022, 0x00096, 0x0, "Merlin+ 132 Dino PS/2 Port"}, 
	{HPHW_FIO, 0x004, 0x00097, 0x0, "Cascade EISA 100VG LAN"}, 
	{HPHW_FIO, 0x023, 0x00099, 0x0, "Rocky1 Wax HPIB"}, 
	{HPHW_FIO, 0x048, 0x00099, 0x0, "Rocky2 120 Clark/Dino HPIB"}, 
	{HPHW_FIO, 0x049, 0x00099, 0x0, "Rocky2 150 Clark/Dino HPIB"}, 
	{HPHW_FIO, 0x004, 0x000A1, 0x0, "SPP2000 Console TTY"}, 
	{HPHW_FIO, 0x004, 0x000A2, 0x0, "Forte Core PCI 10/100BT LAN"}, 
	{HPHW_FIO, 0x005, 0x000A2, 0x0, "AllegroLow PCI 10/100BT LAN"}, 
	{HPHW_FIO, 0x006, 0x000A2, 0x0, "AllegroHIgh Core PCI 10/100BT LAN"}, 
	{HPHW_FIO, 0x007, 0x000A2, 0x0, "PCI Plug-in LAN"}, 
	{HPHW_FIO, 0x00A, 0x000A2, 0x0, "Lego 360 Core PCI 10/100BT LAN"}, 
	{HPHW_FIO, 0x03E, 0x000A2, 0x0, "Merlin+ 132 Core PCI LAN"}, 
	{HPHW_FIO, 0x03F, 0x000A2, 0x0, "Merlin+ 180 Core PCI LAN"}, 
	{HPHW_FIO, 0x056, 0x000A2, 0x0, "Raven+ w SE FWSCSI Core PCI LAN"}, 
	{HPHW_FIO, 0x057, 0x000A2, 0x0, "Raven+ w Diff FWSCSI Core PCI LAN"}, 
	{HPHW_FIO, 0x05E, 0x000A2, 0x0, "Staccato 132 PCI LAN"}, 
	{HPHW_FIO, 0x05F, 0x000A2, 0x0, "Staccato 180 PCI LAN"}, 
	{HPHW_FIO, 0x004, 0x000A3, 0x0, "Forte Core PCI LVD Ultra2 SCSI"}, 
	{HPHW_FIO, 0x004, 0x000A3, 0x0, "Forte Core PCI SE UltraSCSI"}, 
	{HPHW_FIO, 0x004, 0x000A3, 0x0, "Forte Core PCI IDE/ATAPI CD-ROM"}, 
	{HPHW_FIO, 0x005, 0x000A3, 0x0, "AllegroLow Core PCI LVD Ultra2 SCSI"}, 
	{HPHW_FIO, 0x005, 0x000A3, 0x0, "AllegroLow Core PCI IDE/ATAPI CD-ROM"}, 
	{HPHW_FIO, 0x006, 0x000A3, 0x0, "AllegroHigh Core PCI LVD Ultra2 SCSI"}, 
	{HPHW_FIO, 0x006, 0x000A3, 0x0, "AllegroHigh Core PCI IDE/ATAPI CD-ROM"}, 
	{HPHW_FIO, 0x007, 0x000A3, 0x0, "PCI Plug-in Disk"}, 
	{HPHW_FIO, 0x008, 0x000A3, 0x0, "A5158A S FC Tachlite HBA"}, 
	{HPHW_FIO, 0x009, 0x000A3, 0x0, "A5157A D FC HBA"}, 
	{HPHW_FIO, 0x00A, 0x000A3, 0x0, "Lego 360 Core PCI LVD Ultra2 SCSI"}, 
	{HPHW_FIO, 0x00A, 0x000A3, 0x0, "Lego 360 Core PCI NSE UltraSCSI"}, 
	{HPHW_FIO, 0x00A, 0x000A3, 0x0, "Lego 360 Core PCI WSE UltraSCSI"}, 
	{HPHW_FIO, 0x00A, 0x000A3, 0x0, "Lego 360 Core PCI IDE/ATAPI CD-ROM"}, 
	{HPHW_FIO, 0x03E, 0x000A3, 0x0, "Merlin+ 132 Core SE FWSCSI PCI Disk"}, 
	{HPHW_FIO, 0x03F, 0x000A3, 0x0, "Merlin+ 180 Core SE FWSCSI PCI Disk"}, 
	{HPHW_FIO, 0x056, 0x000A3, 0x0, "Raven+ w SE FWSCSI Core PCI Disk"}, 
	{HPHW_FIO, 0x057, 0x000A3, 0x0, "Raven+ w Diff FWSCSI Core PCI Disk"}, 
	{HPHW_FIO, 0x004, 0x000A4, 0x0, "SPP2000 Core BA"}, 
	{HPHW_FIO, 0x004, 0x000A6, 0x0, "Sonic Ethernet 802.3 Card"}, 
	{HPHW_FIO, 0x004, 0x000A9, 0x00, "Forte Core PCI SuperIO RS-232"}, 
	{HPHW_FIO, 0x004, 0x000A9, 0x00, "Forte Core PCI USB KB"}, 
	{HPHW_FIO, 0x005, 0x000A9, 0x00, "AllegroLow Core PCI SuperIO RS-232"}, 
	{HPHW_FIO, 0x005, 0x000A9, 0x00, "AllegroLow Core PCI USB KB"}, 
	{HPHW_FIO, 0x006, 0x000A9, 0x00, "AllegroHigh Core PCI SuperIO RS-232"}, 
	{HPHW_FIO, 0x006, 0x000A9, 0x00, "AllegroHigh Core PCI USB KB"}, 
	{HPHW_FIO, 0x007, 0x000A9, 0x0, "Miscelaneous PCI Plug-in"}, 
	{HPHW_FIO, 0x00A, 0x000A9, 0x0, "Lego 360 Core PCI SuperIO RS-232"}, 
	{HPHW_FIO, 0x00A, 0x000A9, 0x0, "Lego 360 Core PCI USB KB"}, 
	{HPHW_FIO, 0x004, 0x00320, 0x0, "Metheus Frame Buffer"}, 
	{HPHW_FIO, 0x004, 0x00340, 0x0, "BARCO CX4500 VME Grphx Cnsl"}, 
	{HPHW_FIO, 0x004, 0x00360, 0x0, "Hughes TOG VME FDDI"}, 
	{HPHW_IOA, 0x185, 0x0000B, 0x00, "Java BC Summit Port"}, 
	{HPHW_IOA, 0x1FF, 0x0000B, 0x00, "Hitachi Ghostview Summit Port"}, 
	{HPHW_IOA, 0x580, 0x0000B, 0x10, "U2-IOA BC Runway Port"}, 
	{HPHW_IOA, 0x581, 0x0000B, 0x10, "Uturn-IOA BC Runway Port"},
	{HPHW_IOA, 0x582, 0x0000B, 0x10, "Astro BC Runway Port"},
	{HPHW_IOA, 0x700, 0x0000B, 0x00, "NEC-IOS BC System Bus Port"},
	{HPHW_IOA, 0x880, 0x0000C, 0x10, "Pluto BC McKinley Port"},
	{HPHW_MEMORY, 0x002, 0x00008, 0x00, "MID_BUS"}, 
	{HPHW_MEMORY, 0x063, 0x00009, 0x00, "712/132 L2 Upgrade"}, 
	{HPHW_MEMORY, 0x064, 0x00009, 0x00, "712/160 L2 Upgrade"}, 
	{HPHW_MEMORY, 0x065, 0x00009, 0x00, "715/132 L2 Upgrade"}, 
	{HPHW_MEMORY, 0x066, 0x00009, 0x00, "715/160 L2 Upgrade"},
	{HPHW_MEMORY, 0x0AF, 0x00009, 0x00, "Everest Mako Memory"},
	{HPHW_OTHER, 0x004, 0x00030, 0x00, "Master"}, 
	{HPHW_OTHER, 0x004, 0x00034, 0x00, "Slave"}, 
	{HPHW_OTHER, 0x004, 0x00038, 0x00, "EDU"}, 
	{HPHW_OTHER, 0x004, 0x00049, 0x00, "LGB Control"}, 
	{HPHW_MC, 0x004, 0x000C0, 0x00, "BMC IPMI Mgmt Ctlr"}, 
	{HPHW_FAULTY, 0, }  /* Special Marker for last entry */
};


static struct hp_cpu_type_mask {
	unsigned short model;
	unsigned short mask;
	enum cpu_type cpu;
} hp_cpu_type_mask_list[] __initdata = {

	{ 0x0000, 0x0ff0, pcx    },  /* 0x0000 - 0x000f */
	{ 0x0048, 0x0ff0, pcxl   },  /* 0x0040 - 0x004f */
	{ 0x0080, 0x0ff0, pcx    },  /* 0x0080 - 0x008f */
	{ 0x0100, 0x0ff0, pcx    },  /* 0x0100 - 0x010f */
	{ 0x0182, 0x0ffe, pcx    },  /* 0x0182 - 0x0183 */
	{ 0x0182, 0x0ffe, pcxt   },  /* 0x0182 - 0x0183 */
	{ 0x0184, 0x0fff, pcxu   },  /* 0x0184 - 0x0184 */
	{ 0x0200, 0x0ffe, pcxs   },  /* 0x0200 - 0x0201 */
	{ 0x0202, 0x0fff, pcxs   },  /* 0x0202 - 0x0202 */
	{ 0x0203, 0x0fff, pcxt   },  /* 0x0203 - 0x0203 */
	{ 0x0204, 0x0ffc, pcxt   },  /* 0x0204 - 0x0207 */
	{ 0x0280, 0x0ffc, pcxs   },  /* 0x0280 - 0x0283 */
	{ 0x0284, 0x0ffc, pcxt   },  /* 0x0284 - 0x0287 */
	{ 0x0288, 0x0fff, pcxt   },  /* 0x0288 - 0x0288 */
	{ 0x0300, 0x0ffc, pcxs   },  /* 0x0300 - 0x0303 */
	{ 0x0310, 0x0ff0, pcxt   },  /* 0x0310 - 0x031f */
	{ 0x0320, 0x0ff0, pcxt   },  /* 0x0320 - 0x032f */
	{ 0x0400, 0x0ff0, pcxt   },  /* 0x0400 - 0x040f */
	{ 0x0480, 0x0ff0, pcxl   },  /* 0x0480 - 0x048f */
	{ 0x0500, 0x0ff0, pcxl2  },  /* 0x0500 - 0x050f */
	{ 0x0510, 0x0ff0, pcxl2  },  /* 0x0510 - 0x051f */
	{ 0x0580, 0x0ff8, pcxt_  },  /* 0x0580 - 0x0587 */
	{ 0x0588, 0x0ffc, pcxt_  },  /* 0x0588 - 0x058b */
	{ 0x058c, 0x0ffe, pcxt_  },  /* 0x058c - 0x058d */
	{ 0x058e, 0x0fff, pcxt_  },  /* 0x058e - 0x058e */
	{ 0x058f, 0x0fff, pcxu   },  /* 0x058f - 0x058f */
	{ 0x0590, 0x0ffe, pcxu   },  /* 0x0590 - 0x0591 */
	{ 0x0592, 0x0fff, pcxt_  },  /* 0x0592 - 0x0592 */
	{ 0x0593, 0x0fff, pcxu   },  /* 0x0593 - 0x0593 */
	{ 0x0594, 0x0ffc, pcxu   },  /* 0x0594 - 0x0597 */
	{ 0x0598, 0x0ffe, pcxu_  },  /* 0x0598 - 0x0599 */
	{ 0x059a, 0x0ffe, pcxu   },  /* 0x059a - 0x059b */
	{ 0x059c, 0x0fff, pcxu   },  /* 0x059c - 0x059c */
	{ 0x059d, 0x0fff, pcxu_  },  /* 0x059d - 0x059d */
	{ 0x059e, 0x0fff, pcxt_  },  /* 0x059e - 0x059e */
	{ 0x059f, 0x0fff, pcxu   },  /* 0x059f - 0x059f */
	{ 0x05a0, 0x0ffe, pcxt_  },  /* 0x05a0 - 0x05a1 */
	{ 0x05a2, 0x0ffe, pcxu   },  /* 0x05a2 - 0x05a3 */
	{ 0x05a4, 0x0ffc, pcxu   },  /* 0x05a4 - 0x05a7 */
	{ 0x05a8, 0x0ffc, pcxu   },  /* 0x05a8 - 0x05ab */
	{ 0x05ad, 0x0fff, pcxu_  },  /* 0x05ad - 0x05ad */
	{ 0x05ae, 0x0ffe, pcxu_  },  /* 0x05ae - 0x05af */
	{ 0x05b0, 0x0ffe, pcxu_  },  /* 0x05b0 - 0x05b1 */
	{ 0x05b2, 0x0fff, pcxu_  },  /* 0x05b2 - 0x05b2 */
	{ 0x05b3, 0x0fff, pcxu   },  /* 0x05b3 - 0x05b3 */
	{ 0x05b4, 0x0fff, pcxw   },  /* 0x05b4 - 0x05b4 */
	{ 0x05b5, 0x0fff, pcxu_  },  /* 0x05b5 - 0x05b5 */
	{ 0x05b6, 0x0ffe, pcxu_  },  /* 0x05b6 - 0x05b7 */
	{ 0x05b8, 0x0ffe, pcxu_  },  /* 0x05b8 - 0x05b9 */
	{ 0x05ba, 0x0fff, pcxu_  },  /* 0x05ba - 0x05ba */
	{ 0x05bb, 0x0fff, pcxw   },  /* 0x05bb - 0x05bb */
	{ 0x05bc, 0x0ffc, pcxw   },  /* 0x05bc - 0x05bf */
	{ 0x05c0, 0x0ffc, pcxw 	 },  /* 0x05c0 - 0x05c3 */
	{ 0x05c4, 0x0ffe, pcxw 	 },  /* 0x05c4 - 0x05c5 */
	{ 0x05c6, 0x0fff, pcxw 	 },  /* 0x05c6 - 0x05c6 */
	{ 0x05c7, 0x0fff, pcxw_  },  /* 0x05c7 - 0x05c7 */
	{ 0x05c8, 0x0ffc, pcxw 	 },  /* 0x05c8 - 0x05cb */
	{ 0x05cc, 0x0ffe, pcxw 	 },  /* 0x05cc - 0x05cd */
	{ 0x05ce, 0x0ffe, pcxw_  },  /* 0x05ce - 0x05cf */
	{ 0x05d0, 0x0ffc, pcxw_  },  /* 0x05d0 - 0x05d3 */
	{ 0x05d4, 0x0ffe, pcxw_  },  /* 0x05d4 - 0x05d5 */
	{ 0x05d6, 0x0fff, pcxw 	 },  /* 0x05d6 - 0x05d6 */
	{ 0x05d7, 0x0fff, pcxw_  },  /* 0x05d7 - 0x05d7 */
	{ 0x05d8, 0x0ffc, pcxw_  },  /* 0x05d8 - 0x05db */
	{ 0x05dc, 0x0ffe, pcxw2  },  /* 0x05dc - 0x05dd */
	{ 0x05de, 0x0fff, pcxw_  },  /* 0x05de - 0x05de */
	{ 0x05df, 0x0fff, pcxw2  },  /* 0x05df - 0x05df */
	{ 0x05e0, 0x0ffc, pcxw2  },  /* 0x05e0 - 0x05e3 */
	{ 0x05e4, 0x0fff, pcxw2  },  /* 0x05e4 - 0x05e4 */
	{ 0x05e5, 0x0fff, pcxw_  },  /* 0x05e5 - 0x05e5 */
	{ 0x05e6, 0x0ffe, pcxw2  },  /* 0x05e6 - 0x05e7 */
	{ 0x05e8, 0x0ff8, pcxw2  },  /* 0x05e8 - 0x05ef */
	{ 0x05f0, 0x0ff0, pcxw2  },  /* 0x05f0 - 0x05ff */
	{ 0x0600, 0x0fe0, pcxl   },  /* 0x0600 - 0x061f */
	{ 0x0880, 0x0ff0, mako   },  /* 0x0880 - 0x088f */
	{ 0x0000, 0x0000, pcx    }	/* terminate table */
};

char *cpu_name_version[][2] = {
	[pcx]	{ "PA7000 (PCX)",	"1.0" },
	[pcxs]	{ "PA7000 (PCX-S)",	"1.1a" },
	[pcxt]	{ "PA7100 (PCX-T)",	"1.1b" },
	[pcxt_]	{ "PA7200 (PCX-T')",	"1.1c" },
	[pcxl]	{ "PA7100LC (PCX-L)",	"1.1d" },
	[pcxl2]	{ "PA7300LC (PCX-L2)",	"1.1e" },
	[pcxu]	{ "PA8000 (PCX-U)",	"2.0" },
	[pcxu_]	{ "PA8200 (PCX-U+)",	"2.0" },
	[pcxw]	{ "PA8500 (PCX-W)",	"2.0" },
	[pcxw_]	{ "PA8600 (PCX-W+)",	"2.0" },
	[pcxw2]	{ "PA8700 (PCX-W2)",	"2.0" },
	[mako]	{ "PA8800 (Mako)",	"2.0" }
};

const char * __init
parisc_hardware_description(struct parisc_device_id *id)
{
	struct hp_hardware *listptr;
	
	for (listptr = hp_hardware_list; listptr->hw_type != HPHW_FAULTY; listptr++) {
		if ((listptr->hw_type == id->hw_type) &&
				(listptr->hversion == id->hversion) &&
				(listptr->sversion == id->sversion)){
			return listptr->name;
		}
	}

	/*
	 * ok, the above hardware table isn't complete, and we haven't found
	 * our device in this table. So let's now try to find a generic name
	 * to describe the given hardware...
	 */
	switch (id->hw_type) {
		case HPHW_NPROC:
			return "Unknown machine";

		case HPHW_A_DIRECT:
			switch (id->sversion) {
				case 0x0D: return "MUX port";
				case 0x0E: return "RS-232 port";
			}
			break;
			
		case HPHW_MEMORY:
			return "Memory";
			
	}
	
	return "unknown device";
}


/* Interpret hversion (ret[0]) from PDC_MODEL(4)/PDC_MODEL_INFO(0) */
enum cpu_type __init
parisc_get_cpu_type(unsigned long hversion)
{
	struct hp_cpu_type_mask *ptr;
	unsigned short model = ((unsigned short) (hversion)) >> 4;

	for (ptr = hp_cpu_type_mask_list; 0 != ptr->mask; ptr++) {
		if (ptr->model == (model & ptr->mask))
			return ptr->cpu;
	}
	panic("could not identify CPU type\n");

	return pcx;	/* not reached: */
}

