/*
 * HvCallEvent.h
 * Copyright (C) 2001  Mike Corrigan IBM Corporation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

//==================================================================
//
//	This file contains the "hypervisor call" interface which is used to
//	drive the hypervisor from the OS.
//
//==================================================================

//-------------------------------------------------------------------
// Standard Includes
//-------------------------------------------------------------------
#ifndef  _HVCALLSC_H
#include <asm/iSeries/HvCallSc.h>
#endif

#ifndef  _HVTYPES_H
#include <asm/iSeries/HvTypes.h>
#endif

#include <asm/iSeries/LparData.h>

//-------------------------------------------------------------------
// Constants
//-------------------------------------------------------------------
#ifndef _HVCALLEVENT_H
#define _HVCALLEVENT_H

struct HvLpEvent;

typedef u8 HvLpEvent_Type;
typedef u8 HvLpEvent_AckInd;
typedef u8 HvLpEvent_AckType;

struct	HvCallEvent_PackedParms
{
    u8		xAckType:1;
    u8		xAckInd:1;
    u8		xRsvd:1;
    u8		xTargetLp:5;
    u8		xType;
    u16		xSubtype;
    HvLpInstanceId	xSourceInstId;
    HvLpInstanceId	xTargetInstId;
};

typedef u8 HvLpDma_Direction;
typedef u8 HvLpDma_AddressType;

struct	HvCallEvent_PackedDmaParms
{
    u8		xDirection:1;
    u8		xLocalAddrType:1;
    u8		xRemoteAddrType:1;
    u8		xRsvd1:5;
    HvLpIndex	xRemoteLp;
    u8		xType;
    u8		xRsvd2;
    HvLpInstanceId	xLocalInstId;
    HvLpInstanceId	xRemoteInstId;
};

typedef u64 HvLpEvent_Rc;
typedef u64 HvLpDma_Rc;

#define HvCallEventAckLpEvent				HvCallEvent +  0
#define HvCallEventCancelLpEvent			HvCallEvent +  1
#define HvCallEventCloseLpEventPath			HvCallEvent +  2
#define HvCallEventDmaBufList				HvCallEvent +  3
#define HvCallEventDmaSingle				HvCallEvent +  4
#define HvCallEventDmaToSp				HvCallEvent +  5 
#define HvCallEventGetOverflowLpEvents			HvCallEvent +  6
#define HvCallEventGetSourceLpInstanceId		HvCallEvent +  7
#define HvCallEventGetTargetLpInstanceId		HvCallEvent +  8
#define HvCallEventOpenLpEventPath			HvCallEvent +  9
#define HvCallEventSetLpEventStack			HvCallEvent + 10
#define HvCallEventSignalLpEvent			HvCallEvent + 11
#define HvCallEventSignalLpEventParms			HvCallEvent + 12
#define HvCallEventSetInterLpQueueIndex			HvCallEvent + 13
#define HvCallEventSetLpEventQueueInterruptProc		HvCallEvent + 14
#define HvCallEventRouter15				HvCallEvent + 15

//======================================================================
static inline void		HvCallEvent_getOverflowLpEvents(u8 queueIndex)
{
    HvCall1(HvCallEventGetOverflowLpEvents,queueIndex);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
}
//======================================================================
static inline void		HvCallEvent_setInterLpQueueIndex(u8 queueIndex)
{
    HvCall1(HvCallEventSetInterLpQueueIndex,queueIndex);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
}
//======================================================================
static inline void		HvCallEvent_setLpEventStack(u8 queueIndex,
					     char * eventStackAddr,
					     u32 eventStackSize)
{
    u64 abs_addr;
    abs_addr = virt_to_absolute_outline( (unsigned long) eventStackAddr );

    HvCall3(HvCallEventSetLpEventStack, queueIndex, abs_addr, eventStackSize);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
}
//======================================================================
static inline void		HvCallEvent_setLpEventQueueInterruptProc(u8 queueIndex,
							  u16 lpLogicalProcIndex)
{
    HvCall2(HvCallEventSetLpEventQueueInterruptProc,queueIndex,lpLogicalProcIndex);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
}
//=====================================================================
static inline HvLpEvent_Rc HvCallEvent_signalLpEvent(struct HvLpEvent* event)
{
    u64 abs_addr;
    HvLpEvent_Rc retVal;
    abs_addr = virt_to_absolute_outline( (unsigned long) event );
    retVal = (HvLpEvent_Rc)HvCall1(HvCallEventSignalLpEvent, abs_addr);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
    return retVal;
}
//=====================================================================
static inline HvLpEvent_Rc  HvCallEvent_signalLpEventFast(HvLpIndex targetLp,
					   HvLpEvent_Type type,
					   u16 subtype,
					   HvLpEvent_AckInd ackInd,
					   HvLpEvent_AckType ackType,
					   HvLpInstanceId sourceInstanceId,
					   HvLpInstanceId targetInstanceId,
					   u64 correlationToken,
					   u64 eventData1,
					   u64 eventData2,
					   u64 eventData3,
					   u64 eventData4,
					   u64 eventData5)
{
    HvLpEvent_Rc retVal;
    	
    // Pack the misc bits into a single Dword to pass to PLIC
    union
    {
	struct HvCallEvent_PackedParms	parms;
	u64		dword;
    } packed;
    packed.parms.xAckType	= ackType;
    packed.parms.xAckInd	= ackInd;
    packed.parms.xRsvd		= 0;
    packed.parms.xTargetLp	= targetLp;
    packed.parms.xType		= type;
    packed.parms.xSubtype	= subtype;
    packed.parms.xSourceInstId	= sourceInstanceId;
    packed.parms.xTargetInstId	= targetInstanceId;

    retVal = (HvLpEvent_Rc)HvCall7(HvCallEventSignalLpEventParms,
						      packed.dword,
						      correlationToken,
						      eventData1,eventData2,
						      eventData3,eventData4,
						      eventData5);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
    return retVal;
}
//====================================================================
static inline HvLpEvent_Rc	HvCallEvent_ackLpEvent(struct HvLpEvent* event)
{
    u64 abs_addr;
    HvLpEvent_Rc retVal;
    abs_addr = virt_to_absolute_outline( (unsigned long) event );

    retVal = (HvLpEvent_Rc)HvCall1(HvCallEventAckLpEvent, abs_addr);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
    return retVal;
}
//====================================================================
static inline HvLpEvent_Rc   HvCallEvent_cancelLpEvent(struct HvLpEvent* event)
{
    u64 abs_addr;
    HvLpEvent_Rc retVal;
    abs_addr = virt_to_absolute_outline( (unsigned long) event );

    retVal = (HvLpEvent_Rc)HvCall1(HvCallEventCancelLpEvent, abs_addr);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
    return retVal;
}
//===================================================================
static inline HvLpInstanceId	HvCallEvent_getSourceLpInstanceId(HvLpIndex targetLp, HvLpEvent_Type type)
{
    HvLpInstanceId retVal;	
    retVal = HvCall2(HvCallEventGetSourceLpInstanceId,targetLp,type);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
    return retVal;
}
//===================================================================
static inline HvLpInstanceId	HvCallEvent_getTargetLpInstanceId(HvLpIndex targetLp, HvLpEvent_Type type)
{
    HvLpInstanceId retVal;	
    retVal = HvCall2(HvCallEventGetTargetLpInstanceId,targetLp,type);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
    return retVal;
}
//===================================================================
static inline void		HvCallEvent_openLpEventPath(HvLpIndex targetLp,
					     HvLpEvent_Type type)
{
    HvCall2(HvCallEventOpenLpEventPath,targetLp,type);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
}
//===================================================================
static inline void		HvCallEvent_closeLpEventPath(HvLpIndex targetLp,
					      HvLpEvent_Type type)
{
    HvCall2(HvCallEventCloseLpEventPath,targetLp,type);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
}
//===================================================================
static inline HvLpDma_Rc	HvCallEvent_dmaBufList(HvLpEvent_Type type,
					HvLpIndex remoteLp,
					HvLpDma_Direction direction,
					HvLpInstanceId localInstanceId,
					HvLpInstanceId remoteInstanceId,
					HvLpDma_AddressType localAddressType,
					HvLpDma_AddressType remoteAddressType,
					// Do these need to be converted to
					// absolute addresses?
					u64 localBufList,
					u64 remoteBufList,

					u32 transferLength)
{
    HvLpDma_Rc retVal;	
    // Pack the misc bits into a single Dword to pass to PLIC
    union
    {
	struct HvCallEvent_PackedDmaParms	parms;
	u64		dword;
    } packed;
    packed.parms.xDirection	= direction;
    packed.parms.xLocalAddrType	= localAddressType;
    packed.parms.xRemoteAddrType	= remoteAddressType;
    packed.parms.xRsvd1		= 0;
    packed.parms.xRemoteLp	= remoteLp;
    packed.parms.xType		= type;
    packed.parms.xRsvd2		= 0;
    packed.parms.xLocalInstId	= localInstanceId;
    packed.parms.xRemoteInstId	= remoteInstanceId;

    retVal = (HvLpDma_Rc)HvCall4(HvCallEventDmaBufList,
					  packed.dword,
					  localBufList,
					  remoteBufList,
					  transferLength);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
    return retVal;
}
//=================================================================
static inline HvLpDma_Rc	HvCallEvent_dmaSingle(HvLpEvent_Type type,
				       HvLpIndex remoteLp,
				       HvLpDma_Direction direction,
				       HvLpInstanceId localInstanceId,
				       HvLpInstanceId remoteInstanceId,
				       HvLpDma_AddressType localAddressType,
				       HvLpDma_AddressType remoteAddressType,
				       u64 localAddrOrTce,
				       u64 remoteAddrOrTce,
				       u32 transferLength)
{
    HvLpDma_Rc retVal;	
    // Pack the misc bits into a single Dword to pass to PLIC
    union
    {
	struct HvCallEvent_PackedDmaParms	parms;
	u64		dword;
    } packed;
    packed.parms.xDirection	= direction;
    packed.parms.xLocalAddrType	= localAddressType;
    packed.parms.xRemoteAddrType	= remoteAddressType;
    packed.parms.xRsvd1		= 0;
    packed.parms.xRemoteLp	= remoteLp;
    packed.parms.xType		= type;
    packed.parms.xRsvd2		= 0;
    packed.parms.xLocalInstId	= localInstanceId;
    packed.parms.xRemoteInstId	= remoteInstanceId;

    retVal = (HvLpDma_Rc)HvCall4(HvCallEventDmaSingle,
					  packed.dword,
					  localAddrOrTce,
					  remoteAddrOrTce,
					  transferLength);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
    return retVal;
}
//=================================================================
static inline HvLpDma_Rc	HvCallEvent_dmaToSp(void* local, u32 remote, u32 length, HvLpDma_Direction dir)
{
    u64 abs_addr;
    HvLpDma_Rc retVal;
    abs_addr = virt_to_absolute_outline( (unsigned long) local );
    
    retVal = (HvLpDma_Rc)HvCall4(HvCallEventDmaToSp, 
		    			abs_addr,
					remote,
					length,
					dir);
    // getPaca()->adjustHmtForNoOfSpinLocksHeld();
    return retVal;
}
//================================================================

#endif // _HVCALLEVENT_H

