/*************************************************** */
/* Rule Set Based Access Control                     */
/* Implementation of the Access Control Decision     */
/* Facility (ADF) - Main file main.c                 */
/*                                                   */
/* Author and (c) 1999-2003: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 16/Jul/2003                        */
/*************************************************** */

#include <linux/string.h>
#include <linux/init.h>
#include <linux/config.h>
#include <linux/module.h>
#include <rsbac/types.h>
#include <rsbac/aci.h>
#include <rsbac/adf.h>
#include <rsbac/adf_main.h>
#include <rsbac/error.h>
#include <rsbac/helpers.h>
#include <rsbac/getname.h>
#include <rsbac/rkmem.h>
#include <rsbac/network.h>

#ifdef CONFIG_RSBAC_NO_DECISION_ON_NETMOUNT
#include <linux/nfs_fs.h>
#include <linux/coda_psdev.h>
#include <linux/ncp_fs.h>
#include <linux/smb.h>
#endif

#ifdef CONFIG_RSBAC_SECDEL
#include <linux/types.h>
#include <linux/dcache.h>
#include <asm/uaccess.h>
#endif /* SECDEL */

/************************************************* */
/*           Global Variables                      */
/************************************************* */

u_long rsbac_adf_request_count[T_NONE+1] = {0,0,0,0,0,0,0,0};
u_long rsbac_adf_set_attr_count[T_NONE+1] = {0,0,0,0,0,0,0,0};
#ifdef CONFIG_RSBAC_XSTATS
u_long rsbac_adf_request_xcount[T_NONE+1][R_NONE];
u_long rsbac_adf_set_attr_xcount[T_NONE+1][R_NONE];
#endif

#ifdef CONFIG_RSBAC_SWITCH

/******* MAC ********/
#ifdef CONFIG_RSBAC_MAC
boolean rsbac_switch_mac = TRUE;
#endif  /* MAC */

/******* FC ********/
#ifdef CONFIG_RSBAC_FC
boolean rsbac_switch_fc = TRUE;
#endif  /* FC */

/******* SIM ********/
#ifdef CONFIG_RSBAC_SIM
boolean rsbac_switch_sim = TRUE;
#endif  /* SIM */

/******* PM ********/
#ifdef CONFIG_RSBAC_PM
boolean rsbac_switch_pm = TRUE;
#endif  /* PM */

/******* MS ********/
#ifdef CONFIG_RSBAC_MS
boolean rsbac_switch_ms = TRUE;
#endif  /* MS */

/******* FF ********/
#ifdef CONFIG_RSBAC_FF
boolean rsbac_switch_ff = TRUE;
#endif  /* FF */

/******* RC ********/
#ifdef CONFIG_RSBAC_RC
boolean rsbac_switch_rc = TRUE;
#endif  /* RC */

/****** AUTH *******/
#ifdef CONFIG_RSBAC_AUTH
boolean rsbac_switch_auth = TRUE;
#endif  /* AUTH */

/****** ACL *******/
#ifdef CONFIG_RSBAC_ACL
boolean rsbac_switch_acl = TRUE;
#endif  /* ACL */

/****** CAP *******/
#ifdef CONFIG_RSBAC_CAP
boolean rsbac_switch_cap = TRUE;
#endif  /* CAP */

/****** JAIL *******/
#ifdef CONFIG_RSBAC_JAIL
boolean rsbac_switch_jail = TRUE;
#endif  /* JAIL */

/****** RES *******/
#ifdef CONFIG_RSBAC_RES
boolean rsbac_switch_res = TRUE;
#endif  /* RES */

#endif /* SWITCH */

/************************************************* */
/*          Internal Help functions                */
/************************************************* */

/************************************************* */
/*          Externally visible functions           */
/************************************************* */

/* Init function, calls inits for all sub-modules  */

#ifdef CONFIG_RSBAC_INIT_DELAY
void rsbac_init_adf(void)
#else
void __init rsbac_init_adf(void)
#endif
  {
    #if defined(CONFIG_RSBAC_REG)
    rsbac_reg_init();
    #endif
  };

enum rsbac_adf_req_ret_t
    adf_and_plus(enum rsbac_adf_req_ret_t res1,
                 enum rsbac_adf_req_ret_t res2)
  {
    switch (res1)
      {
        case GRANTED:     if (res2 == DO_NOT_CARE)
                            return (GRANTED);
                          else
                            return (res2);
        case NOT_GRANTED: if (res2 == UNDEFINED)
                            return (UNDEFINED);
                          else
                            return (NOT_GRANTED);
        case DO_NOT_CARE: return (res2);
        default:          return (UNDEFINED);
      }
  };

/*********************************************************************/
/* rsbac_adf_request()                                               */
/* This function is the main decision function, called from the AEF. */

EXPORT_SYMBOL(rsbac_adf_request_int);
enum rsbac_adf_req_ret_t
   rsbac_adf_request_int(enum  rsbac_adf_request_t     request,
                               rsbac_pid_t             caller_pid,
                         enum  rsbac_target_t          target,
                         union rsbac_target_id_t       tid,
                         enum  rsbac_attribute_t       attr,
                         union rsbac_attribute_value_t attr_val,
                         enum  rsbac_switch_target_t   ignore_module)
  {
    union rsbac_target_id_t        i_tid;
    union rsbac_attribute_value_t  i_attr_val;
          rsbac_uid_t              owner=0;
    int error=0,tmperr=0;
    enum rsbac_adf_req_ret_t   result = DO_NOT_CARE;
#ifdef CONFIG_RSBAC_SOFTMODE_IND
    enum rsbac_adf_req_ret_t   ret_result = DO_NOT_CARE;
#endif
#ifndef CONFIG_RSBAC_MAINT
    enum rsbac_adf_req_ret_t   mod_result[SW_NONE + 1] = {
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE,
                                   DO_NOT_CARE
                                 };
#endif
    boolean do_log = FALSE;
    boolean log_on_request = TRUE;
/* only if individual logging is enabled */
#if defined(CONFIG_RSBAC_IND_LOG) || defined(CONFIG_RSBAC_IND_NETDEV_LOG) || defined(CONFIG_RSBAC_IND_NETOBJ_LOG)
    union rsbac_attribute_value_t  i_attr_val2;
    enum rsbac_log_level_t log_level;
#endif
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,0) || defined(CONFIG_RSBAC_NO_DECISION_ON_NETMOUNT)
    struct super_block * sb_p;
#endif
#ifdef CONFIG_RSBAC_SOFTMODE
    boolean rsbac_internal = FALSE;
#endif

/* No decision possible before init (called at boot time) -> don't care */
    if (!rsbac_is_initialized())
      return(DO_NOT_CARE);

/* Always granted for kernel (pid 0) and logging daemon */
    if (   !caller_pid
        #if defined(CONFIG_RSBAC_LOG_REMOTE)
        || (caller_pid == rsbaclogd_pid)
        #endif
       )
      return(GRANTED);

/* Checking base values */
    if(   request >= R_NONE
       || target > T_NONE
       || attr > A_none)
      {
        printk(KERN_WARNING
               "rsbac_adf_request(): called with invalid request, target or attribute\n");
        return(NOT_GRANTED);
      }

/* Getting basic information about this request */

    /* only useful for real process, not idle or init */
    if (caller_pid > 1)
      {
        tmperr = rsbac_get_owner(&owner);
        if(tmperr)
          {
            printk(KERN_DEBUG
                   "rsbac_adf_request(): caller_pid %i, RSBAC not initialized, returning DO_NOT_CARE",
                   caller_pid);
            return(DO_NOT_CARE);      /* Startup-Sequence (see above) */
          }
      }
    else  /* caller_pid = 1 -> init, always owned by root */
      owner = 0;


/******************************************************/
/* General work for all modules - before module calls */
    /* test target on rsbac_internal: get its sec_level */
    switch(target)
      {
        case T_FILE:
        case T_DIR:
        case T_FIFO:
        case T_SYMLINK:
#ifdef CONFIG_RSBAC_NO_DECISION_ON_NETMOUNT
          if (   ((sb_p = rsbac_get_super_block(tid.file.device)))
              && (   (sb_p->s_magic == NFS_SUPER_MAGIC)
                  || (sb_p->s_magic == CODA_SUPER_MAGIC)
                  || (sb_p->s_magic == NCP_SUPER_MAGIC)
                  || (sb_p->s_magic == SMB_SUPER_MAGIC)
                 )
             )
            {
              result = DO_NOT_CARE;
              goto log;
            }
#endif
          if(tid.file.device == 0)
            {
              if (   (rsbac_debug_adf_default == 2)
#ifdef CONFIG_RSBAC_DEBUG
                  || rsbac_debug_aef
#endif
                 )
                printk(KERN_DEBUG
                       "rsbac_adf_request(): device is 0 for request %i, target type %i, returning DO_NOT_CARE!\n",
                       request,target);
              result = DO_NOT_CARE;
              goto log;
            }
          switch(request)
            {
              case R_GET_STATUS_DATA:
              #if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,0)
                if(   (target == T_FIFO)
                   && ((sb_p = rsbac_get_super_block(tid.file.device)))
                   && (sb_p->s_magic == PIPEFS_MAGIC)
                  )
                  return DO_NOT_CARE;
                break;
              #endif
              case R_READ_ATTRIBUTE:
#ifdef CONFIG_RSBAC_DAT_VISIBLE
              case R_SEARCH:
              case R_READ:
              case R_CLOSE:
              case R_CHDIR:
              case R_GET_PERMISSIONS_DATA:
#endif
                break;

              default:
                if ((tmperr = rsbac_get_attr(GEN,
                                             target,
                                             tid,
                                             A_internal,
                                             &i_attr_val,
                                             TRUE) ))
                  {
                    if(tmperr == -RSBAC_EINVALIDDEV)
                      {
                        printk(KERN_WARNING
                               "rsbac_adf_request(): rsbac_get_attr() for internal returned EINVALIDDEV!\n");
                        return(DO_NOT_CARE);  /* last calls on shutdown */
                      }
                    else
                      {
                        printk(KERN_WARNING
                               "rsbac_adf_request(): rsbac_get_attr() for internal returned error %i!\n",
                               tmperr);
                        return(NOT_GRANTED);  /* something weird happened */
                      }
                  }
                /* no access to rsbac_internal objects is granted in any case */
                if (i_attr_val.internal)
                  {
                    printk(KERN_WARNING
                           "rsbac_adf_request(): trial to access object declared RSBAC-internal!\n");
                    result = NOT_GRANTED;
                    #ifndef CONFIG_RSBAC_MAINT
                    mod_result[SW_NONE] = NOT_GRANTED;
                    #endif
                    #ifdef CONFIG_RSBAC_SOFTMODE
                    #ifdef CONFIG_RSBAC_SOFTMODE_IND
                    ret_result = NOT_GRANTED;
                    #endif
                    rsbac_internal = TRUE;
                    #endif
                  }
            }
          break;

        default:
          break;
      }
      
/**********************************************************/
/* calling all decision modules, building a common result */

#ifdef CONFIG_RSBAC_DEBUG
/* first, check for valid request/target combination      */
/* (undefined should only happen in _check and means a real bug!) */
  result = adf_and_plus(result,rsbac_adf_request_check(request,
                                                       caller_pid,
                                                       target,
                                                       tid,
                                                       attr,
                                                       attr_val,
                                                       owner) );

if(result == UNDEFINED)
  goto log;
#endif

#if !defined(CONFIG_RSBAC_MAINT)
/******* MAC ********/
#if defined(CONFIG_RSBAC_MAC)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_mac)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != MAC)
    {
      mod_result[MAC] = rsbac_adf_request_mac(request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[MAC]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[MAC])
        ret_result = adf_and_plus(ret_result, mod_result[MAC]);
#endif
    }
#endif  /* MAC */

/******* FC ********/
#if defined(CONFIG_RSBAC_FC)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_fc)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != FC)
    {
      mod_result[FC]  = rsbac_adf_request_fc (request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[FC]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[FC])
        ret_result = adf_and_plus(ret_result, mod_result[FC]);
#endif
    }
#endif  /* FC */

/******* SIM ********/
#if defined(CONFIG_RSBAC_SIM)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_sim)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != SIM)
    {
      mod_result[SIM] = rsbac_adf_request_sim(request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[SIM]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[SIM])
        ret_result = adf_and_plus(ret_result, mod_result[SIM]);
#endif
    }
#endif  /* SIM */

/******* PM ********/
#if defined(CONFIG_RSBAC_PM)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_pm)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != PM)
    {
      mod_result[PM]  = rsbac_adf_request_pm (request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[PM]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[PM])
        ret_result = adf_and_plus(ret_result, mod_result[PM]);
#endif
    }
#endif  /* PM */

/******* MS ********/
#if defined(CONFIG_RSBAC_MS)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_ms)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != MS)
    {
      mod_result[MS]  = rsbac_adf_request_ms (request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[MS]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[MS])
        ret_result = adf_and_plus(ret_result, mod_result[MS]);
#endif
    }
#endif  /* MS */

/******* FF ********/
#if defined(CONFIG_RSBAC_FF)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_ff)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != FF)
    {
      mod_result[FF]  = rsbac_adf_request_ff (request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[FF]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[FF])
        ret_result = adf_and_plus(ret_result, mod_result[FF]);
#endif
    }
#endif  /* FF */

/******* RC ********/
#if defined(CONFIG_RSBAC_RC)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_rc)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != RC)
    {
      mod_result[RC]  = rsbac_adf_request_rc (request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[RC]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[RC])
        ret_result = adf_and_plus(ret_result, mod_result[RC]);
#endif
    }
#endif  /* RC */

/****** AUTH *******/
#if defined(CONFIG_RSBAC_AUTH)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_auth)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != AUTH)
    {
      mod_result[AUTH]= rsbac_adf_request_auth(request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[AUTH]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[AUTH])
        ret_result = adf_and_plus(ret_result, mod_result[AUTH]);
#endif
    }
#endif  /* AUTH */

/****** ACL *******/
#if defined(CONFIG_RSBAC_ACL)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_acl)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != ACL)
    {
      mod_result[ACL] = rsbac_adf_request_acl(request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[ACL]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[ACL])
        ret_result = adf_and_plus(ret_result, mod_result[ACL]);
#endif
    }
#endif  /* ACL */

/****** CAP *******/
#if defined(CONFIG_RSBAC_CAP)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_cap)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != CAP)
    {
      mod_result[CAP] = rsbac_adf_request_cap(request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[CAP]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[CAP])
        ret_result = adf_and_plus(ret_result, mod_result[CAP]);
#endif
    }
#endif  /* CAP */

/****** JAIL *******/
#if defined(CONFIG_RSBAC_JAIL)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_jail)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != JAIL)
    {
      mod_result[JAIL]= rsbac_adf_request_jail(request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[JAIL]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[JAIL])
        ret_result = adf_and_plus(ret_result, mod_result[JAIL]);
#endif
    }
#endif  /* JAIL */

/****** RES *******/
#if defined(CONFIG_RSBAC_RES)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_res)
#endif
  /* no need to call module, if to be ignored */
  if(ignore_module != RES)
    {
      mod_result[RES] = rsbac_adf_request_res(request,
                                              caller_pid,
                                              target,
                                              tid,
                                              attr,
                                              attr_val,
                                              owner);
      result = adf_and_plus(result, mod_result[RES]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
      if(!rsbac_ind_softmode[RES])
        ret_result = adf_and_plus(ret_result, mod_result[RES]);
#endif
    }
#endif  /* RES */

/****** REG *******/
#if defined(CONFIG_RSBAC_REG)
if(ignore_module != REG)
  {
    mod_result[REG]= rsbac_adf_request_reg (request,
                                            caller_pid,
                                            target,
                                            tid,
                                            attr,
                                            attr_val,
                                            owner);
    result = adf_and_plus(result, mod_result[REG]);
#ifdef CONFIG_RSBAC_SOFTMODE_IND
    if(!rsbac_ind_softmode[REG])
      ret_result = adf_and_plus(ret_result, mod_result[REG]);
#endif
  }
#endif  /* REG */

#endif /* !MAINT */

/****************************/

#if defined(CONFIG_RSBAC_DEBUG) && defined(CONFIG_RSBAC_NET)
    if(    rsbac_debug_adf_net
       && (   (target == T_NETDEV)
           || (target == T_NETTEMP)
           || (target == T_NETOBJ)
          )
      )
      do_log = TRUE;
#endif

/* log based on process owner */
#ifdef CONFIG_RSBAC_IND_USER_LOG
    i_tid.user = owner;
    if (rsbac_get_attr(GEN,
                       T_USER,
                       i_tid,
                       A_log_user_based,
                       &i_attr_val,
                       FALSE))
      {
        printk(KERN_WARNING
               "rsbac_adf_request(): rsbac_get_attr() for log_user_based returned error!\n");
      }
    else
      {
        if(((rsbac_request_vector_t) 1 << request) & i_attr_val.log_user_based) 
          do_log = TRUE;
      }
#endif /* CONFIG_RSBAC_IND_USER_LOG */

/* log based on program */
#ifdef CONFIG_RSBAC_IND_PROG_LOG
    if(!do_log)
      {
        i_tid.process = caller_pid;
        if (rsbac_get_attr(GEN,
                           T_PROCESS,
                           i_tid,
                           A_log_program_based,
                           &i_attr_val,
                           FALSE))
          {
            printk(KERN_WARNING
                   "rsbac_adf_request(): rsbac_get_attr() for log_program_based returned error!\n");
          }
        else
          {
            if(((rsbac_request_vector_t) 1 << request) & i_attr_val.log_program_based) 
              do_log = TRUE;
          }
      }
#endif /* CONFIG_RSBAC_IND_PROG_LOG */

/*****************************************************/
/* General work for all modules - after module calls */
/* Note: the process' individual logging attributes are needed above */
    switch(request)
      {
        case R_TERMINATE:
            if (target == T_PROCESS)
              rsbac_remove_target(T_PROCESS,tid);
            break;
        default:
            break;
      }

/* logging request on info level, if requested by file/dir/dev attributes */
/* log_array_low/high, or, if that is requested, if enabled for this request */
/* type (attributes state level, or that request based level is to be taken) */
/* loglevel 2: log everything */
/* loglevel 1: log, if denied */
/* loglevel 0: log nothing */

#ifdef CONFIG_RSBAC_IND_LOG /* only if individual logging is enabled */
    /* if file/dir/dev, depend log on log_arrays */
    /* (but not for file.device = 0) */
    /* log_on_request is TRUE */
    if(   !do_log
       && (   (   (   (target == T_FILE)
                   || (target == T_DIR)
                   || (target == T_FIFO)
                   || (target == T_SYMLINK)
                  )
               && tid.file.device
              )
           || (target == T_DEV)
          )
      )
      {
        if (rsbac_get_attr(GEN,
                           target,
                           tid,
                           A_log_array_low,
                           &i_attr_val,
                           FALSE))
          {
            printk(KERN_WARNING
                   "rsbac_adf_request(): rsbac_get_attr() for log_array_low returned error!\n");
          }
        else
          {
            if (rsbac_get_attr(GEN,
                               target,
                               tid,
                               A_log_array_high,
                               &i_attr_val2,
                               FALSE))
              {
                printk(KERN_WARNING
                       "rsbac_adf_request(): rsbac_get_attr() for log_array_high returned error!\n");
              }
            else
              { /* ll = low-bit for request | (high-bit for request as bit 1) */
                /* WARNING: we deal with u64 here, only logical operations and */
                /* shifts work correctly! */
                log_level =   ((i_attr_val.log_array_low   >> request) & 1)
                          | ( ((i_attr_val2.log_array_high >> request) & 1) << 1);
                if (   log_level == LL_full
                    || (   log_level == LL_denied
                        && (result == NOT_GRANTED
                            || result == UNDEFINED)) )
                  {
                    do_log = TRUE;
                  }
                if(log_level != LL_request)
                  log_on_request = FALSE;
              }
          }
      }
#endif /* CONFIG_RSBAC_IND_LOG */

#ifdef CONFIG_RSBAC_IND_NETDEV_LOG /* only if individual logging for netdev is enabled */
    /* if netdev, depend log on log_arrays */
    /* log_on_request is TRUE */
    if(   !do_log
       && (target == T_NETDEV)
      )
      {
        if (rsbac_get_attr(GEN,
                           target,
                           tid,
                           A_log_array_low,
                           &i_attr_val,
                           FALSE))
          {
            printk(KERN_WARNING
                   "rsbac_adf_request(): rsbac_get_attr() for log_array_low returned error!\n");
          }
        else
          {
            if (rsbac_get_attr(GEN,
                               target,
                               tid,
                               A_log_array_high,
                               &i_attr_val2,
                               FALSE))
              {
                printk(KERN_WARNING
                       "rsbac_adf_request(): rsbac_get_attr() for log_array_high returned error!\n");
              }
            else
              { /* ll = low-bit for request | (high-bit for request as bit 1) */
                /* WARNING: we deal with u64 here, only logical operations and */
                /* shifts work correctly! */
                log_level =   ((i_attr_val.log_array_low   >> request) & 1)
                          | ( ((i_attr_val2.log_array_high >> request) & 1) << 1);
                if (   log_level == LL_full
                    || (   log_level == LL_denied
                        && (result == NOT_GRANTED
                            || result == UNDEFINED)) )
                  {
                    do_log = TRUE;
                  }
                if(log_level != LL_request)
                  log_on_request = FALSE;
              }
          }
      }
#endif /* CONFIG_RSBAC_IND_NETDEV_LOG */

#ifdef CONFIG_RSBAC_IND_NETOBJ_LOG /* only if individual logging for net objects is enabled */
    /* if nettemp, netobj, depend log on log_arrays */
    /* (but not for file.device = 0) */
    /* log_on_request is TRUE */
    if(   !do_log
       && (   (target == T_NETTEMP)
           || (target == T_NETOBJ)
          )
      )
      {
        enum rsbac_attribute_t i_attr1, i_attr2;

        if(target == T_NETOBJ)
          {
            if(rsbac_net_remote_request(request))
              {
                i_attr1 = A_remote_log_array_low;
                i_attr2 = A_remote_log_array_high;
              }
            else
              {
                i_attr1 = A_local_log_array_low;
                i_attr2 = A_local_log_array_high;
              }
          }
        else
          {
            i_attr1 = A_log_array_low;
            i_attr2 = A_log_array_high;
          }
        if (rsbac_get_attr(GEN,
                           target,
                           tid,
                           i_attr1,
                           &i_attr_val,
                           FALSE))
          {
            printk(KERN_WARNING
                   "rsbac_adf_request(): rsbac_get_attr() for log_array_low returned error!\n");
          }
        else
          {
            if (rsbac_get_attr(GEN,
                               target,
                               tid,
                               i_attr2,
                               &i_attr_val2,
                               FALSE))
              {
                printk(KERN_WARNING
                       "rsbac_adf_request(): rsbac_get_attr() for log_array_high returned error!\n");
              }
            else
              { /* ll = low-bit for request | (high-bit for request as bit 1) */
                /* WARNING: we deal with u64 here, only logical operations and */
                /* shifts work correctly! */
                log_level =   ((i_attr_val.log_array_low   >> request) & 1)
                          | ( ((i_attr_val2.log_array_high >> request) & 1) << 1);
                if (   log_level == LL_full
                    || (   log_level == LL_denied
                        && (result == NOT_GRANTED
                            || result == UNDEFINED)) )
                  {
                    do_log = TRUE;
                  }
                if(log_level != LL_request)
                  log_on_request = FALSE;
              }
          }
      }
#endif /* CONFIG_RSBAC_IND_NETOBJ_LOG */

log:
    /* if enabled, try request based log level */
    if (   !do_log
        && log_on_request
        && (   rsbac_log_levels[request][target] == LL_full
            || (   rsbac_log_levels[request][target] == LL_denied
                && (result == NOT_GRANTED
                    || result == UNDEFINED)) ) )
      do_log = TRUE;

    if(do_log)
      {
        char * request_name;
        char * res_name;
        char * res_mods;
        char * target_type_name;
        char * target_id_name;
        char * attr_name;
        char  command[17];
        rsbac_pseudo_t  pseudo = 0;
        rsbac_pid_t parent_pid = 0;

        /* Get owner's logging pseudo */
        if (rsbac_get_attr(GEN,T_USER,i_tid,A_pseudo,&i_attr_val,FALSE))
          {
            printk(KERN_WARNING
              "rsbac_adf_request(): rsbac_get_attr() for pseudo returned error %i",
              error);
            return(NOT_GRANTED);  /* something weird happened */
          }
        /* if pseudo is not registered, return attribute value is 0 (see later) */
        pseudo = i_attr_val.pseudo;

	/* parent pid */
	if(current->p_pptr)
	  parent_pid = current->p_pptr->pid;
        /* rsbac_kmalloc all memory */
        request_name = rsbac_kmalloc(RSBAC_MAXNAMELEN);
        res_name = rsbac_kmalloc(RSBAC_MAXNAMELEN);
        res_mods = rsbac_kmalloc(RSBAC_MAXNAMELEN);
        target_type_name = rsbac_kmalloc(RSBAC_MAXNAMELEN);
        #ifdef CONFIG_RSBAC_LOG_FULL_PATH
        target_id_name
         = rsbac_kmalloc(CONFIG_RSBAC_MAX_PATH_LEN + RSBAC_MAXNAMELEN);
           /* max. path name len + some extra */
        #else
        target_id_name = rsbac_kmalloc(2 * RSBAC_MAXNAMELEN);
           /* max. file name len + some extra */
        #endif
        attr_name = rsbac_kmalloc(RSBAC_MAXNAMELEN);

        request_name[0] = (char) 0;
        target_type_name[0] = (char) 0;
        target_id_name[0] = (char) 0;
        attr_name[0] = (char) 0;
        res_name[0] = (char) 0;
        res_mods[0] = (char) 0;
        command[0] = (char) 0;
        get_request_name(request_name, request);
    #if !defined(CONFIG_RSBAC_MAINT)
        if(result == mod_result[SW_NONE])
          {
            strcat(res_mods, " GEN");
          }
    #if defined(CONFIG_RSBAC_MAC)
        if(result == mod_result[MAC])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[MAC])
              strcat(res_mods, " MAC(Softmode)");
            else
            #endif
              strcat(res_mods, " MAC");
          }
    #endif
    #if defined(CONFIG_RSBAC_FC)
        if(result == mod_result[FC])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[FC])
              strcat(res_mods, " FC(Softmode)");
            else
            #endif
              strcat(res_mods, " FC");
          }
    #endif
    #if defined(CONFIG_RSBAC_SIM)
        if(result == mod_result[SIM])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[SIM])
              strcat(res_mods, " SIM(Softmode)");
            else
            #endif
              strcat(res_mods, " SIM");
          }
    #endif
    #if defined(CONFIG_RSBAC_PM)
        if(result == mod_result[PM])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[PM])
              strcat(res_mods, " PM(Softmode)");
            else
            #endif
              strcat(res_mods, " PM");
          }
    #endif
    #if defined(CONFIG_RSBAC_MS)
        if(result == mod_result[MS])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[MS])
              strcat(res_mods, " MS(Softmode)");
            else
            #endif
              strcat(res_mods, " MS");
          }
    #endif
    #ifdef CONFIG_RSBAC_FF
        if(result == mod_result[FF])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[FF])
              strcat(res_mods, " FF(Softmode)");
            else
            #endif
              strcat(res_mods, " FF");
          }
    #endif
    #ifdef CONFIG_RSBAC_RC
        if(result == mod_result[RC])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[RC])
              strcat(res_mods, " RC(Softmode)");
            else
            #endif
              strcat(res_mods, " RC");
          }
    #endif
    #ifdef CONFIG_RSBAC_AUTH
        if(result == mod_result[AUTH])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[AUTH])
              strcat(res_mods, " AUTH(Softmode)");
            else
            #endif
              strcat(res_mods, " AUTH");
          }
    #endif
    #ifdef CONFIG_RSBAC_ACL
        if(result == mod_result[ACL])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[ACL])
              strcat(res_mods, " ACL(Softmode)");
            else
            #endif
              strcat(res_mods, " ACL");
          }
    #endif
    #ifdef CONFIG_RSBAC_CAP
        if(result == mod_result[CAP])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[CAP])
              strcat(res_mods, " CAP(Softmode)");
            else
            #endif
              strcat(res_mods, " CAP");
          }
    #endif
    #ifdef CONFIG_RSBAC_JAIL
        if(result == mod_result[JAIL])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[JAIL])
              strcat(res_mods, " JAIL(Softmode)");
            else
            #endif
              strcat(res_mods, " JAIL");
          }
    #endif
    #ifdef CONFIG_RSBAC_RES
        if(result == mod_result[RES])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[RES])
              strcat(res_mods, " RES(Softmode)");
            else
            #endif
              strcat(res_mods, " RES");
          }
    #endif
    #ifdef CONFIG_RSBAC_REG
        if(result == mod_result[REG])
          {
            #ifdef CONFIG_RSBAC_SOFTMODE_IND
            if(rsbac_ind_softmode[REG])
              strcat(res_mods, " REG(Softmode)");
            else
            #endif
              strcat(res_mods, " REG");
          }
    #endif
    #endif /* !MAINT */
        if(!res_mods[0])
          strcat(res_mods, " ADF");
        get_target_name(target_type_name, target, target_id_name, tid);
        get_attribute_name(attr_name, attr);
        get_result_name(res_name, result);
        if ((current) && (current->comm))
          {
            strncpy(command,current->comm,16);          
            command[16] = (char) 0;
          }

#ifdef CONFIG_RSBAC_RMSG
        /* if pseudo is set, its value is != 0, else -> use id */
        if (pseudo)
          {
    #ifdef CONFIG_RSBAC_SOFTMODE
            if(rsbac_softmode)
              rsbac_printk(KERN_INFO "rsbac_adf_request(): request %s, pid %u, ppid %u, prog_name %s, pseudo %u, target_type %s, tid %s, attr %s, value %u, result %s (Softmode) by%s\n",
                           request_name, caller_pid, parent_pid, command, i_attr_val.pseudo, target_type_name, target_id_name, attr_name, attr_val.dummy, res_name, res_mods);
            else
    #endif
              rsbac_printk(KERN_INFO "rsbac_adf_request(): request %s, pid %u, ppid %u, prog_name %s, pseudo %u, target_type %s, tid %s, attr %s, value %u, result %s by%s\n",
                           request_name, caller_pid, parent_pid, command, i_attr_val.pseudo, target_type_name, target_id_name, attr_name, attr_val.dummy, res_name, res_mods);
          }
        else
          {
    #ifdef CONFIG_RSBAC_SOFTMODE
            if(rsbac_softmode)
              rsbac_printk(KERN_INFO "rsbac_adf_request(): request %s, pid %u, ppid %u, prog_name %s, uid %u, target_type %s, tid %s, attr %s, value %u, result %s (Softmode) by%s\n",
                           request_name, caller_pid, parent_pid, command, owner, target_type_name, target_id_name, attr_name, attr_val.dummy, res_name, res_mods);
            else
    #endif
              rsbac_printk(KERN_INFO "rsbac_adf_request(): request %s, pid %u, ppid %u, prog_name %s, uid %u, target_type %s, tid %s, attr %s, value %u, result %s by%s\n",
                           request_name, caller_pid, parent_pid, command, owner, target_type_name, target_id_name, attr_name, attr_val.dummy, res_name, res_mods);
          }
#endif
#ifndef CONFIG_RSBAC_RMSG_EXCL
        /* only log to standard syslog, if not disabled by kernel boot parameter */
#ifdef CONFIG_RSBAC_RMSG_NOSYSLOG
	if (!rsbac_nosyslog)
#endif
	  {
            /* if pseudo is set, its value is != 0, else -> use id */
            if (pseudo)
              printk(KERN_INFO
                     "rsbac_adf_request(): request %s, pid %u, ppid %u, prog_name %s, pseudo %u, target_type %s, tid %s, attr %s, value %u, result %s by%s\n",
                     request_name, caller_pid, parent_pid, command, i_attr_val.pseudo, target_type_name, target_id_name, attr_name, attr_val.dummy, res_name, res_mods);
            else
              printk(KERN_INFO
                     "rsbac_adf_request(): request %s, pid %u, ppid %u, prog_name %s, uid %u, target_type %s, tid %s, attr %s, value %u, result %s by%s\n",
                     request_name, caller_pid, parent_pid, command, owner, target_type_name, target_id_name, attr_name, attr_val.dummy, res_name, res_mods);
          }
#endif
        /* rsbac_kfree all helper mem */
        rsbac_kfree(request_name);
        rsbac_kfree(res_name);
        rsbac_kfree(res_mods);
        rsbac_kfree(target_type_name);
        rsbac_kfree(target_id_name);
        rsbac_kfree(attr_name);
      }

/* UNDEFINED must never be returned -> change result */
    if(result == UNDEFINED)
      result = NOT_GRANTED;

/* count */
    rsbac_adf_request_count[target]++;
#ifdef CONFIG_RSBAC_XSTATS
    rsbac_adf_request_xcount[target][request]++;
#endif

/* return result */
    #ifdef CONFIG_RSBAC_SOFTMODE
    if(rsbac_softmode && !rsbac_internal)
      return DO_NOT_CARE;
    else
    #endif
    #ifdef CONFIG_RSBAC_SOFTMODE_IND
      return ret_result;
    #else
      return result; /* change for debugging! */
    #endif
  }; /* end of rsbac_adf_request_int() */


/*********************************************************************/
/* rsbac_adf_request()                                               */
/* This function is the main decision function, called from the AEF. */
/* It is a simple wrapper to the internal function, setting          */
/* ignore_module to SW_NONE.                                         */

EXPORT_SYMBOL(rsbac_adf_request);
enum rsbac_adf_req_ret_t
   rsbac_adf_request( enum  rsbac_adf_request_t     request,
                            rsbac_pid_t             caller_pid,
                      enum  rsbac_target_t          target,
                      union rsbac_target_id_t       tid,
                      enum  rsbac_attribute_t       attr,
                      union rsbac_attribute_value_t attr_val)
  {
    return(rsbac_adf_request_int(request,
                                 caller_pid,
                                 target,
                                 tid,
                                 attr,
                                 attr_val,
                                 SW_NONE));
  }

/*****************************************************************************/
/* If the request returned granted and the operation is performed,           */
/* the following function is called by the AEF to get all aci set correctly. */
/* The second instance of target specification is the new target, if one has */
/* been created, otherwise its values are ignored.                           */
/* It returns 0 on success and an error from error.h otherwise.              */

EXPORT_SYMBOL(rsbac_adf_set_attr);
int  rsbac_adf_set_attr(
                      enum  rsbac_adf_request_t     request,
                            rsbac_pid_t             caller_pid,
                      enum  rsbac_target_t          target,
                      union rsbac_target_id_t       tid,
                      enum  rsbac_target_t          new_target,
                      union rsbac_target_id_t       new_tid,
                      enum  rsbac_attribute_t       attr,
                      union rsbac_attribute_value_t attr_val)
  {
#if defined(CONFIG_RSBAC_IND_PROG_LOG) || defined(CONFIG_RSBAC_IND_USER_LOG)
    union rsbac_target_id_t i_tid;
#endif
    rsbac_uid_t owner;
    int   error = 0;
    boolean do_log = FALSE;
    boolean log_on_request = TRUE;
#ifdef CONFIG_RSBAC_IND_LOG /* only if individual logging is enabled */
    union rsbac_attribute_value_t i_attr_val,i_attr_val2;
    enum rsbac_log_level_t log_level;
#endif
#ifdef CONFIG_RSBAC_NO_DECISION_ON_NETMOUNT
    struct super_block * sb_p;
#endif

/* No attribute setting possible before init (called at boot time) */

   if (!rsbac_is_initialized())
      return(0);

/* kernel (pid 0) is ignored */
    if (   !caller_pid
        #if defined(CONFIG_RSBAC_LOG_REMOTE)
        || (caller_pid == rsbaclogd_pid)
        #endif
       )
      return(0);

/* Checking base values */
    if(   request >= R_NONE
       || target > T_NONE
       || new_target > T_NONE
       || attr > A_none)
      {
        printk(KERN_WARNING
               "rsbac_adf_set_attr(): called with invalid request, target or attribute\n");
        return(-RSBAC_EINVALIDVALUE);
      }

/* Getting basic information about this adf_set_attr-call */

    owner = RSBAC_NO_USER;
    /* only useful for real process, not idle or init */
    if (caller_pid > 1)
      {
        error = rsbac_get_owner(&owner);
        if(error)
          {
            printk(KERN_DEBUG
                   "rsbac_adf_set_attr(): caller_pid %i, RSBAC not initialized, returning 0",
                   caller_pid);
            return(0);      /* Startup-Sequence (see above) */
          }
      }
    else /* caller_pid = 1  -> init -> owner = root */
      owner = 0;

/*************************************************/
/* General work for all modules - before modules */
#ifdef CONFIG_RSBAC_NO_DECISION_ON_NETMOUNT
    if (   (   (target == T_FILE)
            || (target == T_DIR)
            || (target == T_FIFO)
            || (target == T_SYMLINK)
           )
        && ((sb_p = rsbac_get_super_block(tid.file.device)))
        && (   (sb_p->s_magic == NFS_SUPER_MAGIC)
            || (sb_p->s_magic == CODA_SUPER_MAGIC)
            || (sb_p->s_magic == NCP_SUPER_MAGIC)
            || (sb_p->s_magic == SMB_SUPER_MAGIC)
           )
       )
      {
        error = 0;
        goto log;
      }
#endif

/**********************************************************/
/* calling all decision modules, building a common result */


#ifdef CONFIG_RSBAC_DEBUG
/* first, check for valid request/target combination      */
error |= rsbac_adf_set_attr_check(request,
                                  caller_pid,
                                  target,
                                  tid,
                                  new_target,
                                  new_tid,
                                  attr,
                                  attr_val,
                                  owner);
if(error)
  goto general_work;
#endif

#if !defined(CONFIG_RSBAC_MAINT)
/******* MAC ********/
#if defined(CONFIG_RSBAC_MAC)
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_mac)
#endif
  error |= rsbac_adf_set_attr_mac(request,
                                  caller_pid,
                                  target,
                                  tid,
                                  new_target,
                                  new_tid,
                                  attr,
                                  attr_val,
                                  owner);
#endif  /* MAC */

/******* FC ********/
#ifdef CONFIG_RSBAC_FC
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_fc)
#endif
  error |= rsbac_adf_set_attr_fc (request,
                                  caller_pid,
                                  target,
                                  tid,
                                  new_target,
                                  new_tid,
                                  attr,
                                  attr_val,
                                  owner);
#endif  /* FC */

/******* SIM ********/
#ifdef CONFIG_RSBAC_SIM
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_sim)
#endif
  error |= rsbac_adf_set_attr_sim(request,
                                  caller_pid,
                                  target,
                                  tid,
                                  new_target,
                                  new_tid,
                                  attr,
                                  attr_val,
                                  owner);
#endif  /* SIM */

/******* PM ********/
#ifdef CONFIG_RSBAC_PM
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_pm)
#endif
  error |= rsbac_adf_set_attr_pm (request,
                                  caller_pid,
                                  target,
                                  tid,
                                  new_target,
                                  new_tid,
                                  attr,
                                  attr_val,
                                  owner);
#endif  /* PM */

/******* MS ********/
#ifdef CONFIG_RSBAC_MS
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_ms)
#endif
  error |= rsbac_adf_set_attr_ms (request,
                                  caller_pid,
                                  target,
                                  tid,
                                  new_target,
                                  new_tid,
                                  attr,
                                  attr_val,
                                  owner);
#endif  /* MS */

/******* FF ********/
#if 0 /* Nothing to do in there */
#ifdef CONFIG_RSBAC_FF
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_ff)
#endif
  error |= rsbac_adf_set_attr_ff (request,
                                  caller_pid,
                                  target,
                                  tid,
                                  new_target,
                                  new_tid,
                                  attr,
                                  attr_val,
                                  owner);
#endif  /* FF */
#endif /* 0 */

/******* RC ********/
#ifdef CONFIG_RSBAC_RC
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_rc)
#endif
  error |= rsbac_adf_set_attr_rc (request,
                                  caller_pid,
                                  target,
                                  tid,
                                  new_target,
                                  new_tid,
                                  attr,
                                  attr_val,
                                  owner);
#endif  /* RC */

/****** AUTH *******/
#ifdef CONFIG_RSBAC_AUTH
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_auth)
#endif
  error |= rsbac_adf_set_attr_auth(request,
                                   caller_pid,
                                   target,
                                   tid,
                                   new_target,
                                   new_tid,
                                   attr,
                                   attr_val,
                                   owner);
#endif  /* AUTH */

/****** ACL *******/
#ifdef CONFIG_RSBAC_ACL
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_acl)
#endif
  error |= rsbac_adf_set_attr_acl (request,
                                   caller_pid,
                                   target,
                                   tid,
                                   new_target,
                                   new_tid,
                                   attr,
                                   attr_val,
                                   owner);
#endif  /* ACL */

/****** CAP *******/
#ifdef CONFIG_RSBAC_CAP
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_cap)
#endif
  error |= rsbac_adf_set_attr_cap (request,
                                   caller_pid,
                                   target,
                                   tid,
                                   new_target,
                                   new_tid,
                                   attr,
                                   attr_val,
                                   owner);
#endif  /* CAP */

/****** JAIL *******/
#ifdef CONFIG_RSBAC_JAIL
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_jail)
#endif
  error |= rsbac_adf_set_attr_jail(request,
                                   caller_pid,
                                   target,
                                   tid,
                                   new_target,
                                   new_tid,
                                   attr,
                                   attr_val,
                                   owner);
#endif  /* JAIL */

/****** RES *******/
#ifdef CONFIG_RSBAC_RES
#ifdef CONFIG_RSBAC_SWITCH
if (rsbac_switch_res)
#endif
  error |= rsbac_adf_set_attr_res (request,
                                   caller_pid,
                                   target,
                                   tid,
                                   new_target,
                                   new_tid,
                                   attr,
                                   attr_val,
                                   owner);
#endif  /* RES */

/****** REG *******/
#ifdef CONFIG_RSBAC_REG
  error |= rsbac_adf_set_attr_reg (request,
                                   caller_pid,
                                   target,
                                   tid,
                                   new_target,
                                   new_tid,
                                   attr,
                                   attr_val,
                                   owner);
#endif  /* REG */
#endif /* !MAINT */

/* General work for all modules (after set_attr call) */
#ifdef CONFIG_RSBAC_DEBUG
general_work:
#endif
    switch(request)
      {
        /* remove deleted item from rsbac data */
        case R_DELETE :
            switch (target)
              {
                case T_FILE:
                case T_FIFO:
                case T_SYMLINK:
                  /* Only remove file/fifo target on deletion of last link */
                  if (   (attr == A_nlink)
                      && (attr_val.nlink > 1)
                     )
                     break;
                  /* fall through */
                case T_DIR:
                  rsbac_remove_target(target,tid);
                  break;
                case T_IPC:
                  /* shm removal delayed and removed directly, when destroyed */
                  if(tid.ipc.type != I_shm)
                    rsbac_remove_target(target,tid);
                  break;
                default:
                  break;
              }
            break;

#if defined(CONFIG_RSBAC_IND_PROG_LOG)
        case R_CLONE:
            switch (target)
              {
                case T_PROCESS:
                  /* get program based log from old process */
                  if (rsbac_get_attr(GEN,
                                     target,
                                     tid,
                                     A_log_program_based,
                                     &i_attr_val,
                                     FALSE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_request(): rsbac_get_attr() for log_program_based returned error!\n");
                    }
                  else
                    { /* only set, of not default value 0 */
                      if(i_attr_val.log_program_based)
                        {
                          /* set program based log for new process */
                          if (rsbac_set_attr(GEN, new_target,
                                             new_tid,
                                             A_log_program_based,
                                             i_attr_val))
                            {
                              printk(KERN_WARNING
                                     "rsbac_adf_request(): rsbac_set_attr() for log_program_based returned error!\n");
                            }
                        }
                    }
                  break;

                default:
                  break;
              }
            break;
#endif /* CONFIG_RSBAC_IND_PROG_LOG */

#ifdef CONFIG_RSBAC_NET_OBJ
        case R_CLOSE:
            switch (target)
              {
                case T_NETOBJ:
                  rsbac_remove_target(target,tid);
                  break;

                default:
                  break;
              }
            break;
#endif /* CONFIG_RSBAC_IND_PROG_LOG */

#if defined(CONFIG_RSBAC_IND_PROG_LOG)
        case R_EXECUTE :
            switch (target)
              {
                case T_FILE:
                  /* get program based log from file */
                  if (rsbac_get_attr(GEN,
                                     target,
                                     tid,
                                     A_log_program_based,
                                     &i_attr_val,
                                     FALSE))
                    {
                      printk(KERN_WARNING
                             "rsbac_adf_request(): rsbac_get_attr() for log_program_based returned error!\n");
                    }
                  else
                    {
                      /* set program based log for process */
                      i_tid.process = caller_pid;
                      if (rsbac_set_attr(GEN, T_PROCESS,
                                         i_tid,
                                         A_log_program_based,
                                         i_attr_val))
                        {
                          printk(KERN_WARNING
                                 "rsbac_adf_request(): rsbac_set_attr() for log_program_based returned error!\n");
                        }
                    }
                  break;

                default:
                  break;
              }
            break;
#endif /* CONFIG_RSBAC_IND_PROG_LOG */

        default:
            break;
      }

#if defined(CONFIG_RSBAC_DEBUG) && defined(CONFIG_RSBAC_NET)
    if(    rsbac_debug_adf_net
       && (   (target == T_NETDEV)
           || (target == T_NETTEMP)
           || (target == T_NETOBJ)
          )
      )
      do_log = TRUE;
#endif

/* log based on process owner */
#ifdef CONFIG_RSBAC_IND_USER_LOG
    i_tid.user = owner;
    if (rsbac_get_attr(GEN,
                       T_USER,
                       i_tid,
                       A_log_user_based,
                       &i_attr_val,
                       FALSE))
      {
        printk(KERN_WARNING
               "rsbac_adf_request(): rsbac_get_attr() for log_user_based returned error!\n");
      }
    else
      {
        if(((rsbac_request_vector_t) 1 << request) & i_attr_val.log_user_based) 
          do_log = TRUE;
      }
#endif /* CONFIG_RSBAC_IND_USER_LOG */

/* log based on program */
#ifdef CONFIG_RSBAC_IND_PROG_LOG
    if(!do_log)
      {
        i_tid.process = caller_pid;
        if (rsbac_get_attr(GEN,
                           T_PROCESS,
                           i_tid,
                           A_log_program_based,
                           &i_attr_val,
                           FALSE))
          {
            printk(KERN_WARNING
                   "rsbac_adf_request(): rsbac_get_attr() for log_program_based returned error!\n");
          }
        else
          {
            if(((rsbac_request_vector_t) 1 << request) & i_attr_val.log_program_based) 
              do_log = TRUE;
          }
      }
#endif /* CONFIG_RSBAC_IND_PROG_LOG */


/* logging request on info level, if requested by file/dir/dev attributes */
/* log_array_low/high, or, if that is requested, if enabled for this request */
/* type (attributes state level, or that request based level is to be taken) */
/* loglevel 2: log everything */
/* loglevel 1: log, if denied */
/* loglevel 0: log nothing */

#ifdef CONFIG_RSBAC_IND_LOG /* only if individual logging is enabled */
    /* if file/dir/dev, depend log on log_arrays */
    /* (but not for file.device = 0) */
    /* log_on_request is TRUE */
    if(!do_log)
      {
        if(   (   (   (target == T_FILE)
                   || (target == T_DIR)
                   || (target == T_FIFO)
                   || (target == T_SYMLINK)
                  )
               && tid.file.device
              )
           || (target == T_DEV)
          )
          {
            if (rsbac_get_attr(GEN,
                               target,
                               tid,
                               A_log_array_low,
                               &i_attr_val,
                               FALSE))
              {
                printk(KERN_WARNING
                       "rsbac_adf_set_attr(): rsbac_get_attr() for log_array_low returned error!\n");
              }
            else
              {
                if (rsbac_get_attr(GEN,
                                   target,
                                   tid,
                                   A_log_array_high,
                                   &i_attr_val2,
                                   FALSE))
                  {
                    printk(KERN_WARNING
                           "rsbac_adf_set_attr(): rsbac_get_attr() for log_array_high returned error!\n");
                  }
                else
                  { /* ll = low-bit for request | (high-bit for request as bit 1) */
                    log_level =   ((i_attr_val.log_array_low   >> request) & 1)
                              | ( ((i_attr_val2.log_array_high >> request) & 1) << 1);
                    if (   log_level == LL_full
                        || (   log_level == LL_denied
                            && error) )
                      {
                        do_log = TRUE;
                      }
                    if(log_level != LL_request)
                      log_on_request = FALSE;
                  }
              }
          }
      }
#endif /* CONFIG_RSBAC_IND_LOG */

#ifdef CONFIG_RSBAC_NO_DECISION_ON_NETMOUNT
log:
#endif
    /* if enabled, try request based log level */
    if (log_on_request
        && (   rsbac_log_levels[request][target] == LL_full
            || (   rsbac_log_levels[request][target] == LL_denied
                && error) ) )
      do_log = TRUE;

    if(do_log)
      {
        char * request_name;
        char * target_type_name;
        char * new_target_type_name;
        char * target_id_name;
        char * new_target_id_name;
        char * attr_name;

        /* rsbac_kmalloc all memory */
        request_name = rsbac_kmalloc(RSBAC_MAXNAMELEN);
        target_type_name = rsbac_kmalloc(RSBAC_MAXNAMELEN);
        new_target_type_name = rsbac_kmalloc(RSBAC_MAXNAMELEN);
        #ifdef CONFIG_RSBAC_LOG_FULL_PATH
        target_id_name
         = rsbac_kmalloc(CONFIG_RSBAC_MAX_PATH_LEN + RSBAC_MAXNAMELEN);
        new_target_id_name
         = rsbac_kmalloc(CONFIG_RSBAC_MAX_PATH_LEN + RSBAC_MAXNAMELEN);
           /* max. path name len + some extra */
        #else
        target_id_name = rsbac_kmalloc(2 * RSBAC_MAXNAMELEN);
        new_target_id_name = rsbac_kmalloc(2 * RSBAC_MAXNAMELEN);
           /* max. file name len + some extra */
        #endif
        attr_name = rsbac_kmalloc(RSBAC_MAXNAMELEN);

        /* Getting basic information about this request */
        request_name[0] = (char) 0;
        target_type_name[0] = (char) 0;
        target_id_name[0] = (char) 0;
        new_target_type_name[0] = (char) 0;
        new_target_id_name[0] = (char) 0;
        attr_name[0] = (char) 0;
        get_request_name(request_name, request);
        get_target_name(target_type_name, target, target_id_name, tid);
        get_target_name(new_target_type_name, new_target,
                        new_target_id_name, new_tid);
        get_attribute_name(attr_name, attr);

#ifdef CONFIG_RSBAC_RMSG
        /* Logging without pseudo or user id, see previous request log */
        rsbac_printk(KERN_INFO
                 "rsbac_adf_set_attr(): request %s, pid %u, target_type %s, tid %s, new_target_type %s, new_tid %s, attr %s, value %u, error %i\n",
                   request_name, (u_int) caller_pid, target_type_name, target_id_name,
                   new_target_type_name, new_target_id_name, attr_name, attr_val.dummy, error);
#endif
#ifndef CONFIG_RSBAC_RMSG_EXCL
        /* Logging without pseudo or user id, see previous request log */
#ifdef CONFIG_RSBAC_RMSG_NOSYSLOG
	if (!rsbac_nosyslog)
#endif
          printk(KERN_INFO
                 "rsbac_adf_set_attr(): request %s, pid %u, target_type %s, tid %s, new_target_type %s, new_tid %s, attr %s, value %u, error %i\n",
                 request_name, (u_int) caller_pid, target_type_name, target_id_name,
                 new_target_type_name, new_target_id_name, attr_name, attr_val.dummy, error);
#endif
        /* rsbac_kfree all helper mem */
        rsbac_kfree(request_name);
        rsbac_kfree(target_type_name);
        rsbac_kfree(new_target_type_name);
        rsbac_kfree(target_id_name);
        rsbac_kfree(new_target_id_name);
        rsbac_kfree(attr_name);
      }

/* count */
    rsbac_adf_set_attr_count[target]++;
#ifdef CONFIG_RSBAC_XSTATS
    rsbac_adf_set_attr_xcount[target][request]++;
#endif

    return(error);
  }; /* end of rsbac_adf_set_attr() */


/****************
 *
 * Secure Delete
 *
 ****************/

#ifdef CONFIG_RSBAC_SECDEL

/* open_by_dentry */
/* This is done by hand (copy from rsbac_read_open), because system calls */
/* are currently blocked by rsbac */

static int open_by_dentry(struct dentry * file_dentry_p, struct file * file_p)
  {
    int tmperr;

    if ( !(S_ISREG(file_dentry_p->d_inode->i_mode)) )
      { /* this is not a file! -> error! */
        printk(KERN_WARNING
               "open_by_dentry(): expected file is not a file!\n");
        return (-RSBAC_EWRITEFAILED);
      }
    /* Now we fill the file structure, and */
    /* if there is an open func, use it, otherwise ignore */
    if ((tmperr = init_private_file(file_p, file_dentry_p, O_WRONLY)))
      {
        printk(KERN_WARNING
               "open_by_dentry(): could not open file!\n");
        return (-RSBAC_EWRITEFAILED);
      }
    /* Without a write function we get into troubles -> error */
    if ((!file_p->f_op) || (!file_p->f_op->write))
      {
        printk(KERN_WARNING
               "open_by_dentry(): file write function missing!\n");
        if (file_p->f_op && file_p->f_op->release)
          file_p->f_op->release(file_dentry_p->d_inode,file_p);
        return(-RSBAC_EWRITEFAILED);
      }
    /* trying to get write access                                   */
/*
    if (get_write_access(file_dentry_p->d_inode))
      {
        printk(KERN_WARNING
               "rsbac_write_open(): could not get write access!\n");
        if (file_p->f_op->release)
          file_p->f_op->release(file_dentry_p->d_inode,file_p);
        return(-RSBAC_EWRITEFAILED);
      }
*/
    return 0;
  }

/*
 **********************
 * Secure File Truncation
 */

EXPORT_SYMBOL(rsbac_sec_trunc);
int rsbac_sec_trunc(struct dentry * dentry_p,
                    loff_t new_len, loff_t old_len)
  {
#if defined(CONFIG_RSBAC_MAINT)
    return 0;
#else
    int                           err = 0;
    boolean                       need_overwrite = FALSE;

    /* security checks */
    if(   !dentry_p
       || !dentry_p->d_inode)
      return -RSBAC_EINVALIDPOINTER;
    if(!S_ISREG(dentry_p->d_inode->i_mode))
      return -RSBAC_EINVALIDTARGET;
    if(new_len >= old_len)
      return 0;


    /******* MAC ********/
    #ifdef CONFIG_RSBAC_MAC
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_mac)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_mac(dentry_p);
    #endif  /* MAC */

    /******* FC ********/
    #ifdef CONFIG_RSBAC_FC
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_fc)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_fc(dentry_p);
    #endif  /* FC */

    /******* SIM ********/
    #ifdef CONFIG_RSBAC_SIM
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_sim)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_sim(dentry_p);
    #endif  /* SIM */

    /******* PM ********/
    #ifdef CONFIG_RSBAC_PM
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_pm)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_pm(dentry_p);
    #endif  /* PM */

    /******* MS ********/
    #ifdef CONFIG_RSBAC_MS
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_ms)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_ms(dentry_p);
    #endif  /* MS */

    /******* FF ********/
    #ifdef CONFIG_RSBAC_FF
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_ff)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_ff(dentry_p);
    #endif  /* FF */

    /******* RC ********/
    #ifdef CONFIG_RSBAC_RC
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_rc)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_rc(dentry_p);
    #endif  /* RC */

    /****** AUTH *******/
    #ifdef CONFIG_RSBAC_AUTH
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_auth)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_auth(dentry_p);
    #endif  /* AUTH */

    /****** ACL *******/
    #ifdef CONFIG_RSBAC_ACL
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_acl)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_acl(dentry_p);
    #endif  /* ACL */

    /****** CAP *******/
    #ifdef CONFIG_RSBAC_CAP
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_cap)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_cap(dentry_p);
    #endif  /* CAP */

    /****** JAIL *******/
    #ifdef CONFIG_RSBAC_JAIL
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_jail)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_jail(dentry_p);
    #endif  /* JAIL */

    /****** RES *******/
    #ifdef CONFIG_RSBAC_RES
    #ifdef CONFIG_RSBAC_SWITCH
    if (rsbac_switch_res)
    #endif
      /* no need to call module, if already need_overwrite */
      if(!need_overwrite)
        need_overwrite = rsbac_need_overwrite_res(dentry_p);
    #endif  /* RES */

    /****** REG *******/
    #ifdef CONFIG_RSBAC_REG
    if(!need_overwrite)
      need_overwrite = rsbac_need_overwrite_reg(dentry_p);
    #endif  /* REG */

    if(need_overwrite)
      {
        char    * buffer;
        boolean   vmalloc_used;
        struct file                   file;
        int                           tmperr = 0;
        int                           len;
        mm_segment_t                  oldfs;

        buffer = rsbac_vkmalloc(RSBAC_SEC_DEL_CHUNK_SIZE, &vmalloc_used);
        if(!buffer)
          return -RSBAC_ENOMEM;

#ifdef CONFIG_RSBAC_DEBUG
        if(rsbac_debug_write)
          printk(KERN_DEBUG
                 "rsbac_sec_trunc(): zeroing of file %lu on device %02u:%02u from byte %lu to %lu!\n",
                 dentry_p->d_inode->i_ino,
                 MAJOR(dentry_p->d_inode->i_dev),
                 MINOR(dentry_p->d_inode->i_dev),
                 (u_long) new_len,
                 (u_long) old_len-1);
#endif
        /* open */
        err = open_by_dentry(dentry_p, &file);
        if(err)
          {
            rsbac_vkfree(buffer, vmalloc_used);
            return(err);
          }

#ifdef CONFIG_RSBAC_DEBUG
        if(rsbac_debug_write)
          printk(KERN_DEBUG
                 "rsbac_sec_trunc(): file %lu on device %02u:%02u is open, starting to write!\n",
                 dentry_p->d_inode->i_ino,
                 MAJOR(dentry_p->d_inode->i_dev),
                 MINOR(dentry_p->d_inode->i_dev));
#endif

        /* OK, now we can start writing */

        /* Set current user space to kernel space, because write() reads
         * from user space
         */
        oldfs = get_fs();
        set_fs(KERNEL_DS);

        if(file.f_op->llseek)
          file.f_op->llseek(&file,new_len,0);
        else
          { /* taken from fs/read_write.c */
            file.f_pos = new_len;
            file.f_reada = 0;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,14)) && (LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0))
            file.f_version = ++global_event;
#else
            file.f_version = ++event;
#endif
          }
        memset(buffer,0,RSBAC_SEC_DEL_CHUNK_SIZE);

        while(new_len < old_len)
          {
            len = rsbac_min(RSBAC_SEC_DEL_CHUNK_SIZE, old_len-new_len);
            tmperr = file.f_op->write(&file,
                                     buffer,
                                     len,
                                     &file.f_pos);
            /* if none written, end of file is reached -> complain and return */
            if (tmperr <= 0)
              {
                printk(KERN_WARNING
                       "rsbac_sec_trunc(): write error on file!\n");
                err = -RSBAC_EWRITEFAILED;
                goto out;
              }
            new_len += tmperr;
          }

        out:
        /* Set current user space back to user space, because read() writes */
        /* to user space */
        set_fs(oldfs);

#ifdef CONFIG_RSBAC_DEBUG
        if(rsbac_debug_write)
          printk(KERN_DEBUG
                 "rsbac_sec_trunc(): syncing file %lu on device %02u:%02u!\n",
                 dentry_p->d_inode->i_ino,
                 MAJOR(dentry_p->d_inode->i_dev),
                 MINOR(dentry_p->d_inode->i_dev));
#endif

        #if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
        if (file.f_op->fsync)
          err = file.f_op->fsync(&file,dentry_p);
        #else
//        err = fsync_inode_buffers(dentry_p->d_inode);
        err = fsync_inode_data_buffers(dentry_p->d_inode);
        #endif
          
/*
        if (file.f_op->flush)
          file.f_op->flush(&file);
        if (file.f_op->release)
          file.f_op->release(dentry_p->d_inode,&file);
*/
        /* End of write access                     */
//        put_write_access(dentry_p->d_inode);

        rsbac_vkfree(buffer, vmalloc_used);
      }

    /* Ready. */
    return(err);

#endif /* else of MAINT */
  }

EXPORT_SYMBOL(rsbac_sec_del);
int rsbac_sec_del(struct dentry * dentry_p)
  {
    return(rsbac_sec_trunc(dentry_p,
                           0,
                           dentry_p->d_inode->i_size));
  }

#else /* no SECDEL */
EXPORT_SYMBOL(rsbac_sec_trunc);
int rsbac_sec_trunc(struct dentry * dentry_p,
                    loff_t new_len, loff_t old_len)
  {
    return 0;
  }
EXPORT_SYMBOL(rsbac_sec_del);
int rsbac_sec_del(struct dentry * dentry_p)
  {
    return 0;
  }
#endif /* SECDEL */

#ifdef CONFIG_RSBAC_SYM_REDIR
void rsbac_symlink_redirect(struct dentry * dentry_p, char * name)
  {
    int                            err;
    union rsbac_target_id_t        i_tid;
    union rsbac_target_id_t        i_tid2;
    union rsbac_attribute_value_t  i_attr_val;

    if(!name || !dentry_p || !dentry_p->d_inode)
      return;
    if (!rsbac_is_initialized())
      return;

    i_tid.symlink.device = dentry_p->d_inode->i_dev;
    i_tid.symlink.inode = dentry_p->d_inode->i_ino;
    i_tid.symlink.dentry_p = dentry_p;
    if(!S_ISLNK(dentry_p->d_inode->i_mode))
      {
        printk(KERN_DEBUG
               "rsbac_symlink_redirect(): called for non-symlink inode %u on dev %02u:%02u!\n",
               i_tid.symlink.inode,
               MAJOR(i_tid.symlink.device), MINOR(i_tid.symlink.device) );
        return;
      }

#ifdef CONFIG_RSBAC_DEBUG
    if (rsbac_debug_aef)
      printk(KERN_DEBUG
             "rsbac_symlink_redirect(): called for symlink inode %u on dev %02u:%02u!\n",
             i_tid.symlink.inode,
             MAJOR(i_tid.symlink.device), MINOR(i_tid.symlink.device) );
#endif

#ifdef CONFIG_RSBAC_SYM_REDIR_UID
    if ((err = rsbac_get_attr(GEN,
                              T_SYMLINK,
                              i_tid,
                              A_symlink_add_uid,
                              &i_attr_val,
                              FALSE) ))
      {
        printk(KERN_WARNING
               "rsbac_symlink_redirect(): rsbac_get_attr() for symlink_add_uid returned error %i!\n",
               err);
        return;  /* something weird happened */
      }
    if(i_attr_val.symlink_add_uid)
      {
        rsbac_uid_t user;
        u_int len;

        len = strlen(name);
        while(   len
              && (name[len-1] >= '0')
              && (name[len-1] <= '9')
             )
          len--;
        if(len > (PAGE_SIZE - 20))
          return;
        if(!rsbac_get_owner(&user))
          ulongtostr(name+len, user);
      }
#endif

#ifdef CONFIG_RSBAC_SYM_REDIR_MAC
    if ((err = rsbac_get_attr(GEN,
                              T_SYMLINK,
                              i_tid,
                              A_symlink_add_mac_level,
                              &i_attr_val,
                              FALSE) ))
      {
        printk(KERN_WARNING
               "rsbac_symlink_redirect(): rsbac_get_attr() for symlink_add_mac_level returned error %i!\n",
               err);
        return;  /* something weird happened */
      }
    if(i_attr_val.symlink_add_mac_level)
      {
        u_int len;

        len = strlen(name);
        while(   len
              && (   (   (name[len-1] >= '0')
                      && (name[len-1] <= '9')
                     )
#ifdef CONFIG_RSBAC_SYM_REDIR_MAC_CAT
                  || (name[len-1] == ':')
#endif
                 )
             )
          len--;
#ifdef CONFIG_RSBAC_SYM_REDIR_MAC_CAT
        if(len > (PAGE_SIZE - 85))
#else
        if(len > (PAGE_SIZE - 20))
#endif
          return;

        i_tid2.process = current->pid;
        if ((err = rsbac_get_attr(MAC,
                                  T_PROCESS,
                                  i_tid2,
                                  A_current_sec_level,
                                  &i_attr_val,
                                  FALSE) ))
          {
            printk(KERN_WARNING
                   "rsbac_symlink_redirect(): rsbac_get_attr() for current_sec_level returned error %i!\n",
                   err);
            return;  /* something weird happened */
          }

#ifdef CONFIG_RSBAC_SYM_REDIR_MAC_CAT
        len+=sprintf(name+len, "%u:", i_attr_val.current_sec_level);
#else
        len+=sprintf(name+len, "%u", i_attr_val.current_sec_level);
#endif

#ifdef CONFIG_RSBAC_SYM_REDIR_MAC_CAT
        if ((err = rsbac_get_attr(MAC,
                                  T_PROCESS,
                                  i_tid2,
                                  A_mac_curr_categories,
                                  &i_attr_val,
                                  FALSE) ))
          {
            printk(KERN_WARNING
                   "rsbac_symlink_redirect(): rsbac_get_attr() for mac_curr_categories returned error %i!\n",
                   err);
            return;  /* something weird happened */
          }
        u64tostrmac(name+len, i_attr_val.mac_categories);
#endif
      }
#endif

#ifdef CONFIG_RSBAC_SYM_REDIR_RC
    if ((err = rsbac_get_attr(GEN,
                              T_SYMLINK,
                              i_tid,
                              A_symlink_add_rc_role,
                              &i_attr_val,
                              FALSE) ))
      {
        printk(KERN_WARNING
               "rsbac_symlink_redirect(): rsbac_get_attr() for symlink_add_rc_role returned error %i!\n",
               err);
        return;  /* something weird happened */
      }
    if(i_attr_val.symlink_add_rc_role)
      {
        u_int len;

        len = strlen(name);
        while(   len
              && (name[len-1] >= '0')
              && (name[len-1] <= '9')
             )
          len--;
        if(len > (PAGE_SIZE - 20))
          return;

        i_tid2.process = current->pid;
        if ((err = rsbac_get_attr(RC,
                                  T_PROCESS,
                                  i_tid2,
                                  A_rc_role,
                                  &i_attr_val,
                                  FALSE) ))
          {
            printk(KERN_WARNING
                   "rsbac_symlink_redirect(): rsbac_get_attr() for rc_role returned error %i!\n",
                   err);
            return;  /* something weird happened */
          }

        ulongtostr(name+len, i_attr_val.rc_role);
      }
#endif

    return;
  }
#endif

#ifdef CONFIG_RSBAC_ALLOW_DAC_DISABLE_PART
int rsbac_dac_part_disabled(struct dentry * dentry_p)
  {
    int                            err;
    enum  rsbac_target_t           i_target;
    union rsbac_target_id_t        i_tid;
    union rsbac_attribute_value_t  i_attr_val;

    if(!dentry_p || !dentry_p->d_inode || !dentry_p->d_inode->i_dev || !rsbac_is_initialized())
      return FALSE;

    if(S_ISREG(dentry_p->d_inode->i_mode))
      i_target = T_FILE;
    else
    if(S_ISDIR(dentry_p->d_inode->i_mode))
      i_target = T_DIR;
    else
    if(S_ISFIFO(dentry_p->d_inode->i_mode))
      i_target = T_FIFO;
    else
    if(S_ISLNK(dentry_p->d_inode->i_mode))
      i_target = T_SYMLINK;
    else
      return FALSE;

    i_tid.file.device = dentry_p->d_inode->i_dev;
    i_tid.file.inode = dentry_p->d_inode->i_ino;
    i_tid.file.dentry_p = dentry_p;

#ifdef CONFIG_RSBAC_DEBUG
    if (rsbac_debug_aef)
      printk(KERN_DEBUG
             "rsbac_dac_part_disable(): called for dentry_p->d_inode %u on dev %02u:%02u, dentry_p %p!\n",
             i_tid.file.inode,
             MAJOR(i_tid.file.device), MINOR(i_tid.file.device),
             i_tid.file.dentry_p );
#endif

    if ((err = rsbac_get_attr(GEN,
                              i_target,
                              i_tid,
                              A_linux_dac_disable,
                              &i_attr_val,
                              TRUE) ))
      {
        printk(KERN_WARNING
               "rsbac_dac_part_disable(): rsbac_get_attr() for linux_dac_disable returned error %i!\n",
               err);
        return FALSE;  /* something weird happened */
      }
    if(i_attr_val.linux_dac_disable == LDD_true)
      return TRUE;
    else
      return FALSE;
  }
#endif

/* end of rsbac/adf/main.c */
