/* $Id: lr.c,v 1.16 2004/02/21 08:41:28 yumo Exp $ */
/* LR: A virtual ethernet device driver for Linux. 
 *
 * This device driver is implementation of Redundancy of
 * Link Segment (Yumo's original protocol).
 *
 * Author: Yumo (Katsuyuki Yumoto) 2001-2004
 *         yumo@st.rim.or.jp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <linux/config.h>
#include <linux/version.h>

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/string.h>
#include <linux/ptrace.h>
#include <linux/if_ether.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <net/sock.h>
#endif
#include <asm/system.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <linux/in.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/lp.h>
#include <linux/init.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/inetdevice.h>
#include <linux/skbuff.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <linux/tqueue.h>
#endif
#include <linux/ioport.h>
#include <asm/bitops.h>
#include <asm/irq.h>
#include <asm/byteorder.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
#include <asm/spinlock.h>
#else
#include <linux/spinlock.h>
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,5)
#include "miicompat.h"
#else
#include <linux/mii.h>
#endif
#include <linux/proc_fs.h>

#include <linux/timer.h>

#include "lrlink.h"

int nlr = 1;

MODULE_PARM(nlr, "1-" __MODULE_STRING(MAX_PHYIF) "i");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
MODULE_PARM_DESC(nlr, "lr" " number of groups (1-MAX_PHYIF)");
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
/* 
 * Linux module subsystem does not provide module handling
 * facility by corresponding device name.
 * I don't want to make lr driver depend on module name
 * but it depends on device name (e.g. eth0).
 * However there are no way to lock/unlock the module by device name.
 * Therefore I need to convert device name into module name, and
 * then lock/unlock the module...
 */
/* find_module() is exported by patch */
extern struct module *find_module(const char *name);

#define MAX_MODNAME 16
#endif

/* interface name length */
#define MAX_NAME 7

static u16 get_link_stat(struct net_device *dev)
{
	static int (* ioctl)(struct net_device *, struct ifreq *, int);
	struct ifreq ifr;
	struct mii_ioctl_data *data = (struct mii_ioctl_data *)&ifr.ifr_data;
		
	/* Status register is mapped at address 1 in 
	 * MII management register set space.
	 * (IEEE Std 802.3, 1998)*/
	data->reg_num = MII_BMSR; 

	ioctl = dev->do_ioctl;
	if (ioctl && (ioctl(dev, &ifr, SIOCGMIIPHY) == 0)) {
		/* m->val_out contains status
		 * bit 2: 1 = link is up (BMSR_LSTATUS)
		 * bit 5: 1 = auto-negotiation process completed (BMSR_ANEGCOMPLETE)
		 */
		return data->val_out & (BMSR_LSTATUS | BMSR_ANEGCOMPLETE);
	}
	else {
		return BMSR_LSTATUS | BMSR_ANEGCOMPLETE;  /* assume link up... */
	}
}

static u16 get_group_link_stat(struct net_device *dev)
{
	struct lr_private *vp;
	struct group_var *gv;
	int i;
	u16 ret = 0;

	if (!dev)
		return 0;

	vp = (struct lr_private *)dev->priv;
	gv = &(vp->group->var);

	for (i = 0; i < MAX_PHYIF; i++){
		if (gv->port_list[i]){
			ret |= get_link_stat(lr_ports[gv->port_list[i]-1].dev);
		}
	}
	return ret;
}

static int dev_to_port_num(struct net_device *dev)
{
	int i;
	int pn;

	for (i = 0, pn = 0; i < MAX_PHYIF; i++){
		if (lr_ports[i].dev == dev){
			pn = i + 1;
			break;
		}
	}

	return pn;
}

static int hash_mac_addr(u8 *addr)
{
	u32 hash;

	hash = *((u32 *)addr);
	hash ^= hash >>16;
	hash ^= hash >> 8;
	hash %= 32;

	return hash;
}

static struct lr_group *lr_get_group_by_num(int n)
{
	if (n < 1 || MAX_PHYIF < n)
		return NULL;

	return lr_groups + n - 1;
}

static struct lr_group *lr_get_head_group(void)
{
	return lr_groups;
}

static struct lr_group *lr_get_next_group(struct lr_group *g)
{
	int n = ++g - lr_groups;
	if (n < 0 || MAX_PHYIF <= n)
		return NULL;
	return g;
}

static struct lr_port *lr_get_port_by_num(int n)
{
	if (n < 1 || MAX_PHYIF < n)
		return NULL;

	return lr_ports + n - 1;
}

static struct lr_port *lr_get_head_port(void)
{
	return lr_ports;
}

static struct lr_port *lr_get_next_port(struct lr_port *p)
{
	int n = ++p - lr_ports;
	if (n < 0 || MAX_PHYIF <= n)
		return NULL;
	return p;
}

/********************************************************
 *
 *   Link Redundancy Configuration Protocol
 *
 ********************************************************/

/* Group level functions */
static u16 lr_search_for_subg(u8 *da, struct group_var *gv)
{
	int i;

	struct dist_info *h, *p;
	u16 ret = 0;

	i = hash_mac_addr(da);

	/* we need to lock hash list here */
	spin_lock(&gv->dist_lock);

	h = gv->dist_info[i];
	/* then do access */
	for (p = h; p; p = p->next){
		if (!memcmp(da, p->m_addr, ETH_ALEN)){
			ret = p->sgid;
			break;
		}
	}

	/* we need to unlock hash list here */
	spin_unlock(&gv->dist_lock);

	return ret;
}


/* Sub group level functions */
static void
lr_update_distinfo(u8 *sa, int sgid, int is_bfu, u16 age, struct group_var *gv)
{
	struct dist_info *h, *p;
	int i;

	i = hash_mac_addr(sa);

	/* we need to lock hash list here */
	spin_lock(&gv->dist_lock);

	h = gv->dist_info[i];
	/* search for same address on the list */
	for (p = h; p; p = p->next){
		if (!memcmp(sa, p->m_addr, ETH_ALEN)){
			/* found. already registered */
			if (is_bfu){
				p->bfu_mark = _TRUE;
				p->sgid = sgid;
				p->age = age;
			} else {
				if (!p->bfu_mark){
					p->sgid = sgid;
					p->age = age;
				}
			}

			/* we need to unlock hash list here */
			spin_unlock(&gv->dist_lock);
			return;
		}
	}

	/* registering new distributing information */
	h = kmalloc(sizeof(struct dist_info), GFP_ATOMIC);
	if (!h){
		spin_unlock(&gv->dist_lock);
		return;
	}
	memcpy(h->m_addr, sa, ETH_ALEN);
	if (is_bfu){
		h->bfu_mark = 1;
	} else {
		h->bfu_mark = 0;
	}
	h->sgid = sgid;
	h->age = age;
	atomic_set(&h->use_count, 1);

	h->next = gv->dist_info[i];
	gv->dist_info[i] = h;

	/* we need to unlock hash list here */
	spin_unlock(&gv->dist_lock);

	return;
}

static void lr_clear_distinfo(struct group_var *gv)
{
	struct dist_info *h, *p;
	int i;

	/* we need to lock hash list here */
	spin_lock(&gv->dist_lock);

	for (i = 0; i < 32; i++){

		/* then do access */
		h = gv->dist_info[i];
		for (p = h; p; p = h){
			h = p->next;
			if (atomic_dec_and_test(&p->use_count))
				kfree(p);
		}
		gv->dist_info[i] = NULL; /* Don't forget this */

	}

	/* we need to unlock hash list here */
	spin_unlock(&gv->dist_lock);

	return;
}

static void lr_age_distinfo(struct group_var *gv)
{
	struct dist_info *h, *p, **pp;
	int i;

	/* we need to lock hash list here */
	spin_lock(&gv->dist_lock);

	for (i = 0; i < 32; i++){

		/* then do access */
		h = gv->dist_info[i];
		for (p = h, pp = &(gv->dist_info[i]); p; p = h){
			h = p->next;
			if (!p->age--){
				*pp = h;
				if (atomic_dec_and_test(&p->use_count)){
					kfree(p);
				}
				if (h)
					pp = &(h->next);
			} else {
				pp = &(p->next);
			}
		}

	}

	/* we need to unlock hash list here */
	spin_unlock(&gv->dist_lock);

	return;
}


static void
lr_update_sgmt_temp(u16 active_id, u16 port_id, struct group_var *gv)
{
	struct sgmte *sp;
	int i, j, k;

	sp = gv->temp_sgmt;

	/* search for same port_id on the temp SGMT */
	for (i = 0; i < MAX_PHYIF && sp[i].port_id; i++){
		if (sp[i].port_id == port_id){
			/* found. already registered. override it.*/
			sp[i].active_id = active_id;
			return;
		}
	}

	/* registering new sgmt entry */
	for (i = 0; i < MAX_PHYIF; i++){
		if (!sp[i].port_id || sp[i].port_id > port_id){
			for (j = MAX_PHYIF-1, k = MAX_PHYIF-2; j != i; j--, k--){
				sp[j] = sp[k];
			}
			sp[i].active_id = active_id;
			sp[i].port_id = port_id;
			return;
		}
	}

	return;
}

static void lr_clear_sgmt_temp(struct group_var *gv)
{
	struct sgmte *sp;

	sp = gv->temp_sgmt;
	memset(sp, 0, sizeof(struct sgmte)*MAX_PHYIF);

	return;
}
static void lr_clear_sgmt_master(struct group_var *gv)
{
	struct sgmte *sp;

	sp = gv->master_sgmt;
	memset(sp, 0, sizeof(struct sgmte)*MAX_PHYIF);

	return;
}

static void lr_mark_active(struct group_var *gv)
{
	struct sgmte *sp;
	int i, j;
	int sgid = 1;

	sp = gv->temp_sgmt;

	for (i = 0; i < MAX_PHYIF; i++){
		if (sp[i].port_id == sp[i].active_id){
			/* marking active flag */
			sp[i].active = _TRUE;
			/* then set sgid */
			sp[i].sgid = sgid++;
			/* set the sgid to other sgmte which has same active_id */
			for (j = 0; j < MAX_PHYIF; j++){
				if (sp[i].active_id == sp[j].active_id){
					sp[j].sgid = sp[i].sgid;
				}
			}
		}
	}

	return;
}

static bool lr_comp_sgmt(struct group_var *gv)
{
	struct sgmte *msp;
	struct sgmte *tsp;
	int i;

	tsp = gv->temp_sgmt;
	msp = gv->master_sgmt;

	for (i = 0; i < MAX_PHYIF; i++){
		if (memcmp(tsp+i, msp+i, sizeof(struct sgmte)) != 0){
			return _FALSE;
		}
	}

	return _TRUE;
}

static void lr_update_sgmt_master(struct group_var *gv)
{
	struct sgmte *msp;
	struct sgmte *tsp;

	tsp = gv->temp_sgmt;
	msp = gv->master_sgmt;

	memcpy(msp, tsp, sizeof(struct sgmte)*MAX_PHYIF);

	return;
}

static void lr_update_sg_param(struct group_var *gv)
{
	struct sgmte *msp;
	struct subgroup_var *sgp;
	int i, j;

	sgp = gv->subg_list;
	msp = gv->master_sgmt;

	memset(sgp, 0, sizeof(struct subgroup_var)*MAX_PHYIF);

	for (i = 0; i < MAX_PHYIF; i++){
		j = msp[i].sgid;
		if (j){
			j--;
			sgp[j].active_id = msp[i].active_id;
		}
	}

	return;
}

static void lr_update_port_param(struct group_var *gv)
{
	struct sgmte *msp;
	struct lr_port *pe;
	int *pp;
	int i, j;

	pp = gv->port_list;
	msp = gv->master_sgmt;

	for (i = 0; i < MAX_PHYIF && pp[i]; i++){
		for (j = 0; j < MAX_PHYIF; j++){
			if (pp[i] == (msp[j].port_id & 0xff)){
				/* update port's local SGMT */
				pe = lr_get_port_by_num(pp[i]);
				memcpy(&pe->pv.local_sgmt, msp+j, sizeof(struct sgmte));
				break;
			}
		}
	}

	return;
}

static void lr_init_rct_each(struct lr_port *lrp, u16 port_id)
{
	struct rcte *h, *r;
	struct lr_port *p;

	if (!port_id)
		return;
	
	h = lrp->pv.rct;
	/* free all elements */
	for (r = h; r; r = h){
		h = r->next;
		kfree(r);
	}
	lrp->pv.rct = NULL;

	p = lr_get_port_by_num(port_id);
	if (!p->dev)
		return;

	/* alloc new entry */
	lrp->pv.rct = h = kmalloc(sizeof (struct rcte), GFP_ATOMIC);
	if (!h){
		return;
	}
	memset(h, 0, sizeof(struct rcte));

	/* then set port_id */
	h->port_id = port_id;
	h->next = NULL;

	return;
}

static void lr_init_rct(struct group_var *gv)
{
	struct sgmte *msp;
	int *pp;
	int i;

	pp = gv->port_list;
	msp = gv->master_sgmt;

	for (i = 0; i < MAX_PHYIF; i++){
		if (pp[i]){
			lr_init_rct_each(lr_get_port_by_num(pp[i]), pp[i]);
		}
	}

	return;
}

static void lr_register_rcf(struct lr_port *lrp, u16 port_id)
{
	struct rcte *h, *r;

	h = lrp->pv.rct;
	/* search for same port_id in the list */
	for (r = h; r; r = r->next){
		if (r->port_id == port_id){
			/* already registered. do nothing. */
			return;
		}
	}

	/* alloc new entry */
	h = kmalloc(sizeof (struct rcte), GFP_ATOMIC);
	if (!h){
		return;
	}
	memset(h, 0, sizeof(struct rcte));
	h->next = lrp->pv.rct;
	lrp->pv.rct = h;

	/* then set port_id */
	h->port_id = port_id;

	return;
}


/* make all SgN: active_id 0 */
static void lr_clear_sg_active_id(struct group_var *gv)
{
	int i;

	for (i = 0; i < MAX_PHYIF; i++){
		gv->subg_list[i].active_id = 0;
	}
};


/* make all PortN: sgid 0 */
static void lr_clear_port_sgid(struct group_var *gv)
{
	int i;
	int port;
	struct lr_port *p;

	for (i = 0; i < MAX_PHYIF && (port = gv->port_list[i]); i++){
		p = lr_get_port_by_num(port);
		p->pv.local_sgmt.sgid = 0;
	}
};

static void lr_notify_rcf(struct group_var *gv)
{
	int i, j;
	int pn;
	int *pp;
	struct nrcf val;
	struct rcte *r, *h;
	struct lr_port *p;
	int min_id;

	pp = gv->port_list;

	for (i = 0; i < MAX_PHYIF; i++){
		pn = pp[i];
		if (!pn)
			continue;

		p = lr_get_port_by_num(pn);
		if (!p->dev)
			continue;

		val.port_id = p->pv.local_sgmt.port_id;
		min_id = val.port_id;
		
		h = p->pv.rct;
		for (r = h, j = 0; r; r = r->next, j++){
			if (min_id > r->port_id){
				min_id = r->port_id;
			}
		}

		val.active_id = p->pv.local_sgmt.active_id = min_id;
		ring_put(&(gv->rb), (char *)&val, sizeof(struct nrcf));
	}

	return;
}


static void lr_tchg_detected(struct group_var *gv)
{
	unsigned long t_perio;

	gv->tchg_detected = _TRUE;

	if (gv->pii_live && gv->pii_opt){
		gv->perio_time = Periodic_Time;
		return;
	}

	t_perio = gv->admin_perio_time;

	if (gv->tchg_opt && !gv->tchg_notified){
		/* In order to notify topology change to neighbors ASAP,
		   temporarily make perio_time 1/10. */
		gv->perio_time = net_random()%(t_perio/10) + One_Sec;
	} else
		gv->perio_time = net_random()%(t_perio) + One_Sec;

	gv->tchg_notified = _FALSE;
	return;
}


static void lr_tchg_not_detected(struct group_var *gv)
{
	gv->tchg_detected = _FALSE;
	gv->tchg_notified = _FALSE;

	if (gv->pii_live && gv->pii_opt)
		gv->perio_time = Periodic_Time;
	else
		gv->perio_time = gv->admin_perio_time;

	return;
}

static void lr_tchg_notified(struct group_var *gv)
{
	unsigned long t_perio;

	if (gv->pii_live && gv->pii_opt){
		gv->tchg_notified = _FALSE;
		gv->perio_time = Periodic_Time;
		return;
	}

	gv->tchg_notified = _TRUE;
	if (!gv->tchg_detected){
		/* After topology change is notified, in order to make sure
		   it, temporarily make perio_time 1/10 or less. */
		t_perio = gv->admin_perio_time;
		gv->perio_time = net_random()%(t_perio/10) + One_Sec;
	}
}

static struct net_device *lr_search_for_port(u16 port_id)
{
	struct lr_port *p;

	if (!port_id)
		goto nodev;

	for (p = lr_get_head_port(); p; p = lr_get_next_port(p)){
		if (p->pv.local_sgmt.port_id == port_id){
			return p->dev;
		}
	}

nodev:
	return NULL;
}

static bool
lr_subg_distribution(struct sk_buff *skb, struct subgroup_var *sl, u16 sg)
{
	struct net_device *tdev;

	sg--;
	if (!sl[sg].active_id)
		goto freeandout;

	tdev = lr_search_for_port(sl[sg].active_id);
	if (!tdev)
		goto freeandout;

	skb->dev = tdev;
	/* Don't set skb->trans_start ! */
	dev_queue_xmit(skb);
	return _TRUE;

freeandout:
	kfree_skb(skb);
	return _FALSE;
}

static bool
lr_all_subg_distribution(struct sk_buff *skb, struct subgroup_var *sl)
{
	bool ret = _FALSE;
	struct sk_buff *nskb;
	int i;

	if (!sl)
		goto freeandout;

	for (i = 1; i <= MAX_PHYIF; i++){
		nskb = skb_clone(skb, GFP_ATOMIC);
		if (!nskb){
			break;
		}
		ret |= lr_subg_distribution(nskb, sl, i);
		if (!ret)
			break;
	}
freeandout:
	kfree_skb(skb);
	return ret;
}


static int lr_group_distribution(struct sk_buff *skb, struct net_device *dev)
{
	struct lr_private *vp;
	struct ethhdr *eh;
	struct sk_buff *nskb;
	int len;

	u16 sgid; /* sub-group */
	struct group_var *gv;

	if (!skb){
		printk(KERN_CRIT "lr_frame_distribution(): skb is NULL!\n");
		return NET_XMIT_DROP;
	}
	if (!dev){
		printk(KERN_CRIT "lr_frame_distribution(): dev is NULL!\n");
		kfree_skb(skb);
		return NET_XMIT_DROP;
	}

	vp = dev->priv;
	gv = &(vp->group->var);

	/* Search for proper subgroup for transmitting this skb. */

	/* don't use skb->mac.* */
	eh = (struct ethhdr *)skb->data;
	if (eh->h_dest[0]&1)
		sgid = 0;
	else
		sgid = lr_search_for_subg(eh->h_dest, gv);

	/* There may be no ports for distributing this skb. so we need
	 * to clone skb to remove such skb */
	nskb = skb;
	skb = skb_clone(nskb, GFP_ATOMIC);

	if (!skb){
		return NET_XMIT_DROP;
	}

	len = skb->len;

	switch (sgid){
	case 0:
		/* Unknown subgroup for distributing this skb. So, forward
         * this to all the sug-groups.
		 */
		if (!lr_all_subg_distribution(skb, vp->group->var.subg_list)){
			return NET_XMIT_DROP;
		}
		break;
	default:
		if (!lr_subg_distribution(skb, vp->group->var.subg_list, sgid)){
			return NET_XMIT_DROP;
		}
		break;
	}
	kfree_skb(nskb);

	vp->enet_stats.tx_bytes += len;
	vp->enet_stats.tx_packets++;

	return NET_XMIT_SUCCESS;
}

static void inline lr_pass_to_upper(struct sk_buff *skb)
{
	netif_rx(skb);
}

static void
lr_group_collector(struct sk_buff *skb, struct net_device *dev, struct group_var *gv)
{
	struct ethhdr *eh;
	bool is_local_skb;
	struct lr_private *vp;

	vp = dev->priv;

	eh = (struct ethhdr *)skb->mac.raw;
	is_local_skb = 
		(!memcmp(gv->group_addr, eh->h_dest, ETH_ALEN) ||
		eh->h_dest[0]&1)?_TRUE:_FALSE;

	skb->dev = dev;

	if (is_local_skb)
		skb->pkt_type = PACKET_HOST;
	else
		skb->pkt_type = PACKET_OTHERHOST;
		
	if (is_local_skb || dev->flags&IFF_PROMISC){
		vp->enet_stats.rx_bytes += skb->len;
		vp->enet_stats.rx_packets++;
		lr_pass_to_upper(skb);
	} else 
		kfree_skb(skb);

	return;
	
}



static void lr_submit_ntr(struct group_var *gv)
{
	spin_lock(&gv->lock);

	/* make all PortN: sgid 0 */
	lr_clear_sg_active_id(gv);

	/* make ntr true */
	gv->ntr = _TRUE;

	/* make all PortN: sgid 0 */
	lr_clear_port_sgid(gv);

	/* reset all the state machines */
	gv->begin = _TRUE;

	spin_unlock(&gv->lock);

	return;
}

/* BFU */
static bool
lr_bfu_pass(unsigned char *sa, unsigned short sg, struct group_var *gv)
{
	int hash;
	struct glist_ent *p;
	int ret = _TRUE;

	hash = hash_mac_addr(sa);

	spin_lock(&gv->bfu_lock);

	for (p = gv->gl[hash]; p; p = p->next2){
		if (memcmp(p->gaddr, sa, ETH_ALEN) == 0)
			if (p->min_sgid != sg)
				ret = _FALSE;
	}

	spin_unlock(&gv->bfu_lock);

	return ret;
}

static void
lr_subg_collector(struct sk_buff *skb, struct net_device *dev, int sg)
{
	struct ethhdr *eh;
	struct group_var *gv;
	struct lr_private *vp;

	eh = (struct ethhdr *)skb->mac.raw;
	vp = (struct lr_private *)dev->priv;
	gv = &(vp->group->var);

	if (!memcmp(eh->h_source, dev->dev_addr, ETH_ALEN)){
		/* loopback packet is detected. Submit NTR immediately. */
		lr_submit_ntr(gv);
		goto freeandout;
	}

	/* BFU */
	if (eh->h_dest[0]&1)
		if (!lr_bfu_pass(eh->h_source, sg, gv))
			goto freeandout;

	lr_update_distinfo(eh->h_source, sg, 0, gv->age_time, gv);
	lr_group_collector(skb, dev, gv);
	return;

freeandout:
	kfree_skb(skb);
	return;
}


static int lr_nf_receiver(struct sk_buff *skb, struct net_device *dev)
{
	unsigned char *buf;
	int grp, sg, port;
	struct lacpdu *pdu;
	struct lr_port *p;
	struct lr_group *g;

	buf = skb->mac.raw + ETH_HLEN;
	pdu = (struct lacpdu *)buf;

	/* calculate port num */
	port = dev_to_port_num(dev);
	if (!port)
		goto freeandout;
	p = lr_get_port_by_num(port);

	if (!p->pv.local_sgmt.active)
		goto freeandout;

	grp = p->pv.group_id;
	if (!grp)
		goto freeandout;

	g = lr_get_group_by_num(grp);
	sg = p->pv.local_sgmt.sgid;
	if (!sg)
		goto freeandout;

	lr_subg_collector(skb, g->dev, sg);
	return NET_RX_SUCCESS;

freeandout:
	kfree_skb(skb);
	return NET_RX_DROP;
}

/* Broadcast Filtering Unit (BFU) */
static struct sysid_ent *
lr_search_sysid_bfu (unsigned char *sysid, struct group_var *gv)
{
	struct sysid_ent *p;

	for (p = gv->sysidl; p; p = p->next)
		if (memcmp(p->sysid, sysid, ETH_ALEN) == 0)
			return p;

	return NULL;
}

static struct glist_ent *
lr_search_gle_bfu (unsigned short gid, struct sysid_ent *syside)
{
	struct glist_ent *p;

	for (p = syside->gl; p; p = p->next)
		if (p->gid == gid)
			return p;

	return NULL;
}

static struct sglist_ent *
lr_search_sgle_bfu(unsigned short sgid, struct glist_ent *gle)
{
	struct sglist_ent *p;

	for (p = gle->sgl; p; p = p->next)
		if (p->sgid == sgid)
			return p;

	return NULL;
}

static struct sysid_ent *lr_alloc_syside(struct group_var *gv)
{
	struct sysid_ent *p;

	p = (struct sysid_ent *)kmalloc(sizeof(struct sysid_ent), GFP_ATOMIC);
	if (!p)
		return NULL;

	p->next = gv->sysidl;
	gv->sysidl = p;

	p->gv = gv;
	p->gl = NULL;

	return p;
}

static struct glist_ent *
lr_alloc_gle(unsigned char *sa, struct sysid_ent *syside)
{
	struct glist_ent *p;
	int hash;

	hash = hash_mac_addr(sa);

	p = (struct glist_ent *)kmalloc(sizeof(struct glist_ent), GFP_ATOMIC);
	if (!p)
		return NULL;

	p->next = syside->gl;
	syside->gl = p;

	p->next2 = syside->gv->gl[hash];
	syside->gv->gl[hash] = p;
	memcpy(p->gaddr, sa, ETH_ALEN);

	p->gv = syside->gv;
	p->sgl = NULL;
	p->syside = syside;
	atomic_set(&p->use_count, 1);

	return p;
}

static struct sglist_ent *lr_alloc_sgle(struct glist_ent *gle)
{
	struct sglist_ent *p;

	p = (struct sglist_ent *)kmalloc(sizeof(struct sglist_ent), GFP_ATOMIC);
	if (!p)
		return NULL;

	p->next = gle->sgl;
	gle->sgl = p;

	p->gle = gle;

	return p;
}

static void lr_dealloc_sgle(struct sglist_ent *sgle)
{
	struct sglist_ent *p;

	for (p = sgle->gle->sgl; p; p = p->next){
		if (p == sgle){
			sgle->gle->sgl = p->next;
			kfree(sgle);
			return;
		}
		if (p->next == sgle){
			p->next = p->next->next;
			kfree(sgle);
			return;
		}
	}

	return;
}

static void lr_dealloc_gle(struct glist_ent *gle)
{
	struct glist_ent *p;
	int hash;

	hash = hash_mac_addr(gle->gaddr);

	for (p = gle->gv->gl[hash]; p; p = p->next2){
		if (p == gle){
			gle->gv->gl[hash] = p->next2;
			break;
		}
		if (p->next2 == gle){
			p->next2 = p->next2->next2;
			break;
		}
	}

	for (p = gle->syside->gl; p; p = p->next){
		if (p == gle){
			gle->syside->gl = p->next;
			break;
		}
		if (p->next == gle){
			p->next = p->next->next;
			break;
		}
	}

	for (;gle->sgl;)
		lr_dealloc_sgle(gle->sgl);
	if (atomic_dec_and_test(&gle->use_count))
		kfree(gle);
	return;
}

static void lr_dealloc_syside(struct sysid_ent *syside)
{
	struct sysid_ent *p;

	for (p = syside->gv->sysidl; p; p = p->next){
		if (p == syside){
			syside->gv->sysidl = p->next;
			for (;syside->gl;)
				lr_dealloc_gle(syside->gl);
			kfree(syside);
			break;
		}
		if (p->next == syside){
			p->next = p->next->next;
			for (;syside->gl;)
				lr_dealloc_gle(syside->gl);
			kfree(syside);
			break;
		}
	}
	return;
}

static unsigned short lr_min_sgid_bfu(struct glist_ent *gle)
{
	struct sglist_ent *p;
	unsigned short min_sgid = 0xffff;

	for (p = gle->sgl; p; p = p->next)
		if (p->rcv_sgid < min_sgid)
			min_sgid = p->rcv_sgid;

	gle->min_sgid = min_sgid;
	return min_sgid;
}

static void lr_cleanup_bfu(struct group_var *gv)
{
	for (;gv->sysidl;)
		lr_dealloc_syside(gv->sysidl);
}

static bool 
lr_bfu(struct lrcpdu *lrcpdu, unsigned short sgid, struct group_var *gv)
{
	struct sysid_ent *syside;
	struct glist_ent *gle;
	struct sglist_ent *sgle;
	unsigned short perio_time;

	if (!sgid)
		return _FALSE;

	spin_lock(&gv->bfu_lock);

	syside = lr_search_sysid_bfu(lrcpdu->system_id, gv);
	if (!syside){
		syside = lr_alloc_syside(gv);
		if (!syside)
			goto bfufail;
		memcpy(syside->sysid, lrcpdu->system_id, ETH_ALEN);
	}

	gle = lr_search_gle_bfu(lrcpdu->group_id, syside);
	if (!gle){
		gle = lr_alloc_gle(lrcpdu->group_address, syside);
		if (!gle){
			lr_dealloc_syside(syside);
			goto bfufail;
		}
		gle->gid = lrcpdu->group_id;
	}

	sgle = lr_search_sgle_bfu(lrcpdu->subgroup_id, gle);
	if (!sgle){
		sgle = lr_alloc_sgle(gle);
		if (!sgle){
			lr_dealloc_syside(syside);
			goto bfufail;
		}
	}
	sgle->sgid = lrcpdu->subgroup_id;
	sgle->rcv_sgid = sgid;

	gle->min_sgid = lr_min_sgid_bfu(gle);

	/* phase II compatibility */
	if (lrcpdu->version_number < 3 && gv->admin_perio_time > One_Sec){
		gv->pii_live = PII_Live_Time;
		gv->perio_time = One_Sec;
	}
	perio_time = ntohs(lrcpdu->perio_time);
	if (!perio_time)
		perio_time = 1;
	/* end of phase II compatibility */

	if (gle->min_sgid){
		lr_update_distinfo(gle->gaddr, gle->min_sgid, 1, perio_time*4, gv);
	}
	sgle->age = perio_time*4;

	spin_unlock(&gv->bfu_lock);

	return _TRUE;

bfufail:
	spin_unlock(&gv->bfu_lock);
	return _FALSE;
}

static void lr_age_bfu(struct group_var *gv)
{
	struct sglist_ent *sgle;
	struct glist_ent *gle;
	struct sysid_ent *syside;
	int i;

	spin_lock(&gv->bfu_lock);

	for (i = 0; i < 32; i++)
		for (gle = gv->gl[i]; gle; gle = gle->next2){
			for (sgle = gle->sgl; sgle; sgle = sgle->next)
				if (!sgle->age--)
					lr_dealloc_sgle(sgle);
			if (!gle->sgl){
				syside = gle->syside;
				lr_dealloc_gle(gle);
				if (!syside->gl)
					lr_dealloc_syside(syside);
			}
		}

	spin_unlock(&gv->bfu_lock);

	return;
}

static int lr_rcf_receiver(struct sk_buff *skb, struct net_device *dev)
{

	struct lr_port *p;
	unsigned char *buf;
	struct lrcpdu *lrcpdu;
	int pn = 0;

	buf = skb->mac.raw + ETH_HLEN;
	lrcpdu = (struct lrcpdu *)buf;

	pn = dev_to_port_num(skb->dev);
	if (!pn || !lrcpdu->port_id)
		goto freeandout;

	p = lr_get_port_by_num(pn);
	if (memcmp(lrcpdu->system_id, lr_sysvar.system_id, ETH_ALEN) != 0){
		/* other host's RCF */
		int grp = p->pv.group_id;
		struct lr_group *g = lr_get_group_by_num(grp);
		struct lr_private *vp;
		struct group_var *gv;

		vp = (struct lr_private *)(g->dev->priv);
		gv = &(vp->group->var);

		lr_bfu(lrcpdu, p->pv.local_sgmt.sgid, gv);
		if ((lrcpdu->flags & RCFFL_TOPOCHG) &&
			gv->tchg_opt &&
			!gv->tchg_notified)
			lr_tchg_notified(gv);
		kfree_skb(skb);

		return NET_RX_SUCCESS;
	}

	if (p->pv.group_id != lrcpdu->group_id)
		goto freeandout; /* ignore other group RCF */

	if (!p->pv.rcft_running)
		goto freeandout; /* ignore RCF if RCF timer not running */

	lr_register_rcf(p, lrcpdu->port_id);
	kfree_skb(skb);
	return NET_RX_SUCCESS;

freeandout:
	kfree_skb(skb);
	return NET_RX_DROP;
}

/* port parser  */
static int 
lr_port_parser(struct sk_buff *skb, struct net_device *dev, 
			   struct packet_type *pt)
{
	unsigned char *buf;
	int ret;

	if (!skb){
		return NET_RX_BAD;
	}

	buf = skb->mac.raw + ETH_HLEN;
	if (!memcmp(skb->mac.ethernet->h_dest, Slow_Protocols_Multicast, ETH_ALEN) &&
		ntohs(skb->mac.ethernet->h_proto) == Slow_Protocols_Type &&
		buf[0] == LRCP_Subtype){
		ret = lr_rcf_receiver(skb, dev);
	} else {
		ret = lr_nf_receiver(skb, dev);
	}

	return ret;
}


/*-------------- Flush Frame transmitter ---------*/

/* Purpose of this is update filtering info of all the bridges which
 * is conneced to ports belong to the link redundancy group. */


static void lr_xmit_ff_all(struct group_var *gv)
{
	int grp;
	struct lrcpdu pdu;
	struct sk_buff *skb;
	struct ethhdr *eh;
	struct lr_group *g;
	int pdu_len = LRCPDU_LEN;
	int len = pdu_len + ETH_HLEN;

	/* makeup LRCPDU */
	memset(&pdu, 0, sizeof(struct lrcpdu));
	pdu.subtype = FF_Subtype;
	pdu.version_number = LRCP_Version;
	memcpy(pdu.system_id, lr_sysvar.system_id, ETH_ALEN);
	pdu.sys_priority = htons(lr_sysvar.priority);
	pdu.group_id = grp = gv->group_id;
	pdu.subgroup_id = 0;
	pdu.port_id = 0;

	if (!grp)
		return;

	g = lr_get_group_by_num(grp);
	if (!g || !g->dev)
		return;


	skb = dev_alloc_skb(len);

	if (!skb){
		return;
	}
	skb->dev = g->dev;
	skb->mac.raw = skb_put(skb, len);
	eh = (struct ethhdr *)skb->mac.raw;

	memcpy(eh->h_dest, Slow_Protocols_Multicast, ETH_ALEN);
	memcpy(eh->h_source, g->dev->dev_addr, ETH_ALEN);
	eh->h_proto = htons(Slow_Protocols_Type);

	skb->nh.raw = skb->mac.raw + ETH_HLEN;
	memcpy(skb->nh.raw, &pdu, pdu_len);

	lr_all_subg_distribution(skb, gv->subg_list);

	return;
}





/*-------------------- Timers ---------------------*/
/* vt_start(), vt_stop(), is_vt_active() and vt_count()
 * are protected under lr spinlock.
 */
static void inline vt_start(struct lr_timer *vt, int exp)
{
	vt->active = _TRUE;
	vt->exp = exp;
	vt->tim = 0;

	return;
}

static void inline vt_stop(struct lr_timer *vt)
{
	vt->tim = 0;
	vt->active = _FALSE;

	return;
}

static int inline is_vt_active(struct lr_timer *vt)
{
	return vt->active;
}



static void vt_count(struct lr_timer *vt)
{
	if (vt->active){
		if (vt->tim == vt->exp){
			vt->tim = 0;					
			vt->active = _FALSE;
		} else {
			vt->tim++;
		}
	}

	return;
}

#define lr_clear_timer(gv, mem) \
	__lr_clear_timer(&(gv)->mem)
#define lr_register_timer(gv, mem) \
	__lr_register_timer((gv), &(gv)->mem)
#define lr_unregister_timer(gv, mem) \
	__lr_unregister_timer((gv), &(gv)->mem)

static void __lr_clear_timer(struct lr_timer *vt)
{
	memset(vt, 0, sizeof(struct lr_timer));
	return;
}

static void __lr_unregister_timer(struct group_var *gv, struct lr_timer *vt)
{
	struct lr_timer *p, *q;

	if (!vt){
		return;
	}

	if (gv->vth == vt){
		p = gv->vth->next;
		gv->vth->next = 0;
		gv->vth = p;
		return;
	}

	p = gv->vth->next;
	q = gv->vth;
	for (; p; p = p->next, q = q->next){
		if (vt == p){
			q->next = p->next;
			p->next = 0;
		}
	}

	return;
}

static void __lr_register_timer(struct group_var *gv, struct lr_timer *vt)
{
	if (!vt){
		return;
	}

	vt->tick_func = vt_count;
	vt->next = gv->vth;
	gv->vth = vt;

	return;
}

/*-------------------- end of Timers --------------*/




/*-------------- Periodic Machine ---------------*/


/* action at DISABLED state */
static void lr_prm_disabled(struct group_var *gv)
{
	vt_stop(&(gv->periodic_timer));
	return;
}

/* action at WAIT_FOR_TIMEOUT state */
static void lr_prm_wait_timeout(struct group_var *gv)
{
	vt_start(&(gv->periodic_timer), gv->perio_time);
	return;
}

/* action at EXPIRED state */
static void lr_prm_expired(struct group_var *gv)
{
	vt_stop(&(gv->rcf_timer));
	gv->ntr = _TRUE;
	return;
}



static void lr_prm_tick(struct group_var *gv)
{
	struct perio_mach *m = &(gv->prm);
	bool go_init;

	go_init = gv->begin;

	/* state machine */
	if (m->state == DISABLED){
		DO_IT(lr_prm_disabled(gv))

		if (!go_init){
			GO_NEXT_STATE(WAIT_FOR_TIMEOUT)
		}
	} else if (m->state == WAIT_FOR_TIMEOUT){
		if (go_init){
			GO_NEXT_STATE(DISABLED)
		} else {
			DO_IT(lr_prm_wait_timeout(gv))
			if (!is_vt_active(&(gv->periodic_timer))){
				GO_NEXT_STATE(EXPIRED)
			}
		}
	} else if (m->state == EXPIRED){
		if (go_init){
			GO_NEXT_STATE(DISABLED)
		} else {
			DO_IT(lr_prm_expired(gv))
			GO_NEXT_STATE(WAIT_FOR_TIMEOUT)
		}
	}
	return;
}
/*-------------- Periodic Machine ---------------*/

/*-------------- RCF Timer Machine ---------------*/

static void lr_set_port_rcft_running(struct group_var *gv, bool val)
{
	int i, j;
	struct lr_port *p;

	for (i = 0; i < MAX_PHYIF; i++){
		j = gv->port_list[i];
		if (!j)
			break;

		p = lr_get_port_by_num(j);
		if (!p->dev)
			continue;
		p->pv.rcft_running = val;
	}
}

/* action at NOT_RUNNING state */
static void lr_rcftm_not_running(struct group_var *gv)
{
	/* make all PortN:rcft_running FALSE */
	lr_set_port_rcft_running(gv, _FALSE);
	/* make GrpN:rcft_running FALSE */
	gv->rcft_running = _FALSE;

	vt_stop(&(gv->rcf_timer));
	return;
}

/* action at RUNNING state */
static void lr_rcftm_running(struct group_var *gv)
{
	/* make PortN:rcft_running TRUE */
	lr_set_port_rcft_running(gv, _TRUE);
	/* make GrpN:rcft_running TRUE */
	gv->rcft_running = _TRUE;

	lr_init_rct(gv);

	vt_start(&(gv->rcf_timer), Timeout_Time);

	return;
}

/* action at TIMEOUT state */
static void lr_rcftm_timeout(struct group_var *gv)
{
	vt_stop(&(gv->rcf_timer));

	/* make PortN:rcft_running FALSE */
	lr_set_port_rcft_running(gv, _FALSE);

	lr_notify_rcf(gv);
	vt_start(&(gv->rcf_timer), Timeout2_Time - Timeout_Time);
	return;
}

/* action at TIMEOUT2 state */
static void lr_rcftm_timeout2(struct group_var *gv)
{
	vt_stop(&(gv->rcf_timer));

	/* make GrpN:rcft_running FALSE */
	gv->rcft_running = _FALSE;
	return;
}

static void lr_rcftm_tick(struct group_var *gv)
{
	struct rcft_mach *m = &(gv->rcftm);

	bool go_init;

	go_init = gv->begin;

	if (m->state == NOT_RUNNING){
		DO_IT(lr_rcftm_not_running(gv))
		if (!go_init && gv->ntt){
			GO_NEXT_STATE(RUNNING)
		}
	} else if (m->state == RUNNING){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE);
		} else {
			DO_IT(lr_rcftm_running(gv))

			/* timeout */
			if (!is_vt_active(&(gv->rcf_timer))){
				GO_NEXT_STATE(TIMEOUT)
			}
		}
	} else if (m->state == TIMEOUT){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE)
		} else {
			DO_IT(lr_rcftm_timeout(gv))

			if (!is_vt_active(&(gv->rcf_timer))){
				GO_NEXT_STATE(TIMEOUT2)
			}
		}
	} else if (m->state == TIMEOUT2){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE)
		} else {
			DO_IT(lr_rcftm_timeout2(gv))
			GO_NEXT_STATE(NOT_RUNNING)
		}
	}

	return;
}

/*-------------- RCF Timer Machine ---------------*/

/*-------------- SGM machine -------------------*/

/* 
 * SGM machine
 */

/* action at INITIALIZE state */
static void lr_sgmm_init(struct group_var *gv)
{
	gv->sgmm.cnt = 5;
	lr_clear_sg_active_id(gv);
	lr_clear_port_sgid(gv);
	lr_clear_sgmt_temp(gv);
	lr_clear_sgmt_master(gv);
	lr_clear_distinfo(gv);
	return;
}

/* action at WAIT_FOR_NTR state */
static void lr_sgmm_wait_ntr(struct group_var *gv)
{
	return;
}

/* action at REBUILD_START state */
static void lr_sgmm_rebuild_start(struct group_var *gv)
{
	gv->ntt = _TRUE;
	lr_clear_sgmt_temp(gv);
	return;
}

/* action at WAIT_FOR_RCF_NOTIFY state */
static void lr_sgmm_wait_notify(struct group_var *gv)
{
	return;
}

/* action at RCF_RECEIVED state */
static void lr_sgmm_rcf_received(struct group_var *gv)
{
	struct nrcf *nr;
	char buf[sizeof(struct nrcf)+10];

	while (!ring_empty(&(gv->rb))){
		ring_get(&(gv->rb), buf, sizeof(struct nrcf));
		nr = (struct nrcf *)buf;
		lr_update_sgmt_temp(nr->active_id, nr->port_id, gv);
	}
	return;
}

/* action at RCF_TIMEOUT state */
static void lr_sgmm_rcf_timeout(struct group_var *gv)
{
	lr_mark_active(gv);
	gv->sgmt_match = lr_comp_sgmt(gv);
	return;
}


/* action at NOT_CHANGED state */
static void lr_sgmm_not_changed(struct group_var *gv)
{
	lr_tchg_not_detected(gv);
	gv->ntr = _FALSE;
	return;
}

/* action at CHANGED state */
static void lr_sgmm_changed(struct group_var *gv)
{
	/* make all PortN: sgid 0 */
	lr_clear_sg_active_id(gv);
	lr_clear_port_sgid(gv);
	lr_update_sgmt_master(gv);
	lr_clear_distinfo(gv);
	lr_update_sg_param(gv);
	lr_update_port_param(gv);
	lr_xmit_ff_all(gv);
	lr_tchg_detected(gv);
	gv->ntr = _FALSE;
	return;
}

static void lr_sgmm_tick(struct group_var *gv)
{
	struct sgm_mach *m = &(gv->sgmm);

	bool go_init;

	go_init = gv->begin;

	if (m->state == INITIALIZE){
		DO_IT(lr_sgmm_init(gv))
		if (m->cnt)
			m->cnt--;
		else
			gv->begin = _FALSE;

		if (!go_init){
			GO_NEXT_STATE(WAIT_FOR_NTR)
		}
	} else if (m->state == WAIT_FOR_NTR){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE);
		} else {
			DO_IT(lr_sgmm_wait_ntr(gv))

			if (gv->ntr){
				GO_NEXT_STATE(REBUILD_START)
			}
		}
	} else if (m->state == REBUILD_START){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE)
		} else {
			DO_IT(lr_sgmm_rebuild_start(gv))

			if (is_vt_active(&(gv->rcf_timer))){
				GO_NEXT_STATE(WAIT_FOR_RCF_NOTIFY)
			}
		}
	} else if (m->state == WAIT_FOR_RCF_NOTIFY){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE)
		} else {
			DO_IT(lr_sgmm_wait_notify(gv))
			GO_NEXT_STATE(WAIT_FOR_RCF_NOTIFY)

			if (!ring_empty(&(gv->rb))){ /* incoming */
				GO_NEXT_STATE(RCF_RECEIVED)
			}
			if (!is_vt_active(&(gv->rcf_timer))){
				GO_NEXT_STATE(RCF_TIMEOUT)
			}
		}
	} else if (m->state == RCF_RECEIVED){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE)
		} else {
			DO_IT(lr_sgmm_rcf_received(gv))

			GO_NEXT_STATE(WAIT_FOR_RCF_NOTIFY)
		}
	} else if (m->state == RCF_TIMEOUT){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE)
		} else {
			DO_IT(lr_sgmm_rcf_timeout(gv))
			if (gv->sgmt_match){
				GO_NEXT_STATE(NOT_CHANGED)
			} else {
				GO_NEXT_STATE(CHANGED)
			}
		}
	} else if (m->state == NOT_CHANGED){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE)
		} else {
			DO_IT(lr_sgmm_not_changed(gv))
			GO_NEXT_STATE(WAIT_FOR_NTR)
		}
	} else if (m->state == CHANGED){
		if (go_init){
			GO_NEXT_STATE(INITIALIZE)
		} else {
			DO_IT(lr_sgmm_changed(gv))
			GO_NEXT_STATE(WAIT_FOR_NTR)
		}
	}

	return;
}

/*--------------- end of SGM machine -----------*/

/*-------------- RCF Generator Machine ---------------*/


static void lr_xmit_rcf(struct net_device *dev, struct lrcpdu *buf)
{
	int pdu_len = LRCPDU_LEN;
	int len = pdu_len + ETH_HLEN;
	struct ethhdr *eh;
	struct sk_buff *skb;

	if (!dev){
		return;
	}

	skb = dev_alloc_skb(len);

	if (!skb){
		return;
	}
	skb->dev = dev;
	skb->mac.raw = skb_put(skb, len);
	eh = (struct ethhdr *)skb->mac.raw;

	memcpy(eh->h_dest, Slow_Protocols_Multicast, ETH_ALEN);
	memcpy(eh->h_source, dev->dev_addr, ETH_ALEN);
	eh->h_proto = htons(Slow_Protocols_Type);

	skb->nh.raw = skb->mac.raw + ETH_HLEN;
	memcpy(skb->nh.raw, buf, pdu_len);

/*	skb->dev->trans_start = jiffies;*/
	dev_queue_xmit(skb);

	return;
}


static void lr_xmit_rcf_all(struct group_var *gv, struct lrcpdu *pdu)
{
	int i, n;
	struct lr_port *p;

	for (i = 0; (n = gv->port_list[i]) && (i < MAX_PHYIF); i++){
		if (!n)
			break;

		p = lr_get_port_by_num(n);
		if (!p->dev)
			continue;
		pdu->subgroup_id = p->pv.local_sgmt.sgid;
		pdu->port_id = p->pv.local_sgmt.port_id;
		lr_xmit_rcf(p->dev, pdu);
	}

	return;
}

/* action at WAIT_FOR_NTT state */
static void lr_rcfgm_wait_ntt(struct group_var *gv)
{
	return;
}

/* action at RCF_GENERATE state */
static void lr_rcfgm_generate(struct group_var *gv)
{

	struct lrcpdu pdu;

	/* makeup LRCPDU */
	memset(&pdu, 0, sizeof(struct lrcpdu));
	pdu.subtype = LRCP_Subtype;
	pdu.version_number = LRCP_Version;
	memcpy(pdu.system_id, lr_sysvar.system_id, ETH_ALEN);
	memcpy(pdu.group_address, gv->group_addr, ETH_ALEN);
	pdu.sys_priority = htons(lr_sysvar.priority);
	pdu.group_id = gv->group_id;
	pdu.subgroup_id = 0;
	pdu.port_id = 0;

	if (!gv->pii_live || !gv->pii_opt)
		pdu.perio_time = htons(gv->admin_perio_time/One_Sec);
	else
		/* Phase II compatibility */
		pdu.perio_time = Periodic_Time/One_Sec;

	if (gv->tchg_opt)
		pdu.flags = (gv->tchg_detected && !gv->tchg_notified)?RCFFL_TOPOCHG:0;


	lr_xmit_rcf_all(gv, &pdu);

	gv->ntt = _FALSE;
	return;

}


static void lr_rcfgm_tick(struct group_var *gv)
{
	struct rcfg_mach *m = &(gv->rcfgm);
	bool go_init;

	go_init = gv->begin;

	/* state machine */
	if (m->state == WAIT_FOR_NTT){
		DO_IT(lr_rcfgm_wait_ntt(gv))

		if (!go_init && gv->ntt){
			GO_NEXT_STATE(RCF_GENERATE)
		}
	} else if (m->state == RCF_GENERATE){
		if (!go_init){
			DO_IT(lr_rcfgm_generate(gv))
		}
		GO_NEXT_STATE(WAIT_FOR_NTT)
	}
	return;
}
/*-------------- RCF Generator Machine ---------------*/

/*------- Distributing Information aging --------*/
static void lr_age_tick(struct group_var *gv)
{
	if (gv->agecnt++ >= Age_Periodic_Time){
		gv->agecnt = 0;
		lr_age_distinfo(gv);
		lr_age_bfu(gv);
	}
}
/*------- Distributing Information aging --------*/

/*------- Phase II compatibility --------*/
/*------- Phase II absent timer ---------*/
static void lr_pii_tick(struct group_var *gv)
{
	if (gv->pii_live){
		gv->pii_live--;
	}
}
/*------- Phase II live timer    --------*/


static void lr_tick(unsigned long ngrp)
{
	struct lr_group *group;
	struct lr_timer *vt;
	struct group_var *gv;
	int i;

	group = lr_get_group_by_num(ngrp);

	if (!group->dev || !group->dev->priv)
		return;

	gv = &group->var;

	spin_lock(&gv->lock);

	/* periodic machine and other timers */
	for (i = 0, vt = gv->vth; vt;i++, vt = vt->next){
		if (vt->tick_func){
			vt->tick_func(vt);
		}
	}

	/* sgm machine */
	lr_sgmm_tick(gv);

	/* periodic timer machine */
	lr_prm_tick(gv);

	/* RCF generator machine */
	lr_rcfgm_tick(gv);

	/* RCF timer machine */
	lr_rcftm_tick(gv);

	/* Distributing Information aging */
	lr_age_tick(gv);

	/* Phase II compatibility */
	lr_pii_tick(gv);

	group->tl.expires = jiffies + HZ/One_Sec; /* 50 msec */
	add_timer(&group->tl);

	spin_unlock(&gv->lock);

	return;
}

/***************************************************************
 *              End of LRCP
 ***************************************************************/


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
struct packet_type lr_packet_type __initdata = {
	__constant_htons(ETH_P_LR),
	NULL,           /* Listen to all devices */
	lr_port_parser,
	NULL,
	NULL,
};
#else
struct packet_type lr_packet_type = {
   .type = __constant_htons(ETH_P_LR),
   .func = lr_port_parser,
 };
#endif

 inline void ic_lr_init(void)
{
	dev_add_pack(&lr_packet_type);
}
 inline void ic_lr_cleanup(void)
{
	dev_remove_pack(&lr_packet_type);
}



/*
 *  Open/initialize the i/f.  This is called (in the current kernel)
 *  sometime after booting when the 'ifconfig' program is run.
 */
static int lr_open(struct net_device *dev)
{
	/* start transmitting */
	netif_start_queue(dev);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_INC_USE_COUNT;
#endif

	return 0;
}

/* The inverse routine to lr_open (). */
static int lr_close(struct net_device *dev)
{

	/* can't transmit any more */
	netif_stop_queue(dev);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_DEC_USE_COUNT;
#endif

	return 0;
}



static struct net_device_stats *lr_get_stats(struct net_device *dev)
{
	struct lr_private *vp = (struct lr_private *)dev->priv;
	struct net_device_stats *r = &vp->enet_stats;

	return r;
}



static void lr_group_vars_init(struct group_var *gv)
{
	gv->begin = _TRUE;
	
	/* state machines */
	gv->sgmm.state = INITIALIZE;
	gv->rcftm.state = NOT_RUNNING;
	gv->prm.state = DISABLED;

	gv->sgmm.do_it = _TRUE;
	gv->rcftm.do_it = _TRUE;
	gv->prm.do_it = _TRUE;

	gv->admin_perio_time = Periodic_Time;
	gv->perio_time = Periodic_Time;
	gv->age_time = Age_Max;
	gv->tchg_opt = _FALSE;
	gv->pii_opt = _TRUE;
	gv->pii_live = 0;

	memset(gv->port_list, 0, sizeof(int)*MAX_PHYIF);
	memset(gv->subg_list, 0, sizeof(struct subgroup_var)*MAX_PHYIF);

	memset(gv->master_sgmt, 0, sizeof(struct sgmte)*MAX_PHYIF);
	memset(gv->temp_sgmt, 0, sizeof(struct sgmte)*MAX_PHYIF);
	memset(gv->dist_info, 0, sizeof(struct dist_info *)*32);

	gv->ntt = _FALSE;
	gv->ntr = _FALSE;
	gv->rcft_running = _FALSE;
	gv->sgmt_match = _FALSE;

	gv->tchg_notified = _FALSE;
	gv->tchg_detected = _FALSE;
	gv->tchg_opt = _FALSE;

	/* BFU */
	gv->sysidl = NULL;
	memset(gv->gl, 0, sizeof(struct glist_ent *)*32);

	gv->lock = SPIN_LOCK_UNLOCKED;
	gv->dist_lock = SPIN_LOCK_UNLOCKED;
	gv->bfu_lock = SPIN_LOCK_UNLOCKED;

	ring_init(&gv->rb);

	gv->agecnt = 0;

	/* register timers */

	lr_clear_timer(gv, rcf_timer);
	lr_clear_timer(gv, periodic_timer);


	lr_register_timer(gv, rcf_timer);
	lr_register_timer(gv, periodic_timer);
	
	/* start state machines */
	gv->begin = _FALSE;

}


/********************************
 *
 * ioctl() support functions
 *
 ********************************/
static int lr_get_group_id_by_name(char *name)
{
	int i;
	struct lr_group *g;

	if (!name || strlen(name) > MAX_NAME)
		goto exit;

	for (g = lr_get_head_group(), i = 1; g; g = lr_get_next_group(g), i++){
		if (g->dev && g->dev->name)
			if ((strcmp(g->dev->name, name) == 0))
				return i;
	}

exit:
	return 0; /* fail */
}



static void lr_register_port(struct group_var *gv, int n)
{
	int *pl;
	int i, j, k;

	if (!gv || n > MAX_PHYIF || n < 0 )
		return;

	pl = gv->port_list;
	for (i = 0; i < MAX_PHYIF; i++){
		if (pl[i] == n){
			/* already registered */
			return;
		}
	}

	pl = gv->port_list;
	for (i = 0; i < MAX_PHYIF; i++){
		if (!pl[i] || pl[i] > n){
			for (j = MAX_PHYIF-1, k = MAX_PHYIF-2; j != i; j--, k--){
				pl[j] = pl[k];
			}
			pl[i] = n;
			return;
		}
	}
	return;
}

static void lr_unregister_port(struct group_var *gv, int n)
{
	int *pl;
	int j;

	if (!gv || n > MAX_PHYIF || n < 0 )
		return;

	pl = gv->port_list;
	for (j = 0; j < MAX_PHYIF; pl++, j++){
		if ((*pl) == n){
			break;
		}
	}
	for (;j < MAX_PHYIF-1; j++){
		*pl = *(pl+1);
		pl++;
	}
	*pl = 0;
	return;
}


static int lr_get_port_id_by_name(char *ifname)
{
	int i;
	struct lr_port *p;

	if (!ifname || strlen(ifname) > MAX_NAME)
		goto exit;

	/* scan physical i/f name */
	for (p = lr_get_head_port(), i = 1; p; p = lr_get_next_port(p), i++){
		if (p->dev && p->dev->name)
			if (strcmp(p->dev->name, ifname) == 0)
				return i; /* found */
	}

exit:
	return 0; /* not found */
}


static int lr_get_vacant_lr_port(void)
{
	int i;
	struct lr_port *p;

	for (p = lr_get_head_port(), i = 1; p; p = lr_get_next_port(p), i++){
		if (!p->dev)
			return i;
	}
	return 0;
}


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static int lr_catch_device(struct lr_port *p, char *ifname, char *modname)
#else
static int lr_catch_device(struct lr_port *p, char *ifname)
#endif
{
	int ret;

	ret = -EFAULT;
	if (!ifname || !p)
		goto exit;

	ret = -ENODEV;
	if (strlen(ifname) > MAX_NAME)
		goto exit;

	memset(p, 0, sizeof(struct lr_port));

	/* remember this locks device in 2.4 */
	p->dev = dev_get_by_name(ifname);
	if (!p->dev)
		goto exit;		/* invalid ifname */

	if (p->dev->intercepted_ptype ||
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
		p->dev->br_port){
		dev_put(p->dev);
#else
		p->dev->bridge_port_id){
#endif
		p->dev = NULL;
		goto exit;		/* already used by a virtual device driver */
	}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	if (modname && modname[0]){
		strcpy(p->module_name, modname);
		__MOD_INC_USE_COUNT(find_module(p->module_name));
	} else {
		p->module_name[0] = '\0';
	}
#endif

	p->dev->intercepted_ptype = ETH_P_LR;

	/* make the interface to promiscuous mode */
	dev_set_promiscuity(p->dev, 1);
#if 0
	/* Demand loaded and "unused" device driver 
	 * module causes OOPS even though 
	 * set_multicast_list is not NULL!!! */
	if (p->dev->set_multicast_list != NULL){
		p->dev->set_multicast_list(p->dev);
	}
#endif

	ret = 0;

exit:
	return ret;
}

static int lr_release_device(struct lr_port *p)
{
	if (!p || !p->dev)
		return -EFAULT;

	p->dev->intercepted_ptype = 0;

	/* make the interface to promiscuous mode */
	dev_set_promiscuity(p->dev, -1);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	dev_put(p->dev);
#else
	if (p->module_name[0]){
		__MOD_DEC_USE_COUNT(find_module(p->module_name));
		p->module_name[0] = '\0';
	}
#endif
	p->dev = NULL;

	return 0;
}


/* add physical i/f to pool */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
static int lr_add_phyif(char *ifname, int grp)
#else
static int lr_add_phyif(char *ifname, char *modname, int grp)
#endif
{
	int i;
	struct lr_port *p;
	struct lr_group *g;
	struct lr_private *vp;
	int ret;


	ret = -EFAULT;
	if (!ifname)
		goto exit;

	ret = -ENODEV;
	if (strlen(ifname) > MAX_NAME)
		goto exit;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	ret = -EFAULT;
	if (!modname)
		goto exit;
	ret = -ENODEV;
	if (strlen(modname) > MAX_MODNAME)
		goto exit;
#endif

	/* We cannot add a lr i/f to lr
	 * We cannot add a phy i/f already registered
	 * search for vacant lr_ports entry */
	if (lr_get_group_id_by_name(ifname) ||
		lr_get_port_id_by_name(ifname) ||
		!(i = lr_get_vacant_lr_port()))
		goto exit;

	p = lr_get_port_by_num(i);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
	ret = lr_catch_device(p, ifname);
#else
	ret = lr_catch_device(p, ifname, modname);
#endif
	if (ret)
		return ret;

	if (i == 1){
		memcpy(lr_sysvar.system_id, p->dev->dev_addr, ETH_ALEN);
	}

	p->pv.group_id = grp;
	p->pv.local_sgmt.port_id = i;

	g = lr_get_group_by_num(grp);
	vp = (struct lr_private *)g->dev->priv;
	lr_register_port(&vp->group->var, i);

	if (vp->group->var.port_list[0] == i){
		memcpy(vp->group->var.group_addr, p->dev->dev_addr, ETH_ALEN);
		memcpy(g->dev->dev_addr, p->dev->dev_addr, ETH_ALEN);
	}

	ret = 0;

exit:
	return ret;
}


/* delete physical i/f from pool */
static int lr_del_phyif(char *ifname)
{
	int i, j;
	int grp;
	struct lr_port *p;
	struct lr_group *g;
	struct rcte *r, *h;
	struct lr_private *vp;
	struct group_var *gv;
	int ret;

	ret = -EFAULT;
	if (!ifname)
		goto exit;

	ret = -ENODEV;
	if (strlen(ifname) > MAX_NAME)
		goto exit;

	/* We cannot delete a lr i/f 
	 * We cannot delete a phy i/f not registered */
	if (lr_get_group_id_by_name(ifname) ||
		!(i = lr_get_port_id_by_name(ifname)))
		goto exit;

	p = lr_get_port_by_num(i);
	lr_release_device(p);

	grp = p->pv.group_id;

	if (!grp)
		goto exit;

	g = lr_get_group_by_num(grp);
	vp = (struct lr_private *)g->dev->priv;
	gv = &(vp->group->var);
	lr_unregister_port(gv, i);

	j = gv->port_list[0];
	if (j){
		struct net_device *tdev = lr_get_port_by_num(j)->dev;
		memcpy(gv->group_addr, tdev->dev_addr, ETH_ALEN);
		memcpy(g->dev->dev_addr, tdev->dev_addr, ETH_ALEN);
		if (grp == 1)
			memcpy(lr_sysvar.system_id, g->dev->dev_addr, ETH_ALEN);
	}

	h = p->pv.rct;
	for(r = h; r; r = h){
		h = r->next;
		kfree(h);
	}
	memset(p, 0, sizeof(struct lr_port));
	gv->begin = _TRUE;

	ret = 0;

exit:
	return ret;
}

static int lr_get_version(struct u_drv_info *di)
{
 	struct u_drv_info kdi;
 	
	if (!di)
		return -EFAULT;

 	kdi.major = LR_VER_MAJOR;
 	kdi.minor = LR_VER_MINOR;
 	kdi.patch = LR_VER_PATCH;
 	strcpy(kdi.name, "LR");
 	strcpy(kdi.author, "Katsuyuki Yumoto");
 
 	if (copy_to_user(di, &kdi, sizeof(struct u_drv_info)))
 		return -EFAULT;
 
 	return 0;
}

struct net_device *lr_searchfor_vdev(char *vif)
{
	struct net_device *dev = NULL;
	struct lr_group *g;

	if (!vif || strlen(vif) > MAX_NAME)
		return NULL;

	for (g = lr_get_head_group(); g; g = lr_get_next_group(g)){
		if (g->dev && strcmp(vif, g->dev->name) == 0){
			dev = g->dev;
			break;
		}
	}
	return dev;
}


static int lr_get_group_info(struct u_group_info *gi, char *vif)
{
	struct lr_private *vp;
	struct group_var *gv;
	struct net_device *dev;
 	struct u_group_info kgi;
	int ret;

	ret = -EFAULT;
	if (!vif || !gi)
		goto exit;

	ret = -ENODEV;
	if (strlen(vif) > MAX_NAME)
		goto exit;

	dev = lr_searchfor_vdev(vif);
	if (!dev){
		goto exit;
	}

	vp = (struct lr_private *)dev->priv;
	gv = &vp->group->var;

	spin_lock_irq(&gv->lock);

 	kgi.gid = gv->group_id;
 	memcpy(kgi.gaddr, gv->group_addr, ETH_ALEN);
	kgi.perio_time = gv->admin_perio_time/One_Sec;
	kgi.age_time = gv->age_time;
	kgi.tchg_opt = gv->tchg_opt;
	kgi.pii_opt = gv->pii_opt;
	kgi.lstat = get_group_link_stat(dev);

	spin_unlock_irq(&gv->lock);

	ret = -EFAULT;
 	if (copy_to_user(gi, &kgi, sizeof(struct u_group_info)))
 		goto exit;

	ret = 0;
exit:
	return ret;
}

static int lr_set_group_info(struct u_group_info *gi, char *vif)
{
	struct lr_private *vp;
	struct group_var *gv;
	struct net_device *dev;
	struct u_group_info kgi;
	int ret;
	
	ret = -EFAULT;
	if (!vif || !gi)
		goto exit;

	ret = -ENODEV;
	if (strlen(vif) > MAX_NAME)
		goto exit;

	dev = lr_searchfor_vdev(vif);
	if (!dev){
		goto exit;
	}

	vp = (struct lr_private *)dev->priv;
	gv = &vp->group->var;

	ret = -EFAULT;
	if (copy_from_user(&kgi, gi, sizeof(struct u_group_info)))
		goto exit;

	spin_lock_irq(&gv->lock);

	gv->admin_perio_time = kgi.perio_time*One_Sec;
	gv->age_time = kgi.age_time;
	gv->tchg_opt = kgi.tchg_opt;
	gv->pii_opt = kgi.pii_opt;

	spin_unlock_irq(&gv->lock);

	ret = 0;

exit:
	return ret;
}

static int lr_get_port_list(struct u_port_info *pl, char *vif)
{
	struct lr_private *vp;
	struct group_var *gv;
	struct net_device *dev;
	int i, n;
	struct sgmte *sp;
	struct lr_port *p;
 	struct u_port_info kpi;
	int ret = -ENODEV;

	if (!vif || !pl){
		ret = -EFAULT;
		goto out;
	}

	if (strlen(vif) > MAX_NAME)
		goto out;

	dev = lr_searchfor_vdev(vif);
	if (!dev)
		goto out;

	vp = (struct lr_private *)dev->priv;
	gv = &vp->group->var;

	spin_lock_irq(&gv->lock);

	for (i = 0; i < MAX_PHYIF; i++){
		n = gv->port_list[i];
		if (!n || n < 0 || n >= MAX_PHYIF)
			break;

		p = lr_get_port_by_num(n);
		sp = &p->pv.local_sgmt;
 		kpi.gid = p->pv.group_id;
 		kpi.pid = sp->port_id;
 		kpi.aid = sp->active_id;
 		kpi.sgid = sp->sgid;
 		kpi.lstat = get_link_stat(p->dev);
 		strcpy(kpi.dev_name, p->dev->name);
		spin_unlock_irq(&gv->lock);
 		if (copy_to_user(pl, &kpi, sizeof(struct u_port_info))){
 			ret = -EFAULT;
			goto out;
 		}
		spin_lock_irq(&gv->lock);
		pl++;
	}

	spin_unlock_irq(&gv->lock);
	ret = i;
out:
	return ret;
}

static int lr_get_dist_info(struct u_dist_info *di, int size, char *vif)
{
	struct lr_private *vp;
	struct group_var *gv;
	struct net_device *dev;
	struct dist_info *p;
 	struct u_dist_info kdi;
	int i, j;
	unsigned int limit;
	int ret = -ENODEV;

	if (!vif || !di){
		ret = -EFAULT;
		goto out;
	}
	if (size <= 0){
		ret = -EFAULT;
		goto out;
	}

	limit = size/sizeof(struct u_dist_info);

	if (strlen(vif) > MAX_NAME)
		goto out;

	dev = lr_searchfor_vdev(vif);
	if (!dev){
		return -ENODEV;
	}

	vp = (struct lr_private *)dev->priv;
	gv = &vp->group->var;

	spin_lock_irq(&gv->dist_lock);
	for (i = 0, j = 0; i < 32 && j <= limit; i++){
		p = gv->dist_info[i];
		for (; p && j <= limit; p = p->next){
 			memcpy(kdi.m_addr, p->m_addr, ETH_ALEN);
 			kdi.sgid = p->sgid;
 			kdi.age = p->age;
 			kdi.is_bfu = p->bfu_mark;

			atomic_inc(&p->use_count);
			spin_unlock_irq(&gv->dist_lock);
 			if (copy_to_user(di+j, &kdi, sizeof(struct u_dist_info))){
 				spin_unlock_irq(&gv->dist_lock);
 				return -EFAULT;
 			}
			spin_lock_irq(&gv->dist_lock);
			if (atomic_dec_and_test(&p->use_count)){
				/* entry was deleted */
				kfree(p);
				spin_unlock_irq(&gv->dist_lock);
				ret = -EAGAIN;
				goto out;
			}
			j++;
		}
	}
	spin_unlock_irq(&gv->dist_lock);
	ret = j;

out:
	return ret;
}

static int lr_get_bfu_info(struct u_bfu_info *bi, int size, char *vif)
{
	struct lr_private *vp;
	struct group_var *gv;
	struct net_device *dev;
	struct glist_ent *p;
	struct sglist_ent *q;
 	struct u_bfu_info kbi;
	int i, j;
	unsigned int limit;
	int ret;

	ret = -EFAULT;
	if (!vif || !bi){
		goto out;
	}
	if (size <= 0){
		goto out;
	}

	limit = size/sizeof(struct u_dist_info);

	ret = -ENODEV;
	if (strlen(vif) > MAX_NAME)
		goto out;

	dev = lr_searchfor_vdev(vif);
	if (!dev)
		goto out;

	vp = (struct lr_private *)dev->priv;
	gv = &vp->group->var;

	spin_lock_irq(&gv->bfu_lock);

	for (i = 0, j = 0; i < 32 && j <= limit; i++){
		p = gv->gl[i];
		for (; p && j <= limit; p = p->next){
 			memcpy(kbi.gaddr, p->gaddr, ETH_ALEN);
 			kbi.min_sgid = p->min_sgid;
 			kbi.age = 0;
			for (q = p->sgl; q && p->min_sgid; q = q->next)
				if (p->min_sgid == q->rcv_sgid)
 					kbi.age = q->age;
			atomic_inc(&p->use_count);
			spin_unlock_irq(&gv->bfu_lock);

 			if (copy_to_user(bi+j, &kbi, sizeof(struct u_bfu_info))){
 				spin_unlock_irq(&gv->bfu_lock);
 				ret = -EFAULT;
				goto out;
 			}

			spin_lock_irq(&gv->bfu_lock);
			if (atomic_dec_and_test(&p->use_count)){
				/* entry was deleted */
				kfree(p);
				spin_unlock_irq(&gv->bfu_lock);
				ret = -EAGAIN;
				goto out;
			}
			j++;
		}
	}

	spin_unlock_irq(&gv->bfu_lock);
	ret = j;
out:
	return ret;
}

static int lr_get_sys_info(struct u_sys_info *si)
{
 	struct u_sys_info ksi;
 
	if (!si)
		return -EFAULT;

 	memcpy(ksi.sys_id, lr_sysvar.system_id, ETH_ALEN);
 	ksi.prio = lr_sysvar.priority;
 
 	if (copy_to_user(si, &ksi, sizeof(struct u_sys_info)))
 		return -EFAULT;
 
 	return 0;
}

int lr_alloc_private(struct lr_group *g)
{
	struct lr_private *vp;

	if (!g)
		return _FALSE;

	vp = g->dev->priv = 
		(struct lr_private *)kmalloc(sizeof(struct lr_private), GFP_KERNEL);
	
	if (vp == NULL) {
		printk(KERN_ERR "%s: out of memory.\n", g->dev->name);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
		kfree(g->dev->name);
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
		kfree(g->dev);
#else
       free_netdev(g->dev);
#endif
		g->dev = NULL;
		return _FALSE;
	}
	memset(vp, 0, sizeof(struct lr_private));

	vp->group = g;

	return _TRUE;
}

static int lr_accept_fastpath(struct net_device *dev, struct dst_entry *dst)
{
	return -1;
}

extern int lr_ioctl(struct net_device *, struct ifreq *, int );

static int lr_init_group(int n)
{
	struct lr_group *g;
	struct group_var *gv;
	int ret;

	ret = -ENODEV;
	if (n < 0 || n > MAX_PHYIF)
		return ret;

	g = lr_get_group_by_num(n);
	if (!g)
		return ret;

	ret = -ENOMEM;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	g->dev = kmalloc(sizeof(struct net_device), GFP_KERNEL);
#else
	g->dev = alloc_etherdev(sizeof(struct net_device));
#endif
	if (!g->dev)
		goto freeandout3;

	memset(g->dev, 0, sizeof(struct net_device));
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	g->dev->name = kmalloc(strlen("lrXXX"), GFP_KERNEL);
	
	if (!g->dev->name)
		goto freeandout2;
#endif	
	sprintf(g->dev->name, "lr%d", n-1);


	/* Set the private structure */
	if (!lr_alloc_private(g))
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
		goto freeandout1;
#else
		goto freeandout2;
#endif

	gv = &g->var;
	lr_group_vars_init(gv);
	gv->group_id = n;

	/* Fill in the generic fields of the device structure. */
	ether_setup(g->dev);

	/* Then, override parts of it */
	g->dev->hard_start_xmit	= lr_group_distribution;
	g->dev->open		= lr_open;
	g->dev->stop		= lr_close;
	g->dev->get_stats 		= lr_get_stats;
	g->dev->accept_fastpath	= lr_accept_fastpath;
	g->dev->set_mac_address		= NULL;
	g->dev->do_ioctl		= lr_ioctl;
/*	g->dev->flags	        = IFF_BROADCAST | IFF_MULTICAST;*/
	memset(g->dev->dev_addr, 0xfc, ETH_ALEN/2);
	g->dev->dev_addr[ETH_ALEN-1] = n;
	g->dev->tx_queue_len = 0;

	ret = register_netdev(g->dev);
	if (ret)
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
		goto freeandout1;
#else
		goto freeandout2;
#endif
	init_timer(&g->tl);

	g->tl.expires = jiffies + HZ/One_Sec; /* 50 msec */
	g->tl.function = lr_tick;
	g->tl.data = n;
	add_timer(&g->tl);

	return ret;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
freeandout1:
	kfree(g->dev->name);
#endif
freeandout2:
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
    kfree(g->dev);
#else
	free_netdev(g->dev);
#endif
	g->dev = NULL;
freeandout3:
	printk(KERN_ERR "lr: out of memory.\n");
	return ret;
}


static int lr_cleanup_group(struct lr_group *g)
{
	struct group_var *gv;

	if (!g)
		return -EFAULT;

	gv = &g->var;
	/* unregister timers */
	lr_unregister_timer(gv, rcf_timer);
	lr_unregister_timer(gv, periodic_timer);
	ring_flush(&gv->rb);
	unregister_netdev(g->dev);

	lr_cleanup_bfu(gv);
	lr_clear_distinfo(gv);

	kfree(g->dev->priv);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	kfree(g->dev->name);
#endif
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	kfree(g->dev);
#else
	free_netdev(g->dev);
#endif
	g->dev = NULL;
	del_timer_sync(&g->tl);

	return 0;
}

#if 0
static int lr_add_group((struct u_group_info *)gi)
{
}

static int lr_del_group((struct u_group_info *)gi)
{
}
#endif

static int lr_ioctl_dispatch(struct lrconf *vc)
{
	int n;

	switch (vc->action){
		case LRCTL_VERSION: /* get device driver version */
 			return lr_get_version((struct u_drv_info *)vc->buf);

		case LRCTL_ADD_PHYIF: /* add physical i/f to pool*/
			n = lr_get_group_id_by_name(vc->vif_name);
			if (!n)
				return -ENODEV;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
			n = lr_add_phyif(vc->pif_name, n);
#else
			n = lr_add_phyif(vc->pif_name, vc->pif_module, n)
#endif
			return n;

		case LRCTL_DEL_PHYIF: /* delete physical i/f from pool */
			n = lr_del_phyif(vc->pif_name);
			return n;

		case LRCTL_GET_PORTLIST: /* get physical i/f array */
			n = lr_get_port_list((struct u_port_info *)vc->buf, vc->vif_name);
			if (n < 0)
				return n;
			vc->size = n;
			break;
		case LRCTL_GET_DISTINFO:
			n = lr_get_dist_info((struct u_dist_info *)vc->buf, vc->size, vc->vif_name);
			if (n < 0)
				return n;
			vc->size = n;
			break;
		case LRCTL_GET_BFUINFO:
			n = lr_get_bfu_info((struct u_bfu_info *)vc->buf, vc->size, vc->vif_name);
			if (n < 0)
				return n;
			vc->size = n;
			break;
		case LRCTL_GET_SYSINFO:
			return lr_get_sys_info((struct u_sys_info *)vc->buf);

		case LRCTL_GET_GRPINFO:
			n = lr_get_group_info((struct u_group_info *)vc->buf, vc->vif_name);
			if (n < 0)
				return n;
			vc->size = n;
			break;
		case LRCTL_SET_GRPINFO:
			n = lr_set_group_info((struct u_group_info *)vc->buf, vc->vif_name);
			if (n < 0)
				return n;
			vc->size = n;
			break;
#if 0
		case LRCTL_ADD_GRP:
			n = lr_add_group((struct u_group_info *)vc->buf);
			if (n < 0)
				return -EINVAL;
			vc->size = n;
			break;
		case LRCTL_DEL_GRP:
			n = lr_del_group((struct u_group_info *)vc->buf);
			if (n < 0)
				return -EINVAL;
			vc->size = n;
			break;
#endif
	}
	return 0;
}

int lr_ioctl(struct net_device *adev, struct ifreq *rq, int cmd)
{
	struct mii_ioctl_data *data;

	if (!rq || !rq->ifr_data || !adev)
		return -EFAULT;

	data = (struct mii_ioctl_data *)&rq->ifr_data;

	if (cmd == SIOCDEVPRIVATE){
		struct lrconf *vc = (struct lrconf *) rq->ifr_data;
		struct lrconf kvc;
		if (!copy_from_user(&kvc, vc, sizeof(struct lrconf)) &&
			(kvc.magic == LR_MAGIC))
			return lr_ioctl_dispatch(vc);
		/* else
		 *    fall through to keep ioctl() compatibility
		 *    with many ethernet device drivers. They provides
		 *    MII functions commonly using SIOCDEVPRIVATE space. */
	}
	switch (cmd){
		case SIOCGMIIPHY:
			data->phy_id = 0;	
		case SIOCGMIIREG:
			printk("adev->name = %s\n", adev->name);
			data->val_out = get_group_link_stat(adev);
			break;
		default:
			return -EOPNOTSUPP;
	}
	return 0;
}


/* Write the buffer (contents of the port statuses) to a PROCfs file */

static char *state2str_sgm(enum sgmm_state s)
{
	switch (s){
	case INITIALIZE:
		return "INIT";
	case WAIT_FOR_NTR:
		return "W_NTR";
	case REBUILD_START:
		return "RBLD_ST";
	case WAIT_FOR_RCF_NOTIFY:
		return "W_NOTFY";
	case RCF_RECEIVED:
		return "RCF_REC";
	case RCF_TIMEOUT:
		return "RCF_TMO";
	case NOT_CHANGED:
		return "NOT_CHG";
	case CHANGED:
		return "CHANGED";
	}
	return NULL;
}

static char *state2str_rcft(enum rcftm_state s)
{
	switch (s){
	case NOT_RUNNING:
		return "NOT_RUN";
	case RUNNING:
		return "RUNNING";
	case TIMEOUT:
		return "TMOUT";
	case TIMEOUT2:
		return "TMOUT2";
	}
	return NULL;
}

static char *state2str_pr(enum perio_state s)
{
	switch (s){
	case DISABLED:
		return "DISABLE";
	case WAIT_FOR_TIMEOUT:
		return "W_TMOUT";
	case EXPIRED:
		return "EXPIRED";
	}
	return NULL;
}

static int sprintf_a_line(char *buf, struct net_device *devlr)
{
	int len;

	struct lr_private *vp = (struct lr_private *)devlr->priv;
	struct group_var *gv = &(vp->group->var);

	len =  sprintf(buf, "%-8s", devlr->name);
	len += sprintf(buf+len, "%-8s", state2str_sgm(gv->sgmm.state));
	len += sprintf(buf+len, "%-8s", state2str_rcft(gv->rcftm.state));
	len += sprintf(buf+len, "%-8s", state2str_pr(gv->prm.state));
	len += sprintf(buf+len, "\n");

	return len;
}


static int sprintf_a_line_port(char *buf, struct net_device *devlr)
{
	int len = 0;
	int i, n;
	struct sgmte *sp;
	struct lr_private *vp = (struct lr_private *)devlr->priv;
	struct group_var *gv = &(vp->group->var);
	struct lr_port *p;

	spin_lock_irq(&gv->lock);
	for (i = 0; i < MAX_PHYIF; i++){
		n = gv->port_list[i];
		if (!n || n < 0 || n >= MAX_PHYIF)
			break;

		p = lr_get_port_by_num(n);
		sp = &p->pv.local_sgmt;
		len += sprintf(buf+len, "%-8s%c%s(%d) : %d\n",
			devlr->name,
			sp->active?'*':' ',
			lr_get_port_by_num(sp->port_id)->dev->name,
			sp->port_id, sp->sgid);
	}
	spin_unlock_irq(&gv->lock);
	len += sprintf(buf+len, "\n");

	return len;
}
static int sprintf_a_line_distinfo(char *buf, struct net_device *devlr)
{
	int len;
	int i;
	struct dist_info *p;
	struct lr_private *vp = (struct lr_private *)devlr->priv;
	struct group_var *gv = &(vp->group->var);
	char strbuf[16];

	sprintf(strbuf, "%-8s", devlr->name);
	len = sprintf(buf, "%s", "GRP     Neighbors    SubGRP\n");

	spin_lock_irq(&gv->dist_lock);
	for (i = 0; i < 32; i++){
		p = gv->dist_info[i];
		for (; p; p = p->next){
			len += sprintf(buf+len, "%s%02x%02x%02x-%02x%02x%02x %4d %4d\n",
				strbuf,
				p->m_addr[0],
				p->m_addr[1],
				p->m_addr[2],
				p->m_addr[3],
				p->m_addr[4],
				p->m_addr[5],
				p->sgid, p->age);
		}
	}
	spin_unlock_irq(&gv->dist_lock);
	len += sprintf(buf+len, "\n");

	return len;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
static int 
lr_get_info(char *pbuf, char **start, off_t offset, int length, int unused)
#else
static int lr_get_info(char *pbuf, char **start, off_t offset, int length)
#endif
{
#define LIMIT (PAGE_SIZE-80);

	int size;
	int len = 0;
	struct lr_group *g;

	if(!offset){
		/* first time write the header */
		size = sprintf(pbuf,"%s","GRP     SGM     RCFM    PERIO   \n");
		len = size;
	}

	for (g = lr_get_head_group(); g; g = lr_get_next_group(g)){
		if (g->dev){
			len += sprintf_a_line(pbuf+len, g->dev);
		}
	}

	len += sprintf(pbuf+len, "\n Port-Subgroup Mapping\n");
	for (g = lr_get_head_group(); g; g = lr_get_next_group(g)){
		if (g->dev){
			len += sprintf_a_line_port(pbuf+len, g->dev);
		}
	}
	len += sprintf(pbuf+len, "\n Distributing Information\n");
	for (g = lr_get_head_group(); g; g = lr_get_next_group(g)){
		if (g->dev){
			len += sprintf_a_line_distinfo(pbuf+len, g->dev);
		}
	}

	if (len > PAGE_SIZE-80){
		len = PAGE_SIZE-80;
	}

	return len;
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
#ifdef CONFIG_PROC_FS
struct proc_dir_entry proc_net_lr= {
	0, 4, "lrcp",
	S_IFREG | S_IRUGO, 1, 0, 0,
	0, &proc_net_inode_operations,
	lr_get_info
};
#endif
#endif


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
void cleanup_module(void)
#else
 void __exit lr_cleanup_module(void)
#endif
{
	struct lr_port *p;
	struct lr_group *g;

	for (p = lr_get_head_port(); p; p = lr_get_next_port(p)){
		if (p->dev)
			lr_del_phyif(p->dev->name);
	}

	/*
	 * Physical interfaces are already released. 
	 * Now we shouldn't receive any packet on given group.
	 */
	for (g = lr_get_head_group(); g; g = lr_get_next_group(g)) {
		if (!g->dev)
			continue;

		lr_cleanup_group(g);
	}

#ifdef CONFIG_PROC_FS
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	proc_net_unregister(proc_net_lr.low_ino);
#else
	proc_net_remove("lrcp");
#endif
#endif

	/* remove packet handler */
	ic_lr_cleanup();
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
#define lr_init  init_module
#endif


/* 
 * Entry point of LRCP driver.
 * Register/initialize the driver.
 */

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
__initfunc(int lr_init(void))
#else
 int __init lr_init_module(void)
#endif
{
	int i;
	int ret;
	struct lr_group *g;
	struct lr_port *p;

	printk(KERN_INFO "LRCP %d.%d.%d Yumo (Katsuyuki Yumoto) yumo@st.rim.or.jp\n",
		   (int)LR_VER_MAJOR, (int)LR_VER_MINOR, (int)LR_VER_PATCH);

	for (g = lr_get_head_group(); g; g = lr_get_next_group(g)){
		g->dev = NULL;
	}

	for (p = lr_get_head_port(); p; p = lr_get_next_port(p)){
		p->dev = NULL;
	}

	/* nlr comes from module parameter and it is default to 1 */
	for (i = 1; i <= nlr; i++){
		ret = lr_init_group(i);
		if (ret){
			printk(KERN_INFO "lr: lr%d device not registered\n", i-1);
			return ret;
		}
	}

	lr_sysvar.priority = 0x8000;
	
#ifdef CONFIG_PROC_FS
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
	proc_net_register(&proc_net_lr);
#else
	proc_net_create("lrcp", 0, lr_get_info);
#endif
#endif

	/* packet handler registration for receiving */
	ic_lr_init();

	return 0;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
module_init(lr_init_module);
module_exit(lr_cleanup_module);
MODULE_AUTHOR("Katsuyuki Yumoto");
MODULE_DESCRIPTION("Link redundancy configuration protocol driver");
MODULE_LICENSE("GPL");
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#include <linux/vermagic.h>
#include <linux/compiler.h>
MODULE_INFO(vermagic, VERMAGIC_STRING);
MODULE_DESCRIPTION("Link redundancy configuration protocol driver");
MODULE_AUTHOR("Katsuyuki Yumoto");
#endif
