/*
 * Copyright (c) 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses iSCSI Initiator login neogitation "State Machine" specific functions.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_NEGOTIATE_C

#include <linux/string.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>     
#include <iscsi_initiator_debug_opcodes.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_discovery.h>
#include <iscsi_initiator_nego.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_crc.h>
#include <iscsi_initiator_parameters.h>
#include <iscsi_auth_kernel.h>

#undef ISCSI_INITIATOR_NEGOTIATE_C

#define MAX_LOGIN_PDUS  7

extern char *iscsi_ntoa (u32);

typedef struct iscsi_login_s {
	u8 current_stage;
	u8 leading_connection;
	u8 first_response;
	u8 first_auth;
	u8 version_min;
	u8 version_max;
	char isid[6];
	u32 init_task_tag;
	u16 tsih;
	char *req;
	char *rsp;
	char *req_buf;
	char *rsp_buf;
	iscsi_login_holder_t *lh;
} iscsi_login_t;

/*	iscsi_initiator_check_login_response():
 *
 *
 */
static int iscsi_initiator_check_login_response(iscsi_conn_t *conn, iscsi_login_t *login)
{
	int req_csg, req_nsg, rsp_csg, rsp_nsg;
	iscsi_login_holder_t *lh = login->lh;
	struct iscsi_init_login_cmnd *login_req;
	struct iscsi_targ_login_rsp *login_rsp;

	login_req = (struct iscsi_init_login_cmnd *) login->req;
	login_rsp = (struct iscsi_targ_login_rsp *) login->rsp;

	switch (login_rsp->opcode & ISCSI_OPCODE) {
		case ISCSI_TARG_LOGIN_RSP:
			break;
		case ISCSI_TARG_RJT:
			TRACE_ERROR("Got REJECT Opcode\n");
			return(-1);
		default:
			TRACE_ERROR("Got unknown Opcode\n");
			return(-1);
	}

	if (login_rsp->status_class != 0x00) {
		TRACE_ERROR("Login Response with non-zero StatusClass 0x%02x,"
			" StatusDetail 0x%02x.\n", login_rsp->status_class,
				login_rsp->status_detail);
		lh->status_class = login_rsp->status_class;
		lh->status_detail = login_rsp->status_detail;
		print_status_class_and_detail(lh->status_class, lh->status_detail);
		
		/*
		 * Only Normal iSCSI Sessions will have this pointer present.
		 * iscsi_handle_targetaddress_keys() will add the TargetAddress
		 * key returned during in the last Login Response containg
		 * a Redirection Status Class.
		 */
		if ((login_rsp->status_class == 0x01) && conn->target_address)
			lh->login_ret = 1; /* Redirection */
		
		return((lh->login_ret == 1) ? 1 : -1);
	}

	if (login_rsp->status_detail != 0x00) {
		TRACE_ERROR("Login Response with zero StatusClass but non-zero"
			" StatusDetail 0x%02x.\n", login_rsp->status_detail);
		print_status_class_and_detail(login_rsp->status_class,
				login_rsp->status_detail);
		return(-1);
	}

	if ((login_rsp->flags & C_BIT) && (login_rsp->flags & T_BIT)) {
		TRACE_ERROR("Login Response has both C_BIT and T_BIT set,"
				" protocol error.\n");
		return(-1);
	}

	if (!(login_req->flags & T_BIT) && (login_rsp->flags & T_BIT)) {
		TRACE_ERROR("Login Response with T bit set but Initiator's"
			" T bit not set, protocol error.\n");
		return(-1);
	}

	req_csg = (login_req->flags & CSG) >> CSG_SHIFT;
	rsp_csg = (login_rsp->flags & CSG) >> CSG_SHIFT;
	req_nsg = (login_req->flags & NSG);
	rsp_nsg = (login_rsp->flags & NSG);

	if (rsp_csg != login->current_stage) {
		TRACE_ERROR("Target unexpectly changed login stage from"
			" %d to %d, login failed.\n", login->current_stage, rsp_csg);
		return(-1);
	}

	if ((login_rsp->flags & T_BIT) && ((rsp_nsg <= rsp_csg) || (rsp_nsg == 2))) {
		TRACE_ERROR("Illegal login_rsp->flags Combination, CSG: %d,"
			" NSG: %d, T_BIT: %d.\n", rsp_csg, rsp_nsg,
				(login_rsp->flags & T_BIT));
		return(-1);
	}

	if ((login_rsp->version_max != login->version_max) ||
	    (login_rsp->version_active != login->version_min)) {
		TRACE_ERROR("Login request changed Version Max/Active unexpectedly"
			" to 0x%02x/0x%02x, protocol error\n",
			login_rsp->version_max, login_rsp->version_active);
		return(-1);
	}

	if (memcmp(login_rsp->isid, login->isid, 6) != 0) {
		TRACE_ERROR("Login response changed ISID unexpectedly,"
				" protocol error.\n");
		return(-1);
	}

	if (login_req->init_task_tag != login->init_task_tag) {
		TRACE_ERROR("Login response changed ITT unexpectedly to 0x%08x,"
			" protocol error.\n", login_rsp->init_task_tag);
		return(-1);
	}

	if (login_rsp->length > MAX_TEXT_LEN) {
		TRACE_ERROR("Login response payload exceeds default"
			 " MaxRecvDataSegmentLength: %u, protocol error.\n",
			 	MAX_TEXT_LEN);
		return(-1);
	}

	/*
	 * Initiator sets the ExpStatSN from the first login response for
	 * the connection,  and checks against for all subsequent login
	 * responses.
	 *
	 * See iSCSI RFC 3347 10.12.9 and 10.13.4
	 */
	if (login->first_response)
		conn->exp_statsn = (login_rsp->stat_sn + 1);
	else {
		if (login_rsp->stat_sn != conn->exp_statsn) {
			TRACE_ERROR("Login response with incorrect StatSN:"
				" 0x%08x, expected 0x%08x\n", login_rsp->stat_sn,
					conn->exp_statsn);
			return(-1);
		}
		conn->exp_statsn++;
	}

	return(0);
}

/*	iscsi_initiator_check_first_response():
 *
 *
 */
static int iscsi_initiator_check_first_response(
	iscsi_conn_t *conn,
	iscsi_login_t *login)
{
	char *tmpptr;
	u16 TargetPortalGroupTag;
	iscsi_param_t *param = NULL;

	for (param = conn->param_list->param_start; param; param = param->next) {
		if (!strcmp(param->name, TARGETPORTALGROUPTAG)) {
			if (!conn->target_address)
				continue;
			
			if (!IS_PSTATE_ACCEPTOR(param)) {
				TRACE_ERROR("TargetPortalGroupTag key not"
					" received in first login response.\n");
				return(-1);
			}
			TargetPortalGroupTag = simple_strtoul(param->value, &tmpptr, 0);

			if (conn->target_address->need_tpgt) {
				conn->target_address->tpgt = TargetPortalGroupTag;
				conn->target_address->need_tpgt = 0;
			}

			if (conn->target_address->tpgt != TargetPortalGroupTag) {
				TRACE_ERROR("TargetPortalGroupTag %hu is different"
					" from previously received TargetAddress"
					" key %s:%hu,%hu.\n", TargetPortalGroupTag,
			       		iscsi_ntoa(conn->target_address->ipv4_address),
					conn->target_address->port,	
					conn->target_address->tpgt);
				return(-1);
			}
		}	
	}

	login->first_response = 0;

	return(0);
}

static int iscsi_initiator_check_last_response(
	iscsi_conn_t *conn,
	iscsi_login_t *login)
{
	struct iscsi_targ_login_rsp *login_rsp;

	login_rsp = (struct iscsi_targ_login_rsp *) login->rsp;

	if (login->leading_connection) {
		if (!login_rsp->tsih) {
			TRACE_ERROR("Final login response does not contain"
				" a non-zero TSIH for the leading connection"
					" of a session, protocol error.\n");
			return(-1);
		}
	}

	return(0);
}

/*	iscsi_initiator_do_tx_login_io():
 *
 *
 */
static int iscsi_initiator_do_tx_login_io(iscsi_conn_t *conn, iscsi_login_t *login)
{
	u32 padding = 0;
	struct iscsi_init_login_cmnd *login_req;

	login_req = (struct iscsi_init_login_cmnd *) login->req;

	login_req->exp_stat_sn = (login->first_response) ?
			login_req->exp_stat_sn : conn->exp_statsn;

	TRACE(TRACE_LOGIN, "Sending Login Command, Flags: 0x%02x, ITT: 0x%08x,"
		" CmdSN: 0x%08x, ExpStatSN: 0x%08x, Length: %u\n",
		login_req->flags, login_req->init_task_tag, login_req->cmd_sn,
		login_req->exp_stat_sn, login_req->length);

	login_req->opcode		= ISCSI_INIT_LOGIN_CMND | I_BIT;
	login_req->length		= cpu_to_be32(login_req->length);
	memcpy(login_req->isid, login->isid, 6);
	login_req->tsih			= cpu_to_be16(login_req->tsih);
	login_req->init_task_tag	= cpu_to_be32(login_req->init_task_tag);
	login_req->cid			= cpu_to_be16(login_req->cid);
	login_req->cmd_sn		= cpu_to_be32(login_req->cmd_sn);
	login_req->exp_stat_sn		= cpu_to_be32(login_req->exp_stat_sn);

	padding = ((-ntohl(login_req->length)) & 3);

	if (iscsi_login_tx_data(conn, login->req, login->req_buf,
	    ntohl(login_req->length) + padding, INITIATOR) < 0)
		return(-1);

	login_req->length		= 0;	
	login_req->tsih			= be16_to_cpu(login_req->tsih);
	login_req->init_task_tag	= be32_to_cpu(login_req->init_task_tag);
	login_req->cid			= be16_to_cpu(login_req->cid);
	login_req->cmd_sn		= be32_to_cpu(login_req->cmd_sn);
	login_req->exp_stat_sn		= be32_to_cpu(login_req->exp_stat_sn);

	return(0);
}
	
/*	 iscsi_initiator_do_rx_login_io():
 *
 *
 */	
static int iscsi_initiator_do_rx_login_io(iscsi_conn_t *conn, iscsi_login_t *login)
{
	u32 padding = 0, ret = 0;
	struct iscsi_targ_login_rsp *login_rsp;

	if (iscsi_login_rx_data(conn, login->rsp, ISCSI_HDR_LEN, INITIATOR) < 0)
		return(-1);

	login_rsp = (struct iscsi_targ_login_rsp *) login->rsp;
	login_rsp->length               = be32_to_cpu(login_rsp->length);
	login_rsp->tsih                 = be16_to_cpu(login_rsp->tsih);
	login_rsp->init_task_tag        = be32_to_cpu(login_rsp->init_task_tag);
	login_rsp->stat_sn              = be32_to_cpu(login_rsp->stat_sn);
	login_rsp->exp_cmd_sn           = be32_to_cpu(login_rsp->exp_cmd_sn);
	login_rsp->max_cmd_sn           = be32_to_cpu(login_rsp->max_cmd_sn);
        
	TRACE(TRACE_LOGIN, "Got Login Response, Flags: 0x%02x, ITT: 0x%08x,"
		" ExpCmdSN: 0x%08x, MaxCmdSN: 0x%08x, StatSN: 0x%08x,"
		" Length: %u\n", login_rsp->flags, login_rsp->init_task_tag,
		login_rsp->exp_cmd_sn, login_rsp->max_cmd_sn,
		login_rsp->stat_sn, login_rsp->length);

	if ((ret = iscsi_initiator_check_login_response(conn, login)) < 0)
		return(ret);

	padding = ((-login_rsp->length) & 3);
	memset(login->rsp_buf, 0, MAX_TEXT_LEN);

	if (iscsi_login_rx_data(conn, login->rsp_buf,
	    login_rsp->length + padding, INITIATOR) < 0)
		return(-1);

	/*
	 * Perform Redirection
	 */
	if (ret == 1) {
		iscsi_login_holder_t *lh = login->lh;
		iscsi_target_rdr_t rdr;

		memset(&rdr, 0, sizeof(iscsi_target_rdr_t));
		rdr.rdr_tpgt = conn->target_address->tpgt;
		
		if (iscsi_handle_targetaddress_keys(login->rsp_buf, &rdr,
				conn->target_address->target) < 0)
			return(-1);

		conn->login_ip = lh->ipv4_address = rdr.rdr_ipv4_address;
		conn->login_port = lh->port = rdr.rdr_port;
	}
	
	return(ret);
}

/*	iscsi_initiator_do_login_io():
 *
 *
 */
static int iscsi_initiator_do_login_io(iscsi_conn_t *conn, iscsi_login_t *login)
{
	int ret;
	
	if (iscsi_initiator_do_tx_login_io(conn, login) < 0)
		return(-1);

	if ((ret = iscsi_initiator_do_rx_login_io(conn, login)) != 0)
		return(ret);

	return(0);
}

/*	iscsi_initiator_check_key_for_transit():
 *
 *	We use this function to check the outbound buffer that the authentication
 *	daemon has built for keys that signal us to set the NSG1 and T_BIT bits.
 *	See iSCSI Draft v20, Appendix C.
 */
static int iscsi_initiator_check_key_for_transit(iscsi_conn_t *conn, iscsi_login_t *login)
{
	char *CHAP_N = NULL, *SRP_M = NULL;
	iscsi_param_t *param;
	struct iscsi_init_login_cmnd *login_req;

	login_req = (struct iscsi_init_login_cmnd *) login->req;

	if (!(param = iscsi_find_param_from_key(AUTHMETHOD, conn->param_list)))
		return(-1);

	if (strcmp(param->value, CHAP) && strcmp(param->value, SRP))
		return(0);

	if ((CHAP_N = strstr(login->req_buf, "CHAP_N")))
		login_req->flags |= (NSG1 | T_BIT);
	if ((SRP_M = strstr(login->req_buf, "SRP_M")))
		login_req->flags |= (NSG1 | T_BIT);

	return(0);
}

/*	iscsi_initiator_do_authentication():
 *
 *
 */
static int iscsi_initiator_do_authentication(iscsi_conn_t *conn, iscsi_login_t *login)
{
	int authret;
	iscsi_param_t *param;
	struct iscsi_init_login_cmnd *login_req;
	struct iscsi_targ_login_rsp *login_rsp;

	login_req = (struct iscsi_init_login_cmnd *) login->req;
	login_rsp = (struct iscsi_targ_login_rsp *) login->rsp;

	if (!login->first_auth)
		login->first_auth = 1;
	else {
		memset(login->req_buf, 0, MAX_TEXT_LEN);
		memcpy(login->req_buf, login->rsp_buf, login_rsp->length);
		login_req->length = login_rsp->length;
	}

	if (!(param = iscsi_find_param_from_key(AUTHMETHOD, conn->param_list)))
		return(-1);

	authret = iscsi_handle_authetication(
			conn,
			login->req_buf,
			login->rsp_buf,
			login_req->length,
			&login_rsp->length,
			param->value,
			AUTH_CLIENT);
	switch (authret) {
		case 0:
			printk("Received OK response"
			" from Core-iSCSI Authentication"
			" Daemon, continuing.\n");
			break;
		case 1:
			printk("iSCSI security negotiation"
				" completed sucessfully.\n");
			break;
		case 2:
			TRACE_ERROR("Security negotiation"
				" failed.\n");
			return(-1);
		case 3:
			printk("iSCSI mutual"
				" authetication completed"
				" successfully.\n");
			login_req->length = 0;
			return(1);
		default:
			TRACE_ERROR ("Received unknown"
			" error from Core-iSCSI Authentication"
				" Daemon.\n");
			return(-1);
	}

	memset(login->req_buf, 0, MAX_TEXT_LEN); 
	memcpy(login->req_buf, login->rsp_buf, login_rsp->length);
	login_req->length = login_rsp->length;

	return(iscsi_initiator_check_key_for_transit(conn, login));
}

/*	iscsi_initiator_post_csg_zero():
 *
 *
 */
static int iscsi_initiator_post_csg_zero(iscsi_conn_t *conn, iscsi_login_t *login)
{
	int ret;
	struct iscsi_init_login_cmnd *login_req;
	struct iscsi_targ_login_rsp *login_rsp;

	login_req = (struct iscsi_init_login_cmnd *) login->req;
	login_rsp = (struct iscsi_targ_login_rsp *) login->rsp;

	ret = iscsi_encode_text_output(
			PHASE_OPERATIONAL|PHASE_DECLARATIVE,
			SENDER_INITIATOR,
			login->req_buf,
			&login_req->length,
			conn->param_list);
	if (ret < 0)
		return(-1);

	login_req->flags &= ~NSG1;
	login_req->flags |= (CSG1 | NSG3);
	login->current_stage = 1;

	return(0);
}

/*	iscsi_initiator_handle_csg_zero():
 *
 *
 */
static int iscsi_initiator_handle_csg_zero(iscsi_conn_t *conn, iscsi_login_t *login)
{	
	int ret;
	iscsi_param_t *param;
	struct iscsi_init_login_cmnd *login_req;
	struct iscsi_targ_login_rsp *login_rsp;
	
	login_req = (struct iscsi_init_login_cmnd *) login->req;
	login_rsp = (struct iscsi_targ_login_rsp *) login->rsp;

	ret = iscsi_decode_text_input(
			PHASE_SECURITY|PHASE_DECLARATIVE,
			SENDER_TARGET|SENDER_RECEIVER,
			login->rsp_buf,
			login_rsp->length,
			conn->param_list);
	if (ret < 0)
		return(-1);
	else if (ret > 0)
		goto do_auth;

	if (login->first_response)
		if (iscsi_initiator_check_first_response(conn, login) < 0)
			return(-1);

	ret = iscsi_encode_text_output(
			PHASE_SECURITY|PHASE_DECLARATIVE,
			SENDER_INITIATOR,
			login->req_buf,
			&login_req->length,
			conn->param_list);
	if (ret < 0)
		return(-1);

	if (login_rsp->flags & T_BIT)
		return(iscsi_initiator_post_csg_zero(conn, login));

	if (!(param = iscsi_find_param_from_key(AUTHMETHOD, conn->param_list)))
		return(-1);

	if (!strcmp(param->value, NONE))
		return(iscsi_initiator_post_csg_zero(conn, login));

	if ((IS_PSTATE_PROPOSER(param) && IS_PSTATE_RESPONSE_GOT(param)) ||
	     IS_PSTATE_ACCEPTOR(param))
		goto do_auth;

	return(0);
do_auth:
	ret = iscsi_initiator_do_authentication(conn, login);
	if (ret > 0)
		return(iscsi_initiator_post_csg_zero(conn, login));
	return(ret);
}

/*	iscsi_initiator_handle_csg_one():
 *
 *
 */
static int iscsi_initiator_handle_csg_one(iscsi_conn_t *conn, iscsi_login_t *login)
{
	int ret;
	struct iscsi_init_login_cmnd *login_req;
	struct iscsi_targ_login_rsp *login_rsp;

	login_req = (struct iscsi_init_login_cmnd *) login->req;
	login_rsp = (struct iscsi_targ_login_rsp *) login->rsp;

	ret = iscsi_decode_text_input(
			PHASE_OPERATIONAL|PHASE_DECLARATIVE,
			SENDER_TARGET|SENDER_RECEIVER,
			login->rsp_buf,
			login_rsp->length,
			conn->param_list);
	if (ret < 0)
		return(-1);

	if (login->first_response)
		if (iscsi_initiator_check_first_response(conn, login) < 0)
			return(-1);

	ret = iscsi_encode_text_output(
			PHASE_OPERATIONAL|PHASE_DECLARATIVE,
			SENDER_INITIATOR,
			login->req_buf,
			&login_req->length,
			conn->param_list);
	if (ret < 0)
		return(-1);

	return(0);
}

/*	iscsi_initiator_do_login():
 *
 *
 */
static int iscsi_initiator_do_login(iscsi_conn_t *conn, iscsi_login_t *login)
{
	int pdu_count = 0, ret = 0;
	struct iscsi_init_login_cmnd *login_req;
	struct iscsi_targ_login_rsp *login_rsp;

	login_req = (struct iscsi_init_login_cmnd *) login->req;
	login_rsp = (struct iscsi_targ_login_rsp *) login->rsp;

	while (1) {
		if (++pdu_count > MAX_LOGIN_PDUS) {
			TRACE_ERROR("MAX_LOGIN_PDUS count reached.\n");
			return(-1);
		}

		if ((ret = iscsi_initiator_do_login_io(conn, login)) != 0)
			return(ret);

		switch ((login_rsp->flags & CSG) >> CSG_SHIFT) {
			case 0:
				if (iscsi_initiator_handle_csg_zero(conn, login) < 0)
					return(-1);
				break;	
			case 1:
				if (iscsi_initiator_handle_csg_one(conn, login) < 0)
					return(-1);
				if (login_rsp->flags & T_BIT)
					return(iscsi_initiator_check_last_response
							(conn, login));
				break;
			default:
				TRACE_ERROR("Illegal CSG: %d received from"
					" Target, protocol error.\n",
					(login_rsp->flags & CSG) >> CSG_SHIFT);
				return(-1);
		}
	}
		
	return(0);
}

/*	iscsi_initiator_parameter_negotiation():
 *
 *
 */
static int iscsi_initiator_parameter_negotiation(iscsi_conn_t *conn, iscsi_login_t *login)
{
	int ret = 0;
	iscsi_param_t *param;
	iscsi_session_t *sess = SESS(conn);
	iscsi_channel_t *c = sess->channel;
	struct iscsi_init_login_cmnd *login_req;
	struct iscsi_targ_login_rsp *login_rsp;
	
	login->first_response = 1;

	if (!c)
		return(-1);
		
	login_req = (struct iscsi_init_login_cmnd *) login->req;
	login_rsp = (struct iscsi_targ_login_rsp *) login->rsp;	

	login_req->opcode		= ISCSI_INIT_LOGIN_CMND | I_BIT;
	login->current_stage		= 0;
	login_req->version_max		= 0x00;
	login->version_max		= 0x00;
	login_req->version_min		= 0x00;
	login->version_min		= 0x00;
	memcpy(login_req->isid, SESS(conn)->isid, 6);
	memcpy(login->isid, SESS(conn)->isid, 6);
	login_req->tsih			= SESS(conn)->tsih;
	login->tsih			= SESS(conn)->tsih;
	login_req->init_task_tag 	= SESS(conn)->cur_task_tag++;
	login->init_task_tag		= login_req->init_task_tag;
	login_req->cid			= conn->cid;
	login_req->cmd_sn		= SESS(conn)->cur_cmdsn;
	login_req->exp_stat_sn	= 0;
	
	if (!(param = iscsi_find_param_from_key(AUTHMETHOD, conn->param_list)))
		return(-1);

	if (!strcmp(param->value, NONE)) {
		printk("iCHANNEL[%d] - No defined iSCSI Authentication Methods,"
			" skipping SecurityNegotiation phase.\n", c->channel_id);
		login->current_stage = 1;
		login_req->flags |= (CSG1|NSG3|T_BIT);
	}
	
	if (iscsi_encode_text_output(
			(login_req->flags & CSG1) ?
			PHASE_OPERATIONAL|PHASE_DECLARATIVE :
		        PHASE_SECURITY|PHASE_DECLARATIVE,
			SENDER_INITIATOR,
			login->req_buf,
			&login_req->length,
			conn->param_list) < 0)
		return(-1);

	if ((ret = iscsi_initiator_do_login(conn, login)) != 0)
		return(ret);

	if (login->leading_connection) {
		sess->tsih	= login_rsp->tsih;	
		sess->cur_cmdsn	= login_rsp->exp_cmd_sn;
	}
	iscsi_update_cmdsn(sess, login_rsp->exp_cmd_sn, login_rsp->max_cmd_sn, 1);
	
	return(0);
}

/*	iscsi_initiator_start_negotiation():
 *
 *
 */
extern int iscsi_initiator_start_negotiation(iscsi_conn_t *conn, iscsi_login_holder_t *lh)
{
	int ret = 0;
	iscsi_login_t *login;
	
	if (!(login = (iscsi_login_t *) kmalloc(sizeof(iscsi_login_t), GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate memory for iscsi_login_t.\n");
		ret = -1;
		goto out;
	}
	memset(login, 0, sizeof(iscsi_login_t));
		
	if (!(login->req = (char *) kmalloc(ISCSI_HDR_LEN, GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate memory for Login Request.\n");
		ret = -1;
		goto out;
	}
	memset(login->req, 0, ISCSI_HDR_LEN);
	
	if (!(login->rsp = (char *) kmalloc(ISCSI_HDR_LEN, GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate memory for Login Response.\n");
		ret = -1;
		goto out;
	}
	memset(login->rsp, 0, ISCSI_HDR_LEN);

	if (!(login->req_buf = (char *) kmalloc(MAX_TEXT_LEN, GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate memory for response buffer.\n");
		ret = -1;
		goto out;
	}
	memset(login->req_buf, 0, MAX_TEXT_LEN);

	if (!(login->rsp_buf = (char *) kmalloc(MAX_TEXT_LEN, GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate memory for request buffer.\n");
		ret = -1;
		goto out;
	}
	memset(login->rsp_buf, 0, MAX_TEXT_LEN);

	login->lh = lh;
	login->leading_connection = lh->leading_conn;
	
	if ((ret = iscsi_initiator_parameter_negotiation(conn, login)) < 0) {
		TRACE_ERROR("iSCSI Login negotiation failed.\n");
		goto out;
	} else if (ret) {
		printk("iSCSI Login Target Network Portal Redirection to %s:%hu,%hu\n",
			iscsi_ntoa(lh->ipv4_address), lh->port, conn->target_address->tpgt);;
		goto out;
	}
	
out:
	if (ret < 0)
		iscsi_remove_failed_auth_entry(conn, AUTH_CLIENT);
	if (login->req)
		kfree(login->req);
	if (login->rsp)
		kfree(login->rsp);
	if (login->req_buf)
		kfree(login->req_buf);
	if (login->rsp_buf)
		kfree(login->rsp_buf);
	if (login)
		kfree(login);
	
	return(ret);
	
}
