/*
 * Copyright (c) 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *  
 * This file houses the iSCSI Initiator specific SCSI Channel functions.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_LU_C

#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/interrupt.h>
#include <linux/blkdev.h>
#include <scsi.h>
#include <scsi/scsi.h>
#include <scsi/scsi_device.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_lists.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_erl1.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_lu.h>

#undef ISCSI_INITIATOR_LU_C

/*	iscsi_lu_check_access_count():
 *
 *
 */
static int iscsi_lu_check_access_count (int lun, iscsi_channel_t *c)
{
	int ret = 0;
	void *OS_SCSI_lu;
	
	if (!(OS_SCSI_lu = iscsi_OS_get_SAM_lu(lun, (void *)c)))
		return(-ENODEV);
	
	scsi_device_put((struct scsi_device *)OS_SCSI_lu);

	return(ret);	
}

/*	iscsi_lu_check_online():
 *
 *
 */
extern int iscsi_lu_check_online (int lun, iscsi_channel_t *c)
{
	int ret;
	void *OS_SCSI_lu;

	if (!(OS_SCSI_lu = iscsi_OS_get_SAM_lu(lun, (void *)c)))
		return(0);

	ret = iscsi_OS_check_lu_online(OS_SCSI_lu);
	scsi_device_put((struct scsi_device *)OS_SCSI_lu);
	
	return(ret);
}

/*	iscsi_lu_check_access_counts():
 *
 *
 */
extern int iscsi_lu_check_access_counts (iscsi_channel_t *c)
{
	int i, lun_count = 0;

	for (i = 0; i < ISCSI_MAX_LUNS; i++) {
		if (!iscsi_lu_check_online(i, c))
			continue;
		
		if (iscsi_lu_check_access_count(i, c))
			lun_count++;
	}

	return(lun_count);
}

/*	iscsi_lu_fail_scsi_task():
 *
 *
 */
extern void iscsi_lu_fail_scsi_task (iscsi_cmd_t *cmd, iscsi_lun_t *lun)
{
	iscsi_channel_t *ch = lun->channel;
	struct scsi_cmnd *sc;
	
	printk("iCHANNEL[%d]_LU[%d] - ***ERROR*** - Taking SCSI Device OFFLINE\n",
			ch->channel_id, lun->lun);

	if (cmd->state == ISTATE_SEND_RETRY_COMMAND)
		iscsi_dec_scsi_usage_count(cmd);

	sc = cmd->scsi_cmnd;
	iscsi_dec_scsi_usage_count(cmd);
	iscsi_check_scsi_usage_count(cmd, 1);

	iscsi_set_result_from_action_and_complete(sc, ISCSI_COMMAND_FAIL);

	return;
}

/*	iscsi_lu_fail_scsi_task_from_timer():
 *
 *
 */
extern void iscsi_lu_fail_scsi_task_from_timer (iscsi_cmd_t *cmd, iscsi_lun_t *lun)
{
	iscsi_channel_t *ch = lun->channel;
	iscsi_conn_t *conn = CONN(cmd);
	struct scsi_cmnd *sc = cmd->scsi_cmnd;

	printk("iCHANNEL[%d]_LU[%d] - ***ERROR*** - Taking SCSI Device OFFLINE\n",
			ch->channel_id, lun->lun);
	
	if (sc->sc_data_direction == DMA_FROM_DEVICE)
		iscsi_stop_datain_timer(cmd);

	if (cmd->seq_list)
		kfree(cmd->seq_list);
	if (cmd->pdu_list)
		kfree(cmd->pdu_list);
	if (cmd->buf_ptr)
		kfree(cmd->buf_ptr);

	spin_lock_bh(&conn->cmd_lock);
	__iscsi_remove_cmd_from_conn_list(cmd, conn);
	spin_unlock_bh(&conn->cmd_lock);

	iscsi_dec_scsi_usage_count(cmd);
	iscsi_check_scsi_usage_count(cmd, 1);
	
	iscsi_release_cmd_to_pool(cmd, SESS(conn));

	iscsi_set_result_from_action_and_complete(sc, ISCSI_COMMAND_FAIL);

	return;
}

/*	iscsi_channel_check_session_online():
 *
 *
 */
extern iscsi_session_t *iscsi_channel_check_session_online (iscsi_channel_t *c)
{
	iscsi_session_t *sess;
	
	spin_lock_bh(&c->channel_state_lock);
	if (c->status != ISCSI_CHANNEL_ACTIVE) {
		spin_unlock_bh(&c->channel_state_lock);
		return(NULL);
	}

	sess = c->sess;
	spin_unlock_bh(&c->channel_state_lock);

	return(sess);
}

/*	iscsi_lu_scan_full():
 *
 *
 */
extern void iscsi_lu_scan_full (iscsi_channel_t *c)
{
	if (!c->scsi_host) {
		TRACE_ERROR("scsi_host is NULL for iSCSI Channel: %d\n",
			c->channel_id);
		return;
	}

	scsi_scan_host(c->scsi_host);
	c->active_luns = 1;

	return;
}

/*	iscsi_release_all_lus():
 *
 *
 */
extern void iscsi_release_all_lus (iscsi_channel_t *c)
{
	if (!c->scsi_host) {
		TRACE_ERROR("scsi_host is NULL for iSCSI Channel: %d\n",
				c->channel_id);
		return;
	}	

	linux_scsi_forget_host(c->scsi_host);
	c->active_luns = 0;

	return;
}
