/*
 * Copyright (c) 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses the login functions used by the iSCSI Initiator driver.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_LOGIN_C

#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/in.h>
#include <net/sock.h>
#include <net/tcp.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_discovery.h>
#include <iscsi_initiator_nego.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_login.h>
#include <iscsi_initiator_sysfs.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_parameters.h>
#include <iscsi_lists.h>

#undef ISCSI_INITIATOR_LOGIN_C
 
extern iscsi_global_t *iscsi_global;

/*	iscsi_login_timeout():
 *
 *
 */
static void iscsi_login_timeout (unsigned long data)
{
	up((struct semaphore *) data);
	return;
}

/*	iscsi_post_login_failure_handler():
 *
 *
 */
static int iscsi_post_login_failure_handler (iscsi_login_holder_t *lh, int login_ret)
{
	return(login_ret);
}

/*	iscsi_post_login_start_timers():
 *
 *
 */
static void iscsi_post_login_start_timers (iscsi_conn_t *conn, iscsi_channel_t *channel)
{
	iscsi_session_t *sess = SESS(conn);
	
	spin_lock_bh(&conn->netif_lock);
	iscsi_start_netif_timer(conn);
	spin_unlock_bh(&conn->netif_lock);

	/*
	 * Startup the NopOut heartbeat pings now.
	 */
	if (!SESS_OPS(sess)->SessionType) 
		iscsi_start_nopout_timer(conn);

	return;
}

/*	iscsi_post_login_success_handler():
 *
 *
 */
static int iscsi_post_login_success_handler (iscsi_conn_t *conn, iscsi_login_holder_t *lh)
{
	unsigned char buf_ipv4[IPV4_BUF_SIZE];
	iscsi_channel_t *channel = lh->channel;
	iscsi_session_t *sess = SESS(conn);
	iscsi_thread_set_t *ts;
	
	iscsi_inc_conn_usage_count(conn);
	iscsi_inc_session_usage_count(sess);

	sess->channel = lh->channel;
	
	TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_LOGGED_IN.\n");
	conn->conn_state = INIT_CONN_STATE_LOGGED_IN;

	iscsi_set_connection_ops(conn->conn_ops, conn->param_list);
	iscsi_set_sync_and_steering_values(conn);

	ts = iscsi_get_thread_set();

	memset(buf_ipv4, 0, IPV4_BUF_SIZE);
	iscsi_ntoa2(buf_ipv4, conn->login_ip);
	
	if (!lh->leading_conn) {
		if (sess->session_state == INIT_SESS_STATE_FAILED) {
			TRACE(TRACE_STATE, "Moving to INIT_SESS_STATE_LOGGED_IN.\n");
			sess->session_state = INIT_SESS_STATE_LOGGED_IN;
		}

		iscsi_sysfs_register_conn_attributes(conn, sess);

		printk("iCHANNEL[%d] - iSCSI Login successful on CID: %hu to %s:%hu,%hu\n",
			channel->channel_id, conn->cid, buf_ipv4,
			conn->login_port, SESS_OPS(sess)->TargetPortalGroupTag);
		
		iscsi_add_conn_to_list(sess, conn);
		iscsi_post_login_start_timers(conn, sess->channel);
		iscsi_activate_thread_set(conn, ts);

		iscsi_dec_session_usage_count(sess);
		iscsi_dec_conn_usage_count(conn);
		return(0);
	}

	iscsi_set_session_ops(sess->sess_ops, conn->param_list);
	
	TRACE(TRACE_STATE, "Moving to INIT_SESS_STATE_LOGGED_IN.\n");
	sess->session_state = INIT_SESS_STATE_LOGGED_IN;
	sess->sid = iscsi_global->sid++;
	if (!sess->sid)
		sess->sid = iscsi_global->sid++;

	iscsi_sysfs_register_session_attributes(sess, channel);
	iscsi_sysfs_register_conn_attributes(conn, sess);

	printk("iCHANNEL[%d] - iSCSI login successful on CID: %hu to %s:%hu,%hu\n",
		channel->channel_id, conn->cid, buf_ipv4,
		conn->login_port, SESS_OPS(sess)->TargetPortalGroupTag);

	iscsi_add_conn_to_list(sess, conn);
	iscsi_add_session_to_list(sess);

	iscsi_post_login_start_timers(conn, sess->channel);
	iscsi_activate_thread_set(conn, ts);
	
	if (SESS_OPS(sess)->SessionType) {
		iscsi_dec_session_usage_count(sess);
		iscsi_dec_conn_usage_count(conn);
		return(0);
	}
	
	/*
	 * iscsi_dec_session_usage_count() and iscsi_dec_conn_usage_count()
	 * are called inside iscsi_activate_channel() before the LUN Scan.
	 */
	iscsi_activate_channel(conn, lh);

	return(0);
}

/*	iscsi_initiator_post_login_handler():
 *
 *
 */
static int iscsi_initiator_post_login_handler (iscsi_conn_t *conn, iscsi_login_holder_t *lh, int login_ret)
{
	if (login_ret != 0)
		return(iscsi_post_login_failure_handler(lh, login_ret));

	return(iscsi_post_login_success_handler(conn, lh));
}

/*	iscsi_initiator_login():
 *
 *	Called From:	iscsi_create_connection().
 *	Parameters:	iSCSI Connection pointer, iSCSI Login Holder Pointer.
 *	Returns:	0 on success, -1 on error.
 */
extern int iscsi_initiator_login (iscsi_conn_t *conn, iscsi_login_holder_t *lh)
{
	int login_attempts = 0, ret = -1;
	unsigned char buf_ipv4[IPV4_BUF_SIZE];
	iscsi_channel_t *chan = lh->channel;
	iscsi_channel_req_t *cr;
	struct semaphore login_sleep_sem;
	struct timer_list login_timer;
	iscsi_tpg_failover_t tpgf;

	memset((void *)&tpgf, 0, sizeof(iscsi_tpg_failover_t));

	lh->conn = conn;
	
	flush_signals(current);
	
	while (1) {
		/*
		 * We allocate the parameter list here so each time
		 * we have to retry the login we have a fresh list.
		 */
		if (iscsi_copy_param_list(&conn->param_list, chan->param_list,
				lh->leading_conn) < 0)
			goto out;

		if (lh->cc) {
			iscsi_set_connection_parameter_list(conn->param_list,
					&lh->cc->conn_ops);
			iscsi_set_session_parameter_list(conn->param_list,
					&lh->cc->sess_ops);
		}
		
		if (iscsi_check_for_login_targetaddress(lh) < 0)
			goto release_params_and_exit;

		if (iscsi_setup_array_for_tpg_failover(lh, &tpgf) < 0)
			goto release_params_and_exit;
		
		if (!conn->target_address)
			lh->tpg_failover = 0;

		if (!(cr = iscsi_allocate_ch_req(CH_REQ_ACTION_LOGIN,
				(void *)lh, 0, NULL)))
			goto release_params_and_exit;

		ISCSI_ADD_CH_req(cr, chan);
		down_interruptible(&cr->cr_data_sem);
		iscsi_complete_ch_req(cr, chan);
		
		if (signal_pending(current) || iscsi_global->in_shutdown)
			goto release_params_and_exit;
		
		/*
		 * Login was successful.
		 */
		if (lh->login_ret == 0) {
			ret = 0;
			goto out;
		}

		if (lh->login_ret == 1) {
			goto release_params_and_retry;
		}

		/*
		 * Recieved login response with a Initiator Status Class ERROR, 
		 * cease futher login attempts for this connection.
		 */
		if (lh->status_class == STAT_CLASS_INITIATOR) {
			printk("iCHANNEL[%d] - **ERROR** - Not attempting to continue"
				" iSCSI Login\n", chan->channel_id);
			goto release_params_and_exit;
		}

		memset(buf_ipv4, 0, IPV4_BUF_SIZE);
		iscsi_ntoa2(buf_ipv4, conn->login_ip);

		if (lh->tpg_failover) {
			if (++login_attempts <
			    ISCSI_CA(chan)->tpgfailover_login_retries)
				goto release_params_and_retry;
			
			iscsi_update_tpg_failover_array(conn, &tpgf);

			if (!ISCSI_CA(chan)->tpgfailover_attempts ||
			    (++tpgf.failover_attempts <
			     ISCSI_CA(chan)->tpgfailover_attempts)) {
				printk("iCHANNEL[%d] - Doing failover for Target Portal"
					" Group: %hu to %s:%hu\n", chan->channel_id,
					tpgf.tpgt, buf_ipv4, conn->login_port);
				login_attempts = 0;
				goto release_params_and_retry;
			}
			
			printk("iCHANNEL[%d] - **ERROR** - Reached maximum failover"
				" attempts %u for Target Portal Group: %hu\n",
				chan->channel_id, ISCSI_CA(chan)->tpgfailover_attempts,
					tpgf.tpgt);	
				
			goto release_params_and_exit;
		} else {
			if (!ISCSI_CA(chan)->login_retries)
				goto release_params_and_retry;

			if (++login_attempts < ISCSI_CA(chan)->login_retries)
				goto release_params_and_retry;

			printk("iCHANNEL[%d] - **ERROR** - Reached maximum login"
				" attempts %d, aborting iSCSI login to %s:%hu\n",
				chan->channel_id, login_attempts, buf_ipv4,
				conn->login_port); 

			goto release_params_and_exit;
		}	
		
release_params_and_retry:
		/*
		 * Login was not successful, release the old parameters, reset
		 * the initial markerless intervals, kill the login thread if
		 * still present and sleep until we can try again.
		 */
		iscsi_release_param_list(conn->param_list);

		conn->if_marker = conn->of_marker = 0;
		
		init_MUTEX_LOCKED(&login_sleep_sem);
		init_timer(&login_timer);
		login_timer.expires = (jiffies + ISCSI_CA(chan)->login_retry_wait * HZ);
		login_timer.data = (unsigned long)&login_sleep_sem;
		login_timer.function = iscsi_login_timeout;
		
		add_timer(&login_timer);
		down_interruptible(&login_sleep_sem);
		del_timer_sync(&login_timer);

		if (signal_pending(current) || iscsi_global->in_shutdown)
			goto out;
	}

release_params_and_exit:
	iscsi_release_param_list(conn->param_list);
out:
	return(iscsi_initiator_post_login_handler(conn, lh, ret));
}

/*	iscsi_start_login():
 *
 *	Called from iscsi_channel_thread()
 */
extern void iscsi_start_login (iscsi_login_holder_t *lh, iscsi_channel_req_t *cr)
{
	int ip_proto = 0, sock_type = 0, on = 1;
	unsigned char buf_ipv4[IPV4_BUF_SIZE];
	iscsi_channel_t *c = lh->channel;
	iscsi_conn_t *conn = lh->conn;
	mm_segment_t oldfs;
	struct socket *sock;
	struct timer_list login_timer;

	init_timer(&login_timer);
	login_timer.expires = (jiffies + ISCSI_CA(c)->login_timeout * HZ);
	login_timer.data = (unsigned long)&cr->cr_data_sem;
	login_timer.function = iscsi_login_timeout;
	add_timer(&login_timer);
		
	oldfs = get_fs();
	set_fs(get_ds());
	
	switch (lh->network_transport) {
	case ISCSI_TCP:
		ip_proto = IPPROTO_TCP;
		sock_type = SOCK_STREAM;
		break;
	case ISCSI_SCTP_TCP:
		ip_proto = IPPROTO_SCTP;
		sock_type = SOCK_STREAM;
		break;
	case ISCSI_SCTP_UDP:
		ip_proto = IPPROTO_SCTP;
		sock_type = SOCK_SEQPACKET;
		break;
	case ISCSI_IWARP_TCP:
	case ISCSI_IWARP_SCTP:
	case ISCSI_INFINIBAND:
	default:
		TRACE_ERROR("Unsupported network_transport: %d\n", lh->network_transport);
		goto failed;
	}

	conn->network_transport = lh->network_transport;
	
	if (sock_create(AF_INET, sock_type, ip_proto, &conn->sock)) {
		set_fs(oldfs);
		lh->login_ret = -1;
		TRACE_ERROR("Could not create socket\n");
		goto failed;
	}
	sock = conn->sock;

	/*
	 * The SCTP stack needs struct socket->file.
	 */
	if ((lh->network_transport == ISCSI_SCTP_TCP) ||
	    (lh->network_transport == ISCSI_SCTP_UDP)) {
		if (!conn->sock->file) {
			if (!(conn->sock->file = kmalloc(
					sizeof(struct file), GFP_KERNEL))) {
				TRACE_ERROR("Unable to allocate struct file for SCTP\n");
				lh->login_ret = -1;
				set_fs(oldfs);
				goto failed;
			}
			memset(conn->sock->file, 0, sizeof(struct file));

			conn->conn_flags |= CONNFLAG_SCTP_STRUCT_FILE;
		}
	}
	
	/* FIXME: IPv6 goes here */
	conn->s_addr.sin_family = AF_INET;
	conn->s_addr.sin_port           = htons(conn->login_port);
	conn->s_addr.sin_addr.s_addr    = htonl(conn->login_ip);

	/*
	 * Set SO_REUSEADDR, and disable Nagel Algorithm with TCP_NODELAY.
	 */
	sock->ops->setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char *)&on, sizeof(on));
	sock->ops->setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on));

	TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_XPT_WAIT\n");
	conn->conn_state = INIT_CONN_STATE_XPT_WAIT;

        memset(buf_ipv4, 0, IPV4_BUF_SIZE);
	iscsi_ntoa2(buf_ipv4, conn->login_ip); 
	
	if (sock->ops->connect(conn->sock, (struct sockaddr *)&conn->s_addr,
				sizeof(struct sockaddr), 0) < 0) {
		set_fs(oldfs);

		TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_FREE\n");
		conn->conn_state = INIT_CONN_STATE_FREE;

		printk("iCHANNEL[%d] - **ERROR** - Could not establish %s/IP"
			" connection to %s:%d\n", c->channel_id, (ip_proto == IPPROTO_TCP) ?
				"TCP" : "SCTP", buf_ipv4, conn->login_port);

		lh->login_ret = -1;
		goto failed;
	} else {
		TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_IN_LOGIN\n");
		conn->conn_state = INIT_CONN_STATE_IN_LOGIN;
		
		TRACE(TRACE_ISCSI, "Established %s/IP connection to %s:%d\n",
			(ip_proto == IPPROTO_TCP) ? "TCP" : "SCTP",
				buf_ipv4, conn->login_port);
	}

	set_fs(oldfs);
	
	/*
	 * Use the early PHY failure detection when available.
	 */
	iscsi_get_network_interface_from_socket((void *)conn);

	/*
	 * TCP/IP connection established, start the iSCSI login
	 * negotiation process.
	 */
	if (lh->leading_conn)
		memcpy(SESS(conn)->isid, c->ch_isid, 6);

	if ((lh->login_ret = iscsi_initiator_start_negotiation(conn, lh)) < 0)
		goto failed;

	if (signal_pending(current))
		goto failed;
	
	del_timer_sync(&login_timer);
	return;

failed:	
	del_timer_sync(&login_timer);
	
	if (conn->sock) {
		if (conn->conn_flags & CONNFLAG_SCTP_STRUCT_FILE) {
			kfree(conn->sock->file);
			conn->sock->file = NULL;
		}
		sock_release(conn->sock);	
	}

	return;
}

