/*
 * Copyright (c) 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *  
 * This file houses the iSCSI Initiator Information utility functions.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_INFO_C

#include <linux/version.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/in.h>
#include <net/sock.h>
#include <net/tcp.h>
#include <linux/blkdev.h>
#include <linux/kref.h>
#include <scsi/scsi.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_device.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_info.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_lu.h>
#include <iscsi_initiator_scsi.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_parameters.h>

#undef ISCSI_INITIATOR_INFO_C

extern iscsi_global_t *iscsi_global;

static int check_and_copy_buf (
	unsigned char *b,
	unsigned char *tb,
	int *cl,
	int *bl,
	int *ml)
{
	if ((*cl + *bl)> *ml)
		return(-1);
	else {          
		memcpy(b+*cl, tb, *bl);
		*cl += *bl;
	}

	return(0);
}

#define TMP_BUF_LEN	512

/*	iscsi_chan_get_iscsi_info():
 *
 *
 */
extern int iscsi_chan_get_iscsi_info (
	int channel_id,
	char *pb,	/* Pointer to info buffer */
	int ml)		/* Max Length in bytes of buffer space */
{
	unsigned char b[TMP_BUF_LEN];
	int bl = 0, cl = 0, oob = 1;
	iscsi_channel_t *c = &iscsi_global->channels[channel_id];
        iscsi_conn_t *conn;
        iscsi_session_t	*sess = NULL;

	memset(b, 0, TMP_BUF_LEN);
	bl = sprintf(b, "--------------------[iSCSI Session Info for iSCSI Channel %d]"
			"--------------------\n", channel_id);
	if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
		goto done;

	if (!(sess = __iscsi_get_session_from_channel_id(c))) {
		memset(b, 0, TMP_BUF_LEN);
		bl = sprintf(b, "No active iSCSI Session exists on iSCSI"
			" Channel: %d\n", c->channel_id);
		if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
			goto done;
		oob = 0;
		goto done;
	}
	
	if (!SESS_OPS(sess)->SessionType) {
		memset(b, 0, TMP_BUF_LEN);
		bl = sprintf(b, "TargetName: %s\n",
			SESS_OPS(sess)->TargetName);
		if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
			goto done;
		memset(b, 0, TMP_BUF_LEN);
		bl = sprintf(b, "TargetAlias: %s\n",
			SESS_OPS(sess)->TargetAlias);
		if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
			goto done;
	}

	memset(b, 0, TMP_BUF_LEN);
	bl = sprintf(b, "PyX Session ID: %u  "
			 "ISID: 0x%02x %02x %02x %02x %02x %02x  "
			 "TSIH: %hu\n", sess->sid,
			sess->isid[0], sess->isid[1], sess->isid[2],
			sess->isid[3], sess->isid[4], sess->isid[5],
			sess->tsih);
	if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
		goto done;

	memset(b, 0, TMP_BUF_LEN);
	bl = sprintf(b, "Cmds in Session Pool: %d",
		atomic_read(&sess->pool_count));
	bl += sprintf(b+bl, "  Session State: ");
	switch (sess->session_state) {
	case INIT_SESS_STATE_FREE:
		bl += sprintf(b+bl, "INIT_SESS_FREE\n");
		break;
	case INIT_SESS_STATE_LOGGED_IN:
		bl += sprintf(b+bl, "INIT_SESS_LOGGED_IN\n");
	       	break;
	case INIT_SESS_STATE_FAILED:
		bl += sprintf(b+bl, "INIT_SESS_FAILED\n");
		break;
	default:
		bl += sprintf(b+bl, "ERROR: Unknown Session State!\n");
		break;
	}
	if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
		goto done;

	memset(b, 0, TMP_BUF_LEN);
	bl = sprintf(b, "---------------------[iSCSI Session Values]-----------------------\n");
	bl += sprintf(b+bl, "   CmdSN    :  ExpCmdSN  :  MaxCmdSN  :     ITT    :     TTT\n");
	bl += sprintf(b+bl, " 0x%08x   0x%08x   0x%08x   0x%08x   0x%08x\n",
			sess->cur_cmdsn, sess->exp_cmdsn, sess->max_cmdsn,
			sess->cur_task_tag, sess->targ_xfer_tag);
	bl += sprintf(b+bl, "----------------------[iSCSI Connections]-------------------------\n");
	if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
		goto done;
	
	spin_lock_bh(&sess->conn_lock);
	for (conn = sess->conn_head; conn; conn = conn->next) {
		memset(b, 0, TMP_BUF_LEN);
		bl = sprintf(b, "CID: %hu  Connection State: ", conn->cid);
		switch (conn->conn_state) {
		case INIT_CONN_STATE_FREE:
			bl += sprintf(b+bl, "INIT_CONN_FREE\n");
			break;
		case INIT_CONN_STATE_XPT_WAIT:
			bl += sprintf(b+bl, "INIT_CONN_XPT_WAIT\n");
			break;
		case INIT_CONN_STATE_IN_LOGIN:
			bl += sprintf(b+bl, "INIT_CONN_IN_LOGIN\n");
			break;
		case INIT_CONN_STATE_LOGGED_IN:
			bl += sprintf(b+bl, "INIT_CONN_LOGGED_IN\n");
			break;
		case INIT_CONN_STATE_IN_LOGOUT:
			bl += sprintf(b+bl, "INIT_CONN_IN_LOGOUT\n");
			break;
		case INIT_CONN_STATE_LOGOUT_REQUESTED:
			bl += sprintf(b+bl, "INIT_CONN_LOGOUT_REQUESTED\n");
			break;
		case INIT_CONN_STATE_CLEANUP_WAIT:
			bl += sprintf(b+bl, "INIT_CONN_CLEANUP_WAIT\n");
			break;
		default:
			bl += sprintf(b+bl, "ERROR: Unknown Connection State!\n"); 
			break;
		}
		bl += sprintf(b+bl, "	Address %s:%hu,%hu %s",
			iscsi_ntoa(conn->login_ip), conn->login_port,
			SESS_OPS(sess)->TargetPortalGroupTag,
			(conn->network_transport == ISCSI_TCP) ? "TCP" : "SCTP");

		bl += sprintf(b+bl, "  ExpStatSN: 0x%08x\n", conn->exp_statsn);
		if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0) {
			spin_unlock_bh(&sess->conn_lock);
			goto done;
		}
	}
	spin_unlock_bh(&sess->conn_lock);

	oob = 0;
done:
	if (sess)
		iscsi_dec_session_usage_count(sess);
	if (oob) {
		TRACE_ERROR("Ran out of buffer..\n");
	}
	return(cl);
}

/*	iscsi_chan_get_scsi_info():
 *
 *
 */
extern int iscsi_chan_get_scsi_info (
	int channel_id,
	char *pb,       /* Pointer to info buffer */
	int ml)         /* Max Length in bytes of buffer space */
{
	unsigned char b[TMP_BUF_LEN];
	unsigned long flags;
	int bl = 0, cl = 0, oob = 1;
	iscsi_channel_t *c = &iscsi_global->channels[channel_id];
	iscsi_channel_lun_t *lu;
	struct scsi_device *sd, *tmp;
	struct Scsi_Host *sh;

	memset(b, 0, TMP_BUF_LEN);
	bl = sprintf(b, "------------------------[SCSI Info for iSCSI Channel %d]"
			"-------------------------\n", channel_id);
	if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
		goto done;
	
	sh = c->scsi_host;
	memset(b, 0, TMP_BUF_LEN);
	bl = sprintf(b, "SCSI Host No: %d  SCSI-II Host TCQ Count: %d\n",
			sh->host_no, sh->can_queue);
	if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
		goto done;

	memset(b, 0, TMP_BUF_LEN);
	bl = sprintf(b, "Logical Unit TCQ Depth: %d  SGTableSize: %d  MaxSectors: %d\n",
			sh->cmd_per_lun, sh->sg_tablesize, sh->max_sectors);
	if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0)
		goto done;
	
	spin_lock_irqsave(sh->host_lock, flags);
	list_for_each_entry_safe(sd, tmp, &sh->__devices, siblings) {
		if (sd->lun > ISCSI_MAX_LUNS)
			continue;

		lu = &c->ch_lun_list[sd->lun];
		
		memset(b, 0, TMP_BUF_LEN);
		bl = sprintf(b, "iSCSI Logical Unit Number: %d  ", sd->lun);
		bl += sprintf(b+bl, "Status: ONLINE -> %s\n",
			(sd->writeable) ? "READ/WRITE" : "READ");
		if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0) {
			spin_unlock_irqrestore(sh->host_lock, flags);
			goto done;
		}

		memset(b, 0, TMP_BUF_LEN);
		switch (sd->type) {
		case TYPE_DISK:
			if (!lu->OS_device_set) {	
				bl = sprintf(b, "	DISK: sd?");
				break;
			}
			bl = sprintf(b, "	DISK: %s",
					lu->OS_device_name);
			break;
		case TYPE_ROM:
			if (!lu->OS_device_set) {
				bl = sprintf(b, "	CD/DVD: scd?");
				break;
			}
			bl = sprintf(b, "	CD/DVD: %s", lu->OS_device_name);
			break;
		case TYPE_TAPE:
			if (!lu->OS_device_set) {
				bl = sprintf(b, "	TAPE: st?");
				break;
			}
			bl = sprintf(b, "	TAPE: %s", lu->OS_device_name);
			break;
		default:
			bl = sprintf(b, "	UNKNOWN: 0x%02x", sd->type);
			break;
		}

		if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0) {
			spin_unlock_irqrestore(sh->host_lock, flags);
			goto done;
		}
		
		memset(b, 0, TMP_BUF_LEN);
		bl = sprintf(b, "	SCSI BUS Location: %d/%d/%d  Sector Size: %u\n",
				c->channel_id, sd->id, sd->lun, sd->sector_size);
		if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0) {
			spin_unlock_irqrestore(sh->host_lock, flags);
			goto done;
		}
		
		memset(b, 0, TMP_BUF_LEN);
		bl = sprintf(b, "	Active Tasks: %u  Total Tasks: %u	Total Bytes: %lluk\n",
			lu->iscsi_current_tasks, lu->iscsi_tasks,
				(unsigned long long)lu->total_bytes);
		if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0) {
			spin_unlock_irqrestore(sh->host_lock, flags);
			goto done;
		}
#if 0
		/*
		 * Got the Major and Minor.
		 */
		disk = (struct gendisk *) lu->OS_device_data;
		memset(b, 0, TMP_BUF_LEN);
		bl = sprintf(b, "	Major: %d  Minor: %d	Sector Size: %u		Capacity: %luk\n",
				disk->major, disk->first_minor, sd->sector_size, 
				disk->capacity * sd->sector_size);
		bl += sprintf(b+bl, "	Active iSCSI tasks: %u	Total iSCSI tasks: %u		Total Bytes: %lluk\n",
				lu->iscsi_current_tasks, lu->iscsi_tasks, lu->total_bytes);
		if (check_and_copy_buf(pb, b, &cl, &bl, &ml) < 0) {
			scsi_device_put(sd);
			goto done;
		}
#endif
	}
	spin_unlock_irqrestore(sh->host_lock, flags);
	
	oob = 0;
done:
	if (oob) {
		TRACE_ERROR("Ran out of buffer..\n");
	}
	return(cl);
}
