/*
 * Copyright (c) 2002, 2003, 2004 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses error recovery level one functions used by
 * the iSCSI Initiator driver.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_ERL1_C

#include <linux/delay.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/in.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_lists.h>
#include <iscsi_initiator_debug_opcodes.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_erl0.h>
#include <iscsi_initiator_erl1.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_tmr.h>
#include <iscsi_initiator_util.h>

#undef ISCSI_INITIATOR_ERL1_C

extern iscsi_global_t *iscsi_global;

#define OFFLOAD_BUF_SIZE	32768

/*	iscsi_dump_data_payload():
 *
 *	Used to dump excess datain payload for certain error recovery situations.
 *	Receive in OFFLOAD_BUF_SIZE max of datain per rx_data().
 *
 *	dump_padding_digest denotes if padding and data digests need to be dumped.
 */
extern int iscsi_dump_data_payload (
	iscsi_conn_t *conn,
	u32 buf_len,
	int dump_padding_digest)
{
	char *buf, pad_bytes[4];
	int ret = DATAIN_WITHIN_COMMAND_RECOVERY, rx_got;
	u32 length, padding, offset = 0, size;
	struct iovec iov;
	
	length = (buf_len > OFFLOAD_BUF_SIZE) ? OFFLOAD_BUF_SIZE : buf_len;

	if (!(buf = (char *) kmalloc(length, GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate %u bytes for offload"
				" buffer.\n", length);
		return(-1);
	}
	memset(buf, 0, length);
	memset(&iov, 0, sizeof(struct iovec));
	
	while (offset < buf_len) {
		size = ((offset + length) > buf_len) ?
			(buf_len - offset) : length;

		iov.iov_len = size;
		iov.iov_base = buf;

		rx_got = rx_data(conn, &iov, 1, size);
		if (rx_got != size) {
			ret = DATAIN_CANNOT_RECOVER;
			goto out;
		}

		offset += size;
	}

	if (!dump_padding_digest)
		goto out;
	
	if ((padding = ((-buf_len) & 3)) != 0) {
		iov.iov_len = padding;
		iov.iov_base = pad_bytes;
	
		rx_got = rx_data(conn, &iov, 1, padding);
		if (rx_got != padding) {
			ret = DATAIN_CANNOT_RECOVER;
			goto out;
		}
	}
	
	if (CONN_OPS(conn)->DataDigest) {
		u32 data_crc;

		iov.iov_len = CRC_LEN;
		iov.iov_base = &data_crc;

		rx_got = rx_data(conn, &iov, 1, CRC_LEN);
		if (rx_got != CRC_LEN) {
			ret = DATAIN_CANNOT_RECOVER;
			goto out;
		}
	}

out:
	kfree(buf);
	return(ret);
}

/*	iscsi_datain_datasequenceinorder_no_sbit():
 *
 *
 */
extern int iscsi_datain_datasequenceinorder_no_sbit (
	iscsi_cmd_t *cmd)
{
	int i, missing = 0;
	iscsi_seq_t *seq;
	
	/*
	 * This is a paranoid check that is called when a DataIN PDU carrying
	 * a S_BIT arrives while DataSequenceInOrder=No.
	 * It checks each orignal and current offset against total sequence
	 * length to ensure all datain has arrived.
	 */
	for (i = 0; i < cmd->seq_count; i++) {
		seq = &cmd->seq_list[i];
		if (seq->offset < (seq->orig_offset + seq->xfer_len)) {
			TRACE_ERROR("Initiator did not receive ITT: 0x%08x complete"
				" DataIN PDU(s) within sequence Original Offset: %u,"
				" Current Offset: %u, Total Length: %u. Bad Target"
				" DataSequenceInOrder=No implemenation.\n",
					cmd->init_task_tag, seq->orig_offset,
					seq->offset, seq->xfer_len);
			missing++;
		}
	}
		
	return((missing) ? DATAIN_CANNOT_RECOVER : DATAIN_NORMAL);
}

/*	iscsi_datain_datapduinorder_no_fbit():
 *
 *
 */
extern int iscsi_datain_datapduinorder_no_fbit (
	iscsi_cmd_t *cmd,
	iscsi_pdu_t *pdu)
{
	int i, missing = 0;
	u32 pdu_count = 0;
	iscsi_conn_t *conn = CONN(cmd);
	iscsi_pdu_t *first_pdu = NULL;
	
	/*
	 * Get a iscsi_pdu_t pointer to the first PDU, and total PDU count
	 * of the DataIN sequence.
	 */
	if (SESS_OPS_C(conn)->DataSequenceInOrder) {
		for (i = 0; i < cmd->pdu_count; i++) {
			if (cmd->pdu_list[i].seq_no == pdu->seq_no) {
				if (!first_pdu)
					first_pdu = &cmd->pdu_list[i];
				pdu_count++;
			} else if (pdu_count)
				break;
		}
	} else {
		iscsi_seq_t *seq = cmd->seq_ptr;
		first_pdu = &cmd->pdu_list[seq->pdu_start];
		pdu_count = seq->pdu_count;
	}

	if (!first_pdu || !pdu_count)
		return(DATAIN_CANNOT_RECOVER);

	/*
	 * Due the the fact we can only request retransmisson of DataIN PDUs
	 * based on DataSN with a Data SNACK, its basically impossible to recover
	 * from missing offset+length PDU tuples in based purely on DataSNs in
	 * DataPDUInOrder=No.
	 * If the DataSN checks passed for this sequence, but we still detect
	 * missing offset+length tuples the only explanation is a BAD BAD BAD
	 * target implemenation, they should have bought PyX Technologies. :-)
	 */
	for (i = 0; i < pdu_count; i++) {
		if (first_pdu[i].status != ISCSI_PDU_RECEIVED_OK) {
			TRACE_ERROR("Initiator did not receive ITT: 0x%08x DataIN PDU"
				" Offset: %u, Length: %u while DataPDUInOrder=No. Bad"
				" Target DataPDUInOrder=No implementation.\n",
					cmd->init_task_tag, first_pdu[i].offset,
						first_pdu[i].length);
			missing++;
		}
	}
	
	return((missing) ? DATAIN_CANNOT_RECOVER : DATAIN_NORMAL);
}

/*	iscsi_handle_cmdsn_timeout():
 *
 *
 */
static void iscsi_handle_cmdsn_timeout (unsigned long data)
{
	int found_cmd = 0;
	iscsi_channel_t *channel = NULL;
	iscsi_cmd_t *cmd = NULL;
	iscsi_conn_t *conn = NULL;
	iscsi_session_t *sess = (iscsi_session_t *) data;

	iscsi_inc_session_usage_count(sess);
		
	spin_lock_bh(&sess->cmdsn_lock);
	if (sess->cmdsn_timer_flags & CMDSN_TF_STOP) {
		spin_unlock_bh(&sess->cmdsn_lock);
		iscsi_dec_session_usage_count(sess);
		return;
	}
	sess->cmdsn_timer_flags &= ~CMDSN_TF_RUNNING;
	
	channel = sess->channel;

	if (sess->cur_cmdsn <= sess->exp_cmdsn) {
		TRACE(TRACE_TIMER, "Assuming IDLE timeout for ExpCmdSN:"
			" 0x%08x\n", sess->exp_cmdsn);
		goto out;
	}

	if (!SESS_OPS(sess)->ErrorRecoveryLevel) {
		printk("iCHANNEL[%d] - Unable to recover from CmdSN timeout"
			" while in ErorrRecoveryLevel=0\n", channel->channel_id);
		goto failure;
	}

	printk("iCHANNEL[%d] - CmdSN Timer timed out waiting for ExpCmdSN"
		" response: 0x%08x\n", channel->channel_id, sess->exp_cmdsn);
	/*
	 * Look for the non-immediate CmdSN in question.
	 */
	spin_lock(&sess->conn_lock);
	for (conn = sess->conn_head; conn; conn = conn->next) {
		spin_lock(&conn->cmd_lock);
		for (cmd = conn->cmd_head; cmd; cmd = cmd->next) {
			if (cmd->pdu[0] & I_BIT)
				continue;
			if (cmd->cmdsn == sess->exp_cmdsn) {
				__iscsi_inc_conn_usage_count(CONN(cmd));
				found_cmd = 1;
				break;
			}
		}
		spin_unlock(&conn->cmd_lock);
		if (found_cmd)
			break;
	}
	spin_unlock(&sess->conn_lock);

	/*
	 * If the iscsi_cmd_t cannot be located use values in
	 * iscsi_session_t to keep count of how many timeouts
	 * encountered.
	 */
	if (!cmd) {
		if (sess->timed_out_cmdsn != sess->exp_cmdsn) {
			sess->timed_out_cmdsn_count = 0;
			sess->timed_out_cmdsn = sess->exp_cmdsn;
		}

		if (++sess->timed_out_cmdsn_count ==
		    ISCSI_CA(channel)->cmdsn_timeout_retries) {
			printk("iCHANNEL[%d] - CmdSN: 0x%08x exceeded max retries"
			" %u, closing iSCSI session\n", channel->channel_id,
			sess->timed_out_cmdsn, sess->timed_out_cmdsn_count);
			goto failure;
		}

		iscsi_start_cmdsn_timer(sess, sess->exp_cmdsn);
		goto out;
	}

	/*
	 * If the iscsi_cmd_t was located, retry the command up to
	 * ISCSI_CA(channel)->cmdsn_timeout_retries times.
	 */
	if (++cmd->cmdsn_retries < ISCSI_CA(channel)->cmdsn_timeout_retries) {
		TRACE(TRACE_ERL1, "Retrying timed out CmdSN: 0x%08x, attempt"
			" %u\n", cmd->cmdsn, cmd->cmdsn_retries);

		iscsi_start_cmdsn_timer(sess, sess->exp_cmdsn);

		spin_lock(&cmd->state_lock);
		if (cmd->cmd_flags & ICF_CMD_NONIMMEDIATE_ACTIVE) {
			spin_unlock(&cmd->state_lock);
			__iscsi_dec_conn_usage_count(CONN(cmd));
			goto out;
		}
		spin_unlock(&cmd->state_lock);

		iscsi_add_cmd_to_immediate_queue(cmd, conn, ISTATE_SEND_RETRY_COMMAND);
		__iscsi_dec_conn_usage_count(CONN(cmd));
		
		goto out;
	} else {
		printk("iCHANNEL[%d] - CmdSN: 0x%08x exceeded max retires %u, closing"
			" iSCSI session\n", channel->channel_id, cmd->cmdsn,
				cmd->cmdsn_retries);
		__iscsi_dec_conn_usage_count(CONN(cmd));
	}	

failure:
	spin_unlock_bh(&sess->cmdsn_lock);
	__iscsi_cause_session_reinstatement(sess);
	iscsi_dec_session_usage_count(sess);
	return;
out:
	spin_unlock_bh(&sess->cmdsn_lock);
	iscsi_dec_session_usage_count(sess);
	return;
}

/*	iscsi_handle_datain_timeout():
 *
 *
 */
static void iscsi_handle_datain_timeout (unsigned long data)
{
	iscsi_channel_t *channel = NULL;
	iscsi_cmd_t *cmd = (iscsi_cmd_t *) data;
	iscsi_conn_t *conn = CONN(cmd);
	iscsi_session_t *sess = NULL;
	
	iscsi_inc_conn_usage_count(conn);
		
	spin_lock_bh(&cmd->datain_timeout_lock);
	if (cmd->datain_timer_flags & DATAIN_TF_STOP) {
		spin_unlock_bh(&cmd->datain_timeout_lock);
		iscsi_dec_conn_usage_count(conn);
		return;
	}
	cmd->datain_timer_flags &= ~DATAIN_TF_RUNNING;
	
	sess = SESS(conn);
	channel = sess->channel;

	printk("iCHANNEL[%d] - Unable to recover from DataIN timeout"
		" while in ErrorRecoveryLevel=0\n", channel->channel_id);

	spin_unlock_bh(&cmd->datain_timeout_lock);
	iscsi_cause_connection_reinstatement(conn, 0);
	iscsi_dec_conn_usage_count(conn);
	return;
}

/*	iscsi_mod_cmdsn_timer():
 *
 *	Must hold sess->cmdsn_lock while calling.
 */
extern void iscsi_mod_cmdsn_timer (
	 iscsi_session_t *sess,
	 u32 *exp_cmdsn)
{
	iscsi_channel_t *c = sess->channel;
	
	if (!(sess->cmdsn_timer_flags & CMDSN_TF_RUNNING))
		return;

	mod_timer(&sess->cmdsn_timer, jiffies + ISCSI_CA(c)->cmdsn_timeout * HZ);
	TRACE(TRACE_TIMER, "Updated CmdSN timer for ExpCmdSN: 0x%08x on SID:"
			" %u\n", *exp_cmdsn, sess->sid);

	return;
}

/*	iscsi_start_cmdsn_timer():
 *
 *	Must hold sess->cmdsn_lock while calling.
 */
extern void iscsi_start_cmdsn_timer (
	iscsi_session_t *sess,
	u32 cmdsn)
{
	iscsi_channel_t *c = sess->channel;
	
	if (sess->cmdsn_timer_flags & CMDSN_TF_RUNNING)
		return;

	init_timer(&sess->cmdsn_timer);
	sess->cmdsn_timer.expires = (jiffies + ISCSI_CA(c)->cmdsn_timeout * HZ);
	sess->cmdsn_timer.data = (unsigned long) sess;
	sess->cmdsn_timer.function = iscsi_handle_cmdsn_timeout;

	sess->cmdsn_timer_flags |= CMDSN_TF_RUNNING;
	add_timer(&sess->cmdsn_timer);

	TRACE(TRACE_TIMER, "Starting CmdSN timer for CmdSN: 0x%08x on SID:"
			" %u\n", cmdsn, sess->sid);
	return;
}

/*	iscsi_stop_cmdsn_timer():
 *
 *
 */
extern void iscsi_stop_cmdsn_timer (iscsi_session_t *sess)
{
	spin_lock_bh(&sess->cmdsn_lock);
	if (!(sess->cmdsn_timer_flags & CMDSN_TF_RUNNING)) {
		spin_unlock_bh(&sess->cmdsn_lock);
		return;
	}
	sess->cmdsn_timer_flags |= CMDSN_TF_STOP;
	spin_unlock_bh(&sess->cmdsn_lock);
	
	del_timer_sync(&sess->cmdsn_timer);

	spin_lock_bh(&sess->cmdsn_lock);
	sess->cmdsn_timer_flags &= ~CMDSN_TF_RUNNING;
	sess->cmdsn_timer_flags &= ~CMDSN_TF_STOP;
	spin_unlock_bh(&sess->cmdsn_lock);

	TRACE(TRACE_TIMER, "Stopped CmdSN Timer for SID: %u\n", sess->sid);
	return;
}

/*	iscsi_mod_datain_timer():
 *
 *
 */
extern void iscsi_mod_datain_timer (
	iscsi_cmd_t *cmd,
	u32 data_sn)
{
	iscsi_channel_t *channel;
	iscsi_conn_t *conn = cmd->conn;
	iscsi_session_t *sess = SESS(conn);
	
	spin_lock_bh(&cmd->datain_timeout_lock);
	if (!(cmd->datain_timer_flags & DATAIN_TF_RUNNING)) {
		spin_unlock_bh(&cmd->datain_timeout_lock);
		return;
	}
	channel = sess->channel;
	
	mod_timer(&cmd->datain_timer, jiffies + ISCSI_CA(channel)->datain_timeout * HZ);
	TRACE(TRACE_TIMER, "Updated DataIN timer for ITT: 0x%08x, DataSN:"
			" 0x%08x.\n", cmd->init_task_tag, data_sn);
	spin_unlock_bh(&cmd->datain_timeout_lock);
	
	return;
}

/*	iscsi_start_datain_timer():
 *
 *	Called with cmd->datain_timeout_lock held.
 */
extern void iscsi_start_datain_timer (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn)
{
	iscsi_channel_t *channel;
	iscsi_session_t *sess = SESS(conn);
	
	if (cmd->datain_timer_flags & DATAIN_TF_RUNNING)
		return;

	channel = sess->channel;
	
	init_timer(&cmd->datain_timer);
	cmd->datain_timer.expires = (jiffies + ISCSI_CA(channel)->datain_timeout * HZ);
	cmd->datain_timer.data = (unsigned long) cmd;
	cmd->datain_timer.function = iscsi_handle_datain_timeout;

	cmd->datain_timer_flags |= DATAIN_TF_RUNNING; 
	add_timer(&cmd->datain_timer);

	TRACE(TRACE_TIMER, "Starting DataIN timer for ITT: 0x%08x on CID:"
			" %hu.\n", cmd->init_task_tag, conn->cid);
	return;
}

/*	iscsi_stop_datain_timer():
 *
 *
 */
extern void iscsi_stop_datain_timer (iscsi_cmd_t *cmd)
{
	spin_lock_bh(&cmd->datain_timeout_lock);
	if (!(cmd->datain_timer_flags & DATAIN_TF_RUNNING)) {
		spin_unlock_bh(&cmd->datain_timeout_lock);
		return;
	}
	cmd->datain_timer_flags |= DATAIN_TF_STOP;
	spin_unlock_bh(&cmd->datain_timeout_lock);
	
	del_timer_sync(&cmd->datain_timer);

	spin_lock_bh(&cmd->datain_timeout_lock);
	cmd->datain_timer_flags &= ~DATAIN_TF_RUNNING;
	cmd->datain_timer_flags &= ~DATAIN_TF_STOP;
	spin_unlock_bh(&cmd->datain_timeout_lock);

	TRACE(TRACE_TIMER, "Stopped DataIN Timer for ITT: 0x%08x on CID"
		" %hu\n", cmd->init_task_tag, CONN(cmd)->cid);
	return;
}

