/*
 * Copyright (c) 2001, 2002, 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses definitions related to the main iSCSI Initiator driver.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#ifndef ISCSI_INITIATOR_CORE_H
#define ISCSI_INITIATOR_CORE_H

#include <linux/in.h>
#include <net/sock.h>
#include <net/tcp.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_device.h>
#include <scsi/scsi_host.h>

#define PYX_ISCSI_VERSION			"v1.6.2.3"

#define ISCSI_MAX_CHANNELS			32	/* Maximum iSCSI channels */
#define ISCSI_MAX_LUNS				16	/* Maximum iSCSI LUs per iSCSI Channel */
#define ISCSI_MAX_CONNECTIONS			32	/* Maximum iSCSI Connections per Session */
#define ISCSI_MAX_PING_DATA_BYTES		64	/* NOPOUT Ping Data buffer size */
#define ISCSI_MAX_INITIATORNAME_SIZE		256	/* Maximum InitiatorName buffer size */
#define ISCSI_MAX_TARGETNAME_SIZE		256	/* Maximum TargetName buffer size */
#define ISCSI_MAX_TARGETADDRESS_SIZE		256	/* Maximum TargetAddress buffer size */
#define ISCSI_MAX_TPG_FAILOVER_ADDRESSES	64	/* Size of iscsi_tpg_failover_t->tpga_list */
#define ISCSI_IOV_DATA_BUFFER			5	/* Extra iovecs on top of iovecs needed for SCSI request */
#define ISCSI_MISC_IOVECS			5	/* Miscellanous iovec count in iscsi_cmd_t */
#define ISCSI_NO_LOGIN_THREAD_WAIT		2	/* No available login thread timeout in seconds */
#define ISCSI_MAX_SNACKS_PER_CMD		32	/* Maximum used SNACks for single iscsi_cmd_t */
#define ISCSI_LOGIN_THREAD_COUNT		1	/* Login threads created on module init */
#define ISCSI_NETDEV_NAME_SIZE			12	/* Size of the Network Device Name Buffer */
#define ISCSI_OS_DEVICE_NAME			16	/* Size of OS dependant device names */

#define IPV4_BUF_SIZE				18
#define IPV6_BUF_SIZE				256

/* iscsi_channel_attrib_t sanity values */
#define CA_CLOSECONN_REINSTATEMENT		1
#define CA_CMDSN_TIMEOUT			5
#define CA_CMDSN_TIMEOUT_MAX			25
#define CA_CMDSN_TIMEOUT_MIN			2
#define CA_CMDSN_TIMEOUT_RETRIES		4	
#define CA_CMDSN_TIMEOUT_RETRIES_MAX		15
#define CA_CMDSN_TIMEOUT_RETRIES_MIN		1
#define CA_DATAIN_TIMEOUT			3
#define CA_DATAIN_TIMEOUT_MAX			10
#define CA_DATAIN_TIMEOUT_MIN			2
#define CA_DATAIN_TIMEOUT_RETRIES		5
#define CA_DATAIN_TIMEOUT_RETRIES_MAX		15
#define CA_DATAIN_TIMEOUT_RETRIES_MIN		2
#define CA_IMMEDIATE_LOGOUT			1 /* Normally sent as immediate, and not used for REMOVECONNFORRECOVERY */
#define CA_IMMEDIATE_NOP			0 /* Should not be used */
#define CA_IMMEDIATE_TEXT			0 /* Normally sent as non-immediate */
#define CA_LOGIN_RETRIES			5 /* This should always be non-zero :) */
#define CA_LOGIN_RETRY_WAIT			2
#define CA_LOGIN_RETRY_WAIT_MAX			10
#define CA_LOGIN_RETRY_WAIT_MIN			1
#define CA_LOGIN_TIMEOUT			25
#define CA_LOGIN_TIMEOUT_MAX			50
#define CA_LOGIN_TIMEOUT_MIN			5
#define CA_LOGOUT_TIMEOUT			10
#define CA_LOGOUT_TIMEOUT_MAX			25
#define CA_LOGOUT_TIMEOUT_MIN			5
#define CA_LU_SCAN_TIMEOUT			5
#define CA_LU_SCAN_TIMEOUT_MAX			10
#define CA_LU_SCAN_TIMEOUT_MIN			2
#define CA_NETIF_TIMEOUT			2
#define CA_NETIF_TIMEOUT_MAX			255
#define CA_NETIF_TIMEOUT_MIN			1
#define CA_NOPOUT_TIMEOUT			5
#define CA_NOPOUT_TIMEOUT_MAX			25
#define CA_NOPOUT_TIMEOUT_MIN			3
#define CA_NOPOUT_RESPONSE_TIMEOUT		5
#define CA_NOPOUT_RESPONSE_TIMEOUT_MAX		15
#define CA_NOPOUT_RESPONSE_TIMEOUT_MIN		3
#define CA_RANDOM_DATAOUT_PDU_OFFSETS		0
#define CA_SCSI_TASK_ONLINE_TIMEOUT		15
#define CA_SCSI_TASK_ONLINE_TIMEOUT_MIN		10
#define CA_SCSI_TASK_ONLINE_TIMEOUT_MAX		30
#define CA_SCSI_TASK_OFFLINE_TIMEOUT		5
#define CA_SCSI_TASK_OFFLINE_TIMEOUT_MIN	2
#define CA_SCSI_TASK_OFFLINE_TIMEOUT_MAX	10
#define CA_TPGFAILOVER				1
#define CA_TPGFAILOVER_ATTEMPTS			0 /* This should always be zero :) */
#define CA_TPGFAILOVER_LOGIN_RETRIES		25
#define CA_TPGFAILOVER_LOGIN_RETRIES_MAX	255
#define CA_TPGFAILOVER_LOGIN_RETRIES_MIN	5

/* iscsi_conn_t->network_transport */
#define ISCSI_TCP				0
#define ISCSI_SCTP_TCP				1
#define ISCSI_SCTP_UDP				2
#define ISCSI_IWARP_TCP				3
#define ISCSI_IWARP_SCTP			4
#define ISCSI_INFINIBAND			5

/* iscsi_data_count_t->type */
#define ISCSI_RX_DATA				1
#define ISCSI_TX_DATA				2

/* Used in iscsi_cmd_t->cmd_flags */
#define ICF_ATTACHED_TO_CONN			0x00000001
#define ICF_DATAIN_TIMEOUT			0x00000002
#define ICF_GOT_LAST_DATAIN			0x00000004
#define ICF_OUT_OF_ORDER_R2TSN			0x00000008
#define ICF_WITHIN_COMMAND_RECOVERY		0x00000010
#define ICF_CMD_STATE_ACTIVE			0x00000020
#define ICF_CMD_STATE_WAIT_SEM			0x00000040
#define ICF_CMD_NONIMMEDIATE_ACTIVE		0x00000080
#define ICF_ACTIVE_SNACKS			0x00000100

/* Used in iscsi_cmd_t->state */
#define ISTATE_SEND_DATAOUT			1
#define ISTATE_SEND_LAST_DATAOUT		2
#define ISTATE_SENT_LAST_DATAOUT		3
#define ISTATE_SEND_LOGOUT_REQ			4
#define ISTATE_SEND_IMMEDIATE_LOGOUT_REQ	5
#define ISTATE_SENT_LOGOUT_REQ			6
#define ISTATE_SENT_IMMEDIATE_LOGOUT_REQ	7
#define ISTATE_SEND_NOPOUT			8
#define ISTATE_SEND_IMMEDIATE_NOPOUT		9
#define ISTATE_SENT_NOPOUT			10
#define ISTATE_SENT_IMMEDIATE_NOPOUT		11
#define ISTATE_SEND_SCSI_CMD			12
#define ISTATE_SENT_SCSI_CMD			13
#define ISTATE_SEND_TEXT_REQ			16
#define ISTATE_SEND_IMMEDIATE_TEXT_REQ		17
#define ISTATE_SENT_TEXT_REQ			18
#define ISTATE_SENT_IMMEDIATE_TEXT_REQ		19
#define ISTATE_SEND_TMR				20
#define ISTATE_SEND_IMMEDIATE_TMR		21
#define ISTATE_SENT_TMR				22
#define ISTATE_SENT_IMMEDIATE_TMR		23
#define ISTATE_SEND_RETRY_COMMAND		24
#define ISTATE_SENT_RETRY_COMMAND		25
#define ISTATE_SEND_NOPOUT_RESPONSE		28
#define ISTATE_REMOVE				29

/* Used to determine action on iscsi_cmd_t's during recovery */
#define ISCSI_COMMAND_NONE			0
#define ISCSI_COMMAND_RESET			1
#define ISCSI_COMMAND_RETRY			2
#define ISCSI_COMMAND_FAIL			3

/* Used in iscsi_conn_t->conn_flags */
#define CONNFLAG_SEND_NOPOUT			0x01
#define CONNFLAG_TPGF_USE_ORIG			0x02
#define CONNFLAG_SCTP_STRUCT_FILE		0x04

/* Used in iscsi_OS_complete_commands_for_caller(). */
#define SESSION_CALLER				1
#define CONNECTION_CALLER			2

/* Used for iscsi_channel_t->status. */
#define ISCSI_CHANNEL_FREE			1
#define ISCSI_CHANNEL_INACTIVE			2
#define ISCSI_CHANNEL_ACTIVE			3
#define ISCSI_CHANNEL_REESTABLISHING		4
#define ISCSI_CHANNEL_SESSION_LOGGED_OUT	5
#define ISCSI_CHANNEL_FAILED			6

/* Used for iscsi_map_sg_t->map_flags */
#define MAP_SG_KMAP				0x01

/* Used for iscsi_set_channel_status() */
#define SCS_FAIL_OS_SCSI_CMDS			0x00000001
#define SCS_RETRY_OS_SCSI_CMDS			0x00000002
#define SCS_STOP_REINSTATEMENT_THREAD		0x00000004
#define SCS_STOP_REINSTATEMENT_THREAD_AND_SET	0x00000008
#define SCS_LUN_SCAN				0x00000010
#define SCS_RELEASE_LUNS			0x00000020
#define	SCS_FREE_CHANNEL_CONNS			0x00000040
#define SCS_REESTABLISHING_CHECK		0x00000080

/* Used for iscsi_login_holder_t->lh_flags */
#define LH_TARGETNAME_PRESENT			0x01

/* Used for iscsi_OS_queue_SCSI_cmd_to_channel() */
#define OS_SCSI_CMD_FAILED			-1
#define OS_SCSI_CMD_OK				0
#define OS_SCSI_CMD_RETRY			1

/* Used for iscsi_decide_datain_action() return values */
#define DATAIN_CANNOT_RECOVER			-1
#define DATAIN_NORMAL				0
#define DATAIN_DELAYED_PHASE_COLLAPSE		1
#define DATAIN_SEND_TO_SCSI			2
#define DATAIN_WITHIN_COMMAND_RECOVERY		3

/* Used for iscsi_session_t->cmdsn_timer_flags */	
#define CMDSN_TF_RUNNING			0x01
#define CMDSN_TF_STOP				0x02

/* Used for iscsi_cmd_t->datain_timer_flags */
#define DATAIN_TF_RUNNING			0x01
#define DATAIN_TF_STOP				0x02

/* Used for iscsi_conn_t->netif_timer_flags */
#define NETIF_TF_RUNNING			0x01
#define NETIF_TF_STOP				0x02

/* Used for iscsi_conn_t->nopout_timer_flags */
#define NOPOUT_TF_RUNNING			0x01
#define NOPOUT_TF_STOP				0x02

/* Used for iscsi_conn_t->nopout_response_timer_flags */
#define NOPOUT_RESPONSE_TF_RUNNING		0x01
#define NOPOUT_RESPONSE_TF_STOP			0x02

/* Fill this in.. */
#define CS_TF_RUNNING				0x01
#define CS_TF_STOP				0x02

/* Fill this in.. */
#define TIMER_RUNNING				0x01
#define TIMER_STOP				0x02

/* Used for iscsi_channel_req_t->cr_state */
#define	CH_REQ_STATE_ATTACHED			1
#define CH_REQ_STATE_ACTIVE			2
#define CH_REQ_STATE_FREE			3
#define CH_REQ_STATE_STOP_ACTIVE		4
#define CH_REQ_STATE_STOP_COMPLETE		5

/* Used for iscsi_channel_req_t->cr_action */
#define CH_REQ_ACTION_LOGIN			0x01
#define CREQ_LUN_SCAN				0x02
#define CREQ_LUN_UNSCAN				0x04
#define CREQ_STOP_SCSI_CMD			0x08
#define CREQ_STOP_SCSI_CMD_FROM_TIMER		0x10

/* Used for iscsi_update_statsn() return values */
#define STATSN_ERROR_CANNOT_RECOVER		-1
#define STATSN_NORMAL_OPERATION			0

/* SCSI Status Codes */
#define GOOD						0x00
#define CONDITION_MET					0x04
#define INTERMEDIATE					0x10
#define INTERMEDIATE_CONDITION_MET			(INTERMEDIATE|CONDITION_MET)
#define TASK_SET_FULL					QUEUE_FULL

typedef struct iscsi_queue_req_s {
	u8		state;
	struct iscsi_cmd_s *cmd;
	struct iscsi_queue_req_s *next;
	struct iscsi_queue_req_s *prev;
} iscsi_queue_req_t;

typedef struct iscsi_data_count_s {
	int		data_length;		/* Total transfer length */
	int		sync_and_steering;	/* sync and steering is enabled */
	int		type;			/* ISCSI_TX_DATA or ISCSI_RX_DATA */
	u32		iov_count;		/* iovecs count for this IP stack request */
	u32		ss_iov_count;		/* iovecs count with sync and steering accounted for this IP stack request */
	u32		ss_marker_count;	/* sync and steering marker count for this IP stack request */
	struct iovec	*iov;			/* array of iovecs for this IP stack request, number == iov_count */
} iscsi_data_count_t;

typedef struct iscsi_param_list_s {
	struct iscsi_param_s *param_start;	/* start of iSCSI parameter list */
	struct iscsi_extra_response_s *extra_response_start; /* Start of any extra responses not in parameter list */
} iscsi_param_list_t;

typedef struct iscsi_target_rdr_s {
	u16		rdr_port;
	u16		rdr_tpgt;
	u32		rdr_ipv4_address;
} iscsi_target_rdr_t;

typedef struct iscsi_targetaddress_s {
	unsigned char	net_dev[ISCSI_NETDEV_NAME_SIZE];
	u8		need_tpgt;
	u16		port;			/* port address of discovered TargetAddress entry */
	u16		tpgt;			/* target portal group tag of discovered TargetAddress entry */
	u32		ipv4_address;		/* IPv4 address of discovered TargetAddress entry */
	struct iscsi_target_s *target;		/* point to iSCSI Target this TargetAddress entry belongs to */
	struct iscsi_targetaddress_s *next;
	struct iscsi_targetaddress_s *prev;
} iscsi_targetaddress_t;

typedef struct iscsi_target_s {
	unsigned char	targetname[ISCSI_MAX_TARGETNAME_SIZE]; /* TargetName */
	spinlock_t	ta_lock;		/* Lock protecting iscsi_targetaddress_t list */
	iscsi_targetaddress_t *ta_head;		/* Pointer to start of target address list */
	iscsi_targetaddress_t *ta_tail;		/* Pointer to end of target address list */
	struct iscsi_target_s *next;		/* Pointer to next known target */
	struct iscsi_target_s *prev;
} iscsi_target_t;

typedef struct iscsi_tpg_failover_address_s {
	unsigned char	net_dev[ISCSI_NETDEV_NAME_SIZE];
	u16		port;
	u32		ipv4_address;
} iscsi_tpg_failover_address_t;

typedef struct iscsi_tpg_failover_s {
	unsigned char	tpga_list_built;
	u16		tpgt;
	u32		current_tpga;
	u32		failover_attempts;
	u32		tpga_count;
	iscsi_tpg_failover_address_t tpga_list[ISCSI_MAX_TPG_FAILOVER_ADDRESSES];
} iscsi_tpg_failover_t;

typedef struct iscsi_r2t_s {
	u32		data_sn;		/* Incremented from 0 by DataOut PDUs for this R2T */
	u32		pdu_count;		/* Number of iscsi_pdu_t structs for DataPDUInOrder=No */
	u32		pdu_send_count;		/* Number of PDUs sent for DataPDUInOrder=No */
	u32		pdu_send_order;		/* Order of PDUs in iscsi_cmd_t->pdu_list for DataPDUInOrder=No */
	u32		pdu_start;		/* Offset in iscsi_cmd_t->pdu_list for DataPDUInOrder=No */
	u32		offset;			/* Offset in SCSI buffer of next data to send */
	u32		r2t_sn;			/* R2T Sequence Number from Target */
	u32		targ_xfer_tag;		/* Target Transfer Tag */
	u32		xfer_length;		/* Bytes left to send in r2t sequence */
	struct iscsi_seq_s *seq;		/* Pointer for DataSequenceInOrder=No holder */
	struct iscsi_r2t_s *next;		/* Pointer to next iscsi_r2t_t */
	struct iscsi_r2t_s *prev;		/* Pointer to previous iscsi_r2t_t */
} iscsi_r2t_t;

typedef struct iscsi_lun_s {
	int		lun;
	void		*OS_SCSI_lu; /* struct scsi_device goes here */
	struct iscsi_channel_s *channel;        
} iscsi_lun_t;
        
#define LUN(c)                  ((iscsi_lun_t *)(c->lun))

typedef struct iscsi_cmd_s {
	u8		cmdsn_retries;		/* Number of times this CmdSN has been retried */
	u8		datain_retries;		/* Datain Retry Count */
	u8		datain_timer_flags;	/* Datain Timer Flags */
	u8		state;			/* Command state */
	u8		task_mgt_function;	/* Task Management function when opcode == ISCSI_INIT_TASK_MGMT_CMND */
	u8		task_mgt_response;	/* Task Management response when opcode == ISCSI_INIT_TASK_MGMT_CMND */
	u8		unsolicited_data_out;	/* Command contains Unsolicited DataOUT */
	u16		logout_cid;		/* Connection ID contained in PDU when opcode == ISCSI_INIT_LOGOUT_CMND */
	u32		acked_datasn;		/* Last DataSN-1 acknowledged with an DataACK SNACK */
	u32		init_task_tag;		/* Initiator Task Tag for this command */
	u32		cmdsn;			/* iSCSI Command Sequence Number assigned to this command */
	u32		statsn;			/* StatSN assigned from target in an response */
	u32		data_length;		/* Total length of OS SCSI request */
	u32		data_sn;		/* Incremented from 0 by DataIn PDUs for this command */
	u32		data_offset;		/* Offset to next data to receive or R2T */
	u32		data_checksum;		/* Data Digeset for attached data */
	u32		cmd_flags;		/* Command flags */
	u32		iov_data_count;		/* Number of data iovecs used for IP stack calls */
	u32		iov_misc_count;		/* Number of miscellanous iovecs used for IP stack calls */
	u32		next_burst_len;		/* For MaxBurstLength in DataIN Sequences */
	u32		orig_iov_data_count;	/* Number of iovecs allocated for iscsi_cmd_t->iov_data */
	u32  		outstanding_r2ts;	/* Number of outstanding R2Ts */
	u32		pdu_count;		/* Used for DataPDUInOrder=No */
	u32		pdu_send_order;		/* Ordering of Sequence for DataPDUInOrder=No */
	u32		pdu_start;		/* Current iscsi_pdu_t in iscsi_cmd_t->pdu_list */
	u32		read_data_done;		/* Total size in bytes received so far of READ data */
	u32		r2t_sn;			/* Incremented from 0 by R2Ts */
	u32		seq_count;		/* Used for DataSequenceInOrder=No */
	u32		seq_no;			/* Used for DataPDUInOrder=No, DataSequenceInOrder=Yes */
	u32		seq_start_offset;	/* Used for DataSequenceInOrder=Yes */
	u32		seq_end_offset;		/* Used for DataSequenceInOrder=Yes */
	u32		tx_size;		/* Bytes to be transferred for this PDU. */
	u32		buf_ptr_size;		/* Size of *buf_ptr, useful for checking */
	void		*buf_ptr;		/* Pointer to additional data for command */
	struct scsi_cmnd *scsi_cmnd;		/* The SCSI Command :-) */
	unsigned char	pdu[ISCSI_HDR_LEN+CRC_LEN]; /* iSCSI PDU Header + CRC */
	unsigned char	data_pdu[ISCSI_HDR_LEN+CRC_LEN]; /* iSCSI DataOUT PDU Header + CRC */
	atomic_t	cmd_usage_count;	/* Number of pointer accesses related to iSCSI transport handling */
	atomic_t	scsi_usage_count;	/* Number of pointer accesses releated to SCSI transport handling */
	atomic_t	cmd_waiting_on_uc;	/* Waiting for Command usagecount to return to zero */
	atomic_t	scsi_waiting_on_uc;	/* Waiting for SCSI usagecount to return to zero */
	atomic_t	immed_queue_count;	/* Number of times iscsi_cmd_t is present in immediate queue */
	spinlock_t	cmd_usage_lock;		/* spinlock for accessing cmd_usage_lock */
	spinlock_t	scsi_usage_lock;	/* spinlock for accessing scsi usage lock */
	spinlock_t	datain_timeout_lock;	/* spinlock for accessing datain timeout handler */
	spinlock_t	r2t_lock;		/* spinlock for accessing r2t list */
	spinlock_t	state_lock;		/* spinlock for accessing iscsi_cmd_t->state */
	iscsi_lun_t	lun;			/* iSCSI Logical Unit iscsi_cmd_t is destined for */
	iscsi_r2t_t	*r2t_head;		/* First outstanding R2T */
	iscsi_r2t_t	*r2t_tail;		/* Last outstanding R2T */
	struct semaphore	cmd_state_wait_sem;	/* Used for controlling iscsi_cmd_t state changes */
	struct semaphore	cmd_waiting_on_uc_sem;	/* Used for controlling return to zero usage count */
	struct semaphore	scsi_waiting_on_uc_sem;	/* Used for controlling return to zero usage count */
	struct timer_list datain_timer;		/* Timer for DataIN */
	struct iovec	*iov_data;		/* Iovecs allocated from pool */
	struct iovec	iov_misc[ISCSI_MISC_IOVECS]; /* Iovecs for miscellanous purposes */
	struct iscsi_logout_attempt_s *la;	/* Used for logout purposes when opcode == ISCSI_INIT_LOGOUT_CMND */
	struct iscsi_cmd_s *ref_cmd;		/* Used for Task Management Requests */
	struct iscsi_conn_s *conn;		/* Pointer to allegiant connection */
	struct iscsi_pdu_s *pdu_list;		/* Pointer to array of iscsi_pdu_t for DataPDUInOrder=No */
	struct iscsi_pdu_s *pdu_ptr;		/* Pointer to reduce iscsi_get_pdu_holder() calls */
	struct iscsi_seq_s *seq_list;		/* Pointer to array of iscsi_seq_t for DataSequenceInOrder=No */
	struct iscsi_seq_s *seq_ptr;		/* Pointer to reduce iscsi_get_seq_holder() calls */
	iscsi_queue_req_t *i_next;
	struct iscsi_cmd_s *next;		/* Pointer to next iSCSI Command */
	struct iscsi_cmd_s *prev;		/* Pointer to previous iSCSI Command */
} iscsi_cmd_t;

#include <iscsi_initiator_seq_and_pdu_list.h>

typedef struct iscsi_conn_s {
	unsigned char	net_dev[ISCSI_NETDEV_NAME_SIZE]; /* Name of network interface, used for early link failure detection */
	u8		netif_timer_flags;	/* Flags for checking early link layer detection in network PHY */
	u8		nopout_timer_flags;	/* Flags for checking when to send iSCSI NOPOUT Pings */
	u8		nopout_response_timer_flags; /* Flags for checking response to NOPOUT Pings */
	u8		network_transport;
	u8		tx_immediate_queue;	/* Immediate queue is active in tx thread */
	u16		cid;			/* iSCSI Connection ID */
	u16		login_port;		/* Port number of Target this connection is logged into */
	u16		orig_login_port;	/* Used during discovery/recovery */
	u32		auth_id;		/* Unique number used for communcation with authenication daemon */
	u32		conn_flags;		/* Connection flags */
	u32		conn_state;		/* Connection state */
	u32		exp_statsn;		/* Expected Status Sequence Number */
	u32		login_ip;		/* IPv4 Address of Target connection is logged into */
	u32		missing_statsn_count;	/* Missing StatSN count */
	u32		ping_num;		/* Unique counter for NOPOUTs, value is contained in ping data */
	u32		of_marker;		/* OFMarkInt's Current Value */
	u32		orig_login_ip;		/* Used during discovery/recovery */
	u32		if_marker;		/* IFMarkInt's Current Value */
	u32		if_marker_offset;	/* Used for calculating IFMarker offset to next PDU */
	u32		unsolicited_missing_statsn; /* Missing StatSN counter for StatSNs without an iscsi_cmd_t */
	atomic_t	check_immediate_queue;	/* Used for check accessing to immediate execution queue */
	atomic_t	connection_exit;	/* Used to shutdown allocated thread set */
	atomic_t	connection_reinstatement; /* Connection is in connection reinstatement */
	atomic_t	conn_waiting_on_uc;	/* Connection is waiting for conn_usage_count to return to zero before closing the connection */
	atomic_t	conn_usage_count;	/* Accesses to this structure outside of rx/thread threads */
	atomic_t	sleep_on_conn_wait_sem;	/* Used during connection shutdown/failure */
	atomic_t	transport_failed;	/* Network transport failed expectedly */
	iscsi_cmd_t	*cmd_head;		/* Pointer to start of command list */
	iscsi_cmd_t	*cmd_tail;		/* Pointer to end of command list */
	iscsi_queue_req_t *immed_queue_head;	/* Pointer to start of immediate queue */
	iscsi_queue_req_t *immed_queue_tail;	/* Pointer to end of immediate queue */
	iscsi_conn_ops_t *conn_ops;		/* iSCSI connection parameters, used during normal operation */
	spinlock_t	cmd_lock;		/* Spinlock for accessing connection command list */
	spinlock_t	conn_usage_lock;	/* Spinlock for accessing final conn_usage_count */
	spinlock_t	error_lock;		/* Spinlock for accessing missing_statsn list */
	spinlock_t	immed_queue_lock;	/* Spinlock for accessing immediate command list */
	spinlock_t	netif_lock;		/* Spinlock for accessing network interface timeout handler */
	spinlock_t	nopout_timer_lock;	/* Spinlock for accessing NOPOUT timer handler */
	spinlock_t	state_lock;		/* Spinlock for conn->conn_state */
	iscsi_param_list_t *param_list;		/* Linked list of iSCSI connection parameters, only used during login */
	struct kobject		conn_val_obj;
	struct kobject		conn_ops_obj;
	struct semaphore	conn_logout_sem;	/* Used during iSCSI connection logout */
	struct semaphore	conn_waiting_on_uc_sem; /* used for sleeping until conn_usage_count returns to zero */
	struct semaphore	conn_wait_sem;	/* used for sleeping until connection reinstatement is complete */
	struct semaphore	tx_sem;			/* Tx thread sempahore */
	iscsi_targetaddress_t	*target_address; /* Pointer to iSCSI Target connection is logged into */
	struct net_device		*net_if;	/* For early link layer detection */
	/*
	 * IPv6 Access goes here
	 * 
	 * Would also be nice to get struct net_device access from struct socket. :-)
	 */
	struct socket		*sock;		/* Pointer to socket for this connection */
	struct sockaddr_in	s_addr;
	struct timer_list	nopout_timer;	/* Timer for nopout ping sending */
	struct timer_list	nopout_response_timer;	/* Timeout for nopout ping response */
	struct timer_list	transport_timer;	/* Timer used for early link failure detection */
#ifdef CRYPTO_API_CRC32C
	struct crypto_tfm	*conn_rx_tfm;	/* Used for receive CRC32C calculations */
	struct crypto_tfm	*conn_tx_tfm;	/* Used for transmit CRC32C  calculations */
	struct scatterlist	*conn_rx_tfm_sg; /* Used for receive CRC32C calculations */
	struct scatterlist	*conn_tx_tfm_sg; /* Used for transmit CRC32C  calculations */
#endif
	struct iscsi_thread_set_s *thread_set;	/* Pointer to thread set of rx and tx threads */
	struct iscsi_session_s	*session;	/* Pointer to corresponding session for this connection */
	struct iscsi_conn_s	*next;		/* Pointer to next iSCSI Connection */
	struct iscsi_conn_s	*prev;		/* Pointer to previous iSCSI Connection */
} iscsi_conn_t;

#define to_iscsi_conn(d) container_of(d, iscsi_conn_t, conn_class_dev);

#define CONN(cmd)		((iscsi_conn_t *)(cmd)->conn)
#define CONN_OPS(conn)		((iscsi_conn_ops_t *)(conn)->conn_ops)

typedef struct iscsi_session_s {
	u8		cmdsn_timer_flags;	/* CmdSN Timeout is armed */
//#warning FIXME: Get MAC address for hardware NICs
	u8		isid[6];		/* 48-bit Initiator ID */
	u8		reinstatement_set;
	u8		session_state;		/* Current session state */
	u8		no_connection_reinstatement; /* Flag used for forcing session reinstatement */
	u16		cid_counter;		/* Used for assigning connection IDs */
	u16		tsih;			/* Target session identifying handle, assigned by target */
	u32		cur_task_tag;		/* Current Initiator Task Tag */
	u32		targ_xfer_tag;		/* Current Target Xfer Tag (Used for Bookkeeping) */
	u32		timed_out_cmdsn;	/* CmdSN that has timed out */
	u32		timed_out_cmdsn_count;	/* Number of times timed_out_cmdsn has timed out */
	u32		cur_cmdsn;		/* Current Command Sequence Number */
	u32		exp_cmdsn;		/* Expected Cmmand Seqeunce Number */
	u32		max_cmdsn;		/* Maximum Command Sequence Number */
	u32		sid;			/* PyX specific session ID */
	atomic_t	nconn;			/* Number of connections within this session */
	atomic_t	pool_count;		/* Number of iscsi_cmd_ts in session pool */
	atomic_t	session_continuation;	/* session continuation is active */
	atomic_t	session_logout;		/* session is performing logout */
	atomic_t	session_reinstatement;	/* session reinstatement is active */
	atomic_t	session_stop_active;	/* used for stopping active connection when one files in non ERL=2 sessions */
	atomic_t	session_waiting_on_uc;	/* sleeping is waiting for session_usage_count to return to zero before closing the session */
	atomic_t	session_usage_count;	/* accesses to this structure outside of rx/thread threads */
	atomic_t	sleep_on_sess_wait_sem;	/* session is sleeping on session_wait_sem, used for reinstatement */
	atomic_t	stop_reinstatement;	/* Stop session reinstatement from occuring */
	iscsi_cmd_t	*cr_cmd_queue_head;	/* head of successfuly reassigned commands from connection recovery */
	iscsi_cmd_t	*cr_cmd_queue_tail;	/* tail of successfuly reassigned commands from connection recovery */
	iscsi_cmd_t	*pending_head;		/* head of commands waiting for CmdSN window to open */
	iscsi_cmd_t	*pending_tail;		/* tail of commands waiting for CmdSN window to open */
	iscsi_cmd_t	*pool_head;		/* head of unused commands in session pool */
	iscsi_cmd_t	*pool_tail;		/* tail of unused commands in session pool */
	iscsi_conn_t	*conn_head;		/* head of connection list */
	iscsi_conn_t	*conn_tail;		/* tail of connection list */
	iscsi_conn_t	*schedule_conn;		/* connection to activate tx_sem and tx thread for load balancing */
	spinlock_t	cmdsn_lock;		/* used for updating cur_cmdsn, exp_cmdsn, max_cmdsn */
	spinlock_t	conn_lock;		/* used for adding/removing connections */
	spinlock_t	conn_schedule_lock;	/* used for multiple connection scheduling */
	spinlock_t	cr_cmd_queue_lock;	/* used for adding successfully reassigned commands to conn recovery queue */
	spinlock_t	cr_lock;		/* used for connection recovery entries */
	spinlock_t	cr_queue_lock;		/* used for connection recovery queue entries */
	spinlock_t	itt_lock;		/* used for serializing access to iscsi_session_t->cur_task_tag */
	spinlock_t	la_lock;
	spinlock_t	pending_lock;		/* used for adding commands to pending queue */
	spinlock_t	pool_lock;		/* used for session command pool */
	spinlock_t	reinstatement_lock;
	spinlock_t	session_usage_lock;	/* used for controlling final access of session_usage_count */		
	struct kobject		sess_val_obj;
	struct kobject		sess_ops_obj;
	struct semaphore	discovery_sem;		/* used for discovery */
	struct semaphore	session_wait_sem;	/* used for session shutdown */
	struct semaphore	session_waiting_on_uc_sem; /* used for sleeping until session_usage_count returns to zero when closing the session */
	struct semaphore	stop_reinstatement_sem;
	iscsi_sess_ops_t *sess_ops;		/* iSCSI session parameters */
	struct timer_list cmdsn_timer;		/* used for checking CmdSN timeouts */
	struct iscsi_channel_s *channel;	/* SCSI channel this session is assigned to */
	struct iscsi_session_s *next;		/* pointer to next session in global list */
	struct iscsi_session_s *prev;		/* pointer to previous session in global list */
} iscsi_session_t;

#define SESS(conn)		((iscsi_session_t *)(conn)->session)
#define SESS_OPS(sess)		((iscsi_sess_ops_t *)(sess)->sess_ops)
#define SESS_OPS_C(conn)	((iscsi_sess_ops_t *)(conn)->session->sess_ops)

typedef struct iscsi_channel_conn_s {
	unsigned char	net_dev[ISCSI_NETDEV_NAME_SIZE]; /* OS dependant network interface name */
	int		network_transport;	/* network transport type from failed connection */
	u16		cid;			/* iSCSI connection ID of failed connection */
	u16		port;			/* port address of target for failed connection */
	u32		ipv4_address;		/* IPv4 address of target for failed connection */
	iscsi_conn_ops_t conn_ops;		/* iSCSI connection ops from failed connection */
	iscsi_sess_ops_t sess_ops;		/* iSCSI session ops from failed connection */
	struct iscsi_channel_conn_s *next;	/* next iscsi_channel_conn_t entry in iscsi_channel_t list */
	struct iscsi_channel_conn_s *prev;	/* previous iscsi_channel_conn_t entry in iscsi_channel_t list */
} iscsi_channel_conn_t;

typedef struct iscsi_channel_attrib_s {
	u32		closeconn_reinstatement;
	u32		cmdsn_timeout;
	u32		cmdsn_timeout_retries;
	u32		datain_timeout;
	u32		datain_timeout_retries;
	u32		immediate_logout;
	u32		immediate_text;
	u32		login_retries;
	u32		login_retry_wait;
	u32		login_timeout;
	u32		logout_timeout;
	u32		lu_scan_timeout;
	u32		netif_timeout;
	u32		nopout_timeout;
	u32		nopout_response_timeout;
	u32		random_dataout_pdu_offsets;
	u32		scsi_task_online_timeout;
	u32		scsi_task_offline_timeout;
	u32		tpgfailover;
	u32		tpgfailover_attempts;
	u32		tpgfailover_login_retries;
} iscsi_channel_attrib_t;

typedef struct iscsi_channel_req_s {
	u32		cr_req_id;
	int		cr_action;
	int		cr_state;
	void		*cr_data;
	spinlock_t	cr_state_lock;
	struct semaphore	cr_data_sem;
	struct semaphore	cr_stop_sem;
	struct iscsi_channel_req_s *next;
	struct iscsi_channel_req_s *prev;
} iscsi_channel_req_t;

typedef struct iscsi_channel_lun_s {
	unsigned char	OS_device_name[ISCSI_OS_DEVICE_NAME];
	int		OS_device_set;
	int		offline;
	u32		iscsi_tasks;
	u32		iscsi_current_tasks;
	u64		total_bytes;
} iscsi_channel_lun_t;

typedef struct iscsi_channel_s {
	u8		ch_lu_scan_timer_flags;
	u8		active_luns;
	u8		status;			/* channel status */
	u8		ch_isid[6];
	int		channel_id;		/* SCSI channel ID */
	int		host_id;		/* SCSI host ID */
	int		target_id;		/* SCSI target ID */
	u32		cr_req_id_counter;
	u32		ch_max_sectors;
	struct device	dev;
	struct class_device ch_class_dev;
	struct kobject conn_parent;
        struct completion dev_released;
        struct completion class_dev_released;
	struct scsi_cmnd *scsi_cmnd_head;
	struct scsi_cmnd *scsi_cmnd_tail;
	struct Scsi_Host *scsi_host;
	atomic_t	adding_connections;	/* channel is adding additional connections after reinstatement */
	atomic_t	connection_count;	/* number of failed iSCSI connection in list */
	atomic_t	force_channel_offline;
	atomic_t	reinstatement_thread_stop;
	atomic_t	stop_channel;
	iscsi_channel_conn_t *chan_conn_head;	/* pointer to head of failed connections for this channel */
	iscsi_channel_conn_t *chan_conn_tail;	/* pointer to tail of failed connections for this channel */
	iscsi_channel_attrib_t chan_attrib;	/* default channel values for session */
	iscsi_channel_req_t *cr_req_head;
	iscsi_channel_req_t *cr_req_tail;
	iscsi_channel_req_t *cr_scan_head;
	iscsi_channel_req_t *cr_scan_tail;
	spinlock_t	scsi_cmnd_lock;	/* used for accessing channel command list */
	spinlock_t	chan_conn_lock;		/* used for accessing failed connections list */
	spinlock_t	channel_state_lock;
	spinlock_t	cr_req_lock;
	spinlock_t	cr_scan_lock;
	iscsi_channel_lun_t ch_lun_list[ISCSI_MAX_LUNS];
	iscsi_param_list_t *param_list;
	iscsi_target_t *ch_target;
	struct semaphore	ch_access_sem;
	struct semaphore	ct_req_sem;
	struct semaphore	ct_scan_sem;
	struct semaphore	ct_req_done_sem;
	struct semaphore	ct_scan_done_sem;
	struct semaphore	ct_req_start_sem;
	struct semaphore	ct_scan_start_sem;
	struct semaphore	force_offline_sem;
	struct semaphore	reinstatement_thread_stop_sem; /* Used for stopping active reinstatement attempt */
	struct task_struct *channel_req_thread;
	struct task_struct *channel_scan_thread;
	struct task_struct *reinstatement_thread;	/* thread that is performing session reinstatement */
	struct timer_list ch_lu_scan_timer;
	iscsi_session_t *sess;			/* iSCSI session assign to this SCSI channel */
} iscsi_channel_t;

#define dev_to_iscsi_channel(d) container_of(d, iscsi_channel_t, dev)
#define class_dev_to_iscsi_channel(d) container_of(d, iscsi_channel_t, ch_class_dev);

#define ISCSI_CA(c)	(&(c)->chan_attrib)

#include <iscsi_initiator_thread_queue.h>

typedef struct iscsi_logout_attempt_s {
	u8		logout_reason;		/* iSCSI logout reason code */
	u8		logout_response;	/* iSCSI logour response code */
	u16		logout_cid;		/* iSCSI connection ID in logout request */
	u16		logout_on_cid;		/* iSCSI connection ID logout request was sent on */
	int		logout_built;		/* The logout_conn and/or logout_on_conn pointers have been set when logout_immediate != 0 */
	int		logout_lun_remove;	/* Remove iSCSI Logical Units from Host */
	int		logout_force;		/* Force the Logical Unit Remove regardless of Active iSCSI LUNs */
	int		logout_immediate;	/* iSCSI Logout request was sent with I_BIT set */
	atomic_t	logout_got_response;	/* received logout response for request */
	atomic_t	logout_timer_expired;	/* did not receive logout response before timer expired */
	iscsi_conn_t	*logout_conn;		/* iSCSI connection pointer of connection that is being logged out*/
	iscsi_conn_t	*logout_on_conn;	/* iSCSI connection pointer logout request was sent on */
	spinlock_t	logout_state_lock;	/* used for protecting logout_built */
	struct semaphore	logout_sem;		/* used for sleeping until response or timer expires */
	struct semaphore	logout_done_sem;
} iscsi_logout_attempt_t;

typedef struct iscsi_login_holder_s {
	unsigned char	net_dev[ISCSI_NETDEV_NAME_SIZE];
	unsigned char	lh_targetname[ISCSI_MAX_TARGETNAME_SIZE];
	u8		leading_conn;
	u8		network_transport;
	u8		session_reinstatement;
	u8		status_class;
	u8		status_detail;
	u8		tpg_failover;
	u16		conns;
	u16		port;
	int		lh_flags;
	int		login_ret;
	int		scsi_host_id;
	int		scsi_target_id;
	int		scsi_channel_id;
	u32		ipv4_address;
	iscsi_channel_t	*channel;
	iscsi_channel_conn_t *cc;
	iscsi_conn_t	*conn;
} iscsi_login_holder_t;

typedef struct iscsi_linux_map_s {
	int		map_reset;
	u32		iovec_length;
	u32		iscsi_offset;
	u32		current_sg_count;
	u32		current_sg_offset;
	u32		orig_sg_offset;
	u32		sg_count;
	u32		sg_current;
	u32		sg_length;
	struct page	*sg_page;
	struct page	*sg_last_page;
	struct scatterlist *map_sg;
	struct scatterlist *map_orig_sg;
	void		*iovec_base;
} iscsi_linux_map_t;

typedef struct iscsi_map_sg_s {
	int		map_flags;
	u32		data_length;
	u32		data_offset;
	iscsi_cmd_t	*cmd;
	struct iovec    *iov;
} iscsi_map_sg_t;

typedef struct iscsi_unmap_sg_s {
	u32		data_length;
	u32		sg_count;
	u32		sg_offset;
	iscsi_cmd_t	*cmd;
	iscsi_linux_map_t lmap;
} iscsi_unmap_sg_t;

typedef struct iscsi_global_s {
	unsigned char	initiatorname[ISCSI_MAX_INITIATORNAME_SIZE]; /* iSCSI Initiator/Node Name */
	u8		initname_set;		/* Set to 1 once InitiatorName is set via IOCTL */
	u8		in_shutdown;		/* In module removal */
	u8		randomize_pdus;
	u8		randomize_sequences;
	u16		isid_qualifier;
	u32		active_ts;
	u32		auth_id;		/* Unique indentifier used for the authentication daemon */
	u32		inactive_ts;
	u32		nsess;			/* Number of sessions */
	u32		thread_id;		/* Thread ID counter */
	u32		sid;			/* Global PyX specific session ID counter */
	spinlock_t	active_lt_lock;
	spinlock_t	active_ts_lock;
	spinlock_t	channel_lock;
	spinlock_t	inactive_lt_lock;
	spinlock_t	inactive_ts_lock;
	spinlock_t	session_lock;		/* Spinlock used for adding/removing sessions */
	spinlock_t	target_lock;		/* Spinlock used for adding/removing targets */
	iscsi_channel_t channels[ISCSI_MAX_CHANNELS];
	struct scsi_transport_template *iscsi_template;
	struct semaphore auth_sem;		/* Serialize access to userspace daemon */
	iscsi_session_t *sess_head;		/* Pointer to beginning of session list */
	iscsi_session_t *sess_tail;		/* Pointer to send of session list */
	iscsi_target_t	*target_head;		/* Pointer to start of iSCSI targets list */
	iscsi_target_t	*target_tail;		/* Pointer to end of iSCSI targets list */
	iscsi_thread_set_t *active_ts_head;
	iscsi_thread_set_t *active_ts_tail;
	iscsi_thread_set_t *inactive_ts_head;
	iscsi_thread_set_t *inactive_ts_tail;
} iscsi_global_t;

#endif /* ISCSI_INITIATOR_CORE_H */
