/*
 * Copyright (c) 2002, 2003, 2004, 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses the main functions releated to the iSCSI Initiator Core Stack.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_C

#include <linux/version.h>
#include <linux/module.h>
#include <linux/kmod.h>
#include <linux/net.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/crypto.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/in.h>
#include <linux/blkdev.h>
#include <net/sock.h>
#include <net/tcp.h>
#include <scsi.h>
#include <scsi/scsi.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_cmnd.h>
#include <scsi/scsi_device.h>
#define iscsi_scsi_task_t struct scsi_cmnd
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_initiator_serial.h>
#include <iscsi_initiator_crc.h>
#include <iscsi_initiator_debug_opcodes.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator.h>
#include <iscsi_initiator_channel.h>
#include <iscsi_initiator_discovery.h>
#include <iscsi_initiator_erl0.h>
#include <iscsi_initiator_erl1.h>
#include <iscsi_initiator_ioctl_defs.h>
#include <iscsi_initiator_linux.h>
#include <iscsi_initiator_login.h>
#include <iscsi_initiator_nego.h>
#include <iscsi_initiator_scsi.h>
#include <iscsi_initiator_sysfs.h>
#include <iscsi_initiator_tmr.h>
#include <iscsi_initiator_util.h>
#include <iscsi_initiator_parameters.h>

#undef ISCSI_INITIATOR_C

/*      iscsi_schedule_conn_for_work():
 *
 *	Simple round robin scheduling function that activates a connection's
 *	tx thread by upping conn->tx_sem.
 */
static inline void iscsi_schedule_conn_for_work (iscsi_session_t *sess)
{
	unsigned long flags;
	
	spin_lock_irqsave(&sess->conn_schedule_lock, flags);
	if (!sess->schedule_conn) {
		spin_unlock_irqrestore(&sess->conn_schedule_lock, flags);
		return;
	}

	if (atomic_read(&sess->nconn) == 1) {
		up(&sess->schedule_conn->tx_sem);
		spin_unlock_irqrestore(&sess->conn_schedule_lock, flags);
		return;
	}

	sess->schedule_conn = (!sess->schedule_conn->next) ?
		sess->conn_head : sess->schedule_conn->next;
	
	up(&sess->schedule_conn->tx_sem);
	spin_unlock_irqrestore(&sess->conn_schedule_lock, flags);
	
	return;
}       

/*	iscsi_non_scsi_queue_cmd():
 *
 *
 */
extern void iscsi_non_scsi_queue_cmd (
	iscsi_cmd_t *cmd,
	iscsi_session_t *sess)
{
	iscsi_add_cmd_to_session_queue(cmd, sess);
	iscsi_schedule_conn_for_work(sess);

	return;
}

/*	iscsi_scsi_queue_cmd():
 *
 *
 */
extern int iscsi_scsi_queue_cmd (
	struct scsi_cmnd *sc,
	iscsi_session_t *sess)
{
	iscsi_cmd_t *cmd;
	
	if (!(cmd = iscsi_allocate_cmd(sess, sc)))
		return(-1);
	
	cmd->state = ISTATE_SEND_SCSI_CMD;
	cmd->lun.OS_SCSI_lu = iscsi_OS_get_SCSI_lu(sc, &cmd->lun.lun);
	cmd->lun.channel = sess->channel;
	sc->SCp.ptr = (char *) cmd;

	iscsi_add_cmd_to_session_queue(cmd, sess);
	iscsi_schedule_conn_for_work(sess);

	return(0);
}

/*	iscsi_add_nopout():
 *
 *
 */
extern int iscsi_add_nopout (
	iscsi_conn_t *conn)
{
	iscsi_cmd_t *cmd;
	iscsi_session_t *sess = SESS(conn);
			                        
	if (!(cmd = iscsi_allocate_cmd(sess, NULL)))
		return(-1);
				                
	cmd->pdu[0] = ISCSI_INIT_NOP_OUT | I_BIT;
	cmd->state = ISTATE_SEND_IMMEDIATE_NOPOUT;

	iscsi_attach_cmd_to_conn(cmd, conn);
	iscsi_start_nopout_response_timer(conn);
	iscsi_add_cmd_to_immediate_queue(cmd, conn, cmd->state);
	
	return(0);	
}

/*	iscsi_add_nopout_noqueue():
 *
 *
 */
extern void iscsi_add_nopout_noqueue (
	iscsi_conn_t *conn)
{
	conn->conn_flags |= CONNFLAG_SEND_NOPOUT;
	up(&conn->tx_sem);
	
	return;
}

/*	iscsi_add_text_req():
 *
 *
 */
extern int iscsi_add_text_req (
	iscsi_conn_t *conn,
	unsigned char *payload,
	int immediate)
{
        unsigned char *text;
	u32 padding = 0, text_len = (strlen(payload) + 1);
	iscsi_cmd_t *cmd;
	iscsi_session_t *sess = SESS(conn);

	if (text_len > CONN_OPS(conn)->MaxRecvDataSegmentLength) {
		TRACE_ERROR("Cannot send Text Request with payload length: %u"
			" larger than MaxRecvDataSegmentLength: %u\n", text_len,
				CONN_OPS(conn)->MaxRecvDataSegmentLength);
		return(-1);
	}

	if (!(cmd = iscsi_allocate_cmd(sess, NULL)))
		return(-1);

	if ((padding = ((text_len) & 3)) != 0) {
		TRACE(TRACE_ISCSI, "Attaching %u additional bytes for"
			" padding.\n", padding);
	}

	if (!(text = (unsigned char *) kmalloc((text_len + padding), GFP_ATOMIC))) {
		TRACE_ERROR("Unable to allocate memory for outgoing text\n");
		iscsi_release_cmd_to_pool(cmd, sess);
		return(-1);
	}
	memset(text, 0, text_len + padding);
	memcpy(text, payload, text_len);

	cmd->buf_ptr = text;
	cmd->buf_ptr_size = text_len + padding;

	if (!immediate) {
		cmd->state = ISTATE_SEND_TEXT_REQ;
		iscsi_non_scsi_queue_cmd(cmd, sess);
	} else {
		cmd->pdu[0] = ISCSI_INIT_TEXT_CMND | I_BIT;
		cmd->state = ISTATE_SEND_IMMEDIATE_TEXT_REQ;
		iscsi_attach_cmd_to_conn(cmd, conn);
		iscsi_add_cmd_to_immediate_queue(cmd, conn, cmd->state);
	}
				
	return(0);
}

/*	iscsi_decide_list_to_build():
 *
 *	Called from: iscsi_send_cmd().
 */
static inline int iscsi_decide_list_to_build (
	iscsi_cmd_t *cmd,
	u32 immediate_data_length)
{
	iscsi_build_list_t bl;
	iscsi_channel_t *channel;
	iscsi_conn_t *conn = CONN(cmd);
	iscsi_session_t *sess = SESS(conn);
	struct scsi_cmnd *sc = cmd->scsi_cmnd;
	
	if (SESS_OPS(sess)->DataSequenceInOrder &&
	    SESS_OPS(sess)->DataPDUInOrder)
		return(0);

	if (sc->sc_data_direction == DMA_NONE)
		return(0);
	
	channel = sess->channel;
	memset(&bl, 0, sizeof(iscsi_build_list_t));
	
	if (sc->sc_data_direction == DMA_FROM_DEVICE) {
		bl.data_direction = ISCSI_PDU_READ;
		bl.type = PDULIST_NORMAL;
	} else {
		bl.data_direction = ISCSI_PDU_WRITE;
		bl.immediate_data_length = immediate_data_length;
		if (ISCSI_CA(channel)->random_dataout_pdu_offsets)
			 bl.randomize |= RANDOM_DATAOUT_PDU_OFFSETS;
		
		if (!immediate_data_length && !cmd->unsolicited_data_out)
			bl.type = PDULIST_NORMAL;
		else if (immediate_data_length && !cmd->unsolicited_data_out)
			bl.type = PDULIST_IMMEDIATE;
		else if (!immediate_data_length && cmd->unsolicited_data_out)
			bl.type = PDULIST_UNSOLICITED;
		else if (immediate_data_length && cmd->unsolicited_data_out)
			bl.type = PDULIST_IMMEDIATE_AND_UNSOLICITED;
	}
		
	return(iscsi_do_build_list(cmd, &bl));
}

/*	iscsi_build_scsi_cmd():
 *
 *	Main function which builds iSCSI SCSI Command PDUs
 *	(containing the SCSI CDB) which does the following:
 *	
 *	1:  Creates inital ISCSI_INIT_SCSI_CMND PDU.
 *	2:  Calls HOST OS dependant iscsi_OS_get_SCSI_cdb() to copy the
 *	    SCSI CDB into the PDU. (Also calls iscsi_OS_get_SCSI_cdb_len()).
 *	3:  Calls HOST OS dependant iscsi_OS_set_SCSI_tag_attr() to set SAM-2
 *	    Task Attribute flag in iscsi_init_scsi_cmnd->flags.
 *	4:  Determines if the iSCSI command will be a WRITE or READ.
 *	    In the WRITE case builds any immediate data if the session
 *	    allows it.  Also creates Implied R2T if the session allows
 *	    Unsolicited Data OUT.
 *	
 *	FIXME: Add Additional Header Segment for CDBs larger than 16 bytes.
 *	FIXME: Add Additional Header Segment for Bidi Commands.
 */	
static inline int iscsi_build_scsi_cmd (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	iscsi_unmap_sg_t *unmap_sg)
{
	int actual_immediate_data_length = 0;
	int negotiated_immediate_data_length = 0;
	u32 iov_count = 0, padding;
	iscsi_r2t_t *r2t = NULL;
	struct iscsi_init_scsi_cmnd *hdr;
	struct scsi_cmnd *sc = cmd->scsi_cmnd;
	
	cmd->conn		= conn;
	cmd->data_length	= sc->request_bufflen;

//#warning strangeness with CRYPTO_API with DataCRC and 92 sectors and 2.6.10-rc3 ppc
#if defined(CRYPTO_API_CRC32C) && defined(ISCSI_BIG_ENDIAN)
	if (cmd->data_length == 49152) {
		printk("iCHANNEL[%d] - 92 sectors (49152 bytes) to ISCSI_BIG_ENDIAN\n",
				SESS(conn)->channel->channel_id);
	}
#endif

	cmd->tx_size		= ISCSI_HDR_LEN;
	cmd->iov_data_count 	= 1;
	cmd->iov_data[0].iov_base	= cmd->pdu;
	cmd->iov_data[0].iov_len	= ISCSI_HDR_LEN;

	hdr			= (struct iscsi_init_scsi_cmnd *) cmd->pdu;
	hdr->opcode 		= ISCSI_INIT_SCSI_CMND;
	hdr->lun 		= iscsi_put_lun(SCSI_LUN(sc));
	spin_lock(&SESS(conn)->itt_lock);
	cmd->init_task_tag	= SESS(conn)->cur_task_tag++;
	spin_unlock(&SESS(conn)->itt_lock);
	hdr->init_task_tag	= cpu_to_be32(cmd->init_task_tag);
	/*
	 * cmd->cmdsn has already been assigned at this point in
	 * iscsi_initiator_tx_thread() while sess->cmdsn_lock is held.
	 */
	hdr->cmd_sn		= cpu_to_be32(cmd->cmdsn);
	hdr->exp_xfer_len	= cpu_to_be32(cmd->data_length);
	hdr->exp_stat_sn	= cpu_to_be32(conn->exp_statsn);

	if (sc->cmd_len > ISCSI_CDB_LEN) {
		TRACE_ERROR("FIXME: Add iSCSI AHS to handle SCSI CDBs"
			" greater than %d.\n", ISCSI_CDB_LEN);
		return(-1);
	}
	memcpy((void *)hdr->cdb, (void *)sc->cmnd, sc->cmd_len);

	if (sc->cmd_len < ISCSI_CDB_LEN)
		memset(&hdr->cdb[sc->cmd_len], 0, (ISCSI_CDB_LEN - sc->cmd_len));

	iscsi_set_SCSI_tag_attr(sc, hdr);
	
	if (sc->sc_data_direction == DMA_TO_DEVICE) {
		int firstburst_left, write_data_to_send;
		struct iovec *iov = &cmd->iov_data[1];

		hdr->flags |= W_BIT;
		if (cmd->data_length && SESS_OPS_C(conn)->ImmediateData) {
			/* 
			 * Immediate data can be sent 
			 */
			negotiated_immediate_data_length = 
				(CONN_OPS(conn)->MaxRecvDataSegmentLength <
				 SESS_OPS_C(conn)->FirstBurstLength) ?
				 CONN_OPS(conn)->MaxRecvDataSegmentLength : 
				 SESS_OPS_C(conn)->FirstBurstLength;

			/* 
			 * This is the actual datalength that has to be sent. 
			 */
			actual_immediate_data_length =
				(cmd->data_length <
				 negotiated_immediate_data_length) ?
				 cmd->data_length :
				 negotiated_immediate_data_length;

			TRACE(TRACE_ISCSI, "Sending unsolicited immediate data"
				" %d bytes on CID: %hu\n",
				actual_immediate_data_length, conn->cid);

			if (sc->use_sg) {
				int iov_ret;
				iscsi_map_sg_t map_sg;

				memset((void *)&map_sg, 0, sizeof(iscsi_map_sg_t));
				map_sg.cmd = cmd;
				map_sg.map_flags |= MAP_SG_KMAP;
				map_sg.iov = &cmd->iov_data[1];
				map_sg.data_length = actual_immediate_data_length;
				map_sg.data_offset = 0;

				if ((iov_ret = iscsi_map_scatterlists(
						(void *)&map_sg,
						(void *)unmap_sg)) < 0)
					return(-1);
				
				iov_count += iov_ret;
			} else {
				iov[iov_count].iov_len =
					actual_immediate_data_length;
				iov[iov_count++].iov_base = sc->request_buffer;
			}

			hdr->length = cpu_to_be32(actual_immediate_data_length);
			cmd->tx_size += actual_immediate_data_length;
			cmd->data_offset += actual_immediate_data_length;

			if ((padding = ((-actual_immediate_data_length) & 3)) != 0) {
				if (!(cmd->buf_ptr = kmalloc(padding, GFP_ATOMIC))) {
					TRACE_ERROR("Unable to allocate memory"
						" for padding.\n");
					goto failure;
				}
				memset(cmd->buf_ptr, 0, padding);
				TRACE(TRACE_ISCSI, "Adding %u padding bytes to"
					" Immediate Data.\n", padding);
				iov[iov_count].iov_base	= cmd->buf_ptr;
				iov[iov_count++].iov_len = padding;
				cmd->tx_size += padding;
			}

			if (CONN_OPS(conn)->DataDigest) {
#ifndef CRYPTO_API_CRC32C
				int reset_crc = 1;
#endif
				u32 counter = actual_immediate_data_length;
				struct iovec *iov_ptr = &cmd->iov_data[1];
#ifdef CRYPTO_API_CRC32C
				crypto_digest_init(conn->conn_tx_tfm);
#endif				
				while (counter > 0) {
#ifdef CRYPTO_API_CRC32C
					ISCSI_CRC32C_TX_DATA_TX_PATH(iov_ptr->iov_base,
						iov_ptr->iov_len, conn);
#else
					crc32c(iov_ptr->iov_base, iov_ptr->iov_len,
						  reset_crc, &cmd->data_checksum);
					reset_crc = 0;
#endif
					TRACE(TRACE_DIGEST, "Computed CRC32 %d bytes,"
						"CRC 0x%08x\n", iov_ptr->iov_len,
							cmd->data_checksum);
					counter -= iov_ptr->iov_len;
					iov_ptr++;
				}

				if (padding) {
#ifdef CRYPTO_API_CRC32C
					ISCSI_CRC32C_TX_DATA_TX_PATH(cmd->buf_ptr, padding, conn);
#else
					crc32c((u8 *)cmd->buf_ptr, padding,
							reset_crc, &cmd->data_checksum);
					reset_crc = 0;
#endif
					TRACE(TRACE_DIGEST, "Computed CRC32C"
					" DataDigest %d bytes of padding,"
					" CRC 0x%08x\n", padding, cmd->data_checksum);
				}				
#ifdef CRYPTO_API_CRC32C
				ISCSI_CRC32C_TX_DATA_TX_PATH_FINAL(cmd->data_checksum, conn);
#endif
				iov[iov_count].iov_base	= &cmd->data_checksum;
				iov[iov_count++].iov_len = CRC_LEN;
				cmd->tx_size += CRC_LEN;
	
				TRACE(TRACE_DIGEST, "Attaching CRC32C DataDigest"
					" for ImmediateData %u bytes, CRC 0x%08x\n",
					actual_immediate_data_length, cmd->data_checksum);
			}
			cmd->iov_data_count += iov_count;
		}
		
		/* 
		 * Determine if any Write Data is left,
		 * and FirstBurstLength space available.
		 */
		write_data_to_send = (cmd->data_length -
				actual_immediate_data_length);
		firstburst_left =
			(SESS_OPS_C(conn)->FirstBurstLength -
			 	actual_immediate_data_length);

		if (SESS_OPS_C(conn)->InitialR2T ||
		   (write_data_to_send <= 0) || (firstburst_left <= 0)) {	
			/* 
			 * Set the F_BIT as no seperate unsolicated data
			 * PDUs are required 
			 */
			hdr->flags |= F_BIT;
		} else if (!(SESS_OPS_C(conn)->InitialR2T)) {
			/* 
			 * We are allowed to send unsolicited data PDUs
			 * after this command, is there any space left?
			 */
			if ((write_data_to_send > 0) && (firstburst_left > 0)) {
				/* 
				 * Still data to send and there is data left
				 * in FirstBurst, setup to send unsolicted
				 * data PDUs.
				 */
				if (!(r2t = iscsi_allocate_r2t()))
					goto failure;
				
				r2t->offset = cmd->data_offset;
				r2t->r2t_sn = r2t->targ_xfer_tag = 0xFFFFFFFF;
				r2t->xfer_length = (write_data_to_send >
					firstburst_left) ? firstburst_left :
						write_data_to_send;

				if (SESS_OPS_C(conn)->DataSequenceInOrder)
					cmd->data_offset += r2t->xfer_length;

				cmd->unsolicited_data_out = 1;
			}
		}
	} else {
		/* 
		 * Data is a READ, but double check to make sure.
		 */
		if (sc->sc_data_direction == DMA_FROM_DEVICE) {
			hdr->flags |= (R_BIT | F_BIT);

			if (SESS_OPS_C(conn)->DataSequenceInOrder)
				iscsi_set_datain_sequence_values(cmd);
		} else {
			/* 
			 * Data is NOT a READ or WRITE 
			 */
			if (cmd->data_length != 0) {
				TRACE_ERROR("Request_bufflen %u not zero when"
					" Data Direction %d not READ or WRITE\n",
						cmd->data_length,
					sc->sc_data_direction);
				BUG();
			} else
				hdr->flags |= F_BIT;
		}
	}

	if (CONN_OPS(conn)->HeaderDigest) {
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_TX_HDR_TX_PATH(hdr, hdr->header_digest, conn);
#else
		crc32c((u8 *)hdr, ISCSI_HDR_LEN, 1, &hdr->header_digest);
#endif
		cmd->iov_data[0].iov_len += CRC_LEN;
		cmd->tx_size += CRC_LEN;
		TRACE(TRACE_DIGEST, "Attaching CRC32C HeaderDigest 0x%08x\n",
				hdr->header_digest);
	}
	
	TRACE(TRACE_ISCSI, "Built SCSI Cmnd: ITT: 0x%08x, CmdSN: 0x%08x,"
		" ExpXferLen: %u, Length: %u, CID: %hu\n", cmd->init_task_tag,
			cmd->cmdsn, ntohl(hdr->exp_xfer_len),
			actual_immediate_data_length, conn->cid);

	if (iscsi_decide_list_to_build(cmd, actual_immediate_data_length) < 0)
		goto failure;

	if (cmd->unsolicited_data_out) {
		if (!SESS_OPS_C(conn)->DataSequenceInOrder) {
			iscsi_seq_t *seq;
			if (!(seq = iscsi_get_seq_holder(cmd, r2t->offset,
					r2t->xfer_length)))
				goto failure;
			r2t->seq = seq;
		}
		if (!SESS_OPS_C(conn)->DataPDUInOrder)
			if (iscsi_set_pdu_values_for_r2t(cmd, r2t) < 0)
				goto failure;

		iscsi_add_r2t_to_cmd(cmd, r2t);
	}

	if (sc->sc_data_direction == DMA_FROM_DEVICE) {
		spin_lock_bh(&cmd->datain_timeout_lock);
		iscsi_start_datain_timer(cmd, conn);
		spin_unlock_bh(&cmd->datain_timeout_lock);
	}
	
	return(0);
failure:
	if (actual_immediate_data_length)
		iscsi_unmap_scatterlists((void *)&unmap_sg);
	if (r2t)
		kfree(r2t);
	return(-1);
}

/*	iscsi_build_dataout():
 *
 *	Parameters: 	iSCSI Connection Pointer, iSCSI Command Pointer.
 *	Returns:	0 on success, -1 on error. 	
 */
static inline int iscsi_build_dataout (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	iscsi_unmap_sg_t *unmap_sg,
	int *end_of_sequence)
{
	u8 iov_count = 0, final_bit = 0, sg_mapped = 0;
	u32 data_length, data_sn, offset, padding = 0, tx_size;
	iscsi_r2t_t *r2t;
	iscsi_pdu_t *pdu;
	struct iovec *iov;
	struct iscsi_init_scsi_data_out *hdr;
	struct scsi_cmnd *sc = cmd->scsi_cmnd;
	
	if (!(r2t = iscsi_get_r2t_for_cmd(cmd)))
		return(-1);

	if (SESS_OPS_C(conn)->DataPDUInOrder) {
		if (r2t->xfer_length <=
		    CONN_OPS(conn)->MaxRecvDataSegmentLength)
			final_bit = 1;

		data_sn = r2t->data_sn++;
		offset = r2t->offset;
		data_length = (final_bit) ? r2t->xfer_length :
			CONN_OPS(conn)->MaxRecvDataSegmentLength;
	} else {
		if (!(pdu = iscsi_get_pdu_holder_from_r2t(cmd, r2t)))
			return(-1);
		
		if (--r2t->pdu_send_count == 0)
			final_bit = 1;
		
		data_sn = pdu->data_sn = r2t->data_sn++;
		offset = pdu->offset;
		data_length = pdu->length;
	}

        r2t->offset += data_length;
	r2t->xfer_length -= data_length;
	
	hdr			= (struct iscsi_init_scsi_data_out *) cmd->data_pdu;
	memset(hdr, 0, ISCSI_HDR_LEN);

	hdr->opcode		= ISCSI_INIT_SCSI_DATA_OUT;
	hdr->flags		= (final_bit) ? F_BIT : 0;
	if (r2t->targ_xfer_tag != 0xFFFFFFFF)
		hdr->lun	= iscsi_put_lun(SCSI_LUN(sc));
	hdr->init_task_tag	= cpu_to_be32(cmd->init_task_tag);
	hdr->targ_xfer_tag	= cpu_to_be32(r2t->targ_xfer_tag);
	hdr->offset		= cpu_to_be32(offset);
	hdr->exp_stat_sn	= cpu_to_be32(conn->exp_statsn);
	hdr->data_sn		= cpu_to_be32(data_sn);
	hdr->length		= cpu_to_be32(data_length);

	iov			= &cmd->iov_data[0];
	iov[iov_count].iov_base	= cmd->data_pdu;
	iov[iov_count].iov_len	= ISCSI_HDR_LEN;
	tx_size = (ISCSI_HDR_LEN + data_length);
	
	if (CONN_OPS(conn)->HeaderDigest) {
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_TX_HDR_TX_PATH(cmd->data_pdu, hdr->header_digest, conn);
#else
		crc32c((u8 *)cmd->data_pdu, ISCSI_HDR_LEN, 1, &hdr->header_digest);
#endif
		iov[iov_count].iov_len += CRC_LEN;
		tx_size += CRC_LEN;
		TRACE(TRACE_DIGEST, "Attaching HeaderDigest 0x%08x\n",
				hdr->header_digest);
	}	
	iov_count++;

	if (sc->use_sg) {
		int iov_ret;
		iscsi_map_sg_t map_sg;

		memset((void *)&map_sg, 0, sizeof(iscsi_map_sg_t));
		map_sg.cmd = cmd;
		map_sg.map_flags |= MAP_SG_KMAP;
		map_sg.iov = &cmd->iov_data[iov_count];
		map_sg.data_length = data_length;
		map_sg.data_offset = offset;
		
		if ((iov_ret = iscsi_map_scatterlists((void *)&map_sg,
					(void *)unmap_sg)) < 0)
			return(-1);
		
		sg_mapped = 1;
		iov_count += iov_ret;	
	} else {
		iov[iov_count].iov_len = data_length;
		iov[iov_count++].iov_base = sc->request_buffer + offset;
	}

	if ((padding = ((-data_length) & 3)) != 0) {
		if (!(cmd->buf_ptr = kmalloc(padding, GFP_ATOMIC))) {
			TRACE_ERROR("Unable to allocate memory for padding.\n");
			goto failure;
		}

		memset(cmd->buf_ptr, 0, padding);
		TRACE(TRACE_ISCSI, "Attaching %u padding bytes to"
			" DataOUT.\n", padding);
		iov[iov_count].iov_base = cmd->buf_ptr;
		iov[iov_count++].iov_len = padding;
		tx_size += padding;
	}

	if (CONN_OPS(conn)->DataDigest) {
#ifndef CRYPTO_API_CRC32C
		int reset_crc = 1;
#endif
		u32 counter = data_length;
		struct iovec *iov_ptr = &cmd->iov_data[1];
#ifdef CRYPTO_API_CRC32C
		crypto_digest_init(conn->conn_tx_tfm);
#endif	
		while (counter > 0) {
#ifdef CRYPTO_API_CRC32C
			ISCSI_CRC32C_TX_DATA_TX_PATH(iov_ptr->iov_base,
				iov_ptr->iov_len, conn);
#else
			crc32c(iov_ptr->iov_base, iov_ptr->iov_len,
				reset_crc, &cmd->data_checksum);
			reset_crc = 0;
#endif
			TRACE(TRACE_DIGEST, "Computed CRC32C DataDigest %d"
				" bytes, CRC 0x%08x\n", iov_ptr->iov_len,
					cmd->data_checksum);
			counter -= iov_ptr->iov_len;
			iov_ptr++;
		}

		if (padding) {
#ifdef CRYPTO_API_CRC32C
			ISCSI_CRC32C_TX_DATA_TX_PATH(cmd->buf_ptr, padding, conn);
#else
			crc32c((u8 *)cmd->buf_ptr, padding,
				reset_crc, &cmd->data_checksum);
#endif
			TRACE(TRACE_DIGEST, "Computed CRC32C DataDigest %d"
				" bytes of padding, CRC 0x%08x\n", padding,
					cmd->data_checksum);
		}
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_TX_DATA_TX_PATH_FINAL(cmd->data_checksum, conn);
#endif
		iov[iov_count].iov_base = &cmd->data_checksum; 
		iov[iov_count++].iov_len = CRC_LEN;
		tx_size	+= CRC_LEN;
	
		TRACE(TRACE_DIGEST, "Attaching CRC32C DataDigest of DataOut"
			" %u bytes, CRC 0x%08x\n", data_length,
				cmd->data_checksum);
	}

	cmd->tx_size = tx_size;
	cmd->iov_data_count = iov_count;

	if (final_bit) {
		if (cmd->unsolicited_data_out) 
			cmd->unsolicited_data_out = 0;
		else {
			spin_lock(&cmd->r2t_lock);
			cmd->outstanding_r2ts--;
			spin_unlock(&cmd->r2t_lock);
		}

		*end_of_sequence = 1;
		iscsi_free_r2t(cmd, r2t);
	} else
		*end_of_sequence = 0;
	
	TRACE(TRACE_ISCSI, "Built DataOUT ITT: 0x%08x TTT: 0x%08x DataSN:"
		" 0x%08x Offset: %u Length: %u CID: %hu\n", cmd->init_task_tag,
		r2t->targ_xfer_tag, data_sn, offset, data_length, conn->cid);
	
	return(0);

failure:
	if (sg_mapped)
		iscsi_unmap_scatterlists((void *)&unmap_sg);
	return(-1);
} 

/*	iscsi_build_logout_req():
 *
 *
 */
static int iscsi_build_logout_req (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	int immediate)
{
	u8 logout_reason;
	u16 logout_cid;
	u32 exp_statsn;
	int ret = 0;
	iscsi_cmd_t *cmd_p = NULL;
	iscsi_logout_attempt_t *la;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_init_logout_cmnd *hdr;
	
	if (!cmd->la) {
		TRACE_ERROR("iscsi_cmd_t->la is NULL!\n");
		return(-1);
	}
	la = cmd->la;
	
	logout_reason = la->logout_reason;

	if (!immediate) {
		la->logout_on_cid = conn->cid;
		la->logout_on_conn = conn;

		if (logout_reason == CLOSESESSION) {
			logout_cid = conn->cid;
			la->logout_conn = conn;
			la->logout_cid = conn->cid;
			iscsi_inc_conn_usage_count(conn);
			iscsi_set_state_for_connections(sess,
				INIT_CONN_STATE_IN_LOGOUT);
			ret = 1;
		} else {
			logout_cid = la->logout_cid;

			if (logout_cid == conn->cid) {
				TRACE(TRACE_STATE, "Moving to"
					" INIT_CONN_STATE_IN_LOGOUT.\n");
				spin_lock_bh(&conn->state_lock);
				conn->conn_state = INIT_CONN_STATE_IN_LOGOUT;
				spin_unlock_bh(&conn->state_lock);
				ret = 1;
			} else {
				iscsi_conn_t *conn_p = la->logout_conn;
				iscsi_conn_t *on_conn_p = la->logout_on_conn;

				if ((conn->cid != on_conn_p->cid) &&
				    (conn->cid == conn_p->cid)) {
					TRACE(TRACE_STATE, "Moving to"
						" INIT_CONN_STATE_IN_LOGOUT.\n");
					spin_lock_bh(&conn_p->state_lock);
					conn_p->conn_state = INIT_CONN_STATE_IN_LOGOUT;
					spin_unlock_bh(&conn_p->state_lock);

					cmd->la = NULL;

					if (!(cmd_p = iscsi_allocate_cmd(sess, NULL)))
						return(-1);

					cmd_p->la = la;
					cmd_p->pdu[0] = ISCSI_INIT_LOGOUT_CMND | I_BIT;
					cmd_p->state = ISTATE_SEND_IMMEDIATE_LOGOUT_REQ;
					iscsi_attach_cmd_to_conn(cmd_p, on_conn_p);
				
					iscsi_add_cmd_to_immediate_queue(cmd_p, on_conn_p,
							cmd_p->state);
					return(3);
				} else
					iscsi_inc_conn_usage_count(conn);
			}
		}

		spin_lock(&la->logout_state_lock);
		la->logout_built = 1;
		spin_unlock(&la->logout_state_lock);
	} else {
		iscsi_conn_t *conn_p = la->logout_conn;
		iscsi_conn_t *on_conn_p = la->logout_on_conn;
			
		logout_cid = la->logout_cid;
		
		if ((logout_reason == CLOSESESSION) ||
		   ((logout_reason == CLOSECONNECTION) &&
		    (conn->cid == conn_p->cid) && (conn->cid == on_conn_p->cid))) {
			ret = 1;
		}

		if ((logout_reason == CLOSECONNECTION) &&
		    (conn->cid != on_conn_p->cid) && (conn->cid == conn_p->cid)) {
			cmd->la = NULL;

//#warning FIXME: See if doing the TCP half shutdown works OK.
			conn_p->sock->ops->shutdown(conn_p->sock, SEND_SHUTDOWN);

			if (!(cmd_p = iscsi_allocate_cmd(sess, NULL)))
				return(-1);

			cmd_p->la = la; 
			cmd_p->pdu[0] = ISCSI_INIT_LOGOUT_CMND | I_BIT;
			cmd_p->state = ISTATE_SEND_IMMEDIATE_LOGOUT_REQ;
			iscsi_attach_cmd_to_conn(cmd_p, on_conn_p);

			iscsi_add_cmd_to_immediate_queue(cmd_p, on_conn_p, cmd_p->state);
			return(3);
		}
	}
	
	exp_statsn = conn->exp_statsn;
	spin_lock(&SESS(conn)->itt_lock);
	cmd->init_task_tag	= sess->cur_task_tag++;
	spin_unlock(&SESS(conn)->itt_lock);
	if (immediate)
		cmd->cmdsn	= sess->cur_cmdsn;
	hdr			= (struct iscsi_init_logout_cmnd *) cmd->pdu;
	hdr->opcode		= ISCSI_INIT_LOGOUT_CMND;
	if (immediate)	
		hdr->opcode	|= I_BIT;
	hdr->flags		= (F_BIT | logout_reason);
	hdr->length		= cpu_to_be32(0);
	hdr->init_task_tag	= cpu_to_be32(cmd->init_task_tag);
	hdr->cid		= cpu_to_be16(logout_cid);
	hdr->cmd_sn		= cpu_to_be32(cmd->cmdsn);
	hdr->exp_stat_sn	= cpu_to_be32(exp_statsn);

	cmd->iov_misc[0].iov_base = cmd->pdu;
	cmd->iov_misc[0].iov_len = ISCSI_HDR_LEN;
	cmd->iov_misc_count = 1;
	cmd->tx_size = ISCSI_HDR_LEN;
	
	if (CONN_OPS(conn)->HeaderDigest) {
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_TX_HDR_TX_PATH(hdr, hdr->header_digest, conn);
#else
		crc32c((u8 *)hdr, ISCSI_HDR_LEN, 1, &hdr->header_digest);
#endif
		cmd->iov_misc[0].iov_len += CRC_LEN;
		cmd->tx_size += CRC_LEN;
		TRACE(TRACE_DIGEST, "Attaching CRC32C HeaderDigest 0x%08x\n",
				hdr->header_digest);
	}

	TRACE(TRACE_ISCSI, "Built Logout Request: ITT: 0x%08x CmdSN: 0x%08x"
		" ExpStatSN: 0x%08x Reason: 0x%02x CID: %hu on CID: %hu\n",
		cmd->init_task_tag, cmd->cmdsn, exp_statsn, hdr->flags & 0x7f,
			ntohs(hdr->cid), conn->cid);

	return(ret);
}

/*	iscsi_build_nopout():
 *
 *
 */
static int iscsi_build_nopout (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	int immediate)
{
	u32 niov = 0, padding, ping_data_size = 0, tx_size = 0;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_init_nop_out *hdr;

	spin_lock(&SESS(conn)->itt_lock);
	cmd->init_task_tag 	= sess->cur_task_tag++;
	spin_unlock(&SESS(conn)->itt_lock);
	if (immediate)
		cmd->cmdsn	= sess->cur_cmdsn;
	hdr = (struct iscsi_init_nop_out *) cmd->pdu;
	hdr->opcode		= ISCSI_INIT_NOP_OUT;
	if (immediate)
		hdr->opcode	|= I_BIT;
	hdr->flags		|= F_BIT;
	hdr->lun		= cpu_to_be64(0xFFFFFFFFFFFFFFFFULL);
	hdr->init_task_tag	= cpu_to_be32(cmd->init_task_tag);
	hdr->targ_xfer_tag	= cpu_to_be32(0xFFFFFFFF);
	hdr->cmd_sn		= cpu_to_be32(cmd->cmdsn);
	hdr->exp_stat_sn	= cpu_to_be32(conn->exp_statsn);

	if (!(cmd->buf_ptr = (unsigned char *)kmalloc(
			ISCSI_MAX_PING_DATA_BYTES, GFP_ATOMIC))) {
		TRACE_ERROR("Unable to allocate memory for ping data.\n");
		return(-1);
	}
	memset((void *) cmd->buf_ptr, 0, ISCSI_MAX_PING_DATA_BYTES);

	ping_data_size = sprintf(cmd->buf_ptr, "NopOut Ping: %d",
			conn->ping_num++);
	padding = (-ping_data_size) & 3;

	cmd->buf_ptr_size	= ping_data_size + padding;
	hdr->length		= cpu_to_be32(ping_data_size);

	if (CONN_OPS(conn)->HeaderDigest) {
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_TX_HDR_TX_PATH(hdr, hdr->header_digest, conn);
#else
		crc32c((u8 *)hdr, ISCSI_HDR_LEN, 1, &hdr->header_digest);
#endif
		cmd->iov_misc[0].iov_len += CRC_LEN;
		cmd->tx_size += CRC_LEN;
		TRACE(TRACE_DIGEST, "Attaching CRC32C HeaderDigest 0x%08x\n",
			hdr->header_digest);
	}	

	TRACE(TRACE_ISCSI, "Attaching %d bytes of NOPOUT ping"
			" data.\n", ping_data_size);
	if (padding) {
		TRACE(TRACE_ISCSI, "Adding %u padding bytes for"
			" ping data.\n", padding);
	}
			
	cmd->iov_misc[1].iov_base	= cmd->buf_ptr;
	cmd->iov_misc[1].iov_len	= (ping_data_size + padding);
	tx_size = (ping_data_size + padding);
	niov = 1;

	if (CONN_OPS(conn)->DataDigest) {
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_TX_DATA_TX_PATH_NON_SG(cmd->buf_ptr,
			ping_data_size + padding, cmd->data_checksum, conn); 
#else
		crc32c((u8 *) cmd->buf_ptr,
			(ping_data_size + padding), 0x01, &cmd->data_checksum);
#endif
		cmd->iov_misc[2].iov_base	= &cmd->data_checksum;
		cmd->iov_misc[2].iov_len	= CRC_LEN;
		tx_size += CRC_LEN;
		niov++;
		TRACE(TRACE_DIGEST, "Attaching DataDigest for %u bytes"
			" of ping data, CRC 0x%08x\n",
			(ping_data_size + padding), cmd->data_checksum);
	}

	cmd->tx_size += tx_size;
	cmd->iov_misc_count += niov;

	TRACE(TRACE_ISCSI, "Built NOPOUT Ping ITT: 0x%08x, CmdSN: 0x%08x,"
		" ExpStatSN: 0x%08x, Length: %u, CID: %hu\n", cmd->init_task_tag,
		cmd->cmdsn, ntohl(hdr->exp_stat_sn), ping_data_size, conn->cid);

	return(0);
}

/*	iscsi_build_text_req():
 *
 *
 */
extern int iscsi_build_text_req (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	int immediate)
{
	unsigned char *text_buf = (unsigned char *) cmd->buf_ptr;
	u32 text_len = cmd->buf_ptr_size;
	struct iscsi_init_text_cmnd *hdr;

	if (text_len && !text_buf) {
		TRACE_ERROR("Sending Text Request ITT: 0x%08x with Payload"
			" Length: %u, but Payload Buffer is NULL.\n",
			cmd->init_task_tag, text_len);
		return(-1);
	}
		
	spin_lock(&SESS(conn)->itt_lock);
	cmd->init_task_tag	= SESS(conn)->cur_task_tag++;
	spin_unlock(&SESS(conn)->itt_lock);
	if (immediate)
		cmd->cmdsn	= SESS(conn)->cur_cmdsn;
	hdr			= (struct iscsi_init_text_cmnd *) cmd->pdu;
	memset(hdr, 0, ISCSI_HDR_LEN);
	hdr->opcode		= ISCSI_INIT_TEXT_CMND;
	if (immediate)
		hdr->opcode	|= I_BIT;
	hdr->flags		= F_BIT;
	hdr->length		= cpu_to_be32(text_len);
	hdr->init_task_tag	= cpu_to_be32(cmd->init_task_tag);
	hdr->targ_xfer_tag	= cpu_to_be32(0xFFFFFFFF);
	hdr->cmd_sn		= cpu_to_be32(cmd->cmdsn);
	hdr->exp_stat_sn	= cpu_to_be32(conn->exp_statsn);

	if (CONN_OPS(conn)->HeaderDigest) {
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_TX_HDR_TX_PATH(hdr, hdr->header_digest, conn);
#else
		crc32c((u8 *)hdr, ISCSI_HDR_LEN, 1, &hdr->header_digest);
#endif
		cmd->iov_misc[0].iov_len += CRC_LEN;
		cmd->tx_size += CRC_LEN;
		TRACE(TRACE_DIGEST, "Attaching CRC32C HeaderDigest 0x%08x\n",
				hdr->header_digest);
	}
	
	if (text_len) {
		cmd->iov_misc[1].iov_base	= text_buf;
		cmd->iov_misc[1].iov_len	= text_len;
		cmd->tx_size			+= text_len;
		cmd->iov_misc_count++;

		if (CONN_OPS(conn)->DataDigest) {
#ifdef CRYPTO_API_CRC32C			
			ISCSI_CRC32C_TX_DATA_TX_PATH_NON_SG(text_buf, text_len,
					cmd->data_checksum, conn);
#else
			crc32c((u8 *) text_buf, text_len, 0x01,
					&cmd->data_checksum);
#endif
			cmd->iov_misc[2].iov_base	= &cmd->data_checksum;
			cmd->iov_misc[2].iov_len	= CRC_LEN;
			cmd->tx_size		+= CRC_LEN;
			cmd->iov_misc_count++;
			TRACE(TRACE_DIGEST, "Attaching DataDigest for %u"
				" bytes of text data, CRC 0x%08x\n",
				text_len, cmd->data_checksum);
		}
	}

	TRACE(TRACE_ISCSI, "Built Text Request: ITT: 0x%08x, CmdSN: 0x%08x,"
		" ExpStatSN: 0x%08x, Length: %u, on CID: %hu\n",
		cmd->init_task_tag, cmd->cmdsn, ntohl(hdr->exp_stat_sn),
			text_len, conn->cid);

	return(0);
}

/*	iscsi_build_tmr():
 *
 *	
 */
static int iscsi_build_tmr (
	iscsi_cmd_t *cmd,
	iscsi_conn_t *conn,
	int immediate)
{
	switch (cmd->task_mgt_function) {
	case ABORT_TASK:
	case ABORT_TASK_SET:
	case CLEAR_ACA:
	case CLEAR_TASK_SET:
	case LUN_RESET:
	case TARGET_WARM_RESET:
	case TARGET_COLD_RESET:
		break;
	case TASK_REASSIGN:		
		break;
	default:
		TRACE_ERROR("Unknown TMR Function: 0x%02x\n",
			cmd->task_mgt_function);
		return(-1);
	}
		
	return(0);
}

/*	iscsi_handle_data_in():
 * 
 *	Called From:	iscsi_initiator_rx_thread().
 *	Parameters:	iSCSI Connection pointer,
 *			Pointer to buffer containing PDU.
 *	Returns:	0 on success, -1 on error.
 */
static inline int iscsi_handle_data_in (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	unsigned char host_stat = DID_OK;
	int resid = 0, ret = 0, rx_size;
	u8 data_crc_failed = 0;
	u32 checksum = 0, iov_count = 0, padding = 0, pad_bytes = 0;
	unsigned int lun;
	iscsi_cmd_t *cmd;
	iscsi_unmap_sg_t unmap_sg;
	iscsi_session_t *sess = SESS(conn);
	struct iovec *iov;
	struct iscsi_targ_scsi_data_in *hdr;
	struct scsi_cmnd *sc;

	hdr			= (struct iscsi_targ_scsi_data_in *) buf;
	hdr->length		= be32_to_cpu(hdr->length);
	hdr->init_task_tag	= be32_to_cpu(hdr->init_task_tag);
	hdr->targ_xfer_tag	= be32_to_cpu(hdr->targ_xfer_tag);
	hdr->res_count		= be32_to_cpu(hdr->res_count);
	hdr->stat_sn		= be32_to_cpu(hdr->stat_sn);
	hdr->exp_cmd_sn		= be32_to_cpu(hdr->exp_cmd_sn);
	hdr->max_cmd_sn		= be32_to_cpu(hdr->max_cmd_sn);
	hdr->data_sn		= be32_to_cpu(hdr->data_sn);
	hdr->offset		= be32_to_cpu(hdr->offset);

	if (!hdr->length) {
		TRACE_ERROR("DataIN payload is ZERO, protocol error.\n");
		return(-1);
	}

	if (hdr->length > CONN_OPS(conn)->MaxRecvDataSegmentLength) {
		TRACE_ERROR("DataIN payload size %u is greater than"
			" MaxRecvDataSegmentLength %u, protocol error.\n",
		hdr->length, CONN_OPS(conn)->MaxRecvDataSegmentLength);
		return(-1);
	}
	
	if (!(cmd = iscsi_find_cmd_from_itt_or_dump(conn, hdr->init_task_tag,
				hdr->length)))
		return(0);

	cmd->datain_retries = 0;
	cmd->cmd_flags &= ~ICF_DATAIN_TIMEOUT;
	iscsi_mod_datain_timer(cmd, hdr->data_sn);

	TRACE(TRACE_ISCSI, "Got DataIN ITT: 0x%08x, StatSN: 0x%08x, DataSN:"
		" 0x%08x, Offset: %u, Length: %u, CID %hu\n",
		hdr->init_task_tag, hdr->stat_sn, hdr->data_sn,
			hdr->offset, hdr->length, conn->cid);
	
	if (!cmd->scsi_cmnd) { 	
		TRACE_ERROR("OS Dependant SCSI pointer is NULL!\n");
		return(-1); 
	}
	sc = cmd->scsi_cmnd;

	if (cmd->cmd_flags & ICF_GOT_LAST_DATAIN) {
		TRACE_ERROR("Command ITT: 0x%08x received DataIN after last"
			" DataIN received, dumping payload\n",
				cmd->init_task_tag);
		return(iscsi_dump_data_payload(conn, hdr->length, 1));
	}

	if (sc->sc_data_direction != DMA_FROM_DEVICE) {
		TRACE_ERROR("Command ITT: 0x%08x received DataIN for"
			" non READ command.\n", cmd->init_task_tag);
		return(-1);
	}

	/*
	 * When the acknowledge bit is set, the DataIN must contain a valid
	 * TTT and iSCSI LUN.  iSCSI v20 10.7.4
	 */
	if (hdr->flags & A_BIT) {
		if (hdr->targ_xfer_tag == 0xFFFFFFFF) {
			TRACE_ERROR("Acknowledge Bit set but TTT is reserved."
				"  Protocol error.\n");
			return(-1);
		}
		sess->targ_xfer_tag = hdr->targ_xfer_tag;
		lun = iscsi_get_lun((unsigned char *)&hdr->lun);
		if (lun != SCSI_LUN(sc)) {
			TRACE_ERROR("Command ITT: 0x%08x, LUN 0x%08x does not"
				" match LUN 0x%08x from OS Dependant SCSI"
				" pointer.\n", cmd->init_task_tag, lun, SCSI_LUN(sc));
			return(-1);
		}
	}

	if ((hdr->offset + hdr->length) > cmd->data_length) {
		TRACE_ERROR("Command ITT: 0x%08x, received Offset: %u and Length %u,"
		" exceeds OS SCSI request buffer of %u.\n", cmd->init_task_tag,
			hdr->offset, hdr->length, cmd->data_length);
		return(-1);
	}
			
	if (hdr->flags & S_BIT) {
		if (hdr->status != GOOD) {
			TRACE_ERROR("S_BIT MUST only be when with a"
				" non-expection status code, protocol error\n");
			return(-1);
		}
		if (!(hdr->flags & F_BIT)) {
			TRACE_ERROR("F_BIT MUST be set if S_BIT is set,"
				" protocol error\n"); 
			return(-1);
		}

		if (((cmd->read_data_done + hdr->length) < sc->request_bufflen) &&
		      !(hdr->flags & U_BIT)) {
			TRACE_ERROR("U_BIT should have been set as expecting"
				" more data for ITT: 0x%08x\n", cmd->init_task_tag);
			return(-1);
		}

		/*
		 * FIXME: Overflow data
		 */
		if (hdr->flags & O_BIT) {
			resid = sc->request_bufflen;
			host_stat = DID_ERROR;
		} else if (hdr->flags & U_BIT) {
			if (((cmd->read_data_done + hdr->length) + hdr->res_count) !=
			      sc->request_bufflen) {
				TRACE_ERROR("Residual Count %d does not match"
				" expected residual data transfer length %d\n",
					hdr->res_count, sc->request_bufflen -
					(cmd->read_data_done + hdr->length));
			}
			resid = hdr->res_count;
		}
	}

	iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn, 0);
	
	/*
	 * Preform DataSN, DataSequenceInOrder, DataPDUInOrder, and
	 * within-command recovery checks before receiving the payload.
	 */
	ret = iscsi_check_pre_datain(cmd, buf);
	if (ret == DATAIN_CANNOT_RECOVER)
		return(-1);
	
	rx_size = hdr->length;
	iov = &cmd->iov_data[0];

	if (sc->use_sg) {
		int iov_ret;
		iscsi_map_sg_t map_sg;

		memset((void *)&map_sg, 0, sizeof(iscsi_map_sg_t));
		memset((void *)&unmap_sg, 0, sizeof(iscsi_unmap_sg_t));
		map_sg.cmd = cmd;
		map_sg.iov = iov;
		map_sg.map_flags |= MAP_SG_KMAP;
		map_sg.data_length = hdr->length;
		map_sg.data_offset = hdr->offset;
		unmap_sg.cmd = cmd;
		
		if ((iov_ret = iscsi_map_scatterlists((void *)&map_sg,
				(void *)&unmap_sg)) < 0)
			return(-1);

		iov_count += iov_ret;	

		if ((padding = ((-hdr->length) & 3)) != 0) {
			TRACE(TRACE_ISCSI, "Adding %u padding bytes"
				" for DataIN.\n", padding);
			iov[iov_count].iov_len	= padding;
			iov[iov_count++].iov_base = &pad_bytes;
			rx_size += padding;
		}
		
		if (CONN_OPS(conn)->DataDigest) {
			iov[iov_count].iov_len	= CRC_LEN;
			iov[iov_count++].iov_base = &checksum;
			rx_size += CRC_LEN;
		}

		ret = rx_data(conn, cmd->iov_data, iov_count, rx_size);

		/*
		 * If we do not need to get the pages kmapped
		 * (ie: DataDigests do not need to be calcuated) we
		 * can go ahead and unmap them now.
		 */
		if (!CONN_OPS(conn)->DataDigest)
			iscsi_unmap_scatterlists((void *)&unmap_sg);
		
		if (ret != rx_size)
			return(-1);

		TRACE(TRACE_SCSI, "Received %d bytes of SG DataIN"
				" payload\n", hdr->length);
	} else {
		unsigned char *cur_ptr = (unsigned char *)sc->request_buffer + hdr->offset;

		iov[iov_count].iov_len	= hdr->length;
		iov[iov_count++].iov_base = cur_ptr;

		if ((padding = ((-hdr->length) & 3)) != 0) {
			TRACE(TRACE_ISCSI, "Adding %u padding bytes for"
				" DataIN.\n", padding);
			iov[iov_count].iov_len  = padding;
			iov[iov_count++].iov_base = &pad_bytes;
			rx_size += padding;
		}

		if (CONN_OPS(conn)->DataDigest) {
			iov[iov_count].iov_len 	= CRC_LEN;
			iov[iov_count++].iov_base = &checksum;
			rx_size += CRC_LEN;
		}

		ret = rx_data(conn, cmd->iov_data, iov_count, rx_size);
		if (ret != rx_size)
			return(-1);

		TRACE(TRACE_SCSI, "Received %d bytes of non-SG DataIN"
			" payload\n", hdr->length);
	}

	if (CONN_OPS(conn)->DataDigest) {
#ifndef CRYPTO_API_CRC32C
		int reset_crc = 1;
#endif
		u32 counter = hdr->length, data_crc = 0;
		struct iovec *iov_ptr = &cmd->iov_data[0];

		if (sc->use_sg) {
			iscsi_map_sg_t map_sg;
			
			memset((void *)&map_sg, 0, sizeof(iscsi_map_sg_t));
			map_sg.cmd = cmd;
			map_sg.iov = iov;
			map_sg.data_length = hdr->length;
			map_sg.data_offset = hdr->offset;
			
			if (iscsi_map_scatterlists((void *)&map_sg,
					(void *)&unmap_sg) < 0)
				return(-1);
		} else {
			iov->iov_base = (unsigned char *)
				sc->request_buffer + hdr->offset;
			iov->iov_len = hdr->length;
		}
#ifdef CRYPTO_API_CRC32C
		crypto_digest_init(conn->conn_rx_tfm);
#endif			
		while (counter > 0) {
#ifdef CRYPTO_API_CRC32C
			ISCSI_CRC32C_RX_DATA_RX_PATH(iov_ptr->iov_base,
				iov_ptr->iov_len, conn);
#else
			crc32c(iov_ptr->iov_base, iov_ptr->iov_len,
				reset_crc, &data_crc);
			reset_crc = 0;	
#endif
			TRACE(TRACE_DIGEST, "Computed CRC32C DataDigest %d"
				" bytes, CRC 0x%08x\n", iov_ptr->iov_len,
					data_crc);
			counter -= iov_ptr->iov_len;
			iov_ptr++;
		}

		if (padding) {
#ifdef CRYPTO_API_CRC32C
			ISCSI_CRC32C_RX_DATA_RX_PATH((u8 *)&pad_bytes, padding, conn);
#else
			crc32c((u8 *)&pad_bytes, padding,
				reset_crc, &data_crc);
			reset_crc = 0;
#endif
			TRACE(TRACE_DIGEST, "Computed CRC32C DataDigest %d"
				" bytes of padding, CRC 0x%08x\n", padding,
					data_crc);
		}
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_RX_DATA_RX_PATH_FINAL(data_crc, conn);
#endif		
		if (checksum != data_crc) {
			TRACE_ERROR("ITT: 0x%08x, Offset: %u, Length: %u, DataSN:"
			" 0x%08x, CRC32C DataDigest 0x%08x does not match computed"
			" 0x%08x\n", hdr->init_task_tag, hdr->offset, hdr->length,
				hdr->data_sn, checksum, data_crc);
			data_crc_failed = 1;
		}

		/*
		 * Unmap any pages that are still kmapped from the first call
		 * to iscsi_map_scatterlists().
		 */
		if (sc->use_sg)
			iscsi_unmap_scatterlists((void *)&unmap_sg);
	}

	/*
	 * Increment post receieve data and CRC values or perform
	 * within-command recovery.
	 */
	ret = iscsi_check_post_datain(cmd, buf, data_crc_failed);
	if (ret == DATAIN_CANNOT_RECOVER)
		return(-1);
	else if (ret == DATAIN_SEND_TO_SCSI) {
		iscsi_stop_datain_timer(cmd);
		cmd->cmd_flags |= ICF_GOT_LAST_DATAIN;
		sc->resid = resid;
		sc->result = HOST_BYTE(host_stat);
		sc->result = STATUS_BYTE(hdr->status);		
		return(iscsi_process_response(conn, cmd, buf));
	}

	return(0);
}

/*	iscsi_send_nopout_response():
 *
 *
 */
static int iscsi_send_nopout_response (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	int tx_size = ISCSI_HDR_LEN;
	iscsi_cmd_t *cmd;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_init_nop_out *hdr = NULL;
	struct iscsi_targ_nop_in *pdu = (struct iscsi_targ_nop_in *) buf;
	
	if (!(cmd = iscsi_allocate_cmd(sess, NULL)))
		return(-1);
		
	hdr = (struct iscsi_init_nop_out *)&cmd->pdu[0];
	memset(hdr, 0, ISCSI_HDR_LEN);
	hdr->opcode		= ISCSI_INIT_NOP_OUT | I_BIT;
	hdr->flags		= F_BIT;
	hdr->lun		= cpu_to_be64(pdu->lun);
	hdr->init_task_tag	= 0xFFFFFFFF;
	hdr->targ_xfer_tag	= cpu_to_be32(pdu->targ_xfer_tag);
	hdr->cmd_sn		= cpu_to_be32(sess->cur_cmdsn);
	hdr->exp_stat_sn	= cpu_to_be32(conn->exp_statsn);

	if (CONN_OPS(conn)->HeaderDigest) {
#ifdef CRYPTO_API_CRC32C
		ISCSI_CRC32C_TX_HDR_RX_PATH(hdr, hdr->header_digest, conn);
#else
		crc32c((unsigned char *)hdr, ISCSI_HDR_LEN, 0x01, &hdr->header_digest);
#endif
		tx_size += CRC_LEN;
		TRACE(TRACE_DIGEST, "Attaching CRC32C HeaderDigest 0x%08x\n",
				hdr->header_digest);
	}

	TRACE(TRACE_ISCSI, "Sending NopOut Response TTT: 0x%08x, ExpStatSN:"
		" 0x%08x, CID: %hu\n", ntohl(hdr->targ_xfer_tag),
			ntohl(hdr->exp_stat_sn), conn->cid);

	cmd->iov_misc_count = 1;
	cmd->iov_misc[0].iov_base = &cmd->pdu[0];
	cmd->iov_misc[0].iov_len = cmd->tx_size = tx_size;

	iscsi_attach_cmd_to_conn(cmd, conn);
	cmd->state = ISTATE_SEND_NOPOUT_RESPONSE;
	iscsi_add_cmd_to_immediate_queue(cmd, conn, cmd->state);
	
	return(0);
}
		
/*	iscsi_handle_nop_in():
 *
 *	Called From:	iscsi_initiator_rx_thread().
 *	Parameters:	iSCSI Connection Pointer,
 *			Pointer to buffer of received opcode.
 *	Returns:	0 on success, -1 on failure.
 */
static inline int iscsi_handle_nop_in (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	unsigned char *ping_data = NULL;
	iscsi_cmd_t *cmd = NULL;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_targ_nop_in *hdr;

	hdr			= (struct iscsi_targ_nop_in *) buf;
	hdr->length		= be32_to_cpu(hdr->length);
	hdr->lun		= be64_to_cpu(hdr->lun);
	hdr->init_task_tag	= be32_to_cpu(hdr->init_task_tag);
	hdr->targ_xfer_tag	= be32_to_cpu(hdr->targ_xfer_tag);
	hdr->stat_sn		= be32_to_cpu(hdr->stat_sn);
	hdr->exp_cmd_sn		= be32_to_cpu(hdr->exp_cmd_sn);
	hdr->max_cmd_sn		= be32_to_cpu(hdr->max_cmd_sn);

	/*
	 * A Nop-In returing a Nop-Out ping. 
	 */
	if (hdr->init_task_tag != 0xFFFFFFFF) {
		if (!(cmd = iscsi_find_cmd_from_itt_or_dump(conn,
				hdr->init_task_tag, hdr->length)))
			return(0);

		TRACE(TRACE_ISCSI, "Received NOPIN Ping Reply CID: %d, ITT:"
			" 0x%08x, StatSN: 0x%08x, Length: %u\n", conn->cid,
				hdr->init_task_tag, hdr->stat_sn, hdr->length);

		iscsi_stop_nopout_response_timer(conn);

		if (hdr->length > CONN_OPS(conn)->MaxRecvDataSegmentLength) {
			TRACE_ERROR("Ping data payload size %u is greater"
				" than MaxRecvDataSegmentLength %u,"
				" protocol error.\n", hdr->length,
				CONN_OPS(conn)->MaxRecvDataSegmentLength);
			return(-1);
		}
		
		iscsi_start_nopout_timer(conn);
		
		if (hdr->length) {
			if (!(ping_data = iscsi_rx_data_payload(conn, hdr->length)))
				return(0);
			
			TRACE(TRACE_ISCSI, "Received %u bytes of echoed"
					" ping data.\n", hdr->length);

			if (!cmd->buf_ptr) {
				kfree(ping_data);
				TRACE_ERROR("Hmm, original ping data"
					" cmd->buf_ptr is NULL.\n");
				return(0);
			}
			if (memcmp((void *) ping_data,
					(void *) cmd->buf_ptr, hdr->length) != 0) {
				TRACE_ERROR("Received ping data %s does not"
					" equal sent ping data %s.\n", ping_data,
						(unsigned char *) cmd->buf_ptr);
			} else {
				TRACE(TRACE_ISCSI, "Received correct echo"
					" for ping data.\n");
				TRACE(TRACE_ISCSI, "Ping Data: \"%s\"\n", ping_data);
			}
			kfree(ping_data);
		}

		iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn,
				(cmd->pdu[0] & I_BIT));
		if (iscsi_process_response(conn, cmd, buf) < 0)
			return(-1);
	} else {
		/* 
		 * Unsolicited Nop-In,  most likely to notify us about a 
		 * hole in ExpCmdSN or StatSN. 
		 */
		if ((hdr->targ_xfer_tag != 0xFFFFFFFF) &&
		    (hdr->lun == 0xFFFFFFFFFFFFFFFFULL)) {
			TRACE_ERROR("Unsolicited NopIN: Valid TTT, but LUN is reserved!\n");
		}
			
		TRACE(TRACE_ISCSI, "Received Unsolicited NOPIN ITT: 0x%08x,"
			" TTT: 0x%08x, StatSN: 0x%08x on CID: %hu\n",
			hdr->init_task_tag, hdr->targ_xfer_tag, hdr->stat_sn,
				conn->cid); 

		iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn, 1);
     		iscsi_update_statsn(conn, NULL, hdr->stat_sn, 0);
		
		if (hdr->targ_xfer_tag == 0xFFFFFFFF) {
			/* 
			 * This is a NOPIN Not expecting a reply,  Is used to 
			 * asertain holes in CmdSN and StatSN.
			 */
			;
		} else {
			sess->targ_xfer_tag = hdr->targ_xfer_tag;

			/*
			 * This is a ping request,  send a response right away.
			 */
			if (iscsi_send_nopout_response(conn, buf) < 0)
				return(-1);
		}
	}

	return(0);
}

/*	iscsi_logout_closesession():
 *
 *
 */
static int iscsi_logout_closesession (
	iscsi_logout_attempt_t *la)
{
	iscsi_conn_t *conn = la->logout_conn;
	iscsi_session_t *sess = SESS(conn);
	
	switch (la->logout_response) {
	case CONNORSESSCLOSEDSUCCESSFULLY:
		TRACE(TRACE_ISCSI, "iSCSI Session closed successfully"
			" at target.\n");
		break;
	case CIDNOTFOUND:
		TRACE_ERROR("CID was not found on target.\n");
		break;
	case CONNRECOVERYNOTSUPPORTED:
		TRACE_ERROR("Connection Recovery not supported at target.\n");
		break;
	case CLEANUPFAILED:
		TRACE_ERROR("Cleanup failed at Target for various reasons.\n");
		break;
	default:
		TRACE_ERROR("Logout response: 0x%02x unknown, protocol"
			" error.\n", la->logout_response);
		break;
	}

	kfree(la);
	
	iscsi_set_thread_clear(conn, ISCSI_CLEAR_RX_THREAD);
	iscsi_set_thread_set_signal(conn, ISCSI_SIGNAL_RX_THREAD);

	up(&conn->conn_logout_sem);
	
	iscsi_stop_session_and_reinstatement(sess, 0);
		
	return(1);
}

/*	iscsi_logout_closeconn_reinstatement():
 *
 *
 */
static void iscsi_logout_closeconnection_reinstatement (
	iscsi_logout_attempt_t *la,
	iscsi_session_t *sess)
{
	atomic_set(&sess->session_logout, 0);
	atomic_set(&sess->session_reinstatement, 1);
	iscsi_cause_session_reinstatement(sess);
		
	return;
}

/*	iscsi_logout_closeconnection_samecid():
 *
 *
 */
static int iscsi_logout_closeconnection_samecid (
	iscsi_logout_attempt_t *la)
{
	int ret;
	iscsi_channel_t *channel;
	iscsi_conn_t *conn = la->logout_conn;
	iscsi_session_t *sess = SESS(conn);
	
	channel = sess->channel;
		
	switch (la->logout_response) {
	case CONNORSESSCLOSEDSUCCESSFULLY:
		TRACE(TRACE_ISCSI, "iSCSI Connection %hu closed successfully"
				" at target.\n", conn->cid);
		iscsi_set_thread_clear(conn, ISCSI_CLEAR_RX_THREAD);
		iscsi_set_thread_set_signal(conn, ISCSI_SIGNAL_RX_THREAD);

		up(&conn->conn_logout_sem);
		
		if (atomic_read(&sess->session_logout))
			iscsi_stop_session_and_reinstatement(sess, 0);
		else
			iscsi_stop_connection_and_reinstatement(conn);
		kfree(la);
		return(1);
	case CIDNOTFOUND:
		TRACE_ERROR("CID was not found on target.\n");
		break;
	case CONNRECOVERYNOTSUPPORTED:
		TRACE_ERROR("Connection Recovery not supported at target.\n");
		break;
	case CLEANUPFAILED:
		TRACE_ERROR("Cleanup failed at Target for various reasons.\n");
		break;
	default:
		TRACE_ERROR("Logout response: 0x%02x unknown protocol"
			" error.\n", la->logout_response);
		break;
	}

	if (ISCSI_CA(channel)->closeconn_reinstatement &&
	    !atomic_read(&sess->session_logout)) {
		up(&conn->conn_logout_sem);
		iscsi_logout_closeconnection_reinstatement(la, sess);	
		ret = 0;
	} else {
		iscsi_set_thread_clear(conn, ISCSI_CLEAR_RX_THREAD);
		iscsi_set_thread_set_signal(conn, ISCSI_SIGNAL_RX_THREAD);
		up(&conn->conn_logout_sem);
		iscsi_stop_session_and_reinstatement(sess, 0);
		ret = 1;
	}
	kfree(la);
	
	return(ret);
}

/*	iscsi_logout_closeconnection_differentcid():
 *
 *
 */
static int iscsi_logout_closeconnection_differentcid (
	iscsi_logout_attempt_t *la)
{
	int ret;
	iscsi_channel_t *channel;
	iscsi_conn_t *conn = la->logout_conn; /* Connection to be logged out */
	iscsi_conn_t *on_conn = la->logout_on_conn; /* The current connection */
	iscsi_session_t *sess = SESS(conn);
	
	channel = sess->channel;
		
	switch (la->logout_response) {
	case CONNORSESSCLOSEDSUCCESSFULLY:
		TRACE(TRACE_ISCSI, "iSCSI CID %hu on CID: %hu closed"
			" successfully at target.\n", conn->cid, on_conn->cid);
		iscsi_stop_connection_and_reinstatement(conn);
		iscsi_dec_conn_usage_count(conn);
		kfree(la);
		return(0);
	case CIDNOTFOUND:
		TRACE_ERROR("CID %hu was not found on target.\n", conn->cid);
		break;
	case CONNRECOVERYNOTSUPPORTED:
		TRACE_ERROR("Connection Recovery not supported at target.\n");
		break;
	case CLEANUPFAILED:
		TRACE_ERROR("Cleanup failed at Target for various reasons.\n");
		break;
	default:
		TRACE_ERROR("Logout response: 0x%02x unknown, protocol"
			" error.\n", la->logout_response);
		break;
	}

	iscsi_dec_conn_usage_count(conn);
	
	if (ISCSI_CA(channel)->closeconn_reinstatement &&
	    !atomic_read(&sess->session_logout)) {
		iscsi_logout_closeconnection_reinstatement(la, sess);
		ret = 0;
	} else {
		iscsi_set_thread_clear(on_conn, ISCSI_CLEAR_RX_THREAD);
		iscsi_set_thread_set_signal(on_conn, ISCSI_SIGNAL_RX_THREAD);
		iscsi_stop_session_and_reinstatement(sess, 0);
		ret = 1;
	}
	kfree(la);
	
	return(ret);
}

/*	iscsi_handle_logout_rsp():
 *
 *	Called From:	iscsi_initiator_rx_thread().
 *	Parameters:	iSCSI Connection Pointer,
 *			Pointer to buffer of received opcode.
 *	Returns:	1: Goto to restart thread set.
 *			0: Continue as normal.
 *			-1: Cause a transport reset.
 */
static inline int iscsi_handle_logout_rsp (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	iscsi_cmd_t *cmd;
	iscsi_logout_attempt_t *la;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_targ_logout_rsp *hdr;

	hdr			= (struct iscsi_targ_logout_rsp *) buf;
	hdr->length		= be32_to_cpu(hdr->length);
	hdr->init_task_tag	= be32_to_cpu(hdr->init_task_tag);
	hdr->stat_sn		= be32_to_cpu(hdr->stat_sn);
	hdr->exp_cmd_sn		= be32_to_cpu(hdr->exp_cmd_sn);
	hdr->max_cmd_sn		= be32_to_cpu(hdr->max_cmd_sn);
	hdr->time_2_wait	= be16_to_cpu(hdr->time_2_wait);
	hdr->time_2_retain	= be16_to_cpu(hdr->time_2_retain);

	if (!(cmd = iscsi_find_cmd_from_itt(conn, hdr->init_task_tag)))
		return(0);

	TRACE(TRACE_ISCSI, "Got Logout Response: ITT: 0x%08x, StatSN: 0x%08x,"
		" Response: 0x%02x, CID: %hu\n", hdr->init_task_tag,
			hdr->stat_sn, hdr->response, conn->cid);

	iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn,
			(cmd->pdu[0] & I_BIT));
	
	spin_lock(&sess->la_lock);
	if (!cmd->la) {
		TRACE_ERROR("Unable to locate iscsi_logout_attempt_t for"
			" ITT: 0x%08x\n", cmd->init_task_tag);
		spin_unlock(&sess->la_lock);
		return(0);
	}
	la = cmd->la;
	la->logout_response = hdr->response;

	if (atomic_read(&la->logout_timer_expired)) {
		spin_unlock(&sess->la_lock);
		return(0);
	}
	atomic_set(&la->logout_got_response, 1);
	spin_unlock(&sess->la_lock);

	/*
	 * iscsi_start_logout() will be blocked on CLOSESESSION and
	 * CLOSECONNECTION Logout Requests.
	 */
	if (la->logout_reason != REMOVECONNFORRECOVERY) {
		up(&la->logout_sem);
		down(&la->logout_done_sem);
	}
	
	if (iscsi_process_response(conn, cmd, buf) < 0)
		 return(-1);

	switch (la->logout_reason) {
	case CLOSESESSION:
		return(iscsi_logout_closesession(la));
	case CLOSECONNECTION:
		if (la->logout_cid == la->logout_on_cid)
			return(iscsi_logout_closeconnection_samecid(la));
		else
			return(iscsi_logout_closeconnection_differentcid(la));
	case REMOVECONNFORRECOVERY:
		return(0);
	default:
		return(-1);
	}

	return(0);
}

/*	iscsi_handle_r2t():
 *
 * 	Called From:	iscsi_initiator_rx_thread().
 *	Parameters:	iSCSI Connection pointer, Pointer to received buffer.
 *	Returns:	0 on success, -1 on error.
 */
static inline int iscsi_handle_r2t (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	iscsi_cmd_t *cmd;
	iscsi_r2t_t *r2t;
	iscsi_seq_t *seq = NULL;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_targ_r2t *hdr;

	hdr			= (struct iscsi_targ_r2t *) buf;
	hdr->init_task_tag	= be32_to_cpu(hdr->init_task_tag);
	hdr->targ_xfer_tag	= be32_to_cpu(hdr->targ_xfer_tag);
	hdr->stat_sn		= be32_to_cpu(hdr->stat_sn);
	hdr->exp_cmd_sn		= be32_to_cpu(hdr->exp_cmd_sn);
	hdr->max_cmd_sn		= be32_to_cpu(hdr->max_cmd_sn);
	hdr->r2t_sn		= be32_to_cpu(hdr->r2t_sn);
	hdr->offset		= be32_to_cpu(hdr->offset);
	hdr->xfer_len		= be32_to_cpu(hdr->xfer_len);

	if (!(cmd = iscsi_find_cmd_from_itt(conn, hdr->init_task_tag)))
		return(0);

	if (!cmd->scsi_cmnd) {
		TRACE_ERROR("iscsi_cmd_t->scsi_cmnd is NULL!\n");
		return (-1);
	}

	if (!hdr->xfer_len) {
		TRACE_ERROR("ITT: 0x%08x, Received R2Ts DDTL is 0,"
			" protocol error.\n", cmd->init_task_tag);
		return(-1);
	}
	
	if ((hdr->offset + hdr->xfer_len) > cmd->data_length) {
		TRACE_ERROR("ITT: 0x%08x received Offset: %u, XferLen: %u"
			" greater than commands EDTL: %u.\n",
			cmd->init_task_tag, hdr->offset, hdr->xfer_len,
				cmd->data_length);
		return(-1);
	}

	if (hdr->xfer_len > SESS_OPS_C(conn)->MaxBurstLength) {
		TRACE_ERROR("ITT: 0x%08x, R2T DDTL of %u exceeds MaxBurstLength"
			" of %u\n, protocol error", cmd->init_task_tag,
			hdr->xfer_len, SESS_OPS_C(conn)->MaxBurstLength);
		return(-1);
	}

	if (hdr->targ_xfer_tag == 0xFFFFFFFF) {
		TRACE_ERROR("ITT: 0x%08x, R2T has reserved TTT of 0xFFFFFFFF,"
			" but we are expecting a valid value, protocol"
				" error.\n", cmd->init_task_tag);
		return(-1);
	}
	sess->targ_xfer_tag = hdr->targ_xfer_tag;

	if (hdr->r2t_sn > cmd->r2t_sn) {
		TRACE_ERROR("ITT: 0x%08x, contains R2TSN: 0x%08x greater than"
			" expected 0x%08x\n", cmd->init_task_tag, hdr->r2t_sn,
				cmd->r2t_sn);
		if (!SESS_OPS_C(conn)->ErrorRecoveryLevel) {
			TRACE_ERROR("Unable to recover from missing"
				" R2TSN while ERL=0.\n");
			return(-1);
		}
		return(0);
	} else if (hdr->r2t_sn < cmd->r2t_sn) {
		TRACE(TRACE_ERL1, "ITT: 0x%08x contains R2TSN: 0x%08x less than"
			" expected 0x%08x, ignoring.\n", cmd->init_task_tag,
				hdr->r2t_sn, cmd->r2t_sn);
		return(0);
	} else {
		cmd->cmd_flags &= ~ICF_OUT_OF_ORDER_R2TSN;
		cmd->r2t_sn++;
	}

	if (!(r2t = iscsi_allocate_r2t()))
		return(-1);

	r2t->offset = hdr->offset;
	r2t->r2t_sn = hdr->r2t_sn;
	r2t->xfer_length = hdr->xfer_len;
	r2t->targ_xfer_tag = hdr->targ_xfer_tag;
	
	if (SESS_OPS(sess)->DataSequenceInOrder) {
		if (hdr->offset > cmd->data_offset) {
			TRACE_ERROR("ITT: 0x%08x DataSequenceInOrder=Yes but"
				" R2T has Buffer Offset %u and expected %u,"
				" protocol error\n", cmd->init_task_tag,
					hdr->offset, cmd->data_offset);
			goto failure;
		}
		if (hdr->offset < cmd->data_offset) {
			if (!SESS_OPS_C(conn)->ErrorRecoveryLevel) {
				TRACE_ERROR("ITT: 0x%08x, R2T Offset less"
				" than expected (an Recovery R2T), but ERL=0.\n",
					cmd->init_task_tag);
				goto failure;
			}
		}
	} else {
		if (!(seq = iscsi_get_seq_holder(cmd, hdr->offset, hdr->xfer_len)))
			goto failure;

		r2t->seq = seq;
		
		if (seq->status == DATAOUT_SEQUENCE_GOT_R2T) {
			if (!SESS_OPS(sess)->ErrorRecoveryLevel) {
				TRACE_ERROR("ITT: 0x%08x, R2T Offset less"
				" than expected (an Recovery R2T), but ERL=0.\n",
					cmd->init_task_tag);
				goto failure;
			}
		} else
			seq->status = DATAOUT_SEQUENCE_GOT_R2T;
	}

	TRACE(TRACE_ISCSI, "Got R2T: ITT: 0x%08x, StatSN: 0x%08x, R2TSN: 0x%08x,"
		" Offset: %u, DDTL: %u, CID: %hu\n", hdr->init_task_tag,
		hdr->stat_sn, hdr->r2t_sn, hdr->offset, hdr->xfer_len, conn->cid);

	/*
	 * Recovery R2Ts are not counted against MaxOutstandingR2Ts, but we still
	 * keep our own counter for good measure.
	 */
	spin_lock(&cmd->r2t_lock);
	if (++cmd->outstanding_r2ts > SESS_OPS(sess)->MaxOutstandingR2T) {
		TRACE_ERROR("Command ITT: 0x%08x received %u R2Ts, but"
			" MaxOutstandingR2T is %u, protocol error.\n",
			cmd->init_task_tag, cmd->outstanding_r2ts,
			SESS_OPS(sess)->MaxOutstandingR2T);
		spin_unlock(&cmd->r2t_lock);
		goto failure;
	}
	spin_unlock(&cmd->r2t_lock);
	
	if (SESS_OPS(sess)->DataSequenceInOrder)
		cmd->data_offset += r2t->xfer_length;

	/* 
	 * For DataPDUInOrder=No calculate r2t->pdu_start, r2t->pdu_count and
	 * r2t->pdu_send_count.
	 */
	if (!SESS_OPS(sess)->DataPDUInOrder)
		if (iscsi_set_pdu_values_for_r2t(cmd, r2t) < 0)
			goto failure;
	
	iscsi_add_r2t_to_cmd(cmd, r2t);
	
	iscsi_add_cmd_to_immediate_queue(cmd, conn, ISTATE_SEND_DATAOUT);

	iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn, 1);
	iscsi_update_statsn(conn, cmd, hdr->stat_sn, 0);

	return(0);

failure:
	if (r2t)
		kfree(r2t);
	return(-1);
}

/*	iscsi_handle_task_mgt_rsp():
 * 
 * 	Called From:	iscsi_initiator_rx_thread().
 *	Parameters:	iSCSI Connection Pointer,
 *			Point to buffer of received opcode.
 *	Returns:	0 on success, -1 on error.
 */
static inline int iscsi_handle_task_mgt_rsp (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	u8 opcode, task_mgt_function;
	u32 init_task_tag;
	iscsi_cmd_t *cmd, *ref_cmd;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_targ_task_mgt_rsp *hdr;

	hdr 			= (struct iscsi_targ_task_mgt_rsp *) buf;
	hdr->init_task_tag	= be32_to_cpu(hdr->init_task_tag);
	hdr->stat_sn		= be32_to_cpu(hdr->stat_sn);
	hdr->exp_cmd_sn		= be32_to_cpu(hdr->exp_cmd_sn);
	hdr->max_cmd_sn		= be32_to_cpu(hdr->max_cmd_sn);	

	if (!(cmd = iscsi_find_cmd_from_itt(conn, hdr->init_task_tag)))
		return(0);

	TRACE(TRACE_ISCSI, "Got TMR Response: ITT: 0x%08x, StatSN: 0x%08x,"
		" Response: 0x%02x, CID: %hu\n", hdr->init_task_tag,
			hdr->stat_sn, hdr->response, conn->cid);

	iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn,
			(cmd->pdu[0] & I_BIT));
	
	if (!cmd->ref_cmd) {
		TRACE_ERROR("cmd->ref_cmd for ITT: 0x%08x!\n", cmd->init_task_tag);
		return(-1);
	}
	ref_cmd = cmd->ref_cmd;
	
	opcode = ref_cmd->pdu[0];
	init_task_tag = cmd->init_task_tag;
	task_mgt_function = cmd->task_mgt_function;
	cmd->task_mgt_response = hdr->response;
	
	if (iscsi_process_response(conn, cmd, buf) < 0)
		return(-1);            

	switch (hdr->response) {
	case FUNCTION_COMPLETE:
		switch (task_mgt_function) {
		case ABORT_TASK:
		case ABORT_TASK_SET:
		case CLEAR_ACA:
		case CLEAR_TASK_SET:
		case LUN_RESET:
		case TARGET_WARM_RESET:
		case TARGET_COLD_RESET:
			break;
		case TASK_REASSIGN:
			break;
		default:
			TRACE_ERROR("Unknown task_mgt_function: %d\n",
				task_mgt_function);
			goto session_reinstatement;
		}
		break;
	case TASK_DOES_NOT_EXIST:
		switch (task_mgt_function) {
		case ABORT_TASK:
		case ABORT_TASK_SET:
		case CLEAR_ACA:
		case CLEAR_TASK_SET:
			break;
		case TASK_REASSIGN:
			if (!(opcode & I_BIT)) {
				TRACE_ERROR("Got Response TASK_DOES_NOT_EXIST"
				" for TASK_REASSIGN TMR ITT: 0x%08x,"
				" RefTaskTag: 0x%08x, cannot continue with"
				" connection recovery.\n", init_task_tag,
					ref_cmd->init_task_tag);
				goto session_reinstatement;
			}	

			break;
		default:
			goto session_reinstatement;
		}
		break;
	case LUN_DOES_NOT_EXIST:
		switch (task_mgt_function) {
		case ABORT_TASK:
		case ABORT_TASK_SET:
		case CLEAR_ACA:
		case CLEAR_TASK_SET:
		case LUN_RESET:
			break;
		case TASK_REASSIGN:
			TRACE_ERROR("Got Response LUN_DOES_NOT_EXIST for"
			" TASK_REASSIGN TMR ITT: 0x%08x, RefTaskTag: 0x%08x,"
			" cannot continue with connection recovery.\n",
				init_task_tag, ref_cmd->init_task_tag);
			goto session_reinstatement;
		default:
			goto session_reinstatement;
		}
		break;
	case TASK_STILL_ALLEGIANT:
		switch (task_mgt_function) {
		case TASK_REASSIGN:
			TRACE_ERROR("Got Response TASK_STILL_ALLEGIANT for"
			" TASK_REASSIGN TMR ITT: 0x%08x, RefTaskTag: 0x%08x,"
			" cannot continue with connection recovery.\n",
				init_task_tag, ref_cmd->init_task_tag);
			goto session_reinstatement;
		default:
			goto session_reinstatement;
		}
		break;
	case TASK_FAILOVER_NOT_SUPPORTED:
		switch (task_mgt_function) {
		case TASK_REASSIGN:
			TRACE_ERROR("Got Response TASK_FAILOVER_NOT_SUPPORTED for"
			" TASK_REASSIGN TMR ITT: 0x%08x, RefTaskTag: 0x%08x,"
			" cannot continue with connection recovery.\n",
				init_task_tag, ref_cmd->init_task_tag);
			goto session_reinstatement;
		default:
			goto session_reinstatement;
		}
		break;
	case TASK_MGMT_FUNCTION_NOT_SUPPORTED:
		switch (task_mgt_function) {
		case ABORT_TASK:
		case ABORT_TASK_SET:
		case CLEAR_ACA:
		case CLEAR_TASK_SET:
		case LUN_RESET:
		case TARGET_WARM_RESET:
		case TARGET_COLD_RESET:
			break;
		case TASK_REASSIGN:
			TRACE_ERROR("Got Response TASK_MGMT_FUNCTION_NOT_SUPPORTED for"
			" TASK_REASSIGN TMR ITT: 0x%08x, RefTaskTag: 0x%08x,"
			" cannot continue with connection recovery.\n",
				init_task_tag, ref_cmd->init_task_tag);
			goto session_reinstatement;
		default:
			goto session_reinstatement;
		}
		break;
	case FUNCTION_AUTHORIZATION_FAILED:
		switch (task_mgt_function) {
		case LUN_RESET:
		case TARGET_WARM_RESET:
		case TARGET_COLD_RESET:
			break;
		case TASK_REASSIGN:
			TRACE_ERROR("Got Response FUNCTION_AUTHORIZATION_FAILED for"
			" TASK_REASSIGN TMR ITT: 0x%08x, RefTaskTag: 0x%08x,"
			" cannot continue with connection recovery.\n",
				init_task_tag, ref_cmd->init_task_tag);
			goto session_reinstatement;
		default:
			goto session_reinstatement;
		}
		break;
	case FUNCTION_REJECTED:
		switch (task_mgt_function) {
		case TASK_REASSIGN:
			TRACE_ERROR("Got Response FUNCTION_REJECTED for"
			" TASK_REASSIGN TMR ITT: 0x%08x, RefTaskTag: 0x%08x,"
			" cannot continue with connection recovery.\n",
				init_task_tag, ref_cmd->init_task_tag);
			goto session_reinstatement;
		default:
			goto session_reinstatement;
		}
		break;
	default:
		TRACE_ERROR("Unknown TMR Response: 0x%02x, protocol"
				" error.\n", hdr->response);
		goto session_reinstatement;
	}

	return(0);
	
session_reinstatement:
	iscsi_cause_session_reinstatement(sess);

	return(-1);
}

/*	iscsi_handle_async_msg():
 *
 *	Function handles ISCSI_TARG_ASYNC_MSG PDU, and takes appropriate action 
 *	depending on hdr->async_event.
 *
 *	Called From:	iscsi_initiator_rx_thread().
 *	Parameters: 	iSCSI Connection Pointer,
 *			Pointer to buffer of received opcode
 *	Returns:	0 on success, -1 on error
 */
static inline int iscsi_handle_async_msg (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	unsigned char *payload = NULL;
	int ret = 0;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_targ_async_msg *hdr;

	hdr			= (struct iscsi_targ_async_msg *) buf;
	hdr->length		= be32_to_cpu(hdr->length);
	hdr->lun		= be64_to_cpu(hdr->lun);
	hdr->stat_sn		= be32_to_cpu(hdr->stat_sn);
	hdr->exp_cmd_sn		= be32_to_cpu(hdr->exp_cmd_sn);
	hdr->max_cmd_sn		= be32_to_cpu(hdr->max_cmd_sn);
	hdr->parameter1		= be16_to_cpu(hdr->parameter1);
	hdr->parameter2		= be16_to_cpu(hdr->parameter2);
	hdr->parameter3		= be16_to_cpu(hdr->parameter3);

	if (hdr->length > CONN_OPS(conn)->MaxRecvDataSegmentLength) {
		TRACE_ERROR("Reject payload size %u is greater than"
			" MaxRecvDataSegmentLength %u, protocol error.\n",
			hdr->length, CONN_OPS(conn)->MaxRecvDataSegmentLength);
		return(-1);
	}       
	
	if (hdr->length) {
		if (!(payload = iscsi_rx_data_payload(conn, hdr->length)))
			return(0);
	}
	
	iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn, 1);
	iscsi_update_statsn(conn, NULL, hdr->stat_sn, 1);

	switch (hdr->async_event) {
	case ASYNC_EVENT_SCSI_EVENT:
		/* 
		 * FIXME: Receieve sense data a SCSI Asynchronous Event
		 * is reported in the sense data.
		 */
		TRACE_ERROR("SCSI Asynchronous Event not supported\n");
		break;
	case ASYNC_EVENT_REQUEST_LOGOUT:
		TRACE(TRACE_ISCSI, "Got Async Message requesting to"
			" logout within %u seconds\n", hdr->parameter3);
		TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_LOGOUT_REQUESTED.\n");
		spin_lock_bh(&conn->state_lock);
		conn->conn_state = INIT_CONN_STATE_LOGOUT_REQUESTED;
		spin_unlock_bh(&conn->state_lock);

		iscsi_start_logout(sess, 0, 1, CLOSESESSION, conn->cid, conn->cid);
		break;
	case ASYNC_EVENT_DROP_CONNECTION:
		TRACE(TRACE_ERL2, "Got Async Message on CID: %hu indicating"
			" CID: %hu was dropped.\n", conn->cid, hdr->parameter1);
		break;
	case ASYNC_EVENT_DROP_SESSION:
		TRACE(TRACE_ISCSI, "Got Async Message indicating target"
			" will drop all connections of this session,"
				" closing iSCSI session.\n");
		iscsi_cause_session_reinstatement(sess);
		break;
	case ASYNC_EVENT_REQUEST_TEXT:
		/*
		 * FIXME: Send a text request of length 0 here.
		 */
		TRACE_ERROR("Got Async Message requesting a text"
			" request, not implemented yet.\n");
		break;
	case ASYNC_EVENT_VENDOR_SPECIFIC:
		/*
		 * FIXME: Nothing to do here ATM.
		 */
		TRACE(TRACE_ISCSI, "Got Async Message containing a"
			" vendor specific event.\n");
		break;
	default:
		TRACE_ERROR("Unknown asynchronous event: 0x%02x\n",
				hdr->async_event);
		ret = -1;
		break;
	}

	if (payload)
		kfree(payload);
	
	return(ret);
}

/*	iscsi_handle_rjt():
 *	
 * 	Called From:	iscsi_initiator_rx_thread().
 *	Parameters:	iSCSI Connection Pointer,
 *				Pointer to buffer of received opcode.
 *	Returns:	0 on success, -1 on error.
 */
//#warning FIXME: More robust handling of various reject situations.
static inline int iscsi_handle_rjt (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	int ret = 0, update = 0;
	unsigned char *payload = NULL;
	struct iscsi_targ_rjt *hdr;
	iscsi_session_t *sess = SESS(conn);

	hdr 			= (struct iscsi_targ_rjt *) buf;
	hdr->length		= be32_to_cpu(hdr->length);
	hdr->stat_sn		= be32_to_cpu(hdr->stat_sn);
	hdr->exp_cmd_sn		= be32_to_cpu(hdr->exp_cmd_sn);
	hdr->max_cmd_sn		= be32_to_cpu(hdr->max_cmd_sn);
	hdr->data_sn		= be32_to_cpu(hdr->data_sn);

	TRACE(TRACE_ISCSI, "Got Reject Reason: 0x%02x, ExpStatSN: 0x%08x,"
		" CID %hu\n", hdr->reason, hdr->stat_sn, conn->cid);

	print_reject_reason(hdr->reason);

	if (hdr->length > CONN_OPS(conn)->MaxRecvDataSegmentLength) {
		TRACE_ERROR("Reject payload size %u is greater than"
			" MaxRecvDataSegmentLength %u, protocol error.\n",
			hdr->length, CONN_OPS(conn)->MaxRecvDataSegmentLength);
		return(-1);
	}

	if (hdr->length) {
		if (!(payload = iscsi_rx_data_payload(conn, hdr->length)))
			return(0);
	}

	switch (hdr->reason) {
	case REASON_FULL_BEFORE_LOGIN:
		break;
	case REASON_DATA_DIGEST_ERR:
		if (!SESS_OPS(sess)->ErrorRecoveryLevel) {
			TRACE_ERROR("Unable to recover from DataOUT"
				" data digest in ERL=0.\n");
			ret = -1;
			goto out;
		}
		update = 1;
		break;
	case REASON_DATA_SNACK:
		break;
	case REASON_PROTOCOL_ERR:
		ret = -1;
		goto out;
	case REASON_COMMAND_NOT_SUPPORTED:
	case REASON_TOO_MANY_IMMEDIATE_COMMANDS:
	case REASON_TASK_IN_PROGRESS:
	case REASON_INVALID_DATA_ACK:
		break;
	case REASON_INVALID_PDU_FIELD:
	case REASON_OUT_OF_RESOURCES:
		ret = -1;
		goto out;
	case REASON_NEGOTIATION_RESET:
	case REASON_WAITING_FOR_LOGOUT:
	default:
		break;
	}

	if (update) {
		iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn, 1);
		iscsi_update_statsn(conn, NULL, hdr->stat_sn, 1);
	}

out:
	if (payload)
		kfree(payload);
	if (!ret)
		return(0);

	iscsi_cause_session_reinstatement(sess);
	
	return(-1);
}

/*	iscsi_handle_rsp():
 *
 *	Called From:	iscsi_initiator_rx_thread().
 *	Parameters:	iSCSI Connection pointer,
 *			Pointer to buffer containing PDU.
 *	Returns:	0 on success, -1 on error.
 */
static inline int iscsi_handle_rsp (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	unsigned char host_stat = DID_OK, *payload = NULL;
	int missing, resid = 0;
	iscsi_cmd_t *cmd;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_targ_scsi_rsp *hdr;
	struct scsi_cmnd *sc;

	hdr		 	= (struct iscsi_targ_scsi_rsp *) buf;
	hdr->length		= be32_to_cpu(hdr->length);
	hdr->init_task_tag	= be32_to_cpu(hdr->init_task_tag);
	hdr->res_count		= be32_to_cpu(hdr->res_count);
	hdr->stat_sn		= be32_to_cpu(hdr->stat_sn);
	hdr->exp_cmd_sn		= be32_to_cpu(hdr->exp_cmd_sn);
	hdr->max_cmd_sn		= be32_to_cpu(hdr->max_cmd_sn);
	hdr->exp_data_sn	= be32_to_cpu(hdr->exp_data_sn);
	hdr->bidi_res_count	= be32_to_cpu(hdr->bidi_res_count);

	if (hdr->length > CONN_OPS(conn)->MaxRecvDataSegmentLength) {
		TRACE_ERROR("Response payload size %u is greater than"
			" MaxRecvDataSegmentLength %u, protocol error.\n",
			hdr->length, CONN_OPS(conn)->MaxRecvDataSegmentLength);
		return(-1);
	}
	
	if (!(cmd = iscsi_find_cmd_from_itt_or_dump(conn, hdr->init_task_tag,
				hdr->length)))
		return(0);

	if (!cmd->scsi_cmnd) {	
		TRACE_ERROR("iscsi_cmd_t->scsi_cmnd is NULL!\n");
		return(-1);
	}
	sc = cmd->scsi_cmnd;

	TRACE(TRACE_ISCSI, "Got Response ITT: 0x%08x, StatSN: 0x%08x,"
		" Response: 0x%02x, SAM Status: 0x%02x, CID: %hu\n",
		hdr->init_task_tag, hdr->stat_sn, hdr->response,
			hdr->status, conn->cid);

	iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn, 0);
	
	/*
	 * FIXME: Overflow Data
	 * FIXME: Add checks for Bidirectional Read Residual Over/Under Flow
	 */
	if (sc->sc_data_direction == DMA_FROM_DEVICE) {
		iscsi_stop_datain_timer(cmd);

		missing = (sc->request_bufflen - cmd->read_data_done);
		if (hdr->flags & O_BIT) {
			if (missing != 0) {
				TRACE_ERROR("Overflow Bit set, but still"
					" missing %d bytes\n", missing);
				return(-1);
			}
			if (!hdr->res_count) {
				TRACE_ERROR("Overflow Bit set, but Residual"
					" count is 0\n");
				return(-1);
			}
			resid = sc->request_bufflen;
			host_stat = host_byte(DID_ERROR);
		} else if (hdr->flags & U_BIT) {
			if (missing != hdr->res_count) {
				TRACE(TRACE_DEBUG, "Underflow Residual count %u does"
					" NOT match expected %d\n",
						hdr->res_count, missing);
			}
			resid = hdr->res_count;
		} else if (hdr->res_count != 0) {
			TRACE_ERROR("Neither Overflow nor Underflow bit set,"
				" but Residual count is %u\n", hdr->res_count);
			return(-1);
		}
	}

	/*
	 * If the command could not be completed at the Target, fail the connection.
	 */
	if (hdr->response != 0x00) {
		TRACE_ERROR("Command ITT: 0x%08x could not be completed at"
				" target.\n", hdr->response);
		return(-1);
	}

	if (hdr->length) {
		unsigned char *sense_buf = NULL;
		unsigned int sense_buffer_size;
		
		if (!(payload = iscsi_rx_data_payload(conn, hdr->length)))
			return(0);

		sense_buf = payload + 2; /* Skip over SenseLength field */
		
		TRACE(TRACE_ISCSI, "Received %u bytes of SENSE Data.\n",
				hdr->length);

		sense_buffer_size = hdr->length - 2;
		if (sense_buffer_size > SCSI_SENSE_BUFFERSIZE)
			sense_buffer_size = SCSI_SENSE_BUFFERSIZE;

		memcpy((void *)sc->sense_buffer, (void *)sense_buf, sense_buffer_size);
		sc->result = (DRIVER_SENSE << 24);
		
		kfree(payload);
	}

	sc->resid = resid;
	sc->result = HOST_BYTE(host_stat);
	sc->result = STATUS_BYTE(hdr->status);
	
	if (iscsi_process_response(conn, cmd, buf) < 0)
		return(-1);

	return(0);
}

/*	iscsi_handle_text_rsp():
 *
 *	Function which handles a ISCSI_TARG_TEXT_RSP PDU and accompaning
 *	text parameters.  If we are in an discovery session, assume the
 *	response to a SendTargets key, and call
 *	iscsi_handle_sendtargets_response() to do the dirty work.
 *
 *	Called From:	iscsi_initiator_rx_thread().
 *	Parameters:	iSCSI Connection Pointer,
 *			Pointer to buffer containing PDU.
 *	Returns:	0 on success, -1 on error.
 */
 static inline int iscsi_handle_text_rsp (
	iscsi_conn_t *conn,
	unsigned char *buf)
{
	unsigned char *payload = NULL;
	int ret = 0;
	iscsi_cmd_t *cmd;
	iscsi_session_t *sess = SESS(conn);
	struct iscsi_targ_text_rsp *hdr;

	hdr			= (struct iscsi_targ_text_rsp *) buf;
	hdr->length		= be32_to_cpu(hdr->length);
	hdr->init_task_tag	= be32_to_cpu(hdr->init_task_tag);
	hdr->targ_xfer_tag	= be32_to_cpu(hdr->targ_xfer_tag);
	hdr->stat_sn		= be32_to_cpu(hdr->stat_sn);
	hdr->exp_cmd_sn		= be32_to_cpu(hdr->exp_cmd_sn);
	hdr->max_cmd_sn		= be32_to_cpu(hdr->max_cmd_sn);
							
	if (hdr->length > CONN_OPS(conn)->MaxRecvDataSegmentLength) {
		TRACE_ERROR("Response payload size %u is greater than"
			" MaxRecvDataSegmentLength %u, protocol error.\n",
			hdr->length, CONN_OPS(conn)->MaxRecvDataSegmentLength);
		return(-1);
	}

	if (!(cmd = iscsi_find_cmd_from_itt_or_dump(conn, hdr->init_task_tag,
			hdr->length)))
		return(0);

	TRACE(TRACE_ISCSI, "Received Text Response ITT: 0x%08x, StatSN:"
		" 0x%08x, Length: %u, CID: %hu\n", hdr->init_task_tag,
			hdr->stat_sn, hdr->length, conn->cid);
	
	iscsi_update_cmdsn(sess, hdr->exp_cmd_sn, hdr->max_cmd_sn,
			(cmd->pdu[0] & I_BIT));
	
	if (hdr->length) {
		if (!(payload = iscsi_rx_data_payload(conn, hdr->length)))
			return(0);

//#warning FIXME: Move simple discovery logic elseware.
		if (SESS_OPS(sess)->SessionType) {
			if (iscsi_handle_sendtargets_response(conn, payload,
					hdr->length) < 0)
				ret = -1;
			up(&sess->discovery_sem);
			if (ret < 0)
				goto out;
		}
	}
	
	if (iscsi_process_response(conn, cmd, buf) < 0)
		ret = -1;
out:
	if (payload)
		kfree(payload);
	return(ret);
}

/*	iscsi_check_scsi_cmnd_unmap():
 *
 *	Unmap scatterlist segments for unsolicited immediate data.
 */
static inline void iscsi_check_scsi_cmnd_unmap (iscsi_unmap_sg_t *unmap_sg)
{
	iscsi_cmd_t *cmd = unmap_sg->cmd;

	if (!cmd->data_offset)
		return;

	iscsi_unmap_scatterlists((void *)unmap_sg);

	return;
}

/*	iscsi_initiator_tx_thread();
 *
 *	Parameters:	iSCSI Connection Pointer.
 *	Returns:	Nothing.
 */
extern int iscsi_initiator_tx_thread (void *arg)
{
	u8 state;
	int eos = 0, map_sg = 0, ret = 0, scsi_task = 0, use_misc = 0;
	iscsi_cmd_t *cmd = NULL;
	iscsi_conn_t *conn;
	iscsi_queue_req_t *qr = NULL;
	iscsi_session_t *sess;
	iscsi_thread_set_t *ts = (iscsi_thread_set_t *) arg;
	iscsi_unmap_sg_t unmap_sg;

	daemonize(ISCSI_TX_THREAD_NAME);
	spin_lock_irq(&current->sighand->siglock);
	siginitsetinv(&current->blocked, ISCSI_SHUTDOWN_SIGS);
	recalc_sigpending();
	ts->tx_thread = current;
	spin_unlock_irq(&current->sighand->siglock);

restart:
	if (!(conn = iscsi_tx_thread_pre_handler(ts)))
		goto out;

	eos = map_sg = ret = scsi_task = use_misc = 0;
	sess = SESS(conn);
	
	while (1) {
		down_interruptible(&conn->tx_sem);

		if ((ts->status == ISCSI_THREAD_SET_RESET) || signal_pending(current))
			goto transport_err;
get_immediate:
		if (conn->conn_flags & CONNFLAG_SEND_NOPOUT) {
			conn->conn_flags &= ~CONNFLAG_SEND_NOPOUT;
			if (iscsi_add_nopout(conn) < 0)
				goto transport_err;
		}
		
		if ((qr = iscsi_get_cmd_from_immediate_queue(conn))) {
			atomic_set(&conn->check_immediate_queue, 0);
			cmd = qr->cmd;
			state = qr->state;

			spin_lock_bh(&cmd->state_lock);
			cmd->cmd_flags |= ICF_CMD_STATE_ACTIVE;
			atomic_dec(&cmd->immed_queue_count);
			kfree(qr);
check_state:
			switch (state) {
			case ISTATE_SEND_DATAOUT:
				spin_unlock_bh(&cmd->state_lock);
				memset((void *)&unmap_sg, 0, sizeof(iscsi_unmap_sg_t));
				unmap_sg.cmd = cmd;
				map_sg = 1;
				ret = iscsi_build_dataout(cmd, conn, &unmap_sg, &eos);
				break;
			case ISTATE_SEND_IMMEDIATE_LOGOUT_REQ:
				spin_unlock_bh(&cmd->state_lock);
				ret = iscsi_build_logout_req(cmd, conn, 1);
				if (ret == 2) {
					cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
					iscsi_release_cmd_to_pool(cmd, sess);
					goto get_immediate;
				} else if (ret == 3) {
					cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
					iscsi_free_command_from_conn(cmd, conn);
					down_interruptible(&conn->conn_logout_sem);
					goto transport_err;
				}
				break;
			case ISTATE_SEND_IMMEDIATE_NOPOUT:
				spin_unlock_bh(&cmd->state_lock);
				iscsi_mod_nopout_response_timer(conn);
				ret = iscsi_build_nopout(cmd, conn, 1);
				break;
			case ISTATE_SEND_IMMEDIATE_TEXT_REQ:
				spin_unlock_bh(&cmd->state_lock);
				ret = iscsi_build_text_req(cmd, conn, 1);
				break;
			case ISTATE_SEND_IMMEDIATE_TMR:
				spin_unlock_bh(&cmd->state_lock);
				ret = iscsi_build_tmr(cmd, conn, 1);
				break;
			case ISTATE_SEND_NOPOUT_RESPONSE:
				spin_unlock_bh(&cmd->state_lock);
				ret = 0;
				break;
			case ISTATE_SEND_RETRY_COMMAND:
				spin_unlock_bh(&cmd->state_lock);
				memset((void *)&unmap_sg, 0, sizeof(iscsi_unmap_sg_t));
				unmap_sg.cmd = cmd;
				if (cmd->pdu[0] & ISCSI_INIT_SCSI_CMND)
					map_sg = 1;
				ret = iscsi_build_retry_command(cmd, conn, &unmap_sg);
				break;
			case ISTATE_REMOVE:
				cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
				spin_unlock_bh(&cmd->state_lock);
				iscsi_free_command_from_conn(cmd, conn);
				goto get_immediate;
			default:
				TRACE_ERROR("Unknown ITT: 0x%08x, state:"
				" %d on CID: %hu\n", cmd->init_task_tag,
					cmd->state, conn->cid);
				cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
				spin_unlock_bh(&cmd->state_lock);
				conn->tx_immediate_queue = 0;
				goto transport_err;
			}

			if (ret < 0) {
				cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
				conn->tx_immediate_queue = 0;
				goto transport_err;
			}

			if (iscsi_send_tx_data(cmd, conn, use_misc) < 0) {
				if (map_sg)
					iscsi_unmap_scatterlists((void *)&unmap_sg);

				cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
				conn->tx_immediate_queue = 0;
				goto transport_err;
			}

			if (map_sg) {
				iscsi_unmap_scatterlists((void *)&unmap_sg);
				map_sg = 0;
			}

			spin_lock_bh(&cmd->state_lock);
			switch (state) {
			case ISTATE_SEND_DATAOUT:
				if (!eos) 
					goto check_state;       
				eos = 0;
				cmd->state = ISTATE_SENT_LAST_DATAOUT;
				break;
			case ISTATE_SEND_IMMEDIATE_LOGOUT_REQ:
				cmd->state = ISTATE_SENT_IMMEDIATE_LOGOUT_REQ;
				if (ret == 1) {
					cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
					spin_unlock_bh(&cmd->state_lock);
					down(&conn->conn_logout_sem);
					goto transport_err;
				}
				break;
			case ISTATE_SEND_IMMEDIATE_NOPOUT:
				cmd->state = ISTATE_SENT_IMMEDIATE_NOPOUT;
				break;
			case ISTATE_SEND_IMMEDIATE_TEXT_REQ:
				cmd->state = ISTATE_SENT_IMMEDIATE_TEXT_REQ;
				break;
			case ISTATE_SEND_IMMEDIATE_TMR:
				cmd->state = ISTATE_SENT_IMMEDIATE_TMR;
				break;
			case ISTATE_SEND_NOPOUT_RESPONSE:
				spin_unlock_bh(&cmd->state_lock);
				iscsi_free_command_from_conn(cmd, conn);
				goto get_immediate;
			case ISTATE_SEND_RETRY_COMMAND:
				cmd->state = ISTATE_SENT_RETRY_COMMAND;
				spin_unlock_bh(&cmd->state_lock);
				iscsi_check_scsi_cmnd_unmap(&unmap_sg);
				iscsi_remove_cmd_from_immediate_queue(cmd, conn);
				if (cmd->unsolicited_data_out) {
					cmd->state = ISTATE_SEND_DATAOUT;
					iscsi_add_cmd_to_immediate_queue(cmd, conn,
						ISTATE_SEND_DATAOUT);
				}
				spin_lock_bh(&cmd->state_lock);
				break;
			default:
				TRACE_ERROR("Unknown ITT: 0x%08x, state:"
				" %d on CID: %hu\n", cmd->init_task_tag,
					cmd->state, conn->cid);
				cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
				spin_unlock_bh(&cmd->state_lock);
				conn->tx_immediate_queue = 0;
				goto transport_err;
			}

			if (atomic_read(&cmd->immed_queue_count)) {
				spin_unlock_bh(&cmd->state_lock);
				goto get_immediate;
			}
			
			cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
			iscsi_cmd_state_check(cmd);
			spin_unlock_bh(&cmd->state_lock);

			goto get_immediate;
		} else
			conn->tx_immediate_queue = 0;

		/*
		 * If the CmdSN/MaxCmdSN Window is open,  build and send next
		 * command in the session queue.
		 */
get_command:
		spin_lock_bh(&sess->cmdsn_lock);
		if (conn->conn_state == INIT_CONN_STATE_IN_LOGOUT) {
			spin_unlock_bh(&sess->cmdsn_lock);
			continue;
		}
		if (serial_gte(sess->max_cmdsn, sess->cur_cmdsn)) {
			if (!(cmd = iscsi_get_cmd_from_session_queue(sess))) {
				spin_unlock_bh(&sess->cmdsn_lock);
				continue;
			}
			/*
			 * Assign the CmdSN before releasing the lock.
			 *
			 * Release iscsi_session_t->cmdsn_lock as soon as
			 * possible so other iSCSI Connections Transmit
			 * Threads can start builting their ordered
			 * iscsi_cmd_t and send to their respective network
			 * interconnections.
			 *
			 * Note that the CmdSN lock is defined as a 'bottom half'
			 * as desribed per the Linux Kernel OS locking rules in
			 * include/[asm-'uname -m',linux]/spinlock.h
			 */
			cmd->cmdsn = sess->cur_cmdsn++;
			iscsi_start_cmdsn_timer(sess, cmd->cmdsn);
			spin_unlock_bh(&sess->cmdsn_lock);

			/* Dont need to hold iscsi_cmd_t->state_lock as with
			 * immediate queue because no other threads of
			 * execution are doing to touch it.
			 */
			switch (cmd->state) {
			case ISTATE_SEND_SCSI_CMD:
				memset((void *)&unmap_sg, 0, sizeof(iscsi_unmap_sg_t));
				unmap_sg.cmd = cmd;
				scsi_task = 1;
				ret = iscsi_build_scsi_cmd(cmd, conn, &unmap_sg);
				break;
			case ISTATE_SEND_LOGOUT_REQ:
				ret = iscsi_build_logout_req(cmd, conn, 0);
				break;
#if 0
			/*
			 * Non-Immediate NOPOUTs are not pratical as they can cause
			 * connection termination due to Header/Digest Failures
			 * on the iSCSI Target Node, and Initiator Node has to wait
			 * for a CmdSN Acknowledgement timeout to occur.
			 */
			case ISTATE_SEND_NOPOUT:
				iscsi_mod_nopout_response_timer(conn);
				ret = iscsi_build_nopout(cmd, conn, 0);
				break;
#endif
			case ISTATE_SEND_TEXT_REQ:
				ret = iscsi_build_text_req(cmd, conn, 0);
				break;
			case ISTATE_SEND_TMR:
				ret = iscsi_build_tmr(cmd, conn, 0);
				break;
			default:
				TRACE_ERROR("Unknown cmd->state: %d\n",
					cmd->state);
				goto transport_err;
			}
			cmd->cmd_flags |= ICF_CMD_STATE_ACTIVE |
					  ICF_CMD_NONIMMEDIATE_ACTIVE;

			iscsi_attach_cmd_to_conn(cmd, conn);
			
			if (ret < 0) {
				cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
				cmd->cmd_flags &= ~ICF_CMD_NONIMMEDIATE_ACTIVE;
				goto transport_err;
			}

			if (scsi_task) {
				iscsi_channel_t *ch = SESS(conn)->channel;
				iscsi_start_scsi_task_timer(cmd, ch);
			}
			
			if (iscsi_send_tx_data(cmd, conn, 0) < 0) {
				if (scsi_task)
					iscsi_check_scsi_cmnd_unmap(&unmap_sg);
				cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
				cmd->cmd_flags &= ~ICF_CMD_NONIMMEDIATE_ACTIVE;
				goto transport_err;
			}

			if (scsi_task) {
				iscsi_check_scsi_cmnd_unmap(&unmap_sg);
				scsi_task = 0;
			}

			spin_lock_bh(&cmd->state_lock);
			switch (cmd->state) {
			case ISTATE_SEND_SCSI_CMD:
				if (!cmd->unsolicited_data_out) {
					cmd->state = ISTATE_SENT_SCSI_CMD;
					break;
				}
				spin_unlock_bh(&cmd->state_lock);
				iscsi_add_cmd_to_immediate_queue(cmd, conn,
					ISTATE_SEND_DATAOUT);
				spin_lock_bh(&cmd->state_lock);
				break;
			case ISTATE_SEND_LOGOUT_REQ:
				if (ret == 1) {
					cmd->cmd_flags &= ~ICF_CMD_NONIMMEDIATE_ACTIVE;
					cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
					spin_unlock_bh(&cmd->state_lock);
					down(&conn->conn_logout_sem);
					goto transport_err;
				}
				cmd->state = ISTATE_SENT_LOGOUT_REQ;
				break;
#if 0
			/*
			 * See above for why sending non-immediate NOPOUTs is disabled.
			 */
			case ISTATE_SEND_NOPOUT:
				cmd->state = ISTATE_SENT_NOPOUT;
				break;
#endif
			case ISTATE_SEND_TEXT_REQ:
				cmd->state = ISTATE_SENT_TEXT_REQ;
				break;
			case ISTATE_SEND_TMR:
				cmd->state = ISTATE_SENT_TMR;
				break;
			default:
				TRACE_ERROR("Unknown cmd->state: %d\n",
					cmd->state);
				cmd->cmd_flags &= ~ICF_CMD_NONIMMEDIATE_ACTIVE;
				cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
				spin_unlock_bh(&cmd->state_lock);
				goto transport_err;
			}
			cmd->cmd_flags &= ~ICF_CMD_NONIMMEDIATE_ACTIVE;
			cmd->cmd_flags &= ~ICF_CMD_STATE_ACTIVE;
			iscsi_cmd_state_check(cmd);
			spin_unlock_bh(&cmd->state_lock);
			
			if (atomic_read(&conn->check_immediate_queue))
				goto get_immediate;
			
			goto get_command;
		}
		spin_unlock_bh(&sess->cmdsn_lock);
	}

transport_err:
	iscsi_take_action_for_connection_exit(conn);
	goto restart;
out:
	ts->tx_thread = NULL;
	up(&ts->tx_done_sem);
	return(0);
}

/*	iscsi_initiator_rx_thread():
 *
 *	Parameters:	iSCSI Connection pointer.
 *	Returns:	Nothing.
 */
extern int iscsi_initiator_rx_thread (void *arg)
{
	int ret;
	u8 buffer[ISCSI_HDR_LEN], opcode;
	u32 checksum, digest;
	iscsi_conn_t *conn = NULL;
	iscsi_thread_set_t *ts = (iscsi_thread_set_t *) arg;
	struct iovec iov;

	daemonize(ISCSI_RX_THREAD_NAME);
	spin_lock_irq(&current->sighand->siglock);
	siginitsetinv(&current->blocked, ISCSI_SHUTDOWN_SIGS);
	recalc_sigpending();
	ts->rx_thread = current;
	spin_unlock_irq(&current->sighand->siglock);
	
restart:
	if (!(conn = iscsi_rx_thread_pre_handler(ts)))
		goto out;

	while (1) {
		memset((void *)buffer, 0, ISCSI_HDR_LEN);
		memset((void *)&iov, 0, sizeof(struct iovec));

		iov.iov_len	= ISCSI_HDR_LEN;
		iov.iov_base	= buffer;

		ret = rx_data(conn, &iov, 1, ISCSI_HDR_LEN);
		if (ret != ISCSI_HDR_LEN)
			goto transport_err;
			
		opcode = buffer[0];
		if (CONN_OPS(conn)->HeaderDigest) {
			iov.iov_len = CRC_LEN;
			iov.iov_base = &digest;

			ret = rx_data(conn, &iov, 1, CRC_LEN);
			if (ret != CRC_LEN)
				goto transport_err;
#ifdef CRYPTO_API_CRC32C
			ISCSI_CRC32C_RX_HDR_RX_PATH(digest, buffer, checksum, conn);
#else
			crc32c(buffer, ISCSI_HDR_LEN, 0x01, &checksum);
#endif
			if (digest != checksum) {
				TRACE_ERROR("HeaderDigest CRC32C failed, received 0x%08x,"
					" computed 0x%08x\n", digest, checksum);
				/*
				 * Set the PDU to 0 so it will intentially
				 * hit default in the switch below.
				 */
				memset((void *)buffer, 0, ISCSI_HDR_LEN);
			} else {
				TRACE(TRACE_DIGEST, "Got HeaderDigest CRC32C"
					" 0x%08x\n", checksum);
			}
		}

		switch (buffer[0] & ISCSI_OPCODE) {
		case ISCSI_TARG_SCSI_DATA_IN:
			ret = iscsi_handle_data_in(conn, buffer);
			break;
		case ISCSI_TARG_SCSI_RSP:
			ret = iscsi_handle_rsp(conn, buffer);
			break;
		case ISCSI_TARG_R2T:
			ret = iscsi_handle_r2t(conn, buffer);
			break;
		case ISCSI_TARG_TASK_MGMT_RSP:
			ret = iscsi_handle_task_mgt_rsp(conn, buffer);
			break;
		case ISCSI_TARG_TEXT_RSP:
			ret = iscsi_handle_text_rsp(conn, buffer);
			break;
		case ISCSI_TARG_NOP_IN:
			ret = iscsi_handle_nop_in(conn, buffer);
			break;
		case ISCSI_TARG_LOGOUT_RSP:
			ret = iscsi_handle_logout_rsp(conn, buffer);
			break;
		case ISCSI_TARG_ASYNC_MSG:
			ret = iscsi_handle_async_msg(conn, buffer);
			break;
		case ISCSI_TARG_RJT:
			ret = iscsi_handle_rjt(conn, buffer);
			break;
		default:
			TRACE_ERROR("Got unknown iSCSI OpCode 0x%02x\n", buffer[0]);
			if (!SESS_OPS_C(conn)->ErrorRecoveryLevel) {
				TRACE_ERROR("Cannot recover from unknown opcode"
				" while ERL=0, closing iSCSI connection.\n");
				goto transport_err;
			}
			if (!CONN_OPS(conn)->IFMarker) {
				TRACE_ERROR("Unable to recover from unknown"
				" opcode while IFMarker=No, closing iSCSI"
					" connection.\n");
				goto transport_err;
			}
			ret = -1;
			break;
		}

		if (ret < 0)
			goto transport_err;
		else if (ret > 0)
			goto restart;
	}
	
transport_err:
	if (!signal_pending(current))
		atomic_set(&conn->transport_failed, 1);
	iscsi_take_action_for_connection_exit(conn);
	goto restart;
out:
	ts->rx_thread = NULL;
	up(&ts->rx_done_sem);
	return(0);
}

/*	iscsi_init_connection():
 *
 *
 *	Parameters:	iSCSI Connection pointer, iSCSI Session pointer
 *	Returns:	Always 0
 */
static int iscsi_init_connection (
	iscsi_conn_t *conn,
	iscsi_channel_conn_t *cc,
	iscsi_session_t *sess)
{
	conn->session		= sess;
	conn->exp_statsn	= 0;
	init_MUTEX_LOCKED(&conn->conn_logout_sem);
	init_MUTEX_LOCKED(&conn->conn_waiting_on_uc_sem);
	init_MUTEX_LOCKED(&conn->conn_wait_sem);
	init_MUTEX_LOCKED(&conn->tx_sem);
	spin_lock_init(&conn->cmd_lock);
	spin_lock_init(&conn->conn_usage_lock);
	spin_lock_init(&conn->immed_queue_lock);
	spin_lock_init(&conn->netif_lock);
	spin_lock_init(&conn->nopout_timer_lock);
	spin_lock_init(&conn->state_lock);

	TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_FREE.\n");
	conn->conn_state	= INIT_CONN_STATE_FREE;

	if (!(conn->conn_ops = (iscsi_conn_ops_t *) kmalloc(
			sizeof(iscsi_conn_ops_t), GFP_ATOMIC))) {
		TRACE_ERROR("Unable to allocate memory for iscsi_conn_ops_t.\n");
		goto out;
	}
	memset(conn->conn_ops, 0, sizeof(iscsi_conn_ops_t));

	if (cc)
		memcpy(conn->conn_ops, (void *)&cc->conn_ops, sizeof(iscsi_conn_ops_t));
	
#ifdef CRYPTO_API_CRC32C
	if (!(conn->conn_rx_tfm = crypto_alloc_tfm("crc32c", 0))) {
		TRACE_ERROR("Unable to allocate RX crc32c tfm\n");
		goto out;
	}

	if (!(conn->conn_tx_tfm = crypto_alloc_tfm("crc32c", 0))) {
		TRACE_ERROR("Unable to allocate TX crc32c tfm\n");
		goto out;
	}

	if (!(conn->conn_rx_tfm_sg = (struct scatterlist *) kmalloc(
			sizeof(struct scatterlist), GFP_KERNEL))) {
		TRACE_ERROR("kmalloc for conn_rx_tfm_sg failed\n");
		goto out;
	}

	if (!(conn->conn_tx_tfm_sg = (struct scatterlist *) kmalloc(
			sizeof(struct scatterlist), GFP_KERNEL))) {
		TRACE_ERROR("kmalloc for conn_tx_tfm_sg failed\n");
		goto out;
	}
#endif
	return(0);
out:
	if (conn->conn_ops)
		kfree(conn->conn_ops);
#ifdef CRYPTO_API_CRC32C
	if (conn->conn_rx_tfm)
		crypto_free_tfm(conn->conn_rx_tfm);
	if (conn->conn_tx_tfm)
		crypto_free_tfm(conn->conn_tx_tfm);
	if (conn->conn_rx_tfm_sg)
		kfree(conn->conn_rx_tfm_sg);
	if (conn->conn_tx_tfm_sg)
		kfree(conn->conn_tx_tfm_sg);
#endif

	return(-1);
} 

/*	iscsi_init_session():
 *
 *	This function is called from iscsi_create_session() and fills in the
 *	default values for the session struct, as well as making a copy of
 *	the global parameter list for the session to use.
 *
 *	Parameters:	iSCSI Session pointer, iSCSI Login Holder Pointer.
 *	Returns:	0 on success, -1 on error
 */
static int iscsi_init_session (
	iscsi_session_t *sess,
	iscsi_login_holder_t *lh)
{
	iscsi_channel_t *c = lh->channel;
	
	TRACE(TRACE_STATE, "Moving to INIT_SESS_STATE_FREE.\n");
	sess->session_state		= INIT_SESS_STATE_FREE;

	if (!c)
		return(-1);
	
	sess->channel			= c;
	sess->cur_task_tag		= 1;
	sess->cur_cmdsn			= 1;
	sess->exp_cmdsn			= 1;
	sess->max_cmdsn			= 0xFFFFFFFF;
	init_MUTEX_LOCKED(&sess->discovery_sem);
	init_MUTEX_LOCKED(&sess->session_wait_sem);
	init_MUTEX_LOCKED(&sess->session_waiting_on_uc_sem);
	init_MUTEX_LOCKED(&sess->stop_reinstatement_sem);
	spin_lock_init(&sess->cmdsn_lock);
	spin_lock_init(&sess->conn_lock);
	spin_lock_init(&sess->conn_schedule_lock);
	spin_lock_init(&sess->itt_lock);
	spin_lock_init(&sess->la_lock);
	spin_lock_init(&sess->pending_lock);
	spin_lock_init(&sess->pool_lock);
	spin_lock_init(&sess->reinstatement_lock);
	spin_lock_init(&sess->session_usage_lock);

	if (!(sess->sess_ops = (iscsi_sess_ops_t *) kmalloc(
			sizeof(iscsi_sess_ops_t), GFP_ATOMIC))) {
		TRACE_ERROR("Unable to allocate memory for iscsi_sess_ops_t.\n");
		return(-1);
	}
	memset(sess->sess_ops, 0, sizeof(iscsi_sess_ops_t));

	if (lh->cc)
		memcpy(sess->sess_ops, (void *)&lh->cc->sess_ops, sizeof(iscsi_sess_ops_t));
	
	return(0);
}

/*	iscsi_create_connection():
 *
 *	Parameters:	iSCSI session Pointer,  iSCSI Login Holder Pointer,
 *			CID Pointer.
 *	Returns:	0 on success, -1 on error.
 */
extern int iscsi_create_connection (
	iscsi_session_t *sess,
	iscsi_login_holder_t *lh,
	u16 *cid)
{
	int ret = -1;
	iscsi_channel_t *c = lh->channel;
	iscsi_conn_t *conn = NULL;

	if ((atomic_read(&sess->nconn) + atomic_read(&c->connection_count)) >
	     ISCSI_MAX_CONNECTIONS) {
		TRACE_ERROR("Reached ISCSI_MAX_CONNECTIONS: %d on Channel: %d\n",
			ISCSI_MAX_CONNECTIONS, c->channel_id);
		goto out;
	}
	
	if (!(conn = (iscsi_conn_t *) kmalloc(sizeof(iscsi_conn_t), GFP_ATOMIC))) {
		TRACE_ERROR("Could not allocate memory for iscsi_conn_t.\n");
		goto out;
	}
	memset(conn, 0, sizeof(iscsi_conn_t));

	if (iscsi_init_connection(conn, lh->cc, sess) < 0)
		goto out;

	conn->session		= sess;
	conn->auth_id		= iscsi_global->auth_id++;
	conn->login_ip		= lh->ipv4_address;
	conn->login_port	= lh->port;
	spin_lock_bh(&sess->conn_lock);
	conn->cid		= (cid) ? *cid : sess->cid_counter++;
	spin_unlock_bh(&sess->conn_lock);
	strncpy(conn->net_dev, lh->net_dev, strlen(lh->net_dev));
	
	if ((ret = iscsi_initiator_login(conn, lh)) < 0)
		goto out;
	
	return(0);
out:
	if (conn)
		kfree(conn);
	return(ret);
}


/*	iscsi_create_session():
 *
 *	Parameters:	iSCSI Login Holder Pointer.,
 *	Returns:	Newly assigned PyX SID on success, 0 on error
 */
extern u32 iscsi_create_session (iscsi_login_holder_t *lh)
{
	iscsi_session_t *sess = NULL;

	if (!iscsi_global->initname_set) {
		TRACE_ERROR("iSCSI InitiatorName is not"
			" set! Please set it now!\n");
		goto err_out;
	}

	if (!(sess = (iscsi_session_t *) kmalloc(
			sizeof(iscsi_session_t), GFP_ATOMIC))) {
		TRACE_ERROR("Could not allocate memory for iscsi_session_t.\n");
		goto err_out;
	}
	memset(sess, 0, sizeof(iscsi_session_t));

	if (iscsi_init_session(sess, lh) < 0)
		goto err_out;

	lh->leading_conn = lh->tpg_failover = 1;
	
	if (iscsi_create_connection(sess, lh, NULL) < 0)
		goto err_out;
		
	return(sess->sid);

err_out:
	if (sess)
		kfree(sess);
	return(0);
}

/*	iscsi_check_for_connection_reinstatement_wait():
 *
 *
 */
static void iscsi_check_for_connection_reinstatement_wait (iscsi_conn_t *conn)
{
	/*
	 * iscsi_cause_connection_reinstatement() sleeps on this semaphore.
	 */
	spin_lock_bh(&conn->state_lock);
	if (atomic_read(&conn->sleep_on_conn_wait_sem))
		up(&conn->conn_wait_sem);
	else
		atomic_set(&conn->sleep_on_conn_wait_sem, 1);
	spin_unlock_bh(&conn->state_lock);
	
	return;
}

/*	iscsi_close_connection():
 *
 *
 */
extern int iscsi_close_connection (
	iscsi_conn_t *conn)
{
	int ret = 0;
	int conn_logout = (conn->conn_state == INIT_CONN_STATE_IN_LOGOUT);
	iscsi_session_t *sess = SESS(conn);
	iscsi_channel_t *c = sess->channel;
	iscsi_channel_conn_t *cc = NULL;
	
	TRACE(TRACE_ISCSI, "Closing iSCSI connection CID %hu on SID:"
		" %u\n", conn->cid, sess->sid);

	iscsi_stop_netif_timer(conn);

	/*
	 * Always up conn_logout_sem just in case the TX Thread is sleeping
	 * and the logout response never got received because the connection
	 * failed.
	 */
	up(&conn->conn_logout_sem);

	iscsi_check_for_connection_reinstatement_wait(conn);
	iscsi_release_thread_set(conn);

	iscsi_stop_datain_timers_for_cmds(conn);
	iscsi_stop_nopout_response_timer(conn);
	iscsi_stop_nopout_timer(conn);

	/*
	 * Add a iscsi_chan_conn_t entry to the channel list in case
	 * we must fall back to Session Reinstatement later.
	 */
	if (!SESS_OPS(sess)->SessionType &&
	    (conn->conn_state != INIT_CONN_STATE_IN_LOGOUT))
		if (!(cc = iscsi_add_conn_to_channel_for_reinstatement(conn)))
			ret = -1;

	/*
	 * If a CLOSECONNECTION Logout was successful for this iSCSI Connection,
	 * complete the OS dependant SCSI commands with a retry status, otherwise
	 * add the OS dependant SCSI commands to the iscsi_channel_t to be
	 * completed later upon Session Reinstatement.  Both of the above
	 * clear the iscsi_cmd_t->scsi_cmnd pointer.
	 *
	 * For all iSCSI Session Types, finally free the iscsi_cmd_t list.
	 *
	 * iSCSI Sessions used for Discovery will never have any
	 * assoicated SCSI tasks.
	 */
	if (SESS_OPS_C(conn)->SessionType)
		goto free_cmds;
	
	/*
	 * The 3 (Three) different cases for non connection recovery cases.
	 * 
	 * Case A:  We are forcing the entire iSCSI Channel Offline.
	 * Case B:  We are retrying SCSI Commands due to a iSCSI
	 *          Connection Logout.
	 * Case C:  An ERL=0 exception occured and we are adding the passed
	 *          iscsi_conn_t's SCSI Commands to the iSCSI Channel wide
	 *          iSCSI Command Pool of iscsi_cmd_t's to be retried after
	 *          iSCSI Session Reinstatement/Recovery is complete.
	 */
	if (atomic_read(&c->force_channel_offline))
		iscsi_set_conn_scsi_cmnd_results(conn, ISCSI_COMMAND_FAIL);
	else if (conn->conn_state == INIT_CONN_STATE_IN_LOGOUT)
		iscsi_set_conn_scsi_cmnd_results(conn, ISCSI_COMMAND_RETRY);
	else
		iscsi_move_conn_scsi_cmnd_to_channel(conn, c);
free_cmds:
	iscsi_free_all_commands_for_conn(conn);
	
	spin_lock_bh(&sess->conn_lock);
	iscsi_remove_conn_from_list(sess, conn);
	spin_unlock_bh(&sess->conn_lock);

	/*
	 * If any other processes are accessing this connection pointer we
	 * must wait they have completed.
	 */
	iscsi_check_conn_usage_count(conn);

#ifdef CRYPTO_API_CRC32C
	if (conn->conn_rx_tfm_sg)
		kfree(conn->conn_rx_tfm_sg);
	if (conn->conn_tx_tfm_sg)
		kfree(conn->conn_tx_tfm_sg);
	if (conn->conn_rx_tfm)
		crypto_free_tfm(conn->conn_rx_tfm);
	if (conn->conn_tx_tfm)
		crypto_free_tfm(conn->conn_tx_tfm);
#endif
	iscsi_sysfs_unregister_conn_attributes(conn, sess);	

	if (!ret && cc) {
		if (conn->conn_ops) {
			memcpy((void *)&cc->conn_ops, conn->conn_ops, sizeof(iscsi_conn_ops_t));
			kfree(conn->conn_ops);
			conn->conn_ops = NULL;
		}
		if (sess->sess_ops)
			memcpy((void *)&cc->sess_ops, sess->sess_ops, sizeof(iscsi_sess_ops_t));
	} else if (conn->conn_ops) {
		kfree(conn->conn_ops);
		conn->conn_ops = NULL;
	}

	if (conn->param_list) {
		iscsi_release_param_list(conn->param_list);
		conn->param_list = NULL;
	}

	if (conn->sock) {
		if (conn->conn_flags & CONNFLAG_SCTP_STRUCT_FILE) {
			kfree(conn->sock->file);
			conn->sock->file = NULL;
		}

		sock_release(conn->sock);
	}

	spin_lock_bh(&sess->conn_lock);
	TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_FREE\n");
	conn->conn_state = INIT_CONN_STATE_FREE;

	atomic_dec(&sess->nconn);
	printk("iCHANNEL[%d] - Decremented iSCSI connection count to %hu to node: %s\n",
		c->channel_id, atomic_read(&sess->nconn), SESS_OPS(sess)->TargetName);

	kfree(conn);

	/*
	 * Do not set the session reinstatement bit in the following cases:
	 *
	 * A) User is requesting connection logout on this connection.
	 * B) User is requesting session logout on all connections.
	 */
	if (!conn_logout && atomic_read(&sess->session_logout))
		atomic_set(&sess->session_reinstatement, 1);
	
	/*
	 * If this was not the last connection and the session is
	 * in reinstatement, force any other active connections
	 * to shutdown.
	 */
	if (atomic_read(&sess->nconn)) {
		if (!atomic_read(&sess->session_reinstatement)) {
			spin_unlock_bh(&sess->conn_lock);
			return(ret);
		}
		if (!atomic_read(&sess->session_stop_active)) {
			atomic_set(&sess->session_stop_active, 1);
			spin_unlock_bh(&sess->conn_lock);
			iscsi_stop_session(sess, 0, 0);
			return(ret);
		}
		spin_unlock_bh(&sess->conn_lock);
		return(ret);
	}

	/*
	 * This was the last connection in the session, change the
	 * state to failed if we are not performing Session Logout
	 * or Reinstatement.
	 */
	if (!atomic_read(&sess->session_logout) &&
	    !atomic_read(&sess->session_reinstatement)) {
		sess->session_state = INIT_SESS_STATE_FAILED;
		TRACE(TRACE_STATE, "Moving to INIT_SESS_STATE_FAILED.\n");
	}

	/*
	 * Check if iscsi_free_session() or iscsi_stop_session() is sleeping.
	 */
	if (atomic_read(&sess->sleep_on_sess_wait_sem))
		up(&sess->session_wait_sem);

	spin_unlock_bh(&sess->conn_lock);
	
	/*
	 * Returns TRUE if the iSCSI Connection shutdown process finished
	 * with no major exception.
	 */
	return((!ret) ? 1 : ret);
}

/*	iscsi_close_session():
 *	
 *
 */
extern int iscsi_close_session (
	iscsi_session_t *sess)
{
	iscsi_channel_t *channel = sess->channel;
	
	iscsi_stop_cmdsn_timer(sess);
	
	if (atomic_read(&sess->nconn)) {
		TRACE_ERROR("%d connection(s) still exist for iSCSI session"
			" SID: %u\n", atomic_read(&sess->nconn), sess->sid);
		BUG();
	}

	if (iscsi_remove_session_from_list(sess) < 0)
		BUG();

	/*
	 * Skip all handling of SCSI tasks in a Discovery Session.
	 */
	if (SESS_OPS(sess)->SessionType)
		goto free_cmds;

	/*
	 * Stop all SCSI access to the iSCSI Logical Units now that the
	 * IT Nexus is broken.
	 */
	iscsi_scsi_block_devices(channel);
	
	/*
	 * Process the iSCSI Channel Status Now.
	 */
	if (atomic_read(&channel->force_channel_offline)) {
		iscsi_set_channel_status(channel, ISCSI_CHANNEL_FAILED,
				SCS_RELEASE_LUNS | SCS_FAIL_OS_SCSI_CMDS);
	} else if (atomic_read(&channel->stop_channel)) {
		iscsi_set_channel_status(channel, ISCSI_CHANNEL_REESTABLISHING,
				SCS_STOP_REINSTATEMENT_THREAD);
	} else if (atomic_read(&sess->session_reinstatement)) {
		iscsi_set_channel_status(channel, ISCSI_CHANNEL_REESTABLISHING,
				SCS_STOP_REINSTATEMENT_THREAD_AND_SET);
	} else {
		iscsi_set_channel_status(channel, ISCSI_CHANNEL_SESSION_LOGGED_OUT,
				SCS_FREE_CHANNEL_CONNS);
	}
	
	/*
	 * If any other processes are accessing this session pointer we
	 * must wait they have completed.
	 */
	iscsi_check_session_usage_count(sess);
	
	/*
	 * For all Normal sessions:
	 *
	 * Move all commands from the pending list that have not been assigned
	 * to a connection to the channel's command list for ERL=0 recovery.
	 */
	iscsi_move_sess_scsi_cmnd_to_channel(sess, channel);
	
	/*
	 * Now that all outstanding struct scsi_cmnds for this iSCSI Channel have
	 * been put into its channel queue, as well as the SCSI devices having
	 * their queues blocked.  Its safe to complete said struct scsi_cmnds to
	 * the SCSI layer as they will be retried once the IT nexus is reestablished
	 * and SCSI devices unblocked.
	 */
	iscsi_set_scsi_cmnd_results_for_channel(channel,
		(atomic_read(&channel->force_channel_offline)) ?
		 	ISCSI_COMMAND_FAIL : ISCSI_COMMAND_RETRY);

	/*
	 * At this point for normal sessions all OS dependant SCSI pointers
	 * have been moved to the channel list and iscsi_cmd_t->scsi_cmnd'es set to
	 * NULL.  The following two functions will release the iscsi_cmd_t's
	 * back into the session pool, finally iscsi_release_all_cmds_in_pool()
	 * frees all the allocated iscsi_cmd_ts.
	 */
free_cmds:
	iscsi_free_all_commands_for_sess(sess);
	iscsi_release_all_cmds_in_pool(sess);

	/*
	 * If any other processes are accessing this session pointer we
	 * must wait they have completed.  (Only for discovery, for normal
	 * session we call iscsi_check_session_usage_count() above..
	 */
	if (SESS_OPS(sess)->SessionType)
		iscsi_check_session_usage_count(sess);

	iscsi_stop_session(sess, 1, 1);

	if (atomic_read(&channel->force_channel_offline))
		up(&channel->force_offline_sem);

	iscsi_sysfs_unregister_session_attributes(sess, channel);

	/*
	 * Its now safe to clear the iSCSI session's assoicated 
	 * iscsi_channel_t->session for Normal (non Discovery) iSCSI
	 * Sessions.
	 */
	if (!SESS_OPS(sess)->SessionType)
		channel->sess = NULL;
	
	spin_lock(&iscsi_global->session_lock);
	TRACE(TRACE_STATE, "Moving to INIT_SESS_STATE_FREE\n");
	sess->session_state = INIT_SESS_STATE_FREE;

	printk("iCHANNEL[%d] - released iSCSI session to node: %s\n",
		channel->channel_id, SESS_OPS(sess)->TargetName);

	iscsi_global->nsess--;
	printk("iSCSI Core Stack[1] - Decremented number of active iSCSI sessions"
			" to %u\n", iscsi_global->nsess);
	spin_unlock(&iscsi_global->session_lock);
	
	if (sess->sess_ops) {
		kfree(sess->sess_ops);
		sess->sess_ops = NULL;
	}
	
	kfree(sess);
	sess = NULL;
	
	return(0);
}

/*	iscsi_free_session():
 *
 *
 */
extern int iscsi_free_session (iscsi_session_t *sess)
{
	int i = 0, j = 0;
	iscsi_conn_t *conn, *conn_next = NULL;
	iscsi_conn_t *conn_array[ISCSI_MAX_CONNECTIONS];
	
	spin_lock_bh(&sess->conn_lock);	
	sess->no_connection_reinstatement = 1;
	atomic_set(&sess->sleep_on_sess_wait_sem, 1);

	memset(conn_array, 0, (sizeof(iscsi_conn_t *) * ISCSI_MAX_CONNECTIONS));

	conn = sess->conn_head;
	while (conn) {
		conn_next = conn->next;
		__iscsi_inc_conn_usage_count(conn);
		conn_array[i++] = conn;
		conn = conn_next;
	}
	
	for (; i > 0; i--) {
		conn = conn_array[j++];
		spin_unlock_bh(&sess->conn_lock);
		iscsi_stop_connection_and_reinstatement(conn);
		spin_lock_bh(&sess->conn_lock);
		
		__iscsi_dec_conn_usage_count(conn);
	}
	kfree(conn_array);

	if (atomic_read(&sess->nconn)) {
		spin_unlock_bh(&sess->conn_lock);
		down(&sess->session_wait_sem);
	} else
		spin_unlock_bh(&sess->conn_lock);

	iscsi_close_session(sess);

	return(0);
}

/*	iscsi_stop_session():
 *
 *
 */
extern int iscsi_stop_session (iscsi_session_t *sess, int session_sleep, int connection_sleep)
{
	int i = 0, j = 0;
	iscsi_conn_t *conn, *conn_next = NULL;
	iscsi_conn_t *conn_array[ISCSI_MAX_CONNECTIONS];

	spin_lock_bh(&sess->conn_lock);
	sess->no_connection_reinstatement = 1;
	if (session_sleep)
		atomic_set(&sess->sleep_on_sess_wait_sem, 1);

	if (connection_sleep) {
		memset(conn_array, 0, (sizeof(iscsi_conn_t *) * ISCSI_MAX_CONNECTIONS));

		conn = sess->conn_head;
		while (conn) {
			conn_next = conn->next;
			__iscsi_inc_conn_usage_count(conn);
			conn_array[i++] = conn;
			conn = conn_next;
		}
		
		for (; i > 0; i--) {
			conn = conn_array[j++];
			spin_unlock_bh(&sess->conn_lock);
			iscsi_cause_connection_reinstatement(conn, 1);
			spin_lock_bh(&sess->conn_lock);
			
			__iscsi_dec_conn_usage_count(conn);
		}
		
	} else {
		conn = sess->conn_head;
		while (conn) {
			conn_next = conn->next;
			iscsi_cause_connection_reinstatement(conn, 0);
			conn = conn_next;
		}
	}
	
	if (session_sleep && atomic_read(&sess->nconn)) {
		spin_unlock_bh(&sess->conn_lock);
		down(&sess->session_wait_sem);
	} else
		spin_unlock_bh(&sess->conn_lock);
		
	return(0);
}

/*	iscsi_set_state_for_connections():
 *
 *
 */
extern void iscsi_set_state_for_connections (iscsi_session_t *sess, u8 conn_state)
{
	iscsi_conn_t *conn;

	spin_lock_bh(&sess->conn_lock);
	for (conn = sess->conn_head; conn; conn = conn->next) {
		spin_lock(&conn->state_lock);
		if (conn->conn_state == conn_state) {
			spin_unlock(&conn->state_lock);
			continue;
		}
		switch (conn_state) {
		case INIT_CONN_STATE_IN_LOGOUT:
			TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_IN_LOGOUT.\n");
			break;
		default:
			spin_unlock(&conn->state_lock);
			spin_unlock_bh(&sess->conn_lock);
			TRACE_ERROR("Unknown conn_state: %d\n", conn_state);
			return;
		}
		conn->conn_state = conn_state;
		spin_unlock(&conn->state_lock);
	}
	spin_unlock_bh(&sess->conn_lock);

	return;
}

/*      iscsi_no_active_connections_logout():
 *
 *
 */
static int iscsi_no_active_connections_logout (iscsi_session_t *sess)
{
	if (SESS_OPS(sess)->SessionType)
		return(-1);

	if (SESS_OPS(sess)->ErrorRecoveryLevel != 2)
		return(-1);

	iscsi_stop_session_and_reinstatement(sess, 1);
	
	return(-2);
}

/*	iscsi_start_immediate_logout():
 *
 *
 */
static iscsi_cmd_t *iscsi_start_immediate_logout (
	iscsi_session_t *sess,
	iscsi_logout_attempt_t *la)
{
	int session_type = SESS_OPS(sess)->SessionType;
	iscsi_channel_t *channel = sess->channel;
	iscsi_cmd_t *cmd;
	iscsi_conn_t *conn, *on_conn;
	
	if (la->logout_reason == CLOSESESSION) {
		if (!(on_conn = iscsi_get_conn(sess)))
			return(NULL);

		conn = on_conn;
		la->logout_cid = la->logout_on_cid = on_conn->cid;
		
		if (!session_type && la->logout_lun_remove) {
			/*
			 * If the LUN_REMOVE operation was not forced, check to
			 * see if any iSCSI Logical Units are currently in user
			 * by the SCSI subsystem.
			 */

			if (!la->logout_force) {
				if (iscsi_check_active_iSCSI_LUNs(channel, 0) < 0)
					goto error_out;
			}

			iscsi_release_logical_units(channel);

		}

		atomic_set(&sess->session_logout, 1);
		iscsi_stop_channel_reinstatement_process(channel, 0);
	} else if (la->logout_reason == CLOSECONNECTION) {
		if (la->logout_cid == la->logout_on_cid) {
			if (!(conn = iscsi_get_conn_from_cid(sess,
					la->logout_cid))) {
				TRACE_ERROR("Unable to locate active iSCSI"
					" Connection with CID: %hu\n",
						la->logout_cid);
				return(NULL);
			}
			on_conn = conn;
		} else {
			if (!(on_conn = iscsi_get_conn_from_cid(sess,
					la->logout_on_cid))) {
				TRACE_ERROR("Unable to locate active iSCSI"
					" Connection with CID: %hu\n",
						la->logout_on_cid);
				return(NULL);
			}
			if (!(conn = iscsi_get_conn_from_cid(sess,
					la->logout_cid))) {
				iscsi_dec_conn_usage_count(on_conn);
				TRACE_ERROR("Unable to locate active iSCSI"
					" Connection with CID: %hu\n",
						la->logout_cid);
				return(NULL);
			}
		}

		if (!session_type && (atomic_read(&sess->nconn) == 1) &&
		    la->logout_lun_remove) {
			/*
			 * If the LUN_REMOVE operation was not forced, check to
			 * see if any iSCSI Logical Units are currently in user
			 * by the SCSI subsystem.
			 */
			if (!la->logout_force) {
				if (iscsi_check_active_iSCSI_LUNs(channel, 0) < 0)
					goto error_out;
			}
			
			iscsi_release_logical_units(channel);
		}
		
		if (atomic_read(&sess->nconn) == 1) {
			atomic_set(&sess->session_logout, 1);
			iscsi_stop_channel_reinstatement_process(channel, 0);
		}
	} else {
		TRACE_ERROR("Unknown logout reason code 0x%02x, ignoring"
			" request.\n", la->logout_reason);
		return(NULL);
	}

	la->logout_conn		= conn;
	la->logout_on_conn	= on_conn;
	
	if (la->logout_reason == CLOSESESSION)
		iscsi_set_state_for_connections(sess, INIT_CONN_STATE_IN_LOGOUT);
	else if (la->logout_reason == CLOSECONNECTION) {
		TRACE(TRACE_STATE, "Moving to INIT_CONN_STATE_IN_LOGOUT.\n");
		spin_lock_bh(&conn->state_lock);
		conn->conn_state = INIT_CONN_STATE_IN_LOGOUT;
		spin_unlock_bh(&conn->state_lock);
	}

	if (!(cmd = iscsi_allocate_cmd(sess, NULL)))
		goto error_out;

	cmd->la = la;
	cmd->pdu[0] = ISCSI_INIT_LOGOUT_CMND | I_BIT;
	cmd->state = ISTATE_SEND_IMMEDIATE_LOGOUT_REQ;
	iscsi_attach_cmd_to_conn(cmd, conn);

	spin_lock(&la->logout_state_lock);
	la->logout_built = 1;
	spin_unlock(&la->logout_state_lock);
	iscsi_add_cmd_to_immediate_queue(cmd, conn, cmd->state);
	
	return(cmd);

error_out:
	if (la->logout_cid != la->logout_on_cid)
		iscsi_dec_conn_usage_count(on_conn);
	iscsi_dec_conn_usage_count(conn);
	return(NULL);
}

/*	iscsi_start_non_immediate_logout():
 *
 *
 */
static iscsi_cmd_t *iscsi_start_non_immediate_logout (
	iscsi_session_t *sess,
	iscsi_logout_attempt_t *la)
{
	int session_type = SESS_OPS(sess)->SessionType;
	iscsi_channel_t *channel = sess->channel;
	iscsi_cmd_t *cmd;
	iscsi_conn_t *conn = NULL;

	if (la->logout_reason == CLOSESESSION) {
		if (!session_type && la->logout_lun_remove) {
			/*
			 * If the LUN_REMOVE operation was not forced, check to
			 * see if any iSCSI Logical Units are currently in user
			 * by the SCSI subsystem.
			 */
			if (!la->logout_force) {
				if (iscsi_check_active_iSCSI_LUNs(channel, 0) < 0)
					return(NULL);
			}

			iscsi_release_logical_units(channel);
		}
		
		atomic_set(&sess->session_logout, 1);
	} else if (la->logout_reason == CLOSECONNECTION) {
		if (!session_type && (atomic_read(&sess->nconn) == 1) &&
		     la->logout_lun_remove) {
			/*
			 * If the LUN_REMOVE operation was not forced, check to
			 * see if any iSCSI Logical Units are currently in user
			 * by the SCSI subsystem.
			 */
			if (!la->logout_force) {
				if (iscsi_check_active_iSCSI_LUNs(channel, 0) < 0)
					return(NULL);
			}
			
			iscsi_release_logical_units(channel);
		}
		
		if (!(conn = iscsi_get_conn_from_cid(sess, la->logout_cid))) {
			TRACE_ERROR("Unable to locate active iSCSI Connection"
				" with CID: %hu\n", la->logout_cid);
			return(NULL);
		}
		la->logout_conn = conn;

		if (atomic_read(&sess->nconn) == 1)
			atomic_set(&sess->session_logout, 1);
	} else {
		TRACE_ERROR("Unknown logout reason code 0x%02x, ignoring"
			" request.\n", la->logout_reason);
		return(NULL);
	}

	if (!(cmd = iscsi_allocate_cmd(sess, NULL)))
		goto error_out;

	cmd->la = la;
	cmd->state = ISTATE_SEND_LOGOUT_REQ;
	iscsi_non_scsi_queue_cmd(cmd, sess);
	
	return(cmd);

error_out:
	if (la->logout_reason == CLOSECONNECTION)
		iscsi_dec_conn_usage_count(conn);
	return(NULL);
}

/*	iscsi_post_logout_got_response():
 *
 *
 */
static int iscsi_post_logout_got_response (
	iscsi_logout_attempt_t *la,
	iscsi_session_t *sess)
{
	int ret;
	
	if (la->logout_reason == CLOSESESSION) {
		iscsi_dec_conn_usage_count(la->logout_conn);
	} else {
		if (la->logout_cid != la->logout_on_cid)
			iscsi_dec_conn_usage_count(la->logout_on_conn);
		else
			iscsi_dec_conn_usage_count(la->logout_conn);
	}

	ret = (la->logout_response == CONNORSESSCLOSEDSUCCESSFULLY) ? 0 : -1;
	
	up(&la->logout_done_sem);

	return(ret);
}

/*	iscsi_check_expired_logout_conn():
 *
 *
 */
static int iscsi_check_expired_logout_conn (
	iscsi_logout_attempt_t *la)
{
	int ret = 0;
	
	spin_lock(&la->logout_state_lock);
	if (la->logout_immediate || la->logout_built)
		ret = 1;
	spin_unlock(&la->logout_state_lock);
		
	return(ret);
}

/*	iscsi_post_logout_expired_response():
 *
 *
 */
static int iscsi_post_logout_expired_response (
	iscsi_logout_attempt_t *la,
	iscsi_session_t *sess)
{
	int ret = 0;
	iscsi_channel_t *c = sess->channel;
	
	if (la->logout_reason == CLOSESESSION) {
		TRACE_ERROR("Logout Response Timeout for CLOSESESSION Logout"
			" Request on CID: %hu to node: %s\n", la->logout_cid,
				SESS_OPS(sess)->TargetName);
		
		if (iscsi_check_expired_logout_conn(la))
			iscsi_dec_conn_usage_count(la->logout_conn);

		iscsi_stop_session_and_reinstatement(sess, 1);
		ret = -2;
	} else {
		TRACE_ERROR("Logout Response Timeout for CLOSECONNECTION"
			" Logout Request for CID: %hu on CID: %hu to node:"
			" %s\n", la->logout_cid, la->logout_on_cid,
				SESS_OPS(sess)->TargetName);

		if ((la->logout_cid != la->logout_on_cid) &&
		    (iscsi_check_expired_logout_conn(la)))
			iscsi_dec_conn_usage_count(la->logout_on_conn);
		if (iscsi_check_expired_logout_conn(la))
			iscsi_dec_conn_usage_count(la->logout_conn);
		
		if (ISCSI_CA(c)->closeconn_reinstatement) {
			if (atomic_read(&sess->session_logout)) {
				iscsi_stop_session_and_reinstatement(sess, 1);
				ret = -2;
			} else {
				atomic_set(&sess->session_reinstatement, 1);
				iscsi_cause_session_reinstatement(sess);
				ret = -1;
			}
		} else {
			iscsi_stop_session_and_reinstatement(sess, 1);
			ret = -2;
		}
	}

	kfree(la);
	
	return(ret);
}	

/*	iscsi_post_logout_handler():
 *
 *
 */
static int iscsi_post_logout_handler (
	iscsi_logout_attempt_t *la,
	iscsi_session_t *sess)
{
	spin_lock(&sess->la_lock);
	if (atomic_read(&la->logout_got_response)) {
		spin_unlock(&sess->la_lock);
		return(iscsi_post_logout_got_response(la, sess));
	}
	atomic_set(&la->logout_timer_expired, 1);
	spin_unlock(&sess->la_lock);

	return(iscsi_post_logout_expired_response(la, sess));
}

/*	logout_timer_function():
 *
 *
 */
static void logout_timer_function (unsigned long data)
{
	up((struct semaphore *)data);
	return;
}

/*	iscsi_start_logout():
 *
 *
 */
extern int iscsi_start_logout (
	iscsi_session_t *sess,
	int lun_remove,
	int async_logout,
	u8 reason,
	u16 cid,
	u16 on_cid)
{
	iscsi_channel_t *channel = sess->channel;
	iscsi_cmd_t *cmd = NULL;
	iscsi_logout_attempt_t *la = NULL;
	struct timer_list logout_timer;
	
	if (atomic_read(&sess->session_logout)) {
		TRACE_ERROR("iSCSI Session ID: %u already preforming logout,"
			" ignoring request.\n", sess->sid);
		return(-1);
	}

	spin_lock_bh(&sess->conn_lock);
	if (!atomic_read(&sess->nconn)) {
		spin_unlock_bh(&sess->conn_lock);
		return(iscsi_no_active_connections_logout(sess));
	}
	spin_unlock_bh(&sess->conn_lock);

	if (!(la = (iscsi_logout_attempt_t *) kmalloc(
			sizeof(iscsi_logout_attempt_t), GFP_ATOMIC))) {
		TRACE_ERROR("Unable to allocate memory for"
			" iscsi_logout_attempt_t\n");
		return(-1);
	}
	memset(la, 0, sizeof(iscsi_logout_attempt_t));

	init_MUTEX_LOCKED(&la->logout_sem);
	init_MUTEX_LOCKED(&la->logout_done_sem);
	spin_lock_init(&la->logout_state_lock);
	la->logout_lun_remove	= lun_remove;
	la->logout_force	= (lun_remove == 2);
	la->logout_reason	= reason;
	la->logout_cid		= cid;
	la->logout_on_cid	= on_cid;
	la->logout_immediate	= ISCSI_CA(channel)->immediate_logout;
	
	if (la->logout_immediate) {
		if (!(cmd = iscsi_start_immediate_logout(sess, la)))
			goto error_out;
	} else {
		if (!(cmd = iscsi_start_non_immediate_logout(sess, la)))
			goto error_out;
	}
	
	if (async_logout)
		return(0);
	
	init_timer(&logout_timer);
	logout_timer.expires = (jiffies + ISCSI_CA(channel)->logout_timeout * HZ);
	logout_timer.data = (unsigned long)&la->logout_sem;
	logout_timer.function = logout_timer_function;
	add_timer(&logout_timer);
	
	down(&la->logout_sem);
	del_timer_sync(&logout_timer);

	return(iscsi_post_logout_handler(la, sess));
	
error_out:
	if (la)
		kfree(la);	
	return(-1);
}

/*	iscsi_close_all_sessions():
 *
 *	Loop through the driver's session list calling iscsi_close_session()
 *	on each active session.  We dont actually remove the session from the
 *	list here,  iscsi_close_session() takes care of that.    
 */
extern void iscsi_close_all_sessions (void)
{
	iscsi_session_t *sess = NULL, *sess_next = NULL;
	
	spin_lock(&iscsi_global->session_lock);
	sess = iscsi_global->sess_head;
	while (sess) {
		sess_next = sess->next;

		iscsi_inc_session_usage_count(sess);
		spin_unlock(&iscsi_global->session_lock);
		iscsi_stop_session_and_reinstatement(sess, 1);
		spin_lock(&iscsi_global->session_lock);
		
		sess = sess_next;
	}
	spin_unlock(&iscsi_global->session_lock);
	
	return;
}

/*	iscsi_stop_logins_for_channels():
 *
 *
 */
static void iscsi_stop_logins_for_channels (void)
{
	int i;
	iscsi_channel_t *channel;
	
	spin_lock(&iscsi_global->channel_lock);
	for (i = 0; i < ISCSI_MAX_CHANNELS; i++) {
		channel = &iscsi_global->channels[i];
		spin_lock(&channel->channel_state_lock);
		if ((channel->status != ISCSI_CHANNEL_ACTIVE) &&
		    (channel->status != ISCSI_CHANNEL_REESTABLISHING)) {
			spin_unlock(&channel->channel_state_lock);
			continue;
		}
		spin_unlock(&channel->channel_state_lock);

		spin_unlock(&iscsi_global->channel_lock);
		iscsi_stop_channel_reinstatement_process(channel, 0);
		spin_lock(&iscsi_global->channel_lock);
	}
	spin_unlock(&iscsi_global->channel_lock);
		
	return;
}

/*	iscsi_initiator_mod_init():
 *
 *	OS Independant initilization routine.
 */
extern int iscsi_initiator_mod_init (void)
{
	if (!(iscsi_global = (iscsi_global_t *) kmalloc(
			sizeof(iscsi_global_t), GFP_KERNEL))) {
		TRACE_ERROR("Unable to allocate memory for iscsi_global_t\n");
		return(-1);
	}
		
	memset(iscsi_global, 0, sizeof(iscsi_global_t));
	iscsi_global->sid	= 1;
	init_MUTEX(&iscsi_global->auth_sem);
	spin_lock_init(&iscsi_global->active_lt_lock);
	spin_lock_init(&iscsi_global->active_ts_lock);
	spin_lock_init(&iscsi_global->channel_lock);
	spin_lock_init(&iscsi_global->inactive_lt_lock);
	spin_lock_init(&iscsi_global->inactive_ts_lock);
	spin_lock_init(&iscsi_global->session_lock);
	spin_lock_init(&iscsi_global->target_lock);

	iscsi_init_channels();
	
	if (iscsi_allocate_thread_sets(INITIATOR_THREAD_SET_COUNT) !=
			INITIATOR_THREAD_SET_COUNT)
		goto out;

	return(0);
out:
	iscsi_global->in_shutdown = 1;
	iscsi_deallocate_thread_sets();

	kfree(iscsi_global);
	iscsi_global = NULL;

	return(-1);
}

/*	iscsi_initiator_mod_fini():
 *
 *	OS Independant shutdown routine.
 */
extern void iscsi_initiator_mod_fini (void)
{
	iscsi_global->in_shutdown = 1;

	iscsi_stop_logins_for_channels();
	
	iscsi_close_all_sessions();

	iscsi_free_target_entries();
	
	iscsi_deallocate_thread_sets();

	iscsi_free_all_channel_resources();

	kfree(iscsi_global);
	iscsi_global = NULL;
	
	return;
}
