/*
 * Copyright (c) 2003-2004 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 * 
 * This file houses iSCSI Initiator discovery specific functions.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#define ISCSI_INITIATOR_DISCOVERY_C

#include <linux/string.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/inet.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_discovery.h>
#include <iscsi_lists.h>
#include <iscsi_initiator_parameters.h>

#undef ISCSI_INITIATOR_DISCOVERY_C

extern iscsi_global_t *iscsi_global;
extern char *iscsi_ntoa (u32);

/*      iscsi_add_target_to_list():
 *
 *
 */
static void iscsi_add_target_to_list (iscsi_target_t *t)
{
	spin_lock(&iscsi_global->target_lock);
	ADD_ENTRY_TO_LIST(t, iscsi_global->target_head, iscsi_global->target_tail);
	spin_unlock(&iscsi_global->target_lock);

	return;
}

/*	iscsi_add_targetaddress_to_list():
 *
 *
 */
static void iscsi_add_targetaddress_to_list (iscsi_targetaddress_t *ta, iscsi_target_t *t)
{
	spin_lock(&t->ta_lock);
	ADD_ENTRY_TO_LIST(ta, t->ta_head, t->ta_tail);
	spin_unlock(&t->ta_lock);
		
	return;
}

/*	iscsi_free_target_entries():
 *
 *
 */
extern void iscsi_free_target_entries (void)
{
	iscsi_target_t *t, *t_next;
	iscsi_targetaddress_t *ta, *ta_next;
	
	spin_lock(&iscsi_global->target_lock);
	t = iscsi_global->target_head;
	while (t) {
		t_next = t->next;

		spin_lock(&t->ta_lock);
		ta = t->ta_head;
		while (ta) {
			ta_next = ta->next;
			kfree(ta);
			ta = ta_next;
		}
		spin_unlock(&t->ta_lock);
		
		kfree(t);
		t = t_next;
	}
	spin_unlock(&iscsi_global->target_lock);

	return;
}

/*	iscsi_search_for_target():
 *
 *
 */
static iscsi_target_t *iscsi_search_for_target (unsigned char *targetname)
{
	iscsi_target_t *t;

	spin_lock(&iscsi_global->target_lock);
	for (t = iscsi_global->target_head; t; t = t->next) {
		if (!(strncmp(t->targetname, targetname, strlen(targetname)))) {
			spin_unlock(&iscsi_global->target_lock);
			return(t);
		}
	}
	spin_unlock(&iscsi_global->target_lock);

	return(NULL);
}

/*	iscsi_search_for_targetaddress():
 *
 *
 */
static iscsi_targetaddress_t *iscsi_search_for_targetaddress (
	u32 ipv4_address,
	u16 port,
	u16 tpgt,
	iscsi_target_t *t)
{
	iscsi_targetaddress_t *ta;

	spin_lock(&t->ta_lock);
	for (ta = t->ta_head; ta; ta = ta->next) {
		if ((ta->ipv4_address == ipv4_address) &&
		    (ta->port == port) && (ta->tpgt == tpgt)) {
			spin_unlock(&t->ta_lock);
			return(ta);
		}
	}
	spin_unlock(&t->ta_lock);

	return(NULL);
}

/*      iscsi_search_for_target_entry():
 *
 *
 */
extern iscsi_targetaddress_t *iscsi_search_for_login_targetaddress (iscsi_login_holder_t *lh)
{
	iscsi_target_t *t;
 	iscsi_targetaddress_t *ta;
	
	spin_lock(&iscsi_global->target_lock);
	for (t = iscsi_global->target_head; t; t = t->next) {

		if ((lh->lh_flags & LH_TARGETNAME_PRESENT) &&
		    (strncmp(lh->lh_targetname, t->targetname, strlen(t->targetname))))
			continue;
		
		spin_lock(&t->ta_lock);
		for (ta = t->ta_head; ta; ta = ta->next) {
			if ((ta->ipv4_address != lh->ipv4_address) ||
			    (ta->port != lh->port))
				continue;
			
			strncpy(ta->net_dev, lh->net_dev,
				strlen(lh->net_dev));
			spin_unlock(&t->ta_lock);
			spin_unlock(&iscsi_global->target_lock);
			return(ta);
		}
		spin_unlock(&t->ta_lock);
	}
	spin_unlock(&iscsi_global->target_lock);
   
	return(NULL);
}

/*	iscsi_check_for_login_targetaddress():
 *
 *
 */
extern int iscsi_check_for_login_targetaddress (iscsi_login_holder_t *lh)
{
	char buf[512];
	iscsi_channel_t *chan = lh->channel;
	iscsi_conn_t *conn = lh->conn;
	iscsi_session_t *sess = SESS(conn);
	iscsi_targetaddress_t *ta;

	/* 
	 * Search for the IP address and port given in the list
	 * known/discovered targets, if we find a match we may proceed
	 * to Full Feature Phase, else we must start a Discovery session.
	 */
	if (!(ta = iscsi_search_for_login_targetaddress(lh))) {
		if (!lh->leading_conn) {
			TRACE_ERROR("Unable to find TargetAddress entry for"
	   		" %s:%hu while adding a non-leading connection, cannot"
				" continue.\n", iscsi_ntoa(conn->login_ip),
					conn->login_port);
			return(-1);
		}
		
		/*
		 * This will be a Discovery session.
		 */
		TRACE(TRACE_LOGIN, "Unable to find matching iSCSI target,"
			" starting discovery session.\n");
		memset(buf, 0, 512);
		sprintf(buf, "SessionType=%s", "Discovery");

		if (iscsi_change_param_value(buf, INITIATOR,
				conn->param_list) < 0)
                        return(-1);

		iscsi_set_keys_to_negotiate(INITIATOR, 0, conn->param_list);
		iscsi_set_keys_irrelivant_for_discovery(conn->param_list);
	} else {
		/*
		 * This will be a Normal Session.
		 */
		memset(buf, 0, 512);
		sprintf(buf, "TargetName=%s", ta->target->targetname);

		if (iscsi_change_param_value(buf, INITIATOR,
				conn->param_list) < 0)
			return(-1);

		if (lh->leading_conn) {
			if (lh->lh_flags & LH_TARGETNAME_PRESENT) {
				spin_lock_bh(&chan->channel_state_lock);
				chan->ch_target = ta->target;
				spin_unlock_bh(&chan->channel_state_lock);
			}
		} else {
			if (SESS_OPS(sess)->TargetPortalGroupTag != ta->tpgt) {
				TRACE_ERROR("Existing TPGT: %hu does not match"
				" TargetAddress's TPGT: %hu, cannot continue.\n",
				SESS_OPS(sess)->TargetPortalGroupTag, ta->tpgt);
				BUG();
                                return(-1);
			}
		}
		iscsi_set_keys_to_negotiate(INITIATOR, 1, conn->param_list);
		conn->target_address = ta;

		TRACE(TRACE_LOGIN, "Found matching iSCSI target, starting"
				" FFP session.\n");
	}

	return(0);
}

/*	iscsi_setup_array_for_tpg_failover():
 *
 *
 */
extern int iscsi_setup_array_for_tpg_failover (iscsi_login_holder_t *lh, iscsi_tpg_failover_t *tpgf)
{
	int current_tpga_set = 0, i = 0;
	iscsi_channel_t *c = lh->channel;
	iscsi_conn_t *conn = lh->conn;
	iscsi_target_t *login_t;
	iscsi_targetaddress_t *login_ta, *ta;
	iscsi_tpg_failover_address_t *tpga;

	/*
	 * A null conn->target_address denotes that this is a discovery session
	 * and that TPG Failover will not occur.  Also make sure that TPG
	 * failover is enabled on this iSCSI Channel.
	 */
	if (!conn->target_address || !ISCSI_CA(c)->tpgfailover) {
		lh->tpg_failover = 0;
		return(0);
	}

	/*
	 * tpgf->tpga_list_built denotes that this function has already
	 * been called and iscsi_tpg_failover_t initialized.
	 */
	if (tpgf->current_tpga)
		return(0);
	
	login_ta = conn->target_address;
	login_t = login_ta->target;
	tpgf->tpgt = login_ta->tpgt;

	/*
	 * Build up the tpgf->tpga_list array from the list of TargetAddress
	 * keys for the iscsi_target_t.
	 */
	spin_lock(&login_t->ta_lock);
	for (ta = login_t->ta_head; ta; ta = ta->next) {
		if (ta->tpgt != tpgf->tpgt)
			continue;
		tpga = &tpgf->tpga_list[i];
		
		tpga->ipv4_address = ta->ipv4_address;
		tpga->port = ta->port;
		strncpy(tpga->net_dev, ta->net_dev, strlen(ta->net_dev));

		if ((tpga->ipv4_address == conn->login_ip) &&
		    (tpga->port == conn->login_port)) {
			current_tpga_set = 1;
			tpgf->current_tpga = i;
		}
		
		if (++i == ISCSI_MAX_TPG_FAILOVER_ADDRESSES) {
			printk("Reached ISCSI_MAX_TPG_FAILOVER_ADDRESSES:"
				" %u\n", ISCSI_MAX_TPG_FAILOVER_ADDRESSES);
			break;
		}
	}
	spin_unlock(&login_t->ta_lock);

	if (!current_tpga_set) {
		TRACE_ERROR("Unable to set tpgf->current_tpga with %s:%hu,"
		" cannot continue.\n", iscsi_ntoa(conn->login_ip),
				conn->login_port);
		return(-1);
	}

	tpgf->tpga_count = i;
	tpgf->tpga_list_built = 1;

	return(0);
}

/*	iscsi_update_tpg_failover_array():
 *
 *
 */
extern void iscsi_update_tpg_failover_array (iscsi_conn_t *conn, iscsi_tpg_failover_t *tpgf)
{
	iscsi_tpg_failover_address_t *tpga;
	
	if (tpgf->current_tpga == (tpgf->tpga_count - 1))
		tpgf->current_tpga = 0;
	else
		tpgf->current_tpga++;
		
	tpga = &tpgf->tpga_list[tpgf->current_tpga];
	if (!(conn->conn_flags & CONNFLAG_TPGF_USE_ORIG)) {
		conn->conn_flags |= CONNFLAG_TPGF_USE_ORIG;
		conn->orig_login_ip = conn->login_ip;
		conn->orig_login_port = conn->login_port;
	}
	conn->login_ip = tpga->ipv4_address;
	conn->login_port = tpga->port;
	strncpy(conn->net_dev, tpga->net_dev, strlen(tpga->net_dev));
		
	return;
}

/*	iscsi_print_targets():
 *                              
 *	This function will display all targets known that have been
 *	found via iSCSI Discovery sessions.
 */
extern void iscsi_print_targets (void)
{
	iscsi_target_t *t;
	iscsi_targetaddress_t *ta;

	printk("Displaying discovered iSCSI Targets entrys:\n");

	spin_lock(&iscsi_global->target_lock);
	for (t = iscsi_global->target_head; t; t = t->next) {
		printk("TargetName: %s\n", t->targetname);
		spin_lock(&t->ta_lock);
		for (ta = t->ta_head; ta; ta = ta->next) {
			printk("\tTargetAddress: %s:%hu,%hu\n",
				iscsi_ntoa(ta->ipv4_address), ta->port,
					ta->tpgt);
		}
		spin_unlock(&t->ta_lock);
	}
	printk("\n");
	spin_unlock(&iscsi_global->target_lock);

	return;
}

/*	iscsi_handle_sendtargets_response():
 *
 *	Functions adds all TargetName and TargetAddress keys in text_in to the
 *	global iSCSI targets list.
 *
 *	FIXME: Open discovery sessions for TargetNames other than the
 *	       one with the IP we opened this session on.
 */
extern int iscsi_handle_sendtargets_response (
	iscsi_conn_t *conn,
	unsigned char *text_in,
	u32 length_in)
{
	int ret_len = 0;
	char *delim_ptr, *param_ptr;
	char *last_ptr, *targname_ptr;
	char targetname[ISCSI_MAX_TARGETNAME_SIZE];
	iscsi_target_t *target;
	iscsi_targetaddress_t *ta;
	
	param_ptr = text_in;
	last_ptr = (text_in + length_in);
	while (param_ptr < last_ptr) {
		if (!(target = (iscsi_target_t *) kmalloc(
				sizeof(iscsi_target_t), GFP_KERNEL))) {
			TRACE_ERROR("Could not allocate memory for"
					" iscsi_target_t.\n");
			return(-1);
		}
		memset(target, 0, sizeof(iscsi_target_t));
		spin_lock_init(&target->ta_lock);

		if (strncmp(param_ptr, "TargetName", 10)) {
			TRACE_ERROR("Cannot locate TargetName key, exiting\n");
			return(0);
		}

		if (!(delim_ptr = strchr(param_ptr, '='))) {
			TRACE_ERROR("No \"=\" parameter delimitor found,"
					" exiting.\n");
			return(-1);
		}
		targname_ptr = (delim_ptr + 1);

		if (strlen(targname_ptr) > ISCSI_MAX_TARGETNAME_SIZE) {
			TRACE_ERROR("TargetName value is greater than %d, "
				"exiting.\n", ISCSI_MAX_TARGETNAME_SIZE);
			return(-1);
		}

		memset(targetname, 0, ISCSI_MAX_TARGETNAME_SIZE);
		memcpy(targetname, targname_ptr, strlen(targname_ptr));

		if (!(target = iscsi_search_for_target(targetname))) {
			if (!(target = (iscsi_target_t *) kmalloc(
					sizeof(iscsi_target_t), GFP_KERNEL))) {
				TRACE_ERROR("Could not allocate memory for"
					" iscsi_target_t.\n");
				return(-1);
			}
			memset(target, 0, sizeof(iscsi_target_t));
			spin_lock_init(&target->ta_lock);

			memcpy(target->targetname, targetname,
				strlen(targetname));

			iscsi_add_target_to_list(target);

			printk("Discovered iSCSI Target: %s\n", targetname);
		} else {
			printk("Discovered existing iSCSI Target: %s\n",
					targetname);
		}

		param_ptr = (targname_ptr + (strlen(targname_ptr) + 1));
		
		/*
		 * Process all of the TargetAddress keys below the current TargetName
		 * key until a second TargetName key is located or the buffer runs out.
		 */
		if ((ret_len = iscsi_handle_targetaddress_keys(param_ptr, NULL, target)) < 0)
			return(-1);

		if (target->ta_head) {
			param_ptr = param_ptr + ret_len;
			continue;
		}

		/*
		 * No TargetAddress keys where returned, create our own using
		 * the passed ip/port.  We will retrieve the TPGT after its
		 * returned in the first login response.
		 */
		if (!(ta = (iscsi_targetaddress_t *) kmalloc(
				sizeof(iscsi_targetaddress_t), GFP_KERNEL))) {
			TRACE_ERROR("Unable to allocate"
				" iscsi_targetaddress_t\n");
			return(-1);
		}
		memset(ta, 0, sizeof(iscsi_targetaddress_t));
                
		ta->ipv4_address        = conn->login_ip;
		ta->port                = conn->login_port;
		ta->need_tpgt           = 1;
		ta->target              = target;
                
		iscsi_add_targetaddress_to_list(ta, target);
                       
		printk("Discovered TargetAddress: %s:%hu"
			" for %s\n", iscsi_ntoa(conn->login_ip),
			conn->login_port, target->targetname);

		param_ptr = param_ptr + ret_len;
	}

	return(0);
}
	
/*	iscsi_handle_targetaddress_keys():
 *
 *
 */
extern int iscsi_handle_targetaddress_keys (
	char *buf,
	iscsi_target_rdr_t *rdr,
	iscsi_target_t *target)
{
	char *start_ptr = buf, *end_ptr, *tmp_ptr;
	char *delim_ptr, *ipaddress_ptr, *port_ptr, *tpgt_ptr;
	char *colon_ptr, *comma_ptr = NULL;
	char ipv4_address_c[18], port_c[6], tpgt_c[6];
	u16 port, tpgt = 0;
	int len_ret = 0;
	u32 ipv4_address, ip_address_size, tpgt_size;
	u32 targetname_size, port_size, total_size;
	iscsi_targetaddress_t *ta;
		
	while (1) {
		if (!strncmp(start_ptr, "TargetName", 10))
			return(len_ret);

		if (strncmp(start_ptr, "TargetAddress", 13))
			return(len_ret);

		/*
		 * Get passed the TargetName=
		 */
		if (!(delim_ptr = strchr(start_ptr, '='))) {
			TRACE_ERROR("No \"=\" parameter seperator"
				" found, exiting.\n");
			return(-1);
		}
		ipaddress_ptr = (delim_ptr + 1);

		targetname_size = (ipaddress_ptr - start_ptr);
		total_size = targetname_size;
		end_ptr = ipaddress_ptr + strlen(ipaddress_ptr);
			
		/*
		 * Get the IPv4 Address.
		 */
		if (!(colon_ptr = strchr(ipaddress_ptr, ':'))) {
			TRACE_ERROR("No \":\" parameter seperator"
				" found in TargetAddress, exiting.\n");
			return(-1);
		}

		ip_address_size = (colon_ptr - ipaddress_ptr);
		total_size += ip_address_size + 1;

		if (ip_address_size > 18) {
			TRACE_ERROR("IP address of TargetAddress is %u and"
				" greater than 18.\n", ip_address_size);
			return(-1);
		}

		memset(ipv4_address_c, 0, 18);
		strncpy(ipv4_address_c, ipaddress_ptr, ip_address_size);
		ipv4_address_c[ip_address_size] = '\0';
		ipv4_address = ntohl(in_aton(ipv4_address_c));

		/*
		 * Get the unsigned 16-bit Port.
		 */
		port_ptr = (start_ptr + total_size); /* Skip over ":" */

		if (!(comma_ptr = strchr(port_ptr, ','))) {
			if (!rdr) {
				TRACE_ERROR("No \",\" parameter seperator"
					" found in TargetAddress, exiting.\n");
				return(-1);
			}
			tpgt = rdr->rdr_tpgt;
		}
		port_size = (!rdr) ? (comma_ptr - colon_ptr) : (end_ptr - colon_ptr);
		total_size += port_size;

		if (port_size > 6) {
			TRACE_ERROR("Port of TargetAddress is greater"
				" than 6.\n");
			return(-1);
		}
		memset(port_c, 0, 6);
		strncpy(port_c, port_ptr, port_size);
		port_c[port_size] = '\0';
		port = simple_strtoul(port_c, &tmp_ptr, 0);

		if (!comma_ptr)
			goto process_key;
		/*
		 * Get the unsigned 16-bit Target Portal Group Tag.
		 */
		tpgt_ptr = (start_ptr + total_size); /* Skip over "," */
		tpgt_size = (end_ptr - comma_ptr);
		total_size += tpgt_size;
		if (tpgt_size > 6) {
			TRACE_ERROR("TargetPortalGroupTag length: %u in TargetName"
				" is illegal value.\n", tpgt_size);
			return(-1);
		}
		
		memset(tpgt_c, 0, 6);
		strncpy(tpgt_c, tpgt_ptr, tpgt_size);
		tpgt_c[tpgt_size] = '\0';
		tpgt = simple_strtoul(tpgt_c, &tmp_ptr, 0);

process_key:
		if (!(ta = iscsi_search_for_targetaddress(ipv4_address,
				port, tpgt, target))) {
			if (!(ta = (iscsi_targetaddress_t *) kmalloc(
					sizeof(iscsi_targetaddress_t), GFP_KERNEL))) {
				TRACE_ERROR("Unable to allocate"
					" iscsi_targetaddress_t\n");
				return(-1);
			}
			memset(ta, 0, sizeof(iscsi_targetaddress_t));
		
		       	ta->ipv4_address 	= ipv4_address;
			ta->port		= port;
			ta->tpgt		= tpgt;
			ta->target		= target;

			iscsi_add_targetaddress_to_list(ta, target);
			
			printk("Discovered TargetAddress: %s:%hu,%hu"
				" for %s\n", iscsi_ntoa(ipv4_address),
				port, tpgt, target->targetname);
		} else {
			printk("Discovered existing TargetAddress:"
				" %s:%hu,%hu for %s\n",
				iscsi_ntoa(ipv4_address), port, tpgt,
					target->targetname);
		}

		if (rdr) {
			rdr->rdr_ipv4_address = ipv4_address;
			rdr->rdr_port = port;
		}

		start_ptr += total_size;
		len_ret += total_size;
	}
	
	return(len_ret);
}
