/*
 * Copyright (c) 2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 *
 * This file houses scsi_transport_iscsi related abstractions for
 * exporting iSCSI Initiator Core related data to userspace.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#define ISCSI_INITIATOR_CLASS_C

#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/in.h>
#include <scsi/scsi_tcq.h>
#include <scsi/scsi_host.h>
#include <scsi/scsi_transport.h>
#include <scsi/scsi_transport_iscsi.h>
#include <iscsi_linux_os.h>
#include <iscsi_protocol.h>
#include <iscsi_debug.h>
#include <iscsi_initiator_core.h>
#include <iscsi_initiator_linux.h>

#undef ISCSI_INITIATOR_CLASS_C

extern iscsi_global_t *iscsi_global;

#define iscsi_transport_get_conn_fn(field, name)                        \
static void                                                             \
iscsi_get_##field (struct scsi_target *stgt)                            \
{                                                                       \
        struct Scsi_Host *shost = dev_to_shost(stgt->dev.parent);       \
        iscsi_channel_t *c = (iscsi_channel_t *)shost->hostdata;        \
        iscsi_session_t *s = (iscsi_session_t *)c->sess;                \
	iscsi_conn_t *conn = (iscsi_conn_t *)s->conn_head;		\
        iscsi_##name(stgt) = CONN_OPS(conn)->field;                     \
}

iscsi_transport_get_conn_fn(HeaderDigest, header_digest);
iscsi_transport_get_conn_fn(DataDigest, data_digest);
iscsi_transport_get_conn_fn(MaxRecvDataSegmentLength, max_recv_data_segment_len);
/*
 * These need to be added to struct iscsi_function_template
 */
#if 0
iscsi_transport_get_conn_fn(IFMarker, if_marker);
iscsi_transport_get_conn_fn(OFMarker, of_marker);
iscsi_transport_get_conn_fn(IFMarkInt, if_markint);
iscsi_transport_get_conn_fn(OFMarkInt, of_markint);
#endif

#define iscsi_transport_get_sess_fn(field, name)                        \
static void                                                             \
iscsi_get_##field (struct scsi_target *stgt)                            \
{                                                                       \
        struct Scsi_Host *shost = dev_to_shost(stgt->dev.parent);       \
	iscsi_channel_t *c = (iscsi_channel_t *)shost->hostdata;	\
	iscsi_session_t *s = (iscsi_session_t *)c->sess;		\
        iscsi_##name(stgt) = SESS_OPS(s)->field;                        \
}

iscsi_transport_get_sess_fn(InitialR2T, initial_r2t);
iscsi_transport_get_sess_fn(ImmediateData, immediate_data);
iscsi_transport_get_sess_fn(MaxBurstLength, max_burst_len);
iscsi_transport_get_sess_fn(FirstBurstLength, first_burst_len);
iscsi_transport_get_sess_fn(DataPDUInOrder, data_pdu_in_order);
iscsi_transport_get_sess_fn(DataSequenceInOrder, data_sequence_in_order);
iscsi_transport_get_sess_fn(DefaultTime2Wait, def_time2wait);
iscsi_transport_get_sess_fn(DefaultTime2Retain, def_time2retain);
iscsi_transport_get_sess_fn(MaxOutstandingR2T, max_outstanding_r2t);
iscsi_transport_get_sess_fn(TargetPortalGroupTag, tpgt);

static void iscsi_get_tsih (struct scsi_target *stgt)
{
	struct Scsi_Host *shost = dev_to_shost(stgt->dev.parent);
	iscsi_channel_t *c = (iscsi_channel_t *)shost->hostdata;
	iscsi_session_t *s = (iscsi_session_t *)c->sess;

	iscsi_tsih(stgt) = s->tsih;
}

#define iscsi_transport_target_fn(field)                                \
static ssize_t                                                          \
iscsi_get_##field (struct scsi_target *stgt, char *buf, ssize_t count)  \
{                                                                       \
        struct Scsi_Host *shost = dev_to_shost(stgt->dev.parent);       \
	iscsi_channel_t *c = (iscsi_channel_t *)shost->hostdata;	\
	iscsi_session_t *s = (iscsi_session_t *)c->sess;		\
        return snprintf(buf, count - 1, "%s\n", SESS_OPS(s)->field);	\
}

iscsi_transport_target_fn(TargetName);
iscsi_transport_target_fn(TargetAlias);

static void iscsi_get_ip_address(struct scsi_target *starget)
{
	struct Scsi_Host *shost = dev_to_shost(starget->dev.parent);
	iscsi_channel_t *c = (iscsi_channel_t *)shost->hostdata;
	iscsi_session_t *s = (iscsi_session_t *)c->sess;
        iscsi_conn_t *conn = (iscsi_conn_t *)s->conn_head;		
	struct sockaddr_in *addr = (struct sockaddr_in *)&conn->s_addr;

	iscsi_addr_type(starget) = addr->sin_family;
	memcpy(&iscsi_sin_addr(starget), &addr->sin_addr,
		sizeof(struct in_addr));

	return;
}

static void iscsi_get_port(struct scsi_target *starget)
{
	struct Scsi_Host *shost = dev_to_shost(starget->dev.parent);
	iscsi_channel_t *c = (iscsi_channel_t *)shost->hostdata;
	iscsi_session_t *s = (iscsi_session_t *)c->sess;
	iscsi_conn_t *conn = (iscsi_conn_t *)s->conn_head;

	iscsi_port(starget) = conn->login_port;
	return;
}

static void iscsi_get_isid(struct scsi_target *starget)
{
        struct Scsi_Host *shost = dev_to_shost(starget->dev.parent);
	iscsi_channel_t *chan = (iscsi_channel_t *)shost->hostdata;
        iscsi_session_t *sess = (iscsi_session_t *)chan->sess;

        memcpy(iscsi_isid(starget), sess->isid, sizeof(sess->isid));
	return;
}

#define iscsi_transport_initiator_fn(field)                             \
static ssize_t                                                          \
iscsi_get_##field (struct Scsi_Host *shost, char *buf, ssize_t count)   \
{                                                                       \
	iscsi_channel_t *c = (iscsi_channel_t *)shost->hostdata;	\
        iscsi_session_t *s = (iscsi_session_t *)c->sess;		\
        return snprintf(buf, count - 1, "%s\n", SESS_OPS(s)->field);    \
}

iscsi_transport_initiator_fn(InitiatorName);
iscsi_transport_initiator_fn(InitiatorAlias);

struct iscsi_function_template iscsi_fnt = {
        .get_isid = iscsi_get_isid,
        .show_isid = 1,
        .get_tsih = iscsi_get_tsih,
        .show_tsih = 1,
        .get_port = iscsi_get_port,
        .show_port = 1,
        .get_tpgt = iscsi_get_TargetPortalGroupTag,
        .show_tpgt = 1,
        .get_ip_address = iscsi_get_ip_address,
        .show_ip_address = 1,
        .get_initial_r2t = iscsi_get_InitialR2T,
        .show_initial_r2t = 1,
        .get_immediate_data = iscsi_get_ImmediateData,
        .show_immediate_data = 1,
        .get_header_digest = iscsi_get_HeaderDigest,
        .show_header_digest = 1,
        .get_data_digest = iscsi_get_DataDigest,
        .show_data_digest = 1,
/*
 * These need to be added to struct iscsi_function_template
 */
#if 0
	.get_of_marker = iscsi_get_OFMarker,
	.show_of_marker = 1,
	.get_if_marker = iscsi_get_IFMarker,
	.show_if_marker = 1,
	.get_of_markint = iscsi_get_OFMarkInt,
	.show_of_markint = 1,
	.get_if_markint = iscsi_get_IFMarkInt,
	.show_if_markint = 1,
#endif
        .get_max_burst_len = iscsi_get_MaxBurstLength,
        .show_max_burst_len = 1,
        .get_first_burst_len = iscsi_get_FirstBurstLength,
        .show_first_burst_len = 1,
	.get_def_time2wait = iscsi_get_DefaultTime2Wait,
	.show_def_time2wait = 1,
	.get_def_time2retain = iscsi_get_DefaultTime2Retain,
	.show_def_time2retain = 1,
	.get_max_outstanding_r2t = iscsi_get_MaxOutstandingR2T,
	.show_max_outstanding_r2t = 1,
	.get_data_pdu_in_order = iscsi_get_DataPDUInOrder,
	.show_data_pdu_in_order = 1,
	.get_data_sequence_in_order = iscsi_get_DataSequenceInOrder,
	.show_data_sequence_in_order = 1,
        .get_max_recv_data_segment_len = iscsi_get_MaxRecvDataSegmentLength,
        .show_max_recv_data_segment_len = 1,
        .get_target_name = iscsi_get_TargetName,
        .show_target_name = 1,
        .get_target_alias = iscsi_get_TargetAlias,
        .show_target_alias = 1,
        .get_initiator_alias = iscsi_get_InitiatorAlias,
        .show_initiator_alias = 1,
        .get_initiator_name = iscsi_get_InitiatorName,
        .show_initiator_name = 1,
};

