/***********************************************************************
/ Copyright (c) 2001, Nishan Systems, Inc.
/ All rights reserved.
/ 
/ Redistribution and use in source and binary forms, with or without 
/ modification, are permitted provided that the following conditions are 
/ met:
/ 
/ - Redistributions of source code must retain the above copyright notice, 
/   this list of conditions and the following disclaimer. 
/ 
/ - Redistributions in binary form must reproduce the above copyright 
/   notice, this list of conditions and the following disclaimer in the 
/   documentation and/or other materials provided with the distribution. 
/ 
/ - Neither the name of the Nishan Systems, Inc. nor the names of its 
/   contributors may be used to endorse or promote products derived from 
/   this software without specific prior written permission. 
/ 
/ THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
/ AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
/ IMPLIED WARRANTIES OF MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS FOR A 
/ PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL NISHAN SYSTEMS, INC. 
/ OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
/ EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
/ PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
/ OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
/ WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
/ OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF 
/ ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
/
/***********************************************************************/
#include "isns.h"
#include "comm.h"
#include "util.h"

#define MAX_ESI_ENTRIES    (100)

static ISNS_CMD cmd;       /* Used by functions for msgs. */
static int currentXid = 0; /* current XID */
int enableESIFlag;         /* Set to non-zero when enable ESI reponse. */

typedef struct _esi_table_entry
{
   char portal_name[256];
   int port;
   HANDLE t_hdle;
}
ESI_TBL_ENTRY;

/* ESI Table */
ESI_TBL_ENTRY esi_tble[MAX_ESI_ENTRIES];


/* Generates a new transaction id (xid) */
static int
GetXid (void)
{
   //currentXid++;

   return (currentXid++);

}
/*********************************************************************/
/* Zeros out the message and puts a hdr in */
/*********************************************************************/
void
ISNSCreateHdr (ISNS_FUNC_ID func_id, ISNS_CMD * cmd, int b_size, int flags)
{
   memset (cmd, 0, b_size);
   cmd->hdr.len = 0;
   cmd->hdr.flags =
      htons ((short)(flags | ISNS_FLAG_FIRST_PDU | ISNS_FLAG_LAST_PDU |
             ISNS_FLAG_SND_CLIENT));
   cmd->hdr.version = htons ((short)ISNSP_VERSION);
   cmd->hdr.func_id = htons ((short)func_id);
   cmd->hdr.xid = htons ((short)GetXid ());
}

/*********************************************************************/
/* Same as ISNSCreateHdr() but also zeros out the src field */
/*********************************************************************/
void
ISNSInsertQryHdr (ISNS_FUNC_ID func_id, ISNS_CMD * cmd)
{
   ISNSCreateHdr (func_id, cmd, sizeof(ISNS_CMD), 0);
   /* Insert a SRC ATTR of NULL */
   ISNSAppendAttr (cmd, 0, 0, NULL, 0);
}

/*********************************************************************/
/* Appends an attribute to a msg.  Does the necessary, byte 
conversions */
/*********************************************************************/
void
ISNSAppendAttr (ISNS_CMD * cmd, ISNS_TAG tag, int len, char *ptr, int val)
{
   ISNS_TLV_P tlvp = (ISNS_TLV_P)((char *) cmd + cmd->hdr.len + sizeof (ISNS_HDR));

   /* Attr */
   tlvp->attrib_id = htonl (tag);
   tlvp->attrib_len = htonl (len);
   if (ptr != NULL)
      memcpy (&tlvp->attrib_val, ptr, len);
   else
   {
      switch (len)
      {
      case 0:
         break;
      case 2:
         *(uint16_t *) & tlvp->attrib_val = htons ((short)val);
         break;
      case 4:
         *(uint32_t *) & tlvp->attrib_val = htonl (val);
         break;
      default:
         break;
      }
   }
   cmd->hdr.len += ISNS_TAG_LEN + ISNS_LEN_LEN + len;
}

int
get_initiatornode_name (unsigned char *buf)
{
	FILE *f;
	if (!(f = fopen(SYSFS_INITIATOR_NODENAME, "r"))) {
		if (!(f = fopen(PROCFS_INITIATOR_NODENAME, "r"))) {
			printf("Unable to locate iSCSI Initiator Node Name\n");
			return(-1);
		}
	}
	fscanf(f, "iSCSI InitiatorName: %s", buf);
	fclose(f);

	return(0);
}

/*********************************************************************/
/* Appends the SRC field */
/*********************************************************************/
int
GetSrc (ISNS_CMD * p_cmd, int flags)
{
   char cmdLine[256];
   int src_type;

   memset(cmdLine, 0, sizeof(cmdLine));
   if (get_initiatornode_name(cmdLine) < 0)
	   return(-1);

   //printf("iSCSI Node Name: %s\n", cmdLine);
   
   ISNSAppendAttr (p_cmd, ISNS_ISCSI_NODE_ID, PAD4 (strlen (cmdLine)),
		   cmdLine, 0);

   return (0);
}

/*********************************************************************/
/* Disable an ESI */
/*********************************************************************/
void
DisableESI (void)
{
   enableESIFlag = FALSE;
}

/*********************************************************************/
/* Enable an ESI */
/*********************************************************************/
void
EnableESI (void)
{
   enableESIFlag = TRUE;
}

/*********************************************************************/
/* Parses an ESI */
/*********************************************************************/
void
ParseESI (char *buffer)
{
   ISNS_Attr *p_attr;
   ISNS_HDR *pdu;
   char *ptr;

   printf ("Parsing ESI.\n");
   pdu = (struct isns_hdr *)buffer;
   ptr = buffer + sizeof (ISNS_HDR);
   pdu->len = htons (pdu->len);
   while (ptr < (char *) buffer + sizeof (ISNS_HDR) + pdu->len)
   {
      p_attr = (struct ISNS_attr *)ptr;
      p_attr->tag = ntohl (p_attr->tag);
      p_attr->len = ntohl (p_attr->len);
      switch (p_attr->tag)
      {
      case ISNS_ENTITY_ID:
         printf ("---------------------------------\n");
         printf ("Entity ID  : %s\n", &p_attr->val);
         break;
      case ISNS_PORTAL_IP:
         {
            struct in_addr ip;
            ip.s_addr = *(uint32_t *) ((char *) &p_attr->val + 12);
            printf ("Portal IP     : %s.\n", inet_ntoa (ip));
         }
         break;
      case ISNS_PORTAL_PORT:
         p_attr->val.etype = ntohl (p_attr->val.etype);
         printf ("Portal Port    : %d.\n", p_attr->val.etype & 0xFFFF);
         printf ("Portal Port Type : %s.\n",
                 p_attr->val.etype & 0x10000 ? "UDP" : "TCP");
         break;
      default:
         break;
      }
      ptr = (char *) ptr + p_attr->len + 8;
   }
}

/*********************************************************************/
/* Sends an ESI response */
/*********************************************************************/
int
Send_ESI_Response (ISNS_CMD * p_cmd, int size)
{
   char rbuffer[1024];
   ISNS_CMD *p_rcmd;

   if (FALSE == enableESIFlag)
   {
      printf ("ESI Response is disabled.\n");
      return (0);
   }

   /* Verify Msg */
   if (ntohs (p_cmd->hdr.func_id) != ISNS_ESI)
   {
      printf ("***ERROR: Expected ESI, recv %#x.\n", p_cmd->hdr.func_id);
      return (-1);
   }

   /* Send response */
   memset (rbuffer, 0, sizeof (rbuffer));
   ISNSCreateHdr (ISNS_ESI_RSP, (struct cmd *)rbuffer, sizeof (rbuffer), 0);
   p_rcmd = (struct cmd *)rbuffer;
   p_rcmd->hdr.len = ntohs (p_cmd->hdr.len) + 4;
   p_rcmd->hdr.xid = p_cmd->hdr.xid;
   memcpy ((char *) rbuffer + sizeof (ISNS_HDR) + 4,
           (char *) p_cmd + sizeof (ISNS_HDR), ntohs (p_cmd->hdr.len));

   ParseESI ((char *)p_cmd);

   printf ("Rsp --->\n");
   ISNSJustSendCmd ((struct cmd *)rbuffer);

   return (0);
}

/*********************************************************************/
/* Process a SCN and sends a response */
/*********************************************************************/
void
Process_SCN (ISNS_CMD * p_msg, int size)
{
   ISNS_Attr *p_attr;
   char *ptr;

//   printf ("RCV SCN.\n");
   DumpHex (p_msg, size);
   ptr = (char *) p_msg + sizeof (ISNS_HDR);
   printf ("---------------------------------\n");
   while (ptr < (char *) p_msg + sizeof (ISNS_HDR) + ntohs (p_msg->hdr.len))
   {
      p_attr = (struct ISNS_attr *)ptr;
      p_attr->tag = ntohl (p_attr->tag);
      p_attr->len = ntohl (p_attr->len);
      switch (p_attr->tag)
      {
      case ISNS_PORT_NAME:
         {
            char buffer2[256] = { 0 };
            memset (buffer2, 0, sizeof (buffer2));
            printf ("iFCP WWPN  : %s\n",
                    HexToAscii ((char *)&p_attr->val, p_attr->len, buffer2));
            break;
         }
      case ISNS_ENTITY_ID:
      case ISNS_ISCSI_NODE_ID:
         printf ("iSCSI ID  : %s\n", &p_attr->val);
         break;
      case ISNS_TIMESTAMP:
         p_attr->val.timestamp.t_time = ntohl (p_attr->val.timestamp.t_time);
         printf ("TimeStamp: %s", ctime ((time_t *)&p_attr->val.timestamp.t_time));
         break;
      case ISNS_IFCP_SCN_BITMAP:
      case ISNS_ISCSI_SCN_BITMAP:
         p_attr->val.etype=ntohl(p_attr->val.etype);
         printf ("Bitmap: %#x.\n", p_attr->val.etype);
         break;
      default:
         break;
      }
      ptr = (char *) ptr + p_attr->len + 8;
   }


   /* Send a response */
   {
      ISNSCreateHdr(ISNS_SCN_RES, &cmd, sizeof(cmd), 0);

      /* Append Dest Addr */
      p_attr = (struct ISNS_attr *)((char *) p_msg + sizeof (ISNS_HDR));

      /* Insert error code */
      cmd.hdr.len=4;

      cmd.hdr.xid=p_msg->hdr.xid;

      ISNSAppendAttr(&cmd, p_attr->tag, p_attr->len, (char *)&p_attr->val, 0);

      printf("Sending SCN rsp-->\n");
      ISNSJustSendCmd ((struct cmd *)&cmd);
   }

}

/*********************************************************************/
/* Initializes the ESI Table.  This table contains the ESI thread
   handles and other relevant information. */
/*********************************************************************/
void
InitESITable (void)
{
   memset (esi_tble, 0, sizeof (esi_tble));
}

/*********************************************************************/
/* Adds an ESI Entry to the table */
/*********************************************************************/
void
AddESIPortal (char *p_name, int port, HANDLE hdle)
{
   int i;
   for (i = 0; i < MAX_ESI_ENTRIES; i++)
   {
      if (esi_tble[i].t_hdle == 0)
      {
         strcpy (esi_tble[i].portal_name, p_name);
         esi_tble[i].port = port;
         esi_tble[i].t_hdle = hdle;
         break;
      }
   }
}

/*********************************************************************/
/* Kills all ESI Threads */
/*********************************************************************/
void
KillESIThreads (void)
{
   int i;
   printf ("Disabling ALL ESI reponses.\n");
   for (i = 0; i < MAX_ESI_ENTRIES; i++)
   {
      if (esi_tble[i].t_hdle != 0)
      {
#ifdef SNS_LINUX
         pthread_cancel(*(esi_tble[i].t_hdle));
         free(esi_tble[i].t_hdle);
         esi_tble[i].t_hdle = 0;
#else
         TerminateThread (esi_tble[i].t_hdle, -1);
#endif
      }
   }
}
