/*
 * Copyright (c) 2003-2005 PyX Technologies, Inc.
 * Copyright (c) 2005 SBE, Inc.
 * 
 * This file houses the main functions for the
 * iSCSI Initiator Authentication Daemon.
 *
 * Nicholas A. Bellinger <nab@kernel.org>
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <iscsi_auth.h>

auth_conn_t auth_table[MAX_AUTH_CONN];

#ifdef CANSRP
extern char srp_main_loop (auth_conn_t *, int, char *, char *, int *, int *);
#endif /* CANSRP */
extern char chap_main_loop (auth_conn_t *, int, char *, char *, int *, int *);

extern void convert_null_to_semi (char *buf, int len)
{
	int i;

	for (i = 0; i < len; i++)
		if (buf[i] == '\0')
			buf[i] = ';';
	
	return;
}

extern int strlen_semi (char *buf)
{
	int i = 0, ret;

	while (buf) {
		ret = strncmp(buf, ";", 1);
		if (!ret)
			return(i);
		buf++;
		i++;
	}
		
	return(-1);	
}
	
extern int extract_param (const char *in_buf, const char *pattern, unsigned int max_length, char *out_buf, unsigned char *type)
{
	char *ptr;
	int i = 0, j = 0;

	if (!in_buf || !pattern || !out_buf || !type)
		return(-1);

	if (!(ptr = strstr(in_buf, pattern)))
		return(-1);

	ptr = strstr(ptr, "=");
	if (!ptr) 
		return(-1);
		
	ptr += 1;
	if (*ptr== '0' && (*(ptr+1) == 'x' || *(ptr+1) == 'X')) {
		ptr += 2; /* skip 0x */
		*type = HEX;
	} else if (*ptr== '0' && (*(ptr+1) == 'b' || *(ptr+1) == 'B')) {
		ptr += 2; /* skip 0x */
		*type = BASE64;
	} else
		*type = DECIMAL;

	while (ptr[i] != ';') {
		if (j > max_length) {
			printf("Length of input exeeds max_length: %d\n", max_length);
			return(-1);	
		}

		out_buf[j++] = ptr[i++];
	}

	out_buf[j] = '\0';

	return(0);
}

extern void remove_auth_conn (unsigned int auth_id)
{
	int i;

	for (i = 0; i < MAX_AUTH_CONN; i++) {
		if (auth_table[i].auth_id == auth_id) {
			memset(&auth_table[i], 0, sizeof(auth_conn_t));
			break;
		}
	}
	
	return;
}

static auth_conn_t *get_auth_conn (unsigned int auth_id, unsigned int sid, unsigned short cid)
{
	int i;

	for (i = 0; i < MAX_AUTH_CONN; i++) {
		if (auth_table[i].auth_id == auth_id)
			return(&auth_table[i]);
	}

	for (i = 0; i < MAX_AUTH_CONN; i++) {
		if (!auth_table[i].in_use) {
			auth_table[i].in_use = 1;
			auth_table[i].auth_id = auth_id;
			auth_table[i].sid = sid;
			auth_table[i].cid = cid;
			return(&auth_table[i]);
		}
	}

	return(NULL);
}

extern int iscsi_get_client_auth_info (auth_conn_t *auth)
{
	const char delim[] = " ";
	char buf[MAX_BUF], *ptr;
	char *channel, *ma, *user, *passwd, *mauser, *mapasswd;
	FILE *file;
	int channel_id, ma_val;

	if (!(file = fopen(CLIENT_AUTHPATH, "r"))) {
		printf("%s: Unable to fopen() %s\n", INITIATOR_AUTH, CLIENT_AUTHPATH);
		return(-1);
	}

	while (fgets(buf, MAX_BUF, file) != NULL) {
		if (strstr(buf, "#"))
			continue;

		ptr = buf;

		/*
		 * Get past the 'AUTH="'
		 */
		ptr += 6;

		/*
		 * Get rid of the trailing '"'
		 */
		ptr[strlen(ptr) - 2] = '\0';
		
		if (!(channel = (char *)strtok(ptr, delim)))
			continue;
		
		ptr += strlen(channel) + 1;
		channel_id = atoi(channel);

		if (channel_id != auth->channel_id)
			continue;

		if (!(ma = (char *)strtok(ptr, delim)))
			continue;
		ptr += strlen(ma) + 1;
		ma_val = atoi(ma);
		auth->authenticate_target = ma_val;
		
		if (!(user = (char *)strtok(ptr, delim)))
			continue;
		ptr += strlen(user) + 1;
		strncpy(auth->userid, user, strlen(user));
		
		if (!(passwd = (char *)strtok(ptr, delim)))
			continue;
		ptr += strlen(passwd) + 1;
		strncpy(auth->password, passwd, strlen(passwd));
		
		if (!auth->authenticate_target) {
			fclose(file);
			return(0);
		}
		
		if (!(mauser = (char *)strtok(ptr, delim)))
			continue;
		ptr += strlen(mauser) + 1;
		strncpy(auth->ma_userid, mauser, strlen(mauser));
		
		if (!(mapasswd = (char *)strtok(ptr, delim)))
			continue;
		strncpy(auth->ma_password, mapasswd, strlen(mapasswd));

		fclose(file);
		return(0);
	}

	printf("%s: Unable to locate authentication entry for iSCSI Channel: %d\n",
			INITIATOR_AUTH, auth->channel_id);
	
	fclose(file);
	return(-1);
}

#ifdef DEBUG_AUTHD

static int authd_verbose(char *argv, int rc, auth_daemon_t auth, int dir)
{
	printf("-----------------------------------------------------\n");
	if (dir) {
		printf("%s: sent       %d bytes\n", argv, rc);
	} else {
		printf("%s: received   %d bytes\n", argv, rc);
	}
	printf("%s: cid:       %hd\n", argv, auth.cid);
	printf("%s: sid:       %u\n", argv, auth.sid);
	printf("%s: auth_id:   %u\n", argv, auth.auth_id);
	printf("%s: kill_auth_id: %u\n", argv, auth.kill_auth_id);
	printf("%s: type:      %s\n", argv, auth.type);
	printf("%s: in_len:    %d\n", argv, auth.in_len);
	printf("%s: out_len:   %d\n", argv, auth.out_len);
	printf("-----------------------------------------------------\n");
	return 0;
}

#endif /* DEBUG_AUTHD */

extern int main (int argc, char *argv[])
{
	int 	addrlen, sock, rc;
	int	retval = -1;
	auth_conn_t *auth_conn;
	auth_daemon_t iscsi_auth;
	pid_t	pid;
	struct sockaddr_in s_addr, c_addr;

	if ((sock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		printf("%s: socket() failed!\n", argv[0]);
		retval -= 1;
		goto authout;
	}

	memset (&s_addr, 0, sizeof(struct sockaddr_in));
	s_addr.sin_family	= AF_INET;
	s_addr.sin_addr.s_addr	= htonl (INADDR_LOOPBACK);
	s_addr.sin_port		= htons (AUTH_PORT);

	if (bind(sock, (struct sockaddr *)&s_addr, sizeof(struct sockaddr)) < 0) {
		printf("%s: bind() failed!\n", argv[0]);
		retval -= 2;
		goto authout;
	}

	if ((pid = fork()) < 0) {
		retval -= 4;
		goto authout;
	} else if (pid != 0) {
		retval = 0;
		goto authout;
	}

	printf("PyX iSCSI Initiator Authentication Daemon %s spawned at pid %d\n",
		VERSION, getpid());

	while (1) {
		addrlen = sizeof(c_addr);
		
		if ((rc = recvfrom(sock, &iscsi_auth, sizeof(auth_daemon_t), 0,
		    (struct sockaddr *)&c_addr, &addrlen)) < 0) {
			printf("%s: recvfrom() returned %d, exiting\n",
					argv[0], rc);
			continue;
		}

#ifdef DEBUG_AUTHD
		authd_verbose(argv[0], rc, iscsi_auth, 0);
#endif
		memset(iscsi_auth.out_text, 0, TEXT_LEN);

		if (iscsi_auth.kill_auth_id) {
			remove_auth_conn(iscsi_auth.auth_id);
#if 0
			if (strstr("None", iscsi_auth.type))
				iscsi_auth.type[0] = '1';
			else
#endif
				iscsi_auth.type[0] = '2';
			goto out;
		}

		if (!(auth_conn = get_auth_conn(iscsi_auth.auth_id,
				iscsi_auth.sid, iscsi_auth.cid))) {
			iscsi_auth.type[0] = '2';
			goto out;
		}

		auth_conn->channel_id = iscsi_auth.channel_id;
		
		/* Nothing to do */
		if (strstr("None", iscsi_auth.type))
			iscsi_auth.type[0] = '1';
#ifdef CANSRP
		else if (strstr("SRP", iscsi_auth.type))
			iscsi_auth.type[0] = srp_main_loop(auth_conn, iscsi_auth.role,
					iscsi_auth.in_text, iscsi_auth.out_text,
					&iscsi_auth.in_len, &iscsi_auth.out_len);
#endif /* CANSRP */
		else if (strstr("CHAP", iscsi_auth.type))
			iscsi_auth.type[0] = chap_main_loop(auth_conn, iscsi_auth.role,
					iscsi_auth.in_text, iscsi_auth.out_text,
					&iscsi_auth.in_len, &iscsi_auth.out_len);
		else if (strstr("SPKM1", iscsi_auth.type))
			iscsi_auth.type[0] = '2';
		else if (strstr("SPKM2", iscsi_auth.type))
			iscsi_auth.type[0] = '2';
		else if (strstr("KRB5", iscsi_auth.type))
			iscsi_auth.type[0] = '2';
		else
			iscsi_auth.type[0] = '2';
out:
		iscsi_auth.type[1] = '\0';

		if ((rc = sendto(sock, &iscsi_auth, sizeof(auth_daemon_t), 0,
		    (struct sockaddr *)&c_addr, addrlen)) < 0) {
			printf("%s: sendto() returned %d, exiting\n",
				argv[0], rc);
			continue;
		}
#ifdef DEBUG_AUTHD
		authd_verbose(argv[0], rc, iscsi_auth, 1);
#endif
	}
	retval = 0;

authout:
	return retval;
}
