#' Deploy a Document
#'
#' Deploys an application consisting of a single R Markdown document or other
#' single file (such as an HTML or PDF document).
#'
#' @param doc Path to the document to deploy.
#' @param ... Additional arguments to [deployApp()]. Do not supply `appDir`,
#'   `appFiles`, or `appPrimaryDoc`; these three parameters are automatically
#'   generated by `deployDoc` from the document.
#'
#' @details When deploying an R Markdown document, any files which are required
#'   to render and display the file must be deployed.
#'
#'   This method discovers these additional files using
#'   [rmarkdown::find_external_resources()] from \pkg{rmarkdown}.
#'
#'   If you find that the document is missing dependencies, either specify the
#'   dependencies explicitly in the document (the documentation for
#'   [rmarkdown::find_external_resources()] explains how to do this), or call
#'   [deployApp()] directly and specify your own file list in the `appFiles`
#'   parameter.
#'
#' @family Deployment functions
#' @export
deployDoc <- function(doc, ...) {
  # validate inputs
  if (!file.exists(doc)) {
    stop("The document '", doc, "' does not exist.")
  }
  if (!requireNamespace("rmarkdown") ||
      packageVersion("rmarkdown") < "0.5.2") {
    stop("Version 0.5.2 or later of the rmarkdown package is required to ",
         "deploy individual R Markdown documents.")
  }

  # get qualified doc
  qualified_doc <- normalizePath(doc, winslash = "/")

  # see if this doc has runtime: shiny_prerendered, if it does then
  # appFiles will be NULL (bundle the entire directory)
  if (isShinyRmd(doc)) {
    app_files <- NULL
  } else {
    # default to deploying just the single file specified
    app_files <- basename(qualified_doc)

    # if this document's type supports automated resource discovery, do that now,
    # and add the discovered files to the deployment list
    ext <- tolower(tools::file_ext(doc))
    if (ext %in% c("rmd", "qmd", "html", "htm")) {
      message("Discovering document dependencies... ", appendLF = FALSE)
      res <- rmarkdown::find_external_resources(qualified_doc)
      message("OK")
      app_files <- c(app_files, res$path)
    }
  }

  # deploy the document with the discovered dependencies
  deployApp(appDir = dirname(qualified_doc),
            appFiles = app_files,
            appPrimaryDoc = basename(qualified_doc),
            ...)
}
