#!/usr/bin/env python
# Author: Greg Caporaso (gregcaporaso@gmail.com)
# test_gctmpca.py

""" Tests of the Generalized Continuous-Time Markov Process Coevolutionary 
 Algorithm (GCTMPCA) application controller.

"""
from __future__ import division
from os import environ
from cogent.util.unit_test import TestCase, main
from cogent.app.util import ApplicationError
from cogent.util.misc import app_path
from cogent.app.gctmpca import Gctmpca, gctmpca_aa_order, \
 default_gctmpca_aa_priors, gctmpca_base_order

__author__ = "Greg Caporaso"
__copyright__ = "Copyright 2007-2016, The Cogent Project"
__credits__ = ["Greg Caporaso"]
__license__ = "GPL"
__version__ = "1.9"
__maintainer__ = "Greg Caporaso"
__email__ = "gregcaporaso@gmail.com"
__status__ = "Beta"

class GctmpcaTests(TestCase):
    """ Tests of the Gctmpca coevolution detection app controller """

    def setUp(self):
        """ """
        self.run_slow_tests = int(environ.get('TEST_SLOW_APPC',0))
        self.sample_seqs = sample_seqs.split('\n')       
        self.sample_tree = sample_tree.split('\n')        
        self.sample_species_names = sample_species_names.split('\n') 
        self.sample_priors = sample_priors
        self.sample_sub_matrix = sample_sub_matrix
        self.sample_seq_to_species = sample_seq_to_species.split('\n')       
        self.sample_data = {'mol_type':'rna','seqs1':self.sample_seqs,\
         'tree1':self.sample_tree,'seq_names':self.sample_species_names,\
         'species_tree':self.sample_tree,\
         'seq_to_species1':self.sample_seq_to_species,\
         'char_priors':self.sample_priors,'sub_matrix':self.sample_sub_matrix,
         'single_pair_only':1,'pos1':1265,'pos2':1270,'epsilon':0.1}
    
        self.trivial_seqs = trivial_seqs.split('\n')       
        self.trivial_tree = trivial_tree.split('\n')        
        self.trivial_species_names = trivial_species_names.split('\n') 
        self.trivial_seq_to_species = trivial_seq_to_species.split('\n')
        #self.trivial_priors = trivial_priors.split('\n')    
        #self.trivial_sub_matrix = trivial_sub_matrix.split('\n')        
        self.trivial_data = {'mol_type':'protein','seqs1':self.trivial_seqs,\
         'tree1':self.trivial_tree,'seq_names':self.trivial_species_names,\
         'species_tree':self.trivial_tree,\
         'seq_to_species1':self.trivial_seq_to_species,\
         'single_pair_only':0,'epsilon':'0.7'}
     
        self.myog_seqs = myog_seqs.split('\n')       
        self.myog_tree = myog_tree.split('\n')        
        self.myog_species_names = myog_species_names.split('\n') 
        self.myog_seq_to_species = myog_seq_to_species.split('\n')
        #self.myog_priors = myog_priors.split('\n')    
        #self.myog_sub_matrix = myog_sub_matrix.split('\n')        
        self.myog_data = {'mol_type':'protein','seqs1':self.myog_seqs,\
         'tree1':self.myog_tree,'seq_names':self.myog_species_names,\
         'species_tree':self.myog_tree,\
         'seq_to_species1':self.myog_seq_to_species,\
         'single_pair_only':0,'epsilon':'0.1'}
 
        self.app = Gctmpca()

    def test_calculate_likelihood_in_path(self):
        """Gctmpca: calculate_likelihood binary in path """
        assert app_path('calculate_likelihood'),\
            "calculate_likelihood is not in your path -- other tests will fail!"

    def test_trivial_data(self):
        """Gctmpca: full computation works with trivial example """
        if not self.run_slow_tests: return      
        expected = ['position 1\tposition 2\tscore\n','2\t3\t3.705879\n']
        #app = Gctmpca(HALT_EXEC=True)
        #app(self.trivial_data)
        #exit(-1)
        actual = list(self.app(self.trivial_data)['output'])
        self.assertEqual(actual,expected)

    def test_trivial_data_pairwise(self):
        """Gctmpca: pairwise computation works with trivial example """
        if not self.run_slow_tests: return      
        expected = ['position 1\tposition 2\tscore\n']
        self.trivial_data['single_pair_only'] = 1
        self.trivial_data['pos1'] = 1
        self.trivial_data['pos2'] = 2
        actual = list(self.app(self.trivial_data)['output'])
        self.assertEqual(actual,expected)

    def test_myog_data_pairwise(self):
        """Gctmpca: pairwise computation works with myog example """
        if not self.run_slow_tests: return      
        
        expected = ['position 1\tposition 2\tscore\n','44\t108\t0.364928\n']
        self.myog_data['single_pair_only'] = 1
        self.myog_data['pos1'] = 44
        self.myog_data['pos2'] = 108
        actual = list(self.app(self.myog_data)['output'])
        self.assertEqual(actual,expected)

    def test_rna_sample_data(self):
        """Gctmpca: works with provided sample data """
        expected = ['position 1\tposition 2\tscore\n',\
            '1265\t1270\t33.862890\n']
        actual = list(self.app(self.sample_data)['output'])
        self.assertEqual(actual,expected)

    def test_amino_acid_defaults(self):
        """Gctmpca: cogent defaults match those presented by Yeang et al.
        """
        expected_aa_order = 'ARNDCQEGHILKMFPSTWYV'
        self.assertEqual(gctmpca_aa_order,expected_aa_order)
        gctmpca_priors_string = \
            "0.087127\t0.040904\t0.040432\t0.046872\t0.033474\t0.038255\t" +\
            "0.049530\t0.088612\t0.033618\t0.036886\t0.085357\t0.080482\t" +\
            "0.014753\t0.039772\t0.050680\t0.069577\t0.058542\t0.010494\t" +\
            "0.029916\t0.064718\n"
        gctmpca_priors = dict(zip(gctmpca_aa_order,\
            map(float,gctmpca_priors_string.strip().split('\t'))))
        cogent_priors = default_gctmpca_aa_priors
        self.assertFloatEqual(cogent_priors,gctmpca_priors)

    def test_default_sub_matrix_and_priors_used(self):
        """Gctmpca: works with provided sample data mixed w/ defaults"""
        # Works with default priors and sub matrix
        expected = ['position 1\tposition 2\tscore\n',\
            '1265\t1270\t33.862890\n']
        del self.sample_data['char_priors']
        del self.sample_data['sub_matrix']
        actual = list(self.app(self.sample_data)['output'])
        self.assertEqual(actual,expected)

    def test_gctmpca_cl_input_result_sanity(self):
        """Gctmpca: correct number of parameters on command line"""
        actual = self.app._gctmpca_cl_input(self.sample_data)
        self.assertTrue(actual.startswith('0 0 "/tmp/tmp'))
        # Many of the intermediate paramters are randomly generated
        # file names.
        self.assertTrue(actual.endswith('.txt" 1 - 1265 1270'))
        parameter_count = len(actual.split())
        self.assertEqual(parameter_count,22)

    def test_gctmpca_cl_input_result_sanity_w_some_default(self):
        """Gctmpca: correct number of parameters on command line w some defaults
        """
        # remove some parameters and make sure the result changes
        del self.sample_data['single_pair_only']
        del self.sample_data['pos1']
        del self.sample_data['pos2']
        actual = self.app._gctmpca_cl_input(self.sample_data)
        self.assertTrue(actual.startswith('0 0 "/tmp/tmp'))
        # Many of the intermediate paramters are randomly generated
        # file names.
        self.assertTrue(actual.endswith('.txt" 0 -'))
        parameter_count = len(actual.split())
        self.assertEqual(parameter_count,20)

    def test_input_as_gctmpca_char_priors(self):
        """Gctmpca: priors input handler works as expected """
        priors = {'A':0.50,'C':0.25,'G':0.15,'U':0.10}
        expected = ['0.5\t0.25\t0.15\t0.1','']
        actual = self.app._input_as_gctmpca_char_priors(priors,'ACGU')
        self.assertEqual(actual,expected)
        expected = ['0.1\t0.15\t0.25\t0.5','']
        actual = self.app._input_as_gctmpca_char_priors(priors,'UGCA')
        self.assertEqual(actual,expected)
    
    def test_input_as_gctmpca_rate_matrix(self):
        """Gctmpca: rate matrix input handler works as expected """
        m = {'A':{'A':-13.4,'C':3.2,'D':2.2},\
             'C':{'A':3.4,'C':-3.2,'D':22.2},\
             'D':{'A':0.4,'C':0.0,'D':-1.}}
        expected = ['-13.4\t3.2\t2.2','3.4\t-3.2\t22.2','0.4\t0.0\t-1.0']
        actual = self.app._input_as_gctmpca_rate_matrix(m,'ACD')
        self.assertEqual(actual,expected)
        expected = ['-1.0\t0.4\t0.0','2.2\t-13.4\t3.2','22.2\t3.4\t-3.2']
        actual = self.app._input_as_gctmpca_rate_matrix(m,'DAC')
        self.assertEqual(actual,expected)

    def test_unsupported_function_raises_error(self):
        """Gctmpca: attempting an intermolecular experiment causes error """
        self.sample_data['comparison_type'] = 1
        self.assertRaises(NotImplementedError,self.app,self.sample_data)

    def test_gctmpca_cl_input_error(self):
        """Gctmpca: missing required data causes ApplicationError """
        # missing all
        self.assertRaises(ApplicationError,self.app,{})
        # missing one
        # mol_type
        data = {'seqs1':self.sample_seqs,\
         'tree1':self.sample_tree,'seq_names':self.sample_species_names,\
         'species_tree':self.sample_tree,\
         'seq_to_species1':self.sample_seq_to_species} 
        self.assertRaises(ApplicationError,self.app,data)
        # seqs1
        data = {'mol_type':'rna',\
         'tree1':self.sample_tree,'seq_names':self.sample_species_names,\
         'species_tree':self.sample_tree,\
         'seq_to_species1':self.sample_seq_to_species}
        self.assertRaises(ApplicationError,self.app,data)
        # tree1
        data = {'mol_type':'rna','seqs1':self.sample_seqs,\
         'seq_names':self.sample_species_names,\
         'species_tree':self.sample_tree,\
         'seq_to_species1':self.sample_seq_to_species}         
        self.assertRaises(ApplicationError,self.app,data)
        # 'seq_names'
        data = {'mol_type':'rna','seqs1':self.sample_seqs,\
         'tree1':self.sample_tree,\
         'species_tree':self.sample_tree,\
         'seq_to_species1':self.sample_seq_to_species}
        self.assertRaises(ApplicationError,self.app,data)
        # species_tree
        data = {'mol_type':'rna','seqs1':self.sample_seqs,\
         'tree1':self.sample_tree,'seq_names':self.sample_species_names,\
         'seq_to_species1':self.sample_seq_to_species}
        self.assertRaises(ApplicationError,self.app,data)
        # seq_to_species1
        data = {'mol_type':'rna','seqs1':self.sample_seqs,\
         'tree1':self.sample_tree,'seq_names':self.sample_species_names,\
         'species_tree':self.sample_tree}
        self.assertRaises(ApplicationError,self.app,data)
        # pos1 if single_pair_only == 1
        data = {'mol_type':'rna','seqs1':self.sample_seqs,\
         'tree1':self.sample_tree,'seq_names':self.sample_species_names,\
         'species_tree':self.sample_tree,\
         'seq_to_species1':self.sample_seq_to_species,\
         'single_pair_only':1,'pos2':5}
        self.assertRaises(ApplicationError,self.app,data)
        # pos2 if single_pair_only == 2
        data = {'mol_type':'rna','seqs1':self.sample_seqs,\
         'tree1':self.sample_tree,'seq_names':self.sample_species_names,\
         'species_tree':self.sample_tree,\
         'seq_to_species1':self.sample_seq_to_species,\
         'single_pair_only':1,'pos1':5}
        self.assertRaises(ApplicationError,self.app,data)
        # pos1 & pos2 if single_pair_only == 2
        data = {'mol_type':'rna','seqs1':self.sample_seqs,\
         'tree1':self.sample_tree,'seq_names':self.sample_species_names,\
         'species_tree':self.sample_tree,\
         'seq_to_species1':self.sample_seq_to_species,\
         'single_pair_only':1}
        self.assertRaises(ApplicationError,self.app,data)

sample_seqs = """146 1542
cy.Mic.aer  CCGAGCUCGAGUUUGAUCCUGGCUCAGGAUGAACGCUGGCGGCGUGCCUAACACAUGCAAGUCGAACGGGAAUC--------UUCG-------GAUU-CUAGUGGCGGACGGGUGAGUAACGCGUAAGAAUCUAACUUCAGGACGGGGACAACAGUUGGAAACGACUGCUAAUACCCGAUG-UGCCGCAGGGUGAAACC--------UAAU------UGGCCUGGAGAAGAGCUUGCGUCUGAUUAGCUAGUUGGUGGGGUAAGGGCCUACCAAGGCGACGAUCAGUAGCUGGUCUGAG-GGAUGAGCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGAAAGCCUGACGGAGCAACGCCGCGUGAGGGAGGAAGGUCUUUGGAUUGUAAACCUCUUUUCUCAAGGAAGAAG----------UUCU-------------AACGGUUCUUGAGGAAUCAGCCUCGGCUAACUCCGUGCCAGCAGCCCCGGUAAUACGGGGGAGGCAAGCGUUAUCCGGAAUUAUUGGGCGUAAAGCGUCCGCAGGUGGUCAGCCAAGUCUGCCGUCAAAUCAGGUUGCUUAACGACCUAAAGGCGGUGGAAACUGGCAGACUAGAGAGCAGUAGGGGUAGCAGGAAUUCCCAGUGUAGCAGUGAAAUGCGUAGAGAUUGGGAAGAACAUCGGUGGCGAAAGCGUGCUACUGGGCUGUCUCUGACACUCAGGGACGAAAGCUUGGGGAGCGAACGGGAUUAGAUACCCCUGUAGUCCUAGCCGUAAACGAUGGAUACUAGGUGUGGGUUGUAUCGCCCGAGGCGGGCCAAAGAUAACGCGUUAUGUAUCCCCCCUGGAGAGUACGCACGCAAGUUUGAAUCUCAAAGGUAUUGACGGGGGCCCGCACAAGCGGUGGAGUAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCAAG-AUUGACAUGUCGCGAACCCUGGUGAAAGCUGGGGGUGCCUUCGGGAGCGCCAACACAGAUGGUGUAUAGCUGUCGCCAGCUCGUGUCGUGAGAUGUUGGGUUAUGUCCCGCAAAGAGCGCACCCCUCGUUCUUAGUUGCCAGCAUUAAG-UGGACGACUCUAAGGAGACUGCCGGUGACAAACCGGAGGAAGGUGGUGAUGACGUCAAGUCAGCAUGCCCCUUACGUCUUGGGCGACACACGUACUACAGUCGUCGGAACAAAAGGCAGUCAACUCGCGAGAGCCAGCGUAUCCCGCAAACCCGGCCUCAGUUCAGAUUGCAGGCUGCAACUCGCCUGCAUGAAGGAGGAAUCGCUAGUAAUCGCCGGUCAGCAUACGGCGGUGAAUUCGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGAAGCUGGUCACGCCCGAAGUCAUUACCUCAACCGCAAGAGGGGGGAUGCCUAAGGCAGGGCUAGUGACUGGGGUmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
cy.Osc.aga  -----mmmmmmmmmmmmmmmmmmmmmmGAUGAACGCUGGCGGUCUGCUUAACACAUGCAAGUCGAACGGAAUCC--------UUCG--------GGAUUUAGUGGCGGACGGGUGAGUAACACGUAAGAACCUGCCUCUAGGACGGGGACAACAGUUGGAAACGACUGCUAAACCCGGAUG-AGCCGAAAGGUAAAAGA--------UUAA------UCGCCUAGAGAGGGGCUUGCGUCUGAUUAGCUAGUUGGUGAGGUAAGAGCCCACCAAGGCGACGAUCAGUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGAAAGC-UGACGGAGCAAGACCGCGUGGGGGAGGAAGGUUCUUGGAUUGUCAACCCCUUUUCUCAGGGAAGAACA---------CAAU-------------GACGGUACCUGAGGAAAAAGCAUCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGGGGAUGCAAGCGUUAUCCGGAAUGAUUGGGCGUAAAGAGUCCGUAGGUAGUCAUCCAAGUCUGCUGUUAAAGAGCGAGGCUUAACCUCGUAAAGGCAGUGGAAACUGGAAGACUAGAGUGUAGUAGGGGCAGAGGGAAUUCCUGGUGUAGCGGUGAAAUGCGUAGAGAUCAGGAAGAACACCGGUGGCGAAGGCGCUCUGCUGGGCUAUAACUGACACUGAGGGACGAAAGCUAGGGGAGCGAAUGGGAUUAGAUACCCCAGUAGUCCUAGCGGUAAACGAUGGAAACUAGGUGUGGCCUGUAUCGCCCGGGCCGUGCCGAAGCAAACGCGUUAAGUUUCCCGCCUGGGGAGUACGCACGCAAGUGUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGUAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCAGGACUUGACAUCUCUGGAAUCUCCUUGAAAGGGGAGAGUGCCGUAAGGAACCAGAAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUCGUUAGUUGCCAUCAUUAAG-UUGGGAACUCUAGCGAGACUGCCGGUGACAAACCGGAGGAAGGUGAGGAUGACGUCAAGUCAGCAUGGCCCUUACGUCCUGGGCGACACACGUACUACAAUGCGAGGGACAGAGAGCAGCCAACCCGCGAGGGAGAGCGAAUCUCAUAAACCCUGGCACAGUUCAGAUUGCAGGCUGCAACUCGCCUGCAUGAAGGAGGAAUCGCUAGUAAUCGGAGGUCAGCAUACUGCGGUGAAUCCGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGAAGUGAGCCACGCCCGAAGUCAUUACUCUAACCGAAAGGAGGAGGGUGCCGAAGGCGGGGUUGAUGACUGGGGUGAAGUCGUAACAAGGUAGCCGUACCGGAAGGUGUGGCUGGAUCACCUCCUUU
cy.Pho.amb  -----mmmmmmmmmmmmmmmmmmmmmmGAUGAACGCUGGCGGUCUGCUUAACACAUGCAAGUCGAACGAACUC---------UUCG---------GAGUUAGUGGCGGACGGGUGAGUAACGCGUGAGAGUCUAGCUUCAGAACGGGGACAACAGAGGGAAACCACUGCUAAUACCCGAUG-UGCCGAGAGGUGAAAGA--------UUUA------UCAUCUGAAGAUGAGCUCGCGUCCGAUUAGCUAGUUGGUAAGGUAAGAGCUUACCAAGGCGACGAUCGGUAGCUGGUCUGAGAGGAUGACCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGAAAGC-UGACGGAGCAAGACCGCGUGAGGGAGGAAGGCUCUUGGGUUGUAAACCUCUUUUCUCUGGGAAGAAG----------AUCU-------------GACGGUACCAGAGGAAUCAGCAUCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGAUGCAAGCGUUAUCCGGAAUUAUUGGGCGUAAAGCGUCCGUAGGUGGCGGUUCAAGUCUGCCGUUAAAACCAGUAGCUUAACUACUGACAGGCGGUGGAAACUGAACAGCUAGAGUGUGGUAGGGGUAGAGGGAAUUCCCAGUGUAGCGGUGAAAUGCGUAGAGAUUGGGAAGAACACCGGUGGCGAAAGCGCUCUGCUGGACCACAACUGACACUCACGGACGAAAGCUAGGGGAGCGAAAGGGAUUAGAUACCCCAGUAGUCCUAGCCGUAAACGAUGGAUACUAGGUGUAGUAGGUAUCGCCCCUACUGUGCCGUAGCUAACGCGUUAAGUAUCCCGCCUGGGGAGUACGGUCGCAAGAUUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGUAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCAGGACUUGACAUGUCGCGAAUCUCUCUGAAAGGAGAGAGUGCCUUCGGGAGCGCGAACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCACGUUUUUAGUUGCCAUCAUUAAG-UUGGGCACUCUAGAGAGACUGCCGGUGACAAACCGGAGGAAGGUGUGGAUGACGUCAAGUCAGCAUGCCCCUUACGUCCUGGGCUACACACGUACUACAAUGGCUGGGACAAAGAGCUGCAAGCGAGCGAUCGCAAGCCAAUCUCAUAAACCCAGUCUUAGUUCAGAUCGCAGGCUGCAACUCGCCUGCGUGAAGGCGGAAUCGCUAGUAAUCGCAGAUCAGAAUGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGAAGCUGGUCACGCCCGAAGUCGUUAUCCUAACCUUCGGGAGGGAGAUGCCUAAGGCAGGGCUGGUGACUGGGGUGAAGUCGUAACAAGGUAGC-GUACCGGAAGGmmmmmmmmmmmmmmmm-----
cy.Tri.ten  -----mmmmmmmmmmmmmmmmmmmmmmmAUGAACGCUGGCGGUCUGCUUAACACAUGCAAGUCGAACGGACUC---------UUCG---------GAGUUAGUGGCGGACGGGUGAGUAACGCGUGAGAAUCUGCCUUCAGGUCUGGGACAACAGAAGGAAACUUCCGCUAAUCCCGGAUG-AGCCUUAGGGUAAAAGA--------UAAA------UUGCCUGGAGAUGAGCUUGCGUCUGAUUAGCUAGUUGGUGUGGUAAAAGCAUACCAAGGCAACGAUCAGUAGCUGGUCUGAGAGGAUGAGCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGCAAGCCUGACGGAGCAAGACCGCGUGGGGGAGGAAGGCUCUAGGGUUGUAAACCCCUUUUCUUUGGGAAGAAG----------AAAU-------------GACGGUACCAAAGGAAUCAGCCUCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGAGGCAAGCGUUAUCCCGAAUGAUUGGGCGUAAAGCGUCCGCAGGUGGCCAUGUAAGUCUGCUGUUAAAACCCAGGGCUUAACUCUGGUCAGGCAGUGGAAACUACAAAGCUAGAGUUCGGUAGGGGCAAAGGGAAUUCCCGGUGUAGCGGUGAAAUGCGUAGAUAUCGGGAAGAACAUCGGUGGCGAAAGCGCUUUGCUAGACCGAAACUGACGCUCAGGGACGAAAGCUAGGGGAGCGAAUGGGAUUAGAUACCCCAGUAGUCCUAGCCGUAAACGAUGGAUACUAGGUGUUGCCUGUAUCGCCCAGGCAGUGCCGUAGCUAACGCGUUAAGUAUCCCGCCUGGGGAGUACGCACGCAAGUGUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGUAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCAGGGCUUGACAUGUCGCGAAUCUCAAGGAAACUUGAGAGUGCCUUCGGGAGCGCGAACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUCUUUAGUUGCCAUCAUUAAG-UUGGGCACUCUGGAGAGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAGUCAGCAUGCCCCUUACGCUCUGGGCUACACACGUACUACAAUGGUUGGGACAGAGGGCAGCGAACCCGAGAGGGGAAGCCAAUCCCCAAAACCCAGCCUCAGUUCAGAUUGCAGGCUGCAACUCGCCUGCAUGAAGGAGGAAUCGCUAGUAAUCGCAGGUCAGCAUACUGCGGUGAAUCCGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGAAGUUGGCCACGCCCGAAGUCAUUACUCUAACCGAAAGGAGGGGGAUGCCGAAGGCAGGGCUGAUGACUGGGGUGAAGUCGUAACAAGGUAGCCGUACCGGAAmmmmmmmmmmmmmmmmmm-----
pl.Zea.may  -----UACCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUGCAAGUAUGAACUAA-----CGAA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUGUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCAAAACCCUCC-GGGCGGAUCGCA--CGGUUCGCGGCG-ACG-CAUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGGGGCCUACCAUGGUGGUGACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACCG-GGCGCGUUAGUGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGC-CGGGC---------CGGG-UG-CC-GCCGUGGCA----G--AACCA------CC-----AUUAAUAGGGCAGUGGGGGCACGUAUUUCAUAGUCAGAGGUGAAAUUCUUGGAUUUAUGAAAGACGAACAACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACCAGGGAUCAGCGG-UAAUCCCCGCUGGCCCUUAUGAGAAAUCA--AAGUCGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGCGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---CUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGUCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAACUAGCUAUGCCAUC--UUAGCUUCUUAGAGGGACUAUGGCCGUUUAGGCCGCG-AAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGUAUCCAACGAGUACCUUGGCCGACAGGCC-GG-UAAUCUUGGAAAUUUCAUCGUGAUGGGGAUAGAUCAUUGCAAUUGUUGGUCUUCAACGAGGAAUGCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGGUCCGGUGA-AGUGUUCGGACUCGGUUCGCCGGAAGUCCAUUGAACCUUAUCAUUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUG-----
pl.Pan.gin  ----CAACCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUGUAAGUAUGAACUAA-----CAGA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUGUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCAAAACCCUCU-GGACGGAUCGCA--CGGCUCGCGGCG-ACG-CAUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACUAUGGUGGUGACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACCG-GGCUG-AUUCAGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGU-UGGGUC-----GGCCGGUCCG-CC--UCUCGGUG---UG--CACCA------UC-----AUUAACAGGGCAGUGGGGGCACGUAUUUCAUAGUCAGAGGUGAAAUUCUUGGAUUUAUGAAAGACGAACAACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACCAGGGAUCAGUGGAUUUUCUCCACUGGCCCUUAUGAGAAAUCA--AAGUUGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---CUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAAAUAGCUAUGUUAUA--CCAGCUUCUUAGAGGGACUAUGGCCUUUUAGGCCACGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGUAUUCAACGAGUACCUUGGCCGACAGGCCCGGGUAAUCUU-GAAAUUUCAUCGUGAUGGGGAUAGAUCAUUGCAAUUGUUGGUCUUCAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGGUCCGGUGA-AGUGUUCGGAUGCGGUUCGAAAGAAGUCCACUGAACCUUAUCAUUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCAGAAGGAUCAGNN-----
pl.Ara.tha  -----UACCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUGUAAGUAUGAACGAA-----CAGA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUGUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCAAAACCCUAU-GGACGGAUCGCA--UGGUCUGUGGCG-ACG-CAUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACCAUGGUGGUAACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACCG-GGCUCUUUCGAGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGA-UGGGUC-----GGCCGGUCCG-CC--UUU-GGUG---UG--CAUUG------UC-----AUUAACAGGGCAGUGGGGGCACGUAUUUCAUAGUCAGAGGUGAAAUUCUUGGAUUUAUGAAAGACGAACAACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACCAGGGAUCAGCGGAUUAUCUCCGCUGGCCCUUAUGAGAAAUCA--AAGUUGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---CUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAACUAGCUACGUCAUC--CCGGCUUCUUAGAGGGACUAUGGCCGUUUAGGCCAAGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGUAUUCAACGAGACCCUU-GCCGACAGGCCCGGGUAAUCUU-GAAAUUUCAUCGUGAUGGGGAUAGAUCAUUGCAAUUGUUGGUCUUCAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGAUCCGGUGA-AGUGUUCGGACGCGGUUCGCGAGAAGUCCACUAAACCUUAUCAUUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUG-----
pl.Gly.max  -----UACUUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUGUAAGUAUGAACUAA-----CAGA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUGUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCAAAACCCUCU-GGACGGAUCGCA--CGGUUUGCGGCG-ACG-CAUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACCAUGGUGGUGACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACCG-GGCUC-AUUGAGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAUGAUCCAUUGAAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGU-UGGGUC-----GAUCGGUCCG-CC--UCC-GGUG---UG--CACCG------UC-----AUUAACAGGGCAGUGGGGGCACGUAUUUCAUAGUCAGAGGUGAAAUUCUUGGAUUUAUGAAAGACGAACAACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACCAGGGAUCAGCGGAUUUUCUCCGCUGGCCCUUAUGAGAAAUCA--AAGUCGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCU-AAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---CUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAAAUAGCUAUGUUAAC--CCAGCUUCUUAGAGGGACUAUGGCCGCUUAGGCCACGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGUAUUCAACGAGUACCUUGGCCGACAGGUCCGGGUAAUCUU-GAAAUUUCAUCGUGAUGGGGAUAGAUCAUUGCAAUUGUUGGUCUUCAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGGUCCGGUGA-AGUGUUCGGAUGCGGUUCGUGAGAAGUCCACUGAACCUUAUCAUUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUG-----
pl.Pis.sat  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNNNNNNNNAAGCCAUGCAUGUGUAAGUAUGAACUAA-----CAGA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUGUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCAAAACCCUUU-GGACGGAUCGCA--CGGUUUGUGGCG-ACG-CAUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACCAUGGUGGUGACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUAACAC-GGGGAGGUAGUGACAAUAAAUCAAUACCG-GGCUC-AUUGAGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGU-UGGGUU-----GAUCGGUCCG-CC--UCU-GGUG---UG--CACCG------UU-----AUUAACAGGGCAGUGGGGGCACGUAUUUCAUAGUCAGAGGUGAAAUUCUUGGAUUUAUGAAAGACGAACAACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACCAGGGAUCAGCGGAUUUUCUCCGCUGGCCCUUAUGAGAAAUCA--AAGUCGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---CUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAAAUAGCUAUGUUAAC--CCAGCUUCUUAGAGGGACUAUGGCCGCUUAGGCCACGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGUAUUCAACGAGUACCUUGGCCGACAGGCCCGGGUAAUCUU-GAAAUUUCAUCGUGAUGGGGAUAGAUCAUUGCAAUUGUUGGUCUUCAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGGUCCGGUGA-AGUGUUCGGAUGCGGUUCGUGAGAAGUCCACUGAACCUUAUCAUUUAGAGGAAGGAGAAGUCGNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
pl.Lyc.esc  -----UACCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUGUAAGUAUGAACAAA-----CAGA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUGUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCAAAACCCUCUGGGACGGAUCGCA--CGGAUCGCGGCG-ACG-CAUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACCAUGGUGGUGACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACCG-GGCUC-UAUGAGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGA-UGGGCC-----GGCCGGUCCG-CC--CUA-GGUG-UG----CACCG------UC-----AUUAACAGGGCAGUGGGGGCACGUAUUUCAUAGUCAGAGGUGAAAUUCUUGGAUUUAUGAAAGACGAACAACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACCAGGGAUCGGCGGAUUUUCUCCGCCGGCCCUUAUGAGAAAUCA--AAGUUGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---CUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAACUAGCUAUGCUAUC--CCAGCUUCUUAGAGGGACUA-GCCU-UUUAGGCCGCGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCC-CACGCGCGCUACACUGAUGUAUUCAACGAGUACCUU-GCCGACAGGCCCGGGUAAUCUU-GAAAUUUCAUCGUGAUGGGGAUAGAUCAUUGCAAUUGUUGGUCUUCAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGAUCCGGUGA-AAUGUUCGGACGCGGUUCGCGAGAAGUCCAUUGAACCUUAUCAUUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUG-----
pl.Sol.tub  -----UACCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUGUAAGUAUGAACAAA-----CAGA--------UGU-GAAACUGCGAAUGGCUCAU-AAAUCAGUUAAGUUUGUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCAAAACCCUCU-GGACGGAUCGCA--CGGAUCGCGGCG-ACG-CAUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACCAUGGUGGUGACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACCG-GGCUC-UAUGAGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGA-UGGCAC-----GGCCGGUCCG-CC--CUA-GGUG---UG--CACCG------UC-----AUUAACAGGGCAGUGGGGGCACGUAUUUCAUAGUCAGAGGUGAAAUUCUUGGAUUUAUGAAAGACGAACAACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGACCGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACCAGGGGUCGGCGGAUUUUCUCCGCCGGCCCUUAUGAGAAAUCA--AAGUUGGUUCCGGGG-GAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---CUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAACUAGCUAUGCUAUC--CCAGCUUCUUAGAGGGACUACGGCCUUUUAGGCCGCGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGUAUUCAACGAGUACCUUGGCCGACAGGCCCGGGUAAUCUU-GAAAUUUCAUCGUGAUGGGGAUAGAUCAUUGCAAUUGUUGGUCUUCAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGAUCCGGUGA-AAUGUUCGGACGCGGUUCGCGAGAAGUCCAUUGAACCUUAUCAUUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUG-----
pl.Gin.bil  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNUCAAAAUAAGCCAUGCAUGUGUAAGUAUGAACUCU-----CAGA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUCUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCACAAUCCUCU-GGACGGAUCGCA--CGGCUGGCGGCG-ACG-CUUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGAGGCCUACCAUGGUGGUGACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACUG-GGCUC-AUCGAGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGC-CGGGUC-----GGCCGGUCCG-CC-UUUUCGGUG---UG--CACCG------CC-----AUUAAUAGGGCGGUGGGGGCACGUAUUUCAUUGUCAGAGGUGAAAUUCUUGGAUUUAUGAAAGACGAACCACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACUAGGGAUCGGCGGAUUUACUCCGCCGGCCCUUGUGAGAAAUCA--AAGUUGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAACUAGCUAUGCUUCG--CCAGCUUCUUAGAGGGACUAUGGCCCUUCAGGCCAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGUAUUCAACGAGUACCUGGGCCGAGAGGCCCGGGAAAUCUGCGAAAUUUCAUCGUGAUGGGGAUAGAUCAUUGCAAUUAUUGAUCUUAAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAACUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGAUCCGGUGA-AGUGUUCGGACGCGCUUCGCGAGAAGUUCAUUGAACCUUAUCAUUUAGAGGAAGGAGAAGNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
pl.Pin.luc  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNUCAAAAUAAGCCAUGCAUGUCUAAGUAUGAACAU------CAGA--------UGU-GAAACUGCGGAUGGCUCAUUAAAUCAGUUAAGUUUCUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCACAAUCCCUU-GGACGGAUCGCA--CGGUUUGUGGCG-ACG-CUUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGAGGCCUACCAUGGUGGUGACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACUG-GGCUC-AUCGAGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGA-CGUCAC-----GGUCGGUCCG-CC-UUCUUGGUG---UG--CACUG------CC-----AUUAAUAGGGCUGUGGGGGCACGUAUUUCAUUGUCAGAGGUGAAAUUCUUGGAUUUAUGGAAGACGAACCACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACCAGGGAUCGGCGGAUCUACUCCGCCAGCCCUUCUGAGAAAUCA--GAGUGGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAACUAGCUACGCUUCC--CCAGCUUCUUAGAGGGACUAUGGCCUCCUAGGCCAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGCAACCAACGAGCUCCUGGCCCGAAAGGUUUGGGAAAUCUUCCAAAUUGCAUCGUGAUGGGGAUAGACCAUUGCAAUUAUUGAUCUUCAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGAUCCGGUGA-AGUGUUCGGACGCGCUUC-CGAGAAGUUCAUUGAACCUUAUCAUUUAGAGGAAGGAGAAGNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
pl.Tai.cry  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNUCAAAAUAAGCCAUGCAUGUCUAAGUAUGAACUAU-----CAGA--------UGU-GAAACUGCGGAUGGCUCAUUAAAUCAGUUAAGUUUCUUUGAUGGACUCGGAUAACCGUAGUAAUCUAGAGCUAAUACGUGCACAAUCCCUU-GGACGGAUCGCA--CGGUCUGCGGCG-ACG-CUUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGAGGCCUACCAUGGUGGUGACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACUG-GGCUC-AUCGAGUUGGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGU-CGUCAC-----GGUCGGUCCG-CC-ACUC-GGUG---UG--CACUG------CC-----AUUAAUAGGGCUGUGGGGGCACGUAUUUCAUUGUCAGAGGUGAAAUUCUUGGAUUUAUGGAAGACGAACCACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUCAGAUACCGUCCUAGUCUCAACCAUAAACGAUGCCGACCAGGGAUCGGCGGAUCUACUCCGCCAGCCCUUCUGAGAAAUCA--GAGUGGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACGCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUCUAACUAGCUACGCUUCC--CCAGCUUCUUAGAGGGACUAUGGCCGUUUAGGCCAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGCAACCAACGAGCUCCUGGCCCGAAAGGUUCGGGAAAUCUUCCAAAUUGCAUCGUGAUGGGGAUAGACCAUUGCAAUUAUUGAUCUUCAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGAUCCGGUGA-AGUGUUCGGACGCGCUUU-CGAGAAGUUCAUUGAACCUUAUCAUUUAGAGGAAGGAGAAGNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
pr.Sar.neu  -----AACCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUUAAAAUAAGCCAUGCAUGUCUAAGUAUAAGCU-------UAUA--------GGC-GAAACUGCGAAUGGCUCAUUAAAACAGUUAAGUUUAUUUGAUGGACAUGGAUAACCGUGGUAAUCUAUGGCUAAUACAUGCGCAUACCCUCGGGUCGGAUCGCAU-UGGUUUUCGGCG-AUG-GAUCAUUCAAGUUUCUGACCUAUCAGCUUGACGGUACUGUAUUGGACUACCGUGGCAGUGACGGGUACGGGGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCUA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACUC-AGGGAGGUAGUGACAAGAAAUCAACACUG-GAAAU-UAUAUUUUAGUGAUUGAUGAUGGGAAUCCAAACC------------------------------CCUUCAGAGUAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGUGCU-GGAAGC-GCCAGUCCCCC--------UUUG-------------------CACUUGGG--AUUAAUAGGGCAGUGGGGGCACGUAUUUAACUGUCAGAGGUGAAAUUCUUAGAUUUGUUAAAGACGAACUACUGCGAAAGCAUGCCAAAGAUGUUUUAUU---AAUCAAGAACGAAAGUUAGGGGCUCGAAGACGAUCAGAUACCGUCGUAGUCUUAACCAUAAACUAUGCCGACUAGAGAUAGGAAAAUCCUCUUCUCCUGCCCUUAUGAGAAAUCA--AAGUCGGUUCUGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGCGUGGAGC-UGCGGCUUAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAG-AA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUUAACCUCUAAAUAGGGUCAAUUUG-UAUCACUUCUUAGAGGGACUUUGCGUGUCUAACGCAAGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCUGCACGCGCGCUACACUGAUGCAUCCAACGAGCACCCUGGCCGACAGGUCUAGGUAAUCUUUGAGUAUGCAUCGUGAUGGGGAUAGAUUAUUGCAAUUAUUAAUCUUCAACGAGGAAUGCCUAGUAGGCGCAAGUCAGCAGCUUGCGCCGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAGUGUUCCGGUGA-AUUAUUCGGAUGUUCUACCUGGGAAGUUUUGUGAACCUUAACACUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCAGAAGGAUCANNN-----
pr.Pla.mal  -----AACCUGGUUGAUCUUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAAGUGAAAGUAUAUGCAUAU----U--------AUAUGUAGAAACUGCGAACGGCUCAUUAAAACAGUUAAGUCUACUUGACAUAUAAGGAUAACUACGGAAAACUGUAGCUAAUACUUGCUUAAUACUUAUGUAAACACAUAA--UAAUUCG-UAGU-GUG-UAUCAAUCGAGUUUCUGACCUAUCAGCUUGAUGUUAGGGUAUUGGCCUAACAUGGCUAUGACGGGUACGGGGAAUUAGAGUUCGAUUCCGGAGAGGGAGCCUGAGAAAUAGCUACCACAUCUA-GGAAGGCAGCAGGCGCGUAAAUUACCCAAUUCUAAAGA-AGAGAGGUAGUGACAAGAAAUCAAUGCAA-GGCCA-AUUUGGUUUGCAAUUGAUGAUGGGAAUUUAAAAC------------------------------CUCCCAGA-AGGCAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAAUUGUUGCAGUUAAAACGCUCGUAGUUGAAGG-AUC-AAU-AUUUUAAUAAU--GCUU-GCCAUU-UAAUAUCUUCUU-AUU---AUAA--AAUUAAUAGGAUAGCGGGGGCAUGUAUUCAGAUGUCAGAGGUGAAAUUCUUAGAUUUUCUGGAGACAAGCAACUGCGAAAGCAUGCCUAAAAUACUUCAUU---AAUCAAGAACGAAAGUUAAGGGAGUGAAGACGAUCAGAUACCGUCGUAAUCUUAACCAUAAACUAUGCCGACUAGGUGUUGGAUGAAUAUUUCCUUCAGUCCUUAUGAGAAAUCA--AAGUCGGUUCUGGGGCGAGUAUUCGCGCAAGCGAGAAAGUUAAAAGAACCGACGGAAGGGGACAC-AGGCGUGGAGCUUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACUAGUUUAAGACAG-UA---UUAA-------------------------U--UU--CUUGGAUGGUGAUGCAUGGCCGUUUUUAGUUCGUGAAAUGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGAUCUUAACCUCUAAUUAGCGGUAAUAAU---AAGCUUCUUAGAGGAACGAUGUGUGUCUAACACAAGGAAGUUUAAGGCAACAACAGGUCGUGAUGUCCUUAGUGAACUAGGCUGCACGCGUGCUACACUGAUAUGUAUAACGAGGUUGCUCACUGAA-AGUGUAGGUAAUCUUUCAAUAUAUAUCGUGAUGGGGAUAGAUUAUUGCAAUUAUUAAUCUUGAACGAGGAAUGCCUAGUAAGCAUGAUUCAUCAGAUUGUGCUGACUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAAGAUAUGAUGA-AUUGUUUGGAAAGAAUUAUCUGGAAAAAUCGUAAAUCCUAUCUUUUAAAGGAAGGAGAAGUCGUAACAAGGUUUCCGUCGGUGAACCUGCGGAAGGAUCAUUA-----
pr.Sti.hel  -----NNNNNNNNNNNNNNNNNNNNNNNUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUAUAAACUG------UAUA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUAUUUGAUGGACUCGGAUAACUGUAGUAAUCUACAGCUAAUACGUGCUUAGUCCUCU-GGACGAAUCGCA--UAGCUUGUGGCG-AUG-UUUCAUUCAAACUUCUGCCCUAUCAACUUGACGGUAGGAUAGAGGCCUACCAUGGUGGUGACGGGUACGGAGGAUUAGGGUUCGGUUCCGGAGAGGGAGCCUGAGAAAUGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-AGGGAGGUAGUGACAAUAAAUCAAUACCG-GGCAU-UAAAUGUUGGUAAUUGAUGAUAACGAUCUAAACC------------------------------CAUAUAGAGUAUCCAUUGGAGGCAAGUCUGGUGCCNNCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUUNAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGU-GGCUUUC-----UGCGGUCCG-C---UUC-GGUG---UG--CACUG------CU-----AUUAAGAGGGCAGUGGGGGCACGUAUUUCAUUGUCAGAGGUGAAAUUCUUGGAUUUAUGAAAGACGAACUUCUGCGAAAGCAUGCCAAGUAUGUCUUAUU---AAUCAAGAACGAAAGUUGGGGGCUCGAAGACGAUUAGAUACCGUCGUAGUCUCAACCAUAAACGAUGCCGACUAGGGAUCGGUGGGUUCGCUCCUCCGGCCCUUAUGAGAAAUCA--AAGUCGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGCGUGGANNCUGCGGCUUAAUUUGACUCAACACGGGAAAACUUACCAGGUCCAGACAA-GA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGGUUGCCUUGUCAGGUUGAUUCC-GGAACGAACGAGACCUCAGCCUCUAAAUAGUNNNUGUCUU-GCCGACUUCUUAGAGGGACUAUUGUCGUUUAGGCAAUGGAAGUAUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGCAUUCAACGAGUACCUUGGCCGAGAGGCC-GGGUAAUCUU-GAAACUGCAUCGUGAUGGGGAUAGAUUAUUGCAAUUAUUAGUCUUCAACGAGGAAUGCCUAGUAAGCGCGAGUCAUCAGCUCGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGGGUGUGCUGGUGA-AGNNUUCGGAUGAGAUCU-CGAGAAGUUCGUUAAACCCUCCCACCUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
pr.Mal.mat  -----CACCUGGUUGAUCCUGCCAGUAG-CAUACGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUAUAAACAAC-----UGUA--------UGU-GAAACUGCGAAUGGCUCAUUAUAUCAGUUAAGUUUAUUUGAUGGACAUGGAUAACCGUAGUAAUCUAGAGCUAAUACAUGCAUAAGCCUUC-GGCCGGAUCGAU--C--UUCGGAUCG-AUG-CAUCAUUCAAGUUUCUGCCCUAUCAGCUUGAUGGUAGGGUAUUGGCCUACCAUGGCUUUAACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAAUGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGUAAAUUACCCAAUCCUGACAC-AGGGAGGUAGUGACAAUAAAUCAAUGCCG-GGCUUAUUUAAGUUGGCAAUUGAUGAGAACAAUUUAAAUC------------------------------CCUAUCGAGGAUCAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUACUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGUGGU-UGUGGCA-----AACAGUCCG-CCC-AACAGGUGU---GU--GCUA---------U-G-AUUAAUAGGGUAGUGGGGGUACGUAUUCAAUUGUCAGAGGUGAAAUUCUUGGAUUUAUGGAAGACGAACUACUGCGAAAGCAUACCAAAUAUGUUUUAUU---AAUCAAGAACGAAAGUUAGGGGAUCGAAGAUGAUUAGAUACCAUCGUAGUCUUAACCAUAAACUAUGCCGACUAGGGAUUGGUGGCGUUACUCCAUCAGCCCUUAUGAGAAAUCA--AAGUCGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGAAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-GA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCCCCGCCUCUAAAUAGUCGUACUUAGUGUCGGCUUCUUAGAGGGACUUUUGGUGACUAACCGAAGGAAGUUGGGGGCAAUAACAGGUCGUGAUGCCCUUAGUGUCCUGGGCCGCACGCGCGCUACACUGACACACGCAACGAG-UCCUUGAUUGAAAAAUCCGGGUAAUCUUUAAAUGUGUGUCGUGAUAGGGAUAGAUUAUUGCAAUUAUUAAUCUUGAACGAGGAAUUCCUAGUAAAUGCGAGUCAUCAGCUCGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCACCUACCGAUGAAUGAUUCGGUGA-AAAUUCCGGAUGUGGGCAAUAGGAAGUUAUUUAAACCUCAUCAUUUAGAGGAAGGUGAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCAGAAGAUCCAGCUUCCUC
pr.Cyc.gla  -----NNNNNNNNNNNNNNNNNNNNUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUAUAAAU--------AAUA--------AGU-GAAACUGCGAAUGGCUCAUUAAAACAGUUAAGUUUAUUUGAUGAACAUGGAUAACCGUGGUAAUCUAGAGCUAAUACAUGCAUAGGCCUCACGGCCGGAUCCCC-----AACU-GGGG-AUAAGAUCANUCAAGUUUCUGCCCUAUCAGCUUGAUGGUAGUGUAUUGGACUACCAUGGCAGUAACGGGUACGGAGAAUUAGGGUUCGGUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCUA-GGAAGGCAGCAGGCGCGUAAAUUACCCAAUGCAGAUUC-UGCGAGGUAGUGACAAUACAUCAACCUGG-GGUC-UCACGACUACGGGAUUGAUGAGAACAAUUUAAAAA------------------------------CCUAACGAGGAACAAGUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCACUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGUGGU-UGNGCGC------AGUGGUCG-GUC-AUC--GAUC------GCCCG-------------AUUAAUAGGGCAGUGGGGGCAAGUAUUUAAUUGUCAGAGGUGAAAUUCUUGGAUUUAUUAAAGACUAACUUAUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUAGGGGAUCAAAGACGAUCAGAUACCGUCCUAGUCUUAACUAUAAACUAUACCGACUCGGUUUUGGUGGGGUAUCUCUAUCAGCCCGUAUGAGAAAUCA--AAGUCGGUUCUGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGUGUGGACGCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCAAAACAG-UG---UUGA-------------------------U--UC--UGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGCUUAAUUGC-GUAACGAACGAGACCUUAACCUCUUAAUAGAUGCCUGUAC--UAUUCUUCUUAGAGGGACUAUGCGUUUCAAGCGCAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCCUAGUGUCCUGGGCCGCACGCGCGCUACACUGACCCACUCAGCAAGCUCCUGGCCUGGAAAGGUUGGGUAAUCUUGCAAUAUGGGUCGUGUUAGGGAUCGAUCUUUGCAAUUAUAGAUCUUGAACGAGGAAUACCUAGUAAGUGCAAGUCAUCAGCUUGUACUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGGUCCGGUGA-ACCUUCUGGACAGUCGCAACGCGAAGUUAAGUAAACCUUAUCAUUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCGUNNNNNNNNNNNNNNNNNNNNNNNNN-----
pr.Tet.bor  -----AACCUGGUUGAUCCUGCCAGUA--CAUAUGCU-UGUCUUAAAAUAACCCAUGCAUGUGCCAGUUCA-GU--------UGAA--------AGC-GAAACUGCGAAUGGCUCAUUAAAACAGUUAAGUUUAUUUGAUAAACAUGGAUAACCGAGCUAAUGUUGGGCUAAUACAUGCUUAAUUCCCUGGAACGAAUCGAA--G--CUUGCUUCG-AUA-AAUCAUCUAAGUUUCUGCCCUAUCAGCUCGAUGGUAGUGUAUUGGACUACCAUGGCAGUCACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAAGGAGCCUGAGAAACGGCUACUACAACUC-GGUCGGCAGCAGGGAAGAAAAUUGGCCAAUCCUAAUUC-AGGGAGCCAGUGACAAGAAAUCAAGCUGG-GAAACCUA-GUUUUCGGCAUUGAUGAGAAAAGUGUUAAUC------------------------------UCUAGCGAGGAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGUGUU-CAGGUUCU-----UUCGA--------CUCG------------UCG----------G-GAAUUAAUAGGGCAGUGGGGGCAAGUAUUUAAUAGUCAGAGGUGAAAUUCUUGGAUUUAUUAAGGACUAACUAAUGCGAAAGCAUGCCAAAGAUGUUUUAUU---AAUCAAGAACGAAAGUUAGGGGAUCAAAGACGAUCAGAUACCGUCGUAGUCUUAACUAUAAACUAUACCGACUCGGGAUCGGCUGGAUAAUCCAGUCGGCCCGUAUGAGAAAUCA--AAGUCGGUUCUGGGGGAAGUAUGGACGCAAGUCUGAAACUUAAAGGAAUUGACGGAACAGCACACCAGAAGUGGAACCUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACGAGCGCAAGACAA-AG---UUGA-------------------------U--UC--UUUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUUAACCUCUAACUAGUCUGCUAACA--UGUACUUCUUAGAGGGACUAUUGGCAAGAAGCCAAUGGAAGUUUAAGGCAAUAACAGGUCGUGAUGCCCCUAGCGUGCUCGGCCGCACGCGCGUUACAAUGACUGGCGCAGAAAGUUCCUGUCCUGGGAAGGUAGGGUAAUCUUUUAAUACCAGUCGUGUUAGGGAUAGUUCUUUGGAAUUGUGGAUCUUGAACGAGGAAUUUCUAGUAAGUGCAAGUCAUCAGCUUGCGUUGAUUAUGUCCCUGCCGUUUGUACACACCGCCCGUCGCUUGUAGUAACGAAUGGUCUGGUGA-ACCUUCUGGAUGCGGGCAACGGAAAAAUAAGUAAACCCUACCAUUUGGAACAACAAGAAGUCGUAACAAGGUAUCUGUAGGUGAACCUGCAGAUGGAUCAUUA-----
pr.Rho.sal  -----UACCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUUAGUGUAAAUA-------UGU---------UAU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUACGUUUAUUUGAUGGACAUGGAUAACCGUAGUAAUCUAGAGCUAAUACAUGCACAGGUCUUACGACCGAACACAG--C--CUUGAGUGU-GUG-AUUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGAGGCCUACCAUGGUUUUAACGGGUACGGAGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAGACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCC-GACUU-GGGGAGGUAGUGACAAUAAAUCAAUACAG-GGCU--UAC-AGUUUGUUAUUGAUGAGAACAAUUUAAAUC------------------------------UCUUACGAGGAUCAAUUAGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCUAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUCGGGGC-UCGGGGA-----GGCUGUCGG-C---UUCG--------GUGGACGC-----------AUAUUAACAGGGCAGUGGGGCCGUAUAUUUCGUUGUCAGAGGUGAAAUUCUUGGAUUUACGAAAGAUAAACUUCUGCGAAAGCACGGCAAGGAUGUUUUAUU---GAUCAAGAACGAAAGUUAGGGGAUCGAAGACGAUCAGAUACCGUCGUAGUCUUAACCAUAAACUAUGCCGACUAGGGAUCAGUGGAUUUGCUCCAUUGGCCCUUGUGAGAAAUCA--AAGUUGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCUUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCAGCCUGUAACUAGUGACGCUUU--GCCCACUUCUUAGAGGGACUAUUUGUGUUUAAUGAAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGAAUGCAACGAGCUCCUUAUUCGAAAGAAUCGGGUAAACUUUGAAAAUUCAUCGUGAUGGGGAUAGAUUAUUGCAAUUAUUAAUCUUCAACGAGGAAUUCCUAGUAAGCGCGAGUCAUCAGCUCGUGCUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGGUCCGGUGA-AAUCUUCGGAUGCAGUUAUCGAGAAGUUGAUUGAACCUUAUCAUUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUG-----
pr.Por.aca  -----NNNNNNNNNNNNNNNNNNNNNAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUAUAAACGCC-----UAUA-------UCGU-GAAACUGCGAAUGGCUCAUUAAAACAGUUAAGUUCCUUUGGGAAACUUGGAUAGCCGUAGUAAUCUAGAGCUAAUACAUGCCGACGCCUCACGGUCGGAUCGCA--CGGCAUGUGGCG-ACG-CCUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGAGUAUUGGCCUACCAUGGUGUCGACGGGUACGGGGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCCAACAC-GGGGAGGUAGUGACAAAAAAUCAAUAGGG-GGCCCUUUACGGUCUCUAAUUGGUGAGAACAAUUUAAAUU------------------------------CCUAUCAAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAACGCUCGUAGUCGGGAG-CACCUC-------CACGGGCC-GGGUUCUGUUGC---CAG-UUGGG-------------AUUAAUAGGGCAGUGGGGGCACGUAUUUCAUUGUCAGAGGUGAAAUUCUUGGAUUGAUGGAAGACGCACAACUGCGAAAGCAUGCCAUGGAUGUUUUAUU---GAUCAAGAACGAAAGUUAGGGGAUCGAAGACGAUCAGAUACCGUCGUAGUCUUAACCAUAAACGAUGCCGACUGGGGAUUGGCGGGUUUUCUCCGUCAGCCCCUAUGGGAAACCA--AAGUUGGUUCUGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGAAAACUUACCAGGUCCGGACAA-GA---CUGA-------------------------U--UU--UUUGGUUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUAAUUCC-GUAACGAACGAGACCUCGUCCUCUAAAUAGGUGCGCCCCGUCUUACCUUCUUAGAGGGACUAUCCGCGUCUAGCGUAUGGAAGAUUGAGGCAAUAAUAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGCAUUCAACAAGAGCCUGGACCGGAAGGCUCGGGUAACCUUUGAAACUGCAUCGUGAUGGGGAUAGAUCAUUGCAAUUAUUGAUCUUGAACGAGGAAUUCCUUGUAGGCGUAGGUCACUAGCCUGCGCCGAAUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUCCUACCGAUGAAUGGUCCGAUGA-AGUGUUCGGAUGCGGUUA-UUAGAAGUUCAUUAAAUCUUAUCAUUUAGAGGAAGGAGAAGUCGUAACAAGGUUUCCNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
pr.Bla.hom  -----NNNNNNNNNNNNNNNNNNNNNNGUCAUACGCU-CGUCUCAAAAUAAGCCAUGCAUGUGUAAGUGUAAAU--------ACUA--------UUU-GGAACUGCGAAUGGCUCAUUAUAUCAGUUAAGUUUAUUUGGUGAACUUGGAUAACCGUAGUAAUCUAGGGCUAAUACAUGAGAGUCCUUGGUAGGCGUAUCGCA--UGCUUAAUAGCG-AUG-AGUCUUUCAAGUUUCUGCCCUAUCAGCUUGAUGGUAGUAUAUGGGCCUACCAUGGCAGUAACGGGUACGAAGAAUUUGGGUUCGAUUUCGGAGAGGGAGCCUGAGAGAUGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGUAAAUUACCCAAUCCUGACAC-AGGGAGGUAGUGACAAUAAAUCAAUGCGG-GACU--AUC-AGUUUGCAAUUGUUGAGAACAAUGUACAAC------------------------------UCUAUCGAUAAGCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAACGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGU-GAUCGCU-----GUUGUGAGA-C---UUCGUCUCU---CG--ACAU-------------GUUAAAAGGACAGUGGGGGUACAUAUUCACUAGUUAGAGGUGAAAUUCUCGGAUUUAUGGAAGAUGAACAAGUGCGAAAGCAUACCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGCUAGGGGAUCGAAGAGGAUUAGAUACCCUCGUAGUCUUAGCUAUAAACGAUACCGACUAGGGGUUAGUAGA-AAGCUUUAUUAGUCCUUAUGAGAAAUCA--AAGUCGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUUACCAGGUCCAGACAA-AA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGCUAAUUCC-GAAACGAACGAGACCUCCGCCUUUAACUAGUGACGUUGUG---UUGCUUCUUAUAGGGACACUAUAUGUAAAAUGUAGGGAAGCUGGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCUGCACGCGCGCGACACUGAUCUAUUCAACGAG--GCUGGGUCGAGAGACUUGGCAAAUCUUUGAAAGUAGAUCGUGAUGGGGAUUGAUGCUUGUAAUUGUUCAUCAUGAACGAGGAAUUCCUAGUAAACGCAAGUCAUCAACUUGCAUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCACCUACCGAUGAAUGGUCCGGUGA-ACACUUUGGAUUGAGUAAAGGAGAAGUCGUGUAAAUCUUAUCAUUUAGAGGAAGGUGAAGUCGUAACAAGGUUUCCNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
pr.Try.equ  -----NNNNNNNNUGAUUCUGCCAGUAGUCAUAUGCU-UGUUUCAAGAUUAGCCAUGCAUGCCUCAGAAUCACUGC------U----------GCAG-GAAUCUGCGCAUGGCUCAUUACAUCAGACGAAUCUGCGCCAAAAUACUGGAUAACUUGGCGAAAGCCAAGCUAAUACAUGAACUCGGAUUA-UCCGAAAGCCG---GC-CUUGCUCGG-CGUCUACUGAC-GAACAACUGCCCUAUCAGCCAGAUGGCCGUGUAGUGGACUGCCAUGGCGUUGACGGGAGCGGGGGAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAAUAGCUACCACUUCUC-GGAGGGCAGCAGGCGCGCAAAUUGCCCAAUGUCGAAAACGAUGAGGCAGCGAAAAGAAAUGAGCGACC-GUGCCCUAGUGCAGGUUGUUUUAUGGGGGAUACUCAAACC--------------------------CAUCCAUAUCGAGUAACAAUUGGAGACAAGUCUGGUGCCAGCACCCGCGGUAAUUCCAGCUCCAAAAGCGUAUAUUAAUGCUGUUGCUGUUAAAGGGUUCGUAGUUGGGGC-CACGUAG-------UGUGCC--GUGACCUC-GGACGUGUUGACCCACGC--------CCUGAAGGAGGGAGUUGGGGGAGCGUACUGGUGCGUCAGAGGUGAAAUUCUUAGACCGCACCAAGACGAACUACAGCGAAGGCACUUCAAGGAUACCUUCUC---AAUCAAGAACCAAAGUGUGGGGAUCAAAGAUGAUUAGAGACCAUUGUAGUCCACACUGCAAACCAUGACACCCAUGAAUUGGGGAA----UU-UCUUACUCUUCACGCGAAAGCUUGGAGGUGUCUCAGGGGGGAGUACGUUCGCAAGAGUGAAACUUAAAGAAAUUGACGGAAUGGCACACAAGACGUGGAGCGUGCGGUUUAAUUUGACUCAACACGGGGAACUUUACCAGAUCCGGACAG-GA---UGGA-------------------------U--CC--CCUGAAUGGUGGUGCAUGGCCGCUUUUGGUCGGUGGAGUGAUUUGUUUGGUUGAUUCC-GUAACGGACGAGAUCCAAGCUGCCCAGUAGGUGCCGUUUU-GGCCCCUUCUCUGCGGGAUUCCUUGCUUUCGGCAAGGUGAGAUUUUGGGCAACAGCAGGUCGUGAUGCUCCUCAUGUUCUGGGCGACACGCGCACUACAAUGUCAGUGAGAACAGUCCCUUG-AUCAAAAGAGC-GGGGAAACCAAAUCACGUAGACCCACUUGGGACCGAGUAUUGCAAUUAUUGGUCGCCAACGAGGAAUGUCUCGUAGGCGCAGCUCAUCAAACUGUGCCGAUUACGUCCCUGCCAUUUGUACACACCGCCCGUCGUUGUUUCCGAUGAUGGUGCAAUACA-GGUGAUCGGACGUCGUCU-CCGAAAGUUCACCGAUAUUGCUUCAAUAGAGGAAGCAAAAGUCGUAACAAGGUAGCUGUAGNNNNNNNNNNNNNNNNNNNNNNN-----
an.Dro.mel  -----AUUCUGGUUGAUCCUGCCAGUAGUUAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUACACACG-------UUAA--------AGU-GAAACCGCAAAAGGCUCAUUAUAUCAGUUAGGUUCCUUAGAUCGACUUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCAAAACAUUUAUGUGCAGAUCGUA--UGGCUUGCGACG-ACA-GAUCUUUCAAAUGUCUGCCCUAUCAACUUGAUGGUAGUAUCUAGGACUACCAUGGUUGCAACGGGUACGGGGAAUCAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCUA-GGAAGGCAGCAGGCGCGUAAAUUACCCACUCCCAGCUC-GGGGAGGUAGUGACGAAAAAUCAAUACAG-GACUCAUCCGAGCCUGUAAUUGAUGAGUACACUUUAAAUC------------------------------CUUAACAAGGACCAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCGGUUAAAACGUUCGUAGUUGUGCU-UCAUAC-----GGGUAGUACA-ACU-AAUUGUUA-GU-----ACUU----ACCUUUUG-AUUAAUAGAACAGUGGGGGCAAGUAUUACGACGCGAGAGGUGAAAUUCUUGGACCGUCGUAAGACUAACUUAAGCGAAAGCAUGCCAAAGAUGUUUUAUU---AAUCAAGAACGAAAGUUAGAGGUUCGAAGGCGAUCAGAUACCGCCCUAGUUCUAACCAUAAACGAUGCCAGCUAGCAAUUGGGUGUUUUUCUCUCUCAGUGCUUCCGGGAAACCA--AAGCUGGCUCCGGGGGAAGUAUGGUUGCAAAGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGAAAACUUACCAGGUC-GAACAA-UG---UUGA-------------------------U--CC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUCGUGGAGUGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACUCAAAUAUUUAAAUAGAUAC--UGUGCACUAGCUUCUUAAAUGGACAAAUUGCGUCUAGCAAUAUGAGA-UU-GAGCAAUAACAGGUCGUGAUGCCCUUAGUGUCCUGGGCUGCACGCGCGCUACAAUGAAAGUAUCAACGUGUUCCUAGACCGAGAGGUCCGGGUAAACCGUGAACCACUUUCAUGCUUGGGAUUGUGAACUGAAACUG-UUCACAUGAACUUGGAAUUCCCAGUAAGUGUGAGUCAUUAACUCGCAUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUUAUUUAGUGA-GGUCUCCGGAGUGAUCUUGGCAAAAGUUGACCGAACUUGAUUAUUUAGAGGAAGUAAAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUA-----
an.Lam.aep  -----NACCUGGUUGAUCCUGCCAGUAG-CAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUGCGUGCAAACGG------GUUA--------AGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAGGUUCCUUUGAUCGGCUUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCUGAGCGCUCCGGCGCAGAUCGCA--CGGCUGGUGGCG-ACG-UAUCUUUCGAAUGUCUGCCCUAUCAACUUGAUGGCAGGCUCCGUGCCUACCAUGGUGACCACGGGUACGGGGAAUCAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCACUCCCGACUC-GGGGAGGUAGUGACGAAAAAUCAAUACAG-GACUCUUUCGAGCCUGUAAUUGAUGAGUACACUUUAAAUC------------------------------CUUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGCUGUUGCAGUUAAAAAGCUCGUAGUUGGGGA-AGGGGCU-----UGCGGUCCG-CC--GCGAGGUGU--GU--CACUC-------CUG--CAUUAAGAGGGCUGCGGGGGCACGUAUUGUGCCGUUAGAGGUGAAAUUCUUGGAUCGGCGCAAGACGAGCGAAAGCGAAAGCAUGCCAAGAAUGUCUUAUU---AAUCAAGAACGAAAGUCGGAGGUUCGAAGGCGAUCAGAUACCGCCCUAGUUCCGACCAUAAACGAUGCCAACUGGCGAUCAGGCGGUCCCCCUGCCUGGCGCUUGCGGGAAACCA--AAGUGGGUUCCGGGGGAAGUAUGGUUGCAAAGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGAAAACUCACCCGGCCCGGACAG-GA---UUGA-------------------------U--CC--CGUGGGUGGUGGUGCAUGGCCGUUCGUAGUUGGUGGAGCGAUUUGUCUGGUUCAUUCC-GAAACGAACGAGACUCCGACGUCUAACUAGCUACGCUGUG--UGAGCUUCUUAGAGGGACGAGUGGCUUUCAGCCACACGAGA-UU-GAGCAAUAACAGGUCGUGAUGCCCUUAGUGUCCGGGGCCGCACGCGCGCUACACUGAAUGGAUCAGCGUGCGCCUUCGCCGGCAGGCGUGGGUAACCCGUGAAACCCAUUCGUGAUAGGGAUUGGGGAUUGGAACUGUUCCCCAUGAACGAGGAAUUCCCAGUAAGCGCGAGUCAUAAGCUUGCGUUGAUUAAGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGGAUGGUUUAGUGA-GGUCCUCGGAUGGCCGCGAUGAGAAGACGAUCGAACUUGACUGUCUAGAGGAAGUAAAAGUCGUNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
an.Xen.lae  -----UACCUGGUUGAUCCUGCCAGUAG-CAUAUGCU-UGUCUCAAAAUAAGCCAUGCACGUGUAAGUACGCACGG------GGUA--------AGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAGGUUCCUUUGAUCGACUUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCCGAGCGCCCCAGCGCCGAUCGCA--CGUCCCGCGGCG-ACG-AUACAUUCGGAUGUCUGCCCUAUCAACUUGAUGGUACUUUCUGCGCCUACCAUGGUGACCACGGGUACGGGGAAUCAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCACUCCCGACGC-GGGGAGGUAGUGACGAAAAAUCAAUACAG-GACUCUUUCGAGCCUGUAAUUGAUGAGUACACUUUAAAUC------------------------------CUUAACGAGGAUCUAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGCUGCAGUUAAAAAGCUCGUAGUUGGGGA-UCGAGCU-----GGCGGUCCG-CC--GCGAGGCG---GC--UACCC-------CUG--CAUUAAGAGGGCGGCGGGGGCACGUAUUGUGCCGCUAGAGGUGAAAUUCUUGGACCGGCGCAAGACGAACCAAAGCGAAAGCAUGCCAAGAAUGUUUUAUU---AAUCAAGAACGAAAGUCGGAGGUUCGAAGACGAUCAGAUACCGUCGUAGUUCCGACCAUAAACGAUGCCGACUAGCGAUCCGGCGGUCCCCCCGCCGAGCGCUUCCGGGAAACCA--AAGUCGGUUCCGGGGGGAGUAUGGUUGCAAAGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGAAACCUCACCCGGCCCGGACAG-AA---UUGA-------------------------U--CC--UGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACUCCUCCAUCUAACUAGUUACGCCCGG--CCAACUUCUUAGAGGGACAAGUGGCGUUCAGCCACACGAGA-UC-GAGCAAUAACAGGUCGUGAUGCCCUUAGUGUCCGGGGCUGCACGCGCGCUACACUGAACGGAUCAGCGUGUACCUGCGCCGACAGGUGCGGGUAACCCGUGAACCCCGUUCGUGAUAGGGAUCGGGGAUUGCAAUUAUUUCCCAUGAACGAGGAAUUCCCAGUAAGUGCGGGUCAUAAGCUCGCGUUGAUUAAGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGGAUGGUUUAGUGA-GGUCCUCGGACGGCCGCCACGAGAAGACGAUCAAACUUGACUAUCUAGAGGAAGUAAAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUA-----
an.Hom.sap  -----UACCUGGUUGAUCCUGCCAGUAG-CAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUACGCACGG------GGUA--------AGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAGGUUCCUUUGGUCGACUUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCCGGGCGC-CGCGCGCCGAUCGCA--CGCCCCGCGGCG-ACG-ACCCAUUCGAACGUCUGCCCUAUCAACUUGAUGGUAGUCGCCGUGCCUACCAUGGUGACCACGGGUACGGGGAAUCAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCACUCCCGACCC-GGGGAGGUAGUGACGAAAAAUCAAUACAG-GACUCUUUCGAGCCUGUAAUUGAUGAGUCCACUUUAAAUC------------------------------CUUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGCUGCAGUUAAAAAGCUCGUAGUUGGGGA-GCGGGC-----GGGCGGUCCG-CC--GCGAGGCG-AGCC---ACCC------CCGU---AUUAAGAGGGCGGCGGGGGCACGUAUUGCGCCGCUAGAGGUGAAAUUCUUGGACCGGCGCAAGACGGACCAGAGCGAAAGCAUGCCAAGAAUGUUUUAUU---AAUCAAGAACGAAAGUCGGAGGUUCGAAGACGAUCAGAUACCGUCGUAGUUCCGACCAUAAACGAUGCCGACCGGCGAUGCGGCGGUCCCCCCGCCGGGCGCUUCCGGGAAACCA--AAGUCGGUUCCGGGGGGAGUAUGGUUGCAAAGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGAAACCUCACCCGGCCCGGACAG-CA---UUGA-------------------------U--CC--CGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACUCUGGCAUCUAACUAGUUACGCCGAGCCCCAACUUCUUAGAGGGACAAGUGGCGUUCAGCCACCCGAGA-UU-GAGCAAUAACAGGUCGUGAUGCCCUUAGUGUCCGGGGCUGCACGCGCGCUACACUGACUGGCUCAGCGUGUACCUACGCCGGCAGGCGCGGGUAACCCGUGAACCCCAUUCGUGAUGGGGAUCGGGGAUUGCAAUUAUUCCCCAUGAACGAGGAAUUCCCAGUAAGUGCGGGUCAUAAGCUUGCGUUGAUUAAGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGGAUGGUUUAGUGA-GGCCCUCGGACGGCCGCCCUGAGAAGACGGUCGAACUUGACUAUCUAGAGGAAGUAAAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUA-----
an.Mus.mus  -----UACCUGGUUGAUCCUGCCAGUAG-CAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUACGCACGG------GGUA--------AGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAGGUUCCUUUGGUCGACUUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCCGGGCGCUCCCGCGCCGAUCGCA--CGCCCCGCGGCG-ACG-ACCCAUUCGAACGUCUGCCCUAUCAACUUGAUGGUAGUCGCCGUGCCUACCAUGGUGACCACGGGUACGGGGAAUCAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCACUCCCGACCC-GGGGAGGUAGUGACGAAAAAUCAAUACAG-GACUCUUUCGAGCCUGUAAUUGAUGAGUCCACUUUAAAUC------------------------------CUUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGCUGCAGUUAAAAAGCUCGUAGUUGGGGA-GCGGGC-----GGGCGGUCCG-CC--GCGAGGCGA--GU--CACCC------CCGU--CAUUAAGAGGGCGGCGGGGGCACGUAUUGCGCCGCUAGAGGUGAAAUUCUUGGACCGGCGCAAGACGGACCAGAGCGAAAGCAUGCCAAGAAUGUUUUAUU---AAUCAAGAACGAAAGUCGGAGGUUCGAAGACGAUCAGAUACCGUCGUAGUUCCGACCAUAAACGAUGCCGACUGGCGAUGCGGCGGUCCCCCCGCCGGGCGCUUCCGGGAAACCA--AAGUCGGUUCCGGGGGGAGUAUGGUUGCAAAGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGG-GCCUGCGGCUUAAUUUGACUCAACACGGGAAACCUCACCCGGCCCGGACAG-CA---UUGA-------------------------U--CC--CGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACUCUGGCAUCUAACUAGUUACGCCGAGCCCCAACUUCUUAGAGGGACAAGUGGCGUUCAGCCACCCGAGA-UU-GAGCAAUAACAGGUCGUGAUGCCCUUAGUGUCCGGGGCUGCACGCGCGCUACACUGACUGGCUCAGCGUGUACCUGCGCCGGCAGGCGCGGGUAACCCGUGAACCCCAUUCGUGAUGGGGAUCGGGGAUUGCAAUUAUUCCCCAUGAACGAGGAAUUCCCAGUAAGUGCGGGUCAUAAGCUUGCGUUGAUUAAGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGGAUGGUUUAGUGA-GGCCCUCGGACGGCCGCCCUGAGAAGACGGUCGAACUUGACUAUCUAGAGGAAGUAAAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUA-----
an.Rat.nor  -----UACCUGGUUGAUCCUGCCAGUAG-CAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUACGCACGG------GGUA--------AGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAGGUUCCUUUG-UCGACUUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCCGGGCGCUCCCGCGCCGAUCGCA--CGCUCCGCGGCG-ACG-ACCCAUUCGAACGUCUGCCCUAUCAACUUGAUGGUAGUCGCCGUGCCUACCAUGGUGACCACGGGUACGGGGAAUCAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCACUCCCGACCC-GGGGAGGUAGUGACGAAAAAUCAAUACAG-GACUCUUUCGAGCCUGUAAUUGAUGAGUCCACUUUAAAUC------------------------------CUUAACGAGGAUCCAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGCUGCAGUUAAAAAGCUCGUAGUUGGGGA-GCGGGC-----GGGCGGUCCG-CC--GCGAGGCGA--GU--CACCC-------CCG--CAUUAAGAGGGCGGCGGGGGCACGUAUUGCGCCGCUAGAGGUGAAAUUCUUGGACCGGCGCAAGACGGACCAGAGCGAAAGCAUGCCAAGAAUGUUUUAUU---AAUCAAGAACGAAAGUCGGAGGUUCGAAGACGAUCAGAUACCGUCGUAGUUCCGACCAUAAACGAUGCCGACUGGCGAUGCGGCGGUCCCCCCGCCGGGCGCUUCCGGGAAACCA--AAGUCGGUUCCGGGGGGAGUAUGGUUGCAAAGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGG-GCCUGCGGCUUAAUUUGACUCAACACGGGAAACCUCACCCGGCCCGGACAG-CA---UUGA-------------------------U--CC--CGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACUCUGGCAUCUAACUAGUUACGCCGAGCCCCAACUUCUUAGAGGGACAAGUGGCGUUCAGCCACCCGAGA-UU-GAGCAAUAACAGGUCGUGAUGCCCUUAGUGUCCGGGGCUGCACGCGCGCUACACUGACUGGCUCAGCGUGUACCUACGCCGGCAGGCGCGGGUAACCCGUGAACCCCAUUCGUGAUGGGGAUCGGGGAUUGCAAUUAUUCCCCAUGAACGAGGAAUUCCCAGUAAGUGCGGGUCAUAAGCUUGCGUUGAUUAAGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGGAUGGUUUAGUGA-GGCCCUCGGACGGCCGCCCUGAGAAGACGGUCGAACUUGACUAUCUAGAGGAAGUAAAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUA-----
an.Ast.amu  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-------NNAA--------AGU-GAAACUGCGGACGGCUCAUUAAAUCAGUUAGGUUCCUUGGAGCGACAUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCAAAGCGCUUACGCGCCGAUCGCA--CGGUCUGCGGCG-ACG-GAUCCUUCGAAUGUCUGCCCUAUCAACUUGAUGGUACGUUAUGCGCCUACCAUGGUCGUAACGGGUACGGAGAAUCAGGGUUCGAUUCCGGAGAGGGAGCUUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCACUCC-GACAC-GGGGAGGUAGUGACGAAAAAUCAAUACAG-GACUCUUUCGAGCCGGUAAUUGAUGAGUACACUUUAAAUC------------------------------CUUAACGAGGAUCUAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAAUUGCUGCAGUUAAAAAGCUCGUAGUUGGGGC-GCAGCGU-----GGCGGUCCG-C---AAG--GCC---G---CACUC-------CCG---AUUAAGAGGGCUGAGGGGGCACGUAUUGCGGUGUGAGAGGUGAAAUUCUUGGAUCGCCGCAAGACGACCGACUGCGAAAGCAUGCCAAGAAUGUUUUAUU---AAUCAAGAACGAAAGUUAGAGGUUCGAAGGCGAUCAGAUACCGCCCUAGUUCUAACCAUAAACGAUGCCGACU--CGAUC-GCCGGUCCACGCGGCGGGCGUCUGCGGGAAACCA--AAGUCGGUUCCGGGGGAAGUAUGGUUGCAAAGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACCCGGCCCGGACAA-GA---UUGA-------------------------U--UC--UGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACUCUGGCUUCUAAAUAGUCGCGCGCCG--CUGACUUCUUAGAGGGACAAGUGCGGUUCAGCCACGCGAGA-UU-GAGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCGGGGC-GCACGCGCGCUACACUGAAGGGAUCAGCGGGCGCCUUGUCUGGAAGGCCUGGGCAAUCCGUGAACCCCCUUCGUGCUUGGGAUAGGGACUUGCAAUUGUCUCCCUUAAACGAGGAAUUCCCAGUAAGCGCAAGUCAUCAGCUCGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUGGUUUAGUGA-GAUCCUCGGAGGGCCUCUGCCGGAAGACCAUCGAACUUNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
an.Str.int  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-------NNNN--------CGACGAAACUGCGGAUGGCUCAUUAAAUCAGUUAGGUUCAUUGGAUCGACAUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCGUAGCGCUUCCGCGCCGAUCGCA--CGGUUUGCGGCG-ACG-GAUCCUUCGAAUGUCUGCCCUAUCAACUUGAUGGUACGUUAUGCGCCUACCAUGGUCGUCACGGGUACGGAGAAUCAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCACUCCCGACAC-GGGGAGGUAGUGACGAAAAAUCAAUACAG-GACUCUUUCGAGCCUGUAAUUGAUGAGUACACUUUAAAUC------------------------------CUUAACGAGGAUCCACUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAGUAGCGUAUAUUAAAGCUGUUGCAGUUAAAAAGCUCGUAGUUG-GGGCCCAGGC------UGCGGUCCG-C--CGUGU-GUG---U----ACUC-------A-G---AUUAAGAGGGCUGAGGGGGCACGUAUUGCGGUGUGAGAGGUGAAAUUCUUGGAUCGCCGCAAGACGAACGACUGCGAAAGCAUGCCAAGAAUGUUUUAUU---AAUCAAGAACGAAAGUUAGAGGUUCGAAGGCGAUCAGAUACCGCCCUAGUUCUAACCAUAAACGAUGCCGACUGACGAUCCGCCGGUCCCCGCGGCGGGCGUCUAAGGGAAACCA--AAGUCGGUUCCGGGGGAAGUAUGGUUGCAAAGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGAAAACUCACCCGGCCCGGACAA-GA---UUGA-------------------------U--UC--UGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACUCUGGCUUCUAAAUAGUUGCGCGCCG--UCAACUUCUUAGAGGGACAAGUGGCGUUUAGCCAGGCGAGA-UU-GAGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCGGGGC-GCACGCGCGCUACACUGGCGGAAUCAGCGGGUGCCUUGGCCGGAAGGUCUGGGUAAUCCGUGAACCUCCUCCGUGAUGGGGAUAGGGAGUUGCAAUUAUCUCCCUUGAACGAGGAAUUCCCAGUAAGCGCGAGUCAUCAGCUCGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUGGUUUAGUGA-GAUCCUCGGACGUCGUUUGCGAGAAGACGAUCAAANNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
an.Cae.ele  ----AUACCUGAUUGAUUCUGUCAGCGCG-AUAUGCUAAGU--AAAAAUAAGCCAUGCAUGCUUUGAUUCA-----------AA------------U-GAAAUUGCGUACGGCUCAUUAGAGCAGAUACACCUUAUCCGGGAAUAUGGAUAACUGCGGAAAUCUGGAGCUAAUACAUGCAAUACCCGCAAGGGUUUACUGU---CAGUUCGUGACU-CUA-UCCGGAAAGGGUGUCUGCCCUUUCAACUAGAUGGUAGUUUAUUGGACUACCAUGGUUGUUACGGGUACGGAGAAUAAGGGUUCGACUCCGGAGAGGGAGCCUUAGAAACGGCUACCACGUCCA-GGAAGGCAGCAGGCGCGAAACUUAUCCACUGUUG-AGU--AUGAGAUAGUGACUAAAAAUAAAGACUC-AUCC-UUUU-GG-GAGUUAUUUAUGAGUUGAAUACAAAUG------------------------------AUCUUCGAGUAGCAAGGAGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCUCCUAGUGUAUCUCGUUAUUGCUGCGGUUAAAAAGCUCGUAGUUGGGUU-CGUGCCGAGUUCGC------------AUU----------------UGCGU-CAAC-GU-GUUAAGAGGGCAAAGGGGGCACGUAUCAUUACGCGAGAGGUGAAAUUCGUGGACCGUAGUGAGACGCCCAACAGCGAAAGCAUGCCAAGAAUGUCUUAUU---AAUCAAGAACGAAAGUCAGAGGUUCGAAGGCGAUUAGAUACCGCCCUAGUUCUGACCGUAAACGAUGCCAUCUCGCGAUUCGG-AGUUUUCCUGCCGAGGGCUAUCCGGAAACGA--AAGUCGGUUCCGGGGGUAGUAUGGUUGCAAAGCUGAAACUUAAAGAAAUUGACGGAAGGGCACACAAGGCGUGGAGCUUGCGGCUUAAUUUGACUCAACACGGGAAAACUCACCCGGUCCGGACAC-UA---UUGA-------------------------U--CU--GGUGGUUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUUAUUCC-GAAACGAGCGAGACUCUAGCCUCUAAAUAGUUGGCGUUCG--AUAACUUCUUAGAGGGAUAAGCGGUGUUUAGCCGCACGAGA-UU-GAGCGAUAACAGGUCGUGAUGCCCUUAGUGUCCGGGGCUGCACGCGUGCUACACUGGUGGAGUCAGCGGGUUCCUAUGCCGAAAGGUAUCGGUAAACCGUGAAAUUCUUCCAUGUCCGGGAUAGGGUAUUGUAAUUAUUGCCCUUAAACGAGGAAUGCCUAGUAAGUGUGAGUCAUCAGCUCACGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUAUCCGGGACGAACUGAUUCGAGA-AGAGUGGGGAUGUCGUAACCGGAAACCAUUUUUAUCGCAUUGGUUUGAACCGGGUAAAAGUCGUAACAAGGUAGCUGUAGGUGAACCUGCAGCUGGAUCAUCG-----
an.Bra.pli  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNNNNNNNNNNNNNAUGCAUGUCUAAGUACAUACC-------AGCA--------GGU-GAAACCGCGAAUGGCUCAUUAAAUCAGUUAGGUUCCUUAGAUCGACAUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCAAAGCUCGUAUGAGCUGAUCGCA--UGGCUAGCGGCG-ACG-UAUCUUUCAAGUGUCUGCCCUAUCAACUUGAUGGUAAGCGAUUUGCCUACCAUGGUUGUAACGGGUACGGGGAAUCAGGGUUCGAUUCCGGAGAGGGAGCAUGAGAAACGGCUACCACAUCUC-GGAAGGCAGCAGGCGCGCAAAUUACCCACUCCUAGAAC-GGGGAGGUAGUGACGAAAAAUCAAUACCG-GACUCAAU-GAGCCGGUAAUUGAUGAGUACAGUUUAAAAC------------------------------CCUAACGAGGAUCUAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGUAUAGCAUGGGUACGC------------UUUU---------------GCUGU-U-AC-ACAAUUAAUAGGGCAGAGGGGGCACGUAUUGCGGUGUUAGAGGUGAAAUUCUUGGAUCGCCGCAAGACGAACAACUGCGAAAGCAUGCCAAGAAUGUUUUAUU---AAUCAAGAACGAAAGUUGGAGGUUCGAAGACGAUUAGAUACCGUCCUAGUUCCAACCAUAAACGAUGCCAACUAGCGAUUAGCUGCUUUUCACAGCUAGCGCUUCCGGGAAACCA--AAGUUGGUUCCGGGGGAAGUAUGGUUGCAAAGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGUAGCCUGCG-CUUAAUUUGACUCAACACGGGAAAUCUCACCCGGCCCGGACAU-AA---UUGA-------------------------U--UC--GGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGCGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACUCUAGCCUCUAAAUAGUACGUCCUUUUGGGUACUUCUUAGAGGGACAAGUAGCGGUAAGCUACACGAAA-UU-GAGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCGGGGGCAUACGCGCGCUACACUGAAGGGAUAAGCGUGUUCCUGCUCCGAAAGGAGUGGGUAAUCCGUGAAA-CCCUUCGUGAUUGGGAUCGGGGCUUGAAAUUAUUCUCCGUGAACGAGGAAUUCCCAGUAAGCGCGAGUCAUAAGCUCGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUGGUUUAGUGA-GAUCUUCGGAUGGUCGCAACGAGAAGAUGAUCAAACUUGAUCAUUUAGAGGAAGUAAAAGUCGUAACAAGGUUUCCGUANNNNNNNNNNNNNNNNNNNNNNNN-----
fu.Sch.pom  -----UACCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUAUAAGCAAU-----UGUA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUACGUUUAUUUGAUAGACUUGGAUAACCGUGGUAAUCUAGAGCUAAUACAUGCUAAAUCCUUUUGGACGAAUCGCA--UGGCUUGCGGCG-AUG-GUUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGAGGCCUACCAUGGUUUUAACGGGUACGGGGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCCGACAC-GGGGAGGUAGUGACAAGAAAUCAAUGCAG-GGCCCUUUCGGGUUUGUAAUUGAUGAGUACAAUGUAAAUA------------------------------CCUAACGAGGAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGAGC-CUGGUCG-----ACUGGUCCG-CCG--AAGCGUG---UU--UACUG-------UCA--UAUUAAUAGGGUAGUGGGGGCACGUAUUCAAUUGUCAGAGGUGAAAUUCUUGGAUUUAUUGAAGACGAACUACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUAGGGGAUCGAAGACGAUCAGAUACCGUCGUAGUCUUAACCAUAAACUAUGCCGACUAGGGAUCGGGCAAUUUACUUGCUCGGCCCUUACGAGAAAUCA--AAGUCGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACAAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAA-AA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGCUUAAUUGC-GAAACGAACGAGACCUUAACCUCUAAAUAGCUGGAUUUU---UUAGCUUCUUAGAGGGACUAUUGGCAUAAAGCCAAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGACGGAGCCAACGAGUACCUUGGCCGGAAGGUCUGGGUAAUCUUUUAAACUCCGUCGUGCUGGGGAUAGAGCAUUGCAAUUAUUGCUCUUCAACGAGGAAUUCCUAGUAAGCGCAAGUCAUCAGCUUGCGUUGAAUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUGGCUUAGUGA-GGCCUCUGGAUGGCUGCAACGAGAAGUUGGACAAACUUGGUCAUUUAGAGGAAGUAAAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCAGAAGGAUCAUUA-----
fu.Asp.nid  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNNNNNNNNNNNNNNNNNNNNNNNAAGUAUAAGCAA------UAUA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUACGUUUAUUUGAUAGACAUGGAUACCUGUGGUAAUCUAGAGCUAAUACAUGCUAAACCCUUCGGGGCGAAUCGCA--UGGCUUGCGGCG-AUG-GUUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACCAUGGUGGCAACGGGUACGGGGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAA-CC-GACAC-GGGGAGGUAGUGACAAUAAAUUGAUACGG-GGCUCUUUUGGGUUCGUAAUUGAUGAGAACAAUUUAAAUC------------------------------CCUAACGAGGAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGU-CUGGCUG-----GCCGGUCCG-C--CUCAC-GCG---AG--UACUG-------UC---CAUUAAUAGGGUAGUGGGGGCGAGUAUUCAGCUGUCAGAGGUGAAAUUCUUGGAUUUGCUGAAGACUAACUACUGCGAAAGCACGCCAAGGAUGUUUUAUU---AAUCAGGAACGAAAGUUAGGGGAUCGAAGACGAUCAGAUACCGUCGUAGUCUUAACCAUGAACUAUGCCGACUAGGGAUCGGGCGGUUUUCCCGCUCGGCCCUUACGAGAAAUCA--AAGUUGGUUCUGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGAAAUUGACGGAAGGGCACACAAGGCGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAA-AA---UUGA-------------------------U--UU--UUUGGAUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGCUUAAUUGC-GAAACGAACGAGACCUCGGCCCUUAAAUAGCCCGGUGUCC--CUGGCUUCUUAGGGGGACUAUCGCC-UCAAGCCGAUGGAAGUGCGCGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGACAGGGGCAGCGAGUCCCUUGGUCGAGAGGCCCGGGUAAUCUUUUAAACCCUGUCGUGCUGGGGAUAGAGCAUUGCAAUUAUUGCUCUUCAACGAGGAAUGCCUAGUAGGCACGAGUCAUCAGCUGCUGCCGAUUACGUCCCUGCCCUUUGUACACACCGCCUGUCGCUACUACCGAUGAAUGGCUCGGUGA-GGCCUCNGGAUGNCUGCAACGGAAAGCUGGUUAAACCCGNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
fu.Asp.tam  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNNNAAAAUAAGCCAUGCAUGUCUAAGUAUAAGCAC------UAUA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUACGUUUAUUUGAUAGACAUGGAUACCUGUGGUAAUCUAGAGCUAAUACAUGCUAAACCUUUCGGGGCGAAUCGCA--UGGCUUGCGGCG-AUG-GUUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACCAUGGUGGCAACGGGUACGGGGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCCGACAC-GGGGAGGUAGUGACAAUAAAUUGAUACGG-GGCUCUUUUGGGUUCGUAAUUGAUGAGUACAAUCUAAAUC------------------------------CCUAACGAGGAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGU-CUGGCUG-----GCCGGUCCG-C--CUCAC-GCG---AG--UACUG-------UC---CAUUAAUAGGGUAGUGGGGGCGAGUAUUCAGCUGUCAGAGGUGAAAUUCUUGGAUUUGCUGAAGACUAACUACUGCGAAAGCACGCCAAGGAUGUUUUAUU---AAUCAGGAACGAAAGUUAGGGGAUCGAAGACGAUCAGAUACCGUCGUAGUCUUAACCAUAAACUAUGCCGACUAGGGAUCGGGCGGUAUGCCCGCUCGGCCCUUACGAGAAAUCA--AAGUUGGUUCUGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGAAAUUGACGGAAGGGCACACAAGGCGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAA-AA---UUGA-------------------------U--UU--UUUGGAUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGCUUAAUUGC-GAAACGAACGAGACCUCGGCCCUUAAAUAGCCCGGUGUUU--CUGGCUUCUUAGGGGGACUAUCGGC-UCAAGCCGAUGGAAGUGCGCGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGACAGGGCCAGCGAGUCCCUUGACCGAGAGGUCCGGGUAAUCUUUUAAACCCUGUCGUGCUGGGGAUAGAGCAUUGCAAUUAUUGCUCUUCAACGAGGAAUGCCUAGUAGGCACGAGUCAUCAGCUCGUGCCGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUGGCUCGGUGA-GGCCUUCGGAUGGCCGCAACGGAAAGUUGGUCAAACCCGGUCAUUUAGAGGAAGUAAAAGUCGUAACAAGGUUUCNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
fu.Neu.cra  -----UACCUGGUUGAUUCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUUUAAGCA-------UAAA--------CGC-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUAUUUGAUAGACAUGGAUAACCGUGGUAAUCUAGAGCUAAUACAUGCUAAACCCUUCGGGGCGAAUCGCA--UGGCUUGUGGCG-AUG-GUUCAUUCAAAUUUCUGCCCUAUCAACUUGACGGCUGGGUCUUGGCCAGCCAUGGUGACAACGGGUACGGAGGGUUAGGGCUCGACCCCGGAGAAGGAGCCUGAGAAACGGCUACUACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCCGACAC-GGGGAGGUAGUGACAAUAAAUUGAUACAG-GGCUCUUUUGGGUUUGUAAUUGAUGAGUACAAUUUAAAUC------------------------------CCUAACGAGGAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGAGGUUAAAAAGCUCGUAGUUGGGGC-UCGGCC-----GU-CGGUCCG-C--CUCAC-GCG---UG--CACUA-------CU---GAUUAAUAGGGCAGUGGGGGCAAGUAUUCAAUUGUCAGAGGUGAAAUUCUUGGAUUUAUUGAAGACUAACUACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAGGAACGAAAGUUAGGGGAUCGAAGACGAUCAGAUACCGUCGUAGUCUUAACCAUAAACUAUGCCGAUUAGGGAUCGGACGGUUUUCCCGUUCGGCCCUUACGAUAAAUCA--AAAUGGGCUCCUGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGAAAUUGACGGAAGGGCACACCAGGGGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAG-GA---UUGA-------------------------U--UU--CGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGCUUAAUUGC-GAAACGAACGAGACCUUAACCUCUAAAUAGCCCGUAUUUG--CUGGCUUCUUAGAGGGACUAUCGGC-UCAAGCCGAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGACACAGCCAGCGAGCUCCUUGGCCGGAAGGUCCGGGUAAUCUUUUAAACUGUGUCGUGCUGGGGAUAGAGCAUUGCAAUUAUUGCUCUUCAACGAGGAAUCCCUAGUAAGCGCAAGUCAUCAGCUUGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUGGCUCAGUGA-GGCUUCCGGAUGGCCGCAACGGAAAGCUAUCCAAACUCGGUCAUUUAGAGGAAGUAAAAGUCGUAACAAGGUCUCCGUUGGUGAACCAGCGGAGGGAUCAUUA-----
fu.Can.gla  -----UAUCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUAUAAGCAA------UAUA--------AGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUACGUUUAUUUGAUAGACAUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCUUAAUCUUCUUAGACGAAUCGCA--UGGCUUGUGGCG-AUG-GUUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACCAUGGUUUCAACGGGUACGGGGAAUAAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUGACAC-AGGGAGGUAGUGACAAUAAAUCGAUACAG-GGCCCAUUCGGGUUUGUAAUUGAUGAGUACAAUGUAAAUA------------------------------CCUAACGAGGAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGAUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGC-CUGGGU-----AGCCGGUCCG-A--UUUUU-UCG---UG--UACUG------AAU--GCAUUAAUAGGGCGGUGGGGGCAAGUAUUCAAUUGUCAGAGGUGAAAUUCUUGGAUUUAUUGAAGACUAACUACUGCGAAAGCAUGCCAAGGACGUUUUAUU---AAUCAAGAACGAAAGUUAGGGGAUCGAAGAUGAUCAGAUACCGUCGUAGUCUUAACCAUAAACUAUGCCGACUAGGGAUCGGGUGGUUUACCCACUCGGCCCUUACGAGAAAUCA--AAGUCGGUUCUGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCG-CUUAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAA-AA---UUGA-------------------------U--UU--UGUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGCUUAAUUGC-GAAACGAACGAGACCUUAACCUCUAAAUAGUGGUGCAUUU--UCCACUUCUUAGAGGGACUAUCGGUUUCAAGCCGAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGCGUUCUGGGCCGCACGCGCGCUACACUGACGGAGCCAGCGAGUACCUUGGCCGAGAGGUCUUGGUAAUCUUUGAAACUCCGUCGUGCUGGGGAUAGAGCAUUGUAAUUAUUGCUCUUCAACGAGGAAUUCCUAGUAAGCGCAAGUCAUCAGCUUGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUAGUACCGAUGAAUGGCUUAGUGA-GGCCUCAGGACUGCUGCGAGGAGAAUCUGGUCAAACUUGGUCAUUUAGAGGAACUAAAAGUCGUAACAAGGUUUCCGUAGGUGAA-CUGCGGAAGGAUCAUUA-----
fu.Sac.cer  -----UAUCUGGUUGAUCCUGCCAGUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUAUAAGCAA------UAUA--------AGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUACGUUUAUUUGAUAGACAUGGAUAACCGUGGUAAUCUAGAGCUAAUACAUGCUUAAUCUCUUUAGACGAAUCGCA--UGGCUUGUGGCG-AUG-GUUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGUGGCCUACCAUGGUUUCAACGGGUACGGGGAAUAAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCUAAUUC-AGGGAGGUAGUGACAAUAAAUCGAUACAG-GGCCCAUUCGGGUUUGUAAUUGAUGAGUACAAUGUAAAUA------------------------------CCUAACGAGGAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGC-CCGGUU-----GGCCGGUCCG-A--UUUUU-UCG---UG--UACUG-----AUUUC--CAUUAAUAGGGCGGUGGGGGCAGGUAUUCAAUUGUC-GAGGUGAAAUUCUUGGAUUUAUUGAAGACUAACUACUGCGAAAGCAUGCCAAGGACGUUUUAUU---AAUCAAGAACGAAAGUUAGGGGAUCGAAGAUGAUCUGGUACCGUCGUAGUCUUAACCAUAAACUAUGCCGACUAG--AUCGGGUGGUUUACCCACUCGGUCCUUACGAGAAAUCA--AAGUCGGUUCUGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACUAGGAGUGGAGCCUGCGGC-UAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAA-AA---UUGA-------------------------U--UU--UGUGGGUGGUGGUGCAUGGCCGUUCUCAGUUGGUGGAGUGAUUUGUCUGCUUAAUUGC-GAAACGAACGAGACCUUAACCUCUAAAUAGUGGUGCAUUU--UCCACUUCUUAGAGGGACUAUCGGUUUCAAGCCGAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGCGUUCUGGGCCGCACGCGCGCUACACUGACGGAGCCAGCGAGUACCUUGGCCGAGAGGUCUUGGUAAUCUUUGAAACUCCGUCGUGCUGGGGAUAGAGCAUUGUAAUUAUUGCUCUUCAACGAGGAAUUCCUAGUAAGCGCAAGUCAUCAGCUUGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUAGUACCGAUGAAUGGCUUAGUGA-GGCCUCAGGACUGCUGCAAGGAGAAUUUGGACAAACUUGGUCAUUUAGAGGAACUAAAAGUCGUAACAAGGUUUCCGUAGGUGAACCUGCGGAAGGAUCAUUA-----
fu.Bul.hui  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-NNNNNCAAAAUAAGCCAUGCAUGUCUAAGUAUAAACAAA-----CAUA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUAUUUGAUGGACAUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCUGAGCCCUCU-GGGCGAAUCGUA--UGGCUUGCGACG-AUG-CUUCAUUCAAAUAUCUGCCCUAUCAACUUGAUGGUAGGAUAGAGGCCUACCAUGGUAUCAACGGGUACGGGGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCCGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACAG-GGCUCUAUUGGGUUUGUAAUUGAUGAGUACAAUUUAAAUC------------------------------CCUAACGAGGAACAACUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAGUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUCGGGGC-CCGGCGG-----GACGGUCCG-C---CUUAGGUG---UG--UACUU-----CU-----UAUUAAUAGGGCGGUGGGGGCAAGUAUUCCGUUGCUAGAGGUGAAAUUCUUAGGUUUACGGGAGACUAACUACUGCGAAAGCAUGCCAAGGACGUUUUAUU---GAUCAAGAACGAAGGUUAGGGGAUCAAAAACGAUUAGAUACCGUUGUAGUCUUAACAGUAAACUAUGCCGACUAGGGAUCGGUUCCAAUAAGGAAUCGGCCCUUACGAGAAAUCA--AAGUCGGUUCUGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGUGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAA-GA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACCUUAACCUCUAAAUAGCCCGGCUUUG--AUGGCUUCUUAGAGGGACUAACGGCGUUUAGCCGUUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGACUGAGCCAGCGAGUCCCUUGGCCGAGAGGUCUGGGUAAUCUUUGAAACUCAGUCGUGCUGGGGAUAGAGCAUUGCAAUUAUUGCUCUUCAACGAGGAAUACCUAGUAAGCGUGAGUCAUCAGCUCGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUGGCUUAGUGA-GAUCUCCGGAUGGCGGCAAUGAGAAGUUGAUCAAACUUGGUCAUUUAGAGGAAGUAAAAGUCGUAACAAGGUUNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
fu.Den.sul  -----NNNNNNNNNNNNNNNNNNNNUAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUAUAAACAAG-----UGUA--------UGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUAUUUGAUGGACAUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCAAAGCCCUCGCGGGCGAAUCGCA--UGGCUUGCGGCG-AUG-CUUCAUUCAAAUAUCUGCCCUAUCAACUUGAUGGUAGGAUAGAGGCCUACCAUGGUUUCAACGGGUACGGGGAAUAAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCCGACAC-GGGGAGGUAGUGACAAUAAAUCAAUAUAG-GGCUCUUUUGGGUUUAUAAUUGAUGAGUACAAUUUAAAUC------------------------------UCUAACGAGGAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGAGAC-CUGGCCG-----GGCGGUCUG-CC--UCACGGUA---UG--UACUU--------C---UAUUAAUAGGGUAGUGGGGGCAAGUAUUCAGUUGCUAGAGGUGAAAUUCUUGGAUUUACUGAAGACUAACUACUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAGGUUAGGGGAUCGAAAACGAUCAGAUACCGUUGUAGUCUUAACAGUAAACUAUGCCGACUAGGGAUCGGGCGAUCUUGUCGCUCGGCCCUUACGAGAAAUCA--AAGUCGGUUCUGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGAGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAG-UA---UUGA-------------------------A--UU--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACCUUAACCUCUUAAUAGCCAGGCUUUU--CCGGCUUCUUAGAGGGACUGUCUGCGUCUAGCAAACGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGACAGAGCCAGCGAGCACCUUGGCCGGAAGGUCUGGGUAAUCUUUGAAACUCUGUCGUGCUGGGGAUAGAGCAUUGCAAUUAUCGCUCUUCAACGAGGAAUUCCUAGUAAGCGUGAGUCAUCAGCUCGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUGGCUUAGUGA-GGUCUUGGGAUGGCUGCAAUGAGAACUUGAUCAAACUUGGUCAUUUAGAGGAA-UAAAA-UCGUAACAAGG-UUCC-UAGNNNNNNNNNNNNNNNNNNNNNNN-----
fu.Fel.oga  -----NNNNNNNNNNNNNNNNNNNNNAGUCAUAUGCU-UGUCUCAAAAUAAGCCAUGCAUGUCUAAGUAUAAACAAA-----CAUA--------GGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUAUUUGAUGGACAUGGAUAACUGUGGUAAUCUAGAGCUAAUACAUGCUGAGCCCUCU-GGGCGAAUCGCA--UGGCUUGCGGCG-AUG-CUUCAUUCAAAUAUCUGCCCUAUCAACUUGAUGGUAGGAUAGAGACCUACCAUGGUAUCAAC-GGUACGGGGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCCGACAC-GGGGAGGUAGUGACAAUAAAUCAAUACAG-GGCUCUAAUGGGUUUGUAAUUGAUGAGUACAAUUUAAAUC------------------------------CCUAACGAGGAACAACUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAGUACCGUAUAUUAAAGUUGUUGCAGUUAAAACGCUCGUAGUCGGGGC-CUGGCGG-----GGUGGUCCG-CC--UUACGGUG---UG--UACUU--------C---CAUUAAUAGGGCGGUGGGGGCAAGUAUUCCGUUGCUAGAGGUGAAAUUCUUAGAUUUACGGAAGACUAACUUCUGCGAAAGCAUGCCAAGGACGUUUUAUU---GAUCAAGAACGAAGGUUAGGGGAUCAAAAACGAUUAGAUACCGUUGUAGUCUUAACAGUAAACUAUGCCGACUAGGGAUCGG-CCA-UUCCUG-CUCGGCCCU-ACGAG-AAUCA--AAGUCGGUUCUGGGGG-AGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGUGUGGAGCCUGCGGCUUAAUUUGACUCAACACGGGGAAACUCACCAGGUCCAGACAA-AA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGCCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACCUUAACCUCUAAAUAGUCAGGCUCCG--UCGACUUCUUAGAGGGACUGUCGGCGUUUAGCCGACGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGACCGCACGCGCGCUACACUGACUGAGCCAGCGAGUCCCUUGUCCGAGAGGUCUGGGUAAUCUUUG-AA-UCAGUCGUGCUG-GGAUAGAGCAUUGCAAUUAUUGCUCUUCAACGAGGAAUACCUAGUAAGCGUGAGUCAUCAACUCGCGUUGAUUACGUCCCUGCCCUUUGUACACACCGUCUGUCGCUACUACCGAUGAAUGGCUUAGUGA-GGUCUCCGGAUGGCAGCAAUGGGAAGUUGAACAAACUUGGUCAUUUAGAGGAAGUAAAAGUCAUAACAAGGUUUCCGUANNNNNNNNNNNNNNNNNNNNNNNN-----
fu.Gig.mar  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNCU-UGUCUCAAAAUAAGCCAUGCAUGUCU-AGUAUAAUCG-------AUAC--------GGU-GAAACUGCGAAUGGCUCAUUAAAUCAGUUAAGUUUAUUUGAUAGACUUGGAUAACCGUGGUAAUCUAGAGCUAAUACAUGCUAAAUCCUCU-GGACGAAUCGUA--UGGCUUGUGACG-AUG-UAUCAUUCAAAUUUCUGCCCUAUCAACUUGAUGGUAGGAUAGAGGCCUACCAUGGUUUUAACGGGUACGGGGAAUUAGGGUUCGAUUCCGGAGAGGGAGCCUGAGAAACGGCUACCACAUCCA-GGAAGGCAGCAGGCGCGCAAAUUACCCAAUCCCGAUAC-GGGGAGGUAGUGACAAUAAAUCAAUACAG-GGCUCUUAUGGGUUUGUAAUUGAUGAGUACAAUUUAAAUC------------------------------UCUAACGAGGAACAAUUGGAGGCAAGUCUGGUGCCAGCAGCCGCGGUAAUUCCAGCUCCAAUAGCGUAUAUUAAAGUUGUUGCAGUUAAAAAGCUCGUAGUUGGGGG-UUCUACC-----GUUGGUCGG-GC--AAUUGUCU----G--UACUGC------G----UAUUAAUAGGGUAGUGGGGGCAAGUAUUCAAUUGUCAGAGGUGAAAUUCUUGGAUUUAUUGAAGACUAACUUCUGCGAAAGCAUGCCAAGGAUGUUUUAUU---AAUCAAGAACGAAAGUUAGGGGAUCGAAGACGAUCAGAUACCGUCGUAGUCUUAACCAUAAACUAUGCCGACUAGGGAUCGGACGAUUUUCUCGUUCGGC-CUUACGGGAAACCA--AAGUGGGUUCCGGGGGGAGUAUGGUCGCAAGGCUGAAACUUAAAGGAAUUGACGGAAGGGCACACCAGGGGUGGACCGUGCGGCUUAAUUUGACUCAACACGGGAAAACUCACCAGGUCCAGACAA-AA---UUGA-------------------------U--UC--UAUGGGUGGUGGUGCAUGGCCGUUCUUAGUUGGUGGAGUGAUUUGUCUGGUUAAUUCC-GAAACGAACGAGACCUUAACCUCUAAAUAGUCAGGCUUUU--ACGACUUCUUAGAGGGACUAUCGGCGUUUAGCCGAUGGAAGUUUGAGGCAAUAACAGGUCGUGAUGCCCUUAGUGUUCUGGGCCGCACGCGCGCUACACUGAUGAAGUCAUCGAGUUCCUUUACCGGAAGGUAUGGGUAAUCUUUGAAACUUCAUCGUGAUGGGGAUAGAGCAUUGCAAUUAUUGCUCUUCAACGAGGAAUCCCUAGUAAGCAUGAGUCAUCAGCUCGUGCUGAUUACGUCCCUGCCCUUUGUACACACCGCCCGUCGCUACUACCGAUGAAUGGCUUAGUGA-GACCCUCGGACGAUAUCACUGAGAAGUUGGUCAAACUUGGUCAUUUAGAGGAAGUAAAAGUCGUAACAAGGUUUCCGUNNNNNNNNNNNNNNNNNNNNNNNNN-----
fl.Bac.spl  NCUAUGAAGAGUUUGAUCCUGGCUCAGGAUNAACGCUAGCGACAGGCUUAACACAUGCAAGUCGAGGGGCAUCAUGAGGUA-GCAA-UACCUUGAUGGCGACCGGCGCACGGGUGCGUAACGCGUAUGAACCUGCCUGAUACCGGGGUAUAGCCCAUGGAAACGUGGAUUAACACCCCAUAGUACUGCAUAGUUAAAU---------GUUY------AAGGUAUCGGAUGGGCAUGCGUCCUAUUAGUUAGUUGGCGGGGUAACAGCCCACCAAGACGAUGAUAGGUAGGGGUUCUGAGAGGAAGGUCCCCCACAUUGGAACUGAGACACGGUCCAAACUCCUACGGGAGGCAGCAGUGAGGAAUAUUGGUCAAUGGACGUAAGUCUGAACCAGCCAAGUCGCGUGAGGGAAGACUGCCUNUGGGUUGUAAACCNCUUUUAUAAGGGAAGAAUAAGUUCUA--CGUG----UAGAAUGAUGCCUGUACCUUAUGAAUAAGCAUCGGCUNACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGAUGCGAGCGUUAUCCGGAUUUAUUGGGUUUAAAGGGUGCGUAGGCGGUUUAUUAAGUUAGUGGUUAAAUAUUUGAGCUCAACUC-AAUUGUGCCAUUAAUACUGGUAAACUGGAGUACAGGCGAGGUAGGCGGAAUAAGUUAAGUAGCGGUGAAAUGCAUAGAUAUAACUUAGAACUCCGAUUGCGAAGGCAGCUUACCAGACUGUAACUGACGCUGAUGCACGAGAGCGUGGGUAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGCUCACUNGUUCUSUGCGA-UAUA--UUGUACGGGAUUAAGCGAAAGUAUUAAGUGAGCCACCUGGGGAGUACGUCGGCAACGAUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGAGGAACAUGUGGUUUAAUUCGAUGAUACGCGAGGAACCUUACCUGGGUUUAAA-UGAAAUGCCGUAUUUGGAAACAGAUAUUCUC-UUCG-GAGCUUUUUCR-AGGUGCUGCAUGGUUGUCGUCAGCUCGUGCCGUGAGGUGUCGGGUUAAGUCCCAUAACGAGCGCAACCCUUACCGUUAGUUGCUAGCAUAAUG-AUGAGCACUCUAACGGGACUGCCACCGUAA-GGUGGAGGAAGGCGGGGAUGACGUCAAAUCAGCACGGCCCUUACACCCAGGGCUACACACGUGUUACAAUGGCCGGUACAGAGGGCCGCUACCAGGUGACUGGAUGCCAAUCUC-AAAAGCCGGUCGUAGUUCGGAUUGGAGUCUGUAACCCGACUCCAUGAAGUUGGAUUCGCUAGUAAUCGCGCAUCAGCCUGGCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCAAGCCAUGGAAGCCGGGGGUGCCUGAAGUCCGUAACC-----GCGA----GGAUCGGCCUAGGGCAAAACUGGUAACNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
fl.Cyt.mar  ACGAUGAAGAGUUUGAUCCUGGCUCAGGAUNAACGCUAGCGGCAGGCCUAACACAUGCAAGUCGAACGGUAACAGGGAAAAGCUUGCUUUUCUGCUGACGAGUGGCGCACGGGUGCGUAACGCGUAUGAAUCUGCCUUGUACUGGAGUAUAGCCCAGGGAAACUUGGAUUAAUCCUCCAUAGUCUAGCAUUAGUAAAG---------GUUA-------CGGUACAAGAUGAGCAUGCGUCCUAUUAGCUAGUAGGUGUGGUAACGGCACACCUAGGCAACGAUAGGUAGGGGUCCUGAGAGGGAGAUCCCCCACACUGGUACUGAGACACGGACCAGACUCCUACGGGAGGCAGCAGUGAGGAAUAUUGGACAAUGGGCGCAAGCCNNAUCCAGCCAUGCCGCGUGCAGGAAGACUGCCUAUGGGUUGUAAACUNCUUUUAUACGGGAAGAAUAAGGUCUA--CGAG----UAGGCUGAUGACGGUACCGUAAGAAUAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGNNNAUACGGAGNGUGCNAGCGUUAUCCGGAAUCAUUGGGUUUAAAGGGUCCGUAGGUGGACUAAUAAGUCAGUGGUGAAAGUCUGCAGCUUAACUGUAGAAUUGCCAUUGAUACUGUUAGUCUUGAAUUGUUAUGAAGUAACUAGAAUAUGUAGUGUAGCGGUGAAAUGCAUAGAUAUUACAUAGAAUACCGAUUGCGAAGGCAGGUUACUAAUAAUAYAUUGACACUGAUGGACGAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGGUUACUAGCUGUCCGGCCCAUUGAGGGCUGGGUGGCCAAGCGAAAGUGAUAAGUAACCCACCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGGCCNGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGAUACGCGAGGAACCUUACCAGGGCUUAAA-UGGUCUGACAGGAGUGGAAACAUUCUUUUC--UUCG--GACGAUUACA-AGGUGCUGCAUGGUUGUCGUCAGCUCGUGCCGUGAGGUGUCAGGUUAAGUCCUAUAACGAGCGCAACCCCUGUGGUUAGUUGCCAUCGAAAUG-UCGGGAACUCUAGCCAGACUGCCAGUGCAA-ACUGGAGGAAGGUGGGGAUGACGUCAAAUCAUCACGGCCCUUACGUCCUGGGCCACACACGUGCUACAAUGGUAGGUACAGAGAGCAGCCACUUUGCAAAGAGGAGCGAAUCUA-UAAAACCUAUCACAGUUCGGAUCGGGGUCUGCAACUCGACCCCGUGAAGCUGGAAUCGCUAGUAAUCGCAUAUCAGCCUGAUGCGGUGAAUACGUUCCCGGGCNUUGUACACACCGCCCGUCAAGNCAUGGAAGCNNGGGGUACCUGAAGUCCGUCACC-----GUAA----GGAGCGGCCUAGGGUAAGACCGGUAACUGGGGCNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
fl.Fla.xan  -----NNNNNNNNNNNNNNNNNNNNNNNNNNAACGCUAGCGGCAGGCUUAACACAUGCAAGUCGAGGGGUAUAGUUC-----UUCG-----GAAUUAGAGACCGGCGCACGGGUGCGUAACGCGUAUGAAUCUACCUUUCACAGAGGGAUAGCCCAGAGAAAUUUGGAUUAAUACCUUAUAGUAUAGCAUUAUUAAAGA--------UUUA------UCGGUGAAAGAUGAGCAUGCGUCCCAUUAGUUAGUUGGUAAGGUAACGGCUUACCAAGGCAACGAUGGUUAGGGGUCCUGAGAGGGAGAUCCCCCACAUUGGUACUGAGACACGGACCAGACUCCUACGGGAGGCACCAGUGAGGAAUAUUGGACAAUGGGCGCAAGCCUGACCCAGCCAUGCCGCGUGCAGGAUGACGGUCUAUGGAUUGUAAACUGCUUUUAUACAGGAAGAAACAGUUCUA--CGUG----UAGAACCUUGACGGUACUGUAAGAAUAAGGAUCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGAUCCCAAGCUUAUCCGGAAUCAUUGGGUUUAAAGGGUCCGUAGGCGGUCAGAUAAGUCAGUGGUGAAAGCCCAUCGCUCAACGGUGGAACGGCCAUUGAUACUGUCUGACUUGAAUUAUUAGGAAGUAACUAGAAUAUGUAGUGUAGCGGUGAAAUGCUUAGAGAUUACAUGGAAUACCAAUUGCGAAGGCAGGUUACUACUAAUAUAUUGACGCUGAUGGACGAAAGCGUGGGUAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGGAUACUAGCUGUUGGGC--GCAA---GUUCAGUGGCUAAGCGAAAGUGAUAAGUAUCCCACCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGAUACGCGAGGAACCUUACCAAGGCUUAAA-UGGAUUGACCGGUUUGGAAACAGACUUUUC--GCAA--GACAUUUACA-AGGUGCUGCAUGGUUGUCGUCAGCUCGUGCCGUGAGGUGUCAGGUUAAGUCCUAUAACGAGCGCAACCCCUGUUGUUAGUUGCCAGCGACAAG-UCGGGAACUCUAACAAGACUGCCAGUGCAA-ACUGGAAGAAAGUGGGGAUGACGUCAAAUCAUCACGGCCCUUACGCCUUGGGCUACACACGUGCUACAAUGGCCGGUACAGAGAGCAGCCACUGGGCGAGCAGGAGCGAAUCUA-UAAAACCGGUCACAGUUCGGAUCGGAGUCUGCAACUCGACUCCGUGAAGCUGGGAUCGCUAGUAAUCGCAGAUCAGCCUGAUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCAAGCCAUGGAAGCUGGGGGUGCCUGAANUCGGUAACC-----GCAA----GGAGCUGCCUANGGUAAAACUGGUAACUAGGGCUAAGUCGUAACAAGGUANCCNUANNNNNNNNNNNNNNNNNNNNNNNN-----
fl.Mar.psy  -----NNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNGNCCUAAUACAUGCAAGUCGAGGGGUGGCCAGUGUC--CCGN----GNUACAGNNCACCGGCGCACGGGUGNGUAACGCAUAUAAAUCUACCUCGUACUGAGGGAUAACCCACAGAAAUAUGGNUUAAUACCUCAUANUAUAGCCUGAUCAAAG---------UUCG-------CGGUACAAGAUGAGUAUGCGUUCUAUUAGCUAGAUGGUGGGGUAACGGCUCACCAUGACAGCGAUAAAUAGGGGCCCUGAGAGGGGGAUCCCCCACACUGGUACUGAGACACGGACCAGACUCCUACGGGAGGCAGCAGUGAGGAAUAUUGGACAAUGGGCGAGAGCCUGAUCCAGCCAUGCCGCGUGCAGGANGACUGCCUAUGGGUUGUAAACUGCUUUUAUACAGGAAGAAACCCCUCCA--CNUG----UG-AGGAUUGACGGUACNGUAGGAAUAAGGAUCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGAUCCAAGCGUUAUCCGGAAUCAUUGGGUUUAAAGGGUCCGUAGGCGGGACAAUCAGUCAGCGGUGAAAGUCUGUGGCUCAACCAUAGAAUUGCCAUUGAUACUGUUGUUCUUGAAUACUUAUGAAGUGGUUGGAAUAUGUAGUGUAGCGGUGAAAUGCAUAGAUAUUACAUAGAACACCUAUUGCGAAGGCAGGUCACUAAUAAGACACUGACGCUGAUGGACGAAAGCGUGGGUAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGGUUACUAGCUGUUCGGGA-UUCG--GACUGAGUGGCUAAGCGAAAGUGAUAAGUAUUCCACCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGGNCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGAUACGCGAGGAACCUUACCAGGGCUUAAA-UGUAUUGACAGGUUUAGAGAUAUACUUUUC--UUCG--GACAUUUACA-AGGUGCUGCAUGGUUGUCGUCAGCUCGUGCCGUGAGGUGUUAGGUUAAGUCCUAUAACGAGCGCAACCCCUGUUGUUAGUUGCCAGCGGAAUG-UCGGGAACUCUAACAAGACUGCCCGUGCAA-ACUGGAGGAAGGUGGGGAUGACGUCAAAUCAUCACGGCCCUUACGUCAUGGGCUACACACGUGCUACAAUGGUAGGGACAGAGAGCAGCCACUUGGCGAGAAGGAGCGAAUCUA-UAAACCCUAUCACAGUUCGGAUCGGAGUCUGCAACUCGACUCCGUGAAGCUGGAAUCGCUAGUAAUCGCAUAUCAGCCUGAUNCGGUGAAUACGUUCCCGGACCUUGUACACACCGCCCGUCAAGCCAUGGAAGCCGGGAGUGCCUGAAGUCCAUCACC-----GCAA----GGAGCGGCCUAGGGUAAGAUCCGUGACUAGGGNNAAGUCGUAACAAGGUANNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
fl.Pre.eno  -----NNNNNNNNNNNNNNNNNNNNNNGAUGAACGCUAGCUACAGGCCUAACACAUGCAAGUCGAGGGGCAUCA-UGAUGAUCUUGAUCGUU-GAUGGCGACCGGCGCACGGGUGAGUAACGCGUAUCAACCUACCCAUGACUAAGGGAUAACCCGUCGAAAGUCGGCCUAAGACCUUAUGUAAUCGCAUGAUGAAAGA--------UUAA------UUGGUGAUGGAUGGGGAUGCGUCUGAUUAGCUUGUCGGUGAGGUAACGGCUCACCGAGGCAACGAUCAGUAGGGGUUCUGAGAGGAAGGUCCCCCACAUUGGAACUGAGACACGGUCCAAACUCCUACGGGAGGCAGCAGUGAGGAAUAUUGGUCAAUGGACGAUAGUCUGAACCAGCCAAGUAGCGUGCAGGAUGACGGCCUAUGGGUUGUAAACUGCUUUUAUGCGGGGAUAAAGUGAGGGA--CGUG----UCCCUCAUUGCAUGUACCGCAUGAAUAAGGACCGGCUAAUUCCGUGCCAGCAGCCGCGGUAAUACGGAAGGUCCAGGCGUUAUCCGGAUUUAUUGGGUUUAAAGGGAGCGUAGGCUGCCGUUUAAGCGUGUUGUGAAAUGUACCGGCUCAACCGGUGAUGUGCAGCGCGAACUGGAUGGCUUGAGUACGAAGAGGGAAUGCGGAACUCGUGGUGUAGCGGUGAAAUGCUUAGAUAUCACGAGGAACUCCGAUCGCGAAGGCAGCAUUCCGUUUCGUGACUGACGCUGAUGCUCGAAAGUGCGGGUAUCGAACAGGAUUAGAUACCCUGGUAGUCCGCACGGUAAACGAUGGAUGCUCGCUGUUGGAUA-UUUU--UAUUCAGUGGCUAAGUGAAAGCAUUAAGCAUCCCACCUGGGGAGUACGCCGGCAACGGUGAAACUCAAAGGAAUUGACGGGGGCCCGCACNAGCGGAGGAACAUGUGGUUUAAUUCGAUGAUACGCGAGGAAC-UUACCCGGGCUUGAA-UUAGAGGAAAGAUCCAGAGAUGGUGAUGCCC-UUCG-GGGUUCUGUGA-AGGUGCUGCAUGGUUGUCGUCAGCUCGUGCCGUGAGGUGUCGGCUUAAGUGCCAUAACGAGCGCAACCCUUUUCCUUAGUUGCCAUCGGCAAG-CCGGGCACUCUGAGGAUACUGCCUCCGCAA-GGAGGAGGAAGGUGGGGAUGACGUCAAAUCAGCACGGCCCUUACGUCCGGGGCUACACACGUGUUACAAUGGCGCGUACAGAGAGCUGGGUUUGCGCAAGCACUCUCAAAUCUU-UAAAACGCGUCUCAGUUCGGACCGGGGUCUGCAACCCGACCCCGCGAAGCUGGAUUCGCUAGUAAUCGCGCAUCAGCCUGGCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCAAGCCAUGAAAGCNGGGGGCGCCUGAAGUCCGUGACC-----GCAA----GGAUCGGCCUAGGCGAGACCCGGUGAUUNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNNN-----
fu.Fus.gon  CGAACGAAGAGUUUGAUCCUGGCUCAGGAUGAACGCUGACAGAAUGCUUAACACAUGCAAGUCG-ACUCGAGUC--------UUCG--------GACUUGGGUNGCGGACGGGUGAGUAACGCGUAAAAACUUGCCUCAUAGUCUGGGACAACAUCUGGAAACGGAUGCUAAUACCGGAUAUUAUGGCAUUAUGAAAGC--------UAUA-----UGCGCUAUGAGAGAGCUUUGCGUCCCAUUAGCUAGUUGGUGAGGUAACGGCCCACCAAGGCGAUGAUGGGUAGCCGGCCUGAGAGGGUUAACGGCCACAAGGGGACUGAGACACGGCCCNUACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGACCAAAGUCUGAUCCAGCAAUUCUGUGUGCACGAUGACGUUUUUCGGAAUGUAAAGUGCNUUCAGUCGGGAAGAAGC---------AAGU-------------GACGGUACCGACAGAAGAAGCGACGGCUAAAUACGUGCCAGCAGCCGCGGUAAUACGUAUGUCGCNAGCGUUAUCCGGAUUUAUUGGGCGUAAAGCGCGUCUAGGCGGCAAGGAAAGUCUGAUGUGAAAAUGCGGGGCUCAACUCCGUAUU-GCGUUGGAAACUGCCUUNCUAGAGUNCUGGAGAGGUAGGCGGAACUACAAGUGUAGAGGUGAAAUUCGUAGAUAUUUGUAGGAAUGCNNAUGGGGAAGCCAGCCUACUGGACAGAUACUGACGCUAAAGCGCNAAAGCGUGGGUAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCUGUAAACGAUGAUUACUAGGUGUUGGGG--GUCA-AACCUCAGCGCCNAAGCUAACGCGAUAAGUAAUCCGCCUGGGGAGUACGUACGCAAGUAUGAAACUCAAAGGAAUUGACGGGGACCNGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGANNNAACGCGAGGAACCUUACCAGCGUUUGACAUCCUACAAAGAGUGCAGAGAUGCGCUUGUGCUUUCGAGAAUGUAGUGACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUAUCGUAUGUUACCAGCCUUUAG-UUGGGGACUCAUGCGAUACUGCCUGCGACGAGCAGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGCCCCUUAUACGCUGGGCUACACACGUGCUACAAUGGGUAGUACAGAGAGCGGCGAACCCGCGAGGGGGAGCAAAUCUCAGAAAACUAUUCUUAGUUCGGAUUGUACUCUGCAACUCGAGUACAUGAAGUUGGAAUCGCUAGUAAUCGCAAAUCAGCAUGUUGCGGUGAAUACGUUCUCGGGUCUUGUACACACCGCCCGUCACACCACGAGAGUUGGUUGCACCUGAAGUAGCAGGCCUAACCGUAAGGAAGGAUGCUCCGAGGGUGUGGUUAGCGANUGGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
fu.Fus.nul  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-mmmmmmmmmmmmmmm--mmmm--mmmmmmmmmmmGGGUGGCGGACGGGUGAGUAACGCGUAAAAACUUGCCUCACAGCUAGGGACAACAUUUAGAAAUGAAUGCUAAUNCCUNAUAUUAUGGCAUUAUGAAAGC--------UAUA-----AGCACUGUGAGAGAGCUUUGCGUCCCAUUAGCUAGUUGGAGAGGUAACAGCUCACCAAGGCGAUGAUGGGUAGCCGGCCUGAGAGGGUGAACGGCCACAAGGGGACUGAGACACGGCCCNNNCUCCUACGGGAGGCNGCAGUGGGGAAUAUUGGACAAUGGACCGAGGUCUGAUCCAGCAAUUCUGUGUGCACGAUGAAGUUUUUCGGAAUGUAAAGUGCUUUCAGUUGGGAAGAAAG---------AAAU-------------GACGGUACCAACAGAAGAAGUGACGGCUAAAUACGUGCCAGCAGCCGCGGUAAUACGUAUGUCACGAGCGUUAUCCGGAUUUAUUGGGCGUAAAGCGCGUCUAGGUGGUUAUGUAAGUCUGAUGUGAAAAUGCAGGGCUCAACUCUGUAUU-GCGUUGGAAACUGUAUAACUAGAGUACUGGAGAGGUAAGCGGAACUACAAGUGUAGAGGUGAAAUUCGUAGAUAUUUGUAGGAAUGCCGAUGGGGAAGCCAGCUUACUGGACAGAUACUGACGCUAAAGCGCGAAAGCGUGGGUAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAUUACUAGGUGUUGGGG--GUCG-AACCUCAGCGCCCAAGCUAACGCGAUAAGUNAUCCGCCUGGGGAGUACGUACGCAAGUAUGAAACUCAAAGGAAUUGACGGGGAC-CGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGACGNAACGCGAGGAACCUUACCAGCGUUUGACAUCUUAGGAAUGAGAUAGAGAUNUUUCAGUGUCUUCGGAAACCUAAAGACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUNUCGUAUGUUACCAUCAUUAAG-UUGGGGACUCAUGCGAUACUGCCUNCGAUGAGUAGGAGGAAGGUGGGGAUGACGUNNAGUCAUCAUGCCCCUNAUACGCUGGGCUACACACGUGCUACAAUGGGUAGAACAGAGAGUUGCAAAGCCGUGAGGUGGAGCUAAUCUCAGAAAACUNUUCUUAGUUCGGAUUGUACUCUGCAACUCGAGUACAUGAAGUUGGAAUCGCUAGUAAUCGCGAAUCAGCAUGUCGCGGUGAAUACGUUCUCGGGUCUUGUACACACCGCCCGUCACACCACGAGAGUUGGUUGCACCUGAAGUAGCAGGCCUAACCGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
fu.Lep.buc  -----mmmmmmmmmmAUCCUGGCUCAGGAUGAACGCUGACAGAAUGCUUAACACAUGCAAGUCUUUGGC--RAAUCUGUG--CUUG--CACAGCCUAGC-CAAGGCGGACGGGUGAGUAACGCGUAAAAACUUGCCCUGCAGACAGGGAUAACAGACGGAAACGACUGAUAAUACCUGAUAYAAUUGCAUAAUGAAAA---------GUGA-------UGCUGCAGGAGAGCUUUGCGUCCUAUUAGCUUGUUGGUGAGGUAANGGCUCACCAAGGCGAUGAUAGGUAGCCGGCCUGAGAGGGUGAACGGCCACAAGGGGACUGAGAUACGGCCCUUACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGGGCANCCCUGAUCCAGCAAUUCUGUGUGCACGAAGAAGGUUUUCGGAUUGUAAAGUGCUUUCAGCAGGGAAGAAGA---------AAGU-------------GACGGUACCUGCAGAAGAAGCGACGGCUAAAUACGUGCCAGCAGCCGCGGUAAUACGUAUGUCGCAAGCGUUAUCCGGAAUUAUUGGGCAUAAAGGGCAUCUAGGCGGCCAGACAAGUCUGGGGUGAAAACUUGCGGCUCAACCGCAAGCCUGCCCUGGAAACUGUUUGGCUAGAGUGCUGGAGAGGUGGACGGAACUGCACGAGUAGAGGUGAAAUUCGUAGAUAUGUGCAGGAAUGCCGAUGAUGAAGAUAGUUCACUGGACGGNAACUGACGCUGWAGUGCGVAARCUGGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCCACCCGUAAACGAUGAUUACUGGGUGUGGGCAU-GAAG-AGUGUCCGUGCCGAAGCUAAUGCGAUAAGUAAUCCGCCUGGGGAGUACGKCCGCAAGGCUGAAACUCAAAGGAAUUGACGGGGACCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGACGCAACGCGAGGAACCUUACCAGAUCUUGACAUCCUACGAAUGCCUGUGAGAACAGGCAGUGCCUUCGGGAACGUWGAGACAGGUGGUGCAUGGCUGUCGACAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUAUCGCUAGUUGCCAUCAUUAAG-UUGGGGACUCUAGCGAGACUGCCUGCGAAGAGCAGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGCCCCUUAUGAUCUGGGCUACACACGUGCUACAAUGGCCGGNACAAAGAGCUGCAAAACGGUAACGUUUAGCCAAUCUU-UAAAGCCGGUCCAAGUUCGGAUUGAAGUCUGCAACUCGACUUCAUGAAGCCGGAAUCGCUAGUAAUCGCAGAUCAGCAUGCUGCGGUGAAUACGUUCUCGGGUCUUGUACACACCGCCCGUCACACCACGAGAGUUGUUUGNACCUGAAGCCGCCGGUCCAACCGUAAGGAGGAAGGCGUCUAAGGUGUGGAUAGUGAUUGGGGUGAAGUCGUAACAAGGUAAGUCGmmmmmmmmmmmmmmmmmmmmmmmmm-----
fu.Lep.mic  -----mmmmmmmmmmmmmmmGGCUCAGGAUAAACGCUGACAGAAUGCUUAACACAUGCAAGUCG-AUGAUGGGAGCUAG---CUUG---CUAGAAGAAGUCAUGGCGGACGGGUGAGUAACGUGUAAAAACUUACCAUAUAGACUGGGAUAACAGAGGGAAACUUCUGAUAAUACUGGAUA-AGUUGCAUAAUGAAAGUA-------GCAA-----UACGCUAUAUGAGAGCUUUGCAUCCUAUUAGCUAGUUGGUGGGGUAAAAGCCUACCAAGGCGAUGAUAGGUAGCCGGCCUGAGAGGGUGGACGGCCACAAGGGGACUGAGAUACGGCCCUUACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGAGGCAACUCUGAUCCAGCAAUUCUGUGUGUGUGAAGAAGGUUUUAGGACUGUAAAACACUUUUAGUAGGGAAGAAAG---------AAAU-------------GACGGUACCUACAGAAGAAGCGACGGCUAAAUACGUGCCAGCAGCCGCGGUAAUACGUAUGUCGCGAGCGUUAUCCGGAAUUAUUGGGCUUAAAGGGCAUCUAGGCGGUUAAACAAGUUGAAGGUGAAAACCUGUGGCUCAACCAUAGGCUUGCCUACAAAACUGUAUAACUAGAGUACUGGAAAGGUGGGUGGAACUACACGAGUAGAGGUGAAAUUCGUAGAUAUGUGUAGGAAUGCCGAUGAUGAAGAUAACUCACUGGACAGCAACUGACGCUGAAGUGCGAAAGCUAGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCUAGCUGUAAACGAUGAUCACUGGGUGUGGGGAU-UCGA-AGUCUCUGUGCCGAANCAAAAGCGAUAAGUGAUCCGCCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGACCCGCACAAGUGGUGGAGCAUGUGGUUUAAUUCGACGCAACGCGAGGAACCUUACCAGAUCUUGACAUCCUCCGAAGAGCAUAGAAGUAUGCUUGUGCCUACGGGAACGGAGAGACAGGUGGUGCAUGGCUGUCGACAGCUCGUGUUGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGAAACCCCUAUCAUUAGUUACCAUCAUUAAG-UUGGGGACUCUAAUGAAACUGCCUACGAAGAGUAGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGCCCCUUAUGAUCUGGGCUACACACGUGCUACAAUGGAUAGUACAAAGAGAAGCUUUGURGCGAUACAUGNCNANACUAAGAAAGCUAUUCUUAGUUCGGAUUGAAGUCUGCAACUCGACUUCAUGAAGUUGGAAUCACUAGUAAUCGUGAAUCAGCAUGUCACGGUGAAKACGUUNUCGGGUNUUGUACACACCGCCCGUCACACCACGAGAGUUGUUUGCACCUGAAAUUACUGGCCUAACUGUAAAGGGGGAGGUACUGAAGGUGUGGAUAGCGAUUGGGGUGAAGUCGUAACAAGGUAAmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
gr.The.ros  GGGAUGGAGAGUUUGAUCCUGGCUCAGGGGGAACGCUGGCGGCGUGCCUAAUGCAUGCAAGUCGGACGGGANGCACGCN---CUCU---GGCGUGCCGACCGUGGCGGACGGGUGCGUAACACGUGGGAACCCUCCCGGGUGCGGGGGAUAACCCGGGGAAACUCGGGCUAAUACCCCAUACGCUUGGUGAAGGAAAGGCGCAG---GCGA-CUGUGCUGUGCUCGGAGGGCCCUGCGGCCUAUCAGCUAGACGGUAGGGUAACGGCCUACCGUGGCGAUGACGGGUAGCUGGUCUGAGAGGAUGGCCAGCCACACGGGCACUGAGACACGGGCCCGACUCCUACGGGAGGCAGCAGCAGGGAAUCUUCCGCAAUGGGGGCAACCCUGACGGAGCGACGCCGCGUGCGGGAGGAAGCCCUUCGGGGUGUAAACCGCUGUUCGGGGGGACGAUC----------GAGC-------------GACGGUACCCUCGGAGCAAGUCCCGGCUAACUACGUGCCAGCAGCCGCGGUAAGACGUAGGGGGCGAGCGUUACCCGGAGUCACUGGGCGUAAAGGGCGUGUAGGCGGCUGGGUACGCCGCGUGUGAAAGUCCCCGGCUCAACCGGGGAGGGUCGCGCGGGACGGCCUGGCUCGAGGGCGGGAGAGGCGGGUGGAAUUCCCGGUGUAGCGGUGAAAUGCGUAGAGAUCGGGAGGAACGCCGGUGGCGAAGGCGGCCCGCUGGCCCGUACCUGACGCUGAGGCGCGAAGGCGUGGGGAGCGAACCGGAUUAGAUACCCGGGUAGUCCACGCAGUAAACGAUGCGGGCGAGGUGUGGGUGGUUUGACCCCAUCCGUGCCGGCGCCAACGCAGUAAGCCCGCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCAGCGGAGCGUGUGGUUUAAUUCGACGCAACGCGAAGAACCUUACCAGGGCUUGACAUGCACCGAACCUGGCUGAAAGGCUGGGGUGCCGUGAGGAGCGGUGGCACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGGGGUCAGUUAC------GCGG-----GUG-UCUGACCCGACUGCCGGGGAAAGCCCGGAGGAAGGAGGGGAUGACGUCAAGUCAGCAUGGCCCUGACGCCCUGGGCGACACACACGCUACAGUGACCGGGACAGUGGGCAGCGAAGGGGCGACCUGGAGCCAAUCCCGCAAACCCGGUCGUGGUGGGGAUCGCAGGCUGCAACCCGCCUGCGUGAACGCGGAGUUGCUAGUAACCGCCGGUCAGCCUACGGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACGUCACGAAAGCUGGCUUCACCUGAAGCUGGUGGGCCAACCGCACGGGGGCAGCCGUCGAGGGUGGGGCUGGUGAUUGGGACGAAGUCGUAACAAGGUAGCCGUACCGGAAGGNNNNGGUGGAUCACCUCCUUU
pt.Iso.pal  NNUCCGAAGGGUUUGAUCCUGGCUCAGAAUGAACGUUGGCGGCGUGGAUUAGGCAUGCAAGUCGAACGGCC-----------GCAA----------GGCUAGUGGCGAAAGGGUGAGUAAGGCGACGGAACCUACCCCGAGGUUGG-CAUAGCCGGGGGAAACUCCGGGUAAUUCCCAGCGAUGUCGCAUCGC-AAAGGU-------GCAA----UUCCGCCUCGGGACGGGCCGUNGUGGUAUUAGGUAGUUGGUGGGGUCACGGCCCACNAAGCCGACGAUGCCUACCGGGCGUGCGAGCGUGGCCCGGCACACUGGGACUGAGACACUGCCCAGACUCCUACGGGAGGCUGCAGUCGAG-AUCUUCGGCAAUGGGCGCAAGCCUGACCGAGCGACGCGCNGUGGAGGAAGACGGCCCUUGGGUUGUAAACUCCUGNCNNGGGGAAGGAAGGGUCGGC---GAAG---AGCCNAUCUUGACCGC-UCCCUGGAGGAAGCACGGGCUAAGUUCGUGCCAGCAGCCGCGGUAAGACGAACCGUGCAAACGUUAUUCGGAAUCACUGGGCUUAAAGGGCGCGUAGGCGGAAGGGCGCGUCGGCGUUGAAAUNCCCCGGCUCAACCGGGNNAGCGGCGUCGAAACGGCCCUUCNGGAGGGGCGUAGAGGGACUCGGAACUUCCGGUGGAGCGGUGAUAUGCGUUGAGAUCGGAAGGAANGCCCGUGGCGAAAGCGGAGUCCUGGACGCUUACUGACGCUGAGGCGCGAAAGCCAGGGGAGCAAACGGGAUUAGAUACCCCGGUAGUCCUGGCUGUAAACRAUGGGCACUUGGCAGUGGGUUCUCGA-GGGUCCACUGCCNNNGGGAAACCGUGAAGUGCCNNGCCUGGGGAGUAUGGUCGCAAGGCUGAAACUCAAAGGAAUUCACGGGGGCUCACACAAGCGGUGGAGGAUGUGGCUUAAUUCGAANNUACGCGAAAAACCUUACCAGGGCUUGACAUGAGGGAUAGCCGGCGGAAACGUCGGUGCGCCGCAAGUGGAACCUAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGCCGUGAGGUGUCGGGUUAAGUCCCUUAACGAGCGAAACCCCUGCGGCNAGUUGCCAACACUCUG-GUGGGGACUCU-GCCAGACCGCCGGCGUGAAGCCGGAGGAAGGCGGGGAUGACGUCAAGUCCUCAUGGCCCUUAUGCCNAGGGCUGCACACGUGCUACUAUGGCGGGGACAAAGCGUCGCCACGCCGUAAGGCCGAGCCAACCGCGUAAACCCCGCCCCAGUUCGGAUCGAGGGCUGCAACCCGCCCUCGUGAAGCCGGAAUCGCUAGUAAUCGCGGAUCAGCAUGCCGCGGUGAAUGUGUUCCUGAGCCUUGUACACACCGCCCGUCAAGNCACCAAAGGGGGGGGCACCCGAAGUCGAAGA-UC----UCAC---GACGGGCGCCGAAGGUGAAACUCCNNAUGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
pt.Pla.bra  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-------mmmm-------mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-------mmmm-----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmCAAUGGGCGAAAGCCUGACCAAGCGAUGCCGCGUGCGGGAUGAAGGCCCUUGGGUUGUAAACCGCUGUCAGGGGGGAUGAAACU--------UUCG----------GUUGACAGA-GCCCCAGAGGAAGCACGGGCUAAGUACGUGCCAGCAGCCGCGGUAACACGUACUGUGCGAACGUUAUUCGGAAUCACUGGGCUUAAAGGGUGCGUAGGCGGCCUGGAUAGUCAGAUGUGAAAUCCCGCGGCUCAACCGUGGAACUGCAUUUGAAACUGCCAGGCUUGAGUGAGACAGGGGUGUGUGGAACUUCUCGUGGAGCGGUGAAAUGUGUUGAUAUGAGAAGGAACACCGGUGGCGAAAGCGACACACUGGGUCUUAACUGACGCUGAGGCACGAAAGCCAGGGGAGCGAACGGGAUUAGAUACCCCGGUAGUCCUGGCUGUAAACGUUGAGUACUAGUUGGUGGGAACUUCG-GUUCUCACGGACGUAGCAAAAGUGUUAAGUACUCCGCCUGGGGAGUAUGGUCGCAAGGCUGAAACUCAAAGGAAUUGACGGGGGCUCACACAAGCGGUGGAGCAUGUGGCUUAAUUCGAGGCAACGCGAAGAACCUUACCUAGACUUGACAUGAUGGAUAGCUGGCUGAAAGGUCAGUGACGCUUCGGUGGAACAUGCACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCUUGAACGAGCGCAACCCCUGUCGCCAGUUGCCAGCAAUAAUGUUGGGGACUCUGGCGAGACCGCCGGUGUUAAACCGGAGGAAGGUGGGGACGACGUCAAGUCAUCAUGGCCUUUAUGUCUAGGGCUGCACACGUGCUACAAUGCGGCGUACAAAGGGAAGCC-ACCCGCGAGGGGGAGCAAAUCUCAGAAAGCGCCGCUCAGUUCGGAUUGUAGGCUGCAACUCGCCUACAUGAAGCUGGAAUCGCUAGUAAUCGCAGGUCAGCAUACUGCGGUGAAUGUGUUCCUGAGCCUUGUACACACCGCCCGUCAAGCCACGAAAGGGGGGGGCAUCCUAAGUCACUGAGCUA---AUC---UGGCAGGUGCCUAAGAUGAACUCCCUGAUUGGGACUAAGUCGUAACAAGGUAGCCGUAGGGGAACCUGCGGCUGGAUCACCUCCUU-
pt.Pla.mar  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-------mmmm--------mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-------mmmm----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmCUAGGGGGUGUGAGAGCAUGGCCCCCACCACUGGGACUGAGACACUGCCCAGACACCUACGGGUGGCUGCAGUCGAGAAUCUUCGGCAAUGGACGCAAGUCUGACCGAGCGACGCCGCGUGCGGGAUGAAGGCCCUUGGGUUGUAAACCGCUGUCAGAGGGGAUGAAAUGCAGAAGGGUUAU--CCCUUUUGUUUGACAGA-GCCUCAGAGGAAGCACGGGCUAAGUCCGUGCCAGCAGCCGCGGUAACACGUACUGUGCGAACGUUAUUCGGAAUCACUGGGCUUAAAGGGUGCGUAGGCGGUUUAGUAAGUAGGGUGUGAAAUGCCAGGGCUCAACCUUGGCACGGCGCUCUAAACUGCUAAACUUGAGUGAGAUAGGGGUGUACGGAACUUCCGGUGGAGCGGUGAAAUGCGUUGAUAUCGGAAGGAACACCGGUGGCGAAAGCGGUACACUGGGUCUUAACUGACGCUGAGGCACGAAAGCUAGGGUAGCGAACGGGAUUAGAUACCCCGGUAGUCCUAGCCGUAAACGAUGAGUACUAGUUGGGAGGAGCUUCG-GCUCAUCCGGACGUAGCGAAAGCAUUAAGUACUCCGCCUGGGGAGUAUGGUCGCAAGGCUGAAACUCAAAGGAAUUGACGGGGGCUCACACAAGCGGUGGAGCAUGUGGCUUAAUUCGAGGCAACGCGAAGAACCUUACCUGGAUUUGACAUGUUGUAUAGCUCUGUGAAAGCAGAGUGACGCUUCGGUGGAACUUGCACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUCGCGUUAAGUCGCUGAACGAGCGAAACCCCUAUCCUUAGUUGCCAGCACUCAUGGUGGGGACUCUAAGGAGACUGCCGGUGUCAAACCGGAGGAAGGUGGGGACGACGUCAAGUCAUCAUGGCCUUUAUGUCCAGGGCUGCACACGUGCUACAAUGCGGCGUACAAAGGGAAGCAAAAUCGCGAGAUCAAGCAAAUCCCAAAAAGCNUCGCUCAGUUCGGAUUGCAGGCUGCAACUCGCCUGCAUGAAGUUGGAAUCGCUAGUAAUCGCAGGUCAGCUUACUGCGGUGAAUAUGAACCUGAGCCUUGUACACACCGCCCGUCAAGCCACGAAAGCGGGGGGCGUCCAAAGUmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ch.Zea.may  CUCAUGGAGAGUUCGAUCCUGGCUCAGGAUGAACGCUGGCGGCAUGCUUAACACAUGCAAGUCGAACGGGAAGU--------GGU---------GUUUCCAGUGGCGAACGGGUGAGUAACGCGUAAGAACCUGCCCUUGGGAGGGGAACAACAACUGGAAACGGUUGCUAAUACCCCGUA-GGCUGAGGAGCAAAAGGA-------GAAA-----UCCGCCCAAGGAGGGGCUCGCGUCUGAUUAGCUAGUUGGUGAGGCAAUAGCUUACCAAGGCGAUGAUCAGUAGCUGGUCCGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUC-GGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGAAAGCCUGACGGAGCAAUGCCGCGUGGAGGUGGAAGGCCUACGGGUCGUCAACUUCUUUUCUCGGAGAAGAA-----------ACAA------------UGACGGUAUCUGAGGAAUAAGCAUCGGCUAACUCUGUGCCAGCAGCCGCGGUAAGACAGAGGAUGCAAGCGUUAUCCGGAAUGAUUGGGCGUAAAGCGUCUGUAGGUGGCUUUUCAAGUCCGCCGUCAAAUCCCAGGGCUCAACCCUGGACAGGCGGUGGAAACUACCAAGCUGGAGUACGGUAGGGGCAGAGGGAAUUUCCGGUGGAGCGGUGAAAUGCAUUGAGAUCGGAAAGAACACCAACGGCGAAAGCACUCUGCUGGGCCGACACUGACACUGAGAGACGAAAGCUAGGGGAGCAAAUGGGAUUAGAGACCCCAGUAGUCCUAGCCGUAAACGAUGGAUACUAGGUGCUGUGCGACUCGCCCGUGCAGUGCUGUAGCUAACGCGUUAAGUAUCCCGCCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAAGGCGAAGAACCUUACCAGGGCUUGACAUGCCGCGAAUCCUCUUGAAAGAGAGGGGUGCCCUCGGGAACGCGGACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGCCGUAAGGUGUUGGGUUAAGUCUCGCAACGAGCGCAACCCUCGUGUUUAGUUGCCACUA-UGAG-UUUGGAACCCUGAACAGACCGCCGGUGUUAAGCCGGAGGAAGGAGAGGAUGAGGCCAAGUCAUCAUGCCCCUUAUGCCCUGGGCGACACACGUGCUACAAUGGGCGGGACAAAGGGUCGCGAUCUCGCGAGGGUGAGCUAACUCCAAAAACCCGUCCUCAGUUCGGAUUGCAGGCUGCAACUCGCCUGCAUGAAGCAGGAAUCGCUAGUAAUCGCCGGUCAGCCUACGGCGGCGAAUCCGUUCCCGGGCCUUGUACACACCGCCCGUCACACUAUAGGAGCUGGCCAGGUUUGAAGUCAUUAC-CUAACCGUAAGGAGGGGGAUGCCUAAGGCUAGGCUUGCGACUGGAGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGCGGCUGGAUCACCUCCUUU
ch.Dau.car  CUCAUGGAGAGUUCGAUCCUGGCUCAGGAUGAACGCUGGCGGCAUGGAAAACACAUGCAAGUCGGACGGGAAGU--------GGU---------GUUUCCAGUGGCGGACGGGACUGUAACGCGUAAGAACCUGCCCUUGGGUGGGGAACAACAGCUGGAAACGGCUGCUAAUACCCCGUA-GGCUGAGGAGCAAAAGGA-------GGAA-----UCCGCCCGAGGGAGGGCUCGCGUCUGAU-AGCUAGUUGGUGAGGCAAUAGCUUACCAAGGCGAUGAUCAGUAGCUGGUCCGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUC-GGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGAAAGC--GACGGAGCAAUGCCGCGUGGAGGUAGAAGGCCCACGGGUCGUGAACUUCUUUUCCCGGAGAAGAA-----------GCAA------------UGACGGUAUCUGGGGAAUAAGCAUCGGCUAACUCUGUGCCAGCAGCCGCGGUAAUACAGAGGAUGCAAGCGUUAUCCGGAAUGAUUGGGCGUAAAGCGUCUGUAGGUGGCUUUUUAAGUCCGCCGUCAAAUCCCAGGGCUCAACCCUGGACAGGCGGUGGAAACUACCAAGCUGGAGUACGGUAGGGGCAGAGGGAAUUUCCGGUGGAGCGGUGAAAUGCGUAGAGAUCGGAAAGAACACCAACGGCGAAAGCACUCUGCUGGGCCGACAUUGACACUGAGAGACGAAAGCUAGGGGAGCGAAUGGGAUUAAAUACCCCAUUAGUCCUAGCCGUAAACGAUGGAUACUAGGCGCUGUGCG-AUCGCCCGUGCAGUGCUGUAACUACCGCGUUAAGUAUCCCGCCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGGCGCGCACAAUCGGUGGAGCAUGUGGUUUAAUUCGAUGCAAAGCGAAGAACCUUACCAUGGCUUGACAUGCCGCGAAUCCUCUUGAAAGAGAGGGGUGCCUUCGGGAACGCGGACACAGGUGGUGAAUGGCUGUCGUCAGCUCGUGCCGUAAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUGUUUAGUUGCCAUCG-UUGAGUUUGGAACCCUGAACAGACUGCCGGUGAUAAGCCGGAGGAAGGUGAGGAUGACGUCAAGUCAUCAUGCCCCUUAUGCCCUUGGCGACACUCGUGCUACAAUGGCCGGGACAAAGGGUUGCGAUCCCGCGAGGGUGAGCUAACCCCAAAAACCCGUCCUCAGUUGGGAUUGCAGGCUGCAACUCGCCUGCAUGAAGCCGGAAUCGCUAGUAAUCGCCGGUCAGCCUACGGCGGUGAAUUCGGUACCGGGCCUUGUACACACCGCCCGUCACACUAUGGGAGCUGGCCAUGCCCGAAGUCGUUAC-CUAACCGCAAGG-GGGGGAUGCCGAA-GCAGGGCUAGUGACUGGAGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGCGGCUGGAUCACCUCCUmm
ch.Gly.max  CUCAUGGAGAGUUCGAUCCUGGCUCAGGAUGAACGCUGGCGGCAUGCCUUACACAUGCAAGUCGGACGGGAAGU--------GGU---------GUUUCCAGUGGCGGACGGGUGAGUAACGCGUAAGAACCUACCCUUGGGAGGGGAACAACAGCUGGAAACGGCUGCUAAUACCCCGUA-GGCUGAGGAGCAAAAGGA--------GAA-----UCCGCCCGAGGAGGGGCUCGCGUCUGAUUAGCUAGUUGGUGAGGC-AUAGCUUACCAAGGCGAUGAUCAGUAGCUGGUCCGAGA-GAUGAUCAGCCACACUGGGACUGAGACGAGGCCCAGACUCUUC-GGGAGGCAGCAGUGGGGAAUUUUCCGCAAUG-GCGAAAGC-UGACGGAGCAAUGCCGCGUGAAGGUAGAAGGCCUAC-GGUCAUGAACUUCUUUUCCCGGAGAAGAA-----------GCAA------------UGACGGUAUCCGGGGAAUAAGCAUCGGCUAACUCUGUGCCAGCGGCCGCGGUAAGACAGAGGAUGCAAGCGUUAUCCGGAAUGAUUGGGCG-AAAG-GUCUGUAGGUGGCUUUUUAAGUUCGCCGUCAAAUCCCAGGGCUCAACCCUGGACAGGCGGUGGAAACUACCAAGCUGGAGUACGGUAGGGGCAGAGGGAAUUUCCGGUGGAGCGGUGAAAUGCGUAGAGAUCGGAAAGAACACCAACGGCGAAAGCACUCUGCUGGGCCGACACUGACACUGAGAGACGAAACUUAGGGGAGCGAAUGGGAUUAGAUACCCCAGUAGUCCUAGCCGUAAACGAUGGAUACUAGGCGCUGUGCGUAUCGCCCGUGCAAUGCUGUA-CUAACGCGUUAAGUAUCCCGCCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGG-CCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAAAGCGAAGAACCUUACCAGGGCUUGACAUGCCGCGAAUCCUCUUGAAAGAGAGGGGUGC-UUCGGGAACGCGGACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGCCGUAAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUGUUUAGUUGCCAACA-UUUAGUUUGGAACCCUGAGCAGACUGCCGGUGAUAAGCCGGAGGAAGGUGAGGAUGACGUCGAGUCAUCAUGCCCCUUAUGCCCUGGGCGACACACGUGCUACAAU-----GGACAAAGGAUCGCGAUCCCGCGA-GGUGAGCUAACUCCAAAAACCCGUCCUCAGUUCGGAUUGUAGGCUGCAACUCGCCUGCAUGAAGCCGGAAUCGCUAGUAAUCGCCGGUCAGCCUACGGCGGUGAAUUCGUUCCCGGGCCUUGUACACACCGCCCGUCACACUAUGGGAGCUGGCCAUGC-CGAAGUCGUUAC-CUAACCGCAA-GAGGGGGAUGCCGAA-G--GGGCUAGUGACUGGAGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGCGGCUGGAUCACCUCCUUm
ch.Pis.sat  CUCAUGGAGAGUUUGAUCCUGGCUCAGGAUGAACGCUGGCGGCAUGCUUUACACAUGCAAGUCGGACGGGAAGU--------GGU---------GUUUCCAGUGGCGAACGGGUGAGUAACGCGUAAGAACCUGCCCUUGGGAGGGGGACAACAGCUGGAAACGGCUGCUAAUACCCCGUA-GGCUGAGGAGCGAAAGGA-------GGAA-----UCCGCCCAAGGAGGGGCUCGCGUCUGAUUAGCUAGUUGGUGAGGUAAUAGCUUACCAAGGCGAUGAUCAGUAGCUGGUCCGAGAGGAUGAUCAGCCACACUGGGACUGAGACAAGGUCCAGACUCCUC-GGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGAAAGCCUGACGGAGCAAUGCCGCGUGGAGGUAGA-GGCCCCUGGGUCAUGAACUUCUUUUCCCGGAGAAGAA-----------AAAA------------UGACGGUAUCCGGGGAAUAAGCAUCGGCUAACUCUGUGCCAGCAGCCGCGGUAAGACAGAGGAUGCAAGCGUUAUCCGGAAUGAUUGGGCGUAAAGCGUCUGUAGGUGGCUUUUUAAGUUCGCUGUCAAAUACCAGGGCUCAACCCUGGACAGGUGGUGAAAACUACUAAGCUAGAGUACGGUAGGGGCAGAGGGAAUUUCCGGUGGAGCGAUGAAAUGCGUAGAGAUCGGAAGGAACACCAACGGCGAAAGAACUCUGCUGGGCCGACACUGACACUGAGAGACGAAAGCUAGGGGAGCGAAUGGGAUUAGAGACCCCAGUAGUCCUAGCCGUAAACGAUGGAUACUAAGUGCUGUGCGAAUCGCCCGUGCAACGCUGUA-CUAACGCGUUAAGUAUCCCGCCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGG-CCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAAAGCGAAGAACCUUACCAGGGCUUGACAUGC-GCGAAUCCUCUUGAAAGAGAGGAGUGCCUUCGGGAAUGC-GACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUGUUUAGUUGCCAACG-UUUAGUUUGGAACUCUGAACAGACUGCCGGUGAUAAGCCGGAGGAAGGUGAGGAUGACGUCAAGUCAUCAUGCCCCUUAUGCCCUGGGCUACACAC-UGCUACAAUGGACCGGACAAAGGAUCGCGACCCCGCGAGGGUGAGCUAACUUCAAAAACCUGUCCUCAGUUCGGAUUGUAGGCUGCAACUCGCCUACAUGAAGCCGGAAUCGCUAGUAAUCGCCGGUCAGCCUACGGCGGUGAAUUCGUUCCCGGGCCUUGUACACACCGCCCGUCACACUAUGGGAGCUGGCCAUGCCCGAAGUCAUUAC-CUAACCGCAAGGAGGGGGAUGCCGAAGGCAGGGCUAGUGACUGGAGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGCGACUGGAUCACCUCCUUU
ch.Nic.tab  CUCAUGGAGAGUUCGAUCCUGGCUCAGGAUGAACGCUGGCGGCAUGCUUAACACAUGCAAGUCGGACGGGAAGU--------GGU---------GUUUCCAGUGGCGGACGGGUGAGUAACGCGUAAGAACCUGCCCUUGGGAGGGGAACAACAGCUGGAAACGGCUGCUAAUACCCCGUA-GGCUGAGGAGCAAAAGGA-------GGAA-----UCCGCCCGAGGAGGGGCUCGCGUCUGAUUAGCUAGUUGGUGAGGCAAUAGCUUACCAAGGCGAUGAUCAGUAGCUGGUCCGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUC-GGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGAAAGC--GACGGAGCAAUGCCGCGUGGAGGUAGAAGGCCCACGGGUCGUGAACUUCUUUUCCCGGAGAAGAA-----------GCAA------------UGACGGUAUCUGGGGAAUAAGCAUCGGCUAACUCUGUGCCAGCAGCCGCGGUAAUACAGAGGAUGCAAGCGUUAUCCGGAAUGAUUGGGCGUAAAGCGUCUGUAGGUGGCUUUUUAAGUCCGCCGUCAAAUCCCAGGGCUCAACCCUGGACAGGCGGUGGAAACUACCAAGCUGGAGUACGGUAGGGGCAGAGGGAAUUUCCGGUGGAGCGGUGAAAUGCGUAGAGAUCGGAAAGAACACCAACGGCGAAAGCACUCUGCUGGGCCGACACUGACACUGAGAGACGAAAGCUAGGGGAGCGAAUGGGAUUAGAUACCCCAGUAGUCCUAGCCGUAAACGAUGGAUACUAGGCGCUGUGCG-AUCGCCCGUGCAGUGCUGUAGCUAACGCGUUAAGUAUCCCGCCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGGCGCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAAAGCGAAGAACCUUACCAUGGCUUGACAUGCCGCGAAUCCUCUUGAAAGAGAGGGGUGCCUUCGGGAACGCGGACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGCCGUAAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUGUUUAGUUGCCAUCG-UUGAGUUUGGAACCCUGAACAGACUGCCGGUGUUAAGCCGGAGGAAGGUGAGGAUGACGUCAAGUCAUCAUGCCCCUUAUGCCCUUGGCGACACACGUGCUACAAUGGCCGGGACAAAGGGUCGCGAUCCCGCGAGGGUGAGCUAACCCCAAAAACCCGUCCUCAGUUCGGAUUGCAGGCUGCAACUCGCCUGCAUGAAGCCGGAAUCGCUAGUAAUCGCCGGUCAGCCUACGGCGGUGAAUUCGUUCCCGGGCCUUGUACACACCGCCCGUCACACUAUGGGAGCUGGCCAUGCCCGAAGUCGUUAC-CUAACCGCAAGG-GGGGGAUGCCGAA-GCGGGGCUAGUGACUGGAGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGCGGCUGGAUCACCUCCUUU
ch.Pin.thu  CUCAUGGAGAGUUUGAUCCUGGCUCAGGAUGAACGCUGGCGGCAUGCUUAACACAUGCAAGUCGGACGGGAAGU--------GGU---------GUUUCCAGUGGCGGACGGGUGAGUAACGCGUAAGAACCUGCCCUUGGGAGGGGAACAACAGCUGGAAACGGCUGCUAAUACCCCAUA-GGCUGAGGAGCAAAAGGA-------GGAA-----UCCGCCCAAGGAGGGGCUCGCGUCUGAUUAGUUAGUUGGUGAGGCAAUGGCUUACCAAGGCGACGAUCAGUAGCUGGUCCGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUC-GGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGAAAGCCUGACGGAGCAAUGCCGCGUGAAGGCAGAAGGCCCACGGGUCAUGAACUUCUUUUCUCGGAGAAGAA-----------AAAA------------UGACGGUAUCUGAGGAAUAAGCAUCGGCUAACUCUGUGCCAGCAGCCGCGGUAAGACAGAGGAUGCAAGCGUUAUCCGGAAUGAUUGGGCGUAAAGCGUCUGUAGGUGGCUUUUCAAGUCCGCCGUCAAAUCCCAGGGCUCAACCCUGGACAGGCGGUGGAAACUACCAAGCUGGAGUACGGUAGGGGCAGAGGGAAUUUCCGGUGGAGCGGUGAAAUGCGUUGAGAUCGGAAAGAACACCAACGGCGAAAGCACUCUGCUGGGCCGACACUGACACUGAGAGACGAAAGCUAGGGGAGCAAAUGGGAUUAGAUACCCCAGUAGUCCUAGCCGUAAACGAUGGAUACUAAGUGCUGUGCGUAUCGCCCGCGCAGUGCUGUAGCUAACGCGUUAAGUAUCCCGCCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUCAAUUCGAUGCAAAGCGAAGAACCUUACCAGGGCUUGACAUGCCGUGAAUCCUCCCGAAAGAGAGGAGUGCCUUCGGGAACGCGGACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGCCGUAAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUGUUUAGUUGCCAGCA-UUUAGUUUGGAACCCUGAACAGACUGCCGGUGAUAAGCCGGAGGAAGGUGAGGAUGACGUCAAGUCAUCAUGCCCCUUACGCCCUGGGCGACACACGUGCUACAAUGACCGGGACAAAGGGUCGCGACCCCGCGAGGGCAAGCUAACCUCAAAAACCCGGCCUCAGUUCGGAUUGCAGGCUGCAACUCGCCUGCAUGAAGCCGGAAUCGCUAGUAAUCGCCGGUCAGCCUACGGCGGUGAAUCCGUUCCCGGGCCUUGUACACACCGCCCGUCACACUAUGGGAGCUGGCCAUGCCCCAAGUCGUUAC-CUAACCGCAAGGAGGGGGAUGCCGAAGGCUGGGCUAGUGACUGGAGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGCGGCUGGAUCACCUCCUUU
ch.Chl.mir  CCUGCAGAGAGUUYGAUCCUGGCUCAGGAUGAACGCUGGCGGCAUGCUUAACACAUGCAAGUCGAACGGAGAUUUUAUGUUUCUUGAAACAUAAAAUCUUAGUGGCGGACGGGUGAGGAACGCGUAAGAACCUACCUUUAGGGGAGGGACAACAACUGGAAACGGUUGCUAAUACCUCAUA-UGCUGAGGAGUAAAAGGGUUAUAAAUACGAUUAUUCCGCCUAAAGAUGGGCUUGCGUCUGAUUAGCUUGUUGGUGGGGUAAUUGCUUACCAAGGCGACGAUCAGUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUC-GGGAGGCAGCAGURAGGAAUUUUCCGCAAUGGGCGAAAGCCUGACGGAGCAAUGCCGCGUGGAGGAUGACAGCCUGUGGGUCGUAAACUCCUUUUCUCAGAAAAGAA-----------GAUC------------UGACGGUAUCUGAGGAAUAAGCAUCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGAUGCAAGCGUUAUCCGGAAUGAUUGGGCGUAAAGCGUCUGUAGGUGGUUUAUUAAGUCUACUGUUAAAGAUCAGGGCUUAACCCUGAGCCGGCAGUGGAAACUAAUAAGCUUGAGUACGGUAGGGGCAGAGGGAAUUCCCGGUGUAGCGGUGAAAUGCGUAGAGAUCGGGAAGAACACCAAUGGCGAAAGCACUCUGCUGGGCCGAAACUGACACUGAGAGACGAAAGCUAGGGGAGCGAAUGGGAUUAGAUACCCCAGUAGUCCUAGCCGUAAACGAUGGAUACUAAGUGCUGUGCGACUCAACCGUCCAGUACUGUAGCUAACGCGUGAAGUAUCCCGCCUGGGGAGUAUGCUCGCAAGGGUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGYUUAAUUCGAUGCAACGCGAAGAACCUUACCAGGGCUUGACAUGCCAUUUAGUCUUUUGAAAAAAAGACAUAUUUUUAGAUUGGUGGACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCUUUACUUGCCAUUACUAGCUGUUGGGAAAUUAGAGAGACUGCCGGUGAGAAGCCGGRGGAAGGUGAGGAUGACGUCAAGUCAGCAUGCCCCUUAUGCCCUGGGCGACACACGUGCUACAAUGGCCGGGACAAUGAGAUGCAACCUCGCGAGAGCAAGCUAACCUCAAAAACCCGGUCUCAGUUCGGAUUGCAGGCUGCAACUCGCCUGCAUGAAGUCGGAAUCGCUAGUAAUCGCUGGUCAGCCUACAGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGCUGGCUAUGCCCAAAGUCGUUAC-CCAACCGUAAGGAGGGGGAUGCCUAAGGCAGAGCUAGUGACUGGGGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGNGGNUGGAUCACCUCCUUG
ch.Eug.gra  GAAAUGACGAGUUUGAUCCUUGCUCAGGGUGAACGCUGGCGGUAUGCUUAACACAUGCAAGUUGAACGAAAUUACUA-----GCAA-----UAGUAAUUUAGUGGCGGACGGGUGAGUAAUAUGUAAGAAUCUGCGCUUGGGUGAGGAAUAACAGAUGGAAACGUUUGCUAAUGCCUCAUAUUACUGUGAAGUUAAAGA--------GAAU-----UUCGCCUAGGCAUGAGCUUGCAUCUGAUUAGCUUGUUGGUGAGGUAAAGGCUUACCAAGGCGACGAUCAGUAGCUGAUUUGAGAGGAUGAUCAGCCACACUGGGAUUGAGA-ACGGAACAGACUUUUC-GGAAGGCAGCAGUGAGGAAUUUUCCGCAAUGGGCGCAAGCCUGACGGAGCAAUACCGCGUGAAGGAAGAAGGCCUUUGGGUUGUAAACUUCUUUUCUCAAAGAAGAA-----------GAAA------------UGACGGUAUUUGAGGAAUAAGCAUCGGCUAAUUCCGUGCCAGCAGCCGCGGUAAUACGGGAGAUGCGAGCGUUAUCCGGAAUUAUUGGGCGUAAAGAGUUUGUAGGCGGUCAAGUGUGUUUAAUGUUAAAAGUCAAAGCUUAACUUUGGAAGGGCAUUAAAAACUGCUAGACUUGAGUAUGGUAGGGGUGAAGGGAAUUUCCAGUGUAGCGGUGAAAUGCGUAGAGAUUGGAAAGAACACCAAUGGCGAAGGCACUUUUCUAGGCCAAUACUGACGCUGAGAAACGAAAGCUGAGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCUUGGCCGUAAACUAUGGAUACUAAGUGGUGCUG--AAAG----UGCACUGCUGUAGUUAACACGUUAAGUAUCCCGCCUGGGGAGUACGCUUGCACAAGUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAACACGAAGAACCUUACCAGGAUUUGACAGGCUAGGAGGAAGUUUGAAAGAACGCAGUACCUUCGGGUAUCUAGACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUUUUUUUAAUUAAC-----GCUU-----GUCAUUUAGAAAUACUGCUGGUUAUU-ACCAGAGGAAGGUGAGGACGACGUCAAGUCAUCAUGCCCCUUAUAUCCUGGGCUACACACGUGCUACAAUGGUUAAGACAAUAAGUUGCAAUUUUGUGAAAAUGAGCUAAUCUU-AAAACUUAGCCUAAGUUCGGAUUGUAGGCUGAAACUCGCCUACAUGAAGCCGGAAUCGCUAGUAAUCGCCGGUCAGCUUACGGCGGUGAAUACGUUCUCGGGCCUUGUACACACCGCCCGUCACACCAUGGAAGUUGGCUGUGCCCGAAGUUAUUAU-CUGCCUGAAAAGAGGGAAAUACCUAAGGCCUGGCUGGUGACUGGGGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGUGGCUGGAACAAUUCCCmm
ch.Cya.par  ACCAUGGAGAGUUUGAUCCUGGCUCAGGAUGAACGCUGGCGGUAUGCUUAACACAUGCAAGUCGAACGAAGAUC--------GCAA--------GAUCUUAGUGGCGGACGGGUGAGUAACGCGUGAGAAUCUACCCUUAGGAGGGGGACAACAGUUGGAAACGACUGCUAAUACCCCAUA-UGCCUUCGGGUGAAAAGA-------GUAA-----UCUGCCUGAGGAAGAGCUCGCGUCUGAUUAGCUAGUUGGUGGGGUAAAGGCCUACCAAGGCGACGAUCAGUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGGACUGAGAUACGGCCCAGACUCCUC-GGGAGGCAGCAGUGGGGAAUUUUCCGCAAUGGGCGAAAGCCUGACGGAGCAAUACCGCGUGAGGGAAGACGGCCUGUGGGUUGUAAACCUCUUUUCUUAGGGAAGAA-----------UCAA------------UGACGGUACCUAAGGAAUAAGCAUCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGAUGCAAGCGUUAUCCGGAAUCAUUGGGCGUAAAGAGUUCGUAGGUGGCUAAGCAAGUCUGUUGUUAAAGGCUGGGGCUUAACCCCAAAAAGGCAAUGGAAACUGUUUGGCUUGAGUACGGUAGGGGCAGAGGGAAUUCCUGGUGUAGCGGUGAAAUGCGUAGAUAUCAGGAAGAACACCGAUGGCGAAAGCACUCUGCUGGGCCGUUACUGACACUGAGGAACGAAAGCUAGGGUAGCAAAUGGGAUUAGAUACCCCAGUAGUCCUAGCCGUAAACUAUGGAUACUAGGUGUUGUGCGUAUCGCCCGUACAGUACCGUAGCUAACGCGUUAAGUAUCCCGCCUGGGGAGUACGCUCGCAAGAGUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGUAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCAGGGUUUGACAUGUCGCGAAUUUUCUUGAAAGAGAAAAGUGCCUUCGGGAACGCGAACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUUUUUAGUUGCCAUCA-UCAG-UUGGGCACUCUAAAGAGACUGCCGGUGACAAGCCGGAGGAAGGUGGGGAUGACGUCAAGUCAGCAUGCCCCUUAUACCCUGGGCUACACACGUACUACAAUGGUCGGGACAAUAGGUUGCCAACUUGCGAAAGUGAGCUAAUCCGUUAAACCCGGCCUCAGUUCAGAUUGCAGGCUGCAACUCGCCUGCAUGAAGGUGGAAUCGCUAGUAAUCGCCGGUCAGCUUACGGCGGUGAAUUCGUUCCCGGGCCUUGUACACACCGCCCGUCACACCACGGGAGUCGGCCAUGCCCGAAGUCGUUAC-CUAACCAUUUGGAGGGGGAUGCCUAAGGCAGGGCUGGUGACUGGGGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGCGGCUGGAUCACCUCCUUU
ch.Cya.cal  AUAAUGGAGAGUUUGAUCCUGGCUCAGGAUUAACGCUGGCGGUAUGCCUAACACAUGCAAGUCGUACGAGAAU----------UUU---------AUUCUAGUGGCGGACGGGUGAGUAACACGUGAGAAUCUACCUCUAGGAGGGGGAUAACAGUUGGAAACGAUUGCUAAAACCCCAUA-UGCCUUAUGGUGAAAAGA-------UUUA-----UCUGCCUGGAGAUGAGCUCGCGGCUGAUUAGCUAGUUGGUAGGGUAAUGGCUUACCAAGGCAACGAUCAGUAGCUGGUCUUAGAGGAUGAUCAGCCACACUGGAACUGAGAUACGGUCCAGACUCCUC-GGGAGGCAGCAGUGGGGAAUUUUCCACAAUGGGCGAAAGCCUGAUGGAGCAAUACCGUGUGAGGGAUGAAGGCCUGUGGGUUGUAAACCUCUUUUUUCAGGAAAGAA-----------ACUU------------UGACGGUACCUGAAGAAUAAGCAUCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGAUGCAAGCGUUAUCCGGAAUCACUGGGCGUAAAGCGUCUGUAGGUGGUUUAUCAAGUCUGCUGUUAAAGCUUGAGGCUUAACCUCAAAAAAGCAGUGGAAACUGAUAGACUAGAGAAUGGUAGGGGCAGAGAGAAUUCUCAGUGUAGCGGUGAAAUGCGUAGAUAUUGAGAAGAAUACCGAUAGCGAAGGCGCUCUGCUGGGCCAUUACUGACACUCAGAGACGAAAGCUAGGGGAGCAAAUGGGAUUAGAUACCCCAGUAGUCCUAGCCGUAAACUAUGGAUACUAGAUGUUGUGUGAGUAAAUUGUGCAGUAUCGAAGCUAACGCGUUAAGUAUCCCGCCUGGGAAGUACGCUCGCAAGAGUGAAACUCAAAGGAAUUGACGGGGACCCGCACAAGC-GUGGAGCAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCAGGACUUGACAUGUUACUAAUUUCCUUGAAAGAGGAAAGUGCCUUUGGGAAAGUAAACACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCUUUAGUUACCAUCA-UUAAGUUGGGGACUCUAAAGAGACUGCCGGUGAUAAACCGGAGGAAGGUAAGGAUGAGGUCAAGUCAUCAUGCCCCUUAUGUCCUGGGCUACACACGUGCUACAAUGGUUAGGACAAUAAGUCGCAAAUUCGUGAGAACUAGCUAAUCUUAUAAACCUAAUCUCAGUACGGAUUGUAGGCUGCAACUCGCCUACAUGAAGACGGAAUCGCUAGUAAUCGCUGGUCAGCUCACAGCGGUGAAUACGUUCCCGGGUCUUGUACACACCGCCCGUCACACCAUGGGAGCUGGCCAUGUCCGAAAUCAUUAC-UUAACUUAAUGGAGGAGGAUGCUUAAGGCAGGGCUAGUGACUGGGGUGAAGUCGUAACAAGGUAGCCGUACUGGAAGGUGCGGCUGGAUUACCUCCUUU
ga.Esc.col  AAAUUGAAGAGUUUGAUCAUGGCUCAGAUUGAACGCUGGCGGCAGGCCUAACACAUGCAAGUCGAACGGUAACAGGAAGAAGCUUGCUUCUUUGCUGACGAGUGGCGGACGGGUGAGUAAUGUCUGGGAAACUGCCUGAUGGAGGGGGAUAACUACUGGAAACGGUAGCUAAUACCGCAUAACGUCGCAAGACCAAAGAGGGGGACCUUCGGGCCUCUUGCCAUCGGAUGUGCCCAGAUGGGAUUAGCUAGUAGGUGGGGUAACGGCUCACCUAGGCGACGAUCCCUAGCUGGUCUGAGAGGAUGACCAGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGCGCAAGCCUGAUGCAGCCAUGCCGCGUGUAUGAAGAAGGCCUUCGGGUUGUAAAGUACUUUCAGCGGGGAGGAAGGGAGUAAAGUUAAUACCUUUGCUCAUUGACGUUACCCGCAGAAGAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGCACGCAGGCGGUUUGUUAAGUCAGAUGUGAAAUCCCCGGGCUCAACCUGGGAACUGCAUCUGAUACUGGCAAGCUUGAGUCUCGUAGAGGGGGGUAGAAUUCCAGGUGUAGCGGUGAAAUGCGUAGAGAUCUGGAGGAAUACCGGUGGCGAAGGCGGCCCCCUGGACGAAGACUGACGCUCAGGUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGUCGACUUGGAGGUUGUGCCCUUGAGGCGUGGCUUCCGGAGCUAACGCGUUAAGUCGACCGCCUGGGGAGUACGGCCGCAAGGUUAAAACUCAAAUGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCUGGUCUUGACAUCCACGGAAGUUUUCAGAGAUGAGAAUGUGCCUUCGGGAACCGUGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUUGUGAAAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUAUCCUUUGUUGCCAGCGGUCCGGCCGGGAACUCAAAGGAGACUGCCAGUGAUAAACUGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGGCCCUUACGACCAGGGCUACACACGUGCUACAAUGGCGCAUACAAAGAGAAGCGACCUCGCGAGAGCAAGCGGACCUCAUAAAGUGCGUCGUAGUCCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUCGGAAUCGCUAGUAAUCGUGGAUCAGAAUGCCACGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGUUGCAAAAGAAGUAGGUAGCUUAACCUUCGGGAGGGCGCUUACCACUUUGUGAUUCAUGACUGGGGUGAAGUCGUAACAAGGUAACCGUAGGGGAACCUGCGGUUGGAUCACCUCCUUA
ga.Hae.par  --ACUGAAGAGUUUGAUCNUGGCUCAGAUUGAACGCUGGCGGCAGGCUUAACACAUGCAAGUCGAACGGUAACGGGUUGAAACUUGUUUCAAUGCUGACGAGUGGCGGACGGGUGAGUAAUGCUUGGGAAUCUGGCUUAUGGAGGGGGAUAACCAUUGGAAACGAUGGCUAAUACCGCAUAGAAUCGGAAGAUUAAAGGGUGGGACUUUUUAGCCACCUGCCAUAAGAUGAGCCCAAGUGGGAUUAGGUNGUUGGUGAGGUNAAGGCUCACCAAGCCUNCGAUCUCUAGCUNGUCUNAGAGGAUGGCCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCGCNAUGGGGGGAACCCUGACGCAGCNAUGCCGCGUGAAUGAAGAAGGCCUUCGGGUUGUAAAGUUCUUUCGGUGGUGAGGAAGGUUNGUGUGUUAAUAGCACACUAAUUUGACGUUAGCCACAGAAGAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUNAUACGGAGGGUGCGAGCGUUNAUCGGAAUAACUGGGCUUAAAGGGCACGCAGGCGGUAAAUUAAGUGAGAUGUGAAAUCCCCGAGCUUAACUUAGGAAUUGCAUUUCAGACUNGUUUACUAGAGUACUUUAGGGAGGGNUAGAAUUCCACGUGUAGCGGUGAAAUGCGUAGAGAUGUGGAGGAAUACCGAAGGCGAAGGCAGCCCCUUGGGAAGCUACUGACGCUCAUGUGCNNAAGCGUGGGGAGCAAACAGGAUUNGAUACCCUGGUAGUCCACGCUGUAAACGCUGUCGAUUUGGGGAUUGGGC-UUNN-GGCUUGGUGCCCGUAGCUAACGUGAUAAAUCGACCGCCUNGGGAGUACGGCCGCAAGGUUAAAACUCAAAUNAAUUGACGGGGGCCCGCACNAGCGGUGGAGCAUGUGGUUUNAUUCGANNNAACGCGAAGAACCUUACCUACUCUUGACAUCCUAAGAAUCCUGUAGAGAUACGGGAGUGCCUUCGGGAGCUUAGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUUGUGAAAUGUUGGGUUNAGUCCCGCAACGAGCGCAACCCUUAUCCUUUGUUGCCAGCACUUCGGGUGGGAACUCAAAGGAGACUGCCAGUGAUNAACUGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGGCCCUUACGAGUAGGGCUACACACGUGCUACAAUGGUGCAUACAGAGGGAAGCGAGCCUGCGAGGGGGAGCGAAUCUCAGAAAGUGCAUCUAAGUCCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUCGGAAUCGCUAGUAAUCGCAAAUCAGAAUGUUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGUUGUACCAGAAGUAGAUAGCUUAACCUUCGGGAGGGCGUUUACCACGGUAUGAUUCAUGACUmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ga.Leg.lyt  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmACGCUGGCGGCAUGCUUAACACAUGCAAGUCGAACGGCAGCACAGUCUAGCUUGCUAGACGGGUGGCGAGUGGCGAACGGGUGAGUAACGCGUAGGAAUAUACCUUGAAGAGGGGGACAACUUGGGGAAACUCAAGCUAAUACCGCAUAAUGUCUGAGGACGAAAGCCGGGGACCGUAAGGCCUGGCGCUUUAAGAUUAGCCUGCGUCCGAUUAGCUAGUUGGUAGGGAAAGGGCCUACCAAGGCGACGAUCGGUAGCUGGUCUGAGAGGAUGACCAGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGGGGAACCCUGAUCCAGCAAUGCCGCGUGUGUGAAGAAGGCCUGAGGGUUGUAAAGCACUUUCAGUGGGGAGGAGGCUUGUUAGGUUAAGAGCUAAAUAAGUGGACGUUACCCACAGAAGAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUAAUCGGAAUUACUGGGCGUAAAGAGUGCGUAGGUGGUUUGGUAAGUUAUCUGUGAAAUCCCUGGGCUCAACCUGGGCAGGUCAGAUAAUACUGCUGAACUCGAGUAUGGGAGAGGGUAGUGGAAUUUCCGGUGUAGCGGUGAAAUGCGUAGAGAUCGGAAGGAACACCAGUGGCGAAGGCGGCUACCUGGCCUAAUACUGACACUGAGGCACGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGUCAACUAGCUGUUGGUUAUAUGAUAUAAUUAGUGGCGCAGCAAACGCGAUAAGUUGACCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCUACCCUUGACAUACAGUGAACUUUGCAGAGAUGCAUUGGUGCCUUCGGGAACACUGAUACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGUAACGAGCGCAACCCUUAUUGUUCGUUGCCAGCGCUAAUGGCGGGAACUCUAAUGAGACUGCCGGUGACAAACCGGAGGAAGGCGGGGACGACGUCAAGUCAUCAUGGCC-UUACGGGUAGGGCUAGACACGUGCUACAAUGGUUGAUACAGAGGGAAGCGAAGGAGCGAUCUGGAGC-AAUCUUAGAAAGUCAAUCGUAGUCCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUCGGAAUCGCUAGUAAUCGCGAAUCAGCAUGUCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGCUG-ACCAGAAGUAGAUAGUCUAACCGCAAGGGGGACGUUUAmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ga.Mar.psy  -----mmAGAGUUUGAUCAUGGCUCAGAUUGAACGCUGGCGGCAGGCUUAACACAUGCAAGUCGAACGGUAACAGGAAAUAGCUUGCUAUUUUGCUGACGAGUGGCGGACGGGUGAGUAAUGCUUGGGAAUUUGCCGAAAGGUGGGGGACAACAGUUGGAAACGACUGCUAAUACCGCAUAAUGUCUACGGACCAAAGGUGGCCUCUUUUAAUGCUAUCGCCUUUCGAUGAGCCCAAGUGGGAUUAGCUAGUUGGUAAGGUAAUGGCUUACCAAGGCUUCGAUCCCUAGCUGGUCUUAGAGGAUGACCAGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGAGGAAACUCUGAUGCAGCCAUGCCGCGUGUGUGAAGAAGGCUUUCGGGUUGUAAAGCACUUUCAGCGAGGAGGAAAGGGUGUUGGUUAAUAUCCAAUAUCUGUGACGUUACUCGCAGAAGAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAAGGUGCGAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGCGCGUAGGCGGUUUAAUAAGUCAGAUGUGAAAGCCCAGGGCUCAACCUGGGAACUGCAUUUUGAACUGUUAAACUAGAGUUUUGUAGAGGANGGUAGAAUUUCAGGUGUAGCGGUGAAAUGCGUAGAGAUCUGAAGGAAUACCAGUGGCGAAGGCGGCCACCUGGACAAAGACUGACACUGAGGCGCGAAAGCGUGGGUAGCAAACGGGAUUAGAUACCCCGGUAGUCCACGCAGUAAACGAUGUCUAUUAGAAGUUUGUGGCUAUAUGCCGUGGGUUUCAAAGUUAACGCAUUAAAUAGACCGCCUGGGGAGUACGGCCGCAAGGUUAAAACUCAAAUGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCAUCCCUUGACAUCCAGAGAAUCACCUAGAGAUAGAUGAGUGCCUUCGGGAACUCUGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUUGUGAAAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUAUCCUUAGUUGCCAGCACUAAUGGUGGGAACUCUAGGGAGACUGCCGGUGAUAAACCGGAGGAAGGUGGGGACGACGUCAAGUCAUCAUGGCCCUUACGGGAUGGGCUACACACGUGCUACAAUGGCAAAUACAAAGGGUUGCUAACCUGCGAGGGUAUGCGAAUCUCAUAAAGUUUGUCGUAGUCCGGAUCGGAGUCUGCAACUCGACUCCGUGAAGUUGGAAUCGCUAGUAAUCGUGGAUCAGAAUGCCACGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGCUGCACCAGAAGUCAUUAGCUUAACCUUCGGGAUGGCGAUGACCACGGUGUGGUUCAUGACUGGGGUGAAGUCGUAACAAGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ga.Oce.pus  CAACUUGAGAGUUUGAUCCUGGCUCAGAACGAACGCUGGCGGCAGGCCUAACACAUGCAAGUCGAGGGAGAAGCUAUC----UUCG----GAUAGUGGAGACCGGCAGACGGGUGAGUAACACGUGGGAACRUACCGAGUAGUGGGGGAUAACAGUUGGAAACGACUGCUAAUACCGCAUACGCCCUUCGGGGGAAAGA--------UUUA------UCGCUAUUCGAUUGGCCCGCGUUAGAUUAGCUAGUUGGUAAGGUAACGGCUUACCAAGGCGACGAUCUAUAGCUGGUUUGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCGCAAUGGAGGAAACUCUGACGCAGCCAUGCCGCGUGAGUGAAGAAGGCCUUAGGGUUGUAAAGCUCUUUCAGACGUGAU--------------GAUG-----------AUGACAGUAGCGUCAAAAGAAGUUCCGGCUAACUUCGUGCCAGCAGCCGCGGUAAUACGAAGGGAACUAGCGUUGUUCGGAUUUACUGGGCGUAAAGAGCAUGUAGGCGGAUUGGACAGUUGAGGGUGAAAUCCCAGAGCUCAACUCUGGAACGGCCUUCAAUACUUCCAGUCUAGAGUCCGUAAGGGGGUGGUGGAAUUCCGAGUGUAGAGGUGAAAUUCGUAGAUAUUCGGAGGAACACCAGUGGCGAAGGCGACCACCUGGUACGGUACUGACGCUGAGAUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAGUGCUAGUUGUCAGGAUGUUUA-CAUCUUGGUGACGCAGCUAACGCAUUAAGCACUCCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUUGACGGGGGCCNGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAANNAACGCGAAGAACCUUACCNAUUCUUGACAUACCUGUCGAUUUCCAGAGAUGGAUUUCUCAG-UCGCUGGACAGGAUACAGGUGCUGCAUGGCNGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCACCCCUAGUUGCCAGCAUUUAG-UUGGGCACUCUAUGGGAACUGCCGGUGACAAGCNGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGGCCCUUACGGAUUGGGCUACACACGUGCUACAAUGGUAACUACAGUGGGCAGCGACGUCGCGAGGCGAAGCAAAUCUC-CAAAAGUUAUCUCAGUUCGGAUUGUUCUCUGCAACUCGAGAGCAUGAAGUCGGAAUCGCUAGUAAUCGCGGAUCAGCAUGCCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUUGGUUUUACCCGAAGACGGUGGGCUAACCUUUAGGAGGCAGCCGGCCACGGUAAGGUCAGCGACUGGGNmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ga.Pho.pho  -----mmmmmmmmmmmmmmmmmmmmmmAUUGAACGCUGGCGGCAGGCCUAACACAUGCAAGUCGAACGGUAACAGAU-GANACUUGUNUCAU-GCUGACGAACGGCGGACGGGUGAGUAAUGCCUGGGAAUAUACCCUGAUGUGGGGGAUAACUAUUGGAAACGAUAGCUAAUACCGCAUAAUCUCUUCGGAGCAAAGAGGGGGACCUUCGGGCCUCUCGCGUCAGGAUUAGCCCAGGUGGGAUUAGCUAGUUGGUGGGGUAAUGGCUCACCAAGGCGACGAUCCCUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGGGAAACCCUGAUGCAGCCAUGCCGCGUGUAUGAAGAAGGCCUUCGGGUUGUAAAGUACUUUCAGUUGUGAGGAAGGCGUUGGAGUUAAUAGCUUCAGCGCUUGACGUUAGCAACAGAAGAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGUUCCGAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGCAUGCAGGCGGUCUGUUAAGCAAGAUGUGAAAGCCCGGGGCUCAACCUCGGAACAGCAUUUUGAACUGGCAGACUAGAGUCUUGUAGAGGGGGGUAGAAUUUCAGGUGUAGCGGUGAAAUGCGUAGAGAUCUGAAGGAAUACCGGUGGCGAAGGCGGCCCCCUGGACAAAGACUGACGCUCAGAUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGUCUACUUGAAGGUUGUGGCCUUGAGCCGUGGCUUUCGGAGCUAACGCGUUAAGUAGACCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAUGAAUUGACGGAGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCUACUCUUGACAUCCAGAGAAUUCGCUAGAGAUAGCUUAGUGCCUUCGGGAACUCUGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUUGUGAAAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUAUCCUUGUUUGCCAGCACUAAUGGUGGGAACUCCAGGGAGACUGCCGGUGAUAAACCGGAGGAAGGUGGGGACGACGUCAAGUCAUCAUGGCCCUUACGAGUAGGGCUACACACGUGCUAGAAUGGCGUAUAGAGAGGGCUGCAAGCUAGCGAUAGUGAGCGAAUCCGAGAAAGUACGUCGUAGUCCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUCGGAAUCGCUAGUAAUCGUGAAUCAGAAUGUCACGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGCUGCACCAGAAGUAGAUAGCUUAACCUUCGGGAGGGCGUUUACCACGGUGUGGUUCAUGACUGGGGUGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ga.Pse.asp  -----mmmmmmmmmmmmmmmmmmmmmmAUUGAACGCUGGCGGCAGGCCUAACACAUGCAAGUCGAGCGGAUGAGAAGAG---CUUG---CUCUUCGAUUCAGCGGCGGACGGGUGAGUAAUGCCUAGGAAUCUGCCUGGUAGUGGGGGACAACGUUUCGAAAGGAACGCUAAUACCGCAUACGUCCUACGGGAGAAAGCAGGGGACCUUCGGGCCUUGCGCUAUCAGAUGAGCCUAGGUCGGAUUAGCUAGUUGGUGAGGUAAUGGCUCACCAAGGCGACGAUCCGUAACUGGUCUGAGAGGAUGAUCAGUCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGCGAAAGCCUGAUCCAGCCAUGCCGCGUGUGUGAAGAAGGUCUUCGGAUUGUAAAGCACUUUAAGUUGGGAGGAAGGGCAUUAACCUAAUACGUUAGUGUUUUGACGUUACCGACAGAAUAAGCACCGGCUAACUCUGUGCCAGCAGCCGCGGUAAUACAGAGGGUGCAAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGCGCGUAGGUGGUUCGUUAAGUUGGAUGUGAAAUCCCCGGGCUCAACCUGGGAACUGCAUCCAAAACUGGCGAGCUAGAGUAGGGUAGAGGGUGGUGGAAUUUCCUGUGUAGCGGUGAAAUGCGUAGAUAUAGGAAGGAACACCAGUGGCGAAGGCGACCACCUGGACUCAUACUGACACUGAGGUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGUCAACUAGCCGUUGGGAACCUUGAGUUCUUAGUGGCGCAGCUAACGCAUUAAGUUGACCGCCUGGGGAGUACGGCCGCAAGGUUAAAACUCAAAUGUAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGAAGAACCUUACCAGGCCUUGACAUGCAGAGAACUUUCCAGAGAUGGAUUGGUGCCUUCGGGAACUCUGACACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGUAACGAGCGCAACCCUUGUCCUUAGUUACCAGCACUCAUGGUGG-CACUCUAAGGAGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGGCCCUUACGGCCUGGGCUACACACGUGCUACAAUGGUCGGUACAGAGGGUUGCCAAGCCGCGAGGUGGAGCUAAUCCCACAAAACCGAUCGUAGUCCGGAUCGCAGUCUGCAACUCGACUGCGUGAAGUCGGAAUCGCUAGUAAUCRCGAAUCAGAAUGUCGCGGUGAAUACGUUCCCGGGSCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGUUGGACCAGAAGUAGCUAGUCUAACCUUCGGGAGGACGGUUACCACGGUGUGAUUCAUGACUGGGGUGAAGUCGUAACAAGGUAGCCGUAGGGGAACCUGCmmmmmmmmmmmmm-----
ga.Sal.typ  -----mmAGAGUUUGAUCCUGGCUCAGAUUGAACGCUGGCGGCAGGCCUAACACAUGCAAGUCGAACGGUAACAGGAAGCAGCUUGCUGCUUUGCUGACGAGUGGCGGACGGGUGAGUAAUGUCUGGGAAACUGCCUGAUGGAGGGGGAUAACUACUGGAAACGGUGGCUAAUACCGCAUAACGUCGCAAGACCAAAGAGGGGGACCUUCGGGCCUCUUGCCAUCAGAUGUGCCCAGAUGGGAUUAGCUUGUUGGUGAGGUAACGGCUCACCAAGGCGACGAUCCCUAGCUGGUCUGAGAGGAUGACCAGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGCGCAAGCCUGAUGCAGCCAUGCCGCGUGUAUGAAGAAGGCCUUCGGGUUGUAAAGUACUUUCAGCGGGGAGGAAGGUGUUGUGGUUAAUAACCGCAGCAAUUGACGUUACCCGCAGAAGAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGCACGCAGGCGCUCUGUCAAGUCGGAUGUGAAAUCCCCGGGCUCAACCUGGGAACUGCAUUCGAAACUGGCAGGCUUGAGUCUUGUAGAGGGGGGUAGAAUUCCAGGUGUAGCGGUGAAAUGCGUAGAGAUCUGGAGGAAUACCGGUGGCGAAGGCGGCCCCCUGGACAAAGACUGACGCUCAGGUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGUCUACUUGGAGGUUGUGCCCUUGAGGCGUGGCUUCCGGAGCUAACGCGUUAAGUAGACCGCCUGGGGAGUACGGCCGCAAGGUUAAAACUCAAAUGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCUGGUCUUGACAUCCACAGAACUUUCCAGAGAUGGAUUGGUUCCUUCGGGAACUGUGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUUGUGAAAUGUCGGGUUAAGUCCCGCAACGAGCGCAACCCUUAUCCUUUGUUGCCAGCGGUUAGGCCGGGAACUCAAAGGAGACUGCCAGUGAUAAACUGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGGCCCUUACGACCAGGGCUACACACGUGCUACAAUGGCGCAUACAAAGAGAAGCGACCUCGCGAGAGCAAGCGGACCUCAUAAAGUGCGUCGUAGUCCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUCGGAAUCGCUAGUAAUCGUGGAUCAGAAUGCCACGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGUUGCAAAAGAAGUAGGUAGCUUAACCUUCGGGAGGGCGCUUACCACUUUGUGAUUCAUGACUGGGGUGAAGUCGUAACAAGGUAACCGUAGGGGAACCUGCGGCUGGAUCACCUCCUU-
ga.Vib.cho  -----mmAGAGUUUGAUNNUGGCUCAGAUUGAACGCUGGCGGCAGGCCUAACACAUGCAAGUCGAGCGGCAGCACAGAGGAACUUGUUCCUUGGGUGGCGAGCGGCGGACGGGUGAGUAAUGCCUGGGAAAUUGCCCGGUAGAGGGGGAUAACCAUUGGAAACGAUGGCUAAUACCGCAUAACCUCGCAAGAGCAAAGCAGGGGACCUUCGGGCCUUGCGCUACCGGAUAUGCCCAGGUGGGAUUAGCUAGUUGGUGAGGUAAGGGCUCACCAAGGCGACGAUCCCUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGCGCAAGCCUGAUGCAGCCAUGCCGCGUGUAUGAAGAAGGCCUUCGGGUUGUAAAGUACUUUCAGUAGGGAGGAAGGUGGUUAAGUUAAUACCUUAAUCAUUUGACGUUACCUACAGAAGAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGCAUGCAGGUGGUUUGUUAAGUCAGAUGUGAAAGCCCUGGGCUCAACCUAGGAAUCGCAUUUGAAACUGACAAGCUAGAGUACUGUAGAGGGGGGUAGAAUUUCAGGUGUAGCGGUGAAAUGCGUAGAGAUCUGAAGGAAUACCGGUGGCGAAGGCGGCCCCCUGGACAGAUACUGACACUCAGAUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGUCUACUUGGAGGUUGUGCCCUAGAGGCGUGGCUUUCGGAGCUAACGCGUUAAGUAGACCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAUGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCUACUCUUGACAUCCAGAGAAUCUAGCGGAGACGCUGGAGUGCCUUCGGGAGCUCUGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUUGUGAAAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUAUCCUUGUUUGCCAGCACUAAUGGUGGGAACUCCAGGGAGACUGCCGGUGAUAAACCGGAGGAAGGUGGGGACGACGUCAAGUCAUCAUGGCCCUUACGAGUAGGGCUACACACGUGCUACAAUGGCGUAUACAGAGGGCAGCGAUACCGCGAGGUGGAGCGAAUCUCACAAAGUACGUCGUAGUCCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUCGGAAUCGCUAGUAAUCGCAAAUCAGAAUGUUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGCUGCAAAAGAAGCAGGUAGUUUAACCUUCGGGAGGACGCUUGCCACUUUGUGGUUCAUGACUGGGGUGAAGUCGUAACAAGGUAGCGCUAGGGGAACCUGGCGCUGGAUCACCUCCUUU
ga.Xan.pis  -----mmmmmmmmmmmmmmmmmmmmmmAGUGAACGCUGGCGGCAGGCCUAACACAUGCAAGUCGAACGGCAGCACAGUAAGACUUGUCUUAUGGGUGGCGAGUGGCGGACGGGUGAGGAAUACAUCGGAAUCUACUCUUUCGUGGGGGAUAACGUAGGGAAACUUACGCUAAUACCGCAUACGACCUACGGGUGAAAGCGGAGGACCUUCGGGCUUCGCGCGAUUGAAUGAGCCGAUGUCGGAUUAGCUAGUUGGCGGGGUAAAGGCCCACCAAGGCGACGAUCCGUAGCUGGUCUGAGAGGAUGAUCARCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGCGCAAGCCUGAUCCAGCCAUGCCGCGUGGGUGAAGAAGGCCUUCGGGUUGUAAAGCCCUUUUGUUGGGAAAGAAAAGCAGUCGGUUAAUACCCGAUUGUUCUGACGGUACCCAAAGAAUAAGCACCGGCUAACUUCGUGCCAGCAGCCGCGGUAAUACGAAGGGUGCAAGCGUUACUCGGAAUUACUGGGCGUAAAGCGUGCGUAGGUGGUGGUUUAAGUCUGUUGUGAAAGCCCUGGGCUCAACCUGGGAAUUGCAGUGGAUACUGGGUCACUAGAGUGUGGUAGAGGGUAGCGGAAUUCCCGGUGUAGCAGUGAAAUGCGUAGAGAUCGGGAGGAACAUCCGUGGCGAAGGCGGCUACCUGGACCAACACUGACACUGAGGCACGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGCGAACUGGAUGUUGGGUGCUUUG-GCACGCAGUAUCGAAGCUAACGCGUUAAGUUCGCCGCCUGGGGAGUACGGUCGCAAGACUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGUAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCUGGUCUUGACAUCCACGGAACUUUCCAGAGAUGGAUUGGUGCCUUCGGGAACCGUGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCCUUAGUUGCCAGCACUAAUGGUGGGAACUCUAAGGAGACCGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGGCCCUUACGACCAGGGCUACACACGUACUACAAUGGUAGGGACAGAGGGCUGCAAACCCGCGAGGGCAAGCCAAUCCCAGAAACCCUAUCUCAGUCCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUCGGAAUCGCUAGUAAUCGCAGAUCAGCAUGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUUUGUUGCACCAGAAGCAGGUAGCUUAACCUUCGGGAGGGCGCUUGCCACGGUGUGGCCGAUGACUGGGGUGAAGUCGUAACAAGGUAGCCGUAUCGGAAGGUGCmmmmmmmmmmmmm-----
ra.Dei.mur  -----mmmmmmmmmmmmmmmmmmmmAGGGUGAACGCUGGCGGCGUGCUUAAGACAUGCAAGUCGAACGGCCCU---------UUCG--------GAGGGCAGUGGCGCACGGGUGAGUAACGCGUAACGACCUGCCCCCAAGUCUGGAAUAACCCUCCGAAAGGAGGGCUAAUACCGGAUGUGCUGCUUGCAGUAAAGGC-------GCGA-----GCCGCUUGGGGAUGGGGUUGCGUUCCAUCAGCUAGAUGGUGGGGUAAAGGCCUACCAAGGCGACGACGGAUAACCGGCCUGAGAGGGUGGCCGGUCACAGGGGCACUGAGACACGGGUCCCACUCCUACGGGAGGCAGCAGUUAGGAAUCUUCCCCAAUGGACGAAAGUCUGAGGGAGCGACGCCGCGUGAGGGAUGAAGGUCUUCGGAUUGUAAACCUCUGA-AUCAGGGACGAAAGACGCU----UUA------GGCGGGAUGACGGUACCUGAG-UAACAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUACCCGGAAUCACUGGGCGUAAAGGGCGUGUAGGCGGUACUUUAAGUCUGACUUUAAAGACCGUGGCUGAACCACGGAAGUGGGUUGGAUACUGGCGUGCUGGACCUCUGGAGAGACAACCGGAAUUCCUGGUGUAGCGGUGGAAUGCGUAGAUACCAGGAGGAACACCGAUGGCGAAGGCAGGUUCUUGGACAGAAGGUGACGCUGAGGCGCGAAAGUGUGGGGAGCGAACCGGAUUAGAUACCCGGGUAGUCCACACCCUAAACGAUGUACGUUGGCUGAUGGCGG-GAUG--CCGUCAUGGGCGAAGCUAACGCGAUAAACGUACCGCCUGGGAAGUACGGCCGCAAGGUUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGAAGAACCUUACCAGGUCUUGACAUCCCAUGAACCUUGCAGAGAUGUGAGGGUGCCUUCGGGAACAUGGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGCCUUGUGUUGCCAGCAGUUCGGCUGGGCACUCACGAGGGACUGCCUGUGA-AAGCAGGAGGAAGGCGGGGAUGACGUCUAGUCAGCAUGGUCCUUACGACCUGGGCGACACACGUGCUACAAUGGAUGGUACAACGCGCAGCCAAGUCGCGAGACUGAGCGAAUCGCUGAAAGCCAUCCCCAGUUCAGAUCGGAGUCUGCAACUCGACUCCGUGAAGUUGGAAUCGCUAGUAAUCGCGGGUCAGCAUACCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUAGAUUGCAGCUGAAACCGCCGGGAGCU--UCAC---GGCAGGCGUCUAGGCUGUGGUUUAUGACUGGGGUGAAGUCGUAACAAGGUAACUGUACCGGAAGGUGCGGCUGGAmmmmmm-----
ra.Dei.rad  UUUAUGGAGAGUUUGAUCCUGGCUCAGGGUGAACGCUGGCGGCGUGCUUAAGACAUGCAAGUCGAACGCGGUC---------UUCG---------GACCGAGUGGCGCACGGGUGAGUAACACGUAACGACCUACCCAGAAGUCAUGAAUAACUGGCCGAAAGGUCAGCUAAUACGUGAUGUGGUGCGUGCACUAAAGA--------UUUA------UCGCUUCUGGAUGGGGUUGCGUUCCAUCAGCUGGUUGGUGGGGUAAAGGCCUACCAAGGCGACGACGGAUAGCCGGCCUGAGAGGGUGGCCGGCCACAGGGGCACUGAGACACGGGUCCCACUCCUACGGGAGGCAGCAGUUAGGAAUCUUCCACAAUGGGCGCAAGCCUGAUGGAGCGACGCCGCGUGAGGGAUGAAGGUUUUCGGAUCGUAAACCUCUGA-AUCUGGGACGAAAGAGCC-----UUCG------GGCAGAUGACGGUACCAGAG-UAAUAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUACCCGGAAUCACUGGGCGUAAAGGGCGUGUAGGCGGAUAUUUAAGUCUGGUUUUAAAGACCGAGGCUCAACCUCGGGAGUGGACUGGAUACUGGAUGUCUUGACCUCUGGAGAGGUAACUGGAAUUCCUGGUGUAGCGGUGGAAUGCGUAGAUACCAGGAGGAACACCAAUGGCGAAGGCAAGUUACUGGACAGAAGGUGACGCUGAGGCGCGAAAGUGUGGGGAGCAAACCGGAUUAGAUACCCGGGUAGUCCACACCCUAAACGAUGUACGUUGGCUAAGCGCAG-GAUG--CUGUGCUUGGCGAAGCUAACGCGAUAAACGUACCGCCUGGGAAGUACGGCCGCAAGGUUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGAAGAACCUUACCAGGUCUUGACAUGCUAGGAAGGCGCUGGAGACAGCGCCGUGCCUUCGGGAACCUAGACACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGCUUCCAGUUGCCAGCA-UUCAGUUGGGCACUCUGGAGGGACUGCCUGUGA-AAGCAGGAGGAAGGCGGGGAUGACGUCUAGUCAGCAUGGUCCUUACGUCCUGGGCUACACACGUGCUACAAUGGAUAGGACAACGCGCAGCAAACAUGUGAGUGUAAGCGAAUCGCUGAAACCUAUCCCCAGUUCAGAUCGGAGUCUGCAACUCGACUCCGUGAAGUUGGAAUCGCUAGUAAUCGCGGGUCAGCAUACCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUAGAUUGCAGUUGAAACCGCCGGGAGCC--UCAC---GGCAGGCGUCUAGACUGUGGUUUAUGACUGGGGUGAAGUCGUAACAAGGUAACUGUACCGGAAGGUGCGGUUGGAUCACCUCCUUU
sp.Bor.par  -----mmmmAGUUUGAUCCUGGCUUAGAACUAACGCUGGCAGUGCGUCUUAAGCAUGCAAGUCAGACGGAAUGUA-------GCAA-------UACAUUCAGUGGCGAACGGGUGAGUAACGCGUGGAAAUCUACCUAUGAGAUGGGGAUAACUAUUAGAAAUAGUAGCUAAUACCGAAUAAGGUCGAUGGAUGAAAGAAGCCU---UUAA--AGCUUCGCUUGUAGAUGAGUCUGCGUCUUAUUAGCUAGUUGGUAGGGUAAGAGCCUACCAAGGCUAUGAUAAGUAACCGGCCUGAGAGGGUGAACGGUCACACUGGAACUGAGAUACGGUCCAGACUCCUACGGGAGGCAGCAGCUAAGAAUCUUCCGCAAUGGGCGAAAG-CUGACGGAGCGACACUGCGUGAACGAAGAAGGUCGAAAGAUUGUAAAGUUCUUUUAUAAAUGAGGAAUAAGCUUUGUAGAAA-UGACAAGGUGAUGACGUUAAUUUAUGAAUAAGCCCCGGCUAAUUACGUGCCAGCAGCCGCGGUAAUACGUAAGGGGCGAGCGUUGUUCGGGAUUAUUGGGCGUAAAGGGUGAGUAGGCGGAUAUGUAAGUCAUGCGUAAAAUACCACAGCUCAACUGUGGAACUAUGCUGGAAACUGCAUGACUAGAGUCUGAUAGGGGAAGUUAGAAUUCCUGGUGUAAGGGUGGAAUCUGUUGAUAUCAGGAAGAAUACCAGAGGCGAAGGCGAACUUCUGGGUCAAGACUGACGCUGAGUCACGAAAGCGUAGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCUACGCUGUAAACGAUGCACACUUGGUGUUAAUC--GAAA---GGUUAGUACCGAAGCUAACGUGUUAAGUGUGCCGCCUGGGGAGUAUGCUCGCAAGAGUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGAUACGCGAGGAACCUUACCAGGGCUUGACAUAACAGGAUGUAGUUAGAGAUAACUAUUCCCCGUUUGGGGUCUGUAUACAGGUGCUGCAUGGUUGUCGUCAGCUCGUGCUGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUAUUGUCUGUUACCAGCAUUAAAGAUGGGGACUCAGACGAGACUGCCGGUGAUAAGCCGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGGCCCUUAUGUCCUGGGCUACACACGUGCUACAAUGGCCUGUACAAAGCGAUGCGAAACAGUGAUGUGAAGCAAAACGCAUAAAGCAGGUCUCAGUCCAGAUUGAAGUCUGAAACUCGACUUCAUGAAGUUGGAAUCGCUAGUAAUCGUAUAUCAGAAUGAUACGGUGAAUACGUUCUCGGGCCUUGUACACACCGCCCGUCACACCACCCGAGUUGAGGAUACCCGAAGCUAUUAUUCUAACCGCAAGGAGGAAGGUAUCUAAGGUAUGUUUAGUGAGGGGGGUGAAGUCGUAACAAGGUAGCCGUACUGGAAAGUGUGGCUGGAUCACCU-----
sp.Lep.mey  -----mmmmmmmmmmmmmmmmmmmmmmmmSUARCGCUGGCGGCGCGUCUUAAACAUCCAA-UCAAGCGGAGUA---------GCAA---------UACUCAGCGGCGAACGGGUGAGUAACACGUGGGAAUCUUCCUCCGAGUCUGGGAUAACUUUUCGAAAGGGAAGCUAAUACUGGAUAGUCCCAUAAGGGUAAAGA--------UUCA------UUGCUUGGAGAUGAGCCCGCGUCCGAUUAGCUAGUUGGUGAGGUAAUGGCUCACCAAGGCGACGAUCGGUAGCCGGCCUGAGAGGGUGUUCGGCCACAAUGGAACUGAGACACGGUCCAUACUCCUACGGGAGGCAGCAGUUAAGAAUCUUGCUCAAUGGGGGGAACCCUGAAGCAGCGACGCCGCGUGAACGAUGAAGGUCUUCGGAUUGUAAAGUUCAAUAAGCAGGGAAAAAUAAGCA-----GCAA-------UGUGAUGAUGGUACCUGCCUA--AAGCACCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAUGGUGCAAGCGUUGUUCGGAAUCAUUGGGCGUAAAGGGUGCGUAGGCGGACAUGUAAGUCAGGUGUGAAAACUGGGGGCUCAACUCCCAGCCUGCACUUGAAACUAUGUGUCUGGAGUUUGGGAGAGGCAAGUGGAAUUCCAGGUGUAGCGGUGAAAUGCGUAGAUAUCUGGAGGAACACCAGUGGCGAAGGCGACUUGCUGGCCUAAAACUGACGCUGAGGCACGAAAGCGUGGGUAGUGAACGGGAUUAGAUACCCCGGUAAUCCACGCCCUAAACGUUGUCUACCAGUUGUUGGGGGUUUAA--CCCUCAGUAACGAACCUAACGGAUUAAGUAGACCGCCUGGGGACUAUGCUCGCAAGAGUGAAACUCAAAGGAAUUGACGGGGGUCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGAUACGCGAAAAACCUCACCUAGGCUUGACAUGAGUGGAAUCAUGUAGAGAUACAUGAGCC--UUCG--GGCCGCUUCACAGGUGCUGCAUGGUUGUAAUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCACCUUAUGUUGCCAUCA-UUCAGUUGGGCACUCGUAAGGAACUGCCGGUGACAAACCGGAGGAAGGCGGGGAUGACGUCAAAUCCUCAUGGCCUUUAUGUCUAGGGCAACACACGUGCUACAAUGGCCGGUACAAAGGGUAGCCAACUCGCGAGGGGGAGCUAAUCUCAAAAACCCGGUCCCAGUUCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUCGGAAUCGCUAGUAAUCGCGGAUCAGCAUGCCGCGGUGAAUACGUUCCCGGACCUUGUACACACC-CCCGUCACACCACCUGAGUGGGGAGCACCCGAAGUGGUCUUGCCAACCGCAAGGAAGCAGACUACUAAGGUGAAACUCGUGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
sp.Tre.soc  -----mmmmmGUUUGAUCCUGGCUCAGAACGAACGCUGGCGGCGCGUCUUAACCAUGCAAGUCGAGCGGCAGGCA-------GCAA-------UGCCGAGAGCGGCGGACUGGUGAGUAACACGUGGAAACGAACCCCGAUGCCCGGGACAGCCUGUAGAAAUAGAGGGUAAUACCGGAUAGAUCAGGAACGGGAAAGGAGC-----UUCG---GCUCCGCGCCGGGAUCGGUCUGCGGCCCAUCAGCUAGACGGCGGGGUAAGGGCCCGCCGUGGCGAGGACGGGUAUCCGGCCUGAGAGGGCGGACGGACACAUUGGGACUGAGAUACGGCCCAGACUCCUACGGGAGGCAGCAGGUAAGAAUAUUCCGCAAUGGGGGGAACCCUGACGGAGCGACGCCGCGUGAACGAAGAAGGCCGGAAGGUUGUAAAGUUCUUUUCUGUCCGAGGAAUAAGUGUAG-GGAAA-UGCCUGCAUGGUGACGGUAGGGCAGGAAUAAGCACCGGCUAAUUACGUGCCAGCAGCCGCGGUAACACGUAAGGUGCGAGCGUUGUUCGGAAUUAUUGGGCGUAAAGGGCAUGCAGGCGGGUCGCCAAGCUUGGUAAGAAAUACCGGGGCUCAACUCCGGAGCUAUAUUGAGAACUGGCGAGCUAGAGUUGCCGAAGGGUAUCCGGAAUUCCGCGUGAAGGGGUGAAAUCUGUAGAUAUGCGGAAGAACACCGAUGGCGAAGGCAGGAUACCGGCGGACGACUGACGCUGAGGUGCGAAGGUGCGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCGCACAGUCAACGAUGUACACUGGGCGUGUGCGC-AAGA--GCGUGCGUGCCGAAGCAAACGCGAUAAGUGUACCGCCUGGGGAGUAUGCCCGCAAGGGUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAUGGUACGCGAGGAACCUUACCUGGGUUUGACAUCAGAGGGAUCAUAUAGAGAUAUGUGAGCGUAGCAAUACGGCUCUUGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUACUGCCGGUUACUAACAGUAAUGCUGAGGACUCAGGCGGAACUGCCUGCGACAAGCAGGAGGAAGGCGGGGACGACGUCAAGUCAUCAUGGCCCUUAUGUCCAGGGCUACACACGUGCUACAAUGGCCGCCACAGAGCGGGGCGAAGCCGAGAGGCGGAGCAGAACGCAGAAAAGCGGUCGUAGUCCGGAUUGAAGUCUGAAACUCGACUUCAUGAAGCUGGAAUCGCUAGUAACCGCACAUCAGCACGGCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUCCGAGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
sp.Tre.suc  GGCAUGGAGAGUUCGAUCCUGGCUCAGAACGAACGCUGGCGGCGCGUCUUAAGCAUGCAAGUCGGGCGGGAUUCACGCG---CUUG---CGCGUGGUGAGAGCGGCGGACUGGCGAGUAACACGUGGGGACGCGCCCUCCGGACGGGAAUAGCCUGCAGAAAUGCAGGGUAAUGCCGGAUGCGAACUGGAGUGGAAAGCCCC-----CACG---GGGGCGCCGGAGGAGCGGCCCGCGGCCCAUCAGCUGGUAGGCGGUGCAAGGGACCACCUAGGCUACGACGGGUACCCGGCNUAAGAGGGCGGACGGGCACAUUGGGACUGAGAUACGGCCCAGACUCNUACGGGAGGCAGCAGCUAAGAAUAUUCCGCAAUGGGGGGAACNNUGACGGAGCGACGCCGCGUGGGCGAGGAAGGCCGGAAGGUUGUAAAGCCCUUUUGCGCGCGAGGAAUNAGGGGAG-GGGAA-UGCCUUCCCGGUGACUGUAGCGCGCGAAUAAGCGCCGGCUAAUUACGUGCCAGCAGCCGCGGUNACACGUAAGGCGCNAGCGUUGUNCGGAAUNAUUGGGCGUAAAGGGCGUGUAGGCGGCCCUGCAAGCCUNGCGUGAAAUCCCGGGGCNCAACCCCGNAACCGCGCUGGGAACUGCUGGGCUUGAGCCGCUGUGGCGCAGCCGGAAUUNCAGGUGUAGGGGUGAAAUCUGUAGAUAUCUGGAAGAACACCGAUGGCGAAGGCAGGCUGCGAGCGGACGGCUGACGCUGAGGCGCGAAGGCGCGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCGCGCAGUAAACGAUGCACACUGGGUGNUCGGGC-AUGA--GCCCGGNUGCCGAAGCGAACGCGUUAAGUGUGCCGCCUGGGGAGUAUGCCCGCAAGGGUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUAUAAUUCGANNNNUAGCGAGGAACCUUACCUGGGCUUGACAUAACAGGGACCGCCUGGAGACAGGCGGACGCAGCAAUGCGCCUGUGAACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUGCCGCCAGUUGCCAGCAUUAGAGGUGGGCACUCUGGCGGAACUGCCGGCGACAAGCAGGAGGAAGGCGGGGACGACGUCAAGUCAUCAUGGCCCUUAUGUCCAGGGCUACACACGUGCUACAAUGGCAGGCACAGAGUGAAGCGAGGCCGCGAGGCGGAGCGAAACGCAGAAAACCUGCCGUAGUCCGGAUCGGAGUCUGAAACCCGACUCCGUGAAGCUGGAAUCGCUAGUAAUCGCGCAUCAGCACGGCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUCCGAGUCGGGGAUUCCCGAAGCGGGCAGGCCAACCGAAAGGGGGCNNCUCUNNAAGGUNUGCUUmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
th.The.afr  UUCAUGGAGGGUUUGAUCCUGGCUCAGGGUGAACGCUGGCGGCGUGCCUUACACAUGCAAGUCGAGCGAAGCUGCUGGUGGAUUCGANGCCNGUAGACUNAGCGGCGGACGGGUGAGUAACGCGUAGGGACCUACCCCUCAGAGGGGGAUAACUGGGGGAAACCUCAGCUAAUACCCNAUACGUUCUAAGGAAGAAAGGAGC-----AAUA---GCUCUGCUGAGNNNGGGGCCUGCGACCCAUCAGGUAGUUGGUGAGGUAACGGCUCACNAAGCCUACGACGGGUAGCCGGUCUGAGAGGAUGGCCGNCNACAAGGGCACUGAGACACGGGCCNNACUCCUACGGGAGGCAGCAGUGGGGGAUUUUGGACAAUGGGCGAAAGCCUNAUCCAGCGACGCCGNGUGCGGGACGAAGCCCUUCGGGGUGUAAACCGCUGUGGUGGGAGACGAAUAAGGUGAG-GGGAA-AGCCUCACUGAUGACGGUAUCCCACUAGAAAGCCCCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGGGCNAGCGUUACCCGGAUUUACUGGGCGUAAAGGGGGCGUAGGCGGCCGUGAAAGUCCGGUGUGAAAACUCACGGCUCAACCGUGGGACUGCGCUGGAAACUACACGGCUUGAGGACGGUAGAGGGAGACGGAACUGCUGGUGUAGGGGUGAAAUCCGUAGAUAUCAGCAGGAACGCCGGUGGAGAAGUCGGUCUCCUGGGCCGUNCCUGACGCUGAGGCCCNAAAGCUAGGGGAGCAAACCGGAUUAGAUACCCGGGUAGUCCUAGCCGUAAACGAUGCCCNCUAGGUGUGGGGGAG-UAA-UUCCUCCGUGCUGAAGCUAACGCGAUAAGUGGGCCGCCUGGGGAGUAUGCCCGCAAGGGUGAAACUCAAAGGAAUUGACGGGACCCNGCACAAGCGGUGGAGCGUGUGGUUUAAUUGGACGCUAAGCCAAGAACCUUACCAGGGCUUGACAUGUGGUGUACUGNCNCGAAAGGGGAGGGCCUGGUUACGGGAGCCAGCACAGGUGGUGCACGGUCGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUGCCCUUAGUUGCCAGCGGUUCGGCCGGGCACUCUAAGGGNACUGCCGGCGACGAGCCGGAGGAAGGAGGGGAUGACGUCAGAUACUCGUGCNCCUUAUGCCCUNGGCGACACACGCGCUACAAUGGGUGGGACAGCGGGAAGCGAGCCAGCGAUGGUGAGCGAAGCCCUUAAACCCACCCUCAGUUCGGAUUGCAGGCUGAAACCCGCCUGCAUGAAGCCGGAAUCGCUAGUAAUCGCGGAUCAGCCCGCCGCGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm----mmmm---mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
th.The.hyp  -----mmmmGGUUUGAUCCUGGCUCAGGGUGAACGCUGGCGGCGUGCCUAACACAUGCAAGUCGAGCGGGUACCC-------GCAA-------GGGUACCAGCGGCGCACGGGUGAGUAACACGUGGGAAUCUACCCCUCGGAGGGGGAUAACCGGGGGAAACCCCGGCUAAUACCCCAUACGAUCGAAGGA-GAAAGGGGC-----GUUU---CCUCCGCCGAGGGAUGAGCCCGCGUCCCAUCAGGUAGUUGGUGGGGUAAUGGCCCACCAAGCCUACGACGGGUAGCCGGCCUGAGAGGGUGGUCGGCCACAAGGGCACUGAGACACGGGCCCUACUCCUACGGGAGGCAGCAGUGGGGAAUCUUGGACAAUGGGCGAAAGCCUGAUCCAGCGACGCCGCGUGGGGGAUGAAGCCCUUCGGGGUGUAAACCCCUGUUGCGAGGGAGGAAUAAGGCCUG-GGAAA-UGCCAGGCCGAUGACUGUACCUCGCGAGGAAGCCCCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGGGCGAGCGUUACCCGGAUUCACUGGGCGUAAAGCGGGUGUAGGCGGCUCGGUAAGUCGGGUGUGAAAUCCCACAGCUCAACUGUGGAAUUGCGCCCGAAACUGCUGAGCUUGGGGCCGGUAGAGGGAGACGGAACUGCCGGUGUAGGGGUGAAAUCCGUAGAUAUCGGCAGGAACGCCGGUGGGGAAGCCGGUCUCCUGGGCCGCGCCCGACGCUGAGACCCGAAAGCUAGGGGAGCAAACCGGAUUAGAUACCCGGGUAGUCCUAGCCGUAAACGAUGCCCACUAGGUGUGGGGGA-UUAA-UUCCUCCGUGCUGUAGCUAACGCGUUAAGUGGNCCGCCUGGGGAGUACGCCCGCAAGGGUGAAACUCAAAGGAAUUGGCGGGACCCCGCACAAGCGGUGGAGCGCGUGGUUUAAUUGGAUGCUGAGCCAAGAACCUUACCAGGGUUUGACAUGAGGUGUACCGACCCGAAAGGGGAGGGCCCCUUUUGGGGAGGCUGCACAGGUGGUGCACGGCCGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAAUCCCUGCCCUUAGUUGACAACGGUUCGGCCGGGUACUCUAAGGGGACUGCCGGUGACGANUCGGAGGAAGGAGGGGACGACGUCAGGUACUCGUUCCCUUUAUGCCCUGGGCUACACACGCGCUACACUGGGUGGUACAAUGGGUUGCGACCCCNCGAGGGGGAGCCAAUCCC-UAAAACCACCCCCAGUUCGGAUCGCAGGCUGCAACCCGCCUGCGUGAAGCCGGAAUCGCUAGUAAUCGCGGAUCAACCCGCCGCGGUGAAUACGUUCCCGGGGUUUGCACACACCGCCCGUCAAACCACCCGAGUCGGGGGCACCCGAAGACGCUCUCCUAACCGAAAGGAGGGAGGCGUUGAGGGUGAACCUGGUGAGGGGGGCUAAGUCGUAACAAGGUAACCGUACUGGAAGGUGCmmmmmmmmmmmmm-----
th.The.sub  -----mmmmmmmmmmmmmmmmmmmmAGGGUGAACGCUGGCGGCGUGCUUAACACAUGCAAGUCGCGCGGGGA-AACCCC---UUCG---GGGGGAGUACCAGCGGCGCACGGGUGAGUAACACGUGGGAACCUACCCCUCAGCGGGGGAUAACCGGGGGAAACUCCGGCUAAUACCCCAUAUUAUCGACAGAUGAAAGGAGC-----GUUU---GCUUCGUUGAGGGAUGGGCCCGCGGCCCAUCAGGUAGUUGGUGAGGUAAUGGCUCACCAAGCCUACGACGGGUAGCCGACCUGAGAGGGUGACCGGCCACAAGGGCACUGAGACACGGGCCCUACUCCUACGGGAGGCAGCAGUGGGGAAUUUUGGACAAUGGGCGAAAGCCUGAUCCAGCGACGCCGCGUGAGGGACGAAGCCCUUCGGGGUGUAAACCUCUGUUGUGAGGGACGAAUAAGAUCUG-GGAAA-UGCCAGAUCGAUGACGGUACCUCACGAGAAAGCCCCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGGGCAAGCGUUACCCGGAUUCACUGGGCGUAAAGGGGGCGCAGGCGAUCCAGUAUGUCGGGUGUGAAAUCCCACAGCUCAACUGUGGAAUCGCACCCGAAACUACUGGGCUUGGGGCUGGUAGAGGGAGACGGAACUGCUGGUGUAGGGGUGAAAUCCGUAGAUAUCAGCAGGAACGCCGGUGGGGAAGCCGGUCUCCUGGGCCAAGCCCGACGCUGAGGCCCGAAAGCUAGGGGAGCAAACCGGAUUAGAUACCCGGGUAGUCCUAGCCGUAAACGAUGCCCACUAGGUGUGGGGGAG-UCA-UUCCUCCGUGCUGUAGCUAACGCGUUAAGUGGGCCGCCUGGGGAGUACGCCCGCAAGGGUGAAACUCAAAGGAAUUGGCGGGGACCCGCACAAGCGGUGGAGCGUGUGGUUUAAUUGGAUGCUAAGCCAAGAACCUUACCAGGGCUUGACAUGAGGUGUACCAACCCGAAAGGGGAAGGCCCUUUUUAGGGAGCCUGCACAGGUGGUGCACGGCCGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUGCCCUU-GUUGCCAGCGGUUCGGCCGGGCACUCUAAGGGGACCGCCGGCGACGAGCCGGAGGAAGGAGGGGACGACGUCAGGUACUCGUGCCCUUUAUGCCCUGGGCUACACACGCGCUACAAUGGGUGGUACAGUGGGUCGCGACCUCGCGAGAGGGAGCCAAUCCC-CAAAACCAUCCUCAGUUCAGAUCGCAGGCUGCAACCCGCCUGCGUGAAGCCGGAAUCGCUAGUAAUCGCGGAUCAGCCAGCCGCGGUGAAUACGUUCCCGGGGUUUGCACACACCGCCCGUCAAGCCACCCGAGUCGGGGGCACCUGAAGACGCCUUCCUAACCGAAAGGAGGGAGGUGGUGAAGGUGAAUCUGGCGAGGGGGGCUAAmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ar.Sul.sol  -----AUUCCGGUUGAUCCUGCCGGACCC-GACCGCUAUCGGGGUAGGUAAGCCAUGGGAGUCUUACACUCCCG--------GUAA---------GGGAGUGUGGCGGACGGCUGAGUAACACGUGGCAACCUACCCUCGGGACGGGGAUAACCCCGGGAAACUGGGGAUAAUCCCCGAUA-GGGAGGAAUCCUAAAGGCUAUAGGCUUUCUUGUAGCCGCCCGAGGAUGGGGCUACGGCCCAUCAGGCUGUCGGUGGGGUAAAGGCCCACCGAACCUAUAACGGGUAGGGGCCGUGGAAGCGGGAGCCUCCAGUUGGGCACUGAGACAAGGGCCCAGGCCCUACGGGGCGCACCAGGCGCGAAACGUCCCCAAUGCGCGAAAGCGUGAGGGCGCUACCCCGAGUGCCC----------GCAA-------G-AGGCUUUUCCCCGCUCU--------------AAAA-------------------GGCGGGGGAAUAAGCGGGGGCAAGUCUGGUGUCAGCCGCCGCGGUAAUACCAGCUCCGCGAGUGGUCGGGGUGAUUACUGGGCCUAAAGCGCCUGUAGCCGGCCCACCAAGUCGCCCCUUAAAGUCCCCGGCUCAACCGGGGAACUGGGGGCGAUACUGGUGGGCUAGGGGGCGGGAGAGGCGGGGGGUACUCCCGGAGUAGGGGCGAAAUCCUUAGAUACCGGGAGGACCACCAGUGGCGGAAGCGCCCCGCUAGAACGCGCCCGACGGUGAGAGGCGAAAGCCGGGGCAGCAAACGGGAUUAGAUACCCCGGUAGUCCCGGCUGUAAACGAUGCGGGCUAGGUGUCGAGUAGUUAGCCUACUCGGUGCCGCAGGGAAGCCGUUAAGCCCGCCGCCUGGGGAGUACGGUCGCAAGACUGAAACUUAAAGGAAUUGGCGGGGGAGCACACAAGGGGUGGAACCUGCGGCUCAAUUGGAGUCAACGCCUGGAAUCUUACCGGGGG-AGACCC-UA---------------CAAC-------------C--UC--GCGGAGAGGAGGUGCAUGGCCGUCGCCAGCUCGUGUUGUGAAAUGUCCGGUUAAGUCCGGCAACGAGCGAGACCCCCACCCCUAGUUGGUAUUCUUCCG-AGAACCACACUAGGGGGACUGCCGGCGUAA-GCCGGAGGAAGGAGGGGGCCACGGCAGGUCAGCAUGCCCCGAAACUCCCGGGCCGCACGCGGGUUACAAUGGCAGGGACAACGGGAUGCUACCUCGAAAGGGGGAGCCAAUCCU-UAAACCCUGCCGCAGUUGGGAUCGAGGGCUGAAACCCGCCCUCGUGAACGAGGAAUCCCUAGUAACCGCGGGUCAACAACCCGCGGUGAAUACGUCCCUGCUCCUUGCACACACCGCCCGUCGCUCCACCCGAGCGCGAAAGGGGUGA-GGUCCCUGC------GAUA-----AGGGGAUCGAACUCCUUUCCCGCGAGGGGGGAGAAGUCGUAACAAGGUAGCCGUAGGGGAACCUGCGGCUGGAUCACCUCAm--
ar.The.pen  GCCAGACUCCGGUUGAUCCUGCCGGACCC-GACCGCUAUCGGGGUGGGUAACCCAUGGAAGUCUAGGAGCCGGGC-------UACG--------GCCGGCUCCGGCGGACGGCUCAGUAGCACGUGGCAACCUACCCUCGGGAGGGGGAUAACCCCGGGAAACUGGGGAUAAACCCCCAUA-GGCGGGAACGCGAAAGGCCACGGUACAUGCCGUGGCCGCCCGAGGAUGGGGCUGCGCCCUAUCAGGUAGUUGGCGGGGUAACGGCCCGCCAAGCCGAUAACGGGUGGGGGCCGUGAGAGCGGGAGCCCCGAGAUGGGCACUGAGACAAGGGCCCAGGCCCUACGGGGUGCACCAGGGGCGAAACUUCCGCAAUGCGGGAAACCGUGACGGAGUCACCCCGAGUGCCC----------GAAG-------GGUGGCUUUUGCCCGGUCU--------------AAAA-------------------GCCGGGCGAAUAAGCGGGGGCAAGCUUGGUGUCAGCCGCCGCGGUAAUACCAACCCCGCGAGUGGUCGGGACGUUUAUUGGGCCUAAAGCGUCCGUAGCCGGCCCGGUAAGUCCCUCCUUAAAGCCCACGGCUCAACCGUGGGAGCGGAGG-GAUACUGCCGGGCUAGGGGGCGGGAGAGGCCGGGGGUACUCCUGGGGUAGGGGCGAAAUCCUAUAAUCCCAGGAGGACCACCAGUGGCGAAGGCGCCCGGCUAGCACGCGCCCGACGGUGAGGGACGAAAGCUGGGGGAGCAAAGGGGAUUAGAUACCCCCGUAGUCCCAGCUGUAAACGAUGCGGGCUAGGUGUUGGACGGUUCGCCCGUCCAGUGCCGUAGGGAAGCCGUUAAGCCCGCCGCCUGGGGAGUACGGCCGCAAGGCUGAAACUUAAAGGAAUUGGCGGGGGAGCACACAAGGGGUGAAGCUUGCGGUUUAAUUGGAGUCAACGCCGGAAACCUUACCGGGGG-CGACAC-GA---------------CGAC-------------C--AA--GCUGAGAGGAGGUGCAUGGCCGUCGCCGGCUCGUGCCGUGAGGUGUCCUGUUAAGUCAGGGAACGAGCGAGACCCCCGCCCCUAGUUGCUACCCAUUCG-UGGGGCACUCUAGGGGGACUGCCGGCGAUAAGCCGGAGGAAGGUGGGGGCUACGGCAGGUCAGUAUGCCCCGAAACCCCCGGGCUACACGCGAGCUGCAAUGGCGGGGACAGCGGGUUCCGACCCCGAAAGGGGGAGGUAAUCCCUUAAACCCCGCCUCAGUAGGAAUCGAGGGCUGCAACUCGCCCUCGUGAACGUGGAAUCCCUAGUAACCGCGUGUCACCAACGCGCGGUGAAUACGUCCCUGCUCCUUGCACACACCGCCCGUCGCUCCACCCGAGGGAGGCCUAGGUGA-GGCCUCCGCC-----GACG----AGGGAGGUCGAACCUGGGCCUCCCAAGGGGGGAGAAGUCGUAACAAGGUGGCCGUAGGGGAACCUGCGGCCGGAUCACCUCmm--
ar.Hal.val  -----AUUCCGGUUGAUCCUGCCGGAGGC-CAUUGCUAUCGGAGUCCAUUAGCCAUGCUAGUUGCACGAGU-----------UUAG-----------ACUCGUAGCAUAUAGCUCAGUAACACGUGGCAAACUACCCUACAGACCGCAAUAACCUCGGGAAACUGAGGCCAAUAGCGGAUA-UAACGGAGGUUGAAAGU--------UCCG------GCGCUGUAGGAUGUGGCUGCGGCCGAUUAGGUAGAUGGUGGGGUAACGGCCCACCAUGCCGAUAAUCGGUACAGGUUGUGAGAGCAAGAGCCUGGAGACGGUAUCUGAGACAAGAUACCGGGCCCUACGGGGCGCAGCAGGCGCGAAACCUUUACACUGCACGACAGUGCGAUAGGGGGACUCCGAGUGUGG----------AUAU------ACCUCGCUUUUCUGUACCGU--------------AAGG-------------------GGUACAGGAACAAGGACUGGCAAGACCGGUGCCAGCCGCCGCGGUAAUACCGGCAGUCCGAGUGAUGGCCGAUAUUAUUGGGCCUAAAGCGUCCGUAGCUUGCUGUGUAAGUCCAUUGGGAAAUCCACCAGCUCAACUGGUCGGCGUCCGGGGAAACUACACAGCUUGGGGCCGAGAGACUCAACGGGUACGUCCGGGGUAGGAGUGAAAUCCUGUAAUCCUGGACGGACCACCAAUGGGGAAACCACGUUGAGAGACCGGACCCGACAGUGAGGGACGAAAGCCAGGGUCUCGAACCGGAUUAGAUACCCGGGUAGUCCUGGCUGUAAACAAUGCUCGCUAGGUAU-GUCAGCCAUGCACGUAAUGUGCCGUAGUGAAGACGAUAAGCGAGCCGCCUGGGAAGUACGUCCGCAAGGAUGAAACUUAAAGGAAUUGGCGGGGGAGCACACAACCCGAGGAGCCUGCGGUUUAAUUGGACUCAACGCCGGACAUCUCACCGGUCC-CGACAU-UAA--------------UGAC-------------U--CC--ACUGAGAGGAGGUGCAUGGCCGCCGUCAGCUCGUACCGUGAGGCGUCCUGUUAAGUCAGGCAACGAGCGAGACCCACACUUCUAGUUGCCAGCAACCUG-UUGGGUACACUAGGAGGACUGCCAUUGCUAAAAUGGAGGAAGGAAUGGGCAACGGUAGGUCAGUAUGCCCCGAAUGGACCGGGCAACACGCGGGCUACAAUGGCUAUGACAGUGGGAUGCAACGCCGAAAGGCGACGCUAAUCUC-CAAACGUAGUCGUAGUUCGGAUUUCGGGCUGAAACCCGCCCGCAUGAAGCUGGAUUCGGUAGUAAUCGCGUGUCAGAAGCGCGCGGUGAAUACGUCCCUGCUCCUUGCACACACCGCCCGUUAAAGCACCCGAGUGGGGUCCGGAUGA-GGCCGUC--------AUGC-------ACGGUUGAAUCUGG-CUCCGCAAGGGGGCUUAAGUCGUAACAAGGUAGCCGUAGAGGAAUCUGCGGCUGGAUCACCUCCU--
ar.Hal.sac  -----AUUCCGGUUGAUCCUGCCGGAGGC-CAUUGCUAUUGGGAUUCAUUAGCCAUGCUAGUCGCACGAGU-----------UCAG-----------ACUCGUGGCGAAUAGCUCAGUAACACGUGGCAAACUACCCUUCGGAGCUCCAUACCCUCGGGAAACUGAGGCUAAUAGAGCAUA-CCACGGAAGUGCAAAGC--------UCCG------GCGCCGAAGGAUGUGGCUGCGGCCGAUUAGGUAGACGGUGGGGUAACGGCCCACCGUGCCAAUAAUCGGUACGGGUCAUGAGAGUGAGAACCCGGAGACGGAAUCUGAGACAAGAUUCCGGGCCCUACGGGGCGCAGCAGGCGCGAAACCUUUACACUGCACGACAGUGCGAUAAGGGAAUCCCAAGUGCGG----------AUAG------ACUGCGCUUUUGUACACCGU--------------AGGG-------------------GGUGUACGAAUAAGGGCUGGCAAGACCGGUGCCAGCCGCCGCGGUAAUACCGGCAGCCCGAGUGAUGGCCGAUCUUAUUGGGCCUAAAGCGUCCGUAGCUGGCCGCACAAGUCCAUCGGAAAAUCCACCCGCUCAACGGGUGGGCGUCCGGGGAAACUGUGUGGCUUGGGACCGGAAGGCGCGACGGGUACGUCCGGGGUAGGAGUGAAAUCCCGUAAUCCUGGACGGACCGCCGAUGGCGAAAGCACGUCGCGAGGACGGAUCCGACAGUGAGGGACGAAAGCCAGGGUCUCGAACCGGAUUAGAUACCCGGGUAGUCCUGGCCGUAAACAAUGUCUGUUAGGUGU-GGCUCCUACGUGGGUGCUGUGCCGUAGGGAAGCCGCUAAACAGACCGCCUGGGAAGUACGUCCGCAAGGAUGAAACUUAAAGGAAUUGGCGGGGGAGCAUACAACCGGAGGAGCCUGCGGUUUAAUUGGACUCAACGCCGGACAUCUCACCAGCAU-CGACUU-UAA--------------UGAU-------------C--CU--UCAGAGAGGAGGUGCAUGGCCGCCGUCAGCUCGUACCGUGAGGCGUCCUGUUAAGUCAGGCAACGAGCGAGACCCGCAUCCUUACUUGCCAGCAGGCGA-CUGGGGACAGUAGGGAGACCGCCGUGGCCAACACGGAGGAAGGAACGGGCAACGGUAGGUCAGUAUGCCCCGAAUGUGCUGGGCAACACGCGGGCUACAAUGGUCGAGACAAAGGGUUCCAACUCCGAAAGGAGACGGUAAUCUCAGAAACUCGAUCGUAGUUCGGAUUGUGGGCUGCAACUCGCCCACAUGAAGCUGGAUUCGGUAGUAAUCGCGUGUCACAAGCGCGCGGUGAAUACGUCCCUGCUCCUUGCACACACCGCCCGUCAAAGCACCCGAGUGAGGUCCGGAUGA-GGCGCGU---------UCC-------CGCGUCGAAUCUGG-CUUCGCAAGGGGGCUUAAGUCGUAACAAGGUAGCCGUAGGGGAAUCUGCGGCUGGAUCACCUCCU--
ar.Nat.mag  -----AUUCCGGUUGAUCCUGCCGGAGGU-CAUUGCUAUUGGAGUCCAUUAGCCAUGCUAGUUGUACGAGU-----------UUAG-----------ACUCGUAGCAGAUAGCUCAGUAACACGUGGCAAACUACCCUAUGGAUCCAGACAACCUCGG-AAACUGAGGCUAAUCUGGAAUA-CGACGGAGGUCGAAAGC--------UCCG------GCGCCAUAGGAUGUGGC--CGGCCGAUUAGGUAGACGGUGGGGUAACGGCCCACCGUGCCGAUAAUCGGUACGGGUUGUGAGAGCAAGAGCCCGGAGACGGUAUCUGAGACAAGAUACCGGGCCCUACGGGGCGCAGCAGGCGCGAAACCUUUACACUGCACGCCAGUGCGAUAAGGGGACUCCAAGUGCGG----------AUAU------ACCUCGCUUUUUGCGACCGU--------------AAGG-------------------GGUCGCGGAAUAAGUGCUGGCAAGACCGGU-CCAGCCGCCGCGGUAA-ACCGGCAGCACGAG--AUGACCGGUGUUAUUGGGCCUAAAGCGUCCGUAGCUGGCCGCGCAAGUCUAUCGGGAAAUCUCUUCGCUUAACGGAGAGGCGUCCGGGGAAAA-AUGUGGCUUGGGACCGGAAGACCAGAGGGGUACGUCUGGGGUAGGAGUGAAAUCCCGUAAUCCUGGACGGACCACCGGUGGCGAAAGCGCCUCUGGAAGACGGAUCCGACGGUGAGGGACGAAAGCUCGGGUCACGAACCGGAUUAGAUACCCGGGUAGUCCGAGCUGUAAACGAUGUCUGCUAGGUGU-GACAAGUACGCCUGUGUUGUGCCGUAGGGAAGCCGUGAAGCAGACCGCCUGGGAAGUACGUCCGCAAGGAUGAAACUUAAAGGAAUUGGCGGGGGAGCAUACAACCGGAGGAGCCUGCGGUUUAAUUGGACUCAACGCCGGACAUCUCACCAGCAC-CGACAU-UAA--------------GGAU-------------C-UGC--AUUGAGAGGAGGUGCAUGGCCGCCGUCAGCUCGUACCGUGAGGCGUCCUGUUAAGUCAGGCAACGAGCGAGACCCGCUCUCCUAAUUGCCAGCAACUUU-UUGGGUACAUUAGGAGGACUGCCGCUGCCAAAGCGGAGGAAGGAACGGGCAACGGUAGGUCAGUAUGCCCCGAAUGUGCUGGGCAACACGCGGGCUACAAUGGCCACGACAGUGGGAUGCAACGCCGAAAGGCGACGCUAAUCUCCUAAACGUGGUCGUACUUCGGAUUGAGGGCUGAAACUCGCCCUCAUGAAGCUGGAUUCGGUAGUAAUCGCGCCUCAGAAGGGCGCGGUGAAUACGUCCCUGCUCCUUGCACACACCGCCCGUCAAAGCACCCGAGUGGGGUCCGGAUGA-GGCCCGG--------UUUA-------CGGGUCGAAUCUGG-CUCCGCAAGGGGGCUUAAGUCGUAACAAGGUAGCCGUAGGGGAAUCUGCGGCUGGAUCACCUCCA--
ar.Met.fer  CGCCAACUCCGUUUGAUCCUGGCGGAGGC-CACUGCUAUGGGGGUCCAUAAGCCAUGCAAGUCGAACGGGC-----------CUUG----------UGCCCGUGGCGAACGGCUCAGUAACACGUGGAAACCUACCCUGGGGUCCGGGAUAACCCCGGGAAACUGGGGCUAAUCCCGGAUA-GGCGGGAACGCGAAAGGUC------UUUU----GACCGCCCCAGGAUGGGUCUGCGGCCGAUUAGGUAGUUGGUAGGGUAACGGCCUACCAAGCCUACGAUCGGUACGGGUUGUGAGAGCAAGAGCCCGGAGACGGGGCCUGAGACAAGGCCCCGGGCCCUACGGGGCGCAGCAGGCGCGAAAACUCCGCAAUGCGCGAAAGCGCGACGGGGGGACCCCCAGUGCCU----------GUAA-------AGUGGCUUUUCCGGAGUGU--------------AAAA-------------------GCUCCGGGAAUAAGGGCUGGCAAGACCGGUGCCAGCCGCCGCGGUAACACCGGCAGCCCGAGUGGUGGCCGCGUUUAUUGGGCCUAAAGCGUCCGUAGCCGGUCCGGUAAGUCUCCGGUGAAAGCCCGCAGCUCAACUGCGGGAGUAGCCGAGAUACUGCCGGACUUGGGGCCGGGAGAGGCCGGAGGUACCCCCGGGGUAGGGGUGAAAUCCUGUAAUCCCGGGGGGACCACCUGUGGCGAAGGCGUCCGGCUGGAACGGGCCCGACGGUGAGGGACGAAAGCCAGGGGAGCGAACCGGAUUAGAUACCCGGGUAGUCCUGGCCGUAAACGAUGCGGACUUGGUGUUGGGGCACUCGUUGCCCCAGUGCCGAAGGGAAGCCGUUAAGUCCGCCGCCUGGGGAGUACGGCCGCAAGGCUGAAACUUAAAGGAAUUGGCGGGGGAGCACACAACGCGUGGAGCCUGCGGUUUAAUUGGAUUCAACGCCGGACACCUCACCGGGGG-CGACGC-GA---------------UGAU-------------C--UA--GCCGAGAGGAGGUGCAUGGCCGCCGUCAGCUCGUACCGUGAGGCGUCCUGUUAAGUCAGGCAACGAGCGAGACCCGCGCCCCUAGUUGCCAGCGGGUAA-CCGGGCACACUAGGGGGACCGCCAGCGAUAAGCUGGAGGAAGGUGCGGGCGACGGUAGGUCCGUAUGCCCCGAAACCCCCGGGCUACACGCGGGCUACAAUGGCCGGGACAAUGGGUACCGACCCCGAAAGGGGGAGGUAAUCCCAUAAACCCGGCCGUAGUUCGGAUCGAGGGCUGCAACUCGCCCUCGUGAAGCUGGAAUGCGUAGUAAUCGCGGGUCACUAUCCCGCGGUGAAUACGUCCCUGCUCCUUGCACACACCGCCCGUCACGCCACCCAAACGGGGUUCGGAUGA-GGCCAUGCC------UCU-------GAUGGUCGAAUCCGGGCCCCGUGAGGAGGGCGAAGUCGUAACAAGGUAGCCGUAGGGGAACCUGCGGCUGGAUCACmmmmm--
ar.Met.jan  -----AUUCCGGUUGAUCCUGCCGGAGGC-CACUGCUAUCGGGGUCCAUAAGCCAUGCGAGUCA-AGGGGCUCC--------UUCG--------GGGAGCACCGGCGCACGGCUCAGUAACACGUGGCAACCUACCCUCGGGUGGGGGAUAACCUCGGGAAACUGAGGCUAAUCCCCCAUA-GGGGGGAACCCGAAAGGC-------GUAA-----GCUGCCCGAGGAUGGGGCUGCGGCGGAUUAGGUAGUUGGUGGGGUAACGGCCCACCAAGCCUACGAUCCGUACGGGCCCUGAGAGGGGGAGCCCGGAGAUGGACACUGAGACACGGGUCCAGGCC-UACGGGGCGCAGCAGGCGCGAAACCUCCGCAAUGCGCGAAAGCGCGACGGGGGGACCCCGAGUGCCC----------CCU--------GUGGGCUUUUCCGGAGUGU--------------AAAC-------------------GCUCCGGGAAUAAGGGCUGGCAAGUCCGGUGCCAGCAGCCGCGGUAAUACCGGCGGCCCAAGUGGUGGCCACUGUUAUUGGGCCUAAAGCGUCCGUAGCCGGCCCGGUAAGUCUCUGCUUAAAU-CUGCGGCUCAACCGCAGGGCUGGCA-AGAUACUGCCGGGCUUGGGACCGGGAGAGGCCGGGGGUACCCCAGGGGUAGCGGUGAAAUGCGUUGAUCCCUGGGGGACCACCUGUGGCGAAGGCGCCCGGCUGGAACGGGUCCGACGGUGAGGGACGAAGGCCAGGGGAGCAAACCGGAUUAGAUACCCGGGUAGUCCUGGCUGUAAACUCUGCGGACUAGGUGUCGCGUCGUUCGCCGACGCGGUGCCGAAGGGAAGCCGUUAAGUCCGCCGCCUGGGGAGUACGGUCGCAAGACUGAAACUUAAAGGAAUUGGCGGGGGAGCAUACAACGGGUGGAGCCUGCGGUUUAAUUGGAUUCAACGCCGGGCAUCUUACCAGGGG-CGACGC-GA---------------UGAC-------------C--AC--GCCGAGAGGUGGUGCAUGGCCGUCGUCAGCUCGUACCGUGAGGCGUCCUGUUAAGUCAGGUAACGAGCGAGACCCGUGCCCCAUGUUGCUACCUCUCCG-GAGGGCACUCAUGGGGGACCGCCGGCGCUAAGCCGGAGGAAGGUGCGGGCAACGACAGGUCCGCAUGCCCCGAAUCCCCUGGGCUACACGCGGGCUACAAUGGCCGGGACAAUGGGACGCGACCCCGAAAGGGGGAGCGAAUCCCCUAAACCCGGUCGUAGUCCGGAUCGAGGGCUGUAACUCGCCCUCGUGAAGCCGGAAUCCGUAGUAAUCGCGCCUCACCAUGGCGCGGUGAAUGCGUCCCUGCUCCUUGCACACACCGCCCGUCACGCCACCCGAGUUGAGCCCAAGUGA-GGCCCUGCC------GCAA------GAGGGUCGAACUUGGGUUCAGCGAGGGGGGCGAAGUCGUAACAAGGUAGCCGUAGGGGAA-CUGCGGCUGGAUCACCUCCm--
ar.Met.ore  -----NUUCUGGUUGAUCCUGCCAGAGGU-CACUGCUAUCAGUGUUCAUAAGCCAUGCGAGUCAAAUGUUC-----------UUCG----------UGAACAUGGCGUACUGCUCAGUAACNNGUGGAAAUCUGCCCUAAGGUCUGGCAUAACRCCGGGAAACUGAUGAUAAUUCCAGAUG-GACCGGAAGGUAAAAGC--------UCCG------GCGCCUUAGGAUGAAUCUGCGGCCUAUCAGGUUGUAGUGGGUGUAACGUACCUACUAGCCGACGACGGGUACGGGNUGUGAGAGCAAGAGCCCGGAGAUGGAUUCUGAGACAUGAAUCCAGGCCCUACGGGGCGCAGCAGGCGCGAAAACUUUACAAUGCGGGAAACCGCGAUAAGGGGACACUGAGUGCCC----------CUU--------GUUGGC-UGUCCCAUGUAU--------------AAAU-------------------GCAUGUGUUACAAGGGCCGGCAAGACCGGUGCCAGCCGCCGCGGUAACACCGGCGGCCCGAGUGGUGGCCACUAUUAUUGGGUCUAAAGGGUCCGUAGCCGGUUUGAUCAGUCUUCCGGGAAAUCUGACAGCUCAACUGUUAGGCUUCCGGGGAUACUGUCAGGCUUGGGACCGGGAGAGGUAAGAGGUACUACAGGGGUAGGAGUGAAAUCUUGUAAUCCCUGUGGGACCACCAGUGGCGAAGGCGUCUUACCAGAACGGGUCCGACGGUGAGGGACGAAAGCUGGGGGCACGAACCGGAUUAGAUACCCGGGUAGUCCCAGCCGUAAACGAUGCUCGCUAGGUGUCUGGGAUUGCGCGUUUCAGGUGCCGCAGGGAAGCCGUGAAGCGAGCCACCUGGGAAGUACGGCCGCAAGGCUGAAACUUAAAGGAAUUGGCGGGGGAGCAUACAACGGGUGGAGCCUGCGGUUUAAUUGGNNNNAACGCCGGAAAACUCACCUGGGG-CGACAC-UA---------------CGAA-------------C--UC--GCUGAGAGGANGUGCAUGGCCGUCGUCAGUUCGUACUGUGAAGCAUCCUGUUAAGUCAGGCAACGAGCGAGACCCGUGCCCACUGUUGCCAGCAUUUCG-AUGGGUACUCUGUGGGNACCGCCGCUGCUAAAGCGGAGGAAGGUGCGGGCUACGGUAGGUCAGUAUGCCCCGAAUCUCCAGGGCUACACGCGGGCUACAAUGGACGGGACAAUGGGCUCCUACCCCGAAAGGGGNUGGCAAUCUCACAAACCCGGCCGUAGUUCGGAUCGAGGGCUGUAACUCGCCCUCGUGAAGCUGGAAUCCGUAGUAAUCGCGUUUCAAUAUAGCGCGGUGAAUACGUCCCUGCUCCUUGCACACACCGCCCGNCAAACCACCCGAGUGAGGUAUGGGUGA-GGGCACAAC------AAA------GUGUGUUCGAACCUAAAUUUCGCAAGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm--
ar.The.chi  -----mmmmmmmmmmmmmmmmmmGGAGGC-CACUGCUAUGGGGGUCCAUAAGCCAUGCGAGUCAAGGGGGCGUCC-------UUCU------GGGAC-CCACCGGCGGACGGCUCAGUAACACGUCGGAACCUACCCUCGGGAGGGGGAUAACCCCGGGAAACUGGGGCUAAUCCCCCAUA-GGCCGGAAGGCGAAAGGGACC----GUAA--GGUCCCGCCCGAGGAUGGGCCGGCGGCCGAUUAGGUAGUUGGUGGGGUAACGGCCCACCAAGCCGAAGAUCGGUACGGGCCGUGAGAGCGGGAGCCCGGAGAUGGACACUGAGACACGGGUCCAGGNCCUACGGGGCGCAGCAGGCGCGAAACCUCCGCAAUGCGGGAAACCGCGACGGGGGGACCCCCAGUGCCUC---------UCU--------CACGGCUUUUCCGGAGUGU--------------AAAA-------------------GCUCCGGGAAUAAGGGCUGGCNAGGCCGGUGNNAGCCGCCGCGGUAAUACCGGCGGCCCGAGUGGUGGCCACUAUUAUUGGGCCUAAAGCGGCCGUAGCCGGGCCCGUAAGUCCCUGGCGAAAUCCCACGGCUCAACCGUGGGGCUCGCUGGGAUACUGCGGGCCUUGGGACCGGGAGAGGCCGGGGGUACCCCCGGGGUAGGGGUGAAAUCCUAUAAUCCCGGGGGGACCGCCAGUGGCGAAGGCGCCCGGCUGGAACGGGUCCGACGGUGAGGGCCGAAGGCCAGGGGAGCGAACCGGAUUAGAUACCCGGGUAGUCCUGGCUGUAAAGGAUGCGGGCUAGGUGUCGGGCNAUUCG-NCGCCCGGUGCCGUAGGGAAGCCGUUAAGCCCGCCGCCUGGGGAGUACGGCCGCAAGGCUGAAACUUAAAGGAAUUGGCGGGGGAGCAUACAAGGGGUGGAGCGUGCGGUUUAAUUGGAUUCAACGCCGGGAACCUCACCGGGGG-CGACGC-GA---------------CGAA-------------C--GC--GCCGAGAGGAGGUGCAUGGNCGCCGUCAGCUCGUACCGUGAGGCGUCCACUUAAGUGUGGUAACGAGCGAGACCCGCGCCCCCAGUUGCCAGUNCGCU--GGAGGCACUCUGGGGGGACUGCCGGCGAUAAGCCGGAGGAAGGGGCGGGCGACGGUAGGUCAGUAUGCCCCGAAACCCCCGGGCUACACGCGCGCUACAAUGGGCGGGACAAUGGGACCCGACCCCGAAAGGGGAAGGGAAUCCCCUAAACCCGCCCUCAGUUCGGAUCGCGGGCUGCAACUCGCCCGCGUGAAGCUGGAAUCCCUAGUACCCGCGCGUCAUCAUCGCGCGGCGAAUACGUCCCUGCUCCUUGCACACACCGCCCGUCACUCCACCCGAGCGGGGCCCGGGUGA-GGCCCGACUC-----UUCG----GGACGGGUCGAGCCUGGGCUCCGUGAGGGGGGAGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm--
ar.The.ste  -----mmmmmmmmmmmmmmmmmmmmmmmm-mmmmmmmmmmGAGGUCCAUAACCCAUGCGAGUCAUGGGGCGCG---------UCUG---------CGCGCACCGGCGGACGGCUCAGUAACACGUCGGAACCUACCCUCGGGAGGGGGAUAACCCCGGGAAACUGGGGCUAAUCCCCCAUA-GGCCGGAAGGCGAAAGGUC-------UCU----GACCGCCCGAGGAUGGGCCGGCGGCCGAUUAGGUAGUUGGUGGGGUAACGGCCCACCAAGCCGAAGAUCGGUACGGGCCAUGAGAGUGGGAGCCCGGAGAUGGACACUGAGACACGGGUCCAGGCCCUACGGGGCGCAGCAGGCGCGAAACCUCCGCAAUGCGGGCAACCGCGACGGGGGGACCCCCAGUGCCG----------AUA--------CACGGCUUUUCCGGAGUGU--------------AAAA-------------------GCUCCGGGAAUAAGGGCUGGCAAGGCCGGUGGCAGCCGCCGCGGUAAUACCGGCGGCCCGAGUGGUGGCCGCUAUUAUUGGGCCUAAAGCGUCCGUAGCCGGGCCCGUAAGUCCCUGGCGAAAUCCCACGGCUCAACCGUGGGGCUUGCUGGGAUACUGCGGGCCUUGGGACCGGGAGAGGCCGGGGGUACCCCUGGGGUAGGGGUGAAAUCCUAUAAUCCCAGGGGGACCGCCAGUGGCGAAGGCGCCCGGCUGGAACGGGUCCGACGGUGAGGGACGAAGGCCAGGGGAGCGAACCGGAUUAGAUACCCGGGUAGUCCUGGCUGUAAAGGAUGCGGGCUAGGUGUCGGGCGAUUCGCUCGCCCGGUGCCGGAGGGAAGCCGUUAAGCCCGCCGCCUGGGGAGUACGGCCGCAAGGCUGAAACUUAAAGGAAUUGGCGGGGGAGCAUACAAGGGGUGGAGCGUGCGGUUUAAUUGGAUUCAACGCCGGGAACCUCACCGGGGG-CGACGC-GA---------------CGAA-------------C--GC--GCCGAGAGGAGGUGCAUGGCCGCCGUCAGCUCGUACCGUGAGGCGUCCACUUAAGUGUGGUAACGAGCGAGACCCGCGCCCCCAGUUGCCAGUCUGCU--GGAGGCACUCUGGGGGGACCGCCGGCGAUAAGCCGGAGGAAGGAGCGGGCGACGGUAGGUCAGUAUGCCCCGAAACCCCCGGGCUACACGCGCGCUACAAUGGGCGGGACAAUGGGAUCCGACCCCGAAAGGGGAAGGGAAUCCCCUAAACCCGCCCUCAGUUCGGAUUGCGGGCUGCAACUCGCCCGCAUGAAGCUGGAAUCCCUAGUACCCGCGUGUCAUCAUCGCGCGGCGAAUACGUCCCUGCUCCUUGCACACACCGCCCGUCACUCCACCCGAGCGGGGUCCGGAUGA-GGCCUGGCUCC----UUCG---GGGACGGGUCGAGUCUGGGCUCCGUGAGGGGGGAGAAGUCGUAACAAGGUAGCCGUAGGGGAACCUACGGCUCGAUCACCUCCUAU
hi.Act.eur  -----mmmmmmmmmmmmCCUGGCUCAGGACGAACGCUGGCGGCGUGCUUAACACAUGCAAGUCGAACGGGAUCCAAAGGC--UUUU-UGUUUUUGGUGAGAGUGGCGAACGGGUGAGUAACACGUGAGAACCUGCCCCCUUCUUUUGGAUAACCGCAUGAAAGUGUGGCUAAUACAGGAUAUUCCAGCAUUGGGAAAGGU-------UUGG-----UCUGGUGGGGGAUGGGCUCGCGGCCUAUCAGCUUGUUGGUGGGGUGAUGGCCUACCAAGGCGGUGACGGGUAGCCGGCCUGAGAGGGUGGGCGGUCACACUGGGACUGAGAUACGNCCCAGANUCCUACGGGAGGCAGCAGUGGGGGAUUUUGCACAAUGGGCGCAAGCCUGAUGCAGCGACGCCGCGUGAGGGAUGAAGGCCUUCGGGUUGUAAACCUCUUUCGCUGGGUUGAAAGGCCAUGCU---UUG----GGUGUGGUUGAUUUGAACUGGUAAAGAAGUACCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGUACUAGCGUUGUCCGGAUUUAUUGGGCGUAAAGGGCUUGUAGGUGGUUUGUCGCGUCUGUCGUGAAAUCCUGUGGCUUAACCAUGGGCUUGCGGUGGGUACGGGCAGGCUUGAGUGCGGUAGGGGAGACUGGAAUUCCUGGUGUAGCGGUGGAAUGCGCAGAUAUCAGGAGGAACACCGGUGGCGAAGGCGGGUCUCUGGGCCGUUACUGACACUGAGGAGCGAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCAUGCUGUAAACGUUGGGCACUAGGUGUGGGGGCCGUGU-GGUUUCUGCGCCGUAGCUAACGCAUUAAGUGUCCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGGAAUUGACGGGGACCCGCACAAGCGGCGGAGCAUGCGGAUUAAUUCGAUGCAACGCGAAGAACCUUACCAAGGCUUGACAUGACCGGUACAGUGUAGAGAUAUGCUGGCC--UUUU-UGGCUGGUGUGCAGGUGGUGCAUGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCUCGUGUUGCCAGCAUUUGG-UUGGGGACUCUCGAGAGACUGCCGGGGUUAACUCGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGUCUUGGGCUUCACGCAUGCUACAAUGGAUGGUACAGAGGGUUGCGAUACUGUGAGGUGGAGCUAAUCCCUUAAAGCUGUUCUCAGUUCGGAUCGUAGUCUGCAACUCGACUACGUGAAGGUGGAGUCGCUAGUAAUCGCAGAUCAGCACGCUGCGGUGAAUACGUUCUCGGGUCUUGUACACACCGCCCGUCACGUCACGAAAGUUGGUAACACCCGAAmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
hi.Bif.inf  UUUGUGGAGGGUUCGAUUCUGGCUCAGGAUGAACGCUGGCGGCGUGCUUNACNCNUGCNAGUCGAACGGGAUCCAUCGGGC-UUUG--CUUGGUGGUGAGAGUGGCGAACGGGNGAGUNAUGCGUGACGACCUGCCCCAUACACCGGAAUAGCUCCUGGAAACGGGUGGUAAUGCCGGAUGUUCCAGCAUUGGGAAAGU--------UUC-------GCGGUAUGGGAUGGGGUCGCGUCCUAUCAGCUUGACGGCGGGGUAACGGCCCACCGUGGCUUCGACGGGUAGCCGGCCUGAGAGGGCGACCGGCCACNUUGGGACUGAGAUNCGGCCNNGNCUCCUACGGGAGGCNGCNGUGGGNNAUAUUGCACNAUGGGCGCAAGCCUNAUGCAGCGACGCNGCGUGAGGGAUGGAGGCCUUCGGGUUGUNAACCUCUUUUNUCGGGGAGCAAGC---------GUGA-----------GUGAGUUUACCCNUUGAAUNAGCACCGGCUAACUACGUGCCAGCNGCCGCGGUAAUACGUAGGGUGCNAGCGUUAUCCGGAAUUAUUGGGCGUNAAGGGCUCGUAGGCGGUUCGUCGCGUCCGGUGUGAAAGUCCAUCGCUUAACGGUGNAUCCGCGCCGNGUACGGGCGNGCUUGAGUGCGGUAGGGGAGACUGGAAUUCCCGGUGUAACGGUGGAAUGUGUAGAUAUCGGGAAGAACACCAAUGGCGAAGGCAGGUCUCUGGGCUGUUACUGACGCUGAGGAGCNAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUNAACGGUGGAUGCUGGAUGUGGGGCCNUUCCCGGGUUCCGUGUCGGAGCUAACGCGUUAAGCAUCCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGAAAUUGACGGGGGCCNGCACAAGCGGCGGAGCAUGCGGAUUNAUUCGANNNAACGCGAAGAACCUUACCUGGGCUUGACNUGUCCCGACGAUCCCAGAGAUGGNNNN-UCC-UUCG-GGNCGGGNUCACNGGUGGCGCAUGGUCGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGCCCCGUGUUGCCAGCGGUGUG-CCGGNNACUCACGGGNNACCGCCGGGGUUNACUCGGAGGAAGGUGGGGAUGACGUCAGAUCAUCAUGCCCCUUACGUCCAGGGCUUCACGCAUGCUACAAUGGCCGGUACAACGGGAUGCGACGCGGCGACGCGGAGCGGAUCCCUGAAAACCGGNCUCAGUUCGGAUCGCAGUCUGCAACUCGACUGCGUGAAGGCGGAGUCGCUAGUAAUCGCGAAUCAGCACGUCGCGGUGAAUGCGUUCCCNGGCNUUGUACACACNGCCCGUCAAGUCAUGAAAGUGGGCAGCACCCGAAGCCGGUGGCCUAACCCUUGGGANGGAGCCGUCUNAGGUGAGGCUCGUGANNGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
hi.Cel.cel  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmGGCGUGCUUAACACAUGCAAGUCGAACGGUGACGACGGGG--CUUG--CCCUGUCUGAUCAGUGGCGAACGGGUGAGUAACACGUGAGAACCUGCCCUUGACUCUGGGAUAACCGCGGGAAACGGCGGCUAAUACCGGAUAUGAGAGCAUUCUGAAAGA--------UUUA------UCGGUCAAGGAUGGGCUCGCGGCCUAUCAGCUUGUUGGUGGGGUGAUGGCCUACCAAGGCGACGACGGGUAGCCGGCCUGAGAGGGCGACCGGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGCGCAAGCCUGAUGCAGCGACGCCGCGUGAGGGAUGAAGGCCUUCGGGUUGUAAACCUCUUUCAGCAGGGAAGAAGC---------GAAA-----------GUGACGGUACCUGCAGAAGAAGCGCCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGCGCAAGCGUUGUCCGGAAUUAUUGGGCGUAAAGAGCUCGUAGGCGGUUUGUCGCGUCUGCUGUGAAAACCUGAGGCUCAACCUCGGGCUUGCAGUGGGUACGGGCAGACUAGAGUGCGGUAGGGGUGACUGGAAUUCCUGGUGUAGCGGUGGAAUGCGCAGAUAUCAGGAGGAACACCGAUGGCGAAGGCAGGUCACUGGGCCGCAACUGACGCUGAGGAGCGAAAGCAUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCAUGCCGUAAACGUUGGGCACUAGGUGUGGGGCUCUUCCCGAGUUCCGUGCCGCAGCAAACGCAUUAAGUGCCCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGAAAUUGACGGGGGCCCGCACAAGCGGCGGAGCAUGCGGAUUAAUUCGAUGCAACGCGAAGAACCUUACCAAGGCUUGACAUAACCGGAAAAGUGCAGAGAUGUGCUCCCC--GCAA--GGUCGGUGUACAGGUGGUGCAUGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUCCUAUGUUGCCAGCACCAUG-GUGGGGACUCAUAGGAGACUGCCGGGGUCAACUCGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGUCUUGGGCUUCACGCAUGCUACAAUGGCCGGUACAAAGGGCUGCGAUACCGCGAGGUGGAGCGAAUCCCAAAAAGCCGGUCUCAGUUCGGAUUGGGGUCUGCAACUCGACCCCAUGAAGUCGGAGUCGCUAGUAAUCGCAGAUCAGCACGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCAAGUCACGAAAGUCGGUAACACCCGAAGCCGGUGGCCCAACCCUUGGGAGGGAGCCGUCGAAGGUGGGACUGGCGAUUGGGACUAAGUCGUAACAAGGUAGCCGUmmmmmmmmmmmmmmmmmmmmmmmmm-----
hi.Cor.oti  -----mmmmmmmmmmmmmmmmmmmmmmmmCGAACGCUGNCGGCGUGCUUAACACAUGCAAGUCGAACGGAAAGGCCUACUUUCUUGAUUGCGGGUGCUCGAGUGGCGAACGGGUGAGUAACACGUGAGGAUCUGCCCCCAACUUGGGNAUAAGCCUGGGAAACUGGGUCUAAUUCCCGAUAGGACUUGGUGGUGAAAACGA------UUUU---CUAGUGGUUGGGGAUGAGCUCGCGGCCUAUCAGCUUGUUGGUGGGGUAAUGGCCUACCAAGGCGGCGACGGGUAGCCGGCCUGAGAGGGUGGACGGCCACAUUGGGACUGAGAUACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGCGGAAGCCUGAUGCAGCGACGCCGCGUGGGGGAUGACGGCCUUCGGGUUGUAAACUCCUUUCGACCGCGAGGAAGCCGCCUGG--UUGG----AAGGGUGGUGACGGUAGUGGUAGAAGAAGCACCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGUGCGAGCGUUGUCCGGAUUUACUGGGCGUAAAGAGCUCGUAGGUGGCUUGUCGCGUCGUCUGUGAAAGUCUGGGGCUUAACUCCGGGUGUGCAGGCGAUACGGGCUGGCUUGAGUGCUGUAGGGGAGACUGGAAUUCCUGGUGUAGCGGUGGAAUGCGCAGAUAUCAGGAGGAACACCGAUGGCGAAGGCAGGUCUCUGGGCAGUCACUGACGCUGAGGAGCGAGAGCAUGGGUAGCGAACAGGAUUAGAUACCCUGGUAGUCCAUGCUGUAAACGGUGGGCGCUAGGUGUGGGGACUUUCCUGGUUUCCGUGUCCUAGCUAACGCGUUAAGCGCCCCGCCUGGGGAGUACGNCCGCAAGGCUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGCGGAGCAUGUGGAUUAAUUCGAUGCAACGCGAAGAACCUUACCUGGGUUUGACAUGACUAGAUUAGGCGAGAGAUCGUCUGUCCC-UUUG-UGGCUGGUGUGCAGGUGGUGCAUGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGUAACCCUUGUCUUAUGUUGCCAGCACUGUG-GUGGGGACUCGUGAGAGACUGCCGGGGUGAACUCGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGUCCAGGGCUUCACACAUGCUACAAUGGCCGGUACAGUAGGUUGCGAGACCGUGAGGUGGAGCUAAUCCUGUAAAGCUGGUCGUAGUUCGGAUUGGGGUCUGCAACUCGACCCCAUGAAGUCGGAGUCGCUAGUAAUCGCAGAUCACCAUGCUGCGGUGAAUACGUUCUCGGGCCUUGUACACACCGCCCGUCACGUCAUGAAAGUUGGUAACACCCGAAGCCGNUUGCUCC-ACUUAGNUGGGGUGGUNUCGAAGGUGGGAUCGGCGAUmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
hi.Mic.par  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmGCAAGUCGAGCGGAAAGGCCC-----UUCG----GGGGUACUCGAGCGGCGAACGGGUGAGUAACACGUGAGAACCUGCCCCUGACUCUGGGAUAAGCCUGGGAAACCGGGUCUAAUACCGGAUAUGACAGCAUUGUGAAAGU--------UUUU------UCGGUUGGGGAUGGGCUCGCGGCCUAUCAGCUUGUUGGUGGGGUGAUGGCCUACCAAGGCGACGACGGGUAGCCGGCCUGAGAGGGCGACCGGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCGCAAUGGGCGGAAGCCUGAGCCAGCGACGCCGCGUGGGGGAUGACGGCCUUCGGGUUGUAAACCUCUUUCAGCAGGGACGAAGUU--------GACG----------------UGUACCUGUAGAAGAAGCGCCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGCGCGAGCGUUGUCCGGAAUUAUUGGGCGUAAAGAGCUCGUAGGUGGCUUGUUGCGUCUGCCGUGAAAGCCCGUGGCUUAACUACGGGUCUGCGGUGGAUACGGGCAGGCUAGAGGCUGGUAGGGGCAAGCGGAAUUCCUGGUGUAGCGGUGAAAUGCGCAGAUAUCAGGAGGAACACCGGUGGCGAAGGCGGCUUGCUGGGCCAGUUCUGACGCUGAGGAGCGAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCUGUAAACGUUGGGCGCUAGGUGUGGGGGUCUUCCCGAUUCCUGUGCCGUAGCUAACGCAUUAAGCGCCCCGCCUGGGGAGUACGGC-GCAAGGCUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGCGGAGCAUGUUGCUUAAUUCGAGCCAACGCGAAGAACCUUACCAAGGUUUGACAUAACCGGAAACACUCAGAGACGGGUGCCUCC-UUU--GGACUGGUGUACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUUCCAUGUUGCCAGCACUUUGGGUGGGGACUCAUGGGAGACUGCCGGGGUCAACUCGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGCCCCUUAUGUCUUGGGCUGCAAACAUGCUACAAUGGUCGGUACAGAGGGUUGCGAUACCGUGAGGUGGAGCGAAUCCCUAAAAGCCGGUCUCAGUUCGGAUUGGGGUCUGCAACUCGACCCCAUGAAGUCGGAGUCGCUAGUAAUCGCAGAUCAGCACGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACGUCACGAAAGUCGGCAACACCCGAAGCCCGUGGCCCAACCCUUGGGGGGGAGCGGUCGAAGGUGGGGCUGGCGAUUGGGACGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
hi.Mic.ech  -----mmmmmmmmUGAUCCUGGCGCAGGACGAACGCUGGCGGCGUGCUUAACACAUGCAAGUCGAGCGGAAAGGCCC-----UUCG-----GGGUACUCGAGCGGCGAACGGGUGAGUAACACGUGAGAACCUGCCCUAGGCUUUGGGAUAACCCCGGGAAACCGGGCCUAAUACCGAAUAGGACCGCAUGGUGAAAG---------UUUU------UCGGCCUGGGAUGGGCUCGCGGCCUAUCAGCUUGUUGGUGGGGUGAUGGCCUACCAAGCCGACGACGGGUAGCCGGCCUGAGAGGGCGACCGGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGCGGAAGCCUGAUGCAGCGACGCCGCGUGAGGGAUGACGGCCUUCGGGUUGUAAACCUCUUUCAGCAGGGACGAAGC---------GUAA-----------GUGACGGUACCUGCAGAAGAAGCGCCAGCCAACUACGUGCCAGCAGCCGCGGUAAGACGUAGGGCGCGAGCGUUGUCCGGAUUUAUUGGGCGUAAAGAGCUCGUAGGCGGCUUGUCGCGUCGACUGUGAAAACCCGCAGCUCAACUGCGGGCCUGCAGUCGAUACGGGCAGGCUAGAGUUCGGUAGGGGAGACUGGAAUUCCUGGUGUAGCGGUGAAAUGCGCAGAUAUCAGGAGGAACACCGGUGGCGAAGGCGGGUCUCUGGGCCGAUACUGACGCUGAGGAGCGAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCUGUAAACGUUGGGCGCUAGGUGUGGGGGGCCUCCGGUUCUCUGUGCCGCAGCUAACGCAUUAAGCGCCCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGCGGAGCAUGCGGAUUAAUUCGAUGCAACGCGAAGAACCUUACCUGGGUUUGACAUGCCGCAAAACCGGCAGAGAUGUCGGGUCC--UUCG--GGGGCGGUCACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUUCGAUGUUGCCAGCGCUAUG-GCGGGGACUCAUCGAAGACUGCCGGGGUCAACUCGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGCCCCUUAUGUCCAGGGCUUCACGCAUGCUACAAUGGCCGGUACAAUGGGCUGCGAUACCGUGAGGUGGAGCGAAUCCCAAAAAGCCGGUCUCAGUUCGGAUCGGGGUCUGCAACUCGACCCCGUGAAGUCGGAGUCGCUAGUAAUCGCAGAUCAGCACGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACGUCACGAAAGUCGGCAACACCCGAAGCCGGUGGCCCAACCCUUGGGAGGGAGCCGUCGAAGGUGGUGCUGGCGAUUGGGACGAAGUCGUAACAAGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
hi.Myc.cel  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmAACACAUGCAAGUCGAACGGAAAGGCCU-----UUNG----GGGGUGCUCGAGUGGCGAACGGGUGGGUAACACGUGGGGAUCUGCCCUGCACUUCGGGAUAAGCUUGGGAAACUGGGUCUAAUACCGGAUAGGACCGCAUGGUGAAAGC--------UUUU------GCGGUGUGGGAUGGGCCCGCGGCCUAUCAGCUUGUUGGUGGGGUGAUGGCCUACCAAGGCGACGACGGGUAGCCGGCCUGAGAGGGUGUCCGGCCACACUGGGACUGAGAUACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGCGCAAGCCUGAUGCAGCGACGCCGCGUGGGGGAUGACGGCCUUCGGGUUGUAAACCUCUUUCACCAUCGACGAAGCUGCCGG---UUUU-----CCGGUGGUGACGGUAGGUGGAGAAGAAGCACCGGCCAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGUGCGAGCGUUGUCCGGAAUUACUGGGCGUAAAGAGCUCGUAGGUGGUUUGUCGCGUUGUUCGUGAAAUUCCCUGGCUUAACUGGGGGCGUGCGGGCGAUACGGGCAGACUGGAGUACUGCAGGGGAGACUGGAAUUCCUGGUGUAGCGGUGGAAUGCGCAGAUAUCAGGAGGAACACCGGUGGCGAAGGCGGGUCUCUGGGCAGUAACUGACGCUGAGGAGCGAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGGUGGGUACUAGGUGUGGGUUUCUUCCUGGGAUCCGUGCCGUAGCUAACGCAUUAAGUACCCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGCGGAGCAUGUGGAUUAAUUCGAUGCAACGCGAAGAACCUUACCUGGGUUUGACAUGACAGGACGACUGCAGAGAUGUGGUUUCC--CUUG-UGGCCUGUGUGCAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCUCAUGUUGCCAGCGCGAUG-GCGGGGACUCGUGAGAGACUGCCGGGGUCAACUCGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGCCCCUUAUGUCCAGGGCUUCACACAUGCUACAAUGGCCGGUACAAAGGGCUGCGAUGCCGUGAGGUUUAGCGAAUCCUUUAAAGCCGGUCUCAGUUCGGAUCGGGGUCUGCAACUCGACCCCGUGAAGUCGGAGUCGCUAGUAAUCGCAGAUCAGCAUGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACGUCAUGAAAGUCGGUAACACCCGAAGCCAGUGGCCUAACCGCAA-GAGGGAGCUGUCGAAGGUGGGAUCGGCGAUUGGGACGAAGUCGUAACAAGGUAGCCGUACCGGAAGUGmmmmmmmmmmmmmmm-----
hi.Myc.int  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmGGCGGCGUGCUUAACACAUGCAAGUCGAACGGAAAGGCCCC----UUCG----GGGGUACUCGAGUGGCGAACGGGUGAGUAACACGUGGGAAUCUGCCCUGCACACUGGGAUAAGCCUGGGAAACUGGGUCUAAUACCGGAUAGGACCGCAUGGUGAAAGC--------UUUU------GCGGUGUGGGAUGGGCCCGCGGCCUAUCAGCUUGUUGGUGGGGUGACGGCCUACCAAGGCGACGACGGGUAGCCGGCCUGAGAGGGUGUCCGGCCACACUGGGACUGAGAUACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGCGCAAGCCUGAUGCAGCGACGCCGCGUGGGGGAUGACGGCCUUCGGGUUGUAAACCUCUUUCAGCAGGGACGAAGC---------GCAA-----------GUGACGGUACCUGCAGAAGAAGCACCGGCCAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGUGCGAGCGUUGUCCGGAAUUACUGGGCGUAAAGAGCUCGUAGGUGGUUUGUCGCGUUGUUCGUGAAAUCUCACGGCUUAACUGUGAGCGUGCGGGCGAUACGGGCAGACUAGAGUACUGCAGGGGAGACUGGAAUUCCUGGUGUAGCGGUGGAAUGCGCAGAUAUCAGGAGGAACACCGGUGGCGAAGGCGGGUCUCUGGGCAGUAACUGACGCUGAGGAGCGAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGGUGGGUACUAGGUGUGGGUUUCUUCCUGGGAUCCGUGCCGUAGCUAACGCAUUAAGUACCCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGCGGAGCAUGUGGAUUAAUUCGAUGCAACGCGAAGAACCUUACCUGGGUUUGACAUGACAGGACGCGUCUAGAGAUAGGCGUUCC--CUUG-UGGCCUGUGUGCAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCUCAUGUUGCCAGCGGAAUG-CCGGGGACUCGUGAGAGACUGCCGGGGUCAACUCGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGCCCCUUAUGUCCAGGGCUUCACACAUGCUACAAUGGCCGGUACAAAGGGCUGCGAUGCCGCGAGGUUAAGCGAAUCCUUUAAAGCCGGUCUCAGUUCGGAUCGGGGUCUGCAACUCGACCCCGUGAAGUCGGAGUCGCUAGUAAUCGCAGAUCAGCACGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACGUCAUGAAAGUCGGUAACACCCGAAGCCAGUGGCCUAACCUUUGGGAGGGAGCUGUCGAAGGUGGGAUCGGCGAUUGGGACGAAGUCGUAAmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
hi.Noc.fla  -----mmmmmmmmmmmmmmmmmmmmmmGACGAACGCUGGCGGCGUGCUUAACACAUGCAAGUCGAGCGGAAAGGCCC-----UUCG-----GGGUACUCGAGCGGCGAACGGGUGAGUAACACGUGAGAAUCUGCCCCAGGCUCUGGGAUAGCCACCGGAAACGGUGAUUAAUACCGGAUACGACAGCAUUGUGAAAG---------UUUU------UCGGCCUGGGAUGUGCUCGCGGCCUAUCAGCUUGUUGGUGAGGUAAUGGCUCACCAAGGCUUCGACGGGUAGCCGGCCUGAGAGGGUGACCGGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGCGGAAGCCUGAUCCAGCAACGCCGCGUGAGGGAUGACGGCCUUCGGGUUGUAAACCUCUUUCAGCACAGACGAAGC---------GCAA-----------GUGACGGUAUGUGCAGAAGAAGGACCGGCCAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGUCCGAGCGUUGUCCGGAAUUAUUGGGCGUAAAGGGCUCGUAGGCGGUCUGUCGCGUCGGGAGUGAAAACCAGGUGCUUAACACCUGGCCUGCUUUCGAUACGGGCAGACUAGAGGUACUCAGGGGAGAAUGGAAUUCCUGGUGUAGCGGUGAAAUGCGCAGAUAUCAGGAGGAACACCGGUGGCGAAGGCGGUUCUCUGGGAGUAUCCUGACGCUGAGGAGCGAAAGUGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACACCGUAAACGUUGGGCGCUAGGUGUGGGAUCCUUCCCGGGUUCCGUGCCGCAGCUAACGCAUUAAGCGCCCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGCGGAGCAUGCGGAUUAAUUCGAUGCAACGCGAAGAACCUUACCUGGGUUUGACAUAACCGGAAAGCUGCAGAGAUGUAGCCCCU--UUU---AGUCGGUGUACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUCCUAUGUUGCCAGCAAUUCGGUUGGGGACUCAUAGGAGACUGCCGGGGUCAACUCGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGCCCCUUAUGUCCAGGGCUUCACGCAUGCUACAAUGGCCGGUACAAAGGGCUGCGAUCCCGUGAGGGUGAGCGAAUCCCAAAAAGCCGGUCUCAGUUCGGAUUGGGGUCUGCAACUCGACCCCAUGAAGUCGGAGUCGCUAGUAAUCGCAGAUCAGCACGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACGUCACGAAAGUCGGCAACACCCGAAGCCAGUGGCCCAACCCUUGGGGGGGAGCUGUCGAAGGUGGGGCUGGCGAUUGGGACGAAGUCGUAACAAGGUAGCCGUACCGGAAGGUGCmmmmmmmmmmmmm-----
hi.Str.rim  -UCACGGAGAGUUUGAUCCUGGCUCAGGACGAACGCUGGCGGCGUGCUUAACACAUGCAAGUCGAACGAUGAAGCCC-----UUCG-----GGGUGGAUUAGUGGCGAACGGGUGAGUAACACGUGGGAAUCUGCCCUGCGCUCUGGGACAAGCCCUGGAAACGGGGUCUAAUACCGGAUAUGACAGCAUUGUGAAAGC--------UCCG------GCGGUGCAGGAUGAGCCCGCGGCCUAUCAGCUUGUUGGUUAGGUAAUG-CCUACCAAGGCGGCGACGGGUAGCCGGCCUGAGAGGGCGACCGGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGCGCAAGCCUGAUGCAGCGACGCCGCGUGAGGGAUGACGGCCUUCGGGUUGUAAACCUCUUUCAGCAGGGAAGAAGC---------GCAA-----------GUGACGGUACCUGCAGAAGAAGCGCCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGCGCAAGCGUUGUCCGGAAUUAUUGGGCGUAAAGAGCUCGUAGGCGGCUUGUCGCGUCGGAUGUGAAAGCCCGGGGCUUAACCCCGGGUCUGCAUUCGAUACGGGCAGGCUAGAGUUCGGUAGGGGAGAUCGGAAUUCCUGGUGUAGCGGUGAAAUGCGCAGAUAUCAGGAGGAACGCCGGUGGCGAAGGCGGAUCUCUGGGCCGAUACUGACGCUGAGGAGCGAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGUUGGGAACUAGGUGUGGGCGACUUCCCGUCGUCCGUGCCGCAGCUAACGCAUUAAUUGCCCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGCGGAGCAUGUGGCUUAAUUCGACGCAACGCGAAGAACCUUACCAAGGCUUGACAUAACCGGAAACCUCUGGAGACAGGGGCCCC--CUUG-UGGUCGGUGUACAGGUGGUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUG-GCUCUGUUGCCAGCAUUUCGGAUGGGGACUCACAGGAGACCGCCGGGGUCAACUCGGAGGAAGGUGGGGACGACCUCAAGUCAUCAUGCCCCUUAUGUCUUGGGCUGCACACGUGCUACAAUGGCCGGUACAAUGAGCUGCGAUACCGCGAGGUGGAGCGAAUCUCAAAAAGCCGGUCUCAGUUCGGAUUGGGGUCUGCAACUCGACCCCAUGAAGUCGGAGUCGCUAGUAAUCGCAGAUCAGCAUGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACGUCACGAAAGUCGGUAACACCCGAAGCCGGUGGCCCAACCCUUGGGAGGGAAUCGUCGAAGGUGGGACUGGCGAUUGGGACGAAGUCGUAACAAGGUAGCCGUACCGGAAGGUGCGGCUGGAUCACCUCCUUU
lo.Ach.lai  UAUAUGGAGAGUUUGAUCCUGGCUCAGGAUGAACGCUGGCGGCGUGCCUAAUACAUGCAAGUCGAACGAAGCAUC-------UUCG-------GAUGCUUAGUGGCGAACGGGUGAGUAACACGUAGAAACCUACCUUUAACUCGAGGAUAACUCCGGGAAACUGGAGCUAAUACUGGAUA-GGAUGAA-AUUUAAAGA--------UUUA------UCGGUUUAAGAGGGGUCUGCGGCGCAUUAGUUAGUUGGUGGGGUAAGAGCCUACCAAGACGAUGAAUCGUAGCCGGACUGAGAGGUCUACCGGCCACAUUGGGACUGAGA-ACGGCCCAAACUCCUACGGGAGGCAGCAGUAGGGAAUUUUCGGCAAUGGGGGAAACCCUGACCGAGCAACGCCGCGUGAACGACGAAGUACUUCGGUAUGUAAAGUUCUUUUAUAUGGGAAGAAAAAUU------AAA----------AAUUGACGGUACCAUAUGAAUAAGCCCCGGCUAACUAUGUGCCAGCAGCCGCGGUAAUACAUAGGGGGCGAGCGUUAUCCGGAUUUACUGGGCGUAAAGGGUGCGUAGGUGGUUAUAAAAGUUUGUGGUGUAAGUGCAGUGCUUAACGCUGUGA-GGCUAUGAAAACUAUAUAACUAGAGUGAGACAGAGGCAAGUGGAAUUCCAUGUGUAGCGGUAAAAUGCGUAAAUAUAUGGAGGAACACCAGUGGCGAAGGCGGCUUGCUGGGUCUAUACUGACACUGAUGCACGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAGAACUAAGUGUUGGCC--AUAA---GGUCAGUGCUGCAGUUAACGCAUUAAGUUCUCCGCCUGAGUAGUACGUACGCAAGUAUGAAACUCAAAGGAAUUGACGGGACCCCGCACAAGCGGUGGAUCAUGUUGUUUAAUUCGAAGAUACACGAAAAACCUUACCAGGUCUUGACAUACUCUGAAAGGCUUAGAAAUAAGUUC-GG--AGG---CUCAGAUGUACAGGUGGUGCACGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUAUUGCUAGUUACCAUCA-UUAAGUUGGGGACUCUAGCGAGACUGCCAGUGAUAAAUUGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGACCUGGGCUACAAACGUGAUACAAUGGCUGGAACAAAGAGAAGCGAUAGGGUGACCUGGAGCGAAACUCACAAAAACAGUCUCAGUUCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUCGGAAUCGCUAGUAAUCGCAAAUCAGCAUGUUGCGGUGAAUACGUUCUCGGGGUUUGUACACACCGCCCGUCAAACCACGAAAGUGGGCAAUACCCAACGCCGGUGGCCUAACCGAAAGGAGGGAGCCGUCUAAGGUAGGGUCCAUGAUUGGGGUUAAGUCGUAACAAGGUAUCCCUACGGmmmmmmmmmmmmmmmmmmmmm-----
lo.Bac.alc  -----mmmmmmmUUGAUCCUGGCUCAGGACGAACGCUGGCGGCGUGCCUAAUACAUGCAAGUCGAGCGGACCAAAGGGAG--CUUG--CUCCCAGAGGUUAGCGGCGGACGGGUGAGUAACACGUGGGAACCUGCCCUGUAGACUGGGAUAACAUCGAGAAAUCGGUGCUAAUACCGGAUA-AUCAACAUUGUAAAAGAUGGC----UCCG--GCUAUCACUAACAGAU-GGCCUGCGGCGCUUUAGCUAGUUGGUAAGGUAAUGGCUUACCAAGGCGACGAUGCGUAGCCGACCUGAGAGGGUGAUCGGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUAGGGAAUCUUCCGCAAUGGACGAAAGUCUGACGGAGCAACGCCGCGUGAGUGAUGAAGGUUUUCGGAUCGUAAAGCUCUGUUGUUAGGGAAGAACAAGUGCCGUUAAUA-GGUCGGCACCUUGACGGUACCUAACCAGAAAGCCACGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGUGGCAAGCGUUGUCCGGAAUUAUUGGGCGUAAAGCGCGCGCAGGCGGUCUUUUAAGUCUGAUGUGAAAUAUCGGGGCUCAACCCCGAGGGGUCAUUGGAAACUGGGA-GCUUGAGUACAGAAGAGGAGAGUGGAAUUCCNCGUGUNGCGGUGAAAUGCGUAGAUAUGUGGAGGAACACCAGUGGCGAAGGCGACUCUCUGGUCUGUAACUGACGCUGAGGCGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAGUGCUAGGUGUUAGGGGUUUCGUGCCCUUAGUGCCGAAGUUAACACAUUAAGCACUCCGCCUGGGGAGUACGGCCGCAAGGCUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCAGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGAAGAACCUUACCAGGUCUUGACAUCCUUUGACCACUCUAGAGAUNGAGC--UUUCUUCGGGGACAAAGUGACAGGUGGUGCAUGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGAUCUUAGUUGCCAGCA-UUUAGUUGGGCACUCUAAGGUGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAAUCAUNAUGCCCCUUAUGACCUNGGCUACACACGUGCUACAAUGGAUGGUACAAAGGGCAGCGAAACCGCGAGGUCGAGCCAAUCCCAUAAAGCCAUUCUCAGUUCGGAUUGUAGGCUGCAACUCGCCUACAUGAAGCCGGAAUUGCUAGUAAUCGCGGAUCAGCAUGCCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCACGAGAGUUUGUAACACCCmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
lo.Bac.sub  UUAUCGGAGAGUUUGAUCCUGGCUCAGGACGAACGCUGGCGGCGUGCCUAAUACAUGCAAGUCGAGCGGACAG-GUGGGAG-CUUG-CUCC---GAUGUUAGCGGCGGACGGGUGAGUAACACGUGGGAACCUGCCUGUAAGACUGGGAUAACUCCGGGAAACCGGGGCUAAUACCGGAUG-GUUGGCAUCAUAAAAGGUGGC----UUCG--GCUACCACUUACAGAUGGACCCGCGGCGCAUUAGCUAGUUGGUGAGGUAACGGCUCACCAAGGCAACGAUGCGUAGCCGACCUGAGAGGGUGAUCGGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUAGGGAAUCUUCCGCAAUGGACGAAAGUCUGACGGAGCAACGCCGCGUGAGUGAUGAAGGUUUUCGGAUCGUAAAGCUCUGUUGUUAGGGAAGAACAAGUACCGUUAACA-GGGCGGUACCUUGACGGUACCUAACCAGAAAGCCACGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGUGGCAAGCGUUUUCCGGAAUUAUUGGGCGUAAAGGGCUCGCAGGCGGUUUCUUAAGUCUGAUGUGAAAGCCCCCGGCUCAACCGGGGAGGGUCAUUGGAAACUGGGGAACUUGAGUGCAGAAGAGGAGAGUGGAAUUCCACGUGUAGCGGUGAAAUGCGUAGAGAUGUGGAGGAACACCAGUGGCGAAGGCGACUCUCUGGUCUGUAACUGACGCUGAGGAGCGAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAGUGCUAAGUGUUAGGGGGUUCCGCCCCUUAGUGCUGCAGCUAACGCAUUGAGCACUCCGCCUGGGGAGUACGGUCGCAAGACUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGAAGAACCUUACUAGGUCUUGACAUCCUCUGACAAUCCUAGAGAUAGGACGUCCC-UUCG-GGGCAGAGUGACAGGUGGUGCAUGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGAUCUUAGUUGCCAGCA-UUCAGUUGGGCACUCUAAGGUGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGACUUGGGCUACACACGUGCUACAAUGGACAGAACAAAGGGCAGCG-AACCGCGAGGUUAAGCCAAUCCCACAAAUCUGUUCUCAGUUCGGAUCGCAGUCUGCAACUCGACUGCGUGAAGCUGGAAUCGCUAGUAAUCGCGGAUCAGCAUGCCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCACGAGAGUUUGUAACACCCGAAGUCGGUGAGGUAACCUUU-GGAGCCAGCCGCCGAAGGUGGGACAGAUGAUUGGGGUGAAGUCGUAACAAGGUAGCCGUAUCGGAAGGUGCGGCUGGAUCACCUCCUUU
lo.Car.mob  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmGGACGGGUNAGUAACACGUGGGNACCNGCCCAUAAGUGGGGGAUAACAUUCGGAAACGGAUGCUAAUACCGCAUA-ACUCUCCUGAUAAAAGACGGU----UUCC--GCUGUCGCUNAUNGAUNGACCCGCGGCGUNUUAGCUNGUUGGUGAGGUAAUGGCUCACCAAGGCNAUGAUGCNUAGCCGACCUGAGAGGGUNAUCGGCNACACUGGGACUGAGACACGGCCNAGACUCCUACGGGAGGCAGCNGUAGGGAAUCUUCCGCAAUGGACGAAAGUCUGACGGAGCNAUGCCGCGUGAGUGAAGAAGGUUUUCGGAUCGUAAAACUNUGUUGUUAGAGAAGAACNAGGAUGAGCUAAC-UNCUCAUCCCCUGACGGUAUCUNACCAGAAAGCCAUGGCUAACUACGUGCCAGCAGCCGCGGUNAUACGUAGAUGGCNAGCGUUGUCCGGAUUUAUUGGGCGUNAAGCGAGCGCAGGCGGUUCUUUAAGUCUNAUGUGAAAGCCCCCAGCUCNACUGGGNAAGGUCAUUGGAAACUGGGGAACUUGAGUGCAGAAGAGGAGAGUGGAAUUCCACGUGUAGCGGUGAAAUGCGUAGAUAUGUGGAGGAACACCAGUGGCGAAGGCGACUCUCUNGUCUGUNACUGACGCUNAGGCUCGNAAGCGUNGGGAGCAAACAGGAUUAGAUACCCUNGUNGUCCACGCCGUAAACGAUGAGUGCUNAGUGUUGGAGGGUUCCGCCCUUCAGUGCUGCAGCUNACGCAUUAAGCACUNCGCCUNGGGAGUACGGCCGCAAGGCUGAAACUCAAAGGAAUUGACGGGGA-CCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGNAACGCGAAGAACCUUACCAGGUCUUGACAUCCUUUGACCACUCUAGAGAUAGAGCU-UUCCUUCGGGGACAAAGUGACAGGUGGNGCAUNGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUNAAGUCCCGCAACGAGCGCAACCCUNAUUACUAGUUGCCAGCA-UUCAGUUGGGCACUCUAGUGAGACUGCCGGUGAUAAACCGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGACCUNGGCUACACACGUGCUACAAUGGAUGGUACAACGAGUCGCAAGACCGCGAGGUCAAGCUAAUCUCUUAAAGCCAUUCUCNGUUCGGAUUGCAGGCUGCNACUCGCCUNCAUGAAGCCGGAAUCGCUAGUAAUCGCGGAUCNGAACGCCGCGGUNAAUACGUUCCCGGGUCUUGUACACACCGCCCGUCACACCACGAGAGUUUGUAACACCCGAAGUCGGUGAGGUNACCUUUUGGAGCCAGCCGCCUAAGGUNGGAUAGAUmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
lo.Eub.lac  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmCGGCGUGCUUNACMCRUGCGAGUCGAGCGAAGCACUUAGGWAAUUCGUUCCUAUUUGACUGAGCGGCGGACGGGUGAGUAACGCGUGGGAACCUCCCUCAUACAGGGGGAUAACAGUUAGAAAUGACUGCUAAUACCGCAUAAGACCGCAUGGUAAAAAC--------UCCG------GUGGUAUGAGAUGGACCCGCGUCUGAUUAGUUAGUUGGUGGGGUAACGGCCUACCAAGGCGACGAUCAGUAGCCGACCUGAGAGGGUGACCGGCCACAUUGGGACUGAGACACGGCCCAAACUCCUACGGGAGGCAGNAGNGGGGAAUAUUGCACAAUGGGGGANACCCUGAUGCAGCGACGCCGCGUGAGCGAAGAAGUAUUUCGGUAUGUAAAGCUCUAUCAGCAGGGAAG-------------AAAA------------UGACGGUACCUGACUAAGAAGCCCCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGGGCAAGCGUUAUCCGGAUUUACUGGGUGUAAAGGGAGCGUAGACGGAGCAGCAAGUCUGAUGUGAAAACCCGGGGCUCAACCCCGGGACUGCAUUGGAAACUGUUGAUCUGGAGUGCCGGAGAGGUAAGCGGAAUUCCUAGUGUAGCGGUGAAAUGCGUAGAUAUUAGGAGGAACACCAGUGGCGAAGGCGGCUUACUGGACGGUAACUGACGUUGAGGCUCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGACUACUAGGUGUCGGGUGGAAA-GCCAUUCGGUGCCGCAGC-AACGCAAUAAGUAGUCCACCUGGGGAGUACGUUCGCAAGAAUGAAACUCAAAGGAAUUGACGGGGACCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGAANAACCUUACCUGCUCUUGACAUCCGGUGACGGCAGAGUAAUGUCUGCUUUUC-UUCG-GAACCCGGUGACAGGUGGUGCAUGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUAUCUUCAGUAGCCAGCG-GUAAGCCGGGCACUCUGGAGAGACUGCCAGGGAUAACCUGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGAGCAGGGCUACACACGUGCUACAAUGGCGUAAACAAAGGGAAGCGAACCCGCGAGGGUGGGCAAAUCCCANAAAUAACGUCUCAGUUCGGAUUGUAGUCUGCAACUCGACUACAUGAAGCUGGAAUCGCUAGUAAUCGCGAAUCAGAAUGUCGCGGUGAAUACGUUCCCGGGUCUUGUACACACCGCCCGUCACACCAUGGGAGUCAGUAACGCCCGAAGUCAGUGACCCAACCGUmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
lo.Lac.car  UUAUAUGAGAGUUUGAUCCUGGCUCAGGACGAACGCUGGCGGCGUGCCUAAUACAUGCAAGUCGAACGCACGAAGUUGAAGACUUGUCUUUAACCAAGUGAGUGGCGGACGGGUGAGUAACACGUGGGAACCUGCCCAUUAGAGGGGGAUAACAUUCGGAAACGGAUGCUAAUACCGCAUA-GUUUGCAUAAGGAAAGGUGGC----UUCG--GCUACCACUAAUGGAUGGACCCNCGGCGUAUUAGCUAGUUGGUGAGGUAAUGGCUCACNAAGGCAAUGAUACGUAGCCGACCUGAGAGGGUGAUCGGCCACACUGGGACUGAGACACGGCNNNNACUCCUACGGGAGGCAGCAGUAGGGAAUCUUCCGCAAUGGACGAAAGUCUGACGGAGCAACGCCGCGUGAGUGAAGAAGGUUUUCGGAUCGUAAAACUCUNUUGUUAAAGAAGAACAAGGAUGAGAUAAC-UGCUCAUCCCUNNACGGUAUUUAACCAGAAAGCCACGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGUGGCAAGCGUUGUCCGGAUUUAUUGGGCGUAAAGCGAGCGCAGGCGGUUCUUUAAGUCUNAUGUGAAAGCCCNCNGCUCAACCGGGNAGGGUCAUUGGAAACUGGAGAACUUGAGUGCAGAAGAGGAGAGUGGAAUUCCACGUGUAGCGGUGAAAUGCGUAGAUAUGUGGAGGAACACCAGUGGCGAAGGCGACUCUCUNGUCUGUAACUGACGCUGAGGCUCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAGUGCUAAGUGUUGGAGGGUUCCGCCCUUCAGUGCUGCAGCUAACGCAUUAAGCACUCCGCCUGGGGAGUACGGCCGCAAGGCNNNAACUCAAAGGAAUUGNCGGGGACCNGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAANNAACGCGAAGAACCUUACCAGGUCUUGACAUCCUUUGACCACUCUAGAGAUAGAGCUUUCCCUUCGGGGACAAAGUGACAGGUGGNGCAUGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUAUUACUAGUUGCCAGCA-UUUAGUUGGGCACUCUAGUGAGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGACCUGGNCUACACACGUGCUACAAUGGAUGGUACAACGAGUCGCAAGGUCGCGAGGCCAAGCUAAUCUCUUAAAGCCAUUCUCAGUUCGGAUUGUAGGCUGCAACUCGCCUNCAUGAAGCCGGAAUCGCUAGUAAUCGCGGAUCAGAACGCCGCGGUGAAUACGUUCCCGGGUCUUGUACACACCGCNCGUCACACCACGAGAGUUUGUAACACCCGAAGCCGGUGAGGUAACCUUU-GGAGCCAGCCGUCUAAGGUGGGAUAGAUAAUUNGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
lo.Myc.cap  -AAAAUGAGAGUUUGAUCCUGGCUCAGGAUAAACGCUGGCGGCUGGCCUAAUACAUGCAAGUCGAACGGGGGUG--------CUUG--------CACCUCAGUGGCGAACGGGUGAGUAACACGUAUCAACU-ACCUUAUAGCGGGGGAUAACUUUUGGAAACGAAAGAUAAUACCGCAUGUAGAUGCAUAUCAAAAGAACC-----GUUU---GGUUCACUAUGAGAUGGGGAUGCGGCGUAUUAGCUAGUAGGUGAGAUAAUAGCCCACCUAGGCGAUGAUACGUAGCCGAACUGAGAGGUUGAUCGGCCACAUUGGGACUGAAAUACGGCCCAGACUCCUACGGGAGGCAGCAGUAGGGAAUUUUUCACAAUGGACGAAAGUCUGAUGAAGCAAUGCCGCGUGAGUGAUGACGGCCUUCGGGUUGUAAAGCUCUGUUGUAAGGGAAGAAAAAAUAGAGUAGAAA-UGACUUUAUCUUGACAGUACCUUACCAGAAAGCCACGGCUAACUAUGUGCCAGCAGCCGCGGUAAUACAUAGGUGGCAAGCGUUAUCCGGAUUUAUUGGGCGUAUAGGGUGCGUAGGCGGUUUUGCAAGUUUGAGGUUAAAGUCCGGAGCUCAACUCCGGUU-CGCCUUGAAGACUGUUUUACUAGAAUGCAAGAGAGGUAAGCGGAAUUCCAUGUGUAGCGGUGAAAUGCGUAGAUAUAUGGAAGAACACCUGUGGCGAAAGCGGCUUACUGGCUUGUUAUUGACGCUGAGGCACGAAAGCGUGGGGAGCAAAUAGGAUUAGAUACCCUAGUAGUCCACGCCGUAAACGAUGAGUACUAAGUGUUGGG---GUAA----CUCAGCGCUGCAGCUAACGCAUUAAGUACUCCGCCUGAGUAGUAUGCUCGCAAGAGUGAAACUCAAAGGAAUUGACGGGGACCCGCACAAGUGGUGGAGCAUGUGGUUUAAUUCGAAGCAACACGAAGAACCUUACCAGGGCUGACA-UCCAGUGAAAGCUAUAGAGAUAUAGU--AG--AGG---UUCAUUGAGACAGGUGGUGCAUGGUUGUCGUCAGUUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAACGCAACCCUUGUCGUUAGUUACUAACA-UUAAGUUGAGAACUCUAACGAGACUGCUAGUG-UAAGCUAGAGGAAGGUGGGGAUGACGUUAAACUACUAUGCCCUUUAUGUCCUGGGCUACACACGUGCUACAAUGGCUGGUACAAAGAGUUGCAAUCCUGUGAAGGGGAGCUAAUCUCAAAAAACCAGUCUCAGUUCGGAUUGAAGUCUGCAACUCGACUUCAUGAAGCCGGAAUCACUAGUAAUCGCGAAUCAGCUUGUCGCGGUGAAUACGUUCUCGGGUCUUGUACACACCGCCCGUCACACCAUGAGAGUUGGUAAUACCAGAAGUAGGUAGCUUAACCAUUUGGAGAGCGCUUCCCAAGGUAGGACUAGCGAUUGGGGUGAAGUCGUAACAAGGUAUCCGUACGGGAACGUGCGGAUGGAUCACCUCCUUU
lo.Myc.myc  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmCUGGCGGCAUGCCUAAUACAUGCAAGUCGAACGGAGGUG--------CUUG--------CACCUCAGUGGCGAACGGGUGAGUAACACGUAUCAACCUACCUCAUAGCGGGGGAUAACUUUUGGAAACGAAAGAUAAUACCGCAUGUAGAUGCAUAUCAAAAGAACC-----GUUU---GGUUCACUAUGAGAUGGGGAUGCGGCGUAUUAGCUAGUAGGUGAGAUAAUAGCCCACCUAGGCGAUGAUACGUAGCCGAACUGAGAGGUUGAUCGGCCACAUUGGGACUGAGAUACGGCCCAGACUCCUACGGGAGGCAGCAGUAGGGAAUUUUUCACAAUGGACGAAAGUCUGAUGAAGCAAUGCCGCGUGAGUGAUGACGGCCUUCGGGUUGUAAAGCUCUGUUGUAAGGGAAGAAAAAAUAAAGUAGAAA-UGACUUUAUCUUGACAGUACCUUACCAGAAAGCCACGGCUAACUAUGUGCCAGCAGCCGCGGUAAUACAUAGGUGGCAAGCGUUAUCCGGAUUUAUUGGGCGUAUAGGGUGCGUAGGCGGUUUUGCAAGUUUGAGGUUAAAGUCCGGAGCUCAACUCCGGUU-CGCCUUGAAAACUGUAUUACUAGAAUGCAAGAGAGGUAAGCGGAAUUCCAUGUGUAGCGGUGAAAUGCGUAGAUAUAUGGAAGAACACCUGUGGCGAAAGCGGCUUACUGGCUUGUUAUUGACGCUGAGGCACGAAAGCGUGGGGAGCAAAUAGGAUUAGAUACCCUAGUAGUCCACGCCGUAAACGAUGAGUACUAAGUGUUGGG---GUAA----CUCAGCGCUGCAGCUAACGCAUUAAGUACUCCGCCUGAGUAGUAUGCUCGCAAGAGUGAAACUCAAAGGAAUUGACGGGGACCCGCACAAGUGGUGGAGCAUGUGGUUUAAUUCGAAGCAACACGAAGAACCUUACCAGGGCUUGACAUCCAGUGAAAGCUAUAGAGAUAUAGU--AG--AGG---UUCAUUGAGACAGGUGGUGCAUGGUUGUCGUCAGUUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAACGCAACCCUUGUCGUUAGUUACUAACA-UUAAGUUGAGAACUCUAACGAGACUGCUAGUG-UAAGCUAGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGUCCUGGGCUACACACGUGCUACAAUGGCUGGUACAAAGAGUUGCAAUCCUGUGAAGGGGAGCUAAUCUCAAAAAACCAGUCUCAGUUCGGAUUGAAGUCUGCAACUCGACUUCAUGAAGCCGGAAUCACUAGUAAUCGCGAAUCAGCUUGUCGCGGUGAAUACGUUCUCGGGUCUUGUACACACCGCCCGUCACACCAUGAGAGUUGGUAAUACCAGAAGUAGGUAGCUUAACCGUUUGGAGAGCGCUUCCCAAGGUAGGACUAGCGAUUGGGGUGAAGUCGUAACAAGGUAUCCGUACGGGAACmmmmmmmmmmmmmmmmm-----
lo.Spi.cit  UUUAAUGAGAGUUUGAUCCUGGCUCAGGAUGAACGCUGGCGGCAUGCCUAAUACAUGCAAGUCGAACGGGGUG---------CUUG---------CACCCAGUGGCGAACGGGUGAGUAACACGUAUCAAUCUACCCAUUAGCGGGGGAUAACAGUUGGAAACGACUGAUAAUACCGCAUA-CGACGCAUGUUAAAAGGUCC-----GUUU---GGAUCACUAAUGGAUGAGGAUGCGGCGUAUUAGUUAGUUGGUGGGGUAAUGGCCUACCAAGACAAUGAUACGUAGCCGAACUGAGAGGUUGAUCGGCCACAUCGGGACUGAGACACGGCCCGAACUCCUACGGGAGGCAGCAGUAGGGAAUUUUUCACAAUGG-CGAAAGCCUGAUGGAGCAAUGCCGCGUGACUGAAGACGGUCUUCGGAUUGUAAAAGUCUGUUGUAAGGGAAGAACAGUAAGUAUAGAAA-UGAUACUUAUUUGACGGUACCUUACCAGAAAGCCACGGCUAACUAUGUGCCAGCAGCCGCGGUAAUACAUAGGUGGCAAGCGUUAUCCGGAUUUAUUGGGCGUAAAGCGUGCGCAGACGGUUUAACAAGUUUGGGGUCAAAUCCUGGAGCUCAACUCCAGUU-CGCCUUGAAAACUGUUAAGCUAGAGUGUAGGAAAGGUCGAUGGAAUUCCAUGUGUAGCGGUGAAAUGCGUAGAUAUAUGGAGGAACACCAGUGGCGAAGGCGGUCGACUGGCCUAUCACUGACGUUUAGGCACGAAAGCGUAGGGAGCAAAUAGGAUUAGAUACCCUAGUAGUCUACGCCGUAAACGAUGAGUACUAAGUGUCGGAC---UAA---GUUCGGUGCUGCAGCUAACGCAUUAAGUACUCCGCCUGAGUAGUAUGCUCGCAAGAGUGAAACUCAAAGGAAUUGACGGGGACCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGAAGAACCUUACCAAGGCUUGACAUCCAGUGAAAGCUGUAGAAAUACAGU--GG--AGG---UUCAUUGAGACAGGUGGUGCAUGGUUGUCGUCAGCUCGUGCCGUGAGGUGUUUGGUUAAGUCCAGUAACGAGCGCAACCCUUGCCGUUAGUUACUC-CA-UUAAUUGAGAUACUCUAACAGGACUGCUAGUG-UAAGCUAGAGGAAGGUGGGGAUGACGUCAAAUCAGCAUGCCCCUUAUAUCCUGGGCUACACACGUGCUACAAUGGUCGGUACAAACAGUUGCGAUCUCGUAAGAGGGAGCUAAUCUGAAAAAGCCGAUCUCAGUUCGGAUUGAGGGCUGCAACUCGCCCUCAUGAAGCCGGAAUCGCUAGUAAUCGCGAAUCAGCAUGUCGCGGUGAAUACGUUCUCGGGUCUUGUACACACCGCCCGUCACACCAUGAGAGUUGAUAAUACCAGAAGUCGGUAUUCUAACCGCAAGGAGGAAGCCGCCCAAGGUAGGAUUGAUGAUUAGGGUGAAGUCGUAACAAGGUAUCCGUACGAGAACGUGCGGAUGGAUCACCUCCUUU
lo.Sta.hom  -----mmAGAGUUUGAUNNUGGCUCAGGAUGAACGCUGGCGGCGUGCCUAAUACAUGCAAGUCGAGCGAACAGACGAGGAG-CUUG-CUCCUUUGACGUUAGCGGCGGACGGGUGAGUAACACGUGGGAACCUACCUAUAAGACUGGGAUAACUUCGGGAAACCGGAGCUAAUACCGGAUA-AUAUGCAUAGUGAAAGAUGGC----UUU---GCUAUCACUUAUAGAUGGACCUGCGCCGUAUUAGCUAGUUGGUAAGGUAACGGCUUACCAAGGCAACGAUACGUAGCCGACCUGAGAGGGUGAUCGGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUAGGGAAUCUUCCGCAAUGGGCGAAAGCCUGACGGAGCAACGCCGCGUGAGUGAUGAAGGUCUUCGGAUCGUAAAACUCUGUUAUUAGGGAAGAACAAACGUGUAAUAAC-UGUGCACGUCUUGACGGUACCUAAUCAGAAAGCCACGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGUGGCAAGCGUUAUCCGGAAUUAUUGGGCGUAAAGCGCGCGUAGGCGGUUUUUUAAGUCUGAUGUGAAAGCCCACGGCUCAACCGUGGAGGGUCAUUGGAAACUGGAAAACUUGAGUGCAGAAGAGGAAAGUGGAAUUCCAUGUGUAGCGGUGAAAUGCGCAGAGAUAUGGAGGAACACCAGUGGCGAAGGCGACUUUCUGGUCUCUAACUGACGCUGAUGUGCGAAAGCGUGGGGAUCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAGUGCUAAGUGUUAGGGGGUUCCGCCCCUUAGUGCUGCAGCUAACGCAUUAAGCACUCCGCCUGGGGAGUACGACCGCAAGGUUGAAACUCAAAGGAAUUGACGGGGACCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGAAGAACCUUACCAAAUCUUGACAUCCUUUGACCCUUCUAGAGAUAGAAG--UUUCUUCGGGGACAAAGUGACAGGUGGUGCAUGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUAAGCUUAGUUGCCAUCA-UUAAGUUGGGCACUCUAAGUUGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGCCCCUUAUGAUUUGGGCUACACACGUGCUACAAUGGACAAUACAAAGGGCAGCGAAACCGCGAGGUCAAGCAAAUCCCAUAAAGUUGUUCUCAGUUCGGAUUGUAGUCUGCAACUCGACUACAUGAAGCUGGAAUCGCUAGUAAUCGUAGAUCAGCAUGCUACGGUGAAUACGUUCCCGGGUCUUGUACACACCGCCCGUCACACCACGAGAGUUUGUAACACCCGAAGCCGGUGGAGUAACCAUUUGGAGCUAGCCGUCGAAGGUGGGACAAAUGAUUGGGGUGAAGUCGUAACAAGGUAGCCGUAUCGGAAGGUGCGGCUGGAUCACCUCCUUU
mt.Dro.mel  -----UUAAAGUUUUAUUUUGGCUUAAAA-AUUUGUUAUUAAUUUGAUU--UAUAUGUAAAUUU--------------------------------------UUGU------GUGAUU-UUA-UAUU-AUUUAAA-----------------------------------------------------------------------------------------AAAU--AAAUA------U-------------------------------AAA-----AUU---------------------------UAUC--------------------------------------GCAGUAAUUAAUAUUAUU-AAUUA-AAAAAUUUAGAAAUAGCAAUAUAAA---------------------------------------------------------------------------------------------AAAGUAUUGACUAAAUUGGUGCCAGCAGUCGCGGUUAUACCAUUAAUACAAAUAAAUUUUU---------UUAGUAUUAAUUAAAUUAAU---------------------------------------------------------------------------UUAU-------UAA--AAUUAAAUAUAU-A---GUUUUUA-UAUU-----------------UAAUUA----AU-AAAAA------------AAUUGAGUUAAA-UUAA---UAAAAAACUAGGAAUACCC-UA--UAUU-UAA--AAU-GUAAAUAAAUUG--------------------------------------------------------CUAAAG--UAGUA-AUAGUAU-UUCUGAAACUUAAAAAAUUUGGCGGUAUUUUAUCUA-CUAGA--GA-CC-U--GUUUUAUCGUAAUCC-ACGA-GGACCUUACUUAAAU-GUAAU-------------------------------------------------AGUUUAUAUACCGUCGUUAUCAGAAUAUUUU--UAGA-A---AUAAU---AU--------UCAAUAAUUUUAAU--------------------------------------------------------------AAAAA-UUUAUAUCAGAUCAAGGUGUAGCUUAUAUUUAAG--UAAUAAUGGGUUA-AA----UAAA-------------------------------------UUUAUA------------UGAAUAAAAUUAUGAAAA---AAUUUUUGAAAGUGGAUUUAGUAGUAAAAUUAU-AGA---UAAUAAUUUGAUUUUAGCUCUAAAAUAUGUACACAUCGCCCGUCGCUCUUAUU---------------------------------AUUA------------------------------GGUAAGAUAAGUCGUAACAUAGUAGAUGUACUGGAAAGUGUAUCUAGAAUGA-------
mt.Gal.gal  -----AAAAGACUUAGUCCUAACCUUUCU-AUUGGUUUUUGCUAGACAUA-UACAUGCAAGUAU--------------------------------------CCGCAUCCCAGUGAAA-AUGCCCCCACCUUUCUUCCC-------------------------------------------------------------------------------------AAGC-AAAAGGAGCAGGUAUCGGCA-CACU-------------------CAGU----AGCCC---AAGACGCCUU-GCU---UAAGCCAC-----------------------ACCCCCC-GGGUACUAGCAGUAAUUAACCUUAAGCAAUAA-GUUAAACUUGACUUAGCCAUAGCAA--------------------------------------------------------------------------------------------CCCAGGGUUGGUAAAUCUUGUGCCAGCCACCGCGGUCAUACAAGAAACCCAAAUCAAUAGCU-----ACCCGGCGUAAAGAGUGGCCACAU--------------------------------------------------------------------GUUAUCUCACC-------AAA--AUGCAACCAAGCUAUAAGCGAUCC-AC--------CUAACCCAAC--CCAA-----AU-CUUAG------------AAUUUAACCACGAAGG----ACCCAAACUGGGAAUACCC-CA--UAUG-CCU--AGCCCUAAAUCUAGAUACCU-----------------------------------C-CAAC--ACAUGUAUCCGCCU---GAGAACGAGCCAAAGCUUAAAACUCUAAGGACUUGGCGGUGCCCCACCCA-CUAGA--GA-CC-U--GCUAUAUCGUAAUCC-ACGA-UCACCCAACCACCCC-GCCA---GCA--------------------------------------------AGCCUACAUACCGCCGUCGCCAGCCCACCUCAAUGAA-G---ACAACA--GUG------AGCUCAAUAGCCCCU---------------------------------------------------------------CGCU-AAUAAGACAGGUCAAGGUAUAGCCUAUGGGGUGG--GAGAAAUGGGCUA-AUU---UUCUA------------------------------------AGAACA----------AACGAAAAAGGAUGUGAAACC-CGCCCUUAGAAGGAGGAUUUAGCAGUAAAGUGAGAUUACCCAGCUCACUUUAAGACGGCUCUGAGGCACGUACAUACCGCCCGUCACCCUCUUCACAAGCCAUCAACAUCAAUA-------------AAUA--------UAUACUCCCC-UCCCGGCUAAAGACGAGGCAAGUCGUAACAAGGUAAGUGUACCGGAAGGUGCACUUAGACUACN------
mt.Bos.tau  -----CAUAGGUUUGGUCCCAGCCUUCCU-GUUAACUCUUAAUAAACUUA-CACAUGCAAGCAU--------------------------------------CUACACCCCAGUGAGA-AUGCCCUCUGGUUAUUAA---------------------------------------------------------------------------------------AACU-AAGAGGAGCUGGCAUCAGCA-CACA-------------------CUGU----AGCUC---ACGACGCCUU-GC----UUAACCAC-----------------------A-CCCCA-GGGAAACAGCAGUGACAAAAAUUAAGCCAUAA-ACAAAGUUUGACUAAGUUAUAUUAA---------------------------------------------------------------------------------------------UUAGGGUUGGUAAAUCUCGUGCCAGCCACCGCGGUCAUACGAUUAACCCAAGCUAACAGGA-----GUACGGCGUAAAACGUGUUAAAGC----------------------------------------------------------------------ACCAUCCAA-------AAA--UUCUAACUAAGCUAAAAGCAUUAA-AA--------UAAAAAUAAAUGACAAA----AC-CCUAC------------AGCCGA-GCACUAAAG----ACCCAAACUGGGAAUACCC-CA--UAUG-CUU--AGCCCUAAACACAGAUAAUU-----------------------------------C-AUAC--AAAAUUAUUCGCCA---GAGUACUAGC-AC-GCUUAAAACUCAAAGGACUUGGCGGUGCUUUAUCCU-CUAGA--GA-CC-U--GCUAUAUCGUAAACC-CCGA-AAACCUCACCAAUUC-GCUA---AUA--------------------------------------------AGUCUAUAUACCGCCAUCUUCAGCAAACCCU--AAAG-G---AAAAAA--GUA------AGCGUAAUUAUGAUA---------------------------------------------------------------CAUA-AAAACGUUAGGUCAAGGUGUAACCUAUGAAAUGG-GAAGAAAUGGGCUA-AUU---CUCU-------------------------------------AGAGAA------UAAGCACGAAAGUUAUUAUGAAAC--CAAUAACCAAAGGAGGAUUUAGCAGUAAACUAAGAAGA---UGCUUAGUUGAAUUAGGCCAUGAAGCACGCACACACCGCCCGUCACCCUCCUCAAAUAGAUUCAGUGCAU-CUAA-----------CCUA---------UUAAACGCACUAGCUACAUGAGAGGAGACAAGUCGUAACAAGGUAAGCAUACUGGAAAGUGUGCUUGGAUAAAU------
mt.Hom.sap  -----AAUAGGUUUGGUCCUAGCCUUUCU-AUUAGCUCUUAGUAAGAUUA-CACAUGCAAGCAU--------------------------------------CCCCGUUCCAGUGAGU-UCACCCUCUAAUCACCAC---------------------------------------------------------------------------------------GAUC-AAAAGGAACAAGCAUCAGCA-CGCA-------------------AUGC----AGCUC---AAAACGCUUA-GC----CUAGCCAC-----------------------ACCCCCC-GGGAAACAGCAGUGAUUAACCUUUAGCAAUAA-ACAAAGUUUAACUAAGCUAUACUAA-------------------------------------------------------------------------------------------CCCCAGGGUUGGUCAAUUUCGUGCCAGCCACCGCGGUCACACGAUUAACCCAAGUCAAUAGAA-----UC-CGGCGUAAAGAGUGUUUUAGA----------------------------------------------------------------------UCACCCCUCCA-----AAA--ACUCACCUGAGUUAAAAACGUUGA-CA--------C-AAAAUAGACUACAAA----GC-UUUAA------------AUCUGAAACAGAAAAG----ACCCAAACUGGGAAUACCC-CA--UAUG-CUU--AGCCCUAAACCUCAACAGUU-----------------------------------A-AUAC--AAAACUGCUCGCCA---GAACACGAGC-AC-GCUUAAAACUCAAAGGACCUGGCGGUGCUUCAUCCC-CUAGA--GA-CC-U--GCUGUAUCGUAAACC-CCGA-CAACCUCACCACCUC-GCU---------------------------------------------------AGCCUAUAUACCGCCAUCUUCAGCAAACCCU-AUGAG-G---ACAAA---GUA------AGCGCAAGUACCCA----------------------------------------------------------------CGUA-AAGACGUUAGGUCAAGGUGUAGCCCAUGAGGUGG-CAAGAAAUGGGCUA-AUU---UUCU-------------------------------------AGAAAA---------CUACGAUAGCCCUUAUGAAACU-UAAGGGUCGAAGGUGGAUUUAGCAGUAAACUAAGAGGA---UGCUUAGUUGAACAGGGCCCUGAAGCGCGUACACACCGCCCGUCACCCUCCUCAAGUAUACUUCAAAGGACAUUU-----------ACUA---------AAACCCUACGCAUUUAUAU-AGAGGAGACAAGUCGUAACAUGGUAAGUGUACUGGAAAGUGCACUUGGACGAAC------
mt.Mus.mus  -----NAAAGGUUUGGUCCUGGCCUUAUA-AUUAAUUAGAGGUAAAAUUA-CACAUGCAAACCU--------------------------------------CCAUAGACCGGUGUAA-AAUCCCUUAAUUUACUUA---------------------------------------------------------------------------------------AAAU-UUAAGGAGAGGGUAUCAGCA-CAUU-------------------AAAU----AGCUU---AAGACACCUU-GC----CUAGCCAC-----------------------ACCCCCC-GGGACUCAGCAGUGAUAAAUAUUAAGCAAUAA-ACAAAGUUUGACUAAGUUAUACCUC--------------------------------------------------------------------------------------------U-UAGGGUUGGUAAAUUUCGUGCCAGCCACCGCGGUCAUACGAUUAACCCAAACUAAUUAUC------UUCGGCGUAAAACGUGUCAACUA---------------------------------------------------------------------UAAAUAAUAA-------AAA--AUCCAACUUAUAUAAAAUUGUUAG-GA--------CCAAACUCAAUAACAAA----AU-UCUAG------------UUAU-AAACACGAAAG----ACCCAAACUGGGAAUACCC-CA--UAUG-CUU--AGCCAUAAACCUAAAUAAUU----------------------------------AA-UUAC--AAAACUAUUUGCCA---GAGAACUAGC-AU-GCUUAAAACUCAAAGGACUUGGCGGUACUUUAUCCA-CUAGA--GA-CC-U--GCUAUAUCGUAAACC-CCGC-CUACCUCACCAUCUC-GCUA---AUU--------------------------------------------AGCCUAUAUACCGCCAUCUUCAGCAAACCCU--AAAG-G---AUUAAA--GUA------AGCAAAAGAAUCA-A---------------------------------------------------------------CAUA-AAAACGUUAGGUCAAGGUGUAGCCAAUGAAAUGG-GAAGAAAUGGGCUA-AUU---UUCUU------------------------------------AGAACA---------UUACUAUACCCUUUAUGAAAC--UAAAGGACUAAGGAGGAUUUAGUAGUAAAUUAAGAAGA---AGCUUAAUUGAAUUGAGCAAUGAAGUACGCACACACCGCCCGUCACCCUCCUCAAAUUAAAUUAAACUUAACAUA-----------UUAA--------UUUCUAGACA-UCCGUUUAUGAGAGGAGAUAAGUCGUAACAAGGUAAGCAUACUGGAAAGUGUGCUUGGAAUAAU------
mt.Sac.cer  AUUUAUAAGAAUAUGAUGUUGUUUCAGAUUAAGCGCUAAAUAAGGACAUGACGCAUACGAGUCAU-CGU-UU-AUUA-UU------GAUAAUA-AAU--AUGUGGUGUAAACGUGAGUAUUUAUUAGGAAUUAAAACUAUAG-----------AUAAGCUAAACUUAA------UAUA----UUAUAUAAAUA-AAAG-GAUA---UAUA--A-UAUUUACUAUAGUC-AAGCCAAUA-UGGUUUGGUA-GUAGGUUUAUUAGAGUUAAACCUAGCCACGAUCCAU-AAUCGUAAUGAAAGUUAGAACGAUCACGUUGACUCUGAAAUAU-AGUCAA-UAUCUU-AAGAUACAGCAGUGAGGAAUAUUGGACAAUGA-UCAAAGAUUGAUCCAGUUACUUAUUA-GGAUGAU-UAUAAA-AUAUUUUAUUUUAUUUUAUUAAUAUUUAUUAAUAAUAAUAAUA--AAUUAA-UUUAUUAUAUUAAUAUUAUUUAAU---GUCCUGACUAAAUUUGUGCCAGCAGUCGCGGUAACACAAAGAGGGCGAGCGUUAAUCAUAA-----UGGUUUAAAGGAUCCGUAGAUGAAUUAUA------------------------UAU-------------------------------UAGAGUUAAAAAA-------AGAAUUAUAAUAGUAAAGAAAUAAAUAAU-UAUA------AGACUAAUAUAUGUAAA-A--AA-UUAAAUAU-UA------CAUUGA-GGAUUAAGAG---U-GCG-AAACGGAAUAUUC-GU--UAGUUUCU--AGUAGUAAACUAUGAAUAC-AAUUAUUUAUAAU-UAUUAUAUAUAAAAAUAA---U-GAAUG-AAAGUAUUCCACCUGAAGAGUACGUUA-CA-UAAUGAAACUCAAAACAAUAGACGGUUACAGAUUAA-GCAGU--GA-CA-U--GAUUUUUCGUAAUCC-ACGA-UAACUUUACCAUAUU-G----A-AUAU----UAUAAU---AUAUU-AU---------UU---AUA-UUA-AGGCGUUACAUUGUUGUCUUUAGUUCGUGCU--CAAGUU--AGAUUAAA-UGUGCAACGAGCAAAACUCCAUAUAUAAUU--UUAU--------------------AUUA--UUUAU----UA--A-UAUAAAG--AAAGGAAUUAAGACAAAUCAUAAUGAUCCUUAUAAUAUGGGUAAUAGACGUGCUA-A-AUAAAAUGAUAAUAAAUAAUUUAAUU--U-AAUAUAAC----UUUUAAUAUAUUUUUUUUAUAUGAAUUAUAAUCUGAAAUUCGAUUAUAUGAAAAAAGAAUUGCUAGUAAUCGUAAAUGU---UGUUACGGUGAAUAUUC-UAACUGUUUCGCACUAAUCACUCAUCAGGCGUUGAAACAUAUUAUUAUCUUAUUAUUAUAUAAAUUU-UAA-UAUAUUUAUUUAUAUCAGAAAUAAUAUGAAUUAAUGCGAAGUUG-AAUACAGUUACCGUAGGGGAACCUGCGGUGGGCUUAUAAAUAUC
mt.Zea.may  CAAAAUCUGAGUUUGAUCCUGGCUCAGAAGGAACGCUAGCUAUAUGCUUAACACAUGCAAGUCGA-CGU-UGUUUUC-GGGGUUGAGUUGAGA-ACA--AAGUGGCGAACGGGUGCGUAACGCGUGGGAAUCUGCCGAACAG-----------UCGGGCCAAUCCUGAA-----GAAA----GCU-CAAAGC----------------------------CUGUUUGAUGAGCCUGCGUAGUAUUGGUA-GUUGGUCAGGUAAAGGCUGACCAAGCCAUGAUGCUU-AGCUGGUCUUUUCGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCGGACUCCCC-G-GGGGCAGCAGUGGGGAAUCUUGGACAAUGG-GCAAAGCCCGAUCCAGCAAUAUCGC--GAGUGAA-GAAGGCAAUGGCUUGUAAAGCUCUUUCGUCGAGUGCGCGAUC-----------------------AUGACAGGACUCGAGGAA-AAGCCCCGGCUAACUCCGUGCCAGCAGCCGCGGUAAGACGGGGGGGGCAAGUGUUCUUCGGAAUGACUGGGCGUAAAGGGCACGUAGGCGGUGAAUCGGGUUGAAAGUGA-AAGUCGCCAAAAAGGCGGA---A--UGCUU-------CGA-AC-UUGAGUGAGCAGA-------GGAAUUUCGUGUGUAGGGAAAUCCAUCUA-CGAA------GGAACGCCAAAAGCAAG-A--UC-UCUGGGUCCCU------CGCUGG-GUGCGAUGGG---GAGCG-AACAGGAAUACCC-UG--UAGU-CCA--UGCCGUAAACGAUGAGUGUUCGCCCUUGGUCUGUACGGCGGAUCAGGGCCCA---C-UAGCGUGAAACACUCCGCCUGGGGAGUACGGUC-CA-GACCGAAACUCAAAGGAAUUGACGGGGGCCUGACAA-GCGGU--GA-CA-U--GGUUUUUCGUACAAC-GCGC-AAACCUUACCAGCCC-GACA-A-UGAACAAAACCUGU---UAACG-GGA--------ACU--UUC-AUA-AGGUGCUGCAUGGCUGUCGUCAGCUCGUGUC--UGGAUG--UGGUCAAGUCCUAUAACGAGCGAAACCCUCGUUUUGUGUUGCUGAGAAG----UGCCGCACUCACGAGGGACUGCCAGU-GA--ACUGGAGGAAGGUGGGGAUGACGUCAAGUCCGCAUGGCCCUUAUGGGCUGGGCCACACACGUGCUA-AAUGGCAAUGACAAUGGGAAGCAAGGCU--AAGGCGGAGC----UCCGGAAAGAUUGCCUCAGUUCGGAUUGUUCUCUGCAACUCGGGAACAUGAAGUAGAAAUCGCUAGUAAUCGCGGAUGC---UGCCGCGGUGAAUAUGUACCCGGGCCCUGUACACACCGCCCGUCACACCCUGGGAAUUGGUUUCGCCCGAAGCUCGGACCAUGAUCACUAUAUUGGCGCAUACCACGGUGGGGUCUUCGACUGGGGUGAAGUCGUAACAAGGUAGCCGUAGGGGAACCUGUGGCUGGAUUGAAUCC---
mt.Gly.max  CAAAAUAAGAGUUUGAUCCUGGCUCAGAAGGAACGCUAGCUAUAUGCUUAACACAUGCAAGUCGA-CGU-UGUUUUC-GGGGUGC-GUUGAAA-ACA--AAGUGGCGAACGGGUGCGUAAUGCGUGGGAAUCUGCCGAACAG-----------UCGGGCCAAUCCUGAA-----GAAA----GCU-AAAAGC----------------------------CUGUUUGAUGAGCCUGCGUAGUAUUGGUA-GUUGGUCAGGUAAAGGCUGACCAAGCCAUGAUGCUU-AGCUGGUCUUUUCGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCGGACUCCCC-G-GGGGCAGCAGUGGGGAAUCUUGGACAAUGG-GCAAAGCCCGAUCCAGCAAUAUCGC--GAGUGAA-GAAGGCAAUGGCUUGUAAAGCUCUUUCGUCGAGUGCGCGAUC-----------------------AUGACAGGACUCGAGGAA-AAGCCCCGGCUAACUCCGUGCCAGCAGCCGCGGUAAGACGGGGGGGGCAAGUGUUCUUCGGAAUGACUGCGCGUAAAGGGCACGUAGGCGGUGAAUCGGGUUGAAAGUGA-AAGUCGCCAAAAACGUGGA---A--UGCUU-------CGA-AC-UUGAGUGAGCAGA-------GGAAUUUCGUGUGUAGGGAAAUCCAUCUA-CGAA------GGAACGCCAAAAGCAAG-A--UC-UCUGGGUCCCU------CGCUGG-GUGCGAUGGG---GAGCG-AACGGGAAUACCC-UG--UAGU-CCA--UGCCGUAAACGAUGAGUGUUCGCCCUUGGUCU--ACG--GGAUCAGGGCCCA---C-UAGCGUGAAACACUCCGCCUGGGGAGUACGGUC-CA-GACCGAAACUCAAAGGAAUUGACGGGGGCCUGACAA-GCGGU--GA-CA-U--GGUUUUUCGUACAAC-GCGC-AAACCUUACCAGCCC-GACA-A-UGAACAAAACCAGU---UAACG-GGA--------ACU--UUC-AUA-AGGUGCUGCAUGGCUGUCGUCAGCUCGUGUC--UGGAUG--UGGUCAAGUCCUAUAACGAGCGAAACCCUCGUUUUGUGUUGCUGAGAAGAAU-UGCCGCACUCACGAGGGACUGCCAGU-UA--ACUGGAGGAAGGUGGGGAUGACGUCAAGUCCGCAUGGCCCUUAUGGGCUGGGCCACACACGUGCUA-AAUGGCAAUUACAAUGGGAAGCAAGGCU--AAGGCGGAGC----UCCGGAAAGAUUGCCUCAGUUCGGAUUGUUCUCUGCAACUCGGGAACAUGAAGUUGGAAUCGCUAGUAAUCGCGGAUGC---UGCCGCGGUGAAUAUGUACCCGGGCCCUGUACACACCGCCCGUCACACCCUGGGAAUUGGUUUCGCCCGAAGCUCGGACCAUGAUCACUAUAUUGGCGCAUACCACGGUGGGGUCUUCGACUGGGGUGAAGUCGUAACAAGGUAGCCGUAGGGGAACCUGUGGCUGGAUGGAAUCC---
mt.Tet.pyr  AAUUUAUUGAGUGUGGUCAUGGCUCGGGAUUAACGCUAAUUAGACGCCUAACACAUGCGAGUU---UA--UAUAUUA-GAU-----UC---UA-AAU--GUAUAGCGUCAAGGUGAGUAAAUACA--UAAUAUGCCUUUAAGAAAUAAAGUUAAUUUA-AAAAUACAAA--UGCAUAAA---UUAUAAACUAA----G-AUUA--AU-UAA-G-UAAUUAUUUCUAAAAUGUUUAAU--AAAUAAA-U--AUUUAUUUUU---AGUAAAAUA-AGCCUGAUGUUU--AGUUGAAUU-UAGAGUUGUUCAACCACAUACGGGUUGAAAAC-UACCCUA-UCUAU----UGAC-AGCAGUGAGGA-UUUUGG-CAAUAU-GCAAAACA--A-CCAGCGAACUAAU--AAACGAAGAAGUAGCAA-UUGU-A-AAGUUUAAUUAACAAAUG---------------------------------AGUA-UAU-CUAGA-GAAGCUCUGGCUAAACAUGUACCAGCAGCCGCGGUAAUACAUGAGGAGCGAGCGUUACUCAUAUUGACUGAGUGUAUUAAAUACGUAGGAUGUUAUAAUUACUACUUUAAA-AAUGUAAU-AAAAA----UAAAU-C-UGU-------GUUA--U-AAUAAUUAUAUUA-------AGA-CUUAUAGAACAACGAAAUGUACACU-AUAU------AGCUAGCCAUAAACAAG-A--UC-UCUAAUUA-UA------AUCUGA-GUAUGACGGG---UAUCG-AUGAGGAUCUCCC-UA--UAGU-CCG--UACUGUAAAAGAUGAAUACUUUAUGAA---------C-----UUCAGAAU-A---C-UAGCA--AAGUAUUCUGCUUGGGGAGUAUUAUC-CA-GAUUAAAACUUAACUGAAUUGGCGGGAAUUUGUCGA-ACGGU--GA-CA-U--GGUUUUGCGUAAUCC-ACGC-AAAUCUUACCAACGU-UAG--C-UUUAUUAAUAUGGU----CACUA-UA--------AG---UAA-AGU-CGGAAUUGCAUGGCUGUCGUCAGUUCGUGCU--UGAGUU--GGAUUAAGUUCUUUAACGAAUGCAACCCUAUAAAUAAGUCUUUAAUAG-----UUUAUUAUUCUUGUUUAAUCUAUAUCAAUAUGAAUUAGGGGUUUUAGGCUGAAGUCAAGUCCCUAUGGUCUUUAUACGUUGGGCUACACACGUGUUA-AA-GGUAAAAACAAAGAGACGCAAUAGA--AAUCUGGAGC----CUC-AAAAAUUACCU-CAGUUCAGAUUGUCUCAUGAAAUUCUGAGGCAUGAAGAUGAAAUCGUUAGUAAUUGUAAAUAA---UGUUACAGUGAAAUAUUAGUCAAAUUUUGCACACAC-GCCCAUCACGCUCGGAAAGUCAAUAUUAGCGGAAG---AUUUGAAAUCUUUA-GGAAAGUAGUAUCUAAUCUAAUAUUGGUAAUCUGAGUGAAGUUGACACAAGGUACUGGUAGGGGAACCUGUUGGUGGAAUAUAUUAUUA
al.Aci.cry  -----mmmmmmmmmmmmmmmmmmmmmmAGCGAACGCUGGCGGCAUGCUUAACACAUGCAAGUCGCACGGGCAGG--------GCAA--------CCUGUCAGUGGCGGACGGGUGAGUAACGCGUAGGAAUCUAUCCUUGGGUGGGGGACAACCGUGGGAAACUACGGCUAAUACCGCAUGAUCCCUGAGGGGCAAAGGC-------GAAA-----GUCGCCUGAGGAGGAGCCUGCGUCUGAUUAGGUAGUUGGUGGGGUAAAGGCCUACCAAGCCUGCGAUCAGUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGCGAAAGCCUGAUCCAGCAAUGCCGCGUGGAUGAAGAAGGUCUUCGGAUUGUAAAGUCCUUUUGGCGGGGACGA------------UGAU-------------GACGGUACCCGCAGAAUAAGCUCCGGCUAACUUCGUGCCAGCAGCCGCGGUAAUACGAAGGGGGCUAGCGUUGCUCGGAAUGACUGGGCGUAAAGGGCGCGUAGGCGGACGGCACAGUCAGGCGUGAAAUUCCUGGGCUCAACCUGGGGACUGCGUCUGAGACGUGUUGUCUUGAGUAUGGAAGAGGGUUGUGGAAUUUCCAGUGUAGAGGUGAAAUUCGUAGAUAUUGGAAAGAACACCGGUGGCGAAGGCGGCAACCUGGUCCAUUACUGACGCUGAGGCGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCUGUAAACGAUGUGUGCUGGAUGUUGGGGUGUUA-GCACUUCAGUGUCGUAGCUAACGCGGUAAGCACACCGCCUGGGGAGUACGGCCGCAAGGUUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGCAGAACCUUACCAGGAUUUGACAUGGGAGUACCGGUCCAGAGAUGGACUUUCCC-GCAA-GGGGUCCCGCACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGCCUUCAGUUGCCAGCAUUUUGGGUGGGCACUCUGAAGGAACUGCCGGUGACAAGCCGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUAUGUCCUGGGCUACACACGUGCUACAAUGGCGGUGACAGUGGGAAGCCAGGUGGUGACACCGAGCUGAUCUCAAAAA-GCCGUCUCAGUUCGGAUUGCACUCUGCAACUCGAGUGCAUGAAGGUGGAAUCGCUAGUAAUCGCGGAUCAGCAUGCCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAUUUGGUUUGACCUUAAGUUGGUGCGCUAACCGCAAGGAGGCAGCCAACCACGGUCGGGUCAGAGACUGGGGUGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
al.Met.rho  -----mmmmmmmmmmmmmmmmmmmmmmAGCGAACGCUGUCGGCAGGCUUAUCACAUGUAAGUCGAGAGGGCACC--------UUCG--------GGUGUCAGCGGCAGACGGGUGAGUAACACGUGGGAACGUGCCCUUCGGUUCGGAAUAACUCAGGGAAACUUGAGCUAAUACCGGAUACGCCCUUAUGGGGAAAGG--------UUUA------CCGCCGAAGGAUCGGCCCGCGUCUGAUUAGCUUGUUGGUGGGGUAACGGCCUACCAAGGCGACGAUCAGUAGCUGGUCUGAGAGGAUCAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGCGCAAGCCUGAUCCAGCCAUGCCGCGUGAGUGAUGAAGGCCUUAGGGUUGUAAAGCUCUUUUGUCCGGGACGA------------UAAU-------------GACGGUACCGGAAGAAUAAGCCCCGGCUAACUUCGUGCCAGCAGCCGCGGUAAUACGAAGGGGGCUAGCGUUGCUCGGAAUCACUGGGCGUAAAGGGCGCGUAGGCGGCUUUUUAAGUCGGGGGUGAAAGCCUGUGGCUCAACCACAGAAUUGCCUUCGAUACUGGAAAGCUUGAGACCGGAAGAGGACAGCGGAACUGCGAGUGUAGAGGUGAAAUUCGUAGAUAUUCGCAAGAACACCAGUGGCGAAGGCGGCUGUCUGGUCCGGUUCUGACGCUGAGGCGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAAUGCCAGCCGUUGGCCUGCUUG-CAGGUCAGUGGCGCCGCUAACGCAUUAAGCAUUCCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGCAGAACCUUACCAUCCCUUGACAUGGCGUG-UACCCAGGGAGACUUGGGAUCCUCUUCGGAGGCGCGCACACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCACGUCCUUAGUUGCCAUCAUUCAG-UUGGGCACUCUAGGGAGACUGCCGGUGAUAAGCCGGAGGAAGGUGUGGAUGACGUCAAGUCCUCAUGGGCCUUACGGGAUGGGCUACACACGUGCUACAAUGGCGGUGACAGUGGGACGCGAAGGAGCGAUCUGGAGCAAAUCCCCAAAA-GCCGUCUCAGUUCGGAUUGCACUCUGCAACUCGGGUGCAUGAAGGCGGAAUCGCUAGUAAUCGUGGAUCAGCAUGCCACGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUUGGUCUUACCCGACGGCGCUGCGCCAACCGCAAGGAGGCAGGCGACCACGGUAGGGUCAGCGACUGGGGUGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
al.Rhi.fre  -AACUUGAGAGUUUGAUCCUGGCUCAGAACGAACGCUGGCGGCAGGCUUAACACAUGCAAGUCGAGCGCCCC----------GCAA----------GNNNAGCGGCAGACGGGUGAGUAACGCGUGGGAAUCUACCCUUUUCUACGGAAUAACGCAGGGAAACUUGUGCUAAUACCGUAUGAGCCCUUCGGGGGAAAGA--------UUUA------UCGGGAAAGGAUGAGCCCGCGUUGGAUUAGCUAGUUGGUGGGGUAAAGGCCUACCAAGGCGACGAUCCAUAGCUGGUCUGAGAGGAUGAUCAGCCACAUUGGGACUGAGACACGGCCCAAACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGCGCAAGCCUGAUCCAGCCAUGCCGCGUGAGUGAUGAAGGCCCUAGGGUUGUAAAGCUCUUUCACCGGUGAAGA------------UAAU-------------GACGGUAACCGGAGAAGAAGCCCCGGCUAACUUCGUGCCAGCAGCCGCGGUAAUACGAAGGGGGCUAGCGUUGUUCGGAAUUACUGGGCGUAAAGCGCACGUAGGCGGACAUUUAAGUCAGGGGUGAAAUCCCGGGGCUCAACCCCGGAACUGCCUUUGAUACUGGGUGUCUAGAGUCCGGAAGAGGUGAGUGGAAUUCCGAGUGUAGAGGUGAAAUUCGUAGAUAUUCGGAGGAACACCAGUGGCGAAGGCGGCUCACUGGUCCGGUACUGACGCUGAGGUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAAUGUUAGCCGUCGGGCAGUUUA-CUGUUCGGUGGCGCACGUAACGCAUUAAACAUUCCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGCAGAACCUUACCAGCCCUUGACAUCCCGAUCGGAUACGAGAGAUCGUAUCCUCAGUUCGCUGGAUCGGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGCCCUUAGUUGCCAGCAUUUAG-UUGGGCACUCUAAGGGGACUGCCGGUGAUAAGCCGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUACGGGCUGGGCUACACACGUGCUACAAUGGUGGUGACAGUGGGCAGCGAGACCGCGAGGUCGAGCUAAUCUCCAAAA-GCCAUCUCAGUUCGGAUUGCACUCUGCAACUCGAGUGCAUGAAGUUGGAAUCGCUAGUAAUCGCAGAUCAGCAUGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUUGGUUCUACCCGAAGGUAGUGCGCUAACCGCAAGGAGGCAGCUAACCACGGUAGGGUCAGCGACUGGGGUGAAGUCGUAACAAGGUAGCCGUAGGGGAACCUGCGGCUGGAUCACCUCC---
al.Ric.pea  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmGCUUAACACAUGCAAGUCGAACGGACUAAUU---GGGCUUG-CUCCAA-UUAGUUAGUGGCAGACGGGUGAGUAACACGUGGGAAUCUACCCAUCAGUACGGAAUAACUUUUAGAAAUAAAAGCUAAUACCGUAUAUUCUCUGCGGAGGAAAGA--------UUUA------UCGCUGAUGGAUGAGCCCGCGUCAGAUUAGGUAGUUGGUGAGGUAAUGGCUCACCAAGCCGACGAUCUGUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGCGAAAGCCUGAUCCAGCAAUACCGAGUGAGUGAUGAAGGCCUUAGGGUUGUAAAGCUCUUUUAGCAAGGAAGA------------UAAU-------------GACGUUACUUGCAGAAAAAGCCCCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGGGCUAGCGUUGUUCGGAAUUACUGGGCGUAAAGAGUGCGUAGGCGGUUUCGUAAGUUGGAAGUGAAAGCCCGGGGCUUAACCUCGGAAUUGCUUUCAAAACUACUAAUCUAGAGUGUAGUAGGGGAUGAUGGAAUUCCUAGUGUAGAGGUGAAAUUCUUAGAUAUUAGGAGGAACACCGGUGGCGAAGGCGGUCAUCUGGGCUACAACUGACGCUGAUGCACGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAGUGCUAGAUAUCGGAAGAUUC--UCUUUCGGUUUCGCAGCUAACGCAUUAAGCACUCCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUUGACGGGGGCUCGCACAAGCGGUGGAGCAUGCGGUUUAAUUCGAUGUUACGCGAAAAACCUUACCAACCCUUGACAUGGUGGUCGGAUCGCAGAGAUGCUUUUCUCAGCUCGCUGGACCACACACAGGUGUUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCAUUCUUAUUUGCCAGCGGUAAUGCCGGGAACUAUAAGAAAACUGCCGGUGAUAAGCCGGAGGAAGGUGGGGACGACGUCAAGUCAUCAUGGCCCUUACGGGUUGGGCUACACGCGUGCUACAAUGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-mmmm-mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
al.Sph.ter  -----mmmmAGUUUGAUCCUGGCUCAGAACGAACGCUGGCGGCAUGCCUAACACAUGCAAGUCGAACGAGAUC---------UUCG---------GAUCUAGUGGCGCACGGGUGCGUAACGCGUGGGAAUCUGCCCUUGGGUUCGGAAUAACUCAGAGAAAUUUGUGCUAAUACCGUAUAAUGUCUUCGGACCAAAGA--------UUUA------UCGCCCAAGGAUGAGCCCGCGUAGGAUUAGCUAGUUGGUGGGGUAAAGGCUCACCAAGGCGACGAUCCAUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGGACAAUGGGCGAAAGCCUGAUCCAGCAAUGCCGCGUGAGUGAUGAAGGCCCUAGGGUUGUAAAGCUCUUUUACCCGGGAUGA------------UAAU-------------GACAGUACCGGGAGAAUAAGCUCCGGCUAACUUCGUGCCAGCAGCCGCGGUAAUACGAGGGGAGCUAGCGUUGUUCGGAAUUACUGGGCGUAAAGCGCGCGUAGGCGGUUUUUUAAGUCAGAGGUGAAAGCCCGGGGCUCAACCCCGGAAUAGCCUUUGAAACUGGAAAGCUAGAAUCUUGGAGAGGUCAGUGGAAUUCCGAGUGUAGAGGUGAAAUUCGUAGAUAUUCGGAAGAACACCAGUUGCGAAGGCGGCUGACUGGACUGGUAUUGACGCUGAGGUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGAUAACUAGCUGUCCGGGCU-AUAGAGCUUGGGUGGCGCACGUAACGCAUUAAGUUAUCCGCCUGGGGAGUACGGCCGCAAGGUUAAAACUCAAAGGAAUUGACGGGGGCCUGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGCAGAACCUUACCAGCGUUUGACAUCCUGAGCGGUUACCAGAGAUGGUUUCCUUAGUUCGCUGGAUAGGUGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCAUCCCUAGUUGCCAUCAUUAAG-UUGGGCACUCUAAGGAAACUGCCGGUGAUAAGCCGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUACGCGCUGGGCUACACACGUGCUACAAUGGCGGUGACAGUGGGCAGCAACCUCGCGAGAGGUAGCUAAUCUCCAAAA-GCCGUCUCAGUUCGGAUUGUUCUCUGCAACUCGAGAGCAUGAAGGCGGAAUCGCUAGUAAUCGCGGAUCAGCAUGCCGCGGUGAAUACGUUCCCAGGCCUUGUACACACCGCCCGUCACACCAUGGGAGUUGGUUUCACCCGAAGGCCCAGAGCCAACCGCACGGAGGAAAGG------UGAAUGCCUAGUUA-UGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
be.Hyd.pal  -----mmmmmmmmmmmmmmmmmmmmmmAUUGAACGCUGGCGGCAUGCUUUACACAUGCAAGUCGAACGGUAACAGGCC----GCAA----GGUGCUGACGAGUGGCGAACGGGUGAGUAAUGCAUCGGAACGUGCCCAGUCGUGGGGGAUAACGCAGCGAAAGCUGCGCUAAUACCGCAUACGAUCUAUGGAUGAAAGCGGGGGACCGUAAGGCCUCGCGCGAUUGGAGCGGCCGAUGUCAGAUUAGGUAGUUGGUGGGGUAAAGGCUCACCAAGCCAACGAUCUGUAGCUGGUCUGAGAGGACGACCAGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUUUUGGACAAUGGGCGCMAGCCUGAUCCAGCAAUGCCGCGUGCAGGAAGAAGGCCUUCGGGUUGUAAACUGCUUUUGUACGGAACGAAAAGGCUCUGGUUAAUACCUGGGGCACAUGACGGUACCGUAAGAAUAAGCACCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGUGCAAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGUGCGCAGGCGGUUUUGUAAGACAGGCGUGAAAUCCCCGGGCUCAACCUGGGAAUUGCGCUUGUGACUGCAAGGCUGGAGUGCGGCAGAGGGGGAUGGAAUUCCGCGUGUAGCAGUGAAAUGCGUAGAUAUGCGGAGGAACACCGAUGGCGAAGGCAAUCCCCUGGGCCUGCACUGACGCUCAUGCACGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGUCAACUGGUUGUUGGGUCUCUUC-UGACUCAGUAACGAAGCUAACGCGUGAAGUUGACCGCCUGGGGAGUACGGCCGCAAGGUUGAAACUCAAAGGAAUUGACGGGGACCCGCACAAGCGGUGGAUGAUGUGGUUUAAUUCGAUGCAACGCGAAAAACCUUACCCACCUUUGACAUGUACGGAAUUUGCCAGAGAUGGCUUAGUGCUGAAAAGAGCCGUAACACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCAUUAGUUGCUAC---GAAA---GGGCACUCUAAUGAGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUAUAGGUGGGGCUACACACGUCAUACAAUGGCCGGUACAAAGGGUCGCAAACCCGCGAGGGGGAGCUAAUCCAUCAAAGCCGGUCGUAGUCCGGAUCGCAGUCUGCAACUCGACUGCGUGAAGUCGGAAUCGCUAGUAAUCGUGGAUCAGCAUGUCACGGUGAAUACGUUCCCGGGUCUUGUACACACCGCCCGUCACACCAUGGGAGCGGGUCUCGCCAGAAGUAGUUAGCCUAACCGCAAGGAGGGCGAUUACCACGGCGGGGUUCGUGACUGGGGUGGAGUCGUAACAAGGUAGCCGUAUCGGAAGGmmmmmmmmmmmmmmmm-----
be.Nei.fla  -AACAUAAGAGUUUGAUCCUGGCUCAGAUUGAACGCUGGCGGCAUGCUUUACACAUGCAAGUCGAACGGCAGCACAGAGAAGCUUGCUUCUUGGGUGGCGAGUGGCGAACGGGUGAGUAAUAUAUCGGAACGUACCGAGUAAUGGGGGAUAACUAAUCGAAAGAUUAGCUAAUACCGCAUAUUCUCUGAGGAGGAAAGCAGGGGACCUUCGGGCCNUGCGUUAUUCGAGCGGCCGAUAUCUGAUUAGUUAGUUGGUGGGGUAAAGGCCUACCAAGGCGACGAUCAGUAGCGGGUCUGAGAGGAUGAUCCGCCACACUGGGACUGAGACACGGCCNNGNCUCCUACGGGAGGCAGCAGUGGGGAAUUUUGGACAAUGGGCGCAAGCCUNAUCCAGCCAUGCCGCGUGUCUGAAGAAGGCCUUCGGGUUGUAAAGGACUUUUGUCAGGGAAGAAAAGGCUGUUGCUAAUAUCGACAGCUGAUGACGGUACCUGAAGAAUAAGCACCGGCUAACUACGUGCCAGCAGCCGCGNNNAUACGUAGGGUGCNAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGAGCGCAGACGGUUACUUAAGCAGGAUGUGAAAU-CCCGGGCUCAACCCGGNNACUGCGUUCUGAACUGGGUAACUAGAGUGUGUCAGAGGGAGGUAGAAUUCCACGUGUAGCAGUGAAAUGCGUAGAGAUGUGGAGGAAUACCGAUGGCGAAGGCAGCCUCCUGGGAUAACACUGACGUUCAUGCUCGAAAGCGUGGGUAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGUCAAUUAGCUGUUGGGCAACUUGAUUGCUUAGUAGCGUAGCUAACGCGUGAAAUUGACCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUUGACGGGGACCCGCACAAGCGGUGGAUGAUGUGGAUUAAUUCGAUGNAACGCGAAGAACCUUACCUGGUCUUGACAUGUACGGAAUCCUCCGGAGACGGAGGNGUGCCUUCGGGAGCCGUAACACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCRACGAGCGCAACCCUUGUCAUUAGUUGCCAUCAUUUAG-UUGGGCACUCUAAUGAGACUGCCGGUGACAAGCCGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUAUGACCAGGGCUUCACACGUCAUACAAUGGUCGGUACAGAGGGUAGCCAAGCCGCGAGGUGGAGCCAAUCUCACAAAACCGAUCGUAGUCCGGAUUGCACUCUGCAACUCGAGUGCAUGAAGUCGGAAUCGCUAGUAAUCGCAGGUCAGCAUACUGCGGUGAAUACGUUCCCGGGUCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGGGAUACCAGAAGUAGGUAGGGUAACCGCAAGGAGUCCGCUUACCACGGUAUGCUUCAUGACUGGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
be.Pse.syz  -----mmmmmmmmmmmmmmmmmmmmmmmmmmAACGCUGGCGGCAUGCCUUACACAUGCAAGUCGAACGGCAGCGGGGGUAG-CUUG--CUACCUCCGGCGAGUGGCGAACGGGUGAGUAAUACAUCGGAACGUGCCCUGUAGUGGGGGAUAACUA-UCGAAAGACUGGCUAAUACCGCAUACGACCUGAGGGUGAAAGUGGGGC-CCGCAAGGCCUCAUGCUAUAGGAGCGGCCGAUGUCUGAUUAGCUAGUUGGUGGGGUAAAGGCCCACCAAGGCGACGAUCAGUAGCUGGUCUGAGAGGACGAUCAGCCACACUGGGACUGAGAGACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUUUUGGACAAUGGGGGCAACCCUGAUCCAGCAAUGCCGCGUGUGUGAAGAAGGCCUUCGGGUUGUAAAGCACUUUUGUCCGGAAAGAAAUCGCUUCGGUUAAUACCUGGAGUGGAUGACGGUACCGGAAGAAUAAGGACCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGUCCAAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGUGCGCAGGCGGUUGUGCAAGACCGAUGUGAAAUCCCCGGGCUUAACCUGGGAAUUGCAUUGGUGACUGCACGGCUAGAGUGUGUCAGAGGGGGGUAGAAUUCCACGUGUAGCAGUGAAAUGCGUAGAGAUGUGGAGGAAUACCGAUGGCGAAGGCAGCCCCCUGGGAUAACACUGACGCUCAUGCACGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGUCAACUAGUUGUUGGGGA-UUCAUUUCCUUAGUAACGUAGCUAACGCGUGAAGUUGACCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUUGACGGGGACCCGCACAAGCGGUGGAUGAUGUGGAUUAAUUCGAUGCAACGCGAAAAACCUUACCUACCCUUGACAUGCCACUAACGAAGCAGAGAUGCAUUAGUG-UGAAAAGAAAGUGGACACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCUCUAGUUGCUA-C--GAAA---GGGCACUCUAGAGAGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUAUGGGUAGGGCUUCACACGUCAUACAAUGGUGCAUACAGAGGGUUGCCAAGCCGCGAGGUGGAGCUAAUCCCAGAAAAUGCAUCGUAGUCCGGAUCGUAGUCUGCAACUCGACUACGUGAAGCUGGAAUCGCUAGUAAUCGCGGAUCAGCAUGCCGCGGUGAAUACGUUCCCGGGUCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGCUUUACCAGAAGUAGUUAGCCUAACCGCAAGGAGGGCGAUUACCAC-GUAGGGUUCAUGACUGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
be.Rho.pur  GAACUGAAGAGUUUGAUCCUGGCUCAGAUUGAACGCUGGCGGCAUGCCUUACACAUGCAAGUCGAACGGUAACGGGNCC---UUCG---GGCGCCGAACGAGUGGCGAACGGGUGAGUAAUGCAUCGGAACAUGCCCUGAAGUGGGGGAUAACGUAGCGAAAGUUACGCUAAUACCGCAUAUUCUGUGAGCAGGAAAGCAGGGGACCUUCGGGCCUUGCGCUUUGGGAGUGGCCGAUGUCGGAUUAGCUAGUUGGUGGGGUAAAAGCCUACCAAGGCAACGAUCCGUAGCGGGUCUGAGAGGAUGAUCCGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUUUUGGACAAUGGGCGAAAGCCUGAUCCAGCCAUGCCGCGUGAGUGAAGAAGGCCUUCGGGUUGUAAAGCUCUUUCGGCGGGGAAGAAAUCGGGUUUCCUAAUACGGAACCCGGAUGACGGUACCCGAAGAAGAAGCACCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGUGCNAGCGUUAAUCGGAAUUACUGGGCGUAAAGCGUGCGCAGGCGGUUGUGUAAGACAGACGUGAAAUCCCCGGGCUCAACCUGGGAACUGCGUUUGUGACUGCACAGCUAGAGUACGGCAGAGGGGGGUGGAAUUCCACGUGUAGCAGUGAAAUGCGUAGAGAUGUGGAGGAACACCGAUGGCGAAGGCAGCCCCCUGGGCCAAUACUGACGCUCAUGCACGNAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGUCAACUAGGUGUUGGUGGGGUUAACCCAUUAGUGCCGUAGCUAACGCGUGAAGUUGACCGCCUGGGGAGUACGGCGGCAAGGUUAAAACUCAAAGGAAUUGACGGGGANCCGCACAAGCGGUGGAUGAUGUGGAUUAAUUCGAUGCAACGCGAAAAACCUUACCUACCCUUGACAUGUCAGGAAUCCUGAGGAGACUCGGGAGUGCCGAAAGGNACCUGAACACAGGUGCUGCAUGGCNGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCAUUAAUUGCCAUCAUUCAG-UUGGGCACUUUAAUGAAACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUAUGGGUAGGGCUUCACACGUCAUACAAUGGUCGGUCCAUAGGGUUGCNAACCCGCGAGGGGGAGCUAAUCCCAGAAAGCCGAUCGUAGUCCGGAUUGCAGUCUGCAACUCGACUGCAUGAAGUCGGAAUCGCUAGUAAUCGCGGAUCAGCAUGUCGCGGUGAAUACGUUCCCGGGUCUUGUACACACCGCCCGUCACACCAUGGGAGCGGGUUCUGCCAGAAGUAGUUAGCCUAACCGCAAGGAGGGCGAUUACCACGGCAGCGUUCGUmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
be.Tha.ter  -----mmmmmmmmmmmmmmmmmmmmmmmmmmAACGCUGGCGGCAUGCUUUACACAUGCAAGUCGAACGGCAGCACGGGC---UUCG---GCCUGGUGGCGAGUGGCGAACGGGUGAGUAAUGCAUCGGAACGUGCCCAUUUGUGGGGGAUAACUACGCGAAAGCGUGGCUAAUACCGCAUACGCCCUGAGGGGGAAAGCGGGGGAUUUUCGAACCUCGCGCAAUUGGAGCGGCCGAUGUCAGAUUAGCUUGUUGGUGAGGUAAAGGCUCACCAAGGCGACGAUCUGUAGCGGGUCUGAGAGGAUGAUCCGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUUUUGGACAAUGGGGGCAACCCUGAUCCAGCCAUGCCGCGUGAGUGAAGAAGGCCUUCGGGUUGUAAAGCUCUUUCAGCCGGGAAGAAAACGCACUCUCUAACAUAGGGUGUGGAUGACGGUACCGGAAGAAGAAGCACCGGCUAACUACGUGCCAGCAGCCGCGGUAAUACGUAGGGUGCGAGCGUUAAUCGGAAUUACUGGGCGUAAAGGGUGCGCAGGCGGUUUUGUAAGACAGAUGUGAAAUCCCCGGGCUUAACCUGGGAACUGCGUUUGUGACUGCAAGGCUAGAGUACGGCAGAGGGGGGUGGAAUUCCGCGUGUAGCAGUGAAAUGCGUAGAUAUGCGGAGGAACACCGAUGGCGAAGGCAACCCCCUGGGCCUGUACUGACGCUCAUGCACGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGUCAACUAGUCGUUCGGAGCGCAAUGCACUGAGUGACGCAGCUAACGCGUGAAGUUGACCGCCUGGGGAGUACGGCCGCAAGGUUAAAACUCAAAGGAAUUGACGGGGACCCGCACAAGCGGUGGAUGAUGUGGAUUAAUUCGAUGCAACGCGAAAAACCUUACCUACCCUUGACAUGUCUGGAACCUUGGUGAGAGCCGAGGGUGCCUUCGGGAGCCAGAACACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCAUUAGUUGCCAUCAUUUAG-UUGGGCACUCUAAUGAGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUAUGGGUAGGGCUUCACACGUCAUACAAUGGUCGGUACAGAGGGUUGCCAAGCCGCGAGGUGGAGCCAAUCCCACAAAGCCGAUCGUAGUCCGGAUCGUAGUCUGCAACUCGACUACGUGAAGUCGGAAUCGCUAGUAAUCGCAGAUCAGCAUGCUGCGGUGAAUACGUUCCCGGGUCUUGUACACACCGCCCGUCACACCAUGGGAGUGGGUUUCACCAGAAGUAGGUAGCUUAACCUUCGGGAGGGCGCUUACCACGGUGAGAUUCAUGACUGGGGUGAAGUCGUAACAAGGUAGCCGUAUCGGAAmmmmmmmmmmmmmmmmmm-----
de.Des.vib  -----mmmmmmmmmmmmmmUGGCUCAGAAUGAACGCUGGCGGCGUGCUUAACACAUGCAAGUCGAACGAGAAAGGGAUUG--CUUG-CAAUCCUGAGUAGAGUGGCGCACGGGUGAGUAACACGUAGAAAUCUGUCUCCAAGCCUGGGAUAACUAUUCGAAAGGGUAGCUAAUACCGGAUAAAGUUAUAAAAUGAAAGAUUGCCUCUCNUGAAGCAAUUGUUUGGGGGUGAGUUUGCGUACCAUUAGCUUGUUGGUGGGGUAAAGGCCUACCAAGGCAACGAUGGUUAGCUGGUCUGAGAGGAUGAUCAGUCACACUGGAACUGGAACACGGUCCAGACUCCUACGGGAGGCAGCAGUGAGGAAUUUUGCGCAAUGGGGGCAACCCUGACGCAGCAACGCCGCGUGAGUGAAGAAGGCCUUUGGGUCGUAAAGCUCUGUCAACAGGGAAGAAGUUAUUAUCUAUAAUAGUGGAUACUAUUGACGGUACCUGUGGAGGAAGCGCCGGCUAACUCCGUGCCAGCAGCCGCGGUAACACGGGGGGCGCAAGCGUUAUUCGGAAUUAUUGGGCGUAAAGGGCGCGUAGGCGGUCUUGUCCGUCAGGUGUGAAAGCCCGGGGCUCAACCCCGGAAGUGCACUUGAAACAGCAAGACUUGAAUACGGGAGAGGAAAGCGGAAUUCCUGGUGUAGAGGUGAAAUUCGUAGAUAUCAGGAGGAACACCGAUGGCGAAGGCAGCUUUCUGGACCGAUAUUGACGCUGAGGCGCGAAGGCGUGGGUAGCGAACGGGAUUAGAUACCCCGGUAGUCCACGCAGUAAACGUUGUACACUCGGUGUGGCGGAUAUUAAAUCUGCUGUGCCCAAGCUAACGCAUUAAGUGUACCGCCUGGGAAGUACGGUCGCAAGACUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGACGCAACGCGAAGAACCUUACCUGGGUUUGACAUCCUGUGAAUAUUGGGUAAUUGCCAUAGUGCCUUCGGGAGCACAGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUUGGUUAAGUCCAGCAACGAGCGCAACCCUUAUCGUCAGUUGCCAGCACUAAUGGUGGGAACUCUGGCGAGACUGCCCCGGUCAACGGGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUAUAUCCAGGGCUACACACGUGCUACAAUGGUAGGUACAAAGGGCAGCGACUCUGCAAAAAGAAGCGAAUCCCAAAA-GCCUAUCUCAGUCCGGAUUGGGGUCUGCAACUCGACCCCAUGAAGUUGGAAUCGCUAGUAAUCGCGGAUCAGCAUGCCGCGGUGAAUAUGUUCCCGGGCCUUGUACACACCGCCCGUCACACCAUGGAAGUUGAUUAUACCCGACGUCGCUGGGCUAACUUUUAAGGGGCAGGCGCCUAAGGUAUGGUCGAUAACUGGGGUGAAGUCGUAACAAGGUAGCCGUUGGAGmmmmmmmmmmmmmmmmmmmm-----
de.Des.ter  -----mmmmmmmmmmmmmmmmmmmmmmAUUGAACGCUGGCGGCGUGCCUAACACAUGCAAGUCGUGCGUGAAAGGGC-----UUCG----GCCUGAGUAAAGCGGCGCACGGGUGAGUAACGCGUGGAGAUCUGCCCAUGAGUUGGGAAUAACGGCUGGAAACGGUCGCUAAUACCGAAUACGCUC-CAUGGGGAAAGGUGGCCUCUCUUGAAGCUACCGCUCAUGGAUGAGUCCGCGUCCCAUUAGCUUGUUGGCGGGGUAAUGGCCCACCAAGGCGACGAUGGGUAGCCGACCUGAGAGGGUGAUCGGCCACACUGGGACUGGAACACGGCCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCGCAAUGGGCGAAAGCCUGACGCAGCGACGCCGCGUGAGGGAUGAAGGCCUUCGGGUCGUAAACCUCUGUCAGGAGGGAAGAACCGCCAUGGUGUAAU-CAGCCAUGGUCUGACGGUACCUCCAAAGGAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUAAUCGGAAUCACUGGGCGUAAAGCGCACGUAGGCUGUUUGGUAAGUCAGGGGUGAAAUCCCGCAGCUCAACUGCGGGAUUGCCCUUGAUACUGCUGGACUUGAGUUCGGGAGAGGGUGGCGGAAUUCCAGGUGUAGGAGUGAAAUCCGUAGAUAUCUGGAGGAACAUCAGUGGCGAAGGCGGCCACCUGGACCGAUACUGACGCUGAGGUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCUGUAAACGAUGGAUGCUAGGUGUCGGGGC-CUUG-AGCUUCGGUNCCGCAGCUAACGCGUNAAGCAUCCCGCCUGGGGAGUACGGUCGNNAGGNUGAAACUCAAAGAAAUUGACGGGNNCCCGCACAAGC-GUGGAGUAUGUGGUUUAAUUCGAUGCAACGCGAAGAACCUUACCCAGGCUUGACAUCCGGGAAACCCUCCCGAAAAGGAGGGGUGCUUUCGAGAAUCCCGAGACAGGUGCUGCNUGGNUGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUGUUCAUAGUUGCUACCAGUAAUGCUGGGCACUCUAUGGAGNNNNCCCCGGUUAACGGGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGGCCCUUACNNCUNNGGCUACACACGUACUACAAUGGCGCACACAAAGGGCAGCGAUACCGUGAGGUGGAGCCAAUCCCAAAAAAUGCGUCCCAGUCCGGAUUGCAGUCUGCAACUCGACUGCAUGAAGUUGGAAUCGCUAGUAAUUCGAGAUCAGCAUGCUCGGGUGAAUGCGUUCCCGGGCCUUGUACACACCGCCCGUCACACCACGAAAGUCGGUUUUACCCGAUACCGGUGAGCCAACCGCAAGGAGGCAGCCGUCUACGGUAGGGCCGAUGAUUGGGGUGAAGUCGUAACNNNGUAGCCGUAGGGGAACCUGCGGCUmmmmmmmmm-----
de.Des.suc  -----mmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmGUAUGCCUAACACAUGCAAGUCGAACGCGAACGGAAUC---UUCG--GAUUCCNAGUAGAGUGGCGCACGGGUGAGUAACACGUGGAAACCUGCCCUAGAGUCUGGGAUAACACUCCGAAAGGAGUGCUAAUACCGGAUAAGACCCUCGGGGAAAAGA--------UUUA------UUGCUCUAGGAUGGGUCCGCGGUCCAUUAGCUAGUUGGUGGGGUAAUGGCCUACCAAGGCUGCGAUGGAUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUUUUGCGCAAUGGGCGAAAGCCUGACGCAGCAAUACCGCGUGAAGGAUGAAGGCCCUUGGGUCGUAAACUUCUGUCAGAGGGGAAGAAAUACUC-----GCAA------GAGUGCUGACGGUACCCUCAAAGGAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUGUUCGGAAUUAUUGGGCGUAAAGAGCAUGUAGGCGGUCUGUUAAGUCUGGUGUGAAAGCCCGGGGCUCAACCCCGGAAGUGCAUUGGAUACUGGC-GACUUGAGUAUGGGAGAGGAAAGUGGAAUUCCGAGUGUAGGAGUGAAAUCCGUAGAUAUUCGGAGGAACACCAGUGGCGAAGGCGGCUUUCUGGACCAAUACUGACGCUGAGAUGCGAAAGCGUGGGGAGCGAACAGGAUUAGAUACCCUGGUAGUCCACGCCGUAAACGAUGGGUACUAGGUGUUGCGGGUAACCCUCCUGCAGUGCCGCAGCUAACGCAUUAAGUACCCCGCCUGGGGAGUACGGCCGCAAGGCUAAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGACGCAACGCGAAGAACCUUACCUAGGCUUGACAUCCCGAUCACUCUAUGGAAAAAUAGAGGUCAGCUUGCUGGAUCGGUGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUGUCCUUAGUUGCCAUCAUUAAG-UUGGGCACUCUAGGGAGACUGCCGGUGUUAAACCGGAGGAAGGUGGGGAUGACGUCAAGUCCUCAUGGCCCUUAUGCCUAGGGCUACACACGUGCUACAAUGGCCGGUACAAAGGGCAGCAAUACCGCGAGGUGGAGCGAAUCCCAAAAAGCCGGUCUCAGUUCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGUUGGAAUCGCUAGUAAUCGCGUAUCAGCAUGACGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCACGGGAGUCUAUUGUACCGGAAACCGGUGGGCUAACCUUCGGGGAGCAGCCGUUUAUGGUAUGAUCGGUAACUGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ep.Alv.pom  -----mmAGAGUUUGAUCAUGGCUCAGAGUGAACGCUGGCGGCGUGCUUAACACAUGCAAGUCGAACGGUAACAGGUC----UUCG---GAGUGCUGACGAGUGGCGAACGGGUGAGUAAUGUAUAGUAAUUUUCCCCUUGGAGAGGGAUAGCCACUGGAAACGGUGAUUAAUACCUCAUACUCCUAAAAAGGGAAAGG--------UUUA----UUCUGCCAAGGGAUAAGACUAUAUCCUAUCAGCUAGUUGGUAGUGUAAGGGACUACCAAGGCAAUGACGGGUAGCUGGUCUGAGAGGAUGAUCAGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGGGGAAUAUUGCACAAUGGGGGAAACCCUGAUGCAGCAACGCCGCGUGGAGGAAGAAGCAUUUAGGUGUGUAAACUCCUUUUAUCAAGGAAGAA-----------GA-C-------------GACGGUACUUGAUGAAUAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUACUCGGAAUCACUGGGCGUAAAGCGCAUGUAGGCGGUUGAUUAAGUUAGAAGUGAAAGCCUACGGCUCAACCGUAGAACAGCUUCUAAAACUGAUUAACUAGAGUCUGGGAGAGGAAGAUGGAAUUAGUAGUGUAGGGGUAAAAUCCGUAGAGAUUACUAGGAAUACCGAAAGCGAAGGCGAUCUUCUGGAACAGGACUGACGCUGAGAUGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGAAUGUUAGUCGUUGGAGUGCUAGCCACUUCAGCGAUGCAGCUAACGCAUUAAACAUUCCGCCUGGGGAGUACGSCCGCAAGGUUAAAACUCAAAGGAAUAGACGGGGACCCGCACAAGUGGUGGAGCAUGUGGUUUAAUUCGAAGAUACGCGAAGAACCUUACCUGGCCUUGACAUAAUCAGAACCCACCAGAGAUGGUGGGGUGCCUUCGGGAGCUGAUAUACAGGUGCUGCACGGCUGUCGUCAGCUCGUGUCGUGUGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUGACUAGUUACUAACAGUUAGGCUGAGGACUCUAGUCAGACUGCCUUCG-UAAGGAGGAGGAAGGUAAGGACGACGUCAAGUCAUCAUGGCCCUUACGGCCAGGGCGACACACGUGCUACAAUGGGAAGGACAAAGAGAAGCAAAACUGCGAGGUGGAGYAAAUCU-AUAAACCUUCUCUCAGUUCGGAUUGCAGUCUGCAACUCGACUGCAUGAAGCUGGAAUCACUAGUAAUCGUGAAUCAGCCUGUCACGGUGAAUACGUUCCCGGGUCUUGUACUCACCGCCCGUCACACCAUGGGAGUUGAUUUCACCCGAAGUGGGGAUGCUA---AAUA---GGCUACCCACCACGGUGGAAUUAGCGACUGGGGUGAAGUCGUAACAAGGUAACCGUAGGAGAACCUGCGGCUGGAUCACCUCCUUA
ep.Cam.fet  -UUAUGGAGAGUUUGAUCCUGGCUCAGAGUGAACGCNGGCGGCGUGCCUAAUACAUGCAAGUCGAACGGAGUAUUAAGAGAGCUUGNU--NUUAAUACNUAGUGGCGCACGGGUGAGUAAUGUAUAGUAAUCUGCCCUACACUGGAGGACAACAGUUAGAAAUGACUGCUAAUACUCCAUACUCCUAUAACGGGAAAG---------UNUU------UCGGUGUAGGAUGAGACUAUAUUGUAUCAGCUAGUUGGUAAGGUAAUGGCUUACCAAGGCUNUGACGCAUAACUGGUCUGAGAGGAUGAUCAGUCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUAGGGAAUAUUGCUCAAUGGGGGAAACCCUGAAGCAGCAACGCCGCGUGGAGGAUGACACUUUUCGGAGCGUAAACUCCNUUUGUUAGGGAAGAA-----------CCAU-------------GACGGUACCUAACGAAUAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCNAGCGUUACUCGGAAUCACUGGGCGUAAAGGACGCGUAGGCGGAUUAUCAAGUCUUUUGUGAAAUCUAACAGCUAAACUGUUAAACUGCUUGAGAAACUGAUAAUCUAGAGUGAGGGAGAGGCAGAUGGAAUUGGUGGUGUAGGGGUAAAAUCCGUAGAGAUCACCAGGAAUACCCAUUGCGAAGGCGAUCUGCUGGAACUCAACUGACGCUAAUGCGUGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCUCUAAACGAUGUAUACUAGUUGUUGCUGUGCUAGUCACGGCAGUAAUGCACCUAACGGAUUAAGUAUACCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUAGACGGGGACCCGCACAAGCGGUGGAGNNNGUGGUUUAAUUCGANNNUACGCGAAGAACCUUACCUGGGCUUGAUAUCCAACUAAUCUCUUAGAGAUAAGAGAGUGCUCUUGAGAAAGUUGAGACAGGUGCUGCACGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCACGUAUUUAGUUGCUAACAGUUCGGCUGAGCACUCUAAAUAGACUGCCUUCG-CAAGGAGGAGGAAGGUGUGGACGACGUCAAGUCAUCAUGGCCCUUAUGCCCAGGGCGACACACGUGCUACAAUGGCAUAUACAAUGAGAUGCAAUAUCGCGAGAUGGAGCAAAUCUAUAAA-AUAUGUCCCAGUUCGGAUUGGAGUCUGCAACUCGACUCCAUGAAGCCGGAAUCGCUAGUAAUCGUAGAUCAGCCUGCUACGGUGAAUACGUUCCCGGGUCUUGUACUCACCGCCCGUCACACCAUGGGAGUUGAUUUCACUCGAAGUCGGAAUGCUA---AAC---UAGCUACCGCCCACAGUGGAAUCAGCGACUGGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ep.Cam.lar  -UUAUGGAGAGUUUGAUCCUGGCUCAGAGUGAACGCUGGCGGCGUGCCUAAUACAUGCAAGUCGNACGAUGAAGCGA-CUAGCUUGAG---UUGUGGAUUAGUGGCGCACGGGUGAGUAAGGUAUAGUAAUCUGCCCUACACAAGAGGACAACAGUUGGAAACGACUGCUNAUACUCUAUACUCCUACAAAGGGAAAG---------UUUU------UCGGUGUAGGAUGAGACUAUAUAGUAUCAGCUAGUUGGUGAGGUAAUGGCUCACCAAGGCUNUGACGCUUAACUGGUCUGAGAGGAUGAUCAGUCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUAGGGAAUAUUGCGCAAUGGGGGAAACCCUGACGCAGCAACGCCGCGUGGAGGAUGACACUUUUCGGAGCGUAAACUCCUUUUCUUAGGGAAGAA-----------UUCU-------------GACGGUACCUAAGGAAUAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCNAGCGUUACUCGGAAUCACUGGGCGUAAAGGGCGCGUAGGCGGAUUAUCAAGUCUCUUGUGAAAUCCAACGGCUUAACCGUUGAACUGCUUGGGAAACUGGUAAUCUAGAGUGGGGGAGAGGCAGAUGGAAUUGGUGGUGUAGGGGUAAAAUCCGUAGAUAUCACCAAGAAUACCCAUUGCGAAGGCGAUCUGCUGGAACUUAACUGACGCUAAGGCGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGUAUGCUAGUUGUUGGGGUGCUAGUCAUCUCAGUAAUGCAGCUAACGCAUUAAGCAUACCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUAGACGGGGACCCGCACAAGCGGUGGAGCAUGUGGUUUNAUNCGAAGAUACGCGAAGAACCUUACCUGGGCUUGAUAUCCUAAGAACCUUAUAGAGAUAUGAGGGUGCUCUUGAGAACUUAGAGACAGGUGCUGCACGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCACGUAUUUAGUUGCUAACACUUCGGGUGAGCACUCUAAAUAGACUGCCUUCG-UAAGGAGGAGGAAGGUGUGGACGACGUCAAGUCAUCAUGGCCCUUAUGCCCAGGGCGACACACGUGCUACAAUGGCAUAUACAAUGAGACGCAAUACCGCGAGGUGGAGCAAAUCUAUAAA-AUAUGUCCCAGUUCGGAUUGUUCUCUGCAACUCGAGAGCAUGAAGCCGGAAUCGCUAGUAAUCGUAGAUCAGCCUGCUACGGUGAAUACGUUCCCGGGUCUUGUACUCACCGCCCGUCACACCAUGGGAGUUGAUUUCACUCGAAGCCGGAAUACUA---AAC---UAGUUACCGUCCACAGUGGAAUCAGCGACUGGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ep.Hel.pyl  UUUAUGGAGAGUUUGAUCCUGGCUCAGAGUGAACGCUGGCGGCGUGCCUAAUACAUGCAAGUCGAACGAUGAAGCUU-CUAGCUUGAG--AGUGCUGAUUAGUGGCGCACGGGUGAGUAACGCAUAGGCAUGUGCCUCUUAGUUUGGGAUAGCCAUUGGAAACGAUGAUUAAUACCAGAUACUCC-UACGGGGGAAAGA--------UUUA------UCGCUAAGAGAUCAGCCUAUGUCCUAUCAGCUUGUUGGUAAGGUAAUGGCUUACCAAGGCUAUGACGGGUAUCCGGCCUGAGAGGGUGAACGGACACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUAGGGAAUAUUGCUCAAUGGGGGAAACCCUGAAGCAGCAACGCCGCGUGGAGGAUGAAGGUUUUAGGAUUGUAAACUCCUUUUGUUAGAGAAGA------------UAAU-------------GACGGUAUCUAACGAAUAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUACUCGGAAUCACUGGGCGUAAAGAGCGCGUAGGCGGGAUAGUCAGUCAGGUGUGAAAUCCUAUGGCUUAACCAUAGAACUGCAUUUGAAACUACUAUUCUAGAGUGUGGGAGAGGUAGGUGGAAUUCUUGGUGUAGGGGUAAAAUCCGUAGAGAUCAAGAGGAAUACUCAUUGCGAAGGCGACCUGCUGGAACAUUACUGACGCUGAUGCGCUAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGGAUGCUAGUUGUUGGAGGGUUAGUCUCUCCAGUAAUGCAGCUAACGCAUUAAGCAUCCCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUAGACGGGGACCCGCACAAGCGGUGGAGCANGUGGUUUAAUUCGANNNNACACGAAGAACCUUACCUAGGCUUGACAUUGAGAGAAUCCGCUAGAAAUAGUGGAGGUCUCUUGAGACCUUGAAAACAGGUGCUGCACGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCNUUUCUUAGUUGCUAACAGUUAUGCUGAGAACUCUAAGGAUACUGCCUCCG-UAAGGAGGAGGAAGGUGGGGACGACGUCAAGUCAUCAUGGCCCUUACGCCUAGGGCUACACACGUGCUACAAUGGGGUGCACAAAGAGAAGCAAUACUGUGAAGUGGAGCCAAUCUUCAAA-ACACCUCUCAGUUCGGAUUGUAGGCUGCAACUCGCCUGCAUGAAGCUGGAAUCGCUAGUAAUCGCAAAUCAGCCUGUUGCGGUGAAUACGUUCCCGGGUCUUGUACUCACCGCCCGUCACACCAUGGGAGUUGUGUUUGCCUUAAGUCAGGAUGCUA---AAU---UGGCUACUGCCCACGGCACACACAGCGACUGGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
ep.Hel.tro  -----mmmmmmmmmmmmmmmmmmUCAGAGUGAACGCUGGCGGCGUGCCUAAUACAUGCAAGUCGAACGAUGAAGCUU-UUAGCUUGAG---AAGUGGAUUAGUGGCGCACGGGUGAGUAAUGCAUAGGAACAUGCCCCAUAGUCUGGGAUAGCCACUGGAAACGGUGAUUAAUACCGGAUACUCCUUACGAGGGAAAG---------UUUU------UCGCUAUGGGAUUGGCCUAUGUCCUAUCAGCUUGUUGGUGAGGUAAUGGCUCACCAAGGCUAUGACGGGUAUCCGGCCUGAGAGGGUGAUCGGACACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUAGGGAAUAUUGCUCAAUGGGCGAAAGCCUGAAGCAGCAACGCCGCGUGGAGGAUGAAGGUUUUAGGAUUGUAAACUCCUUUUCUAAGAGAAGA------------UUAU-------------GACGGUAUCUUAGGAAUAAGCACCGGCUAACUCCGUGCCAGCAGCCGCGGUAAUACGGAGGGUGCAAGCGUUACUCGGAAUCACUGGGCGUAAAGAGCGCGUAGGCGGGGUAAUAAGUCAGAUGUGAAAUCCUGUAGCUUAACUACAGAACUGCAUUUGAAACUGUUAUUCUAGAGUGUGGGAGAGGUAGGUGGAAUUCUUGGUGUAGGGGUAAAAUCCGUAGAGAUCAAGAGGAAUACUCAUUGCGAAGGCGACCUGCUGGAACAUUACUGACGCUGAUGCGCGAAAGCGUGGGGAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGAUGAAUGCUAGUUGUUGCCCUGCUUGUCAGGGCAGUAAUGCAGCUAACGCAUUAAGCAUUCCGCCUGGGGAGUACGGUCGCAAGAUUAAAACUCAAAGGAAUAGACGGGGACCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGAUACGCGAAGAACCUUACCUAGGCUUGACAUUGAUAGAAUCUGCUAGAGAUAGCGGAGUGCCUUCGGGAGCUUGAAAACAGGUGCUGCACGGCUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUCGUCCUUAGUUGCUAGCAGUUCGGCUGAGCACUCUAAGGAGACUGCCUUCG-UAAGGAGGAGGAAGGUGAGGACGACGUCAAGUCAUCAUGGCCCUUACGCCUAGGGCUACACACGUGCUACAAUGGGGCGCACAAAGAGGAGCAAUAUCGUGAGAUGGAGCAAAUCUCAAAA-ACGUCUCUCAGUUCGGAUUGUAGUCUGCAACUCGACUACAUAAAGCUGGAAUCGCUAGUAAUCGCAAAUCAGCAUGUUGCGGUGAAUACGUUCCCGGGUCUUGUACUCACCGCCCGUCACACCAUGGGAGUUGUAUUCGCCUUAAGUCGGAAUGCCA---AAC---UGGCUACCGCCCACGGCGGAUGCAGCGACUGGGGUGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
un.Aer.mar  -----mmmmmmmmmmmmmmUGGCUCAGGACGAACGCUGGCGGCGUGCCUAAGACAUGCAAGUCGAGCGGGGAGAUUGGGGAGCUUGCUCUCUGGUCUCCUAGCGGCGGACGGGUGAGUAACACGUGGGAACCUGCCCGGCAGUGGGGGAUAACCCUGGGAAACUGGGGCUAAUACCGCAUACGGUCGCAUGAAGAAAGGCCGUC---GUGA-GGCGGUCGCUGCCGGAGGGGCCCGCGGCCCAUCAGCUCGUUGGUGGGGUAACGGCCCACCAAGGCGACGACGGGUAGCCGGCCUGAGAGGGUGGUCGGCCACACUGGGACUGAGACACGGCCCAGACUCCUACGGGAGGCAGCAGUCGGGAAUCUUGCGCAAUGGGCGAAAGCCUGACGCAGCGACGCCGCGUGGGGGAGGAAGCCCUUCGGGGUGUAAACCCCUGUCGUCCGGGACGAAGGUGGGGGG-UGAAA--GGCCCUCUGCUGACGGUACCGGAGGAGGAAGCCCCGGCUAACUACNUGCCAGCAGCCGCGGUAAGACGUAGGGGGCGAGCGUUGUCCGGAAUCACUGGGCGUAAAGGGCGCGUAGGCGGCCUGGCAAGUCGGAUGUGAAAGGUCCCGGCUCAACCGGGGAGGUGCAUUCGAAACUGCCGGGCUUGAGGGCAGGAGAGGGCAGCGGAAUUCCCGGUGGAGCGGUGAAAUGCGUAGAGAUCGGGAGGAACACCAGUGGCGAAGGCGGCUGCCUGGCCUGGCCCUGACGCUGAGGCGCGACAGCGUGGGGAGCGAACGGGAUUAGAUACCCCGGUAGUCCACGCCGUAAACGAUGGGUGCUAGGUGUGGGAGGUUCGACCCCUUCCGUGCCGGAGCUAACGCACUAAGCACCCCGCCUGGGGAGUACGGCCGCAAGGCUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGAAGCAACGCGAAGAACCUUACCUGGGCUUGACAUCCCGCGAACCUGGCCGAAAGGCUGGGGUGCCUUCGGGAGCGCGGUGACAGGUGCUGCAUGGUUGUCGUCAGCUCGUGUCGUGAGAUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCCGCCCUGUGUUGCCAGCGGUACGGCCGGGCACUCACAGGGGACUGCCGGUGACAAACCGGAGGAAGGUGGGGAUGACGUCAAAUCAUCAUGGCCCUUAUGCCCAGGGCUACACACGUGCUACAAUGGCCGGUACAACGGGUUGCGAACCCGCGAGGGGGAGCCAAUCCCUAAAAGCCGGUCUCAGUUCGGAUCGCAGGCUGCAACUCGCCUGCGUGAAGCCGGAAUCGCUAGUAAUCGCGGAUCAGAAUGCCGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCACGGGAGCCGGCAACACCCGAAGCCGGUGGCCCAACCGUAAGGAGGGAGCCGUCGAAGGUGGGGCCGGUGACUGGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
un.Fle.sin  UGGAUGGAGAGUUUGAUCCUGGCUCAGAACGAACGCUGGCGGCGUGCUUAACACAUGCAAGUCAGGGGGAAUCCUCCC----UUCG--GGGAGGUAGUACNCCGGCGCACGGGUGAGUAACGCGUGAGGACCUACCCAUUAGACCUGGAUAACCCGGCGAAAGUUGGGCUAAGACAGGAUGUGUUAUCAUUGAUAAAGCAGGU----UUCG--GCUUGUGCUAAUGGAUGGGCUCGCGUCUGAUUAGCUAGUUGGUAAGGUAAAGGCUUACCAAGGCGACGAUCAUUAGGCGGCCUGAGAGGGUGGUCGCCCACACUGGAACUGAGACACGGUCCAGNCUCCUACGGGAGGCAGCAGUGGGGAAUUUUGCGCAAUGGGCGAGAGCCUGACGCAGCGACGCCGCGUGGACGAGGAAGGCCUUCGGGUCGUAAAGUCCNUUUUUACGGGAAGAAAGUUAUUAACAUAAC-UGGUUAAUAUUUGACGGUACCGUAAGAAUAAGCCCCGGCUAACUCCGUGCCAGCAGCCGNNNUNAUACGGAGGGGGCNAGCGUUGUUCGGAAUUACUGGGCGUAAAGCGUACGUAGGCGGCGUGGUAAGUCGGAGGUUAAAGGCUACGGCUNAACCGUAGUAGGGCNUUUGAUACUAUCAUGCUAGAGUGUCGGAGGGGGNAGCGGAAUUCCCUGUGUAGCGGUGAAAUGCGUAGAUAUAGGGAAGAACNCCAGUAGCGAAGGCGGCUACCUGGCCGAUCACUGACGCUGAGGUACGAGAGCGUGGGUAGCAAACAGGAUUAGAUACCCUGGUAGUCCACGCUGUAAACGAUGGACGCUAGGUGUUGGAGGAACCGCCCCUUCAGUGCCGAAGCCAACGCGUUAAGCGUCCNGCCUGGGGAGUACGACCGCAAGGUUAAAACUCAAAGGAAUUGACGGGGGCCNGCACAAGCGGUGGAGCACGUNGUUUAAUUCGAUGNUAACCGAAGAACCUUACCUGGUCUUGACAUCCUCCGGAGGGACUAGAGAUAGUUCUGNUGCNUNGGNAGCGGAGAGACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCCUAUUCAUAGUUGCCAUCGGUAAGGCCGGGCACUCUAUGGAUACUGCCCGUGAUAAACGGGAGGAAGGUGGGGACGACGUCAAGUCAUCAUGGCCUUUAUGACCAGGGCUACACACGUGCUACAAUGGUACGUACAGAGGGCAGCGAAGCCGCGAGGUGNAGCGAAUCCCUAAAAGCGUGCCUUAGUUCGGAUUGUAGUCUGCAAUUCGACUACAUGAAGGCGGAAUCGCUAGUAAUCGCAGGUCAGCAAACUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCNCNUCACACCACGGGAGUUAGUUGUACCUGAAGCCGGUGGCCCAAC-UGCG-GAGGGAGCCGNUUAUGGUAUGACUGGCAACNGGGGmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmmm-----
un.Mag.bav  -----mmAGAGUUUGAUUUUGGCUCAGAGCGAACGCUGGCGGCGUGCCUAACACAUGCAAGUCGAACGGAUCAUGAGUC---GCAA---GAUUCAUGGUUAGUGGCGCACGGGUGAGUAACGCGUAGGAAUCUGCCUUCAGGACCGGAACAACCAUUGGAAACGAUGGCUAAUCCCGGAUAAGACCGCAAGGUAAAAGGA-------GCAA-----UUCACCUGGAGAUGAGCCUGCGUCCUAUCAGGUAGUUGGUAAGGUAACGGCUUACCAAGCCUAAGACGGGUAGCCGGUCUGAGGGGAUGAACGGCCACACUGGAACUGAGACACGGUCCAGACUCCUACGGGAGGCAGCAGUGAGGAAUUUUGCGCAAUGGGGGAAACCCUGACGCAGCGACGCCGCGUGGAGGAAGAAGGCCUUCGGGUUGUAAACUCCUUUUGUAGGGAAAGA------------UGAU-------------GACGGUACCUUACGAAUAAGCCACGGCUAACUCUGUGCCAGCAGCCGCGGUAAGACAGAGGUGGCAAGCGUUGCUCGGAAUUACUGGGCUUAAAGGGUGCGUAGGCGGUUAGAUAAGUUUGGGGUGGAAUGCUCGGGCUCAACCCGGGAAUUGCCUUGAAAACUGUUUAACUUGAGUAAGCGAGGGGAUGGCGGAAUUCCUGGUGUAGCGGUGAAAUGCGUAGAUAUCAGGAGGAAGGCCGGUGGCGAAGGCGGCCAUCAGGCGCUUAACUGACGCUGAGGCACGAAAGCGUGGGGAUCAAACAGGAUUAGAUACCCUGGUAGUCCACGCCCUAAACGGUGGGUACUAGGUAUAGGGCUCGAUA-GGGUUCUGUGCCGAAGGGAAACCAAUAAGUACCCCGCCUGGGAAGUACGGCCGCAAGGUUGAAACUCAAAGGAAUUGACGGGGGCCCGCACAAGCGGUGGAGCAUGUGGUUUAAUUCGACGCAACGCGAGGAACCUUACCUGGGUUUGACAUGAAGUGUAGGAGUCCGAAAGGAUAACGCUCCGCAAGGAGAGCUUGCACAGGUGCUGCAUGGCUGUCGUCAGCUCGUGCCGUGAGGUGUUGGGUUAAGUCCCGCAACGAGCGCAACCCUUAUUCUCUGUUGCCAUCGGUAAAGCCGGGCGCUCUGAGAAAACUGCCGGCGAUGAGUUGGAGGAAGGUGGGGAUGACGUCAAGUCAUCAUGGCCCUUAUGUCCAGGGCUACACACGUGCUACAAUGGCUAUUACAGAGGGAAGCAAGAUCGCGAGGUGGAGCAAAUCCC-UAAAAAUAGUCUUAGUUGGGAUCGGAGUCUGCAACUCGACUCCGUGAACGUGGAAUCGCUAGUAAUCGCAGAUCACUACGCUGCGGUGAAUACGUUCCCGGGCCUUGUACACACCGCCCGUCACACCACGAAAGUCUGCUGUACCCGAAGCGGGUGAGCUAACCGCAAGGAGGCAGCUCACGAAGGUAUGGUAGGUGAUUGGGGUGAAGUCGUAACAAGGUAGCCGUAGGGGAACCUGCGGCUGGAUCACCUCCUUU
"""

sample_tree = """((((ch.Cya.cal:0.01909,ch.Cya.par:0.00974):0.00562,((ch.Eug.gra:0.03679,ch.Chl.mir:0.03448):0.00792,(ch.Pin.thu:0.00289,(((ch.Pis.sat:0.00728,ch.Gly.max:0.00560):0.00207,(ch.Nic.tab:0.00032,ch.Dau.car:0.00404):0.00191):0.00189,ch.Zea.may:0.00644):0.00207):0.00992):0.00674):0.00976,(((al.Aci.cry:0.02022,(al.Met.rho:0.01428,((al.Sph.ter:0.01856,al.Rhi.fre:0.01560):0.00577,(al.Ric.pea:0.03909,ga.Oce.pus:0.02219):0.00897):0.00740):0.00884):0.01161,(((be.Rho.pur:0.01507,be.Nei.fla:0.01667):0.00857,(be.Tha.ter:0.01119,(be.Pse.syz:0.01745,be.Hyd.pal:0.01564):0.00716):0.00536):0.01090,(ga.Xan.pis:0.01723,(ga.Pse.asp:0.01325,(ga.Leg.lyt:0.02104,(ga.Mar.psy:0.01684,((ga.Pho.pho:0.01267,ga.Hae.par:0.02272):0.00386,(ga.Vib.cho:0.01167,(ga.Sal.typ:0.00374,ga.Esc.col:0.00568):0.00939):0.00726):0.00652):0.01339):0.00892):0.00829):0.00892):0.01675):0.00927,((((de.Des.suc:0.02147,(de.Des.ter:0.02837,de.Des.vib:0.02735):0.01258):0.01172,(ra.Dei.rad:0.01597,ra.Dei.mur:0.01346):0.02678):0.00841,(((lo.Eub.lac:0.03050,sp.Lep.mey:0.03062):0.01102,((((lo.Lac.car:0.00581,lo.Car.mob:0.01332):0.01048,(lo.Sta.hom:0.01349,lo.Bac.sub:0.01148):0.01015):0.00661,lo.Bac.alc:0.01312):0.01687,((hi.Bif.inf:0.02108,((((hi.Myc.int:0.00581,hi.Myc.cel:0.00922):0.00960,hi.Cor.oti:0.01993):0.00689,(hi.Mic.par:0.01467,(hi.Mic.ech:0.00986,(hi.Noc.fla:0.01240,(hi.Str.rim:0.01912,hi.Cel.cel:0.01370):0.00521):0.00437):0.00594):0.00601):0.01040,hi.Act.eur:0.02348):0.00923):0.01618,((un.Fle.sin:0.03037,(((th.The.sub:0.01159,th.The.hyp:0.01299):0.01239,th.The.afr:0.02464):0.02101,(sp.Tre.suc:0.02298,sp.Tre.soc:0.02704):0.02202):0.01363):0.01096,(un.Aer.mar:0.01630,(((pt.Pla.mar:0.01738,pt.Pla.bra:0.02107):0.03072,pt.Iso.pal:0.03183):0.02494,gr.The.ros:0.03608):0.01201):0.01061):0.00862):0.01055):0.00806):0.00855,(((lo.Spi.cit:0.02091,(lo.Myc.myc:0.00530,lo.Myc.cap:0.00444):0.01268):0.01693,lo.Ach.lai:0.02594):0.01425,((fu.Lep.mic:0.01902,fu.Lep.buc:0.01886):0.01380,(fu.Fus.nul:0.01817,fu.Fus.gon:0.01127):0.01341):0.01792):0.00949):0.00816):0.00698,(((un.Mag.bav:0.02918,sp.Bor.par:0.03882):0.01142,((fl.Fla.xan:0.01205,(fl.Mar.psy:0.01497,fl.Cyt.mar:0.01590):0.00762):0.01329,(fl.Pre.eno:0.03031,fl.Bac.spl:0.02532):0.01223):0.02551):0.01015,((((ep.Hel.tro:0.01074,ep.Hel.pyl:0.01136):0.01095,(ep.Cam.lar:0.00615,ep.Cam.fet:0.00894):0.01470):0.01449,ep.Alv.pom:0.01917):0.01597,(((mt.Gly.max:0.02673,mt.Zea.may:0.02336):0.10722,((mt.Tet.pyr:0.10091,mt.Sac.cer:0.10584):0.04143,(((mt.Hom.sap:0.01752,(mt.Mus.mus:0.02097,mt.Bos.tau:0.01663):0.01043):0.01436,mt.Gal.gal:0.02956):0.02679,mt.Dro.mel:0.05506):0.07927):0.04096):0.05555,((ar.Met.ore:0.02764,((ar.Hal.sac:0.01549,(ar.Nat.mag:0.01515,ar.Hal.val:0.01643):0.00712):0.02421,(ar.Met.fer:0.01448,((ar.The.ste:0.00873,ar.The.chi:0.00962):0.01420,(ar.Met.jan:0.01611,(ar.The.pen:0.01914,ar.Sul.sol:0.02566):0.02002):0.00655):0.00998):0.01266):0.01176):0.03812,(pr.Try.equ:0.13823,((pr.Tet.bor:0.03717,pr.Cyc.gla:0.02400):0.01721,((pr.Pla.mal:0.11785,pr.Sar.neu:0.02921):0.01926,(pr.Por.aca:0.03748,((pr.Rho.sal:0.02632,(pr.Bla.hom:0.03851,pr.Mal.mat:0.03042):0.01478):0.00979,(((fu.Gig.mar:0.01786,((fu.Den.sul:0.00925,(fu.Fel.oga:0.00867,fu.Bul.hui:0.00910):0.00830):0.00765,(((fu.Sac.cer:0.00875,fu.Can.gla:0.00407):0.01253,(fu.Neu.cra:0.01565,(fu.Asp.tam:0.00163,fu.Asp.nid:0.00344):0.01020):0.00840):0.00546,fu.Sch.pom:0.01836):0.00563):0.00700):0.00993,((an.Bra.pli:0.02731,an.Cae.ele:0.06299):0.01757,(((an.Str.int:0.01379,an.Ast.amu:0.01332):0.01183,((((an.Rat.nor:0.00272,an.Mus.mus:0.00282):0.00216,an.Hom.sap:0.00628):0.01405,an.Xen.lae:0.00769):0.00993,an.Lam.aep:0.01238):0.01437):0.01557,an.Dro.mel:0.06939):0.01211):0.01858):0.00917,(pr.Sti.hel:0.02282,(((pl.Tai.cry:0.00186,pl.Pin.luc:0.00267):0.00699,pl.Gin.bil:0.00367):0.00467,((pl.Ara.tha:0.00614,((pl.Pis.sat:0.00081,pl.Gly.max:0.00234):0.00290,((pl.Sol.tub:0.00344,pl.Lyc.esc:0.00378):0.00372,pl.Pan.gin:0.00361):0.00180):0.00148):0.00251,pl.Zea.may:0.01435):0.00597):0.01283):0.01173):0.00837):0.01080):0.01141):0.01545):0.04435):0.08751):0.03645):0.02229):0.00860):0.00878):0.00657):0.01503):0.00749,(cy.Pho.amb:0.01236,(cy.Tri.ten:0.01069,cy.Osc.aga:0.01588):0.00501):0.00887,cy.Mic.aer:0.03275);
"""

sample_species_names = """cy.Mic.aer
cy.Osc.aga
cy.Pho.amb
cy.Tri.ten
pl.Zea.may
pl.Pan.gin
pl.Ara.tha
pl.Gly.max
pl.Pis.sat
pl.Lyc.esc
pl.Sol.tub
pl.Gin.bil
pl.Pin.luc
pl.Tai.cry
pr.Sar.neu
pr.Pla.mal
pr.Sti.hel
pr.Mal.mat
pr.Cyc.gla
pr.Tet.bor
pr.Rho.sal
pr.Por.aca
pr.Bla.hom
pr.Try.equ
an.Dro.mel
an.Lam.aep
an.Xen.lae
an.Hom.sap
an.Mus.mus
an.Rat.nor
an.Ast.amu
an.Str.int
an.Cae.ele
an.Bra.pli
fu.Sch.pom
fu.Asp.nid
fu.Asp.tam
fu.Neu.cra
fu.Can.gla
fu.Sac.cer
fu.Bul.hui
fu.Den.sul
fu.Fel.oga
fu.Gig.mar
fl.Bac.spl
fl.Cyt.mar
fl.Fla.xan
fl.Mar.psy
fl.Pre.eno
fu.Fus.gon
fu.Fus.nul
fu.Lep.buc
fu.Lep.mic
gr.The.ros
pt.Iso.pal
pt.Pla.bra
pt.Pla.mar
ch.Zea.may
ch.Dau.car
ch.Gly.max
ch.Pis.sat
ch.Nic.tab
ch.Pin.thu
ch.Chl.mir
ch.Eug.gra
ch.Cya.par
ch.Cya.cal
ga.Esc.col
ga.Hae.par
ga.Leg.lyt
ga.Mar.psy
ga.Oce.pus
ga.Pho.pho
ga.Pse.asp
ga.Sal.typ
ga.Vib.cho
ga.Xan.pis
ra.Dei.mur
ra.Dei.rad
sp.Bor.par
sp.Lep.mey
sp.Tre.soc
sp.Tre.suc
th.The.afr
th.The.hyp
th.The.sub
ar.Sul.sol
ar.The.pen
ar.Hal.val
ar.Hal.sac
ar.Nat.mag
ar.Met.fer
ar.Met.jan
ar.Met.ore
ar.The.chi
ar.The.ste
hi.Act.eur
hi.Bif.inf
hi.Cel.cel
hi.Cor.oti
hi.Mic.par
hi.Mic.ech
hi.Myc.cel
hi.Myc.int
hi.Noc.fla
hi.Str.rim
lo.Ach.lai
lo.Bac.alc
lo.Bac.sub
lo.Car.mob
lo.Eub.lac
lo.Lac.car
lo.Myc.cap
lo.Myc.myc
lo.Spi.cit
lo.Sta.hom
mt.Dro.mel
mt.Gal.gal
mt.Bos.tau
mt.Hom.sap
mt.Mus.mus
mt.Sac.cer
mt.Zea.may
mt.Gly.max
mt.Tet.pyr
al.Aci.cry
al.Met.rho
al.Rhi.fre
al.Ric.pea
al.Sph.ter
be.Hyd.pal
be.Nei.fla
be.Pse.syz
be.Rho.pur
be.Tha.ter
de.Des.vib
de.Des.ter
de.Des.suc
ep.Alv.pom
ep.Cam.fet
ep.Cam.lar
ep.Hel.pyl
ep.Hel.tro
un.Aer.mar
un.Fle.sin
un.Mag.bav
"""

sample_seq_to_species = """cy.Mic.aer\tcy.Mic.aer
cy.Osc.aga\tcy.Osc.aga
cy.Pho.amb\tcy.Pho.amb
cy.Tri.ten\tcy.Tri.ten
pl.Zea.may\tpl.Zea.may
pl.Pan.gin\tpl.Pan.gin
pl.Ara.tha\tpl.Ara.tha
pl.Gly.max\tpl.Gly.max
pl.Pis.sat\tpl.Pis.sat
pl.Lyc.esc\tpl.Lyc.esc
pl.Sol.tub\tpl.Sol.tub
pl.Gin.bil\tpl.Gin.bil
pl.Pin.luc\tpl.Pin.luc
pl.Tai.cry\tpl.Tai.cry
pr.Sar.neu\tpr.Sar.neu
pr.Pla.mal\tpr.Pla.mal
pr.Sti.hel\tpr.Sti.hel
pr.Mal.mat\tpr.Mal.mat
pr.Cyc.gla\tpr.Cyc.gla
pr.Tet.bor\tpr.Tet.bor
pr.Rho.sal\tpr.Rho.sal
pr.Por.aca\tpr.Por.aca
pr.Bla.hom\tpr.Bla.hom
pr.Try.equ\tpr.Try.equ
an.Dro.mel\tan.Dro.mel
an.Lam.aep\tan.Lam.aep
an.Xen.lae\tan.Xen.lae
an.Hom.sap\tan.Hom.sap
an.Mus.mus\tan.Mus.mus
an.Rat.nor\tan.Rat.nor
an.Ast.amu\tan.Ast.amu
an.Str.int\tan.Str.int
an.Cae.ele\tan.Cae.ele
an.Bra.pli\tan.Bra.pli
fu.Sch.pom\tfu.Sch.pom
fu.Asp.nid\tfu.Asp.nid
fu.Asp.tam\tfu.Asp.tam
fu.Neu.cra\tfu.Neu.cra
fu.Can.gla\tfu.Can.gla
fu.Sac.cer\tfu.Sac.cer
fu.Bul.hui\tfu.Bul.hui
fu.Den.sul\tfu.Den.sul
fu.Fel.oga\tfu.Fel.oga
fu.Gig.mar\tfu.Gig.mar
fl.Bac.spl\tfl.Bac.spl
fl.Cyt.mar\tfl.Cyt.mar
fl.Fla.xan\tfl.Fla.xan
fl.Mar.psy\tfl.Mar.psy
fl.Pre.eno\tfl.Pre.eno
fu.Fus.gon\tfu.Fus.gon
fu.Fus.nul\tfu.Fus.nul
fu.Lep.buc\tfu.Lep.buc
fu.Lep.mic\tfu.Lep.mic
gr.The.ros\tgr.The.ros
pt.Iso.pal\tpt.Iso.pal
pt.Pla.bra\tpt.Pla.bra
pt.Pla.mar\tpt.Pla.mar
ch.Zea.may\tch.Zea.may
ch.Dau.car\tch.Dau.car
ch.Gly.max\tch.Gly.max
ch.Pis.sat\tch.Pis.sat
ch.Nic.tab\tch.Nic.tab
ch.Pin.thu\tch.Pin.thu
ch.Chl.mir\tch.Chl.mir
ch.Eug.gra\tch.Eug.gra
ch.Cya.par\tch.Cya.par
ch.Cya.cal\tch.Cya.cal
ga.Esc.col\tga.Esc.col
ga.Hae.par\tga.Hae.par
ga.Leg.lyt\tga.Leg.lyt
ga.Mar.psy\tga.Mar.psy
ga.Oce.pus\tga.Oce.pus
ga.Pho.pho\tga.Pho.pho
ga.Pse.asp\tga.Pse.asp
ga.Sal.typ\tga.Sal.typ
ga.Vib.cho\tga.Vib.cho
ga.Xan.pis\tga.Xan.pis
ra.Dei.mur\tra.Dei.mur
ra.Dei.rad\tra.Dei.rad
sp.Bor.par\tsp.Bor.par
sp.Lep.mey\tsp.Lep.mey
sp.Tre.soc\tsp.Tre.soc
sp.Tre.suc\tsp.Tre.suc
th.The.afr\tth.The.afr
th.The.hyp\tth.The.hyp
th.The.sub\tth.The.sub
ar.Sul.sol\tar.Sul.sol
ar.The.pen\tar.The.pen
ar.Hal.val\tar.Hal.val
ar.Hal.sac\tar.Hal.sac
ar.Nat.mag\tar.Nat.mag
ar.Met.fer\tar.Met.fer
ar.Met.jan\tar.Met.jan
ar.Met.ore\tar.Met.ore
ar.The.chi\tar.The.chi
ar.The.ste\tar.The.ste
hi.Act.eur\thi.Act.eur
hi.Bif.inf\thi.Bif.inf
hi.Cel.cel\thi.Cel.cel
hi.Cor.oti\thi.Cor.oti
hi.Mic.par\thi.Mic.par
hi.Mic.ech\thi.Mic.ech
hi.Myc.cel\thi.Myc.cel
hi.Myc.int\thi.Myc.int
hi.Noc.fla\thi.Noc.fla
hi.Str.rim\thi.Str.rim
lo.Ach.lai\tlo.Ach.lai
lo.Bac.alc\tlo.Bac.alc
lo.Bac.sub\tlo.Bac.sub
lo.Car.mob\tlo.Car.mob
lo.Eub.lac\tlo.Eub.lac
lo.Lac.car\tlo.Lac.car
lo.Myc.cap\tlo.Myc.cap
lo.Myc.myc\tlo.Myc.myc
lo.Spi.cit\tlo.Spi.cit
lo.Sta.hom\tlo.Sta.hom
mt.Dro.mel\tmt.Dro.mel
mt.Gal.gal\tmt.Gal.gal
mt.Bos.tau\tmt.Bos.tau
mt.Hom.sap\tmt.Hom.sap
mt.Mus.mus\tmt.Mus.mus
mt.Sac.cer\tmt.Sac.cer
mt.Zea.may\tmt.Zea.may
mt.Gly.max\tmt.Gly.max
mt.Tet.pyr\tmt.Tet.pyr
al.Aci.cry\tal.Aci.cry
al.Met.rho\tal.Met.rho
al.Rhi.fre\tal.Rhi.fre
al.Ric.pea\tal.Ric.pea
al.Sph.ter\tal.Sph.ter
be.Hyd.pal\tbe.Hyd.pal
be.Nei.fla\tbe.Nei.fla
be.Pse.syz\tbe.Pse.syz
be.Rho.pur\tbe.Rho.pur
be.Tha.ter\tbe.Tha.ter
de.Des.vib\tde.Des.vib
de.Des.ter\tde.Des.ter
de.Des.suc\tde.Des.suc
ep.Alv.pom\tep.Alv.pom
ep.Cam.fet\tep.Cam.fet
ep.Cam.lar\tep.Cam.lar
ep.Hel.pyl\tep.Hel.pyl
ep.Hel.tro\tep.Hel.tro
un.Aer.mar\tun.Aer.mar
un.Fle.sin\tun.Fle.sin
un.Mag.bav\tun.Mag.bav
"""

sample_priors = {'A':0.2528,'C':0.2372,'G':0.3099,'U':0.2001}

sub_matrix = """-1.4150\t0.2372\t0.9777\t0.2001
0.2528\t-1.1940\t0.3099\t0.6313
0.7976\t0.2372\t-1.2349\t0.2001
0.2528\t0.7484\t0.3099\t-1.3111
"""
sample_sub_matrix = {}
for row_c,row in zip(gctmpca_base_order,sub_matrix.split('\n')):
    sample_sub_matrix[row_c] = dict(zip(gctmpca_base_order,row.split()))

trivial_seqs = """3 4
A1..  AACF
A12.  AADF
A123  ADCF
"""      

trivial_tree = """(A1..:0.5,(A12.,A123):0.5);
"""

trivial_species_names = """A1..
A12.
A123
"""

trivial_seq_to_species = """A1..\tA1..
A12.\tA12.
A123\tA123
"""
    
myog_seqs = """42 153
Alligator......  ELSDQEWKHVLDIWTKVESKLPEHGHEVIIRLLQEHPETQERFEKFKHMKTADEMKSSEKMKQHGNTVFTALGNILKQKGNHAEVLKPLAKSHALEHKIPVKYLEFISEIIVKVIAEKYPADFGADSQAAMRKALELFRNDMASKYKEFGYQG
Aptenodytes....  GLNDQEWQQVLTMWGKVESDLAGHGHAVLMRLFKSHPETMDRFDKFRGLKTPDEMRGSEDMKKHGVTVLT-LGQILKKKGHHEAELKPLSQTHATKHKVPVKYLEFISEAIMKVIAQKHASNFGADAQEAMKKALELFRNDMASKYKEFGFQG
Balaenoptera...  VLTDAEWHLVLNIWAKVEADVAGHGQDILISLFKGHPETLEKFDKFKHLKTEAEMKASEDLKKHGNTVLTALGGILKKKGHHEAELKPLAQSHATKHKIPIKYLEFISDAIIHVLHSRHPADFGADAQAAMNKALELFRKDIAAKYKELGFQG
Bos............  GLSDGEWQLVLNAWGKVEADVAGHGQEVLIRLFTGHPETLEKFDKFKHLKTEAEMKASEDLKKHGNTVLTALGGILKKKGHHEAEVKHLAESHANKHKIPVKYLEFISDAIIHVLHAKHPSDFGADAQAAMSKALELFRNDMAAQYKVLGFHG
Callithrix.....  GLSDGEWQLVLNVWGKVEADIPSHGQEVLISLFKGHPETLEKFDKFKHLKSEDEMKASEELKKHGVTVLTALGGILKKKGHHEAELKPLAQSHATKHKIPVKYLEFISDAIVHVLQKKHPGDFGADAQGAMKKALELFRNDMAAKYKELGFQG
Canis..........  GLSDGEWQIVLNIWGKVETDLAGHGQEVLIRLFKNHPETLDKFDKFKHLKTEDEMKGSEDLKKHGNTVLTALGGILKKKGHHEAELKPLAQSHATKHKIPVKYLEFISDAIIQVLQSKHSGDFHADTEAAMKKALELFRNDIAAKYKELGFQG
Caretta........  GLSDDEWNHVLGIWAKVEPDLSAHGQEVIIRLFQLHPETQERFAKFKNLTTIDALKSSEEVKKHGTTVLTALGRILKQKNNHEQELKPLAESHATKHKIPVKYLEFICEIIVKVIAEKHPSDFGADSQAAMKKALELFRNDMASKYKEFGFQG
Castor.........  GLSDGEWQLVLHVWGKVEADLAGHGQEVLIRLFKGHPETLEKFNKFKHIKSEDEMKASEDLKKHGVTVLTALGGVLKKKGHHEAEIKPLAQSHATKHKIPIKYLEFISEAIIHVLQSKHPGBFGADABGAMNKALELFRKDIAAKYKELGFQG
Ctenodactylus..  GLSDGEWQLVLNAWGKVETDIGGHGQEVLIRLFKGHPETLEKFDKFKHLKSEDEMKASEDLKKHGTTVLTALGNILKKKGQHEAELAPLAQSHATKHKIPVKYLEFISEAIIQVLESKHPGDFGADAQGAMSKALELFRNDIAAKYKELGFQG
Didelphis......  GLSDGEWQLVLNAWGKVEADIPGHGQEVLIRLFKGHPETLEKFDKFKHLKSEDEMKASEDLKKHGATVLTALGNILKKKGNHEAELKPLAQSHATKHKISVQFLEFISEAIIQVIQSKHPGDFGGDAQAAMGKALELFRNDMAAKYKELGFQG
Elephas........  GLSDGEWELVLKTWGKVEADIPGHGETVFVRLFTGHPETLEKFDKFKHLKTEGEMKASEDLKKQGVTVLTALGGILKKKGHHEAEIQPLAQSHATKHKIPIKYLEFISDAIIHVLQSKHPAEFGADAQGAMKKALELFRNDIAAKYKELGFQG
Equus..........  GLSDGEWQQVLNVWGKVEADIAGHGQEVLIRLFTGHPETLEKFDKFKHLKTEAEMKASEDLKKHGTVVLTALGGILKKKGHHEAELKPLAQSHATKHKIPIKYLEFISDAIIHVLHSKHPGDFGADAQGAMTKALELFRNDIAAKYKELGFQG
Gallus.........  GLSDQEWQQVLTIWGKVEADIAGHGHEVLMRLFHDHPETLDRFDKFKGLKTPDQMKGSEDLKKHGATVLTQLGKILKQKGNHESELKPLAQTHATKHKIPVKYLEFISEVIIKVIAEKHAADFGADSQAAMKKALELFRNDMASKYKEFGFQG
Graptemys......  GLSDDEWHHVLGIWAKVEPDLSAHGQEVIIRLFQVHPETQERFAKFKNLKTIDELRSSEEVKKHGTTVLTALGRILKLKNNHEPELKPLAESHATKHKIPVKYLEFICEIIVKVIAEKHPSDFGADSQAAMRKALELFRNDMASKYKEFGFQG
Inia...........  GLSDGEWQLVLNIWGKVEADLAGHGQDVLIRLFKGHPETLEKFDKFKHLKTEAEMKASEDLKKHGNTVLTALGGILKKKGHHEAELKPLAQSHATKHKIPIKYLEFISEAIIHVLHSRHPGDFGADAQAAMNKALELFRKDIAAKYKELGFHG
Kogia..........  VLSEGEWQLVLHVWAKVEADIAGHGQDILIRLFKHHPETLEKFDRFKHLKSEAEMKASEDLKKHGVTVLTALGAILKKKGHHEAELKPLAQSHATKHKIPIKYLEFISEAIIHVLHSRHPADFGADAQGAMSKALELFRKDIAAKYKELGYQG
Lagostomus.....  GLSDGEWQLVLNVWGKVEADLGGHGQEVLIRLFKGHPETLEKFDKFKHLKAEDEMRASEDLKKHGTTVLTALGGILKKRGQHAAELAPLAQSHATKHKIPVKYLEFISEAIIQVLQSKHPGDFGADAQAAMSKALELFRNDIAAKYKELGFQG
Lagothrix......  GLSDGEWQLVLNIWGKVEADIPSHGQEVLISLFKGHPETLEKFDKFKHLKSEDEMKASEELKKHGVTVLTALGGILKKKGQHEAELKPLAQSHATKHKIPVKYLEFISDAIIHALQKKHPGDFGADAQGAMKKALELFRNDMAAKYKELGFQG
Lutra..........  GLSDGEWQLVLNVWGKVEADLAGHGQEVLIRLFKGHPETLEKFDKFKHLKSEDEMKGSEDLKKHGNTVLTALGGILKKKGKHEAELKPLAQSHATKHKIPIKYLEFISEAIIQVLQSKHPGBFGADAQGAMKRALELFRNDIAAKYKELGFQG
Macropus.......  GLSDGEWQLVLNIWGKVETDEGGHGKDVLIRLFKGHPETLEKFDKFKHLKSEDEMKASEDLKKHGITVLTALGNILKKKGHHEAELKPLAQSHATKHKIPVQFLEFISDAIIQVIQSKHAGNFGADAQAAMKKALELFRHDMAAKYKEFGFQG
Meles..........  GLSDGEWQLVLNVWGKVEADLAGHGQEVLIRLFKGHPETLEKFDKFKHLKSEDEMKGSEDLKKHGNTVLTALGGILKKKGHQEAELKPLAQSHATKHKIPVKYLEFISDAIAQVLQSKHPGNFAAEAQGAMKKALELFRNDIAAKYKELGFQG
Mus............  GLSDGEWQLVLNVWGKVEADLAGHGQEVLIGLFKTHPETLDKFDKFKNLKSEEDMKGSEDLKKHGCTVLTALGTILKKKGQHAAEIQPLAQSHATKHKIPVKYLEFISEIIIEVLKKRHSGDFGADAQGAMSKALELFRNDIAAKYKELGFQG
Ochotona.......  GLSDGEWQLVLNVWGKVEADLAGHGQEVLIRLFKNHPETLEKFDKFKNLKSEDEMKGSDDLKKHGNTVLSALGGILKKKGQHEAELKPLAQSHATKHKIPVKYLEFISEAIIQVLQSKHPGDFGADAQGAMSKALELFRNDMAAKYKELGFQG
Ondatra........  GLSDGEWQLVLHVWGKVEADLAGHGQDVLIRLFKAHPETLEKFDKFKHIKSEDEMKGSEDLKKHGBTVLTALGGILKKKGHHEAEIKPLAQSHATKHKIPIKYLEFISEAIIHVLZSKHPSBFGADVZGAMKRALELFRNDIAAKYKELGFQG
Orcinus........  GLSDGEWQLVLNVWGKVEADLAGHGQDILIRLFKGHPETLEKFDKFKHLKTEADMKASEDLKKHGNTVLTALGAILKKKGHHDAELKPLAQSHATKHKIPIKYLEFISEAIIHVLHSRHPAEFGADAQGAMNKALELFRKDIAAKYKELGFHG
Ornithorhynchus  GLSDGEWQLVLKVWGKVEGDLPGHGQEVLIRLFKTHPETLEKFDKFKGLKTEDEMKASADLKKHGGTVLTALGNILKKKGQHEAELKPLAQSHATKHKISIKFLEYISEAIIHVLQSKHSADFGADAQAAMGKALELFRNDMAAKYKEFGFQG
Orycteropus....  GLSDAEWQLVLNVWGKVEADIPGHGQDVLIRLFKGHPETLEKFDRFKHLKTEDEMKASEDLKKHGTTVLTALGGILKKKGQHEAEIQPLAQSHATKHKIPVKYLEFISEAIIQVIQSKHSGDFGADAQGAMSKALELFRNDIAAKYKELGFQG
Ovis...........  GLSDGEWQLVLNAWGKVEADVAGHGQEVLIRLFTGHPETLEKFDKFKHLKTEAEMKASEDLKKHGNTVLTALGGILKKKGHHEAEVKHLAESHANKHKIPVKYLEFISDAIIHVLHAKHPSNFGADAQGAMSKALELFRNDMAAEYKVLGFQG
Pan............  GLSDGEWQLVLNVWGKVEADIPGHGQEVLIRLFKGHPETLEKFDKFKHLKSEDEMKASEDLKKHGATVLTALGGILKKKGHHEAEIKPLAQSHATKHKIPVKYLEFISECIIQVLHSKHPGDFGADAQGAMNKALELFRKDMASNYKELGFQG
Papio..........  GLSDGEWQLVLNVWGKVEADIPSHGQEVLIRLFKGHPETLEKFDKFKHLKSEDEMKASEDLKKHGATVLTALGGILKKKGHHEAEIKPLAQSHATKHKIPVKYLELISESIIQVLQSKHPGDFGADAQGAMNKALELFRNDMAAKYKELGFQG
Phocoenoides...  GLSEGEWQLVLNVWGKVEADLAGHGQDVLIRLFKGHPETLEKFDKFKHLKTEAEMKASEDLKKHGNTVLTALGGILKKKGHHDAELKPLAQSHATKHKIPIKYLEFISEAIIHVLHSRHPAEFGADAQGAMNKALELFRKDIATKYKELGFHG
Physeter.......  VLSEGEWQLVLHVWAKVEADVAGHGQDILIRLFKSHPETLEKFDRFKHLKTEAEMKASEDLKKHGVTVLTALGAILKKKGHHEAELKPLAQSHATKHKIPIKYLEFISEAIIHVLHSRHPGDFGADAQGAMNKALELFRKDIAAKYKELGYQG
Pongo..........  GLSDGEWQLVLNVWGKVEADIPSHGQEVLIRLFKGHPETLEKFDKFKHLKSEDEMKASEDLKKHGATVLTALGGILKKKGHHEAEIKPLAQSHATKHKIPVKYLEFISESIIQVLQSKHPGDFGADAQGAMNKALELFRKDMASNYKELGFQG
Proechimys.....  GLSDGEWQLVLNVWGKVEGDLSGHGQEVLIRLFKGHPETLEKFDKFKHLKAEDEMRASEELKKHGTTVLTALGGILKKKGQHAAELAPLAQSHATKHKIPVKYLEFISEAIIQVLQSKHPGDFGADAQGAMSKALELFRNDIAAKYKELGFQG
Rousettus......  GLSDGEWQLVLNVWGKVEADIPGHGQEVLIRLFKGHPETLEKFDKFKHLKSEDEMKASEDLKKHGATVLTALGGILKKKGQHEAQLKPLAQSHATKHKIPVKYLEFISEVIIQVLQSKHPGDFGADAQGAMGKALELFRNDIAAKYKELGFQG
Saimiri........  GLSDGEWQLVLNIWGKVEADIPSHGQEVLISLFKGHPETLEKFDKFKHLKSEDEMKASEELKKHGTTVLTALGGILKKKGQHEAELKPLAQSHATKHKIPVKYLELISDAIVHVLQKKHPGDFGADAQGAMKKALELFRNDMAAKYKELGFQG
Spalax.........  GLSDGEWQLVLNVWGKVEGDLAGHGQEVLIKLFKNHPETLEKFDKFKHLKSEDEMKGSEDLKKHGNTVLTALGGILKKKGQHAAEIQPLAQSHATKHKIPIKYLEFISEAIIQVLQSKHPGDFGADAQGAMSKALELFRNDIAAKYKELGFQG
Tachyglossus...  GLSDGEWQLVLKVWGKVETDITGHGQDVLIRLFKTHPETLEKFDKFKHLKTEDEMKASADLKKHGGVVLTALGSILKKKGQHEAELKPLAQSHATKHKISIKFLEFISEAIIHVLQSKHSADFGADAQAAMGKALELFRNDMATKYKEFGFQG
Tupaia.........  GLSDGEWQLVLNVWGKVEADVAGHGQEVLIRLFKGHPETLEKFDKFKHLKTEDEMKASEDLKKHGNTVLSALGGILKKKGQHEAEIKPLAQSHATKHKIPVKYLEFISEAIIQVLQSKHPGDFGADAQAAMSKALELFRNDIAAKYKELGFQG
Varanus........  GLSDEEWKKVVDIWGKVEPDLPSHGQEVIIRMFQNHPETQDRFAKFKNLKTLDEMKNSEDLKKHGTTVLTALGRILKQKGHHEAEIAPLAQTHANTHKIPIKYLEFICEVIVGVIAEKHSADFGADSQEAMRKALELFRNDMASRYKELGFQG
Zalophus.......  GLSDGEWQLVLNIWGKVEADLVGHGQEVLIRLFKGHPETLEKFDKFKHLKSEDEMKRSEDLKKHGKTVLTALGGILKKKGHHDAELKPLAQSHATKHKIPIKYLEFISEAIIHVLQSKHPGDFGADTHAAMKKALELFRNDIAAKYRELGFQG
Ziphius........  GLSEAEWQLVLHVWAKVEADLSGHGQEILIRLFKGHPETLEKFDKFKHLKSEAEMKASEDLKKHGHTVLTALGGILKKKGHHEAELKPLAQSHATKHKIPIKYLEFISDAIIHVLHSRHPSDFGADAQAAMTKALELFRKDIAAKYKELGFHG
"""
myog_seq_to_species = """Alligator......	Alligator......
Aptenodytes....	Aptenodytes....
Balaenoptera...	Balaenoptera...
Bos............	Bos............
Callithrix.....	Callithrix.....
Canis..........	Canis..........
Caretta........	Caretta........
Castor.........	Castor.........
Ctenodactylus..	Ctenodactylus..
Didelphis......	Didelphis......
Elephas........	Elephas........
Equus..........	Equus..........
Gallus.........	Gallus.........
Graptemys......	Graptemys......
Inia...........	Inia...........
Kogia..........	Kogia..........
Lagostomus.....	Lagostomus.....
Lagothrix......	Lagothrix......
Lutra..........	Lutra..........
Macropus.......	Macropus.......
Meles..........	Meles..........
Mus............	Mus............
Ochotona.......	Ochotona.......
Ondatra........	Ondatra........
Orcinus........	Orcinus........
Ornithorhynchus	Ornithorhynchus
Orycteropus....	Orycteropus....
Ovis...........	Ovis...........
Pan............	Pan............
Papio..........	Papio..........
Phocoenoides...	Phocoenoides...
Physeter.......	Physeter.......
Pongo..........	Pongo..........
Proechimys.....	Proechimys.....
Rousettus......	Rousettus......
Saimiri........	Saimiri........
Spalax.........	Spalax.........
Tachyglossus...	Tachyglossus...
Tupaia.........	Tupaia.........
Varanus........	Varanus........
Zalophus.......	Zalophus.......
Ziphius........	Ziphius........
"""

myog_tree = """((((((((Alligator......:0.14157,(Caretta........:0.02868,Graptemys......:0.02361):0.07084):0.02380,Varanus........:0.11672):0.04036,(Aptenodytes....:0.09769,Gallus.........:0.07336):0.03201):0.06365,(Ornithorhynchus:0.02648,Tachyglossus...:0.03235):0.05190):0.00632,(Didelphis......:0.03986,Macropus.......:0.06472):0.01338):0.01350,((((((Balaenoptera...:0.05296,(Kogia..........:0.01638,Physeter.......:0.01630):0.03528):0.00341,Ziphius........:0.04397):0.01061,(Inia...........:0.01297,(Orcinus........:0.01876,Phocoenoides...:0.01392):0.01971):0.00818):0.02315,((Bos............:0.01434,Ovis...........:0.01180):0.06631,Equus..........:0.03500):0.00381):0.00467,Elephas........:0.09030):0.01398,((Canis..........:0.05867,Zalophus.......:0.04591):0.00949,(((Castor.........:0.03609,Ondatra........:0.04888):0.01934,Lutra..........:0.01988):0.00609,Meles..........:0.04293):0.00277):0.00653):0.00260):0.00531,(((Callithrix.....:0.00920,(Lagothrix......:0.01063,Saimiri........:0.01551):0.00387):0.03812,((Pan............:0.00842,Pongo..........:0.01118):0.01307,Papio..........:0.01307):0.01907):0.00421,Rousettus......:0.02439):0.00691):0.00513,((Ctenodactylus..:0.03010,(Lagostomus.....:0.01528,Proechimys.....:0.01740):0.01892):0.00914,Orycteropus....:0.04642):0.00078,(((Mus............:0.07859,Spalax.........:0.02599):0.01114,Ochotona.......:0.03134):0.00969,Tupaia.........:0.02462):0.00576);
"""

myog_species_names = """Alligator......
Aptenodytes....
Balaenoptera...
Bos............
Callithrix.....
Canis..........
Caretta........
Castor.........
Ctenodactylus..
Didelphis......
Elephas........
Equus..........
Gallus.........
Graptemys......
Inia...........
Kogia..........
Lagostomus.....
Lagothrix......
Lutra..........
Macropus.......
Meles..........
Mus............
Ochotona.......
Ondatra........
Orcinus........
Ornithorhynchus
Orycteropus....
Ovis...........
Pan............
Papio..........
Phocoenoides...
Physeter.......
Pongo..........
Proechimys.....
Rousettus......
Saimiri........
Spalax.........
Tachyglossus...
Tupaia.........
Varanus........
Zalophus.......
Ziphius........
"""
if __name__ == "__main__":
    main()
