## __init__.py
## 
## Copyright (C) 2005-2007 Mykola Lynnyk <pydefiner@gmail.com>
## Copyright (C) 2008-2009 Stefan J. Betz <stefan_betz@gmx.net>
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published
## by the Free Software Foundation; version 3 only.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

import ConfigParser
from ConfigParser import SafeConfigParser
from os import environ, F_OK, R_OK, X_OK, access, mkdir, chmod, stat
from os.path import isdir
from stat import S_IRUSR, S_IWUSR, S_ISUID, ST_MODE
from sqlite3 import connect, register_adapter, version_info
from threading import RLock
from pyneighborhood.misc import homeconfdir, print_debug, parse_args, get_desktop_environ
from pyneighborhood.database import init_database

# DO NOT move this !!!
# Parsing arguments
parse_args()

# Memory database of SQLite 3
db = connect(":memory:", check_same_thread = False)
dblock = RLock()
# Initialise the database
init_database(db)
# Fixing SQLite 2.4.1 behavior first occured in Python 2.6:
if version_info >= (2,4,1):
    register_adapter(str, lambda s:s.decode('utf-8'))

if not isdir(homeconfdir()):
    print_debug("Creating Config Directory...")
    mkdir(homeconfdir())

config = SafeConfigParser()

if not access(homeconfdir() + "/config", F_OK):
    print_debug("Creating Default Config...")
    config.add_section("Main")
    config.set("Main", "mount_directory", environ["HOME"] + "/pyNeighborhood")
    if not isdir(environ["HOME"] + "/pyNeighborhood"):
        mkdir(environ["HOME"] + "/pyNeighborhood")
    config.set("Main", "cleanup_dirs", "True")
    config.set("Main", "enable_file_manager", "True")
    config.set("Main", "smbclient", "/usr/bin/smbclient")
    config.set("Main", "nmblookup", "/usr/bin/nmblookup")
    config.set("Main", "smbtree", "/usr/bin/smbtree")
    config.set("Main", "username", "")
    config.set("Main", "password", "")
    config.set("Main", "legacy_browsing", "False")
    config.set("Main", "workgroup", "")
    config.set("Main", "filemanager", "/usr/bin/nautilus")
    config.set("Main", "enable_sudo", "True")
    config.set("Main", "sudo", "/usr/bin/gksu")
    config.add_section("CIFS")
    config.set("CIFS", "mount", "/sbin/mount.cifs")
    config.set("CIFS", "unmount", "/sbin/umount.cifs")
    config.set("CIFS", "options", "")
    f = file(homeconfdir() + "/config", "w")
    config.write(f)
    f.flush()
    f.close()
    del f
    chmod(homeconfdir() + "/config", S_IRUSR + S_IWUSR)
else:
    f = file(homeconfdir() + "/config", "r")
    config.readfp(f)
    f.close()
    del f

# Ensure that executables like smbclient, nmblookup, ... really exist
# The conditions are valid defaults defined below...
defaults =  {"nmblookup":   "/usr/bin/nmblookup",
             "smbclient":   "/usr/bin/smbclient",
             "smbtree":     "/usr/bin/smbtree",
             "filemanager": "/usr/bin/nautilus"}

for file in defaults.keys():
    if not access( config.get("Main", file), F_OK | X_OK ):
        print_debug( "%s isn't accessable, falling back to default: %s"
                    % ( config.get("Main", file), defaults[file] ) )
        config.set("Main", file, defaults[file])

for utility in ["mount", "unmount"]:
    print_debug("Checking for %s" % config.get('CIFS',utility))
    if not access( config.get("CIFS", utility), F_OK ):
        config.set('CIFS',utility,'/sbin/%s.cifs' % ('mount' if utility == 'mount' else 'umount'))
    print_debug("Checking for %s" % config.get('CIFS',utility))
    if not access( config.get("CIFS", utility), F_OK):
        config.set('CIFS',utility,'/bin/%s' % ('mount' if utility == 'mount' else 'umount'))
        print_debug("Using %s for future operations." % config.get('CIFS',utility))
        
sudo_defaults = { "gnome":   "/usr/bin/gksu",
                  "kde":     "/usr/bin/kdesudo",
                  "lxde":    "/usr/bin/gksu",
                  "xfce":    "/usr/bin/gksu",
                  "default": "/usr/bin/gksu" }
        
if not access( config.get("Main", "sudo"), F_OK | X_OK):
    print_debug("%s isn't accessable... trying to determine correct sudo wrapper..." % config.get("Main", "sudo"))
    if get_desktop_environ() in sudo_defaults.keys():
        print_debug("Detected %s desktop environment... falling back to %s as sudo wrapper..." % ( get_desktop_environ(), sudo_defaults[get_desktop_environ()] ))
        config.set("Main", "sudo", sudo_defaults[get_desktop_environ()])
    else:
        print_debug("Couldn't get valid sudo wrapper for detected desktop environment %s, falling back to default %s..." % (get_desktop_environ(), sudo_defaults["default"]))
        config.set("Main", "sudo", sudo_defaults["default"])

fileman_defaults = { "gnome":   "/usr/bin/nautilus",
                     "kde":     "/usr/bin/dolphin",
                     "lxde":    "/usr/bin/pcmanfm",
                     "xfce":    "/usr/bin/thunar",
                     "default": "/usr/bin/nautilus" }
        
if not access( config.get("Main", "filemanager"), F_OK | X_OK):
    print_debug("%s isn't accessable... trying to determine correct filemanager..." % config.get("Main", "filemanager"))
    if get_desktop_environ() in fileman_defaults.keys():
        print_debug("Detected %s desktop environment... falling back to %s as filemanager..." % ( get_desktop_environ(), fileman_defaults[get_desktop_environ()] ))
        config.set("Main", "filemanager", fileman_defaults[get_desktop_environ()])
    else:
        print_debug("Couldn't get valid filemanager for detected desktop environment %s, falling back to default %s..." % (get_desktop_environ(), fileman_defaults["default"]))
        config.set("Main", "filemanager", fileman_defaults["default"])
     

# Checking for SUID bit
# If it's not set, we'll force sudo
if ( stat( config.get('CIFS','mount') )[ST_MODE] & S_ISUID ) == 0:
    config.set("Main", "enable_sudo", "True")
    print_debug( "SUID bit on %s is not set, forcing sudo usage..." % config.get('CIFS', 'mount') )
