#!/usr/bin/python

"""Device to represent CPU control."""
# (C) Copyright IBM Corp. 2008-2009
# Licensed under the GPLv2.
import discovery
import os
import subprocess
import pwrkap_data
import util
import datetime
import dircache

SYSFS_CPU_DIR = "/sys/devices/system/cpu/"
SYSFS_CPUFREQ_DIR = "/cpufreq/"
PROC_CPUINFO = "/proc/cpuinfo"
SYSFS_CPU_PACKAGE_ID_FILE = "/topology/physical_package_id"
SYSFS_CPU_CPUFREQ_DIR = "cpufreq"
SYSFS_CPU_AFFINITY_FILE = "/" + SYSFS_CPU_CPUFREQ_DIR + "/affected_cpus"
SYSFS_CPU_RELATED_FILE = "/" + SYSFS_CPU_CPUFREQ_DIR + "/related_cpus"
SYSFS_CPU_HOTPLUG_FILE = "/online"
SYSFS_CPU_GOVERNOR_FILE = "/" + SYSFS_CPU_CPUFREQ_DIR + "/scaling_governor"
DEFAULT_CPU_GOVERNOR = "ondemand"

class cpu_device(pwrkap_data.device):
	"""Driver for a real CPU, as controlled by /proc and /sys."""

	def __init__(self, id, stat_reader):
		"""Create a CPU device."""
		global SYSFS_CPU_DIR, SYSFS_CPUFREQ_DIR

		self.id = id
		self.cpufreq_dir = SYSFS_CPU_DIR + "cpu" + str(id) + SYSFS_CPUFREQ_DIR
		self.stat_reader = stat_reader
		self.last_util = None
		self.last_util_time = None
		self.load_process = None

	def get_power_states(self):
		"""Return the power states supported by this CPU."""
		res = []
		states = util.read_line_as_array(self.cpufreq_dir + "scaling_available_frequencies")
		if states == None:
			return []

		states.sort(cmp = util.cmp_string_as_number)
		max_state = int(states[len(states) - 1])
		for state in states:
			res.append((int(state), float(state) / max_state))
		return res
		
	def get_max_power_state(self):
		"""Return the maximum power state."""
		data = util.read_line_as_array(self.cpufreq_dir + "scaling_max_freq")
		return int(data[0])

	def set_max_power_state(self, max_pstate):
		"""Set the maximum power state."""
		pstates = self.get_power_states()
		for (pstate, junk) in pstates:
			if pstate == max_pstate:
				util.write_file(self.cpufreq_dir + "scaling_max_freq", str(max_pstate))
				return True
		return False

	def get_current_power_state(self):
		"""Return the current power state."""
		data = util.read_line_as_array(self.cpufreq_dir + "scaling_cur_freq")
		return int(data[0])

	def get_power_state_performance_potential(self, state):
		"""Return the performance potential of a given state."""
		for (freq, potential) in self.get_power_states():
			if freq == state:
				return potential
		return None

	def get_utilization_details(self):
		"""Return details of the device's utilization."""
		now = datetime.datetime.utcnow()
		if self.last_util_time == None or \
		   self.last_util == None or \
		   (now - self.last_util_time) > pwrkap_data.ONE_SECOND:
			(u, i) = self.stat_reader.read(self.id)
			if u + i == 0:
				i = 1

			pot = self.get_power_state_performance_potential(self.get_current_power_state())
			if pot == None:
				pot = 1.0

			self.last_util = (float(u) / (i + u)) * pot
			self.last_util_time = now

		return {self.get_name(): self.last_util}

	def inventory(self):
		key = self.get_name()
		obj = {"states": self.get_power_states()}

		return (key, obj)

	def get_prefix(self):
		return "cpu"

	def get_name(self):
		return self.get_prefix() + str(self.id)

	def start_load(self):
		def try_run_process(args):
			"""Try to run a process."""
			try:
				self.load_process = subprocess.Popen(args)
			except:
				return False
			return True

		if self.load_process != None:
			return True

		if try_run_process("burnP6"):
			return True

		if try_run_process(["dd", "if=/dev/zero", "of=/dev/null", "bs=1"]):
			return True

		return False

	def stop_load(self):
		if self.load_process == None:
			return
		os.kill(self.load_process.pid, 9)
		self.load_process.wait()
		self.load_process = None

PROC_CPUINFO = "/proc/cpuinfo"
class fixed_cpu_device(cpu_device):
	"""Driver for a fixed-frequency CPU, as controlled by /proc."""

	def __init__(self, id, stat_reader):
		"""Create a CPU device."""
		def extract_cpu_speed(data):
			"""Figure out the CPU's speed."""

			# Skip down to the processor entry
			for line in data:
				if len(line) > 2 and line[0] == "procesor" and int(line[2]) == id:
					break;
			for line in data:
				if len(line) > 3 and line[0] == "cpu" and line[1] == "MHz":
					return int(float(line[3]) * 1000)

			return None
		global PROC_CPUINFO
		cpu_device.__init__(self, id, stat_reader)

		self.speed = extract_cpu_speed(util.read_file_as_array(PROC_CPUINFO))

	def get_power_states(self):
		"""Return the power states supported by this CPU."""
		return [(self.speed, 1.0)]
		
	def get_max_power_state(self):
		"""Return the maximum power state."""
		return self.speed

	def set_max_power_state(self, max_pstate):
		"""Set the maximum power state."""
		if max_pstate == self.speed:
			return True
		return False

	def get_prefix(self):
		return "fixedcpu"

	def get_current_power_state(self):
		"""Return the current power state."""
		return self.speed

PROC_STAT_UPDATE_INTERVAL = 500
class proc_stat_reader:
	"""Common reader to cache accesses to /proc/stat."""

	def __init__(self):
		self.old_lines = None
		self.lines = None
		self.next_update = None
		self.update()
		self.update()

	def update(self):
		"""Update data."""
		global PROC_STAT_UPDATE_INTERVAL

		self.old_lines = self.lines
		self.lines = util.read_file_as_array("/proc/stat")
		x = datetime.datetime.utcnow()
		self.next_update = x + datetime.timedelta(milliseconds = PROC_STAT_UPDATE_INTERVAL)

	def read(self, cpu):
		"""Read usage data for a given CPU."""
		def find_cpu_use(lines, cpukey):
			"""Find the array corresponding to a CPU."""
			for line in lines:
				if line[0] == cpukey:
					return line
			return None
		now = datetime.datetime.utcnow()
		if now > self.next_update:
			self.update()
		cpukey = "cpu" + str(cpu)
		old = find_cpu_use(self.old_lines, cpukey)
		new = find_cpu_use(self.lines, cpukey)
		assert len(old) == len(new)
		
		# Column 4 is idle time; the rest indicate use.
		use = 0
		idle = 0
		for i in range(1, len(old)):
			if i == 4:
				idle = int(new[i]) - int(old[i])
			else:
				use = use + int(new[i]) - int(old[i])
		return (use, idle)

def cpus_reset():
	"""Bring all CPUs online and set them to default cpufreq control."""
	global SYSFS_CPU_DIR, SYSFS_CPU_HOTPLUG_FILE, SYSFS_CPU_GOVERNOR_FILE, DEFAULT_CPU_GOVERNOR

	for cpu in dircache.listdir(SYSFS_CPU_DIR):
		if not cpu.startswith("cpu"):
			continue
		try:
			util.write_file(SYSFS_CPU_DIR + cpu + SYSFS_CPU_HOTPLUG_FILE, "1")
			util.write_file(SYSFS_CPU_DIR + cpu + SYSFS_CPU_GOVERNOR_FILE, DEFAULT_CPU_GOVERNOR)
		except:
			pass

def get_proc_cpu_info(cpuid, attributes):
	"""Search /proc/cpuinfo for data about a particular CPU."""
	global PROC_CPUINFO

	proc_cpuinfo = util.read_file_as_array(PROC_CPUINFO, ":")

	# Find start of this cpu's section
	cpu_start = None
	for i in range(0, len(proc_cpuinfo)):
		if len(proc_cpuinfo[i]) < 2:
			continue
		if proc_cpuinfo[i][0].strip() == "processor" and \
		   int(proc_cpuinfo[i][1]) == cpuid:
			cpu_start = i
			break
	if cpu_start == None:
		return {}

	# Now loop through attributes
	res = {}
	for i in range(cpu_start + 1, len(proc_cpuinfo)):
		key = proc_cpuinfo[i][0].strip()
		if key == "processor":
			break
		if key in attributes:
			res[key] = proc_cpuinfo[i][1].strip()
	return res

seen_cpu_affinity_bug = False
def get_cpu_affinities(cpuid):
	"""Determine CPU affinity data."""
	global SYSFS_CPU_DIR, SYSFS_CPU_AFFINITY_FILE, SYSFS_CPU_PACKAGE_ID_FILE
	global seen_cpu_affinity_bug, SYSFS_CPU_RELATED_FILE

	set_all = False

	# Grab data from cpufreq sysfs attributes
	affinity = util.read_line_as_array(SYSFS_CPU_DIR + "cpu" + \
					   str(cpuid) + \
					   SYSFS_CPU_RELATED_FILE)

	if affinity == None:
		affinity = util.read_line_as_array(SYSFS_CPU_DIR + "cpu" + \
						   str(cpuid) + \
						   SYSFS_CPU_AFFINITY_FILE)

	# Errata report: If we have a Core2 (family 6, model 15+) system with
	# no coordination reported via sysfs, we have to calculate our own
	# affinity data because the kernel misreports hw coordinated cores as 
	# if they were totally independent (there are only half as many power
	# planes as cores).  For now we'll simply assume that all cores on a
	# package must run at the same speed--for quad-cores this isn't true;
	# they're dual dual-cores and if we could figure out which cores map
	# to which planes we could enable more fine-toothed control.

	cpuinfo = get_proc_cpu_info(cpuid, ["cpu family", "model"])
	if len(affinity) == 1 and int(cpuinfo["cpu family"]) == 6 and \
	   int(cpuinfo["model"]) >= 15:
		if not seen_cpu_affinity_bug:
			print "Kernel coordination bug found, applying workaround."
		seen_cpu_affinity_bug = True
		set_all = True
		this_cpu_package = util.read_line_as_array(SYSFS_CPU_DIR + "cpu" + \
							   str(cpuid) + \
							   SYSFS_CPU_PACKAGE_ID_FILE)
		affinity = [cpuid]
		for dir in dircache.listdir(SYSFS_CPU_DIR):
			if not dir.startswith("cpu"):
				continue
			try:
				other_cpu_id = int(dir[3:])
			except:
				continue
			if other_cpu_id == cpuid:
				continue
			cpu_package = util.read_line_as_array(SYSFS_CPU_DIR + dir + SYSFS_CPU_PACKAGE_ID_FILE)
			if cpu_package == this_cpu_package:
				affinity.append(other_cpu_id)
	

	# Sort data so that the lowest number CPU comes first
	affinity.sort(cmp = util.cmp_string_as_number)

	return (set_all, affinity)

def cpu_discover():
	"""Discover system CPUs."""
	global SYSFS_CPU_DIR, SYSFS_CPU_AFFINITY_FILE, SYSFS_CPU_CPUFREQ_DIR

	psr = proc_stat_reader()
	cpu_map = {}
	cpus = []
	domains = []
	fixed_cpus = []

	# Bring all CPUs online
	cpus_reset()
	
	# Find CPUs
	for cpu in dircache.listdir(SYSFS_CPU_DIR):
		if not cpu.startswith("cpu"):
			continue

		# Check for the presence of "cpufreq" file
		cpufreq_found = False
		try:
			cpu_id = int(cpu[3:])
		except:
			continue
		for handle in dircache.listdir(SYSFS_CPU_DIR + cpu + "/"):
			if handle.startswith(SYSFS_CPU_CPUFREQ_DIR):
				cpufreq_found = True
				break
		if not cpufreq_found:
			cpu_dev = fixed_cpu_device(cpu_id, psr)
			fixed_cpus.append(cpu_dev)
			discovery.PWRKAP_DEVICES.append(cpu_dev)
			continue

		# Proceed with cpu device construction
		cpu_dev = cpu_device(cpu_id, psr)
		cpu_map[cpu_id] = cpu_dev
		discovery.PWRKAP_DEVICES.append(cpu_dev)
		cpus.append(cpu_dev)

	# Map CPU affinities into domains
	for cpu_dev in cpus:
		(set_all, affinity) = get_cpu_affinities(cpu_dev.id)
		if int(affinity[0]) != cpu_dev.id:
			continue
		domain_list = []
		for cpu_id in affinity:
			domain_list.append(cpu_map[int(cpu_id)])
		domain = pwrkap_data.device_domain(domain_list)
		domain.must_set_all = set_all
		discovery.PWRKAP_DEVICE_DOMAINS.append(domain)

	# Put all the fixed CPUs into a separate domain
	if len(fixed_cpus) > 0:
		domain = pwrkap_data.device_domain(fixed_cpus)
		discovery.PWRKAP_DEVICE_DOMAINS.append(domain)

def cpu_init():
	"""Set up CPU device discovery function."""
	discovery.PWRKAP_DEVICE_DISCOVERY.append(cpu_discover)
	return True

cpu_init()
