# Copyright (c) 1999 Sun Microsystems, Inc.
# All rights reserved.
# 
# Permission is hereby granted, without written agreement and without
# license or royalty fees, to use, copy, modify, and distribute this
# software and its documentation for any purpose, provided that the
# above copyright notice and the following two paragraphs appear in
# all copies of this software.
# 
# IN NO EVENT SHALL SUN MICROSYSTEMS, INC. BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
# OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF SUN
# MICROSYSTEMS, INC. HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# 
# SUN MICROSYSTEMS, INC. SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THE SOFTWARE PROVIDED
# HEREUNDER IS ON AN "AS IS" BASIS, AND SUN MICROSYSTEMS, INC. HAS NO
# OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
# MODIFICATIONS.
#



#
# SyncMail conduit for PilotManager
# Mark Giles (mark.giles@east.sun.com)
#
# Based on SyncMemo by Alan Harder
# For PilotManager by Bharat Mediratta
#
# SyncMail will synchronize mail from a local mailbox (or nfs-mounted
# mailbox) to the Pilot mail program.  It will also send any mail
# composed on the Pilot.  As mail is read or deleted on the workstation
# its status is updated on the Pilot on the next sync.  This update is
# in one direction only -- No changes are made to the mailbox on the
# workstation.
#
# The local mailbox doesn't have to be your inbox.  Some people,
# especially those with large inboxes who don't want to synchronize
# the whole mess to their Pilot, may want to create a local mailbox
# and only save messages there that they want to have downloaded to
# their pilot.  Simply enter the path to this file as the mailbox
# location on the configuration screen.
#
# This conduit will not work with IMAP -- I'll try to add some sort
# of IMAP support in a future release when I have time, but I really
# know nothing about IMAP so it may be awhile.  Keep in mind that in
# some places, IMAP mailboxes are also accessible via NFS -- if this
# is the case for you, SyncMail may still work for you.
#
# This tool uses the mail app on the Pilot.  If your Pilot version
# doesn't have this app you're out of luck.
#
#
# It includes the following features:
#  - Gracefully handles textual enclosures and MIME multipart messages.
#    (generally ignores non-text portions)
#  - Limited filtering of messages (as provided through HotSync prefs
#    on Pilot).  Warning: This feature has not been fully tested and
#    may contain bugs.
#  - Since it doesn't (yet) update your mail on your workstation, there
#    is minimal risk of doing any real damage, even though it is alpha
#    software.
#  - Having said that, keep in mind:
#
#
#            THIS IS AN ALPHA VERSION.  USE AT YOUR OWN RISK
#
#
# Version History:
#  Version 0.1 (alpha release), 3/99
#  Version 0.2 (alpha release), 4/99
#    - Added patch from George Caswell to read mail files without the
#      Content-Length header.  If Content-Length is not present it finds
#      the end of the body by matching the next "From" line.
#
#
#  Things that still don't work and features to add:
#    - Filtering (as provided through HotSync preferences on Pilot)
#      is not thoroughly tested yet.
#    - Can't read Pilot sig pref (I don't know how to retrieve this)
#    - Doesn't add signature to outgoing mail (see previous item)
#    - Cleanup of old memo stuff in code
#    - All messages always modified on Pilot (slows fast sync)?
#    - Handle other mailboxes (adding category on Pilot)
#      For now, you can set inbox to another mail folder and it will load it
#      to the Pilot (replacing inbox of course).
#    - Changes made on Pilot (delete or category change) not copied to
#      workstation
#      This is kind of a messy area to deal with since most mail readers
#      don't like other programs messing with their mail folders so this
#      may never be fixed without IMAP support.
#    - IMAP support
#    - Mail mark.giles@east.sun.com with feature requests or bug reports
#      (hopfully less of the latter)
#
package SyncMail;

use Tk;
use TkUtils;
use strict;
use Data::Dumper;
use Carp;

my ($gConfigDialog);		# configuration dialog
my ($gDismissBtn);		# dismiss button on config dialog
my ($gYesNoDialog);		# yes/no dialog
my ($RCFILE);			# configuration file
my ($MIDFILE);			# MessageID file
my ($PREFS);			# configuration variables
my ($VERSION) = "0.2";		# SyncMail version

sub conduitInit
{
    $RCFILE = "SyncMail/SyncMail.prefs";
    $MIDFILE = "SyncMail/SyncMail.mid";

    &loadPrefs;
    $PREFS->{"inboxType"} = "local"
	unless (defined($PREFS->{"inboxType"}));
    $PREFS->{"homeDirectory"} = "$ENV{HOME}/.pilotmgr/SyncMail"
	unless (defined($PREFS->{"homeDirectory"}));
    $PREFS->{"sendPilotMail"} = 1
	unless (defined($PREFS->{"sendPilotMail"}));
    $PREFS->{"loadMailToPilot"} = 1
	unless (defined($PREFS->{"loadMailToPilot"}));
    $PREFS->{"useDTMailStatus"} = 1
	unless (defined($PREFS->{"useDTMailStatus"}));
    $PREFS->{"ignoreNonText"} = 1
	unless (defined($PREFS->{"ignoreNonText"}));
    $PREFS->{"inboxLoc"} = "$ENV{MAIL}"
	unless (defined($PREFS->{"inboxLoc"}));
    $PREFS->{"usePilotSettings"} = 1
	unless (defined($PREFS->{"usePilotSettings"}));
    $PREFS->{"truncateLength"} = 4000
	unless (defined($PREFS->{"truncateLength"}));
    $PREFS->{"doOneFullsync"} = 1
	unless (defined($PREFS->{"doOneFullsync"}));
    $PREFS->{"fullsync"} = 0
	unless (defined($PREFS->{"fullsync"}));
}

sub conduitQuit
{
    &savePrefs;
}

sub conduitInfo
{
    return { "database" =>
		{
		    "name" => PDA::Pilot::MailDatabase->dbname,
		    "creator" => PDA::Pilot::MailDatabase->creator,
		    "type" => "DATA",
		    "flags" => 0,
		    "version" => 0,
		},
	     "version" => $VERSION,
	     "author" => "Mark Giles",
	     "email" => "Mark.Giles\@Sun.COM" };
}

sub conduitConfigure
{
    my ($this, $wm) = @_;
    my ($frame, $obj, $subfr, @objs);

    unless (defined($gConfigDialog) && $gConfigDialog->Exists)
    {
	$gConfigDialog = $wm->Toplevel(-title => "Configuring SyncMail");
	$gConfigDialog->transient($wm);


	$frame = $gConfigDialog->Frame;
	
	$subfr = $frame->Frame(-relief => "ridge", -bd => 4);
	$obj = TkUtils::Label($subfr, "Inbox Location:");
	$obj->pack(-anchor => "c");

	$obj = $subfr->Radiobutton(-text => "Local File",
				   -variable => \$PREFS->{"inboxType"},
				   -value => "local",
				   -state => "normal");
	$obj->pack(-side => "top", -anchor => "w");

	@objs = TkUtils::LabelEntry($subfr, "Location: ", \$PREFS->{"inboxLoc"});
	$objs[0]->parent->pack(-padx => 25);

	$obj = $subfr->Radiobutton(-text => "IMAP (not available yet)",
				   -variable => \$PREFS->{"inboxType"},
				   -value => "imap",
				   -state => "disabled");
	$obj->pack(-side => "top", -anchor => "w");
	$subfr->pack(-side => "left", -anchor => "w", -expand => "true", -fill => "both");


	$subfr = $frame->Frame(-relief => "ridge", -bd => 4);
	$obj = TkUtils::Label($subfr, "Sync Type:");
	$obj->pack(-anchor => "c");

	$obj = $subfr->Radiobutton(-text => "Full Sync",
				   -variable => \$PREFS->{"fullsync"},
				   -value => 1);
	$obj->pack(-side => "top", -anchor => "c");

	$obj = $subfr->Radiobutton(-text => "Fast Sync",
				   -variable => \$PREFS->{"fullsync"},
				   -value => 0);
	$obj->pack(-side => "top", -anchor => "c");
	$subfr->pack(-side => "left", -anchor => "w", -expand => "true", -fill => "both");
	
	$frame->pack(-side => "top", -expand => "true", -fill => "both");
	

	$frame = $gConfigDialog->Frame(-relief => "ridge", -bd => 4);
	$obj = TkUtils::Label($frame, "General Settings:");
	$obj->pack(-anchor => "c");
	
	$obj = TkUtils::Checkbutton($frame, "Send mail composed on Pilot",
				    \$PREFS->{"sendPilotMail"});
	$obj->pack(-side => "top");
	$obj = TkUtils::Checkbutton($frame, "Ignore non-text messages and enclosures",
				    \$PREFS->{"ignoreNonText"});
	$obj->pack(-side => "top");

	# use up some space
	$obj = TkUtils::Label($frame, "");


	$obj = TkUtils::Label($frame, "This setting controls whether the Pilot's HotSync settings\nare used to control aspects such as type of sync and message\ntruncation, or whether the settins in this panel should\noverride the settings on your Pilot.");
	$obj->configure(-justify => "left");
	$obj->pack(-side => "top", -anchor => "center");

	$subfr = $frame->Frame;
	$obj = $subfr->Radiobutton(-text => "Use Pilot Settings",
				   -variable => \$PREFS->{"usePilotSettings"},
				   -value => 1);
	$obj->pack(-side => "left", -padx => 25);
	$obj = $subfr->Radiobutton(-text => "Override Pilot Settings",
				   -variable => \$PREFS->{"usePilotSettings"},
				   -value => 0);
	$obj->pack(-side => "left", -padx => 25);
	$subfr->pack(-side => "top");

	$obj = TkUtils::Checkbutton($frame, "Update Pilot from workstation",
				    \$PREFS->{"loadMailToPilot"});
	$obj->pack(-side => "top", -anchor => "w", -padx => 75);
	$subfr = $frame->Frame;
	$obj = $subfr->Label(-text => "Truncate message after: ");
	$obj->pack(-side => "left");
	$obj = $subfr->Entry(-textvariable => \$PREFS->{"truncateLength"},
			     -width => 6, -relief => "sunken");
	$obj->pack(-side => "left", -fill => "x");
        $obj = $subfr->Label(-text => " chars.");
	$obj->pack(-side => "left");
	$subfr->pack(-side => "top", -padx => 75);
	
	$obj = $frame->Label(-text => "");
	$obj->pack;

	$frame->pack(-side => "top");
	

        # Special options
        #
        $frame = $gConfigDialog->Frame(-relief => 'ridge',
                                       -bd => 4);
        $obj = TkUtils::Label($frame, "Special Options");
        $obj->pack(-anchor => 'center');

        my ($text);
        $text = ("If something drastic happens to either your Pilot mail\ndatabase or the .pilotmgr/SyncMail directory on your\nworkstation that leads to problems with SyncMail, you\ncan reset the conduit.  This forces the conduit to\nexamine all mail messages, rather than just those that\nhave been moved or deleted.");

        $obj = TkUtils::Label($frame, $text);
        $obj->configure(-justify => 'left');
        $obj->pack(-anchor => 'center');

        $obj = TkUtils::Button($frame, "Reset SyncMail", 
                               sub{ &resetSyncMail; });
        $obj->pack(-side => "top");

	$obj = $frame->Label(-text => "");
	$obj->pack;

        $frame->pack(-expand => 'true',
		     -fill => 'both');


	$gDismissBtn = TkUtils::Button($gConfigDialog, "Dismiss",
		sub{ &savePrefs; $gConfigDialog->withdraw });
	$gDismissBtn->pack(-side => "bottom", -anchor => "c");
	PilotMgr::setColors($gConfigDialog);
    }

    $gConfigDialog->Popup(-popanchor => 'c',
			  -popover => $wm,
			  -overanchor => 'c');
}


sub resetSyncMail
{
    $PREFS->{"doOneFullsync"} = 1;
}


sub conduitSync
{
    my ($this, $dlp, $info) = @_;
    my ($record, %pilot, $fullsync);
    my ($appinfo, $ret, $file, $i, $lastsync);
    my (@cats, %catcase, %catindx, $cat, @cat_ids, $pilot_dbhandle, $key);
    my ($dbinfo) = &conduitInfo->{database};
    my ($dbname) = $dbinfo->{name};
    my ($sendcnt, $delboxcnt, $incnt, $delcnt, $filtercnt);
    my ($bytesRead, $fileSize);

    $sendcnt = $delboxcnt = $incnt = $delcnt = $filtercnt = 0;
    $fullsync = 0;



    # Open or create DB:
    eval
    {
	$pilot_dbhandle = $dlp->open($dbname);
    };
    if ($@ =~ /read-only value/)
    {
      PilotMgr::msg("Pilot database '$dbname' does not exist.\n" .
		    "Skipping.");
	return;
    }
    elsif ($@)
    {
	croak($@);
    }

    if (!defined($pilot_dbhandle))
    {
	PilotMgr::msg("Unable to open '$dbname'.  Aborting!");
	return;
    }

    # Let the user know what we're doing
    #
    $dlp->getStatus();



    # Get prefs from Pilot
    my $truncate = $PREFS->{"truncateLength"};
    my $wsToPilot = $PREFS->{"loadMailToPilot"};
    my ($filter, $tofilter, $fromfilter, $subjfilter);
    $filter = 0;
    $tofilter = $fromfilter = $subjfilter = "";
    if ($PREFS->{"usePilotSettings"}) {
	my $pref = $pilot_dbhandle->getPref(1);
	$truncate = $pref->{'truncate'};
	$wsToPilot = ($pref->{'syncType'} ne "Send");
	if ($pref->{'syncType'} eq "Filter") {
            # filter = 1 for excluding matches, 2 for getting only matches
	    $filter = $pref->{'getContaining'} + 1;
	    $tofilter = $pref->{'filterTo'} if (defined($pref->{'filterTo'}));
	    $fromfilter = $pref->{'filterFrom'} if (defined($pref->{'filterFrom'}));
	    $subjfilter = $pref->{'filterSubject'} if (defined($pref->{'filterSubject'}));
	}
    }


    # Figure out if we need to do a full sync (only applies if doing ws->pilot)
    if ($wsToPilot) {
	$fullsync = $PREFS->{"fullsync"};
	if (! -e $MIDFILE) {
	    open (MID, ">$MIDFILE") || return;
	    close MID;
	    $PREFS->{"doOneFullsync"} = 1;
	    PilotMgr::msg("SyncMail MessageID file not found.  Must recreate with full sync.");
	}
	if ($PREFS->{"doOneFullsync"}) {
	    $fullsync = 1;
	    $PREFS->{"doOneFullsync"} = 0;
	}

	if ($fullsync) {
	  PilotMgr::msg("Doing full sync.");
	} else {
	  PilotMgr::msg("Doing fast sync.");
	}
    } elsif ($PREFS->{"sendPilotMail"}) {
      PilotMgr::msg("Sending mail composed on Pilot only.");
    } else {
      PilotMgr::msg("SyncMail: Nothing to do.");
	return;
    }
    

    # Get categories
    $appinfo = $pilot_dbhandle->getAppBlock();

    @cats = @{$appinfo->{categoryName}};
    @cat_ids = @{$appinfo->{categoryID}};
    foreach $i ($[..$#cats)
    {
	next unless (length($cats[$i])>0);
	$cat = $cats[$i];
	$cat =~ tr/A-Z/a-z/;
	$catcase{$cat} = $cats[$i];	# translate lowercase to anycase
	$catindx{$cat} = $i;	# translate anycase to index
    }

    
    
    if ($fullsync) {
	%pilot = &readAllMail($dlp, $pilot_dbhandle);
    } else {
	%pilot = &readMailInCategory($pilot_dbhandle, $catindx{"outbox"});
    }

    #compare and update
    if ($fullsync || $PREFS->{"sendPilotMail"}) {
	my ($id, $count, @keys, %record);
	$count = 0;
	@keys = keys(%pilot);
	foreach $id (@keys)
	{
	    if ($#keys == 0)
	    {
	      PilotMgr::status("Handling Mail From Pilot", 0);
	    }
	    else {
		unless ($count % &my_ceil($#keys / 20))
		{
		  PilotMgr::status("Handling Mail From Pilot", 
				   int (100 * $count / $#keys));
		}
	    }
	    $count++;
	    
	    $record = $pilot{$id};
	    if ($cats[$record->{category}] eq "Outbox") {
		if ($PREFS->{"sendPilotMail"}) {
		    PilotMgr::msg("Sending mail to $record->{'to'} about $record->{'subject'}");
		    # To, cc and bcc may have multiple lines.
		    $record->{'to'} =~ s/\n/\n\t/g;
		      $record->{'cc'} =~ s/\n/\n\t/g if (defined($record->{'cc'}));
		      $record->{'bcc'} =~ s/\n/\n\t/g if (defined($record->{'bcc'}));
		    
		    $file = &mktemp("/tmp");
		    open(FD, ">$file") || return;
		    print FD "To: $record->{'to'}\n";
		    print FD "Cc: $record->{'cc'}\n" if (defined($record->{cc}));
		    print FD "Bcc: $record->{'Bcc'}\n" if (defined($record->{bcc}));
		    print FD "Subject: $record->{'subject'}\n" if (defined($record->{subject}));
		    print FD "From: $record->{'from'}\n" if (defined($record->{from}));
		    print FD "X-Mailer: SyncMail $VERSION with PilotManager\n\n";
		    print FD "$record->{'body'}\n";
		    close FD;
		    
		    system("/usr/lib/sendmail -bm -t < $file");
		    unlink($file);
		    $sendcnt++;
		
		    # And remove it from the pilot
		    $ret = &deleteRecord($dlp, $pilot_dbhandle, $record->{'pilot_id'});
		    $delcnt++;
		    if ($ret < 0)
		    {
			$ret = PDA::Pilot::errorText($ret);
		        PilotMgr::msg("Error $ret deleting pilot record $record->{'pilot_id'}!!");
			$dlp->log("SyncMail: Error $ret deleting pilot record $record->{'pilot_id'}.");
		    }
		}

	    } elsif ($fullsync) {
		# Full sync: Remove from Pilot
		$ret = &deleteRecord($dlp, $pilot_dbhandle, $record->{'pilot_id'});
		$delcnt++;
		if ($ret < 0)
		{
		    $ret = PDA::Pilot::errorText($ret);
		    PilotMgr::msg("Error $ret deleting pilot record $record->{'pilot_id'}!!");
		    $dlp->log("SyncMail: Error $ret deleting pilot record $record->{'pilot_id'}.");
		}
	    }
	}
    }
    


    # Read new mail and add to Pilot
    if ($wsToPilot)
    {
        PilotMgr::status("Sending Mail To Pilot", 0);
		       
	open (IN, "$PREFS->{inboxLoc}") || return;
	$fileSize = -s IN;
	$bytesRead = 0;
	$file = &mktemp("/tmp");
	open (MIDIN, "$MIDFILE") || return;
	open (MIDOUT, ">$file") || return;

      READ:
	while(1)
	{
	    # Read a header
	    my ($hf, $hd, $msginfo, $ct, $cl, $body, $line, $firstRcv, $ch,
		$mmb, $mid);
	    $msginfo = {
		pilot_id => 0,
		to => "",
		cc => "",
		body => "",
		subject => "<None>",
		from => "???",
		replyTo => "",
		"read" => 0,
		sentTo => "",
		category => $catindx{"inbox"},
		date => [1, 2, 3, 4, 5, 6, 7, 8],
	    };
	    $ct = "text/plain";
	    $cl = 0;
	    $firstRcv = 1;
	    $ch = "";		# Current header (allows multiline headers)
	    $mmb = "";
	    $mid = "<0>";

	  LINE:
	    while($line = <IN>) {
		$bytesRead += length($line);
		chop $line;

		if ($line =~ /^\s/) { # Heading continuation
		    $ch .= $line;
		    next LINE;
		}

		if (($line eq "") || ($line =~ /^[^:]*:\s*/)) {
		    $ch =~ s/\s+/ /g;	# Strip extra spaces
		    if ($ch =~ /^([^:]*):\s*(.*)$/) {
			$hf = $1;	# Header field
			$hd = $2;	# Header data
			$hf =~ tr/A-Z/a-z/;
			
			if ($hf eq "from") {
			    $msginfo->{from} = $hd;
			} elsif (($hf eq "received") && ($firstRcv)) {
			    # Only use first received header which is the time
			    # received by users mail host.  This trick, used by
			    # many mail readers, works around any timezone
			    # issues.
			    $firstRcv = 0;
			    $hd =~ s/^[^;]*;\s*(.*)$/$1/;
			    $msginfo->{date} = &dateToPilot($hd);
			} elsif ($hf eq "to") {
			    $msginfo->{to} = $hd;
			} elsif ($hf eq "cc") {
			    $msginfo->{cc} = $hd;
			} elsif ($hf eq "reply-to") {
			    $msginfo->{replyTo} = $hd;
			} elsif ($hf eq "subject") {
			    $msginfo->{subject} = $hd;
			} elsif ($hf eq "status") {
			    if ($hd =~ /(R|r)/) {
				$msginfo->{"read"} = 1;
			    }  
			} elsif (($hf eq "x-status") && ($PREFS->{"useDTMailStatus"})) {
			    if ($hd eq "D\$\$\$") {
				$msginfo->{category} = $catindx{"deleted"};
			    }
			} elsif ($hf eq "message-id") {
			    $hd =~ s/^.*(<.*>).*$/$1/;
			    $mid = $hd;
			} elsif ($hf eq "content-length") {
			    $cl = $hd;
			} elsif ($hf eq "content-type") {
			    # split content type and rest of header
			    $hd =~ /^([^;]*);\s*(.*)$/;
			    $ct = $1; # Content type
			    $hd = $2; # Remainder
			    $ct =~ tr/A-Z/a-z/;

			    # deal with content-type multipart/mixed
			    if ($ct eq "multipart/mixed") {
				my ($attr, $val); #MIME attribute and value

				# split attribute/value pair and rest of line
				while ($hd =~ /^\s*([^=]*)=((\"[^\"]*\")|(\S*))\s*(.*)$/) {
				    $attr = $1;
				    $val = $2;
				    $hd = $5;
				    $attr =~ tr/A-Z/a-z/;
				    # set boundary if appropriate
				    $mmb = $val if ($attr eq "boundary");
				}
			    }
			}
		    }
		    
		    $ch = $line;
		}
		
		last LINE if ($line eq "");
	    }

	    # Read body
	    $body = "";
	    if ($cl > 0) {
		# Read body from content-length value:
		read (IN, $body, $cl);
	    } else {
		# Read body, matching end at next From line:
		my @body;
		while (($line = <IN>) && !( $line =~ /^From\s+\S+\s+\S+\s\S+\s.\d\s\d\d:\d\d:\d\d\s\d\d\d\d\s/)) {
		    push(@body, $line);
		}
		$body = join('', @body);
	    }
	    if (length($body) > 0) {
		$bytesRead += length($body);
	        PilotMgr::status("Sending Mail To Pilot", (100*$bytesRead)/$fileSize);

		# Find line in MID file
		$ret = 0;
		$cat = -1;
	      MID:
		while (<MIDIN>) {
		    chop;
		    /^(<.*>),(\d*),(\d*)$/;
		    if ($1 eq $mid) {
			# if not fullsync, recycle id (unless deleted)
			if ((!$fullsync)) {
			    $msginfo->{pilot_id} = $2;
			}
		        $cat = $3;
		        last MID;
			
		    } elsif (!$fullsync) {

			# Delete from Pilot (already deleted if fullsync)
			$ret = &deleteRecord($dlp, $pilot_dbhandle, $2);
			$delcnt++;
			if ($ret < 0)
			{
			    $ret = PDA::Pilot::errorText($ret);
			    PilotMgr::msg("Error $ret deleting pilot record $2!!");
			    $dlp->log("SyncMail: Error $ret deleting pilot record $2.");
			}
		    }
	        }

		# if category modified (or not found), we have to write
		if (($cat != $msginfo->{'category'}) || $fullsync) {
		    # Deal w/ multipart/mixed messages
		    if (($PREFS->{"ignoreNonText"}) && ($ct eq "multipart/mixed")
			&& ($mmb ne "")) {
			$body = &mimeMultipart($body, $mmb);
			# if successful pretend it's text/plain
			$ct = "text/plain" if ($body ne "");
		    }
		    

		    if (length($body) > $truncate) {
			substr($body, $truncate) = "\n\n--- TRUNCATED ---\n";
		    }

		    $msginfo->{body} = $body;


		    # Write to pilot if content-type text
		    if ((!$PREFS->{"ignoreNonText"}) || ($ct =~ /^text/)) {
			# check filters
			my $match = 1;
			# Must match all criteria -- is this right?
			if ($filter > 0) {
			    if ($tofilter ne "") {
				$match = 0 unless ($msginfo->{to} =~ /$tofilter/);
			    }
			    if ($fromfilter ne "") {
				$match = 0 unless ($msginfo->{from} =~ /$fromfilter/);
			    }
			    if ($subjfilter ne "") {
				$match = 0 unless ($msginfo->{subject} =~ /$subjfilter/);
			    }
			}
			my $put = 1;
			if ($filter == 1) {
			    $put = ! $match;
			} elsif ($filter == 2) {
			    $put = $match;
			}
			
			if ($put) {
			    $delboxcnt++ if ($msginfo->{category} == $catindx{"deleted"});
			    $incnt++ if ($msginfo->{category} == $catindx{"inbox"});
			
			    $ret = &writeMail($msginfo, $dlp, $pilot_dbhandle);
			    if ($ret < 0)
			    {
				$ret = PDA::Pilot::errorText($ret);
			      PilotMgr::msg("Error $ret writing mail to pilot.");
				$dlp->log("SyncMail: Error $ret writing mail to pilot.");
			    }
			    # Write info to MIDFILE
			    print MIDOUT "$mid,$ret,$msginfo->{'category'}\n";
			} else {
			    $filtercnt++;
			}
		    } else {
			# Non text -- Write info to MIDFILE so we won't
                        # consider again
			print MIDOUT "$mid,$ret,$msginfo->{'category'}\n";
		    }
		} else {
		    # Already in Pilot -- write to MIDFILE
		    print MIDOUT "$mid,$msginfo->{'pilot_id'},$cat\n";
		}
	    }

	    last READ unless (defined $line);
	}
	close IN;
	close MIDIN;
	close MIDOUT;
	if ((system "/bin/mv -f $file $MIDFILE") > 0) {
	    PilotMgr::msg("Error saving message-id file.  Will do full sync next time.");
	    $PREFS->{"doOneFullsync"} = 1;
	}

	PilotMgr::status("Sending Mail to Pilot", 100);
    }



    &cleanupPilot($dlp, $pilot_dbhandle)
	if ($PREFS->{"gDoPilotToFile"});

    $pilot_dbhandle->close();
    
    PilotMgr::msg("Sent $sendcnt mail messages.") if ($sendcnt);
    PilotMgr::msg("Deleted $delcnt mesages from Pilot.") if ($delcnt);
    PilotMgr::msg("Placed $incnt messages in Pilot inbox.") if ($incnt);
    PilotMgr::msg("Placed $delboxcnt messages in Pilot's deleted mailbox.") if ($delboxcnt);
    PilotMgr::msg("Ignored $filtercnt messages due to filters.") if ($filtercnt);
    PilotMgr::msg("No changes made.") if (($sendcnt + $delcnt + $incnt + $delboxcnt) == 0);
}

##########################################################################

sub loadPrefs
{
    my ($line);

    eval `cat $RCFILE`;

    # For some reason, we need to reference $PREFS here
    # or the preferences won't get loaded properly.
    #
    $PREFS;
}

sub savePrefs
{
    my ($var);

    $Data::Dumper::Purity = 1;
    $Data::Dumper::Deepcopy = 1;

    if (open(FD, ">$RCFILE"))
    {
	if (defined &Data::Dumper::Dumpxs)
	{
	    print FD Data::Dumper->Dumpxs([$PREFS], ['PREFS']);
	}
	else
	{
	    print FD Data::Dumper->Dump([$PREFS], ['PREFS']);
	}

	print FD "1;\n";
	close(FD);
    }
    else
    {
	PilotMgr::msg("Unable to save preferences to $RCFILE!");
    }
}

sub readAllMail
{
    my ($sock, $dbhandle) = @_;
    my ($i, $record, $id, $count_max);
    my (%db);

    $i = 0;
    PilotMgr::status("Reading Pilot Mail [full sync]", 0);
    $count_max = $dbhandle->getRecords();

    while (1)
    {
	$record = $dbhandle->getRecord($i);
	last if (!defined($record));

	unless ($i % &my_ceil($count_max / 20))
	{
	    PilotMgr::status("Reading Pilot Mail [full sync]",
			     int(100 * $i / $count_max));
	}

	$i++;

	next if ($record->{"deleted"} || $record->{"archived"} ||
		 $record->{"busy"});

	$id = $record->{"id"};
	$db{$id} = &pilotToNeutral($id, $record->{"category"}, $record);
    }
    PilotMgr::status("Reading Pilot Mail [full sync]", 100);

    return(%db);
}

sub readChangedMail
{
    my ($sock, $dbhandle) = @_;
    my ($i, $record, $id);
    my (%db);

    PilotMgr::status("Reading Pilot Mail [fast sync]", 0);
    while (1)
    {
	$record = $dbhandle->getNextModRecord();
	last if (!defined($record));

	$id = $record->{"id"};

	$db{$id} = &pilotToNeutral($id, $record->{"category"}, $record);
    }
    PilotMgr::status("Reading Pilot Mail [fast sync]", 100);

    return(%db);
}

sub readMailInCategory
{
    my ($dbhandle, $categ) = @_;
    my ($record, $id);
    my (%db);

    while (1)
    {
	$record = $dbhandle->getNextRecord($categ);
	last if (!defined($record));

	$id = $record->{"id"};
	next if (defined($db{$id}));

	$db{$id} = &pilotToNeutral($id, $record->{"category"}, $record);
    }

    return(%db);
}

sub recsDiffer
{
    my ($a, $b) = @_;
    my (%seen, $key);

    foreach $key (keys %$a, keys %$b)
    {
	next if $seen{$key}++;
	next if (&isSpecialSyncField($key));
	return 1 unless (defined($a->{$key}) && defined($b->{$key}) &&
			 $a->{$key} eq $b->{$key});
    }
    return 0;
}

sub matchRec
{
    my ($pilot, $rec) = @_;
    my ($pi);

    foreach $pi (values(%$pilot))
    {
	return $pi->{pilot_id} unless (&recsDiffer($rec, $pi));
    }
    return 0;
}

sub writeMail
{
    my ($record, $dlp, $pilot_dbhandle) = @_;
    my ($pi_rec, $id);

    $record->{pilot_id} ||= 0;
    $record->{category} ||= 0;

    $pi_rec = &neutralToPilot($record, $pilot_dbhandle);

    $pi_rec->{attr} ||= 0; # Why isn't this set?
    $pi_rec->{category} ||= 0; # Why isn't this set?

    # Turn watchdog off for this write.
    #
    #PilotMgr::watchdog($dlp, 0);
    $id = $pilot_dbhandle->setRecord($pi_rec);
    #PilotMgr::watchdog($dlp, 1);

    return $id;
}

sub deleteRecord
{
    my ($dlp, $pilot_dbhandle, $record_id) = @_;
    my ($ret);

    $ret = $pilot_dbhandle->deleteRecord($record_id);

    return ($ret);
}

sub cleanupPilot
{
    my ($dlp, $pilot_dbhandle) = @_;
    $pilot_dbhandle->purge();
    $pilot_dbhandle->resetFlags();
}

sub mktemp
{
    my ($dir, $basename) = @_;
    my ($fname, $fext) = ("Mail0000", "");
    
    $basename ||= "";
    return ("$dir/$basename") if ($basename && (! -e "$dir/$basename"));
    $fext = $1 if ($basename =~ /(\.\w+)$/);
    $fext ||= ".txt";
    
    while (1)
    {
        return ("$dir/$fname$fext") if (! -e "$dir/$fname$fext");
        $fname++;
    }
}

sub mimeMultipart
{
    my ($msg, $boundary) = @_;
    my ($text, $textType, $hf, $hd, $pl, $line, $inheader);

    $text = "";
    $inheader = $textType = 0;
    foreach $line (split (/\n/, $msg)) {
	#while ($msg =~ s/^(.*)\n((.|\n)*)/$2/) {
	#$line = $1;
	if ($inheader) {
	    if ($line =~ /^([^:]+):\s*([^;]*)(;\s*(.*))?$/) {
		$hf = $1;		# Header field
		$hd = $2;		# Header data
		$pl = $4;		# MIME Param list
		$hf =~ tr/A-Z/a-z/;
		if ($hf eq "content-type") {
		    $hd =~ tr/A-Z/a-z/;
		    $textType = 1 if ($hd eq "text/plain"); #=~ /^\s*text/g);
		    if ($pl =~ /^(.*\s)*name=((\"[^\"]*\")|(\S*))/) {
			if ($textType) {
			    $text .= "\n\n--- Document: $2\n\n";
			} else {
			    $text .= "\n\n--- Document $2 Ignored ---\n\n";
			}
		    }
		}
	    } elsif ($line eq "") {
		# End of header
		$inheader = 0;
	    }
	} elsif ($line =~ /^--${boundary}$/) {
	    $textType = 0;
	    $inheader = 1;
        } else {
	    $text .= ($line . "\n") if ($textType);
	}
    }

    return($text);
}


sub dateToPilot
{
    my ($str) = @_;
    my ($pilot, %months);
    %months = (
	       'Jan', 0,
	       'Feb', 1,
	       'Mar', 2,
	       'Apr', 3,
	       'May', 4,
	       'Jun', 5,
	       'Jul', 6,
	       'Aug', 7,
	       'Sep', 8,
	       'Oct', 9,
	       'Nov', 10,
	       'Dec', 11
	       );

    $str =~ /^((...),\s*)?(\d*)\s*(...)\s*(\d\d)?(\d\d)\s*(..):(..):(..)\s*(.*)/;
    # TTD: What are the last two numbers in the pilot's array?
    $pilot = [$9, $8, $7, $3, $months{$4}, $6, 0, 0];

    return ($pilot);
}

##########################################################################

sub pilotToNeutral
{
    my ($id, $category, $pi_rec) = @_;
    my ($record, $key);
    
    $record = {
	pilot_id => $id,
	category => $category,
    };
    #Copy all relevant fields here:
    $record->{modified} = $pi_rec->{modified};
    $record->{priority} = $pi_rec->{priority};
    $record->{to} = $pi_rec->{to};
    $record->{confirmRead} = $pi_rec->{confirmRead};
    $record->{body} = $pi_rec->{body};
    $record->{subject} = $pi_rec->{subject};
    $record->{confirmDelivery} = $pi_rec->{confirmDelivery};
    $record->{signature} = $pi_rec->{signature};
    $record->{from} = $pi_rec->{from};
    $record->{"read"} = $pi_rec->{"read"};
    $record->{sentTo} = $pi_rec->{sentTo};
    $record->{date} = $pi_rec->{date};
    $record->{category} = $pi_rec->{category};
    $record->{cc} = $pi_rec->{cc};
    $record->{bcc} = $pi_rec->{bcc};

    return($record);
}

sub neutralToPilot
{
    my ($record, $db_handle) = @_;
    my ($pi_rec) = $db_handle->newRecord();

    $pi_rec->{id} = $record->{pilot_id};
    $pi_rec->{category} = $record->{category};
    #Copy all relevant fields here:
    $pi_rec->{modified} = $record->{modified};
    $pi_rec->{priority} = $record->{priority};
    $pi_rec->{to} = $record->{to};
    $pi_rec->{confirmRead} = $record->{confirmRead};
    $pi_rec->{body} = $record->{body};
    $pi_rec->{subject} = $record->{subject};
    $pi_rec->{confirmDelivery} = $record->{confirmDelivery};
    $pi_rec->{signature} = $record->{signature};
    $pi_rec->{from} = $record->{from};
    $pi_rec->{"read"} = $record->{"read"};
    $pi_rec->{sentTo} = $record->{sentTo};
    $pi_rec->{date} = $record->{date};
    $pi_rec->{cc} = $record->{cc};
    # No bcc going to pilot
    
    # If reply-to is set, we should set from to this since the Pilot doesn't
    # know about reply-to
    $pi_rec->{from} = $record->{replyTo} if ($record->{replyTo} ne "");

    $pi_rec->{deleted} = 0;
    $pi_rec->{modified} = 0;
    $pi_rec->{priority} = 1;
    $pi_rec->{busy} = 0;
    $pi_rec->{archived} = 0;
    $pi_rec->{secret} = 0;
    $pi_rec->{confirmRead} = 0;
    $pi_rec->{signature} = 0;
    $pi_rec->{confirmDelivery} = 0;
    $pi_rec->{"index"} = 0;

    return $pi_rec;
}

sub fileToNeutral
{
    my ($id, $category, $filetext) = @_;
    my ($record);

    $record = {};
    $record->{pilot_id} = $id if ($id);
    $record->{category} = $category if (length($category) > 0);

    #Parse file and fill all relevant fields here:
    $record->{text} = $filetext;

    return($record);
}

sub neutralToFile
{
    my ($record) = @_;
    my ($filetext) = "";

    #Pull out record fields and encode into filetext:
    $filetext = $record->{text};

    return($filetext);
}

sub isSpecialSyncField
{
    my ($field) = @_;

    # Fields that should NOT be compared between Pilot and file.
    #     (ie info stored only on one side or the other, but not both)
    #     "pilot_id" should always be one of these fields.
    # return ($field eq "pilot_id" || $field eq "createdate");

    return ($field eq "pilot_id");
}

sub mergeRecs
{
    my ($dlp, $record, $pirec) = @_;
    my ($ret);

    #Merge method.  Modify fields in $record to get final result.

    $ret = "";
    if ($PREFS->{"gUseFilemerge"})
    {
	PilotMgr::msg("Running filemerge..");
	$ret = &fileMerge($dlp, $record->{text}, $pirec->{text});
	PilotMgr::msg("Using filemerged text.") if ($ret);
    }
    unless ($ret)
    {
	PilotMgr::msg("Concatenating both versions!  Merge " .
	    "changes before your next hotsync!");
	$ret = $record->{text} .
	       "\n--- ^^ File ^^ === Version Separator === vv Pilot vv ---\n" .
	       $pirec->{text};
    }
    $record->{text} = $ret;
}

sub makeFilename
{
    my ($record) = @_;
    my ($filename) = "";

    #Make a potential filename from the given record:
    if ($record->{text} =~ /^\s*<HTML>/i)
    {
	if ($record->{text} =~ /<TITLE\s*[^>]*>\s*(.*)<\/TITLE\s*[^>]*>/si)
	{
	    # Try to use the title of HTML docs
	    $filename = substr($1, 0, 16);
	}
	elsif ($record->{text} =~ /(\s*<[^>]*>)*\s*([^<]*)\s*/si)
	{
	    # Use first non-html text
	    $filename = substr($2, 0, 16);
	}
	$filename =~ s/\s+$//;
	$filename =~ s/\s+/_/g;
	$filename .= ".html" if (length($filename) > 0);
    }
    unless ($filename)
    {
	$filename = substr($record->{text}, 0, 16);
	$filename =~ s/\n.*//g;
	$filename =~ s/\s+$//;
	$filename =~ s/^\s+//;
	$filename =~ s/\s+/_/g;
	$filename .= ".txt";
    }

    return($filename);
}

sub my_ceil
{
    my ($val) = int($_[0]);

    return 1 if ($val == 0);
    return $val;
}

1;
