/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmimage
 * @short_description: An image drawable supporting various media.
 * @see_also: #PgmDrawable, #PgmText, #PgmCanvas.
 *
 * <refsect2>
 * <para>
 * #PgmImage is a drawable displaying media. It supports various ways of loading
 * images through buffers, file paths or videos through GStreamer.
 * </para>
 * <title>Loading image data</title>
 * <para>
 * Image data loading can happen with three different functions:
 * <itemizedlist>
 * <listitem>
 * <para>
 * pgm_image_set_from_buffer() takes a pre-loaded data buffer and sets it as
 * the currently displayed image. This is useful when you want to use an image
 * loading library (GdkPixbuf, FreeImage, etc) in your application and just
 * provide the pixels to #PgmImage for display. The data buffer containing the
 * pixels is copied internally, you can free the data buffer from the
 * application side as soon as the function returns.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * pgm_image_set_from_gst_buffer() takes a GStreamer #GstBuffer and sets it as
 * the currently displayed image. This is mostly used to do video rendering.
 * There's no copying of the buffer data to optimize performances, indeed the
 * reference count of the buffer is going to be increased to keep the buffer
 * around while it's needed for rendering. When you call pgm_image_clear() the
 * reference to the buffer will be decreased and the buffer can get freed. Note
 * that this method is used by #PgmSink to render video frames directly in a
 * #PgmImage when the pipeline is playing.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * pgm_image_set_from_file() takes a path to an image file delegating image
 * loading to Pigment. Thus the loading is asynchronous and won't block the
 * Pigment main loop.
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * <title>Sharing image data between #PgmImage objects</title>
 * <para>
 * pgm_image_set_from_image() is a convenient system to slave an image to
 * another one. Indeed you might want to load an image data once and then use
 * it in multiple image objects. In that case this image becomes a slave to the
 * one that has the image data loaded internally and each time it needs to draw
 * it will use that data.
 * </para>
 * <para>
 * Layout settings of the drawable are independent from one image to another.
 * That means that even if two image objects are using the same image, they can
 * have different colors, different #PgmDrawableLayoutType or different
 * #PgmDrawableAlignment.
 * </para>
 * <para>
 * Each time a new image data buffer is loaded in the master image object, all
 * the slave image objects are automatically updated. That means you can render
 * a video clip in ten different drawables without doing anything else than
 * slaving nine image objects to the one that's receiving the image data.
 * </para>
 * <title>Image data aspect ratio</title>
 * <para>
 * This rarely happens with normal images but video rendering often has non
 * square pixels video frames coming out of the video decoders (DVD, DV cameras,
 * etc). In that case a calculation has to be done when projecting to the
 * viewport so that we put in adequation both the pixel aspect ratio and the
 * source video aspect ratio. You can set the image aspect ratio using
 * pgm_image_set_aspect_ratio() and be assured that Pigment is going to render
 * that image correctly on the viewport.
 * </para>
 * <title>Benefitting from hardware acceleration</title>
 * <para>
 * Depending on the viewport implementation, some #PgmImagePixelFormat (color
 * space) can be supported or not. When it comes to video rendering, hardware
 * acceleration is very important and you need to know what kind of pixel
 * formats are convenient for the rendering backend. you can get a list of
 * supported (accelerated) pixel formats using
 * pgm_viewport_get_pixel_formats().
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-11-08 (0.3.2)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include "pgmimage.h"
#include "pgmmarshal.h"
#include <stdlib.h>     /* atoi */

GST_DEBUG_CATEGORY_STATIC (pgm_image_debug);
#define GST_CAT_DEFAULT pgm_image_debug

/* Image signals */
enum {
  FILE_LOADED,
  CLONED,
  UN_CLONED,
  LAST_SIGNAL
};

/* Structure used by the thread pool to load images */
typedef struct {
  PgmImage  *image;
  gchar     *filename;
  GdkPixbuf *pixbuf;
  gboolean   emit_signal;
  guint      max_size;
  guint16    id;
} PixbufLoaderData;

/* Image internal flags */
enum {
  PGM_IMAGE_STORED_FROM_FILE_LOADED = (1 << 0),
  PGM_IMAGE_VISIBLE                 = (1 << 1)
};

static PgmDrawableClass *parent_class = NULL;
static guint pgm_image_signals[LAST_SIGNAL] = { 0 };

/* Private methods */

/* Update the ratio of the slaves. Called with Image LOCK. */
static void
update_slaves_ratio (PgmImage *image)
{
  PgmImage *slave;
  GList *walk;

  walk = image->slaves;
  while (walk)
    {
      slave = walk->data;
      slave->par_n = image->par_n;
      slave->par_d = image->par_d;
      walk = walk->next;
    }
}

/* Cancel loading tasks in the thread pool for an image.
 * Called with Image LOCK. */
static void
cancel_async_loading_task (PgmImage *image)
{
  image->loader_id = 0;
}

/* Get whether or not a loading has been cancelled */
static gboolean
is_loading_cancelled (PixbufLoaderData *loader)
{
  PgmImage *image = loader->image;
  guint id;

  GST_OBJECT_LOCK (image);
  id = image->loader_id;
  GST_OBJECT_UNLOCK (image);

  return (id != loader->id);
}

/* Function used by the image loader system to finish the loading in the
 * Pigment main loop */
static gboolean
pixbuf_loaded_from_file (gpointer data)
{
  PixbufLoaderData *loader = (PixbufLoaderData*) data;
  GdkPixbuf *pixbuf = loader->pixbuf;
  PgmImage *image = loader->image;

  /* Check for cancellation before loading */
  if (is_loading_cancelled (loader))
    {
      GST_LOG_OBJECT (image, "\"%s\" loading has been cancelled",
                      loader->filename);
      g_free (loader->filename);
      goto cancel;
    }

  /* Clear the current storage */
  pgm_image_clear (image);

  /* Setup the new storage */
  GST_OBJECT_LOCK (image);
  image->storage_type = PGM_IMAGE_FILE;
  image->data.file.pixbuf = gdk_pixbuf_ref (pixbuf);
  image->data.file.filename = loader->filename;
  image->data.file.width = gdk_pixbuf_get_width (pixbuf);
  image->data.file.height = gdk_pixbuf_get_height (pixbuf);
  image->flags |= PGM_IMAGE_STORED_FROM_FILE_LOADED;
  image->par_n = image->data.file.width;
  image->par_d = image->data.file.height;
  update_slaves_ratio (image);
  GST_OBJECT_UNLOCK (image);

  /* Emit signals */
  if (loader->emit_signal)
    g_signal_emit (G_OBJECT (image), pgm_image_signals[FILE_LOADED], 0);
  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_FILE);

  /* Clean up */
 cancel:
  g_slice_free (PixbufLoaderData, loader);
  g_object_unref (pixbuf);
  gst_object_unref (image);

  /* Remove the idle source from the Pigment main loop */
  return FALSE;
}

/* Function used by the thread pool to load images */
static void
pixbuf_loader (gpointer data,
               gpointer user_data)
{
  PixbufLoaderData *loader = (PixbufLoaderData*) data;
  PgmImage *image = loader->image;

  /* Check for cancellation before loading */
  if (is_loading_cancelled (loader))
    {
      GST_LOG_OBJECT (image, "\"%s\" loading has been cancelled",
                      loader->filename);
      goto cancel;
    }

  /* Load the image with requested scaling keeping the apect-ratio */
  loader->pixbuf = gdk_pixbuf_new_from_file_at_size (loader->filename,
                                                     loader->max_size,
                                                     loader->max_size, NULL);

  /* Check validity */
  if (loader->pixbuf == NULL)
    {
      GST_WARNING_OBJECT (image, "can't load image from \"%s\"",
                          loader->filename);
      goto cancel;
    }
  /* Check for cancellation after loading */
  else if (is_loading_cancelled (loader))
    {
      GST_LOG_OBJECT (image, "\"%s\" loading has been cancelled",
                      loader->filename);
      g_object_unref (loader->pixbuf);
      goto cancel;
    }
  /* Finish the loading in the Pigment main loop */
  else
    {
      g_idle_add (pixbuf_loaded_from_file, loader);
      return;
    }

 cancel:
  g_free (loader->filename);
  g_slice_free (PixbufLoaderData, loader);
  gst_object_unref (image);
}

/* Load an image from a file.
 * FIXME: The static counter is not thread-safe, locking the counter
 *        assignment and incrementation would be safer here. */
static void
set_from_file (PgmImage *image,
               const gchar *filename,
               guint max_size,
               gboolean emit_signal)
{
  PixbufLoaderData *loader;
  static guint counter = 1;

  /* Create and fill loader data */
  loader = g_slice_new (PixbufLoaderData);
  loader->image = gst_object_ref (image);
  loader->filename = g_strdup (filename);
  loader->max_size = (max_size < 1) ? -1 : max_size;
  loader->emit_signal = emit_signal;
  GST_OBJECT_LOCK (image);
  image->loader_id = counter;
  GST_OBJECT_UNLOCK (image);
  loader->id = counter;

  /* Increment counter for the next loader, avoid zero since it's used by
   * images to cancel loading tasks */
  if (G_UNLIKELY (++counter == 0))
    counter++;

  /* Push the loading in the thread pool */
  g_thread_pool_push (PGM_IMAGE_GET_CLASS (image)->pixbuf_loader_pool,
                      loader, NULL);
}

/* Do the actual clearing of image. Called with Image LOCK. */
static PgmError
do_clear (PgmImage *image)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  switch (image->storage_type)
    {
    case PGM_IMAGE_FILE:
      GST_DEBUG_OBJECT (image, "Clearing file image");
      if (image->flags & PGM_IMAGE_STORED_FROM_FILE_LOADED)
        {
          gdk_pixbuf_unref ((GdkPixbuf*) (image->data.file.pixbuf));
          image->data.file.pixbuf = NULL;
          g_free (image->data.file.filename);
          image->data.file.filename = NULL;
          image->flags &= ~PGM_IMAGE_STORED_FROM_FILE_LOADED;
        }
      break;

    case PGM_IMAGE_BUFFER:
      GST_DEBUG_OBJECT (image, "Clearing buffer image");
      g_free (image->data.buffer.buffer);
      image->data.buffer.buffer = NULL;
      break;

    case PGM_IMAGE_GST_BUFFER:
      GST_DEBUG_OBJECT (image, "Clearing GstBuffer image");
      gst_buffer_unref (image->data.gst_buffer.gst_buffer);
      image->data.gst_buffer.gst_buffer = NULL;
      break;

    case PGM_IMAGE_PIXBUF:
      GST_DEBUG_OBJECT (image, "Clearing GdkPixbuf image");
      gdk_pixbuf_unref ((GdkPixbuf*) (image->data.pixbuf.pixbuf));
      image->data.pixbuf.pixbuf = NULL;
      break;

    case PGM_IMAGE_SYSTEM_BUFFER:
      GST_DEBUG_OBJECT (image, "Clearing system buffer image");
      image->data.system_buffer.system_buffer = NULL;
      break;

    case PGM_IMAGE_IMAGE:
      GST_DEBUG_OBJECT (image, "Clearing slaved image");
      if (image->master)
        {
          PgmImage *master;

          /* Remove ourself from the master's slave list. */
          GST_OBJECT_LOCK (image->master);
          image->master->slaves = g_list_remove (image->master->slaves, image);
          master = image->master;
          GST_OBJECT_UNLOCK (image->master);
          image->master = NULL;
          g_signal_emit (G_OBJECT (master), pgm_image_signals[UN_CLONED], 0,
                         g_list_length (master->slaves));
        }
      break;

    default:
      break;
    }

  image->storage_type = PGM_IMAGE_EMPTY;
  image->par_n = 0;
  image->par_d = 1;
  update_slaves_ratio (image);

  return PGM_ERROR_OK;
}

/* Gets the width and height in pixel of the original image buffer. Called
 * with LOCK. */
static void
image_get_buffer_pixel_size (PgmImage *image,
                             guint *width,
                             guint *height)
{
  switch (image->storage_type)
    {
    case PGM_IMAGE_FILE:
      *width = image->data.file.width;
      *height = image->data.file.height;
      break;

    case PGM_IMAGE_BUFFER:
      *width = image->data.buffer.width;
      *height = image->data.buffer.height;
      break;

    case PGM_IMAGE_GST_BUFFER:
      *width = image->data.gst_buffer.width;
      *height = image->data.gst_buffer.height;
      break;

    case PGM_IMAGE_PIXBUF:
      *width = gdk_pixbuf_get_width (image->data.pixbuf.pixbuf);
      *height = gdk_pixbuf_get_height (image->data.pixbuf.pixbuf);
      break;

    case PGM_IMAGE_IMAGE:
      image_get_buffer_pixel_size (image->master, width, height);
      break;

    case PGM_IMAGE_SYSTEM_BUFFER:
      *width = image->data.system_buffer.width;
      *height = image->data.system_buffer.height;
      break;

    default:
      *width = *height = 0;
    }
}

/* Gets the position of the upper left corner of the image in drawable
 * coordinates (in x_offset and y_offset), and the resolution of the image, in
 * pixel per canvas unit. Called with LOCK. */
static void
image_get_offset_and_resolution (PgmImage *image,
                                 gfloat *x_offset,
                                 gfloat *y_offset,
                                 gfloat *x_resolution,
                                 gfloat *y_resolution)
{
  PgmDrawable *drawable = PGM_DRAWABLE (image);
  guint image_width, image_height;
  gboolean image_wider, fill_height;
  gfloat image_aspect_ratio;

  image_get_buffer_pixel_size (image, &image_width, &image_height);

  if (image->layout == PGM_IMAGE_FILLED)
    {
      *x_offset = 0;
      *y_offset = 0;
      *x_resolution = image_width / drawable->width;
      *y_resolution = image_height / drawable->height;
      return;
    }

  image_aspect_ratio = ((gfloat) image->par_n) / image->par_d;
  image_wider = image_aspect_ratio > drawable->width / drawable->height;
  fill_height = (image->layout == PGM_IMAGE_SCALED && !image_wider)
                || (image->layout == PGM_IMAGE_ZOOMED && image_wider);

  /* Resolution */
  if (fill_height)
    {
      *x_resolution = image_width / (image_aspect_ratio * drawable->height);
      *y_resolution = image_height / drawable->height;
    }
  else /* fill width */
    {
      *x_resolution = image_width / drawable->width;
      *y_resolution = image_height * image_aspect_ratio / drawable->width;
    }

  /* x_offset */
  if (!fill_height || image->align & PGM_IMAGE_LEFT)
    *x_offset = 0;
  else if (image->align & PGM_IMAGE_RIGHT)
    *x_offset = drawable->width - image_width / *x_resolution;
  else /* PGM_IMAGE_CENTER */
    *x_offset = 0.5f * (drawable->width - image_width / *x_resolution);

  /* y_offset */
  if (fill_height || image->align & PGM_IMAGE_TOP)
    *y_offset = 0;
  else if (image->align & PGM_IMAGE_BOTTOM)
    *y_offset = drawable->height - image_height / *y_resolution;
  else /* PGM_IMAGE_CENTER */
    *y_offset = 0.5f * (drawable->height - image_height / *y_resolution);
}

/* PgmImage virtual functions */

static PgmError
pgm_image_show (PgmDrawable *drawable)
{
  PgmImage *image = PGM_IMAGE (drawable);
  gboolean emit = FALSE;

  /* Flag the object as visible and update the GstBuffer if it's the current
   * storage */

  GST_OBJECT_LOCK (image);
  image->flags |= PGM_IMAGE_VISIBLE;
  if (image->storage_type == PGM_IMAGE_GST_BUFFER)
    emit = TRUE;
  GST_OBJECT_UNLOCK (image);

  if (emit)
    _pgm_drawable_emit_changed (drawable, PGM_IMAGE_DATA_GST_BUFFER);

  return PGM_ERROR_OK;
}

static PgmError
pgm_image_hide (PgmDrawable *drawable)
{
  PgmImage *image = PGM_IMAGE (drawable);

  /* Flag the object as not visible, it allows not to emit the changed signal
   * when a new GstBuffer is set and the drawable is not visible */

  GST_OBJECT_LOCK (image);
  image->flags &= ~PGM_IMAGE_VISIBLE;
  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/* GObject stuff */

/* Forward declarations */
static void pgm_image_base_class_init (gpointer g_class);
static void pgm_image_base_class_finalize (gpointer g_class);
static void pgm_image_class_init (PgmImageClass *klass);
static void pgm_image_init (PgmImage *image);

GType
pgm_image_get_type (void)
{
  static GType image_type = 0;

  if (G_UNLIKELY (image_type == 0))
    {
      static const GTypeInfo image_info = {
        sizeof (PgmImageClass),
        (GBaseInitFunc) pgm_image_base_class_init,
        (GBaseFinalizeFunc) pgm_image_base_class_finalize,
        (GClassInitFunc) pgm_image_class_init,
        NULL,
        NULL,
        sizeof (PgmImage),
        0,
        (GInstanceInitFunc) pgm_image_init,
        NULL
      };

      image_type = g_type_register_static (PGM_TYPE_DRAWABLE, "PgmImage",
                                           &image_info, 0);
    }

  GST_DEBUG_CATEGORY_INIT (pgm_image_debug, "pgm_image", 0,
                           "Pigment Image object");

  return image_type;
}

static void
pgm_image_dispose (GObject *object)
{
  PgmImage *image = PGM_IMAGE (object);

  /* Image is a master */
  if (image->slaves)
    {
      GList *copy = NULL, *walk = NULL;

      /* We need a copy of the slaves list since it gets modified in
       * pgm_image_clear */
      copy = g_list_copy (image->slaves);

      for (walk = copy; walk; walk = walk->next)
        pgm_image_clear (walk->data);

      g_list_free (copy);

      if (image->slaves)
        GST_DEBUG_OBJECT (image, "Slave list not completely cleared");
    }

  GST_OBJECT_LOCK (image);
  do_clear (image);
  GST_OBJECT_UNLOCK (image);

  pgm_mat4x4_free (image->mapping_matrix);
  image->mapping_matrix = NULL;

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_image_base_class_init (gpointer g_class)
{
  PgmImageClass *klass = PGM_IMAGE_CLASS (g_class);
  const gchar *env_var = NULL;
  gint max_threads = 1;

  /* Get the number of threads to use in the pool from the environment. The
   * number is 1 by default. This is there to ease us to make a decision on
   * the default number to use depending on the CPU technology (multi-core,
   * hyper-threading, ...). */
  env_var = g_getenv ("PGM_IMAGE_THREAD_POOL_SIZE");
  if (env_var)
    max_threads = CLAMP (atoi (env_var), 1, 16);
  klass->pixbuf_loader_pool =
    g_thread_pool_new (pixbuf_loader, NULL, max_threads, TRUE, NULL);

  GST_DEBUG ("Image thread pool size: %d", max_threads);
}

static void
pgm_image_base_class_finalize (gpointer g_class)
{
  PgmImageClass *klass = PGM_IMAGE_CLASS (g_class);

  g_thread_pool_free (klass->pixbuf_loader_pool, TRUE, TRUE);
  klass->pixbuf_loader_pool = NULL;
}

static void
pgm_image_class_init (PgmImageClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;
  PgmDrawableClass *drawable_class = (PgmDrawableClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  /**
   * PgmImage::file-loaded:
   * @image: the #PgmImage
   *
   * Will be emitted after @image has finished to load its data from the file
   * path given in the pgm_image_set_from_file() method.
   */
  pgm_image_signals[FILE_LOADED] =
    g_signal_new ("file-loaded", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmImageClass, file_loaded),
                  NULL, NULL, pgm_marshal_VOID__VOID, G_TYPE_NONE, 0);
  /**
   * PgmImage::cloned:
   * @image: the #PgmImage
   * @clone: the cloned #PgmImage
   * @n_clones: the number of cloned images
   *
   * Will be emitted after @image has been cloned by another #PgmImage through
   * the pgm_image_new_from_image() or pgm_image_set_from_image() functions.
   */
  pgm_image_signals[CLONED] =
    g_signal_new ("cloned", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmImageClass, cloned),
                  NULL, NULL, pgm_marshal_VOID__OBJECT_UINT, G_TYPE_NONE, 2,
                  PGM_TYPE_IMAGE, G_TYPE_UINT);
  /**
   * PgmImage::un-cloned:
   * @image: the #PgmImage
   * @n_clones: the number of cloned images
   *
   * Will be emitted after @image has got a cloned #PgmImage removed.
   */
  pgm_image_signals[UN_CLONED] =
    g_signal_new ("un-cloned", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmImageClass, un_cloned),
                  NULL, NULL, pgm_marshal_VOID__UINT, G_TYPE_NONE, 1,
                  G_TYPE_UINT);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_image_dispose);
  drawable_class->show = GST_DEBUG_FUNCPTR (pgm_image_show);
  drawable_class->hide = GST_DEBUG_FUNCPTR (pgm_image_hide);
}

static void
pgm_image_init (PgmImage *image)
{
  /* Public data */
  /* image->master = NULL; */
  /* image->slaves = NULL; */
  image->mapping_matrix = pgm_mat4x4_new_identity ();
  /* image->storage_type = PGM_IMAGE_EMPTY; */
  image->layout = PGM_IMAGE_SCALED;
  image->align = PGM_IMAGE_CENTER;
  image->interp = PGM_IMAGE_BILINEAR;
  /* image->wrap_s = PGM_IMAGE_CLAMP; */
  /* image->wrap_t = PGM_IMAGE_CLAMP; */
  /* image->par_n = 0; */
  image->par_d = 1;
  /* image->border_width = 0.0f; */
  image->border_inner_r = 255;
  image->border_inner_g = 255;
  image->border_inner_b = 255;
  image->border_inner_a = 255;
  image->border_outer_r = 255;
  image->border_outer_g = 255;
  image->border_outer_b = 255;
  image->border_outer_a = 255;

  /* Private data */
  /* image->loader_id = 0; */
  /* image->flags = 0; */
}

/* public methods */

/**
 * pgm_image_new:
 *
 * Creates a new #PgmImage instance.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new (void)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_file:
 * @filename: the filename.
 * @max_size: the maximum size of the image in pixels before loading it in the
 * #PgmImage or 0 to not constrain the size.
 *
 * Creates a new #PgmImage instance loading an image from the given @filename.
 * It optionally pre-scales the image so that it has a maximum width and height
 * of @max_size.
 *
 * The loading is done asynchronously, it means that this function returns right
 * after the image object is created, the loading and decoding of the image
 * being done in another thread. The <link linkend="PgmImage-file-loaded">
 * 'file-loaded'</link> signal is emitted from the Pigment main loop (the
 * thread running pgm_main() or the thread running a #GMainLoop using the
 * default #GMainContext) once the loading finishes successfully.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_file (const gchar *filename,
                         guint max_size)
{
  PgmImage *image;
  PgmError ret = PGM_ERROR_OK;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image from file");

  ret = pgm_image_set_from_file (image, filename, max_size);
  if (PGM_ERROR_OK != ret)
    {
      gst_object_unref (image);
      return NULL;
    }

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_pixbuf:
 * @pixbuf: the #GdkPixbuf.
 *
 * Creates a new #PgmImage instance using @pixbuf as stored image.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_pixbuf (GdkPixbuf *pixbuf)
{
  PgmImage *image;
  PgmError ret = PGM_ERROR_OK;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image from pixbuf");

  ret = pgm_image_set_from_pixbuf (image, pixbuf);
  if (PGM_ERROR_OK != ret)
    {
      gst_object_unref (image);
      return NULL;
    }

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_buffer:
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @stride: the image rowstride in bytes (number of bytes per line).
 * @size: the buffer size in bytes.
 * @data: a pointer to the data buffer.
 *
 * Creates a new #PgmImage instance with the image from the given buffer.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_buffer (PgmImagePixelFormat format,
                           guint width,
                           guint height,
                           guint stride,
                           guint size,
                           gconstpointer data)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  pgm_image_set_from_buffer (image, format, width, height, stride, size, data);

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_new_from_image:
 * @src_image: a #PgmImage which will be used as the master image.
 *
 * Creates a new #PgmImage instance with an image slaved from the
 * image of @src_image.
 *
 * MT safe.
 *
 * Returns: a new #PgmImage instance.
 */
PgmDrawable *
pgm_image_new_from_image (PgmImage *src_image)
{
  PgmImage *image;

  image = g_object_new (PGM_TYPE_IMAGE, NULL);
  GST_DEBUG_OBJECT (image, "created new image");

  pgm_image_set_from_image (image, src_image);

  return PGM_DRAWABLE (image);
}

/**
 * pgm_image_set_from_file:
 * @image: a #PgmImage object.
 * @filename: a filename.
 * @max_size: the maximum size of the image in pixels before loading it in the
 * #PgmImage or 0 to not constrain the size.
 *
 * Loads an image from the file @filename. It optionally pre-scales the image so
 * it has a maximum width and height of @max_size.
 *
 * The loading is done asynchronously, it means that this function returns right
 * after the image object is created, the loading and decoding of the image
 * being done in another thread. The <link linkend="PgmImage-pixbuf-loaded">
 * 'pixbuf-loaded'</link> signal is emitted from the Pigment main loop (the
 * thread running pgm_main() or the thread running a #GMainLoop using the
 * default #GMainContext) once the loading finishes successfully.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_file (PgmImage *image,
                         const gchar *filename,
                         guint max_size)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  if (filename == NULL)
    return PGM_ERROR_X;

  set_from_file (image, filename, max_size, TRUE);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_pixbuf:
 * @image: a #PgmImage object.
 * @pixbuf: the #GdkPixbuf.

 * Loads an image in @image from a pixbuf. @pixbuf will have its reference
 * count increased by 1 and will not get freed until the drawable gets cleaned
 * up or that a new buffer is loaded.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_pixbuf (PgmImage *image,
                           GdkPixbuf *pixbuf)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (pixbuf != NULL, PGM_ERROR_X);

  pgm_image_clear (image);

  GST_OBJECT_LOCK (image);
  image->storage_type = PGM_IMAGE_PIXBUF;
  image->data.pixbuf.pixbuf = gdk_pixbuf_ref (pixbuf);
  image->par_n = gdk_pixbuf_get_width (pixbuf);
  image->par_d = gdk_pixbuf_get_height (pixbuf);
  update_slaves_ratio (image);
  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_PIXBUF);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_buffer:
 * @image: a #PgmImage object.
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @stride: the rowstride of the image in bytes (number of bytes per line).
 * @size: the buffer size in bytes.
 * @data: a pointer to the data buffer.
 *
 * Loads an image in @image from an existing buffer using pixel format @format.
 * If you don't know the rowstride of the image you can set stride to 0. @data
 * is copied internally you can free it right after the function call returns.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_buffer (PgmImage *image,
                           PgmImagePixelFormat format,
                           guint width,
                           guint height,
                           guint stride,
                           guint size,
                           gconstpointer data)
{
  gpointer _data = NULL;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (data != NULL, PGM_ERROR_X);

  /* Let's set the storage data */
  GST_OBJECT_LOCK (image);

  cancel_async_loading_task (image);

  /* The buffer sent is not the first of this size, it's not needed to call
   * a clear, just free the buffer. */
  if (G_LIKELY (image->storage_type == PGM_IMAGE_BUFFER
                && image->data.buffer.width == width
                && image->data.buffer.height == height
                && image->data.buffer.format == format
                && image->data.buffer.stride == stride))
    {
      if (G_LIKELY (image->data.buffer.buffer))
        g_free (image->data.buffer.buffer);
    }

  /* It's the first */
  else
    {
      /* Let's clear if needed */
      if (image->storage_type != PGM_IMAGE_EMPTY)
        {
          do_clear (image);
          GST_OBJECT_UNLOCK (image);
          _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_EMPTY);
          GST_OBJECT_LOCK (image);
        }

      /* Store buffer informations */
      image->storage_type = PGM_IMAGE_BUFFER;
      image->data.buffer.format = format;
      image->data.buffer.width = width;
      image->data.buffer.height = height;
      image->data.buffer.stride = stride;
      image->data.buffer.size = size;

      /* Store ratio */
      image->par_n = width;
      image->par_d = height;
      update_slaves_ratio (image);
    }

  /* Try to copy the given buffer */
  _data = g_memdup (data, size);

  image->data.buffer.buffer = (guint8 *) _data;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_BUFFER);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_gst_buffer:
 * @image: a #PgmImage object.
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @stride: the rowstride of the image in bytes (number of bytes per line).
 * @buffer: A #GstBuffer reference containing the video frame.
 *
 * Loads an image in @image from an existing #GstBuffer using the pixel format
 * @format. If you don't know the rowstride of the image you can set stride
 * to 0. @buffer will have its reference count increased by 1 and will not get
 * freed until the drawable gets cleaned up or that a new buffer is loaded.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_gst_buffer (PgmImage *image,
                               PgmImagePixelFormat format,
                               guint width,
                               guint height,
                               guint stride,
                               GstBuffer *buffer)
{
  gboolean visible;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (GST_IS_BUFFER (buffer), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  visible = image->flags & PGM_IMAGE_VISIBLE;

  cancel_async_loading_task (image);

  /* The GstBuffer sent is not the first, it's not needed to call a clear,
   * just unref the GstBuffer. I mean that the implementation should not be
   * signaled to clear its allocated size/format for the stream of buffers
   * since it's really heavy to clear for each new one. */
  if (G_LIKELY (image->storage_type == PGM_IMAGE_GST_BUFFER
                && image->data.gst_buffer.width == width
                && image->data.gst_buffer.height == height
                && image->data.gst_buffer.format == format
                && image->data.gst_buffer.stride == stride))
    {
      if (G_LIKELY (image->data.gst_buffer.gst_buffer))
        gst_buffer_unref (image->data.gst_buffer.gst_buffer);
    }

  /* It's the first */
  else
    {
      /* Let's clear if needed */
      if (image->storage_type != PGM_IMAGE_EMPTY)
        {
          do_clear (image);
          GST_OBJECT_UNLOCK (image);
          _pgm_drawable_emit_changed (PGM_DRAWABLE (image),
                                      PGM_IMAGE_DATA_EMPTY);
          GST_OBJECT_LOCK (image);
        }

      /* Store buffer informations */
      image->storage_type = PGM_IMAGE_GST_BUFFER;
      image->data.gst_buffer.format = format;
      image->data.gst_buffer.width = width;
      image->data.gst_buffer.height = height;
      image->data.gst_buffer.stride = stride;

      /* Store ratio */
      image->par_n = width;
      image->par_d = height;
      update_slaves_ratio (image);
    }

  /* Take a ref on the GstBuffer */
  image->data.gst_buffer.gst_buffer = gst_buffer_ref (buffer);

  GST_OBJECT_UNLOCK (image);

  if (visible)
    _pgm_drawable_emit_changed (PGM_DRAWABLE (image),
                                PGM_IMAGE_DATA_GST_BUFFER);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_system_buffer:
 * @image: a #PgmImage object.
 * @format: the pixel format of the buffer.
 * @width: the image width in pixels.
 * @height: the image height in pixels.
 * @system_buffer: a pointer to the system buffer.
 *
 * Loads an image in @image from an existing system buffer.
 *
 * A system buffer depends on the platform, for instance on UNIX with an X11
 * server running, @system_buffer can be set to an X11 Pixmap.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_system_buffer (PgmImage *image,
                                  PgmImagePixelFormat format,
                                  guint width,
                                  guint height,
                                  gconstpointer system_buffer)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (system_buffer != NULL, PGM_ERROR_X);

  /* Let's set the storage data */
  GST_OBJECT_LOCK (image);

  cancel_async_loading_task (image);

  /* Let's clear if needed */
  if (image->storage_type != PGM_IMAGE_EMPTY)
    {
      do_clear (image);
      GST_OBJECT_UNLOCK (image);
      _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_EMPTY);
      GST_OBJECT_LOCK (image);
    }

  /* Store buffer informations */
  image->storage_type = PGM_IMAGE_SYSTEM_BUFFER;
  image->data.system_buffer.format = format;
  image->data.system_buffer.width = width;
  image->data.system_buffer.height = height;
  image->data.system_buffer.system_buffer = system_buffer;

  /* Store ratio */
  image->par_n = width;
  image->par_d = height;
  update_slaves_ratio (image);

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image),
                              PGM_IMAGE_DATA_SYSTEM_BUFFER);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_from_image:
 * @image: a #PgmImage object.
 * @src_image: the source #PgmImage object to use as a master.
 *
 * Slaves @image to @src_image. Every change to @src_image is reflected on
 * @image until you remove @image from the canvas or you call pgm_image_clear()
 * on @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_from_image (PgmImage *image,
                          PgmImage *src_image)
{
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (PGM_IS_IMAGE (src_image), PGM_ERROR_X);

  GST_DEBUG_OBJECT (image, "using image from %s", GST_OBJECT_NAME (src_image));

  GST_OBJECT_LOCK (image);

  /* Do not do anything if the image is already a slave of the source image */
  if (G_UNLIKELY (src_image == image->master))
    {
      GST_OBJECT_UNLOCK (image);
      ret = PGM_ERROR_OK;
      goto beach;
    }

  /* Trying to deadlock us? :) */
  if (G_UNLIKELY (image == src_image || image == src_image->master))
    {
      GST_WARNING_OBJECT (image, "trying to do a master/slave loop!");
      GST_OBJECT_UNLOCK (image);
      ret = PGM_ERROR_X;
      goto beach;
    }

  cancel_async_loading_task (image);
  do_clear (image);
  GST_OBJECT_UNLOCK (image);
  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_EMPTY);
  GST_OBJECT_LOCK (image);

  image->storage_type = PGM_IMAGE_IMAGE;

  GST_OBJECT_LOCK (src_image);

  /* The master requested is already a slave */
  if (G_UNLIKELY (src_image->master))
    {
      GST_DEBUG_OBJECT (image, "%s is already a slave to %s, using its master",
                        GST_OBJECT_NAME (src_image),
                        GST_OBJECT_NAME (src_image->master));

      GST_OBJECT_LOCK (src_image->master);

      src_image->master->slaves = g_list_append (src_image->master->slaves,
                                                 image);
      image->master = src_image->master;
      GST_OBJECT_UNLOCK (src_image->master);
    }

  /* The master requested is not a slave */
  else
    {
      /* Add ourself to the slave list with an increased reference */
      src_image->slaves = g_list_append (src_image->slaves, image);
      image->master = src_image;
    }

  image->par_n = src_image->par_n;
  image->par_d = src_image->par_d;

  GST_OBJECT_UNLOCK (src_image);
  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_IMAGE);

  g_signal_emit (G_OBJECT (image->master), pgm_image_signals[CLONED], 0,
                 image, g_list_length (image->master->slaves));

beach:
  return ret;
}

/**
 * pgm_image_to_pixbuf:
 * @image: a #PgmImage object.
 * @pixbuf: a double pointer to #GdkPixbuf. Unref after usage.
 *
 * Retrieves a #GdkPixbuf of the image currently stored in @image.
 *
 * Only the images stored as pixbuf can be retrieved at the moment, the
 * conversion for the other storage types is not implemented yet.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_to_pixbuf (PgmImage *image,
                     GdkPixbuf **pixbuf)
{
  PgmError ret = PGM_ERROR_X;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (image != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  switch (image->storage_type)
    {
    case PGM_IMAGE_PIXBUF:
      *pixbuf = gdk_pixbuf_ref (image->data.pixbuf.pixbuf);
      ret = PGM_ERROR_OK;
      break;

      /* FIXME: Not implemented yet */
    case PGM_IMAGE_FILE:
    case PGM_IMAGE_IMAGE:
    case PGM_IMAGE_BUFFER:
    case PGM_IMAGE_GST_BUFFER:
    case PGM_IMAGE_SYSTEM_BUFFER:
      GST_WARNING_OBJECT (image, "Conversion to pixbuf not implemented for raw "
                          "buffers, GStreamer buffers, system buffers and "
                          "cloned images");
      break;

    case PGM_IMAGE_EMPTY:
      GST_WARNING_OBJECT (image, "Cannot convert empty image to pixbuf");
      break;

    default:
      break;
    }

  GST_OBJECT_UNLOCK (image);

  return ret;
}

/**
 * pgm_image_clear:
 * @image: a #PgmImage object.
 *
 * Removes any image from @image. If @image had some image data loaded, it's
 * cleared, if there was a #GstBuffer used, it's unreffed and if the @image was
 * a slave to another it is not anymore. If @image has slave images they all
 * get cleared but they still are slaves to @image. So if you load a new image
 * to @image, all the slaves will load it too.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_clear (PgmImage *image)
{
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);
  cancel_async_loading_task (image);
  ret = do_clear (image);
  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_DATA_EMPTY);

  return ret;
}

/**
 * pgm_image_get_storage_type:
 * @image: a #PgmImage object.
 * @storage: a #PgmImageStorageType where the storage type is going to be
 * stored.
 *
 * Retrieves the type of representation being used by @image to store image
 * data. If @image has no image data, the return value will be
 * #PGM_IMAGE_EMPTY.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_storage_type (PgmImage *image,
                            PgmImageStorageType *storage)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (storage != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *storage = image->storage_type;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_system_buffer_lock:
 * @image: a #PgmImage object.
 *
 * Lock the system buffer set as the image content, ensuring the image is not
 * going to be drawn while the system buffer content changes. This function
 * guaranties tear free system buffer updates.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_system_buffer_lock (PgmImage *image)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  /* FIXME: lock */

  return PGM_ERROR_OK;
}

/**
 * pgm_image_system_buffer_unlock:
 * @image: a #PgmImage object.
 *
 * Unlock the system buffer set as the image content.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_system_buffer_unlock (PgmImage *image)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  /* FIXME: unlock */

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image),
                              PGM_IMAGE_SYSTEM_BUFFER_CONTENT);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_mapping_matrix:
 * @image: a #PgmImage object.
 * @mapping_matrix: a #PgmMat4x4 object.
 *
 * Defines the transformation to apply to the stored image when it is rendered.
 * You can make the stored image slide over the drawable, rotate around it,
 * stretch and shrink, or any combination of the three.
 *
 * Each point in an image can be defined by an (x, y) vector, which we call
 * the source position, representing the horizontal (x) and vertical (y)
 * positions (with values between 0 for left/top and 1 for right/bottom).
 * When the image is drawn on a surface, each point (x, y) is drawn on the
 * (x', y') coordinate of the surface. We call (x', y') the destination
 * position. The default mapping matrix is the identity, you have
 * (x', y') == (x, y). Once you have called the function, the destination
 * position is calculated by multiplying @mapping_matrix with the source
 * position vector. To reset the mapping matrix, just set the identity.
 *
 * @mapping_matrix is a 4x4 matrix since the source and destination positions
 * can be represented as 4 coordinate vectors (x, y, z, w) and (x', y', z', w').
 * Unless you know what you are doing, you should not worry about z and w, and
 * arrange for your matrix to leave them unchanged.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_mapping_matrix (PgmImage *image,
                              PgmMat4x4 *mapping_matrix)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (mapping_matrix != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);
  pgm_mat4x4_set_from_mat4x4 (image->mapping_matrix, mapping_matrix);
  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_MAPPING_MATRIX);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_mapping_matrix:
 * @image: a #PgmImage object.
 * @mapping_matrix: a pointer to a #PgmMat4x4 pointer where the mapping matrix
 * is going to be stored. pgm_mat4x4_free() after use.
 *
 * Retrieves in @matrix the current mapping matrix applied to @image. 
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_mapping_matrix (PgmImage *image,
                              PgmMat4x4 **mapping_matrix)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (mapping_matrix != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);
  *mapping_matrix = pgm_mat4x4_copy (image->mapping_matrix);
  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_alignment:
 * @image: a #PgmImage object.
 * @align: a #PgmImageAlignment combination of flags.
 *
 * Defines the way @image aligns the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_alignment (PgmImage *image,
                         PgmImageAlignment align)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->align = align;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_ALIGNMENT);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_alignment:
 * @image: a #PgmImage object.
 * @align: a pointer to a #PgmImageAlignment where alignment flags are going
 * to be stored.
 *
 * Retrieves in @align the way @image aligns the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_alignment (PgmImage *image,
                         PgmImageAlignment *align)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (align != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *align = image->align;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_layout:
 * @image: a #PgmImage object.
 * @layout: a #PgmImageLayoutType layout type.
 *
 * Defines the way @image layouts its stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_layout (PgmImage *image,
                      PgmImageLayoutType layout)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->layout = layout;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_LAYOUT);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_layout:
 * @image: a #PgmImage object.
 * @layout: a pointer to a #PgmImageLayoutType where the layout type is going
 * to be stored.
 *
 * Retrieves in @layout the way @image layouts its its stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */

PgmError
pgm_image_get_layout (PgmImage *image,
                      PgmImageLayoutType *layout)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (layout != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *layout = image->layout;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_interp:
 * @image: a #PgmImage object.
 * @interp: the interpolation type.
 *
 * Defines that @image will be rendered using @interp as its interpolation
 * type.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_interp (PgmImage *image,
                      PgmImageInterpType interp)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->interp = interp;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_INTERP);

  return  PGM_ERROR_OK;
}

/**
 * pgm_image_get_interp:
 * @image: a #PgmImage object.
 * @interp: a pointer to a #PgmImageInterpType where the interpolation type
 * is going to be stored.
 *
 * Retrieves in @interp the current interpolation type of @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_interp (PgmImage *image,
                      PgmImageInterpType *interp)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (interp != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *interp = image->interp;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_wrapping:
 * @image: a #PgmImage object.
 * @wrap_s: the wrapping on the s coordinate. #PGM_IMAGE_REPEAT by default.
 * @wrap_t: the wrapping on the t coordinate. #PGM_IMAGE_REPEAT by default.
 *
 * Sets the horizontal and vertical wrapping modes used for the mapping of
 * @image. When the mapping matrix is modified, pixels outside the range of the
 * image content are accessed, the wrapping mode defines the behaviour of the
 * mapping in that particular case.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_wrapping (PgmImage *image,
                        PgmImageWrapping wrap_s,
                        PgmImageWrapping wrap_t)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->wrap_s = wrap_s;
  image->wrap_t = wrap_t;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_WRAPPING);

  return  PGM_ERROR_OK;
}

/**
 * pgm_image_get_wrapping:
 * @image: a #PgmImage object.
 * @wrap_s: a pointer to a #PgmImageInterpType where the wrapping mode on the s
 * coordinate is going to be stored.
 * @wrap_t: a pointer to a #PgmImageInterpType where the wrapping mode on the t
 * coordinate is going to be stored.
 *
 * Retrieves in @wrap_s and @wrap_t the current wrapping modes of @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_wrapping (PgmImage *image,
                        PgmImageWrapping *wrap_s,
                        PgmImageWrapping *wrap_t)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (wrap_s != NULL, PGM_ERROR_X);
  g_return_val_if_fail (wrap_t != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *wrap_s = image->wrap_s;
  *wrap_t = image->wrap_t;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_aspect_ratio:
 * @image: a #PgmImage object.
 * @numerator: the numerator of the aspect ratio fraction.
 * @denominator: the denominator of the aspect ratio fraction.
 *
 * Customizes the aspect ratio of the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_aspect_ratio (PgmImage *image,
                            guint numerator,
                            guint denominator)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  if (numerator == image->par_n && denominator == image->par_d)
    {
      GST_OBJECT_UNLOCK (image);
      return PGM_ERROR_OK;
    }

  image->par_n = numerator;
  image->par_d = MAX (denominator, 1);

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_ASPECT_RATIO);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_aspect_ratio:
 * @image: a #PgmImage object.
 * @numerator: a pointer to a #guint where the numerator of the aspect ratio
 * fraction will be stored.
 * @denominator: a pointer to a #guint where the denominator of the aspect
 * ratio fraction will be stored.
 *
 * Retrieves the aspect ratio of the stored image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_aspect_ratio (PgmImage *image,
                            guint *numerator,
                            guint *denominator)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (numerator != NULL, PGM_ERROR_X);
  g_return_val_if_fail (denominator != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *numerator = image->par_n;
  *denominator = image->par_d;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_border_width:
 * @image: a #PgmImage object.
 * @width: the border with. 0.0 by default.
 *
 * Defines the border width drawn around @image.
 *
 * Note that the border is drawn around the image, inside the drawable. When
 * you change the size of the border, the image will be down-scaled.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_border_width (PgmImage *image,
                            gfloat width)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  /* Avoid further computations if the width doesn't change */
  if (width == image->border_width)
    {
      GST_OBJECT_UNLOCK (image);
      return PGM_ERROR_OK;
    }

  /* Clamp width to [0, MAX_FLOAT] */
  image->border_width = MAX (width, 0);

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image), PGM_IMAGE_BORDER_WIDTH);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_border_width:
 * @image: a #PgmImage object.
 * @width: a pointer to a #gfloat where the border width will be stored.
 *
 * Retrieves the width of the border drawn around #image inside the drawable.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_border_width (PgmImage *image,
                            gfloat *width)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *width = image->border_width;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_border_inner_color:
 * @image: a #PgmImage object.
 * @red: the border inner red color. 255 by default.
 * @green: the border inner green color. 255 by default.
 * @blue: the border inner blue color. 255 by default.
 * @alpha: the border inner alpha color. 255 by default.
 *
 * Defines the border inner color drawn around @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_border_inner_color (PgmImage *image,
                                  guchar red,
                                  guchar green,
                                  guchar blue,
                                  guchar alpha)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->border_inner_r = red;
  image->border_inner_g = green;
  image->border_inner_b = blue;
  image->border_inner_a = alpha;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image),
                              PGM_IMAGE_BORDER_INNER_COLOR);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_border_inner_color:
 * @image: a #PgmImage object.
 * @red: a pointer to a #guchar where the border inner red color will be stored.
 * @green: a pointer to a #guchar where the border inner green color will be
 * stored.
 * @blue: a pointer to a #guchar where the border inner blue color will be
 * stored.
 * @alpha: a pointer to a #guchar where the border inner alpha color will be
 * stored.
 *
 * Retrieves the inner color of the border drawn around #image inside the
 * drawable.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_border_inner_color (PgmImage *image,
                                  guchar *red,
                                  guchar *green,
                                  guchar *blue,
                                  guchar *alpha)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (red != NULL, PGM_ERROR_X);
  g_return_val_if_fail (green != NULL, PGM_ERROR_X);
  g_return_val_if_fail (blue != NULL, PGM_ERROR_X);
  g_return_val_if_fail (alpha != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *red = image->border_inner_r;
  *green = image->border_inner_g;
  *blue = image->border_inner_b;
  *alpha = image->border_inner_a;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_set_border_outer_color:
 * @image: a #PgmImage object.
 * @red: the border outer red color. 255 by default.
 * @green: the border outer green color. 255 by default.
 * @blue: the border outer blue color. 255 by default.
 * @alpha: the border outer alpha color. 255 by default.
 *
 * Defines the border outer color drawn around @image.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_set_border_outer_color (PgmImage *image,
                                  guchar red,
                                  guchar green,
                                  guchar blue,
                                  guchar alpha)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  image->border_outer_r = red;
  image->border_outer_g = green;
  image->border_outer_b = blue;
  image->border_outer_a = alpha;

  GST_OBJECT_UNLOCK (image);

  _pgm_drawable_emit_changed (PGM_DRAWABLE (image),
                              PGM_IMAGE_BORDER_OUTER_COLOR);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_get_border_outer_color:
 * @image: a #PgmImage object.
 * @red: a pointer to a #guchar where the border outer red color will be stored.
 * @green: a pointer to a #guchar where the border outer green color will be
 * stored.
 * @blue: a pointer to a #guchar where the border outer blue color will be
 * stored.
 * @alpha: a pointer to a #guchar where the border outer alpha color will be
 * stored.
 *
 * Retrieves the outer color of the border drawn around #image inside the
 * drawable.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_get_border_outer_color (PgmImage *image,
                                  guchar *red,
                                  guchar *green,
                                  guchar *blue,
                                  guchar *alpha)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (red != NULL, PGM_ERROR_X);
  g_return_val_if_fail (green != NULL, PGM_ERROR_X);
  g_return_val_if_fail (blue != NULL, PGM_ERROR_X);
  g_return_val_if_fail (alpha != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  *red = image->border_outer_r;
  *green = image->border_outer_g;
  *blue = image->border_outer_b;
  *alpha = image->border_outer_a;

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_from_drawable:
 * @image: a #PgmImage instance
 * @x_image: where to return the x coordinate in the @image plane.
 * @y_image: where to return the y coordinate in the @image plane.
 * @x_drawable: the x coordinate in drawable space.
 * @y_drawable: the y coordinate in drawable space.
 *
 * Converts a position in drawable space (canvas units, origin at top left of
 * drawable) into @image space (image pixel units, origin at top left of image).
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_from_drawable (PgmImage *image,
                         gint *x_image,
                         gint *y_image,
                         gfloat x_drawable,
                         gfloat y_drawable)
{
  gfloat x_offset, y_offset;
  gfloat x_resolution, y_resolution;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (NULL != x_image, PGM_ERROR_X);
  g_return_val_if_fail (NULL != y_image, PGM_ERROR_X);
  g_return_val_if_fail (image->layout != PGM_IMAGE_CENTERED
                        || image->layout != PGM_IMAGE_TILED, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);
  image_get_offset_and_resolution (image, &x_offset, &y_offset,
                                   &x_resolution, &y_resolution);
  GST_OBJECT_UNLOCK (image);

  *x_image = (gint) ((x_drawable - x_offset) * x_resolution);
  *y_image = (gint) ((y_drawable - y_offset) * y_resolution);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_to_drawable:
 * @image: a #PgmImage instance
 * @x_drawable: where to return the x coordinate in the @drawable plane.
 * @y_drawable: where to return the y coordinate in the @drawable plane.
 * @x_image: the x coordinate in image space.
 * @y_image: the y coordinate in image space.
 *
 * Converts a position in @image space (image pixel units, origin at top left of
 * image) into drawable space (canvas units, origin at top left of drawable).
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_to_drawable (PgmImage *image,
                       gfloat *x_drawable,
                       gfloat *y_drawable,
                       gint x_image,
                       gint y_image)
{
  gfloat x_offset, y_offset;
  gfloat x_resolution, y_resolution;

  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);
  g_return_val_if_fail (NULL != x_drawable, PGM_ERROR_X);
  g_return_val_if_fail (NULL != y_drawable, PGM_ERROR_X);
  g_return_val_if_fail (image->layout != PGM_IMAGE_CENTERED
                        || image->layout != PGM_IMAGE_TILED, PGM_ERROR_X);

  GST_OBJECT_LOCK (image);
  image_get_offset_and_resolution (image, &x_offset, &y_offset,
                                   &x_resolution, &y_resolution);
  GST_OBJECT_UNLOCK (image);

  *x_drawable = x_offset + x_image / x_resolution;
  *y_drawable = y_offset + y_image / y_resolution;

  return PGM_ERROR_OK;
}

/* Protected functions */

/**
 * _pgm_image_stored_from_file_free:
 * @image: a #PgmImage instance
 *
 * Free the image buffer allocated by a pgm_image_set_from_file(). Can be used
 * by a plugin to clean up the central memory after the image buffer has been
 * sent to the video memory.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
_pgm_image_stored_from_file_free (PgmImage *image)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  /* Free the pixbuf if current storage has been loaded from a file */
  if (image->storage_type == PGM_IMAGE_FILE
      && image->flags & PGM_IMAGE_STORED_FROM_FILE_LOADED)
    {
      gdk_pixbuf_unref ((GdkPixbuf*) (image->data.file.pixbuf));
      image->data.file.pixbuf = NULL;
      image->flags &= ~PGM_IMAGE_STORED_FROM_FILE_LOADED;
    }

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}

/**
 * _pgm_image_stored_from_file_load:
 * @image: a #PgmImage instance
 *
 * Reload the image buffer set through pgm_image_set_from_file().
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
_pgm_image_stored_from_file_load (PgmImage *image)
{
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  GST_OBJECT_LOCK (image);

  if (image->storage_type != PGM_IMAGE_FILE)
    {
      GST_OBJECT_UNLOCK (image);
      return PGM_ERROR_X;
    }

  if (! (image->flags & PGM_IMAGE_STORED_FROM_FILE_LOADED))
    {
      GST_OBJECT_UNLOCK (image);
      set_from_file (image, image->data.file.filename,
                     MAX (image->data.file.width, image->data.file.height),
                     FALSE);
      return PGM_ERROR_OK;
    }

  GST_OBJECT_UNLOCK (image);

  return PGM_ERROR_OK;
}
