/***********************************************************************************************************************************
Test Build Config
***********************************************************************************************************************************/
#include "common/harnessStorage.h"

/***********************************************************************************************************************************
Test Run
***********************************************************************************************************************************/
static void
testRun(void)
{
    FUNCTION_HARNESS_VOID();

    // Create default storage object for testing
    Storage *storageTest = storagePosixNewP(TEST_PATH_STR, .write = true);

    // *****************************************************************************************************************************
    if (testBegin("bldCfgParse() and bldCfgRender()"))
    {
        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("render with label");

        TEST_RESULT_STR_Z(
            bldCfgRenderLabel(STRDEF("line"), true, STRDEF("label")),
            "line                                                                                                                  "
            "      // label",
            "render");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("render without label due to long line");

        TEST_RESULT_STR_Z(
            bldCfgRenderLabel(
                STRDEF(
                    "12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890"
                    "12345678901234567890"),
                true, STRDEF("label")),
            "1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678"
            "901234567890",
            "render");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("command parse errors");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            "command:\n"
            "  archive-get:\n"
            "    bogus: test\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "unknown command definition 'bogus'");

        #define TEST_COMMAND_VALID                                                                                                 \
            "command:\n"                                                                                                           \
            "  archive-get:\n"                                                                                                     \
            "    internal: true\n"                                                                                                 \
            "\n"

        #define TEST_OPTION_GROUP_VALID                                                                                            \
            "optionGroup:\n"                                                                                                       \
            "  repo: {}\n"                                                                                                         \
            "\n"

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("option parse errors");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    bogus: test\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "unknown option definition 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    section: global\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'config' requires 'type'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    depend:\n"
            "      bogus: test\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "unknown depend definition 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    command:\n"
            "      backup:\n"
            "        bogus: test\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "unknown option command definition 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    depend: bogus\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "dependency inherited from option 'bogus' before it is defined");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "    command:\n"
            "      bogus: {}\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "invalid command 'bogus' in option 'config' command list");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "    depend:\n"
            "      option: bogus\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "dependency on undefined option 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "    group: bogus\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'config' has invalid group 'bogus'");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  stanza:\n"
            "    type: string\n"
            "  config:\n"
            "    type: string\n"
            "    depend:\n"
            "      option: online\n"
            "  online:\n"
            "    type: boolean\n"
            "    depend:\n"
            "      option: config\n");

        TEST_ERROR(
            bldCfgParse(storageTest), FormatError,
            "unable to resolve dependencies for option(s) 'config, online'\n"
            "HINT: are there circular dependencies?");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'stanza' must exist");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "  stanza:\n"
            "    type: string\n"
            "    depend:\n"
            "      option: config\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'stanza' may not depend on other option");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: string\n"
            "  stanza:\n"
            "    type: string\n"
            "    command:\n"
            "      archive-get:\n"
            "        depend:\n"
            "          option: config\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "option 'stanza' command 'archive-get' may not depend on other option");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            TEST_COMMAND_VALID
            TEST_OPTION_GROUP_VALID
            "option:\n"
            "  config:\n"
            "    type: bool\n"
            "  stanza:\n"
            "    type: string\n"
            "    command:\n"
            "      archive-get:\n"
            "        depend:\n"
            "          option: config\n"
            "          default: false\n");

        TEST_ERROR(bldCfgParse(storageTest), FormatError, "dependency default invalid for non-boolean option 'stanza'");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("parse and render config");

        HRN_STORAGE_PUT_Z(
            storageTest, "src/build/config/config.yaml",
            "command:\n"
            "  archive-get:\n"
            "    command-role:\n"
            "      async: {}\n"
            "      local: {}\n"
            "      remote: {}\n"
            "    lock-type: archive\n"
            "    log-file: false\n"
            "    log-level-default: debug\n"
            "    parameter-allowed: true\n"
            "\n"
            "  backup:\n"
            "    internal: true\n"
            "    command-role:\n"
            "      local: {}\n"
            "      remote: {}\n"
            "    lock-type: backup\n"
            "    lock-required: true\n"
            "    lock-remote-required: true\n"
            "\n"
            "  help: {}\n"
            "\n"
            "  version: {}\n"
            "\n"
            "optionGroup:\n"
            "  pg: {}\n"
            "  repo: {}\n"
            "\n"
            "option:\n"
            "  timeout:\n"
            "    type: time\n"
            "    default: 10s\n"
            "    command:\n"
            "      backup:\n"
            "        default: 100ms\n"
            "      archive-get:\n"
            "        default: 30m\n"
            "    allow-range: [100ms, 7w]\n"
            "\n"
            "  buffer-size:\n"
            "    section: global\n"
            "    type: size\n"
            "    command:\n"
            "      backup: {}\n"
            "      archive-get:\n"
            "        default: 32KiB\n"
            "        allow-list:\n"
            "          - 32KiB\n"
            "    allow-list:\n"
            "      - 8KiB\n"
            "      - 16KiB: HAVE_LARGE_MEM\n"
            "    command-role:\n"
            "      main: {}\n"
            "\n"
            "  compress-network:\n"
            "    section: global\n"
            "    type: string-id\n"
            "    command:\n"
            "      backup:\n"
            "        allow-list:\n"
            "          - gz\n"
            "    command-role:\n"
            "      main: {}\n"
            "\n"
            "  compress-type:\n"
            "    section: global\n"
            "    type: string\n"
            "    default: gz\n"
            "    command:\n"
            "      backup: {}\n"
            "      archive-get:\n"
            "        depend:\n"
            "          option: config\n"
            "    command-role: {}\n"
            "    deprecate:\n"
            "      compress: {}\n"
            "\n"
            "  compress-level:\n"
            "    section: global\n"
            "    type: integer\n"
            "    required: false\n"
            "    allow-range: [0, 9]\n"
            "    command: compress-type\n"
            "    depend: compress-type\n"
            "\n"
            "  compress-level-network:\n"
            "    inherit: compress-level\n"
            "    internal: true\n"
            "    secure: true\n"
            "    default: ~\n"
            "    depend:\n"
            "      option: compress-type\n"
            "      list:\n"
            "        - none\n"
            "        - gz\n"
            "\n"
            "  config:\n"
            "    type: string\n"
            "    default: CFGOPTDEF_CONFIG_PATH \"/\" PROJECT_CONFIG_FILE\n"
            "    default-literal: true\n"
            "    negate: true\n"
            "    command:\n"
            "      backup: {}\n"
            "      archive-get:\n"
            "        default: CFGOPTDEF_CONFIG_PATH \"/.-\" PROJECT_CONFIG_FILE\n"
            "\n"
            "  config-include:\n"
            "    section: global\n"
            "    type: path\n"
            "    default: /include\n"
            "    command-role:\n"
            "      main: {}\n"
            "\n"
            "  log-level-console:\n"
            "    section: global\n"
            "    type: string-id\n"
            "    default: warn\n"
            "    allow-list:\n"
            "      - off\n"
            "      - error\n"
            "      - warn\n"
            "      - debug1\n"
            "\n"
            "  log-level-file:\n"
            "    section: global\n"
            "    type: string-id\n"
            "    default: info\n"
            "    allow-list: log-level-console\n"
            "    command:\n"
            "      backup:\n"
            "        internal: true\n"
            "        required: false\n"
            "        default: warn\n"
            "        allow-list:\n"
            "          - off\n"
            "          - warn\n"
            "        depend:\n"
            "          option: log-level-console\n"
            "          list:\n"
            "            - warn\n"
            "        command-role:\n"
            "          main: {}\n"
            "      help: {}\n"
            "      archive-get:\n"
            "        default: ~\n"
            "\n"
            "  stanza:\n"
            "    type: list\n"
            "    command:\n"
            "      archive-get: {}\n"
            "      backup: {}\n"
            "\n"
            "  online:\n"
            "    type: boolean\n"
            "    default: true\n"
            "    command:\n"
            "      backup:\n"
            "        default: false\n"
            "    command-role:\n"
            "      main: {}\n"
            "    deprecate:\n"
            "      online-old: {}\n"
            "\n"
            "  backup-standby:\n"
            "    section: global\n"
            "    type: boolean\n"
            "    beta: true\n"
            "    default: false\n"
            "    depend:\n"
            "      option: online\n"
            "      default: true\n"
            "      list:\n"
            "        - true\n"
            "    command:\n"
            "      backup: {}\n"
            "    command-role:\n"
            "      async: {}\n"
            "      main: {}\n"
            "    deprecate:\n"
            "      backup-standby-old: {}\n"
            "\n"
            "  pg-path:\n"
            "    section: stanza\n"
            "    group: pg\n"
            "    type: hash\n"
            "    deprecate:\n"
            "      pg-path: {}\n"
            "      pg?-path-indexed: {}\n"
            "      db-path: {}\n"
            "      db?-path: {}\n"
            "\n"
            "  pg-host:\n"
            "    section: stanza\n"
            "    group: pg\n"
            "    type: string\n"
            "    command:\n"
            "      archive-get:\n"
            "        default: host1\n"
            "      backup: {}\n"
            "    deprecate:\n"
            "      pg?-host: {}\n");

        TEST_RESULT_VOID(bldCfgRender(storageTest, bldCfgParse(storageTest), false), "parse and render");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("bldCfgRenderVar128Size()");

        TEST_RESULT_UINT(bldCfgRenderVar128Size(10000), 2, "check size");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("check config.auto.h");

        TEST_STORAGE_GET(
            storageTest,
            "src/config/config.auto.h",
            COMMENT_BLOCK_BEGIN "\n"
            "Command and Option Configuration\n"
            "\n"
            "Automatically generated by 'build-code config' -- do not modify directly.\n"
            COMMENT_BLOCK_END "\n"
            "#ifndef CONFIG_CONFIG_AUTO_H\n"
            "#define CONFIG_CONFIG_AUTO_H\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Command constants\n"
            COMMENT_BLOCK_END "\n"
            "#define CFGCMD_ARCHIVE_GET                                          \"archive-get\"\n"
            "#define CFGCMD_BACKUP                                               \"backup\"\n"
            "#define CFGCMD_HELP                                                 \"help\"\n"
            "#define CFGCMD_VERSION                                              \"version\"\n"
            "\n"
            "#define CFG_COMMAND_TOTAL                                           4\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option group constants\n"
            COMMENT_BLOCK_END "\n"
            "#define CFG_OPTION_GROUP_TOTAL                                      2\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option constants\n"
            COMMENT_BLOCK_END "\n"
            "#define CFGOPT_BACKUP_STANDBY                                       \"backup-standby\"\n"
            "#define CFGOPT_BUFFER_SIZE                                          \"buffer-size\"\n"
            "#define CFGOPT_COMPRESS_LEVEL                                       \"compress-level\"\n"
            "#define CFGOPT_COMPRESS_LEVEL_NETWORK                               \"compress-level-network\"\n"
            "#define CFGOPT_COMPRESS_NETWORK                                     \"compress-network\"\n"
            "#define CFGOPT_COMPRESS_TYPE                                        \"compress-type\"\n"
            "#define CFGOPT_CONFIG                                               \"config\"\n"
            "#define CFGOPT_CONFIG_INCLUDE                                       \"config-include\"\n"
            "#define CFGOPT_LOG_LEVEL_CONSOLE                                    \"log-level-console\"\n"
            "#define CFGOPT_LOG_LEVEL_FILE                                       \"log-level-file\"\n"
            "#define CFGOPT_ONLINE                                               \"online\"\n"
            "#define CFGOPT_STANZA                                               \"stanza\"\n"
            "#define CFGOPT_TIMEOUT                                              \"timeout\"\n"
            "\n"
            "#define CFG_OPTION_TOTAL                                            15\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option value constants\n"
            COMMENT_BLOCK_END "\n"
            "#define CFGOPTVAL_COMPRESS_NETWORK_GZ                               STRID5(\"gz\", 0x3470)\n"
            "#define CFGOPTVAL_COMPRESS_NETWORK_GZ_Z                             \"gz\"\n"
            "\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_DEBUG1                          STRID6(\"debug1\", 0x7475421441)\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_DEBUG1_Z                        \"debug1\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_ERROR                           STRID5(\"error\", 0x127ca450)\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_ERROR_Z                         \"error\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_OFF                             STRID5(\"off\", 0x18cf0)\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_OFF_Z                           \"off\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_WARN                            STRID5(\"warn\", 0x748370)\n"
            "#define CFGOPTVAL_LOG_LEVEL_CONSOLE_WARN_Z                          \"warn\"\n"
            "\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_DEBUG1                             STRID6(\"debug1\", 0x7475421441)\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_DEBUG1_Z                           \"debug1\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_ERROR                              STRID5(\"error\", 0x127ca450)\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_ERROR_Z                            \"error\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_OFF                                STRID5(\"off\", 0x18cf0)\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_OFF_Z                              \"off\"\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_WARN                               STRID5(\"warn\", 0x748370)\n"
            "#define CFGOPTVAL_LOG_LEVEL_FILE_WARN_Z                             \"warn\"\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Command enum\n"
            COMMENT_BLOCK_END "\n"
            "typedef enum\n"
            "{\n"
            "    cfgCmdArchiveGet,\n"
            "    cfgCmdBackup,\n"
            "    cfgCmdHelp,\n"
            "    cfgCmdVersion,\n"
            "    cfgCmdNone,\n"
            "} ConfigCommand;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option group enum\n"
            COMMENT_BLOCK_END "\n"
            "typedef enum\n"
            "{\n"
            "    cfgOptGrpPg,\n"
            "    cfgOptGrpRepo,\n"
            "} ConfigOptionGroup;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option enum\n"
            COMMENT_BLOCK_END "\n"
            "typedef enum\n"
            "{\n"
            "    cfgOptBackupStandby,\n"
            "    cfgOptBufferSize,\n"
            "    cfgOptCompressLevel,\n"
            "    cfgOptCompressLevelNetwork,\n"
            "    cfgOptCompressNetwork,\n"
            "    cfgOptCompressType,\n"
            "    cfgOptConfig,\n"
            "    cfgOptConfigInclude,\n"
            "    cfgOptLogLevelConsole,\n"
            "    cfgOptLogLevelFile,\n"
            "    cfgOptOnline,\n"
            "    cfgOptPgHost,\n"
            "    cfgOptPgPath,\n"
            "    cfgOptStanza,\n"
            "    cfgOptTimeout,\n"
            "} ConfigOption;\n"
            "\n"
            "#endif\n");

        // -------------------------------------------------------------------------------------------------------------------------
        TEST_TITLE("check parse.auto.c.inc");

        TEST_STORAGE_GET(
            storageTest,
            "src/config/parse.auto.c.inc",
            COMMENT_BLOCK_BEGIN "\n"
            "Config Parse Rules\n"
            "\n"
            "Automatically generated by 'build-code config' -- do not modify directly.\n"
            COMMENT_BLOCK_END "\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Rule Strings\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_STR(value)                                   PARSE_RULE_U32_1(parseRuleValStr##value)\n"
            "\n"
            "static const StringPub parseRuleValueStr[] =\n"
            "{\n"
            "    PARSE_RULE_STRPUB(\"/include\"),\n"
            "    PARSE_RULE_STRPUB(\"0\"),\n"
            "    PARSE_RULE_STRPUB(\"100ms\"),\n"
            "    PARSE_RULE_STRPUB(\"10s\"),\n"
            "    PARSE_RULE_STRPUB(\"16KiB\"),\n"
            "    PARSE_RULE_STRPUB(\"30m\"),\n"
            "    PARSE_RULE_STRPUB(\"32KiB\"),\n"
            "    PARSE_RULE_STRPUB(\"7w\"),\n"
            "    PARSE_RULE_STRPUB(\"8KiB\"),\n"
            "    PARSE_RULE_STRPUB(\"9\"),\n"
            "    PARSE_RULE_STRPUB(\"debug1\"),\n"
            "    PARSE_RULE_STRPUB(\"error\"),\n"
            "    PARSE_RULE_STRPUB(\"gz\"),\n"
            "    PARSE_RULE_STRPUB(\"host1\"),\n"
            "    PARSE_RULE_STRPUB(\"info\"),\n"
            "    PARSE_RULE_STRPUB(\"off\"),\n"
            "    PARSE_RULE_STRPUB(\"warn\"),\n"
            "    PARSE_RULE_STRPUB(CFGOPTDEF_CONFIG_PATH \"/\" PROJECT_CONFIG_FILE),\n"
            "    PARSE_RULE_STRPUB(CFGOPTDEF_CONFIG_PATH \"/.-\" PROJECT_CONFIG_FILE),\n"
            "};\n"
            "\n"
            "typedef enum\n"
            "{\n"
            "    parseRuleValStrQT_FS_include_QT,\n"
            "    parseRuleValStrQT_0_QT,\n"
            "    parseRuleValStrQT_100ms_QT,\n"
            "    parseRuleValStrQT_10s_QT,\n"
            "    parseRuleValStrQT_16KiB_QT,\n"
            "    parseRuleValStrQT_30m_QT,\n"
            "    parseRuleValStrQT_32KiB_QT,\n"
            "    parseRuleValStrQT_7w_QT,\n"
            "    parseRuleValStrQT_8KiB_QT,\n"
            "    parseRuleValStrQT_9_QT,\n"
            "    parseRuleValStrQT_debug1_QT,\n"
            "    parseRuleValStrQT_error_QT,\n"
            "    parseRuleValStrQT_gz_QT,\n"
            "    parseRuleValStrQT_host1_QT,\n"
            "    parseRuleValStrQT_info_QT,\n"
            "    parseRuleValStrQT_off_QT,\n"
            "    parseRuleValStrQT_warn_QT,\n"
            "    parseRuleValStrCFGOPTDEF_CONFIG_PATH_SP_QT_FS_QT_SP_PROJECT_CONFIG_FILE,\n"
            "    parseRuleValStrCFGOPTDEF_CONFIG_PATH_SP_QT_FS_DT_DS_QT_SP_PROJECT_CONFIG_FILE,\n"
            "} ParseRuleValueStr;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Rule StringIds\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_STRID(value)                                 PARSE_RULE_U32_1(parseRuleValStrId##value)\n"
            "\n"
            "static const StringId parseRuleValueStrId[] =\n"
            "{\n"
            "    STRID6(\"debug1\", 0x7475421441),\n"
            "    STRID5(\"error\", 0x127ca450),\n"
            "    STRID5(\"gz\", 0x3470),\n"
            "    STRID5(\"info\", 0x799c90),\n"
            "    STRID5(\"off\", 0x18cf0),\n"
            "    STRID5(\"warn\", 0x748370),\n"
            "};\n"
            "\n"
            "static const uint8_t parseRuleValueStrIdStrMap[] =\n"
            "{\n"
            "    parseRuleValStrQT_debug1_QT,\n"
            "    parseRuleValStrQT_error_QT,\n"
            "    parseRuleValStrQT_gz_QT,\n"
            "    parseRuleValStrQT_info_QT,\n"
            "    parseRuleValStrQT_off_QT,\n"
            "    parseRuleValStrQT_warn_QT,\n"
            "};\n"
            "\n"
            "typedef enum\n"
            "{\n"
            "    parseRuleValStrIdDebug1,\n"
            "    parseRuleValStrIdError,\n"
            "    parseRuleValStrIdGz,\n"
            "    parseRuleValStrIdInfo,\n"
            "    parseRuleValStrIdOff,\n"
            "    parseRuleValStrIdWarn,\n"
            "} ParseRuleValueStrId;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Rule Ints\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_INT(value)                                   PARSE_RULE_U32_1(parseRuleValInt##value)\n"
            "\n"
            "static const int parseRuleValueInt[] =\n"
            "{\n"
            "    0,\n"
            "    9,\n"
            "};\n"
            "\n"
            "static const uint8_t parseRuleValueIntStrMap[] =\n"
            "{\n"
            "    parseRuleValStrQT_0_QT,\n"
            "    parseRuleValStrQT_9_QT,\n"
            "};\n"
            "\n"
            "typedef enum\n"
            "{\n"
            "    parseRuleValInt0,\n"
            "    parseRuleValInt9,\n"
            "} ParseRuleValueInt;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Rule Sizes\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_SIZE(value)                                  PARSE_RULE_U32_1(parseRuleValSize##value)\n"
            "\n"
            "static const int64_t parseRuleValueSize[] =\n"
            "{\n"
            "    8192,\n"
            "    16384,\n"
            "    32768,\n"
            "};\n"
            "\n"
            "static const uint8_t parseRuleValueSizeStrMap[] =\n"
            "{\n"
            "    parseRuleValStrQT_8KiB_QT,\n"
            "    parseRuleValStrQT_16KiB_QT,\n"
            "    parseRuleValStrQT_32KiB_QT,\n"
            "};\n"
            "\n"
            "typedef enum\n"
            "{\n"
            "    parseRuleValSize8KiB,\n"
            "    parseRuleValSize16KiB,\n"
            "    parseRuleValSize32KiB,\n"
            "} ParseRuleValueSize;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Rule Times\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_TIME(value)                                  PARSE_RULE_U32_1(parseRuleValTime##value)\n"
            "\n"
            "static const unsigned int parseRuleValueTime[] =\n"
            "{\n"
            "    100,\n"
            "    10000,\n"
            "    1800000,\n"
            "    4233600000,\n"
            "};\n"
            "\n"
            "static const uint8_t parseRuleValueTimeStrMap[] =\n"
            "{\n"
            "    parseRuleValStrQT_100ms_QT,\n"
            "    parseRuleValStrQT_10s_QT,\n"
            "    parseRuleValStrQT_30m_QT,\n"
            "    parseRuleValStrQT_7w_QT,\n"
            "};\n"
            "\n"
            "typedef enum\n"
            "{\n"
            "    parseRuleValTime100ms,\n"
            "    parseRuleValTime10s,\n"
            "    parseRuleValTime30m,\n"
            "    parseRuleValTime7w,\n"
            "} ParseRuleValueTime;\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Command parse data\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_CMD(value)                                   PARSE_RULE_U32_1(cfgCmd##value)\n"
            "\n"
            "static const ParseRuleCommand parseRuleCommand[CFG_COMMAND_TOTAL] =\n"
            "{\n"
            "    PARSE_RULE_COMMAND\n"
            "    (\n"
            "        PARSE_RULE_COMMAND_NAME(\"archive-get\"),\n"
            "        PARSE_RULE_COMMAND_LOCK_TYPE(Archive),\n"
            "        PARSE_RULE_COMMAND_LOG_LEVEL_DEFAULT(Debug),\n"
            "        PARSE_RULE_COMMAND_PARAMETER_ALLOWED(true),\n"
            "\n"
            "        PARSE_RULE_COMMAND_ROLE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_COMMAND_ROLE(Async)\n"
            "            PARSE_RULE_COMMAND_ROLE(Local)\n"
            "            PARSE_RULE_COMMAND_ROLE(Main)\n"
            "            PARSE_RULE_COMMAND_ROLE(Remote)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_COMMAND\n"
            "    (\n"
            "        PARSE_RULE_COMMAND_NAME(\"backup\"),\n"
            "        PARSE_RULE_COMMAND_LOCK_REQUIRED(true),\n"
            "        PARSE_RULE_COMMAND_LOCK_REMOTE_REQUIRED(true),\n"
            "        PARSE_RULE_COMMAND_LOCK_TYPE(Backup),\n"
            "        PARSE_RULE_COMMAND_LOG_FILE(true),\n"
            "        PARSE_RULE_COMMAND_LOG_LEVEL_DEFAULT(Info),\n"
            "\n"
            "        PARSE_RULE_COMMAND_ROLE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_COMMAND_ROLE(Local)\n"
            "            PARSE_RULE_COMMAND_ROLE(Main)\n"
            "            PARSE_RULE_COMMAND_ROLE(Remote)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_COMMAND\n"
            "    (\n"
            "        PARSE_RULE_COMMAND_NAME(\"help\"),\n"
            "        PARSE_RULE_COMMAND_LOCK_TYPE(None),\n"
            "        PARSE_RULE_COMMAND_LOG_FILE(true),\n"
            "        PARSE_RULE_COMMAND_LOG_LEVEL_DEFAULT(Info),\n"
            "\n"
            "        PARSE_RULE_COMMAND_ROLE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_COMMAND_ROLE(Main)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_COMMAND\n"
            "    (\n"
            "        PARSE_RULE_COMMAND_NAME(\"version\"),\n"
            "        PARSE_RULE_COMMAND_LOCK_TYPE(None),\n"
            "        PARSE_RULE_COMMAND_LOG_FILE(true),\n"
            "        PARSE_RULE_COMMAND_LOG_LEVEL_DEFAULT(Info),\n"
            "\n"
            "        PARSE_RULE_COMMAND_ROLE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_COMMAND_ROLE(Main)\n"
            "        ),\n"
            "    ),\n"
            "};\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option group parse data\n"
            COMMENT_BLOCK_END "\n"
            "static const ParseRuleOptionGroup parseRuleOptionGroup[CFG_OPTION_GROUP_TOTAL] =\n"
            "{\n"
            "    PARSE_RULE_OPTION_GROUP\n"
            "    (\n"
            "        PARSE_RULE_OPTION_GROUP_NAME(\"pg\"),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION_GROUP\n"
            "    (\n"
            "        PARSE_RULE_OPTION_GROUP_NAME(\"repo\"),\n"
            "    ),\n"
            "};\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option parse data\n"
            COMMENT_BLOCK_END "\n"
            "#define PARSE_RULE_VAL_OPT(value)                                   PARSE_RULE_U32_1(cfgOpt##value)\n"
            "\n"
            "static const ParseRuleOption parseRuleOption[CFG_OPTION_TOTAL] =\n"
            "{\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"backup-standby\"),\n"
            "        PARSE_RULE_OPTION_TYPE(Boolean),\n"
            "        PARSE_RULE_OPTION_BETA(true),\n"
            "        PARSE_RULE_OPTION_NEGATE(true),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(Global),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_OPTIONAL_DEPEND_DEFAULT(PARSE_RULE_VAL_BOOL_TRUE),\n"
            "                    PARSE_RULE_VAL_OPT(Online),\n"
            "                    PARSE_RULE_VAL_BOOL_TRUE,\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_BOOL_FALSE,\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"buffer-size\"),\n"
            "        PARSE_RULE_OPTION_TYPE(Size),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(Global),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(ArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_SIZE(32KiB),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_SIZE(32KiB),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_SIZE(8KiB),\n"
            "                    PARSE_RULE_VAL_SIZE(16KiB),\n"
            "#ifndef HAVE_LARGE_MEM\n"
            "                        PARSE_RULE_BOOL_FALSE,\n"
            "#endif\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"compress-level\"),\n"
            "        PARSE_RULE_OPTION_TYPE(Integer),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(false),\n"
            "        PARSE_RULE_OPTION_SECTION(Global),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(ArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(Config),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(0),\n"
            "                    PARSE_RULE_VAL_INT(9),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(0),\n"
            "                    PARSE_RULE_VAL_INT(9),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"compress-level-network\"),\n"
            "        PARSE_RULE_OPTION_TYPE(Integer),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(false),\n"
            "        PARSE_RULE_OPTION_SECTION(Global),\n"
            "        PARSE_RULE_OPTION_SECURE(true),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(ArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(Config),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(0),\n"
            "                    PARSE_RULE_VAL_INT(9),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(CompressType),\n"
            "                    PARSE_RULE_VAL_STRID(None),\n"
            "                    PARSE_RULE_VAL_STRID(Gz),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_INT(0),\n"
            "                    PARSE_RULE_VAL_INT(9),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"compress-network\"),\n"
            "        PARSE_RULE_OPTION_TYPE(StringId),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(Global),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(Backup),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(Gz),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"compress-type\"),\n"
            "        PARSE_RULE_OPTION_TYPE(String),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(Global),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(ArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(Config),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(QT_gz_QT),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(QT_gz_QT),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"config\"),\n"
            "        PARSE_RULE_OPTION_TYPE(String),\n"
            "        PARSE_RULE_OPTION_NEGATE(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(CommandLine),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(ArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(CFGOPTDEF_CONFIG_PATH_SP_QT_FS_DT_DS_QT_SP_PROJECT_CONFIG_FILE),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(CFGOPTDEF_CONFIG_PATH_SP_QT_FS_QT_SP_PROJECT_CONFIG_FILE),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"config-include\"),\n"
            "        PARSE_RULE_OPTION_TYPE(Path),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(Global),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(QT_FS_include_QT),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"log-level-console\"),\n"
            "        PARSE_RULE_OPTION_TYPE(StringId),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(Global),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(Off),\n"
            "                    PARSE_RULE_VAL_STRID(Error),\n"
            "                    PARSE_RULE_VAL_STRID(Warn),\n"
            "                    PARSE_RULE_VAL_STRID(Debug1),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(Warn),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"log-level-file\"),\n"
            "        PARSE_RULE_OPTION_TYPE(StringId),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(Global),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "            PARSE_RULE_OPTION_COMMAND(Help)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(Backup),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEPEND\n"
            "                (\n"
            "                    PARSE_RULE_VAL_OPT(LogLevelConsole),\n"
            "                    PARSE_RULE_VAL_STRID(Warn),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(Off),\n"
            "                    PARSE_RULE_VAL_STRID(Warn),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(Warn),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_NOT_REQUIRED(),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_LIST\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(Off),\n"
            "                    PARSE_RULE_VAL_STRID(Error),\n"
            "                    PARSE_RULE_VAL_STRID(Warn),\n"
            "                    PARSE_RULE_VAL_STRID(Debug1),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STRID(Info),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"online\"),\n"
            "        PARSE_RULE_OPTION_TYPE(Boolean),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(CommandLine),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(Backup),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_BOOL_FALSE,\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_BOOL_TRUE,\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"pg-host\"),\n"
            "        PARSE_RULE_OPTION_TYPE(String),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(Stanza),\n"
            "        PARSE_RULE_OPTION_GROUP_ID(Pg),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(ArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_STR(QT_host1_QT),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"pg-path\"),\n"
            "        PARSE_RULE_OPTION_TYPE(Hash),\n"
            "        PARSE_RULE_OPTION_RESET(true),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(Stanza),\n"
            "        PARSE_RULE_OPTION_MULTI(true),\n"
            "        PARSE_RULE_OPTION_GROUP_ID(Pg),\n"
            "        PARSE_RULE_OPTION_DEPRECATE_MATCH(true),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"stanza\"),\n"
            "        PARSE_RULE_OPTION_TYPE(List),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(CommandLine),\n"
            "        PARSE_RULE_OPTION_MULTI(true),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "    ),\n"
            COMMENT_SEPARATOR "\n"
            "    PARSE_RULE_OPTION\n"
            "    (\n"
            "        PARSE_RULE_OPTION_NAME(\"timeout\"),\n"
            "        PARSE_RULE_OPTION_TYPE(Time),\n"
            "        PARSE_RULE_OPTION_REQUIRED(true),\n"
            "        PARSE_RULE_OPTION_SECTION(CommandLine),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_MAIN_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_ASYNC_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_LOCAL_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTION_COMMAND_ROLE_REMOTE_VALID_LIST\n"
            "        (\n"
            "            PARSE_RULE_OPTION_COMMAND(ArchiveGet)\n"
            "            PARSE_RULE_OPTION_COMMAND(Backup)\n"
            "        ),\n"
            "\n"
            "        PARSE_RULE_OPTIONAL\n"
            "        (\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(ArchiveGet),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_TIME(100ms),\n"
            "                    PARSE_RULE_VAL_TIME(7w),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_TIME(30m),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_FILTER_CMD\n"
            "                (\n"
            "                    PARSE_RULE_VAL_CMD(Backup),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_TIME(100ms),\n"
            "                    PARSE_RULE_VAL_TIME(7w),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_TIME(100ms),\n"
            "                ),\n"
            "            ),\n"
            "\n"
            "            PARSE_RULE_OPTIONAL_GROUP\n"
            "            (\n"
            "                PARSE_RULE_OPTIONAL_ALLOW_RANGE\n"
            "                (\n"
            "                    PARSE_RULE_VAL_TIME(100ms),\n"
            "                    PARSE_RULE_VAL_TIME(7w),\n"
            "                ),\n"
            "\n"
            "                PARSE_RULE_OPTIONAL_DEFAULT\n"
            "                (\n"
            "                    PARSE_RULE_VAL_TIME(10s),\n"
            "                ),\n"
            "            ),\n"
            "        ),\n"
            "    ),\n"
            "};\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Option deprecations\n"
            COMMENT_BLOCK_END "\n"
            "#define CFG_OPTION_DEPRECATE_TOTAL                                  7\n"
            "\n"
            "static const ParseRuleOptionDeprecate parseRuleOptionDeprecate[CFG_OPTION_DEPRECATE_TOTAL] =\n"
            "{\n"
            "    {\n"
            "        .name = \"backup-standby-old\",\n"
            "        .id = cfgOptBackupStandby,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"compress\",\n"
            "        .id = cfgOptCompressType,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"db-path\",\n"
            "        .id = cfgOptPgPath,\n"
            "        .indexed = true,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"online-old\",\n"
            "        .id = cfgOptOnline,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"pg-host\",\n"
            "        .id = cfgOptPgHost,\n"
            "        .indexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"pg-path\",\n"
            "        .id = cfgOptPgPath,\n"
            "        .unindexed = true,\n"
            "    },\n"
            COMMENT_SEPARATOR "\n"
            "    {\n"
            "        .name = \"pg-path-indexed\",\n"
            "        .id = cfgOptPgPath,\n"
            "        .indexed = true,\n"
            "    },\n"
            "};\n"
            "\n"
            COMMENT_BLOCK_BEGIN "\n"
            "Order for option parse resolution\n"
            COMMENT_BLOCK_END "\n"
            "static const uint8_t optionResolveOrder[] =\n"
            "{\n"
            "    cfgOptStanza,\n"
            "    cfgOptBufferSize,\n"
            "    cfgOptCompressNetwork,\n"
            "    cfgOptConfig,\n"
            "    cfgOptConfigInclude,\n"
            "    cfgOptLogLevelConsole,\n"
            "    cfgOptLogLevelFile,\n"
            "    cfgOptOnline,\n"
            "    cfgOptPgHost,\n"
            "    cfgOptPgPath,\n"
            "    cfgOptTimeout,\n"
            "    cfgOptBackupStandby,\n"
            "    cfgOptCompressLevel,\n"
            "    cfgOptCompressType,\n"
            "    cfgOptCompressLevelNetwork,\n"
            "};\n");
    }

    FUNCTION_HARNESS_RETURN_VOID();
}
