// -*- Mode: C++; -*-
//                            Package   : omniORB
// tcpSocketVaxRoutines.cc    Created on: 2000
//                            Author    : Bruce Visscher (visschb@rjrt.com)
//
//    Copyright (C) 1999-2000 Bruce Visscher
//
//    This file is part of the omniORB library
//
//    The omniORB library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  
//    02111-1307, USA
//
// Revision 1.1  2000/05/12 15:18:18   bcv
//
// Simplified mutex initialization.  Modified to not set errno except when an
// error occurs.
//
#include <errno.h>
#include <iodef.h>
#include <ssdef.h>
#include <ucx$inetdef.h>    // UCX network definitions
#include <pthread.h>
#include <starlet.h>
#include "TcpSocketVaxRoutines.h"

class tcpSocketVaxAstArgument {
public:
  tcpSocketVaxAstArgument() : pd_complete(0) {
#if __VMS_VER >= 70000000
    pthread_mutex_init(&pd_mutex, 0);
#else
    pthread_mutex_init(&pd_mutex, pthread_mutexattr_default);
#endif
#if __VMS_VER >= 70000000
    pthread_cond_init(&pd_condition, 0);
#else
    pthread_cond_init(&pd_condition, pthread_condattr_default);
#endif

  }
  ~tcpSocketVaxAstArgument() {
    pthread_cond_destroy(&pd_condition);
    pthread_mutex_destroy(&pd_mutex);
  }
  void Signal() {
    pd_complete=1;
    pthread_cond_signal_int_np(&pd_condition);
  }
  void Wait() {
    pthread_mutex_lock(&pd_mutex);
    while (!pd_complete) {
      pthread_cond_wait(&pd_condition, &pd_mutex);
    }
    pthread_mutex_unlock(&pd_mutex);
  }
private:
  pthread_mutex_t pd_mutex;
  pthread_cond_t pd_condition;
  int pd_complete;
};

static void tcpSocketVaxAstCompletion(tcpSocketVaxAstArgument*);

struct item_list_2 {
  unsigned int pd_length;
  char* pd_address;
  unsigned int* pd_retlen;
};

#include <descrip.h>	// for $assign test.

int tcpSocketVaxAccept(int s, sockaddr* addr, size_t* size) {

  short channel(decc$get_sdc(s));
  tcpSocketVaxAstArgument astArgument;
  short newChannel;
  dsc$descriptor inet_dev = {10, DSC$K_CLASS_S, DSC$K_DTYPE_T, "UCX$DEVICE"};
  sys$assign(&inet_dev, &newChannel, 0, 0);
  short	iosb[4];	// I/O status block
  // Accept a connection from a client.
  unsigned int retlen;
  item_list_2 addrList;
  addrList.pd_length=*size;
  addrList.pd_address=(char*)addr;
  addrList.pd_retlen=&retlen;
  int status = sys$qio(0,			// Event flag
		    channel,		// Channel number
		    IO$_ACCESS|IO$M_ACCEPT,	// I/O function
		    iosb,			// I/O status block
		    &tcpSocketVaxAstCompletion, &astArgument,
		    0, 0,			// P1, P2 (not used)
		    &addrList,			// P3 Remote IP address
		    &newChannel,		// P4 Channel for new socket
		    0, 0);			// P5, P6 (not used)
  if (!(status&1)) {
    errno=EBADF;
    return -1;
  }

  astArgument.Wait();
  status = iosb[0];

  if (status & 1) {
    int newSocket=socket_fd(newChannel);
    return newSocket;
  }

  vaxc$errno=status;
  switch (status) {
    case SS$_ACCVIO:
      errno=EFAULT;
      return -1;
    case SS$_BADPARAM:
      errno=ENOTSOCK;
      return -1;
    case SS$_SUSPENDED:
      errno=EWOULDBLOCK;
      return -1;
    case SS$_REJECT:
      errno=EOPNOTSUPP;
      return -1;
    default:
      errno=EVMSERR;
      return -1;
  }
}

int tcpSocketVaxRecv(int s, void* buf, size_t length, int) {
  short channel(decc$get_sdc(s));
  short iosb[4];
  tcpSocketVaxAstArgument astArgument;
  int status = sys$qio(0,     // Event flag
		channel,      // Channel number
		IO$_READVBLK, //  I/O function
		iosb,	      //  I/O status block
		&tcpSocketVaxAstCompletion, &astArgument,
		buf,	      // P1 buffer
		length,	      // P2 buffer length
		0, 0, 0, 0);  // P3-P6
  if (!(status&1)) {
    errno=EBADF;
    return -1;
  }

  astArgument.Wait();
  status = iosb[0];

  if (status & 1) {
    return iosb[1];
  }

  vaxc$errno=status;
  //  My best guess in how the QIO return code should translate:
  switch (status) {
    case SS$_ACCVIO:
      errno=EFAULT;
      return -1;
    case SS$_BADPARAM:
      errno=ENOTSOCK;
      return -1;
    case SS$_LINKDISCON:
      return 0;
    case SS$_SUSPENDED:
      errno=EWOULDBLOCK;
      return -1;
    case SS$_NOLINKS:
      errno=EPIPE;
      return -1;
    default:
      errno=EVMSERR;
      return -1;
  }
}

int tcpSocketVaxSend(int s, const void* buf, size_t length, int) {
  short channel(decc$get_sdc(s));
  short iosb[4];
  tcpSocketVaxAstArgument astArgument;
  int status = sys$qio(	0,			// event flag
			channel,
			IO$_WRITEVBLK,		// function
                        iosb,			// iosb
			&tcpSocketVaxAstCompletion, &astArgument,
			buf,			// IO buffer 
			length,			// IO buffer size
			0,			// p3 UNUSED
			0,			// p4 IO options flag
                        0, 0			// p5, p6 UNUSED
  );
  if (status & 1) {
    astArgument.Wait();
    status=iosb[0];
  } else {
    vaxc$errno=status;
    errno=EBADF;
    return -1;
  }
  if (status & 1) {
    return iosb[1];
  }

  vaxc$errno=status;
  switch (status) {
    case SS$_ACCVIO:
      errno=EFAULT;
      return -1;
    case SS$_BADPARAM:
      errno=ENOTSOCK;
      return -1;
    case SS$_SUSPENDED:
      errno=EWOULDBLOCK;
      return -1;
    case SS$_TOOMUCHDATA:
      errno=EMSGSIZE;
      return -1;
    default:
      errno=EVMSERR;
      return -1;
  }
}

static void tcpSocketVaxAstCompletion(tcpSocketVaxAstArgument* arg) {
  arg->Signal();
}
