/**
 * Copyright (C) 2006 Steve Ratcliffe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 * Author: steve
 * Date: 26-Dec-2006
 */
package uk.me.parabola.mkgmap.general;

import java.util.Map;
import java.util.HashMap;

import uk.me.parabola.imgfmt.app.Coord;

/**
 * A map element is a point, line or shape that appears on the map.  This
 * class holds all the common routines that are shared across all elements.
 * 
 * @author Steve Ratcliffe.
 */
public abstract class MapElement {
	private String name;
	private String ref;
	private int type;

	private int minResolution = 24;
	private int maxResolution = 24;
	
	private String zipCode;
	private String city;
	private String region;
	private String country;	
	
	private final Map<String, String> attributes = new HashMap<String, String>();

	protected MapElement() {
	}

	protected MapElement(MapElement orig) {
		name = orig.name;
		ref = orig.ref;
		type = orig.type;
		minResolution = orig.minResolution;
		maxResolution = orig.maxResolution;
	}

	/**
	 * Provide a copy of this MapElement without geometry. This is used
	 * when filtering and clipping to create modified versions.
	 *
	 * @return the copy;
	 */
	public abstract MapElement copy();

	public String getName() {
		return name;
	}

	public String getRef() {
		return ref;
	}

	public void setName(String name) {
	  if(name != null)
		this.name = name.toUpperCase();
	}

	public void setRef(String ref) {
	  if(ref != null)
		this.ref = ref.toUpperCase();
	}

	public String getCity() {
		return city;
	}

	public void setCity(String city) {
	  if(city != null)
		this.city = city.toUpperCase();
	}
	
	public String getZip() {
		return zipCode;
	}

	public void setZip(String zip) {
		this.zipCode = zip;
	}

	public String getCountry() {
		return country;
	}

	public void setCountry(String country) {
	  if(country != null)
		this.country = country.toUpperCase();
	}
	
	public String getRegion() {
		return region;
	}

	public void setRegion(String region) {
	  if(region != null)
  		this.region = region.toUpperCase();
	}	
	
	public String getStreet() {
		return attributes.get("street");
	}

	public void setStreet(String street) {
		attributes.put("street", street);	
	}

	public String getPhone() {
		return attributes.get("phone");
	}

	public void setPhone(String phone) {
	
	  if(phone.startsWith("00"))
		{
		  phone = phone.replaceFirst("00","+");
		}
		attributes.put("phone", phone);	
	}

	public String getHouseNumber() {
		return attributes.get("houseNumber");
	}

	public void setHouseNumber(String houseNumber) {
		attributes.put("houseNumber", houseNumber);		
	}
	
	public String getIsIn() {
		return attributes.get("isIn");
	}

	public void setIsIn(String isIn) {
	  if(isIn != null)
		attributes.put("isIn", isIn.toUpperCase());
	}	


	/**
	 * This is the type code that goes in the .img file so that the GPS device
	 * knows what to display.
	 *
	 * @return the type.
	 */
	public int getType() {
		return type;
	}

	public void setType(int type) {
		this.type = type;
	}

	/**
	 * Get the 'location' of the element.  This is the mid point of the bounding
	 * box for the element.  For a point, this will be the coordinates of the
	 * point itself of course.
	 *
	 * @return Co-ordinate of the mid-point of the bounding box of the element.
	 */
	public abstract Coord getLocation();

	/**
	 * Get the resolutions that an element should be displayed at.
	 * It will return the minimum resolution at which this element should be
	 * displayed at.
	 *
	 * @return The lowest resolution at which the element will be visible.
	 */
	public int getMinResolution() {
		return minResolution;
	}

	public void setMinResolution(int minResolution) {
		this.minResolution = minResolution;
	}

	public int getMaxResolution() {
		return maxResolution;
	}

	public void setMaxResolution(int maxResolution) {
		this.maxResolution = maxResolution;
	}
}
