/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * Copyright (C) 2020 - 2024 Intel Corporation
 */

#ifndef IPU7_FW_BOOT_ABI_H
#define IPU7_FW_BOOT_ABI_H

#include "ipu7_fw_common_abi.h"
#include "ipu7_fw_syscom_abi.h"

#define IA_GOFO_FWLOG_SEVERITY_CRIT			(0U)
#define IA_GOFO_FWLOG_SEVERITY_ERROR			(1U)
#define IA_GOFO_FWLOG_SEVERITY_WARNING			(2U)
#define IA_GOFO_FWLOG_SEVERITY_INFO			(3U)
#define IA_GOFO_FWLOG_SEVERITY_DEBUG			(4U)
#define IA_GOFO_FWLOG_SEVERITY_VERBOSE			(5U)
#define IA_GOFO_FWLOG_MAX_LOGGER_SOURCES		(64U)

#define LOGGER_CONFIG_CHANNEL_ENABLE_HWPRINTF_BITMASK	BIT(0)
#define LOGGER_CONFIG_CHANNEL_ENABLE_SYSCOM_BITMASK	BIT(1)
#define LOGGER_CONFIG_CHANNEL_ENABLE_ALL_BITMASK \
	(LOGGER_CONFIG_CHANNEL_ENABLE_HWPRINTF_BITMASK | \
	 LOGGER_CONFIG_CHANNEL_ENABLE_SYSCOM_BITMASK)

struct ia_gofo_logger_config {
	u8 use_source_severity;
	u8 source_severity[IA_GOFO_FWLOG_MAX_LOGGER_SOURCES];
	u8 use_channels_enable_bitmask;
	u8 channels_enable_bitmask;
	u8 padding[1];
	ia_gofo_addr_t hw_printf_buffer_base_addr;
	u32 hw_printf_buffer_size_bytes;
};

static inline void ia_gofo_logger_config_abi_test_func(void)
{
	CHECK_ALIGN32(struct ia_gofo_logger_config);
}

#pragma pack(push, 1)

#define IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_MAX_REG_IDX_PER_APP \
	((u32)IA_GOFO_FW_BOOT_ID_MAX)
#define IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_IS_OFFSET (0U)
#define IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_PS_OFFSET \
	((IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_IS_OFFSET) + \
	(u32)(IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_MAX_REG_IDX_PER_APP))
#define IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_PRIMARY_OFFSET (0U)
#define IA_GOFO_CCG_IPU_BUTTRESS_FW_BOOT_PARAMS_SECONDARY_OFFSET (0x3000U / 4U)
#define IA_GOFO_HKR_IPU_BUTTRESS_FW_BOOT_PARAMS_SECONDARY_OFFSET \
	(IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_MAX_REG_IDX_PER_APP * 2U)
#define IA_GOFO_HKR_HIF_BUTTRESS_FW_BOOT_PARAMS_SECONDARY_OFFSET \
	(IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_MAX_REG_IDX_PER_APP)
#define IA_GOFO_CCG_IPU_BUTTRESS_FW_BOOT_PARAMS_MAX_REG_IDX \
	(IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_MAX_REG_IDX_PER_APP * 4U)
#define IA_GOFO_HKR_IPU_BUTTRESS_FW_BOOT_PARAMS_MAX_REG_IDX \
	(IA_GOFO_BUTTRESS_FW_BOOT_PARAMS_MAX_REG_IDX_PER_APP * 4U)

#define IA_GOFO_BOOT_RESERVED_SIZE (58U)
#define IA_GOFO_BOOT_SECONDARY_RESERVED_SIZE (IA_GOFO_BOOT_RESERVED_SIZE)
#define IA_GOFO_BOOT_SECONDARY_RESERVED_FIELDS \
	(sizeof(ia_gofo_addr_t) + sizeof(ia_gofo_addr_t) + sizeof(u32))

enum ia_gofo_buttress_reg_id {
	IA_GOFO_FW_BOOT_CONFIG_ID = 0,
	IA_GOFO_FW_BOOT_STATE_ID = 1,
	IA_GOFO_FW_BOOT_RESERVED1_ID = IA_GOFO_FW_BOOT_STATE_ID,
	IA_GOFO_FW_BOOT_SYSCOM_QUEUE_INDICES_BASE_ID = 2,
	IA_GOFO_FW_BOOT_UNTRUSTED_ADDR_MIN_ID = 3,
	IA_GOFO_FW_BOOT_RESERVED0_ID = IA_GOFO_FW_BOOT_UNTRUSTED_ADDR_MIN_ID,
	IA_GOFO_FW_BOOT_MESSAGING_VERSION_ID = 4,
	IA_GOFO_FW_BOOT_ID_MAX
};

#define IA_GOFO_FW_BOOT_STATE_IS_CRITICAL(boot_state) \
	(0xdead0000U == ((boot_state) & 0xffff0000U))

struct ia_gofo_boot_config {
	u32 length;
	struct ia_gofo_version_s config_version;
	struct ia_gofo_msg_version_list client_version_support;
	ia_gofo_addr_t pkg_dir;
	ia_gofo_addr_t subsys_config;
	u32 uc_tile_frequency_mhz;
	u16 checksum;
	u8 padding[2];
	u32 reserved[IA_GOFO_BOOT_RESERVED_SIZE];
	struct syscom_config_s syscom_context_config[1];
};

struct ia_gofo_secondary_boot_config {
	u32 length;
	struct ia_gofo_version_s config_version;
	struct ia_gofo_msg_version_list client_version_support;
	u8 reserved1[IA_GOFO_BOOT_SECONDARY_RESERVED_FIELDS];
	u16 checksum;
	u8 padding[2];
	u32 reserved2[IA_GOFO_BOOT_SECONDARY_RESERVED_SIZE];
	struct syscom_config_s syscom_context_config[1];
};

#define FW_BOOT_CONFIG_ALLOC_SIZE(num_queues) \
	((sizeof(struct ia_gofo_boot_config) + \
	(sizeof(struct syscom_queue_params_config) * num_queues)))

#pragma pack(pop)

static inline void ia_gofo_boot_config_abi_test_func(void)
{
	CHECK_ALIGN32(struct ia_gofo_boot_config);
	CHECK_ALIGN32(struct ia_gofo_secondary_boot_config);
}

#define IA_GOFO_WDT_TIMEOUT_ERR			0xdead0401U
#define IA_GOFO_MEM_FATAL_DME_ERR		0xdead0801U
#define IA_GOFO_MEM_UNCORRECTABLE_LOCAL_ERR	0xdead0802U
#define IA_GOFO_MEM_UNCORRECTABLE_DIRTY_ERR	0xdead0803U
#define IA_GOFO_MEM_UNCORRECTABLE_DTAG_ERR	0xdead0804U
#define IA_GOFO_MEM_UNCORRECTABLE_CACHE_ERR	0xdead0805U
#define IA_GOFO_DOUBLE_EXCEPTION_ERR		0xdead0806U
#define IA_GOFO_BIST_DMEM_FAULT_DETECTION_ERR	0xdead1000U
#define IA_GOFO_BIST_DATA_INTEGRITY_FAILURE	0xdead1010U

enum ia_gofo_boot_state {
	IA_GOFO_FW_BOOT_STATE_SECONDARY_BOOT_CONFIG_READY = 0x57a7b000U,
	IA_GOFO_FW_BOOT_STATE_UNINIT = 0x57a7e000U,
	IA_GOFO_FW_BOOT_STATE_STARTING_0 = 0x57a7d000U,
	IA_GOFO_FW_BOOT_STATE_MEM_INIT_DONE = 0x57a7d100U,
	IA_GOFO_FW_BOOT_STATE_BOOT_CONFIG_START = 0x57a7d200U,
	IA_GOFO_FW_BOOT_STATE_QUEUE_INIT_DONE = 0x57a7d300U,
	IA_GOFO_FW_BOOT_STATE_READY = 0x57a7e100U,
	IA_GOFO_FW_BOOT_STATE_CRIT_UNSPECIFIED = 0xdead0001U,
	IA_GOFO_FW_BOOT_STATE_CRIT_CFG_PTR = 0xdead0101U,
	IA_GOFO_FW_BOOT_STATE_CRIT_CFG_VERSION = 0xdead0201U,
	IA_GOFO_FW_BOOT_STATE_CRIT_MSG_VERSION = 0xdead0301U,
	IA_GOFO_FW_BOOT_STATE_CRIT_WDT_TIMEOUT = IA_GOFO_WDT_TIMEOUT_ERR,
	IA_GOFO_FW_BOOT_STATE_WRONG_DATA_SECTION_UNPACKING = 0xdead0501U,
	IA_GOFO_FW_BOOT_STATE_WRONG_RO_DATA_SECTION_UNPACKING = 0xdead0601U,
	IA_GOFO_FW_BOOT_STATE_INVALID_UNTRUSTED_ADDR_MIN = 0xdead0701U,
	IA_GOFO_FW_BOOT_STATE_CRIT_MEM_FATAL_DME = IA_GOFO_MEM_FATAL_DME_ERR,
	IA_GOFO_FW_BOOT_STATE_CRIT_MEM_UNCORRECTABLE_LOCAL =
	IA_GOFO_MEM_UNCORRECTABLE_LOCAL_ERR,
	IA_GOFO_FW_BOOT_STATE_CRIT_MEM_UNCORRECTABLE_DIRTY =
	IA_GOFO_MEM_UNCORRECTABLE_DIRTY_ERR,
	IA_GOFO_FW_BOOT_STATE_CRIT_MEM_UNCORRECTABLE_DTAG =
	IA_GOFO_MEM_UNCORRECTABLE_DTAG_ERR,
	IA_GOFO_FW_BOOT_STATE_CRIT_MEM_UNCORRECTABLE_CACHE =
	IA_GOFO_MEM_UNCORRECTABLE_CACHE_ERR,
	IA_GOFO_FW_BOOT_STATE_CRIT_DOUBLE_EXCEPTION =
	IA_GOFO_DOUBLE_EXCEPTION_ERR,
	IA_GOFO_FW_BOOT_STATE_SHUTDOWN_CMD = 0x57a7f001U,
	IA_GOFO_FW_BOOT_STATE_SHUTDOWN_START = 0x57a7e200U,
	IA_GOFO_FW_BOOT_STATE_INACTIVE = 0x57a7e300U,
	IA_GOFO_FW_BOOT_HW_CMD_ACK_TIMEOUT = 0x57a7e400U
};

#endif