/****************************************************************************
 *    lib/c/Types.cpp - This file is part of coala							*
 *																			*
 *    Copyright (C) 2009  Torsten Grote										*
 *																			*
 *    This program is free software; you can redistribute it and/or modify	*
 *    it under the terms of the GNU General Public License as published by	*
 *    the Free Software Foundation; either version 3 of the License, or		*
 *    (at your option) any later version.									*
 *																			*
 *    This program is distributed in the hope that it will be useful,		*
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *    GNU General Public License for more details.							*
 *																			*
 *    You should have received a copy of the GNU General Public License		*
 *    along with this program; if not, see http://www.gnu.org/licenses		*
 ****************************************************************************/

#include "Types.h"

using namespace C;

Types::Types() {
	types_ = vector<Type*>();
	constraints_ = vector<Constraint*>();
}

Types::Types(Type* identifier) {
	types_ = vector<Type*>(1, identifier);
	constraints_ = vector<Constraint*>();
}

Types::Types(Constraint* constraint) {
	types_ = vector<Type*>();
	constraints_ = vector<Constraint*>(1, constraint);
}

Types::~Types ( ) { }

set<Variable*>* Types::getVariables() {
	set<Variable*>* vars = new set<Variable*>();

	if(!types_.empty()) {
		for(vector<Type*>::iterator n = types_.begin(); n!= types_.end(); ++n) {
			// only return variables that don't need additional type information
			if((*n)->isTrue()) {
				set<Variable*>* tmp_vars = (*n)->getVariables();
			
				if(!tmp_vars->empty()) {
					vars->insert(tmp_vars->begin(), tmp_vars->end());
				}
			}
		} // end for
	}
	
	if(!constraints_.empty()) {
		for(vector<Constraint*>::iterator n = constraints_.begin(); n!= constraints_.end(); ++n) {
			// only return variables that don't need additional type information
			if((*n)->getType() == "==") {
				set<Variable*>* tmp_vars = (*n)->getVariables();
				
				if(!tmp_vars->empty()) {
					vars->insert(tmp_vars->begin(), tmp_vars->end());
				}
				delete tmp_vars;
			}
		} // end for constraint loop
	}
	
	return vars;
}

void Types::addType(Type* identifier) {
	types_.push_back(identifier);	
}

void Types::addType(Constraint* constraint) {
	constraints_.push_back(constraint);
}

string Types::getAsString(Printer* p, set<Variable*>* variables, int line) {
	std::stringstream tmp_stream;

	Printer* tmp_p = new Printer(	&tmp_stream, p->debug, p->fake_neg, p->inc,
									p->rev, p->no_w_chk, p->no_direct_enc,
									p->lang);
	print(tmp_p, variables, line);
	delete tmp_p;

	return tmp_stream.str();
}

void Types::print(Printer* p, set<Variable*>* variables, int line) {
	// take variables from constraints into account
	for(vector<Constraint*>::iterator constraint = constraints_.begin(); constraint != constraints_.end(); ++constraint) {
		set<Variable*>* constraint_vars = (*constraint)->getVariables();
		variables->insert(constraint_vars->begin(), constraint_vars->end());
		delete constraint_vars;
	}
	
	if(!p->no_w_chk) {
		if(variables) check(variables, line);
		else {
			set<Variable*>* vars = new set<Variable*>();
			check(vars, line);
			delete vars;
		}
	}
	
	bool printed = printTypes(p, variables, &types_);
	printConstraints(p, variables, &constraints_, printed);
}

bool Types::printTypes(Printer* p, set<Variable*>* variables, vector<Type*>* vec) {
	bool printed = false;
	
	if(!vec->empty()) {
		for(vector<Type*>::iterator n = vec->begin(); n != vec->end(); ++n) {
			//set<Variable*>* type_vars = (*n)->getVariables();
			//set<Variable*> result;
			//insert_iterator< set<Variable*> > res_ins(result, result.begin());
			
			//set_intersection(type_vars->begin(), type_vars->end(), variables->begin(), variables->end(), res_ins);
			
			// only add types that have variables that are actually used and drop the rest
			//if(!result.empty()) {
				string type = (*n)->print(p);
				string restricted = "";
				int min = (*n)->getMin();
				int max = (*n)->getMax();
				
				// type grounding has been restricted
				if(min != -1 || max != -1) {
					restricted = "restricted_";
					string min_str;
					string max_str;
					stringstream ss;
					ss << min; ss >> min_str; ss.clear();
					ss << max; ss >> max_str;
					
					if(min != -1) p->addExtra(min_str + " ");
					p->addExtra("{ " + restricted + type + " : " + type + " }");
					if(max != -1) p->addExtra(" " + max_str);
					p->addExtra(".\n");
				}
				
				if(printed) p->add(", ");
				if(!(*n)->isTrue())
					p->add("not ");
				p->add(restricted + type);
				printed = true;
			//}
		} // end for
	}

	return printed;
}

void Types::printConstraints(Printer* p, set<Variable*>* variables, vector<Constraint*>* vec, bool printed) {
	if(!vec->empty()) {
		for(vector<Constraint*>::iterator n = vec->begin(); n != vec->end(); ++n) {
			//set<Variable*>* type_vars = (*n)->getVariables();
			//set<Variable*> result;
			//insert_iterator< set<Variable*> > res_ins(result, result.begin());
			
			//set_intersection(type_vars->begin(), type_vars->end(), variables->begin(), variables->end(), res_ins);

			// only add types that have variables that are actually used and drop the rest
			//if(!result.empty()) {
				if(printed) p->add(", ");
				p->add((*n)->print(p));
				printed = true;
			//}
			//delete type_vars;
		}
	}
}


void Types::check(set<Variable*>* vars, int line) {
	set<Variable*>* type_vars = getVariables();
	
	if(*vars != *type_vars) {
		string error; stringstream ss; ss << line; ss >> error;
		
		if(!type_vars->empty()) {
			set<Variable*>::iterator type = type_vars->begin();
			
			for(set<Variable*>::iterator var = vars->begin(); var != vars->end(); ++var) {
				Variable* v = *var;
				Variable* t = *type;
				bool out = false;
				
				if(v != t) {
					for(; type != type_vars->end();) {
						if(v == *type) {
							out = true;
							break;
						}
						else if(++type == type_vars->end()) {
							throw std::runtime_error("Variable "+v->getText()+" not defined in line "+error+".\n");
						}
					}
				}
				
				if(out) break;
				
				if(++type == type_vars->end()) {
					string var_name;
					if(var == vars->end()) { var_name = v->getText(); }
					else { var_name = (*++var)->getText(); }
					
					throw std::runtime_error("Variable "+var_name+" not defined in line "+error+".\n");
				}
			} // end for
		}
		else {
			throw std::runtime_error("No Variables defined at line "+error+". Hint: Types with 'not' are not defining.\n");
		}
	}
	delete type_vars;
}
