package charactermanaj.model.util;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;

import charactermanaj.model.CharacterData;
import charactermanaj.model.io.CharacterDataPersistent;
import charactermanaj.model.io.CharacterDataPersistent.DocInfo;

/**
 * 埋め込みリソースを生成するためのビルド時のヘルパ.<br>
 * @author seraphy
 */
public class MakeEmbeddedResource {

	public static void main(String[] args) throws Exception {
		// 出力するデフォルトのファイル名
		String[] fileNames = {
				"resources/schema/character.xml.ser",
				"resources/resource_list.txt"
		};
		// 引数があれば、そちらを優先
		for (int idx = 0; idx < Math.min(args.length, fileNames.length); idx++) {
			fileNames[idx] = args[idx];
		}
		
		// デフォルトのキャラクター定義の事前シリアライズバージョン作成
		createDefaultCharacterSer(fileNames[0]);
		
		// 展開対象リソースファイル名の出力
		createResourceList(fileNames[1]);
	}
	
	/**
	 * 展開対象リソースファイル名の出力
	 * @param fileName ファィル名
	 * @throws Exception 失敗
	 */
	public static void createResourceList(String fileName) throws Exception {
		File outiFile = new File(fileName);
		ArrayList<String> resourceList = new ArrayList<String>();

		resourceList.add("_HOW_TO_LOCALIZE.txt");
		
		File resourceDir = new File("resources");
		for (String dirName : new String[] {"appinfo", "languages", "menu"}) {
			File dir = new File(resourceDir, dirName);
			for (File file : dir.listFiles()) {
				if ( !file.isFile()) {
					continue;
				}
				String resourceName = dirName + "/" + file.getName();
				resourceList.add(resourceName);
			}
		}
		
		String br = "\r\n";
		FileOutputStream fos = new FileOutputStream(outiFile);
		try {
			Writer wr = new OutputStreamWriter(fos, Charset.forName("UTF-8"));
			try {
				wr.write("# localized resources" + br);
				for (String resource : resourceList) {
					wr.write(resource + br);
				}
				
			} finally {
				wr.close();
			}
		} finally {
			fos.close();
		}
		System.out.println("make embedded resource : succeeded. file=" + outiFile);
	}

	/**
	 * デフォルトのキャラクター定義の事前シリアライズバージョン作成
	 * @param fileName ファィル名
	 * @throws Exception 失敗
	 */
	public static void createDefaultCharacterSer(String fileName) throws Exception {
		File outiFile = new File(fileName);
		
		// 読み込む言語データ(ENは標準)
		Locale[] locales = new Locale[] {
				Locale.ENGLISH, // デフォルト
				Locale.JAPANESE, // 日本語固有、(現状、2つしか定義さてれいない)
		};

		// 各言語ごとのデフォルトキャラクターデータを読み込む
		HashMap<String, CharacterData> characterDataMap = new HashMap<String, CharacterData>();
		for (Locale locale : locales) {
			String lang = locale.getLanguage();
			if (characterDataMap.containsKey(lang)) {
				continue; // すでに読み込み済みなのでスキップする.
			}
			CharacterData cd = loadDefaultCharacterData(locale);
			characterDataMap.put(lang, cd);
		}

		// シリアライズする.
		FileOutputStream fos = new FileOutputStream(outiFile);
		try {
			ObjectOutputStream oos = new ObjectOutputStream(fos);
			try {
				oos.writeObject(characterDataMap);
			} finally {
				oos.close();
			}
		} finally {
			fos.close();
		}
		System.out.println("make embedded resource : succeeded. file=" + outiFile);
	}
	
	protected static CharacterData loadDefaultCharacterData(Locale locale) throws Exception {
		// 埋め込みリソースからデフォルトキャラクターデータを構築する.
		CharacterData cd;
		URL defaultCharacter = getEmbeddedResourceURL(CharacterDataPersistent.DEFAULT_CHARACTER_XML);
		InputStream is = defaultCharacter.openStream();
		try {
			DocInfo docInfo = new DocInfo();
			docInfo.setFirstElementName("character");
			docInfo.setNamespace(CharacterDataPersistent.NS);
			docInfo.setVersion("1.0");
			
			CharacterDataPersistent persist = CharacterDataPersistent.getInstance();
			cd = persist.loadCharacterDataFromXML(is, null, docInfo, locale);

		} finally {
			is.close();
		}
		return cd;
	}

	protected static URL getEmbeddedResourceURL(String schemaName) {
		return CharacterDataPersistent.class.getResource(schemaName);
	}
	
}
