/*
  This file is part of CDO. CDO is a collection of Operators to manipulate and analyse Climate model Data.

  Author: Uwe Schulzweida

*/
#ifndef REMAP_H
#define REMAP_H

#include <cstdint>
#include <cmath>

#include "varray.h"
#include "remap_vars.h"
#include "remap_grid.h"
#include "remap_grid_cell_search.h"
#include "grid_point_search.h"
#include "mpim_grid/grid_healpix.h"

#ifndef M_PI
#define M_PI 3.14159265358979323846264338327950288
#endif

constexpr double PI = M_PI;
constexpr double PI2 = (2.0 * PI);
constexpr double PIH = (0.5 * PI);
constexpr float PI_f = PI;
constexpr float PI2_f = PI2;
constexpr float PIH_f = PIH;

constexpr double TINY = 1.e-14;

#define REMAP_GRID_BASIS_SRC 1
#define REMAP_GRID_BASIS_TGT 2

struct LonLatPoint
{
  double lon = 0.0, lat = 0.0;
  LonLatPoint(){};
  LonLatPoint(double _lon, double _lat) : lon(_lon), lat(_lat){};
};

// clang-format off
struct  // GridSearchBins
#ifdef WARN_UNUSED
[[gnu::warn_unused]]
#endif
GridSearchBins
// clang-format on
{
  unsigned nbins;                // num of bins for restricted search
  size_t ncells;                 // total number of grid cells (cell_bound_box)
  Varray<size_t> bin_addr;       // min,max adds for grid cells in this lat bin
  Varray<float> bin_lats;        // min,max latitude for each search bin
  Varray<float> cell_bound_box;  // lon/lat bounding box for use
};

// clang-format off
struct  // RemapSearch
#ifdef WARN_UNUSED
[[gnu::warn_unused]]
#endif
RemapSearch
// clang-format on
{
  RemapGrid *srcGrid;
  RemapGrid *tgtGrid;

  GridSearchBins srcBins;
  GridSearchBins tgtBins;

  GridPointSearch gps;
  GridCellSearch gcs;
};

// clang-format off
struct  // RemapType
#ifdef WARN_UNUSED
[[gnu::warn_unused]]
#endif
RemapType
// clang-format on
{
  int nused = 0;
  int gridID = -1;
  size_t gridsize = 0;
  size_t numMissVals = 0;
  RemapGrid srcGrid;
  RemapGrid tgtGrid;
  RemapVars vars;
  RemapSearch search;
};

#define REMAP_WRITE_REMAP 2
#define REMAP_MAX_ITER 3
#define REMAP_NUM_SRCH_BINS 4
#define REMAP_GENWEIGHTS 5

int remap_check_mask_indices(const size_t (&indices)[4], const Varray<int8_t> &mask);

void remap_set_int(int remapvar, int value);

void remap_init_grids(RemapMethod mapType, bool doExtrapolate, int gridID1, RemapGrid &srcGrid, int gridID2, RemapGrid &tgtGrid);

void remap_grid_free(RemapGrid &grid, bool removeMask = true);
void remap_grid_alloc(RemapMethod mapType, RemapGrid &grid);
void remap_search_init(RemapMethod mapType, RemapSearch &search, RemapGrid &srcGrid, RemapGrid &tgtGrid);
void remap_search_free(RemapSearch &search);

void remap_search_points(RemapSearch &rsearch, const LonLatPoint &llpoint, knnWeightsType &knnWeights);
int remap_search_square(RemapSearch &rsearch, const LonLatPoint &llpoint, SquareCorners &squareCorners);
size_t remap_search_cells(RemapSearch &rsearch, bool isReg2dCell, GridCell &gridCell, Varray<size_t> &srchAddr);

void remap_bilinear_weights(RemapSearch &rsearch, RemapVars &rv);
void remap_bicubic_weights(RemapSearch &rsearch, RemapVars &rv);
void remap_distwgt_weights(size_t numNeighbors, RemapSearch &rsearch, RemapVars &rv);
void remap_conserv_weights(RemapSearch &rsearch, RemapVars &rv);

void remap_bilinear(RemapSearch &rsearch, const Field &field1, Field &field2);
void remap_bicubic(RemapSearch &rsearch, const Field &field1, Field &field2);
void remap_distwgt(size_t numNeighbors, RemapSearch &rsearch, const Field &field1, Field &field2);
void remap_conserv(NormOpt normOpt, RemapSearch &rsearch, const Field &field1, Field &field2);

void remap_stat(int remapOrder, RemapGrid &srcGrid, RemapGrid &tgtGrid, RemapVars &rv, const Field &field1, const Field &field2);

template <typename T>
void remap_gradients(const Varray<T> &array, RemapGrid &grid, const Varray<int8_t> &mask, RemapGradients &gradients);
void remap_gradients(const Field &field, RemapGrid &grid, RemapGradients &gradients);

void remap_write_data_scrip(const std::string &weightsfile, const RemapSwitches &remapSwitches, RemapGrid &srcGrid,
                            RemapGrid &tgtGrid, RemapVars &rv);
RemapSwitches remap_read_data_scrip(const std::string &weightsfile, int gridID1, int gridID2, RemapGrid &srcGrid,
                                    RemapGrid &tgtGrid, RemapVars &rv);

void calc_lat_bins(GridSearchBins &searchBins);
size_t get_srch_cells(size_t tgtCellIndex, GridSearchBins &tgtBins, GridSearchBins &srcBins, float *tgt_cell_bound_box,
                      Varray<size_t> &srchIndices);

int grid_search_square_reg_2d_NN(size_t nx, size_t ny, size_t *nbrIndices, double *nbrDistance, double plat, double plon,
                                 const Varray<double> &src_center_lat, const Varray<double> &src_center_lon);

int grid_search_square_reg_2d(RemapGrid *srcGrid, SquareCorners &squareCorners, double plat, double plon);

int grid_search_square_curv_2d_scrip(RemapGrid *srcGrid, SquareCorners &squareCorners, double plat, double plon,
                                     GridSearchBins &srcBins);

std::pair<double, double> remap_find_weights(const LonLatPoint &llpoint, const SquareCorners &squareCorners);

int rect_grid_search(size_t &ii, size_t &jj, double x, double y, size_t nxm, size_t nym, const Varray<double> &xm,
                     const Varray<double> &ym);

LonLatPoint remapgrid_get_lonlat(const RemapGrid *grid, size_t index);

void remap_check_area(size_t grid_size, const Varray<double> &cell_area, const char *name);

template <typename T>
void remap_set_mask(const Varray<T> &v, size_t gridsize, size_t numMissVals, double mv, Varray<int8_t> &mask);

void remap_set_mask(const Field &field1, size_t gridsize, size_t numMissVals, double missval, Varray<int8_t> &imask);

#endif /* REMAP_H */
