(* :Name: SurfaceGraphics3D *)

(* :Title: SurfaceGraphics3D *)

(* :Author: Tom Wickham-Jones*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 2.2 *)

(* :Summary:
	This package provides general three-dimensional surface plotting
	functions.
*)

(* :History:
	Created summer 1993 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)

(*:Warnings:
	This package requires the triangulation package Delaunay.m

	In order for it to work properly it must be loaded directly
	and not through the Master file.
	
*)

If[ Context[ GridSort] =!= "ExtendGraphics`SurfaceGraphics3D`",
		Remove[ GridSort]]

If[ Context[ TriangleSort] =!= "ExtendGraphics`SurfaceGraphics3D`",
		Remove[ TriangleSort]]
		
If[ Context[ ListSurfacePlot3D] =!= "ExtendGraphics`SurfaceGraphics3D`",
		Remove[ ListSurfacePlot3D]]
		

BeginPackage[ "ExtendGraphics`SurfaceGraphics3D`", 
				"ExtendGraphics`Delaunay`", 
				"Utilities`FilterOptions`"]

(* In case old version exists *)



ListSurfacePlot3D::usage = 
	"ListSurfacePlot3D[ data] generates a surface plot of data.  
	 Data should be of the form { p1, p2, ...} where each point is a 
	 triple of numbers {x, y, z} or a quadruple {x,y,z,pt}.  
	 The data does not have to regularly spaced.   If the data
	 is a quadruple the ColorFunction option is applied to
	 this last element."

SurfaceGraphics3D::usage = 
	"SurfaceGraphics3D is a graphics object that represents a surface plot." 


GridSort::usage =
	"GridSort[ data, n] takes a three-dimensional array of
	 data and sort it into n by n rectangular bins containing an
	 average of the original data.  The matrix of values is returned. 
	 Bins which remain empty are returned with Indeterminate."

TriangleSort::usage =
	"TriangleSort[ data, n] takes a three-dimensional array of
	 data and sort it into n by n rectangular bins containing an
	 average of the original data.  The data is combined with its
	 {x,y} values, empty bins dropped and the result returned as
	 a new three-dimensional set."

Begin["`Private`"]

Clear[ ListSurfacePlot3D]


SurfaceGraphics3D /:
Show[ SurfaceGraphics3D[ stuff_, opts___], nopts___] :=
	ListSurfacePlot3D[ stuff, Join[ {nopts}, {opts}]]

SurfaceGraphics3D /:
Graphics3D[ SurfaceGraphics3D[ stuff_, opts___]] :=
	Graphics3DFromSurface[ stuff, opts]
	
Format[ SurfaceGraphics3D[ ___]] := "-SurfaceGraphics3D-"


Options[ ListSurfacePlot3D] =
	Options[ ListPlot3D]

ListSurfacePlot3D[ pts_List /; TensorRank[ N[pts]] === 3, 
				   opts___ /; OptionQ[ {opts}]] :=
    Block[{tri, n, m},
    	n = Dimensions[ pts] ;
	m = Part[ n, 2] ;
	n = Part[ n, 1] ;
    	tri = Table[
		  {(i-1)*m+j, (i-1)*m+j+1, i*m+j+1, i*m+j},
		  {i,n-1}, {j,m-1}] ;
	tri = Partition[ Flatten[ tri], 4] ;
	ListSurfacePlot3D[ {Flatten[ N[pts], 1], tri}, opts]
	]

ListSurfacePlot3D[ 
			{
		   	pts_List,
		   	tri_ /; Apply[ And, Map[ VectorQ[#, IntegerQ]&, tri]]
		   	}, opts___ /; OptionQ[ {opts}]] :=
	(
	Show[ Graphics3DFromSurface[ {pts, tri}, opts]] ;
	SurfaceGraphics3D[ 
		{pts, tri}, 
		Join[ {opts}, Options[ ListSurfacePlot]]]
	)


ListSurfacePlot3D[ pts_List /; TensorRank[ pts] === 2, 
				   opts___ /; OptionQ[ {opts}]] :=
    Block[{tri},
    	tri = Delaunay[ N[ pts]] ;
		If[ tri =!= $Failed,
			ListSurfacePlot3D[ {N[ pts], Last[tri]}, opts]]
	]

Graphics3DFromSurface[ {pts_, tri_}, opts___] :=
    Block[{data, mesh, cfun, opt, polys, z0, z1},

		opt = Join[ Flatten[ {opts}], Options[ ListSurfacePlot3D]] ;
    	mesh = Mesh /. opt ;
		If[ mesh, mesh = MeshStyle /. opt] ;
		cfun = ColorFunction /. opt ;

		polys = Map[ Take[ #, 3]&, pts] ;
		polys = Map[ Polygon[ Part[ polys, #]]&, tri] ;

		If[ cfun =!= Automatic,
			opt = Prepend[ opt, Lighting -> False] ;
	    	If[ Length[ First[ pts]] === 3,
	    		data = Map[ Last, pts] ;
	    		z0 = Min[ data] ;
	    		z1 = Max[ data] ;
	    		polys = Map[ SetColor[#, z0, z1, cfun]&, polys],
		
		(* else *)
		
			data = Map[ Last, pts] ;
			data = Map[ Part[ data,#]&, tri] ;
			polys = Transpose[ {polys, data}] ;
			polys = Map[ SetColorData[#, cfun]&, polys]]] ;

		polys =
	    	Switch[ mesh,
	    		Automatic, polys,
				None,  Prepend[ polys, EdgeForm[]],
				False, Prepend[ polys, EdgeForm[]],
				_, Prepend[ polys, EdgeForm[ mesh]]] ;

		opt = If[ cfun === Automatic,
		    		opt,
					Append[opt, Lighting -> False]] ;

		opt = Sequence @@ opt ;
      	Graphics3D[ 
        	polys,
			FilterOptions[ Graphics3D, opt]]
	]


CheckColor[ GrayLevel[ g_]] := 
	GrayLevel[ FixArg[g]]

CheckColor[ RGBColor[ r_, g_, b_]] := 
	RGBColor[ FixArg[r], FixArg[g], FixArg[b]]

CheckColor[ Hue[ h_]] := 
	Hue[ h]

CheckColor[ Hue[ h_, s_, b_]] := 
	Hue[ h, FixArg[s], FixArg[b]]

CheckColor[ CMYKColor[ c_, m_, y_, k_]] := 
	CMYKColor[ FixArg[c], FixArg[m], FixArg[y], FixArg[ k]]

CheckColor[ _] :=
	GrayLevel[ 0]

FixArg[ val_] :=
	Which[ val < 0, 0,
	       val > 1, 1,
	       True, val]

SetColor[ Polygon[ poly_], z0_, z1_, cfun_] :=
    Block[{vals, heads, col},
		vals = Map[ ((Last[#]-z0)/(z1-z0))&, poly] ;
		vals = Map[ CheckColor[ cfun[#]]&, vals] ;
		heads = Map[ Head, vals] ;
		vals = Map[ Apply[ List, #]&, vals] ;
		col = If[ Apply[ SameQ, heads],
					vals = Apply[ Plus, vals]/Length[ vals] ;
					Apply[ First[ heads], vals],
					GrayLevel[ 0]] ;
		{FaceForm[ col], Polygon[ poly]}
	]
    	
SetColorData[ {Polygon[ poly_], data_}, cfun_] :=
    Block[{vals, heads, col},
	vals = Map[ cfun, data] ;
	vals = Map[ CheckColor, vals] ;
	heads = Map[ Head, vals] ;
	vals = Map[ Apply[ List, #]&, vals] ;
	col = If[ Apply[ SameQ, heads],
		vals = Apply[ Plus, vals]/Length[ vals] ;
		Apply[ First[ heads], vals],
		GrayLevel[ 0]] ;
	{FaceForm[ col], Polygon[ poly]}
	]

GridSort[ data_ /; 
      MatrixQ[ data] &&
      Length[ First[ data]] === 3, n_] :=
    Block[{tmp, x0, x1, y0, y1, xi, yi, d, res},
		{x0, x1} = Map[ #[Map[ First, data]]&, {Min, Max}] ;
        tmp = Map[ Part[#,2]&, data] ;
		{y0, y1} = Map[ #[tmp]&, {Min, Max}] ;
        res = Table[ {0,0}, {n}, {n}] ;
        Do[
            d = Part[ data, i] ;
            xi = Round[(n-1)( Part[d,1]-x0)/(x1-x0)]+1;
            yi = Round[(n-1)( Part[d,2]-y0)/(y1-y0)]+1;
            res[[yi,xi,1]] += Part[d,3];
            res[[yi,xi,2]]++,
            {i,Length[ data]}] ;
        res = res /. {0,0} -> Indeterminate ;
        Map[ Apply[ Divide, #]&, res, {2}]
        ]


TriangleSort[ data_ /; 
    MatrixQ[ data] &&
    Length[ First[ data]] === 3, n_] :=
        Block[{tmp, x0, x1, y0, y1, xi, yi, d, res, grid},
			{x0, x1} = Map[ #[Map[ First, data]]&, {Min, Max}] ;
        	tmp = Map[ Part[#,2]&, data] ;
			{y0, y1} = Map[ #[tmp]&, {Min, Max}] ;
			xi = (x1-x0)/n ;
			yi = (y1-y0)/n ;
			grid = Table[ {x,y},
						{x,x0+xi/2,x1,xi},
						{y,y0+yi/2,y1,yi}] ;
			grid = Flatten[ grid, 1] ;
            res = Table[ {0,0}, {n}, {n}] ;
            Do[
                d = Part[ data, i] ;
                xi = Round[(n-1)( Part[d,1]-x0)/(x1-x0)]+1;
                yi = Round[(n-1)( Part[d,2]-y0)/(y1-y0)]+1;
                res[[xi,yi,1]] += Part[d,3];
                res[[xi,yi,2]]++,
                {i,Length[ data]}] ;
			res = Flatten[ res, 1] ;
            res = res /. {0,0} -> Indeterminate ;
            res = Map[ Apply[ Divide, #]&, res] ;
			res = Apply[ Append, Transpose[ {grid, res}], {1}] ;
			DeleteCases[ res, {_,_,Indeterminate}]
        ]

  	
End[]

EndPackage[]

(*

<<ExtendGraphics/SurfaceGraphics3D.m

d = Table[ { x = 4 Random[]-2, y = 4 Random[]-2, Exp[ -Sqrt[x^2+y^2]]},
	 {500}];

d = 
 Flatten[ Table[ {x, y, Exp[ -Sqrt[x^2+y^2]]}, {x, -2,2,.2},{y,-2,2,.2}], 1];


ListSurfacePlot3D[ d]

GridSort[ d, 20]

ListPlot3D[ %]

TriangleSort[ d,20]

ListSurfacePlot3D[ %]


Show[ %, 
	Mesh -> {Thickness[0.0001], GrayLevel[ 0.75]}]

Show[ %, 
	ColorFunction -> Hue]

ListSurfacePlot3D[ d, 
	Mesh -> {Thickness[0.0001], GrayLevel[ 0.75]}]

ListSurfacePlot3D[ d, 
	ColorFunction -> GrayLevel]

d = Table[ {x, y, x y/2.}, {x,2}, {y,3}];

d = Table[ {x, y, Sin[ N[ x y]]}, {x,0,2Pi,.2}, {y,0,Pi,.2}];

ListSurfacePlot3D[ d]


d = Table[ {x, y, x y/2., {x/5, y/5}}, {x,0,5}, {y,0,5}] ;

d = Table[ {x, y, Sin[x y], {x/(2Pi), y/Pi}}, {x,0,2Pi,.25}, {y,0,Pi,.25}] ;

ListSurfacePlot3D[ d]

Show[%, 
	ColorFunction -> (RGBColor[ #[[1]], #[[2]],0]&)]




*)



