# search, a bzr plugin for searching within bzr branches/repositories.
# Copyright (C) 2008 Robert Collins
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

"""Tests for the inventory specific logic."""

from bzrlib.plugins.search import inventory
from bzrlib.tests import TestCaseWithTransport


class TestPathFromInventory(TestCaseWithTransport):

    def get_inventory(self, tree, revid):
        tree.lock_read()
        try:
            inventories = tree.branch.repository.inventories
            return inventories.get_record_stream([(revid,)], 'unordered',
                True).next().get_bytes_as('fulltext')
        finally:
            tree.unlock()

    def test_paths_from_ids_basic(self):
        tree = self.make_branch_and_tree('foo', format="1.14")
        t = self.get_transport('foo')
        t.mkdir('subdir')
        t.mkdir('subdir/nested-dir')
        t.put_bytes('subdir/nested-dir/the file', 'content')
        # The ids are in reverse sort order to try to exercise corner cases in
        # xml processing.
        tree.add(['subdir', 'subdir/nested-dir', 'subdir/nested-dir/the file'],
            ['3dir', '2dir', '1file'])
        revid = tree.commit('first post')
        xml = self.get_inventory(tree, revid)
        serializer = tree.branch.repository._serializer
        # All individually:
        self.assertEqual({'1file':'subdir/nested-dir/the file'},
            inventory.paths_from_ids(xml, serializer, ['1file']))
        self.assertEqual({'2dir':'subdir/nested-dir'},
            inventory.paths_from_ids(xml, serializer, ['2dir']))
        self.assertEqual({'3dir':'subdir'},
            inventory.paths_from_ids(xml, serializer, ['3dir']))
        # In twos:
        self.assertEqual({'1file':'subdir/nested-dir/the file',
            '2dir':'subdir/nested-dir'},
            inventory.paths_from_ids(xml, serializer, ['1file', '2dir']))
        self.assertEqual({'1file':'subdir/nested-dir/the file',
            '3dir':'subdir'},
            inventory.paths_from_ids(xml, serializer, ['1file', '3dir']))
        self.assertEqual({'2dir':'subdir/nested-dir',
            '3dir':'subdir'},
            inventory.paths_from_ids(xml, serializer, ['3dir', '2dir']))
        # All together
        self.assertEqual({'1file':'subdir/nested-dir/the file',
            '2dir':'subdir/nested-dir',
            '3dir':'subdir'},
            inventory.paths_from_ids(xml, serializer, ['1file', '2dir', '3dir']))

    def test_paths_from_ids_rich_root(self):
        tree = self.make_branch_and_tree('foo', format="rich-root-pack")
        tree.set_root_id('a-root')
        t = self.get_transport('foo')
        t.mkdir('subdir')
        tree.add(['subdir'], ['3dir'])
        revid = tree.commit('first post')
        xml = self.get_inventory(tree, revid)
        serializer = tree.branch.repository._serializer
        # All individually:
        self.assertEqual({'3dir':'subdir'},
            inventory.paths_from_ids(xml, serializer, ['3dir']))
        self.assertEqual({'a-root':''},
            inventory.paths_from_ids(xml, serializer, ['a-root']))
        # In twos:
        self.assertEqual({'3dir':'subdir',
            'a-root':''},
            inventory.paths_from_ids(xml, serializer, ['3dir', 'a-root']))

    def test_format_5_unique_root(self):
        from bzrlib.xml5 import Serializer_v5
        serializer = Serializer_v5()
        xml = [
            '<inventory file_id="root" format="5" revision_id="rev1">\n',
            '<file file_id="__init__" name="__init__.py" parent_id="root" revision="rev1" text_sha1="f2bcc80a5f26cba9ce168ddfe038562c658e2d37" text_size="638" />\n',
            '<directory file_id="tests" name="tests" parent_id="root" revision="rev1" />\n',
            '<file file_id="test.py" name="test.py" parent_id="tests" revision="rev1" text_sha1="a76a16cfe684b4f34374a110cb7bd4482de533d1" text_size="697" />\n',
            '</inventory>\n'
            ]
        # All individually is enough to test the lookup of the root id/path:
        self.assertEqual({'__init__':'__init__.py'},
            inventory.paths_from_ids(xml, serializer, ['__init__']))
        self.assertEqual({'tests':'tests'},
            inventory.paths_from_ids(xml, serializer, ['tests']))
        self.assertEqual({'test.py':'tests/test.py'},
            inventory.paths_from_ids(xml, serializer, ['test.py']))

    def test_escaped_chars(self):
        """Inventories with escaping attributes (&'"<>) are matched ok."""
        from bzrlib.xml5 import Serializer_v5
        serializer = Serializer_v5()
        xml = [
            '<inventory file_id="root" format="5" revision_id="rev1">\n',
            '<file file_id="&amp;&apos;&quot;&lt;&gt;" name="__init__.py" parent_id="root" revision="rev1" />\n',
            '<directory file_id="tests" name="&amp;&apos;&quot;&lt;&gt;" parent_id="root" revision="rev1" />\n',
            '<file file_id="test.py" name="&gt;&lt;&quot;&apos;&amp;" parent_id="tests" revision="rev1"  />\n',
            '</inventory>\n'
            ]
        # Lookup an id that has every escape
        self.assertEqual({'&\'"<>':'__init__.py'},
            inventory.paths_from_ids(xml, serializer, ['&\'"<>']))
        # Get the path for a name which is escaped
        self.assertEqual({'test.py':'&\'"<>/><"\'&'},
            inventory.paths_from_ids(xml, serializer, ['test.py']))
