package org.bouncycastle.pqc.jcajce.provider.test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.bc.BCObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.pqc.asn1.PQCObjectIdentifiers;
import org.bouncycastle.pqc.jcajce.spec.QTESLAParameterSpec;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.test.FixedSecureRandom;

public class QTESLATest
    extends KeyPairGeneratorTest
{
    static SecureRandom secureRandom = new SecureRandom();

    private void doTestSig(KeyPair kp)
        throws Exception
    {
        byte[] seed = Hex.decode("061550234D158C5EC95595FE04EF7A25767F2E24CC2BC479D09D86DC9ABCFDE7056A8C266F9EF97ED08541DBD2E1FFA1");
        byte[] msg = Hex.decode("D81C4D8D734FCBFBEADE3D3F8A039FAA2A2C9957E835AD55B22E75BF57BB556AC8");

        Signature signer = Signature.getInstance(kp.getPublic().getAlgorithm(), "BCPQC");

        signer.initSign(kp.getPrivate(), new FixedSecureRandom(seed));

        signer.update(msg);

        byte[] sig = signer.sign();

        signer = Signature.getInstance("qTESLA", "BCPQC");

        signer.initVerify(kp.getPublic());

        signer.update(msg);

        assertTrue(signer.verify(sig));
    }

    private void doTestKey(KeyPair kp)
        throws Exception
    {
        KeyFactory keyFact = KeyFactory.getInstance("qTESLA", "BCPQC");

        PublicKey pubKey = keyFact.generatePublic(new X509EncodedKeySpec(kp.getPublic().getEncoded()));
        PrivateKey privKey = keyFact.generatePrivate(new PKCS8EncodedKeySpec(kp.getPrivate().getEncoded()));

        assertEquals(kp.getPublic(), pubKey);
        assertEquals(kp.getPrivate(), privKey);
        assertEquals(kp.getPublic().hashCode(), pubKey.hashCode());
        assertEquals(kp.getPrivate().hashCode(), privKey.hashCode());
        assertEquals(kp.getPublic(), serialiseDeserialise(kp.getPublic()));
        assertEquals(kp.getPrivate(), serialiseDeserialise(kp.getPrivate()));
    }

    private Object serialiseDeserialise(Object o)
        throws Exception
    {
        ByteArrayOutputStream bOut = new ByteArrayOutputStream();
        ObjectOutputStream oOut = new ObjectOutputStream(bOut);

        oOut.writeObject(o);

        oOut.close();

        ObjectInputStream oIn = new ObjectInputStream(new ByteArrayInputStream(bOut.toByteArray()));

        return oIn.readObject();
    }


    public void testGenerateKeyPairSigningVerifyingPI()
        throws Exception
    {
        KeyPairGenerator kpGen = KeyPairGenerator.getInstance("qTESLA", "BCPQC");

        kpGen.initialize(new QTESLAParameterSpec(QTESLAParameterSpec.PROVABLY_SECURE_I), secureRandom);

        KeyPair kp = kpGen.generateKeyPair();

        kf = KeyFactory.getInstance("qTESLA");

        performKeyPairEncodingTest(kp);

        assertEquals(BCObjectIdentifiers.qTESLA_p_I, SubjectPublicKeyInfo.getInstance(kp.getPublic().getEncoded()).getAlgorithm().getAlgorithm());

        doTestSig(kp);
        doTestKey(kp);
    }

    public void testGenerateKeyPairSigningVerifyingPIII()
        throws Exception
    {

        KeyPairGenerator kpGen = KeyPairGenerator.getInstance("qTESLA", "BCPQC");

        kpGen.initialize(new QTESLAParameterSpec(QTESLAParameterSpec.PROVABLY_SECURE_III), secureRandom);

        KeyPair kp = kpGen.generateKeyPair();

        kf = KeyFactory.getInstance("qTESLA");

        performKeyPairEncodingTest(kp);

        assertEquals(BCObjectIdentifiers.qTESLA_p_III, SubjectPublicKeyInfo.getInstance(kp.getPublic().getEncoded()).getAlgorithm().getAlgorithm());

        doTestSig(kp);
        doTestKey(kp);
    }

    private void doTestKAT(ASN1ObjectIdentifier alg, byte[] pubKey, byte[] privKey, byte[] seed, byte[] msg, byte[] expected)
        throws Exception
    {
        KeyFactory keyFact = KeyFactory.getInstance("qTESLA", "BCPQC");

        PublicKey qPub = keyFact.generatePublic(new X509EncodedKeySpec(new SubjectPublicKeyInfo(new AlgorithmIdentifier(alg), pubKey).getEncoded()));
        PrivateKey qPriv = keyFact.generatePrivate(new PKCS8EncodedKeySpec(new PrivateKeyInfo(new AlgorithmIdentifier(alg), new DEROctetString(privKey)).getEncoded()));

        Signature signer = Signature.getInstance("qTESLA", "BCPQC");

        signer.initSign(qPriv, QTESLASecureRandomFactory.getFixed(seed, 256));

        signer.update(msg);

        byte[] sig = signer.sign();

        assertTrue(Arrays.areEqual(expected, Arrays.concatenate(sig, msg)));

        signer = Signature.getInstance("qTESLA", "BCPQC");

        signer.initVerify(qPub);

        signer.update(msg);

        assertTrue(signer.verify(sig));
    }


    /*
    # qTesla-p-I
    */
    public void testCatPIVector0()
        throws Exception
    {
        byte[] seed = Hex
            .decode("061550234D158C5EC95595FE04EF7A25767F2E24CC2BC479D09D86DC9ABCFDE7056A8C266F9EF97ED08541DBD2E1FFA1");
//        int mlen = 33;
        byte[] msg = Hex.decode("D81C4D8D734FCBFBEADE3D3F8A039FAA2A2C9957E835AD55B22E75BF57BB556AC8");
        byte[] pk = Hex.decode(
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
        byte[] sk = Hex.decode(
            "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");
//        int smlen = 2625;
        byte[] sm = Hex.decode(
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

        doTestKAT(PQCObjectIdentifiers.qTESLA_p_I, pk, sk, seed, msg, sm);
    }

    /*
    # qTesla-p-III
    */
    public void testCatPIIIVector0()
        throws Exception
    {
        byte[] seed = Hex
            .decode("061550234D158C5EC95595FE04EF7A25767F2E24CC2BC479D09D86DC9ABCFDE7056A8C266F9EF97ED08541DBD2E1FFA1");
//        int mlen = 33;
        byte[] msg = Hex.decode("D81C4D8D734FCBFBEADE3D3F8A039FAA2A2C9957E835AD55B22E75BF57BB556AC8");
        byte[] pk = Arrays.concatenate(
            Hex.decode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
            Hex.decode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
        byte[] sk = Hex.decode(
            "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");
//        int smlen = 5697;
        byte[] sm = Hex.decode(
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

        doTestKAT(PQCObjectIdentifiers.qTESLA_p_III, pk, sk, seed, msg, sm);
    }
}
