/*################################################################################
# Linux Management Providers (LMP), Simple Identity Management provider package
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr, desmons_frederic@yahoo.fr>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by 2009 Khahramon NURIDDINOV, TUIT <qahramon0786@gmail.com>
# Modified by 2009 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr>
################################################################################*/

#include "OpenDRIM_AccountCapabilitiesAccess.h"
#include "OpenDRIM_AccountAccess.h"

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_unload(string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_AccountCapabilities>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	// Get the default minimum length for an account password
	vector<string> etc_login_defs_entries, etc_login_defs_values;
	etc_login_defs_entries.push_back("PASS_MIN_LEN");
	
	CF_assert(Simple_Identity_Management_OpenDRIM_Account_extractLoginDefsValue(etc_login_defs_values, etc_login_defs_entries, errorMessage));
	
	vector<unsigned short> ComplexPasswordRulesSupported;
	ComplexPasswordRulesSupported.push_back(2);
	// Read the /etc/passwd file
	string etc_passwd;
	
	CF_assert(CF_readTextFile("/etc/passwd", etc_passwd, errorMessage));
	
	vector<string> etc_passwd_lines;
	CF_splitText(etc_passwd_lines, etc_passwd, '\n');
	// Read the /etc/shadow file
	string etc_shadow;
	vector<string> etc_shadow_lines;
	if (discriminant=="ei" || discriminant=="gi")
	{
		CF_assert(CF_readTextFile("/etc/shadow", etc_shadow, errorMessage));
		
		CF_splitText(etc_shadow_lines, etc_shadow, '\n');
		if (etc_shadow_lines.size()!=etc_passwd_lines.size()) {
			errorMessage="Wrong format (missmatch with /etc/passwd): /etc/shadow";
			return FAILED;
		}
	}
	// In gi (getInstance) mode the first instance in the vector contains
	// the objectpath of the instance we're supposed to get
	string gi_instance_id;
	if (discriminant=="gi")
		result[0].getInstanceID(gi_instance_id);
	// For each line of /etc/passwd and /etc/shadow
	for (size_t i=0; i<etc_passwd_lines.size(); i++) {
		// Check the format of the line in /etc/passwd
		if (etc_passwd_lines[i]=="")
			break;
		vector<string> current_etc_passwd_line;
		CF_splitText(current_etc_passwd_line, etc_passwd_lines[i], ':');
		if (current_etc_passwd_line.size()!=7) {
			errorMessage="Wrong format (at line "+CF_intToStr((unsigned long) i+1)+"): /etc/passwd";
			return FAILED;
		}
		// Assign the key values
		OpenDRIM_AccountCapabilities instance;
		instance.setInstanceID(current_etc_passwd_line[0]);
		// If we need the content of the instance(s)
		// We check if the current line matches with the requested instance (gi mode)
		if (discriminant=="ei" || (discriminant=="gi" && CF_strCmpNoCase(gi_instance_id, current_etc_passwd_line[0]))) {
			// Check the format of the line of /etc/shadow
			vector<string> current_etc_shadow_line;
			CF_splitText(current_etc_shadow_line, etc_shadow_lines[i], ':');
			if (current_etc_shadow_line.size()!=9) {
				errorMessage="Wrong format (at line "+CF_intToStr((unsigned long) i+1)+"): /etc/shadow";
				return FAILED;
			}
			// If we're in gi mode, fill in the first instance of the vector
			if (discriminant=="gi") {
				if (etc_login_defs_values[0]!="")
					result[0].setComplexPasswordRulesSupported(ComplexPasswordRulesSupported);
				
				CF_assert(Simple_Identity_Management_OpenDRIM_AccountCapabilities_extractAccountCapabilitiesInfo(result[0], current_etc_passwd_line, current_etc_shadow_line, errorMessage));
				
				// We found the instance so we can exit here
				return OK;
			}
			// Else we fill the current instance
			if (etc_login_defs_values[0]!="")
				instance.setComplexPasswordRulesSupported(ComplexPasswordRulesSupported);
				
			CF_assert(Simple_Identity_Management_OpenDRIM_AccountCapabilities_extractAccountCapabilitiesInfo(instance, current_etc_passwd_line, current_etc_shadow_line, errorMessage));
		}
		// Add the instance to the vector
		if (discriminant=="ei" || discriminant=="ein")
			result.push_back(instance);
	}
	// If we get here it means we didn't find the instance
	if (discriminant=="gi")
	{
		errorMessage = "No instance";
		return NOT_FOUND;
	}
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_AccountCapabilities& instance, const char** properties, string& errorMessage) {
	_E_;
	vector<OpenDRIM_AccountCapabilities> instances;
	instances.push_back(instance);
	
	// Get the instance (it also checks if it exists)
	CF_assert(Simple_Identity_Management_OpenDRIM_AccountCapabilities_retrieve(broker, ctx, instances, properties, errorMessage, "gi"));
	
	instance=instances[0];
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_AccountCapabilities& newInstance, const OpenDRIM_AccountCapabilities& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	// We are going to modify the instance by executing the 'passwd' command
	string InstanceID;
	oldInstance.getInstanceID(InstanceID);
	string command="passwd ";
	string newMaximumPasswordAge, oldMaximumPasswordAge;
	// Check which properties have been modified and form passwd options
	if (newInstance.getMaximumPasswordAge(newMaximumPasswordAge)!=NOT_FOUND) {
		if (oldInstance.getMaximumPasswordAge(oldMaximumPasswordAge)==NOT_FOUND || newMaximumPasswordAge!=oldMaximumPasswordAge) {
			if (newMaximumPasswordAge.substr(21,4)!=":000") {
				errorMessage="Maximum password age MUST be an interval.";
				return FAILED;
			}
			newMaximumPasswordAge=newMaximumPasswordAge.substr(0,8);
			while (newMaximumPasswordAge[0]=='0')
				newMaximumPasswordAge.erase(0,1);
			command+="-x "+newMaximumPasswordAge+" ";
		}
	}
	string newDisableTimeoutAfterPasswordExpiration, oldDisableTimeoutAfterPasswordExpiration;
	if (newInstance.getDisableTimeoutAfterPasswordExpiration(newDisableTimeoutAfterPasswordExpiration)!=NOT_FOUND) {
		if (oldInstance.getDisableTimeoutAfterPasswordExpiration(oldDisableTimeoutAfterPasswordExpiration)==NOT_FOUND || newDisableTimeoutAfterPasswordExpiration!=oldDisableTimeoutAfterPasswordExpiration) {
			if (newDisableTimeoutAfterPasswordExpiration.substr(21,4)!=":000") {
				errorMessage="Password disable timeout age MUST be an interval.";
				return FAILED;
			}
			newDisableTimeoutAfterPasswordExpiration=newDisableTimeoutAfterPasswordExpiration.substr(0,8);
			while (newDisableTimeoutAfterPasswordExpiration[0]=='0')
				newDisableTimeoutAfterPasswordExpiration.erase(0,1);
			command+="-i "+newDisableTimeoutAfterPasswordExpiration+" ";
		}
	}
	// Execute passwd
	if (command!="passwd ")
	{
		string stdOut, stdErr;
		CF_assert(CF_runCommand(command+InstanceID, stdOut, stdErr, errorMessage));
	}
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_AccountCapabilities& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_AccountCapabilities& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_CreateGoalSettings(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_AccountCapabilities& instance, unsigned short& returnValue, const OpenDRIM_AccountCapabilities_CreateGoalSettings_In& in, OpenDRIM_AccountCapabilities_CreateGoalSettings_Out& out, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_extractAccountCapabilitiesInfo(OpenDRIM_AccountCapabilities& instance, const vector<string>& current_etc_passwd_line, const vector<string>& current_etc_shadow_line, string& errorMessage) {
	_E_;
	// Assign account capabilities information
	instance.setDisableOnInactivity(false);
	string MaximumPasswordAge=current_etc_shadow_line[4];
	if (MaximumPasswordAge.size()<=8 && MaximumPasswordAge!="99999" && MaximumPasswordAge!="") {
		unsigned long initial_size=MaximumPasswordAge.size();
		for (size_t i=0; i<8-initial_size; i++)
			MaximumPasswordAge="0"+MaximumPasswordAge;
		MaximumPasswordAge+="000000.000000:000";
		instance.setMaximumPasswordAge(MaximumPasswordAge);
	}
	if (current_etc_shadow_line[6]!="") {
		string DisableTimeoutAfterPasswordExpiration=current_etc_shadow_line[6];
		unsigned long initial_size=DisableTimeoutAfterPasswordExpiration.size();
		for (size_t i=0; i<8-initial_size; i++)
			DisableTimeoutAfterPasswordExpiration="0"+DisableTimeoutAfterPasswordExpiration;
		DisableTimeoutAfterPasswordExpiration+="000000.000000:000";
		instance.setDisableTimeoutAfterPasswordExpiration(DisableTimeoutAfterPasswordExpiration);
		instance.setDisableOnInactivity(true);
	}
	// set ElementName
	instance.setElementName(instance.InstanceID+" account capabilities");
	_L_;
	return OK;
}

int Simple_Identity_Management_OpenDRIM_AccountCapabilities_populate(OpenDRIM_AccountCapabilities& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

