/*
  Simple DirectMedia Layer
  Copyright (C) 1997-2023 Sam Lantinga <slouken@libsdl.org>

  This software is provided 'as-is', without any express or implied
  warranty.  In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

  1. The origin of this software must not be misrepresented; you must not
     claim that you wrote the original software. If you use this software
     in a product, an acknowledgment in the product documentation would be
     appreciated but is not required.
  2. Altered source versions must be plainly marked as such, and must not be
     misrepresented as being the original software.
  3. This notice may not be removed or altered from any source distribution.
*/
#include "../../SDL_internal.h"

#if SDL_VIDEO_RENDER_D3D12 && !SDL_RENDER_DISABLED && !defined(__XBOXONE__) && !defined(__XBOXSERIES__)

#include "SDL_stdinc.h"

#include "../../core/windows/SDL_windows.h"
#include <d3d12.h>

#include "SDL_shaders_d3d12.h"

#define SDL_COMPOSE_ERROR(str) SDL_STRINGIFY_ARG(__FUNCTION__) ", " str

/* Direct3D 12 shaders

   SDL's shaders are compiled into SDL itself, to simplify distribution.

   All Direct3D 12 shaders were compiled with the following:

   dxc -E <FUNC> -T <TYPE> -Fo <OUTPUT FILE> <INPUT FILE>

     Variables:
     - <FUNC>: the function. This is main for the pixel shader, and one of
       four options for the vertex shader
     - <TYPE>: the type of shader.  A table of utilized shader types is
       listed below.
     - <OUTPUT FILE>: where to store compiled output
     - <INPUT FILE>: where to read shader source code from

     Shader types:
     - ps_6_0: Pixel shader
     - vs_6_0: Vertex shader


   Shader object code was converted to unsigned chars via the following
   *nix style command (available separately from Windows + MSVC):

     xxd --include <FILE>
  */

/* The color-only-rendering pixel shader:

   --- D3D12_PixelShader_Colors.hlsl ---
   struct PixelShaderInput
   {
       float4 pos : SV_POSITION;
       float2 tex : TEXCOORD0;
       float4 color : COLOR0;
   };

   #define ColorRS \
       "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
       "DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
       "DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
       "DENY_HULL_SHADER_ROOT_ACCESS )," \
       "RootConstants(num32BitConstants=32, b0)"

   [RootSignature(ColorRS)]
   float4 main(PixelShaderInput input) : SV_TARGET0
   {
       return input.color;
   }
*/

static unsigned char D3D12_PixelShader_Colors[] = {
    0x44, 0x58, 0x42, 0x43, 0x38, 0xfe, 0xa0, 0xc7, 0x8e, 0xb3, 0x78, 0x02,
    0x7e, 0x71, 0xa9, 0x1c, 0x23, 0x3e, 0x23, 0xa2, 0x01, 0x00, 0x00, 0x00,
    0xe5, 0x0c, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0xd9, 0x01, 0x00, 0x00, 0x11, 0x02, 0x00, 0x00, 0xa9, 0x07, 0x00, 0x00,
    0xc5, 0x07, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0xbc, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x54, 0x45,
    0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44,
    0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x0a, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03, 0x02, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44,
    0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x08, 0x00, 0x00, 0x00, 0x52, 0x54, 0x53, 0x30, 0x30, 0x00, 0x00,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
    0x00, 0x53, 0x54, 0x41, 0x54, 0x90, 0x05, 0x00, 0x00, 0x60, 0x00, 0x00,
    0x00, 0x64, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x78, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0,
    0xde, 0x21, 0x0c, 0x00, 0x00, 0x5b, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23,
    0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84,
    0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45,
    0x02, 0x42, 0x92, 0x0b, 0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18,
    0x4b, 0x0a, 0x32, 0x42, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88,
    0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22, 0xc4,
    0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46,
    0x06, 0x51, 0x18, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0,
    0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff,
    0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00,
    0x00, 0x89, 0x20, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08,
    0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x22, 0xa4, 0x84, 0x04, 0x13, 0x22,
    0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84, 0x84,
    0x4c, 0x10, 0x3c, 0x23, 0x00, 0x25, 0x00, 0x8a, 0x19, 0x80, 0x39, 0x02,
    0x30, 0x98, 0x23, 0x40, 0x8a, 0x31, 0x44, 0x54, 0x44, 0x56, 0x0c, 0x20,
    0xa2, 0x1a, 0xc2, 0x81, 0x80, 0x61, 0x04, 0x62, 0x18, 0x46, 0x10, 0x86,
    0xbb, 0xa4, 0x29, 0xa2, 0x84, 0xc9, 0x4f, 0x91, 0x8b, 0x58, 0xd8, 0x03,
    0x18, 0x88, 0x48, 0x6c, 0x1e, 0x6a, 0x42, 0xc3, 0xf2, 0xa6, 0x03, 0x01,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10,
    0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x02,
    0x01, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0xa2, 0x12, 0x18, 0x01,
    0x28, 0x86, 0x02, 0x2b, 0x83, 0x42, 0x28, 0x87, 0x92, 0x28, 0x90, 0x02,
    0x2a, 0x82, 0xf2, 0xa0, 0x2a, 0x89, 0x32, 0x28, 0x84, 0x11, 0x80, 0x22,
    0x28, 0x10, 0xea, 0x19, 0x00, 0xf2, 0xb1, 0x1c, 0x86, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x79, 0x18, 0x00,
    0x00, 0x85, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
    0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b,
    0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b,
    0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9,
    0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13,
    0x04, 0x62, 0x98, 0x20, 0x10, 0xc4, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04,
    0x41, 0x01, 0x6e, 0x6e, 0x82, 0x40, 0x14, 0x1b, 0x86, 0x03, 0x21, 0x26,
    0x08, 0x4d, 0x35, 0x41, 0x20, 0x0c, 0x0e, 0x70, 0x6f, 0x73, 0x13, 0x04,
    0xe2, 0x98, 0x20, 0x10, 0x08, 0x97, 0x29, 0xab, 0x2f, 0xa8, 0xa7, 0xa9,
    0x24, 0xaa, 0xa4, 0x27, 0xa7, 0x09, 0x02, 0x91, 0x4c, 0x10, 0x08, 0x65,
    0x03, 0x82, 0x30, 0x0d, 0xe1, 0x3c, 0x50, 0xc4, 0x81, 0xae, 0x0c, 0x6f,
    0x82, 0x40, 0x2c, 0x4c, 0xa8, 0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88,
    0x60, 0x36, 0x20, 0xc8, 0xd4, 0x50, 0x4e, 0x05, 0x45, 0x2c, 0xc6, 0xde,
    0xd8, 0xde, 0xe4, 0x26, 0x08, 0x04, 0x43, 0x63, 0xe8, 0x89, 0xe9, 0x49,
    0x0a, 0x66, 0x03, 0x82, 0x5c, 0x0d, 0xe6, 0x64, 0x50, 0xb4, 0x81, 0x58,
    0x24, 0x4b, 0xdb, 0x30, 0x10, 0xca, 0x36, 0x41, 0x10, 0x80, 0x0d, 0xc0,
    0x86, 0x81, 0xf0, 0xbc, 0x0d, 0xc1, 0xb7, 0x61, 0x18, 0x3a, 0x30, 0x98,
    0x20, 0x38, 0xd6, 0x86, 0x40, 0x0c, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x71,
    0x99, 0xb2, 0xfa, 0x82, 0x7a, 0x9b, 0x4b, 0xa3, 0x4b, 0x7b, 0x73, 0x9b,
    0x20, 0x14, 0xcf, 0x04, 0xa1, 0x80, 0x36, 0x04, 0xc4, 0x04, 0xa1, 0x88,
    0x26, 0x08, 0x85, 0xb4, 0x61, 0x21, 0xca, 0xc0, 0x0c, 0xce, 0x00, 0x0d,
    0xd2, 0x60, 0x48, 0x03, 0x42, 0x0d, 0x00, 0x22, 0x54, 0x45, 0x58, 0x43,
    0x4f, 0x4f, 0x52, 0x44, 0x13, 0x84, 0x62, 0xda, 0xb0, 0x0c, 0x6c, 0x60,
    0x06, 0x6a, 0x80, 0x06, 0x6d, 0x30, 0xb4, 0xc1, 0xa0, 0x06, 0xc0, 0x04,
    0x81, 0x68, 0x58, 0x0c, 0x3d, 0x31, 0x3d, 0x49, 0x4d, 0x10, 0x08, 0x67,
    0x83, 0xd0, 0xc4, 0xc1, 0x86, 0xe5, 0x0d, 0xe0, 0xc0, 0x0c, 0xd4, 0x00,
    0x0d, 0xda, 0x60, 0x48, 0x83, 0x37, 0x50, 0x03, 0x39, 0xd8, 0x30, 0xac,
    0x81, 0x1b, 0xcc, 0x01, 0x93, 0x29, 0xab, 0x2f, 0xaa, 0x30, 0xb9, 0xb3,
    0x32, 0xba, 0x09, 0x42, 0x41, 0x6d, 0x58, 0x88, 0x3a, 0x30, 0x03, 0x3b,
    0x40, 0x03, 0x35, 0x18, 0xd2, 0x80, 0x50, 0x03, 0x39, 0xd8, 0x10, 0xdc,
    0xc1, 0x86, 0x81, 0x0e, 0xf0, 0x00, 0xd8, 0x50, 0x74, 0x64, 0x90, 0x07,
    0x00, 0xc0, 0x22, 0xcd, 0x6d, 0x8e, 0x6e, 0x8e, 0xc6, 0x5c, 0xda, 0xd9,
    0x17, 0x1b, 0x19, 0x8d, 0xb9, 0xb4, 0xb3, 0xaf, 0x39, 0xba, 0x0d, 0xc6,
    0x1e, 0x44, 0x7c, 0xe0, 0xf4, 0x81, 0x53, 0x85, 0x8d, 0xcd, 0xae, 0xcd,
    0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73,
    0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8,
    0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x45, 0x1d,
    0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9, 0xa6, 0x37, 0xb2,
    0x32, 0xb6, 0x29, 0x01, 0x52, 0x89, 0x0c, 0xcf, 0x85, 0x2e, 0x0f, 0xae,
    0x2c, 0xc8, 0xcd, 0xed, 0x8d, 0x2e, 0x8c, 0x2e, 0xed, 0xcd, 0x6d, 0x6e,
    0x8a, 0xb0, 0x81, 0x41, 0x1d, 0x32, 0x3c, 0x17, 0xbb, 0xb4, 0xb2, 0xbb,
    0x24, 0xb2, 0x29, 0xba, 0x30, 0xba, 0xb2, 0x29, 0x81, 0x18, 0xd4, 0x21,
    0xc3, 0x73, 0x29, 0x73, 0xa3, 0x93, 0xcb, 0x83, 0x7a, 0x4b, 0x73, 0xa3,
    0x9b, 0x9b, 0x12, 0xe4, 0x41, 0x17, 0x32, 0x3c, 0x97, 0xb1, 0xb7, 0x3a,
    0x37, 0xba, 0x32, 0xb9, 0xb9, 0x29, 0x41, 0x1f, 0x00, 0x79, 0x18, 0x00,
    0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c,
    0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80,
    0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed,
    0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d,
    0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83,
    0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78,
    0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70,
    0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc,
    0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3,
    0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c,
    0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83,
    0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03,
    0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68,
    0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60,
    0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80,
    0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98,
    0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec,
    0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c,
    0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d,
    0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43,
    0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03,
    0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03,
    0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28,
    0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4,
    0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1,
    0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00,
    0x00, 0x0b, 0x00, 0x00, 0x00, 0x16, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe,
    0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x02, 0xd5,
    0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4,
    0xe4, 0x17, 0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xe1, 0x4a, 0xd5, 0x78, 0xe4, 0x4f, 0xd6,
    0x12, 0x0d, 0x86, 0x23, 0xf7, 0x0e, 0x82, 0x21, 0x40, 0x44, 0x58, 0x49,
    0x4c, 0x18, 0x05, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x46, 0x01, 0x00,
    0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00,
    0x00, 0x3d, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04,
    0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08,
    0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x10, 0x45, 0x02, 0x42, 0x92, 0x0b,
    0x42, 0x84, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x42,
    0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32,
    0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81,
    0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x21, 0x46, 0x06, 0x51, 0x18, 0x00,
    0x00, 0x06, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff,
    0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03,
    0x20, 0x01, 0x00, 0x00, 0x00, 0x49, 0x18, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00,
    0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x22, 0x08, 0x09, 0x20, 0x64, 0x85,
    0x04, 0x13, 0x22, 0xa4, 0x84, 0x04, 0x13, 0x22, 0xe3, 0x84, 0xa1, 0x90,
    0x14, 0x12, 0x4c, 0x88, 0x8c, 0x0b, 0x84, 0x84, 0x4c, 0x10, 0x30, 0x23,
    0x00, 0x25, 0x00, 0x8a, 0x19, 0x80, 0x39, 0x02, 0x30, 0x98, 0x23, 0x40,
    0x8a, 0x31, 0x44, 0x54, 0x44, 0x56, 0x0c, 0x20, 0xa2, 0x1a, 0xc2, 0x81,
    0x80, 0x74, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60,
    0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf,
    0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a,
    0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
    0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
    0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6,
    0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73,
    0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74,
    0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71,
    0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43,
    0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xc8, 0x02, 0x01, 0x0d, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98,
    0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43,
    0xa2, 0x12, 0x18, 0x01, 0x28, 0x86, 0x22, 0x28, 0x83, 0xf2, 0xa0, 0x2a,
    0x89, 0x32, 0x28, 0x84, 0x11, 0x80, 0x22, 0x28, 0x10, 0xda, 0xb1, 0x1c,
    0x86, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x04, 0x02, 0x01, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c,
    0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03,
    0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01,
    0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a,
    0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b,
    0xd9, 0x10, 0x04, 0x13, 0x04, 0x62, 0x98, 0x20, 0x10, 0xc4, 0x06, 0x61,
    0x20, 0x26, 0x08, 0x44, 0xb1, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b,
    0x06, 0xc4, 0x20, 0x26, 0x08, 0xcb, 0xb3, 0x21, 0x50, 0x26, 0x08, 0x02,
    0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x8d, 0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb,
    0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb, 0x04, 0xa1, 0x48, 0x26, 0x08, 0x85,
    0xb2, 0x21, 0x20, 0x26, 0x08, 0xc5, 0x32, 0x41, 0x28, 0x98, 0x0d, 0x0b,
    0xe1, 0x3c, 0x50, 0x24, 0x0d, 0x12, 0x31, 0x01, 0x44, 0xa8, 0x8a, 0xb0,
    0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26, 0x08, 0x45, 0xb3, 0x61, 0x19, 0xaa,
    0x67, 0x8a, 0xac, 0xc1, 0x1a, 0x26, 0x60, 0x82, 0x40, 0x18, 0x2c, 0x86,
    0x9e, 0x98, 0x9e, 0xa4, 0x26, 0x08, 0xc4, 0x31, 0x41, 0x20, 0x90, 0x0d,
    0x82, 0xb6, 0x6d, 0x58, 0xb0, 0xec, 0x99, 0x22, 0x6b, 0x90, 0xb0, 0x89,
    0xdb, 0x30, 0x50, 0x57, 0xc7, 0x64, 0xca, 0xea, 0x8b, 0x2a, 0x4c, 0xee,
    0xac, 0x8c, 0x6e, 0x82, 0x50, 0x38, 0x1b, 0x16, 0xe2, 0x7b, 0xc0, 0x20,
    0x9a, 0x06, 0x89, 0x98, 0xb8, 0x0d, 0x41, 0x18, 0x6c, 0x18, 0x3c, 0x31,
    0x00, 0x36, 0x14, 0x4c, 0x33, 0x06, 0x00, 0x50, 0x85, 0x8d, 0xcd, 0xae,
    0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3,
    0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d,
    0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x46,
    0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9, 0xa6, 0x37,
    0xb2, 0x32, 0xb6, 0x29, 0x01, 0x52, 0x87, 0x0c, 0xcf, 0xc5, 0x2e, 0xad,
    0xec, 0x2e, 0x89, 0x6c, 0x8a, 0x2e, 0x8c, 0xae, 0x6c, 0x4a, 0xa0, 0xd4,
    0x21, 0xc3, 0x73, 0x29, 0x73, 0xa3, 0x93, 0xcb, 0x83, 0x7a, 0x4b, 0x73,
    0xa3, 0x9b, 0x9b, 0x12, 0x8c, 0x01, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00,
    0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c,
    0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80,
    0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed,
    0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d,
    0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83,
    0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78,
    0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70,
    0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc,
    0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3,
    0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c,
    0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83,
    0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03,
    0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68,
    0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60,
    0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80,
    0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98,
    0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec,
    0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c,
    0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d,
    0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43,
    0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03,
    0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03,
    0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28,
    0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4,
    0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1,
    0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00,
    0x00, 0x0b, 0x00, 0x00, 0x00, 0x16, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe,
    0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x02, 0xd5,
    0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4,
    0xe4, 0x17, 0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00,
    0x00, 0x61, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41,
    0x2c, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x44, 0x85, 0x30,
    0x03, 0x50, 0x0a, 0x54, 0x25, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x60, 0x4c, 0x84, 0x03, 0x21, 0xc3, 0x88, 0x41, 0x02,
    0x80, 0x20, 0x18, 0x18, 0x54, 0xf1, 0x44, 0x02, 0x31, 0x62, 0x90, 0x00,
    0x20, 0x08, 0x06, 0x46, 0x65, 0x40, 0x52, 0x52, 0x8c, 0x18, 0x24, 0x00,
    0x08, 0x82, 0x81, 0x61, 0x1d, 0xd1, 0xd4, 0x18, 0x23, 0x06, 0x09, 0x00,
    0x82, 0x60, 0x80, 0x58, 0x06, 0x45, 0x31, 0xc4, 0x88, 0x41, 0x02, 0x80,
    0x20, 0x18, 0x20, 0x96, 0x41, 0x51, 0xc5, 0x30, 0x62, 0x90, 0x00, 0x20,
    0x08, 0x06, 0x88, 0x65, 0x50, 0xd4, 0x22, 0x8c, 0x18, 0x24, 0x00, 0x08,
    0x82, 0x01, 0x62, 0x19, 0x14, 0xe5, 0x04, 0x08, 0x00, 0x00, 0x00, 0x00,
    0x00
};

/* The texture-rendering pixel shader:

    --- D3D12_PixelShader_Textures.hlsl ---
    Texture2D theTexture : register(t0);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define TextureRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(TextureRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        return theTexture.Sample(theSampler, input.tex) * input.color;
    }
*/

static unsigned char D3D12_PixelShader_Textures[] = {
    0x44, 0x58, 0x42, 0x43, 0x7e, 0x22, 0x8c, 0xc3, 0x6b, 0xf9, 0xa8, 0x10,
    0xf1, 0x5f, 0x55, 0xde, 0x62, 0x56, 0x81, 0x2d, 0x01, 0x00, 0x00, 0x00,
    0x7d, 0x11, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0xfd, 0x01, 0x00, 0x00, 0x8d, 0x02, 0x00, 0x00, 0x6d, 0x0a, 0x00, 0x00,
    0x89, 0x0a, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0xe0, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x54, 0x45,
    0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44,
    0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x0a, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03, 0x02, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44,
    0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00,
    0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x08, 0x00, 0x00, 0x00, 0x52, 0x54, 0x53, 0x30, 0x88, 0x00, 0x00,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0x70, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0xd8, 0x07, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0xf6, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xc0, 0x07, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xed, 0x01, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4a, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x9c, 0x23, 0x08, 0x8a, 0x21, 0xc5, 0x12, 0xa2, 0x92, 0x1d,
    0x08, 0x18, 0x46, 0x10, 0x80, 0xbb, 0xa4, 0x29, 0xa2, 0x84, 0xc9, 0x4f,
    0x91, 0x8b, 0x58, 0xd8, 0x03, 0x18, 0x88, 0x48, 0x6c, 0x1e, 0x6a, 0x42,
    0x43, 0xc8, 0x21, 0x1d, 0x04, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72,
    0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72,
    0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e,
    0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07,
    0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
    0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07,
    0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07,
    0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07,
    0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07,
    0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06,
    0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e,
    0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07,
    0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21, 0x23, 0x45, 0x44, 0x00,
    0x72, 0x00, 0xc0, 0xf4, 0x00, 0x80, 0x87, 0x3c, 0x09, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x18, 0x20, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x38, 0x40,
    0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x99,
    0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8,
    0x63, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40,
    0x16, 0x08, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98,
    0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43,
    0x5a, 0x05, 0x42, 0xa2, 0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2,
    0x50, 0xca, 0xa0, 0x1c, 0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8,
    0x3c, 0x8a, 0x85, 0x56, 0x11, 0x8c, 0x00, 0x14, 0x42, 0x19, 0x94, 0x04,
    0x9d, 0x19, 0x00, 0x2a, 0x33, 0x00, 0x44, 0x66, 0x00, 0x68, 0xcc, 0x00,
    0x50, 0x98, 0x01, 0x20, 0x3d, 0x03, 0x40, 0x7b, 0x2c, 0x87, 0x21, 0x00,
    0x00, 0x00, 0x9e, 0x07, 0x00, 0x02, 0x81, 0x40, 0x00, 0x79, 0x18, 0x00,
    0x00, 0xb7, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
    0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b,
    0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b,
    0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9,
    0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13,
    0x04, 0x01, 0x99, 0x20, 0x08, 0xc9, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04,
    0x41, 0x01, 0x6e, 0x6e, 0x82, 0x20, 0x28, 0x1b, 0x86, 0x03, 0x21, 0x26,
    0x08, 0x45, 0x47, 0x85, 0x0e, 0xad, 0x8c, 0xaa, 0x0c, 0x8f, 0xae, 0x4e,
    0xae, 0x6c, 0x82, 0x20, 0x2c, 0x13, 0x04, 0x81, 0xd9, 0x20, 0x10, 0xcd,
    0x86, 0x84, 0x50, 0x16, 0x82, 0x18, 0x18, 0xc2, 0xd9, 0x10, 0x3c, 0x13,
    0x84, 0x83, 0xa3, 0x42, 0x87, 0x56, 0x36, 0x15, 0xd6, 0x06, 0xc7, 0x56,
    0x26, 0xb7, 0x01, 0x21, 0x22, 0x89, 0x20, 0x06, 0x02, 0xd8, 0x10, 0x4c,
    0x1b, 0x08, 0x08, 0x00, 0xa8, 0x09, 0x02, 0xe1, 0x4d, 0x10, 0x84, 0x86,
    0x01, 0xda, 0x04, 0x41, 0x70, 0x26, 0x08, 0xc2, 0xb3, 0xc1, 0x40, 0xb0,
    0x8c, 0xd0, 0x1a, 0x12, 0x6d, 0x69, 0x70, 0x73, 0x13, 0x04, 0x01, 0xda,
    0x40, 0x20, 0x5c, 0xd6, 0x4d, 0x10, 0x02, 0x30, 0xd8, 0x20, 0x10, 0xdf,
    0x86, 0x00, 0x0c, 0x36, 0x08, 0x44, 0x18, 0x6c, 0x20, 0xae, 0xcd, 0x13,
    0x83, 0x09, 0xc2, 0xf0, 0x4d, 0x10, 0x84, 0x88, 0x06, 0x5a, 0x98, 0x1b,
    0x19, 0x5b, 0xd9, 0x04, 0x41, 0x90, 0x36, 0x18, 0x88, 0x19, 0x64, 0x84,
    0x76, 0x06, 0x1b, 0x84, 0x32, 0x40, 0x83, 0x09, 0x82, 0x16, 0x06, 0x13,
    0x04, 0x61, 0xe2, 0x00, 0xf7, 0x36, 0xc7, 0x65, 0xca, 0xea, 0x0b, 0xea,
    0x69, 0x2a, 0x89, 0x2a, 0xe9, 0xc9, 0x69, 0x03, 0x82, 0xb0, 0x41, 0x46,
    0x94, 0x41, 0x1b, 0x68, 0x0d, 0x07, 0xba, 0x32, 0x3c, 0x26, 0x54, 0x45,
    0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x30, 0x1b, 0x10, 0xe4, 0x0d, 0xb2,
    0xae, 0x0c, 0xe0, 0x40, 0x6b, 0x58, 0x8c, 0xbd, 0xb1, 0xbd, 0xc9, 0x4d,
    0x10, 0x04, 0x8a, 0xc6, 0xd0, 0x13, 0xd3, 0x93, 0x14, 0xcc, 0x06, 0x04,
    0x91, 0x83, 0x6c, 0x0e, 0xca, 0x80, 0x0e, 0xb4, 0x66, 0x03, 0xb1, 0x06,
    0x6e, 0x10, 0x07, 0x75, 0xb0, 0xe1, 0x20, 0xac, 0x31, 0x20, 0x83, 0x34,
    0x50, 0x03, 0x3b, 0x98, 0x20, 0x28, 0xc2, 0x06, 0x60, 0xc3, 0x40, 0xe4,
    0x41, 0x1e, 0x6c, 0x08, 0xf4, 0x60, 0xc3, 0x30, 0xe0, 0xc1, 0x1e, 0x4c,
    0x10, 0x36, 0x31, 0xd8, 0x10, 0xf4, 0x01, 0x89, 0xb6, 0xb0, 0x34, 0x37,
    0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e,
    0x13, 0x84, 0x65, 0x9b, 0x20, 0x2c, 0xda, 0x86, 0x80, 0x98, 0x20, 0x2c,
    0xd9, 0x04, 0x61, 0xc1, 0x36, 0x2c, 0x04, 0x28, 0x84, 0x82, 0x28, 0x8c,
    0x02, 0x29, 0x0c, 0xa4, 0x40, 0x94, 0x02, 0x40, 0x84, 0xaa, 0x08, 0x6b,
    0xe8, 0xe9, 0x49, 0x8a, 0x68, 0x82, 0xb0, 0x5c, 0x1b, 0x84, 0x2c, 0xdb,
    0xb0, 0x0c, 0xa7, 0x10, 0x0a, 0xa5, 0x30, 0x0a, 0xa8, 0x30, 0xa0, 0xc2,
    0x50, 0x0a, 0xa9, 0xc0, 0x62, 0xe8, 0x89, 0xe9, 0x49, 0x6a, 0x82, 0x20,
    0x54, 0x1b, 0x84, 0x8c, 0x15, 0x36, 0x2c, 0xcc, 0x2a, 0x84, 0x42, 0x29,
    0x8c, 0x02, 0x2a, 0x0c, 0xa4, 0xc0, 0x94, 0x42, 0x2b, 0x6c, 0x18, 0x4c,
    0x41, 0x15, 0x5c, 0x81, 0xc9, 0x94, 0xd5, 0x17, 0x55, 0x98, 0xdc, 0x59,
    0x19, 0xdd, 0x04, 0x61, 0x39, 0x36, 0x2c, 0x04, 0x2c, 0x84, 0x42, 0x2c,
    0x8c, 0x42, 0x29, 0x0c, 0xa4, 0x40, 0x94, 0x42, 0x2b, 0x6c, 0x08, 0x64,
    0x61, 0xc3, 0xf0, 0x0a, 0xb3, 0x00, 0x6c, 0x28, 0xf0, 0xe0, 0x0f, 0x68,
    0xa1, 0x02, 0x68, 0x98, 0xb1, 0xbd, 0x85, 0xd1, 0xcd, 0xb1, 0x48, 0x73,
    0x9b, 0xa3, 0x9b, 0x9b, 0x20, 0x08, 0x16, 0x8d, 0xb9, 0xb4, 0xb3, 0x2f,
    0x36, 0x32, 0x1a, 0x73, 0x69, 0x67, 0x5f, 0x73, 0x74, 0x44, 0xe8, 0xca,
    0xf0, 0xbe, 0xdc, 0xde, 0xe4, 0xda, 0x36, 0x28, 0xb6, 0x50, 0x06, 0xb7,
    0x80, 0x0b, 0xb9, 0x80, 0xe8, 0x42, 0x19, 0xec, 0xc2, 0x50, 0x85, 0x8d,
    0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54,
    0x21, 0xc3, 0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12,
    0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6,
    0x04, 0x45, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9,
    0xa6, 0x37, 0xb2, 0x32, 0xb6, 0x29, 0x01, 0x52, 0x86, 0x0c, 0xcf, 0x45,
    0xae, 0x6c, 0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0x40, 0x55,
    0x22, 0xc3, 0x73, 0xa1, 0xcb, 0x83, 0x2b, 0x0b, 0x72, 0x73, 0x7b, 0xa3,
    0x0b, 0xa3, 0x4b, 0x7b, 0x73, 0x9b, 0x9b, 0x22, 0xd8, 0xc1, 0x1e, 0xd4,
    0x21, 0xc3, 0x73, 0xb1, 0x4b, 0x2b, 0xbb, 0x4b, 0x22, 0x9b, 0xa2, 0x0b,
    0xa3, 0x2b, 0x9b, 0x12, 0xf4, 0x41, 0x1d, 0x32, 0x3c, 0x97, 0x32, 0x37,
    0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba, 0xb9, 0x29, 0x01, 0x2d,
    0x74, 0x21, 0xc3, 0x73, 0x19, 0x7b, 0xab, 0x73, 0xa3, 0x2b, 0x93, 0x9b,
    0x9b, 0x12, 0xec, 0x02, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x15, 0xf0, 0x05, 0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9,
    0x65, 0x39, 0x10, 0x38, 0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f,
    0xd2, 0xc3, 0xf4, 0x32, 0x10, 0x18, 0x2c, 0x80, 0x37, 0x08, 0xfc, 0xe8,
    0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7, 0xe5, 0x65, 0x20, 0x70, 0x66, 0xfd,
    0x91, 0xa8, 0x65, 0x3c, 0xbd, 0x2e, 0x2f, 0xcb, 0x88, 0x40, 0xeb, 0x8f,
    0x64, 0x2f, 0x8f, 0xe9, 0x6f, 0x39, 0xb0, 0x49, 0x82, 0xcd, 0x80, 0x40,
    0x20, 0x30, 0x68, 0x06, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x11, 0x01,
    0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0xd8, 0xc0, 0x34, 0x5c, 0xbe, 0xf3,
    0xf8, 0x8b, 0x03, 0x0c, 0x62, 0xf3, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x15,
    0x40, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x01, 0xcc, 0xb3, 0x10, 0x7e, 0x71,
    0xdb, 0x46, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x34, 0x39, 0x11, 0x81,
    0x52, 0xd3, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x26, 0x40, 0x30, 0x00, 0xd2,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0xa4, 0x51, 0x7f, 0xb4, 0x3e, 0x6a,
    0xe5, 0x04, 0x66, 0x2e, 0x73, 0x80, 0x43, 0xf2, 0xb2, 0x44, 0x58, 0x49,
    0x4c, 0xec, 0x06, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xbb, 0x01, 0x00,
    0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0xd4, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00,
    0x00, 0xb2, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04,
    0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08,
    0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b,
    0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52,
    0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32,
    0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81,
    0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00,
    0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff,
    0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03,
    0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8,
    0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60,
    0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00,
    0x00, 0x43, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85,
    0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90,
    0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x68, 0x23,
    0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00,
    0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06,
    0xa1, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56,
    0x62, 0xf2, 0x8b, 0xdb, 0x46, 0xc5, 0x18, 0x63, 0x10, 0x2a, 0xf7, 0x0c,
    0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82,
    0x55, 0x18, 0x45, 0x18, 0x1b, 0x63, 0x0c, 0x42, 0xc8, 0xa0, 0x36, 0x47,
    0x10, 0x14, 0x83, 0x91, 0x42, 0xc8, 0x23, 0x38, 0x10, 0x30, 0x8c, 0x40,
    0x0c, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc, 0xc3, 0x3c, 0xb8,
    0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8,
    0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0, 0xc3, 0x3b, 0xc8,
    0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0, 0x03, 0x1b, 0x80,
    0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8, 0x41, 0x3b, 0xa4,
    0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0, 0x43, 0x39, 0xa0,
    0x80, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30, 0x0f, 0xf3, 0xe0,
    0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20, 0x0f, 0xe5, 0x20,
    0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40, 0x0f, 0xef, 0x20,
    0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40, 0x0f, 0x6c, 0x00,
    0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x98, 0x94, 0xea, 0x4d,
    0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c, 0x44, 0xc4, 0x4e,
    0xc0, 0x44, 0xa0, 0x80, 0xd0, 0x4d, 0x07, 0x02, 0x00, 0x13, 0x14, 0x72,
    0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72,
    0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e,
    0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07,
    0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
    0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07,
    0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07,
    0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07,
    0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07,
    0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06,
    0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e,
    0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07,
    0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00,
    0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02,
    0x50, 0x0c, 0x45, 0x50, 0x12, 0x65, 0x50, 0x1e, 0x54, 0x4a, 0xa2, 0x0c,
    0x0a, 0x61, 0x04, 0xa0, 0x08, 0x0a, 0x84, 0xec, 0x0c, 0x00, 0xe1, 0x19,
    0x00, 0xca, 0x63, 0x39, 0x0c, 0x01, 0x00, 0x00, 0xf0, 0x3c, 0x00, 0x10,
    0x08, 0x04, 0x02, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x63, 0x00, 0x00,
    0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63,
    0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03,
    0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a,
    0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b,
    0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0xa2, 0x98, 0x20,
    0x10, 0xc6, 0x06, 0x61, 0x20, 0x26, 0x08, 0xc4, 0xb1, 0x41, 0x18, 0x0c,
    0x0a, 0x70, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0x96, 0x44, 0x60,
    0x82, 0x40, 0x20, 0x13, 0x04, 0x22, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84,
    0x50, 0x16, 0x82, 0x18, 0x18, 0xc2, 0xd9, 0x10, 0x3c, 0x13, 0x04, 0x6c,
    0xda, 0x80, 0x10, 0xd1, 0x42, 0x10, 0x03, 0x01, 0x6c, 0x08, 0xa4, 0x0d,
    0x04, 0x04, 0x00, 0xd3, 0x04, 0x21, 0xa3, 0x36, 0x04, 0xd5, 0x04, 0x41,
    0x00, 0x48, 0xb4, 0x85, 0xa5, 0xb9, 0x71, 0x99, 0xb2, 0xfa, 0x82, 0x7a,
    0x9b, 0x4b, 0xa3, 0x4b, 0x7b, 0x73, 0x9b, 0x20, 0x14, 0xcc, 0x04, 0xa1,
    0x68, 0x36, 0x04, 0xc4, 0x04, 0xa1, 0x70, 0x26, 0x08, 0xc5, 0xb3, 0x61,
    0x21, 0x32, 0x6d, 0xe3, 0xba, 0xa1, 0x23, 0x3c, 0x80, 0x08, 0x55, 0x11,
    0xd6, 0xd0, 0xd3, 0x93, 0x14, 0xd1, 0x04, 0xa1, 0x80, 0x26, 0x08, 0x84,
    0xb2, 0x41, 0x10, 0x03, 0x31, 0xd8, 0xb0, 0x0c, 0x60, 0xa0, 0x79, 0x5c,
    0x18, 0x0c, 0x61, 0x30, 0x78, 0x63, 0xc0, 0x62, 0xe8, 0x89, 0xe9, 0x49,
    0x6a, 0x82, 0x40, 0x2c, 0x1b, 0x04, 0x31, 0x30, 0x83, 0x0d, 0x0b, 0x53,
    0x06, 0x9a, 0xc7, 0x85, 0xc1, 0xd0, 0x31, 0xde, 0x19, 0x6c, 0x18, 0x3e,
    0x32, 0x40, 0x03, 0x26, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65,
    0x74, 0x13, 0x84, 0x22, 0xda, 0xb0, 0x10, 0x6a, 0xa0, 0xad, 0x01, 0xe7,
    0x0d, 0x1d, 0xe1, 0x9d, 0xc1, 0x86, 0x80, 0x0d, 0x36, 0x0c, 0x69, 0xd0,
    0x06, 0xc0, 0x86, 0xe2, 0xc2, 0xdc, 0x80, 0x02, 0xaa, 0xb0, 0xb1, 0xd9,
    0xb5, 0xb9, 0xa4, 0x91, 0x95, 0xb9, 0xd1, 0x4d, 0x09, 0x82, 0x2a, 0x64,
    0x78, 0x2e, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x53, 0x02, 0xa2,
    0x09, 0x19, 0x9e, 0x8b, 0x5d, 0x18, 0x9b, 0x5d, 0x99, 0xdc, 0x94, 0xc0,
    0xa8, 0x43, 0x86, 0xe7, 0x32, 0x87, 0x16, 0x46, 0x56, 0x26, 0xd7, 0xf4,
    0x46, 0x56, 0xc6, 0x36, 0x25, 0x40, 0xca, 0x90, 0xe1, 0xb9, 0xc8, 0x95,
    0xcd, 0xbd, 0xd5, 0xc9, 0x8d, 0x95, 0xcd, 0x4d, 0x09, 0xa6, 0x3a, 0x64,
    0x78, 0x2e, 0x76, 0x69, 0x65, 0x77, 0x49, 0x64, 0x53, 0x74, 0x61, 0x74,
    0x65, 0x53, 0x82, 0xaa, 0x0e, 0x19, 0x9e, 0x4b, 0x99, 0x1b, 0x9d, 0x5c,
    0x1e, 0xd4, 0x5b, 0x9a, 0x1b, 0xdd, 0xdc, 0x94, 0xc0, 0x0d, 0x00, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80,
    0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84,
    0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c,
    0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42,
    0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88,
    0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c,
    0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79,
    0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70,
    0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f,
    0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4,
    0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30,
    0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc,
    0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b,
    0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70,
    0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76,
    0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72,
    0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e,
    0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1,
    0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21,
    0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8,
    0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94,
    0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc,
    0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c,
    0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e,
    0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e,
    0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00,
    0x00, 0x71, 0x20, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x46, 0x20, 0x0d,
    0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c,
    0x84, 0x05, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0xbf, 0x38, 0xc0, 0x20, 0x36,
    0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0xdb, 0x00, 0x34, 0x5c, 0xbe, 0xf3, 0xf8,
    0x12, 0xc0, 0x3c, 0x0b, 0xe1, 0x17, 0xb7, 0x6d, 0x02, 0xd5, 0x70, 0xf9,
    0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4, 0x17,
    0xb7, 0x6d, 0x00, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x61, 0x20, 0x00,
    0x00, 0x3a, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0xf4, 0x46, 0x00, 0x88, 0xcc, 0x00, 0x14,
    0x42, 0x29, 0x94, 0x5c, 0xe1, 0x51, 0x29, 0x01, 0x1a, 0x33, 0x00, 0x00,
    0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x00, 0x65, 0x84, 0x73, 0x5d,
    0xc8, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x40, 0x5a, 0x11, 0x61, 0x58,
    0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x87, 0x5c, 0x19, 0x94,
    0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xf1, 0x25, 0x98, 0x56, 0x28,
    0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x80, 0x81, 0x92, 0x6d, 0xd1,
    0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46, 0x18, 0x2c, 0x1a, 0x57,
    0x31, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x88, 0x01, 0xe3, 0x75,
    0x54, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc6, 0x18, 0x34, 0x9f,
    0x97, 0x38, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0x88, 0xc1, 0x82,
    0x1c, 0x42, 0x90, 0x24, 0xdf, 0xf7, 0x24, 0xa3, 0x09, 0x01, 0x30, 0x9a,
    0x20, 0x04, 0xa3, 0x09, 0x83, 0x30, 0x9a, 0x40, 0x0c, 0x46, 0x2c, 0xf2,
    0x31, 0x62, 0x91, 0x8f, 0x11, 0x8b, 0x7c, 0x8c, 0x58, 0xe4, 0x33, 0x62,
    0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0x1b, 0x58, 0x68, 0x80, 0x06, 0x1f,
    0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0x1b, 0x58, 0x68, 0x80,
    0x06, 0xd3, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0x1b, 0x58,
    0x68, 0x80, 0x06, 0x9e, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08,
    0x1b, 0x58, 0x68, 0x80, 0x06, 0x61, 0x10, 0x20, 0x00, 0x00, 0x00, 0x00,
    0x00
};

/* The yuv-rendering pixel shader:

    --- D3D12_PixelShader_YUV_JPEG.hlsl ---
    Texture2D theTextureY : register(t0);
    Texture2D theTextureU : register(t1);
    Texture2D theTextureV : register(t2);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define YUVRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t2), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(YUVRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        const float3 offset = {0.0, -0.501960814, -0.501960814};
        const float3 Rcoeff = {1.0000,  0.0000,  1.4020};
        const float3 Gcoeff = {1.0000, -0.3441, -0.7141};
        const float3 Bcoeff = {1.0000,  1.7720,  0.0000};

        float4 Output;

        float3 yuv;
        yuv.x = theTextureY.Sample(theSampler, input.tex).r;
        yuv.y = theTextureU.Sample(theSampler, input.tex).r;
        yuv.z = theTextureV.Sample(theSampler, input.tex).r;

        yuv += offset;
        Output.r = dot(yuv, Rcoeff);
        Output.g = dot(yuv, Gcoeff);
        Output.b = dot(yuv, Bcoeff);
        Output.a = 1.0f;

        return Output * input.color;
    }

*/

static unsigned char D3D12_PixelShader_YUV_JPEG[] = {
    0x44, 0x58, 0x42, 0x43, 0xb1, 0xef, 0x39, 0xc1, 0xc8, 0x72, 0xbb, 0xb0,
    0x20, 0x73, 0xd6, 0x2a, 0xc9, 0xba, 0xcb, 0x70, 0x01, 0x00, 0x00, 0x00,
    0x95, 0x13, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0x1d, 0x02, 0x00, 0x00, 0x05, 0x03, 0x00, 0x00, 0x9d, 0x0b, 0x00, 0x00,
    0xb9, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0x00, 0x01, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f,
    0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42,
    0x00, 0x03, 0x02, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x02, 0x44, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x52, 0x54, 0x53, 0x30, 0xe0, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xe0, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x01, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0xc8, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0x90, 0x08, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0x24, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x78, 0x08, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x1b, 0x02, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4b, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x2c, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x39, 0x82,
    0xa0, 0x18, 0x52, 0x2c, 0x21, 0x2e, 0xe1, 0x81, 0x80, 0x61, 0x04, 0x01,
    0xb8, 0x4b, 0x9a, 0x22, 0x4a, 0x98, 0xfc, 0x14, 0xb9, 0x88, 0x85, 0x3d,
    0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x84, 0x1d, 0xd2, 0x41,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21,
    0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xc9, 0x01,
    0x00, 0xd3, 0x03, 0x00, 0x1e, 0xf2, 0x24, 0x40, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x61, 0x80, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0xe3, 0x00, 0x01, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x67, 0x02, 0x02,
    0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x4f, 0x05,
    0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e,
    0x0c, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2,
    0x40, 0x16, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x5a, 0x05, 0x42, 0xa2,
    0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c,
    0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8, 0x3c, 0x0a, 0xa2, 0x74,
    0x68, 0x15, 0xc1, 0x08, 0x40, 0x21, 0x94, 0x41, 0x49, 0xd0, 0x99, 0x01,
    0xa0, 0x32, 0x03, 0x40, 0x64, 0x06, 0x80, 0xc6, 0x0c, 0x00, 0x85, 0x19,
    0x00, 0xe2, 0x33, 0x00, 0xd4, 0xc7, 0x72, 0x18, 0x02, 0x00, 0x00, 0xe0,
    0x38, 0x00, 0x20, 0x10, 0x08, 0x04, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00,
    0x00, 0xc5, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
    0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b,
    0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b,
    0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9,
    0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13,
    0x04, 0x61, 0x99, 0x20, 0x08, 0xcc, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04,
    0x41, 0x01, 0x6e, 0x6e, 0x82, 0x20, 0x34, 0x1b, 0x86, 0x03, 0x21, 0x26,
    0x08, 0x45, 0x18, 0x70, 0xa1, 0x43, 0x2b, 0xa3, 0x2a, 0xc3, 0xa3, 0xab,
    0x93, 0x2b, 0xcb, 0x9a, 0x20, 0x08, 0xce, 0x04, 0x41, 0x78, 0x36, 0x08,
    0x44, 0xb3, 0x21, 0x21, 0x94, 0x85, 0x20, 0x06, 0x86, 0x70, 0xb8, 0xd0,
    0xa1, 0x95, 0x51, 0x95, 0xe1, 0xd1, 0xd5, 0xc9, 0x95, 0x55, 0x6d, 0x48,
    0x06, 0x05, 0x22, 0x86, 0x81, 0x21, 0x1c, 0x2e, 0x74, 0x68, 0x65, 0x54,
    0x65, 0x78, 0x74, 0x75, 0x72, 0x65, 0x56, 0x1b, 0x12, 0x46, 0x91, 0x08,
    0x66, 0x60, 0x08, 0x67, 0xc3, 0xf0, 0x44, 0xd3, 0x04, 0xe1, 0x00, 0x03,
    0x2a, 0x74, 0x68, 0x65, 0x53, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x72, 0x1b,
    0x10, 0xa2, 0xb2, 0x08, 0x62, 0x20, 0x80, 0x0d, 0xc1, 0xb5, 0x81, 0xa0,
    0x00, 0x00, 0x9b, 0x20, 0x10, 0x62, 0x30, 0x41, 0x10, 0x20, 0x06, 0x68,
    0x13, 0x04, 0x21, 0x9a, 0x20, 0x08, 0xd2, 0x06, 0x03, 0xe1, 0x3a, 0xc2,
    0x6b, 0x48, 0xb4, 0xa5, 0xc1, 0xcd, 0x4d, 0x10, 0x84, 0x69, 0x03, 0x81,
    0x80, 0x41, 0x17, 0x06, 0x13, 0x84, 0x80, 0x0c, 0x36, 0x08, 0xc4, 0x18,
    0x6c, 0x08, 0xc8, 0x60, 0x83, 0x40, 0x94, 0xc1, 0x06, 0x62, 0xfb, 0xc4,
    0xc0, 0x0c, 0x26, 0x08, 0xc3, 0x18, 0x4c, 0x10, 0x04, 0x8a, 0x06, 0x5a,
    0x98, 0x1b, 0x19, 0x5b, 0xd9, 0x04, 0x41, 0xa8, 0x36, 0x18, 0x88, 0x1a,
    0x74, 0x84, 0xb7, 0x06, 0x1b, 0x84, 0x34, 0x60, 0x83, 0x09, 0x02, 0x57,
    0x06, 0x13, 0x04, 0xc1, 0xe2, 0x00, 0xf7, 0x36, 0xc7, 0x65, 0xca, 0xea,
    0x0b, 0xea, 0x69, 0x2a, 0x89, 0x2a, 0xe9, 0xc9, 0x69, 0x03, 0x82, 0xc0,
    0x41, 0x47, 0xa4, 0x41, 0x1c, 0x78, 0x0d, 0x07, 0xba, 0x32, 0x3c, 0x26,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x30, 0x1b, 0x10, 0x64,
    0x0e, 0xba, 0x30, 0x48, 0x03, 0x3a, 0xf0, 0x1a, 0x16, 0x63, 0x6f, 0x6c,
    0x6f, 0x72, 0x13, 0x04, 0xe1, 0xa2, 0x31, 0xf4, 0xc4, 0xf4, 0x24, 0x05,
    0xb3, 0x01, 0x41, 0xec, 0xa0, 0xbb, 0x83, 0x34, 0xc0, 0x03, 0xaf, 0xd9,
    0x40, 0xbc, 0x81, 0x1c, 0xd4, 0x41, 0x1e, 0x6c, 0x38, 0x08, 0xed, 0x0c,
    0xd0, 0xa0, 0x0d, 0xdc, 0x40, 0x0f, 0x26, 0x08, 0x0a, 0xb1, 0x01, 0xd8,
    0x30, 0x10, 0x7d, 0xd0, 0x07, 0x1b, 0x02, 0x3f, 0xd8, 0x30, 0x0c, 0x7c,
    0xf0, 0x07, 0x13, 0x84, 0xce, 0x0c, 0x36, 0x04, 0xa1, 0x40, 0xa2, 0x2d,
    0x2c, 0xcd, 0x8d, 0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb, 0x5c, 0x1a, 0x5d,
    0xda, 0x9b, 0xdb, 0x04, 0x61, 0xf9, 0x26, 0x08, 0x8b, 0xb7, 0x21, 0x20,
    0x26, 0x08, 0x4b, 0x37, 0x41, 0x58, 0xb8, 0x0d, 0x0b, 0x41, 0x0a, 0xa5,
    0x60, 0x0a, 0xa7, 0x80, 0x0a, 0x03, 0x2a, 0x10, 0xa9, 0x00, 0x10, 0xa1,
    0x2a, 0xc2, 0x1a, 0x7a, 0x7a, 0x92, 0x22, 0x9a, 0x20, 0x2c, 0xdb, 0x06,
    0xa1, 0xeb, 0x36, 0x2c, 0xc3, 0x2a, 0x94, 0x42, 0x2a, 0x9c, 0x02, 0x2b,
    0x0c, 0xac, 0x30, 0xa4, 0x42, 0x2b, 0xb0, 0x18, 0x7a, 0x62, 0x7a, 0x92,
    0x9a, 0x20, 0x08, 0xd8, 0x06, 0xa1, 0x83, 0x85, 0x0d, 0x0b, 0xf3, 0x0a,
    0xa5, 0x90, 0x0a, 0xa7, 0xc0, 0x0a, 0x03, 0x2a, 0x30, 0xa9, 0x10, 0x0b,
    0x1b, 0x06, 0x55, 0x70, 0x05, 0x59, 0x60, 0x32, 0x65, 0xf5, 0x45, 0x15,
    0x26, 0x77, 0x56, 0x46, 0x37, 0x41, 0x58, 0x94, 0x0d, 0x0b, 0x41, 0x0b,
    0xa5, 0x50, 0x0b, 0xa7, 0x90, 0x0a, 0x03, 0x2a, 0x10, 0xa9, 0x10, 0x0b,
    0x1b, 0x02, 0x5b, 0xd8, 0x30, 0xcc, 0xc2, 0x2d, 0x00, 0x1b, 0x0a, 0x3e,
    0x18, 0x05, 0x5c, 0xc8, 0x00, 0x1a, 0x66, 0x6c, 0x6f, 0x61, 0x74, 0x73,
    0x13, 0x04, 0x21, 0x63, 0x91, 0xe6, 0x36, 0x47, 0x37, 0x37, 0x41, 0x10,
    0x34, 0x1a, 0x73, 0x69, 0x67, 0x5f, 0x6c, 0x64, 0x34, 0xe6, 0xd2, 0xce,
    0xbe, 0xe6, 0xe8, 0x88, 0xd0, 0x95, 0xe1, 0x7d, 0xb9, 0xbd, 0xc9, 0xb5,
    0x6d, 0x50, 0x74, 0x61, 0x17, 0x78, 0xa1, 0x17, 0x7c, 0x01, 0xf9, 0x85,
    0x34, 0x00, 0x87, 0xae, 0x0a, 0x1b, 0x9b, 0x5d, 0x9b, 0x4b, 0x1a, 0x59,
    0x99, 0x1b, 0xdd, 0x94, 0x20, 0xa8, 0x42, 0x86, 0xe7, 0x62, 0x57, 0x26,
    0x37, 0x97, 0xf6, 0xe6, 0x36, 0x25, 0x20, 0x9a, 0x90, 0xe1, 0xb9, 0xd8,
    0x85, 0xb1, 0xd9, 0x95, 0xc9, 0x4d, 0x09, 0x8a, 0x3a, 0x64, 0x78, 0x2e,
    0x73, 0x68, 0x61, 0x64, 0x65, 0x72, 0x4d, 0x6f, 0x64, 0x65, 0x6c, 0x53,
    0x02, 0xa4, 0x0c, 0x19, 0x9e, 0x8b, 0x5c, 0xd9, 0xdc, 0x5b, 0x9d, 0xdc,
    0x58, 0xd9, 0xdc, 0x94, 0x00, 0xab, 0x44, 0x86, 0xe7, 0x42, 0x97, 0x07,
    0x57, 0x16, 0xe4, 0xe6, 0xf6, 0x46, 0x17, 0x46, 0x97, 0xf6, 0xe6, 0x36,
    0x37, 0x45, 0xd0, 0x83, 0x3f, 0xa8, 0x43, 0x86, 0xe7, 0x62, 0x97, 0x56,
    0x76, 0x97, 0x44, 0x36, 0x45, 0x17, 0x46, 0x57, 0x36, 0x25, 0x08, 0x85,
    0x3a, 0x64, 0x78, 0x2e, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x50, 0x6f, 0x69,
    0x6e, 0x74, 0x73, 0x53, 0x02, 0x5c, 0xe8, 0x42, 0x86, 0xe7, 0x32, 0xf6,
    0x56, 0xe7, 0x46, 0x57, 0x26, 0x37, 0x37, 0x25, 0x00, 0x07, 0x00, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80,
    0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84,
    0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c,
    0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42,
    0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88,
    0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c,
    0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79,
    0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70,
    0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f,
    0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4,
    0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30,
    0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc,
    0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b,
    0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70,
    0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76,
    0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72,
    0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e,
    0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1,
    0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21,
    0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8,
    0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94,
    0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc,
    0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c,
    0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e,
    0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e,
    0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00,
    0x00, 0x71, 0x20, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x35, 0xf0, 0x05,
    0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9, 0x65, 0x39, 0x10, 0x38,
    0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f, 0xd2, 0xc3, 0xf4, 0x32,
    0x10, 0x18, 0xac, 0x00, 0x38, 0x08, 0xfc, 0xe8, 0xe8, 0x32, 0xb5, 0x8c,
    0xa7, 0xd7, 0xe5, 0xe5, 0x2a, 0x10, 0x38, 0xb3, 0xfe, 0x48, 0xd4, 0x32,
    0x9e, 0x5e, 0x97, 0x97, 0x65, 0x44, 0xa0, 0xf5, 0x47, 0xb2, 0x97, 0xc7,
    0xf4, 0xb7, 0x1c, 0xd8, 0x24, 0xc1, 0x66, 0x40, 0x20, 0x10, 0x18, 0x2c,
    0x01, 0x38, 0x08, 0xfc, 0xe8, 0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7, 0xe5,
    0x65, 0x2b, 0x10, 0x38, 0xb3, 0xfe, 0x48, 0xd4, 0x32, 0x9e, 0x5e, 0x97,
    0x97, 0x65, 0x44, 0xa0, 0xf5, 0x47, 0xb2, 0x97, 0xc7, 0xf4, 0xb7, 0x1c,
    0xd8, 0x24, 0xc1, 0x66, 0x40, 0x20, 0x10, 0x18, 0x2c, 0x00, 0x38, 0x08,
    0xfc, 0xe8, 0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7, 0xe5, 0xe5, 0x2c, 0x10,
    0x38, 0xb3, 0xfe, 0x48, 0xd4, 0x32, 0x9e, 0x5e, 0x97, 0x97, 0x65, 0x44,
    0xa0, 0xf5, 0x47, 0xb2, 0x97, 0xc7, 0xf4, 0xb7, 0x1c, 0xd8, 0x24, 0xc1,
    0x66, 0x40, 0x20, 0x10, 0x18, 0xb4, 0x04, 0x69, 0xb8, 0x7c, 0xe7, 0xf1,
    0x85, 0x88, 0x00, 0x26, 0x22, 0x04, 0x9a, 0x61, 0x21, 0x0c, 0xc1, 0x19,
    0x2e, 0xdf, 0x79, 0xfc, 0xc1, 0x99, 0x6e, 0xbf, 0xb8, 0x6d, 0x2b, 0x98,
    0x86, 0xcb, 0x77, 0x1e, 0x7f, 0x71, 0x80, 0x41, 0x6c, 0x1e, 0x6a, 0xf2,
    0x8b, 0xdb, 0xb6, 0x03, 0x68, 0xb8, 0x7c, 0xe7, 0xf1, 0x25, 0x80, 0x79,
    0x16, 0xc2, 0x2f, 0x6e, 0xdb, 0x0c, 0xaa, 0xe1, 0xf2, 0x9d, 0xc7, 0x97,
    0x26, 0x27, 0x22, 0x50, 0x6a, 0x7a, 0xa8, 0xc9, 0x2f, 0x6e, 0xdb, 0x08,
    0x08, 0x06, 0x40, 0x1a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53,
    0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd2, 0x76, 0x77,
    0xba, 0xe6, 0x06, 0xb3, 0xd2, 0x85, 0x4b, 0xcf, 0xc4, 0x54, 0x26, 0x79,
    0xad, 0x44, 0x58, 0x49, 0x4c, 0xd4, 0x07, 0x00, 0x00, 0x60, 0x00, 0x00,
    0x00, 0xf5, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0xbc, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0,
    0xde, 0x21, 0x0c, 0x00, 0x00, 0xec, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23,
    0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84,
    0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45,
    0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18,
    0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88,
    0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4,
    0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46,
    0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0,
    0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff,
    0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff,
    0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00,
    0x00, 0x89, 0x20, 0x00, 0x00, 0x45, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48,
    0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22,
    0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4,
    0x4c, 0x10, 0x70, 0x23, 0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08,
    0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18,
    0x80, 0x10, 0x52, 0x06, 0xa1, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90,
    0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x8b, 0xdb, 0x46, 0xc5, 0x18, 0x63,
    0x10, 0x2a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0,
    0x0c, 0x0b, 0x81, 0x82, 0x55, 0x18, 0x45, 0x18, 0x1b, 0x63, 0x0c, 0x42,
    0xc8, 0xa0, 0x56, 0x90, 0x41, 0xc6, 0x18, 0x63, 0x0c, 0x7a, 0x73, 0x04,
    0x41, 0x31, 0x18, 0x29, 0x84, 0x44, 0x92, 0x03, 0x01, 0xc3, 0x08, 0xc4,
    0x30, 0x53, 0x1b, 0x8c, 0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b,
    0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c,
    0x90, 0x02, 0x1f, 0xd8, 0x43, 0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c,
    0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xb0, 0x01, 0x18,
    0xd0, 0x81, 0x1f, 0x80, 0x81, 0x1f, 0xe8, 0x81, 0x1e, 0xb4, 0x43, 0x3a,
    0xc0, 0xc3, 0x3c, 0xfc, 0x02, 0x3d, 0xe4, 0x03, 0x3c, 0x94, 0x03, 0x0a,
    0xc8, 0x4c, 0x62, 0x30, 0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e,
    0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72,
    0x40, 0x0a, 0x7c, 0x60, 0x0f, 0xe5, 0x30, 0x0e, 0xf4, 0xf0, 0x0e, 0xf2,
    0xc0, 0x07, 0xe6, 0xc0, 0x0e, 0xef, 0x10, 0x0e, 0xf4, 0xc0, 0x06, 0x60,
    0x40, 0x07, 0x7e, 0x00, 0x06, 0x7e, 0x80, 0x84, 0x6a, 0xe9, 0xde, 0x24,
    0x4d, 0x11, 0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04,
    0x4c, 0x04, 0x0a, 0x08, 0xe5, 0x74, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72,
    0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72,
    0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e,
    0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07,
    0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
    0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07,
    0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07,
    0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07,
    0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07,
    0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06,
    0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e,
    0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07,
    0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01,
    0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08,
    0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02,
    0x50, 0x0c, 0x45, 0x50, 0x12, 0x65, 0x50, 0x1e, 0x54, 0x4a, 0xa2, 0x0c,
    0x0a, 0x61, 0x04, 0xa0, 0x08, 0x0a, 0x84, 0xf0, 0x0c, 0x00, 0xe9, 0x19,
    0x00, 0xda, 0x63, 0x39, 0x0c, 0x01, 0x00, 0x00, 0x70, 0x1c, 0x00, 0x10,
    0x08, 0x04, 0x02, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x69, 0x00, 0x00,
    0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63,
    0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03,
    0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a,
    0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b,
    0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0xc2, 0x98, 0x20,
    0x10, 0xc7, 0x06, 0x61, 0x20, 0x26, 0x08, 0x04, 0xb2, 0x41, 0x18, 0x0c,
    0x0a, 0x70, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0xd8, 0x44, 0x60,
    0x82, 0x40, 0x24, 0x13, 0x04, 0x42, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84,
    0x50, 0x16, 0x82, 0x18, 0x18, 0xc2, 0xd9, 0x90, 0x0c, 0xca, 0x42, 0x0c,
    0x03, 0x43, 0x38, 0x1b, 0x12, 0x46, 0x59, 0x08, 0x66, 0x60, 0x08, 0x67,
    0xc3, 0xf0, 0x40, 0xd1, 0x04, 0x41, 0xa3, 0x36, 0x20, 0xc4, 0xb4, 0x10,
    0xc4, 0x40, 0x00, 0x1b, 0x02, 0x6a, 0x03, 0x21, 0x01, 0x40, 0x35, 0x41,
    0xd8, 0xaa, 0x0d, 0xc1, 0x35, 0x41, 0x10, 0x00, 0x12, 0x6d, 0x61, 0x69,
    0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6, 0xd2, 0xe8, 0xd2, 0xde,
    0xdc, 0x26, 0x08, 0x45, 0x33, 0x41, 0x28, 0x9c, 0x0d, 0x01, 0x31, 0x41,
    0x28, 0x9e, 0x09, 0x42, 0x01, 0x6d, 0x58, 0x88, 0x8d, 0xeb, 0xbc, 0x6f,
    0xf8, 0x08, 0x30, 0x00, 0x88, 0x50, 0x15, 0x61, 0x0d, 0x3d, 0x3d, 0x49,
    0x11, 0x4d, 0x10, 0x8a, 0x68, 0x82, 0x40, 0x2c, 0x1b, 0x04, 0x32, 0x20,
    0x83, 0x0d, 0xcb, 0x20, 0x06, 0x1c, 0x18, 0x78, 0x63, 0x30, 0x8c, 0xc1,
    0x00, 0x06, 0x65, 0xc0, 0x62, 0xe8, 0x89, 0xe9, 0x49, 0x6a, 0x82, 0x40,
    0x30, 0x1b, 0x04, 0x32, 0x40, 0x83, 0x0d, 0x0b, 0x73, 0x06, 0x1c, 0x18,
    0x78, 0x63, 0x30, 0x7c, 0x0c, 0x18, 0xa4, 0xc1, 0x86, 0x21, 0x0c, 0xcc,
    0x40, 0x0d, 0x98, 0x4c, 0x59, 0x7d, 0x51, 0x85, 0xc9, 0x9d, 0x95, 0xd1,
    0x4d, 0x10, 0x0a, 0x69, 0xc3, 0x42, 0xb0, 0x01, 0xd7, 0x06, 0x1e, 0x18,
    0x0c, 0x1f, 0x01, 0x06, 0x69, 0xb0, 0x21, 0x70, 0x83, 0x0d, 0xc3, 0x1a,
    0xbc, 0x01, 0xb0, 0xa1, 0xc8, 0x34, 0x38, 0xb0, 0x80, 0x2a, 0x6c, 0x6c,
    0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x53, 0x82, 0xa0, 0x0a,
    0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda, 0x9b, 0xdb, 0x94, 0x80,
    0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66, 0x57, 0x26, 0x37, 0x25,
    0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85, 0x91, 0x95, 0xc9, 0x35,
    0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x90, 0x32, 0x64, 0x78, 0x2e, 0x72,
    0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73, 0x53, 0x82, 0xaa, 0x0e,
    0x19, 0x9e, 0x8b, 0x5d, 0x5a, 0xd9, 0x5d, 0x12, 0xd9, 0x14, 0x5d, 0x18,
    0x5d, 0xd9, 0x94, 0xe0, 0xaa, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27,
    0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37, 0x25, 0x80, 0x03, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80,
    0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84,
    0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c,
    0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42,
    0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88,
    0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c,
    0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79,
    0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70,
    0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f,
    0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4,
    0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30,
    0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc,
    0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b,
    0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70,
    0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76,
    0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72,
    0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e,
    0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1,
    0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21,
    0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8,
    0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94,
    0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc,
    0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c,
    0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e,
    0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e,
    0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00,
    0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x56, 0x20, 0x0d,
    0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c,
    0x84, 0x11, 0x38, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x38, 0xd3, 0xed, 0x17,
    0xb7, 0x6d, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e, 0x30, 0x88,
    0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x00, 0x0d, 0x97, 0xef, 0x3c,
    0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d, 0x9b, 0x40, 0x35, 0x5c,
    0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a, 0x4d, 0x0f, 0x35, 0xf9,
    0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x61, 0x20, 0x00,
    0x00, 0x66, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x14, 0x47, 0x00, 0xa8, 0x94, 0x00, 0x91,
    0x72, 0x2b, 0xbc, 0x52, 0x28, 0xb9, 0x42, 0x98, 0x01, 0xa0, 0x31, 0x46,
    0x50, 0x9e, 0x74, 0xe9, 0x7f, 0x63, 0x04, 0xa2, 0x3e, 0xb7, 0xf3, 0x2f,
    0x8c, 0x11, 0x80, 0xef, 0x0a, 0xae, 0xbf, 0x30, 0x46, 0xc0, 0x97, 0xbd,
    0xb9, 0x7f, 0x23, 0x00, 0x63, 0x04, 0x21, 0x09, 0x86, 0xf0, 0x2f, 0x8c,
    0x11, 0x80, 0x20, 0x08, 0xe2, 0xdf, 0x0c, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x20, 0x85, 0xc1, 0x62, 0x75, 0x1d, 0x34, 0x62, 0x90,
    0x00, 0x20, 0x08, 0x06, 0x92, 0x18, 0x30, 0x57, 0x18, 0x84, 0x41, 0x34,
    0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xd2, 0x18, 0x34, 0x58, 0x18, 0x84,
    0x81, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x12, 0x19, 0x38, 0x9b,
    0x18, 0x88, 0xc1, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46, 0x1a,
    0x38, 0x61, 0x30, 0x06, 0x5a, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06,
    0x86, 0x1a, 0x3c, 0x62, 0x40, 0x06, 0x94, 0x33, 0x62, 0x90, 0x00, 0x20,
    0x08, 0x06, 0xc6, 0x1a, 0x40, 0x63, 0x50, 0x06, 0xdb, 0x33, 0x62, 0x90,
    0x00, 0x20, 0x08, 0x06, 0x06, 0x1b, 0x44, 0x64, 0x60, 0x06, 0x1f, 0x34,
    0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46, 0x1b, 0x48, 0x68, 0x70, 0x06,
    0x5e, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x1b, 0x4c, 0x69,
    0x80, 0x06, 0x98, 0x34, 0x62, 0xf0, 0x00, 0x20, 0x08, 0x06, 0x4d, 0x1b,
    0x5c, 0xc8, 0x21, 0x04, 0xcb, 0x92, 0x06, 0x69, 0x30, 0x2d, 0xa3, 0x09,
    0x01, 0x30, 0x62, 0xf0, 0x00, 0x20, 0x08, 0x06, 0xcd, 0x1b, 0x64, 0x4b,
    0x42, 0x0c, 0x4d, 0xb3, 0x06, 0x6b, 0x50, 0x35, 0xa3, 0x09, 0x01, 0x30,
    0x62, 0xf0, 0x00, 0x20, 0x08, 0x06, 0x4d, 0x1c, 0x6c, 0xce, 0x62, 0x14,
    0xcf, 0xd3, 0x06, 0x6d, 0x70, 0x3d, 0xa3, 0x09, 0x01, 0x60, 0xc3, 0x04,
    0x1f, 0x13, 0x28, 0xf8, 0x8c, 0x18, 0x2c, 0x00, 0x08, 0x82, 0xc1, 0x43,
    0x07, 0x60, 0x70, 0x08, 0x01, 0x65, 0x5d, 0x23, 0x06, 0x0b, 0x00, 0x82,
    0x60, 0xf0, 0xd4, 0x41, 0x18, 0x20, 0x83, 0x50, 0x65, 0xda, 0x88, 0xc1,
    0x02, 0x80, 0x20, 0x18, 0x3c, 0x76, 0x20, 0x06, 0x09, 0x31, 0x58, 0x1c,
    0x66, 0x43, 0x24, 0x1f, 0x1b, 0x22, 0xf9, 0xd8, 0x10, 0xc9, 0x67, 0xc4,
    0x20, 0x01, 0x40, 0x10, 0x0c, 0x10, 0x3e, 0x20, 0x83, 0x3b, 0xb8, 0x03,
    0x37, 0x18, 0x46, 0x0c, 0x12, 0x00, 0x04, 0xc1, 0x00, 0xe1, 0x03, 0x32,
    0xb8, 0x83, 0x3b, 0x38, 0x03, 0x61, 0xc4, 0x20, 0x01, 0x40, 0x10, 0x0c,
    0x10, 0x3e, 0x20, 0x83, 0x3b, 0xb8, 0x83, 0x36, 0x08, 0x46, 0x0c, 0x12,
    0x00, 0x04, 0xc1, 0x00, 0xe1, 0x03, 0x32, 0xb8, 0x83, 0x3b, 0x80, 0x83,
    0x08, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* The yuv-rendering pixel shader:

    --- D3D12_PixelShader_YUV_BT601.hlsl ---
    Texture2D theTextureY : register(t0);
    Texture2D theTextureU : register(t1);
    Texture2D theTextureV : register(t2);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define YUVRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t2), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(YUVRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        const float3 offset = {-0.0627451017, -0.501960814, -0.501960814};
        const float3 Rcoeff = {1.1644,  0.0000,  1.5960};
        const float3 Gcoeff = {1.1644, -0.3918, -0.8130};
        const float3 Bcoeff = {1.1644,  2.0172,  0.0000};

        float4 Output;

        float3 yuv;
        yuv.x = theTextureY.Sample(theSampler, input.tex).r;
        yuv.y = theTextureU.Sample(theSampler, input.tex).r;
        yuv.z = theTextureV.Sample(theSampler, input.tex).r;

        yuv += offset;
        Output.r = dot(yuv, Rcoeff);
        Output.g = dot(yuv, Gcoeff);
        Output.b = dot(yuv, Bcoeff);
        Output.a = 1.0f;

        return Output * input.color;
    }

*/

static unsigned char D3D12_PixelShader_YUV_BT601[] = {
    0x44, 0x58, 0x42, 0x43, 0x15, 0xda, 0x8f, 0x26, 0x88, 0x1f, 0xc5, 0x79,
    0x25, 0xc3, 0x83, 0x74, 0xb6, 0xe8, 0x4f, 0x43, 0x01, 0x00, 0x00, 0x00,
    0x99, 0x13, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0x1d, 0x02, 0x00, 0x00, 0x05, 0x03, 0x00, 0x00, 0x95, 0x0b, 0x00, 0x00,
    0xb1, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0x00, 0x01, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f,
    0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42,
    0x00, 0x03, 0x02, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x02, 0x44, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x52, 0x54, 0x53, 0x30, 0xe0, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xe0, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x01, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0xc8, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0x88, 0x08, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0x22, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x70, 0x08, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x19, 0x02, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4b, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x2c, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x39, 0x82,
    0xa0, 0x18, 0x52, 0x2c, 0x21, 0x2e, 0xe1, 0x81, 0x80, 0x61, 0x04, 0x01,
    0xb8, 0x4b, 0x9a, 0x22, 0x4a, 0x98, 0xfc, 0x14, 0xb9, 0x88, 0x85, 0x3d,
    0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x84, 0x1d, 0xd2, 0x41,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21,
    0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xc9, 0x01,
    0x00, 0xd3, 0x03, 0x00, 0x1e, 0xf2, 0x24, 0x40, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x61, 0x80, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0xe3, 0x00, 0x01, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x67, 0x02, 0x02,
    0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x4f, 0x05,
    0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e,
    0x0c, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2,
    0x40, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x5a, 0x05, 0x42, 0xa2,
    0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c,
    0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8, 0x3c, 0x8a, 0x87, 0x56,
    0x11, 0x8c, 0x00, 0x14, 0x42, 0x19, 0x94, 0x04, 0x9d, 0x19, 0x00, 0x2a,
    0x33, 0x00, 0x44, 0x66, 0x00, 0x68, 0xcc, 0x00, 0x50, 0x98, 0x01, 0x20,
    0x3e, 0x03, 0x40, 0x7d, 0x2c, 0x87, 0x21, 0x00, 0x00, 0x00, 0x8e, 0x03,
    0x00, 0x02, 0x81, 0x40, 0x00, 0x79, 0x18, 0x00, 0x00, 0xc4, 0x00, 0x00,
    0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63,
    0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03,
    0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a,
    0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b,
    0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0x61, 0x99, 0x20,
    0x08, 0xcc, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04, 0x41, 0x01, 0x6e, 0x6e,
    0x82, 0x20, 0x34, 0x1b, 0x86, 0x03, 0x21, 0x26, 0x08, 0x05, 0x18, 0x70,
    0xa1, 0x43, 0x2b, 0xa3, 0x2a, 0xc3, 0xa3, 0xab, 0x93, 0x2b, 0xcb, 0x9a,
    0x20, 0x08, 0xce, 0x04, 0x41, 0x78, 0x36, 0x08, 0x44, 0xb3, 0x21, 0x21,
    0x94, 0x85, 0x20, 0x06, 0x86, 0x70, 0xb8, 0xd0, 0xa1, 0x95, 0x51, 0x95,
    0xe1, 0xd1, 0xd5, 0xc9, 0x95, 0x55, 0x6d, 0x48, 0x06, 0x05, 0x22, 0x86,
    0x81, 0x21, 0x1c, 0x2e, 0x74, 0x68, 0x65, 0x54, 0x65, 0x78, 0x74, 0x75,
    0x72, 0x65, 0x56, 0x1b, 0x12, 0x46, 0x91, 0x08, 0x66, 0x60, 0x08, 0x67,
    0xc3, 0xf0, 0x44, 0xd3, 0x04, 0xe1, 0xf8, 0xa8, 0xd0, 0xa1, 0x95, 0x4d,
    0x85, 0xb5, 0xc1, 0xb1, 0x95, 0xc9, 0x6d, 0x40, 0x88, 0xca, 0x22, 0x88,
    0x81, 0x00, 0x36, 0x04, 0xd7, 0x06, 0x82, 0x02, 0x00, 0x6c, 0x82, 0x40,
    0x84, 0xc1, 0x04, 0x41, 0x80, 0x18, 0xa0, 0x4d, 0x10, 0x84, 0x68, 0x82,
    0x20, 0x48, 0x1b, 0x0c, 0x84, 0xeb, 0x08, 0xaf, 0x21, 0xd1, 0x96, 0x06,
    0x37, 0x37, 0x41, 0x10, 0xa6, 0x0d, 0x04, 0x02, 0x06, 0x5d, 0x18, 0x4c,
    0x10, 0x82, 0x31, 0xd8, 0x20, 0x10, 0x63, 0xb0, 0x21, 0x20, 0x83, 0x0d,
    0x02, 0x51, 0x06, 0x1b, 0x88, 0xed, 0x13, 0x03, 0x33, 0x98, 0x20, 0x0c,
    0x62, 0x30, 0x41, 0x10, 0x28, 0x1a, 0x68, 0x61, 0x6e, 0x64, 0x6c, 0x65,
    0x13, 0x04, 0xa1, 0xda, 0x60, 0x20, 0x6a, 0xd0, 0x11, 0xde, 0x1a, 0x6c,
    0x10, 0xd2, 0x80, 0x0d, 0x26, 0x08, 0x1c, 0x19, 0x4c, 0x10, 0x04, 0x8b,
    0x03, 0xdc, 0xdb, 0x1c, 0x97, 0x29, 0xab, 0x2f, 0xa8, 0xa7, 0xa9, 0x24,
    0xaa, 0xa4, 0x27, 0xa7, 0x0d, 0x08, 0x02, 0x07, 0x1d, 0x91, 0x06, 0x71,
    0xe0, 0x35, 0x1c, 0xe8, 0xca, 0xf0, 0x98, 0x50, 0x15, 0x61, 0x0d, 0x3d,
    0x3d, 0x49, 0x11, 0xc1, 0x6c, 0x40, 0x90, 0x39, 0xe8, 0xc2, 0x20, 0x0d,
    0xe8, 0xc0, 0x6b, 0x58, 0x8c, 0xbd, 0xb1, 0xbd, 0xc9, 0x4d, 0x10, 0x84,
    0x8b, 0xc6, 0xd0, 0x13, 0xd3, 0x93, 0x14, 0xcc, 0x06, 0x04, 0xb1, 0x83,
    0xee, 0x0e, 0xd2, 0x00, 0x0f, 0xbc, 0x66, 0x03, 0xf1, 0x06, 0x72, 0x50,
    0x07, 0x79, 0xb0, 0xe1, 0x20, 0xb4, 0x33, 0x40, 0x83, 0x36, 0x70, 0x03,
    0x3d, 0x98, 0x20, 0x28, 0xc4, 0x06, 0x60, 0xc3, 0x40, 0xf4, 0x41, 0x1f,
    0x6c, 0x08, 0xfc, 0x60, 0xc3, 0x30, 0xf0, 0xc1, 0x1f, 0x4c, 0x10, 0xba,
    0x32, 0xd8, 0x10, 0x84, 0x02, 0x89, 0xb6, 0xb0, 0x34, 0x37, 0x2e, 0x53,
    0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x13, 0x84,
    0xc5, 0x9b, 0x20, 0x2c, 0xdd, 0x86, 0x80, 0x98, 0x20, 0x2c, 0xdc, 0x04,
    0x61, 0xd9, 0x36, 0x2c, 0x04, 0x29, 0x94, 0x82, 0x29, 0x9c, 0x02, 0x2a,
    0x0c, 0xa8, 0x40, 0xa4, 0x02, 0x40, 0x84, 0xaa, 0x08, 0x6b, 0xe8, 0xe9,
    0x49, 0x8a, 0x68, 0x82, 0xb0, 0x68, 0x1b, 0x84, 0xae, 0xdb, 0xb0, 0x0c,
    0xab, 0x50, 0x0a, 0xa9, 0x70, 0x0a, 0xac, 0x30, 0xb0, 0xc2, 0x90, 0x0a,
    0xad, 0xc0, 0x62, 0xe8, 0x89, 0xe9, 0x49, 0x6a, 0x82, 0x20, 0x60, 0x1b,
    0x84, 0x0e, 0x16, 0x36, 0x2c, 0xcc, 0x2b, 0x94, 0x42, 0x2a, 0x9c, 0x02,
    0x2b, 0x0c, 0xa8, 0xc0, 0xa4, 0x42, 0x2c, 0x6c, 0x18, 0x54, 0xc1, 0x15,
    0x64, 0x81, 0xc9, 0x94, 0xd5, 0x17, 0x55, 0x98, 0xdc, 0x59, 0x19, 0xdd,
    0x04, 0x61, 0x51, 0x36, 0x2c, 0x04, 0x2d, 0x94, 0x42, 0x2d, 0x9c, 0x42,
    0x2a, 0x0c, 0xa8, 0x40, 0xa4, 0x42, 0x2c, 0x6c, 0x08, 0x6c, 0x61, 0xc3,
    0x30, 0x0b, 0xb7, 0x00, 0x6c, 0x28, 0xf8, 0x60, 0x14, 0x70, 0x21, 0x03,
    0x68, 0x98, 0xb1, 0xbd, 0x85, 0xd1, 0xcd, 0xb1, 0x48, 0x73, 0x9b, 0xa3,
    0x9b, 0x9b, 0x20, 0x08, 0x19, 0x8d, 0xb9, 0xb4, 0xb3, 0x2f, 0x36, 0x32,
    0x1a, 0x73, 0x69, 0x67, 0x5f, 0x73, 0x74, 0x44, 0xe8, 0xca, 0xf0, 0xbe,
    0xdc, 0xde, 0xe4, 0xda, 0x36, 0x28, 0xba, 0xd0, 0xec, 0x02, 0x2f, 0xf4,
    0x02, 0xe2, 0x0b, 0x69, 0xf0, 0x0b, 0x5d, 0x15, 0x36, 0x36, 0xbb, 0x36,
    0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf,
    0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40, 0x34, 0x21,
    0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x14, 0x75,
    0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8,
    0xca, 0xd8, 0xa6, 0x04, 0x48, 0x19, 0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9,
    0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0x01, 0x56, 0x89, 0x0c, 0xcf,
    0x85, 0x2e, 0x0f, 0xae, 0x2c, 0xc8, 0xcd, 0xed, 0x8d, 0x2e, 0x8c, 0x2e,
    0xed, 0xcd, 0x6d, 0x6e, 0x8a, 0xa0, 0x07, 0x7f, 0x50, 0x87, 0x0c, 0xcf,
    0xc5, 0x2e, 0xad, 0xec, 0x2e, 0x89, 0x6c, 0x8a, 0x2e, 0x8c, 0xae, 0x6c,
    0x4a, 0x10, 0x0a, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2,
    0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0xb8, 0xd0, 0x85, 0x0c,
    0xcf, 0x65, 0xec, 0xad, 0xce, 0x8d, 0xae, 0x4c, 0x6e, 0x6e, 0x4a, 0xf0,
    0x0b, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x3e, 0x00, 0x00,
    0x00, 0x35, 0xf0, 0x05, 0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9,
    0x65, 0x39, 0x10, 0x38, 0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f,
    0xd2, 0xc3, 0xf4, 0x32, 0x10, 0x18, 0xac, 0x00, 0x38, 0x08, 0xfc, 0xe8,
    0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7, 0xe5, 0xe5, 0x2a, 0x10, 0x38, 0xb3,
    0xfe, 0x48, 0xd4, 0x32, 0x9e, 0x5e, 0x97, 0x97, 0x65, 0x44, 0xa0, 0xf5,
    0x47, 0xb2, 0x97, 0xc7, 0xf4, 0xb7, 0x1c, 0xd8, 0x24, 0xc1, 0x66, 0x40,
    0x20, 0x10, 0x18, 0x2c, 0x01, 0x38, 0x08, 0xfc, 0xe8, 0xe8, 0x32, 0xb5,
    0x8c, 0xa7, 0xd7, 0xe5, 0x65, 0x2b, 0x10, 0x38, 0xb3, 0xfe, 0x48, 0xd4,
    0x32, 0x9e, 0x5e, 0x97, 0x97, 0x65, 0x44, 0xa0, 0xf5, 0x47, 0xb2, 0x97,
    0xc7, 0xf4, 0xb7, 0x1c, 0xd8, 0x24, 0xc1, 0x66, 0x40, 0x20, 0x10, 0x18,
    0x2c, 0x00, 0x38, 0x08, 0xfc, 0xe8, 0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7,
    0xe5, 0xe5, 0x2c, 0x10, 0x38, 0xb3, 0xfe, 0x48, 0xd4, 0x32, 0x9e, 0x5e,
    0x97, 0x97, 0x65, 0x44, 0xa0, 0xf5, 0x47, 0xb2, 0x97, 0xc7, 0xf4, 0xb7,
    0x1c, 0xd8, 0x24, 0xc1, 0x66, 0x40, 0x20, 0x10, 0x18, 0xb4, 0x04, 0x69,
    0xb8, 0x7c, 0xe7, 0xf1, 0x85, 0x88, 0x00, 0x26, 0x22, 0x04, 0x9a, 0x61,
    0x21, 0x0c, 0xc1, 0x19, 0x2e, 0xdf, 0x79, 0xfc, 0xc1, 0x99, 0x6e, 0xbf,
    0xb8, 0x6d, 0x2b, 0x98, 0x86, 0xcb, 0x77, 0x1e, 0x7f, 0x71, 0x80, 0x41,
    0x6c, 0x1e, 0x6a, 0xf2, 0x8b, 0xdb, 0xb6, 0x03, 0x68, 0xb8, 0x7c, 0xe7,
    0xf1, 0x25, 0x80, 0x79, 0x16, 0xc2, 0x2f, 0x6e, 0xdb, 0x0c, 0xaa, 0xe1,
    0xf2, 0x9d, 0xc7, 0x97, 0x26, 0x27, 0x22, 0x50, 0x6a, 0x7a, 0xa8, 0xc9,
    0x2f, 0x6e, 0xdb, 0x08, 0x08, 0x06, 0x40, 0x1a, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x57, 0xe4, 0xe0, 0xfd, 0x5b, 0x8a, 0xbf, 0x47, 0x55, 0x60, 0xe1,
    0xa5, 0xe1, 0x4c, 0x5a, 0xa7, 0x44, 0x58, 0x49, 0x4c, 0xe0, 0x07, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0xf8, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xc8, 0x07, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xef, 0x01, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x45, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23, 0x00, 0x25, 0x00, 0x14,
    0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84,
    0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0xa1, 0x9b, 0x86, 0xcb,
    0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x8b, 0xdb,
    0x46, 0xc5, 0x18, 0x63, 0x10, 0x2a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f,
    0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82, 0x55, 0x18, 0x45, 0x18,
    0x1b, 0x63, 0x0c, 0x42, 0xc8, 0xa0, 0x56, 0x90, 0x41, 0xc6, 0x18, 0x63,
    0x0c, 0x7a, 0x73, 0x04, 0x41, 0x31, 0x18, 0x29, 0x84, 0x44, 0x92, 0x03,
    0x01, 0xc3, 0x08, 0xc4, 0x30, 0x53, 0x1b, 0x8c, 0x03, 0x3b, 0x84, 0xc3,
    0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83,
    0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8, 0x43, 0x39, 0x8c, 0x03,
    0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03,
    0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80, 0x81, 0x1f, 0xe8, 0x81,
    0x1e, 0xb4, 0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc, 0x02, 0x3d, 0xe4, 0x03,
    0x3c, 0x94, 0x03, 0x0a, 0xc8, 0x4c, 0x62, 0x30, 0x0e, 0xec, 0x10, 0x0e,
    0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f,
    0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60, 0x0f, 0xe5, 0x30, 0x0e,
    0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xe6, 0xc0, 0x0e, 0xef, 0x10, 0x0e,
    0xf4, 0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00, 0x06, 0x7e, 0x80, 0x84,
    0x6a, 0xe9, 0xde, 0x24, 0x4d, 0x11, 0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf,
    0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x08, 0xe5, 0x74, 0x20, 0x00,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10,
    0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2,
    0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30,
    0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x40, 0x16, 0x08, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98,
    0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43,
    0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50, 0x12, 0x65, 0x50, 0x1e,
    0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0, 0x08, 0x0a, 0x84, 0xf0,
    0x0c, 0x00, 0xe9, 0x19, 0x00, 0xda, 0x63, 0x39, 0x0c, 0x01, 0x00, 0x00,
    0x70, 0x1c, 0x00, 0x10, 0x08, 0x04, 0x02, 0x00, 0x00, 0x79, 0x18, 0x00,
    0x00, 0x69, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
    0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b,
    0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b,
    0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9,
    0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13,
    0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61, 0x20, 0x26, 0x08, 0x04,
    0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26,
    0x08, 0xd8, 0x44, 0x60, 0x82, 0x40, 0x24, 0x13, 0x04, 0x42, 0xd9, 0x20,
    0x10, 0xcd, 0x86, 0x84, 0x50, 0x16, 0x82, 0x18, 0x18, 0xc2, 0xd9, 0x90,
    0x0c, 0xca, 0x42, 0x0c, 0x03, 0x43, 0x38, 0x1b, 0x12, 0x46, 0x59, 0x08,
    0x66, 0x60, 0x08, 0x67, 0xc3, 0xf0, 0x40, 0xd1, 0x04, 0x41, 0xa3, 0x36,
    0x20, 0xc4, 0xb4, 0x10, 0xc4, 0x40, 0x00, 0x1b, 0x02, 0x6a, 0x03, 0x21,
    0x01, 0x40, 0x35, 0x41, 0xd8, 0xaa, 0x0d, 0xc1, 0x35, 0x41, 0x10, 0x00,
    0x12, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6,
    0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0x45, 0x33, 0x41, 0x28, 0x9c,
    0x0d, 0x01, 0x31, 0x41, 0x28, 0x9e, 0x09, 0x42, 0x01, 0x6d, 0x58, 0x88,
    0x8d, 0xeb, 0xbc, 0x6f, 0xf8, 0x08, 0x30, 0x00, 0x88, 0x50, 0x15, 0x61,
    0x0d, 0x3d, 0x3d, 0x49, 0x11, 0x4d, 0x10, 0x8a, 0x68, 0x82, 0x40, 0x2c,
    0x1b, 0x04, 0x32, 0x20, 0x83, 0x0d, 0xcb, 0x20, 0x06, 0x1c, 0x18, 0x78,
    0x63, 0x30, 0x8c, 0xc1, 0x00, 0x06, 0x65, 0xc0, 0x62, 0xe8, 0x89, 0xe9,
    0x49, 0x6a, 0x82, 0x40, 0x30, 0x1b, 0x04, 0x32, 0x40, 0x83, 0x0d, 0x0b,
    0x73, 0x06, 0x1c, 0x18, 0x78, 0x63, 0x30, 0x7c, 0x0c, 0x18, 0xa4, 0xc1,
    0x86, 0x21, 0x0c, 0xcc, 0x40, 0x0d, 0x98, 0x4c, 0x59, 0x7d, 0x51, 0x85,
    0xc9, 0x9d, 0x95, 0xd1, 0x4d, 0x10, 0x0a, 0x69, 0xc3, 0x42, 0xb0, 0x01,
    0xd7, 0x06, 0x1e, 0x18, 0x0c, 0x1f, 0x01, 0x06, 0x69, 0xb0, 0x21, 0x70,
    0x83, 0x0d, 0xc3, 0x1a, 0xbc, 0x01, 0xb0, 0xa1, 0xc8, 0x34, 0x38, 0xb0,
    0x80, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74,
    0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda,
    0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66,
    0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85,
    0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x90, 0x32,
    0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73,
    0x53, 0x82, 0xaa, 0x0e, 0x19, 0x9e, 0x8b, 0x5d, 0x5a, 0xd9, 0x5d, 0x12,
    0xd9, 0x14, 0x5d, 0x18, 0x5d, 0xd9, 0x94, 0xe0, 0xaa, 0x43, 0x86, 0xe7,
    0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37,
    0x25, 0x80, 0x03, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00,
    0x00, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44,
    0x84, 0x40, 0x33, 0x2c, 0x84, 0x11, 0x38, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f,
    0x38, 0xd3, 0xed, 0x17, 0xb7, 0x6d, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3,
    0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x00,
    0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d,
    0x9b, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a,
    0x4d, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03,
    0x00, 0x61, 0x20, 0x00, 0x00, 0x69, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41,
    0x2c, 0x10, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x14, 0x47, 0x00,
    0xa8, 0x94, 0x00, 0x91, 0x72, 0x2b, 0xbc, 0x52, 0x28, 0xb9, 0x42, 0x98,
    0x01, 0xa0, 0x31, 0x46, 0xe0, 0xba, 0xa6, 0x08, 0x82, 0xc1, 0x18, 0x41,
    0x69, 0xa2, 0x60, 0xfd, 0x0b, 0x63, 0x04, 0x22, 0x6c, 0xc6, 0xec, 0x2f,
    0x8c, 0x11, 0xe8, 0xad, 0x8c, 0xf3, 0xdf, 0x18, 0x41, 0x48, 0x82, 0x21,
    0xee, 0x0b, 0x23, 0x00, 0x63, 0x04, 0x21, 0x09, 0x86, 0xf0, 0x2f, 0x8c,
    0x11, 0xbc, 0xb8, 0xa8, 0xe6, 0xdf, 0x0c, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x20, 0x89, 0x01, 0x73, 0x79, 0x5e, 0x34, 0x62, 0x90,
    0x00, 0x20, 0x08, 0x06, 0xd2, 0x18, 0x34, 0x98, 0x18, 0x88, 0x81, 0x34,
    0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x12, 0x19, 0x38, 0x99, 0x18, 0x88,
    0xc1, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x52, 0x19, 0x3c, 0xdc,
    0x18, 0x8c, 0x01, 0x35, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x1a,
    0x3c, 0x62, 0x40, 0x06, 0x9b, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06,
    0xc6, 0x1a, 0x40, 0x63, 0x50, 0x06, 0xd5, 0x33, 0x62, 0x90, 0x00, 0x20,
    0x08, 0x06, 0x06, 0x1b, 0x44, 0x64, 0x60, 0x06, 0x1c, 0x34, 0x62, 0x90,
    0x00, 0x20, 0x08, 0x06, 0x46, 0x1b, 0x48, 0x65, 0x70, 0x06, 0x60, 0x10,
    0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xe1, 0x06, 0x53, 0x1a, 0xa0,
    0xc1, 0x27, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xf1, 0x06, 0x94,
    0x1a, 0xa4, 0x41, 0x36, 0x8d, 0x18, 0x3c, 0x00, 0x08, 0x82, 0x41, 0xe3,
    0x06, 0x18, 0x72, 0x08, 0xc1, 0xb2, 0xa8, 0x81, 0x1a, 0x50, 0xcb, 0x68,
    0x42, 0x00, 0x8c, 0x18, 0x3c, 0x00, 0x08, 0x82, 0x41, 0x03, 0x07, 0xda,
    0x92, 0x10, 0x43, 0xd3, 0xb0, 0x01, 0x1b, 0x58, 0xcd, 0x68, 0x42, 0x00,
    0x8c, 0x18, 0x3c, 0x00, 0x08, 0x82, 0x41, 0x23, 0x07, 0x9c, 0xb3, 0x18,
    0xc5, 0xf3, 0xb8, 0x81, 0x1b, 0x60, 0xcf, 0x68, 0x42, 0x00, 0x58, 0x51,
    0xc1, 0xc7, 0x08, 0x0a, 0x3e, 0x36, 0x54, 0xf0, 0x19, 0x31, 0x58, 0x00,
    0x10, 0x04, 0x83, 0xc7, 0x0e, 0xc4, 0x60, 0x10, 0x82, 0xea, 0xca, 0x46,
    0x0c, 0x16, 0x00, 0x04, 0xc1, 0xe0, 0xb9, 0x83, 0x31, 0x20, 0x06, 0xc1,
    0xda, 0xb8, 0x11, 0x83, 0x05, 0x00, 0x41, 0x30, 0x78, 0xf0, 0x80, 0x0c,
    0x0a, 0x62, 0xb8, 0xbc, 0xcc, 0x06, 0x49, 0x3e, 0x36, 0x48, 0xf2, 0xb1,
    0x41, 0x92, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x7e, 0x60,
    0x06, 0x79, 0x90, 0x07, 0x70, 0x30, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82,
    0x01, 0xe2, 0x07, 0x66, 0x90, 0x07, 0x79, 0x90, 0x06, 0xc2, 0x88, 0x41,
    0x02, 0x80, 0x20, 0x18, 0x20, 0x7e, 0x60, 0x06, 0x79, 0x90, 0x07, 0x6f,
    0x10, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xe2, 0x07, 0x66, 0x90,
    0x07, 0x79, 0x20, 0x07, 0x12, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00
};

/* The yuv-rendering pixel shader:

    --- D3D12_PixelShader_YUV_BT709.hlsl ---
    Texture2D theTextureY : register(t0);
    Texture2D theTextureU : register(t1);
    Texture2D theTextureV : register(t2);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define YUVRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t2), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(YUVRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        const float3 offset = {-0.0627451017, -0.501960814, -0.501960814};
        const float3 Rcoeff = {1.1644,  0.0000,  1.7927};
        const float3 Gcoeff = {1.1644, -0.2132, -0.5329};
        const float3 Bcoeff = {1.1644,  2.1124,  0.0000};

        float4 Output;

        float3 yuv;
        yuv.x = theTextureY.Sample(theSampler, input.tex).r;
        yuv.y = theTextureU.Sample(theSampler, input.tex).r;
        yuv.z = theTextureV.Sample(theSampler, input.tex).r;

        yuv += offset;
        Output.r = dot(yuv, Rcoeff);
        Output.g = dot(yuv, Gcoeff);
        Output.b = dot(yuv, Bcoeff);
        Output.a = 1.0f;

        return Output * input.color;
    }

*/

static unsigned char D3D12_PixelShader_YUV_BT709[] = {
    0x44, 0x58, 0x42, 0x43, 0xe6, 0x97, 0xfb, 0x70, 0x81, 0x58, 0x82, 0x84,
    0xfd, 0xe7, 0xb6, 0x4d, 0x56, 0x90, 0xf8, 0x87, 0x01, 0x00, 0x00, 0x00,
    0x99, 0x13, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0x1d, 0x02, 0x00, 0x00, 0x05, 0x03, 0x00, 0x00, 0x95, 0x0b, 0x00, 0x00,
    0xb1, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0x00, 0x01, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f,
    0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42,
    0x00, 0x03, 0x02, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x02, 0x44, 0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x52, 0x54, 0x53, 0x30, 0xe0, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xe0, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x01, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0xc8, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0x88, 0x08, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0x22, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x70, 0x08, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x19, 0x02, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4b, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x2c, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x39, 0x82,
    0xa0, 0x18, 0x52, 0x2c, 0x21, 0x2e, 0xe1, 0x81, 0x80, 0x61, 0x04, 0x01,
    0xb8, 0x4b, 0x9a, 0x22, 0x4a, 0x98, 0xfc, 0x14, 0xb9, 0x88, 0x85, 0x3d,
    0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x84, 0x1d, 0xd2, 0x41,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21,
    0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xc9, 0x01,
    0x00, 0xd3, 0x03, 0x00, 0x1e, 0xf2, 0x24, 0x40, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x61, 0x80, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0xe3, 0x00, 0x01, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0x67, 0x02, 0x02,
    0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x4f, 0x05,
    0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x1e,
    0x0c, 0x08, 0x80, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb2,
    0x40, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x5a, 0x05, 0x42, 0xa2,
    0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c,
    0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8, 0x3c, 0x8a, 0x87, 0x56,
    0x11, 0x8c, 0x00, 0x14, 0x42, 0x19, 0x94, 0x04, 0x9d, 0x19, 0x00, 0x2a,
    0x33, 0x00, 0x44, 0x66, 0x00, 0x68, 0xcc, 0x00, 0x50, 0x98, 0x01, 0x20,
    0x3e, 0x03, 0x40, 0x7d, 0x2c, 0x87, 0x21, 0x00, 0x00, 0x00, 0x8e, 0x03,
    0x00, 0x02, 0x81, 0x40, 0x00, 0x79, 0x18, 0x00, 0x00, 0xc4, 0x00, 0x00,
    0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63,
    0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03,
    0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a,
    0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b,
    0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0x61, 0x99, 0x20,
    0x08, 0xcc, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04, 0x41, 0x01, 0x6e, 0x6e,
    0x82, 0x20, 0x34, 0x1b, 0x86, 0x03, 0x21, 0x26, 0x08, 0x05, 0x18, 0x70,
    0xa1, 0x43, 0x2b, 0xa3, 0x2a, 0xc3, 0xa3, 0xab, 0x93, 0x2b, 0xcb, 0x9a,
    0x20, 0x08, 0xce, 0x04, 0x41, 0x78, 0x36, 0x08, 0x44, 0xb3, 0x21, 0x21,
    0x94, 0x85, 0x20, 0x06, 0x86, 0x70, 0xb8, 0xd0, 0xa1, 0x95, 0x51, 0x95,
    0xe1, 0xd1, 0xd5, 0xc9, 0x95, 0x55, 0x6d, 0x48, 0x06, 0x05, 0x22, 0x86,
    0x81, 0x21, 0x1c, 0x2e, 0x74, 0x68, 0x65, 0x54, 0x65, 0x78, 0x74, 0x75,
    0x72, 0x65, 0x56, 0x1b, 0x12, 0x46, 0x91, 0x08, 0x66, 0x60, 0x08, 0x67,
    0xc3, 0xf0, 0x44, 0xd3, 0x04, 0xe1, 0xf8, 0xa8, 0xd0, 0xa1, 0x95, 0x4d,
    0x85, 0xb5, 0xc1, 0xb1, 0x95, 0xc9, 0x6d, 0x40, 0x88, 0xca, 0x22, 0x88,
    0x81, 0x00, 0x36, 0x04, 0xd7, 0x06, 0x82, 0x02, 0x00, 0x6c, 0x82, 0x40,
    0x84, 0xc1, 0x04, 0x41, 0x80, 0x18, 0xa0, 0x4d, 0x10, 0x84, 0x68, 0x82,
    0x20, 0x48, 0x1b, 0x0c, 0x84, 0xeb, 0x08, 0xaf, 0x21, 0xd1, 0x96, 0x06,
    0x37, 0x37, 0x41, 0x10, 0xa6, 0x0d, 0x04, 0x02, 0x06, 0x5d, 0x18, 0x4c,
    0x10, 0x82, 0x31, 0xd8, 0x20, 0x10, 0x63, 0xb0, 0x21, 0x20, 0x83, 0x0d,
    0x02, 0x51, 0x06, 0x1b, 0x88, 0xed, 0x13, 0x03, 0x33, 0x98, 0x20, 0x0c,
    0x62, 0x30, 0x41, 0x10, 0x28, 0x1a, 0x68, 0x61, 0x6e, 0x64, 0x6c, 0x65,
    0x13, 0x04, 0xa1, 0xda, 0x60, 0x20, 0x6a, 0xd0, 0x11, 0xde, 0x1a, 0x6c,
    0x10, 0xd2, 0x80, 0x0d, 0x26, 0x08, 0x1c, 0x19, 0x4c, 0x10, 0x04, 0x8b,
    0x03, 0xdc, 0xdb, 0x1c, 0x97, 0x29, 0xab, 0x2f, 0xa8, 0xa7, 0xa9, 0x24,
    0xaa, 0xa4, 0x27, 0xa7, 0x0d, 0x08, 0x02, 0x07, 0x1d, 0x91, 0x06, 0x71,
    0xe0, 0x35, 0x1c, 0xe8, 0xca, 0xf0, 0x98, 0x50, 0x15, 0x61, 0x0d, 0x3d,
    0x3d, 0x49, 0x11, 0xc1, 0x6c, 0x40, 0x90, 0x39, 0xe8, 0xc2, 0x20, 0x0d,
    0xe8, 0xc0, 0x6b, 0x58, 0x8c, 0xbd, 0xb1, 0xbd, 0xc9, 0x4d, 0x10, 0x84,
    0x8b, 0xc6, 0xd0, 0x13, 0xd3, 0x93, 0x14, 0xcc, 0x06, 0x04, 0xb1, 0x83,
    0xee, 0x0e, 0xd2, 0x00, 0x0f, 0xbc, 0x66, 0x03, 0xf1, 0x06, 0x72, 0x50,
    0x07, 0x79, 0xb0, 0xe1, 0x20, 0xb4, 0x33, 0x40, 0x83, 0x36, 0x70, 0x03,
    0x3d, 0x98, 0x20, 0x28, 0xc4, 0x06, 0x60, 0xc3, 0x40, 0xf4, 0x41, 0x1f,
    0x6c, 0x08, 0xfc, 0x60, 0xc3, 0x30, 0xf0, 0xc1, 0x1f, 0x4c, 0x10, 0xba,
    0x32, 0xd8, 0x10, 0x84, 0x02, 0x89, 0xb6, 0xb0, 0x34, 0x37, 0x2e, 0x53,
    0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x13, 0x84,
    0xc5, 0x9b, 0x20, 0x2c, 0xdd, 0x86, 0x80, 0x98, 0x20, 0x2c, 0xdc, 0x04,
    0x61, 0xd9, 0x36, 0x2c, 0x04, 0x29, 0x94, 0x82, 0x29, 0x9c, 0x02, 0x2a,
    0x0c, 0xa8, 0x40, 0xa4, 0x02, 0x40, 0x84, 0xaa, 0x08, 0x6b, 0xe8, 0xe9,
    0x49, 0x8a, 0x68, 0x82, 0xb0, 0x68, 0x1b, 0x84, 0xae, 0xdb, 0xb0, 0x0c,
    0xab, 0x50, 0x0a, 0xa9, 0x70, 0x0a, 0xac, 0x30, 0xb0, 0xc2, 0x90, 0x0a,
    0xad, 0xc0, 0x62, 0xe8, 0x89, 0xe9, 0x49, 0x6a, 0x82, 0x20, 0x60, 0x1b,
    0x84, 0x0e, 0x16, 0x36, 0x2c, 0xcc, 0x2b, 0x94, 0x42, 0x2a, 0x9c, 0x02,
    0x2b, 0x0c, 0xa8, 0xc0, 0xa4, 0x42, 0x2c, 0x6c, 0x18, 0x54, 0xc1, 0x15,
    0x64, 0x81, 0xc9, 0x94, 0xd5, 0x17, 0x55, 0x98, 0xdc, 0x59, 0x19, 0xdd,
    0x04, 0x61, 0x51, 0x36, 0x2c, 0x04, 0x2d, 0x94, 0x42, 0x2d, 0x9c, 0x42,
    0x2a, 0x0c, 0xa8, 0x40, 0xa4, 0x42, 0x2c, 0x6c, 0x08, 0x6c, 0x61, 0xc3,
    0x30, 0x0b, 0xb7, 0x00, 0x6c, 0x28, 0xf8, 0x60, 0x14, 0x70, 0x21, 0x03,
    0x68, 0x98, 0xb1, 0xbd, 0x85, 0xd1, 0xcd, 0xb1, 0x48, 0x73, 0x9b, 0xa3,
    0x9b, 0x9b, 0x20, 0x08, 0x19, 0x8d, 0xb9, 0xb4, 0xb3, 0x2f, 0x36, 0x32,
    0x1a, 0x73, 0x69, 0x67, 0x5f, 0x73, 0x74, 0x44, 0xe8, 0xca, 0xf0, 0xbe,
    0xdc, 0xde, 0xe4, 0xda, 0x36, 0x28, 0xba, 0xd0, 0xec, 0x02, 0x2f, 0xf4,
    0x02, 0xe2, 0x0b, 0x69, 0xf0, 0x0b, 0x5d, 0x15, 0x36, 0x36, 0xbb, 0x36,
    0x97, 0x34, 0xb2, 0x32, 0x37, 0xba, 0x29, 0x41, 0x50, 0x85, 0x0c, 0xcf,
    0xc5, 0xae, 0x4c, 0x6e, 0x2e, 0xed, 0xcd, 0x6d, 0x4a, 0x40, 0x34, 0x21,
    0xc3, 0x73, 0xb1, 0x0b, 0x63, 0xb3, 0x2b, 0x93, 0x9b, 0x12, 0x14, 0x75,
    0xc8, 0xf0, 0x5c, 0xe6, 0xd0, 0xc2, 0xc8, 0xca, 0xe4, 0x9a, 0xde, 0xc8,
    0xca, 0xd8, 0xa6, 0x04, 0x48, 0x19, 0x32, 0x3c, 0x17, 0xb9, 0xb2, 0xb9,
    0xb7, 0x3a, 0xb9, 0xb1, 0xb2, 0xb9, 0x29, 0x01, 0x56, 0x89, 0x0c, 0xcf,
    0x85, 0x2e, 0x0f, 0xae, 0x2c, 0xc8, 0xcd, 0xed, 0x8d, 0x2e, 0x8c, 0x2e,
    0xed, 0xcd, 0x6d, 0x6e, 0x8a, 0xa0, 0x07, 0x7f, 0x50, 0x87, 0x0c, 0xcf,
    0xc5, 0x2e, 0xad, 0xec, 0x2e, 0x89, 0x6c, 0x8a, 0x2e, 0x8c, 0xae, 0x6c,
    0x4a, 0x10, 0x0a, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2,
    0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0xb8, 0xd0, 0x85, 0x0c,
    0xcf, 0x65, 0xec, 0xad, 0xce, 0x8d, 0xae, 0x4c, 0x6e, 0x6e, 0x4a, 0xf0,
    0x0b, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x3e, 0x00, 0x00,
    0x00, 0x35, 0xf0, 0x05, 0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9,
    0x65, 0x39, 0x10, 0x38, 0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f,
    0xd2, 0xc3, 0xf4, 0x32, 0x10, 0x18, 0xac, 0x00, 0x38, 0x08, 0xfc, 0xe8,
    0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7, 0xe5, 0xe5, 0x2a, 0x10, 0x38, 0xb3,
    0xfe, 0x48, 0xd4, 0x32, 0x9e, 0x5e, 0x97, 0x97, 0x65, 0x44, 0xa0, 0xf5,
    0x47, 0xb2, 0x97, 0xc7, 0xf4, 0xb7, 0x1c, 0xd8, 0x24, 0xc1, 0x66, 0x40,
    0x20, 0x10, 0x18, 0x2c, 0x01, 0x38, 0x08, 0xfc, 0xe8, 0xe8, 0x32, 0xb5,
    0x8c, 0xa7, 0xd7, 0xe5, 0x65, 0x2b, 0x10, 0x38, 0xb3, 0xfe, 0x48, 0xd4,
    0x32, 0x9e, 0x5e, 0x97, 0x97, 0x65, 0x44, 0xa0, 0xf5, 0x47, 0xb2, 0x97,
    0xc7, 0xf4, 0xb7, 0x1c, 0xd8, 0x24, 0xc1, 0x66, 0x40, 0x20, 0x10, 0x18,
    0x2c, 0x00, 0x38, 0x08, 0xfc, 0xe8, 0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7,
    0xe5, 0xe5, 0x2c, 0x10, 0x38, 0xb3, 0xfe, 0x48, 0xd4, 0x32, 0x9e, 0x5e,
    0x97, 0x97, 0x65, 0x44, 0xa0, 0xf5, 0x47, 0xb2, 0x97, 0xc7, 0xf4, 0xb7,
    0x1c, 0xd8, 0x24, 0xc1, 0x66, 0x40, 0x20, 0x10, 0x18, 0xb4, 0x04, 0x69,
    0xb8, 0x7c, 0xe7, 0xf1, 0x85, 0x88, 0x00, 0x26, 0x22, 0x04, 0x9a, 0x61,
    0x21, 0x0c, 0xc1, 0x19, 0x2e, 0xdf, 0x79, 0xfc, 0xc1, 0x99, 0x6e, 0xbf,
    0xb8, 0x6d, 0x2b, 0x98, 0x86, 0xcb, 0x77, 0x1e, 0x7f, 0x71, 0x80, 0x41,
    0x6c, 0x1e, 0x6a, 0xf2, 0x8b, 0xdb, 0xb6, 0x03, 0x68, 0xb8, 0x7c, 0xe7,
    0xf1, 0x25, 0x80, 0x79, 0x16, 0xc2, 0x2f, 0x6e, 0xdb, 0x0c, 0xaa, 0xe1,
    0xf2, 0x9d, 0xc7, 0x97, 0x26, 0x27, 0x22, 0x50, 0x6a, 0x7a, 0xa8, 0xc9,
    0x2f, 0x6e, 0xdb, 0x08, 0x08, 0x06, 0x40, 0x1a, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x28, 0xea, 0x91, 0x74, 0x07, 0xef, 0x37, 0x34, 0xc5, 0x63, 0xc4,
    0x30, 0x49, 0xbf, 0x93, 0x8b, 0x44, 0x58, 0x49, 0x4c, 0xe0, 0x07, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0xf8, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xc8, 0x07, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xef, 0x01, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x45, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23, 0x00, 0x25, 0x00, 0x14,
    0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84,
    0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0xa1, 0x9b, 0x86, 0xcb,
    0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x8b, 0xdb,
    0x46, 0xc5, 0x18, 0x63, 0x10, 0x2a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f,
    0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82, 0x55, 0x18, 0x45, 0x18,
    0x1b, 0x63, 0x0c, 0x42, 0xc8, 0xa0, 0x56, 0x90, 0x41, 0xc6, 0x18, 0x63,
    0x0c, 0x7a, 0x73, 0x04, 0x41, 0x31, 0x18, 0x29, 0x84, 0x44, 0x92, 0x03,
    0x01, 0xc3, 0x08, 0xc4, 0x30, 0x53, 0x1b, 0x8c, 0x03, 0x3b, 0x84, 0xc3,
    0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83,
    0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8, 0x43, 0x39, 0x8c, 0x03,
    0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03,
    0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80, 0x81, 0x1f, 0xe8, 0x81,
    0x1e, 0xb4, 0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc, 0x02, 0x3d, 0xe4, 0x03,
    0x3c, 0x94, 0x03, 0x0a, 0xc8, 0x4c, 0x62, 0x30, 0x0e, 0xec, 0x10, 0x0e,
    0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f,
    0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60, 0x0f, 0xe5, 0x30, 0x0e,
    0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xe6, 0xc0, 0x0e, 0xef, 0x10, 0x0e,
    0xf4, 0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00, 0x06, 0x7e, 0x80, 0x84,
    0x6a, 0xe9, 0xde, 0x24, 0x4d, 0x11, 0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf,
    0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x08, 0xe5, 0x74, 0x20, 0x00,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10,
    0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2,
    0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30,
    0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x40, 0x16, 0x08, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98,
    0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43,
    0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50, 0x12, 0x65, 0x50, 0x1e,
    0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0, 0x08, 0x0a, 0x84, 0xf0,
    0x0c, 0x00, 0xe9, 0x19, 0x00, 0xda, 0x63, 0x39, 0x0c, 0x01, 0x00, 0x00,
    0x70, 0x1c, 0x00, 0x10, 0x08, 0x04, 0x02, 0x00, 0x00, 0x79, 0x18, 0x00,
    0x00, 0x69, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
    0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b,
    0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b,
    0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9,
    0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13,
    0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61, 0x20, 0x26, 0x08, 0x04,
    0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26,
    0x08, 0xd8, 0x44, 0x60, 0x82, 0x40, 0x24, 0x13, 0x04, 0x42, 0xd9, 0x20,
    0x10, 0xcd, 0x86, 0x84, 0x50, 0x16, 0x82, 0x18, 0x18, 0xc2, 0xd9, 0x90,
    0x0c, 0xca, 0x42, 0x0c, 0x03, 0x43, 0x38, 0x1b, 0x12, 0x46, 0x59, 0x08,
    0x66, 0x60, 0x08, 0x67, 0xc3, 0xf0, 0x40, 0xd1, 0x04, 0x41, 0xa3, 0x36,
    0x20, 0xc4, 0xb4, 0x10, 0xc4, 0x40, 0x00, 0x1b, 0x02, 0x6a, 0x03, 0x21,
    0x01, 0x40, 0x35, 0x41, 0xd8, 0xaa, 0x0d, 0xc1, 0x35, 0x41, 0x10, 0x00,
    0x12, 0x6d, 0x61, 0x69, 0x6e, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0xde, 0xe6,
    0xd2, 0xe8, 0xd2, 0xde, 0xdc, 0x26, 0x08, 0x45, 0x33, 0x41, 0x28, 0x9c,
    0x0d, 0x01, 0x31, 0x41, 0x28, 0x9e, 0x09, 0x42, 0x01, 0x6d, 0x58, 0x88,
    0x8d, 0xeb, 0xbc, 0x6f, 0xf8, 0x08, 0x30, 0x00, 0x88, 0x50, 0x15, 0x61,
    0x0d, 0x3d, 0x3d, 0x49, 0x11, 0x4d, 0x10, 0x8a, 0x68, 0x82, 0x40, 0x2c,
    0x1b, 0x04, 0x32, 0x20, 0x83, 0x0d, 0xcb, 0x20, 0x06, 0x1c, 0x18, 0x78,
    0x63, 0x30, 0x8c, 0xc1, 0x00, 0x06, 0x65, 0xc0, 0x62, 0xe8, 0x89, 0xe9,
    0x49, 0x6a, 0x82, 0x40, 0x30, 0x1b, 0x04, 0x32, 0x40, 0x83, 0x0d, 0x0b,
    0x73, 0x06, 0x1c, 0x18, 0x78, 0x63, 0x30, 0x7c, 0x0c, 0x18, 0xa4, 0xc1,
    0x86, 0x21, 0x0c, 0xcc, 0x40, 0x0d, 0x98, 0x4c, 0x59, 0x7d, 0x51, 0x85,
    0xc9, 0x9d, 0x95, 0xd1, 0x4d, 0x10, 0x0a, 0x69, 0xc3, 0x42, 0xb0, 0x01,
    0xd7, 0x06, 0x1e, 0x18, 0x0c, 0x1f, 0x01, 0x06, 0x69, 0xb0, 0x21, 0x70,
    0x83, 0x0d, 0xc3, 0x1a, 0xbc, 0x01, 0xb0, 0xa1, 0xc8, 0x34, 0x38, 0xb0,
    0x80, 0x2a, 0x6c, 0x6c, 0x76, 0x6d, 0x2e, 0x69, 0x64, 0x65, 0x6e, 0x74,
    0x53, 0x82, 0xa0, 0x0a, 0x19, 0x9e, 0x8b, 0x5d, 0x99, 0xdc, 0x5c, 0xda,
    0x9b, 0xdb, 0x94, 0x80, 0x68, 0x42, 0x86, 0xe7, 0x62, 0x17, 0xc6, 0x66,
    0x57, 0x26, 0x37, 0x25, 0x30, 0xea, 0x90, 0xe1, 0xb9, 0xcc, 0xa1, 0x85,
    0x91, 0x95, 0xc9, 0x35, 0xbd, 0x91, 0x95, 0xb1, 0x4d, 0x09, 0x90, 0x32,
    0x64, 0x78, 0x2e, 0x72, 0x65, 0x73, 0x6f, 0x75, 0x72, 0x63, 0x65, 0x73,
    0x53, 0x82, 0xaa, 0x0e, 0x19, 0x9e, 0x8b, 0x5d, 0x5a, 0xd9, 0x5d, 0x12,
    0xd9, 0x14, 0x5d, 0x18, 0x5d, 0xd9, 0x94, 0xe0, 0xaa, 0x43, 0x86, 0xe7,
    0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96, 0xe6, 0x46, 0x37, 0x37,
    0x25, 0x80, 0x03, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00,
    0x00, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44,
    0x84, 0x40, 0x33, 0x2c, 0x84, 0x11, 0x38, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f,
    0x38, 0xd3, 0xed, 0x17, 0xb7, 0x6d, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3,
    0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x00,
    0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d,
    0x9b, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a,
    0x4d, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03,
    0x00, 0x61, 0x20, 0x00, 0x00, 0x69, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41,
    0x2c, 0x10, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x14, 0x47, 0x00,
    0xa8, 0x94, 0x00, 0x91, 0x72, 0x2b, 0xbc, 0x52, 0x28, 0xb9, 0x42, 0x98,
    0x01, 0xa0, 0x31, 0x46, 0x00, 0xb3, 0xac, 0x0b, 0x82, 0xc1, 0x18, 0x81,
    0x18, 0xee, 0x30, 0xfc, 0x0b, 0x63, 0x04, 0x3a, 0x4a, 0xd3, 0xe5, 0x2f,
    0x8c, 0x11, 0xc8, 0x79, 0xaf, 0xfa, 0xdf, 0x18, 0x41, 0x48, 0x82, 0x21,
    0xee, 0x0b, 0x23, 0x00, 0x63, 0x04, 0x21, 0x09, 0x86, 0xf0, 0x2f, 0x8c,
    0x11, 0xbc, 0xb8, 0xa8, 0xe6, 0xdf, 0x0c, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x20, 0x89, 0x01, 0x73, 0x79, 0x5e, 0x34, 0x62, 0x90,
    0x00, 0x20, 0x08, 0x06, 0xd2, 0x18, 0x34, 0x98, 0x18, 0x88, 0x81, 0x34,
    0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x12, 0x19, 0x38, 0x99, 0x18, 0x88,
    0xc1, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x52, 0x19, 0x3c, 0xdc,
    0x18, 0x8c, 0x01, 0x35, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x1a,
    0x3c, 0x62, 0x40, 0x06, 0x9b, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06,
    0xc6, 0x1a, 0x40, 0x63, 0x50, 0x06, 0xd5, 0x33, 0x62, 0x90, 0x00, 0x20,
    0x08, 0x06, 0x06, 0x1b, 0x44, 0x64, 0x60, 0x06, 0x1c, 0x34, 0x62, 0x90,
    0x00, 0x20, 0x08, 0x06, 0x46, 0x1b, 0x48, 0x65, 0x70, 0x06, 0x60, 0x10,
    0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xe1, 0x06, 0x53, 0x1a, 0xa0,
    0xc1, 0x27, 0x8d, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xf1, 0x06, 0x94,
    0x1a, 0xa4, 0x41, 0x36, 0x8d, 0x18, 0x3c, 0x00, 0x08, 0x82, 0x41, 0xe3,
    0x06, 0x18, 0x72, 0x08, 0xc1, 0xb2, 0xa8, 0x81, 0x1a, 0x50, 0xcb, 0x68,
    0x42, 0x00, 0x8c, 0x18, 0x3c, 0x00, 0x08, 0x82, 0x41, 0x03, 0x07, 0xda,
    0x92, 0x10, 0x43, 0xd3, 0xb0, 0x01, 0x1b, 0x58, 0xcd, 0x68, 0x42, 0x00,
    0x8c, 0x18, 0x3c, 0x00, 0x08, 0x82, 0x41, 0x23, 0x07, 0x9c, 0xb3, 0x18,
    0xc5, 0xf3, 0xb8, 0x81, 0x1b, 0x60, 0xcf, 0x68, 0x42, 0x00, 0x58, 0x51,
    0xc1, 0xc7, 0x08, 0x0a, 0x3e, 0x36, 0x54, 0xf0, 0x19, 0x31, 0x58, 0x00,
    0x10, 0x04, 0x83, 0xc7, 0x0e, 0xc4, 0x60, 0x10, 0x82, 0xea, 0xca, 0x46,
    0x0c, 0x16, 0x00, 0x04, 0xc1, 0xe0, 0xb9, 0x83, 0x31, 0x20, 0x06, 0xc1,
    0xda, 0xb8, 0x11, 0x83, 0x05, 0x00, 0x41, 0x30, 0x78, 0xf0, 0x80, 0x0c,
    0x0a, 0x62, 0xb8, 0xbc, 0xcc, 0x06, 0x49, 0x3e, 0x36, 0x48, 0xf2, 0xb1,
    0x41, 0x92, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20, 0x7e, 0x60,
    0x06, 0x79, 0x90, 0x07, 0x70, 0x30, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82,
    0x01, 0xe2, 0x07, 0x66, 0x90, 0x07, 0x79, 0x90, 0x06, 0xc2, 0x88, 0x41,
    0x02, 0x80, 0x20, 0x18, 0x20, 0x7e, 0x60, 0x06, 0x79, 0x90, 0x07, 0x6f,
    0x10, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xe2, 0x07, 0x66, 0x90,
    0x07, 0x79, 0x20, 0x07, 0x12, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00
};

/* The yuv-rendering pixel shader:

    --- D3D12_PixelShader_NV12_JPEG.hlsl ---
    Texture2D theTextureY : register(t0);
    Texture2D theTextureUV : register(t1);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define NVRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(NVRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        const float3 offset = {0.0, -0.501960814, -0.501960814};
        const float3 Rcoeff = {1.0000,  0.0000,  1.4020};
        const float3 Gcoeff = {1.0000, -0.3441, -0.7141};
        const float3 Bcoeff = {1.0000,  1.7720,  0.0000};

        float4 Output;

        float3 yuv;
        yuv.x = theTextureY.Sample(theSampler, input.tex).r;
        yuv.yz = theTextureUV.Sample(theSampler, input.tex).rg;

        yuv += offset;
        Output.r = dot(yuv, Rcoeff);
        Output.g = dot(yuv, Gcoeff);
        Output.b = dot(yuv, Bcoeff);
        Output.a = 1.0f;

        return Output * input.color;
    }

*/

static unsigned char D3D12_PixelShader_NV12_JPEG[] = {
    0x44, 0x58, 0x42, 0x43, 0x4d, 0x84, 0xab, 0xd3, 0xf9, 0x2a, 0x29, 0x46,
    0xc8, 0x39, 0x22, 0xd6, 0x7a, 0xdf, 0x22, 0x16, 0x01, 0x00, 0x00, 0x00,
    0xe1, 0x12, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0x0d, 0x02, 0x00, 0x00, 0xc9, 0x02, 0x00, 0x00, 0x1d, 0x0b, 0x00, 0x00,
    0x39, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0xf0, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43,
    0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00,
    0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44,
    0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x52, 0x54, 0x53,
    0x30, 0xb4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00,
    0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x01, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0x9c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0x4c, 0x08, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0x13, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x34, 0x08, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x0a, 0x02, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4b, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x2c, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x39, 0x82,
    0xa0, 0x18, 0x52, 0x2c, 0x21, 0x2e, 0xe1, 0x81, 0x80, 0x61, 0x04, 0x01,
    0xb8, 0x4b, 0x9a, 0x22, 0x4a, 0x98, 0xfc, 0x14, 0xb9, 0x88, 0x85, 0x3d,
    0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x84, 0x1d, 0xd2, 0x41,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21,
    0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xe9, 0x01,
    0x00, 0x0f, 0x79, 0x12, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0xc0,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00,
    0x00, 0x16, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x5a, 0x05, 0x42, 0xa2,
    0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c,
    0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8, 0x3c, 0x0a, 0xa2, 0x6c,
    0x68, 0x15, 0xc1, 0x08, 0x40, 0x21, 0x94, 0x41, 0x49, 0xd0, 0x99, 0x01,
    0xa0, 0x32, 0x03, 0x40, 0x64, 0x06, 0x80, 0xc6, 0x0c, 0x00, 0x85, 0x19,
    0x00, 0xe2, 0x33, 0x00, 0xd4, 0xc7, 0x72, 0x18, 0x02, 0x00, 0x00, 0xe0,
    0x38, 0x00, 0x20, 0x10, 0x08, 0x04, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00,
    0x00, 0xbf, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
    0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b,
    0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b,
    0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9,
    0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13,
    0x04, 0x41, 0x99, 0x20, 0x08, 0xcb, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04,
    0x41, 0x01, 0x6e, 0x6e, 0x82, 0x20, 0x30, 0x1b, 0x86, 0x03, 0x21, 0x26,
    0x08, 0x05, 0x18, 0x70, 0xa1, 0x43, 0x2b, 0xa3, 0x2a, 0xc3, 0xa3, 0xab,
    0x93, 0x2b, 0xcb, 0x9a, 0x20, 0x08, 0xcd, 0x04, 0x41, 0x70, 0x36, 0x08,
    0x44, 0xb3, 0x21, 0x21, 0x94, 0x85, 0x20, 0x06, 0x86, 0x70, 0xc8, 0xd0,
    0xa1, 0x95, 0x51, 0x95, 0xe1, 0xd1, 0xd5, 0xc9, 0x95, 0x55, 0x59, 0x6d,
    0x48, 0x06, 0x05, 0x22, 0x86, 0x81, 0x21, 0x9c, 0x0d, 0xc2, 0x13, 0x4d,
    0x10, 0x8e, 0x8f, 0x0a, 0x1d, 0x5a, 0xd9, 0x54, 0x58, 0x1b, 0x1c, 0x5b,
    0x99, 0xdc, 0x06, 0x84, 0x98, 0x28, 0x82, 0x18, 0x08, 0x60, 0x43, 0x50,
    0x6d, 0x20, 0x24, 0x00, 0xb0, 0x26, 0x08, 0x44, 0x18, 0x4c, 0x10, 0x84,
    0x87, 0x01, 0xda, 0x04, 0x41, 0x80, 0x26, 0x08, 0x42, 0xb4, 0xc1, 0x40,
    0xb4, 0x8d, 0xe0, 0x1a, 0x12, 0x6d, 0x69, 0x70, 0x73, 0x13, 0x04, 0x41,
    0xda, 0x40, 0x20, 0xde, 0xf6, 0x4d, 0x10, 0x82, 0x31, 0xd8, 0x20, 0x10,
    0x61, 0xb0, 0x21, 0x10, 0x83, 0x0d, 0x02, 0x31, 0x06, 0x1b, 0x88, 0xac,
    0x03, 0x03, 0x32, 0x98, 0x20, 0x0c, 0x62, 0x30, 0x41, 0x10, 0x26, 0x1a,
    0x68, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x13, 0x04, 0x81, 0xda, 0x60, 0x20,
    0x68, 0xb0, 0x11, 0x5c, 0x1a, 0x6c, 0x10, 0xce, 0x40, 0x0d, 0x26, 0x08,
    0x1c, 0x19, 0x4c, 0x10, 0x84, 0x8a, 0x03, 0xdc, 0xdb, 0x1c, 0x97, 0x29,
    0xab, 0x2f, 0xa8, 0xa7, 0xa9, 0x24, 0xaa, 0xa4, 0x27, 0xa7, 0x0d, 0x08,
    0xe2, 0x06, 0x1b, 0x71, 0x06, 0x6f, 0xc0, 0x35, 0x1c, 0xe8, 0xca, 0xf0,
    0x98, 0x50, 0x15, 0x61, 0x0d, 0x3d, 0x3d, 0x49, 0x11, 0xc1, 0x6c, 0x40,
    0x90, 0x38, 0xd8, 0xbe, 0x33, 0x90, 0x03, 0xae, 0x61, 0x31, 0xf6, 0xc6,
    0xf6, 0x26, 0x37, 0x41, 0x10, 0x2c, 0x1a, 0x43, 0x4f, 0x4c, 0x4f, 0x52,
    0x30, 0x1b, 0x10, 0x84, 0x0e, 0xb6, 0x3a, 0x38, 0x03, 0x3b, 0xe0, 0x9a,
    0x0d, 0x44, 0x1b, 0xc0, 0xc1, 0x1c, 0xdc, 0xc1, 0x86, 0x83, 0xc0, 0xca,
    0xc0, 0x0c, 0xd6, 0x80, 0x0d, 0xf0, 0x60, 0x82, 0xa0, 0x0c, 0x1b, 0x80,
    0x0d, 0x03, 0xb1, 0x07, 0x7b, 0xb0, 0x21, 0xe0, 0x83, 0x0d, 0xc3, 0xa0,
    0x07, 0x7d, 0x30, 0x41, 0xe8, 0xca, 0x60, 0x43, 0xf0, 0x07, 0x24, 0xda,
    0xc2, 0xd2, 0xdc, 0xb8, 0x4c, 0x59, 0x7d, 0x41, 0xbd, 0xcd, 0xa5, 0xd1,
    0xa5, 0xbd, 0xb9, 0x4d, 0x10, 0x16, 0x6f, 0x82, 0xb0, 0x74, 0x1b, 0x02,
    0x62, 0x82, 0xb0, 0x70, 0x13, 0x84, 0x65, 0xdb, 0xb0, 0x10, 0xa2, 0x30,
    0x0a, 0xa4, 0x50, 0x0a, 0xa6, 0x30, 0x98, 0x02, 0x71, 0x0a, 0x00, 0x11,
    0xaa, 0x22, 0xac, 0xa1, 0xa7, 0x27, 0x29, 0xa2, 0x09, 0xc2, 0xa2, 0x6d,
    0x10, 0xb6, 0x6d, 0xc3, 0x32, 0xa4, 0xc2, 0x28, 0x9c, 0x42, 0x29, 0xa8,
    0xc2, 0xa0, 0x0a, 0xc3, 0x29, 0xac, 0x02, 0x8b, 0xa1, 0x27, 0xa6, 0x27,
    0xa9, 0x09, 0x82, 0x70, 0x6d, 0x10, 0x36, 0x57, 0xd8, 0xb0, 0x30, 0xad,
    0x30, 0x0a, 0xa7, 0x50, 0x0a, 0xaa, 0x30, 0x98, 0x02, 0x73, 0x0a, 0xaf,
    0xb0, 0x61, 0x40, 0x05, 0x56, 0x80, 0x05, 0x26, 0x53, 0x56, 0x5f, 0x54,
    0x61, 0x72, 0x67, 0x65, 0x74, 0x13, 0x84, 0x25, 0xd9, 0xb0, 0x10, 0xb2,
    0x30, 0x0a, 0xb3, 0x50, 0x0a, 0xa7, 0x30, 0x98, 0x02, 0x71, 0x0a, 0xaf,
    0xb0, 0x21, 0xa0, 0x85, 0x0d, 0x43, 0x2c, 0xd4, 0x02, 0xb0, 0xa1, 0xd0,
    0x83, 0x50, 0xb0, 0x85, 0x0b, 0xa0, 0x61, 0xc6, 0xf6, 0x16, 0x46, 0x37,
    0x37, 0x41, 0x10, 0x30, 0x16, 0x69, 0x6e, 0x73, 0x74, 0x73, 0x13, 0x04,
    0x21, 0xa3, 0x31, 0x97, 0x76, 0xf6, 0xc5, 0x46, 0x46, 0x63, 0x2e, 0xed,
    0xec, 0x6b, 0x8e, 0x8e, 0x08, 0x5d, 0x19, 0xde, 0x97, 0xdb, 0x9b, 0x5c,
    0xdb, 0x06, 0x05, 0x17, 0x72, 0x41, 0x17, 0x76, 0x81, 0x17, 0x90, 0x5e,
    0x38, 0x03, 0x5f, 0x60, 0xaa, 0xb0, 0xb1, 0xd9, 0xb5, 0xb9, 0xa4, 0x91,
    0x95, 0xb9, 0xd1, 0x4d, 0x09, 0x82, 0x2a, 0x64, 0x78, 0x2e, 0x76, 0x65,
    0x72, 0x73, 0x69, 0x6f, 0x6e, 0x53, 0x02, 0xa2, 0x09, 0x19, 0x9e, 0x8b,
    0x5d, 0x18, 0x9b, 0x5d, 0x99, 0xdc, 0x94, 0xa0, 0xa8, 0x43, 0x86, 0xe7,
    0x32, 0x87, 0x16, 0x46, 0x56, 0x26, 0xd7, 0xf4, 0x46, 0x56, 0xc6, 0x36,
    0x25, 0x40, 0xca, 0x90, 0xe1, 0xb9, 0xc8, 0x95, 0xcd, 0xbd, 0xd5, 0xc9,
    0x8d, 0x95, 0xcd, 0x4d, 0x09, 0xac, 0x4a, 0x64, 0x78, 0x2e, 0x74, 0x79,
    0x70, 0x65, 0x41, 0x6e, 0x6e, 0x6f, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
    0x73, 0x53, 0x04, 0x3c, 0xe8, 0x83, 0x3a, 0x64, 0x78, 0x2e, 0x76, 0x69,
    0x65, 0x77, 0x49, 0x64, 0x53, 0x74, 0x61, 0x74, 0x65, 0x53, 0x82, 0x3f,
    0xa8, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96,
    0xe6, 0x46, 0x37, 0x37, 0x25, 0xb0, 0x85, 0x2e, 0x64, 0x78, 0x2e, 0x63,
    0x6f, 0x75, 0x6e, 0x74, 0x65, 0x72, 0x73, 0x53, 0x02, 0x5f, 0x00, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80,
    0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84,
    0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c,
    0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42,
    0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88,
    0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c,
    0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79,
    0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70,
    0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f,
    0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4,
    0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30,
    0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc,
    0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b,
    0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70,
    0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76,
    0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72,
    0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e,
    0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1,
    0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21,
    0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8,
    0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94,
    0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc,
    0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c,
    0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e,
    0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e,
    0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00,
    0x00, 0x71, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x25, 0xf0, 0x05,
    0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9, 0x65, 0x39, 0x10, 0x38,
    0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f, 0xd2, 0xc3, 0xf4, 0x32,
    0x10, 0x18, 0xac, 0x80, 0x38, 0x08, 0xfc, 0xe8, 0xe8, 0x32, 0xb5, 0x8c,
    0xa7, 0xd7, 0xe5, 0xe5, 0xaa, 0x15, 0x08, 0x9c, 0x59, 0x7f, 0x24, 0x6a,
    0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0, 0xfa, 0x23, 0xd9, 0xcb,
    0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x10, 0x08, 0x0c,
    0x16, 0x00, 0x1c, 0x04, 0x7e, 0x74, 0x74, 0x99, 0x5a, 0xc6, 0xd3, 0xeb,
    0xf2, 0x72, 0x16, 0x08, 0x9c, 0x59, 0x7f, 0x24, 0x6a, 0x19, 0x4f, 0xaf,
    0xcb, 0xcb, 0x32, 0x22, 0xd0, 0xfa, 0x23, 0xd9, 0xcb, 0x63, 0xfa, 0x5b,
    0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x10, 0x08, 0x0c, 0x1a, 0x82, 0x34,
    0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0,
    0x10, 0x76, 0xe0, 0x0c, 0x97, 0xef, 0x3c, 0xfe, 0xe0, 0x4c, 0xb7, 0x5f,
    0xdc, 0xb6, 0x11, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0xbf, 0x38, 0xc0, 0x20,
    0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x9b, 0x01, 0x34, 0x5c, 0xbe, 0xf3,
    0xf8, 0x12, 0xc0, 0x3c, 0x0b, 0xe1, 0x17, 0xb7, 0x6d, 0x05, 0xd5, 0x70,
    0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4,
    0x17, 0xb7, 0x6d, 0x03, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xa0, 0x59, 0x53, 0xa0, 0x6c, 0x9f, 0xab, 0x00, 0x3c, 0x49, 0x69,
    0x91, 0x4f, 0x9f, 0xaf, 0x32, 0x44, 0x58, 0x49, 0x4c, 0xa0, 0x07, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0xe8, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x88, 0x07, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xdf, 0x01, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x45, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23, 0x00, 0x25, 0x00, 0x14,
    0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84,
    0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0xa1, 0x9b, 0x86, 0xcb,
    0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x8b, 0xdb,
    0x46, 0xc5, 0x18, 0x63, 0x10, 0x2a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f,
    0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82, 0x55, 0x18, 0x45, 0x18,
    0x1b, 0x63, 0x0c, 0x42, 0xc8, 0xa0, 0x56, 0x90, 0x41, 0xc6, 0x18, 0x63,
    0x0c, 0x7a, 0x73, 0x04, 0x41, 0x31, 0x18, 0x29, 0x84, 0x44, 0x92, 0x03,
    0x01, 0xc3, 0x08, 0xc4, 0x30, 0x53, 0x1b, 0x8c, 0x03, 0x3b, 0x84, 0xc3,
    0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83,
    0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8, 0x43, 0x39, 0x8c, 0x03,
    0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03,
    0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80, 0x81, 0x1f, 0xe8, 0x81,
    0x1e, 0xb4, 0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc, 0x02, 0x3d, 0xe4, 0x03,
    0x3c, 0x94, 0x03, 0x0a, 0xc8, 0x4c, 0x62, 0x30, 0x0e, 0xec, 0x10, 0x0e,
    0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f,
    0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60, 0x0f, 0xe5, 0x30, 0x0e,
    0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xe6, 0xc0, 0x0e, 0xef, 0x10, 0x0e,
    0xf4, 0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00, 0x06, 0x7e, 0x80, 0x84,
    0x6a, 0xe9, 0xde, 0x24, 0x4d, 0x11, 0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf,
    0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x08, 0xe5, 0x74, 0x20, 0x00,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10,
    0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2,
    0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30,
    0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x40, 0x16, 0x08, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98,
    0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43,
    0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50, 0x12, 0x65, 0x50, 0x1e,
    0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0, 0x08, 0x0a, 0x84, 0xf0,
    0x0c, 0x00, 0xe9, 0x19, 0x00, 0xda, 0x63, 0x39, 0x0c, 0x01, 0x00, 0x00,
    0x70, 0x1c, 0x00, 0x10, 0x08, 0x04, 0x02, 0x00, 0x00, 0x79, 0x18, 0x00,
    0x00, 0x65, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
    0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b,
    0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b,
    0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9,
    0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13,
    0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61, 0x20, 0x26, 0x08, 0x04,
    0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26,
    0x08, 0xd8, 0x44, 0x60, 0x82, 0x40, 0x24, 0x13, 0x04, 0x42, 0xd9, 0x20,
    0x10, 0xcd, 0x86, 0x84, 0x50, 0x16, 0x82, 0x18, 0x18, 0xc2, 0xd9, 0x90,
    0x0c, 0xca, 0x42, 0x0c, 0x03, 0x43, 0x38, 0x1b, 0x84, 0x07, 0x9a, 0x20,
    0x68, 0xd4, 0x06, 0x84, 0x90, 0x16, 0x82, 0x18, 0x08, 0x60, 0x43, 0x30,
    0x6d, 0x20, 0x22, 0x00, 0xa0, 0x26, 0x08, 0x5b, 0xb5, 0x21, 0xb0, 0x26,
    0x08, 0x02, 0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x8d, 0xcb, 0x94, 0xd5, 0x17,
    0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb, 0x04, 0xa1, 0x68, 0x26,
    0x08, 0x85, 0xb3, 0x21, 0x20, 0x26, 0x08, 0xc5, 0x33, 0x41, 0x28, 0xa0,
    0x0d, 0x0b, 0xa1, 0x6d, 0x5c, 0xe7, 0x0d, 0x1e, 0xf1, 0x01, 0x44, 0xa8,
    0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26, 0x08, 0x45, 0x34, 0x41,
    0x20, 0x96, 0x0d, 0xc2, 0x18, 0x8c, 0xc1, 0x86, 0x65, 0x08, 0x83, 0xed,
    0xeb, 0xc4, 0x60, 0x10, 0x83, 0xe1, 0x23, 0x03, 0x16, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x13, 0x04, 0x82, 0xd9, 0x20, 0x8c, 0xc1, 0x19, 0x6c, 0x58,
    0x18, 0x33, 0xd8, 0xbe, 0x4e, 0x0c, 0x06, 0x8f, 0xf9, 0xd0, 0x60, 0xc3,
    0x00, 0x06, 0x65, 0x90, 0x06, 0x4c, 0xa6, 0xac, 0xbe, 0xa8, 0xc2, 0xe4,
    0xce, 0xca, 0xe8, 0x26, 0x08, 0x85, 0xb4, 0x61, 0x21, 0xd6, 0x60, 0x63,
    0x83, 0xee, 0x1b, 0x3c, 0xe2, 0x43, 0x83, 0x0d, 0x41, 0x1b, 0x6c, 0x18,
    0xd4, 0xc0, 0x0d, 0x80, 0x0d, 0x05, 0x96, 0xbd, 0x41, 0x05, 0x54, 0x61,
    0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04,
    0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6,
    0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9,
    0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c,
    0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x80, 0x94, 0x21, 0xc3, 0x73,
    0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x50,
    0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8,
    0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x56, 0x1d, 0x32, 0x3c, 0x97, 0x32, 0x37,
    0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba, 0xb9, 0x29, 0xc1, 0x1b,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80,
    0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84,
    0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c,
    0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42,
    0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88,
    0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c,
    0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79,
    0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70,
    0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f,
    0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4,
    0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30,
    0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc,
    0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b,
    0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70,
    0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76,
    0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72,
    0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e,
    0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1,
    0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21,
    0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8,
    0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94,
    0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc,
    0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c,
    0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e,
    0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e,
    0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00,
    0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x56, 0x20, 0x0d,
    0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c,
    0x84, 0x11, 0x38, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x38, 0xd3, 0xed, 0x17,
    0xb7, 0x6d, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e, 0x30, 0x88,
    0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x00, 0x0d, 0x97, 0xef, 0x3c,
    0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d, 0x9b, 0x40, 0x35, 0x5c,
    0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a, 0x4d, 0x0f, 0x35, 0xf9,
    0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x61, 0x20, 0x00,
    0x00, 0x5d, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x14, 0x47, 0x00, 0xa8, 0x94, 0x00, 0x91,
    0xc2, 0x2b, 0xb7, 0x92, 0x2b, 0x85, 0x42, 0x98, 0x01, 0xa0, 0x31, 0x46,
    0x50, 0x9e, 0x74, 0xe9, 0x7f, 0x63, 0x04, 0xa2, 0x3e, 0xb7, 0xf3, 0x2f,
    0x8c, 0x11, 0x80, 0xef, 0x0a, 0xae, 0xbf, 0x30, 0x46, 0xc0, 0x97, 0xbd,
    0xb9, 0x7f, 0x23, 0x00, 0x63, 0x04, 0x21, 0x09, 0x86, 0xf0, 0x2f, 0x8c,
    0x11, 0x80, 0x20, 0x08, 0xe2, 0xdf, 0x0c, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x20, 0x85, 0x01, 0x63, 0x81, 0x01, 0x18, 0x40, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x89, 0x41, 0x73, 0x81, 0x01, 0x18,
    0x44, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x8d, 0x81, 0xa3, 0x85,
    0x41, 0x18, 0x48, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xa0, 0x41,
    0x03, 0x06, 0x62, 0x90, 0x31, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60,
    0xa4, 0x81, 0x13, 0x06, 0x63, 0x30, 0x35, 0x23, 0x06, 0x09, 0x00, 0x82,
    0x60, 0x60, 0xa8, 0xc1, 0x23, 0x06, 0x64, 0xa0, 0x39, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x60, 0xac, 0x01, 0x34, 0x06, 0x65, 0xe0, 0x3d, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb0, 0x41, 0x74, 0x06, 0x66, 0xd0,
    0x41, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb4, 0x81, 0x84, 0x06,
    0x67, 0x70, 0x45, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xb0, 0xc1,
    0x85, 0x1c, 0x42, 0xa0, 0x28, 0x68, 0x80, 0x06, 0x92, 0x32, 0x9a, 0x10,
    0x00, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xb8, 0x41, 0xb6, 0x24,
    0xc4, 0xc0, 0x30, 0x6a, 0xa0, 0x06, 0x14, 0x33, 0x9a, 0x10, 0x00, 0xa3,
    0x09, 0x42, 0x60, 0x42, 0x04, 0x1f, 0x13, 0x24, 0xf8, 0x8c, 0x18, 0x2c,
    0x00, 0x08, 0x82, 0xc1, 0x23, 0x07, 0x9d, 0x21, 0x04, 0x12, 0x55, 0x8d,
    0x18, 0x2c, 0x00, 0x08, 0x82, 0xc1, 0x33, 0x07, 0xde, 0x31, 0x08, 0xd3,
    0x85, 0x8d, 0x18, 0x2c, 0x00, 0x08, 0x82, 0xc1, 0x43, 0x07, 0x1f, 0x42,
    0x0c, 0x94, 0x66, 0xd9, 0x00, 0xc9, 0xc7, 0x06, 0x48, 0x3e, 0x36, 0x40,
    0xf2, 0x19, 0x31, 0x48, 0x00, 0x10, 0x04, 0x03, 0x44, 0x0f, 0xc2, 0xa0,
    0x0e, 0xea, 0x80, 0x0d, 0x86, 0x11, 0x83, 0x04, 0x00, 0x41, 0x30, 0x40,
    0xf4, 0x20, 0x0c, 0xea, 0xa0, 0x0e, 0xca, 0x40, 0x18, 0x31, 0x48, 0x00,
    0x10, 0x04, 0x03, 0x44, 0x0f, 0xc2, 0xa0, 0x0e, 0xea, 0x60, 0x0d, 0x82,
    0x11, 0x83, 0x04, 0x00, 0x41, 0x30, 0x40, 0xf4, 0x20, 0x0c, 0xea, 0xa0,
    0x0e, 0xdc, 0x00, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* The yuv-rendering pixel shader:

    --- D3D12_PixelShader_NV12_BT601.hlsl ---
    Texture2D theTextureY : register(t0);
    Texture2D theTextureUV : register(t1);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define NVRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(NVRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        const float3 offset = {-0.0627451017, -0.501960814, -0.501960814};
        const float3 Rcoeff = {1.1644,  0.0000,  1.5960};
        const float3 Gcoeff = {1.1644, -0.3918, -0.8130};
        const float3 Bcoeff = {1.1644,  2.0172,  0.0000};

        float4 Output;

        float3 yuv;
        yuv.x = theTextureY.Sample(theSampler, input.tex).r;
        yuv.yz = theTextureUV.Sample(theSampler, input.tex).rg;

        yuv += offset;
        Output.r = dot(yuv, Rcoeff);
        Output.g = dot(yuv, Gcoeff);
        Output.b = dot(yuv, Bcoeff);
        Output.a = 1.0f;

        return Output * input.color;
    }

*/

static unsigned char D3D12_PixelShader_NV12_BT601[] = {
    0x44, 0x58, 0x42, 0x43, 0xd0, 0xda, 0x37, 0x96, 0x52, 0x6b, 0x77, 0x3b,
    0x82, 0x71, 0x03, 0xf9, 0x0a, 0x76, 0x4b, 0xd7, 0x01, 0x00, 0x00, 0x00,
    0xe9, 0x12, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0x0d, 0x02, 0x00, 0x00, 0xc9, 0x02, 0x00, 0x00, 0x15, 0x0b, 0x00, 0x00,
    0x31, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0xf0, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43,
    0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00,
    0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44,
    0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x52, 0x54, 0x53,
    0x30, 0xb4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00,
    0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x01, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0x9c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0x44, 0x08, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0x11, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x2c, 0x08, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x08, 0x02, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4b, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x2c, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x39, 0x82,
    0xa0, 0x18, 0x52, 0x2c, 0x21, 0x2e, 0xe1, 0x81, 0x80, 0x61, 0x04, 0x01,
    0xb8, 0x4b, 0x9a, 0x22, 0x4a, 0x98, 0xfc, 0x14, 0xb9, 0x88, 0x85, 0x3d,
    0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x84, 0x1d, 0xd2, 0x41,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21,
    0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xe9, 0x01,
    0x00, 0x0f, 0x79, 0x12, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0xc0,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00,
    0x00, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x5a, 0x05, 0x42, 0xa2,
    0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c,
    0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8, 0x3c, 0x0a, 0x87, 0x56,
    0x11, 0x8c, 0x00, 0x14, 0x42, 0x19, 0x94, 0x04, 0x9d, 0x19, 0x00, 0x2a,
    0x33, 0x00, 0x44, 0x66, 0x00, 0x68, 0xcc, 0x00, 0x50, 0x98, 0x01, 0x20,
    0x3e, 0x03, 0x40, 0x7d, 0x2c, 0x87, 0x21, 0x00, 0x00, 0x00, 0x8e, 0x03,
    0x00, 0x02, 0x81, 0x40, 0x00, 0x79, 0x18, 0x00, 0x00, 0xbe, 0x00, 0x00,
    0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63,
    0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03,
    0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a,
    0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b,
    0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0x41, 0x99, 0x20,
    0x08, 0xcb, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04, 0x41, 0x01, 0x6e, 0x6e,
    0x82, 0x20, 0x30, 0x1b, 0x86, 0x03, 0x21, 0x26, 0x08, 0xc5, 0xc7, 0x85,
    0x0e, 0xad, 0x8c, 0xaa, 0x0c, 0x8f, 0xae, 0x4e, 0xae, 0x2c, 0x6b, 0x82,
    0x20, 0x34, 0x13, 0x04, 0xc1, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84, 0x50,
    0x16, 0x82, 0x18, 0x18, 0xc2, 0x21, 0x43, 0x87, 0x56, 0x46, 0x55, 0x86,
    0x47, 0x57, 0x27, 0x57, 0x56, 0x65, 0xb5, 0x21, 0x19, 0x14, 0x88, 0x18,
    0x06, 0x86, 0x70, 0x36, 0x08, 0x4f, 0x34, 0x41, 0x38, 0x3c, 0x2a, 0x74,
    0x68, 0x65, 0x53, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x72, 0x1b, 0x10, 0x62,
    0xa2, 0x08, 0x62, 0x20, 0x80, 0x0d, 0x41, 0xb5, 0x81, 0x90, 0x00, 0xc0,
    0x9a, 0x20, 0x10, 0x60, 0x30, 0x41, 0x10, 0x1e, 0x06, 0x68, 0x13, 0x04,
    0x01, 0x9a, 0x20, 0x08, 0xd1, 0x06, 0x03, 0xd1, 0x36, 0x82, 0x6b, 0x48,
    0xb4, 0xa5, 0xc1, 0xcd, 0x4d, 0x10, 0x04, 0x69, 0x03, 0x81, 0x78, 0xdb,
    0x37, 0x41, 0x08, 0xc4, 0x60, 0x83, 0x40, 0x84, 0xc1, 0x86, 0x40, 0x0c,
    0x36, 0x08, 0xc4, 0x18, 0x6c, 0x20, 0xb2, 0x0e, 0x0c, 0xc8, 0x60, 0x82,
    0x30, 0x84, 0xc1, 0x04, 0x41, 0x98, 0x68, 0xa0, 0x85, 0xb9, 0x91, 0xb1,
    0x95, 0x4d, 0x10, 0x04, 0x6a, 0x83, 0x81, 0xa0, 0xc1, 0x46, 0x70, 0x69,
    0xb0, 0x41, 0x38, 0x03, 0x35, 0x98, 0x20, 0x70, 0x63, 0x30, 0x41, 0x10,
    0x2a, 0x0e, 0x70, 0x6f, 0x73, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0x9e, 0xa6,
    0x92, 0xa8, 0x92, 0x9e, 0x9c, 0x36, 0x20, 0x88, 0x1b, 0x6c, 0xc4, 0x19,
    0xbc, 0x01, 0xd7, 0x70, 0xa0, 0x2b, 0xc3, 0x63, 0x42, 0x55, 0x84, 0x35,
    0xf4, 0xf4, 0x24, 0x45, 0x04, 0xb3, 0x01, 0x41, 0xe2, 0x60, 0xfb, 0xce,
    0x40, 0x0e, 0xb8, 0x86, 0xc5, 0xd8, 0x1b, 0xdb, 0x9b, 0xdc, 0x04, 0x41,
    0xb0, 0x68, 0x0c, 0x3d, 0x31, 0x3d, 0x49, 0xc1, 0x6c, 0x40, 0x10, 0x3a,
    0xd8, 0xea, 0xe0, 0x0c, 0xec, 0x80, 0x6b, 0x36, 0x10, 0x6d, 0x00, 0x07,
    0x73, 0x70, 0x07, 0x1b, 0x0e, 0x02, 0x2b, 0x03, 0x33, 0x58, 0x03, 0x36,
    0xc0, 0x83, 0x09, 0x82, 0x32, 0x6c, 0x00, 0x36, 0x0c, 0xc4, 0x1e, 0xec,
    0xc1, 0x86, 0x80, 0x0f, 0x36, 0x0c, 0x83, 0x1e, 0xf4, 0xc1, 0x04, 0xa1,
    0x23, 0x83, 0x0d, 0xc1, 0x1f, 0x90, 0x68, 0x0b, 0x4b, 0x73, 0xe3, 0x32,
    0x65, 0xf5, 0x05, 0xf5, 0x36, 0x97, 0x46, 0x97, 0xf6, 0xe6, 0x36, 0x41,
    0x58, 0xba, 0x09, 0xc2, 0xc2, 0x6d, 0x08, 0x88, 0x09, 0xc2, 0xb2, 0x4d,
    0x10, 0x16, 0x6d, 0xc3, 0x42, 0x88, 0xc2, 0x28, 0x90, 0x42, 0x29, 0x98,
    0xc2, 0x60, 0x0a, 0xc4, 0x29, 0x00, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e,
    0x9e, 0xa4, 0x88, 0x26, 0x08, 0x4b, 0xb6, 0x41, 0xd8, 0xb6, 0x0d, 0xcb,
    0x90, 0x0a, 0xa3, 0x70, 0x0a, 0xa5, 0xa0, 0x0a, 0x83, 0x2a, 0x0c, 0xa7,
    0xb0, 0x0a, 0x2c, 0x86, 0x9e, 0x98, 0x9e, 0xa4, 0x26, 0x08, 0xc2, 0xb5,
    0x41, 0xd8, 0x5c, 0x61, 0xc3, 0xc2, 0xb4, 0xc2, 0x28, 0x9c, 0x42, 0x29,
    0xa8, 0xc2, 0x60, 0x0a, 0xcc, 0x29, 0xbc, 0xc2, 0x86, 0x01, 0x15, 0x58,
    0x01, 0x16, 0x98, 0x4c, 0x59, 0x7d, 0x51, 0x85, 0xc9, 0x9d, 0x95, 0xd1,
    0x4d, 0x10, 0x96, 0x64, 0xc3, 0x42, 0xc8, 0xc2, 0x28, 0xcc, 0x42, 0x29,
    0x9c, 0xc2, 0x60, 0x0a, 0xc4, 0x29, 0xbc, 0xc2, 0x86, 0x80, 0x16, 0x36,
    0x0c, 0xb1, 0x50, 0x0b, 0xc0, 0x86, 0x42, 0x0f, 0x42, 0xc1, 0x16, 0x2e,
    0x80, 0x86, 0x19, 0xdb, 0x5b, 0x18, 0xdd, 0x1c, 0x8b, 0x34, 0xb7, 0x39,
    0xba, 0xb9, 0x09, 0x82, 0x80, 0xd1, 0x98, 0x4b, 0x3b, 0xfb, 0x62, 0x23,
    0xa3, 0x31, 0x97, 0x76, 0xf6, 0x35, 0x47, 0x47, 0x84, 0xae, 0x0c, 0xef,
    0xcb, 0xed, 0x4d, 0xae, 0x6d, 0x83, 0x82, 0x0b, 0x4d, 0x2e, 0xe8, 0xc2,
    0x2e, 0x20, 0xbc, 0x70, 0x06, 0xbd, 0xc0, 0x54, 0x61, 0x63, 0xb3, 0x6b,
    0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0,
    0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13,
    0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x41, 0x51,
    0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d,
    0xac, 0x8c, 0x6d, 0x4a, 0x80, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b,
    0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x58, 0x95, 0xc8, 0xf0,
    0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2, 0xe8,
    0xd2, 0xde, 0xdc, 0xe6, 0xa6, 0x08, 0x78, 0xd0, 0x07, 0x75, 0xc8, 0xf0,
    0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca,
    0xa6, 0x04, 0x7f, 0x50, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e,
    0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x60, 0x0b, 0x5d, 0xc8,
    0xf0, 0x5c, 0xc6, 0xde, 0xea, 0xdc, 0xe8, 0xca, 0xe4, 0xe6, 0xa6, 0x04,
    0xbd, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00,
    0x00, 0x25, 0xf0, 0x05, 0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9,
    0x65, 0x39, 0x10, 0x38, 0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f,
    0xd2, 0xc3, 0xf4, 0x32, 0x10, 0x18, 0xac, 0x80, 0x38, 0x08, 0xfc, 0xe8,
    0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7, 0xe5, 0xe5, 0xaa, 0x15, 0x08, 0x9c,
    0x59, 0x7f, 0x24, 0x6a, 0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0,
    0xfa, 0x23, 0xd9, 0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33,
    0x20, 0x10, 0x08, 0x0c, 0x16, 0x00, 0x1c, 0x04, 0x7e, 0x74, 0x74, 0x99,
    0x5a, 0xc6, 0xd3, 0xeb, 0xf2, 0x72, 0x16, 0x08, 0x9c, 0x59, 0x7f, 0x24,
    0x6a, 0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0, 0xfa, 0x23, 0xd9,
    0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x10, 0x08,
    0x0c, 0x1a, 0x82, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13,
    0x11, 0x02, 0xcd, 0xb0, 0x10, 0x76, 0xe0, 0x0c, 0x97, 0xef, 0x3c, 0xfe,
    0xe0, 0x4c, 0xb7, 0x5f, 0xdc, 0xb6, 0x11, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f,
    0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x9b, 0x01,
    0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x12, 0xc0, 0x3c, 0x0b, 0xe1, 0x17, 0xb7,
    0x6d, 0x05, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28,
    0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x03, 0x04, 0x03, 0x20, 0x0d,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xd6, 0x59, 0x5f, 0x75, 0xd6, 0x73, 0xad,
    0xd4, 0x07, 0x66, 0xfb, 0xaa, 0x3b, 0xb4, 0x38, 0x99, 0x44, 0x58, 0x49,
    0x4c, 0xb0, 0x07, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xec, 0x01, 0x00,
    0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x98, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00,
    0x00, 0xe3, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04,
    0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08,
    0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b,
    0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52,
    0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32,
    0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81,
    0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00,
    0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff,
    0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03,
    0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8,
    0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60,
    0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00,
    0x00, 0x45, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85,
    0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90,
    0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23,
    0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00,
    0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06,
    0xa1, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56,
    0x62, 0xf2, 0x8b, 0xdb, 0x46, 0xc5, 0x18, 0x63, 0x10, 0x2a, 0xf7, 0x0c,
    0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82,
    0x55, 0x18, 0x45, 0x18, 0x1b, 0x63, 0x0c, 0x42, 0xc8, 0xa0, 0x56, 0x90,
    0x41, 0xc6, 0x18, 0x63, 0x0c, 0x7a, 0x73, 0x04, 0x41, 0x31, 0x18, 0x29,
    0x84, 0x44, 0x92, 0x03, 0x01, 0xc3, 0x08, 0xc4, 0x30, 0x53, 0x1b, 0x8c,
    0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0,
    0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8,
    0x43, 0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0,
    0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80,
    0x81, 0x1f, 0xe8, 0x81, 0x1e, 0xb4, 0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc,
    0x02, 0x3d, 0xe4, 0x03, 0x3c, 0x94, 0x03, 0x0a, 0xc8, 0x4c, 0x62, 0x30,
    0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80,
    0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60,
    0x0f, 0xe5, 0x30, 0x0e, 0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xe6, 0xc0,
    0x0e, 0xef, 0x10, 0x0e, 0xf4, 0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00,
    0x06, 0x7e, 0x80, 0x84, 0x6a, 0xe9, 0xde, 0x24, 0x4d, 0x11, 0x25, 0x4c,
    0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x08,
    0xe5, 0x74, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60,
    0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf,
    0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a,
    0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
    0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
    0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6,
    0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73,
    0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74,
    0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71,
    0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43,
    0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x0f, 0x00, 0x00,
    0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26,
    0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50,
    0x12, 0x65, 0x50, 0x1e, 0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0,
    0x08, 0x0a, 0x84, 0xf0, 0x0c, 0x00, 0xe9, 0x19, 0x00, 0xda, 0x63, 0x39,
    0x0c, 0x01, 0x00, 0x00, 0x70, 0x1c, 0x00, 0x10, 0x08, 0x04, 0x02, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c,
    0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03,
    0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01,
    0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a,
    0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b,
    0xd9, 0x10, 0x04, 0x13, 0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61,
    0x20, 0x26, 0x08, 0x04, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b,
    0x06, 0xc4, 0x20, 0x26, 0x08, 0xd8, 0x44, 0x60, 0x82, 0x40, 0x24, 0x13,
    0x04, 0x42, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84, 0x50, 0x16, 0x82, 0x18,
    0x18, 0xc2, 0xd9, 0x90, 0x0c, 0xca, 0x42, 0x0c, 0x03, 0x43, 0x38, 0x1b,
    0x84, 0x07, 0x9a, 0x20, 0x68, 0xd4, 0x06, 0x84, 0x90, 0x16, 0x82, 0x18,
    0x08, 0x60, 0x43, 0x30, 0x6d, 0x20, 0x22, 0x00, 0xa0, 0x26, 0x08, 0x5b,
    0xb5, 0x21, 0xb0, 0x26, 0x08, 0x02, 0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x8d,
    0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb,
    0x04, 0xa1, 0x68, 0x26, 0x08, 0x85, 0xb3, 0x21, 0x20, 0x26, 0x08, 0xc5,
    0x33, 0x41, 0x28, 0xa0, 0x0d, 0x0b, 0xa1, 0x6d, 0x5c, 0xe7, 0x0d, 0x1e,
    0xf1, 0x01, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26,
    0x08, 0x45, 0x34, 0x41, 0x20, 0x96, 0x0d, 0xc2, 0x18, 0x8c, 0xc1, 0x86,
    0x65, 0x08, 0x83, 0xed, 0xeb, 0xc4, 0x60, 0x10, 0x83, 0xe1, 0x23, 0x03,
    0x16, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x13, 0x04, 0x82, 0xd9, 0x20, 0x8c,
    0xc1, 0x19, 0x6c, 0x58, 0x18, 0x33, 0xd8, 0xbe, 0x4e, 0x0c, 0x06, 0x8f,
    0xf9, 0xd0, 0x60, 0xc3, 0x00, 0x06, 0x65, 0x90, 0x06, 0x4c, 0xa6, 0xac,
    0xbe, 0xa8, 0xc2, 0xe4, 0xce, 0xca, 0xe8, 0x26, 0x08, 0x85, 0xb4, 0x61,
    0x21, 0xd6, 0x60, 0x63, 0x83, 0xee, 0x1b, 0x3c, 0xe2, 0x43, 0x83, 0x0d,
    0x41, 0x1b, 0x6c, 0x18, 0xd4, 0xc0, 0x0d, 0x80, 0x0d, 0x05, 0x96, 0xbd,
    0x41, 0x05, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73,
    0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6,
    0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
    0x36, 0xbb, 0x32, 0xb9, 0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e,
    0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x80,
    0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b,
    0x9b, 0x9b, 0x12, 0x50, 0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee,
    0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x56, 0x1d, 0x32,
    0x3c, 0x97, 0x32, 0x37, 0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba,
    0xb9, 0x29, 0xc1, 0x1b, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00,
    0x00, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44,
    0x84, 0x40, 0x33, 0x2c, 0x84, 0x11, 0x38, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f,
    0x38, 0xd3, 0xed, 0x17, 0xb7, 0x6d, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3,
    0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x00,
    0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d,
    0x9b, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a,
    0x4d, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03,
    0x00, 0x61, 0x20, 0x00, 0x00, 0x61, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41,
    0x2c, 0x10, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x14, 0x47, 0x00,
    0xa8, 0x94, 0x00, 0x91, 0xc2, 0x2b, 0xb7, 0x92, 0x2b, 0x85, 0x42, 0x98,
    0x01, 0xa0, 0x31, 0x46, 0xe0, 0xba, 0xa6, 0x08, 0x82, 0xc1, 0x18, 0x41,
    0x69, 0xa2, 0x60, 0xfd, 0x0b, 0x63, 0x04, 0x22, 0x6c, 0xc6, 0xec, 0x2f,
    0x8c, 0x11, 0xe8, 0xad, 0x8c, 0xf3, 0xdf, 0x18, 0x41, 0x48, 0x82, 0x21,
    0xee, 0x0b, 0x23, 0x00, 0x63, 0x04, 0x21, 0x09, 0x86, 0xf0, 0x2f, 0x8c,
    0x11, 0xbc, 0xb8, 0xa8, 0xe6, 0xdf, 0x0c, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x20, 0x89, 0x41, 0x73, 0x85, 0x41, 0x18, 0x44, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x8d, 0x81, 0x83, 0x85, 0x41, 0x18,
    0x48, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x91, 0xc1, 0xb3, 0x89,
    0x81, 0x18, 0x4c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xa4, 0x81,
    0x13, 0x06, 0x63, 0xa0, 0x35, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60,
    0xa8, 0xc1, 0x23, 0x06, 0x64, 0x40, 0x39, 0x23, 0x06, 0x09, 0x00, 0x82,
    0x60, 0x60, 0xac, 0x01, 0x34, 0x06, 0x65, 0xb0, 0x3d, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x60, 0xb0, 0x41, 0x44, 0x06, 0x66, 0xf0, 0x41, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb4, 0x81, 0x84, 0x06, 0x67, 0xe0,
    0x45, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb8, 0xc1, 0x94, 0x06,
    0x68, 0x80, 0x49, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xb4, 0x01,
    0x86, 0x1c, 0x42, 0xa0, 0x28, 0x69, 0x90, 0x06, 0x93, 0x32, 0x9a, 0x10,
    0x00, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xbc, 0x81, 0xb6, 0x24,
    0xc4, 0xc0, 0x30, 0x6b, 0xb0, 0x06, 0x15, 0x33, 0x9a, 0x10, 0x00, 0xa3,
    0x09, 0x42, 0x60, 0xc4, 0x04, 0x1f, 0x1b, 0x24, 0xf8, 0xd8, 0x30, 0xc1,
    0x67, 0xc4, 0x60, 0x01, 0x40, 0x10, 0x0c, 0x1e, 0x3a, 0xf8, 0x06, 0x21,
    0x98, 0xaa, 0x6b, 0xc4, 0x60, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x3a, 0x00,
    0x03, 0x62, 0x10, 0xa8, 0x4c, 0x1b, 0x31, 0x58, 0x00, 0x10, 0x04, 0x83,
    0xc7, 0x0e, 0xc2, 0xa0, 0x20, 0x86, 0x8a, 0xbb, 0x6c, 0x88, 0xe4, 0x63,
    0x43, 0x24, 0x1f, 0x1b, 0x22, 0xf9, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82,
    0x01, 0xc2, 0x07, 0x63, 0x70, 0x07, 0x77, 0xe0, 0x06, 0xc3, 0x88, 0x41,
    0x02, 0x80, 0x20, 0x18, 0x20, 0x7c, 0x30, 0x06, 0x77, 0x70, 0x07, 0x67,
    0x20, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x07, 0x63, 0x70,
    0x07, 0x77, 0xd0, 0x06, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20,
    0x7c, 0x30, 0x06, 0x77, 0x70, 0x07, 0x70, 0x10, 0x21, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00
};

/* The yuv-rendering pixel shader:

    --- D3D12_PixelShader_NV12_BT709.hlsl ---
    Texture2D theTextureY : register(t0);
    Texture2D theTextureUV : register(t1);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define NVRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(NVRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        const float3 offset = {-0.0627451017, -0.501960814, -0.501960814};
        const float3 Rcoeff = {1.1644,  0.0000,  1.7927};
        const float3 Gcoeff = {1.1644, -0.2132, -0.5329};
        const float3 Bcoeff = {1.1644,  2.1124,  0.0000};

        float4 Output;

        float3 yuv;
        yuv.x = theTextureY.Sample(theSampler, input.tex).r;
        yuv.yz = theTextureUV.Sample(theSampler, input.tex).rg;

        yuv += offset;
        Output.r = dot(yuv, Rcoeff);
        Output.g = dot(yuv, Gcoeff);
        Output.b = dot(yuv, Bcoeff);
        Output.a = 1.0f;

        return Output * input.color;
    }

*/

static unsigned char D3D12_PixelShader_NV12_BT709[] = {
    0x44, 0x58, 0x42, 0x43, 0x45, 0xb4, 0xa5, 0xa8, 0x01, 0xc5, 0xab, 0xf5,
    0xc0, 0x51, 0x32, 0xc2, 0xc3, 0x60, 0x64, 0xf1, 0x01, 0x00, 0x00, 0x00,
    0xe9, 0x12, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0x0d, 0x02, 0x00, 0x00, 0xc9, 0x02, 0x00, 0x00, 0x15, 0x0b, 0x00, 0x00,
    0x31, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0xf0, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43,
    0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00,
    0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44,
    0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x52, 0x54, 0x53,
    0x30, 0xb4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00,
    0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x01, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0x9c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0x44, 0x08, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0x11, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x2c, 0x08, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x08, 0x02, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4b, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x2c, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x39, 0x82,
    0xa0, 0x18, 0x52, 0x2c, 0x21, 0x2e, 0xe1, 0x81, 0x80, 0x61, 0x04, 0x01,
    0xb8, 0x4b, 0x9a, 0x22, 0x4a, 0x98, 0xfc, 0x14, 0xb9, 0x88, 0x85, 0x3d,
    0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x84, 0x1d, 0xd2, 0x41,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21,
    0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xe9, 0x01,
    0x00, 0x0f, 0x79, 0x12, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0xc0,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00,
    0x00, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x5a, 0x05, 0x42, 0xa2,
    0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c,
    0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8, 0x3c, 0x0a, 0x87, 0x56,
    0x11, 0x8c, 0x00, 0x14, 0x42, 0x19, 0x94, 0x04, 0x9d, 0x19, 0x00, 0x2a,
    0x33, 0x00, 0x44, 0x66, 0x00, 0x68, 0xcc, 0x00, 0x50, 0x98, 0x01, 0x20,
    0x3e, 0x03, 0x40, 0x7d, 0x2c, 0x87, 0x21, 0x00, 0x00, 0x00, 0x8e, 0x03,
    0x00, 0x02, 0x81, 0x40, 0x00, 0x79, 0x18, 0x00, 0x00, 0xbe, 0x00, 0x00,
    0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63,
    0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03,
    0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a,
    0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b,
    0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0x41, 0x99, 0x20,
    0x08, 0xcb, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04, 0x41, 0x01, 0x6e, 0x6e,
    0x82, 0x20, 0x30, 0x1b, 0x86, 0x03, 0x21, 0x26, 0x08, 0xc5, 0xc7, 0x85,
    0x0e, 0xad, 0x8c, 0xaa, 0x0c, 0x8f, 0xae, 0x4e, 0xae, 0x2c, 0x6b, 0x82,
    0x20, 0x34, 0x13, 0x04, 0xc1, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84, 0x50,
    0x16, 0x82, 0x18, 0x18, 0xc2, 0x21, 0x43, 0x87, 0x56, 0x46, 0x55, 0x86,
    0x47, 0x57, 0x27, 0x57, 0x56, 0x65, 0xb5, 0x21, 0x19, 0x14, 0x88, 0x18,
    0x06, 0x86, 0x70, 0x36, 0x08, 0x4f, 0x34, 0x41, 0x38, 0x3c, 0x2a, 0x74,
    0x68, 0x65, 0x53, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x72, 0x1b, 0x10, 0x62,
    0xa2, 0x08, 0x62, 0x20, 0x80, 0x0d, 0x41, 0xb5, 0x81, 0x90, 0x00, 0xc0,
    0x9a, 0x20, 0x10, 0x60, 0x30, 0x41, 0x10, 0x1e, 0x06, 0x68, 0x13, 0x04,
    0x01, 0x9a, 0x20, 0x08, 0xd1, 0x06, 0x03, 0xd1, 0x36, 0x82, 0x6b, 0x48,
    0xb4, 0xa5, 0xc1, 0xcd, 0x4d, 0x10, 0x04, 0x69, 0x03, 0x81, 0x78, 0xdb,
    0x37, 0x41, 0x08, 0xc4, 0x60, 0x83, 0x40, 0x84, 0xc1, 0x86, 0x40, 0x0c,
    0x36, 0x08, 0xc4, 0x18, 0x6c, 0x20, 0xb2, 0x0e, 0x0c, 0xc8, 0x60, 0x82,
    0x30, 0x84, 0xc1, 0x04, 0x41, 0x98, 0x68, 0xa0, 0x85, 0xb9, 0x91, 0xb1,
    0x95, 0x4d, 0x10, 0x04, 0x6a, 0x83, 0x81, 0xa0, 0xc1, 0x46, 0x70, 0x69,
    0xb0, 0x41, 0x38, 0x03, 0x35, 0x98, 0x20, 0x70, 0x63, 0x30, 0x41, 0x10,
    0x2a, 0x0e, 0x70, 0x6f, 0x73, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0x9e, 0xa6,
    0x92, 0xa8, 0x92, 0x9e, 0x9c, 0x36, 0x20, 0x88, 0x1b, 0x6c, 0xc4, 0x19,
    0xbc, 0x01, 0xd7, 0x70, 0xa0, 0x2b, 0xc3, 0x63, 0x42, 0x55, 0x84, 0x35,
    0xf4, 0xf4, 0x24, 0x45, 0x04, 0xb3, 0x01, 0x41, 0xe2, 0x60, 0xfb, 0xce,
    0x40, 0x0e, 0xb8, 0x86, 0xc5, 0xd8, 0x1b, 0xdb, 0x9b, 0xdc, 0x04, 0x41,
    0xb0, 0x68, 0x0c, 0x3d, 0x31, 0x3d, 0x49, 0xc1, 0x6c, 0x40, 0x10, 0x3a,
    0xd8, 0xea, 0xe0, 0x0c, 0xec, 0x80, 0x6b, 0x36, 0x10, 0x6d, 0x00, 0x07,
    0x73, 0x70, 0x07, 0x1b, 0x0e, 0x02, 0x2b, 0x03, 0x33, 0x58, 0x03, 0x36,
    0xc0, 0x83, 0x09, 0x82, 0x32, 0x6c, 0x00, 0x36, 0x0c, 0xc4, 0x1e, 0xec,
    0xc1, 0x86, 0x80, 0x0f, 0x36, 0x0c, 0x83, 0x1e, 0xf4, 0xc1, 0x04, 0xa1,
    0x23, 0x83, 0x0d, 0xc1, 0x1f, 0x90, 0x68, 0x0b, 0x4b, 0x73, 0xe3, 0x32,
    0x65, 0xf5, 0x05, 0xf5, 0x36, 0x97, 0x46, 0x97, 0xf6, 0xe6, 0x36, 0x41,
    0x58, 0xba, 0x09, 0xc2, 0xc2, 0x6d, 0x08, 0x88, 0x09, 0xc2, 0xb2, 0x4d,
    0x10, 0x16, 0x6d, 0xc3, 0x42, 0x88, 0xc2, 0x28, 0x90, 0x42, 0x29, 0x98,
    0xc2, 0x60, 0x0a, 0xc4, 0x29, 0x00, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e,
    0x9e, 0xa4, 0x88, 0x26, 0x08, 0x4b, 0xb6, 0x41, 0xd8, 0xb6, 0x0d, 0xcb,
    0x90, 0x0a, 0xa3, 0x70, 0x0a, 0xa5, 0xa0, 0x0a, 0x83, 0x2a, 0x0c, 0xa7,
    0xb0, 0x0a, 0x2c, 0x86, 0x9e, 0x98, 0x9e, 0xa4, 0x26, 0x08, 0xc2, 0xb5,
    0x41, 0xd8, 0x5c, 0x61, 0xc3, 0xc2, 0xb4, 0xc2, 0x28, 0x9c, 0x42, 0x29,
    0xa8, 0xc2, 0x60, 0x0a, 0xcc, 0x29, 0xbc, 0xc2, 0x86, 0x01, 0x15, 0x58,
    0x01, 0x16, 0x98, 0x4c, 0x59, 0x7d, 0x51, 0x85, 0xc9, 0x9d, 0x95, 0xd1,
    0x4d, 0x10, 0x96, 0x64, 0xc3, 0x42, 0xc8, 0xc2, 0x28, 0xcc, 0x42, 0x29,
    0x9c, 0xc2, 0x60, 0x0a, 0xc4, 0x29, 0xbc, 0xc2, 0x86, 0x80, 0x16, 0x36,
    0x0c, 0xb1, 0x50, 0x0b, 0xc0, 0x86, 0x42, 0x0f, 0x42, 0xc1, 0x16, 0x2e,
    0x80, 0x86, 0x19, 0xdb, 0x5b, 0x18, 0xdd, 0x1c, 0x8b, 0x34, 0xb7, 0x39,
    0xba, 0xb9, 0x09, 0x82, 0x80, 0xd1, 0x98, 0x4b, 0x3b, 0xfb, 0x62, 0x23,
    0xa3, 0x31, 0x97, 0x76, 0xf6, 0x35, 0x47, 0x47, 0x84, 0xae, 0x0c, 0xef,
    0xcb, 0xed, 0x4d, 0xae, 0x6d, 0x83, 0x82, 0x0b, 0x4d, 0x2e, 0xe8, 0xc2,
    0x2e, 0x20, 0xbc, 0x70, 0x06, 0xbd, 0xc0, 0x54, 0x61, 0x63, 0xb3, 0x6b,
    0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0,
    0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13,
    0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x41, 0x51,
    0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d,
    0xac, 0x8c, 0x6d, 0x4a, 0x80, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b,
    0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x58, 0x95, 0xc8, 0xf0,
    0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2, 0xe8,
    0xd2, 0xde, 0xdc, 0xe6, 0xa6, 0x08, 0x78, 0xd0, 0x07, 0x75, 0xc8, 0xf0,
    0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca,
    0xa6, 0x04, 0x7f, 0x50, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e,
    0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x60, 0x0b, 0x5d, 0xc8,
    0xf0, 0x5c, 0xc6, 0xde, 0xea, 0xdc, 0xe8, 0xca, 0xe4, 0xe6, 0xa6, 0x04,
    0xbd, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00,
    0x00, 0x25, 0xf0, 0x05, 0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9,
    0x65, 0x39, 0x10, 0x38, 0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f,
    0xd2, 0xc3, 0xf4, 0x32, 0x10, 0x18, 0xac, 0x80, 0x38, 0x08, 0xfc, 0xe8,
    0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7, 0xe5, 0xe5, 0xaa, 0x15, 0x08, 0x9c,
    0x59, 0x7f, 0x24, 0x6a, 0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0,
    0xfa, 0x23, 0xd9, 0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33,
    0x20, 0x10, 0x08, 0x0c, 0x16, 0x00, 0x1c, 0x04, 0x7e, 0x74, 0x74, 0x99,
    0x5a, 0xc6, 0xd3, 0xeb, 0xf2, 0x72, 0x16, 0x08, 0x9c, 0x59, 0x7f, 0x24,
    0x6a, 0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0, 0xfa, 0x23, 0xd9,
    0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x10, 0x08,
    0x0c, 0x1a, 0x82, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13,
    0x11, 0x02, 0xcd, 0xb0, 0x10, 0x76, 0xe0, 0x0c, 0x97, 0xef, 0x3c, 0xfe,
    0xe0, 0x4c, 0xb7, 0x5f, 0xdc, 0xb6, 0x11, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f,
    0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x9b, 0x01,
    0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x12, 0xc0, 0x3c, 0x0b, 0xe1, 0x17, 0xb7,
    0x6d, 0x05, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28,
    0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x03, 0x04, 0x03, 0x20, 0x0d,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xd1, 0x3f, 0x63, 0xb2, 0x44, 0xb0, 0x28,
    0x54, 0xc4, 0x46, 0x87, 0x44, 0x49, 0x1b, 0x1b, 0xb5, 0x44, 0x58, 0x49,
    0x4c, 0xb0, 0x07, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xec, 0x01, 0x00,
    0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x98, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00,
    0x00, 0xe3, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04,
    0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08,
    0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b,
    0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52,
    0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32,
    0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81,
    0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00,
    0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff,
    0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03,
    0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8,
    0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60,
    0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00,
    0x00, 0x45, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85,
    0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90,
    0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23,
    0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00,
    0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06,
    0xa1, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56,
    0x62, 0xf2, 0x8b, 0xdb, 0x46, 0xc5, 0x18, 0x63, 0x10, 0x2a, 0xf7, 0x0c,
    0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82,
    0x55, 0x18, 0x45, 0x18, 0x1b, 0x63, 0x0c, 0x42, 0xc8, 0xa0, 0x56, 0x90,
    0x41, 0xc6, 0x18, 0x63, 0x0c, 0x7a, 0x73, 0x04, 0x41, 0x31, 0x18, 0x29,
    0x84, 0x44, 0x92, 0x03, 0x01, 0xc3, 0x08, 0xc4, 0x30, 0x53, 0x1b, 0x8c,
    0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0,
    0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8,
    0x43, 0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0,
    0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80,
    0x81, 0x1f, 0xe8, 0x81, 0x1e, 0xb4, 0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc,
    0x02, 0x3d, 0xe4, 0x03, 0x3c, 0x94, 0x03, 0x0a, 0xc8, 0x4c, 0x62, 0x30,
    0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80,
    0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60,
    0x0f, 0xe5, 0x30, 0x0e, 0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xe6, 0xc0,
    0x0e, 0xef, 0x10, 0x0e, 0xf4, 0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00,
    0x06, 0x7e, 0x80, 0x84, 0x6a, 0xe9, 0xde, 0x24, 0x4d, 0x11, 0x25, 0x4c,
    0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x08,
    0xe5, 0x74, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60,
    0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf,
    0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a,
    0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
    0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
    0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6,
    0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73,
    0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74,
    0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71,
    0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43,
    0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x0f, 0x00, 0x00,
    0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26,
    0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50,
    0x12, 0x65, 0x50, 0x1e, 0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0,
    0x08, 0x0a, 0x84, 0xf0, 0x0c, 0x00, 0xe9, 0x19, 0x00, 0xda, 0x63, 0x39,
    0x0c, 0x01, 0x00, 0x00, 0x70, 0x1c, 0x00, 0x10, 0x08, 0x04, 0x02, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c,
    0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03,
    0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01,
    0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a,
    0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b,
    0xd9, 0x10, 0x04, 0x13, 0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61,
    0x20, 0x26, 0x08, 0x04, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b,
    0x06, 0xc4, 0x20, 0x26, 0x08, 0xd8, 0x44, 0x60, 0x82, 0x40, 0x24, 0x13,
    0x04, 0x42, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84, 0x50, 0x16, 0x82, 0x18,
    0x18, 0xc2, 0xd9, 0x90, 0x0c, 0xca, 0x42, 0x0c, 0x03, 0x43, 0x38, 0x1b,
    0x84, 0x07, 0x9a, 0x20, 0x68, 0xd4, 0x06, 0x84, 0x90, 0x16, 0x82, 0x18,
    0x08, 0x60, 0x43, 0x30, 0x6d, 0x20, 0x22, 0x00, 0xa0, 0x26, 0x08, 0x5b,
    0xb5, 0x21, 0xb0, 0x26, 0x08, 0x02, 0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x8d,
    0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb,
    0x04, 0xa1, 0x68, 0x26, 0x08, 0x85, 0xb3, 0x21, 0x20, 0x26, 0x08, 0xc5,
    0x33, 0x41, 0x28, 0xa0, 0x0d, 0x0b, 0xa1, 0x6d, 0x5c, 0xe7, 0x0d, 0x1e,
    0xf1, 0x01, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26,
    0x08, 0x45, 0x34, 0x41, 0x20, 0x96, 0x0d, 0xc2, 0x18, 0x8c, 0xc1, 0x86,
    0x65, 0x08, 0x83, 0xed, 0xeb, 0xc4, 0x60, 0x10, 0x83, 0xe1, 0x23, 0x03,
    0x16, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x13, 0x04, 0x82, 0xd9, 0x20, 0x8c,
    0xc1, 0x19, 0x6c, 0x58, 0x18, 0x33, 0xd8, 0xbe, 0x4e, 0x0c, 0x06, 0x8f,
    0xf9, 0xd0, 0x60, 0xc3, 0x00, 0x06, 0x65, 0x90, 0x06, 0x4c, 0xa6, 0xac,
    0xbe, 0xa8, 0xc2, 0xe4, 0xce, 0xca, 0xe8, 0x26, 0x08, 0x85, 0xb4, 0x61,
    0x21, 0xd6, 0x60, 0x63, 0x83, 0xee, 0x1b, 0x3c, 0xe2, 0x43, 0x83, 0x0d,
    0x41, 0x1b, 0x6c, 0x18, 0xd4, 0xc0, 0x0d, 0x80, 0x0d, 0x05, 0x96, 0xbd,
    0x41, 0x05, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73,
    0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6,
    0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
    0x36, 0xbb, 0x32, 0xb9, 0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e,
    0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x80,
    0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b,
    0x9b, 0x9b, 0x12, 0x50, 0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee,
    0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x56, 0x1d, 0x32,
    0x3c, 0x97, 0x32, 0x37, 0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba,
    0xb9, 0x29, 0xc1, 0x1b, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00,
    0x00, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44,
    0x84, 0x40, 0x33, 0x2c, 0x84, 0x11, 0x38, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f,
    0x38, 0xd3, 0xed, 0x17, 0xb7, 0x6d, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3,
    0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x00,
    0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d,
    0x9b, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a,
    0x4d, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03,
    0x00, 0x61, 0x20, 0x00, 0x00, 0x61, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41,
    0x2c, 0x10, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x14, 0x47, 0x00,
    0xa8, 0x94, 0x00, 0x91, 0xc2, 0x2b, 0xb7, 0x92, 0x2b, 0x85, 0x42, 0x98,
    0x01, 0xa0, 0x31, 0x46, 0x00, 0xb3, 0xac, 0x0b, 0x82, 0xc1, 0x18, 0x81,
    0x18, 0xee, 0x30, 0xfc, 0x0b, 0x63, 0x04, 0x3a, 0x4a, 0xd3, 0xe5, 0x2f,
    0x8c, 0x11, 0xc8, 0x79, 0xaf, 0xfa, 0xdf, 0x18, 0x41, 0x48, 0x82, 0x21,
    0xee, 0x0b, 0x23, 0x00, 0x63, 0x04, 0x21, 0x09, 0x86, 0xf0, 0x2f, 0x8c,
    0x11, 0xbc, 0xb8, 0xa8, 0xe6, 0xdf, 0x0c, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x20, 0x89, 0x41, 0x73, 0x85, 0x41, 0x18, 0x44, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x8d, 0x81, 0x83, 0x85, 0x41, 0x18,
    0x48, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x91, 0xc1, 0xb3, 0x89,
    0x81, 0x18, 0x4c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xa4, 0x81,
    0x13, 0x06, 0x63, 0xa0, 0x35, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60,
    0xa8, 0xc1, 0x23, 0x06, 0x64, 0x40, 0x39, 0x23, 0x06, 0x09, 0x00, 0x82,
    0x60, 0x60, 0xac, 0x01, 0x34, 0x06, 0x65, 0xb0, 0x3d, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x60, 0xb0, 0x41, 0x44, 0x06, 0x66, 0xf0, 0x41, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb4, 0x81, 0x84, 0x06, 0x67, 0xe0,
    0x45, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb8, 0xc1, 0x94, 0x06,
    0x68, 0x80, 0x49, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xb4, 0x01,
    0x86, 0x1c, 0x42, 0xa0, 0x28, 0x69, 0x90, 0x06, 0x93, 0x32, 0x9a, 0x10,
    0x00, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xbc, 0x81, 0xb6, 0x24,
    0xc4, 0xc0, 0x30, 0x6b, 0xb0, 0x06, 0x15, 0x33, 0x9a, 0x10, 0x00, 0xa3,
    0x09, 0x42, 0x60, 0xc4, 0x04, 0x1f, 0x1b, 0x24, 0xf8, 0xd8, 0x30, 0xc1,
    0x67, 0xc4, 0x60, 0x01, 0x40, 0x10, 0x0c, 0x1e, 0x3a, 0xf8, 0x06, 0x21,
    0x98, 0xaa, 0x6b, 0xc4, 0x60, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x3a, 0x00,
    0x03, 0x62, 0x10, 0xa8, 0x4c, 0x1b, 0x31, 0x58, 0x00, 0x10, 0x04, 0x83,
    0xc7, 0x0e, 0xc2, 0xa0, 0x20, 0x86, 0x8a, 0xbb, 0x6c, 0x88, 0xe4, 0x63,
    0x43, 0x24, 0x1f, 0x1b, 0x22, 0xf9, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82,
    0x01, 0xc2, 0x07, 0x63, 0x70, 0x07, 0x77, 0xe0, 0x06, 0xc3, 0x88, 0x41,
    0x02, 0x80, 0x20, 0x18, 0x20, 0x7c, 0x30, 0x06, 0x77, 0x70, 0x07, 0x67,
    0x20, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x07, 0x63, 0x70,
    0x07, 0x77, 0xd0, 0x06, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20,
    0x7c, 0x30, 0x06, 0x77, 0x70, 0x07, 0x70, 0x10, 0x21, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00
};

/* The yuv-rendering pixel shader:

    --- D3D12_PixelShader_NV21_JPEG.hlsl ---
    Texture2D theTextureY : register(t0);
    Texture2D theTextureUV : register(t1);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define NVRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(NVRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        const float3 offset = {0.0, -0.501960814, -0.501960814};
        const float3 Rcoeff = {1.0000,  0.0000,  1.4020};
        const float3 Gcoeff = {1.0000, -0.3441, -0.7141};
        const float3 Bcoeff = {1.0000,  1.7720,  0.0000};

        float4 Output;

        float3 yuv;
        yuv.x = theTextureY.Sample(theSampler, input.tex).r;
        yuv.yz = theTextureUV.Sample(theSampler, input.tex).gr;

        yuv += offset;
        Output.r = dot(yuv, Rcoeff);
        Output.g = dot(yuv, Gcoeff);
        Output.b = dot(yuv, Bcoeff);
        Output.a = 1.0f;

        return Output * input.color;
    }

*/

static unsigned char D3D12_PixelShader_NV21_JPEG[] = {
    0x44, 0x58, 0x42, 0x43, 0xcf, 0xc7, 0x5e, 0x20, 0x3e, 0x0d, 0xdf, 0x60,
    0x41, 0xdf, 0xe0, 0xb4, 0x61, 0xb5, 0x7b, 0x53, 0x01, 0x00, 0x00, 0x00,
    0xe1, 0x12, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0x0d, 0x02, 0x00, 0x00, 0xc9, 0x02, 0x00, 0x00, 0x1d, 0x0b, 0x00, 0x00,
    0x39, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0xf0, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43,
    0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00,
    0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44,
    0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x52, 0x54, 0x53,
    0x30, 0xb4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00,
    0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x01, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0x9c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0x4c, 0x08, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0x13, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x34, 0x08, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x0a, 0x02, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4b, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x2c, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x39, 0x82,
    0xa0, 0x18, 0x52, 0x2c, 0x21, 0x2e, 0xe1, 0x81, 0x80, 0x61, 0x04, 0x01,
    0xb8, 0x4b, 0x9a, 0x22, 0x4a, 0x98, 0xfc, 0x14, 0xb9, 0x88, 0x85, 0x3d,
    0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x84, 0x1d, 0xd2, 0x41,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21,
    0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xe9, 0x01,
    0x00, 0x0f, 0x79, 0x12, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0xc0,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00,
    0x00, 0x16, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x5a, 0x05, 0x42, 0xa2,
    0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c,
    0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8, 0x3c, 0x0a, 0xa2, 0x6c,
    0x68, 0x15, 0xc1, 0x08, 0x40, 0x21, 0x94, 0x41, 0x49, 0xd0, 0x99, 0x01,
    0xa0, 0x32, 0x03, 0x40, 0x64, 0x06, 0x80, 0xc6, 0x0c, 0x00, 0x85, 0x19,
    0x00, 0xe2, 0x33, 0x00, 0xd4, 0xc7, 0x72, 0x18, 0x02, 0x00, 0x00, 0xe0,
    0x38, 0x00, 0x20, 0x10, 0x08, 0x04, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00,
    0x00, 0xbf, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
    0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b,
    0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b,
    0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9,
    0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13,
    0x04, 0x41, 0x99, 0x20, 0x08, 0xcb, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04,
    0x41, 0x01, 0x6e, 0x6e, 0x82, 0x20, 0x30, 0x1b, 0x86, 0x03, 0x21, 0x26,
    0x08, 0x05, 0x18, 0x70, 0xa1, 0x43, 0x2b, 0xa3, 0x2a, 0xc3, 0xa3, 0xab,
    0x93, 0x2b, 0xcb, 0x9a, 0x20, 0x08, 0xcd, 0x04, 0x41, 0x70, 0x36, 0x08,
    0x44, 0xb3, 0x21, 0x21, 0x94, 0x85, 0x20, 0x06, 0x86, 0x70, 0xc8, 0xd0,
    0xa1, 0x95, 0x51, 0x95, 0xe1, 0xd1, 0xd5, 0xc9, 0x95, 0x55, 0x59, 0x6d,
    0x48, 0x06, 0x05, 0x22, 0x86, 0x81, 0x21, 0x9c, 0x0d, 0xc2, 0x13, 0x4d,
    0x10, 0x8e, 0x8f, 0x0a, 0x1d, 0x5a, 0xd9, 0x54, 0x58, 0x1b, 0x1c, 0x5b,
    0x99, 0xdc, 0x06, 0x84, 0x98, 0x28, 0x82, 0x18, 0x08, 0x60, 0x43, 0x50,
    0x6d, 0x20, 0x24, 0x00, 0xb0, 0x26, 0x08, 0x44, 0x18, 0x4c, 0x10, 0x84,
    0x87, 0x01, 0xda, 0x04, 0x41, 0x80, 0x26, 0x08, 0x42, 0xb4, 0xc1, 0x40,
    0xb4, 0x8d, 0xe0, 0x1a, 0x12, 0x6d, 0x69, 0x70, 0x73, 0x13, 0x04, 0x41,
    0xda, 0x40, 0x20, 0xde, 0xf6, 0x4d, 0x10, 0x82, 0x31, 0xd8, 0x20, 0x10,
    0x61, 0xb0, 0x21, 0x10, 0x83, 0x0d, 0x02, 0x31, 0x06, 0x1b, 0x88, 0xac,
    0x03, 0x03, 0x32, 0x98, 0x20, 0x0c, 0x62, 0x30, 0x41, 0x10, 0x26, 0x1a,
    0x68, 0x61, 0x6e, 0x64, 0x6c, 0x65, 0x13, 0x04, 0x81, 0xda, 0x60, 0x20,
    0x68, 0xb0, 0x11, 0x5c, 0x1a, 0x6c, 0x10, 0xce, 0x40, 0x0d, 0x26, 0x08,
    0x1c, 0x19, 0x4c, 0x10, 0x84, 0x8a, 0x03, 0xdc, 0xdb, 0x1c, 0x97, 0x29,
    0xab, 0x2f, 0xa8, 0xa7, 0xa9, 0x24, 0xaa, 0xa4, 0x27, 0xa7, 0x0d, 0x08,
    0xe2, 0x06, 0x1b, 0x71, 0x06, 0x6f, 0xc0, 0x35, 0x1c, 0xe8, 0xca, 0xf0,
    0x98, 0x50, 0x15, 0x61, 0x0d, 0x3d, 0x3d, 0x49, 0x11, 0xc1, 0x6c, 0x40,
    0x90, 0x38, 0xd8, 0xbe, 0x33, 0x90, 0x03, 0xae, 0x61, 0x31, 0xf6, 0xc6,
    0xf6, 0x26, 0x37, 0x41, 0x10, 0x2c, 0x1a, 0x43, 0x4f, 0x4c, 0x4f, 0x52,
    0x30, 0x1b, 0x10, 0x84, 0x0e, 0xb6, 0x3a, 0x38, 0x03, 0x3b, 0xe0, 0x9a,
    0x0d, 0x44, 0x1b, 0xc0, 0xc1, 0x1c, 0xdc, 0xc1, 0x86, 0x83, 0xc0, 0xca,
    0xc0, 0x0c, 0xd6, 0x80, 0x0d, 0xf0, 0x60, 0x82, 0xa0, 0x0c, 0x1b, 0x80,
    0x0d, 0x03, 0xb1, 0x07, 0x7b, 0xb0, 0x21, 0xe0, 0x83, 0x0d, 0xc3, 0xa0,
    0x07, 0x7d, 0x30, 0x41, 0xe8, 0xca, 0x60, 0x43, 0xf0, 0x07, 0x24, 0xda,
    0xc2, 0xd2, 0xdc, 0xb8, 0x4c, 0x59, 0x7d, 0x41, 0xbd, 0xcd, 0xa5, 0xd1,
    0xa5, 0xbd, 0xb9, 0x4d, 0x10, 0x16, 0x6f, 0x82, 0xb0, 0x74, 0x1b, 0x02,
    0x62, 0x82, 0xb0, 0x70, 0x13, 0x84, 0x65, 0xdb, 0xb0, 0x10, 0xa2, 0x30,
    0x0a, 0xa4, 0x50, 0x0a, 0xa6, 0x30, 0x98, 0x02, 0x71, 0x0a, 0x00, 0x11,
    0xaa, 0x22, 0xac, 0xa1, 0xa7, 0x27, 0x29, 0xa2, 0x09, 0xc2, 0xa2, 0x6d,
    0x10, 0xb6, 0x6d, 0xc3, 0x32, 0xa4, 0xc2, 0x28, 0x9c, 0x42, 0x29, 0xa8,
    0xc2, 0xa0, 0x0a, 0xc3, 0x29, 0xac, 0x02, 0x8b, 0xa1, 0x27, 0xa6, 0x27,
    0xa9, 0x09, 0x82, 0x70, 0x6d, 0x10, 0x36, 0x57, 0xd8, 0xb0, 0x30, 0xad,
    0x30, 0x0a, 0xa7, 0x50, 0x0a, 0xaa, 0x30, 0x98, 0x02, 0x73, 0x0a, 0xaf,
    0xb0, 0x61, 0x40, 0x05, 0x56, 0x80, 0x05, 0x26, 0x53, 0x56, 0x5f, 0x54,
    0x61, 0x72, 0x67, 0x65, 0x74, 0x13, 0x84, 0x25, 0xd9, 0xb0, 0x10, 0xb2,
    0x30, 0x0a, 0xb3, 0x50, 0x0a, 0xa7, 0x30, 0x98, 0x02, 0x71, 0x0a, 0xaf,
    0xb0, 0x21, 0xa0, 0x85, 0x0d, 0x43, 0x2c, 0xd4, 0x02, 0xb0, 0xa1, 0xd0,
    0x83, 0x50, 0xb0, 0x85, 0x0b, 0xa0, 0x61, 0xc6, 0xf6, 0x16, 0x46, 0x37,
    0x37, 0x41, 0x10, 0x30, 0x16, 0x69, 0x6e, 0x73, 0x74, 0x73, 0x13, 0x04,
    0x21, 0xa3, 0x31, 0x97, 0x76, 0xf6, 0xc5, 0x46, 0x46, 0x63, 0x2e, 0xed,
    0xec, 0x6b, 0x8e, 0x8e, 0x08, 0x5d, 0x19, 0xde, 0x97, 0xdb, 0x9b, 0x5c,
    0xdb, 0x06, 0x05, 0x17, 0x72, 0x41, 0x17, 0x76, 0x81, 0x17, 0x90, 0x5e,
    0x38, 0x03, 0x5f, 0x60, 0xaa, 0xb0, 0xb1, 0xd9, 0xb5, 0xb9, 0xa4, 0x91,
    0x95, 0xb9, 0xd1, 0x4d, 0x09, 0x82, 0x2a, 0x64, 0x78, 0x2e, 0x76, 0x65,
    0x72, 0x73, 0x69, 0x6f, 0x6e, 0x53, 0x02, 0xa2, 0x09, 0x19, 0x9e, 0x8b,
    0x5d, 0x18, 0x9b, 0x5d, 0x99, 0xdc, 0x94, 0xa0, 0xa8, 0x43, 0x86, 0xe7,
    0x32, 0x87, 0x16, 0x46, 0x56, 0x26, 0xd7, 0xf4, 0x46, 0x56, 0xc6, 0x36,
    0x25, 0x40, 0xca, 0x90, 0xe1, 0xb9, 0xc8, 0x95, 0xcd, 0xbd, 0xd5, 0xc9,
    0x8d, 0x95, 0xcd, 0x4d, 0x09, 0xac, 0x4a, 0x64, 0x78, 0x2e, 0x74, 0x79,
    0x70, 0x65, 0x41, 0x6e, 0x6e, 0x6f, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e,
    0x73, 0x53, 0x04, 0x3c, 0xe8, 0x83, 0x3a, 0x64, 0x78, 0x2e, 0x76, 0x69,
    0x65, 0x77, 0x49, 0x64, 0x53, 0x74, 0x61, 0x74, 0x65, 0x53, 0x82, 0x3f,
    0xa8, 0x43, 0x86, 0xe7, 0x52, 0xe6, 0x46, 0x27, 0x97, 0x07, 0xf5, 0x96,
    0xe6, 0x46, 0x37, 0x37, 0x25, 0xb0, 0x85, 0x2e, 0x64, 0x78, 0x2e, 0x63,
    0x6f, 0x75, 0x6e, 0x74, 0x65, 0x72, 0x73, 0x53, 0x02, 0x5f, 0x00, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80,
    0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84,
    0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c,
    0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42,
    0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88,
    0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c,
    0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79,
    0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70,
    0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f,
    0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4,
    0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30,
    0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc,
    0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b,
    0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70,
    0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76,
    0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72,
    0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e,
    0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1,
    0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21,
    0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8,
    0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94,
    0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc,
    0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c,
    0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e,
    0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e,
    0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00,
    0x00, 0x71, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x25, 0xf0, 0x05,
    0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9, 0x65, 0x39, 0x10, 0x38,
    0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f, 0xd2, 0xc3, 0xf4, 0x32,
    0x10, 0x18, 0xac, 0x80, 0x38, 0x08, 0xfc, 0xe8, 0xe8, 0x32, 0xb5, 0x8c,
    0xa7, 0xd7, 0xe5, 0xe5, 0xaa, 0x15, 0x08, 0x9c, 0x59, 0x7f, 0x24, 0x6a,
    0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0, 0xfa, 0x23, 0xd9, 0xcb,
    0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x10, 0x08, 0x0c,
    0x16, 0x00, 0x1c, 0x04, 0x7e, 0x74, 0x74, 0x99, 0x5a, 0xc6, 0xd3, 0xeb,
    0xf2, 0x72, 0x16, 0x08, 0x9c, 0x59, 0x7f, 0x24, 0x6a, 0x19, 0x4f, 0xaf,
    0xcb, 0xcb, 0x32, 0x22, 0xd0, 0xfa, 0x23, 0xd9, 0xcb, 0x63, 0xfa, 0x5b,
    0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x10, 0x08, 0x0c, 0x1a, 0x82, 0x34,
    0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0,
    0x10, 0x76, 0xe0, 0x0c, 0x97, 0xef, 0x3c, 0xfe, 0xe0, 0x4c, 0xb7, 0x5f,
    0xdc, 0xb6, 0x11, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f, 0xbf, 0x38, 0xc0, 0x20,
    0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x9b, 0x01, 0x34, 0x5c, 0xbe, 0xf3,
    0xf8, 0x12, 0xc0, 0x3c, 0x0b, 0xe1, 0x17, 0xb7, 0x6d, 0x05, 0xd5, 0x70,
    0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28, 0x35, 0x3d, 0xd4, 0xe4,
    0x17, 0xb7, 0x6d, 0x03, 0x04, 0x03, 0x20, 0x0d, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x3d, 0xf1, 0x38, 0x49, 0x7f, 0x91, 0x6b, 0xbb, 0x2e, 0x0c, 0x31,
    0x06, 0xa9, 0xa3, 0xcd, 0x81, 0x44, 0x58, 0x49, 0x4c, 0xa0, 0x07, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0xe8, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x88, 0x07, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xdf, 0x01, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x45, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23, 0x00, 0x25, 0x00, 0x14,
    0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84,
    0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0xa1, 0x9b, 0x86, 0xcb,
    0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56, 0x62, 0xf2, 0x8b, 0xdb,
    0x46, 0xc5, 0x18, 0x63, 0x10, 0x2a, 0xf7, 0x0c, 0x97, 0x3f, 0x61, 0x0f,
    0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82, 0x55, 0x18, 0x45, 0x18,
    0x1b, 0x63, 0x0c, 0x42, 0xc8, 0xa0, 0x56, 0x90, 0x41, 0xc6, 0x18, 0x63,
    0x0c, 0x7a, 0x73, 0x04, 0x41, 0x31, 0x18, 0x29, 0x84, 0x44, 0x92, 0x03,
    0x01, 0xc3, 0x08, 0xc4, 0x30, 0x53, 0x1b, 0x8c, 0x03, 0x3b, 0x84, 0xc3,
    0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0, 0x03, 0x3d, 0xd4, 0x83,
    0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8, 0x43, 0x39, 0x8c, 0x03,
    0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03,
    0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80, 0x81, 0x1f, 0xe8, 0x81,
    0x1e, 0xb4, 0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc, 0x02, 0x3d, 0xe4, 0x03,
    0x3c, 0x94, 0x03, 0x0a, 0xc8, 0x4c, 0x62, 0x30, 0x0e, 0xec, 0x10, 0x0e,
    0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f,
    0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60, 0x0f, 0xe5, 0x30, 0x0e,
    0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xe6, 0xc0, 0x0e, 0xef, 0x10, 0x0e,
    0xf4, 0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00, 0x06, 0x7e, 0x80, 0x84,
    0x6a, 0xe9, 0xde, 0x24, 0x4d, 0x11, 0x25, 0x4c, 0x3e, 0x0b, 0x30, 0xcf,
    0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x08, 0xe5, 0x74, 0x20, 0x00,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10,
    0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2,
    0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30,
    0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x40, 0x16, 0x08, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98,
    0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43,
    0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50, 0x12, 0x65, 0x50, 0x1e,
    0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0, 0x08, 0x0a, 0x84, 0xf0,
    0x0c, 0x00, 0xe9, 0x19, 0x00, 0xda, 0x63, 0x39, 0x0c, 0x01, 0x00, 0x00,
    0x70, 0x1c, 0x00, 0x10, 0x08, 0x04, 0x02, 0x00, 0x00, 0x79, 0x18, 0x00,
    0x00, 0x65, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13,
    0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b,
    0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b,
    0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9,
    0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13,
    0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61, 0x20, 0x26, 0x08, 0x04,
    0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26,
    0x08, 0xd8, 0x44, 0x60, 0x82, 0x40, 0x24, 0x13, 0x04, 0x42, 0xd9, 0x20,
    0x10, 0xcd, 0x86, 0x84, 0x50, 0x16, 0x82, 0x18, 0x18, 0xc2, 0xd9, 0x90,
    0x0c, 0xca, 0x42, 0x0c, 0x03, 0x43, 0x38, 0x1b, 0x84, 0x07, 0x9a, 0x20,
    0x68, 0xd4, 0x06, 0x84, 0x90, 0x16, 0x82, 0x18, 0x08, 0x60, 0x43, 0x30,
    0x6d, 0x20, 0x22, 0x00, 0xa0, 0x26, 0x08, 0x5b, 0xb5, 0x21, 0xb0, 0x26,
    0x08, 0x02, 0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x8d, 0xcb, 0x94, 0xd5, 0x17,
    0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb, 0x04, 0xa1, 0x68, 0x26,
    0x08, 0x85, 0xb3, 0x21, 0x20, 0x26, 0x08, 0xc5, 0x33, 0x41, 0x28, 0xa0,
    0x0d, 0x0b, 0xa1, 0x6d, 0x5c, 0xe7, 0x0d, 0x1e, 0xf1, 0x01, 0x44, 0xa8,
    0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26, 0x08, 0x45, 0x34, 0x41,
    0x20, 0x96, 0x0d, 0xc2, 0x18, 0x8c, 0xc1, 0x86, 0x65, 0x08, 0x83, 0xed,
    0xeb, 0xc4, 0x60, 0x10, 0x83, 0xe1, 0x23, 0x03, 0x16, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x13, 0x04, 0x82, 0xd9, 0x20, 0x8c, 0xc1, 0x19, 0x6c, 0x58,
    0x18, 0x33, 0xd8, 0xbe, 0x4e, 0x0c, 0x06, 0x8f, 0xf9, 0xd0, 0x60, 0xc3,
    0x00, 0x06, 0x65, 0x90, 0x06, 0x4c, 0xa6, 0xac, 0xbe, 0xa8, 0xc2, 0xe4,
    0xce, 0xca, 0xe8, 0x26, 0x08, 0x85, 0xb4, 0x61, 0x21, 0xd6, 0x60, 0x63,
    0x83, 0xee, 0x1b, 0x3c, 0xe2, 0x43, 0x83, 0x0d, 0x41, 0x1b, 0x6c, 0x18,
    0xd4, 0xc0, 0x0d, 0x80, 0x0d, 0x05, 0x96, 0xbd, 0x41, 0x05, 0x54, 0x61,
    0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04,
    0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6,
    0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9,
    0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c,
    0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x80, 0x94, 0x21, 0xc3, 0x73,
    0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x50,
    0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8,
    0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x56, 0x1d, 0x32, 0x3c, 0x97, 0x32, 0x37,
    0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba, 0xb9, 0x29, 0xc1, 0x1b,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80,
    0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84,
    0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c,
    0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42,
    0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88,
    0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c,
    0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79,
    0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70,
    0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f,
    0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4,
    0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30,
    0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc,
    0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b,
    0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70,
    0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76,
    0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72,
    0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e,
    0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1,
    0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21,
    0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8,
    0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94,
    0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc,
    0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c,
    0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e,
    0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e,
    0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00,
    0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x56, 0x20, 0x0d,
    0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44, 0x84, 0x40, 0x33, 0x2c,
    0x84, 0x11, 0x38, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x38, 0xd3, 0xed, 0x17,
    0xb7, 0x6d, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e, 0x30, 0x88,
    0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x00, 0x0d, 0x97, 0xef, 0x3c,
    0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d, 0x9b, 0x40, 0x35, 0x5c,
    0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a, 0x4d, 0x0f, 0x35, 0xf9,
    0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03, 0x00, 0x61, 0x20, 0x00,
    0x00, 0x5d, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x14, 0x47, 0x00, 0xa8, 0x94, 0x00, 0x91,
    0xc2, 0x2b, 0xb7, 0x92, 0x2b, 0x85, 0x42, 0x98, 0x01, 0xa0, 0x31, 0x46,
    0x50, 0x9e, 0x74, 0xe9, 0x7f, 0x63, 0x04, 0xa2, 0x3e, 0xb7, 0xf3, 0x2f,
    0x8c, 0x11, 0x80, 0xef, 0x0a, 0xae, 0xbf, 0x30, 0x46, 0xc0, 0x97, 0xbd,
    0xb9, 0x7f, 0x23, 0x00, 0x63, 0x04, 0x21, 0x09, 0x86, 0xf0, 0x2f, 0x8c,
    0x11, 0x80, 0x20, 0x08, 0xe2, 0xdf, 0x0c, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x20, 0x85, 0x01, 0x63, 0x81, 0x01, 0x18, 0x40, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x89, 0x41, 0x73, 0x81, 0x01, 0x18,
    0x44, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x8d, 0x81, 0xa3, 0x85,
    0x41, 0x18, 0x48, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xa0, 0x41,
    0x03, 0x06, 0x62, 0x90, 0x31, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60,
    0xa4, 0x81, 0x13, 0x06, 0x63, 0x30, 0x35, 0x23, 0x06, 0x09, 0x00, 0x82,
    0x60, 0x60, 0xa8, 0xc1, 0x23, 0x06, 0x64, 0xa0, 0x39, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x60, 0xac, 0x01, 0x34, 0x06, 0x65, 0xe0, 0x3d, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb0, 0x41, 0x74, 0x06, 0x66, 0xd0,
    0x41, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb4, 0x81, 0x84, 0x06,
    0x67, 0x70, 0x45, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xb0, 0xc1,
    0x85, 0x1c, 0x42, 0xa0, 0x28, 0x68, 0x80, 0x06, 0x92, 0x32, 0x9a, 0x10,
    0x00, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xb8, 0x41, 0xb6, 0x24,
    0xc4, 0xc0, 0x30, 0x6a, 0xa0, 0x06, 0x14, 0x33, 0x9a, 0x10, 0x00, 0xa3,
    0x09, 0x42, 0x60, 0x41, 0x04, 0x1f, 0x1b, 0x24, 0xf8, 0x8c, 0x18, 0x2c,
    0x00, 0x08, 0x82, 0xc1, 0x23, 0x07, 0x9d, 0x21, 0x04, 0x12, 0x55, 0x8d,
    0x18, 0x2c, 0x00, 0x08, 0x82, 0xc1, 0x33, 0x07, 0xde, 0x31, 0x08, 0xd3,
    0x85, 0x8d, 0x18, 0x2c, 0x00, 0x08, 0x82, 0xc1, 0x43, 0x07, 0x1f, 0x42,
    0x0c, 0x94, 0x66, 0xd9, 0x00, 0xc9, 0xc7, 0x06, 0x48, 0x3e, 0x36, 0x40,
    0xf2, 0x19, 0x31, 0x48, 0x00, 0x10, 0x04, 0x03, 0x44, 0x0f, 0xc2, 0xa0,
    0x0e, 0xea, 0x80, 0x0d, 0x86, 0x11, 0x83, 0x04, 0x00, 0x41, 0x30, 0x40,
    0xf4, 0x20, 0x0c, 0xea, 0xa0, 0x0e, 0xca, 0x40, 0x18, 0x31, 0x48, 0x00,
    0x10, 0x04, 0x03, 0x44, 0x0f, 0xc2, 0xa0, 0x0e, 0xea, 0x60, 0x0d, 0x82,
    0x11, 0x83, 0x04, 0x00, 0x41, 0x30, 0x40, 0xf4, 0x20, 0x0c, 0xea, 0xa0,
    0x0e, 0xdc, 0x00, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* The yuv-rendering pixel shader:

    --- D3D12_PixelShader_NV21_BT601.hlsl ---
    Texture2D theTextureY : register(t0);
    Texture2D theTextureUV : register(t1);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define NVRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(NVRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        const float3 offset = {-0.0627451017, -0.501960814, -0.501960814};
        const float3 Rcoeff = {1.1644,  0.0000,  1.5960};
        const float3 Gcoeff = {1.1644, -0.3918, -0.8130};
        const float3 Bcoeff = {1.1644,  2.0172,  0.0000};

        float4 Output;

        float3 yuv;
        yuv.x = theTextureY.Sample(theSampler, input.tex).r;
        yuv.yz = theTextureUV.Sample(theSampler, input.tex).gr;

        yuv += offset;
        Output.r = dot(yuv, Rcoeff);
        Output.g = dot(yuv, Gcoeff);
        Output.b = dot(yuv, Bcoeff);
        Output.a = 1.0f;

        return Output * input.color;
    }

*/

static unsigned char D3D12_PixelShader_NV21_BT601[] = {
    0x44, 0x58, 0x42, 0x43, 0x06, 0x29, 0xca, 0x47, 0x9e, 0x47, 0xed, 0x99,
    0xcf, 0xa5, 0xa6, 0xdb, 0x0b, 0x0d, 0xa5, 0x31, 0x01, 0x00, 0x00, 0x00,
    0xe9, 0x12, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0x0d, 0x02, 0x00, 0x00, 0xc9, 0x02, 0x00, 0x00, 0x15, 0x0b, 0x00, 0x00,
    0x31, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0xf0, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43,
    0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00,
    0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44,
    0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x52, 0x54, 0x53,
    0x30, 0xb4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00,
    0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x01, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0x9c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0x44, 0x08, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0x11, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x2c, 0x08, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x08, 0x02, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4b, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x2c, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x39, 0x82,
    0xa0, 0x18, 0x52, 0x2c, 0x21, 0x2e, 0xe1, 0x81, 0x80, 0x61, 0x04, 0x01,
    0xb8, 0x4b, 0x9a, 0x22, 0x4a, 0x98, 0xfc, 0x14, 0xb9, 0x88, 0x85, 0x3d,
    0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x84, 0x1d, 0xd2, 0x41,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21,
    0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xe9, 0x01,
    0x00, 0x0f, 0x79, 0x12, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0xc0,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00,
    0x00, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x5a, 0x05, 0x42, 0xa2,
    0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c,
    0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8, 0x3c, 0x0a, 0x87, 0x56,
    0x11, 0x8c, 0x00, 0x14, 0x42, 0x19, 0x94, 0x04, 0x9d, 0x19, 0x00, 0x2a,
    0x33, 0x00, 0x44, 0x66, 0x00, 0x68, 0xcc, 0x00, 0x50, 0x98, 0x01, 0x20,
    0x3e, 0x03, 0x40, 0x7d, 0x2c, 0x87, 0x21, 0x00, 0x00, 0x00, 0x8e, 0x03,
    0x00, 0x02, 0x81, 0x40, 0x00, 0x79, 0x18, 0x00, 0x00, 0xbe, 0x00, 0x00,
    0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63,
    0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03,
    0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a,
    0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b,
    0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0x41, 0x99, 0x20,
    0x08, 0xcb, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04, 0x41, 0x01, 0x6e, 0x6e,
    0x82, 0x20, 0x30, 0x1b, 0x86, 0x03, 0x21, 0x26, 0x08, 0xc5, 0xc7, 0x85,
    0x0e, 0xad, 0x8c, 0xaa, 0x0c, 0x8f, 0xae, 0x4e, 0xae, 0x2c, 0x6b, 0x82,
    0x20, 0x34, 0x13, 0x04, 0xc1, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84, 0x50,
    0x16, 0x82, 0x18, 0x18, 0xc2, 0x21, 0x43, 0x87, 0x56, 0x46, 0x55, 0x86,
    0x47, 0x57, 0x27, 0x57, 0x56, 0x65, 0xb5, 0x21, 0x19, 0x14, 0x88, 0x18,
    0x06, 0x86, 0x70, 0x36, 0x08, 0x4f, 0x34, 0x41, 0x38, 0x3c, 0x2a, 0x74,
    0x68, 0x65, 0x53, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x72, 0x1b, 0x10, 0x62,
    0xa2, 0x08, 0x62, 0x20, 0x80, 0x0d, 0x41, 0xb5, 0x81, 0x90, 0x00, 0xc0,
    0x9a, 0x20, 0x10, 0x60, 0x30, 0x41, 0x10, 0x1e, 0x06, 0x68, 0x13, 0x04,
    0x01, 0x9a, 0x20, 0x08, 0xd1, 0x06, 0x03, 0xd1, 0x36, 0x82, 0x6b, 0x48,
    0xb4, 0xa5, 0xc1, 0xcd, 0x4d, 0x10, 0x04, 0x69, 0x03, 0x81, 0x78, 0xdb,
    0x37, 0x41, 0x08, 0xc4, 0x60, 0x83, 0x40, 0x84, 0xc1, 0x86, 0x40, 0x0c,
    0x36, 0x08, 0xc4, 0x18, 0x6c, 0x20, 0xb2, 0x0e, 0x0c, 0xc8, 0x60, 0x82,
    0x30, 0x84, 0xc1, 0x04, 0x41, 0x98, 0x68, 0xa0, 0x85, 0xb9, 0x91, 0xb1,
    0x95, 0x4d, 0x10, 0x04, 0x6a, 0x83, 0x81, 0xa0, 0xc1, 0x46, 0x70, 0x69,
    0xb0, 0x41, 0x38, 0x03, 0x35, 0x98, 0x20, 0x70, 0x63, 0x30, 0x41, 0x10,
    0x2a, 0x0e, 0x70, 0x6f, 0x73, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0x9e, 0xa6,
    0x92, 0xa8, 0x92, 0x9e, 0x9c, 0x36, 0x20, 0x88, 0x1b, 0x6c, 0xc4, 0x19,
    0xbc, 0x01, 0xd7, 0x70, 0xa0, 0x2b, 0xc3, 0x63, 0x42, 0x55, 0x84, 0x35,
    0xf4, 0xf4, 0x24, 0x45, 0x04, 0xb3, 0x01, 0x41, 0xe2, 0x60, 0xfb, 0xce,
    0x40, 0x0e, 0xb8, 0x86, 0xc5, 0xd8, 0x1b, 0xdb, 0x9b, 0xdc, 0x04, 0x41,
    0xb0, 0x68, 0x0c, 0x3d, 0x31, 0x3d, 0x49, 0xc1, 0x6c, 0x40, 0x10, 0x3a,
    0xd8, 0xea, 0xe0, 0x0c, 0xec, 0x80, 0x6b, 0x36, 0x10, 0x6d, 0x00, 0x07,
    0x73, 0x70, 0x07, 0x1b, 0x0e, 0x02, 0x2b, 0x03, 0x33, 0x58, 0x03, 0x36,
    0xc0, 0x83, 0x09, 0x82, 0x32, 0x6c, 0x00, 0x36, 0x0c, 0xc4, 0x1e, 0xec,
    0xc1, 0x86, 0x80, 0x0f, 0x36, 0x0c, 0x83, 0x1e, 0xf4, 0xc1, 0x04, 0xa1,
    0x23, 0x83, 0x0d, 0xc1, 0x1f, 0x90, 0x68, 0x0b, 0x4b, 0x73, 0xe3, 0x32,
    0x65, 0xf5, 0x05, 0xf5, 0x36, 0x97, 0x46, 0x97, 0xf6, 0xe6, 0x36, 0x41,
    0x58, 0xba, 0x09, 0xc2, 0xc2, 0x6d, 0x08, 0x88, 0x09, 0xc2, 0xb2, 0x4d,
    0x10, 0x16, 0x6d, 0xc3, 0x42, 0x88, 0xc2, 0x28, 0x90, 0x42, 0x29, 0x98,
    0xc2, 0x60, 0x0a, 0xc4, 0x29, 0x00, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e,
    0x9e, 0xa4, 0x88, 0x26, 0x08, 0x4b, 0xb6, 0x41, 0xd8, 0xb6, 0x0d, 0xcb,
    0x90, 0x0a, 0xa3, 0x70, 0x0a, 0xa5, 0xa0, 0x0a, 0x83, 0x2a, 0x0c, 0xa7,
    0xb0, 0x0a, 0x2c, 0x86, 0x9e, 0x98, 0x9e, 0xa4, 0x26, 0x08, 0xc2, 0xb5,
    0x41, 0xd8, 0x5c, 0x61, 0xc3, 0xc2, 0xb4, 0xc2, 0x28, 0x9c, 0x42, 0x29,
    0xa8, 0xc2, 0x60, 0x0a, 0xcc, 0x29, 0xbc, 0xc2, 0x86, 0x01, 0x15, 0x58,
    0x01, 0x16, 0x98, 0x4c, 0x59, 0x7d, 0x51, 0x85, 0xc9, 0x9d, 0x95, 0xd1,
    0x4d, 0x10, 0x96, 0x64, 0xc3, 0x42, 0xc8, 0xc2, 0x28, 0xcc, 0x42, 0x29,
    0x9c, 0xc2, 0x60, 0x0a, 0xc4, 0x29, 0xbc, 0xc2, 0x86, 0x80, 0x16, 0x36,
    0x0c, 0xb1, 0x50, 0x0b, 0xc0, 0x86, 0x42, 0x0f, 0x42, 0xc1, 0x16, 0x2e,
    0x80, 0x86, 0x19, 0xdb, 0x5b, 0x18, 0xdd, 0x1c, 0x8b, 0x34, 0xb7, 0x39,
    0xba, 0xb9, 0x09, 0x82, 0x80, 0xd1, 0x98, 0x4b, 0x3b, 0xfb, 0x62, 0x23,
    0xa3, 0x31, 0x97, 0x76, 0xf6, 0x35, 0x47, 0x47, 0x84, 0xae, 0x0c, 0xef,
    0xcb, 0xed, 0x4d, 0xae, 0x6d, 0x83, 0x82, 0x0b, 0x4d, 0x2e, 0xe8, 0xc2,
    0x2e, 0x20, 0xbc, 0x70, 0x06, 0xbd, 0xc0, 0x54, 0x61, 0x63, 0xb3, 0x6b,
    0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0,
    0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13,
    0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x41, 0x51,
    0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d,
    0xac, 0x8c, 0x6d, 0x4a, 0x80, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b,
    0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x58, 0x95, 0xc8, 0xf0,
    0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2, 0xe8,
    0xd2, 0xde, 0xdc, 0xe6, 0xa6, 0x08, 0x78, 0xd0, 0x07, 0x75, 0xc8, 0xf0,
    0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca,
    0xa6, 0x04, 0x7f, 0x50, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e,
    0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x60, 0x0b, 0x5d, 0xc8,
    0xf0, 0x5c, 0xc6, 0xde, 0xea, 0xdc, 0xe8, 0xca, 0xe4, 0xe6, 0xa6, 0x04,
    0xbd, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00,
    0x00, 0x25, 0xf0, 0x05, 0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9,
    0x65, 0x39, 0x10, 0x38, 0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f,
    0xd2, 0xc3, 0xf4, 0x32, 0x10, 0x18, 0xac, 0x80, 0x38, 0x08, 0xfc, 0xe8,
    0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7, 0xe5, 0xe5, 0xaa, 0x15, 0x08, 0x9c,
    0x59, 0x7f, 0x24, 0x6a, 0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0,
    0xfa, 0x23, 0xd9, 0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33,
    0x20, 0x10, 0x08, 0x0c, 0x16, 0x00, 0x1c, 0x04, 0x7e, 0x74, 0x74, 0x99,
    0x5a, 0xc6, 0xd3, 0xeb, 0xf2, 0x72, 0x16, 0x08, 0x9c, 0x59, 0x7f, 0x24,
    0x6a, 0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0, 0xfa, 0x23, 0xd9,
    0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x10, 0x08,
    0x0c, 0x1a, 0x82, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13,
    0x11, 0x02, 0xcd, 0xb0, 0x10, 0x76, 0xe0, 0x0c, 0x97, 0xef, 0x3c, 0xfe,
    0xe0, 0x4c, 0xb7, 0x5f, 0xdc, 0xb6, 0x11, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f,
    0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x9b, 0x01,
    0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x12, 0xc0, 0x3c, 0x0b, 0xe1, 0x17, 0xb7,
    0x6d, 0x05, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28,
    0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x03, 0x04, 0x03, 0x20, 0x0d,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x6a, 0x46, 0xfe, 0xd5, 0x21, 0x7d,
    0xa1, 0xb7, 0x81, 0xb9, 0x35, 0xf9, 0x0f, 0x28, 0x1d, 0x44, 0x58, 0x49,
    0x4c, 0xb0, 0x07, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xec, 0x01, 0x00,
    0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x98, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00,
    0x00, 0xe3, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04,
    0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08,
    0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b,
    0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52,
    0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32,
    0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81,
    0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00,
    0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff,
    0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03,
    0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8,
    0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60,
    0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00,
    0x00, 0x45, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85,
    0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90,
    0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23,
    0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00,
    0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06,
    0xa1, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56,
    0x62, 0xf2, 0x8b, 0xdb, 0x46, 0xc5, 0x18, 0x63, 0x10, 0x2a, 0xf7, 0x0c,
    0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82,
    0x55, 0x18, 0x45, 0x18, 0x1b, 0x63, 0x0c, 0x42, 0xc8, 0xa0, 0x56, 0x90,
    0x41, 0xc6, 0x18, 0x63, 0x0c, 0x7a, 0x73, 0x04, 0x41, 0x31, 0x18, 0x29,
    0x84, 0x44, 0x92, 0x03, 0x01, 0xc3, 0x08, 0xc4, 0x30, 0x53, 0x1b, 0x8c,
    0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0,
    0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8,
    0x43, 0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0,
    0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80,
    0x81, 0x1f, 0xe8, 0x81, 0x1e, 0xb4, 0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc,
    0x02, 0x3d, 0xe4, 0x03, 0x3c, 0x94, 0x03, 0x0a, 0xc8, 0x4c, 0x62, 0x30,
    0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80,
    0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60,
    0x0f, 0xe5, 0x30, 0x0e, 0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xe6, 0xc0,
    0x0e, 0xef, 0x10, 0x0e, 0xf4, 0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00,
    0x06, 0x7e, 0x80, 0x84, 0x6a, 0xe9, 0xde, 0x24, 0x4d, 0x11, 0x25, 0x4c,
    0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x08,
    0xe5, 0x74, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60,
    0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf,
    0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a,
    0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
    0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
    0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6,
    0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73,
    0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74,
    0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71,
    0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43,
    0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x0f, 0x00, 0x00,
    0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26,
    0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50,
    0x12, 0x65, 0x50, 0x1e, 0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0,
    0x08, 0x0a, 0x84, 0xf0, 0x0c, 0x00, 0xe9, 0x19, 0x00, 0xda, 0x63, 0x39,
    0x0c, 0x01, 0x00, 0x00, 0x70, 0x1c, 0x00, 0x10, 0x08, 0x04, 0x02, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c,
    0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03,
    0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01,
    0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a,
    0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b,
    0xd9, 0x10, 0x04, 0x13, 0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61,
    0x20, 0x26, 0x08, 0x04, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b,
    0x06, 0xc4, 0x20, 0x26, 0x08, 0xd8, 0x44, 0x60, 0x82, 0x40, 0x24, 0x13,
    0x04, 0x42, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84, 0x50, 0x16, 0x82, 0x18,
    0x18, 0xc2, 0xd9, 0x90, 0x0c, 0xca, 0x42, 0x0c, 0x03, 0x43, 0x38, 0x1b,
    0x84, 0x07, 0x9a, 0x20, 0x68, 0xd4, 0x06, 0x84, 0x90, 0x16, 0x82, 0x18,
    0x08, 0x60, 0x43, 0x30, 0x6d, 0x20, 0x22, 0x00, 0xa0, 0x26, 0x08, 0x5b,
    0xb5, 0x21, 0xb0, 0x26, 0x08, 0x02, 0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x8d,
    0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb,
    0x04, 0xa1, 0x68, 0x26, 0x08, 0x85, 0xb3, 0x21, 0x20, 0x26, 0x08, 0xc5,
    0x33, 0x41, 0x28, 0xa0, 0x0d, 0x0b, 0xa1, 0x6d, 0x5c, 0xe7, 0x0d, 0x1e,
    0xf1, 0x01, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26,
    0x08, 0x45, 0x34, 0x41, 0x20, 0x96, 0x0d, 0xc2, 0x18, 0x8c, 0xc1, 0x86,
    0x65, 0x08, 0x83, 0xed, 0xeb, 0xc4, 0x60, 0x10, 0x83, 0xe1, 0x23, 0x03,
    0x16, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x13, 0x04, 0x82, 0xd9, 0x20, 0x8c,
    0xc1, 0x19, 0x6c, 0x58, 0x18, 0x33, 0xd8, 0xbe, 0x4e, 0x0c, 0x06, 0x8f,
    0xf9, 0xd0, 0x60, 0xc3, 0x00, 0x06, 0x65, 0x90, 0x06, 0x4c, 0xa6, 0xac,
    0xbe, 0xa8, 0xc2, 0xe4, 0xce, 0xca, 0xe8, 0x26, 0x08, 0x85, 0xb4, 0x61,
    0x21, 0xd6, 0x60, 0x63, 0x83, 0xee, 0x1b, 0x3c, 0xe2, 0x43, 0x83, 0x0d,
    0x41, 0x1b, 0x6c, 0x18, 0xd4, 0xc0, 0x0d, 0x80, 0x0d, 0x05, 0x96, 0xbd,
    0x41, 0x05, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73,
    0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6,
    0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
    0x36, 0xbb, 0x32, 0xb9, 0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e,
    0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x80,
    0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b,
    0x9b, 0x9b, 0x12, 0x50, 0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee,
    0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x56, 0x1d, 0x32,
    0x3c, 0x97, 0x32, 0x37, 0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba,
    0xb9, 0x29, 0xc1, 0x1b, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00,
    0x00, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44,
    0x84, 0x40, 0x33, 0x2c, 0x84, 0x11, 0x38, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f,
    0x38, 0xd3, 0xed, 0x17, 0xb7, 0x6d, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3,
    0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x00,
    0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d,
    0x9b, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a,
    0x4d, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03,
    0x00, 0x61, 0x20, 0x00, 0x00, 0x61, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41,
    0x2c, 0x10, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x14, 0x47, 0x00,
    0xa8, 0x94, 0x00, 0x91, 0xc2, 0x2b, 0xb7, 0x92, 0x2b, 0x85, 0x42, 0x98,
    0x01, 0xa0, 0x31, 0x46, 0xe0, 0xba, 0xa6, 0x08, 0x82, 0xc1, 0x18, 0x41,
    0x69, 0xa2, 0x60, 0xfd, 0x0b, 0x63, 0x04, 0x22, 0x6c, 0xc6, 0xec, 0x2f,
    0x8c, 0x11, 0xe8, 0xad, 0x8c, 0xf3, 0xdf, 0x18, 0x41, 0x48, 0x82, 0x21,
    0xee, 0x0b, 0x23, 0x00, 0x63, 0x04, 0x21, 0x09, 0x86, 0xf0, 0x2f, 0x8c,
    0x11, 0xbc, 0xb8, 0xa8, 0xe6, 0xdf, 0x0c, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x20, 0x89, 0x41, 0x73, 0x85, 0x41, 0x18, 0x44, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x8d, 0x81, 0x83, 0x85, 0x41, 0x18,
    0x48, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x91, 0xc1, 0xb3, 0x89,
    0x81, 0x18, 0x4c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xa4, 0x81,
    0x13, 0x06, 0x63, 0xa0, 0x35, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60,
    0xa8, 0xc1, 0x23, 0x06, 0x64, 0x40, 0x39, 0x23, 0x06, 0x09, 0x00, 0x82,
    0x60, 0x60, 0xac, 0x01, 0x34, 0x06, 0x65, 0xb0, 0x3d, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x60, 0xb0, 0x41, 0x44, 0x06, 0x66, 0xf0, 0x41, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb4, 0x81, 0x84, 0x06, 0x67, 0xe0,
    0x45, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb8, 0xc1, 0x94, 0x06,
    0x68, 0x80, 0x49, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xb4, 0x01,
    0x86, 0x1c, 0x42, 0xa0, 0x28, 0x69, 0x90, 0x06, 0x93, 0x32, 0x9a, 0x10,
    0x00, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xbc, 0x81, 0xb6, 0x24,
    0xc4, 0xc0, 0x30, 0x6b, 0xb0, 0x06, 0x15, 0x33, 0x9a, 0x10, 0x00, 0xa3,
    0x09, 0x42, 0x60, 0xc4, 0x04, 0x1f, 0x13, 0x24, 0xf8, 0x18, 0x31, 0xc1,
    0x67, 0xc4, 0x60, 0x01, 0x40, 0x10, 0x0c, 0x1e, 0x3a, 0xf8, 0x06, 0x21,
    0x98, 0xaa, 0x6b, 0xc4, 0x60, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x3a, 0x00,
    0x03, 0x62, 0x10, 0xa8, 0x4c, 0x1b, 0x31, 0x58, 0x00, 0x10, 0x04, 0x83,
    0xc7, 0x0e, 0xc2, 0xa0, 0x20, 0x86, 0x8a, 0xbb, 0x6c, 0x88, 0xe4, 0x63,
    0x43, 0x24, 0x1f, 0x1b, 0x22, 0xf9, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82,
    0x01, 0xc2, 0x07, 0x63, 0x70, 0x07, 0x77, 0xe0, 0x06, 0xc3, 0x88, 0x41,
    0x02, 0x80, 0x20, 0x18, 0x20, 0x7c, 0x30, 0x06, 0x77, 0x70, 0x07, 0x67,
    0x20, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x07, 0x63, 0x70,
    0x07, 0x77, 0xd0, 0x06, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20,
    0x7c, 0x30, 0x06, 0x77, 0x70, 0x07, 0x70, 0x10, 0x21, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00
};

/* The yuv-rendering pixel shader:

    --- D3D12_PixelShader_NV21_BT709.hlsl ---
    Texture2D theTextureY : register(t0);
    Texture2D theTextureUV : register(t1);
    SamplerState theSampler : register(s0);

    struct PixelShaderInput
    {
        float4 pos : SV_POSITION;
        float2 tex : TEXCOORD0;
        float4 color : COLOR0;
    };

    #define NVRS \
        "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
        "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
        "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
        "            DENY_HULL_SHADER_ROOT_ACCESS )," \
        "RootConstants(num32BitConstants=32, b0),"\
        "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
        "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

    [RootSignature(NVRS)]
    float4 main(PixelShaderInput input) : SV_TARGET
    {
        const float3 offset = {-0.0627451017, -0.501960814, -0.501960814};
        const float3 Rcoeff = {1.1644,  0.0000,  1.7927};
        const float3 Gcoeff = {1.1644, -0.2132, -0.5329};
        const float3 Bcoeff = {1.1644,  2.1124,  0.0000};

        float4 Output;

        float3 yuv;
        yuv.x = theTextureY.Sample(theSampler, input.tex).r;
        yuv.yz = theTextureUV.Sample(theSampler, input.tex).gr;

        yuv += offset;
        Output.r = dot(yuv, Rcoeff);
        Output.g = dot(yuv, Gcoeff);
        Output.b = dot(yuv, Bcoeff);
        Output.a = 1.0f;

        return Output * input.color;
    }

*/

static unsigned char D3D12_PixelShader_NV21_BT709[] = {
    0x44, 0x58, 0x42, 0x43, 0x5c, 0x52, 0x50, 0xb3, 0x0c, 0x9c, 0x81, 0x8a,
    0x2d, 0x9e, 0x1b, 0x5d, 0x0b, 0xcb, 0x2a, 0xa0, 0x01, 0x00, 0x00, 0x00,
    0xe9, 0x12, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xdb, 0x00, 0x00, 0x00, 0x15, 0x01, 0x00, 0x00,
    0x0d, 0x02, 0x00, 0x00, 0xc9, 0x02, 0x00, 0x00, 0x15, 0x0b, 0x00, 0x00,
    0x31, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x32, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x54, 0x61, 0x72, 0x67, 0x65, 0x74,
    0x00, 0x50, 0x53, 0x56, 0x30, 0xf0, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x03, 0x01, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43,
    0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00,
    0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44,
    0x00, 0x03, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x44, 0x10, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x07, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x52, 0x54, 0x53,
    0x30, 0xb4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00,
    0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x5c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff,
    0xff, 0x01, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00,
    0x00, 0x9c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0x44, 0x08, 0x00,
    0x00, 0x60, 0x00, 0x00, 0x00, 0x11, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49,
    0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x2c, 0x08, 0x00,
    0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x08, 0x02, 0x00,
    0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00,
    0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32,
    0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b,
    0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32,
    0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14,
    0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e,
    0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5,
    0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8,
    0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86,
    0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08,
    0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4b, 0x00, 0x00,
    0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4,
    0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a,
    0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x78, 0x33, 0x00, 0xc3, 0x08, 0x04,
    0x30, 0x47, 0x00, 0x06, 0x33, 0xb5, 0xc1, 0x38, 0xb0, 0x43, 0x38, 0xcc,
    0xc3, 0x3c, 0xb8, 0x01, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8,
    0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x3d, 0x94, 0xc3, 0x38, 0xd0,
    0xc3, 0x3b, 0xc8, 0x03, 0x1f, 0x98, 0x03, 0x3b, 0xbc, 0x43, 0x38, 0xd0,
    0x03, 0x1b, 0x80, 0x01, 0x1d, 0xf8, 0x01, 0x18, 0xf8, 0x81, 0x1e, 0xe8,
    0x41, 0x3b, 0xa4, 0x03, 0x3c, 0xcc, 0xc3, 0x2f, 0xd0, 0x43, 0x3e, 0xc0,
    0x43, 0x39, 0xa0, 0x40, 0xcc, 0x24, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30,
    0x0f, 0xf3, 0xe0, 0x06, 0xb4, 0x50, 0x0e, 0xf8, 0x40, 0x0f, 0xf5, 0x20,
    0x0f, 0xe5, 0x20, 0x07, 0xa4, 0xc0, 0x07, 0xf6, 0x50, 0x0e, 0xe3, 0x40,
    0x0f, 0xef, 0x20, 0x0f, 0x7c, 0x60, 0x0e, 0xec, 0xf0, 0x0e, 0xe1, 0x40,
    0x0f, 0x6c, 0x00, 0x06, 0x74, 0xe0, 0x07, 0x60, 0xe0, 0x07, 0x48, 0x08,
    0x83, 0xc8, 0x4d, 0xd2, 0x14, 0x51, 0xc2, 0xe4, 0xb3, 0x00, 0xf3, 0x2c,
    0x44, 0xc4, 0x4e, 0xc0, 0x44, 0xa0, 0x40, 0x90, 0x19, 0x01, 0x28, 0x01,
    0xa2, 0x34, 0x47, 0x80, 0x14, 0x03, 0x08, 0x21, 0x96, 0x20, 0x56, 0x0c,
    0x24, 0x84, 0x58, 0x80, 0xdc, 0x4d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48,
    0xfe, 0x4a, 0x48, 0x2b, 0x31, 0xf9, 0xc5, 0x6d, 0xa3, 0x02, 0x00, 0x00,
    0x04, 0xad, 0x7b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x10, 0x68,
    0x86, 0x85, 0x40, 0x41, 0x2c, 0x0c, 0x14, 0x52, 0x02, 0x00, 0x80, 0x10,
    0x02, 0xd0, 0x2c, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0xd5, 0x39, 0x82,
    0xa0, 0x18, 0x52, 0x2c, 0x21, 0x2e, 0xe1, 0x81, 0x80, 0x61, 0x04, 0x01,
    0xb8, 0x4b, 0x9a, 0x22, 0x4a, 0x98, 0xfc, 0x14, 0xb9, 0x88, 0x85, 0x3d,
    0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x84, 0x1d, 0xd2, 0x41,
    0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87,
    0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0,
    0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40,
    0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21,
    0x23, 0x45, 0x44, 0x00, 0x72, 0x00, 0xc0, 0xe4, 0x00, 0x80, 0xe9, 0x01,
    0x00, 0x0f, 0x79, 0x12, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x30, 0x40, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x71, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x33, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0x0f, 0x06, 0x04, 0xc0,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x20, 0x00, 0x00,
    0x00, 0x15, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c,
    0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x5a, 0x05, 0x42, 0xa2,
    0x04, 0x46, 0x00, 0x8a, 0xa1, 0x08, 0x4a, 0xa2, 0x50, 0xca, 0xa0, 0x1c,
    0x0a, 0xa4, 0x10, 0x4a, 0xa1, 0xc0, 0x0a, 0xa8, 0x3c, 0x0a, 0x87, 0x56,
    0x11, 0x8c, 0x00, 0x14, 0x42, 0x19, 0x94, 0x04, 0x9d, 0x19, 0x00, 0x2a,
    0x33, 0x00, 0x44, 0x66, 0x00, 0x68, 0xcc, 0x00, 0x50, 0x98, 0x01, 0x20,
    0x3e, 0x03, 0x40, 0x7d, 0x2c, 0x87, 0x21, 0x00, 0x00, 0x00, 0x8e, 0x03,
    0x00, 0x02, 0x81, 0x40, 0x00, 0x79, 0x18, 0x00, 0x00, 0xbe, 0x00, 0x00,
    0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63,
    0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03,
    0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a,
    0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b,
    0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0x41, 0x99, 0x20,
    0x08, 0xcb, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04, 0x41, 0x01, 0x6e, 0x6e,
    0x82, 0x20, 0x30, 0x1b, 0x86, 0x03, 0x21, 0x26, 0x08, 0xc5, 0xc7, 0x85,
    0x0e, 0xad, 0x8c, 0xaa, 0x0c, 0x8f, 0xae, 0x4e, 0xae, 0x2c, 0x6b, 0x82,
    0x20, 0x34, 0x13, 0x04, 0xc1, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84, 0x50,
    0x16, 0x82, 0x18, 0x18, 0xc2, 0x21, 0x43, 0x87, 0x56, 0x46, 0x55, 0x86,
    0x47, 0x57, 0x27, 0x57, 0x56, 0x65, 0xb5, 0x21, 0x19, 0x14, 0x88, 0x18,
    0x06, 0x86, 0x70, 0x36, 0x08, 0x4f, 0x34, 0x41, 0x38, 0x3c, 0x2a, 0x74,
    0x68, 0x65, 0x53, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x72, 0x1b, 0x10, 0x62,
    0xa2, 0x08, 0x62, 0x20, 0x80, 0x0d, 0x41, 0xb5, 0x81, 0x90, 0x00, 0xc0,
    0x9a, 0x20, 0x10, 0x60, 0x30, 0x41, 0x10, 0x1e, 0x06, 0x68, 0x13, 0x04,
    0x01, 0x9a, 0x20, 0x08, 0xd1, 0x06, 0x03, 0xd1, 0x36, 0x82, 0x6b, 0x48,
    0xb4, 0xa5, 0xc1, 0xcd, 0x4d, 0x10, 0x04, 0x69, 0x03, 0x81, 0x78, 0xdb,
    0x37, 0x41, 0x08, 0xc4, 0x60, 0x83, 0x40, 0x84, 0xc1, 0x86, 0x40, 0x0c,
    0x36, 0x08, 0xc4, 0x18, 0x6c, 0x20, 0xb2, 0x0e, 0x0c, 0xc8, 0x60, 0x82,
    0x30, 0x84, 0xc1, 0x04, 0x41, 0x98, 0x68, 0xa0, 0x85, 0xb9, 0x91, 0xb1,
    0x95, 0x4d, 0x10, 0x04, 0x6a, 0x83, 0x81, 0xa0, 0xc1, 0x46, 0x70, 0x69,
    0xb0, 0x41, 0x38, 0x03, 0x35, 0x98, 0x20, 0x70, 0x63, 0x30, 0x41, 0x10,
    0x2a, 0x0e, 0x70, 0x6f, 0x73, 0x5c, 0xa6, 0xac, 0xbe, 0xa0, 0x9e, 0xa6,
    0x92, 0xa8, 0x92, 0x9e, 0x9c, 0x36, 0x20, 0x88, 0x1b, 0x6c, 0xc4, 0x19,
    0xbc, 0x01, 0xd7, 0x70, 0xa0, 0x2b, 0xc3, 0x63, 0x42, 0x55, 0x84, 0x35,
    0xf4, 0xf4, 0x24, 0x45, 0x04, 0xb3, 0x01, 0x41, 0xe2, 0x60, 0xfb, 0xce,
    0x40, 0x0e, 0xb8, 0x86, 0xc5, 0xd8, 0x1b, 0xdb, 0x9b, 0xdc, 0x04, 0x41,
    0xb0, 0x68, 0x0c, 0x3d, 0x31, 0x3d, 0x49, 0xc1, 0x6c, 0x40, 0x10, 0x3a,
    0xd8, 0xea, 0xe0, 0x0c, 0xec, 0x80, 0x6b, 0x36, 0x10, 0x6d, 0x00, 0x07,
    0x73, 0x70, 0x07, 0x1b, 0x0e, 0x02, 0x2b, 0x03, 0x33, 0x58, 0x03, 0x36,
    0xc0, 0x83, 0x09, 0x82, 0x32, 0x6c, 0x00, 0x36, 0x0c, 0xc4, 0x1e, 0xec,
    0xc1, 0x86, 0x80, 0x0f, 0x36, 0x0c, 0x83, 0x1e, 0xf4, 0xc1, 0x04, 0xa1,
    0x23, 0x83, 0x0d, 0xc1, 0x1f, 0x90, 0x68, 0x0b, 0x4b, 0x73, 0xe3, 0x32,
    0x65, 0xf5, 0x05, 0xf5, 0x36, 0x97, 0x46, 0x97, 0xf6, 0xe6, 0x36, 0x41,
    0x58, 0xba, 0x09, 0xc2, 0xc2, 0x6d, 0x08, 0x88, 0x09, 0xc2, 0xb2, 0x4d,
    0x10, 0x16, 0x6d, 0xc3, 0x42, 0x88, 0xc2, 0x28, 0x90, 0x42, 0x29, 0x98,
    0xc2, 0x60, 0x0a, 0xc4, 0x29, 0x00, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e,
    0x9e, 0xa4, 0x88, 0x26, 0x08, 0x4b, 0xb6, 0x41, 0xd8, 0xb6, 0x0d, 0xcb,
    0x90, 0x0a, 0xa3, 0x70, 0x0a, 0xa5, 0xa0, 0x0a, 0x83, 0x2a, 0x0c, 0xa7,
    0xb0, 0x0a, 0x2c, 0x86, 0x9e, 0x98, 0x9e, 0xa4, 0x26, 0x08, 0xc2, 0xb5,
    0x41, 0xd8, 0x5c, 0x61, 0xc3, 0xc2, 0xb4, 0xc2, 0x28, 0x9c, 0x42, 0x29,
    0xa8, 0xc2, 0x60, 0x0a, 0xcc, 0x29, 0xbc, 0xc2, 0x86, 0x01, 0x15, 0x58,
    0x01, 0x16, 0x98, 0x4c, 0x59, 0x7d, 0x51, 0x85, 0xc9, 0x9d, 0x95, 0xd1,
    0x4d, 0x10, 0x96, 0x64, 0xc3, 0x42, 0xc8, 0xc2, 0x28, 0xcc, 0x42, 0x29,
    0x9c, 0xc2, 0x60, 0x0a, 0xc4, 0x29, 0xbc, 0xc2, 0x86, 0x80, 0x16, 0x36,
    0x0c, 0xb1, 0x50, 0x0b, 0xc0, 0x86, 0x42, 0x0f, 0x42, 0xc1, 0x16, 0x2e,
    0x80, 0x86, 0x19, 0xdb, 0x5b, 0x18, 0xdd, 0x1c, 0x8b, 0x34, 0xb7, 0x39,
    0xba, 0xb9, 0x09, 0x82, 0x80, 0xd1, 0x98, 0x4b, 0x3b, 0xfb, 0x62, 0x23,
    0xa3, 0x31, 0x97, 0x76, 0xf6, 0x35, 0x47, 0x47, 0x84, 0xae, 0x0c, 0xef,
    0xcb, 0xed, 0x4d, 0xae, 0x6d, 0x83, 0x82, 0x0b, 0x4d, 0x2e, 0xe8, 0xc2,
    0x2e, 0x20, 0xbc, 0x70, 0x06, 0xbd, 0xc0, 0x54, 0x61, 0x63, 0xb3, 0x6b,
    0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0,
    0x5c, 0xec, 0xca, 0xe4, 0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13,
    0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x41, 0x51,
    0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d,
    0xac, 0x8c, 0x6d, 0x4a, 0x80, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b,
    0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x58, 0x95, 0xc8, 0xf0,
    0x5c, 0xe8, 0xf2, 0xe0, 0xca, 0x82, 0xdc, 0xdc, 0xde, 0xe8, 0xc2, 0xe8,
    0xd2, 0xde, 0xdc, 0xe6, 0xa6, 0x08, 0x78, 0xd0, 0x07, 0x75, 0xc8, 0xf0,
    0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca,
    0xa6, 0x04, 0x7f, 0x50, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e,
    0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x60, 0x0b, 0x5d, 0xc8,
    0xf0, 0x5c, 0xc6, 0xde, 0xea, 0xdc, 0xe8, 0xca, 0xe4, 0xe6, 0xa6, 0x04,
    0xbd, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x33, 0x00, 0x00,
    0x00, 0x25, 0xf0, 0x05, 0x7e, 0x74, 0x74, 0x79, 0x1a, 0x6e, 0xc3, 0xd9,
    0x65, 0x39, 0x10, 0x38, 0xab, 0x4e, 0xc3, 0x6d, 0x38, 0xbb, 0x2c, 0x9f,
    0xd2, 0xc3, 0xf4, 0x32, 0x10, 0x18, 0xac, 0x80, 0x38, 0x08, 0xfc, 0xe8,
    0xe8, 0x32, 0xb5, 0x8c, 0xa7, 0xd7, 0xe5, 0xe5, 0xaa, 0x15, 0x08, 0x9c,
    0x59, 0x7f, 0x24, 0x6a, 0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0,
    0xfa, 0x23, 0xd9, 0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33,
    0x20, 0x10, 0x08, 0x0c, 0x16, 0x00, 0x1c, 0x04, 0x7e, 0x74, 0x74, 0x99,
    0x5a, 0xc6, 0xd3, 0xeb, 0xf2, 0x72, 0x16, 0x08, 0x9c, 0x59, 0x7f, 0x24,
    0x6a, 0x19, 0x4f, 0xaf, 0xcb, 0xcb, 0x32, 0x22, 0xd0, 0xfa, 0x23, 0xd9,
    0xcb, 0x63, 0xfa, 0x5b, 0x0e, 0x6c, 0x92, 0x60, 0x33, 0x20, 0x10, 0x08,
    0x0c, 0x1a, 0x82, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13,
    0x11, 0x02, 0xcd, 0xb0, 0x10, 0x76, 0xe0, 0x0c, 0x97, 0xef, 0x3c, 0xfe,
    0xe0, 0x4c, 0xb7, 0x5f, 0xdc, 0xb6, 0x11, 0x4c, 0xc3, 0xe5, 0x3b, 0x8f,
    0xbf, 0x38, 0xc0, 0x20, 0x36, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x9b, 0x01,
    0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x12, 0xc0, 0x3c, 0x0b, 0xe1, 0x17, 0xb7,
    0x6d, 0x05, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4b, 0x93, 0x13, 0x11, 0x28,
    0x35, 0x3d, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x03, 0x04, 0x03, 0x20, 0x0d,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xf3, 0x33, 0x44, 0x4f, 0x49, 0x10, 0x21,
    0xcb, 0x08, 0xc8, 0x3e, 0x2d, 0x19, 0x9c, 0x34, 0xa7, 0x44, 0x58, 0x49,
    0x4c, 0xb0, 0x07, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xec, 0x01, 0x00,
    0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00,
    0x00, 0x98, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00,
    0x00, 0xe3, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04,
    0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08,
    0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b,
    0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52,
    0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32,
    0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81,
    0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00,
    0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff,
    0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03,
    0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8,
    0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60,
    0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00,
    0x00, 0x45, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85,
    0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90,
    0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x70, 0x23,
    0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00,
    0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06,
    0xa1, 0x9b, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0xfc, 0x95, 0x90, 0x56,
    0x62, 0xf2, 0x8b, 0xdb, 0x46, 0xc5, 0x18, 0x63, 0x10, 0x2a, 0xf7, 0x0c,
    0x97, 0x3f, 0x61, 0x0f, 0x21, 0xf9, 0x21, 0xd0, 0x0c, 0x0b, 0x81, 0x82,
    0x55, 0x18, 0x45, 0x18, 0x1b, 0x63, 0x0c, 0x42, 0xc8, 0xa0, 0x56, 0x90,
    0x41, 0xc6, 0x18, 0x63, 0x0c, 0x7a, 0x73, 0x04, 0x41, 0x31, 0x18, 0x29,
    0x84, 0x44, 0x92, 0x03, 0x01, 0xc3, 0x08, 0xc4, 0x30, 0x53, 0x1b, 0x8c,
    0x03, 0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xd0, 0x42, 0x39, 0xe0,
    0x03, 0x3d, 0xd4, 0x83, 0x3c, 0x94, 0x83, 0x1c, 0x90, 0x02, 0x1f, 0xd8,
    0x43, 0x39, 0x8c, 0x03, 0x3d, 0xbc, 0x83, 0x3c, 0xf0, 0x81, 0x39, 0xb0,
    0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xb0, 0x01, 0x18, 0xd0, 0x81, 0x1f, 0x80,
    0x81, 0x1f, 0xe8, 0x81, 0x1e, 0xb4, 0x43, 0x3a, 0xc0, 0xc3, 0x3c, 0xfc,
    0x02, 0x3d, 0xe4, 0x03, 0x3c, 0x94, 0x03, 0x0a, 0xc8, 0x4c, 0x62, 0x30,
    0x0e, 0xec, 0x10, 0x0e, 0xf3, 0x30, 0x0f, 0x6e, 0x40, 0x0b, 0xe5, 0x80,
    0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0x7c, 0x60,
    0x0f, 0xe5, 0x30, 0x0e, 0xf4, 0xf0, 0x0e, 0xf2, 0xc0, 0x07, 0xe6, 0xc0,
    0x0e, 0xef, 0x10, 0x0e, 0xf4, 0xc0, 0x06, 0x60, 0x40, 0x07, 0x7e, 0x00,
    0x06, 0x7e, 0x80, 0x84, 0x6a, 0xe9, 0xde, 0x24, 0x4d, 0x11, 0x25, 0x4c,
    0x3e, 0x0b, 0x30, 0xcf, 0x42, 0x44, 0xec, 0x04, 0x4c, 0x04, 0x0a, 0x08,
    0xe5, 0x74, 0x20, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60,
    0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf,
    0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a,
    0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
    0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
    0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6,
    0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73,
    0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74,
    0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71,
    0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43,
    0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x0f, 0x00, 0x00,
    0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26,
    0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50,
    0x12, 0x65, 0x50, 0x1e, 0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0xa0,
    0x08, 0x0a, 0x84, 0xf0, 0x0c, 0x00, 0xe9, 0x19, 0x00, 0xda, 0x63, 0x39,
    0x0c, 0x01, 0x00, 0x00, 0x70, 0x1c, 0x00, 0x10, 0x08, 0x04, 0x02, 0x00,
    0x00, 0x79, 0x18, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c,
    0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03,
    0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01,
    0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a,
    0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b,
    0xd9, 0x10, 0x04, 0x13, 0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61,
    0x20, 0x26, 0x08, 0x04, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b,
    0x06, 0xc4, 0x20, 0x26, 0x08, 0xd8, 0x44, 0x60, 0x82, 0x40, 0x24, 0x13,
    0x04, 0x42, 0xd9, 0x20, 0x10, 0xcd, 0x86, 0x84, 0x50, 0x16, 0x82, 0x18,
    0x18, 0xc2, 0xd9, 0x90, 0x0c, 0xca, 0x42, 0x0c, 0x03, 0x43, 0x38, 0x1b,
    0x84, 0x07, 0x9a, 0x20, 0x68, 0xd4, 0x06, 0x84, 0x90, 0x16, 0x82, 0x18,
    0x08, 0x60, 0x43, 0x30, 0x6d, 0x20, 0x22, 0x00, 0xa0, 0x26, 0x08, 0x5b,
    0xb5, 0x21, 0xb0, 0x26, 0x08, 0x02, 0x40, 0xa2, 0x2d, 0x2c, 0xcd, 0x8d,
    0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xdb, 0x5c, 0x1a, 0x5d, 0xda, 0x9b, 0xdb,
    0x04, 0xa1, 0x68, 0x26, 0x08, 0x85, 0xb3, 0x21, 0x20, 0x26, 0x08, 0xc5,
    0x33, 0x41, 0x28, 0xa0, 0x0d, 0x0b, 0xa1, 0x6d, 0x5c, 0xe7, 0x0d, 0x1e,
    0xf1, 0x01, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26,
    0x08, 0x45, 0x34, 0x41, 0x20, 0x96, 0x0d, 0xc2, 0x18, 0x8c, 0xc1, 0x86,
    0x65, 0x08, 0x83, 0xed, 0xeb, 0xc4, 0x60, 0x10, 0x83, 0xe1, 0x23, 0x03,
    0x16, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x13, 0x04, 0x82, 0xd9, 0x20, 0x8c,
    0xc1, 0x19, 0x6c, 0x58, 0x18, 0x33, 0xd8, 0xbe, 0x4e, 0x0c, 0x06, 0x8f,
    0xf9, 0xd0, 0x60, 0xc3, 0x00, 0x06, 0x65, 0x90, 0x06, 0x4c, 0xa6, 0xac,
    0xbe, 0xa8, 0xc2, 0xe4, 0xce, 0xca, 0xe8, 0x26, 0x08, 0x85, 0xb4, 0x61,
    0x21, 0xd6, 0x60, 0x63, 0x83, 0xee, 0x1b, 0x3c, 0xe2, 0x43, 0x83, 0x0d,
    0x41, 0x1b, 0x6c, 0x18, 0xd4, 0xc0, 0x0d, 0x80, 0x0d, 0x05, 0x96, 0xbd,
    0x41, 0x05, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73,
    0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4, 0xe6,
    0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30,
    0x36, 0xbb, 0x32, 0xb9, 0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e,
    0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d, 0xac, 0x8c, 0x6d, 0x4a, 0x80,
    0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b,
    0x9b, 0x9b, 0x12, 0x50, 0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee,
    0x92, 0xc8, 0xa6, 0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x56, 0x1d, 0x32,
    0x3c, 0x97, 0x32, 0x37, 0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba,
    0xb9, 0x29, 0xc1, 0x1b, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c, 0x00, 0x00,
    0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d,
    0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07,
    0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80,
    0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66,
    0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d,
    0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07,
    0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03,
    0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90,
    0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50,
    0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2,
    0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39,
    0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14,
    0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07,
    0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07,
    0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87,
    0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0,
    0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc,
    0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6,
    0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f,
    0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c,
    0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87,
    0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0,
    0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50,
    0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00,
    0x00, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x11, 0xc0, 0x44,
    0x84, 0x40, 0x33, 0x2c, 0x84, 0x11, 0x38, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f,
    0x38, 0xd3, 0xed, 0x17, 0xb7, 0x6d, 0x01, 0xd3, 0x70, 0xf9, 0xce, 0xe3,
    0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x00,
    0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x04, 0x30, 0xcf, 0x42, 0xf8, 0xc5, 0x6d,
    0x9b, 0x40, 0x35, 0x5c, 0xbe, 0xf3, 0xf8, 0xd2, 0xe4, 0x44, 0x04, 0x4a,
    0x4d, 0x0f, 0x35, 0xf9, 0xc5, 0x6d, 0x1b, 0x00, 0xc1, 0x00, 0x48, 0x03,
    0x00, 0x61, 0x20, 0x00, 0x00, 0x61, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41,
    0x2c, 0x10, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x14, 0x47, 0x00,
    0xa8, 0x94, 0x00, 0x91, 0xc2, 0x2b, 0xb7, 0x92, 0x2b, 0x85, 0x42, 0x98,
    0x01, 0xa0, 0x31, 0x46, 0x00, 0xb3, 0xac, 0x0b, 0x82, 0xc1, 0x18, 0x81,
    0x18, 0xee, 0x30, 0xfc, 0x0b, 0x63, 0x04, 0x3a, 0x4a, 0xd3, 0xe5, 0x2f,
    0x8c, 0x11, 0xc8, 0x79, 0xaf, 0xfa, 0xdf, 0x18, 0x41, 0x48, 0x82, 0x21,
    0xee, 0x0b, 0x23, 0x00, 0x63, 0x04, 0x21, 0x09, 0x86, 0xf0, 0x2f, 0x8c,
    0x11, 0xbc, 0xb8, 0xa8, 0xe6, 0xdf, 0x0c, 0x00, 0x00, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x20, 0x89, 0x41, 0x73, 0x85, 0x41, 0x18, 0x44, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x8d, 0x81, 0x83, 0x85, 0x41, 0x18,
    0x48, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x91, 0xc1, 0xb3, 0x89,
    0x81, 0x18, 0x4c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xa4, 0x81,
    0x13, 0x06, 0x63, 0xa0, 0x35, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60,
    0xa8, 0xc1, 0x23, 0x06, 0x64, 0x40, 0x39, 0x23, 0x06, 0x09, 0x00, 0x82,
    0x60, 0x60, 0xac, 0x01, 0x34, 0x06, 0x65, 0xb0, 0x3d, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x60, 0xb0, 0x41, 0x44, 0x06, 0x66, 0xf0, 0x41, 0x23,
    0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb4, 0x81, 0x84, 0x06, 0x67, 0xe0,
    0x45, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0xb8, 0xc1, 0x94, 0x06,
    0x68, 0x80, 0x49, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xb4, 0x01,
    0x86, 0x1c, 0x42, 0xa0, 0x28, 0x69, 0x90, 0x06, 0x93, 0x32, 0x9a, 0x10,
    0x00, 0x23, 0x06, 0x0f, 0x00, 0x82, 0x60, 0xd0, 0xbc, 0x81, 0xb6, 0x24,
    0xc4, 0xc0, 0x30, 0x6b, 0xb0, 0x06, 0x15, 0x33, 0x9a, 0x10, 0x00, 0xa3,
    0x09, 0x42, 0x60, 0xc4, 0x04, 0x1f, 0x13, 0x24, 0xf8, 0x18, 0x31, 0xc1,
    0x67, 0xc4, 0x60, 0x01, 0x40, 0x10, 0x0c, 0x1e, 0x3a, 0xf8, 0x06, 0x21,
    0x98, 0xaa, 0x6b, 0xc4, 0x60, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x3a, 0x00,
    0x03, 0x62, 0x10, 0xa8, 0x4c, 0x1b, 0x31, 0x58, 0x00, 0x10, 0x04, 0x83,
    0xc7, 0x0e, 0xc2, 0xa0, 0x20, 0x86, 0x8a, 0xbb, 0x6c, 0x88, 0xe4, 0x63,
    0x43, 0x24, 0x1f, 0x1b, 0x22, 0xf9, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82,
    0x01, 0xc2, 0x07, 0x63, 0x70, 0x07, 0x77, 0xe0, 0x06, 0xc3, 0x88, 0x41,
    0x02, 0x80, 0x20, 0x18, 0x20, 0x7c, 0x30, 0x06, 0x77, 0x70, 0x07, 0x67,
    0x20, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xc2, 0x07, 0x63, 0x70,
    0x07, 0x77, 0xd0, 0x06, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20,
    0x7c, 0x30, 0x06, 0x77, 0x70, 0x07, 0x70, 0x10, 0x21, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00
};

/* The sole vertex shader (have to build 4x for different root signatures):

   --- D3D12_VertexShader.hlsl ---
   #pragma pack_matrix( row_major )

   cbuffer VertexShaderConstants : register(b0)
   {
       matrix model;
       matrix projectionAndView;
   };

   struct VertexShaderInput
   {
       float3 pos : POSITION;
       float2 tex : TEXCOORD0;
       float4 color : COLOR0;
   };

   struct VertexShaderOutput
   {
       float4 pos : SV_POSITION;
       float2 tex : TEXCOORD0;
       float4 color : COLOR0;
   };

   #define ColorRS \
       "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
       "DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
       "DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
       "DENY_HULL_SHADER_ROOT_ACCESS )," \
       "RootConstants(num32BitConstants=32, b0)"

   #define TextureRS \
       "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
       "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
       "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
       "            DENY_HULL_SHADER_ROOT_ACCESS )," \
       "RootConstants(num32BitConstants=32, b0),"\
       "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
       "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

   #define YUVRS \
       "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
       "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
       "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
       "            DENY_HULL_SHADER_ROOT_ACCESS )," \
       "RootConstants(num32BitConstants=32, b0),"\
       "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
       "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
       "DescriptorTable ( SRV(t2), visibility = SHADER_VISIBILITY_PIXEL ),"\
       "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

   #define NVRS \
       "RootFlags ( ALLOW_INPUT_ASSEMBLER_INPUT_LAYOUT |" \
       "            DENY_DOMAIN_SHADER_ROOT_ACCESS |" \
       "            DENY_GEOMETRY_SHADER_ROOT_ACCESS |" \
       "            DENY_HULL_SHADER_ROOT_ACCESS )," \
       "RootConstants(num32BitConstants=32, b0),"\
       "DescriptorTable ( SRV(t0), visibility = SHADER_VISIBILITY_PIXEL ),"\
       "DescriptorTable ( SRV(t1), visibility = SHADER_VISIBILITY_PIXEL ),"\
       "DescriptorTable ( Sampler(s0), visibility = SHADER_VISIBILITY_PIXEL )"

   [RootSignature(ColorRS)]
   VertexShaderOutput mainColor(VertexShaderInput input)
   {
       VertexShaderOutput output;
       float4 pos = float4(input.pos, 1.0f);

       // Transform the vertex position into projected space.
       pos = mul(pos, model);
       pos = mul(pos, projectionAndView);
       output.pos = pos;

       // Pass through texture coordinates and color values without transformation
       output.tex = input.tex;
       output.color = input.color;

       return output;
   }

   [RootSignature(TextureRS)]
   VertexShaderOutput mainTexture(VertexShaderInput input)
   {
       return mainColor(input);
   }

   [RootSignature(YUVRS)]
   VertexShaderOutput mainYUV(VertexShaderInput input)
   {
       return mainColor(input);
   }

   [RootSignature(NVRS)]
   VertexShaderOutput mainNV(VertexShaderInput input)
   {
       return mainColor(input);
   }
*/

static unsigned char D3D12_VertexShader_Colors[] = {
    0x44, 0x58, 0x42, 0x43, 0xbd, 0x28, 0xdc, 0xb4, 0x40, 0x82, 0x5f, 0xf6,
    0x92, 0x33, 0x60, 0xe7, 0x83, 0x5b, 0x8f, 0x2d, 0x01, 0x00, 0x00, 0x00,
    0xeb, 0x13, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x63, 0x01, 0x00, 0x00,
    0x73, 0x02, 0x00, 0x00, 0xab, 0x02, 0x00, 0x00, 0xa7, 0x0a, 0x00, 0x00,
    0xc3, 0x0a, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x80, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x71, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x54, 0x45, 0x58,
    0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00,
    0x4f, 0x53, 0x47, 0x31, 0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x03, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x7d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
    0x69, 0x6f, 0x6e, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44,
    0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x50, 0x53, 0x56, 0x30, 0x08,
    0x01, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x03,
    0x03, 0x00, 0x03, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00,
    0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x54, 0x45, 0x58,
    0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x43, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x22,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03,
    0x02, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x0f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x52, 0x54, 0x53, 0x30, 0x30, 0x00, 0x00, 0x00, 0x02,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x53,
    0x54, 0x41, 0x54, 0xf4, 0x07, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0xfd,
    0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0xdc, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21,
    0x0c, 0x00, 0x00, 0xf4, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02,
    0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41,
    0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25,
    0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42,
    0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a,
    0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00,
    0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41,
    0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51,
    0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff,
    0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff,
    0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00,
    0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13,
    0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89,
    0x20, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20,
    0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84,
    0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10,
    0x88, 0xc1, 0x0c, 0xc0, 0x30, 0x02, 0x01, 0x24, 0x41, 0x70, 0x70, 0x30,
    0x5c, 0x3e, 0xb0, 0x20, 0x46, 0xc3, 0x10, 0xcd, 0xe4, 0x2f, 0x84, 0x01,
    0x08, 0x98, 0x2f, 0x4d, 0x11, 0x25, 0x4c, 0xfe, 0x4b, 0x44, 0x13, 0x71,
    0xb1, 0x07, 0x30, 0x10, 0x11, 0xe7, 0x34, 0xd2, 0x04, 0x34, 0x93, 0x84,
    0x04, 0x41, 0xb8, 0x6e, 0xb8, 0x7c, 0x60, 0x41, 0x8c, 0x86, 0x21, 0x9a,
    0xc9, 0x5f, 0x08, 0x03, 0x10, 0x30, 0x9f, 0x73, 0x1a, 0x69, 0x02, 0x9a,
    0x49, 0x42, 0xc1, 0x40, 0xc4, 0x08, 0x40, 0x09, 0x0c, 0x3a, 0xe6, 0x08,
    0xc0, 0x60, 0x8e, 0x00, 0x29, 0x06, 0x90, 0x24, 0x89, 0x92, 0xd0, 0x52,
    0x0c, 0x23, 0x49, 0x12, 0x05, 0xa0, 0xe6, 0xa8, 0xe1, 0xf2, 0x27, 0xec,
    0x21, 0x24, 0x9f, 0xdb, 0xa8, 0x62, 0x25, 0x26, 0xbf, 0xb8, 0x6d, 0x44,
    0x00, 0x00, 0x00, 0x90, 0x72, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92,
    0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x80, 0x0a, 0xf1, 0x24, 0x51, 0x42,
    0x52, 0x29, 0x80, 0x04, 0x00, 0x00, 0xa2, 0xe6, 0x08, 0x82, 0x62, 0x44,
    0x89, 0x92, 0x24, 0x16, 0x5d, 0x03, 0x01, 0x67, 0x09, 0x0b, 0x20, 0x49,
    0x3e, 0x03, 0x4c, 0x11, 0x72, 0xf9, 0xc5, 0xe2, 0x00, 0x93, 0x8f, 0xfb,
    0x38, 0x0a, 0x84, 0xe3, 0xa4, 0x29, 0xa2, 0x84, 0xc9, 0x7f, 0x89, 0x68,
    0x22, 0x2e, 0xf6, 0x00, 0x06, 0x22, 0xe2, 0x9c, 0x46, 0x9a, 0x80, 0x66,
    0x92, 0x90, 0xa0, 0x69, 0xc3, 0x08, 0x02, 0x70, 0x99, 0x34, 0x45, 0x94,
    0x30, 0xf9, 0x2f, 0x11, 0x4d, 0xc4, 0xc5, 0x1e, 0xc0, 0x40, 0x44, 0x28,
    0x35, 0x3d, 0xd4, 0x84, 0x86, 0x80, 0x0b, 0x86, 0x11, 0x06, 0xe0, 0x30,
    0x69, 0x8a, 0x28, 0x61, 0xf2, 0x5f, 0x22, 0x9a, 0x88, 0x8b, 0x3d, 0x80,
    0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x78, 0x5c, 0x70, 0x09, 0xe7,
    0x34, 0xd2, 0x04, 0x34, 0x93, 0x84, 0x82, 0x2d, 0x1d, 0x12, 0x00, 0x13,
    0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68,
    0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a,
    0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
    0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
    0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72,
    0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a,
    0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d,
    0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21, 0x23, 0x25,
    0x40, 0x00, 0x52, 0x00, 0xc0, 0x90, 0xe7, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0xcf, 0x02, 0x04, 0x40, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x06, 0x08, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x12, 0x10,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x28,
    0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2,
    0x5c, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90,
    0x05, 0x02, 0x00, 0x19, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19,
    0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x52, 0x0a,
    0xa1, 0x08, 0xca, 0x80, 0x92, 0x12, 0x18, 0x01, 0x28, 0x86, 0x02, 0x14,
    0x28, 0x82, 0x42, 0x28, 0x83, 0x72, 0x28, 0x89, 0x02, 0x0c, 0x28, 0xb0,
    0x02, 0x29, 0xa0, 0xf2, 0x28, 0x8c, 0xd2, 0x0d, 0x28, 0x0a, 0x52, 0x4a,
    0x62, 0x04, 0x80, 0x8a, 0x19, 0x00, 0x22, 0x66, 0x00, 0x68, 0x98, 0x01,
    0xa0, 0x6d, 0x06, 0x80, 0xba, 0x19, 0x00, 0xfa, 0x66, 0x00, 0x08, 0x9c,
    0x01, 0xa0, 0x70, 0x2c, 0x87, 0x61, 0x9e, 0xe7, 0x01, 0x20, 0x30, 0x00,
    0x00, 0x10, 0x01, 0x21, 0x10, 0x0c, 0x40, 0x50, 0x00, 0x00, 0x00, 0x79,
    0x18, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46,
    0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b,
    0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1,
    0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa,
    0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10,
    0x04, 0x13, 0x84, 0x44, 0x99, 0x20, 0x24, 0xcb, 0x06, 0x61, 0x20, 0x36,
    0x08, 0x04, 0x41, 0xc1, 0x6e, 0x6e, 0x82, 0x90, 0x30, 0x1b, 0x86, 0x03,
    0x21, 0x26, 0x08, 0x45, 0xc7, 0x64, 0xe8, 0xcd, 0x6d, 0x8e, 0x2e, 0xcc,
    0x8d, 0x6e, 0x6e, 0x82, 0x90, 0x34, 0x1b, 0x10, 0x42, 0x59, 0x08, 0x62,
    0x60, 0x80, 0x0d, 0x41, 0xb3, 0x81, 0x00, 0x00, 0x07, 0x98, 0x20, 0x6c,
    0x60, 0xc0, 0xa2, 0xed, 0x8d, 0xac, 0x8c, 0x6d, 0x82, 0x90, 0x38, 0x13,
    0x84, 0xe4, 0xd9, 0x30, 0x4c, 0xd3, 0x30, 0x41, 0x48, 0xa0, 0x09, 0x42,
    0x12, 0x4d, 0x10, 0x12, 0x69, 0x03, 0x82, 0x44, 0x12, 0x55, 0x11, 0xd6,
    0xc5, 0x08, 0x4e, 0xee, 0x4d, 0xad, 0x6c, 0x8c, 0x2e, 0xed, 0xcd, 0x2d,
    0xc8, 0x8d, 0xcc, 0x2a, 0xad, 0xec, 0x6e, 0x82, 0x90, 0x4c, 0x1b, 0x10,
    0x24, 0x93, 0xa8, 0x4a, 0xb3, 0xae, 0x0d, 0x03, 0x83, 0x6d, 0x13, 0x84,
    0x2e, 0x0c, 0x26, 0x08, 0x09, 0xc5, 0x01, 0xee, 0x6d, 0x8e, 0xcb, 0x94,
    0xd5, 0x17, 0xd4, 0xd3, 0x54, 0x12, 0x55, 0xd2, 0x93, 0xd3, 0x06, 0x04,
    0xf9, 0x2a, 0x62, 0x02, 0x03, 0xeb, 0xe2, 0x40, 0x57, 0x86, 0x37, 0x41,
    0x48, 0x2a, 0x26, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x30,
    0x1b, 0x10, 0x44, 0x0c, 0xaa, 0x31, 0x98, 0xc8, 0xc0, 0xba, 0x58, 0x8c,
    0xbd, 0xb1, 0xbd, 0xc9, 0x4d, 0x10, 0x12, 0x8b, 0xc6, 0xd0, 0x13, 0xd3,
    0x93, 0x14, 0xcc, 0x06, 0x04, 0x31, 0x83, 0xea, 0x0c, 0x26, 0x34, 0xb0,
    0xae, 0x0d, 0x84, 0x17, 0x06, 0x65, 0x90, 0x06, 0x13, 0x84, 0x4f, 0x0c,
    0x88, 0x40, 0x3d, 0x4d, 0x25, 0x51, 0x25, 0x3d, 0x39, 0x6d, 0x40, 0x90,
    0xaf, 0x22, 0x26, 0x36, 0xb0, 0xae, 0x0d, 0x84, 0xd7, 0x06, 0x65, 0x90,
    0x06, 0x13, 0x04, 0x30, 0x18, 0x83, 0x0d, 0x04, 0xb2, 0x54, 0xc4, 0x06,
    0x81, 0x81, 0x83, 0x09, 0xc2, 0xf0, 0x4d, 0x10, 0x08, 0x6f, 0x43, 0x43,
    0x40, 0x5c, 0xa7, 0x06, 0x6b, 0xe0, 0x06, 0x6f, 0x10, 0x07, 0x72, 0xc0,
    0xcd, 0x41, 0x1c, 0x4c, 0x10, 0x90, 0x60, 0x03, 0xb0, 0x61, 0x20, 0xec,
    0xc0, 0x0e, 0x36, 0x04, 0x77, 0xb0, 0x61, 0x18, 0xea, 0x00, 0x0f, 0x26,
    0x08, 0x61, 0x40, 0x06, 0x1b, 0x02, 0x3d, 0x60, 0xd2, 0x16, 0x96, 0xe6,
    0x36, 0xf4, 0xc6, 0xf6, 0x26, 0x37, 0x41, 0x50, 0xb6, 0x09, 0x82, 0xc2,
    0x6d, 0x08, 0x88, 0x09, 0x82, 0x92, 0x6c, 0x10, 0x2a, 0x6b, 0xc3, 0x42,
    0xb0, 0x41, 0x1f, 0xf8, 0xc1, 0x1f, 0xf8, 0xc1, 0x00, 0x0a, 0x84, 0x1f,
    0x84, 0x02, 0x11, 0xaa, 0x22, 0xac, 0xa1, 0xa7, 0x27, 0x29, 0xa2, 0x09,
    0x82, 0x82, 0x6c, 0x10, 0xaa, 0x6a, 0xc3, 0x32, 0x8c, 0x42, 0x1f, 0xf8,
    0xc1, 0x1f, 0xf8, 0xc1, 0x40, 0x0a, 0x83, 0x1f, 0x94, 0x02, 0x8b, 0xa1,
    0x27, 0xa6, 0x27, 0xa9, 0x09, 0x82, 0x72, 0x4c, 0x10, 0x92, 0x6b, 0x83,
    0x50, 0xa5, 0xc2, 0x86, 0x45, 0x3a, 0x85, 0x3e, 0xf0, 0x83, 0x3f, 0xf0,
    0x83, 0x01, 0x15, 0x24, 0x3f, 0x50, 0x85, 0x0d, 0x83, 0x28, 0x98, 0xc2,
    0x2a, 0x70, 0x99, 0xb2, 0xfa, 0x82, 0x7a, 0x9b, 0x4b, 0xa3, 0x4b, 0x7b,
    0x73, 0xdb, 0xb0, 0x10, 0xad, 0xd0, 0x07, 0xa0, 0xf0, 0x07, 0xa8, 0x30,
    0xa0, 0x02, 0xe1, 0x07, 0xaa, 0xb0, 0x61, 0x19, 0x46, 0xa1, 0x0f, 0xfc,
    0xe0, 0x0f, 0x48, 0x61, 0x20, 0x85, 0xc1, 0x0f, 0x4a, 0x61, 0xc3, 0x22,
    0x9d, 0x42, 0x1f, 0xf8, 0xc1, 0x1f, 0x90, 0xc2, 0x80, 0x0a, 0x92, 0x1f,
    0xa8, 0xc2, 0x86, 0xc1, 0x15, 0x5e, 0x01, 0x16, 0x36, 0x0c, 0xac, 0x10,
    0x0b, 0xc0, 0x86, 0xa2, 0x0e, 0xf8, 0x40, 0x16, 0x1e, 0x80, 0x86, 0x19,
    0xdb, 0x5b, 0x18, 0xdd, 0xdc, 0x04, 0x21, 0xc1, 0x58, 0xa4, 0xb9, 0xcd,
    0xd1, 0xcd, 0x4d, 0x10, 0x92, 0x8c, 0xc6, 0x5c, 0xda, 0xd9, 0x17, 0x1b,
    0x19, 0x8d, 0xb9, 0xb4, 0xb3, 0xaf, 0x39, 0xba, 0x09, 0x42, 0xa2, 0x6d,
    0x40, 0x68, 0xa1, 0x16, 0x6c, 0xe1, 0x16, 0x70, 0xe1, 0xca, 0x05, 0x5d,
    0xa8, 0xc2, 0xc6, 0x66, 0xd7, 0xe6, 0x92, 0x46, 0x56, 0xe6, 0x46, 0x37,
    0x25, 0x08, 0xaa, 0x90, 0xe1, 0xb9, 0xd8, 0x95, 0xc9, 0xcd, 0xa5, 0xbd,
    0xb9, 0x4d, 0x09, 0x88, 0x26, 0x64, 0x78, 0x2e, 0x76, 0x61, 0x6c, 0x76,
    0x65, 0x72, 0x53, 0x82, 0xa2, 0x0e, 0x19, 0x9e, 0xcb, 0x1c, 0x5a, 0x18,
    0x59, 0x99, 0x5c, 0xd3, 0x1b, 0x59, 0x19, 0xdb, 0x94, 0x00, 0x29, 0x43,
    0x86, 0xe7, 0x22, 0x57, 0x36, 0xf7, 0x56, 0x27, 0x37, 0x56, 0x36, 0x37,
    0x25, 0x70, 0x2a, 0x91, 0xe1, 0xb9, 0xd0, 0xe5, 0xc1, 0x95, 0x05, 0xb9,
    0xb9, 0xbd, 0xd1, 0x85, 0xd1, 0xa5, 0xbd, 0xb9, 0xcd, 0x4d, 0x11, 0xe6,
    0x00, 0x0f, 0xea, 0x90, 0xe1, 0xb9, 0xd8, 0xa5, 0x95, 0xdd, 0x25, 0x91,
    0x4d, 0xd1, 0x85, 0xd1, 0x95, 0x4d, 0x09, 0xf4, 0xa0, 0x0e, 0x19, 0x9e,
    0x4b, 0x99, 0x1b, 0x9d, 0x5c, 0x1e, 0xd4, 0x5b, 0x9a, 0x1b, 0xdd, 0xdc,
    0x94, 0x40, 0x16, 0xba, 0x90, 0xe1, 0xb9, 0x8c, 0xbd, 0xd5, 0xb9, 0xd1,
    0x95, 0xc9, 0xcd, 0x4d, 0x09, 0x74, 0x01, 0x79, 0x18, 0x00, 0x00, 0x49,
    0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14,
    0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79,
    0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e,
    0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1,
    0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc,
    0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74,
    0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a,
    0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e,
    0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e,
    0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21,
    0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0,
    0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc,
    0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72,
    0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76,
    0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f,
    0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c,
    0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03,
    0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1,
    0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61,
    0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8,
    0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94,
    0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0,
    0xc3, 0x8c, 0xc8, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x72, 0x10, 0x87, 0x73,
    0x70, 0x03, 0x7b, 0x08, 0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77,
    0xa8, 0x07, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1d,
    0x00, 0x00, 0x00, 0x06, 0x00, 0x71, 0x4e, 0x23, 0x4d, 0x40, 0x33, 0x49,
    0xff, 0x42, 0x18, 0x80, 0x80, 0x19, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8,
    0x42, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0x94, 0x84, 0x01, 0x08,
    0x98, 0x5f, 0xdc, 0xb6, 0x19, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44,
    0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0x61, 0x02, 0xd3, 0x70, 0xf9,
    0xce, 0xe3, 0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb,
    0x36, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x34, 0x39, 0x11, 0x81, 0x52,
    0xd3, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c,
    0xfe, 0x44, 0x44, 0x13, 0x02, 0x44, 0x98, 0x5f, 0xdc, 0xb6, 0x05, 0x24,
    0x0c, 0x80, 0x34, 0x9c, 0xb3, 0x38, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x46, 0x05, 0xa8, 0xdb, 0xf4, 0x51, 0x8f, 0x4d, 0x34,
    0xfb, 0x4e, 0x9d, 0x82, 0x8b, 0x6e, 0x9e, 0x44, 0x58, 0x49, 0x4c, 0x20,
    0x09, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0x48, 0x02, 0x00, 0x00, 0x44,
    0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x08,
    0x09, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x3f,
    0x02, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13,
    0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06,
    0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e,
    0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4,
    0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48,
    0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4,
    0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1,
    0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08,
    0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40,
    0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d,
    0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49,
    0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20,
    0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x30,
    0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93,
    0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12,
    0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x6c, 0x23, 0x00, 0x25,
    0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6,
    0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0xa1, 0xa3,
    0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0x7c, 0x6e, 0xa3, 0x8a, 0x95, 0x98,
    0xfc, 0xe2, 0xb6, 0x11, 0x31, 0xc6, 0x18, 0x54, 0xee, 0x19, 0x2e, 0x7f,
    0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16, 0x02, 0x05, 0xab, 0x10,
    0x8a, 0x30, 0x42, 0xad, 0x14, 0x83, 0x8c, 0x31, 0xe8, 0xcd, 0x11, 0x04,
    0xc5, 0x60, 0xa4, 0x10, 0x12, 0x49, 0x0e, 0x04, 0x0c, 0x23, 0x10, 0x43,
    0x12, 0xd4, 0x83, 0x83, 0xe1, 0xf2, 0x81, 0x05, 0x31, 0x1a, 0x86, 0x68,
    0x26, 0x7f, 0x21, 0x0c, 0x40, 0xc0, 0x7c, 0x69, 0x8a, 0x28, 0x61, 0xf2,
    0x5f, 0x22, 0x9a, 0x88, 0x8b, 0x3d, 0x80, 0x81, 0x88, 0x38, 0xa7, 0x91,
    0x26, 0xa0, 0x99, 0x24, 0x24, 0x58, 0x7b, 0xdd, 0x70, 0xf9, 0xc0, 0x82,
    0x18, 0x0d, 0x43, 0x34, 0x93, 0xbf, 0x10, 0x06, 0x20, 0x60, 0x3e, 0xe7,
    0x34, 0xd2, 0x04, 0x34, 0x93, 0x84, 0x82, 0x4b, 0x38, 0x1d, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36,
    0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e,
    0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07,
    0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07,
    0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07,
    0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06,
    0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
    0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07,
    0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07,
    0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07,
    0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07,
    0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c,
    0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c,
    0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32,
    0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6,
    0x04, 0x43, 0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x05, 0x28, 0x50, 0x06,
    0xe5, 0x50, 0x04, 0xe5, 0x41, 0xa5, 0x24, 0x46, 0x00, 0xca, 0xa0, 0x08,
    0x0a, 0x81, 0xf2, 0x0c, 0x00, 0xe9, 0xb1, 0x1c, 0x86, 0x79, 0x9e, 0x07,
    0x80, 0xc0, 0x00, 0x00, 0x40, 0x04, 0x84, 0x40, 0x30, 0x00, 0x41, 0x01,
    0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x66, 0x00, 0x00, 0x00, 0x1a,
    0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73,
    0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71,
    0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a,
    0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b,
    0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7,
    0x06, 0x61, 0x20, 0x26, 0x08, 0x04, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x76,
    0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0xd9, 0x44, 0x60, 0x82, 0x40,
    0x24, 0x1b, 0x10, 0x42, 0x59, 0x08, 0x62, 0x60, 0x80, 0x0d, 0x41, 0xb3,
    0x81, 0x00, 0x00, 0x07, 0x98, 0x20, 0x68, 0xd4, 0x86, 0x00, 0x9a, 0x20,
    0x08, 0x00, 0x93, 0xb6, 0xb0, 0x34, 0xb7, 0xa1, 0x37, 0xb6, 0x37, 0x39,
    0x22, 0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x13, 0x84, 0xc2,
    0x99, 0x20, 0x14, 0xcf, 0x86, 0x80, 0x98, 0x20, 0x14, 0xd0, 0x04, 0x81,
    0x50, 0x26, 0x08, 0xc4, 0xb2, 0x41, 0xc8, 0xb4, 0x0d, 0x0b, 0x41, 0x55,
    0xd6, 0x65, 0x0d, 0x18, 0x61, 0x6d, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e,
    0x9e, 0xa4, 0x88, 0x26, 0x08, 0x45, 0xb4, 0x41, 0xc8, 0xb2, 0x0d, 0xcb,
    0xd0, 0x55, 0xd6, 0x65, 0x0d, 0xde, 0x60, 0x7d, 0x13, 0x04, 0x82, 0x61,
    0x31, 0xf4, 0xc4, 0xf4, 0x24, 0x35, 0x41, 0x28, 0xa4, 0x09, 0x02, 0xd1,
    0x6c, 0x10, 0x32, 0x32, 0xd8, 0xb0, 0x84, 0x81, 0x18, 0x54, 0xd6, 0x65,
    0x0d, 0x63, 0x10, 0x06, 0x56, 0x19, 0x6c, 0x18, 0x38, 0x30, 0x30, 0x03,
    0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e,
    0x1b, 0x16, 0x02, 0x0d, 0x2a, 0xec, 0x1a, 0x83, 0x61, 0x0c, 0x08, 0xab,
    0x0c, 0x36, 0x2c, 0x43, 0x57, 0x59, 0x97, 0x37, 0x78, 0x83, 0xf5, 0x6d,
    0x58, 0xc2, 0x40, 0x0c, 0x2a, 0xeb, 0xf2, 0x86, 0x31, 0x08, 0x03, 0xab,
    0x0c, 0x36, 0x0c, 0x69, 0xa0, 0x06, 0x6b, 0xb0, 0x61, 0x38, 0x03, 0x36,
    0x00, 0x36, 0x14, 0xd2, 0xd4, 0x06, 0x0f, 0x50, 0x85, 0x8d, 0xcd, 0xae,
    0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3,
    0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d,
    0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x46,
    0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9, 0xa6, 0x37,
    0xb2, 0x32, 0xb6, 0x29, 0x01, 0x52, 0x86, 0x0c, 0xcf, 0x45, 0xae, 0x6c,
    0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0xe0, 0xd4, 0x21, 0xc3,
    0x73, 0xb1, 0x4b, 0x2b, 0xbb, 0x4b, 0x22, 0x9b, 0xa2, 0x0b, 0xa3, 0x2b,
    0x9b, 0x12, 0x40, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2,
    0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x6d, 0x00, 0x00, 0x79,
    0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4,
    0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c,
    0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00,
    0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2,
    0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38,
    0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d,
    0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87,
    0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87,
    0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30,
    0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde,
    0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b,
    0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c,
    0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07,
    0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87,
    0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87,
    0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87,
    0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0,
    0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4,
    0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38,
    0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b,
    0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8, 0x21, 0x07, 0x7c, 0x70, 0x03,
    0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b, 0x08, 0x07, 0x79, 0x60, 0x87,
    0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x71,
    0x20, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x36, 0xb0, 0x0d, 0x97, 0xef,
    0x3c, 0xbe, 0x10, 0x50, 0x45, 0x41, 0x44, 0xa5, 0x03, 0x0c, 0x25, 0x61,
    0x00, 0x02, 0xe6, 0x17, 0xb7, 0x6d, 0x05, 0xd2, 0x70, 0xf9, 0xce, 0xe3,
    0x0b, 0x11, 0x01, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x58, 0xc0, 0x34,
    0x5c, 0xbe, 0xf3, 0xf8, 0x8b, 0x03, 0x0c, 0x62, 0xf3, 0x50, 0x93, 0x5f,
    0xdc, 0xb6, 0x09, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44,
    0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x11, 0x48, 0xc3, 0xe5,
    0x3b, 0x8f, 0x3f, 0x11, 0xd1, 0x84, 0x00, 0x11, 0xe6, 0x17, 0xb7, 0x6d,
    0x00, 0x09, 0x03, 0x20, 0x0d, 0xe7, 0x2c, 0x4e, 0x04, 0x00, 0x00, 0x61,
    0x20, 0x00, 0x00, 0xcf, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x44, 0x8a, 0xab, 0x14, 0x0a,
    0x61, 0x06, 0xa0, 0xec, 0x4a, 0x8e, 0x4a, 0x09, 0x50, 0x1c, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x61, 0x03,
    0x63, 0x59, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0xdd, 0x21,
    0x5d, 0x8f, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x87, 0x4c,
    0x18, 0x71, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xf1, 0x25, 0x54,
    0xf6, 0x20, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x80, 0x81, 0x52,
    0x69, 0x51, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46, 0x18, 0x2c,
    0xdc, 0x36, 0x29, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x88, 0x01,
    0xd3, 0x71, 0xc8, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc6, 0x18,
    0x34, 0x5d, 0x57, 0x31, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x90,
    0x81, 0xe3, 0x79, 0x4a, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46,
    0x19, 0x3c, 0xdf, 0x57, 0x39, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0,
    0x90, 0x81, 0xa3, 0x80, 0xc1, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1,
    0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0xc1, 0x01, 0x80, 0x20,
    0x18, 0x34, 0x69, 0x30, 0x3d, 0x66, 0x30, 0x9a, 0x10, 0x00, 0xa3, 0x09,
    0x42, 0x30, 0x9a, 0x30, 0x08, 0xa3, 0x09, 0xc4, 0x30, 0x62, 0x70, 0x00,
    0x20, 0x08, 0x06, 0x8d, 0x1b, 0x60, 0x54, 0x19, 0x8c, 0x26, 0x04, 0xc0,
    0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0x8c, 0x18,
    0x1c, 0x00, 0x08, 0x82, 0x41, 0x33, 0x07, 0x5d, 0xc6, 0x06, 0xa3, 0x09,
    0x01, 0x30, 0x9a, 0x20, 0x04, 0xa3, 0x09, 0x83, 0x30, 0x9a, 0x40, 0x0c,
    0x36, 0x5d, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x07, 0x0f,
    0xce, 0xe0, 0x7a, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41, 0x30, 0x78, 0xf2,
    0x00, 0x0d, 0xae, 0x25, 0xb0, 0xe0, 0x80, 0x8e, 0x59, 0x9b, 0x7c, 0x46,
    0x0c, 0x10, 0x00, 0x04, 0xc1, 0xe0, 0xe1, 0x83, 0x35, 0xd8, 0xa4, 0x60,
    0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x3e, 0x60, 0x83, 0xcd, 0x09,
    0x2c, 0x50, 0xa0, 0x63, 0xd9, 0x27, 0x9f, 0x11, 0x03, 0x04, 0x00, 0x41,
    0x30, 0x78, 0x40, 0xe1, 0x0d, 0xbe, 0x2a, 0x18, 0x31, 0x40, 0x00, 0x10,
    0x04, 0x83, 0x27, 0x14, 0xe0, 0xe0, 0x8b, 0x02, 0x0b, 0x1a, 0xe8, 0x18,
    0x37, 0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x87, 0x14,
    0xe6, 0x60, 0x0c, 0xb0, 0x60, 0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e,
    0x52, 0xa0, 0x83, 0x31, 0xa0, 0x02, 0x0b, 0x20, 0xe8, 0x8c, 0x18, 0x1c,
    0x00, 0x08, 0x82, 0x41, 0x83, 0x0a, 0x72, 0xe0, 0x06, 0x74, 0x30, 0x9a,
    0x10, 0x00, 0xa3, 0x09, 0x42, 0x30, 0x9a, 0x30, 0x08, 0xa3, 0x09, 0xc4,
    0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x4d, 0x2b, 0xdc, 0xc1, 0x1c,
    0xe8, 0xc1, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20,
    0x8c, 0x26, 0x10, 0xc3, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x34, 0xb2,
    0xc0, 0x07, 0x78, 0xd0, 0x0a, 0xa3, 0x09, 0x01, 0x30, 0x9a, 0x20, 0x04,
    0xa3, 0x09, 0x83, 0x30, 0x9a, 0x40, 0x0c, 0x23, 0x06, 0x07, 0x00, 0x82,
    0x60, 0xd0, 0xdc, 0x42, 0x28, 0xf4, 0xc1, 0x2b, 0x8c, 0x26, 0x04, 0xc0,
    0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0xd8, 0x14,
    0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x87, 0x17, 0x56,
    0xc1, 0x7b, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41, 0x30, 0x78, 0x7a, 0x81,
    0x15, 0xb6, 0x25, 0x18, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0xc7, 0x17,
    0x5a, 0x01, 0x3b, 0x02, 0xb3, 0xca, 0x40, 0x3e, 0x23, 0x06, 0x08, 0x00,
    0x82, 0x60, 0xf0, 0x80, 0xc3, 0x2b, 0x88, 0x81, 0x14, 0x8c, 0x18, 0x20,
    0x00, 0x08, 0x82, 0xc1, 0x13, 0x0e, 0xb0, 0xf0, 0x39, 0xc1, 0x88, 0x01,
    0x02, 0x80, 0x20, 0x18, 0x3c, 0xe2, 0x10, 0x0b, 0x9c, 0x12, 0x58, 0x96,
    0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x87, 0x1c, 0x66,
    0xc1, 0x0c, 0xaa, 0x60, 0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x72,
    0xa0, 0x85, 0x31, 0x88, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41, 0x30, 0x78,
    0xcc, 0xa1, 0x16, 0xc0, 0xa0, 0x09, 0x8c, 0x6b, 0x03, 0xf9, 0x8c, 0x18,
    0x20, 0x00, 0x08, 0x82, 0xc1, 0x83, 0x0e, 0xb7, 0xa0, 0x06, 0x58, 0x30,
    0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0x4f, 0x3a, 0xe0, 0xc2, 0x19, 0x50,
    0xc1, 0x88, 0x01, 0x02, 0x80, 0x20, 0x18, 0x3c, 0xea, 0x90, 0x0b, 0x64,
    0x00, 0x05, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e,
    0xa0, 0x03, 0x3a, 0x80, 0x43, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06,
    0x48, 0x3b, 0xe4, 0x02, 0x3a, 0xa0, 0x03, 0x2d, 0x24, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e, 0xa0, 0x03, 0x3a, 0xf8, 0x42,
    0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0x3b, 0xe4, 0x02, 0x3a,
    0xa0, 0xc3, 0x2f, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xb4,
    0x43, 0x2e, 0xa4, 0x03, 0x3a, 0x80, 0x43, 0x2b, 0x8c, 0x18, 0x24, 0x00,
    0x08, 0x82, 0x01, 0xd2, 0x0e, 0xb9, 0x90, 0x0e, 0xe8, 0x40, 0x0b, 0xac,
    0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0x3b, 0xe4, 0xc2, 0x38,
    0xa0, 0x03, 0x38, 0xc4, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20,
    0xed, 0x90, 0x0b, 0xe3, 0x80, 0x0e, 0xb4, 0x00, 0x0b, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e, 0x8c, 0x03, 0x3a, 0xf8, 0xc2,
    0x2b, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xd2, 0x0e, 0xb9, 0x30,
    0x0e, 0xe8, 0xf0, 0x0b, 0xae, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00
};

static unsigned char D3D12_VertexShader_Textures[] = {
    0x44, 0x58, 0x42, 0x43, 0x64, 0x78, 0x9b, 0xaf, 0x0d, 0x1f, 0x5b, 0x33,
    0xd2, 0x5e, 0x6d, 0x7f, 0x5d, 0x5c, 0xd1, 0x9d, 0x01, 0x00, 0x00, 0x00,
    0x4b, 0x14, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x63, 0x01, 0x00, 0x00,
    0x73, 0x02, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00, 0x03, 0x0b, 0x00, 0x00,
    0x1f, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x80, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x71, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x54, 0x45, 0x58,
    0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00,
    0x4f, 0x53, 0x47, 0x31, 0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x03, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x7d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
    0x69, 0x6f, 0x6e, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44,
    0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x50, 0x53, 0x56, 0x30, 0x08,
    0x01, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x03,
    0x03, 0x00, 0x03, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00,
    0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x54, 0x45, 0x58,
    0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x43, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x22,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03,
    0x02, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x0f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x52, 0x54, 0x53, 0x30, 0x88, 0x00, 0x00, 0x00, 0x02,
    0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x70,
    0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
    0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0xf8, 0x07, 0x00, 0x00, 0x60,
    0x00, 0x01, 0x00, 0xfe, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00,
    0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00, 0x42,
    0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xf5, 0x01, 0x00, 0x00, 0x0b,
    0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07,
    0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92,
    0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80,
    0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38,
    0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43,
    0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11,
    0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04,
    0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b,
    0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84,
    0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff,
    0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x32,
    0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04,
    0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b,
    0x84, 0xc4, 0x4c, 0x10, 0x88, 0xc1, 0x0c, 0xc0, 0x30, 0x02, 0x01, 0x24,
    0x41, 0x70, 0x70, 0x30, 0x5c, 0x3e, 0xb0, 0x20, 0x46, 0xc3, 0x10, 0xcd,
    0xe4, 0x2f, 0x84, 0x01, 0x08, 0x98, 0x2f, 0x4d, 0x11, 0x25, 0x4c, 0xfe,
    0x4b, 0x44, 0x13, 0x71, 0xb1, 0x07, 0x30, 0x10, 0x11, 0xe7, 0x34, 0xd2,
    0x04, 0x34, 0x93, 0x84, 0x04, 0x41, 0xb8, 0x6e, 0xb8, 0x7c, 0x60, 0x41,
    0x8c, 0x86, 0x21, 0x9a, 0xc9, 0x5f, 0x08, 0x03, 0x10, 0x30, 0x9f, 0x73,
    0x1a, 0x69, 0x02, 0x9a, 0x49, 0x42, 0xc1, 0x40, 0xc4, 0x08, 0x40, 0x09,
    0x0c, 0x3a, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0x06, 0x90, 0x24,
    0x89, 0x92, 0xd0, 0x52, 0x0c, 0x23, 0x49, 0x12, 0x05, 0xa0, 0xe6, 0xa8,
    0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x9f, 0xdb, 0xa8, 0x62, 0x25, 0x26,
    0xbf, 0xb8, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x90, 0x72, 0xcf, 0x70, 0xf9,
    0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x80, 0x0a,
    0xf1, 0x24, 0x51, 0x42, 0x52, 0x29, 0x80, 0x04, 0x00, 0x00, 0xa2, 0xe6,
    0x08, 0x82, 0x62, 0x44, 0x89, 0x92, 0x24, 0x16, 0x5d, 0x03, 0x01, 0x67,
    0x09, 0x0b, 0x20, 0x49, 0x3e, 0x03, 0x4c, 0x11, 0x72, 0xf9, 0xc5, 0xe2,
    0x00, 0x93, 0x8f, 0xfb, 0x38, 0x0a, 0x84, 0xe3, 0xa4, 0x29, 0xa2, 0x84,
    0xc9, 0x7f, 0x89, 0x68, 0x22, 0x2e, 0xf6, 0x00, 0x06, 0x22, 0xe2, 0x9c,
    0x46, 0x9a, 0x80, 0x66, 0x92, 0x90, 0xa0, 0x69, 0xc3, 0x08, 0x02, 0x70,
    0x99, 0x34, 0x45, 0x94, 0x30, 0xf9, 0x2f, 0x11, 0x4d, 0xc4, 0xc5, 0x1e,
    0xc0, 0x40, 0x44, 0x28, 0x35, 0x3d, 0xd4, 0x84, 0x86, 0x80, 0x0b, 0x86,
    0x11, 0x06, 0xe0, 0x30, 0x69, 0x8a, 0x28, 0x61, 0xf2, 0x5f, 0x22, 0x9a,
    0x88, 0x8b, 0x3d, 0x80, 0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x78,
    0x5c, 0x70, 0x09, 0xe7, 0x34, 0xd2, 0x04, 0x34, 0x93, 0x84, 0x82, 0x2d,
    0x1d, 0x12, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36,
    0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e,
    0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07,
    0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07,
    0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07,
    0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06,
    0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
    0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07,
    0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07,
    0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07,
    0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07,
    0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64,
    0x90, 0x21, 0x23, 0x25, 0x40, 0x00, 0x52, 0x00, 0xc0, 0x90, 0xe7, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0xcf,
    0x02, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43,
    0x9e, 0x06, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x86, 0x3c, 0x12, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0c, 0x79, 0x28, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x18, 0xf2, 0x5c, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x90, 0x05, 0x02, 0x00, 0x19, 0x00, 0x00, 0x00, 0x32,
    0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6,
    0x04, 0x43, 0x52, 0x0a, 0xa1, 0x08, 0xca, 0x80, 0x92, 0x12, 0x18, 0x01,
    0x28, 0x86, 0x02, 0x14, 0x28, 0x82, 0x42, 0x28, 0x83, 0x72, 0x28, 0x89,
    0x02, 0x0c, 0x28, 0xb0, 0x02, 0x29, 0xa0, 0xf2, 0x28, 0x8c, 0xd2, 0x0d,
    0x28, 0x0a, 0x52, 0x4a, 0x62, 0x04, 0x80, 0x8a, 0x19, 0x00, 0x22, 0x66,
    0x00, 0x68, 0x98, 0x01, 0xa0, 0x6d, 0x06, 0x80, 0xba, 0x19, 0x00, 0xfa,
    0x66, 0x00, 0x08, 0x9c, 0x01, 0xa0, 0x70, 0x2c, 0x87, 0x61, 0x9e, 0xe7,
    0x01, 0x20, 0x30, 0x00, 0x00, 0x10, 0x01, 0x21, 0x10, 0x0c, 0x40, 0x50,
    0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0xc1, 0x00, 0x00, 0x00, 0x1a,
    0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73,
    0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71,
    0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a,
    0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b,
    0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x44, 0x99, 0x20, 0x24, 0xcb,
    0x06, 0x61, 0x20, 0x36, 0x08, 0x04, 0x41, 0xc1, 0x6e, 0x6e, 0x82, 0x90,
    0x30, 0x1b, 0x86, 0x03, 0x21, 0x26, 0x08, 0x45, 0xc7, 0x64, 0xe8, 0xcd,
    0x6d, 0x8e, 0x2e, 0xcc, 0x8d, 0x6e, 0x6e, 0x82, 0x90, 0x34, 0x1b, 0x10,
    0x42, 0x59, 0x08, 0x62, 0x60, 0x80, 0x0d, 0x41, 0xb3, 0x81, 0x00, 0x00,
    0x07, 0x98, 0x20, 0x6c, 0x60, 0xc0, 0xa2, 0xed, 0x8d, 0xac, 0x8c, 0x6d,
    0x82, 0x90, 0x38, 0x13, 0x84, 0xe4, 0xd9, 0x30, 0x4c, 0xd3, 0x30, 0x41,
    0x48, 0xa0, 0x09, 0x42, 0x12, 0x4d, 0x10, 0x12, 0x69, 0x03, 0x82, 0x44,
    0x12, 0x55, 0x11, 0xd6, 0xc5, 0x08, 0x4e, 0xee, 0x4d, 0xad, 0x6c, 0x8c,
    0x2e, 0xed, 0xcd, 0x2d, 0xc8, 0x8d, 0xcc, 0x2a, 0xad, 0xec, 0x6e, 0x82,
    0x90, 0x4c, 0x1b, 0x10, 0x24, 0x93, 0xa8, 0x4a, 0xb3, 0xae, 0x0d, 0x03,
    0x83, 0x6d, 0x13, 0x84, 0x2e, 0x0c, 0x26, 0x08, 0x09, 0xc5, 0x01, 0xee,
    0x6d, 0x8e, 0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xd3, 0x54, 0x12, 0x55, 0xd2,
    0x93, 0xd3, 0x06, 0x04, 0xf9, 0x2a, 0x62, 0x02, 0x03, 0xeb, 0xe2, 0x40,
    0x57, 0x86, 0x37, 0x41, 0x48, 0x2a, 0x26, 0x54, 0x45, 0x58, 0x43, 0x4f,
    0x4f, 0x52, 0x44, 0x30, 0x1b, 0x10, 0x44, 0x0c, 0xaa, 0x31, 0x98, 0xc8,
    0xc0, 0xba, 0x58, 0x8c, 0xbd, 0xb1, 0xbd, 0xc9, 0x4d, 0x10, 0x12, 0x8b,
    0xc6, 0xd0, 0x13, 0xd3, 0x93, 0x14, 0xcc, 0x06, 0x04, 0x31, 0x83, 0xea,
    0x0c, 0x26, 0x34, 0xb0, 0xae, 0x0d, 0x84, 0x17, 0x06, 0x65, 0x90, 0x06,
    0x13, 0x84, 0x4f, 0x0c, 0x88, 0x40, 0x3d, 0x4d, 0x25, 0x51, 0x25, 0x3d,
    0x39, 0x6d, 0x40, 0x90, 0xaf, 0x22, 0x26, 0x36, 0xb0, 0xae, 0x0d, 0x84,
    0xd7, 0x06, 0x65, 0x90, 0x06, 0x13, 0x04, 0x30, 0x18, 0x83, 0x0d, 0x04,
    0xb2, 0x54, 0xc4, 0x06, 0x81, 0x81, 0x83, 0x09, 0xc2, 0xf0, 0x4d, 0x10,
    0x08, 0x6f, 0x43, 0x43, 0x40, 0x5c, 0xa7, 0x06, 0x6b, 0xe0, 0x06, 0x6f,
    0x10, 0x07, 0x72, 0xc0, 0xcd, 0x41, 0x1c, 0x4c, 0x10, 0x90, 0x60, 0x03,
    0xb0, 0x61, 0x20, 0xec, 0xc0, 0x0e, 0x36, 0x04, 0x77, 0xb0, 0x61, 0x18,
    0xea, 0x00, 0x0f, 0x26, 0x08, 0x61, 0x40, 0x06, 0x1b, 0x02, 0x3d, 0xe0,
    0xd2, 0x16, 0x96, 0xe6, 0x46, 0x55, 0x86, 0x47, 0x57, 0x27, 0x57, 0x36,
    0x41, 0x50, 0xb6, 0x09, 0x82, 0xc2, 0x6d, 0x08, 0x88, 0x09, 0x82, 0x92,
    0x6c, 0x10, 0x2a, 0x6b, 0xc3, 0x42, 0xb0, 0x41, 0x1f, 0xf8, 0xc1, 0x1f,
    0xf8, 0xc1, 0x00, 0x0a, 0x84, 0x1f, 0x84, 0x02, 0x11, 0xaa, 0x22, 0xac,
    0xa1, 0xa7, 0x27, 0x29, 0xa2, 0x09, 0x82, 0x82, 0x6c, 0x10, 0xaa, 0x6a,
    0xc3, 0x32, 0x8c, 0x42, 0x1f, 0xf8, 0xc1, 0x1f, 0xf8, 0xc1, 0x40, 0x0a,
    0x83, 0x1f, 0x94, 0x02, 0x8b, 0xa1, 0x27, 0xa6, 0x27, 0xa9, 0x09, 0x82,
    0x72, 0x4c, 0x10, 0x92, 0x6b, 0x83, 0x50, 0xa5, 0xc2, 0x86, 0x45, 0x3a,
    0x85, 0x3e, 0xf0, 0x83, 0x3f, 0xf0, 0x83, 0x01, 0x15, 0x24, 0x3f, 0x50,
    0x85, 0x0d, 0x83, 0x28, 0x98, 0xc2, 0x2a, 0x70, 0x99, 0xb2, 0xfa, 0x82,
    0x7a, 0x9b, 0x4b, 0xa3, 0x4b, 0x7b, 0x73, 0xdb, 0xb0, 0x10, 0xad, 0xd0,
    0x07, 0xa0, 0xf0, 0x07, 0xa8, 0x30, 0xa0, 0x02, 0xe1, 0x07, 0xaa, 0xb0,
    0x61, 0x19, 0x46, 0xa1, 0x0f, 0xfc, 0xe0, 0x0f, 0x48, 0x61, 0x20, 0x85,
    0xc1, 0x0f, 0x4a, 0x61, 0xc3, 0x22, 0x9d, 0x42, 0x1f, 0xf8, 0xc1, 0x1f,
    0x90, 0xc2, 0x80, 0x0a, 0x92, 0x1f, 0xa8, 0xc2, 0x86, 0xc1, 0x15, 0x5e,
    0x01, 0x16, 0x36, 0x0c, 0xac, 0x10, 0x0b, 0xc0, 0x86, 0xa2, 0x0e, 0xf8,
    0x40, 0x16, 0x1e, 0x80, 0x86, 0x19, 0xdb, 0x5b, 0x18, 0xdd, 0xdc, 0x04,
    0x21, 0xc1, 0x58, 0xa4, 0xb9, 0xcd, 0xd1, 0xcd, 0x4d, 0x10, 0x92, 0x8c,
    0xc6, 0x5c, 0xda, 0xd9, 0x17, 0x1b, 0x19, 0x8d, 0xb9, 0xb4, 0xb3, 0xaf,
    0x39, 0xba, 0x09, 0x42, 0xa2, 0x6d, 0x40, 0x68, 0xa1, 0x16, 0x6c, 0xe1,
    0x16, 0x70, 0xe1, 0xca, 0x05, 0x5d, 0xa8, 0xc2, 0xc6, 0x66, 0xd7, 0xe6,
    0x92, 0x46, 0x56, 0xe6, 0x46, 0x37, 0x25, 0x08, 0xaa, 0x90, 0xe1, 0xb9,
    0xd8, 0x95, 0xc9, 0xcd, 0xa5, 0xbd, 0xb9, 0x4d, 0x09, 0x88, 0x26, 0x64,
    0x78, 0x2e, 0x76, 0x61, 0x6c, 0x76, 0x65, 0x72, 0x53, 0x82, 0xa2, 0x0e,
    0x19, 0x9e, 0xcb, 0x1c, 0x5a, 0x18, 0x59, 0x99, 0x5c, 0xd3, 0x1b, 0x59,
    0x19, 0xdb, 0x94, 0x00, 0x29, 0x43, 0x86, 0xe7, 0x22, 0x57, 0x36, 0xf7,
    0x56, 0x27, 0x37, 0x56, 0x36, 0x37, 0x25, 0x70, 0x2a, 0x91, 0xe1, 0xb9,
    0xd0, 0xe5, 0xc1, 0x95, 0x05, 0xb9, 0xb9, 0xbd, 0xd1, 0x85, 0xd1, 0xa5,
    0xbd, 0xb9, 0xcd, 0x4d, 0x11, 0xe6, 0x00, 0x0f, 0xea, 0x90, 0xe1, 0xb9,
    0xd8, 0xa5, 0x95, 0xdd, 0x25, 0x91, 0x4d, 0xd1, 0x85, 0xd1, 0x95, 0x4d,
    0x09, 0xf4, 0xa0, 0x0e, 0x19, 0x9e, 0x4b, 0x99, 0x1b, 0x9d, 0x5c, 0x1e,
    0xd4, 0x5b, 0x9a, 0x1b, 0xdd, 0xdc, 0x94, 0x40, 0x16, 0xba, 0x90, 0xe1,
    0xb9, 0x8c, 0xbd, 0xd5, 0xb9, 0xd1, 0x95, 0xc9, 0xcd, 0x4d, 0x09, 0x74,
    0x01, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33,
    0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43,
    0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98,
    0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33,
    0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05,
    0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43,
    0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08,
    0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78,
    0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1,
    0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33,
    0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e,
    0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03,
    0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60,
    0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80,
    0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8,
    0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18,
    0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee,
    0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c,
    0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c,
    0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43,
    0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3,
    0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83,
    0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8, 0x21,
    0x07, 0x7c, 0x70, 0x03, 0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b, 0x08,
    0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00,
    0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x06,
    0x00, 0x71, 0x4e, 0x23, 0x4d, 0x40, 0x33, 0x49, 0xff, 0x42, 0x18, 0x80,
    0x80, 0x19, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x40, 0x15, 0x05,
    0x11, 0x95, 0x0e, 0x30, 0x94, 0x84, 0x01, 0x08, 0x98, 0x5f, 0xdc, 0xb6,
    0x19, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21,
    0xd0, 0x0c, 0x0b, 0x61, 0x02, 0xd3, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e,
    0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x50, 0x0d, 0x97,
    0xef, 0x3c, 0xbe, 0x34, 0x39, 0x11, 0x81, 0x52, 0xd3, 0x43, 0x4d, 0x7e,
    0x71, 0xdb, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x44, 0x44, 0x13,
    0x02, 0x44, 0x98, 0x5f, 0xdc, 0xb6, 0x05, 0x2c, 0x0c, 0x80, 0x34, 0x2d,
    0x71, 0x4d, 0x54, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48,
    0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54,
    0xa0, 0xdd, 0xa0, 0xcf, 0x1a, 0x5f, 0x65, 0x5e, 0xda, 0x32, 0xf3, 0x80,
    0x17, 0x51, 0x70, 0x44, 0x58, 0x49, 0x4c, 0x24, 0x09, 0x00, 0x00, 0x60,
    0x00, 0x01, 0x00, 0x49, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00,
    0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x0c, 0x09, 0x00, 0x00, 0x42,
    0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x40, 0x02, 0x00, 0x00, 0x0b,
    0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07,
    0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92,
    0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80,
    0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38,
    0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43,
    0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91,
    0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04,
    0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b,
    0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84,
    0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff,
    0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x32,
    0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04,
    0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b,
    0x84, 0xa4, 0x4c, 0x10, 0x6c, 0x23, 0x00, 0x25, 0x00, 0x14, 0x66, 0x00,
    0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84, 0x14, 0x42,
    0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0xa1, 0xa3, 0x86, 0xcb, 0x9f, 0xb0,
    0x87, 0x90, 0x7c, 0x6e, 0xa3, 0x8a, 0x95, 0x98, 0xfc, 0xe2, 0xb6, 0x11,
    0x31, 0xc6, 0x18, 0x54, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2,
    0x43, 0xa0, 0x19, 0x16, 0x02, 0x05, 0xab, 0x10, 0x8a, 0x30, 0x42, 0xad,
    0x14, 0x83, 0x8c, 0x31, 0xe8, 0xcd, 0x11, 0x04, 0xc5, 0x60, 0xa4, 0x10,
    0x12, 0x49, 0x0e, 0x04, 0x0c, 0x23, 0x10, 0x43, 0x12, 0xd4, 0x83, 0x83,
    0xe1, 0xf2, 0x81, 0x05, 0x31, 0x1a, 0x86, 0x68, 0x26, 0x7f, 0x21, 0x0c,
    0x40, 0xc0, 0x7c, 0x69, 0x8a, 0x28, 0x61, 0xf2, 0x5f, 0x22, 0x9a, 0x88,
    0x8b, 0x3d, 0x80, 0x81, 0x88, 0x38, 0xa7, 0x91, 0x26, 0xa0, 0x99, 0x24,
    0x24, 0x58, 0x7b, 0xdd, 0x70, 0xf9, 0xc0, 0x82, 0x18, 0x0d, 0x43, 0x34,
    0x93, 0xbf, 0x10, 0x06, 0x20, 0x60, 0x3e, 0xe7, 0x34, 0xd2, 0x04, 0x34,
    0x93, 0x84, 0x82, 0x4b, 0x38, 0x1d, 0x08, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68,
    0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a,
    0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
    0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
    0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72,
    0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a,
    0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d,
    0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40,
    0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79,
    0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8,
    0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40,
    0x16, 0x08, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19,
    0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x25,
    0x30, 0x02, 0x50, 0x0c, 0x05, 0x28, 0x50, 0x06, 0xe5, 0x50, 0x04, 0xe5,
    0x41, 0xa5, 0x24, 0x46, 0x00, 0xca, 0xa0, 0x08, 0x0a, 0x81, 0xf2, 0x0c,
    0x00, 0xe9, 0xb1, 0x1c, 0x86, 0x79, 0x9e, 0x07, 0x80, 0xc0, 0x00, 0x00,
    0x40, 0x04, 0x84, 0x40, 0x30, 0x00, 0x41, 0x01, 0x00, 0x00, 0x00, 0x79,
    0x18, 0x00, 0x00, 0x67, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46,
    0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b,
    0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1,
    0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa,
    0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10,
    0x04, 0x13, 0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61, 0x20, 0x26,
    0x08, 0x04, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x76, 0x73, 0x1b, 0x06, 0xc4,
    0x20, 0x26, 0x08, 0xd9, 0x44, 0x60, 0x82, 0x40, 0x24, 0x1b, 0x10, 0x42,
    0x59, 0x08, 0x62, 0x60, 0x80, 0x0d, 0x41, 0xb3, 0x81, 0x00, 0x00, 0x07,
    0x98, 0x20, 0x68, 0xd4, 0x86, 0x00, 0x9a, 0x20, 0x08, 0x00, 0x97, 0xb6,
    0xb0, 0x34, 0x37, 0xaa, 0x32, 0x3c, 0xba, 0x3a, 0xb9, 0x32, 0x22, 0x50,
    0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x13, 0x84, 0xc2, 0x99, 0x20,
    0x14, 0xcf, 0x86, 0x80, 0x98, 0x20, 0x14, 0xd0, 0x04, 0x81, 0x50, 0x26,
    0x08, 0xc4, 0xb2, 0x41, 0xc8, 0xb4, 0x0d, 0x0b, 0x41, 0x55, 0xd6, 0x65,
    0x0d, 0x18, 0x61, 0x6d, 0x44, 0xa8, 0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4,
    0x88, 0x26, 0x08, 0x45, 0xb4, 0x41, 0xc8, 0xb2, 0x0d, 0xcb, 0xd0, 0x55,
    0xd6, 0x65, 0x0d, 0xde, 0x60, 0x7d, 0x13, 0x04, 0x82, 0x61, 0x31, 0xf4,
    0xc4, 0xf4, 0x24, 0x35, 0x41, 0x28, 0xa4, 0x09, 0x02, 0xd1, 0x6c, 0x10,
    0x32, 0x32, 0xd8, 0xb0, 0x84, 0x81, 0x18, 0x54, 0xd6, 0x65, 0x0d, 0x63,
    0x10, 0x06, 0x56, 0x19, 0x6c, 0x18, 0x38, 0x30, 0x30, 0x03, 0x2e, 0x53,
    0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x1b, 0x16,
    0x02, 0x0d, 0x2a, 0xec, 0x1a, 0x83, 0x61, 0x0c, 0x08, 0xab, 0x0c, 0x36,
    0x2c, 0x43, 0x57, 0x59, 0x97, 0x37, 0x78, 0x83, 0xf5, 0x6d, 0x58, 0xc2,
    0x40, 0x0c, 0x2a, 0xeb, 0xf2, 0x86, 0x31, 0x08, 0x03, 0xab, 0x0c, 0x36,
    0x0c, 0x69, 0xa0, 0x06, 0x6b, 0xb0, 0x61, 0x38, 0x03, 0x36, 0x00, 0x36,
    0x14, 0xd2, 0xd4, 0x06, 0x0f, 0x50, 0x85, 0x8d, 0xcd, 0xae, 0xcd, 0x25,
    0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73, 0xb1,
    0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0,
    0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6, 0x04, 0x46, 0x1d, 0x32,
    0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9, 0xa6, 0x37, 0xb2, 0x32,
    0xb6, 0x29, 0x01, 0x52, 0x86, 0x0c, 0xcf, 0x45, 0xae, 0x6c, 0xee, 0xad,
    0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0xe0, 0xd4, 0x21, 0xc3, 0x73, 0xb1,
    0x4b, 0x2b, 0xbb, 0x4b, 0x22, 0x9b, 0xa2, 0x0b, 0xa3, 0x2b, 0x9b, 0x12,
    0x40, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde,
    0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x6d, 0x00, 0x00, 0x00, 0x00, 0x79,
    0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4,
    0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c,
    0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00,
    0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2,
    0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38,
    0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d,
    0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87,
    0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87,
    0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30,
    0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde,
    0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b,
    0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c,
    0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07,
    0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87,
    0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87,
    0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87,
    0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0,
    0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc,
    0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4,
    0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39,
    0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38,
    0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b,
    0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8, 0x21, 0x07, 0x7c, 0x70, 0x03,
    0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b, 0x08, 0x07, 0x79, 0x60, 0x87,
    0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x71,
    0x20, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x36, 0xb0, 0x0d, 0x97, 0xef,
    0x3c, 0xbe, 0x10, 0x50, 0x45, 0x41, 0x44, 0xa5, 0x03, 0x0c, 0x25, 0x61,
    0x00, 0x02, 0xe6, 0x17, 0xb7, 0x6d, 0x05, 0xd2, 0x70, 0xf9, 0xce, 0xe3,
    0x0b, 0x11, 0x01, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x58, 0xc0, 0x34,
    0x5c, 0xbe, 0xf3, 0xf8, 0x8b, 0x03, 0x0c, 0x62, 0xf3, 0x50, 0x93, 0x5f,
    0xdc, 0xb6, 0x09, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44,
    0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x11, 0x48, 0xc3, 0xe5,
    0x3b, 0x8f, 0x3f, 0x11, 0xd1, 0x84, 0x00, 0x11, 0xe6, 0x17, 0xb7, 0x6d,
    0x00, 0x0b, 0x03, 0x20, 0x4d, 0x4b, 0x5c, 0x13, 0x15, 0x11, 0x00, 0x61,
    0x20, 0x00, 0x00, 0xcf, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x44, 0x8a, 0xab, 0x14, 0x0a,
    0x61, 0x06, 0xa0, 0xec, 0x4a, 0x8e, 0x4a, 0x09, 0x50, 0x1c, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x61, 0x03,
    0x63, 0x59, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0xdd, 0x21,
    0x5d, 0x8f, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x87, 0x4c,
    0x18, 0x71, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xf1, 0x25, 0x54,
    0xf6, 0x20, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x80, 0x81, 0x52,
    0x69, 0x51, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46, 0x18, 0x2c,
    0xdc, 0x36, 0x29, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x88, 0x01,
    0xd3, 0x71, 0xc8, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc6, 0x18,
    0x34, 0x5d, 0x57, 0x31, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x90,
    0x81, 0xe3, 0x79, 0x4a, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46,
    0x19, 0x3c, 0xdf, 0x57, 0x39, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0,
    0x90, 0x81, 0xa3, 0x80, 0xc1, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1,
    0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0xc1, 0x01, 0x80, 0x20,
    0x18, 0x34, 0x69, 0x30, 0x3d, 0x66, 0x30, 0x9a, 0x10, 0x00, 0xa3, 0x09,
    0x42, 0x30, 0x9a, 0x30, 0x08, 0xa3, 0x09, 0xc4, 0x30, 0x62, 0x70, 0x00,
    0x20, 0x08, 0x06, 0x8d, 0x1b, 0x60, 0x54, 0x19, 0x8c, 0x26, 0x04, 0xc0,
    0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0x8c, 0x18,
    0x1c, 0x00, 0x08, 0x82, 0x41, 0x33, 0x07, 0x5d, 0xc6, 0x06, 0xa3, 0x09,
    0x01, 0x30, 0x9a, 0x20, 0x04, 0xa3, 0x09, 0x83, 0x30, 0x9a, 0x40, 0x0c,
    0x36, 0x5d, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x07, 0x0f,
    0xce, 0xe0, 0x7a, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41, 0x30, 0x78, 0xf2,
    0x00, 0x0d, 0xae, 0x25, 0xb0, 0xe0, 0x80, 0x8e, 0x59, 0x9b, 0x7c, 0x46,
    0x0c, 0x10, 0x00, 0x04, 0xc1, 0xe0, 0xe1, 0x83, 0x35, 0xd8, 0xa4, 0x60,
    0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x3e, 0x60, 0x83, 0xcd, 0x09,
    0x2c, 0x50, 0xa0, 0x63, 0xd9, 0x27, 0x9f, 0x11, 0x03, 0x04, 0x00, 0x41,
    0x30, 0x78, 0x40, 0xe1, 0x0d, 0xbe, 0x2a, 0x18, 0x31, 0x40, 0x00, 0x10,
    0x04, 0x83, 0x27, 0x14, 0xe0, 0xe0, 0x8b, 0x02, 0x0b, 0x1a, 0xe8, 0x18,
    0x37, 0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x87, 0x14,
    0xe6, 0x60, 0x0c, 0xb0, 0x60, 0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e,
    0x52, 0xa0, 0x83, 0x31, 0xa0, 0x02, 0x0b, 0x20, 0xe8, 0x8c, 0x18, 0x1c,
    0x00, 0x08, 0x82, 0x41, 0x83, 0x0a, 0x72, 0xe0, 0x06, 0x74, 0x30, 0x9a,
    0x10, 0x00, 0xa3, 0x09, 0x42, 0x30, 0x9a, 0x30, 0x08, 0xa3, 0x09, 0xc4,
    0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x4d, 0x2b, 0xdc, 0xc1, 0x1c,
    0xe8, 0xc1, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20,
    0x8c, 0x26, 0x10, 0xc3, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x34, 0xb2,
    0xc0, 0x07, 0x78, 0xd0, 0x0a, 0xa3, 0x09, 0x01, 0x30, 0x9a, 0x20, 0x04,
    0xa3, 0x09, 0x83, 0x30, 0x9a, 0x40, 0x0c, 0x23, 0x06, 0x07, 0x00, 0x82,
    0x60, 0xd0, 0xdc, 0x42, 0x28, 0xf4, 0xc1, 0x2b, 0x8c, 0x26, 0x04, 0xc0,
    0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0xd8, 0x14,
    0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x87, 0x17, 0x56,
    0xc1, 0x7b, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41, 0x30, 0x78, 0x7a, 0x81,
    0x15, 0xb6, 0x25, 0x18, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0xc7, 0x17,
    0x5a, 0x01, 0x3b, 0x02, 0xb3, 0xca, 0x40, 0x3e, 0x23, 0x06, 0x08, 0x00,
    0x82, 0x60, 0xf0, 0x80, 0xc3, 0x2b, 0x88, 0x81, 0x14, 0x8c, 0x18, 0x20,
    0x00, 0x08, 0x82, 0xc1, 0x13, 0x0e, 0xb0, 0xf0, 0x39, 0xc1, 0x88, 0x01,
    0x02, 0x80, 0x20, 0x18, 0x3c, 0xe2, 0x10, 0x0b, 0x9c, 0x12, 0x58, 0x96,
    0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x87, 0x1c, 0x66,
    0xc1, 0x0c, 0xaa, 0x60, 0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x72,
    0xa0, 0x85, 0x31, 0x88, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41, 0x30, 0x78,
    0xcc, 0xa1, 0x16, 0xc0, 0xa0, 0x09, 0x8c, 0x6b, 0x03, 0xf9, 0x8c, 0x18,
    0x20, 0x00, 0x08, 0x82, 0xc1, 0x83, 0x0e, 0xb7, 0xa0, 0x06, 0x58, 0x30,
    0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0x4f, 0x3a, 0xe0, 0xc2, 0x19, 0x50,
    0xc1, 0x88, 0x01, 0x02, 0x80, 0x20, 0x18, 0x3c, 0xea, 0x90, 0x0b, 0x64,
    0x00, 0x05, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e,
    0xa0, 0x03, 0x3a, 0x80, 0x43, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06,
    0x48, 0x3b, 0xe4, 0x02, 0x3a, 0xa0, 0x03, 0x2d, 0x24, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e, 0xa0, 0x03, 0x3a, 0xf8, 0x42,
    0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0x3b, 0xe4, 0x02, 0x3a,
    0xa0, 0xc3, 0x2f, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xb4,
    0x43, 0x2e, 0xa4, 0x03, 0x3a, 0x80, 0x43, 0x2b, 0x8c, 0x18, 0x24, 0x00,
    0x08, 0x82, 0x01, 0xd2, 0x0e, 0xb9, 0x90, 0x0e, 0xe8, 0x40, 0x0b, 0xac,
    0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0x3b, 0xe4, 0xc2, 0x38,
    0xa0, 0x03, 0x38, 0xc4, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20,
    0xed, 0x90, 0x0b, 0xe3, 0x80, 0x0e, 0xb4, 0x00, 0x0b, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e, 0x8c, 0x03, 0x3a, 0xf8, 0xc2,
    0x2b, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xd2, 0x0e, 0xb9, 0x30,
    0x0e, 0xe8, 0xf0, 0x0b, 0xae, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00
};

static unsigned char D3D12_VertexShader_YUV[] = {
    0x44, 0x58, 0x42, 0x43, 0xec, 0x10, 0x35, 0x4f, 0x25, 0x8f, 0xde, 0xa0,
    0xad, 0x2c, 0x2c, 0xb7, 0x5e, 0xd4, 0x57, 0xdd, 0x01, 0x00, 0x00, 0x00,
    0x97, 0x14, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x63, 0x01, 0x00, 0x00,
    0x73, 0x02, 0x00, 0x00, 0x5b, 0x03, 0x00, 0x00, 0x53, 0x0b, 0x00, 0x00,
    0x6f, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x80, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x71, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x54, 0x45, 0x58,
    0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00,
    0x4f, 0x53, 0x47, 0x31, 0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x03, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x7d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
    0x69, 0x6f, 0x6e, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44,
    0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x50, 0x53, 0x56, 0x30, 0x08,
    0x01, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x03,
    0x03, 0x00, 0x03, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00,
    0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x54, 0x45, 0x58,
    0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x43, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x22,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03,
    0x02, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x0f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x52, 0x54, 0x53, 0x30, 0xe0, 0x00, 0x00, 0x00, 0x02,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xe0, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x88,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
    0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01,
    0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x53, 0x54, 0x41, 0x54, 0xf0,
    0x07, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0xfc, 0x01, 0x00, 0x00, 0x44,
    0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xd8,
    0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xf3,
    0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13,
    0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06,
    0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e,
    0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xc4,
    0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48,
    0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4,
    0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1,
    0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08,
    0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40,
    0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d,
    0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49,
    0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20,
    0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x4c,
    0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13,
    0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12,
    0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x88, 0xc1, 0x0c, 0xc0,
    0x30, 0x02, 0x01, 0x24, 0x41, 0x70, 0x70, 0x30, 0x5c, 0x3e, 0xb0, 0x20,
    0x46, 0xc3, 0x10, 0xcd, 0xe4, 0x2f, 0x84, 0x01, 0x08, 0x98, 0x2f, 0x4d,
    0x11, 0x25, 0x4c, 0xfe, 0x4b, 0x44, 0x13, 0x71, 0xb1, 0x07, 0x30, 0x10,
    0x11, 0xe7, 0x34, 0xd2, 0x04, 0x34, 0x93, 0x84, 0x04, 0x41, 0xb8, 0x6e,
    0xb8, 0x7c, 0x60, 0x41, 0x8c, 0x86, 0x21, 0x9a, 0xc9, 0x5f, 0x08, 0x03,
    0x10, 0x30, 0x9f, 0x73, 0x1a, 0x69, 0x02, 0x9a, 0x49, 0x42, 0xc1, 0x40,
    0xc4, 0x08, 0x40, 0x09, 0x0c, 0x3a, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00,
    0x29, 0x06, 0x90, 0x24, 0x89, 0x92, 0xd0, 0x52, 0x0c, 0x23, 0x49, 0x12,
    0x05, 0xa0, 0xe6, 0xa8, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x9f, 0xdb,
    0xa8, 0x62, 0x25, 0x26, 0xbf, 0xb8, 0x6d, 0x44, 0x00, 0x00, 0x00, 0x90,
    0x72, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0,
    0x10, 0x28, 0x80, 0x0a, 0xf1, 0x24, 0x51, 0x42, 0x52, 0x29, 0x80, 0x04,
    0x00, 0x00, 0xa2, 0xe6, 0x08, 0x82, 0x62, 0x44, 0x89, 0x92, 0x24, 0x16,
    0x5d, 0x03, 0x01, 0x67, 0x09, 0x0b, 0x20, 0x49, 0x3e, 0x03, 0x4c, 0x11,
    0x72, 0xf9, 0xc5, 0xe2, 0x00, 0x93, 0x8f, 0xfb, 0x38, 0x0a, 0x84, 0xe3,
    0xa4, 0x29, 0xa2, 0x84, 0xc9, 0x7f, 0x89, 0x68, 0x22, 0x2e, 0xf6, 0x00,
    0x06, 0x22, 0xe2, 0x9c, 0x46, 0x9a, 0x80, 0x66, 0x92, 0x90, 0xa0, 0x69,
    0xc3, 0x08, 0x02, 0x70, 0x99, 0x34, 0x45, 0x94, 0x30, 0xf9, 0x2f, 0x11,
    0x4d, 0xc4, 0xc5, 0x1e, 0xc0, 0x40, 0x44, 0x28, 0x35, 0x3d, 0xd4, 0x84,
    0x86, 0x80, 0x0b, 0x86, 0x11, 0x06, 0xe0, 0x30, 0x69, 0x8a, 0x28, 0x61,
    0xf2, 0x5f, 0x22, 0x9a, 0x88, 0x8b, 0x3d, 0x80, 0x81, 0x88, 0xc4, 0xe6,
    0xa1, 0x26, 0x34, 0x78, 0x5c, 0x70, 0x09, 0xe7, 0x34, 0xd2, 0x04, 0x34,
    0x93, 0x84, 0x82, 0x2d, 0x1d, 0x12, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87,
    0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87,
    0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00,
    0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0,
    0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60,
    0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60,
    0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0,
    0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21, 0x23, 0x25, 0x40, 0x00, 0x52, 0x00,
    0xc0, 0x90, 0xe7, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x80, 0x21, 0xcf, 0x02, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x43, 0x9e, 0x06, 0x08, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x12, 0x10, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x28, 0x20, 0x00, 0x02, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x5c, 0x40, 0x00, 0x0c,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90, 0x05, 0x02, 0x00, 0x19,
    0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19, 0x11, 0x4c, 0x90, 0x8c,
    0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x52, 0x0a, 0xa1, 0x08, 0xca, 0x80,
    0x92, 0x12, 0x18, 0x01, 0x28, 0x86, 0x02, 0x14, 0x28, 0x82, 0x42, 0x28,
    0x83, 0x72, 0x28, 0x89, 0x02, 0x0c, 0x28, 0xb0, 0x02, 0x29, 0xa0, 0xf2,
    0x28, 0x8c, 0xd2, 0x0d, 0x28, 0x0a, 0x52, 0x4a, 0x62, 0x04, 0x80, 0x8a,
    0x19, 0x00, 0x22, 0x66, 0x00, 0x68, 0x98, 0x01, 0xa0, 0x6d, 0x06, 0x80,
    0xba, 0x19, 0x00, 0xfa, 0x66, 0x00, 0x08, 0x9c, 0x01, 0xa0, 0x70, 0x2c,
    0x87, 0x61, 0x9e, 0xe7, 0x01, 0x20, 0x30, 0x00, 0x00, 0x10, 0x01, 0x21,
    0x10, 0x0c, 0x40, 0x50, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0xc0,
    0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35,
    0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b,
    0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b,
    0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79,
    0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0x44,
    0x99, 0x20, 0x24, 0xcb, 0x06, 0x61, 0x20, 0x36, 0x08, 0x04, 0x41, 0xc1,
    0x6e, 0x6e, 0x82, 0x90, 0x30, 0x1b, 0x86, 0x03, 0x21, 0x26, 0x08, 0x45,
    0xc7, 0x64, 0xe8, 0xcd, 0x6d, 0x8e, 0x2e, 0xcc, 0x8d, 0x6e, 0x6e, 0x82,
    0x90, 0x34, 0x1b, 0x10, 0x42, 0x59, 0x08, 0x62, 0x60, 0x80, 0x0d, 0x41,
    0xb3, 0x81, 0x00, 0x00, 0x07, 0x98, 0x20, 0x6c, 0x60, 0xc0, 0xa2, 0xed,
    0x8d, 0xac, 0x8c, 0x6d, 0x82, 0x90, 0x38, 0x13, 0x84, 0xe4, 0xd9, 0x30,
    0x4c, 0xd3, 0x30, 0x41, 0x48, 0xa0, 0x09, 0x42, 0x12, 0x4d, 0x10, 0x12,
    0x69, 0x03, 0x82, 0x44, 0x12, 0x55, 0x11, 0xd6, 0xc5, 0x08, 0x4e, 0xee,
    0x4d, 0xad, 0x6c, 0x8c, 0x2e, 0xed, 0xcd, 0x2d, 0xc8, 0x8d, 0xcc, 0x2a,
    0xad, 0xec, 0x6e, 0x82, 0x90, 0x4c, 0x1b, 0x10, 0x24, 0x93, 0xa8, 0x4a,
    0xb3, 0xae, 0x0d, 0x03, 0x83, 0x6d, 0x13, 0x84, 0x2e, 0x0c, 0x26, 0x08,
    0x09, 0xc5, 0x01, 0xee, 0x6d, 0x8e, 0xcb, 0x94, 0xd5, 0x17, 0xd4, 0xd3,
    0x54, 0x12, 0x55, 0xd2, 0x93, 0xd3, 0x06, 0x04, 0xf9, 0x2a, 0x62, 0x02,
    0x03, 0xeb, 0xe2, 0x40, 0x57, 0x86, 0x37, 0x41, 0x48, 0x2a, 0x26, 0x54,
    0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x30, 0x1b, 0x10, 0x44, 0x0c,
    0xaa, 0x31, 0x98, 0xc8, 0xc0, 0xba, 0x58, 0x8c, 0xbd, 0xb1, 0xbd, 0xc9,
    0x4d, 0x10, 0x12, 0x8b, 0xc6, 0xd0, 0x13, 0xd3, 0x93, 0x14, 0xcc, 0x06,
    0x04, 0x31, 0x83, 0xea, 0x0c, 0x26, 0x34, 0xb0, 0xae, 0x0d, 0x84, 0x17,
    0x06, 0x65, 0x90, 0x06, 0x13, 0x84, 0x4f, 0x0c, 0x88, 0x40, 0x3d, 0x4d,
    0x25, 0x51, 0x25, 0x3d, 0x39, 0x6d, 0x40, 0x90, 0xaf, 0x22, 0x26, 0x36,
    0xb0, 0xae, 0x0d, 0x84, 0xd7, 0x06, 0x65, 0x90, 0x06, 0x13, 0x04, 0x30,
    0x18, 0x83, 0x0d, 0x04, 0xb2, 0x54, 0xc4, 0x06, 0x81, 0x81, 0x83, 0x09,
    0xc2, 0xf0, 0x4d, 0x10, 0x08, 0x6f, 0x43, 0x43, 0x40, 0x5c, 0xa7, 0x06,
    0x6b, 0xe0, 0x06, 0x6f, 0x10, 0x07, 0x72, 0xc0, 0xcd, 0x41, 0x1c, 0x4c,
    0x10, 0x90, 0x60, 0x03, 0xb0, 0x61, 0x20, 0xec, 0xc0, 0x0e, 0x36, 0x04,
    0x77, 0xb0, 0x61, 0x18, 0xea, 0x00, 0x0f, 0x26, 0x08, 0x61, 0x40, 0x06,
    0x1b, 0x02, 0x3d, 0xe0, 0xd1, 0x16, 0x96, 0xe6, 0x96, 0x55, 0x65, 0x35,
    0x41, 0x50, 0xb6, 0x09, 0x82, 0xc2, 0x6d, 0x08, 0x88, 0x09, 0x82, 0x92,
    0x6c, 0x10, 0x2a, 0x6b, 0xc3, 0x42, 0xb0, 0x41, 0x1f, 0xf8, 0xc1, 0x1f,
    0xf8, 0xc1, 0x00, 0x0a, 0x84, 0x1f, 0x84, 0x02, 0x11, 0xaa, 0x22, 0xac,
    0xa1, 0xa7, 0x27, 0x29, 0xa2, 0x09, 0x82, 0x82, 0x6c, 0x10, 0xaa, 0x6a,
    0xc3, 0x32, 0x8c, 0x42, 0x1f, 0xf8, 0xc1, 0x1f, 0xf8, 0xc1, 0x40, 0x0a,
    0x83, 0x1f, 0x94, 0x02, 0x8b, 0xa1, 0x27, 0xa6, 0x27, 0xa9, 0x09, 0x82,
    0x72, 0x4c, 0x10, 0x92, 0x6b, 0x83, 0x50, 0xa5, 0xc2, 0x86, 0x45, 0x3a,
    0x85, 0x3e, 0xf0, 0x83, 0x3f, 0xf0, 0x83, 0x01, 0x15, 0x24, 0x3f, 0x50,
    0x85, 0x0d, 0x83, 0x28, 0x98, 0xc2, 0x2a, 0x70, 0x99, 0xb2, 0xfa, 0x82,
    0x7a, 0x9b, 0x4b, 0xa3, 0x4b, 0x7b, 0x73, 0xdb, 0xb0, 0x10, 0xad, 0xd0,
    0x07, 0xa0, 0xf0, 0x07, 0xa8, 0x30, 0xa0, 0x02, 0xe1, 0x07, 0xaa, 0xb0,
    0x61, 0x19, 0x46, 0xa1, 0x0f, 0xfc, 0xe0, 0x0f, 0x48, 0x61, 0x20, 0x85,
    0xc1, 0x0f, 0x4a, 0x61, 0xc3, 0x22, 0x9d, 0x42, 0x1f, 0xf8, 0xc1, 0x1f,
    0x90, 0xc2, 0x80, 0x0a, 0x92, 0x1f, 0xa8, 0xc2, 0x86, 0xc1, 0x15, 0x5e,
    0x01, 0x16, 0x36, 0x0c, 0xac, 0x10, 0x0b, 0xc0, 0x86, 0xa2, 0x0e, 0xf8,
    0x40, 0x16, 0x1e, 0x80, 0x86, 0x19, 0xdb, 0x5b, 0x18, 0xdd, 0xdc, 0x04,
    0x21, 0xc1, 0x58, 0xa4, 0xb9, 0xcd, 0xd1, 0xcd, 0x4d, 0x10, 0x92, 0x8c,
    0xc6, 0x5c, 0xda, 0xd9, 0x17, 0x1b, 0x19, 0x8d, 0xb9, 0xb4, 0xb3, 0xaf,
    0x39, 0xba, 0x09, 0x42, 0xa2, 0x6d, 0x40, 0x68, 0xa1, 0x16, 0x6c, 0xe1,
    0x16, 0x70, 0xe1, 0xca, 0x05, 0x5d, 0xa8, 0xc2, 0xc6, 0x66, 0xd7, 0xe6,
    0x92, 0x46, 0x56, 0xe6, 0x46, 0x37, 0x25, 0x08, 0xaa, 0x90, 0xe1, 0xb9,
    0xd8, 0x95, 0xc9, 0xcd, 0xa5, 0xbd, 0xb9, 0x4d, 0x09, 0x88, 0x26, 0x64,
    0x78, 0x2e, 0x76, 0x61, 0x6c, 0x76, 0x65, 0x72, 0x53, 0x82, 0xa2, 0x0e,
    0x19, 0x9e, 0xcb, 0x1c, 0x5a, 0x18, 0x59, 0x99, 0x5c, 0xd3, 0x1b, 0x59,
    0x19, 0xdb, 0x94, 0x00, 0x29, 0x43, 0x86, 0xe7, 0x22, 0x57, 0x36, 0xf7,
    0x56, 0x27, 0x37, 0x56, 0x36, 0x37, 0x25, 0x70, 0x2a, 0x91, 0xe1, 0xb9,
    0xd0, 0xe5, 0xc1, 0x95, 0x05, 0xb9, 0xb9, 0xbd, 0xd1, 0x85, 0xd1, 0xa5,
    0xbd, 0xb9, 0xcd, 0x4d, 0x11, 0xe6, 0x00, 0x0f, 0xea, 0x90, 0xe1, 0xb9,
    0xd8, 0xa5, 0x95, 0xdd, 0x25, 0x91, 0x4d, 0xd1, 0x85, 0xd1, 0x95, 0x4d,
    0x09, 0xf4, 0xa0, 0x0e, 0x19, 0x9e, 0x4b, 0x99, 0x1b, 0x9d, 0x5c, 0x1e,
    0xd4, 0x5b, 0x9a, 0x1b, 0xdd, 0xdc, 0x94, 0x40, 0x16, 0xba, 0x90, 0xe1,
    0xb9, 0x8c, 0xbd, 0xd5, 0xb9, 0xd1, 0x95, 0xc9, 0xcd, 0x4d, 0x09, 0x74,
    0x01, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33,
    0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43,
    0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98,
    0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33,
    0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05,
    0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43,
    0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08,
    0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78,
    0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1,
    0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33,
    0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e,
    0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03,
    0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60,
    0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80,
    0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8,
    0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18,
    0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee,
    0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c,
    0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c,
    0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43,
    0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3,
    0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83,
    0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8, 0x21,
    0x07, 0x7c, 0x70, 0x03, 0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b, 0x08,
    0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00,
    0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x06,
    0x00, 0x71, 0x4e, 0x23, 0x4d, 0x40, 0x33, 0x49, 0xff, 0x42, 0x18, 0x80,
    0x80, 0x19, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x40, 0x15, 0x05,
    0x11, 0x95, 0x0e, 0x30, 0x94, 0x84, 0x01, 0x08, 0x98, 0x5f, 0xdc, 0xb6,
    0x19, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44, 0x04, 0x30, 0x11, 0x21,
    0xd0, 0x0c, 0x0b, 0x61, 0x02, 0xd3, 0x70, 0xf9, 0xce, 0xe3, 0x2f, 0x0e,
    0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x36, 0x50, 0x0d, 0x97,
    0xef, 0x3c, 0xbe, 0x34, 0x39, 0x11, 0x81, 0x52, 0xd3, 0x43, 0x4d, 0x7e,
    0x71, 0xdb, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0x44, 0x44, 0x13,
    0x02, 0x44, 0x98, 0x5f, 0xdc, 0xb6, 0x05, 0x1c, 0x0c, 0x80, 0x34, 0xb2,
    0xfb, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7b, 0x37, 0x12, 0xc2, 0x76,
    0x04, 0xc6, 0xc0, 0xad, 0xe5, 0xc2, 0x1e, 0x4d, 0x70, 0xa6, 0x71, 0x44,
    0x58, 0x49, 0x4c, 0x20, 0x09, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0x48,
    0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x08, 0x09, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21,
    0x0c, 0x00, 0x00, 0x3f, 0x02, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02,
    0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41,
    0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25,
    0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42,
    0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a,
    0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00,
    0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41,
    0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51,
    0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff,
    0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff,
    0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00,
    0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13,
    0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89,
    0x20, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20,
    0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84,
    0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10,
    0x6c, 0x23, 0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60,
    0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84, 0x14, 0x42, 0xa6, 0x18, 0x80, 0x10,
    0x52, 0x06, 0xa1, 0xa3, 0x86, 0xcb, 0x9f, 0xb0, 0x87, 0x90, 0x7c, 0x6e,
    0xa3, 0x8a, 0x95, 0x98, 0xfc, 0xe2, 0xb6, 0x11, 0x31, 0xc6, 0x18, 0x54,
    0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16,
    0x02, 0x05, 0xab, 0x10, 0x8a, 0x30, 0x42, 0xad, 0x14, 0x83, 0x8c, 0x31,
    0xe8, 0xcd, 0x11, 0x04, 0xc5, 0x60, 0xa4, 0x10, 0x12, 0x49, 0x0e, 0x04,
    0x0c, 0x23, 0x10, 0x43, 0x12, 0xd4, 0x83, 0x83, 0xe1, 0xf2, 0x81, 0x05,
    0x31, 0x1a, 0x86, 0x68, 0x26, 0x7f, 0x21, 0x0c, 0x40, 0xc0, 0x7c, 0x69,
    0x8a, 0x28, 0x61, 0xf2, 0x5f, 0x22, 0x9a, 0x88, 0x8b, 0x3d, 0x80, 0x81,
    0x88, 0x38, 0xa7, 0x91, 0x26, 0xa0, 0x99, 0x24, 0x24, 0x58, 0x7b, 0xdd,
    0x70, 0xf9, 0xc0, 0x82, 0x18, 0x0d, 0x43, 0x34, 0x93, 0xbf, 0x10, 0x06,
    0x20, 0x60, 0x3e, 0xe7, 0x34, 0xd2, 0x04, 0x34, 0x93, 0x84, 0x82, 0x4b,
    0x38, 0x1d, 0x08, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87,
    0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87,
    0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00,
    0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90,
    0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0,
    0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30,
    0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20,
    0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0,
    0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60,
    0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60,
    0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0,
    0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40,
    0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x30,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c,
    0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02, 0x50, 0x0c,
    0x05, 0x28, 0x50, 0x06, 0xe5, 0x50, 0x04, 0xe5, 0x41, 0xa5, 0x24, 0x46,
    0x00, 0xca, 0xa0, 0x08, 0x0a, 0x81, 0xf2, 0x0c, 0x00, 0xe9, 0xb1, 0x1c,
    0x86, 0x79, 0x9e, 0x07, 0x80, 0xc0, 0x00, 0x00, 0x40, 0x04, 0x84, 0x40,
    0x30, 0x00, 0x41, 0x01, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x66,
    0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35,
    0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b,
    0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b,
    0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79,
    0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0xc2,
    0x98, 0x20, 0x10, 0xc7, 0x06, 0x61, 0x20, 0x26, 0x08, 0x04, 0xb2, 0x41,
    0x18, 0x0c, 0x0a, 0x76, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0xd9,
    0x44, 0x60, 0x82, 0x40, 0x24, 0x1b, 0x10, 0x42, 0x59, 0x08, 0x62, 0x60,
    0x80, 0x0d, 0x41, 0xb3, 0x81, 0x00, 0x00, 0x07, 0x98, 0x20, 0x68, 0xd4,
    0x86, 0x00, 0x9a, 0x20, 0x08, 0x00, 0x8f, 0xb6, 0xb0, 0x34, 0xb7, 0xac,
    0x2a, 0x2b, 0x22, 0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x13,
    0x84, 0xc2, 0x99, 0x20, 0x14, 0xcf, 0x86, 0x80, 0x98, 0x20, 0x14, 0xd0,
    0x04, 0x81, 0x50, 0x26, 0x08, 0xc4, 0xb2, 0x41, 0xc8, 0xb4, 0x0d, 0x0b,
    0x41, 0x55, 0xd6, 0x65, 0x0d, 0x18, 0x61, 0x6d, 0x44, 0xa8, 0x8a, 0xb0,
    0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26, 0x08, 0x45, 0xb4, 0x41, 0xc8, 0xb2,
    0x0d, 0xcb, 0xd0, 0x55, 0xd6, 0x65, 0x0d, 0xde, 0x60, 0x7d, 0x13, 0x04,
    0x82, 0x61, 0x31, 0xf4, 0xc4, 0xf4, 0x24, 0x35, 0x41, 0x28, 0xa4, 0x09,
    0x02, 0xd1, 0x6c, 0x10, 0x32, 0x32, 0xd8, 0xb0, 0x84, 0x81, 0x18, 0x54,
    0xd6, 0x65, 0x0d, 0x63, 0x10, 0x06, 0x56, 0x19, 0x6c, 0x18, 0x38, 0x30,
    0x30, 0x03, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69,
    0x6f, 0x6e, 0x1b, 0x16, 0x02, 0x0d, 0x2a, 0xec, 0x1a, 0x83, 0x61, 0x0c,
    0x08, 0xab, 0x0c, 0x36, 0x2c, 0x43, 0x57, 0x59, 0x97, 0x37, 0x78, 0x83,
    0xf5, 0x6d, 0x58, 0xc2, 0x40, 0x0c, 0x2a, 0xeb, 0xf2, 0x86, 0x31, 0x08,
    0x03, 0xab, 0x0c, 0x36, 0x0c, 0x69, 0xa0, 0x06, 0x6b, 0xb0, 0x61, 0x38,
    0x03, 0x36, 0x00, 0x36, 0x14, 0xd2, 0xd4, 0x06, 0x0f, 0x50, 0x85, 0x8d,
    0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e, 0x4a, 0x10, 0x54,
    0x21, 0xc3, 0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x9b, 0x12,
    0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec, 0xca, 0xe4, 0xa6,
    0x04, 0x46, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30, 0xb2, 0x32, 0xb9,
    0xa6, 0x37, 0xb2, 0x32, 0xb6, 0x29, 0x01, 0x52, 0x86, 0x0c, 0xcf, 0x45,
    0xae, 0x6c, 0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e, 0x4a, 0xe0, 0xd4,
    0x21, 0xc3, 0x73, 0xb1, 0x4b, 0x2b, 0xbb, 0x4b, 0x22, 0x9b, 0xa2, 0x0b,
    0xa3, 0x2b, 0x9b, 0x12, 0x40, 0x75, 0xc8, 0xf0, 0x5c, 0xca, 0xdc, 0xe8,
    0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6, 0x04, 0x6d, 0x00,
    0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33,
    0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43,
    0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98,
    0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33,
    0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05,
    0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43,
    0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08,
    0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78,
    0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1,
    0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33,
    0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e,
    0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03,
    0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60,
    0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80,
    0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8,
    0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18,
    0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee,
    0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c,
    0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c,
    0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43,
    0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3,
    0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83,
    0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8, 0x21,
    0x07, 0x7c, 0x70, 0x03, 0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b, 0x08,
    0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00,
    0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x36,
    0xb0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x50, 0x45, 0x41, 0x44, 0xa5,
    0x03, 0x0c, 0x25, 0x61, 0x00, 0x02, 0xe6, 0x17, 0xb7, 0x6d, 0x05, 0xd2,
    0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x11, 0x01, 0x4c, 0x44, 0x08, 0x34, 0xc3,
    0x42, 0x58, 0xc0, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x8b, 0x03, 0x0c, 0x62,
    0xf3, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x09, 0x54, 0xc3, 0xe5, 0x3b, 0x8f,
    0x2f, 0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0xb6,
    0x11, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x11, 0xd1, 0x84, 0x00, 0x11,
    0xe6, 0x17, 0xb7, 0x6d, 0x00, 0x07, 0x03, 0x20, 0x8d, 0xec, 0xbe, 0x00,
    0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0xcf, 0x00, 0x00, 0x00, 0x13,
    0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x44,
    0x8a, 0xab, 0x14, 0x0a, 0x61, 0x06, 0xa0, 0xec, 0x4a, 0x8e, 0x4a, 0x09,
    0x50, 0x1c, 0x01, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82,
    0x60, 0x20, 0x61, 0x03, 0x63, 0x59, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20,
    0x18, 0x18, 0xdd, 0x21, 0x5d, 0x8f, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08,
    0x06, 0x86, 0x87, 0x4c, 0x18, 0x71, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82,
    0x81, 0xf1, 0x25, 0x54, 0xf6, 0x20, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
    0x60, 0x80, 0x81, 0x52, 0x69, 0x51, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08,
    0x06, 0x46, 0x18, 0x2c, 0xdc, 0x36, 0x29, 0x23, 0x06, 0x09, 0x00, 0x82,
    0x60, 0x60, 0x88, 0x01, 0xd3, 0x71, 0xc8, 0x32, 0x62, 0x90, 0x00, 0x20,
    0x08, 0x06, 0xc6, 0x18, 0x34, 0x5d, 0x57, 0x31, 0x23, 0x06, 0x09, 0x00,
    0x82, 0x60, 0x60, 0x90, 0x81, 0xe3, 0x79, 0x4a, 0x33, 0x62, 0x90, 0x00,
    0x20, 0x08, 0x06, 0x46, 0x19, 0x3c, 0xdf, 0x57, 0x39, 0x23, 0x06, 0x07,
    0x00, 0x82, 0x60, 0xd0, 0x90, 0x81, 0xa3, 0x80, 0xc1, 0x68, 0x42, 0x00,
    0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88,
    0xc1, 0x01, 0x80, 0x20, 0x18, 0x34, 0x69, 0x30, 0x3d, 0x66, 0x30, 0x9a,
    0x10, 0x00, 0xa3, 0x09, 0x42, 0x30, 0x9a, 0x30, 0x08, 0xa3, 0x09, 0xc4,
    0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x8d, 0x1b, 0x60, 0x54, 0x19,
    0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68,
    0x02, 0x31, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x41, 0x33, 0x07, 0x5d,
    0xc6, 0x06, 0xa3, 0x09, 0x01, 0x30, 0x9a, 0x20, 0x04, 0xa3, 0x09, 0x83,
    0x30, 0x9a, 0x40, 0x0c, 0x36, 0x5d, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10,
    0x04, 0x83, 0x07, 0x0f, 0xce, 0xe0, 0x7a, 0x82, 0x11, 0x03, 0x04, 0x00,
    0x41, 0x30, 0x78, 0xf2, 0x00, 0x0d, 0xae, 0x25, 0xb0, 0xe0, 0x80, 0x8e,
    0x59, 0x9b, 0x7c, 0x46, 0x0c, 0x10, 0x00, 0x04, 0xc1, 0xe0, 0xe1, 0x83,
    0x35, 0xd8, 0xa4, 0x60, 0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x3e,
    0x60, 0x83, 0xcd, 0x09, 0x2c, 0x50, 0xa0, 0x63, 0xd9, 0x27, 0x9f, 0x11,
    0x03, 0x04, 0x00, 0x41, 0x30, 0x78, 0x40, 0xe1, 0x0d, 0xbe, 0x2a, 0x18,
    0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x27, 0x14, 0xe0, 0xe0, 0x8b, 0x02,
    0x0b, 0x1a, 0xe8, 0x18, 0x37, 0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10,
    0x04, 0x83, 0x87, 0x14, 0xe6, 0x60, 0x0c, 0xb0, 0x60, 0xc4, 0x00, 0x01,
    0x40, 0x10, 0x0c, 0x9e, 0x52, 0xa0, 0x83, 0x31, 0xa0, 0x02, 0x0b, 0x20,
    0xe8, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x41, 0x83, 0x0a, 0x72, 0xe0,
    0x06, 0x74, 0x30, 0x9a, 0x10, 0x00, 0xa3, 0x09, 0x42, 0x30, 0x9a, 0x30,
    0x08, 0xa3, 0x09, 0xc4, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x4d,
    0x2b, 0xdc, 0xc1, 0x1c, 0xe8, 0xc1, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08,
    0xc1, 0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0xc1, 0x01, 0x80,
    0x20, 0x18, 0x34, 0xb2, 0xc0, 0x07, 0x78, 0xd0, 0x0a, 0xa3, 0x09, 0x01,
    0x30, 0x9a, 0x20, 0x04, 0xa3, 0x09, 0x83, 0x30, 0x9a, 0x40, 0x0c, 0x23,
    0x06, 0x07, 0x00, 0x82, 0x60, 0xd0, 0xdc, 0x42, 0x28, 0xf4, 0xc1, 0x2b,
    0x8c, 0x26, 0x04, 0xc0, 0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68,
    0x02, 0x31, 0xd8, 0x14, 0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04,
    0x83, 0x87, 0x17, 0x56, 0xc1, 0x7b, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41,
    0x30, 0x78, 0x7a, 0x81, 0x15, 0xb6, 0x25, 0x18, 0x31, 0x40, 0x00, 0x10,
    0x04, 0x83, 0xc7, 0x17, 0x5a, 0x01, 0x3b, 0x02, 0xb3, 0xca, 0x40, 0x3e,
    0x23, 0x06, 0x08, 0x00, 0x82, 0x60, 0xf0, 0x80, 0xc3, 0x2b, 0x88, 0x81,
    0x14, 0x8c, 0x18, 0x20, 0x00, 0x08, 0x82, 0xc1, 0x13, 0x0e, 0xb0, 0xf0,
    0x39, 0xc1, 0x88, 0x01, 0x02, 0x80, 0x20, 0x18, 0x3c, 0xe2, 0x10, 0x0b,
    0x9c, 0x12, 0x58, 0x96, 0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04,
    0x83, 0x87, 0x1c, 0x66, 0xc1, 0x0c, 0xaa, 0x60, 0xc4, 0x00, 0x01, 0x40,
    0x10, 0x0c, 0x9e, 0x72, 0xa0, 0x85, 0x31, 0x88, 0x82, 0x11, 0x03, 0x04,
    0x00, 0x41, 0x30, 0x78, 0xcc, 0xa1, 0x16, 0xc0, 0xa0, 0x09, 0x8c, 0x6b,
    0x03, 0xf9, 0x8c, 0x18, 0x20, 0x00, 0x08, 0x82, 0xc1, 0x83, 0x0e, 0xb7,
    0xa0, 0x06, 0x58, 0x30, 0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0x4f, 0x3a,
    0xe0, 0xc2, 0x19, 0x50, 0xc1, 0x88, 0x01, 0x02, 0x80, 0x20, 0x18, 0x3c,
    0xea, 0x90, 0x0b, 0x64, 0x00, 0x05, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60,
    0x80, 0xb4, 0x43, 0x2e, 0xa0, 0x03, 0x3a, 0x80, 0x43, 0x33, 0x62, 0x90,
    0x00, 0x20, 0x08, 0x06, 0x48, 0x3b, 0xe4, 0x02, 0x3a, 0xa0, 0x03, 0x2d,
    0x24, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e, 0xa0,
    0x03, 0x3a, 0xf8, 0x42, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48,
    0x3b, 0xe4, 0x02, 0x3a, 0xa0, 0xc3, 0x2f, 0x04, 0x23, 0x06, 0x09, 0x00,
    0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e, 0xa4, 0x03, 0x3a, 0x80, 0x43, 0x2b,
    0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xd2, 0x0e, 0xb9, 0x90, 0x0e,
    0xe8, 0x40, 0x0b, 0xac, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48,
    0x3b, 0xe4, 0xc2, 0x38, 0xa0, 0x03, 0x38, 0xc4, 0xc2, 0x88, 0x41, 0x02,
    0x80, 0x20, 0x18, 0x20, 0xed, 0x90, 0x0b, 0xe3, 0x80, 0x0e, 0xb4, 0x00,
    0x0b, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e, 0x8c,
    0x03, 0x3a, 0xf8, 0xc2, 0x2b, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01,
    0xd2, 0x0e, 0xb9, 0x30, 0x0e, 0xe8, 0xf0, 0x0b, 0xae, 0x80, 0x00, 0x00,
    0x00, 0x00, 0x00
};

static unsigned char D3D12_VertexShader_NV[] = {
    0x44, 0x58, 0x42, 0x43, 0x52, 0x51, 0xf4, 0x61, 0xe0, 0x16, 0x3e, 0xc4,
    0xd9, 0x83, 0xc5, 0x7a, 0x0a, 0xa7, 0x7f, 0x7c, 0x01, 0x00, 0x00, 0x00,
    0x63, 0x14, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x63, 0x01, 0x00, 0x00,
    0x73, 0x02, 0x00, 0x00, 0x2f, 0x03, 0x00, 0x00, 0x27, 0x0b, 0x00, 0x00,
    0x43, 0x0b, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31,
    0x80, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x71, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7a, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0f, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x54, 0x45, 0x58,
    0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00,
    0x4f, 0x53, 0x47, 0x31, 0x83, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x03, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x7d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
    0x69, 0x6f, 0x6e, 0x00, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44,
    0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00, 0x50, 0x53, 0x56, 0x30, 0x08,
    0x01, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x03,
    0x03, 0x00, 0x03, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00,
    0x50, 0x4f, 0x53, 0x49, 0x54, 0x49, 0x4f, 0x4e, 0x00, 0x54, 0x45, 0x58,
    0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c, 0x4f, 0x52, 0x00,
    0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x00, 0x43, 0x4f, 0x4c,
    0x4f, 0x52, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x43, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x42, 0x00, 0x03, 0x02, 0x00, 0x00, 0x22,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x44, 0x00, 0x03,
    0x02, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x0f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x52, 0x54, 0x53, 0x30, 0xb4, 0x00, 0x00, 0x00, 0x02,
    0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x7c,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
    0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x53,
    0x54, 0x41, 0x54, 0xf0, 0x07, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0xfc,
    0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10,
    0x00, 0x00, 0x00, 0xd8, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21,
    0x0c, 0x00, 0x00, 0xf3, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02,
    0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41,
    0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25,
    0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42,
    0x92, 0x0b, 0x42, 0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a,
    0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00,
    0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50, 0x41,
    0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51,
    0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff,
    0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff,
    0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff, 0xff, 0xff, 0x1f, 0x00,
    0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13,
    0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89,
    0x20, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20,
    0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3, 0x84,
    0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10,
    0x88, 0xc1, 0x0c, 0xc0, 0x30, 0x02, 0x01, 0x24, 0x41, 0x70, 0x70, 0x30,
    0x5c, 0x3e, 0xb0, 0x20, 0x46, 0xc3, 0x10, 0xcd, 0xe4, 0x2f, 0x84, 0x01,
    0x08, 0x98, 0x2f, 0x4d, 0x11, 0x25, 0x4c, 0xfe, 0x4b, 0x44, 0x13, 0x71,
    0xb1, 0x07, 0x30, 0x10, 0x11, 0xe7, 0x34, 0xd2, 0x04, 0x34, 0x93, 0x84,
    0x04, 0x41, 0xb8, 0x6e, 0xb8, 0x7c, 0x60, 0x41, 0x8c, 0x86, 0x21, 0x9a,
    0xc9, 0x5f, 0x08, 0x03, 0x10, 0x30, 0x9f, 0x73, 0x1a, 0x69, 0x02, 0x9a,
    0x49, 0x42, 0xc1, 0x40, 0xc4, 0x08, 0x40, 0x09, 0x0c, 0x3a, 0xe6, 0x08,
    0xc0, 0x60, 0x8e, 0x00, 0x29, 0x06, 0x90, 0x24, 0x89, 0x92, 0xd0, 0x52,
    0x0c, 0x23, 0x49, 0x12, 0x05, 0xa0, 0xe6, 0xa8, 0xe1, 0xf2, 0x27, 0xec,
    0x21, 0x24, 0x9f, 0xdb, 0xa8, 0x62, 0x25, 0x26, 0xbf, 0xb8, 0x6d, 0x44,
    0x00, 0x00, 0x00, 0x90, 0x72, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10, 0x92,
    0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x80, 0x0a, 0xf1, 0x24, 0x51, 0x42,
    0x52, 0x29, 0x80, 0x04, 0x00, 0x00, 0xa2, 0xe6, 0x08, 0x82, 0x62, 0x44,
    0x89, 0x92, 0x24, 0x16, 0x5d, 0x03, 0x01, 0x67, 0x09, 0x0b, 0x20, 0x49,
    0x3e, 0x03, 0x4c, 0x11, 0x72, 0xf9, 0xc5, 0xe2, 0x00, 0x93, 0x8f, 0xfb,
    0x38, 0x0a, 0x84, 0xe3, 0xa4, 0x29, 0xa2, 0x84, 0xc9, 0x7f, 0x89, 0x68,
    0x22, 0x2e, 0xf6, 0x00, 0x06, 0x22, 0xe2, 0x9c, 0x46, 0x9a, 0x80, 0x66,
    0x92, 0x90, 0xa0, 0x69, 0xc3, 0x08, 0x02, 0x70, 0x99, 0x34, 0x45, 0x94,
    0x30, 0xf9, 0x2f, 0x11, 0x4d, 0xc4, 0xc5, 0x1e, 0xc0, 0x40, 0x44, 0x28,
    0x35, 0x3d, 0xd4, 0x84, 0x86, 0x80, 0x0b, 0x86, 0x11, 0x06, 0xe0, 0x30,
    0x69, 0x8a, 0x28, 0x61, 0xf2, 0x5f, 0x22, 0x9a, 0x88, 0x8b, 0x3d, 0x80,
    0x81, 0x88, 0xc4, 0xe6, 0xa1, 0x26, 0x34, 0x78, 0x5c, 0x70, 0x09, 0xe7,
    0x34, 0xd2, 0x04, 0x34, 0x93, 0x84, 0x82, 0x2d, 0x1d, 0x12, 0x00, 0x13,
    0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68,
    0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a,
    0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
    0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
    0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72,
    0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a,
    0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d,
    0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xa0, 0x07, 0x76, 0x40, 0x07, 0x3a, 0x0f, 0x64, 0x90, 0x21, 0x23, 0x25,
    0x40, 0x00, 0x52, 0x00, 0xc0, 0x90, 0xe7, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x21, 0xcf, 0x02, 0x04, 0x40, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x9e, 0x06, 0x08, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x12, 0x10,
    0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x28,
    0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2,
    0x5c, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x90,
    0x05, 0x02, 0x00, 0x19, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x18, 0x19,
    0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x52, 0x0a,
    0xa1, 0x08, 0xca, 0x80, 0x92, 0x12, 0x18, 0x01, 0x28, 0x86, 0x02, 0x14,
    0x28, 0x82, 0x42, 0x28, 0x83, 0x72, 0x28, 0x89, 0x02, 0x0c, 0x28, 0xb0,
    0x02, 0x29, 0xa0, 0xf2, 0x28, 0x8c, 0xd2, 0x0d, 0x28, 0x0a, 0x52, 0x4a,
    0x62, 0x04, 0x80, 0x8a, 0x19, 0x00, 0x22, 0x66, 0x00, 0x68, 0x98, 0x01,
    0xa0, 0x6d, 0x06, 0x80, 0xba, 0x19, 0x00, 0xfa, 0x66, 0x00, 0x08, 0x9c,
    0x01, 0xa0, 0x70, 0x2c, 0x87, 0x61, 0x9e, 0xe7, 0x01, 0x20, 0x30, 0x00,
    0x00, 0x10, 0x01, 0x21, 0x10, 0x0c, 0x40, 0x50, 0x00, 0x00, 0x00, 0x79,
    0x18, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46,
    0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b,
    0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1,
    0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa,
    0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10,
    0x04, 0x13, 0x84, 0x44, 0x99, 0x20, 0x24, 0xcb, 0x06, 0x61, 0x20, 0x36,
    0x08, 0x04, 0x41, 0xc1, 0x6e, 0x6e, 0x82, 0x90, 0x30, 0x1b, 0x86, 0x03,
    0x21, 0x26, 0x08, 0x45, 0xc7, 0x64, 0xe8, 0xcd, 0x6d, 0x8e, 0x2e, 0xcc,
    0x8d, 0x6e, 0x6e, 0x82, 0x90, 0x34, 0x1b, 0x10, 0x42, 0x59, 0x08, 0x62,
    0x60, 0x80, 0x0d, 0x41, 0xb3, 0x81, 0x00, 0x00, 0x07, 0x98, 0x20, 0x6c,
    0x60, 0xc0, 0xa2, 0xed, 0x8d, 0xac, 0x8c, 0x6d, 0x82, 0x90, 0x38, 0x13,
    0x84, 0xe4, 0xd9, 0x30, 0x4c, 0xd3, 0x30, 0x41, 0x48, 0xa0, 0x09, 0x42,
    0x12, 0x4d, 0x10, 0x12, 0x69, 0x03, 0x82, 0x44, 0x12, 0x55, 0x11, 0xd6,
    0xc5, 0x08, 0x4e, 0xee, 0x4d, 0xad, 0x6c, 0x8c, 0x2e, 0xed, 0xcd, 0x2d,
    0xc8, 0x8d, 0xcc, 0x2a, 0xad, 0xec, 0x6e, 0x82, 0x90, 0x4c, 0x1b, 0x10,
    0x24, 0x93, 0xa8, 0x4a, 0xb3, 0xae, 0x0d, 0x03, 0x83, 0x6d, 0x13, 0x84,
    0x2e, 0x0c, 0x26, 0x08, 0x09, 0xc5, 0x01, 0xee, 0x6d, 0x8e, 0xcb, 0x94,
    0xd5, 0x17, 0xd4, 0xd3, 0x54, 0x12, 0x55, 0xd2, 0x93, 0xd3, 0x06, 0x04,
    0xf9, 0x2a, 0x62, 0x02, 0x03, 0xeb, 0xe2, 0x40, 0x57, 0x86, 0x37, 0x41,
    0x48, 0x2a, 0x26, 0x54, 0x45, 0x58, 0x43, 0x4f, 0x4f, 0x52, 0x44, 0x30,
    0x1b, 0x10, 0x44, 0x0c, 0xaa, 0x31, 0x98, 0xc8, 0xc0, 0xba, 0x58, 0x8c,
    0xbd, 0xb1, 0xbd, 0xc9, 0x4d, 0x10, 0x12, 0x8b, 0xc6, 0xd0, 0x13, 0xd3,
    0x93, 0x14, 0xcc, 0x06, 0x04, 0x31, 0x83, 0xea, 0x0c, 0x26, 0x34, 0xb0,
    0xae, 0x0d, 0x84, 0x17, 0x06, 0x65, 0x90, 0x06, 0x13, 0x84, 0x4f, 0x0c,
    0x88, 0x40, 0x3d, 0x4d, 0x25, 0x51, 0x25, 0x3d, 0x39, 0x6d, 0x40, 0x90,
    0xaf, 0x22, 0x26, 0x36, 0xb0, 0xae, 0x0d, 0x84, 0xd7, 0x06, 0x65, 0x90,
    0x06, 0x13, 0x04, 0x30, 0x18, 0x83, 0x0d, 0x04, 0xb2, 0x54, 0xc4, 0x06,
    0x81, 0x81, 0x83, 0x09, 0xc2, 0xf0, 0x4d, 0x10, 0x08, 0x6f, 0x43, 0x43,
    0x40, 0x5c, 0xa7, 0x06, 0x6b, 0xe0, 0x06, 0x6f, 0x10, 0x07, 0x72, 0xc0,
    0xcd, 0x41, 0x1c, 0x4c, 0x10, 0x90, 0x60, 0x03, 0xb0, 0x61, 0x20, 0xec,
    0xc0, 0x0e, 0x36, 0x04, 0x77, 0xb0, 0x61, 0x18, 0xea, 0x00, 0x0f, 0x26,
    0x08, 0x61, 0x40, 0x06, 0x1b, 0x02, 0x3d, 0xa0, 0xd1, 0x16, 0x96, 0xe6,
    0xe6, 0x64, 0x35, 0x41, 0x50, 0xb6, 0x09, 0x82, 0xc2, 0x6d, 0x08, 0x88,
    0x09, 0x82, 0x92, 0x6c, 0x10, 0x2a, 0x6b, 0xc3, 0x42, 0xb0, 0x41, 0x1f,
    0xf8, 0xc1, 0x1f, 0xf8, 0xc1, 0x00, 0x0a, 0x84, 0x1f, 0x84, 0x02, 0x11,
    0xaa, 0x22, 0xac, 0xa1, 0xa7, 0x27, 0x29, 0xa2, 0x09, 0x82, 0x82, 0x6c,
    0x10, 0xaa, 0x6a, 0xc3, 0x32, 0x8c, 0x42, 0x1f, 0xf8, 0xc1, 0x1f, 0xf8,
    0xc1, 0x40, 0x0a, 0x83, 0x1f, 0x94, 0x02, 0x8b, 0xa1, 0x27, 0xa6, 0x27,
    0xa9, 0x09, 0x82, 0x72, 0x4c, 0x10, 0x92, 0x6b, 0x83, 0x50, 0xa5, 0xc2,
    0x86, 0x45, 0x3a, 0x85, 0x3e, 0xf0, 0x83, 0x3f, 0xf0, 0x83, 0x01, 0x15,
    0x24, 0x3f, 0x50, 0x85, 0x0d, 0x83, 0x28, 0x98, 0xc2, 0x2a, 0x70, 0x99,
    0xb2, 0xfa, 0x82, 0x7a, 0x9b, 0x4b, 0xa3, 0x4b, 0x7b, 0x73, 0xdb, 0xb0,
    0x10, 0xad, 0xd0, 0x07, 0xa0, 0xf0, 0x07, 0xa8, 0x30, 0xa0, 0x02, 0xe1,
    0x07, 0xaa, 0xb0, 0x61, 0x19, 0x46, 0xa1, 0x0f, 0xfc, 0xe0, 0x0f, 0x48,
    0x61, 0x20, 0x85, 0xc1, 0x0f, 0x4a, 0x61, 0xc3, 0x22, 0x9d, 0x42, 0x1f,
    0xf8, 0xc1, 0x1f, 0x90, 0xc2, 0x80, 0x0a, 0x92, 0x1f, 0xa8, 0xc2, 0x86,
    0xc1, 0x15, 0x5e, 0x01, 0x16, 0x36, 0x0c, 0xac, 0x10, 0x0b, 0xc0, 0x86,
    0xa2, 0x0e, 0xf8, 0x40, 0x16, 0x1e, 0x80, 0x86, 0x19, 0xdb, 0x5b, 0x18,
    0xdd, 0xdc, 0x04, 0x21, 0xc1, 0x58, 0xa4, 0xb9, 0xcd, 0xd1, 0xcd, 0x4d,
    0x10, 0x92, 0x8c, 0xc6, 0x5c, 0xda, 0xd9, 0x17, 0x1b, 0x19, 0x8d, 0xb9,
    0xb4, 0xb3, 0xaf, 0x39, 0xba, 0x09, 0x42, 0xa2, 0x6d, 0x40, 0x68, 0xa1,
    0x16, 0x6c, 0xe1, 0x16, 0x70, 0xe1, 0xca, 0x05, 0x5d, 0xa8, 0xc2, 0xc6,
    0x66, 0xd7, 0xe6, 0x92, 0x46, 0x56, 0xe6, 0x46, 0x37, 0x25, 0x08, 0xaa,
    0x90, 0xe1, 0xb9, 0xd8, 0x95, 0xc9, 0xcd, 0xa5, 0xbd, 0xb9, 0x4d, 0x09,
    0x88, 0x26, 0x64, 0x78, 0x2e, 0x76, 0x61, 0x6c, 0x76, 0x65, 0x72, 0x53,
    0x82, 0xa2, 0x0e, 0x19, 0x9e, 0xcb, 0x1c, 0x5a, 0x18, 0x59, 0x99, 0x5c,
    0xd3, 0x1b, 0x59, 0x19, 0xdb, 0x94, 0x00, 0x29, 0x43, 0x86, 0xe7, 0x22,
    0x57, 0x36, 0xf7, 0x56, 0x27, 0x37, 0x56, 0x36, 0x37, 0x25, 0x70, 0x2a,
    0x91, 0xe1, 0xb9, 0xd0, 0xe5, 0xc1, 0x95, 0x05, 0xb9, 0xb9, 0xbd, 0xd1,
    0x85, 0xd1, 0xa5, 0xbd, 0xb9, 0xcd, 0x4d, 0x11, 0xe6, 0x00, 0x0f, 0xea,
    0x90, 0xe1, 0xb9, 0xd8, 0xa5, 0x95, 0xdd, 0x25, 0x91, 0x4d, 0xd1, 0x85,
    0xd1, 0x95, 0x4d, 0x09, 0xf4, 0xa0, 0x0e, 0x19, 0x9e, 0x4b, 0x99, 0x1b,
    0x9d, 0x5c, 0x1e, 0xd4, 0x5b, 0x9a, 0x1b, 0xdd, 0xdc, 0x94, 0x40, 0x16,
    0xba, 0x90, 0xe1, 0xb9, 0x8c, 0xbd, 0xd5, 0xb9, 0xd1, 0x95, 0xc9, 0xcd,
    0x4d, 0x09, 0x74, 0x01, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x49,
    0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14,
    0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79,
    0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e,
    0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1,
    0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc,
    0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74,
    0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a,
    0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e,
    0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e,
    0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21,
    0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0,
    0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc,
    0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72,
    0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76,
    0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f,
    0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c,
    0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03,
    0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1,
    0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61,
    0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8,
    0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94,
    0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0,
    0xc3, 0x8c, 0xc8, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x72, 0x10, 0x87, 0x73,
    0x70, 0x03, 0x7b, 0x08, 0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77,
    0xa8, 0x07, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1c,
    0x00, 0x00, 0x00, 0x06, 0x00, 0x71, 0x4e, 0x23, 0x4d, 0x40, 0x33, 0x49,
    0xff, 0x42, 0x18, 0x80, 0x80, 0x19, 0xc1, 0x36, 0x5c, 0xbe, 0xf3, 0xf8,
    0x42, 0x40, 0x15, 0x05, 0x11, 0x95, 0x0e, 0x30, 0x94, 0x84, 0x01, 0x08,
    0x98, 0x5f, 0xdc, 0xb6, 0x19, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x44,
    0x04, 0x30, 0x11, 0x21, 0xd0, 0x0c, 0x0b, 0x61, 0x02, 0xd3, 0x70, 0xf9,
    0xce, 0xe3, 0x2f, 0x0e, 0x30, 0x88, 0xcd, 0x43, 0x4d, 0x7e, 0x71, 0xdb,
    0x36, 0x50, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x34, 0x39, 0x11, 0x81, 0x52,
    0xd3, 0x43, 0x4d, 0x7e, 0x71, 0xdb, 0x56, 0x20, 0x0d, 0x97, 0xef, 0x3c,
    0xfe, 0x44, 0x44, 0x13, 0x02, 0x44, 0x98, 0x5f, 0xdc, 0xb6, 0x05, 0x18,
    0x0c, 0x80, 0x34, 0xe7, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48,
    0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4e,
    0x7e, 0xdd, 0xd9, 0x5c, 0x85, 0x79, 0x2c, 0xb5, 0xd6, 0x2b, 0x4f, 0x5a,
    0xb5, 0xd4, 0xd5, 0x44, 0x58, 0x49, 0x4c, 0x18, 0x09, 0x00, 0x00, 0x60,
    0x00, 0x01, 0x00, 0x46, 0x02, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00,
    0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x42,
    0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x3d, 0x02, 0x00, 0x00, 0x0b,
    0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07,
    0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92,
    0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80,
    0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38,
    0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43,
    0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91,
    0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04,
    0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b,
    0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84,
    0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff, 0xff,
    0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03,
    0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00,
    0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x32,
    0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04,
    0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b,
    0x84, 0xa4, 0x4c, 0x10, 0x6c, 0x23, 0x00, 0x25, 0x00, 0x14, 0x66, 0x00,
    0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84, 0x14, 0x42,
    0xa6, 0x18, 0x80, 0x10, 0x52, 0x06, 0xa1, 0xa3, 0x86, 0xcb, 0x9f, 0xb0,
    0x87, 0x90, 0x7c, 0x6e, 0xa3, 0x8a, 0x95, 0x98, 0xfc, 0xe2, 0xb6, 0x11,
    0x31, 0xc6, 0x18, 0x54, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2,
    0x43, 0xa0, 0x19, 0x16, 0x02, 0x05, 0xab, 0x10, 0x8a, 0x30, 0x42, 0xad,
    0x14, 0x83, 0x8c, 0x31, 0xe8, 0xcd, 0x11, 0x04, 0xc5, 0x60, 0xa4, 0x10,
    0x12, 0x49, 0x0e, 0x04, 0x0c, 0x23, 0x10, 0x43, 0x12, 0xd4, 0x83, 0x83,
    0xe1, 0xf2, 0x81, 0x05, 0x31, 0x1a, 0x86, 0x68, 0x26, 0x7f, 0x21, 0x0c,
    0x40, 0xc0, 0x7c, 0x69, 0x8a, 0x28, 0x61, 0xf2, 0x5f, 0x22, 0x9a, 0x88,
    0x8b, 0x3d, 0x80, 0x81, 0x88, 0x38, 0xa7, 0x91, 0x26, 0xa0, 0x99, 0x24,
    0x24, 0x58, 0x7b, 0xdd, 0x70, 0xf9, 0xc0, 0x82, 0x18, 0x0d, 0x43, 0x34,
    0x93, 0xbf, 0x10, 0x06, 0x20, 0x60, 0x3e, 0xe7, 0x34, 0xd2, 0x04, 0x34,
    0x93, 0x84, 0x82, 0x4b, 0x38, 0x1d, 0x08, 0x00, 0x00, 0x00, 0x00, 0x13,
    0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68,
    0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a,
    0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d,
    0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71,
    0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72,
    0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a,
    0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73,
    0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d,
    0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72,
    0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40,
    0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79,
    0x80, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8,
    0x23, 0x01, 0x01, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40,
    0x16, 0x08, 0x00, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19,
    0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x22, 0x25,
    0x30, 0x02, 0x50, 0x0c, 0x05, 0x28, 0x50, 0x06, 0xe5, 0x50, 0x04, 0xe5,
    0x41, 0xa5, 0x24, 0x46, 0x00, 0xca, 0xa0, 0x08, 0x0a, 0x81, 0xf2, 0x0c,
    0x00, 0xe9, 0xb1, 0x1c, 0x86, 0x79, 0x9e, 0x07, 0x80, 0xc0, 0x00, 0x00,
    0x40, 0x04, 0x84, 0x40, 0x30, 0x00, 0x41, 0x01, 0x00, 0x00, 0x00, 0x79,
    0x18, 0x00, 0x00, 0x65, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46,
    0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b,
    0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1,
    0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa,
    0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10,
    0x04, 0x13, 0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61, 0x20, 0x26,
    0x08, 0x04, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x76, 0x73, 0x1b, 0x06, 0xc4,
    0x20, 0x26, 0x08, 0xd9, 0x44, 0x60, 0x82, 0x40, 0x24, 0x1b, 0x10, 0x42,
    0x59, 0x08, 0x62, 0x60, 0x80, 0x0d, 0x41, 0xb3, 0x81, 0x00, 0x00, 0x07,
    0x98, 0x20, 0x68, 0xd4, 0x86, 0x00, 0x9a, 0x20, 0x08, 0x00, 0x8d, 0xb6,
    0xb0, 0x34, 0x37, 0x27, 0x2b, 0x22, 0x50, 0x4f, 0x53, 0x49, 0x54, 0x49,
    0x4f, 0x4e, 0x13, 0x84, 0xc2, 0x99, 0x20, 0x14, 0xcf, 0x86, 0x80, 0x98,
    0x20, 0x14, 0xd0, 0x04, 0x81, 0x50, 0x26, 0x08, 0xc4, 0xb2, 0x41, 0xc8,
    0xb4, 0x0d, 0x0b, 0x41, 0x55, 0xd6, 0x65, 0x0d, 0x18, 0x61, 0x6d, 0x44,
    0xa8, 0x8a, 0xb0, 0x86, 0x9e, 0x9e, 0xa4, 0x88, 0x26, 0x08, 0x45, 0xb4,
    0x41, 0xc8, 0xb2, 0x0d, 0xcb, 0xd0, 0x55, 0xd6, 0x65, 0x0d, 0xde, 0x60,
    0x7d, 0x13, 0x04, 0x82, 0x61, 0x31, 0xf4, 0xc4, 0xf4, 0x24, 0x35, 0x41,
    0x28, 0xa4, 0x09, 0x02, 0xd1, 0x6c, 0x10, 0x32, 0x32, 0xd8, 0xb0, 0x84,
    0x81, 0x18, 0x54, 0xd6, 0x65, 0x0d, 0x63, 0x10, 0x06, 0x56, 0x19, 0x6c,
    0x18, 0x38, 0x30, 0x30, 0x03, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73,
    0x69, 0x74, 0x69, 0x6f, 0x6e, 0x1b, 0x16, 0x02, 0x0d, 0x2a, 0xec, 0x1a,
    0x83, 0x61, 0x0c, 0x08, 0xab, 0x0c, 0x36, 0x2c, 0x43, 0x57, 0x59, 0x97,
    0x37, 0x78, 0x83, 0xf5, 0x6d, 0x58, 0xc2, 0x40, 0x0c, 0x2a, 0xeb, 0xf2,
    0x86, 0x31, 0x08, 0x03, 0xab, 0x0c, 0x36, 0x0c, 0x69, 0xa0, 0x06, 0x6b,
    0xb0, 0x61, 0x38, 0x03, 0x36, 0x00, 0x36, 0x14, 0xd2, 0xd4, 0x06, 0x0f,
    0x50, 0x85, 0x8d, 0xcd, 0xae, 0xcd, 0x25, 0x8d, 0xac, 0xcc, 0x8d, 0x6e,
    0x4a, 0x10, 0x54, 0x21, 0xc3, 0x73, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b,
    0x73, 0x9b, 0x12, 0x10, 0x4d, 0xc8, 0xf0, 0x5c, 0xec, 0xc2, 0xd8, 0xec,
    0xca, 0xe4, 0xa6, 0x04, 0x46, 0x1d, 0x32, 0x3c, 0x97, 0x39, 0xb4, 0x30,
    0xb2, 0x32, 0xb9, 0xa6, 0x37, 0xb2, 0x32, 0xb6, 0x29, 0x01, 0x52, 0x86,
    0x0c, 0xcf, 0x45, 0xae, 0x6c, 0xee, 0xad, 0x4e, 0x6e, 0xac, 0x6c, 0x6e,
    0x4a, 0xe0, 0xd4, 0x21, 0xc3, 0x73, 0xb1, 0x4b, 0x2b, 0xbb, 0x4b, 0x22,
    0x9b, 0xa2, 0x0b, 0xa3, 0x2b, 0x9b, 0x12, 0x40, 0x75, 0xc8, 0xf0, 0x5c,
    0xca, 0xdc, 0xe8, 0xe4, 0xf2, 0xa0, 0xde, 0xd2, 0xdc, 0xe8, 0xe6, 0xa6,
    0x04, 0x6d, 0x00, 0x79, 0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33,
    0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43,
    0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98,
    0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33,
    0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05,
    0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43,
    0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08,
    0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78,
    0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1,
    0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33,
    0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e,
    0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03,
    0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60,
    0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80,
    0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8,
    0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18,
    0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee,
    0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c,
    0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c,
    0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43,
    0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3,
    0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83,
    0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8, 0x21,
    0x07, 0x7c, 0x70, 0x03, 0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b, 0x08,
    0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00,
    0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x36,
    0xb0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x50, 0x45, 0x41, 0x44, 0xa5,
    0x03, 0x0c, 0x25, 0x61, 0x00, 0x02, 0xe6, 0x17, 0xb7, 0x6d, 0x05, 0xd2,
    0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x11, 0x01, 0x4c, 0x44, 0x08, 0x34, 0xc3,
    0x42, 0x58, 0xc0, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x8b, 0x03, 0x0c, 0x62,
    0xf3, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x09, 0x54, 0xc3, 0xe5, 0x3b, 0x8f,
    0x2f, 0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4, 0x50, 0x93, 0x5f, 0xdc, 0xb6,
    0x11, 0x48, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x11, 0xd1, 0x84, 0x00, 0x11,
    0xe6, 0x17, 0xb7, 0x6d, 0x00, 0x06, 0x03, 0x20, 0xcd, 0xf9, 0x02, 0x61,
    0x20, 0x00, 0x00, 0xcf, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10,
    0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x44, 0x8a, 0xab, 0x14, 0x0a,
    0x61, 0x06, 0xa0, 0xec, 0x4a, 0x8e, 0x4a, 0x09, 0x50, 0x1c, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x20, 0x61, 0x03,
    0x63, 0x59, 0xc1, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x18, 0xdd, 0x21,
    0x5d, 0x8f, 0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x86, 0x87, 0x4c,
    0x18, 0x71, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x81, 0xf1, 0x25, 0x54,
    0xf6, 0x20, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x80, 0x81, 0x52,
    0x69, 0x51, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46, 0x18, 0x2c,
    0xdc, 0x36, 0x29, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x88, 0x01,
    0xd3, 0x71, 0xc8, 0x32, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0xc6, 0x18,
    0x34, 0x5d, 0x57, 0x31, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x90,
    0x81, 0xe3, 0x79, 0x4a, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x46,
    0x19, 0x3c, 0xdf, 0x57, 0x39, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0xd0,
    0x90, 0x81, 0xa3, 0x80, 0xc1, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1,
    0x68, 0xc2, 0x20, 0x8c, 0x26, 0x10, 0xc3, 0x88, 0xc1, 0x01, 0x80, 0x20,
    0x18, 0x34, 0x69, 0x30, 0x3d, 0x66, 0x30, 0x9a, 0x10, 0x00, 0xa3, 0x09,
    0x42, 0x30, 0x9a, 0x30, 0x08, 0xa3, 0x09, 0xc4, 0x30, 0x62, 0x70, 0x00,
    0x20, 0x08, 0x06, 0x8d, 0x1b, 0x60, 0x54, 0x19, 0x8c, 0x26, 0x04, 0xc0,
    0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0x8c, 0x18,
    0x1c, 0x00, 0x08, 0x82, 0x41, 0x33, 0x07, 0x5d, 0xc6, 0x06, 0xa3, 0x09,
    0x01, 0x30, 0x9a, 0x20, 0x04, 0xa3, 0x09, 0x83, 0x30, 0x9a, 0x40, 0x0c,
    0x36, 0x5d, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x07, 0x0f,
    0xce, 0xe0, 0x7a, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41, 0x30, 0x78, 0xf2,
    0x00, 0x0d, 0xae, 0x25, 0xb0, 0xe0, 0x80, 0x8e, 0x59, 0x9b, 0x7c, 0x46,
    0x0c, 0x10, 0x00, 0x04, 0xc1, 0xe0, 0xe1, 0x83, 0x35, 0xd8, 0xa4, 0x60,
    0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x3e, 0x60, 0x83, 0xcd, 0x09,
    0x2c, 0x50, 0xa0, 0x63, 0xd9, 0x27, 0x9f, 0x11, 0x03, 0x04, 0x00, 0x41,
    0x30, 0x78, 0x40, 0xe1, 0x0d, 0xbe, 0x2a, 0x18, 0x31, 0x40, 0x00, 0x10,
    0x04, 0x83, 0x27, 0x14, 0xe0, 0xe0, 0x8b, 0x02, 0x0b, 0x1a, 0xe8, 0x18,
    0x37, 0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x87, 0x14,
    0xe6, 0x60, 0x0c, 0xb0, 0x60, 0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e,
    0x52, 0xa0, 0x83, 0x31, 0xa0, 0x02, 0x0b, 0x20, 0xe8, 0x8c, 0x18, 0x1c,
    0x00, 0x08, 0x82, 0x41, 0x83, 0x0a, 0x72, 0xe0, 0x06, 0x74, 0x30, 0x9a,
    0x10, 0x00, 0xa3, 0x09, 0x42, 0x30, 0x9a, 0x30, 0x08, 0xa3, 0x09, 0xc4,
    0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0x4d, 0x2b, 0xdc, 0xc1, 0x1c,
    0xe8, 0xc1, 0x68, 0x42, 0x00, 0x8c, 0x26, 0x08, 0xc1, 0x68, 0xc2, 0x20,
    0x8c, 0x26, 0x10, 0xc3, 0x88, 0xc1, 0x01, 0x80, 0x20, 0x18, 0x34, 0xb2,
    0xc0, 0x07, 0x78, 0xd0, 0x0a, 0xa3, 0x09, 0x01, 0x30, 0x9a, 0x20, 0x04,
    0xa3, 0x09, 0x83, 0x30, 0x9a, 0x40, 0x0c, 0x23, 0x06, 0x07, 0x00, 0x82,
    0x60, 0xd0, 0xdc, 0x42, 0x28, 0xf4, 0xc1, 0x2b, 0x8c, 0x26, 0x04, 0xc0,
    0x68, 0x82, 0x10, 0x8c, 0x26, 0x0c, 0xc2, 0x68, 0x02, 0x31, 0xd8, 0x14,
    0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x87, 0x17, 0x56,
    0xc1, 0x7b, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41, 0x30, 0x78, 0x7a, 0x81,
    0x15, 0xb6, 0x25, 0x18, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0xc7, 0x17,
    0x5a, 0x01, 0x3b, 0x02, 0xb3, 0xca, 0x40, 0x3e, 0x23, 0x06, 0x08, 0x00,
    0x82, 0x60, 0xf0, 0x80, 0xc3, 0x2b, 0x88, 0x81, 0x14, 0x8c, 0x18, 0x20,
    0x00, 0x08, 0x82, 0xc1, 0x13, 0x0e, 0xb0, 0xf0, 0x39, 0xc1, 0x88, 0x01,
    0x02, 0x80, 0x20, 0x18, 0x3c, 0xe2, 0x10, 0x0b, 0x9c, 0x12, 0x58, 0x96,
    0x06, 0xf2, 0x19, 0x31, 0x40, 0x00, 0x10, 0x04, 0x83, 0x87, 0x1c, 0x66,
    0xc1, 0x0c, 0xaa, 0x60, 0xc4, 0x00, 0x01, 0x40, 0x10, 0x0c, 0x9e, 0x72,
    0xa0, 0x85, 0x31, 0x88, 0x82, 0x11, 0x03, 0x04, 0x00, 0x41, 0x30, 0x78,
    0xcc, 0xa1, 0x16, 0xc0, 0xa0, 0x09, 0x8c, 0x6b, 0x03, 0xf9, 0x8c, 0x18,
    0x20, 0x00, 0x08, 0x82, 0xc1, 0x83, 0x0e, 0xb7, 0xa0, 0x06, 0x58, 0x30,
    0x62, 0x80, 0x00, 0x20, 0x08, 0x06, 0x4f, 0x3a, 0xe0, 0xc2, 0x19, 0x50,
    0xc1, 0x88, 0x01, 0x02, 0x80, 0x20, 0x18, 0x3c, 0xea, 0x90, 0x0b, 0x64,
    0x00, 0x05, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e,
    0xa0, 0x03, 0x3a, 0x80, 0x43, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06,
    0x48, 0x3b, 0xe4, 0x02, 0x3a, 0xa0, 0x03, 0x2d, 0x24, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e, 0xa0, 0x03, 0x3a, 0xf8, 0x42,
    0x31, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0x3b, 0xe4, 0x02, 0x3a,
    0xa0, 0xc3, 0x2f, 0x04, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xb4,
    0x43, 0x2e, 0xa4, 0x03, 0x3a, 0x80, 0x43, 0x2b, 0x8c, 0x18, 0x24, 0x00,
    0x08, 0x82, 0x01, 0xd2, 0x0e, 0xb9, 0x90, 0x0e, 0xe8, 0x40, 0x0b, 0xac,
    0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x48, 0x3b, 0xe4, 0xc2, 0x38,
    0xa0, 0x03, 0x38, 0xc4, 0xc2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x20,
    0xed, 0x90, 0x0b, 0xe3, 0x80, 0x0e, 0xb4, 0x00, 0x0b, 0x23, 0x06, 0x09,
    0x00, 0x82, 0x60, 0x80, 0xb4, 0x43, 0x2e, 0x8c, 0x03, 0x3a, 0xf8, 0xc2,
    0x2b, 0x8c, 0x18, 0x24, 0x00, 0x08, 0x82, 0x01, 0xd2, 0x0e, 0xb9, 0x30,
    0x0e, 0xe8, 0xf0, 0x0b, 0xae, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* Root signature blobs extracted from Vertex Shader
   dxc command line is:
   dxc -E <RS> -T rootsig_1_1 -rootsig-define <RS> -Fo <OUTFILE> D3D12_VertexShader.hlsl

    Variables:
     - <RS>: the root signature define
     - <OUTFILE>: the output file name.

*/

static unsigned char D3D12_RootSig_Color[] = {
    0x44, 0x58, 0x42, 0x43, 0x16, 0xb1, 0xe2, 0x29, 0x64, 0xb1, 0xb2, 0xe8,
    0x3c, 0xf7, 0xf6, 0x21, 0xad, 0x40, 0xa9, 0x50, 0x01, 0x00, 0x00, 0x00,
    0x5c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x52, 0x54, 0x53, 0x30, 0x30, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00
};

static unsigned char D3D12_RootSig_Texture[] = {
    0x44, 0x58, 0x42, 0x43, 0x7a, 0x3c, 0xe5, 0xd1, 0x3c, 0x38, 0x44, 0x34,
    0xda, 0xbc, 0x9b, 0xfc, 0xb3, 0xe7, 0x95, 0xaa, 0x01, 0x00, 0x00, 0x00,
    0xb4, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x52, 0x54, 0x53, 0x30, 0x88, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x88, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x50, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff
};

static unsigned char D3D12_RootSig_YUV[] = {
    0x44, 0x58, 0x42, 0x43, 0x65, 0xa2, 0x64, 0x32, 0x6e, 0x35, 0x8d, 0x76,
    0xf7, 0x54, 0x3c, 0x62, 0x2e, 0x8e, 0xa4, 0xbc, 0x01, 0x00, 0x00, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x52, 0x54, 0x53, 0x30, 0xe0, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xe0, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0x01, 0x00, 0x00, 0x00, 0xa8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00,
    0xc8, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff
};

static unsigned char D3D12_RootSig_NV[] = {
    0x44, 0x58, 0x42, 0x43, 0x22, 0xd8, 0x7c, 0x4d, 0xb5, 0x44, 0x4a, 0xcd,
    0x25, 0xc2, 0x80, 0xac, 0x89, 0xb0, 0xa6, 0xfe, 0x01, 0x00, 0x00, 0x00,
    0xe0, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x52, 0x54, 0x53, 0x30, 0xb4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xb4, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x5c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0x01, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff
};

static struct
{
    const void *ps_shader_data;
    SIZE_T ps_shader_size;
    const void *vs_shader_data;
    SIZE_T vs_shader_size;
    D3D12_RootSignature root_sig;
} D3D12_shaders[NUM_SHADERS] = {
    { D3D12_PixelShader_Colors, sizeof(D3D12_PixelShader_Colors),
      D3D12_VertexShader_Colors, sizeof(D3D12_VertexShader_Colors),
      ROOTSIG_COLOR },
    { D3D12_PixelShader_Textures, sizeof(D3D12_PixelShader_Textures),
      D3D12_VertexShader_Textures, sizeof(D3D12_VertexShader_Textures),
      ROOTSIG_TEXTURE },
#if SDL_HAVE_YUV
    { D3D12_PixelShader_YUV_JPEG, sizeof(D3D12_PixelShader_YUV_JPEG),
      D3D12_VertexShader_YUV, sizeof(D3D12_VertexShader_YUV),
      ROOTSIG_YUV },
    { D3D12_PixelShader_YUV_BT601, sizeof(D3D12_PixelShader_YUV_BT601),
      D3D12_VertexShader_YUV, sizeof(D3D12_VertexShader_YUV),
      ROOTSIG_YUV },
    { D3D12_PixelShader_YUV_BT709, sizeof(D3D12_PixelShader_YUV_BT709),
      D3D12_VertexShader_YUV, sizeof(D3D12_VertexShader_YUV),
      ROOTSIG_YUV },
    { D3D12_PixelShader_NV12_JPEG, sizeof(D3D12_PixelShader_NV12_JPEG),
      D3D12_VertexShader_NV, sizeof(D3D12_VertexShader_NV),
      ROOTSIG_NV },
    { D3D12_PixelShader_NV12_BT601, sizeof(D3D12_PixelShader_NV12_BT601),
      D3D12_VertexShader_NV, sizeof(D3D12_VertexShader_NV),
      ROOTSIG_NV },
    { D3D12_PixelShader_NV12_BT709, sizeof(D3D12_PixelShader_NV12_BT709),
      D3D12_VertexShader_NV, sizeof(D3D12_VertexShader_NV),
      ROOTSIG_NV },
    { D3D12_PixelShader_NV21_JPEG, sizeof(D3D12_PixelShader_NV21_JPEG),
      D3D12_VertexShader_NV, sizeof(D3D12_VertexShader_NV),
      ROOTSIG_NV },
    { D3D12_PixelShader_NV21_BT601, sizeof(D3D12_PixelShader_NV21_BT601),
      D3D12_VertexShader_NV, sizeof(D3D12_VertexShader_NV),
      ROOTSIG_NV },
    { D3D12_PixelShader_NV21_BT709, sizeof(D3D12_PixelShader_NV21_BT709),
      D3D12_VertexShader_NV, sizeof(D3D12_VertexShader_NV),
      ROOTSIG_NV },
#endif
};

static struct
{
    const void *rs_shader_data;
    SIZE_T rs_shader_size;
} D3D12_rootsigs[NUM_ROOTSIGS] = {
    { D3D12_RootSig_Color, sizeof(D3D12_RootSig_Color) },
    { D3D12_RootSig_Texture, sizeof(D3D12_RootSig_Texture) },
#if SDL_HAVE_YUV
    { D3D12_RootSig_YUV, sizeof(D3D12_RootSig_YUV) },
    { D3D12_RootSig_NV, sizeof(D3D12_RootSig_NV) },
#endif
};

void D3D12_GetVertexShader(D3D12_Shader shader, D3D12_SHADER_BYTECODE *outBytecode)
{
    outBytecode->pShaderBytecode = D3D12_shaders[shader].vs_shader_data;
    outBytecode->BytecodeLength = D3D12_shaders[shader].vs_shader_size;
}

void D3D12_GetPixelShader(D3D12_Shader shader, D3D12_SHADER_BYTECODE *outBytecode)
{
    outBytecode->pShaderBytecode = D3D12_shaders[shader].ps_shader_data;
    outBytecode->BytecodeLength = D3D12_shaders[shader].ps_shader_size;
}

D3D12_RootSignature D3D12_GetRootSignatureType(D3D12_Shader shader)
{
    return D3D12_shaders[shader].root_sig;
}

void D3D12_GetRootSignatureData(D3D12_RootSignature rootSig, D3D12_SHADER_BYTECODE *outBytecode)
{
    outBytecode->pShaderBytecode = D3D12_rootsigs[rootSig].rs_shader_data;
    outBytecode->BytecodeLength = D3D12_rootsigs[rootSig].rs_shader_size;
}

#endif /* SDL_VIDEO_RENDER_D3D12 && !SDL_RENDER_DISABLED  && !defined(__XBOXONE__) && !defined(__XBOXSERIES__) */

/* vi: set ts=4 sw=4 expandtab: */
