/*
 * Unit tests for metafile functions
 *
 * Copyright (c) 2002 Dmitry Timoshkov
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <assert.h>
#include <stdio.h>
#include <math.h>

#include "wine/test.h"
#include "winbase.h"
#include "wingdi.h"
#include "winuser.h"
#include "winerror.h"

static LOGFONTA orig_lf;
static BOOL emr_processed = FALSE;

/* Arbitrarily chosen values for the second co-ordinate of a metafile line */
#define LINE_X 55.0f
#define LINE_Y 15.0f

static INT (WINAPI * pGetRelAbs)(HDC, DWORD);
static INT (WINAPI * pSetRelAbs)(HDC, INT);
static COLORREF (WINAPI *pSetDCBrushColor)(HDC,COLORREF);
static COLORREF (WINAPI *pSetDCPenColor)(HDC,COLORREF);

#define GDI_GET_PROC(func)                                     \
    p ## func = (void *)GetProcAddress(hGDI, #func);           \
    if(!p ## func)                                             \
        trace("GetProcAddress(hGDI, \"%s\") failed\n", #func); \

static void init_function_pointers(void)
{
    HMODULE hGDI;

    pGetRelAbs = NULL;
    pSetRelAbs = NULL;

    hGDI = GetModuleHandleA("gdi32.dll");
    assert(hGDI);
    GDI_GET_PROC(GetRelAbs);
    GDI_GET_PROC(SetRelAbs);
    GDI_GET_PROC(SetDCBrushColor);
    GDI_GET_PROC(SetDCPenColor);
}

static int CALLBACK eto_emf_enum_proc(HDC hdc, HANDLETABLE *handle_table,
    const ENHMETARECORD *emr, int n_objs, LPARAM param)
{
    static int n_record;
    DWORD i;
    const INT *dx;
    INT *orig_dx = (INT *)param;
    LOGFONTA device_lf;
    INT ret;

    if(!hdc) return 1;

    PlayEnhMetaFileRecord(hdc, handle_table, emr, n_objs);

    switch (emr->iType)
    {
    case EMR_HEADER:
        ok(GetTextAlign(hdc) == 0, "text align %08x\n", GetTextAlign(hdc));
        ok(GetBkColor(hdc) == RGB(0xff, 0xff, 0xff), "bk color %08lx\n", GetBkColor(hdc));
        ok(GetTextColor(hdc) == RGB(0x0, 0x0, 0x0), "text color %08lx\n", GetTextColor(hdc));
        ok(GetROP2(hdc) == R2_COPYPEN, "rop %d\n", GetROP2(hdc));
        ok(GetArcDirection(hdc) == AD_COUNTERCLOCKWISE, "arc dir %d\n", GetArcDirection(hdc));
        ok(GetPolyFillMode(hdc) == ALTERNATE, "poly fill %d\n", GetPolyFillMode(hdc));
        ok(GetStretchBltMode(hdc) == BLACKONWHITE, "stretchblt mode %d\n", GetStretchBltMode(hdc));

        /* GetBkMode, GetRelAbs do not get reset to the default value */
        ok(GetBkMode(hdc) == OPAQUE, "bk mode %d\n", GetBkMode(hdc));
        if(pSetRelAbs && pGetRelAbs)
            ok(pGetRelAbs(hdc, 0) == RELATIVE, "relabs %d\n", pGetRelAbs(hdc, 0));

        n_record = 0;
        break;

    case EMR_EXTTEXTOUTA:
    {
        const EMREXTTEXTOUTA *emr_ExtTextOutA = (const EMREXTTEXTOUTA *)emr;
        dx = (const INT *)((const char *)emr + emr_ExtTextOutA->emrtext.offDx);

        ret = GetObjectA(GetCurrentObject(hdc, OBJ_FONT), sizeof(device_lf), &device_lf);
        ok( ret == sizeof(device_lf), "GetObjectA error %ld\n", GetLastError());

        /* compare up to lfOutPrecision, other values are not interesting,
         * and in fact sometimes arbitrary adapted by Win9x.
         */
        ok(!memcmp(&orig_lf, &device_lf, FIELD_OFFSET(LOGFONTA, lfOutPrecision)), "fonts don't match\n");
        ok(!lstrcmpA(orig_lf.lfFaceName, device_lf.lfFaceName), "font names don't match\n");

        for(i = 0; i < emr_ExtTextOutA->emrtext.nChars; i++)
        {
            ok(orig_dx[i] == dx[i], "pass %d: dx[%ld] (%d) didn't match %d\n",
                                     n_record, i, dx[i], orig_dx[i]);
        }
        n_record++;
        emr_processed = TRUE;
        break;
    }

    case EMR_EXTTEXTOUTW:
    {
        const EMREXTTEXTOUTW *emr_ExtTextOutW = (const EMREXTTEXTOUTW *)emr;
        dx = (const INT *)((const char *)emr + emr_ExtTextOutW->emrtext.offDx);

        SetLastError(0xdeadbeef);
        ret = GetObjectA(GetCurrentObject(hdc, OBJ_FONT), sizeof(device_lf), &device_lf);
        ok( ret == sizeof(device_lf) ||
            broken(ret == (sizeof(device_lf) - LF_FACESIZE + strlen(device_lf.lfFaceName) + 1)), /* NT4 */
            "GetObjectA error %ld\n", GetLastError());

        /* compare up to lfOutPrecision, other values are not interesting,
         * and in fact sometimes arbitrary adapted by Win9x.
         */
        ok(!memcmp(&orig_lf, &device_lf, FIELD_OFFSET(LOGFONTA, lfOutPrecision)), "fonts don't match\n");
        ok(!lstrcmpA(orig_lf.lfFaceName, device_lf.lfFaceName), "font names don't match\n");

        ok(!emr_ExtTextOutW->rclBounds.left, "emr_ExtTextOutW->rclBounds.left = %ld\n",
                emr_ExtTextOutW->rclBounds.left);
        ok(emr_ExtTextOutW->rclBounds.right != -1, "emr_ExtTextOutW->rclBounds.right = %ld\n",
                emr_ExtTextOutW->rclBounds.right);
        ok(emr_ExtTextOutW->rclBounds.bottom != -1, "emr_ExtTextOutW->rclBounds.bottom = %ld\n",
                emr_ExtTextOutW->rclBounds.bottom);

        for(i = 0; i < emr_ExtTextOutW->emrtext.nChars; i++)
        {
            ok(orig_dx[i] == dx[i], "pass %d: dx[%ld] (%d) didn't match %d\n",
                                     n_record, i, dx[i], orig_dx[i]);
        }
        n_record++;
        emr_processed = TRUE;
        break;
    }

    default:
        break;
    }

    return 1;
}

static void test_ExtTextOut(void)
{
    HWND hwnd;
    HDC hdcDisplay, hdcMetafile;
    HENHMETAFILE hMetafile;
    HFONT hFont;
    static const char text[] = "Simple text to test ExtTextOut on metafiles";
    INT i, len, dx[256];
    static const RECT rc = { 0, 0, 100, 100 };
    BOOL ret;

    assert(ARRAY_SIZE(dx) >= lstrlenA(text));

    /* Win9x doesn't play EMFs on invisible windows */
    hwnd = CreateWindowExA(0, "static", NULL, WS_POPUP | WS_VISIBLE,
                           0, 0, 200, 200, 0, 0, 0, NULL);
    ok(hwnd != 0, "CreateWindowExA error %ld\n", GetLastError());

    hdcDisplay = GetDC(hwnd);
    ok(hdcDisplay != 0, "GetDC error %ld\n", GetLastError());

    SetMapMode(hdcDisplay, MM_TEXT);

    memset(&orig_lf, 0, sizeof(orig_lf));

    orig_lf.lfCharSet = ANSI_CHARSET;
    orig_lf.lfClipPrecision = CLIP_DEFAULT_PRECIS;
    orig_lf.lfWeight = FW_DONTCARE;
    orig_lf.lfHeight = 7;
    orig_lf.lfQuality = DEFAULT_QUALITY;
    lstrcpyA(orig_lf.lfFaceName, "Arial");
    hFont = CreateFontIndirectA(&orig_lf);
    ok(hFont != 0, "CreateFontIndirectA error %ld\n", GetLastError());

    hFont = SelectObject(hdcDisplay, hFont);

    len = lstrlenA(text);
    for (i = 0; i < len; i++)
    {
        ret = GetCharWidthA(hdcDisplay, text[i], text[i], &dx[i]);
        ok( ret, "GetCharWidthA error %ld\n", GetLastError());
    }
    hFont = SelectObject(hdcDisplay, hFont);

    hdcMetafile = CreateEnhMetaFileA(hdcDisplay, NULL, NULL, NULL);
    ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    ok(GetDeviceCaps(hdcMetafile, TECHNOLOGY) == DT_RASDISPLAY,
       "GetDeviceCaps(TECHNOLOGY) has to return DT_RASDISPLAY for a display based EMF\n");

    hFont = SelectObject(hdcMetafile, hFont);

    /* 1. pass NULL lpDx */
    ret = ExtTextOutA(hdcMetafile, 0, 0, 0, &rc, text, lstrlenA(text), NULL);
    ok( ret, "ExtTextOutA error %ld\n", GetLastError());

    /* 2. pass custom lpDx */
    ret = ExtTextOutA(hdcMetafile, 0, 20, 0, &rc, text, lstrlenA(text), dx);
    ok( ret, "ExtTextOutA error %ld\n", GetLastError());

    /* 3. pass NULL lprc */
    ret = ExtTextOutA(hdcMetafile, 0, 40, 0, NULL, text, lstrlenA(text), NULL);
    ok( ret, "ExtTextOutA error %ld\n", GetLastError());

    /* 4. test with unmatched BeginPath/EndPath calls */
    ret = BeginPath(hdcMetafile);
    ok( ret, "BeginPath error %ld\n", GetLastError());
    ret = BeginPath(hdcMetafile);
    ok( ret, "BeginPath error %ld\n", GetLastError());
    ret = EndPath(hdcMetafile);
    ok( ret, "BeginPath error %ld\n", GetLastError());
    ret = ExtTextOutA(hdcMetafile, 0, 60, 0, NULL, text, lstrlenA(text), NULL);
    ok( ret, "ExtTextOutA error %ld\n", GetLastError());

    hFont = SelectObject(hdcMetafile, hFont);
    ret = DeleteObject(hFont);
    ok( ret, "DeleteObject error %ld\n", GetLastError());

    hMetafile = CloseEnhMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    ok(!GetObjectType(hdcMetafile), "CloseEnhMetaFile has to destroy metafile hdc\n");

    ret = PlayEnhMetaFile(hdcDisplay, hMetafile, &rc);
    ok( ret, "PlayEnhMetaFile error %ld\n", GetLastError());

    SetTextAlign(hdcDisplay, TA_UPDATECP | TA_CENTER | TA_BASELINE | TA_RTLREADING );
    SetBkColor(hdcDisplay, RGB(0xff, 0, 0));
    SetTextColor(hdcDisplay, RGB(0, 0xff, 0));
    SetROP2(hdcDisplay, R2_NOT);
    SetArcDirection(hdcDisplay, AD_CLOCKWISE);
    SetPolyFillMode(hdcDisplay, WINDING);
    SetStretchBltMode(hdcDisplay, HALFTONE);

    if(pSetRelAbs) pSetRelAbs(hdcDisplay, RELATIVE);
    SetBkMode(hdcDisplay, OPAQUE);

    ret = EnumEnhMetaFile(hdcDisplay, hMetafile, eto_emf_enum_proc, dx, &rc);
    ok( ret, "EnumEnhMetaFile error %ld\n", GetLastError());

    ok( GetTextAlign(hdcDisplay) == (TA_UPDATECP | TA_CENTER | TA_BASELINE | TA_RTLREADING),
        "text align %08x\n", GetTextAlign(hdcDisplay));
    ok( GetBkColor(hdcDisplay) == RGB(0xff, 0, 0), "bk color %08lx\n", GetBkColor(hdcDisplay));
    ok( GetTextColor(hdcDisplay) == RGB(0, 0xff, 0), "text color %08lx\n", GetTextColor(hdcDisplay));
    ok( GetROP2(hdcDisplay) == R2_NOT, "rop2 %d\n", GetROP2(hdcDisplay));
    ok( GetArcDirection(hdcDisplay) == AD_CLOCKWISE, "arc dir  %d\n", GetArcDirection(hdcDisplay));
    ok( GetPolyFillMode(hdcDisplay) == WINDING, "poly fill %d\n", GetPolyFillMode(hdcDisplay));
    ok( GetStretchBltMode(hdcDisplay) == HALFTONE, "stretchblt mode %d\n", GetStretchBltMode(hdcDisplay));

    ok(emr_processed, "EnumEnhMetaFile couldn't find EMR_EXTTEXTOUTA or EMR_EXTTEXTOUTW record\n");

    ok(!EnumEnhMetaFile(hdcDisplay, hMetafile, eto_emf_enum_proc, dx, NULL),
       "A valid hdc has to require a valid rc\n");

    ok(EnumEnhMetaFile(NULL, hMetafile, eto_emf_enum_proc, dx, NULL),
       "A null hdc does not require a valid rc\n");

    ret = DeleteEnhMetaFile(hMetafile);
    ok( ret, "DeleteEnhMetaFile error %ld\n", GetLastError());
    ret = ReleaseDC(hwnd, hdcDisplay);
    ok( ret, "ReleaseDC error %ld\n", GetLastError());
    DestroyWindow(hwnd);
}

struct eto_scale_test_record
{
    INT graphics_mode;
    INT map_mode;
    double ex_scale;
    double ey_scale;
    BOOL processed;
};

static int CALLBACK eto_scale_enum_proc(HDC hdc, HANDLETABLE *handle_table,
    const ENHMETARECORD *emr, int n_objs, LPARAM param)
{
    struct eto_scale_test_record *test = (struct eto_scale_test_record*)param;

    if (emr->iType == EMR_EXTTEXTOUTW)
    {
        const EMREXTTEXTOUTW *pExtTextOutW = (const EMREXTTEXTOUTW *)emr;
        ok(fabs(test->ex_scale - pExtTextOutW->exScale) < 0.001,
           "Got exScale %f, expected %f\n", pExtTextOutW->exScale, test->ex_scale);
        ok(fabs(test->ey_scale - pExtTextOutW->eyScale) < 0.001,
           "Got eyScale %f, expected %f\n", pExtTextOutW->eyScale, test->ey_scale);
        test->processed = TRUE;
    }

    return 1;
}

static void test_ExtTextOutScale(void)
{
    const RECT rc = { 0, 0, 100, 100 };
    struct eto_scale_test_record test;
    HDC hdcDisplay, hdcMetafile;
    HENHMETAFILE hMetafile;
    HWND hwnd;
    SIZE wndext, vportext;
    int horzSize, vertSize, horzRes, vertRes;
    int ret;
    int i;

    hwnd = CreateWindowExA(0, "static", NULL, WS_POPUP | WS_VISIBLE,
                           0, 0, 200, 200, 0, 0, 0, NULL);
    ok(hwnd != 0, "CreateWindowExA failed\n");

    hdcDisplay = GetDC(hwnd);
    ok(hdcDisplay != 0, "GetDC failed\n");

    horzSize = GetDeviceCaps(hdcDisplay, HORZSIZE);
    horzRes  = GetDeviceCaps(hdcDisplay, HORZRES);
    vertSize = GetDeviceCaps(hdcDisplay, VERTSIZE);
    vertRes  = GetDeviceCaps(hdcDisplay, VERTRES);
    ok(horzSize && horzRes && vertSize && vertRes, "GetDeviceCaps failed\n");

    for (i = 0; i < 16; i++)
    {
        test.graphics_mode = i / 8 + 1;
        test.map_mode      = i % 8 + 1;

        ret = SetGraphicsMode(hdcDisplay, test.graphics_mode);
        ok(ret, "SetGraphicsMode failed\n");
        ret = SetMapMode(hdcDisplay, test.map_mode);
        ok(ret, "SetMapMode failed\n");

        if ((test.map_mode == MM_ISOTROPIC) || (test.map_mode == MM_ANISOTROPIC))
        {
            ret = SetWindowExtEx(hdcDisplay, 1, 1, NULL);
            ok(ret, "SetWindowExtEx failed\n");
            ret = SetViewportExtEx(hdcDisplay, -20, -10, NULL);
            ok(ret, "SetViewportExtEx failed\n");
        }

        ret = GetViewportExtEx(hdcDisplay, &vportext);
        ok(ret, "GetViewportExtEx failed\n");
        ret = GetWindowExtEx(hdcDisplay, &wndext);
        ok(ret, "GetWindowExtEx failed\n");

        if (winetest_debug > 1)
            trace("gm %d, mm %d, wnd %ld,%ld, vp %ld,%ld horz %d,%d vert %d,%d\n",
                  test.graphics_mode, test.map_mode,
                  wndext.cx, wndext.cy, vportext.cx, vportext.cy,
                  horzSize, horzRes, vertSize, vertRes);

        if (test.graphics_mode == GM_COMPATIBLE)
        {
            test.ex_scale = 100.0 * ((FLOAT)horzSize  / (FLOAT)horzRes) /
                                    ((FLOAT)wndext.cx / (FLOAT)vportext.cx);
            test.ey_scale = 100.0 * ((FLOAT)vertSize  / (FLOAT)vertRes) /
                                    ((FLOAT)wndext.cy / (FLOAT)vportext.cy);
        }
        else
        {
            test.ex_scale = 0.0;
            test.ey_scale = 0.0;
        }

        hdcMetafile = CreateEnhMetaFileA(hdcDisplay, NULL, NULL, NULL);
        ok(hdcMetafile != 0, "CreateEnhMetaFileA failed\n");

        ret = SetGraphicsMode(hdcMetafile, test.graphics_mode);
        ok(ret, "SetGraphicsMode failed\n");
        ret = SetMapMode(hdcMetafile, test.map_mode);
        ok(ret, "SetMapMode failed\n");

        if ((test.map_mode == MM_ISOTROPIC) || (test.map_mode == MM_ANISOTROPIC))
        {
            ret = SetWindowExtEx(hdcMetafile, 1, 1, NULL);
            ok(ret, "SetWindowExtEx failed\n");
            ret = SetViewportExtEx(hdcMetafile, -20, -10, NULL);
            ok(ret, "SetViewportExtEx failed\n");
        }

        ret = ExtTextOutW(hdcMetafile, 0, 0, 0, 0, L"a", 1, NULL);
        ok(ret, "ExtTextOutW failed\n");

        hMetafile = CloseEnhMetaFile(hdcMetafile);
        ok(hMetafile != 0, "CloseEnhMetaFile failed\n");

        test.processed = 0;
        ret = EnumEnhMetaFile(hdcDisplay, hMetafile, eto_scale_enum_proc, &test, &rc);
        ok(ret, "EnumEnhMetaFile failed\n");
        ok(test.processed, "EnumEnhMetaFile couldn't find EMR_EXTTEXTOUTW record\n");

        ret = DeleteEnhMetaFile(hMetafile);
        ok(ret, "DeleteEnhMetaFile failed\n");
    }

    ret = ReleaseDC(hwnd, hdcDisplay);
    ok(ret, "ReleaseDC failed\n");
    DestroyWindow(hwnd);
}


static void check_dc_state(HDC hdc, int restore_no,
                           int wnd_org_x, int wnd_org_y, int wnd_ext_x, int wnd_ext_y,
                           int vp_org_x, int vp_org_y, int vp_ext_x, int vp_ext_y)
{
    BOOL ret;
    XFORM xform;
    FLOAT xscale, yscale, edx, edy;

    SetLastError(0xdeadbeef);
    ret = GetWorldTransform(hdc, &xform);
    ok(ret, "GetWorldTransform error %lu\n", GetLastError());

    if (winetest_debug > 1)
        trace("%d: eM11 %f, eM22 %f, eDx %f, eDy %f\n", restore_no, xform.eM11,
              xform.eM22, xform.eDx, xform.eDy);

    ok(xform.eM12 == 0.0, "%d: expected eM12 0.0, got %f\n", restore_no, xform.eM12);
    ok(xform.eM21 == 0.0, "%d: expected eM21 0.0, got %f\n", restore_no, xform.eM21);

    xscale = (FLOAT)vp_ext_x / (FLOAT)wnd_ext_x;
    if (winetest_debug > 1) trace("x scale %f\n", xscale);
    ok(fabs(xscale - xform.eM11) < 0.01, "%d: vp_ext_x %d, wnd_ext_cx %d, eM11 %f\n",
       restore_no, vp_ext_x, wnd_ext_x, xform.eM11);

    yscale = (FLOAT)vp_ext_y / (FLOAT)wnd_ext_y;
    if (winetest_debug > 1) trace("y scale %f\n", yscale);
    ok(fabs(yscale - xform.eM22) < 0.01, "%d: vp_ext_y %d, wnd_ext_y %d, eM22 %f\n",
       restore_no, vp_ext_y, wnd_ext_y, xform.eM22);

    edx = (FLOAT)vp_org_x - xform.eM11 * (FLOAT)wnd_org_x;
    ok(fabs(edx - xform.eDx) < 0.01, "%d: edx %f != eDx %f\n", restore_no, edx, xform.eDx);
    edy = (FLOAT)vp_org_y - xform.eM22 * (FLOAT)wnd_org_y;
    ok(fabs(edy - xform.eDy) < 0.01, "%d: edy %f != eDy %f\n", restore_no, edy, xform.eDy);
}

static int CALLBACK savedc_emf_enum_proc(HDC hdc, HANDLETABLE *handle_table,
                                         const ENHMETARECORD *emr, int n_objs, LPARAM param)
{
    BOOL ret;
    XFORM xform;
    POINT pt;
    SIZE size;
    static int save_state;
    static int restore_no;
    static int select_no;

    if (winetest_debug > 1)
        trace("hdc %p, emr->iType %ld, emr->nSize %ld, param %p\n",
              hdc, emr->iType, emr->nSize, (void *)param);

    SetLastError(0xdeadbeef);
    ret = GetWorldTransform(hdc, &xform);
    if (!ret && GetLastError() == ERROR_CALL_NOT_IMPLEMENTED)
    {
        ret = GetWindowOrgEx(hdc, &pt);
        ok(ret, "GetWindowOrgEx error %lu\n", GetLastError());
        trace("window org (%ld,%ld)\n", pt.x, pt.y);
        ret = GetViewportOrgEx(hdc, &pt);
        ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
        trace("vport org (%ld,%ld)\n", pt.x, pt.y);
        ret = GetWindowExtEx(hdc, &size);
        ok(ret, "GetWindowExtEx error %lu\n", GetLastError());
        trace("window ext (%ld,%ld)\n", size.cx, size.cy);
        ret = GetViewportExtEx(hdc, &size);
        ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
        trace("vport ext (%ld,%ld)\n", size.cx, size.cy);
    }
    else
    {
        ok(ret, "GetWorldTransform error %lu\n", GetLastError());
        if (winetest_debug > 1)
            trace("eM11 %f, eM22 %f, eDx %f, eDy %f\n", xform.eM11, xform.eM22, xform.eDx, xform.eDy);
    }

    PlayEnhMetaFileRecord(hdc, handle_table, emr, n_objs);

    switch (emr->iType)
    {
    case EMR_HEADER:
    {
        static RECT exp_bounds = { 0, 0, 150, 150 };
        RECT bounds;
        const ENHMETAHEADER *emf = (const ENHMETAHEADER *)emr;

        if (winetest_debug > 1)
        {
            trace("bounds %ld,%ld-%ld,%ld, frame %ld,%ld-%ld,%ld\n",
                  emf->rclBounds.left, emf->rclBounds.top, emf->rclBounds.right, emf->rclBounds.bottom,
                  emf->rclFrame.left, emf->rclFrame.top, emf->rclFrame.right, emf->rclFrame.bottom);
            trace("mm %ld x %ld, device %ld x %ld\n", emf->szlMillimeters.cx, emf->szlMillimeters.cy,
                  emf->szlDevice.cx, emf->szlDevice.cy);
        }

        SetRect(&bounds, emf->rclBounds.left, emf->rclBounds.top, emf->rclBounds.right, emf->rclBounds.bottom);
        ok(EqualRect(&bounds, &exp_bounds), "wrong bounds\n");

        save_state = 0;
        restore_no = 0;
        select_no = 0;
        check_dc_state(hdc, restore_no, 0, 0, 1, 1, 0, 0, 1, 1);
        break;
    }

    case EMR_LINETO:
        {
            const EMRLINETO *line = (const EMRLINETO *)emr;
            if (winetest_debug > 1) trace("EMR_LINETO %ld,%ld\n", line->ptl.x, line->ptl.x);
            break;
        }
    case EMR_SETWINDOWORGEX:
        {
            const EMRSETWINDOWORGEX *org = (const EMRSETWINDOWORGEX *)emr;
            if (winetest_debug > 1)
                trace("EMR_SETWINDOWORGEX: %ld,%ld\n", org->ptlOrigin.x, org->ptlOrigin.y);
            break;
        }
    case EMR_SETWINDOWEXTEX:
        {
            const EMRSETWINDOWEXTEX *ext = (const EMRSETWINDOWEXTEX *)emr;
            if (winetest_debug > 1)
                trace("EMR_SETWINDOWEXTEX: %ld,%ld\n", ext->szlExtent.cx, ext->szlExtent.cy);
            break;
        }
    case EMR_SETVIEWPORTORGEX:
        {
            const EMRSETVIEWPORTORGEX *org = (const EMRSETVIEWPORTORGEX *)emr;
            if (winetest_debug > 1)
                trace("EMR_SETVIEWPORTORGEX: %ld,%ld\n", org->ptlOrigin.x, org->ptlOrigin.y);
            break;
        }
    case EMR_SETVIEWPORTEXTEX:
        {
            const EMRSETVIEWPORTEXTEX *ext = (const EMRSETVIEWPORTEXTEX *)emr;
            if (winetest_debug > 1)
                trace("EMR_SETVIEWPORTEXTEX: %ld,%ld\n", ext->szlExtent.cx, ext->szlExtent.cy);
            break;
        }
    case EMR_SAVEDC:
        save_state++;
        if (winetest_debug > 1) trace("EMR_SAVEDC\n");
        break;

    case EMR_RESTOREDC:
        {
            const EMRRESTOREDC *restoredc = (const EMRRESTOREDC *)emr;
            if (winetest_debug > 1) trace("EMR_RESTOREDC: %ld\n", restoredc->iRelative);

            switch(++restore_no)
            {
            case 1:
                ok(restoredc->iRelative == -1, "first restore %ld\n", restoredc->iRelative);
                check_dc_state(hdc, restore_no, -2, -2, 8192, 8192, 20, 20, 20479, 20478);
                break;
            case 2:
                ok(restoredc->iRelative == -3, "second restore %ld\n", restoredc->iRelative);
                check_dc_state(hdc, restore_no, 0, 0, 16384, 16384, 0, 0, 17873, 17872);
                break;
            case 3:
                ok(restoredc->iRelative == -2, "third restore %ld\n", restoredc->iRelative);
                check_dc_state(hdc, restore_no, -4, -4, 32767, 32767, 40, 40, 3276, 3276);
                break;
            }
            ok(restore_no <= 3, "restore_no %d\n", restore_no);
            save_state += restoredc->iRelative;
            break;
        }
    case EMR_SELECTOBJECT:
        {
            const EMRSELECTOBJECT *selectobj = (const EMRSELECTOBJECT*)emr;
            if (winetest_debug > 1) trace("EMR_SELECTOBJECT: %lx\n",selectobj->ihObject);
            select_no ++;
            break;
        }
    case EMR_EOF:
        ok(save_state == 0, "EOF save_state %d\n", save_state);
        ok(select_no == 3, "Too many/few selects  %i\n",select_no);
        break;
    }

    SetLastError(0xdeadbeef);
    ret = GetWorldTransform(hdc, &xform);
    if (!ret && GetLastError() == ERROR_CALL_NOT_IMPLEMENTED)
    {
        ret = GetWindowOrgEx(hdc, &pt);
        ok(ret, "GetWindowOrgEx error %lu\n", GetLastError());
        if (winetest_debug > 1) trace("window org (%ld,%ld)\n", pt.x, pt.y);
        ret = GetViewportOrgEx(hdc, &pt);
        ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
        if (winetest_debug > 1) trace("vport org (%ld,%ld)\n", pt.x, pt.y);
        ret = GetWindowExtEx(hdc, &size);
        ok(ret, "GetWindowExtEx error %lu\n", GetLastError());
        if (winetest_debug > 1) trace("window ext (%ld,%ld)\n", size.cx, size.cy);
        ret = GetViewportExtEx(hdc, &size);
        ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
        if (winetest_debug > 1) trace("vport ext (%ld,%ld)\n", size.cx, size.cy);
    }
    else
    {
        ok(ret, "GetWorldTransform error %lu\n", GetLastError());
        if (winetest_debug > 1)
            trace("eM11 %f, eM22 %f, eDx %f, eDy %f\n", xform.eM11, xform.eM22, xform.eDx, xform.eDy);
    }

    return 1;
}

static void test_SaveDC(void)
{
    HDC hdcMetafile, hdcDisplay;
    HENHMETAFILE hMetafile;
    HWND hwnd;
    int ret;
    POINT pt;
    SIZE size;
    HFONT hFont,hFont2,hFontOld,hFontCheck;
    static const RECT rc = { 0, 0, 150, 150 };

    /* Win9x doesn't play EMFs on invisible windows */
    hwnd = CreateWindowExA(0, "static", NULL, WS_POPUP | WS_VISIBLE,
                           0, 0, 200, 200, 0, 0, 0, NULL);
    ok(hwnd != 0, "CreateWindowExA error %ld\n", GetLastError());

    hdcDisplay = GetDC(hwnd);
    ok(hdcDisplay != 0, "GetDC error %ld\n", GetLastError());

    hdcMetafile = CreateEnhMetaFileA(hdcDisplay, NULL, NULL, NULL);
    ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    SetMapMode(hdcMetafile, MM_ANISOTROPIC);

    /* Need to write something to the emf, otherwise Windows won't play it back */
    LineTo(hdcMetafile, 150, 150);

    SetWindowOrgEx(hdcMetafile, 0, 0, NULL);
    SetViewportOrgEx(hdcMetafile, 0, 0, NULL);
    SetWindowExtEx(hdcMetafile, 110, 110, NULL );
    SetViewportExtEx(hdcMetafile, 120, 120, NULL );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 0,"Expecting ViewportOrg x of 0, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 120,"Expecting ViewportExt cx of 120, got %li\n",size.cx);
    ret = SaveDC(hdcMetafile);
    ok(ret == 1, "ret = %d\n", ret);

    SetWindowOrgEx(hdcMetafile, -1, -1, NULL);
    SetViewportOrgEx(hdcMetafile, 10, 10, NULL);
    SetWindowExtEx(hdcMetafile, 150, 150, NULL );
    SetViewportExtEx(hdcMetafile, 200, 200, NULL );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 10,"Expecting ViewportOrg x of 10, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 200,"Expecting ViewportExt cx of 200, got %li\n",size.cx);
    ret = SaveDC(hdcMetafile);
    ok(ret == 2, "ret = %d\n", ret);

    SetWindowOrgEx(hdcMetafile, -2, -2, NULL);
    SetViewportOrgEx(hdcMetafile, 20, 20, NULL);
    SetWindowExtEx(hdcMetafile, 120, 120, NULL );
    SetViewportExtEx(hdcMetafile, 300, 300, NULL );
    SetPolyFillMode( hdcMetafile, ALTERNATE );
    SetBkColor( hdcMetafile, 0 );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 20,"Expecting ViewportOrg x of 20, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 300,"Expecting ViewportExt cx of 300, got %li\n",size.cx);
    ret = SaveDC(hdcMetafile);
    ok(ret == 3, "ret = %d\n", ret);

    SetWindowOrgEx(hdcMetafile, -3, -3, NULL);
    SetViewportOrgEx(hdcMetafile, 30, 30, NULL);
    SetWindowExtEx(hdcMetafile, 200, 200, NULL );
    SetViewportExtEx(hdcMetafile, 400, 400, NULL );

    SetPolyFillMode( hdcMetafile, WINDING );
    SetBkColor( hdcMetafile, 0x123456 );
    ok( GetPolyFillMode( hdcMetafile ) == WINDING, "PolyFillMode not restored\n" );
    ok( GetBkColor( hdcMetafile ) == 0x123456, "Background color not restored\n" );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 30,"Expecting ViewportOrg x of 30, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 400,"Expecting ViewportExt cx of 400, got %li\n",size.cx);
    ret = RestoreDC(hdcMetafile, -1);
    ok(ret, "ret = %d\n", ret);

    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 20,"Expecting ViewportOrg x of 20, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 300,"Expecting ViewportExt cx of 300, got %li\n",size.cx);
    ok( GetPolyFillMode( hdcMetafile ) == ALTERNATE, "PolyFillMode not restored\n" );
    ok( GetBkColor( hdcMetafile ) == 0, "Background color not restored\n" );
    ret = SaveDC(hdcMetafile);
    ok(ret == 3, "ret = %d\n", ret);

    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 20,"Expecting ViewportOrg x of 20, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 300,"Expecting ViewportExt cx of 300, got %li\n",size.cx);
    ret = RestoreDC(hdcMetafile, 1);
    ok(ret, "ret = %d\n", ret);
    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 0,"Expecting ViewportOrg x of 0, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 120,"Expecting ViewportExt cx of 120, got %li\n",size.cx);

    SetWindowOrgEx(hdcMetafile, -4, -4, NULL);
    SetViewportOrgEx(hdcMetafile, 40, 40, NULL);
    SetWindowExtEx(hdcMetafile, 500, 500, NULL );
    SetViewportExtEx(hdcMetafile, 50, 50, NULL );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 40,"Expecting ViewportOrg x of 40, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 50,"Expecting ViewportExt cx of 50, got %li\n",size.cx);
    ret = SaveDC(hdcMetafile);
    ok(ret == 1, "ret = %d\n", ret);

    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 40,"Expecting ViewportOrg x of 40, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 50,"Expecting ViewportExt cx of 50, got %li\n",size.cx);
    ret = SaveDC(hdcMetafile);
    ok(ret == 2, "ret = %d\n", ret);

    memset(&orig_lf, 0, sizeof(orig_lf));
    orig_lf.lfCharSet = ANSI_CHARSET;
    orig_lf.lfClipPrecision = CLIP_DEFAULT_PRECIS;
    orig_lf.lfWeight = FW_DONTCARE;
    orig_lf.lfHeight = 7;
    orig_lf.lfQuality = DEFAULT_QUALITY;
    lstrcpyA(orig_lf.lfFaceName, "Arial");
    hFont = CreateFontIndirectA(&orig_lf);
    ok(hFont != 0, "CreateFontIndirectA error %ld\n", GetLastError());

    hFontOld = SelectObject(hdcMetafile, hFont);

    hFont2 = CreateFontIndirectA(&orig_lf);
    ok(hFont2 != 0, "CreateFontIndirectA error %ld\n", GetLastError());
    hFontCheck = SelectObject(hdcMetafile, hFont2);
    ok(hFontCheck == hFont, "Font not selected\n");

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = RestoreDC(hdcMetafile, 1);
    ok(ret, "ret = %d\n", ret);
    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(ret, "GetViewportOrgEx error %lu\n", GetLastError());
    ok(pt.x == 40,"Expecting ViewportOrg x of 40, got %li\n",pt.x);
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(ret, "GetViewportExtEx error %lu\n", GetLastError());
    ok(size.cx == 50,"Expecting ViewportExt cx of 50, got %li\n",size.cx);

    hFontCheck = SelectObject(hdcMetafile, hFontOld);
    ok(hFontOld == hFontCheck && hFontCheck != hFont && hFontCheck != hFont2,
       "Font not reverted with DC Restore\n");

    ret = RestoreDC(hdcMetafile, -20);
    ok(!ret, "ret = %d\n", ret);
    ret = RestoreDC(hdcMetafile, 20);
    ok(!ret, "ret = %d\n", ret);

    hMetafile = CloseEnhMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    ret = EnumEnhMetaFile(hdcDisplay, hMetafile, savedc_emf_enum_proc, 0, &rc);
    ok( ret == 1, "EnumEnhMetaFile rets %d\n", ret);

    ret = DeleteObject(hFont);
    ok( ret, "DeleteObject error %ld\n", GetLastError());
    ret = DeleteObject(hFont2);
    ok( ret, "DeleteObject error %ld\n", GetLastError());
    ret = DeleteEnhMetaFile(hMetafile);
    ok( ret, "DeleteEnhMetaFile error %ld\n", GetLastError());
    ret = ReleaseDC(hwnd, hdcDisplay);
    ok( ret, "ReleaseDC error %ld\n", GetLastError());
    DestroyWindow(hwnd);
}

static void test_mf_SaveDC(void)
{
    HDC hdcMetafile;
    HMETAFILE hMetafile;
    int ret;
    POINT pt;
    SIZE size;
    HFONT hFont,hFont2,hFontOld,hFontCheck;

    hdcMetafile = CreateMetaFileA(NULL);
    ok(hdcMetafile != 0, "CreateMetaFileA error %ld\n", GetLastError());

    ret = SetMapMode(hdcMetafile, MM_ANISOTROPIC);
    ok (ret, "SetMapMode should not fail\n");

    /* Need to write something to the emf, otherwise Windows won't play it back */
    LineTo(hdcMetafile, 150, 150);

    pt.x = pt.y = 5555;
    SetWindowOrgEx(hdcMetafile, 0, 0, &pt);
    ok( pt.x == 5555 && pt.y == 5555, "wrong origin %ld,%ld\n", pt.x, pt.y);
    pt.x = pt.y = 5555;
    SetViewportOrgEx(hdcMetafile, 0, 0, &pt);
    ok( pt.x == 5555 && pt.y == 5555, "wrong origin %ld,%ld\n", pt.x, pt.y);
    size.cx = size.cy = 5555;
    SetWindowExtEx(hdcMetafile, 110, 110, &size );
    ok( size.cx == 5555 && size.cy == 5555, "wrong size %ld,%ld\n", size.cx, size.cy );
    size.cx = size.cy = 5555;
    SetViewportExtEx(hdcMetafile, 120, 120, &size );
    ok( size.cx == 5555 && size.cy == 5555, "wrong size %ld,%ld\n", size.cx, size.cy );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = GetViewportOrgEx(hdcMetafile, &pt);
    ok(!ret, "GetViewportOrgEx should fail\n");
    ret = GetViewportExtEx(hdcMetafile, &size);
    ok(!ret, "GetViewportExtEx should fail\n");
    ret = SaveDC(hdcMetafile);
    ok(ret == 1, "ret = %d\n", ret);

    SetWindowOrgEx(hdcMetafile, -1, -1, NULL);
    SetViewportOrgEx(hdcMetafile, 10, 10, NULL);
    SetWindowExtEx(hdcMetafile, 150, 150, NULL );
    SetViewportExtEx(hdcMetafile, 200, 200, NULL );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = SaveDC(hdcMetafile);
    ok(ret == 1, "ret = %d\n", ret);

    SetWindowOrgEx(hdcMetafile, -2, -2, NULL);
    SetViewportOrgEx(hdcMetafile, 20, 20, NULL);
    SetWindowExtEx(hdcMetafile, 120, 120, NULL );
    SetViewportExtEx(hdcMetafile, 300, 300, NULL );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);
    SetPolyFillMode( hdcMetafile, ALTERNATE );
    SetBkColor( hdcMetafile, 0 );

    ret = SaveDC(hdcMetafile);
    ok(ret == 1, "ret = %d\n", ret);

    SetWindowOrgEx(hdcMetafile, -3, -3, NULL);
    SetViewportOrgEx(hdcMetafile, 30, 30, NULL);
    SetWindowExtEx(hdcMetafile, 200, 200, NULL );
    SetViewportExtEx(hdcMetafile, 400, 400, NULL );

    SetPolyFillMode( hdcMetafile, WINDING );
    SetBkColor( hdcMetafile, 0x123456 );
    ok( !GetPolyFillMode( hdcMetafile ), "GetPolyFillMode succeeded\n" );
    ok( GetBkColor( hdcMetafile ) == CLR_INVALID, "GetBkColor succeeded\n" );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = RestoreDC(hdcMetafile, -1);
    ok(ret, "ret = %d\n", ret);

    ret = SaveDC(hdcMetafile);
    ok(ret == 1, "ret = %d\n", ret);

    ret = RestoreDC(hdcMetafile, 1);
    ok(ret, "ret = %d\n", ret);

    SetWindowOrgEx(hdcMetafile, -4, -4, NULL);
    SetViewportOrgEx(hdcMetafile, 40, 40, NULL);
    SetWindowExtEx(hdcMetafile, 500, 500, NULL );
    SetViewportExtEx(hdcMetafile, 50, 50, NULL );

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = SaveDC(hdcMetafile);
    ok(ret == 1, "ret = %d\n", ret);

    ret = SaveDC(hdcMetafile);
    ok(ret == 1, "ret = %d\n", ret);

    memset(&orig_lf, 0, sizeof(orig_lf));
    orig_lf.lfCharSet = ANSI_CHARSET;
    orig_lf.lfClipPrecision = CLIP_DEFAULT_PRECIS;
    orig_lf.lfWeight = FW_DONTCARE;
    orig_lf.lfHeight = 7;
    orig_lf.lfQuality = DEFAULT_QUALITY;
    lstrcpyA(orig_lf.lfFaceName, "Arial");
    hFont = CreateFontIndirectA(&orig_lf);
    ok(hFont != 0, "CreateFontIndirectA error %ld\n", GetLastError());

    hFontOld = SelectObject(hdcMetafile, hFont);

    hFont2 = CreateFontIndirectA(&orig_lf);
    ok(hFont2 != 0, "CreateFontIndirectA error %ld\n", GetLastError());
    hFontCheck = SelectObject(hdcMetafile, hFont2);
    ok(hFontCheck == hFont, "Font not selected\n");

    /* Force Win9x to update DC state */
    SetPixelV(hdcMetafile, 50, 50, 0);

    ret = RestoreDC(hdcMetafile, 1);
    ok(ret, "ret = %d\n", ret);

    hFontCheck = SelectObject(hdcMetafile, hFontOld);
    ok(hFontOld != hFontCheck && hFontCheck == hFont2, "Font incorrectly reverted with DC Restore\n");

    /* restore level is ignored */
    ret = RestoreDC(hdcMetafile, -20);
    ok(ret, "ret = %d\n", ret);
    ret = RestoreDC(hdcMetafile, 20);
    ok(ret, "ret = %d\n", ret);
    ret = RestoreDC(hdcMetafile, 0);
    ok(ret, "ret = %d\n", ret);

    hMetafile = CloseMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseMetaFile error %ld\n", GetLastError());

    ret = DeleteMetaFile(hMetafile);
    ok( ret, "DeleteMetaFile error %ld\n", GetLastError());
    ret = DeleteObject(hFont);
    ok( ret, "DeleteObject error %ld\n", GetLastError());
    ret = DeleteObject(hFont2);
    ok( ret, "DeleteObject error %ld\n", GetLastError());
}


/* Win-format metafile (mfdrv) tests */
/* These tests compare the generated metafiles byte-by-byte */
/* with the nominal results. */

/* Maximum size of sample metafiles in bytes. */
#define MF_BUFSIZE 2048

/* 8x8 bitmap data for a pattern brush */
static const unsigned char SAMPLE_PATTERN_BRUSH[] = {
    0x01, 0x00, 0x02, 0x00,
    0x03, 0x00, 0x04, 0x00,
    0x05, 0x00, 0x06, 0x00,
    0x07, 0x00, 0x08, 0x00
};

/* Sample metafiles to be compared to the outputs of the
 * test functions.
 */

static const unsigned char MF_BLANK_BITS[] = {
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x0c, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00,
};

static const unsigned char MF_GRAPHICS_BITS[] = {
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x22, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x14, 0x02,
    0x01, 0x00, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x13, 0x02, 0x02, 0x00, 0x02, 0x00, 0x05, 0x00,
    0x00, 0x00, 0x14, 0x02, 0x01, 0x00, 0x01, 0x00,
    0x07, 0x00, 0x00, 0x00, 0x18, 0x04, 0x02, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00
};

static const unsigned char MF_PATTERN_BRUSH_BITS[] = {
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x9e, 0x00,
    0x00, 0x00, 0x02, 0x00, 0x5d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x2d, 0x00, 0x00, 0x00, 0x42, 0x01,
    0x03, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x2d, 0x01, 0x00, 0x00, 0x5d, 0x00, 0x00, 0x00,
    0x42, 0x01, 0x05, 0x00, 0x00, 0x00, 0x28, 0x00,
    0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x20, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00,
    0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x06, 0x00,
    0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x0a, 0x00,
    0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x0c, 0x00,
    0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x0e, 0x00,
    0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x10, 0x00,
    0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x12, 0x00,
    0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x14, 0x00,
    0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x16, 0x00,
    0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x1a, 0x00,
    0x00, 0x00, 0x1b, 0x00, 0x00, 0x00, 0x1c, 0x00,
    0x00, 0x00, 0x1d, 0x00, 0x00, 0x00, 0x1e, 0x00,
    0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x20, 0x00,
    0x00, 0x00, 0x21, 0x00, 0x00, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x2d, 0x01, 0x01, 0x00, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00
};

static const unsigned char MF_DCBRUSH_BITS[] =
{
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x2a, 0x00,
    0x00, 0x00, 0x02, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0xfc, 0x02,
    0x00, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x2d, 0x01, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0xfa, 0x02, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x2d, 0x01, 0x01, 0x00,
    0x07, 0x00, 0x00, 0x00, 0x1b, 0x04, 0x14, 0x00,
    0x14, 0x00, 0x0a, 0x00, 0x0a, 0x00, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00
};

static const unsigned char MF_TEXTOUT_ON_PATH_BITS[] =
{
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x19, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x32, 0x0a,
    0x16, 0x00, 0x0b, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x54, 0x65, 0x73, 0x74, 0x03, 0x00, 0x05, 0x00,
    0x08, 0x00, 0x0c, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00
};

static const unsigned char EMF_TEXTOUT_ON_PATH_BITS[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xe7, 0xff, 0xff, 0xff, 0xe9, 0xff, 0xff, 0xff,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xf4, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x40, 0x01, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xe2, 0x04, 0x00,
    0x80, 0xa9, 0x03, 0x00, 0x3b, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xc8, 0x41, 0x00, 0x80, 0xbb, 0x41,
    0x0b, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x54, 0x00, 0x00, 0x00,
    0x54, 0x00, 0x65, 0x00, 0x73, 0x00, 0x74, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x3c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_TEXTOUT_OUTLINE_ON_PATH_BITS[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xdd, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x90, 0x06, 0x00, 0x00, 0x1a, 0x04, 0x00, 0x00,
    0x51, 0x02, 0x00, 0x00, 0x72, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x1a, 0x0b, 0x09, 0x00,
    0xf0, 0xa6, 0x05, 0x00, 0x25, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x80,
    0x3b, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x54, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x01, 0x00, 0x00, 0x00, 0xc3, 0x30, 0x0d, 0x42,
    0xcf, 0xf3, 0x0c, 0x42, 0x0b, 0x00, 0x00, 0x00,
    0x16, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x4c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x54, 0x00, 0x00, 0x00, 0x54, 0x00, 0x65, 0x00,
    0x73, 0x00, 0x74, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x80,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char MF_LINETO_BITS[] = {
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x11, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x13, 0x02,
    0x0f, 0x00, 0x37, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00
};

static const unsigned char EMF_LINETO_BITS[] = {
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x37, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x61, 0x06, 0x00, 0x00, 0xb7, 0x01, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x38, 0x01, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x7c, 0x01, 0x00, 0x00, 0x2c, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xcc, 0x05, 0x00,
    0xe0, 0x93, 0x04, 0x00, 0x46, 0x00, 0x00, 0x00,
    0x48, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00,
    0x47, 0x44, 0x49, 0x43, 0x01, 0x00, 0x00, 0x80,
    0x00, 0x03, 0x00, 0x00, 0x60, 0xe5, 0xf4, 0x73,
    0x00, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x11, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x13, 0x02,
    0x0f, 0x00, 0x37, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x0b, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x09, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x36, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x37, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x80, 0x25, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
    0x30, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x80, 0x4b, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_LINETO_MM_ANISOTROPIC_BITS[] = {
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x37, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x38, 0x01, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x7c, 0x01, 0x00, 0x00, 0x2c, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xcc, 0x05, 0x00,
    0xe0, 0x93, 0x04, 0x00, 0x46, 0x00, 0x00, 0x00,
    0x48, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00,
    0x47, 0x44, 0x49, 0x43, 0x01, 0x00, 0x00, 0x80,
    0x00, 0x03, 0x00, 0x00, 0xa4, 0xfe, 0xf4, 0x73,
    0x00, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x11, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x13, 0x02,
    0x0f, 0x00, 0x37, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x0b, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x09, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x36, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x37, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x80, 0x25, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
    0x30, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x80, 0x4b, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_LINETO_MM_TEXT_BITS[] = {
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x37, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x61, 0x06, 0x00, 0x00, 0xb7, 0x01, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xe4, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x7c, 0x01, 0x00, 0x00, 0x2c, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xcc, 0x05, 0x00,
    0xe0, 0x93, 0x04, 0x00, 0x0b, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x80,
    0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x80, 0x30, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x80,
    0x4b, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_DCBRUSH_BITS[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x13, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
    0x39, 0x01, 0x00, 0x00, 0x39, 0x01, 0x00, 0x00,
    0x52, 0x02, 0x00, 0x00, 0x52, 0x02, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x44, 0x01, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x40, 0x01, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xe2, 0x04, 0x00,
    0x80, 0xa9, 0x03, 0x00, 0x25, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x80,
    0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x13, 0x00, 0x00, 0x80, 0x27, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x55, 0x55, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x26, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x33, 0x44, 0x55, 0x00, 0x25, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x2b, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x13, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x12, 0x34, 0x56, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_BEZIER_BITS[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x80, 0x00, 0x00, 0x01, 0x80, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x1a, 0x2a, 0x0d, 0x00, 0x1a, 0x2f, 0x0d, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x44, 0x01, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x51, 0x01, 0x00, 0x00, 0x0e, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x24, 0x05, 0x00,
    0xb0, 0x1e, 0x04, 0x00, 0x58, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x0a, 0x00, 0x14, 0x00, 0x14, 0x00,
    0x0f, 0x00, 0x0f, 0x00, 0x55, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00,
    0x19, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x0a, 0x00, 0x14, 0x00, 0x14, 0x00,
    0x0f, 0x00, 0x0f, 0x00, 0x19, 0x00, 0x19, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x01, 0x80, 0x00, 0x00, 0x01, 0x80, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x80, 0x00, 0x00,
    0x01, 0x80, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00,
    0x19, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x01, 0x80, 0x00, 0x00,
    0x01, 0x80, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x80, 0x00, 0x00, 0x01, 0x80, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_POLYPOLYLINE_BITS[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x90, 0x00, 0x00, 0xf4, 0x01, 0x00, 0x00,
    0x61, 0x01, 0x00, 0x00, 0xc2, 0x02, 0x00, 0x00,
    0x7a, 0xd4, 0x13, 0x00, 0xe8, 0x44, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x84, 0x01, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xa1, 0x05, 0x00, 0x00, 0x47, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x28, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xc0, 0xc1, 0x07, 0x00,
    0x2c, 0x84, 0x04, 0x00, 0x5a, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x02, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x14, 0x00,
    0x5a, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x14, 0x00, 0x64, 0x00, 0xc8, 0x00,
    0x07, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00,
    0x00, 0x90, 0x00, 0x00, 0x2c, 0x01, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x90, 0x00, 0x00, 0xf4, 0x01, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00,
    0x00, 0x90, 0x00, 0x00, 0x2c, 0x01, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0xf4, 0x01, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_GRADIENTFILL_BITS[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x2b, 0x01, 0x00, 0x00, 0x35, 0x01, 0x00, 0x00,
    0x23, 0x00, 0x00, 0x00, 0x61, 0x01, 0x00, 0x00,
    0x31, 0x29, 0x00, 0x00, 0xa3, 0x2a, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x40, 0x05, 0x00, 0x00, 0x46, 0x03, 0x00, 0x00,
    0xda, 0x01, 0x00, 0x00, 0x28, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x15, 0x3c, 0x07, 0x00,
    0xcb, 0x82, 0x04, 0x00, 0x76, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x2b, 0x01, 0x00, 0x00,
    0x35, 0x01, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x00, 0xff, 0x00, 0x80, 0x00, 0x00, 0x01, 0x80,
    0xc8, 0x00, 0x00, 0x00, 0xd2, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff,
    0xb4, 0x00, 0x00, 0x00, 0xbe, 0x00, 0x00, 0x00,
    0x34, 0x12, 0x78, 0x56, 0xbc, 0x9a, 0xf0, 0xde,
    0x2c, 0x01, 0x00, 0x00, 0x36, 0x01, 0x00, 0x00,
    0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x9a, 0x01, 0x00, 0x00,
    0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char emf_pattern_brush_bits[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xe6, 0xff, 0xff, 0xff, 0xe5, 0xff, 0xff, 0xff,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xd8, 0x01, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x77, 0x07, 0x00, 0x00, 0x3b, 0x04, 0x00, 0x00,
    0xfa, 0x01, 0x00, 0x00, 0x1f, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x12, 0xb7, 0x07, 0x00,
    0x4f, 0x5f, 0x04, 0x00, 0x5d, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x06, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x00,
    0xd4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00,
    0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
    0x28, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x06, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00,
    0x12, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00,
    0x16, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00,
    0x1a, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00,
    0x1c, 0x00, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x00,
    0x1e, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00,
    0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char mf_palette_brush_bits[] =
{
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0xb2, 0x03,
    0x00, 0x00, 0x02, 0x00, 0x05, 0x02, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x02, 0x00, 0x00, 0xf7, 0x00,
    0x00, 0x03, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x02, 0x02, 0x00, 0x04, 0x04, 0x04, 0x00,
    0x06, 0x06, 0x06, 0x00, 0x08, 0x08, 0x08, 0x00,
    0x0a, 0x0a, 0x0a, 0x00, 0x0c, 0x0c, 0x0c, 0x00,
    0x0e, 0x0e, 0x0e, 0x00, 0x10, 0x10, 0x10, 0x00,
    0x12, 0x12, 0x12, 0x00, 0x14, 0x14, 0x14, 0x00,
    0x16, 0x16, 0x16, 0x00, 0x18, 0x18, 0x18, 0x00,
    0x1a, 0x1a, 0x1a, 0x00, 0x1c, 0x1c, 0x1c, 0x00,
    0x1e, 0x1e, 0x1e, 0x00, 0x20, 0x20, 0x20, 0x00,
    0x22, 0x22, 0x22, 0x00, 0x24, 0x24, 0x24, 0x00,
    0x26, 0x26, 0x26, 0x00, 0x28, 0x28, 0x28, 0x00,
    0x2a, 0x2a, 0x2a, 0x00, 0x2c, 0x2c, 0x2c, 0x00,
    0x2e, 0x2e, 0x2e, 0x00, 0x30, 0x30, 0x30, 0x00,
    0x32, 0x32, 0x32, 0x00, 0x34, 0x34, 0x34, 0x00,
    0x36, 0x36, 0x36, 0x00, 0x38, 0x38, 0x38, 0x00,
    0x3a, 0x3a, 0x3a, 0x00, 0x3c, 0x3c, 0x3c, 0x00,
    0x3e, 0x3e, 0x3e, 0x00, 0x40, 0x40, 0x40, 0x00,
    0x42, 0x42, 0x42, 0x00, 0x44, 0x44, 0x44, 0x00,
    0x46, 0x46, 0x46, 0x00, 0x48, 0x48, 0x48, 0x00,
    0x4a, 0x4a, 0x4a, 0x00, 0x4c, 0x4c, 0x4c, 0x00,
    0x4e, 0x4e, 0x4e, 0x00, 0x50, 0x50, 0x50, 0x00,
    0x52, 0x52, 0x52, 0x00, 0x54, 0x54, 0x54, 0x00,
    0x56, 0x56, 0x56, 0x00, 0x58, 0x58, 0x58, 0x00,
    0x5a, 0x5a, 0x5a, 0x00, 0x5c, 0x5c, 0x5c, 0x00,
    0x5e, 0x5e, 0x5e, 0x00, 0x60, 0x60, 0x60, 0x00,
    0x62, 0x62, 0x62, 0x00, 0x64, 0x64, 0x64, 0x00,
    0x66, 0x66, 0x66, 0x00, 0x68, 0x68, 0x68, 0x00,
    0x6a, 0x6a, 0x6a, 0x00, 0x6c, 0x6c, 0x6c, 0x00,
    0x6e, 0x6e, 0x6e, 0x00, 0x70, 0x70, 0x70, 0x00,
    0x72, 0x72, 0x72, 0x00, 0x74, 0x74, 0x74, 0x00,
    0x76, 0x76, 0x76, 0x00, 0x78, 0x78, 0x78, 0x00,
    0x7a, 0x7a, 0x7a, 0x00, 0x7c, 0x7c, 0x7c, 0x00,
    0x7e, 0x7e, 0x7e, 0x00, 0x80, 0x80, 0x80, 0x00,
    0x82, 0x82, 0x82, 0x00, 0x84, 0x84, 0x84, 0x00,
    0x86, 0x86, 0x86, 0x00, 0x88, 0x88, 0x88, 0x00,
    0x8a, 0x8a, 0x8a, 0x00, 0x8c, 0x8c, 0x8c, 0x00,
    0x8e, 0x8e, 0x8e, 0x00, 0x90, 0x90, 0x90, 0x00,
    0x92, 0x92, 0x92, 0x00, 0x94, 0x94, 0x94, 0x00,
    0x96, 0x96, 0x96, 0x00, 0x98, 0x98, 0x98, 0x00,
    0x9a, 0x9a, 0x9a, 0x00, 0x9c, 0x9c, 0x9c, 0x00,
    0x9e, 0x9e, 0x9e, 0x00, 0xa0, 0xa0, 0xa0, 0x00,
    0xa2, 0xa2, 0xa2, 0x00, 0xa4, 0xa4, 0xa4, 0x00,
    0xa6, 0xa6, 0xa6, 0x00, 0xa8, 0xa8, 0xa8, 0x00,
    0xaa, 0xaa, 0xaa, 0x00, 0xac, 0xac, 0xac, 0x00,
    0xae, 0xae, 0xae, 0x00, 0xb0, 0xb0, 0xb0, 0x00,
    0xb2, 0xb2, 0xb2, 0x00, 0xb4, 0xb4, 0xb4, 0x00,
    0xb6, 0xb6, 0xb6, 0x00, 0xb8, 0xb8, 0xb8, 0x00,
    0xba, 0xba, 0xba, 0x00, 0xbc, 0xbc, 0xbc, 0x00,
    0xbe, 0xbe, 0xbe, 0x00, 0xc0, 0xc0, 0xc0, 0x00,
    0xc2, 0xc2, 0xc2, 0x00, 0xc4, 0xc4, 0xc4, 0x00,
    0xc6, 0xc6, 0xc6, 0x00, 0xc8, 0xc8, 0xc8, 0x00,
    0xca, 0xca, 0xca, 0x00, 0xcc, 0xcc, 0xcc, 0x00,
    0xce, 0xce, 0xce, 0x00, 0xd0, 0xd0, 0xd0, 0x00,
    0xd2, 0xd2, 0xd2, 0x00, 0xd4, 0xd4, 0xd4, 0x00,
    0xd6, 0xd6, 0xd6, 0x00, 0xd8, 0xd8, 0xd8, 0x00,
    0xda, 0xda, 0xda, 0x00, 0xdc, 0xdc, 0xdc, 0x00,
    0xde, 0xde, 0xde, 0x00, 0xe0, 0xe0, 0xe0, 0x00,
    0xe2, 0xe2, 0xe2, 0x00, 0xe4, 0xe4, 0xe4, 0x00,
    0xe6, 0xe6, 0xe6, 0x00, 0xe8, 0xe8, 0xe8, 0x00,
    0xea, 0xea, 0xea, 0x00, 0xec, 0xec, 0xec, 0x00,
    0xee, 0xee, 0xee, 0x00, 0xf0, 0xf0, 0xf0, 0x00,
    0xf2, 0xf2, 0xf2, 0x00, 0xf4, 0xf4, 0xf4, 0x00,
    0xf6, 0xf6, 0xf6, 0x00, 0xf8, 0xf8, 0xf8, 0x00,
    0xfa, 0xfa, 0xfa, 0x00, 0xfc, 0xfc, 0xfc, 0x00,
    0xfe, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x02, 0x02, 0x00, 0x04, 0x04, 0x04, 0x00,
    0x06, 0x06, 0x06, 0x00, 0x08, 0x08, 0x08, 0x00,
    0x0a, 0x0a, 0x0a, 0x00, 0x0c, 0x0c, 0x0c, 0x00,
    0x0e, 0x0e, 0x0e, 0x00, 0x10, 0x10, 0x10, 0x00,
    0x12, 0x12, 0x12, 0x00, 0x14, 0x14, 0x14, 0x00,
    0x16, 0x16, 0x16, 0x00, 0x18, 0x18, 0x18, 0x00,
    0x1a, 0x1a, 0x1a, 0x00, 0x1c, 0x1c, 0x1c, 0x00,
    0x1e, 0x1e, 0x1e, 0x00, 0x20, 0x20, 0x20, 0x00,
    0x22, 0x22, 0x22, 0x00, 0x24, 0x24, 0x24, 0x00,
    0x26, 0x26, 0x26, 0x00, 0x28, 0x28, 0x28, 0x00,
    0x2a, 0x2a, 0x2a, 0x00, 0x2c, 0x2c, 0x2c, 0x00,
    0x2e, 0x2e, 0x2e, 0x00, 0x30, 0x30, 0x30, 0x00,
    0x32, 0x32, 0x32, 0x00, 0x34, 0x34, 0x34, 0x00,
    0x36, 0x36, 0x36, 0x00, 0x38, 0x38, 0x38, 0x00,
    0x3a, 0x3a, 0x3a, 0x00, 0x3c, 0x3c, 0x3c, 0x00,
    0x3e, 0x3e, 0x3e, 0x00, 0x40, 0x40, 0x40, 0x00,
    0x42, 0x42, 0x42, 0x00, 0x44, 0x44, 0x44, 0x00,
    0x46, 0x46, 0x46, 0x00, 0x48, 0x48, 0x48, 0x00,
    0x4a, 0x4a, 0x4a, 0x00, 0x4c, 0x4c, 0x4c, 0x00,
    0x4e, 0x4e, 0x4e, 0x00, 0x50, 0x50, 0x50, 0x00,
    0x52, 0x52, 0x52, 0x00, 0x54, 0x54, 0x54, 0x00,
    0x56, 0x56, 0x56, 0x00, 0x58, 0x58, 0x58, 0x00,
    0x5a, 0x5a, 0x5a, 0x00, 0x5c, 0x5c, 0x5c, 0x00,
    0x5e, 0x5e, 0x5e, 0x00, 0x60, 0x60, 0x60, 0x00,
    0x62, 0x62, 0x62, 0x00, 0x64, 0x64, 0x64, 0x00,
    0x66, 0x66, 0x66, 0x00, 0x68, 0x68, 0x68, 0x00,
    0x6a, 0x6a, 0x6a, 0x00, 0x6c, 0x6c, 0x6c, 0x00,
    0x6e, 0x6e, 0x6e, 0x00, 0x70, 0x70, 0x70, 0x00,
    0x72, 0x72, 0x72, 0x00, 0x74, 0x74, 0x74, 0x00,
    0x76, 0x76, 0x76, 0x00, 0x78, 0x78, 0x78, 0x00,
    0x7a, 0x7a, 0x7a, 0x00, 0x7c, 0x7c, 0x7c, 0x00,
    0x7e, 0x7e, 0x7e, 0x00, 0x80, 0x80, 0x80, 0x00,
    0x82, 0x82, 0x82, 0x00, 0x84, 0x84, 0x84, 0x00,
    0x86, 0x86, 0x86, 0x00, 0x88, 0x88, 0x88, 0x00,
    0x8a, 0x8a, 0x8a, 0x00, 0x8c, 0x8c, 0x8c, 0x00,
    0x8e, 0x8e, 0x8e, 0x00, 0x90, 0x90, 0x90, 0x00,
    0x92, 0x92, 0x92, 0x00, 0x94, 0x94, 0x94, 0x00,
    0x96, 0x96, 0x96, 0x00, 0x98, 0x98, 0x98, 0x00,
    0x9a, 0x9a, 0x9a, 0x00, 0x9c, 0x9c, 0x9c, 0x00,
    0x9e, 0x9e, 0x9e, 0x00, 0xa0, 0xa0, 0xa0, 0x00,
    0xa2, 0xa2, 0xa2, 0x00, 0xa4, 0xa4, 0xa4, 0x00,
    0xa6, 0xa6, 0xa6, 0x00, 0xa8, 0xa8, 0xa8, 0x00,
    0xaa, 0xaa, 0xaa, 0x00, 0xac, 0xac, 0xac, 0x00,
    0xae, 0xae, 0xae, 0x00, 0xb0, 0xb0, 0xb0, 0x00,
    0xb2, 0xb2, 0xb2, 0x00, 0xb4, 0xb4, 0xb4, 0x00,
    0xb6, 0xb6, 0xb6, 0x00, 0xb8, 0xb8, 0xb8, 0x00,
    0xba, 0xba, 0xba, 0x00, 0xbc, 0xbc, 0xbc, 0x00,
    0xbe, 0xbe, 0xbe, 0x00, 0xc0, 0xc0, 0xc0, 0x00,
    0xc2, 0xc2, 0xc2, 0x00, 0xc4, 0xc4, 0xc4, 0x00,
    0xc6, 0xc6, 0xc6, 0x00, 0xc8, 0xc8, 0xc8, 0x00,
    0xca, 0xca, 0xca, 0x00, 0xcc, 0xcc, 0xcc, 0x00,
    0xce, 0xce, 0xce, 0x00, 0xd0, 0xd0, 0xd0, 0x00,
    0xd2, 0xd2, 0xd2, 0x00, 0xd4, 0xd4, 0xd4, 0x00,
    0xd6, 0xd6, 0xd6, 0x00, 0xd8, 0xd8, 0xd8, 0x00,
    0xda, 0xda, 0xda, 0x00, 0xdc, 0xdc, 0xdc, 0x00,
    0xde, 0xde, 0xde, 0x00, 0xe0, 0xe0, 0xe0, 0x00,
    0xe2, 0xe2, 0xe2, 0x00, 0xe4, 0xe4, 0xe4, 0x00,
    0xe6, 0xe6, 0xe6, 0x00, 0xe8, 0xe8, 0xe8, 0x00,
    0xea, 0xea, 0xea, 0x00, 0xec, 0xec, 0xec, 0x00,
    0xee, 0xee, 0xee, 0x00, 0xf0, 0xf0, 0xf0, 0x00,
    0xf2, 0xf2, 0xf2, 0x00, 0xf4, 0xf4, 0xf4, 0x00,
    0xf6, 0xf6, 0xf6, 0x00, 0xf8, 0xf8, 0xf8, 0x00,
    0xfa, 0xfa, 0xfa, 0x00, 0xfc, 0xfc, 0xfc, 0x00,
    0xfe, 0xfe, 0xfe, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x34, 0x02, 0x00, 0x00, 0x99, 0x01, 0x00, 0x00,
    0x42, 0x01, 0x05, 0x00, 0x01, 0x00, 0x28, 0x00,
    0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
    0xfe, 0x00, 0xfd, 0x00, 0xfc, 0x00, 0xfb, 0x00,
    0xfa, 0x00, 0xf9, 0x00, 0xf8, 0x00, 0xf7, 0x00,
    0xf6, 0x00, 0xf5, 0x00, 0xf4, 0x00, 0xf3, 0x00,
    0xf2, 0x00, 0xf1, 0x00, 0xf0, 0x00, 0xef, 0x00,
    0xee, 0x00, 0xed, 0x00, 0xec, 0x00, 0xeb, 0x00,
    0xea, 0x00, 0xe9, 0x00, 0xe8, 0x00, 0xe7, 0x00,
    0xe6, 0x00, 0xe5, 0x00, 0xe4, 0x00, 0xe3, 0x00,
    0xe2, 0x00, 0xe1, 0x00, 0xe0, 0x00, 0xdf, 0x00,
    0xde, 0x00, 0xdd, 0x00, 0xdc, 0x00, 0xdb, 0x00,
    0xda, 0x00, 0xd9, 0x00, 0xd8, 0x00, 0xd7, 0x00,
    0xd6, 0x00, 0xd5, 0x00, 0xd4, 0x00, 0xd3, 0x00,
    0xd2, 0x00, 0xd1, 0x00, 0xd0, 0x00, 0xcf, 0x00,
    0xce, 0x00, 0xcd, 0x00, 0xcc, 0x00, 0xcb, 0x00,
    0xca, 0x00, 0xc9, 0x00, 0xc8, 0x00, 0xc7, 0x00,
    0xc6, 0x00, 0xc5, 0x00, 0xc4, 0x00, 0xc3, 0x00,
    0xc2, 0x00, 0xc1, 0x00, 0xc0, 0x00, 0xbf, 0x00,
    0xbe, 0x00, 0xbd, 0x00, 0xbc, 0x00, 0xbb, 0x00,
    0xba, 0x00, 0xb9, 0x00, 0xb8, 0x00, 0xb7, 0x00,
    0xb6, 0x00, 0xb5, 0x00, 0xb4, 0x00, 0xb3, 0x00,
    0xb2, 0x00, 0xb1, 0x00, 0xb0, 0x00, 0xaf, 0x00,
    0xae, 0x00, 0xad, 0x00, 0xac, 0x00, 0xab, 0x00,
    0xaa, 0x00, 0xa9, 0x00, 0xa8, 0x00, 0xa7, 0x00,
    0xa6, 0x00, 0xa5, 0x00, 0xa4, 0x00, 0xa3, 0x00,
    0xa2, 0x00, 0xa1, 0x00, 0xa0, 0x00, 0x9f, 0x00,
    0x9e, 0x00, 0x9d, 0x00, 0x9c, 0x00, 0x9b, 0x00,
    0x9a, 0x00, 0x99, 0x00, 0x98, 0x00, 0x97, 0x00,
    0x96, 0x00, 0x95, 0x00, 0x94, 0x00, 0x93, 0x00,
    0x92, 0x00, 0x91, 0x00, 0x90, 0x00, 0x8f, 0x00,
    0x8e, 0x00, 0x8d, 0x00, 0x8c, 0x00, 0x8b, 0x00,
    0x8a, 0x00, 0x89, 0x00, 0x88, 0x00, 0x87, 0x00,
    0x86, 0x00, 0x85, 0x00, 0x84, 0x00, 0x83, 0x00,
    0x82, 0x00, 0x81, 0x00, 0x80, 0x00, 0x7f, 0x00,
    0x7e, 0x00, 0x7d, 0x00, 0x7c, 0x00, 0x7b, 0x00,
    0x7a, 0x00, 0x79, 0x00, 0x78, 0x00, 0x77, 0x00,
    0x76, 0x00, 0x75, 0x00, 0x74, 0x00, 0x73, 0x00,
    0x72, 0x00, 0x71, 0x00, 0x70, 0x00, 0x6f, 0x00,
    0x6e, 0x00, 0x6d, 0x00, 0x6c, 0x00, 0x6b, 0x00,
    0x6a, 0x00, 0x69, 0x00, 0x68, 0x00, 0x67, 0x00,
    0x66, 0x00, 0x65, 0x00, 0x64, 0x00, 0x63, 0x00,
    0x62, 0x00, 0x61, 0x00, 0x60, 0x00, 0x5f, 0x00,
    0x5e, 0x00, 0x5d, 0x00, 0x5c, 0x00, 0x5b, 0x00,
    0x5a, 0x00, 0x59, 0x00, 0x58, 0x00, 0x57, 0x00,
    0x56, 0x00, 0x55, 0x00, 0x54, 0x00, 0x53, 0x00,
    0x52, 0x00, 0x51, 0x00, 0x50, 0x00, 0x4f, 0x00,
    0x4e, 0x00, 0x4d, 0x00, 0x4c, 0x00, 0x4b, 0x00,
    0x4a, 0x00, 0x49, 0x00, 0x48, 0x00, 0x47, 0x00,
    0x46, 0x00, 0x45, 0x00, 0x44, 0x00, 0x43, 0x00,
    0x42, 0x00, 0x41, 0x00, 0x40, 0x00, 0x3f, 0x00,
    0x3e, 0x00, 0x3d, 0x00, 0x3c, 0x00, 0x3b, 0x00,
    0x3a, 0x00, 0x39, 0x00, 0x38, 0x00, 0x37, 0x00,
    0x36, 0x00, 0x35, 0x00, 0x34, 0x00, 0x33, 0x00,
    0x32, 0x00, 0x31, 0x00, 0x30, 0x00, 0x2f, 0x00,
    0x2e, 0x00, 0x2d, 0x00, 0x2c, 0x00, 0x2b, 0x00,
    0x2a, 0x00, 0x29, 0x00, 0x28, 0x00, 0x27, 0x00,
    0x26, 0x00, 0x25, 0x00, 0x24, 0x00, 0x23, 0x00,
    0x22, 0x00, 0x21, 0x00, 0x20, 0x00, 0x1f, 0x00,
    0x1e, 0x00, 0x1d, 0x00, 0x1c, 0x00, 0x1b, 0x00,
    0x1a, 0x00, 0x19, 0x00, 0x18, 0x00, 0x17, 0x00,
    0x16, 0x00, 0x15, 0x00, 0x14, 0x00, 0x13, 0x00,
    0x12, 0x00, 0x11, 0x00, 0x10, 0x00, 0x0f, 0x00,
    0x0e, 0x00, 0x0d, 0x00, 0x0c, 0x00, 0x0b, 0x00,
    0x0a, 0x00, 0x09, 0x00, 0x08, 0x00, 0x07, 0x00,
    0x06, 0x00, 0x05, 0x00, 0x04, 0x00, 0x03, 0x00,
    0x02, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11,
    0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21,
    0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
    0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30, 0x31,
    0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
    0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41,
    0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
    0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51,
    0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
    0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61,
    0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
    0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71,
    0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79,
    0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81,
    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
    0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91,
    0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1,
    0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9,
    0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1,
    0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9,
    0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0, 0xc1,
    0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9,
    0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1,
    0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9,
    0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1,
    0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
    0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0, 0xf1,
    0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9,
    0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff, 0x04, 0x00,
    0x00, 0x00, 0x2d, 0x01, 0x01, 0x00, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00
};

static const unsigned char emf_blit_bits[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x1a, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x84, 0x00, 0x00, 0x00, 0xba, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x20, 0x04, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x77, 0x07, 0x00, 0x00, 0x3b, 0x04, 0x00, 0x00,
    0xfa, 0x01, 0x00, 0x00, 0x1f, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x12, 0xb7, 0x07, 0x00,
    0x4f, 0x5f, 0x04, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x21, 0x00, 0xf0, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4d, 0x00, 0x00, 0x00, 0xb8, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00,
    0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b, 0x0c, 0x0d,
    0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
    0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
    0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25,
    0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
    0x4c, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
    0x21, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x4d, 0x00, 0x00, 0x00,
    0xb8, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x94, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11,
    0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21,
    0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
    0x2a, 0x2b, 0x2c, 0x2d, 0x4c, 0x00, 0x00, 0x00,
    0xb0, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11,
    0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21,
    0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
    0x2a, 0x2b, 0x2c, 0x2d, 0x74, 0x00, 0x00, 0x00,
    0xb8, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x0a, 0x14, 0x1e, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x94, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11,
    0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21,
    0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29,
    0x2a, 0x2b, 0x2c, 0x2d, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char emf_mask_blit_bits[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x58, 0x03, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x77, 0x07, 0x00, 0x00, 0x3b, 0x04, 0x00, 0x00,
    0xfa, 0x01, 0x00, 0x00, 0x1f, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x12, 0xb7, 0x07, 0x00,
    0x4f, 0x5f, 0x04, 0x00, 0x4e, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0xa8, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0xf4, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b, 0x0c, 0x0d,
    0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
    0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
    0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25,
    0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xf0, 0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00,
    0xf0, 0x00, 0x00, 0x00, 0x4e, 0x00, 0x00, 0x00,
    0xcc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0xa8, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b, 0x0c, 0x0d,
    0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
    0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
    0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25,
    0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
    0x4f, 0x00, 0x00, 0x00, 0x0c, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0xb4, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b, 0x0c, 0x0d,
    0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
    0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
    0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25,
    0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xf0, 0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00,
    0xf0, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

/* For debugging or dumping the raw metafiles produced by
 * new test functions.
 */
static INT CALLBACK mf_enum_proc(HDC hdc, HANDLETABLE *ht, METARECORD *mr,
                                 INT nobj, LPARAM param)
{
    trace("hdc %p, mr->rdFunction %04x, mr->rdSize %lu, param %p\n",
           hdc, mr->rdFunction, mr->rdSize, (void *)param);
    return TRUE;
}

/* For debugging or dumping the raw metafiles produced by
 * new test functions.
 */

static void dump_mf_bits (const HMETAFILE mf, const char *desc)
{
    BYTE buf[MF_BUFSIZE];
    UINT mfsize, i;

    if (!winetest_debug) return;

    mfsize = GetMetaFileBitsEx (mf, MF_BUFSIZE, buf);
    ok (mfsize > 0, "%s: GetMetaFileBitsEx failed.\n", desc);

    printf ("MetaFile %s has bits:\n{\n    ", desc);
    for (i=0; i<mfsize; i++)
    {
        printf ("0x%02x", buf[i]);
        if (i == mfsize-1)
            printf ("\n");
        else if (i % 8 == 7)
            printf (",\n    ");
        else
            printf (", ");
    }
    printf ("};\n");
}

/* Compare the metafile produced by a test function with the
 * expected raw metafile data in "bits".
 * Return value is 0 for a perfect match,
 * -1 if lengths aren't equal,
 * otherwise returns the number of non-matching bytes.
 */

static int compare_mf_bits (const HMETAFILE mf, const unsigned char *bits, UINT bsize,
    const char *desc)
{
    unsigned char buf[MF_BUFSIZE];
    UINT mfsize, i;
    int diff;

    mfsize = GetMetaFileBitsEx (mf, MF_BUFSIZE, buf);
    ok (mfsize > 0, "%s: GetMetaFileBitsEx failed.\n", desc);
    if (mfsize < MF_BUFSIZE)
        ok (mfsize == bsize, "%s: mfsize=%d, bsize=%d.\n",
            desc, mfsize, bsize);
    else
        ok (bsize >= MF_BUFSIZE, "%s: mfsize > bufsize (%d bytes), bsize=%d.\n",
            desc, mfsize, bsize);
    if (mfsize != bsize)
        return -1;

    diff = 0;
    for (i=0; i<bsize; i++)
    {
       if (buf[i] !=  bits[i])
           diff++;
    }
    ok (diff == 0, "%s: mfsize=%d, bsize=%d, diff=%d\n",
        desc, mfsize, bsize, diff);

    return diff; 
}

static int compare_mf_disk_bits(LPCSTR name, const BYTE *bits, UINT bsize, const char *desc)
{
    unsigned char buf[MF_BUFSIZE];
    DWORD mfsize, rd_size, i;
    int diff;
    HANDLE hfile;
    BOOL ret;

    hfile = CreateFileA(name, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, 0);
    assert(hfile != INVALID_HANDLE_VALUE);

    mfsize = GetFileSize(hfile, NULL);
    assert(mfsize <= MF_BUFSIZE);

    ret = ReadFile(hfile, buf, sizeof(buf), &rd_size, NULL);
    ok( ret && rd_size == mfsize, "ReadFile: error %ld\n", GetLastError());

    CloseHandle(hfile);

    ok(mfsize == bsize, "%s: mfsize=%ld, bsize=%d.\n", desc, mfsize, bsize);

    if (mfsize != bsize)
        return -1;

    diff = 0;
    for (i=0; i<bsize; i++)
    {
        if (buf[i] != bits[i])
            diff++;
    }
    ok(diff == 0, "%s: mfsize=%ld, bsize=%d, diff=%d\n",
        desc, mfsize, bsize, diff);

    return diff; 
}

/* For debugging or dumping the raw EMFs produced by
 * new test functions.
 */
static void dump_emf_bits(const HENHMETAFILE mf, const char *desc)
{
    BYTE buf[MF_BUFSIZE];
    UINT mfsize, i;

    if (!winetest_debug) return;

    mfsize = GetEnhMetaFileBits(mf, MF_BUFSIZE, buf);
    ok (mfsize > 0, "%s: GetEnhMetaFileBits failed\n", desc);

    printf("EMF %s has bits:\n{\n    ", desc);
    for (i = 0; i < mfsize; i++)
    {
        printf ("0x%02x", buf[i]);
        if (i == mfsize-1)
            printf ("\n");
        else if (i % 8 == 7)
            printf (",\n    ");
        else
            printf (", ");
    }
    printf ("};\n");
}

static void dump_emf_records(const HENHMETAFILE mf, const char *desc)
{
    BYTE *emf;
    BYTE buf[MF_BUFSIZE];
    UINT mfsize, offset;

    if (!winetest_debug) return;

    mfsize = GetEnhMetaFileBits(mf, MF_BUFSIZE, buf);
    ok (mfsize > 0, "%s: GetEnhMetaFileBits error %ld\n", desc, GetLastError());

    printf("EMF %s has records:\n", desc);

    emf = buf;
    offset = 0;
    while(offset < mfsize)
    {
        EMR *emr = (EMR *)(emf + offset);
        printf("emr->iType %ld, emr->nSize %lu\n", emr->iType, emr->nSize);
        /*trace("emr->iType 0x%04lx, emr->nSize 0x%04lx\n", emr->iType, emr->nSize);*/
        offset += emr->nSize;
    }
}

static void dump_emf_record(const ENHMETARECORD *emr, const char *desc)
{
    const BYTE *buf;
    DWORD i;

    if (!winetest_debug) return;

    printf ("%s: EMF record %lu has bits:\n{\n", desc, emr->iType);
    buf = (const BYTE *)emr;
    for (i = 0; i < emr->nSize; i++)
    {
        printf ("0x%02x", buf[i]);
        if (i == emr->nSize - 1)
            printf ("\n");
        else if (i % 8 == 7)
            printf (",\n");
        else
            printf (", ");
    }
    printf ("};\n");
}

static void dump_EMREXTTEXTOUT(const EMREXTTEXTOUTW *eto)
{
    trace("rclBounds %ld,%ld - %ld,%ld\n", eto->rclBounds.left, eto->rclBounds.top,
          eto->rclBounds.right, eto->rclBounds.bottom);
    trace("iGraphicsMode %lu\n", eto->iGraphicsMode);
    trace("exScale: %f\n", eto->exScale);
    trace("eyScale: %f\n", eto->eyScale);
    trace("emrtext.ptlReference %ld,%ld\n", eto->emrtext.ptlReference.x, eto->emrtext.ptlReference.y);
    trace("emrtext.nChars %lu\n", eto->emrtext.nChars);
    trace("emrtext.offString %#lx\n", eto->emrtext.offString);
    trace("emrtext.fOptions %#lx\n", eto->emrtext.fOptions);
    trace("emrtext.rcl %ld,%ld - %ld,%ld\n", eto->emrtext.rcl.left, eto->emrtext.rcl.top,
          eto->emrtext.rcl.right, eto->emrtext.rcl.bottom);
    trace("emrtext.offDx %#lx\n", eto->emrtext.offDx);
}

static BOOL match_emf_record(const ENHMETARECORD *emr1, const ENHMETARECORD *emr2,
                             const char *desc, BOOL ignore_scaling)
{
    int diff;

    ok(emr1->iType == emr2->iType, "%s: emr->iType %lu != %lu\n",
       desc, emr1->iType, emr2->iType);

    ok(emr1->nSize == emr2->nSize, "%s: emr->nSize %lu != %lu\n",
       desc, emr1->nSize, emr2->nSize);

    /* iType and nSize mismatches are fatal */
    if (emr1->iType != emr2->iType || emr1->nSize != emr2->nSize) return FALSE;

    /* contents of EMR_GDICOMMENT are not interesting */
    if (emr1->iType == EMR_GDICOMMENT) return TRUE;

    /* different Windows versions setup DC scaling differently when
     * converting an old style metafile to an EMF.
     */
    if (ignore_scaling && (emr1->iType == EMR_SETWINDOWEXTEX ||
                           emr1->iType == EMR_SETVIEWPORTEXTEX))
        return TRUE;

    if (emr1->iType == EMR_EXTTEXTOUTW || emr1->iType == EMR_EXTTEXTOUTA)
    {
        EMREXTTEXTOUTW *eto1, *eto2;

        eto1 = HeapAlloc(GetProcessHeap(), 0, emr1->nSize);
        memcpy(eto1, emr1, emr1->nSize);
        eto2 = HeapAlloc(GetProcessHeap(), 0, emr2->nSize);
        memcpy(eto2, emr2, emr2->nSize);

        /* different Windows versions setup DC scaling differently */
        eto1->exScale = eto1->eyScale = 0.0;
        eto2->exScale = eto2->eyScale = 0.0;

        diff = memcmp(eto1, eto2, emr1->nSize);
        if (diff)
        {
            dump_EMREXTTEXTOUT(eto1);
            dump_EMREXTTEXTOUT(eto2);
        }
        HeapFree(GetProcessHeap(), 0, eto1);
        HeapFree(GetProcessHeap(), 0, eto2);
    }
    else if (emr1->iType == EMR_EXTSELECTCLIPRGN && !lstrcmpA(desc, "emf_clipping"))
    {
        diff = memcmp(emr1, emr2, emr1->nSize);
    }
    else if (emr1->iType == EMR_POLYBEZIERTO16 || emr1->iType == EMR_POLYBEZIER16)
    {
        EMRPOLYBEZIER16 *eto1, *eto2;

        eto1 = (EMRPOLYBEZIER16*)emr1;
        eto2 = (EMRPOLYBEZIER16*)emr2;

        diff = eto1->cpts != eto2->cpts;
        if(!diff)
            diff = memcmp(eto1->apts, eto2->apts, eto1->cpts * sizeof(POINTS));
    }
    else if (emr1->iType == EMR_POLYBEZIERTO || emr1->iType == EMR_POLYBEZIER)
    {
        EMRPOLYBEZIER *eto1, *eto2;

        eto1 = (EMRPOLYBEZIER*)emr1;
        eto2 = (EMRPOLYBEZIER*)emr2;

        diff = eto1->cptl != eto2->cptl;
        if(!diff)
            diff = memcmp(eto1->aptl, eto2->aptl, eto1->cptl * sizeof(POINTL));
    }
    else if (emr1->iType == EMR_PLGBLT && emr2->iType == EMR_PLGBLT)
    {
        EMRPLGBLT *eto1 = (EMRPLGBLT *)emr1;
        EMRPLGBLT *eto2 = (EMRPLGBLT *)emr2;
        diff = memcmp(emr1, emr2, sizeof(*emr1)) ||
            memcmp(&eto1->aptlDest, &eto2->aptlDest, emr1->nSize - FIELD_OFFSET(EMRPLGBLT, aptlDest));
    }
    else
        diff = memcmp(emr1, emr2, emr1->nSize);

    ok(diff == 0, "%s: contents of record %lu don't match\n", desc, emr1->iType);

    if (diff)
    {
        dump_emf_record(emr1, "expected bits");
        dump_emf_record(emr2, "actual bits");
    }

    return diff == 0; /* report all non-fatal record mismatches */
}

/* Compare the EMF produced by a test function with the
 * expected raw EMF data in "bits".
 * Return value is 0 for a perfect match,
 * -1 if lengths aren't equal,
 * otherwise returns the number of non-matching bytes.
 */
static int compare_emf_bits(const HENHMETAFILE mf, const unsigned char *bits,
                            UINT bsize, const char *desc,
                            BOOL ignore_scaling)
{
    unsigned char buf[MF_BUFSIZE];
    UINT mfsize, offset1, offset2;
    const ENHMETAHEADER *emh1, *emh2;

    mfsize = GetEnhMetaFileBits(mf, MF_BUFSIZE, buf);
    ok (mfsize > 0, "%s: GetEnhMetaFileBits error %ld\n", desc, GetLastError());

    if (mfsize < MF_BUFSIZE)
    {
        ok(mfsize == bsize, "%s: mfsize=%d, bsize=%d\n", desc, mfsize, bsize);
    }
    else
        ok(bsize >= MF_BUFSIZE, "%s: mfsize > bufsize (%d bytes), bsize=%d\n",
           desc, mfsize, bsize);

    /* basic things must match */
    emh1 = (const ENHMETAHEADER *)bits;
    emh2 = (const ENHMETAHEADER *)buf;
    ok(emh1->iType == EMR_HEADER, "expected EMR_HEADER, got %lu\n", emh1->iType);
    ok(emh1->nSize == sizeof(ENHMETAHEADER), "expected sizeof(ENHMETAHEADER), got %lu\n", emh1->nSize);
    ok(emh2->nBytes == mfsize, "expected emh->nBytes %u, got %lu\n", mfsize, emh2->nBytes);
    ok(emh1->dSignature == ENHMETA_SIGNATURE, "expected ENHMETA_SIGNATURE, got %lu\n", emh1->dSignature);

    ok(emh1->iType == emh2->iType, "expected EMR_HEADER, got %lu\n", emh2->iType);
    ok(emh1->nSize == emh2->nSize,
       "expected nSize %lu, got %lu\n", emh1->nSize, emh2->nSize);
    ok(emh1->rclBounds.left == emh2->rclBounds.left, "%s: expected rclBounds.left = %ld, got %ld\n",
            desc, emh1->rclBounds.left, emh2->rclBounds.left);
    ok(emh1->rclBounds.top == emh2->rclBounds.top, "%s: expected rclBounds.top = %ld, got %ld\n",
            desc, emh1->rclBounds.top, emh2->rclBounds.top);
    ok(emh1->rclBounds.right == emh2->rclBounds.right, "%s: expected rclBounds.right = %ld, got %ld\n",
            desc, emh1->rclBounds.right, emh2->rclBounds.right);
    ok(emh1->rclBounds.bottom == emh2->rclBounds.bottom, "%s: expected rclBounds.bottom = %ld, got %ld\n",
            desc, emh1->rclBounds.bottom, emh2->rclBounds.bottom);
    ok(emh1->dSignature == emh2->dSignature, "expected dSignature %lu, got %lu\n", emh1->dSignature, emh2->dSignature);
    ok(emh1->nBytes == emh2->nBytes,
       "expected nBytes %lu, got %lu\n", emh1->nBytes, emh2->nBytes);
    ok(emh1->nRecords == emh2->nRecords, "expected nRecords %lu, got %lu\n", emh1->nRecords, emh2->nRecords);

    offset1 = emh1->nSize;
    offset2 = emh2->nSize; /* Needed for Win9x/WinME/NT4 */
    while (offset1 < emh1->nBytes)
    {
        const ENHMETARECORD *emr1 = (const ENHMETARECORD *)(bits + offset1);
        const ENHMETARECORD *emr2 = (const ENHMETARECORD *)(buf + offset2);

        if (winetest_debug > 1)
            trace("%s: EMF record %lu, size %lu/record %lu, size %lu\n",
                  desc, emr1->iType, emr1->nSize, emr2->iType, emr2->nSize);

        if (!match_emf_record(emr1, emr2, desc, ignore_scaling)) return -1;

        /* We have already bailed out if iType or nSize don't match */
        offset1 += emr1->nSize;
        offset2 += emr2->nSize;
    }
    return 0;
}

static const unsigned char EMF_STRETCHBLT_1BIT_3X3[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x2c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4d, 0x00, 0x00, 0x00,
    0xac, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x9c, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHBLT_4BIT_3X3[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x64, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4d, 0x00, 0x00, 0x00,
    0xe4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00,
    0xd4, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHBLT_8BIT_3X3[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x05, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4d, 0x00, 0x00, 0x00,
    0xa4, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x28, 0x04, 0x00, 0x00,
    0x94, 0x04, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHBLT_16BIT_555_3X3[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x40, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4d, 0x00, 0x00, 0x00,
    0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
    0xa0, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x7c, 0x00, 0x00, 0xe0, 0x03, 0x00, 0x00,
    0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHBLT_24BIT_3X3[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x44, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4d, 0x00, 0x00, 0x00,
    0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x94, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHBLT_32BIT_888_3X3[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x60, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4d, 0x00, 0x00, 0x00,
    0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
    0xa0, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHBLT_16BIT_3X3[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x40, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4d, 0x00, 0x00, 0x00,
    0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
    0xa0, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xf8, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00,
    0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHBLT_32BIT_3X3[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x60, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4d, 0x00, 0x00, 0x00,
    0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x6c, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
    0xa0, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xff, 0x00, 0xe0, 0xff, 0x00, 0x00,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_BITBLT_1BIT_4X4[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0xa4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x94, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_BITBLT_4BIT_4X4[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x5c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00,
    0xcc, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_BITBLT_8BIT_4X4[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x1c, 0x05, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x9c, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x28, 0x04, 0x00, 0x00,
    0x8c, 0x04, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_BITBLT_16BIT_555_4X4[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x38, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0xb8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x7c, 0x00, 0x00, 0xe0, 0x03, 0x00, 0x00,
    0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_BITBLT_24BIT_4X4[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x3c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0xbc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_BITBLT_32BIT_888_4X4[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x58, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_BITBLT_16BIT_4X4[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x38, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0xb8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xf8, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00,
    0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_BITBLT_32BIT_4X4[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x58, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0xcc, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xff, 0x00, 0xe0, 0xff, 0x00, 0x00,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

/* tests blitting to an EMF */
static void test_emf_BitBlt(void)
{
    static const XFORM xform = {0.5f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f};
    static const int bitmap_width = 4, bitmap_height = 4;
    unsigned char bmi_buffer[FIELD_OFFSET(BITMAPINFO, bmiColors[256])];
    BITMAPINFO *bmi = (BITMAPINFO *)bmi_buffer;
    HDC hdc, hdc_emf, hdc_emf2, hdc_bitmap;
    HBITMAP hbitmap, old_hbitmap;
    HENHMETAFILE hemf, hemf2;
    int ret, test_idx;
    void *bits;

    static const struct
    {
        WORD bpp;
        int src_width;
        int src_height;
        WORD compression;
        const void *bits;
        size_t bits_count;
        DWORD used_color_count;
        DWORD color_count;
        RGBQUAD colors[3];
    }
    tests[] =
    {
        {1, 3, 3, BI_RGB, EMF_STRETCHBLT_1BIT_3X3, sizeof(EMF_STRETCHBLT_1BIT_3X3), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 3, 3, BI_RGB, EMF_STRETCHBLT_4BIT_3X3, sizeof(EMF_STRETCHBLT_4BIT_3X3), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 3, 3, BI_RGB, EMF_STRETCHBLT_8BIT_3X3, sizeof(EMF_STRETCHBLT_8BIT_3X3), 1, 1, {{0xff, 0xff, 0xff}}},
        {16, 3, 3, BI_RGB, EMF_STRETCHBLT_16BIT_555_3X3, sizeof(EMF_STRETCHBLT_16BIT_555_3X3)},
        {24, 3, 3, BI_RGB, EMF_STRETCHBLT_24BIT_3X3, sizeof(EMF_STRETCHBLT_24BIT_3X3)},
        {32, 3, 3, BI_RGB, EMF_STRETCHBLT_32BIT_888_3X3, sizeof(EMF_STRETCHBLT_32BIT_888_3X3)},
        {16, 3, 3, BI_BITFIELDS, EMF_STRETCHBLT_16BIT_3X3, sizeof(EMF_STRETCHBLT_16BIT_3X3), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {32, 3, 3, BI_BITFIELDS, EMF_STRETCHBLT_32BIT_3X3, sizeof(EMF_STRETCHBLT_32BIT_3X3), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
        {1, 4, 4, BI_RGB, EMF_BITBLT_1BIT_4X4, sizeof(EMF_BITBLT_1BIT_4X4), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 4, 4, BI_RGB, EMF_BITBLT_4BIT_4X4, sizeof(EMF_BITBLT_4BIT_4X4), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 4, 4, BI_RGB, EMF_BITBLT_8BIT_4X4, sizeof(EMF_BITBLT_8BIT_4X4), 1, 1, {{0xff, 0xff, 0xff}}},
        {16, 4, 4, BI_RGB, EMF_BITBLT_16BIT_555_4X4, sizeof(EMF_BITBLT_16BIT_555_4X4)},
        {24, 4, 4, BI_RGB, EMF_BITBLT_24BIT_4X4, sizeof(EMF_BITBLT_24BIT_4X4)},
        {32, 4, 4, BI_RGB, EMF_BITBLT_32BIT_888_4X4, sizeof(EMF_BITBLT_32BIT_888_4X4)},
        {16, 4, 4, BI_BITFIELDS, EMF_BITBLT_16BIT_4X4, sizeof(EMF_BITBLT_16BIT_4X4), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {32, 4, 4, BI_BITFIELDS, EMF_BITBLT_32BIT_4X4, sizeof(EMF_BITBLT_32BIT_4X4), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
    };

    hdc = GetDC(0);

    /* Test that source DC cannot be an enhanced metafile */
    hdc_emf = CreateEnhMetaFileW(hdc, NULL, NULL, NULL);
    ok(!!hdc_emf, "CreateEnhMetaFileW failed, error %ld\n", GetLastError());
    hdc_emf2 = CreateEnhMetaFileW(hdc, NULL, NULL, NULL);
    ok(!!hdc_emf2, "CreateEnhMetaFileW failed, error %ld\n", GetLastError());

    ret = StretchBlt(hdc_emf, 0, 0, 1, 1, hdc_emf2, 0, 0, 1, 1, SRCCOPY);
    ok(!ret, "StretchBlt succeeded\n");

    hemf2 = CloseEnhMetaFile(hdc_emf2);
    ok(!!hemf2, "CloseEnhMetaFile failed, error %ld\n", GetLastError());
    hemf = CloseEnhMetaFile(hdc_emf);
    ok(!!hemf, "CloseEnhMetaFile failed, error %ld\n", GetLastError());
    DeleteEnhMetaFile(hemf2);
    DeleteEnhMetaFile(hemf);

    for (test_idx = 0; test_idx < ARRAY_SIZE(tests); ++test_idx)
    {
        winetest_push_context("Test %d", test_idx);

        memset(bmi_buffer, 0, sizeof(bmi_buffer));
        bmi->bmiHeader.biSize = sizeof(bmi->bmiHeader);
        bmi->bmiHeader.biHeight = bitmap_width;
        bmi->bmiHeader.biWidth = bitmap_height;
        bmi->bmiHeader.biBitCount = tests[test_idx].bpp;
        bmi->bmiHeader.biPlanes = 1;
        bmi->bmiHeader.biCompression = tests[test_idx].compression;
        bmi->bmiHeader.biClrUsed = tests[test_idx].used_color_count;
        memcpy(bmi->bmiColors, tests[test_idx].colors, sizeof(RGBQUAD) * tests[test_idx].color_count);

        hbitmap = CreateDIBSection(hdc, bmi, DIB_RGB_COLORS, &bits, NULL, 0);
        ok(!!hbitmap, "CreateDIBSection failed, error %ld\n", GetLastError());
        hdc_bitmap = CreateCompatibleDC(hdc);
        ok(!!hdc_bitmap, "CreateCompatibleDC failed, error %ld\n", GetLastError());
        old_hbitmap = SelectObject(hdc_bitmap, hbitmap);

        SetBkColor(hdc_bitmap, RGB(0xff, 0xff, 0xff));
        ret = SetGraphicsMode(hdc_bitmap, GM_ADVANCED);
        ok(ret, "SetGraphicsMode failed, error %ld\n", GetLastError());
        ret = SetWorldTransform(hdc_bitmap, &xform);
        ok(ret, "SetWorldTransform failed, error %ld\n", GetLastError());

        hdc_emf = CreateEnhMetaFileW(hdc, NULL, NULL, NULL);
        ok(!!hdc_emf, "CreateEnhMetaFileW failed, error %ld\n", GetLastError());

        ret = BitBlt(hdc_bitmap, 0, 0, bitmap_width, bitmap_height, 0, 0, 0, BLACKNESS);
        ok(ret, "BitBlt failed, error %ld\n", GetLastError());
        if (tests[test_idx].src_width == bitmap_width && tests[test_idx].src_height == bitmap_height)
        {
            ret = BitBlt(hdc_emf, 0, 0, bitmap_width, bitmap_height, hdc_bitmap, 0, 0, SRCCOPY);
            ok(ret, "BitBlt failed, error %ld\n", GetLastError());
        }
        else
        {
            ret = StretchBlt(hdc_emf, 0, 0, bitmap_width, bitmap_height, hdc_bitmap, 0, 0,
                             tests[test_idx].src_width, tests[test_idx].src_height, SRCCOPY);
            ok(ret, "StretchBlt failed, error %ld\n", GetLastError());
        }

        hemf = CloseEnhMetaFile(hdc_emf);
        ok(!!hemf, "CloseEnhMetaFile failed, error %ld\n", GetLastError());

        ret = compare_emf_bits(hemf, tests[test_idx].bits, tests[test_idx].bits_count,
                               "test_emf_BitBlt", FALSE);
        if (ret)
        {
            dump_emf_bits(hemf, "test_emf_BitBlt");
            dump_emf_records(hemf, "test_emf_BitBlt");
        }

        DeleteEnhMetaFile(hemf);
        SelectObject(hdc_bitmap, old_hbitmap);
        DeleteDC(hdc_bitmap);
        DeleteObject(hbitmap);

        winetest_pop_context();
    }

    ReleaseDC(0, hdc);
}

static void test_emf_DCBrush(void)
{
    HDC hdcMetafile;
    HENHMETAFILE hMetafile;
    HBRUSH hBrush;
    HPEN hPen;
    BOOL ret;
    COLORREF color;

    if (!pSetDCBrushColor || !pSetDCPenColor)
    {
        win_skip( "SetDCBrush/PenColor not supported\n" );
        return;
    }

    hdcMetafile = CreateEnhMetaFileA(GetDC(0), NULL, NULL, NULL);
    ok( hdcMetafile != 0, "CreateEnhMetaFileA failed\n" );

    hBrush = SelectObject(hdcMetafile, GetStockObject(DC_BRUSH));
    ok(hBrush != 0, "SelectObject error %ld.\n", GetLastError());

    hPen = SelectObject(hdcMetafile, GetStockObject(DC_PEN));
    ok(hPen != 0, "SelectObject error %ld.\n", GetLastError());

    color = pSetDCBrushColor( hdcMetafile, RGB(0x55,0x55,0x55) );
    ok( color == 0xffffff, "SetDCBrushColor returned %lx\n", color );

    color = pSetDCPenColor( hdcMetafile, RGB(0x33,0x44,0x55) );
    ok( color == 0, "SetDCPenColor returned %lx\n", color );

    Rectangle( hdcMetafile, 10, 10, 20, 20 );

    color = pSetDCBrushColor( hdcMetafile, RGB(0x12,0x34,0x56) );
    ok( color == 0x555555, "SetDCBrushColor returned %lx\n", color );

    hMetafile = CloseEnhMetaFile(hdcMetafile);
    ok( hMetafile != 0, "CloseEnhMetaFile failed\n" );

    if (compare_emf_bits (hMetafile, EMF_DCBRUSH_BITS, sizeof(EMF_DCBRUSH_BITS),
                          "emf_DC_Brush", FALSE ) != 0)
    {
        dump_emf_bits(hMetafile, "emf_DC_Brush");
        dump_emf_records(hMetafile, "emf_DC_Brush");
    }
    ret = DeleteEnhMetaFile(hMetafile);
    ok( ret, "DeleteEnhMetaFile error %ld\n", GetLastError());
    ret = DeleteObject(hBrush);
    ok( ret, "DeleteObject(HBRUSH) error %ld\n", GetLastError());
    ret = DeleteObject(hPen);
    ok( ret, "DeleteObject(HPEN) error %ld\n", GetLastError());
}

/* Test a blank metafile.  May be used as a template for new tests. */

static void test_mf_Blank(void)
{
    HDC hdcMetafile;
    HMETAFILE hMetafile;
    INT caps;
    BOOL ret;
    INT type;

    hdcMetafile = CreateMetaFileA(NULL);
    ok(hdcMetafile != 0, "CreateMetaFileA(NULL) error %ld\n", GetLastError());

/* Tests on metafile initialization */
    caps = GetDeviceCaps (hdcMetafile, TECHNOLOGY);
    ok (caps == DT_METAFILE,
        "GetDeviceCaps: TECHNOLOGY=%d != DT_METAFILE.\n", caps);

    hMetafile = CloseMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseMetaFile error %ld\n", GetLastError());
    type = GetObjectType(hMetafile);
    ok(type == OBJ_METAFILE, "CloseMetaFile created object with type %d\n", type);
    type = GetObjectType(hdcMetafile);
    ok(type == 0 || broken(type == OBJ_METADC) /* win10 >=1607 */,
       "CloseMetaFile has to destroy metafile hdc (%d)\n", type);

    if (compare_mf_bits (hMetafile, MF_BLANK_BITS, sizeof(MF_BLANK_BITS),
        "mf_blank") != 0)
    {
        dump_mf_bits(hMetafile, "mf_Blank");
        EnumMetaFile(0, hMetafile, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hMetafile);
    ok( ret, "DeleteMetaFile(%p) error %ld\n", hMetafile, GetLastError());
}

static void test_metafile_file(void)
{
    char temp_path[MAX_PATH];
    char mf_name[MAX_PATH];
    char buf[4096];
    HMETAFILE metafile;
    POINT oldpoint;
    DWORD size;
    HANDLE file;
    HDC dc;
    BOOL ret;

    GetTempPathA(MAX_PATH, temp_path);
    GetTempFileNameA(temp_path, "wmf", 0, mf_name);

    dc = CreateMetaFileA(mf_name);
    ok(dc != 0, "CreateMetaFileA(NULL) error %ld\n", GetLastError());

    file = CreateFileA(mf_name, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                       OPEN_EXISTING, 0, NULL);
    ok(file == INVALID_HANDLE_VALUE && GetLastError() == ERROR_SHARING_VIOLATION,
       "CreateFile returned: %p %lu\n", file, GetLastError());

    file = CreateFileA(mf_name, 0, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                       OPEN_EXISTING, 0, NULL);
    ok(file != INVALID_HANDLE_VALUE, "CreateFile failed: %lu\n", GetLastError());

    size = GetFileSize(file, NULL);
    ok(!size, "size = %lu\n", size);

    ret = MoveToEx(dc, 1, 1, NULL);
    ok( ret, "MoveToEx error %ld.\n", GetLastError());
    ret = LineTo(dc, 2, 2);
    ok( ret, "LineTo error %ld.\n", GetLastError());
    ret = MoveToEx(dc, 1, 1, &oldpoint);
    ok( ret, "MoveToEx error %ld.\n", GetLastError());
    ret = Ellipse(dc, 0, 0, 2, 2);
    ok( ret, "Ellipse error %ld.\n", GetLastError());

    ret = GetCurrentPositionEx(dc, &oldpoint);
    ok(!ret, "GetCurrentPositionEx succeeded\n");

    size = GetFileSize(file, NULL);
    ok(!size, "size = %lu\n", size);

    metafile = CloseMetaFile(dc);
    size = GetFileSize(file, NULL);
    ok(size == sizeof(MF_GRAPHICS_BITS), "size = %lu\n", size);

    CloseHandle(file);
    file = CreateFileA(mf_name, GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                       OPEN_EXISTING, 0, NULL);
    todo_wine
    ok(file == INVALID_HANDLE_VALUE && GetLastError() == ERROR_SHARING_VIOLATION,
       "CreateFile returned: %p %lu\n", file, GetLastError());
    if (file != INVALID_HANDLE_VALUE) CloseHandle(file);
    file = CreateFileA(mf_name, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                       OPEN_EXISTING, 0, NULL);
    ok(file != INVALID_HANDLE_VALUE, "CreateFile failed: %lu\n", GetLastError());
    ret = ReadFile(file, buf, sizeof(buf), &size, NULL);
    ok(ret, "ReadFile failed: %lu\n", GetLastError());
    ok(size == sizeof(MF_GRAPHICS_BITS), "size = %lu\n", size);
    ok(!memcmp(buf, MF_GRAPHICS_BITS, sizeof(MF_GRAPHICS_BITS)), "unexpected file content\n");
    CloseHandle(file);

    if (compare_mf_bits(metafile, MF_GRAPHICS_BITS, sizeof(MF_GRAPHICS_BITS), "mf_Graphics") != 0)
    {
        dump_mf_bits(metafile, "mf_Graphics");
        EnumMetaFile(0, metafile, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(metafile);
    ok(ret, "Could not delete metafile: %lu\n", GetLastError());

    SetLastError(0xdeadbeef);
    ret = DeleteMetaFile(metafile);
    ok(!ret, "DeleteMetaFile succeeded\n");
    ok(GetLastError() == ERROR_INVALID_HANDLE, "GetLastError() = %lu\n", GetLastError());

    SetLastError(0xdeadbeef);
    metafile = CloseMetaFile(dc);
    ok(!metafile && GetLastError() == ERROR_INVALID_HANDLE, "CloseMetaFile returned %p (%lu)\n",
       metafile, GetLastError());

    ret = DeleteFileA(mf_name);
    ok(ret, "Could not delete file: %lu\n", GetLastError());

    SetLastError(0xdeadbeef);
    ret = DeleteMetaFile(ULongToHandle(0xdeadbeef));
    ok(!ret, "DeleteMetaFile succeeded\n");
    ok(GetLastError() == ERROR_INVALID_HANDLE, "GetLastError() = %lu\n", GetLastError());

    SetLastError(0xdeadbeef);
    size = GetMetaFileBitsEx(ULongToHandle(0xdeadbeef), 0, NULL);
    ok(!size, "GetMetaFileBitsEx returned %lu\n", size);
    ok(GetLastError() == ERROR_INVALID_HANDLE, "GetLastError() = %lu\n", GetLastError());
}

static void test_mf_SetPixel(void)
{
    HMETAFILE mf;
    COLORREF c;
    BOOL ret;
    HDC hdc;

    hdc = CreateMetaFileW(NULL);
    ok(hdc != 0, "CreateEnhMetaFile failed\n");

    c = GetPixel(hdc, 5, 5);
    ok(c == CLR_INVALID, "c = %lx\n", c);

    c = SetPixel(hdc, 5, 5, RGB(1,2,3));
    ok(c == 1, "c = %lx\n", c);

    c = SetPixel(hdc, 5, 5, RGB(1,2,3));
    ok(c == 1, "c = %lx\n", c);

    ret = SetPixelV(hdc, 5, 5, RGB(1,2,3));
    ok(ret, "ret = %x\n", ret);

    c = GetPixel(hdc, 5, 5);
    ok(c == CLR_INVALID, "c = %lx\n", c);

    ret = Rectangle(hdc, 1, 1, 10, 10);
    ok(ret, "Rectangle failed\n");

    c = GetPixel(hdc, 5, 5);
    ok(c == CLR_INVALID, "c = %lx\n", c);

    mf = CloseMetaFile(hdc);
    ok(mf != 0, "CloseEnhMetaFile failed\n");
    DeleteMetaFile(mf);
}

static void test_mf_attrs(void)
{
    HMETAFILE mf;
    UINT attr;
    HDC hdc;

    hdc = CreateMetaFileW(NULL);
    ok(hdc != 0, "CreateEnhMetaFile failed\n");

    attr = SetTextAlign(hdc, TA_BOTTOM);
    ok(attr == TRUE, "attr = %x\n", attr);
    attr = SetTextAlign(hdc, TA_TOP);
    ok(attr == TRUE, "attr = %x\n", attr);

    attr = SetBkMode(hdc, TRANSPARENT);
    ok(attr == TRUE, "attr = %x\n", attr);
    attr = SetBkMode(hdc, OPAQUE);
    ok(attr == TRUE, "attr = %x\n", attr);

    mf = CloseMetaFile(hdc);
    ok(mf != 0, "CloseEnhMetaFile failed\n");
    DeleteMetaFile(mf);
}

static void test_enhmetafile_file(void)
{
    char temp_path[MAX_PATH];
    char mf_name[MAX_PATH];
    char buf[4096];
    HENHMETAFILE metafile;
    POINT pts[4];
    DWORD size;
    HANDLE file;
    HDC dc;
    BOOL ret;

    GetTempPathA(MAX_PATH, temp_path);
    GetTempFileNameA(temp_path, "wmf", 0, mf_name);

    dc = CreateEnhMetaFileA(NULL, mf_name, NULL, NULL);
    ok(dc != 0, "CreateMetaFileA(NULL) error %ld\n", GetLastError());

    file = CreateFileA(mf_name, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                       OPEN_EXISTING, 0, NULL);
    ok(file == INVALID_HANDLE_VALUE && GetLastError() == ERROR_SHARING_VIOLATION,
       "CreateFile returned: %p %lu\n", file, GetLastError());

    file = CreateFileA(mf_name, 0, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                       OPEN_EXISTING, 0, NULL);
    ok(file != INVALID_HANDLE_VALUE, "CreateFile failed: %lu\n", GetLastError());

    size = GetFileSize(file, NULL);
    ok(!size, "size = %lu\n", size);

    pts[0].x = pts[0].y = 10;
    pts[1].x = pts[1].y = 20;
    pts[2].x = pts[2].y = 15;
    pts[3].x = pts[3].y = 25;
    ret = PolyBezierTo(dc, pts, 3);  /* EMR_POLYBEZIERTO16 */
    ok( ret, "PolyBezierTo failed\n" );
    ret = PolyBezier(dc, pts, 4);    /* EMR_POLYBEZIER16   */
    ok( ret, "PolyBezier failed\n" );

    pts[0].x = pts[0].y = 32769;
    ret = PolyBezier(dc, pts, 4);    /* EMR_POLYBEZIER   */
    ok( ret, "PolyBezier failed\n" );
    ret = PolyBezierTo(dc, pts, 3);  /* EMR_POLYBEZIERTO */
    ok( ret, "PolyBezierTo failed\n" );

    size = GetFileSize(file, NULL);
    ok(!size, "size = %lu\n", size);

    metafile = CloseEnhMetaFile(dc);
    size = GetFileSize(file, NULL);
    ok(size == sizeof(EMF_BEZIER_BITS), "size = %lu\n", size);

    CloseHandle(file);
    file = CreateFileA(mf_name, GENERIC_WRITE, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                       OPEN_EXISTING, 0, NULL);
    todo_wine
    ok(file == INVALID_HANDLE_VALUE && GetLastError() == ERROR_SHARING_VIOLATION,
       "CreateFile returned: %p %lu\n", file, GetLastError());
    if (file != INVALID_HANDLE_VALUE) CloseHandle(file);
    file = CreateFileA(mf_name, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
                       OPEN_EXISTING, 0, NULL);
    ok(file != INVALID_HANDLE_VALUE, "CreateFile failed: %lu\n", GetLastError());
    ret = ReadFile(file, buf, sizeof(buf), &size, NULL);
    ok(ret, "ReadFile failed: %lu\n", GetLastError());
    ok(size == sizeof(EMF_BEZIER_BITS), "size = %lu\n", size);
    CloseHandle(file);

    if (compare_emf_bits(metafile, EMF_BEZIER_BITS, sizeof(EMF_BEZIER_BITS), "emf_Bezier", FALSE) != 0)
    {
        dump_emf_bits(metafile, "emf_Bezier");
        dump_emf_records(metafile, "emf_Bezier");
    }

    ret = DeleteEnhMetaFile(metafile);
    ok(ret, "Could not delete emf: %lu\n", GetLastError());

    SetLastError(0xdeadbeef);
    ret = DeleteEnhMetaFile(metafile);
    ok(!ret, "DeleteEnhMetaFile succeeded\n");
    ok(GetLastError() == ERROR_INVALID_HANDLE, "GetLastError() = %lu\n", GetLastError());

    SetLastError(0xdeadbeef);
    metafile = CloseEnhMetaFile(dc);
    ok(!metafile && GetLastError() == ERROR_INVALID_HANDLE, "CloseMetaFile returned %p (%lu)\n",
       metafile, GetLastError());

    ret = DeleteFileA(mf_name);
    ok(ret, "Could not delete file: %lu\n", GetLastError());

    SetLastError(0xdeadbeef);
    ret = DeleteEnhMetaFile(ULongToHandle(0xdeadbeef));
    ok(!ret, "DeleteEnhMetaFile succeeded\n");
    ok(GetLastError() == ERROR_INVALID_HANDLE, "GetLastError() = %lu\n", GetLastError());

    SetLastError(0xdeadbeef);
    size = GetEnhMetaFileBits(ULongToHandle(0xdeadbeef), 0, NULL);
    ok(!size, "GetEnhMetaFileBitsEx returned %lu\n", size);
    ok(GetLastError() == ERROR_INVALID_HANDLE, "GetLastError() = %lu\n", GetLastError());
}

static void test_emf_SetPixel(void)
{
    HENHMETAFILE emf;
    COLORREF c;
    BOOL ret;
    HDC hdc;

    hdc = CreateEnhMetaFileW(NULL, NULL, NULL, NULL);
    ok(hdc != 0, "CreateEnhMetaFile failed\n");

    c = GetPixel(hdc, 5, 5);
    ok(c == CLR_INVALID, "c = %lx\n", c);

    c = SetPixel(hdc, 5, 5, RGB(1,2,3));
    ok(c == CLR_INVALID, "c = %lx\n", c);

    ret = SetPixelV(hdc, 5, 5, RGB(1,2,3));
    ok(!ret, "ret = %x\n", ret);

    c = GetPixel(hdc, 5, 5);
    ok(c == CLR_INVALID, "c = %lx\n", c);

    ret = Rectangle(hdc, 1, 1, 10, 10);
    ok(ret, "Rectangle failed\n");

    c = GetPixel(hdc, 5, 5);
    ok(c == CLR_INVALID, "c = %lx\n", c);

    emf = CloseEnhMetaFile(hdc);
    ok(emf != 0, "CloseEnhMetaFile failed\n");
    DeleteEnhMetaFile(emf);
}

static void test_emf_attrs(void)
{
    HENHMETAFILE mf;
    UINT attr;
    HDC hdc;

    hdc = CreateEnhMetaFileW(NULL, NULL, NULL, NULL);
    ok(hdc != 0, "CreateEnhMetaFile failed\n");

    attr = SetTextAlign(hdc, TA_BOTTOM);
    ok(attr == 0, "attr = %x\n", attr);
    attr = SetTextAlign(hdc, TA_TOP);
    ok(attr == TA_BOTTOM, "attr = %x\n", attr);

    attr = SetBkMode(hdc, TRANSPARENT);
    ok(attr == OPAQUE, "attr = %x\n", attr);
    attr = SetBkMode(hdc, OPAQUE);
    ok(attr == TRANSPARENT, "attr = %x\n", attr);

    mf = CloseEnhMetaFile(hdc);
    ok(mf != 0, "CloseEnhMetaFile failed\n");
    DeleteEnhMetaFile(mf);
}

static void test_CopyMetaFile(void)
{
    HDC hdcMetafile;
    HMETAFILE hMetafile, hmf_copy;
    BOOL ret;
    char temp_path[MAX_PATH];
    char mf_name[MAX_PATH];
    INT type;

    hdcMetafile = CreateMetaFileA(NULL);
    ok(hdcMetafile != 0, "CreateMetaFileA(NULL) error %ld\n", GetLastError());

    hMetafile = CloseMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseMetaFile error %ld\n", GetLastError());
    type = GetObjectType(hMetafile);
    ok(type == OBJ_METAFILE, "CloseMetaFile created object with type %d\n", type);

    if (compare_mf_bits (hMetafile, MF_BLANK_BITS, sizeof(MF_BLANK_BITS),
        "mf_blank") != 0)
    {
        dump_mf_bits(hMetafile, "mf_Blank");
        EnumMetaFile(0, hMetafile, mf_enum_proc, 0);
    }

    GetTempPathA(MAX_PATH, temp_path);
    GetTempFileNameA(temp_path, "wmf", 0, mf_name);

    hmf_copy = CopyMetaFileA(hMetafile, mf_name);
    ok(hmf_copy != 0, "CopyMetaFile error %ld\n", GetLastError());

    type = GetObjectType(hmf_copy);
    ok(type == OBJ_METAFILE, "CopyMetaFile created object with type %d\n", type);

    ret = DeleteMetaFile(hMetafile);
    ok( ret, "DeleteMetaFile(%p) error %ld\n", hMetafile, GetLastError());

    if (compare_mf_disk_bits(mf_name, MF_BLANK_BITS, sizeof(MF_BLANK_BITS), "mf_blank") != 0)
    {
        dump_mf_bits(hmf_copy, "mf_Blank");
        EnumMetaFile(0, hmf_copy, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hmf_copy);
    ok( ret, "DeleteMetaFile(%p) error %ld\n", hmf_copy, GetLastError());

    DeleteFileA(mf_name);
}

static void test_SetMetaFileBits(void)
{
    HMETAFILE hmf;
    INT type;
    BOOL ret;
    BYTE buf[256];
    METAHEADER *mh;

    hmf = SetMetaFileBitsEx(sizeof(MF_GRAPHICS_BITS), MF_GRAPHICS_BITS);
    ok(hmf != 0, "SetMetaFileBitsEx error %ld\n", GetLastError());
    type = GetObjectType(hmf);
    ok(type == OBJ_METAFILE, "SetMetaFileBitsEx created object with type %d\n", type);

    if (compare_mf_bits(hmf, MF_GRAPHICS_BITS, sizeof(MF_GRAPHICS_BITS), "mf_Graphics") != 0)
    {
        dump_mf_bits(hmf, "mf_Graphics");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hmf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hmf, GetLastError());

    /* NULL data crashes XP SP1 */
    /*hmf = SetMetaFileBitsEx(sizeof(MF_GRAPHICS_BITS), NULL);*/

    /* Now with zero size */
    SetLastError(0xdeadbeef);
    hmf = SetMetaFileBitsEx(0, MF_GRAPHICS_BITS);
    ok(!hmf, "SetMetaFileBitsEx should fail\n");
    ok(GetLastError() == ERROR_INVALID_DATA, "wrong error %ld\n", GetLastError());

    /* Now with odd size */
    SetLastError(0xdeadbeef);
    hmf = SetMetaFileBitsEx(sizeof(MF_GRAPHICS_BITS) - 1, MF_GRAPHICS_BITS);
    ok(!hmf, "SetMetaFileBitsEx should fail\n");
    ok(GetLastError() == 0xdeadbeef /* XP SP1 */, "wrong error %ld\n", GetLastError());

    /* Now with zeroed out header fields */
    assert(sizeof(buf) >= sizeof(MF_GRAPHICS_BITS));
    memcpy(buf, MF_GRAPHICS_BITS, sizeof(MF_GRAPHICS_BITS));
    mh = (METAHEADER *)buf;
    /* corruption of any of the below fields leads to a failure */
    mh->mtType = 0;
    mh->mtVersion = 0;
    mh->mtHeaderSize = 0;
    SetLastError(0xdeadbeef);
    hmf = SetMetaFileBitsEx(sizeof(MF_GRAPHICS_BITS), buf);
    ok(!hmf, "SetMetaFileBitsEx should fail\n");
    ok(GetLastError() == ERROR_INVALID_DATA, "wrong error %ld\n", GetLastError());

    /* Now with corrupted mtSize field */
    memcpy(buf, MF_GRAPHICS_BITS, sizeof(MF_GRAPHICS_BITS));
    mh = (METAHEADER *)buf;
    /* corruption of mtSize doesn't lead to a failure */
    mh->mtSize *= 2;
    hmf = SetMetaFileBitsEx(sizeof(MF_GRAPHICS_BITS), buf);
    ok(hmf != 0, "SetMetaFileBitsEx error %ld\n", GetLastError());

    if (compare_mf_bits(hmf, MF_GRAPHICS_BITS, sizeof(MF_GRAPHICS_BITS), "mf_Graphics") != 0)
    {
        dump_mf_bits(hmf, "mf_Graphics");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hmf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hmf, GetLastError());

#ifndef _WIN64 /* Generates access violation on XP x64 and Win2003 x64 */
    /* Now with zeroed out mtSize field */
    memcpy(buf, MF_GRAPHICS_BITS, sizeof(MF_GRAPHICS_BITS));
    mh = (METAHEADER *)buf;
    /* zeroing mtSize doesn't lead to a failure */
    mh->mtSize = 0;
    hmf = SetMetaFileBitsEx(sizeof(MF_GRAPHICS_BITS), buf);
    ok(hmf != 0, "SetMetaFileBitsEx error %ld\n", GetLastError());

    if (compare_mf_bits(hmf, MF_GRAPHICS_BITS, sizeof(MF_GRAPHICS_BITS), "mf_Graphics") != 0)
    {
        dump_mf_bits(hmf, "mf_Graphics");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hmf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hmf, GetLastError());
#endif
}

/* Simple APIs from mfdrv/graphics.c
 */

static void test_mf_Graphics(void)
{
    HDC hdcMetafile;
    HMETAFILE hMetafile;
    POINT oldpoint;
    INT type;
    BOOL ret;

    static const POINT points[] = { {1, 1}, {2, 2} };
    static const BYTE types[] = { PT_MOVETO, PT_LINETO };

    hdcMetafile = CreateMetaFileA(NULL);
    ok(hdcMetafile != 0, "CreateMetaFileA(NULL) error %ld\n", GetLastError());

    ret = MoveToEx(hdcMetafile, 1, 1, NULL);
    ok( ret, "MoveToEx error %ld.\n", GetLastError());
    ret = LineTo(hdcMetafile, 2, 2);
    ok( ret, "LineTo error %ld.\n", GetLastError());

    oldpoint.x = oldpoint.y = 0xdeadbeef;
    ret = MoveToEx(hdcMetafile, 1, 1, &oldpoint);
    ok( ret, "MoveToEx error %ld.\n", GetLastError());
    ok(oldpoint.x == 0xdeadbeef && oldpoint.y == 0xdeadbeef,
       "MoveToEx modified oldpoint\n");

    ret = Ellipse(hdcMetafile, 0, 0, 2, 2);
    ok( ret, "Ellipse error %ld.\n", GetLastError());

    ret = ArcTo(hdcMetafile, 1, 2, 30, 40, 11, 12, 23, 24 );
    ok( !ret, "ArcTo succeeded\n" );

    SetLastError(0xdeadbeef);
    ret = PolyDraw(hdcMetafile, points, types, ARRAY_SIZE(points));
    ok(!ret, "PolyDraw succeeded\n");
    todo_wine
    ok(GetLastError() == 0xdeadbeef, "GetLastError() = %lu\n", GetLastError());

    hMetafile = CloseMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseMetaFile error %ld\n", GetLastError());
    type = GetObjectType(hdcMetafile);
    ok(type == 0 || broken(type == OBJ_METADC) /* win10 >=1607 */,
       "CloseMetaFile has to destroy metafile hdc (%d)\n", type);

    if (compare_mf_bits (hMetafile, MF_GRAPHICS_BITS, sizeof(MF_GRAPHICS_BITS),
        "mf_Graphics") != 0)
    {
        dump_mf_bits(hMetafile, "mf_Graphics");
        EnumMetaFile(0, hMetafile, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hMetafile);
    ok( ret, "DeleteMetaFile(%p) error %ld\n",
        hMetafile, GetLastError());
}

static void test_mf_FloodFill(void)
{
    HMETAFILE mf;
    HDC hdc;
    BOOL ret;

    static const unsigned char floodfill_bits[] =
    {
        0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x24, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x48, 0x05,
        0x00, 0x00, 0x03, 0x04, 0x05, 0x00, 0x02, 0x00,
        0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x48, 0x05,
        0x00, 0x00, 0x08, 0x09, 0x0a, 0x00, 0x07, 0x00,
        0x06, 0x00, 0x08, 0x00, 0x00, 0x00, 0x48, 0x05,
        0x01, 0x00, 0x12, 0x13, 0x14, 0x00, 0x11, 0x00,
        0x10, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    hdc = CreateMetaFileA(NULL);
    ok(hdc != 0, "CreateMetaFileA(NULL) error %ld\n", GetLastError());

    ret = FloodFill(hdc, 1, 2, RGB(3,4,5));
    ok(ret, "FloodFill failed\n");

    ret = ExtFloodFill(hdc, 6, 7, RGB(8,9,10), FLOODFILLBORDER);
    ok(ret, "FloodFill failed\n");

    ret = ExtFloodFill(hdc, 16, 17, RGB(18,19,20), FLOODFILLSURFACE);
    ok(ret, "FloodFill failed\n");

    mf = CloseMetaFile(hdc);
    ok(mf != 0, "CloseMetaFile error %ld\n", GetLastError());

    if (compare_mf_bits (mf, floodfill_bits, sizeof(floodfill_bits),
        "mf_FloodFill") != 0)
    {
        dump_mf_bits(mf, "mf_FloodFill");
        EnumMetaFile(0, mf, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(mf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", mf, GetLastError());
}

static void test_mf_PatternBrush(void)
{
    char buffer[sizeof(BITMAPINFOHEADER) + (2 + 32 * 32 / 8) * sizeof(RGBQUAD)];
    BITMAPINFO *info = (BITMAPINFO *)buffer;
    HDC hdcMetafile;
    HMETAFILE hMetafile;
    LOGBRUSH *orig_lb;
    HBRUSH hBrush, dib_brush;
    DWORD *bits;
    unsigned int i;
    BOOL ret;
    INT type;

    orig_lb = HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof(LOGBRUSH));

    orig_lb->lbStyle = BS_PATTERN;
    orig_lb->lbColor = RGB(0, 0, 0);
    orig_lb->lbHatch = (ULONG_PTR)CreateBitmap (8, 8, 1, 1, SAMPLE_PATTERN_BRUSH);
    ok((HBITMAP)orig_lb->lbHatch != NULL, "CreateBitmap error %ld.\n", GetLastError());

    hBrush = CreateBrushIndirect (orig_lb);
    ok(hBrush != 0, "CreateBrushIndirect error %ld\n", GetLastError());

    hdcMetafile = CreateMetaFileA(NULL);
    ok(hdcMetafile != 0, "CreateMetaFileA error %ld\n", GetLastError());

    hBrush = SelectObject(hdcMetafile, hBrush);
    ok(hBrush != 0, "SelectObject error %ld.\n", GetLastError());

    memset(info, 0, sizeof(buffer));
    info->bmiHeader.biSize = sizeof(info->bmiHeader);
    info->bmiHeader.biHeight = 32;
    info->bmiHeader.biWidth = 32;
    info->bmiHeader.biBitCount = 1;
    info->bmiHeader.biPlanes = 1;
    info->bmiHeader.biCompression = BI_RGB;
    bits = (DWORD *)info->bmiColors;
    for (i = 0; i < 32 * 32 / 8; i++) bits[i] = i;

    dib_brush = CreateDIBPatternBrushPt(info, DIB_RGB_COLORS);
    ok(dib_brush != NULL, "CreatePatternBrush failed\n");

    dib_brush = SelectObject(hdcMetafile, dib_brush);
    ok(dib_brush != 0, "SelectObject error %ld.\n", GetLastError());

    hMetafile = CloseMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseMetaFile error %ld\n", GetLastError());
    type = GetObjectType(hdcMetafile);
    ok(type == 0 || broken(type == OBJ_METADC) /* win10 >=1607 */,
       "CloseMetaFile has to destroy metafile hdc (%d)\n", type);

    if (compare_mf_bits (hMetafile, MF_PATTERN_BRUSH_BITS, sizeof(MF_PATTERN_BRUSH_BITS),
        "mf_Pattern_Brush") != 0)
    {
        dump_mf_bits(hMetafile, "mf_Pattern_Brush");
        EnumMetaFile(0, hMetafile, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hMetafile);
    ok( ret, "DeleteMetaFile error %ld\n", GetLastError());
    ret = DeleteObject(hBrush);
    ok( ret, "DeleteObject(HBRUSH) error %ld\n", GetLastError());
    ret = DeleteObject(dib_brush);
    ok(ret, "DeleteObject failed\n");
    ret = DeleteObject((HBITMAP)orig_lb->lbHatch);
    ok( ret, "DeleteObject(HBITMAP) error %ld\n",
        GetLastError());
    HeapFree (GetProcessHeap(), 0, orig_lb);
}

static int CALLBACK pattern_brush_emf_enum_proc(HDC hdc, HANDLETABLE *htable,
        const ENHMETARECORD *rec, int n, LPARAM arg)
{
    LOGBRUSH brush;
    BOOL ret;

    switch (rec->iType)
    {
    case EMR_HEADER:
    case EMR_SELECTOBJECT:
    case EMR_EOF:
    case EMR_CREATEMONOBRUSH:
        return 1;
    case EMR_CREATEDIBPATTERNBRUSHPT:
        ok(!htable->objectHandle[2], "objectHandle[2] already used\n");
        ret = PlayEnhMetaFileRecord(hdc, htable, rec, n);
        ok(ret, "PlayEnhMetaFileRecord failed\n");
        ok(htable->objectHandle[2] != NULL, "objectHandle[2] not created\n");

        ret = GetObjectW(htable->objectHandle[2], sizeof(brush), &brush);
        ok(ret, "GetObjectW failed\n");
        ok(brush.lbStyle == BS_DIBPATTERN, "brush.lbStyle = %d\n", brush.lbStyle);
        ok(brush.lbHatch > (ULONG_PTR)rec && brush.lbHatch < (ULONG_PTR)rec + rec->nSize,
                "brush.lbHatch = %p, not in %p-%p range\n",
                (void *)brush.lbHatch, rec, (const BYTE *)rec + rec->nSize);
        return 1;
    default:
        ok(0, "unexpected record %lu\n", rec->iType);
        return 0;
    }
}

static void test_emf_pattern_brush(void)
{
    char buffer[sizeof(BITMAPINFOHEADER) + (2 + 32 * 32 / 8) * sizeof(RGBQUAD)];
    BITMAPINFO *info = (BITMAPINFO *)buffer;
    HDC hdc;
    HENHMETAFILE emf;
    LOGBRUSH *orig_lb;
    HBRUSH bitmap_brush, dib_brush;
    DWORD *bits;
    unsigned int i;
    BOOL ret;

    orig_lb = HeapAlloc(GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof(LOGBRUSH));

    orig_lb->lbStyle = BS_PATTERN;
    orig_lb->lbColor = RGB(0, 0, 0);
    orig_lb->lbHatch = (ULONG_PTR)CreateBitmap(8, 8, 1, 1, SAMPLE_PATTERN_BRUSH);
    ok((HBITMAP)orig_lb->lbHatch != NULL, "CreateBitmap error %ld.\n", GetLastError());

    bitmap_brush = CreateBrushIndirect(orig_lb);
    ok(bitmap_brush != 0, "CreateBrushIndirect error %ld\n", GetLastError());

    hdc = CreateEnhMetaFileA(NULL, NULL, NULL, NULL);
    ok(hdc != 0, "CreateMetaFileA error %ld\n", GetLastError());

    bitmap_brush = SelectObject(hdc, bitmap_brush);
    ok(bitmap_brush != 0, "SelectObject error %ld.\n", GetLastError());

    memset(info, 0, sizeof(buffer));
    info->bmiHeader.biSize = sizeof(info->bmiHeader);
    info->bmiHeader.biHeight = 32;
    info->bmiHeader.biWidth = 32;
    info->bmiHeader.biBitCount = 1;
    info->bmiHeader.biPlanes = 1;
    info->bmiHeader.biCompression = BI_RGB;
    bits = (DWORD *)info->bmiColors;
    for (i = 0; i < 32 * 32 / 8; i++) bits[i] = i;

    dib_brush = CreateDIBPatternBrushPt(info, DIB_RGB_COLORS);
    ok(dib_brush != NULL, "CreatePatternBrush failed\n");

    dib_brush = SelectObject(hdc, dib_brush);
    ok(dib_brush != 0, "SelectObject error %ld.\n", GetLastError());

    emf = CloseEnhMetaFile(hdc);
    ok(emf != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    if (compare_emf_bits(emf, emf_pattern_brush_bits, sizeof(emf_pattern_brush_bits),
                         "emf_pattern_brush", FALSE))
    {
        dump_emf_bits(emf, "emf_pattern_brush");
        dump_emf_records(emf, "emf_pattern_brush");
    }

    ret = DeleteObject(bitmap_brush);
    ok(ret, "DeleteObject failed\n");
    ret = DeleteObject(dib_brush);
    ok(ret, "DeleteObject failed\n");
    ret = DeleteObject((HBITMAP)orig_lb->lbHatch);
    ok(ret, "DeleteObject failed\n");
    HeapFree(GetProcessHeap(), 0, orig_lb);

    ret = EnumEnhMetaFile(NULL, emf, pattern_brush_emf_enum_proc, NULL, NULL);
    ok(ret, "EnumEnhMetaFile error %ld\n", GetLastError());
    ret = DeleteEnhMetaFile(emf);
    ok(ret, "DeleteMetaFile error %ld\n", GetLastError());
}

static void test_mf_palette_brush(void)
{
    char buffer[sizeof(BITMAPINFOHEADER) + 256 * sizeof(RGBQUAD) + 16 * 16];
    BITMAPINFO *info = (BITMAPINFO *)buffer;
    WORD *indices = (WORD *)info->bmiColors;
    char pal_buffer[sizeof(LOGPALETTE) + 256 * sizeof(PALETTEENTRY)];
    LOGPALETTE *pal = (LOGPALETTE *)pal_buffer;
    HDC hdc = CreateMetaFileA(NULL);
    HMETAFILE metafile;
    HBRUSH brush;
    int i;
    HPALETTE palette;

    memset( info, 0, sizeof(*info) );
    info->bmiHeader.biSize        = sizeof(info->bmiHeader);
    info->bmiHeader.biWidth       = 16;
    info->bmiHeader.biHeight      = 16;
    info->bmiHeader.biPlanes      = 1;
    info->bmiHeader.biBitCount    = 8;
    info->bmiHeader.biCompression = BI_RGB;

    for (i = 0; i < 256; i++) indices[i] = 255 - i;
    for (i = 0; i < 256; i++) ((BYTE *)(indices + 256))[i] = i;
    brush = CreateDIBPatternBrushPt(info, DIB_PAL_COLORS);
    ok( brush != NULL, "CreateDIBPatternBrushPt failed\n" );

    pal->palVersion = 0x300;
    pal->palNumEntries = 256;
    for (i = 0; i < 256; i++)
    {
        pal->palPalEntry[i].peRed = i * 2;
        pal->palPalEntry[i].peGreen = i * 2;
        pal->palPalEntry[i].peBlue = i * 2;
        pal->palPalEntry[i].peFlags = 0;
    }
    palette = CreatePalette(pal);

    ok(SelectPalette(hdc, palette, 0) != NULL, "SelectPalette failed\n");
    ok(SelectObject(hdc, brush) != NULL, "SelectObject failed\n");

    DeleteObject(brush);
    DeleteObject(palette);

    metafile = CloseMetaFile(hdc);
    if (compare_mf_bits(metafile, mf_palette_brush_bits, sizeof(mf_palette_brush_bits),
        "mf_Palette_Pattern_Brush") != 0)
    {
        dump_mf_bits(metafile, "mf_Palette_Pattern_Brush");
        EnumMetaFile(0, metafile, mf_enum_proc, 0);
    }
    DeleteObject(metafile);
}

static void test_mf_DCBrush(void)
{
    HDC hdcMetafile;
    HMETAFILE hMetafile;
    HBRUSH hBrush;
    HPEN hPen;
    BOOL ret;
    COLORREF color;

    if (!pSetDCBrushColor || !pSetDCPenColor)
    {
        win_skip( "SetDCBrush/PenColor not supported\n" );
        return;
    }

    hdcMetafile = CreateMetaFileA(NULL);
    ok( hdcMetafile != 0, "CreateMetaFileA failed\n" );

    hBrush = SelectObject(hdcMetafile, GetStockObject(DC_BRUSH));
    ok(hBrush != 0, "SelectObject error %ld.\n", GetLastError());

    hPen = SelectObject(hdcMetafile, GetStockObject(DC_PEN));
    ok(hPen != 0, "SelectObject error %ld.\n", GetLastError());

    color = pSetDCBrushColor( hdcMetafile, RGB(0x55,0x55,0x55) );
    ok( color == CLR_INVALID, "SetDCBrushColor returned %lx\n", color );

    color = pSetDCPenColor( hdcMetafile, RGB(0x33,0x44,0x55) );
    ok( color == CLR_INVALID, "SetDCPenColor returned %lx\n", color );

    Rectangle( hdcMetafile, 10, 10, 20, 20 );

    color = pSetDCBrushColor( hdcMetafile, RGB(0x12,0x34,0x56) );
    ok( color == CLR_INVALID, "SetDCBrushColor returned %lx\n", color );

    hMetafile = CloseMetaFile(hdcMetafile);
    ok( hMetafile != 0, "CloseMetaFile failed\n" );

    if (compare_mf_bits(hMetafile, MF_DCBRUSH_BITS, sizeof(MF_DCBRUSH_BITS), "mf_DCBrush") != 0)
    {
        dump_mf_bits(hMetafile, "mf_DCBrush");
        EnumMetaFile(0, hMetafile, mf_enum_proc, 0);
    }
    ret = DeleteMetaFile(hMetafile);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hMetafile, GetLastError());
}

static void test_mf_select(void)
{
    HMETAFILE hmf;
    HDC hdc, hdc2;
    HGDIOBJ obj;
    HPEN pen;
    int cnt;
    BOOL ret;

    static const unsigned char select_bits[] =
    {
        0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x4d, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0xfc, 0x02,
        0x00, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x2d, 0x01, 0x00, 0x00,
        0x07, 0x00, 0x00, 0x00, 0xfc, 0x02, 0x00, 0x00,
        0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x2d, 0x01, 0x01, 0x00, 0x08, 0x00,
        0x00, 0x00, 0xfa, 0x02, 0x00, 0x00, 0x01, 0x00,
        0x00, 0x00, 0x01, 0x01, 0x01, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x2d, 0x01, 0x02, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x1e, 0x00, 0x08, 0x00, 0x00, 0x00,
        0xfa, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x2d, 0x01, 0x03, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x27, 0x01, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00,
        0x2d, 0x01, 0x03, 0x00, 0x04, 0x00, 0x00, 0x00,
        0xf0, 0x01, 0x02, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x27, 0x01, 0xfb, 0xff, 0x03, 0x00, 0x00, 0x00,
        0x00, 0x00
    };

    static const unsigned char delete_not_selected_bits[] =
    {
        0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x28, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0xfa, 0x02,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01,
        0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x2d, 0x01,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0xfa, 0x02,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x2d, 0x01,
        0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0xf0, 0x01,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    static const unsigned char delete_selected_bits[] =
    {
        0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x18, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0xfa, 0x02,
        0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01,
        0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x2d, 0x01,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    hdc = CreateMetaFileA(NULL);
    ok(hdc != 0, "CreateMetaFileA failed\n");

    obj = SelectObject(hdc, GetStockObject(DC_BRUSH));
    ok(obj == GetStockObject(WHITE_BRUSH), "brush is not a stock WHITE_BRUSH: %p\n", obj);

    obj = SelectObject(hdc, GetStockObject(WHITE_BRUSH));
    ok(obj == GetStockObject(DC_BRUSH), "brush is not a stock DC_BRUSH: %p\n", obj);

    pen = CreatePen(PS_SOLID, 1, RGB(1,1,1));
    obj = SelectObject(hdc, pen);
    ok(obj == GetStockObject(BLACK_PEN), "pen is not a stock BLACK_PEN: %p\n", obj);

    cnt = SaveDC(hdc);
    ok(cnt == 1, "cnt = %d\n", cnt);

    obj = SelectObject(hdc, GetStockObject(BLACK_PEN));
    ok(obj == pen, "unexpected pen: %p\n", obj);

    ret = RestoreDC(hdc, -1);
    ok(ret, "RestoreDC failed\n");

    obj = SelectObject(hdc, GetStockObject(BLACK_PEN));
    /* pen is still black after RestoreDC */
    ok(obj == GetStockObject(BLACK_PEN), "unexpected pen: %p\n", obj);
    ret = DeleteObject(pen);
    ok(ret, "DeleteObject failed: %lu\n", GetLastError());

    obj = GetCurrentObject(hdc, OBJ_PEN);
    ok(!obj, "GetCurrentObject succeeded\n");

    SetLastError(0xdeadbeef);
    obj = SelectObject(hdc, GetStockObject(DEFAULT_PALETTE));
    ok(!obj && GetLastError() == ERROR_INVALID_FUNCTION,
       "SelectObject returned %p (%lu).\n", obj, GetLastError());

    ret = RestoreDC(hdc, -5);
    ok(ret, "RestoreDC failed\n");

    hmf = CloseMetaFile(hdc);
    ok(hmf != 0, "CloseMetaFile failed\n");

    if (compare_mf_bits(hmf, select_bits, sizeof(select_bits), "mf_select"))
    {
        dump_mf_bits(hmf, "mf_select");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hmf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hmf, GetLastError());

    /* create two metafiles, select the same pen to both of them,
     * unselect it from only one and then delete */
    hdc = CreateMetaFileA(NULL);
    ok(hdc != 0, "CreateMetaFileA failed\n");
    hdc2 = CreateMetaFileA(NULL);
    ok(hdc2 != 0, "CreateMetaFileA failed\n");

    pen = CreatePen(PS_SOLID, 1, RGB(1,1,1));
    obj = SelectObject(hdc, pen);
    ok(obj == GetStockObject(BLACK_PEN), "pen is not a stock BLACK_PEN: %p\n", obj);
    obj = SelectObject(hdc2, pen);
    ok(obj == GetStockObject(BLACK_PEN), "pen is not a stock BLACK_PEN: %p\n", obj);

    obj = SelectObject(hdc, GetStockObject(BLACK_PEN));
    ok(obj == pen, "unexpected pen: %p\n", obj);
    ret = DeleteObject(pen);
    ok(ret, "DeleteObject failed: %lu\n", GetLastError());

    hmf = CloseMetaFile(hdc);
    ok(hmf != 0, "CloseMetaFile failed\n");
    if (compare_mf_bits(hmf, delete_not_selected_bits,
                        sizeof(delete_not_selected_bits), "mf_delete_not_selected"))
    {
        dump_mf_bits(hmf, "mf_delete_not_selected");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }
    ret = DeleteMetaFile(hmf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hmf, GetLastError());

    hmf = CloseMetaFile(hdc2);
    ok(hmf != 0, "CloseMetaFile failed\n");
    if (compare_mf_bits(hmf, delete_selected_bits,
                        sizeof(delete_selected_bits), "mf_delete_selected"))
    {
        dump_mf_bits(hmf, "mf_delete_selected");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }
    ret = DeleteMetaFile(hmf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hmf, GetLastError());
}

static void test_emf_select(void)
{
    HENHMETAFILE hemf;
    HDC hdc, hdc2;
    HGDIOBJ obj;
    HPEN pen;
    int cnt;
    BOOL ret;

    static const unsigned char select_bits[] =
    {
        0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xe6, 0xff, 0xff, 0xff, 0xe6, 0xff, 0xff, 0xff,
        0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
        0xf8, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x09, 0x06, 0x00, 0x00, 0x07, 0x04, 0x00, 0x00,
        0x99, 0x01, 0x00, 0x00, 0x11, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xcd, 0x3c, 0x06, 0x00,
        0x91, 0x29, 0x04, 0x00, 0x25, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x80,
        0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x80, 0x26, 0x00, 0x00, 0x00,
        0x1c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x00,
        0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x00,
        0x08, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x80,
        0x22, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0x25, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x80,
        0x28, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
        0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
    };

    static const unsigned char delete_not_selected_bits[] =
    {
        0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xe6, 0xff, 0xff, 0xff, 0xe6, 0xff, 0xff, 0xff,
        0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
        0xc0, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x09, 0x06, 0x00, 0x00, 0x07, 0x04, 0x00, 0x00,
        0x99, 0x01, 0x00, 0x00, 0x11, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xcd, 0x3c, 0x06, 0x00,
        0x91, 0x29, 0x04, 0x00, 0x26, 0x00, 0x00, 0x00,
        0x1c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x00,
        0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x80,
        0x28, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
        0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
    };

    static const unsigned char delete_selected_bits[] =
    {
        0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xe6, 0xff, 0xff, 0xff, 0xe6, 0xff, 0xff, 0xff,
        0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
        0xb4, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x09, 0x06, 0x00, 0x00, 0x07, 0x04, 0x00, 0x00,
        0x99, 0x01, 0x00, 0x00, 0x11, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0xcd, 0x3c, 0x06, 0x00,
        0x91, 0x29, 0x04, 0x00, 0x26, 0x00, 0x00, 0x00,
        0x1c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x00,
        0x25, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
        0x14, 0x00, 0x00, 0x00
    };

    hdc = CreateEnhMetaFileA(NULL, NULL, NULL, NULL);
    ok(hdc != 0, "CreateEnhMetaFileA failed\n");

    obj = SelectObject(hdc, GetStockObject(DC_BRUSH));
    ok(obj == GetStockObject(WHITE_BRUSH), "brush is not a stock WHITE_BRUSH: %p\n", obj);

    obj = SelectObject(hdc, GetStockObject(WHITE_BRUSH));
    ok(obj == GetStockObject(DC_BRUSH), "brush is not a stock DC_BRUSH: %p\n", obj);

    pen = CreatePen(PS_SOLID, 1, RGB(1,1,1));
    obj = SelectObject(hdc, pen);
    ok(obj == GetStockObject(BLACK_PEN), "pen is not a stock BLACK_PEN: %p\n", obj);

    cnt = SaveDC(hdc);
    ok(cnt == 1, "cnt = %d\n", cnt);

    obj = SelectObject(hdc, GetStockObject(BLACK_PEN));
    ok(obj == pen, "unexpected pen: %p\n", obj);

    ret = RestoreDC(hdc, -1);
    ok(ret, "RestoreDC failed\n");

    obj = SelectObject(hdc, GetStockObject(BLACK_PEN));
    /* pen is still black after RestoreDC */
    ok(obj == pen, "unexpected pen: %p\n", obj);
    ret = DeleteObject(pen);
    ok(ret, "DeleteObject failed: %lu\n", GetLastError());

    obj = GetCurrentObject(hdc, OBJ_PEN);
    ok(obj == GetStockObject(BLACK_PEN), "GetCurrentObject returned %p\n", obj);

    SetLastError(0xdeadbeef);
    obj = SelectObject(hdc, GetStockObject(DEFAULT_PALETTE));
    ok(!obj, "SelectObject returned %p (%lu).\n", obj, GetLastError());

    ret = RestoreDC(hdc, -5);
    ok(!ret, "RestoreDC succeeded\n");

    hemf = CloseEnhMetaFile(hdc);
    ok(hemf != 0, "CloseEnhMetaFile failed\n");

    if (compare_emf_bits(hemf, select_bits, sizeof(select_bits), "emf_select", FALSE))
    {
        dump_emf_bits(hemf, "emf_select");
        dump_emf_records(hemf, "emf_select");
    }

    ret = DeleteEnhMetaFile(hemf);
    ok(ret, "DeleteEnhMetaFile(%p) error %ld\n", hemf, GetLastError());

    /* create two EMFs, select the same pen to both of them,
     * unselect it from only one and then delete */
    hdc = CreateEnhMetaFileA(NULL, NULL, NULL, NULL);
    ok(hdc != 0, "CreateEnhMetaFileA failed\n");
    hdc2 = CreateEnhMetaFileA(NULL, NULL, NULL, NULL);
    ok(hdc2 != 0, "CreateEnhMetaFileA failed\n");

    pen = CreatePen(PS_SOLID, 1, RGB(1,1,1));
    obj = SelectObject(hdc, pen);
    ok(obj == GetStockObject(BLACK_PEN), "pen is not a stock BLACK_PEN: %p\n", obj);
    obj = SelectObject(hdc2, pen);
    ok(obj == GetStockObject(BLACK_PEN), "pen is not a stock BLACK_PEN: %p\n", obj);

    obj = SelectObject(hdc, GetStockObject(BLACK_PEN));
    ok(obj == pen, "unexpected pen: %p\n", obj);
    ret = DeleteObject(pen);
    ok(ret, "DeleteObject failed: %lu\n", GetLastError());

    hemf = CloseEnhMetaFile(hdc);
    ok(hemf != 0, "CloseEnhMetaFile failed\n");
    if (compare_emf_bits(hemf, delete_not_selected_bits, sizeof(delete_not_selected_bits),
                         "emf_delete_not_selected", FALSE))
    {
        dump_emf_bits(hemf, "emf_delete_not_selected");
        dump_emf_records(hemf, "emf_delete_not_selected");
    }
    ret = DeleteEnhMetaFile(hemf);
    ok(ret, "DeleteEnhMetaFile(%p) error %ld\n", hemf, GetLastError());

    hemf = CloseEnhMetaFile(hdc2);
    ok(hemf != 0, "CloseEnhMetaFile failed\n");
    if (compare_emf_bits(hemf, delete_selected_bits, sizeof(delete_selected_bits),
                         "emf_delete_selected", FALSE))
    {
        dump_emf_bits(hemf, "emf_delete_selected");
        dump_emf_records(hemf, "emf_delete_selected");
    }
    ret = DeleteEnhMetaFile(hemf);
    ok(ret, "DeleteEnhMetaFile(%p) error %ld\n", hemf, GetLastError());
}

static const PALETTEENTRY logpalettedata[8] = {
    { 0x10, 0x20, 0x30, PC_NOCOLLAPSE },
    { 0x20, 0x30, 0x40, PC_NOCOLLAPSE },
    { 0x30, 0x40, 0x50, PC_NOCOLLAPSE },
    { 0x40, 0x50, 0x60, PC_NOCOLLAPSE },
    { 0x50, 0x60, 0x70, PC_NOCOLLAPSE },
    { 0x60, 0x70, 0x80, PC_NOCOLLAPSE },
    { 0x70, 0x80, 0x90, PC_NOCOLLAPSE },
    { 0x80, 0x90, 0xA0, PC_NOCOLLAPSE },
};

static void test_mf_palette(void)
{
    char logpalettebuf[sizeof(LOGPALETTE) + sizeof(logpalettedata)];
    LOGPALETTE *logpalette = (PLOGPALETTE)logpalettebuf;
    HPALETTE hpal, old_pal;
    HMETAFILE hmf;
    HDC hdc;
    BOOL ret;

    static const unsigned char palette_bits[] =
    {
        0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x5c, 0x00,
        0x00, 0x00, 0x02, 0x00, 0x2d, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0xf7, 0x00,
        0x00, 0x03, 0x08, 0x00, 0x10, 0x20, 0x30, 0x04,
        0x20, 0x30, 0x40, 0x04, 0x30, 0x40, 0x50, 0x04,
        0x40, 0x50, 0x60, 0x04, 0x50, 0x60, 0x70, 0x04,
        0x60, 0x70, 0x80, 0x04, 0x70, 0x80, 0x90, 0x04,
        0x80, 0x90, 0xa0, 0x04, 0x04, 0x00, 0x00, 0x00,
        0x34, 0x02, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x35, 0x00, 0x2d, 0x00, 0x00, 0x00, 0xf7, 0x00,
        0x00, 0x03, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00,
        0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00,
        0x80, 0x00, 0x80, 0x00, 0x00, 0x80, 0x80, 0x00,
        0xc0, 0xc0, 0xc0, 0x00, 0xc0, 0xdc, 0xc0, 0x00,
        0xa6, 0xca, 0xf0, 0x00, 0xff, 0xfb, 0xf0, 0x00,
        0xa0, 0xa0, 0xa4, 0x00, 0x80, 0x80, 0x80, 0x00,
        0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00,
        0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
        0xff, 0x00, 0xff, 0x00, 0x00, 0xff, 0xff, 0x00,
        0xff, 0xff, 0xff, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x34, 0x02, 0x01, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x35, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    hdc = CreateMetaFileA(NULL);
    ok( hdc != 0, "CreateMetaFileA failed\n" );

    /* Initialize the logical palette with a few colours */
    logpalette->palVersion = 0x300;
    logpalette->palNumEntries = ARRAY_SIZE(logpalettedata);
    memcpy( logpalette->palPalEntry, logpalettedata, sizeof(logpalettedata) );
    hpal = CreatePalette( logpalette );
    old_pal = SelectPalette( hdc, hpal, FALSE );
    ok(old_pal == ULongToHandle(TRUE), "old_pal=%p\n", old_pal);

    ret = RealizePalette(hdc);
    ok(ret, "RealizePalette failed\n");

    old_pal = SelectPalette( hdc, GetStockObject(DEFAULT_PALETTE), FALSE );
    ok(old_pal == ULongToHandle(TRUE), "old_pal=%p\n", old_pal);

    ret = RealizePalette(hdc);
    ok(ret, "RealizePalette failed\n");

    hmf = CloseMetaFile(hdc);
    ok( hmf != 0, "CloseMetaFile failed\n" );

    if (compare_mf_bits(hmf, palette_bits, sizeof(palette_bits), "mf_palette"))
    {
        dump_mf_bits(hmf, "mf_palette");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hmf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hmf, GetLastError());
}

static void test_emf_palette(void)
{
    char logpalettebuf[sizeof(LOGPALETTE) + sizeof(logpalettedata)];
    LOGPALETTE *logpalette = (PLOGPALETTE)logpalettebuf;
    HPALETTE hpal, old_pal, def_pal;
    HENHMETAFILE emf;
    HDC hdc;
    BOOL ret;

    static const unsigned char palette_bits[] =
    {
        0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xe6, 0xff, 0xff, 0xff, 0xe6, 0xff, 0xff, 0xff,
        0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
        0xf0, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x80, 0x07, 0x00, 0x00, 0xcd, 0x03, 0x00, 0x00,
        0xfc, 0x01, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
        0x9f, 0xed, 0x03, 0x00, 0x31, 0x00, 0x00, 0x00,
        0x30, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x03, 0x08, 0x00, 0x10, 0x20, 0x30, 0x00,
        0x20, 0x30, 0x40, 0x00, 0x30, 0x40, 0x50, 0x00,
        0x40, 0x50, 0x60, 0x00, 0x50, 0x60, 0x70, 0x00,
        0x60, 0x70, 0x80, 0x00, 0x70, 0x80, 0x90, 0x00,
        0x80, 0x90, 0xa0, 0x00, 0x30, 0x00, 0x00, 0x00,
        0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x30, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00,
        0x0f, 0x00, 0x00, 0x80, 0x0e, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x10, 0x20, 0x30, 0x00,
        0x20, 0x30, 0x40, 0x00, 0x30, 0x40, 0x50, 0x00,
        0x40, 0x50, 0x60, 0x00, 0x50, 0x60, 0x70, 0x00,
        0x60, 0x70, 0x80, 0x00, 0x70, 0x80, 0x90, 0x00,
        0x80, 0x90, 0xa0, 0x00, 0x34, 0x00, 0x00, 0x00
    };

    hdc = CreateEnhMetaFileA(NULL, NULL, NULL, NULL);
    ok(hdc != 0, "CreateMetaFileA error %ld\n", GetLastError());

    /* Initialize the logical palette with a few colours */
    logpalette->palVersion = 0x300;
    logpalette->palNumEntries = ARRAY_SIZE(logpalettedata);
    memcpy(logpalette->palPalEntry, logpalettedata, sizeof(logpalettedata));
    hpal = CreatePalette(logpalette);
    old_pal = SelectPalette(hdc, hpal, FALSE);
    def_pal = GetStockObject(DEFAULT_PALETTE);
    ok(old_pal == def_pal, "old_pal=%p, expected %p\n", old_pal, def_pal);

    ret = RealizePalette(hdc);
    ok(!ret, "RealizePalette returned %d\n", ret);

    old_pal = SelectPalette(hdc, GetStockObject(DEFAULT_PALETTE), FALSE);
    ok(old_pal == hpal, "old_pal=%p, expected %p\n", old_pal, hpal);

    ret = RealizePalette(hdc);
    ok(!ret, "RealizePalette returned %d\n", ret);

    emf = CloseEnhMetaFile(hdc);
    ok(emf != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    if (compare_emf_bits(emf, palette_bits, sizeof(palette_bits), "emf_palette", FALSE))
    {
        dump_emf_bits(emf, "emf_palette");
        dump_emf_records(emf, "emf_palette");
    }

    ret = DeleteEnhMetaFile(emf);
    ok(ret, "DeleteMetaFile error %ld\n", GetLastError());
}

static void test_mf_blit(void)
{
    BITMAPINFO bmi = {{ sizeof(bmi) }};
    HMETAFILE hmf;
    HBITMAP bitmap, old_bitmap;
    void *bits;
    HDC hdc, dib_hdc;
    unsigned int i;
    BOOL ret;

    static const unsigned char blit_bits[] =
    {
        0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0xdf, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x1d, 0x06,
        0x21, 0x00, 0xf0, 0x00, 0x05, 0x00, 0x04, 0x00,
        0x03, 0x00, 0x02, 0x00, 0x33, 0x00, 0x00, 0x00,
        0x41, 0x0b, 0x20, 0x00, 0xcc, 0x00, 0x03, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00,
        0x04, 0x00, 0x02, 0x00, 0x01, 0x00, 0x28, 0x00,
        0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00,
        0x00, 0x00, 0x01, 0x00, 0x18, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b,
        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
        0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
        0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b,
        0x2c, 0x2d, 0x33, 0x00, 0x00, 0x00, 0x41, 0x0b,
        0x21, 0x00, 0xf0, 0x00, 0x03, 0x00, 0x02, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x04, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b, 0x0c, 0x0d,
        0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
        0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
        0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25,
        0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
        0x33, 0x00, 0x00, 0x00, 0x41, 0x0b, 0x20, 0x00,
        0xcc, 0x00, 0x03, 0x00, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x02, 0x00, 0x02, 0x00,
        0x01, 0x00, 0x28, 0x00, 0x00, 0x00, 0x04, 0x00,
        0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00,
        0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x24, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
        0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
        0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x31, 0x00,
        0x00, 0x00, 0x40, 0x09, 0x20, 0x00, 0xcc, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x02, 0x00,
        0x02, 0x00, 0x01, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b, 0x0c, 0x0d,
        0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
        0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d,
        0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25,
        0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d,
        0x03, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    hdc = CreateMetaFileA(NULL);
    ok(hdc != 0, "CreateMetaFileA failed\n");

    ret = PatBlt(hdc, 2, 3, 4, 5, PATCOPY);
    ok(ret, "PatBlt failed\n");

    dib_hdc = CreateCompatibleDC(NULL);
    ok(!!dib_hdc, "CreateCompatibleDC failed\n");

    bmi.bmiHeader.biHeight = 3;
    bmi.bmiHeader.biWidth = 4;
    bmi.bmiHeader.biBitCount = 24;
    bmi.bmiHeader.biPlanes = 1;
    bmi.bmiHeader.biCompression = BI_RGB;
    bitmap = CreateDIBSection(dib_hdc, &bmi, DIB_RGB_COLORS, &bits, NULL, 0);
    ok(!!bitmap, "CreateDIBSection failed, error %ld\n", GetLastError());
    for (i = 0; i < bmi.bmiHeader.biHeight * bmi.bmiHeader.biWidth * 3; i++)
        ((BYTE *)bits)[i] = i + 10;

    old_bitmap = SelectObject(dib_hdc, bitmap);
    ok(!!old_bitmap, "SelectObject failed\n");

    ret = StretchBlt(hdc, 1, 2, 4, 6, dib_hdc, 0, 0, 2, 3, SRCCOPY);
    ok(ret, "StretchBlt failed\n");

    /* PATCOPY is recorded as StretchBlt */
    ret = StretchBlt(hdc, 1, 2, 4, 6, dib_hdc, 0, 0, 2, 3, PATCOPY);
    ok(ret, "StretchBlt failed\n");

    /* same src and dst size, recorded as StretchBlt */
    ret = StretchBlt(hdc, 1, 2, 2, 3, dib_hdc, 0, 0, 2, 3, SRCCOPY);
    ok(ret, "StretchBlt failed\n");

    ret = BitBlt(hdc, 1, 2, 2, 3, dib_hdc, 0, 0, SRCCOPY);
    ok(ret, "BitBlt failed\n");

    DeleteDC(dib_hdc);
    DeleteObject(bitmap);

    hmf = CloseMetaFile(hdc);
    ok(hmf != 0, "CloseMetaFile failed\n");

    if (compare_mf_bits(hmf, blit_bits, sizeof(blit_bits), "mf_blit"))
    {
        dump_mf_bits(hmf, "mf_blit");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hmf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hmf, GetLastError());
}

static void test_emf_blit(void)
{
    BITMAPINFO bmi = {{ sizeof(bmi) }};
    HENHMETAFILE hmf;
    HBITMAP bitmap, old_bitmap;
    void *bits;
    HDC hdc, dib_hdc;
    unsigned int i;
    BOOL ret;

    hdc = CreateEnhMetaFileA(NULL, NULL, NULL, NULL);
    ok(hdc != 0, "CreateMetaFileA failed\n");

    ret = PatBlt(hdc, 2, 3, 4, 5, PATCOPY);
    ok(ret, "PatBlt failed\n");

    dib_hdc = CreateCompatibleDC(NULL);
    ok(!!dib_hdc, "CreateCompatibleDC failed\n");

    bmi.bmiHeader.biHeight = 3;
    bmi.bmiHeader.biWidth = 4;
    bmi.bmiHeader.biBitCount = 24;
    bmi.bmiHeader.biPlanes = 1;
    bmi.bmiHeader.biCompression = BI_RGB;
    bitmap = CreateDIBSection(dib_hdc, &bmi, DIB_RGB_COLORS, &bits, NULL, 0);
    ok(!!bitmap, "CreateDIBSection failed, error %ld\n", GetLastError());
    for (i = 0; i < bmi.bmiHeader.biHeight * bmi.bmiHeader.biWidth * 3; i++)
        ((BYTE *)bits)[i] = i + 10;

    old_bitmap = SelectObject(dib_hdc, bitmap);
    ok(!!old_bitmap, "SelectObject failed\n");

    ret = StretchBlt(hdc, 1, 2, 4, 6, dib_hdc, 0, 0, 2, 3, SRCCOPY);
    ok(ret, "StretchBlt failed\n");

    /* PATCOPY is recorded as PatBlt */
    ret = StretchBlt(hdc, 1, 2, 4, 6, dib_hdc, 0, 0, 2, 3, PATCOPY);
    ok(ret, "StretchBlt failed\n");

    /* same src and dst size, still recorded as StretchBlt */
    ret = StretchBlt(hdc, 1, 2, 2, 3, dib_hdc, 0, 0, 2, 3, SRCCOPY);
    ok(ret, "StretchBlt failed\n");

    ret = BitBlt(hdc, 1, 2, 2, 3, dib_hdc, 0, 0, SRCCOPY);
    ok(ret, "BitBlt failed\n");

    ret = GdiTransparentBlt(hdc, 1, 2, 2, 3, dib_hdc, 0, 0, 2, 3, RGB(10,20,30));
    ok(ret, "GdiTransparentBlt failed\n");

    DeleteDC(dib_hdc);
    DeleteObject(bitmap);

    hmf = CloseEnhMetaFile(hdc);
    ok(hmf != 0, "CloseMetaFile failed\n");

    if (compare_emf_bits(hmf, emf_blit_bits, sizeof(emf_blit_bits), "emf_blit", FALSE) != 0)
    {
        dump_emf_bits(hmf, "emf_blit");
        dump_emf_records(hmf, "emf_blit");
    }

    ret = DeleteEnhMetaFile(hmf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hmf, GetLastError());
}

struct check_mf_arg
{
    int pos;
    const WORD *records;
};

static int WINAPI check_mf_records(HDC hdc, HANDLETABLE *htable,
        METARECORD *rec, int n, LPARAM ctx)
{
    struct check_mf_arg *records = (struct check_mf_arg *)ctx;

    ok(rec->rdFunction == records->records[records->pos], "got record %x, expected %x\n",
            rec->rdFunction, records->records[records->pos]);
    return rec->rdFunction == records->records[records->pos++];
}

static void test_PlayMetaFile(void)
{
    static WORD line_records[] = { META_LINETO, 0 };
    struct check_mf_arg records;
    HMETAFILE hmeta;
    HDC dc, metadc;
    BOOL r;

    dc = CreateCompatibleDC(NULL);
    ok(dc != NULL, "CreateCompatibleDC failed\n");

    metadc = CreateMetaFileW(NULL);
    ok(metadc != NULL, "CreateMetaFile failed\n");
    r = LineTo(metadc, 10, 10);
    ok(r, "LineTo failed\n");
    hmeta = CloseMetaFile(metadc);
    ok(hmeta != NULL, "CloseMetaFile failed\n");

    records.pos = 0;
    records.records = line_records;
    r = EnumMetaFile(dc, hmeta, check_mf_records, (LPARAM)&records);
    ok(r, "EnumMetaFile failed\n");

    metadc = CreateMetaFileW(NULL);
    ok(metadc != NULL, "CreateMetaFile failed\n");
    r = PlayMetaFile(metadc, hmeta);
    ok(r, "PlayMetaFile failed\n");
    r = DeleteMetaFile(hmeta);
    ok(r, "DeleteMetaFile failed\n");
    hmeta = CloseMetaFile(metadc);
    ok(hmeta != NULL, "CloseMetaFile failed\n");

    records.pos = 0;
    records.records = line_records;
    r = EnumMetaFile(dc, hmeta, check_mf_records, (LPARAM)&records);
    ok(r, "EnumMetaFile failed\n");

    r = DeleteMetaFile(hmeta);
    ok(r, "DeleteMetaFile failed\n");
    r = DeleteDC(dc);
    ok(r, "DeleteDC failed\n");
}

static void test_emf_mask_blit(void)
{
    BITMAPINFO bmi = {{ sizeof(bmi) }};
    HBITMAP bitmap, mask_bitmap;
    HENHMETAFILE emf;
    HDC hdc, dib_hdc;
    unsigned int i;
    void *bits;
    BOOL ret;

    static const POINT plg_points[] = { { 2, 1 }, { 1, 1 }, { 1, 2 } };

    hdc = CreateEnhMetaFileA(NULL, NULL, NULL, NULL);
    ok(hdc != 0, "CreateMetaFileA failed\n");

    dib_hdc = CreateCompatibleDC(NULL);
    ok(!!dib_hdc, "CreateCompatibleDC failed\n");

    bmi.bmiHeader.biHeight = 3;
    bmi.bmiHeader.biWidth = 4;
    bmi.bmiHeader.biBitCount = 24;
    bmi.bmiHeader.biPlanes = 1;
    bmi.bmiHeader.biCompression = BI_RGB;
    bitmap = CreateDIBSection(dib_hdc, &bmi, DIB_RGB_COLORS, &bits, NULL, 0);
    ok(!!bitmap, "CreateDIBSection failed, error %ld\n", GetLastError());
    for (i = 0; i < bmi.bmiHeader.biHeight * bmi.bmiHeader.biWidth * 3; i++)
        ((BYTE *)bits)[i] = i + 10;

    bmi.bmiHeader.biHeight = 3;
    bmi.bmiHeader.biWidth = 4;
    bmi.bmiHeader.biBitCount = 1;
    bmi.bmiHeader.biPlanes = 1;
    bmi.bmiHeader.biCompression = BI_RGB;
    mask_bitmap = CreateDIBSection(dib_hdc, &bmi, DIB_RGB_COLORS, &bits, NULL, 0);
    ok(!!mask_bitmap, "CreateDIBSection failed, error %ld\n", GetLastError());
    memset(bits, ~0, bmi.bmiHeader.biHeight * 4);

    SelectObject(dib_hdc, bitmap);

    ret = MaskBlt(hdc, 0, 0, 3, 4, dib_hdc, 0, 0, mask_bitmap, 0, 0, SRCCOPY);
    ok(ret, "MaskBlt failed\n");

    ret = MaskBlt(hdc, 0, 0, 3, 4, dib_hdc, 0, 0, NULL, 0, 0, SRCCOPY);
    ok(ret, "MaskBlt failed\n");

    ret = PlgBlt(hdc, plg_points, dib_hdc, 0, 0, 3, 4, mask_bitmap, 0, 0);
    ok(ret, "PlgBlt failed\n");

    DeleteDC(dib_hdc);
    DeleteObject(bitmap);
    DeleteObject(mask_bitmap);

    emf = CloseEnhMetaFile(hdc);
    ok(emf != 0, "CloseMetaFile failed\n");

    if (compare_emf_bits(emf, emf_mask_blit_bits, sizeof(emf_mask_blit_bits),
                         "emf_mask_blit", FALSE) != 0)
    {
        dump_emf_bits(emf, "emf_mask_blit");
        dump_emf_records(emf, "emf_mask_blit");
    }

    ret = DeleteEnhMetaFile(emf);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", emf, GetLastError());
}

static const unsigned char EMF_STRETCHDIBITS_1BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_1BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_4BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_4BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_8BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_8BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_16BIT_555_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x18, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_16BIT_555_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x18, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_24BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x28, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xa8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_24BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x28, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xa8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_32BIT_888_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x38, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xb8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_32BIT_888_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x38, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xb8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_16BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xa4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 0x00,
    0xe0, 0x07, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_16BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xa4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 0x00,
    0xe0, 0x07, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_32BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x44, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
    0xe0, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_32BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x44, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
    0xe0, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_1BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_1BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_4BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_4BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_8BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_8BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_16BIT_555_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x18, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_16BIT_555_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x18, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_24BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x28, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xa8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_24BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x28, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xa8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_32BIT_888_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x38, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xb8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_32BIT_888_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x38, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xb8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_16BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xa4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 0x00,
    0xe0, 0x07, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_16BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xa4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 0x00,
    0xe0, 0x07, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_32BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x44, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
    0xe0, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_STRETCHDIBITS_32BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x44, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x51, 0x00, 0x00, 0x00,
    0xc4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x00, 0xcc, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
    0xe0, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static void test_emf_StretchDIBits(void)
{
    static const unsigned char dib_bits[4 * 4 * 4] = { 0 };
    static const int bitmap_width = 4, bitmap_height = 4;
    union {
        unsigned char b[FIELD_OFFSET(BITMAPINFO, bmiColors[256])];
        BITMAPINFO i;
        BITMAPCOREINFO c;
    } bmi;
    HDC hdc, hdc_emf;
    HENHMETAFILE hemf;
    int ret, test_idx, color_idx;

    static const struct
    {
        WORD bpp;
        int src_width;
        int src_height;
        WORD compression;
        unsigned char infomode;
        const void *bits;
        size_t bits_count;
        DWORD used_color_count;
        DWORD color_count;
        RGBQUAD colors[3];
    }
    tests[] =
    {
        {1, 3, 3, BI_RGB, 0, EMF_STRETCHDIBITS_1BIT_3X3_NOSIZE, sizeof(EMF_STRETCHDIBITS_1BIT_3X3_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {1, 3, 3, BI_RGB, 1, EMF_STRETCHDIBITS_1BIT_3X3_SIZED, sizeof(EMF_STRETCHDIBITS_1BIT_3X3_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 3, 3, BI_RGB, 0, EMF_STRETCHDIBITS_4BIT_3X3_NOSIZE, sizeof(EMF_STRETCHDIBITS_4BIT_3X3_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 3, 3, BI_RGB, 1, EMF_STRETCHDIBITS_4BIT_3X3_SIZED, sizeof(EMF_STRETCHDIBITS_4BIT_3X3_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 3, 3, BI_RGB, 0, EMF_STRETCHDIBITS_8BIT_3X3_NOSIZE, sizeof(EMF_STRETCHDIBITS_8BIT_3X3_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 3, 3, BI_RGB, 1, EMF_STRETCHDIBITS_8BIT_3X3_SIZED, sizeof(EMF_STRETCHDIBITS_8BIT_3X3_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {16, 3, 3, BI_RGB, 0, EMF_STRETCHDIBITS_16BIT_555_3X3_NOSIZE, sizeof(EMF_STRETCHDIBITS_16BIT_555_3X3_NOSIZE)},
        {16, 3, 3, BI_RGB, 1, EMF_STRETCHDIBITS_16BIT_555_3X3_SIZED, sizeof(EMF_STRETCHDIBITS_16BIT_555_3X3_SIZED)},
        {24, 3, 3, BI_RGB, 0, EMF_STRETCHDIBITS_24BIT_3X3_NOSIZE, sizeof(EMF_STRETCHDIBITS_24BIT_3X3_NOSIZE)},
        {24, 3, 3, BI_RGB, 1, EMF_STRETCHDIBITS_24BIT_3X3_SIZED, sizeof(EMF_STRETCHDIBITS_24BIT_3X3_SIZED)},
        {24, 3, 3, BI_RGB, 2, EMF_STRETCHDIBITS_24BIT_3X3_NOSIZE, sizeof(EMF_STRETCHDIBITS_24BIT_3X3_NOSIZE)},
        {32, 3, 3, BI_RGB, 0, EMF_STRETCHDIBITS_32BIT_888_3X3_NOSIZE, sizeof(EMF_STRETCHDIBITS_32BIT_888_3X3_NOSIZE)},
        {32, 3, 3, BI_RGB, 1, EMF_STRETCHDIBITS_32BIT_888_3X3_SIZED, sizeof(EMF_STRETCHDIBITS_32BIT_888_3X3_SIZED)},
        {16, 3, 3, BI_BITFIELDS, 0, EMF_STRETCHDIBITS_16BIT_3X3_NOSIZE, sizeof(EMF_STRETCHDIBITS_16BIT_3X3_NOSIZE), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {16, 3, 3, BI_BITFIELDS, 1, EMF_STRETCHDIBITS_16BIT_3X3_SIZED, sizeof(EMF_STRETCHDIBITS_16BIT_3X3_SIZED), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {32, 3, 3, BI_BITFIELDS, 0, EMF_STRETCHDIBITS_32BIT_3X3_NOSIZE, sizeof(EMF_STRETCHDIBITS_32BIT_3X3_NOSIZE), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
        {32, 3, 3, BI_BITFIELDS, 1, EMF_STRETCHDIBITS_32BIT_3X3_SIZED, sizeof(EMF_STRETCHDIBITS_32BIT_3X3_SIZED), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
        {1, 4, 4, BI_RGB, 0, EMF_STRETCHDIBITS_1BIT_4X4_NOSIZE, sizeof(EMF_STRETCHDIBITS_1BIT_4X4_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {1, 4, 4, BI_RGB, 1, EMF_STRETCHDIBITS_1BIT_4X4_SIZED, sizeof(EMF_STRETCHDIBITS_1BIT_4X4_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 4, 4, BI_RGB, 0, EMF_STRETCHDIBITS_4BIT_4X4_NOSIZE, sizeof(EMF_STRETCHDIBITS_4BIT_4X4_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 4, 4, BI_RGB, 1, EMF_STRETCHDIBITS_4BIT_4X4_SIZED, sizeof(EMF_STRETCHDIBITS_4BIT_4X4_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 4, 4, BI_RGB, 0, EMF_STRETCHDIBITS_8BIT_4X4_NOSIZE, sizeof(EMF_STRETCHDIBITS_8BIT_4X4_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 4, 4, BI_RGB, 1, EMF_STRETCHDIBITS_8BIT_4X4_SIZED, sizeof(EMF_STRETCHDIBITS_8BIT_4X4_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {16, 4, 4, BI_RGB, 0, EMF_STRETCHDIBITS_16BIT_555_4X4_NOSIZE, sizeof(EMF_STRETCHDIBITS_16BIT_555_4X4_NOSIZE)},
        {16, 4, 4, BI_RGB, 1, EMF_STRETCHDIBITS_16BIT_555_4X4_SIZED, sizeof(EMF_STRETCHDIBITS_16BIT_555_4X4_SIZED)},
        {24, 4, 4, BI_RGB, 0, EMF_STRETCHDIBITS_24BIT_4X4_NOSIZE, sizeof(EMF_STRETCHDIBITS_24BIT_4X4_NOSIZE)},
        {24, 4, 4, BI_RGB, 1, EMF_STRETCHDIBITS_24BIT_4X4_SIZED, sizeof(EMF_STRETCHDIBITS_24BIT_4X4_SIZED)},
        {24, 4, 4, BI_RGB, 2, EMF_STRETCHDIBITS_24BIT_4X4_NOSIZE, sizeof(EMF_STRETCHDIBITS_24BIT_4X4_NOSIZE)},
        {32, 4, 4, BI_RGB, 0, EMF_STRETCHDIBITS_32BIT_888_4X4_NOSIZE, sizeof(EMF_STRETCHDIBITS_32BIT_888_4X4_NOSIZE)},
        {32, 4, 4, BI_RGB, 1, EMF_STRETCHDIBITS_32BIT_888_4X4_SIZED, sizeof(EMF_STRETCHDIBITS_32BIT_888_4X4_SIZED)},
        {16, 4, 4, BI_BITFIELDS, 0, EMF_STRETCHDIBITS_16BIT_4X4_NOSIZE, sizeof(EMF_STRETCHDIBITS_16BIT_4X4_NOSIZE), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {16, 4, 4, BI_BITFIELDS, 1, EMF_STRETCHDIBITS_16BIT_4X4_SIZED, sizeof(EMF_STRETCHDIBITS_16BIT_4X4_SIZED), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {32, 4, 4, BI_BITFIELDS, 0, EMF_STRETCHDIBITS_32BIT_4X4_NOSIZE, sizeof(EMF_STRETCHDIBITS_32BIT_4X4_NOSIZE), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
        {32, 4, 4, BI_BITFIELDS, 1, EMF_STRETCHDIBITS_32BIT_4X4_SIZED, sizeof(EMF_STRETCHDIBITS_32BIT_4X4_SIZED), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
    };

    hdc = GetDC(0);

    for (test_idx = 0; test_idx < ARRAY_SIZE(tests); ++test_idx)
    {
        winetest_push_context("Test %d", test_idx);

        memset(&bmi, 0, sizeof(bmi));
        if (tests[test_idx].infomode < 2)
        {
            bmi.i.bmiHeader.biSize = sizeof(bmi.i.bmiHeader);
            bmi.i.bmiHeader.biWidth = bitmap_width;
            bmi.i.bmiHeader.biHeight = bitmap_height;
            bmi.i.bmiHeader.biBitCount = tests[test_idx].bpp;
            bmi.i.bmiHeader.biPlanes = 1;
            bmi.i.bmiHeader.biCompression = tests[test_idx].compression;
            bmi.i.bmiHeader.biClrUsed = tests[test_idx].used_color_count;
            if (tests[test_idx].infomode == 1)
                bmi.i.bmiHeader.biSizeImage = (((bitmap_width * tests[test_idx].bpp + 31) >> 3) & ~3) * bitmap_height;
            memcpy(bmi.i.bmiColors, tests[test_idx].colors, sizeof(RGBQUAD) * tests[test_idx].color_count);
        }
        else
        {
            bmi.c.bmciHeader.bcSize = sizeof(bmi.c.bmciHeader);
            bmi.c.bmciHeader.bcWidth = bitmap_width;
            bmi.c.bmciHeader.bcHeight = bitmap_height;
            bmi.c.bmciHeader.bcPlanes = 1;
            bmi.c.bmciHeader.bcBitCount = tests[test_idx].bpp;
            for (color_idx = 0; color_idx < tests[test_idx].color_count; color_idx++)
            {
                bmi.c.bmciColors[color_idx].rgbtBlue = tests[test_idx].colors[color_idx].rgbBlue;
                bmi.c.bmciColors[color_idx].rgbtGreen = tests[test_idx].colors[color_idx].rgbGreen;
                bmi.c.bmciColors[color_idx].rgbtRed = tests[test_idx].colors[color_idx].rgbRed;
            }
        }

        hdc_emf = CreateEnhMetaFileW(hdc, NULL, NULL, NULL);
        ok(!!hdc_emf, "CreateEnhMetaFileW failed, error %ld\n", GetLastError());

        ret = StretchDIBits(hdc_emf, 0, 0, bitmap_width, bitmap_height, 0, 0,
                            tests[test_idx].src_width, tests[test_idx].src_height,
                            dib_bits, &bmi.i, DIB_RGB_COLORS, SRCCOPY);
        ok(ret == bitmap_height, "expected StretchDIBits to return %d, got %d (lasterr %lu)\n", bitmap_height, ret, GetLastError());

        hemf = CloseEnhMetaFile(hdc_emf);
        ok(!!hemf, "CloseEnhMetaFile failed, error %ld\n", GetLastError());

        ret = compare_emf_bits(hemf, tests[test_idx].bits, tests[test_idx].bits_count,
                               "test_emf_StretchDIBits", FALSE);
        if (ret)
        {
            dump_emf_bits(hemf, "test_emf_StretchDIBits");
            dump_emf_records(hemf, "test_emf_StretchDIBits");
        }

        DeleteEnhMetaFile(hemf);

        winetest_pop_context();
    }

    ReleaseDC(0, hdc);
}

static const unsigned char EMF_SETDIBITSTODEVICE_1BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x04, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_1BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x04, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_4BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x04, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_4BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x04, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_8BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x04, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_8BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x04, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x84, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_16BIT_555_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_16BIT_555_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x0c, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x8c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_24BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x18, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_24BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x18, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_32BIT_888_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xa4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_32BIT_888_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xa4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_16BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x18, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xf8, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00,
    0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_16BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x18, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xf8, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00,
    0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_32BIT_3X3_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x30, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xb0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xff, 0x00, 0xe0, 0xff, 0x00, 0x00,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_32BIT_3X3_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x35, 0x00, 0x00, 0x00, 0x35, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x30, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xb0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xff, 0x00, 0xe0, 0xff, 0x00, 0x00,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_1BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x08, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_1BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x08, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_4BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x08, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_4BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x08, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_8BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x08, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_8BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x08, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_16BIT_555_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x14, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_16BIT_555_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x14, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_24BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xa4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_24BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x24, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xa4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x18, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_32BIT_888_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x34, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xb4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_32BIT_888_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x34, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xb4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x74, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_16BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x20, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xf8, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00,
    0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_16BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x20, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x10, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xf8, 0x00, 0x00, 0xe0, 0x07, 0x00, 0x00,
    0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_32BIT_4X4_NOSIZE[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x40, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xff, 0x00, 0xe0, 0xff, 0x00, 0x00,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_SETDIBITSTODEVICE_32BIT_4X4_SIZED[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x40, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x0f, 0x01, 0x00, 0x00, 0xcb, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x22, 0x04, 0x00,
    0xc0, 0x19, 0x03, 0x00, 0x50, 0x00, 0x00, 0x00,
    0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x20, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0xff, 0x00, 0xe0, 0xff, 0x00, 0x00,
    0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static void test_emf_SetDIBitsToDevice(void)
{
    static const unsigned char dib_bits[4 * 4 * 4] = { 0 };
    static const int bitmap_width = 4, bitmap_height = 4;
    union {
        unsigned char b[FIELD_OFFSET(BITMAPINFO, bmiColors[256])];
        BITMAPINFO i;
        BITMAPCOREINFO c;
    } bmi;
    HDC hdc, hdc_emf;
    HENHMETAFILE hemf;
    int ret, test_idx, color_idx;

    static const struct
    {
        WORD bpp;
        int width;
        int height;
        WORD compression;
        unsigned char infomode;
        const void *bits;
        size_t bits_count;
        DWORD used_color_count;
        DWORD color_count;
        RGBQUAD colors[3];
    }
    tests[] =
    {
        {1, 3, 3, BI_RGB, 0, EMF_SETDIBITSTODEVICE_1BIT_3X3_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_1BIT_3X3_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {1, 3, 3, BI_RGB, 1, EMF_SETDIBITSTODEVICE_1BIT_3X3_SIZED, sizeof(EMF_SETDIBITSTODEVICE_1BIT_3X3_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 3, 3, BI_RGB, 0, EMF_SETDIBITSTODEVICE_4BIT_3X3_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_4BIT_3X3_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 3, 3, BI_RGB, 1, EMF_SETDIBITSTODEVICE_4BIT_3X3_SIZED, sizeof(EMF_SETDIBITSTODEVICE_4BIT_3X3_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 3, 3, BI_RGB, 0, EMF_SETDIBITSTODEVICE_8BIT_3X3_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_8BIT_3X3_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 3, 3, BI_RGB, 1, EMF_SETDIBITSTODEVICE_8BIT_3X3_SIZED, sizeof(EMF_SETDIBITSTODEVICE_8BIT_3X3_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {16, 3, 3, BI_RGB, 0, EMF_SETDIBITSTODEVICE_16BIT_555_3X3_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_16BIT_555_3X3_NOSIZE)},
        {16, 3, 3, BI_RGB, 1, EMF_SETDIBITSTODEVICE_16BIT_555_3X3_SIZED, sizeof(EMF_SETDIBITSTODEVICE_16BIT_555_3X3_SIZED)},
        {24, 3, 3, BI_RGB, 0, EMF_SETDIBITSTODEVICE_24BIT_3X3_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_24BIT_3X3_NOSIZE)},
        {24, 3, 3, BI_RGB, 1, EMF_SETDIBITSTODEVICE_24BIT_3X3_SIZED, sizeof(EMF_SETDIBITSTODEVICE_24BIT_3X3_SIZED)},
        {24, 3, 3, BI_RGB, 2, EMF_SETDIBITSTODEVICE_24BIT_3X3_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_24BIT_3X3_NOSIZE)},
        {32, 3, 3, BI_RGB, 0, EMF_SETDIBITSTODEVICE_32BIT_888_3X3_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_32BIT_888_3X3_NOSIZE)},
        {32, 3, 3, BI_RGB, 1, EMF_SETDIBITSTODEVICE_32BIT_888_3X3_SIZED, sizeof(EMF_SETDIBITSTODEVICE_32BIT_888_3X3_SIZED)},
        {16, 3, 3, BI_BITFIELDS, 0, EMF_SETDIBITSTODEVICE_16BIT_3X3_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_16BIT_3X3_NOSIZE), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {16, 3, 3, BI_BITFIELDS, 1, EMF_SETDIBITSTODEVICE_16BIT_3X3_SIZED, sizeof(EMF_SETDIBITSTODEVICE_16BIT_3X3_SIZED), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {32, 3, 3, BI_BITFIELDS, 0, EMF_SETDIBITSTODEVICE_32BIT_3X3_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_32BIT_3X3_NOSIZE), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
        {32, 3, 3, BI_BITFIELDS, 1, EMF_SETDIBITSTODEVICE_32BIT_3X3_SIZED, sizeof(EMF_SETDIBITSTODEVICE_32BIT_3X3_SIZED), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
        {1, 4, 4, BI_RGB, 0, EMF_SETDIBITSTODEVICE_1BIT_4X4_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_1BIT_4X4_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {1, 4, 4, BI_RGB, 1, EMF_SETDIBITSTODEVICE_1BIT_4X4_SIZED, sizeof(EMF_SETDIBITSTODEVICE_1BIT_4X4_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 4, 4, BI_RGB, 0, EMF_SETDIBITSTODEVICE_4BIT_4X4_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_4BIT_4X4_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, 4, 4, BI_RGB, 1, EMF_SETDIBITSTODEVICE_4BIT_4X4_SIZED, sizeof(EMF_SETDIBITSTODEVICE_4BIT_4X4_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 4, 4, BI_RGB, 0, EMF_SETDIBITSTODEVICE_8BIT_4X4_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_8BIT_4X4_NOSIZE), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, 4, 4, BI_RGB, 1, EMF_SETDIBITSTODEVICE_8BIT_4X4_SIZED, sizeof(EMF_SETDIBITSTODEVICE_8BIT_4X4_SIZED), 1, 1, {{0xff, 0xff, 0xff}}},
        {16, 4, 4, BI_RGB, 0, EMF_SETDIBITSTODEVICE_16BIT_555_4X4_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_16BIT_555_4X4_NOSIZE)},
        {16, 4, 4, BI_RGB, 1, EMF_SETDIBITSTODEVICE_16BIT_555_4X4_SIZED, sizeof(EMF_SETDIBITSTODEVICE_16BIT_555_4X4_SIZED)},
        {24, 4, 4, BI_RGB, 0, EMF_SETDIBITSTODEVICE_24BIT_4X4_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_24BIT_4X4_NOSIZE)},
        {24, 4, 4, BI_RGB, 1, EMF_SETDIBITSTODEVICE_24BIT_4X4_SIZED, sizeof(EMF_SETDIBITSTODEVICE_24BIT_4X4_SIZED)},
        {24, 4, 4, BI_RGB, 2, EMF_SETDIBITSTODEVICE_24BIT_4X4_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_24BIT_4X4_NOSIZE)},
        {32, 4, 4, BI_RGB, 0, EMF_SETDIBITSTODEVICE_32BIT_888_4X4_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_32BIT_888_4X4_NOSIZE)},
        {32, 4, 4, BI_RGB, 1, EMF_SETDIBITSTODEVICE_32BIT_888_4X4_SIZED, sizeof(EMF_SETDIBITSTODEVICE_32BIT_888_4X4_SIZED)},
        {16, 4, 4, BI_BITFIELDS, 0, EMF_SETDIBITSTODEVICE_16BIT_4X4_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_16BIT_4X4_NOSIZE), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {16, 4, 4, BI_BITFIELDS, 1, EMF_SETDIBITSTODEVICE_16BIT_4X4_SIZED, sizeof(EMF_SETDIBITSTODEVICE_16BIT_4X4_SIZED), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {32, 4, 4, BI_BITFIELDS, 0, EMF_SETDIBITSTODEVICE_32BIT_4X4_NOSIZE, sizeof(EMF_SETDIBITSTODEVICE_32BIT_4X4_NOSIZE), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
        {32, 4, 4, BI_BITFIELDS, 1, EMF_SETDIBITSTODEVICE_32BIT_4X4_SIZED, sizeof(EMF_SETDIBITSTODEVICE_32BIT_4X4_SIZED), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
    };

    hdc = GetDC(0);

    for (test_idx = 0; test_idx < ARRAY_SIZE(tests); ++test_idx)
    {
        winetest_push_context("Test %d", test_idx);

        memset(&bmi, 0, sizeof(bmi));
        if (tests[test_idx].infomode < 2)
        {
            bmi.i.bmiHeader.biSize = sizeof(bmi.i.bmiHeader);
            bmi.i.bmiHeader.biWidth = bitmap_width;
            bmi.i.bmiHeader.biHeight = bitmap_height;
            bmi.i.bmiHeader.biBitCount = tests[test_idx].bpp;
            bmi.i.bmiHeader.biPlanes = 1;
            bmi.i.bmiHeader.biCompression = tests[test_idx].compression;
            bmi.i.bmiHeader.biClrUsed = tests[test_idx].used_color_count;
            if (tests[test_idx].infomode == 1)
                bmi.i.bmiHeader.biSizeImage = (((bitmap_width * tests[test_idx].bpp + 31) >> 3) & ~3) * bitmap_height;
            memcpy(bmi.i.bmiColors, tests[test_idx].colors, sizeof(RGBQUAD) * tests[test_idx].color_count);
        }
        else
        {
            bmi.c.bmciHeader.bcSize = sizeof(bmi.c.bmciHeader);
            bmi.c.bmciHeader.bcWidth = bitmap_width;
            bmi.c.bmciHeader.bcHeight = bitmap_height;
            bmi.c.bmciHeader.bcPlanes = 1;
            bmi.c.bmciHeader.bcBitCount = tests[test_idx].bpp;
            for (color_idx = 0; color_idx < tests[test_idx].color_count; color_idx++)
            {
                bmi.c.bmciColors[color_idx].rgbtBlue = tests[test_idx].colors[color_idx].rgbBlue;
                bmi.c.bmciColors[color_idx].rgbtGreen = tests[test_idx].colors[color_idx].rgbGreen;
                bmi.c.bmciColors[color_idx].rgbtRed = tests[test_idx].colors[color_idx].rgbRed;
            }
        }

        hdc_emf = CreateEnhMetaFileW(hdc, NULL, NULL, NULL);
        ok(!!hdc_emf, "CreateEnhMetaFileW failed, error %ld\n", GetLastError());

        ret = SetDIBitsToDevice(hdc_emf, 0, 0,
                                tests[test_idx].width, tests[test_idx].height,
                                0, 0, 0, bitmap_height,
                                dib_bits, &bmi.i, DIB_RGB_COLORS);
        ok(ret == tests[test_idx].height, "expected SetDIBitsToDevice to return %d, got %d (lasterr %lu)\n", tests[test_idx].height, ret, GetLastError());

        hemf = CloseEnhMetaFile(hdc_emf);
        ok(!!hemf, "CloseEnhMetaFile failed, error %ld\n", GetLastError());

        ret = compare_emf_bits(hemf, tests[test_idx].bits, tests[test_idx].bits_count,
                               "test_emf_SetDIBitsToDevice", FALSE);
        if (ret)
        {
            dump_emf_bits(hemf, "test_emf_SetDIBitsToDevice");
            dump_emf_records(hemf, "test_emf_SetDIBitsToDevice");
        }

        DeleteEnhMetaFile(hemf);

        winetest_pop_context();
    }

    ReleaseDC(0, hdc);
}

static void test_mf_ExtTextOut_on_path(void)
{
    HDC hdcMetafile;
    HMETAFILE hMetafile;
    BOOL ret;
    static const INT dx[4] = { 3, 5, 8, 12 };

    hdcMetafile = CreateMetaFileA(NULL);
    ok(hdcMetafile != 0, "CreateMetaFileA(NULL) error %ld\n", GetLastError());

    ret = BeginPath(hdcMetafile);
    ok(!ret, "BeginPath on metafile DC should fail\n");

    ret = ExtTextOutA(hdcMetafile, 11, 22, 0, NULL, "Test", 4, dx);
    ok(ret, "ExtTextOut error %ld\n", GetLastError());

    ret = EndPath(hdcMetafile);
    ok(!ret, "EndPath on metafile DC should fail\n");

    hMetafile = CloseMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseMetaFile error %ld\n", GetLastError());

    if (compare_mf_bits(hMetafile, MF_TEXTOUT_ON_PATH_BITS, sizeof(MF_TEXTOUT_ON_PATH_BITS),
        "mf_TextOut_on_path") != 0)
    {
        dump_mf_bits(hMetafile, "mf_TextOut_on_path");
        EnumMetaFile(0, hMetafile, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hMetafile);
    ok(ret, "DeleteMetaFile(%p) error %ld\n", hMetafile, GetLastError());
}

static const unsigned char EMF_EMPTY_BITS[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xdd, 0xff, 0xff, 0xff, 0xdd, 0xff, 0xff, 0xff,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x80, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x90, 0x06, 0x00, 0x00, 0xba, 0x03, 0x00, 0x00,
    0x4e, 0x02, 0x00, 0x00, 0x4a, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xb0, 0x00, 0x09, 0x00,
    0x10, 0x09, 0x05, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static void test_emf_ExtTextOut_on_path(void)
{
    HWND hwnd;
    HDC hdcDisplay, hdcMetafile;
    HENHMETAFILE hMetafile;
    BOOL ret;
    LOGFONTA lf;
    HFONT hFont;
    static const INT dx[4] = { 3, 5, 8, 12 };
    RECT rect = { 10, 20, 30, 40 };

    /* Win9x doesn't play EMFs on invisible windows */
    hwnd = CreateWindowExA(0, "static", NULL, WS_POPUP | WS_VISIBLE,
                           0, 0, 200, 200, 0, 0, 0, NULL);
    ok(hwnd != 0, "CreateWindowExA error %ld\n", GetLastError());

    hdcDisplay = GetDC(hwnd);
    ok(hdcDisplay != 0, "GetDC error %ld\n", GetLastError());

    /* with default font */
    ret = BeginPath(hdcDisplay);
    ok(ret, "BeginPath error %ld\n", GetLastError());

    ret = ExtTextOutA(hdcDisplay, 11, 22, 0, NULL, "Test", 4, dx);
    ok(ret, "ExtTextOut error %ld\n", GetLastError());

    ret = EndPath(hdcDisplay);
    ok(ret, "EndPath error %ld\n", GetLastError());

    ret = GetPath(hdcDisplay, NULL, NULL, 0);
    ok(!ret, "expected 0, got %d\n", ret);

    hdcMetafile = CreateEnhMetaFileA(hdcDisplay, NULL, NULL, NULL);
    ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    ret = BeginPath(hdcMetafile);
    ok(ret, "BeginPath error %ld\n", GetLastError());

    ret = ExtTextOutA(hdcMetafile, 11, 22, 0, NULL, "Test", 4, dx);
    ok(ret, "ExtTextOut error %ld\n", GetLastError());

    ret = EndPath(hdcMetafile);
    ok(ret, "EndPath error %ld\n", GetLastError());

    ret = GetPath(hdcMetafile, NULL, NULL, 0);
    ok(!ret, "expected 0, got %d\n", ret);

    hMetafile = CloseEnhMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    /* this doesn't succeed yet: EMF has correct size, all EMF records
     * are there, but their contents don't match for different reasons.
     */
    if (compare_emf_bits(hMetafile, EMF_TEXTOUT_ON_PATH_BITS, sizeof(EMF_TEXTOUT_ON_PATH_BITS),
        "emf_TextOut_on_path", FALSE) != 0)
    {
        dump_emf_bits(hMetafile, "emf_TextOut_on_path");
        dump_emf_records(hMetafile, "emf_TextOut_on_path");
    }

    ret = DeleteEnhMetaFile(hMetafile);
    ok(ret, "DeleteEnhMetaFile error %ld\n", GetLastError());

    /* with outline font */
    memset(&lf, 0, sizeof(lf));
    lf.lfCharSet = ANSI_CHARSET;
    lf.lfClipPrecision = CLIP_DEFAULT_PRECIS;
    lf.lfWeight = FW_DONTCARE;
    lf.lfHeight = 7;
    lf.lfQuality = DEFAULT_QUALITY;
    lstrcpyA(lf.lfFaceName, "Tahoma");
    hFont = CreateFontIndirectA(&lf);
    ok(hFont != 0, "CreateFontIndirectA error %ld\n", GetLastError());
    hFont = SelectObject(hdcDisplay, hFont);

    ret = BeginPath(hdcDisplay);
    ok(ret, "BeginPath error %ld\n", GetLastError());

    ret = ExtTextOutA(hdcDisplay, 11, 22, 0, NULL, "Test", 4, dx);
    ok(ret, "ExtTextOut error %ld\n", GetLastError());

    ret = EndPath(hdcDisplay);
    ok(ret, "EndPath error %ld\n", GetLastError());

    ret = GetPath(hdcDisplay, NULL, NULL, 0);
    ok(ret != 0, "expected != 0\n");

    SelectObject(hdcDisplay, hFont);

    hdcMetafile = CreateEnhMetaFileA(hdcDisplay, NULL, NULL, NULL);
    ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    hFont = SelectObject(hdcMetafile, hFont);

    ret = BeginPath(hdcMetafile);
    ok(ret, "BeginPath error %ld\n", GetLastError());

    ret = ExtTextOutA(hdcMetafile, 11, 22, 0, NULL, "Test", 4, dx);
    ok(ret, "ExtTextOut error %ld\n", GetLastError());

    ret = EndPath(hdcMetafile);
    ok(ret, "EndPath error %ld\n", GetLastError());

    ret = GetPath(hdcMetafile, NULL, NULL, 0);
    ok(!ret, "expected 0, got %d\n", ret);

    hFont = SelectObject(hdcMetafile, hFont);
    DeleteObject(hFont);

    hMetafile = CloseEnhMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    if (compare_emf_bits(hMetafile, EMF_TEXTOUT_OUTLINE_ON_PATH_BITS, sizeof(EMF_TEXTOUT_OUTLINE_ON_PATH_BITS),
        "emf_TextOut_outline_on_path", FALSE) != 0)
    {
        dump_emf_bits(hMetafile, "emf_TextOut_outline_on_path");
        dump_emf_records(hMetafile, "emf_TextOut_outline_on_path");
    }

    ret = DeleteEnhMetaFile(hMetafile);
    ok(ret, "DeleteEnhMetaFile error %ld\n", GetLastError());

    /* test ExtTextOut with count == -1 doesn't get written */
    hdcMetafile = CreateEnhMetaFileA(hdcDisplay, NULL, NULL, NULL);
    ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    ret = ExtTextOutA(hdcMetafile, 11, 22, ETO_OPAQUE, &rect, "Test", -1, dx);
    ok(!ret, "ExtTextOut error %ld\n", GetLastError());

    hMetafile = CloseEnhMetaFile(hdcMetafile);
    ok(hMetafile != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    if (compare_emf_bits(hMetafile, EMF_EMPTY_BITS, sizeof(EMF_EMPTY_BITS),
        "emf_TextOut_negative_count", FALSE) != 0)
    {
        dump_emf_bits(hMetafile, "emf_TextOut_negative_count");
        dump_emf_records(hMetafile, "emf_TextOut_negative_count");
    }

    ret = DeleteEnhMetaFile(hMetafile);
    ok(ret, "DeleteEnhMetaFile error %ld\n", GetLastError());

    ret = ReleaseDC(hwnd, hdcDisplay);
    ok(ret, "ReleaseDC error %ld\n", GetLastError());
    DestroyWindow(hwnd);
}

static const unsigned char EMF_CLIPPING[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x1a, 0x00, 0x00, 0x00, 0x1a, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x04, 0x01, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xb0, 0x04, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x3e, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0x30, 0xda, 0x04, 0x00, 0x36, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x4b, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x04, 0x00, 0x00, 0x3b, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0xff, 0x03, 0x00, 0x00,
    0xff, 0x03, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x43, 0x00, 0x00, 0x00,
    0x0c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static void translate( POINT *pt, UINT count, const XFORM *xform )
{
    while (count--)
    {
        FLOAT x = (FLOAT)pt->x;
        FLOAT y = (FLOAT)pt->y;
        pt->x = (LONG)floor( x * xform->eM11 + y * xform->eM21 + xform->eDx + 0.5 );
        pt->y = (LONG)floor( x * xform->eM12 + y * xform->eM22 + xform->eDy + 0.5 );
        pt++;
    }
}

/* Compare rectangles allowing rounding errors */
static BOOL is_equal_rect(const RECT *rc1, const RECT *rc2)
{
    return abs(rc1->left - rc2->left) <= 1 &&
           abs(rc1->top - rc2->top) <= 1 &&
           abs(rc1->right - rc2->right) <= 1 &&
           abs(rc1->bottom - rc2->bottom) <= 1;
}

static int CALLBACK clip_emf_enum_proc(HDC hdc, HANDLETABLE *handle_table,
                                       const ENHMETARECORD *emr, int n_objs, LPARAM param)
{
    if (emr->iType == EMR_EXTSELECTCLIPRGN)
    {
        const EMREXTSELECTCLIPRGN *clip = (const EMREXTSELECTCLIPRGN *)emr;
        union _rgn
        {
            RGNDATA data;
            char buf[sizeof(RGNDATAHEADER) + sizeof(RECT)];
        };
        const union _rgn *rgn1;
        union _rgn rgn2;
        RECT rect, rc_transformed;
        const RECT *rc = (const RECT *)param;
        HRGN hrgn;
        XFORM xform;
        INT ret;

        if (winetest_debug > 1)
            trace("EMR_EXTSELECTCLIPRGN: cbRgnData %#lx, iMode %lu\n", clip->cbRgnData, clip->iMode);

        ok(clip->iMode == RGN_COPY, "expected RGN_COPY, got %lu\n", clip->iMode);
        ok(clip->cbRgnData >= sizeof(RGNDATAHEADER) + sizeof(RECT),
           "too small data block: %lu bytes\n", clip->cbRgnData);
        if (clip->cbRgnData < sizeof(RGNDATAHEADER) + sizeof(RECT))
            return 0;

        rgn1 = (const union _rgn *)clip->RgnData;

        if (winetest_debug > 1)
            trace("size %lu, type %lu, count %lu, rgn size %lu, bound %s\n",
                  rgn1->data.rdh.dwSize, rgn1->data.rdh.iType,
                  rgn1->data.rdh.nCount, rgn1->data.rdh.nRgnSize,
                  wine_dbgstr_rect(&rgn1->data.rdh.rcBound));

        ok(EqualRect(&rgn1->data.rdh.rcBound, rc), "rects don't match\n");

        rect = *(const RECT *)rgn1->data.Buffer;
        if (winetest_debug > 1) trace("rect %s\n", wine_dbgstr_rect(&rect));
        ok(EqualRect(&rect, rc), "rects don't match\n");

        ok(rgn1->data.rdh.dwSize == sizeof(rgn1->data.rdh), "expected sizeof(rdh), got %lu\n", rgn1->data.rdh.dwSize);
        ok(rgn1->data.rdh.iType == RDH_RECTANGLES, "expected RDH_RECTANGLES, got %lu\n", rgn1->data.rdh.iType);
        ok(rgn1->data.rdh.nCount == 1, "expected 1, got %lu\n", rgn1->data.rdh.nCount);
        ok(rgn1->data.rdh.nRgnSize == sizeof(RECT),
           "expected sizeof(RECT), got %lu\n", rgn1->data.rdh.nRgnSize);

        hrgn = CreateRectRgn(0, 0, 0, 0);

        memset(&xform, 0, sizeof(xform));
        SetLastError(0xdeadbeef);
        ret = GetWorldTransform(hdc, &xform);
        ok(ret, "GetWorldTransform error %lu\n", GetLastError());

        if (winetest_debug > 1) trace("xform.eM11 %f, xform.eM22 %f\n", xform.eM11, xform.eM22);

        ret = GetClipRgn(hdc, hrgn);
        ok(ret == 0, "GetClipRgn returned %d, expected 0\n", ret);

        PlayEnhMetaFileRecord(hdc, handle_table, emr, n_objs);

        ret = GetClipRgn(hdc, hrgn);
        ok(ret == 1, "GetClipRgn returned %d, expected 1\n", ret);

        ret = GetRegionData(hrgn, 0, NULL);
        ok(ret == sizeof(rgn2.data.rdh) + sizeof(RECT), "expected sizeof(rgn), got %u\n", ret);

        ret = GetRegionData(hrgn, sizeof(rgn2), &rgn2.data);
        ok(ret == sizeof(rgn2), "expected sizeof(rgn2), got %u\n", ret);

        if (winetest_debug > 1)
            trace("size %lu, type %lu, count %lu, rgn size %lu, bound %s\n", rgn2.data.rdh.dwSize,
                  rgn2.data.rdh.iType, rgn2.data.rdh.nCount, rgn2.data.rdh.nRgnSize,
                  wine_dbgstr_rect(&rgn2.data.rdh.rcBound));

        rect = rgn2.data.rdh.rcBound;
        rc_transformed = *rc;
        translate((POINT *)&rc_transformed, 2, &xform);
        if (winetest_debug > 1)
            trace("transformed %s\n", wine_dbgstr_rect(&rc_transformed));
        ok(is_equal_rect(&rect, &rc_transformed), "rects don't match\n");

        rect = *(const RECT *)rgn2.data.Buffer;
        if (winetest_debug > 1) trace("rect %s\n", wine_dbgstr_rect(&rect));
        rc_transformed = *rc;
        translate((POINT *)&rc_transformed, 2, &xform);
        if (winetest_debug > 1) trace("transformed %s\n", wine_dbgstr_rect(&rc_transformed));
        ok(is_equal_rect(&rect, &rc_transformed), "rects don't match\n");

        ok(rgn2.data.rdh.dwSize == sizeof(rgn1->data.rdh), "expected sizeof(rdh), got %lu\n", rgn2.data.rdh.dwSize);
        ok(rgn2.data.rdh.iType == RDH_RECTANGLES, "expected RDH_RECTANGLES, got %lu\n", rgn2.data.rdh.iType);
        ok(rgn2.data.rdh.nCount == 1, "expected 1, got %lu\n", rgn2.data.rdh.nCount);
        ok(rgn2.data.rdh.nRgnSize == sizeof(RECT),
           "expected sizeof(RECT), got %lu\n", rgn2.data.rdh.nRgnSize);

        DeleteObject(hrgn);
    }
    return 1;
}

static void test_emf_clipping(void)
{
    char buffer[100];
    RGNDATA *rgndata = (RGNDATA *)buffer;
    static const RECT rc = { 0, 0, 100, 100 };
    RECT rc_clip = { 100, 100, 1024, 1024 };
    HWND hwnd;
    HDC hdc;
    HENHMETAFILE hemf;
    HRGN hrgn;
    INT ret;
    RECT rc_res, rc_sclip;

    SetLastError(0xdeadbeef);
    hdc = CreateEnhMetaFileA(0, NULL, NULL, NULL);
    ok(hdc != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    /* Need to write something to the emf, otherwise Windows won't play it back */
    LineTo(hdc, 1, 1);

    hrgn = CreateRectRgn(rc_clip.left, rc_clip.top, rc_clip.right, rc_clip.bottom);
    ret = SelectClipRgn(hdc, hrgn);
    ok(ret == SIMPLEREGION, "expected SIMPLEREGION, got %d\n", ret);

    BeginPath(hdc);
    Rectangle(hdc, rc_clip.left, rc_clip.top, rc_clip.right, rc_clip.bottom);
    EndPath(hdc);
    ret = SelectClipPath(hdc, RGN_AND);
    ok(ret, "SelectClipPath error %ld\n", GetLastError());

    SetLastError(0xdeadbeef);
    hemf = CloseEnhMetaFile(hdc);
    ok(hemf != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    if (compare_emf_bits(hemf, EMF_CLIPPING, sizeof(EMF_CLIPPING),
        "emf_clipping", FALSE) != 0)
    {
        dump_emf_bits(hemf, "emf_clipping");
        dump_emf_records(hemf, "emf_clipping");
    }

    DeleteObject(hrgn);

    /* Win9x doesn't play EMFs on invisible windows */
    hwnd = CreateWindowExA(0, "static", NULL, WS_POPUP | WS_VISIBLE,
                           0, 0, 200, 200, 0, 0, 0, NULL);
    ok(hwnd != 0, "CreateWindowExA error %ld\n", GetLastError());

    hdc = GetDC(hwnd);

    ret = EnumEnhMetaFile(hdc, hemf, clip_emf_enum_proc, &rc_clip, &rc);
    ok(ret, "EnumEnhMetaFile error %ld\n", GetLastError());

    DeleteEnhMetaFile(hemf);
    ReleaseDC(hwnd, hdc);
    DestroyWindow(hwnd);

    hdc = CreateEnhMetaFileA(0, NULL, NULL, NULL);

    SetRect(&rc_sclip, 100, 100, GetSystemMetrics(SM_CXSCREEN), GetSystemMetrics(SM_CYSCREEN));
    hrgn = CreateRectRgn(rc_sclip.left, rc_sclip.top, rc_sclip.right, rc_sclip.bottom);
    SelectClipRgn(hdc, hrgn);
    SetRect(&rc_res, -1, -1, -1, -1);
    ret = GetClipBox(hdc, &rc_res);
    ok(ret == SIMPLEREGION, "got %d\n", ret);
    ok(EqualRect(&rc_res, &rc_sclip), "expected %s, got %s\n", wine_dbgstr_rect(&rc_sclip),
       wine_dbgstr_rect(&rc_res));

    OffsetRect(&rc_sclip, -100, -100);
    ret = OffsetClipRgn(hdc, -100, -100);
    ok(ret == SIMPLEREGION, "got %d\n", ret);
    SetRect(&rc_res, -1, -1, -1, -1);
    ret = GetClipBox(hdc, &rc_res);
    ok(ret == SIMPLEREGION, "got %d\n", ret);
    ok(EqualRect(&rc_res, &rc_sclip), "expected %s, got %s\n", wine_dbgstr_rect(&rc_sclip),
       wine_dbgstr_rect(&rc_res));

    ret = IntersectClipRect(hdc, 0, 0, 100, 100);
    /* all versions of Windows return COMPLEXREGION despite the region comprising one rectangle */
    ok(ret == SIMPLEREGION || broken(ret == COMPLEXREGION), "wrong region type %d\n", ret);
    ret = GetClipRgn(hdc, hrgn);
    ok(ret == 1, "expected 1, got %d\n", ret);
    ret = GetRegionData(hrgn, sizeof(buffer), rgndata);
    ok(ret == sizeof(RGNDATAHEADER) + sizeof(RECT), "got %u\n", ret);
    ok(rgndata->rdh.nCount == 1, "got %lu rectangles\n", rgndata->rdh.nCount);
    ok(EqualRect((RECT *)rgndata->Buffer, &rc), "got rect %s\n", wine_dbgstr_rect((RECT *)rgndata->Buffer));
    SetRect(&rc_res, -1, -1, -1, -1);
    ret = GetClipBox(hdc, &rc_res);
    ok(ret == SIMPLEREGION, "got %d\n", ret);
    ok(EqualRect(&rc_res, &rc), "expected %s, got %s\n", wine_dbgstr_rect(&rc),
       wine_dbgstr_rect(&rc_res));

    SetRect(&rc_sclip, 0, 0, 100, 50);
    ret = ExcludeClipRect(hdc, 0, 50, 100, 100);
    /* all versions of Windows return COMPLEXREGION despite the region comprising one rectangle */
    ok(ret == SIMPLEREGION || broken(ret == COMPLEXREGION), "wrong region type %d\n", ret);
    ret = GetClipRgn(hdc, hrgn);
    ok(ret == 1, "expected 1, got %d\n", ret);
    ret = GetRegionData(hrgn, sizeof(buffer), rgndata);
    ok(ret == sizeof(RGNDATAHEADER) + sizeof(RECT), "got %u\n", ret);
    ok(rgndata->rdh.nCount == 1, "got %lu rectangles\n", rgndata->rdh.nCount);
    ok(EqualRect((RECT *)rgndata->Buffer, &rc_sclip), "got rect %s\n", wine_dbgstr_rect((RECT *)rgndata->Buffer));
    SetRect(&rc_res, -1, -1, -1, -1);
    ret = GetClipBox(hdc, &rc_res);
    ok(ret == SIMPLEREGION, "got %d\n", ret);
    ok(EqualRect(&rc_res, &rc_sclip), "expected %s, got %s\n", wine_dbgstr_rect(&rc_sclip),
       wine_dbgstr_rect(&rc_res));

    hemf = CloseEnhMetaFile(hdc);
    DeleteEnhMetaFile(hemf);
    DeleteObject(hrgn);
}

static const unsigned char MF_CLIP_BITS[] = {
    /* METAHEADER */
    0x01, 0x00,             /* mtType */
    0x09, 0x00,             /* mtHeaderSize */
    0x00, 0x03,             /* mtVersion */
    0x32, 0x00, 0x00, 0x00, /* mtSize */
    0x01, 0x00,             /* mtNoObjects */
    0x14, 0x00, 0x00, 0x00, /* mtMaxRecord (size in words of longest record) */
    0x00, 0x00,             /* reserved */

    /* METARECORD for CreateRectRgn(0x11, 0x22, 0x33, 0x44) */
    0x14, 0x00, 0x00, 0x00, /* rdSize in words */
    0xff, 0x06,             /* META_CREATEREGION */
    0x00, 0x00, 0x06, 0x00, 0xf6, 0x02, 0x00, 0x00,
    0x24, 0x00, 0x01, 0x00, 0x02, 0x00, 0x11, 0x00,
    0x22, 0x00, 0x33, 0x00, 0x44, 0x00, 0x02, 0x00,
    0x22, 0x00, 0x44, 0x00, 0x11, 0x00, 0x33, 0x00,
    0x02, 0x00,

    /* METARECORD for SelectObject */
    0x04, 0x00, 0x00, 0x00,
    0x2d, 0x01,             /* META_SELECTOBJECT (not META_SELECTCLIPREGION?!) */
    0x00, 0x00,

    /* METARECORD */
    0x04, 0x00, 0x00, 0x00,
    0xf0, 0x01,             /* META_DELETEOBJECT */
    0x00, 0x00,

    /* METARECORD for MoveTo(1,0x30) */
    0x05, 0x00, 0x00, 0x00, /* rdSize in words */
    0x14, 0x02,             /* META_MOVETO */
    0x30, 0x00,             /* y */
    0x01, 0x00,             /* x */

    /* METARECORD for LineTo(0x20, 0x30) */
    0x05, 0x00, 0x00, 0x00, /* rdSize in words */
    0x13, 0x02,             /* META_LINETO */
    0x30, 0x00,             /* y */
    0x20, 0x00,             /* x */

    /* EOF */
    0x03, 0x00, 0x00, 0x00,
    0x00, 0x00
};

static int clip_mf_enum_proc_seen_selectclipregion;
static int clip_mf_enum_proc_seen_selectobject;

static int CALLBACK clip_mf_enum_proc(HDC hdc, HANDLETABLE *handle_table,
                                       METARECORD *mr, int n_objs, LPARAM param)
{
    switch (mr->rdFunction) {
    case META_SELECTCLIPREGION:
        clip_mf_enum_proc_seen_selectclipregion++;
        break;
    case META_SELECTOBJECT:
        clip_mf_enum_proc_seen_selectobject++;
        break;
    }
    return 1;
}

static void test_mf_clipping(void)
{
                          /* left top right bottom */
    static RECT rc_clip = { 0x11, 0x22, 0x33, 0x44 };
    HWND hwnd;
    HDC hdc;
    HMETAFILE hmf;
    HRGN hrgn;
    INT ret;

    SetLastError(0xdeadbeef);
    hdc = CreateMetaFileA(NULL);
    ok(hdc != 0, "CreateMetaFileA error %ld\n", GetLastError());

    hrgn = CreateRectRgn(rc_clip.left, rc_clip.top, rc_clip.right, rc_clip.bottom);
    ret = SelectClipRgn(hdc, hrgn);
    /* Seems like it should be SIMPLEREGION, but windows returns NULLREGION? */
    ok(ret == NULLREGION, "expected NULLREGION, got %d\n", ret);

    /* Draw a line that starts off left of the clip region and ends inside it */
    MoveToEx(hdc, 0x1, 0x30, NULL);
    LineTo(hdc,  0x20, 0x30);

    SetLastError(0xdeadbeef);
    hmf = CloseMetaFile(hdc);
    ok(hmf != 0, "CloseMetaFile error %ld\n", GetLastError());

    if (compare_mf_bits(hmf, MF_CLIP_BITS, sizeof(MF_CLIP_BITS),
        "mf_clipping") != 0)
    {
        dump_mf_bits(hmf, "mf_clipping");
    }

    DeleteObject(hrgn);

    hwnd = CreateWindowExA(0, "static", NULL, WS_POPUP | WS_VISIBLE,
                           0, 0, 200, 200, 0, 0, 0, NULL);
    ok(hwnd != 0, "CreateWindowExA error %ld\n", GetLastError());

    hdc = GetDC(hwnd);

    ret = EnumMetaFile(hdc, hmf, clip_mf_enum_proc, (LPARAM)&rc_clip);
    ok(ret, "EnumMetaFile error %ld\n", GetLastError());

    /* Oddly, windows doesn't seem to use META_SELECTCLIPREGION */
    ok(clip_mf_enum_proc_seen_selectclipregion == 0,
       "expected 0 selectclipregion, saw %d\n", clip_mf_enum_proc_seen_selectclipregion);
    ok(clip_mf_enum_proc_seen_selectobject == 1,
       "expected 1 selectobject, saw %d\n", clip_mf_enum_proc_seen_selectobject);

    DeleteMetaFile(hmf);
    ReleaseDC(hwnd, hdc);
    DestroyWindow(hwnd);
}

static const unsigned char MF_PATH_BITS[] =
{
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x2c, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x14, 0x02,
    0x32, 0x00, 0x32, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x13, 0x02, 0x96, 0x00, 0x32, 0x00, 0x05, 0x00,
    0x00, 0x00, 0x13, 0x02, 0x96, 0x00, 0x96, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x13, 0x02, 0x32, 0x00,
    0x96, 0x00, 0x05, 0x00, 0x00, 0x00, 0x13, 0x02,
    0x32, 0x00, 0x32, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x1b, 0x04, 0x14, 0x00, 0x14, 0x00, 0x0a, 0x00,
    0x0a, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00
};

static void test_mf_GetPath(void)
{
    HDC hdc;
    HMETAFILE hmf;
    BOOL ret;
    int size;

    SetLastError(0xdeadbeef);
    hdc = CreateMetaFileA(NULL);
    ok(hdc != 0, "CreateMetaFileA error %ld\n", GetLastError());

    ret = BeginPath(hdc);
    ok(!ret, "BeginPath on metafile DC should fail\n");
    ret = MoveToEx(hdc, 50, 50, NULL);
    ok( ret, "MoveToEx error %ld.\n", GetLastError());
    ret = LineTo(hdc, 50, 150);
    ok( ret, "LineTo error %ld.\n", GetLastError());
    ret = LineTo(hdc, 150, 150);
    ok( ret, "LineTo error %ld.\n", GetLastError());
    ret = LineTo(hdc, 150, 50);
    ok( ret, "LineTo error %ld.\n", GetLastError());
    ret = LineTo(hdc, 50, 50);
    ok( ret, "LineTo error %ld.\n", GetLastError());
    Rectangle(hdc, 10, 10, 20, 20);
    EndPath(hdc);

    size = GetPath(hdc, NULL, NULL, 0);
    ok( size == -1, "GetPath returned %d.\n", size);

    hmf = CloseMetaFile(hdc);
    ok(hmf != 0, "CloseMetaFile error %ld\n", GetLastError());

    if (compare_mf_bits (hmf, MF_PATH_BITS, sizeof(MF_PATH_BITS), "mf_GetPath") != 0)
    {
        dump_mf_bits(hmf, "mf_GetPath");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }

    ret = DeleteMetaFile(hmf);
    ok( ret, "DeleteMetaFile error %ld\n", GetLastError());
}

static INT CALLBACK EmfEnumProc(HDC hdc, HANDLETABLE *lpHTable, const ENHMETARECORD *lpEMFR, INT nObj, LPARAM lpData)
{
    LPMETAFILEPICT lpMFP = (LPMETAFILEPICT)lpData;
    POINT mapping[2] = { { 0, 0 }, { 10, 10 } };
    /* When using MM_TEXT Win9x does not update the mapping mode 
     * until a record is played which actually outputs something */
    PlayEnhMetaFileRecord(hdc, lpHTable, lpEMFR, nObj);
    LPtoDP(hdc, mapping, 2);
    if (winetest_debug > 1)
        trace("EMF record: iType %ld, nSize %ld, (%ld,%ld)-(%ld,%ld)\n",
              lpEMFR->iType, lpEMFR->nSize,
              mapping[0].x, mapping[0].y, mapping[1].x, mapping[1].y);

    if (lpEMFR->iType == EMR_LINETO)
    {
        INT x0, y0, x1, y1;
        if (!lpMFP || lpMFP->mm == MM_TEXT)
        {
            x0 = 0;
            y0 = 0;
            x1 = (INT)floor(10 * 100.0 / LINE_X + 0.5);
            y1 = (INT)floor(10 * 100.0 / LINE_Y + 0.5);
        }
        else
        {
            ok(lpMFP->mm == MM_ANISOTROPIC, "mm=%ld\n", lpMFP->mm);
            
            x0 = MulDiv(0, GetDeviceCaps(hdc, HORZSIZE) * 100, GetDeviceCaps(hdc, HORZRES));
            y0 = MulDiv(0, GetDeviceCaps(hdc, VERTSIZE) * 100, GetDeviceCaps(hdc, VERTRES));
            x1 = MulDiv(10, GetDeviceCaps(hdc, HORZSIZE) * 100, GetDeviceCaps(hdc, HORZRES));
            y1 = MulDiv(10, GetDeviceCaps(hdc, VERTSIZE) * 100, GetDeviceCaps(hdc, VERTRES));
        }
        ok(mapping[0].x == x0 && mapping[0].y == y0 && mapping[1].x == x1 && mapping[1].y == y1,
            "(%ld,%ld)->(%ld,%ld), expected (%d,%d)->(%d,%d)\n",
            mapping[0].x, mapping[0].y, mapping[1].x, mapping[1].y,
            x0, y0, x1, y1);
    }
    return TRUE;
}

static HENHMETAFILE create_converted_emf(const METAFILEPICT *mfp)
{
    HDC hdcMf;
    HMETAFILE hmf;
    HENHMETAFILE hemf;
    BOOL ret;
    UINT size;
    LPBYTE pBits;

    hdcMf = CreateMetaFileA(NULL);
    ok(hdcMf != NULL, "CreateMetaFile failed with error %ld\n", GetLastError());
    ret = LineTo(hdcMf, (INT)LINE_X, (INT)LINE_Y);
    ok(ret, "LineTo failed with error %ld\n", GetLastError());
    hmf = CloseMetaFile(hdcMf);
    ok(hmf != NULL, "CloseMetaFile failed with error %ld\n", GetLastError());

    if (compare_mf_bits (hmf, MF_LINETO_BITS, sizeof(MF_LINETO_BITS), "mf_LineTo") != 0)
    {
        dump_mf_bits(hmf, "mf_LineTo");
        EnumMetaFile(0, hmf, mf_enum_proc, 0);
    }

    size = GetMetaFileBitsEx(hmf, 0, NULL);
    ok(size, "GetMetaFileBitsEx failed with error %ld\n", GetLastError());
    pBits = HeapAlloc(GetProcessHeap(), 0, size);
    GetMetaFileBitsEx(hmf, size, pBits);
    DeleteMetaFile(hmf);
    hemf = SetWinMetaFileBits(size, pBits, NULL, mfp);
    HeapFree(GetProcessHeap(), 0, pBits);
    return hemf;
}

static void test_mf_conversions(void)
{
    trace("Testing MF->EMF conversion (MM_ANISOTROPIC)\n");
    {
        HDC hdcOffscreen = CreateCompatibleDC(NULL);
        HENHMETAFILE hemf;
        METAFILEPICT mfp;
        RECT rect = { 0, 0, 100, 100 };
        mfp.mm = MM_ANISOTROPIC;
        mfp.xExt = 100;
        mfp.yExt = 100;
        mfp.hMF = NULL;
        hemf = create_converted_emf(&mfp);

        if (compare_emf_bits(hemf, EMF_LINETO_MM_ANISOTROPIC_BITS, sizeof(EMF_LINETO_MM_ANISOTROPIC_BITS),
                             "emf_LineTo MM_ANISOTROPIC", TRUE) != 0)
        {
            dump_emf_bits(hemf, "emf_LineTo MM_ANISOTROPIC");
            dump_emf_records(hemf, "emf_LineTo MM_ANISOTROPIC");
        }

        EnumEnhMetaFile(hdcOffscreen, hemf, EmfEnumProc, &mfp, &rect);

        DeleteEnhMetaFile(hemf);
        DeleteDC(hdcOffscreen);
    }

    trace("Testing MF->EMF conversion (MM_TEXT)\n");
    {
        HDC hdcOffscreen = CreateCompatibleDC(NULL);
        HENHMETAFILE hemf;
        METAFILEPICT mfp;
        RECT rect = { 0, 0, 100, 100 };
        mfp.mm = MM_TEXT;
        mfp.xExt = 0;
        mfp.yExt = 0;
        mfp.hMF = NULL;
        hemf = create_converted_emf(&mfp);

        if (compare_emf_bits(hemf, EMF_LINETO_MM_TEXT_BITS, sizeof(EMF_LINETO_MM_TEXT_BITS),
                             "emf_LineTo MM_TEXT", TRUE) != 0)
        {
            dump_emf_bits(hemf, "emf_LineTo MM_TEXT");
            dump_emf_records(hemf, "emf_LineTo MM_TEXT");
        }

        EnumEnhMetaFile(hdcOffscreen, hemf, EmfEnumProc, &mfp, &rect);

        DeleteEnhMetaFile(hemf);
        DeleteDC(hdcOffscreen);
    }

    trace("Testing MF->EMF conversion (NULL mfp)\n");
    {
        HDC hdcOffscreen = CreateCompatibleDC(NULL);
        HENHMETAFILE hemf;
        RECT rect = { 0, 0, 100, 100 };
        hemf = create_converted_emf(NULL);

        if (compare_emf_bits(hemf, EMF_LINETO_BITS, sizeof(EMF_LINETO_BITS),
                             "emf_LineTo NULL", TRUE) != 0)
        {
            dump_emf_bits(hemf, "emf_LineTo NULL");
            dump_emf_records(hemf, "emf_LineTo NULL");
        }

        EnumEnhMetaFile(hdcOffscreen, hemf, EmfEnumProc, NULL, &rect);

        DeleteEnhMetaFile(hemf);
        DeleteDC(hdcOffscreen);
    }
}

static BOOL getConvertedFrameAndBounds(UINT buffer_size, BYTE * buffer, BOOL mfpIsNull,
                                       LONG mm, LONG xExt, LONG yExt,
                                       RECTL * rclBounds, RECTL * rclFrame)
{
  METAFILEPICT mfp;
  METAFILEPICT * mfpPtr = NULL;
  HENHMETAFILE emf;
  ENHMETAHEADER header;
  UINT res;

  if (!mfpIsNull)
  {
    mfp.mm = mm;
    mfp.xExt = xExt;
    mfp.yExt = yExt;
    mfpPtr = &mfp;
  }

  emf = SetWinMetaFileBits(buffer_size, buffer, NULL, mfpPtr);
  ok(emf != NULL, "SetWinMetaFileBits failed\n");
  if (!emf) return FALSE;
  res = GetEnhMetaFileHeader(emf, sizeof(header), &header);
  ok(res != 0, "GetEnhMetaHeader failed\n");
  DeleteEnhMetaFile(emf);
  if (!res) return FALSE;

  *rclBounds = header.rclBounds;
  *rclFrame = header.rclFrame;
  return TRUE;
}

static void checkConvertedFrameAndBounds(UINT buffer_size, BYTE * buffer, BOOL mfpIsNull,
                                         LONG mm, LONG xExt, LONG yExt,
                                         RECTL * rclBoundsExpected, RECTL * rclFrameExpected)
{
  RECTL rclBounds, rclFrame;

  if (getConvertedFrameAndBounds(buffer_size, buffer, mfpIsNull, mm, xExt, yExt, &rclBounds, &rclFrame))
  {
    const char * msg;
    char buf[64];

    if (mfpIsNull)
    {
       msg = "mfp == NULL";
    }
    else
    {
      const char * mm_str;
      switch (mm)
      {
         case MM_ANISOTROPIC: mm_str = "MM_ANISOTROPIC"; break;
         case MM_ISOTROPIC:   mm_str = "MM_ISOTROPIC"; break;
         default:             mm_str = "Unexpected";
      }
      sprintf(buf, "mm=%s, xExt=%ld, yExt=%ld", mm_str, xExt, yExt);
      msg = buf;
    }

    ok(rclBounds.left == rclBoundsExpected->left, "rclBounds.left: Expected %ld, got %ld (%s)\n", rclBoundsExpected->left, rclBounds.left, msg);
    ok(rclBounds.top == rclBoundsExpected->top, "rclBounds.top: Expected %ld, got %ld (%s)\n", rclBoundsExpected->top, rclBounds.top, msg);
    ok(rclBounds.right == rclBoundsExpected->right, "rclBounds.right: Expected %ld, got %ld (%s)\n", rclBoundsExpected->right, rclBounds.right, msg);
    ok(rclBounds.bottom == rclBoundsExpected->bottom, "rclBounds.bottom: Expected %ld, got %ld (%s)\n", rclBoundsExpected->bottom, rclBounds.bottom, msg);
    ok(rclFrame.left == rclFrameExpected->left, "rclFrame.left: Expected %ld, got %ld (%s)\n", rclFrameExpected->left, rclFrame.left, msg);
    ok(rclFrame.top == rclFrameExpected->top, "rclFrame.top: Expected %ld, got %ld (%s)\n", rclFrameExpected->top, rclFrame.top, msg);
    ok(rclFrame.right == rclFrameExpected->right, "rclFrame.right: Expected %ld, got %ld (%s)\n", rclFrameExpected->right, rclFrame.right, msg);
    ok(rclFrame.bottom == rclFrameExpected->bottom, "rclFrame.bottom: Expected %ld, got %ld (%s)\n", rclFrameExpected->bottom, rclFrame.bottom, msg);
  }
}

static void test_SetWinMetaFileBits(void)
{
  HMETAFILE wmf;
  HDC wmfDC;
  BYTE * buffer;
  UINT buffer_size;
  RECT rect;
  UINT res;
  RECTL rclBoundsAnisotropic, rclFrameAnisotropic;
  RECTL rclBoundsIsotropic, rclFrameIsotropic;
  RECTL rclBounds, rclFrame;
  HDC dc;
  LONG diffx, diffy;

  wmfDC = CreateMetaFileA(NULL);
  ok(wmfDC != NULL, "CreateMetaFile failed\n");
  if (!wmfDC) return;

  SetWindowExtEx(wmfDC, 100, 100, NULL);
  SetRect(&rect, 0, 0, 50, 50);
  FillRect(wmfDC, &rect, GetStockObject(BLACK_BRUSH));
  wmf = CloseMetaFile(wmfDC);
  ok(wmf != NULL, "Metafile creation failed\n");
  if (!wmf) return;

  buffer_size = GetMetaFileBitsEx(wmf, 0, NULL);
  ok(buffer_size != 0, "GetMetaFileBitsEx failed\n");
  if (buffer_size == 0)
  {
    DeleteMetaFile(wmf);
    return;
  }

  buffer = HeapAlloc(GetProcessHeap(), 0, buffer_size);
  ok(buffer != NULL, "HeapAlloc failed\n");
  if (!buffer)
  {
    DeleteMetaFile(wmf);
    return;
  }

  res = GetMetaFileBitsEx(wmf, buffer_size, buffer);
  ok(res == buffer_size, "GetMetaFileBitsEx failed\n");
  DeleteMetaFile(wmf);
  if (res != buffer_size)
  {
     HeapFree(GetProcessHeap(), 0, buffer);
     return;
  }

  /* Get the reference bounds and frame */
  getConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ANISOTROPIC, 0, 0, &rclBoundsAnisotropic, &rclFrameAnisotropic);
  getConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ISOTROPIC, 0, 0,  &rclBoundsIsotropic, &rclFrameIsotropic);

  ok(rclBoundsAnisotropic.left == 0 && rclBoundsAnisotropic.top == 0 &&
     rclBoundsIsotropic.left == 0 && rclBoundsIsotropic.top == 0,
     "SetWinMetaFileBits: Reference bounds: Left and top bound must be zero\n");

  ok(rclBoundsAnisotropic.right >= rclBoundsIsotropic.right, "SetWinMetaFileBits: Reference bounds: Invalid right bound\n");
  ok(rclBoundsAnisotropic.bottom >= rclBoundsIsotropic.bottom, "SetWinMetaFileBits: Reference bounds: Invalid bottom bound\n");
  diffx = rclBoundsIsotropic.right - rclBoundsIsotropic.bottom;
  if (diffx < 0) diffx = -diffx;
  ok(diffx <= 1, "SetWinMetaFileBits (MM_ISOTROPIC): Reference bounds are not isotropic\n");

  dc = CreateCompatibleDC(NULL);

  /* Allow 1 mm difference (rounding errors) */
  diffx = rclBoundsAnisotropic.right - GetDeviceCaps(dc, HORZRES) / 2;
  diffy = rclBoundsAnisotropic.bottom - GetDeviceCaps(dc, VERTRES) / 2;
  if (diffx < 0) diffx = -diffx;
  if (diffy < 0) diffy = -diffy;
  todo_wine
  {
  ok(diffx <= 1 && diffy <= 1,
     "SetWinMetaFileBits (MM_ANISOTROPIC): Reference bounds: The whole device surface must be used (%dx%d), but got (%ldx%ld)\n",
     GetDeviceCaps(dc, HORZRES) / 2, GetDeviceCaps(dc, VERTRES) / 2, rclBoundsAnisotropic.right, rclBoundsAnisotropic.bottom);
  }

  /* Allow 1 mm difference (rounding errors) */
  diffx = rclFrameAnisotropic.right / 100 - GetDeviceCaps(dc, HORZSIZE) / 2;
  diffy = rclFrameAnisotropic.bottom / 100 - GetDeviceCaps(dc, VERTSIZE) / 2;
  if (diffx < 0) diffx = -diffx;
  if (diffy < 0) diffy = -diffy;
  todo_wine
  {
  ok(diffx <= 1 && diffy <= 1,
     "SetWinMetaFileBits (MM_ANISOTROPIC): Reference frame: The whole device surface must be used (%dx%d), but got (%ldx%ld)\n",
     GetDeviceCaps(dc, HORZSIZE) / 2, GetDeviceCaps(dc, VERTSIZE) / 2, rclFrameAnisotropic.right / 100, rclFrameAnisotropic.bottom / 100);
  }
  DeleteDC(dc);

  /* If the METAFILEPICT pointer is NULL, the MM_ANISOTROPIC mapping mode and the whole device surface are used */
  checkConvertedFrameAndBounds(buffer_size, buffer, TRUE, 0, 0, 0, &rclBoundsAnisotropic, &rclFrameAnisotropic);

  /* If xExt or yExt is zero or negative, the whole device surface is used */
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ANISOTROPIC, 10000, 0, &rclBoundsAnisotropic, &rclFrameAnisotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ISOTROPIC, 10000, 0, &rclBoundsIsotropic, &rclFrameIsotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ANISOTROPIC, 0, 10000, &rclBoundsAnisotropic, &rclFrameAnisotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ISOTROPIC, 0, 10000, &rclBoundsIsotropic, &rclFrameIsotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ANISOTROPIC, -10000, 0, &rclBoundsAnisotropic, &rclFrameAnisotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ISOTROPIC, -10000, 0, &rclBoundsIsotropic, &rclFrameIsotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ANISOTROPIC, 0, -10000, &rclBoundsAnisotropic, &rclFrameAnisotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ISOTROPIC, 0, -10000, &rclBoundsIsotropic, &rclFrameIsotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ANISOTROPIC, -10000, 10000, &rclBoundsAnisotropic, &rclFrameAnisotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ISOTROPIC, -10000, 10000, &rclBoundsIsotropic, &rclFrameIsotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ANISOTROPIC, 10000, -10000, &rclBoundsAnisotropic, &rclFrameAnisotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ISOTROPIC, 10000, -10000, &rclBoundsIsotropic, &rclFrameIsotropic);

  /* MSDN says that negative xExt and yExt values specify a ratio.
     Check that this is wrong and the whole device surface is used */
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ANISOTROPIC, -1000, -100, &rclBoundsAnisotropic, &rclFrameAnisotropic);
  checkConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ISOTROPIC, -1000, -100, &rclBoundsIsotropic, &rclFrameIsotropic);

  /* Ordinary conversions */

  if (getConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ANISOTROPIC, 30000, 20000, &rclBounds, &rclFrame))
  {
    ok(rclFrame.left == 0 && rclFrame.top == 0 && rclFrame.right == 30000 && rclFrame.bottom == 20000,
       "SetWinMetaFileBits (MM_ANISOTROPIC): rclFrame contains invalid values\n");
    ok(rclBounds.left == 0 && rclBounds.top == 0 && rclBounds.right > rclBounds.bottom,
       "SetWinMetaFileBits (MM_ANISOTROPIC): rclBounds contains invalid values\n");
  }

  if (getConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_ISOTROPIC, 30000, 20000, &rclBounds, &rclFrame))
  {
    ok(rclFrame.left == 0 && rclFrame.top == 0 && rclFrame.right == 30000 && rclFrame.bottom == 20000,
       "SetWinMetaFileBits (MM_ISOTROPIC): rclFrame contains invalid values\n");
    ok(rclBounds.left == 0 && rclBounds.top == 0,
       "SetWinMetaFileBits (MM_ISOTROPIC): rclBounds contains invalid values\n");

    /* Wine has a rounding error */
    diffx = rclBounds.right - rclBounds.bottom;
    if (diffx < 0) diffx = -diffx;
    ok(diffx <= 1, "SetWinMetaFileBits (MM_ISOTROPIC): rclBounds is not isotropic\n");
  }

  if (getConvertedFrameAndBounds(buffer_size, buffer, FALSE, MM_HIMETRIC, 30000, 20000, &rclBounds, &rclFrame))
  {
    ok(rclFrame.right - rclFrame.left != 30000 && rclFrame.bottom - rclFrame.top != 20000,
       "SetWinMetaFileBits: xExt and yExt must be ignored for mapping modes other than MM_ANISOTROPIC and MM_ISOTROPIC\n");
  }

  HeapFree(GetProcessHeap(), 0, buffer);
}

static BOOL near_match(int x, int y)
{
    int epsilon = min(abs(x), abs(y));

    epsilon = max(epsilon/100, 2);

    if(x < y - epsilon || x > y + epsilon) return FALSE;
    return TRUE;
}

static void getwinmetafilebits(UINT mode, int scale, RECT *rc)
{
    HENHMETAFILE emf, emf2;
    HDC display_dc, emf_dc;
    ENHMETAHEADER *enh_header, *enh2_header;
    UINT size, emf_size, i, emf2_size;
    WORD check = 0;
    DWORD rec_num = 0;
    METAHEADER *mh = NULL;
    METARECORD *rec;
    INT horz_res, vert_res, horz_size, vert_size;
    INT curve_caps, line_caps, poly_caps;
    METAFILEPICT mfp;

    display_dc = GetDC(NULL);
    ok(display_dc != NULL, "display_dc is NULL\n");

    horz_res = GetDeviceCaps(display_dc, HORZRES);
    vert_res = GetDeviceCaps(display_dc, VERTRES);
    horz_size = GetDeviceCaps(display_dc, HORZSIZE);
    vert_size = GetDeviceCaps(display_dc, VERTSIZE);

    emf_dc = CreateEnhMetaFileA(display_dc, NULL, rc, NULL);
    ok(emf_dc != NULL, "emf_dc is NULL\n");

    curve_caps = GetDeviceCaps(emf_dc, CURVECAPS);
    ok(curve_caps == 511, "expect 511 got %d\n", curve_caps);

    line_caps = GetDeviceCaps(emf_dc, LINECAPS);
    ok(line_caps == 254, "expect 254 got %d\n", line_caps);

    poly_caps = GetDeviceCaps(emf_dc, POLYGONALCAPS);
    ok(poly_caps == 255, "expect 511 got %d\n", poly_caps);

    for(i = 0; i < 3000; i++) /* This is enough to take emf_size > 0xffff */
        Rectangle(emf_dc, 0, 0, 1000, 20);
    emf = CloseEnhMetaFile(emf_dc);
    ok(emf != NULL, "emf is NULL\n");

    emf_size = GetEnhMetaFileBits(emf, 0, NULL);
    enh_header = HeapAlloc(GetProcessHeap(), 0, emf_size);
    emf_size = GetEnhMetaFileBits(emf, emf_size, (BYTE*)enh_header);
    DeleteEnhMetaFile(emf);
    /* multiply szlDevice.cx by scale, when scale != 1 the recording and playback dcs
       have different resolutions */
    enh_header->szlDevice.cx *= scale;
    emf = SetEnhMetaFileBits(emf_size, (BYTE*)enh_header);
    ok(emf != NULL, "emf is NULL\n");
    ok(EqualRect((RECT*)&enh_header->rclFrame, rc), "Frame rectangles differ\n");

    size = GetWinMetaFileBits(emf, 0, NULL, mode, display_dc);
    ok(size ||
       broken(size == 0), /* some versions of winxp fail for some reason */
       "GetWinMetaFileBits returns 0\n");
    if(!size) goto end;
    mh = HeapAlloc(GetProcessHeap(), 0, size);
    GetWinMetaFileBits(emf, size, (BYTE*)mh, mode, display_dc);

    for(i = 0; i < size / 2; i++) check += ((WORD*)mh)[i];
    ok(check == 0, "check %04x\n", check);

    rec = (METARECORD*)(mh + 1);

    while(rec->rdSize && rec->rdFunction)
    {
        const DWORD chunk_size = 0x2000;
        DWORD mfcomment_chunks = (emf_size + chunk_size - 1) / chunk_size;

        if(rec_num < mfcomment_chunks)
        {
            DWORD this_chunk_size = chunk_size;

            if(rec_num == mfcomment_chunks - 1)
                this_chunk_size = emf_size - rec_num * chunk_size;

            ok(rec->rdSize == (this_chunk_size + 44) / 2, "%04lx: got %04lx expected %04lx\n", rec_num, rec->rdSize, (this_chunk_size + 44) / 2);
            ok(rec->rdFunction == META_ESCAPE, "%04lx: got %04x\n", rec_num, rec->rdFunction);
            if(rec->rdSize < (this_chunk_size + 44) / 2) break;
            ok(rec->rdParm[0] == MFCOMMENT, "got %04x\n", rec->rdParm[0]);
            ok(rec->rdParm[1] == this_chunk_size + 34, "got %04x %x\n", rec->rdParm[1], emf_size + 34);
            ok(rec->rdParm[2] == 0x4d57, "got %04x\n", rec->rdParm[2]); /* WMFC */
            ok(rec->rdParm[3] == 0x4346, "got %04x\n", rec->rdParm[3]); /*  "   */
            ok(rec->rdParm[4] == 1, "got %04x\n", rec->rdParm[4]);
            ok(rec->rdParm[5] == 0, "got %04x\n", rec->rdParm[5]);
            ok(rec->rdParm[6] == 0, "got %04x\n", rec->rdParm[6]);
            ok(rec->rdParm[7] == 1, "got %04x\n", rec->rdParm[7]);
            /* parm[8] is the checksum, tested above */
            if(rec_num > 0) ok(rec->rdParm[8] == 0, "got %04x\n", rec->rdParm[8]);
            ok(rec->rdParm[9] == 0, "got %04x\n", rec->rdParm[9]);
            ok(rec->rdParm[10] == 0, "got %04x\n", rec->rdParm[10]);
            ok(rec->rdParm[11] == mfcomment_chunks, "got %04x\n", rec->rdParm[11]); /* num chunks */
            ok(rec->rdParm[12] == 0, "got %04x\n", rec->rdParm[12]);
            ok(rec->rdParm[13] == this_chunk_size, "got %04x expected %04lx\n", rec->rdParm[13], this_chunk_size);
            ok(rec->rdParm[14] == 0, "got %04x\n", rec->rdParm[14]);
            ok(*(DWORD*)(rec->rdParm + 15) == emf_size - this_chunk_size - rec_num * chunk_size, "got %08lx\n", *(DWORD*)(rec->rdParm + 15));  /* DWORD size remaining after current chunk */
            ok(*(DWORD*)(rec->rdParm + 17) == emf_size, "got %08lx emf_size %08x\n", *(DWORD*)(rec->rdParm + 17), emf_size);
            ok(!memcmp(rec->rdParm + 19, (char*)enh_header + rec_num * chunk_size, this_chunk_size), "bits mismatch\n");
        }

        else if(rec_num == mfcomment_chunks)
        {
            ok(rec->rdFunction == META_SETMAPMODE, "got %04x\n", rec->rdFunction);
            ok(rec->rdParm[0] == mode, "got %04x\n", rec->rdParm[0]);
        }
        else if(rec_num == mfcomment_chunks + 1)
        {
            POINT pt;
            ok(rec->rdFunction == META_SETWINDOWORG, "got %04x\n", rec->rdFunction);
            switch(mode)
            {
            case MM_TEXT:
            case MM_ISOTROPIC:
            case MM_ANISOTROPIC:
                pt.y = MulDiv(rc->top, vert_res, vert_size * 100) + 1;
                pt.x = MulDiv(rc->left, horz_res, horz_size * 100);
                break;
            case MM_LOMETRIC:
                pt.y = MulDiv(-rc->top, 1, 10) + 1;
                pt.x = MulDiv( rc->left, 1, 10);
                break;
            case MM_HIMETRIC:
                pt.y = -rc->top + 1;
                pt.x = (rc->left >= 0) ? rc->left : rc->left + 1; /* strange but true */
                break;
            case MM_LOENGLISH:
                pt.y = MulDiv(-rc->top, 10, 254) + 1;
                pt.x = MulDiv( rc->left, 10, 254);
                break;
            case MM_HIENGLISH:
                pt.y = MulDiv(-rc->top, 100, 254) + 1;
                pt.x = MulDiv( rc->left, 100, 254);
                break;
            case MM_TWIPS:
                pt.y = MulDiv(-rc->top, 72 * 20, 2540) + 1;
                pt.x = MulDiv( rc->left, 72 * 20, 2540);
                break;
            default:
                pt.x = pt.y = 0;
            }
            ok(near_match((short)rec->rdParm[0], pt.y), "got %d expect %ld\n", (short)rec->rdParm[0], pt.y);
            ok(near_match((short)rec->rdParm[1], pt.x), "got %d expect %ld\n", (short)rec->rdParm[1], pt.x);
        }
        if(rec_num == mfcomment_chunks + 2)
        {
            ok(rec->rdFunction == META_SETWINDOWEXT, "got %04x\n", rec->rdFunction);
            ok(near_match((short)rec->rdParm[0], MulDiv(rc->bottom - rc->top, vert_res, vert_size * 100)),
               "got %d\n", (short)rec->rdParm[0]);
            ok(near_match((short)rec->rdParm[1], MulDiv(rc->right - rc->left, horz_res, horz_size * 100)),
               "got %d\n", (short)rec->rdParm[1]);
        }

        rec_num++;
        rec = (METARECORD*)((WORD*)rec + rec->rdSize);
    }

    /* Show that we get the original back when we do the reverse conversion.
       mfp is ignored in this case. */
    mfp.mm = MM_ISOTROPIC;
    mfp.xExt = 0xcafe;
    mfp.yExt = 0xbeef;
    emf2 = SetWinMetaFileBits( size, (BYTE*)mh, NULL, &mfp );
    ok( !!emf2, "got NULL\n" );
    emf2_size = GetEnhMetaFileBits( emf2, 0, NULL );
    enh2_header = HeapAlloc( GetProcessHeap(), 0, emf2_size );
    emf2_size = GetEnhMetaFileBits( emf2, emf2_size, (BYTE*)enh2_header );
    ok( emf_size == emf2_size, "%d %d\n", emf_size, emf2_size );
    ok( !memcmp( enh_header, enh2_header, emf_size ), "mismatch\n" );
    HeapFree( GetProcessHeap(), 0, enh2_header );
    DeleteEnhMetaFile( emf2 );

end:
    HeapFree(GetProcessHeap(), 0, mh);
    HeapFree(GetProcessHeap(), 0, enh_header);
    DeleteEnhMetaFile(emf);

    ReleaseDC(NULL, display_dc);
}

static void test_GetWinMetaFileBits(void)
{
    UINT mode;
    RECT frames[] =
    {
        { 1000,  2000, 3000, 6000},
        {-1000,  2000, 3000, 6000},
        { 1000, -2000, 3000, 6000},
        { 1005,  2005, 3000, 6000},
        {-1005, -2005, 3000, 6000},
        {-1005, -2010, 3000, 6000},
        {-1005,  2010, 3000, 6000},
        {    0,     0,    1,    1},
        {   -1,    -1,    1,    1},
        {    0,     0,    0,    0}
    };

    for(mode = MM_MIN; mode <= MM_MAX; mode++)
    {
        RECT *rc;
        for(rc = frames; rc->right - rc->left > 0; rc++)
        {
            getwinmetafilebits(mode, 1, rc);
            getwinmetafilebits(mode, 2, rc);
        }
    }
}

static BOOL (WINAPI *pGdiIsMetaPrintDC)(HDC);
static BOOL (WINAPI *pGdiIsMetaFileDC)(HDC);
static BOOL (WINAPI *pGdiIsPlayMetafileDC)(HDC);

static void test_gdiis(void)
{
    RECT rect = {0,0,100,100};
    HDC hdc, hemfDC, hmfDC;
    HENHMETAFILE hemf;
    HMODULE hgdi32;

    /* resolve all the functions */
    hgdi32 = GetModuleHandleA("gdi32.dll");
    pGdiIsMetaPrintDC = (void*) GetProcAddress(hgdi32, "GdiIsMetaPrintDC");
    pGdiIsMetaFileDC = (void*) GetProcAddress(hgdi32, "GdiIsMetaFileDC");
    pGdiIsPlayMetafileDC = (void*) GetProcAddress(hgdi32, "GdiIsPlayMetafileDC");

    if(!pGdiIsMetaPrintDC || !pGdiIsMetaFileDC || !pGdiIsPlayMetafileDC)
    {
        win_skip("Needed GdiIs* functions are not available\n");
        return;
    }

    /* try with nothing */
    ok(!pGdiIsMetaPrintDC(NULL), "ismetaprint with NULL parameter\n");
    ok(!pGdiIsMetaFileDC(NULL), "ismetafile with NULL parameter\n");
    ok(!pGdiIsPlayMetafileDC(NULL), "isplaymetafile with NULL parameter\n");

    /* try with a metafile */
    hmfDC = CreateMetaFileA(NULL);
    ok(!pGdiIsMetaPrintDC(hmfDC), "ismetaprint on metafile\n");
    ok(pGdiIsMetaFileDC(hmfDC), "ismetafile on metafile\n");
    ok(!pGdiIsPlayMetafileDC(hmfDC), "isplaymetafile on metafile\n");
    DeleteMetaFile(CloseMetaFile(hmfDC));

    /* try with an enhanced metafile */
    hdc = GetDC(NULL);
    hemfDC = CreateEnhMetaFileW(hdc, NULL, &rect, NULL);
    ok(hemfDC != NULL, "failed to create emf\n");

    ok(!pGdiIsMetaPrintDC(hemfDC), "ismetaprint on emf\n");
    ok(pGdiIsMetaFileDC(hemfDC), "ismetafile on emf\n");
    ok(!pGdiIsPlayMetafileDC(hemfDC), "isplaymetafile on emf\n");

    hemf = CloseEnhMetaFile(hemfDC);
    ok(hemf != NULL, "failed to close EMF\n");
    DeleteEnhMetaFile(hemf);
    ReleaseDC(NULL,hdc);
}

static void test_SetEnhMetaFileBits(void)
{
    BYTE data[256];
    HENHMETAFILE hemf;
    ENHMETAHEADER *emh;

    memset(data, 0xAA, sizeof(data));
    SetLastError(0xdeadbeef);
    hemf = SetEnhMetaFileBits(sizeof(data), data);
    ok(!hemf, "SetEnhMetaFileBits should fail\n");
    ok(GetLastError() == ERROR_INVALID_DATA,
       "expected ERROR_INVALID_DATA, got %lu\n", GetLastError());

    emh = (ENHMETAHEADER *)data;
    memset(emh, 0, sizeof(*emh));

    emh->iType = EMR_HEADER;
    emh->nSize = sizeof(*emh);
    emh->dSignature = ENHMETA_SIGNATURE;
    /* emh->nVersion  = 0x10000; XP doesn't care about version */
    emh->nBytes = sizeof(*emh);
    /* emh->nRecords = 1; XP doesn't care about records */
    emh->nHandles = 1; /* XP refuses to load a EMF if nHandles == 0 */

    SetLastError(0xdeadbeef);
    hemf = SetEnhMetaFileBits(emh->nBytes, data);
    ok(hemf != 0, "SetEnhMetaFileBits error %lu\n", GetLastError());
    DeleteEnhMetaFile(hemf);

    /* XP refuses to load unaligned EMF */
    emh->nBytes++;
    SetLastError(0xdeadbeef);
    hemf = SetEnhMetaFileBits(emh->nBytes, data);
    ok(!hemf, "SetEnhMetaFileBits should fail\n");
    ok(GetLastError() == 0xdeadbeef, "Expected deadbeef, got %lu\n", GetLastError());
    DeleteEnhMetaFile(hemf);

    emh->dSignature = 0;
    emh->nBytes--;
    SetLastError(0xdeadbeef);
    hemf = SetEnhMetaFileBits(emh->nBytes, data);
    ok(!hemf, "SetEnhMetaFileBits should fail\n");
    ok(GetLastError() == 0xdeadbeef, "Expected deadbeef, got %lu\n", GetLastError());
    DeleteEnhMetaFile(hemf);
}

static void test_emf_polybezier(void)
{
    HDC hdcMetafile;
    HENHMETAFILE hemf;
    POINT pts[4];
    BOOL ret;

    SetLastError(0xdeadbeef);
    hdcMetafile = CreateEnhMetaFileA(GetDC(0), NULL, NULL, NULL);
    ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    pts[0].x = pts[0].y = 10;
    pts[1].x = pts[1].y = 20;
    pts[2].x = pts[2].y = 15;
    pts[3].x = pts[3].y = 25;
    ret = PolyBezierTo(hdcMetafile, pts, 3);  /* EMR_POLYBEZIERTO16 */
    ok( ret, "PolyBezierTo failed\n" );
    ret = PolyBezier(hdcMetafile, pts, 4);    /* EMR_POLYBEZIER16   */
    ok( ret, "PolyBezier failed\n" );

    pts[0].x = pts[0].y = 32769;
    ret = PolyBezier(hdcMetafile, pts, 4);    /* EMR_POLYBEZIER   */
    ok( ret, "PolyBezier failed\n" );
    ret = PolyBezierTo(hdcMetafile, pts, 3);  /* EMR_POLYBEZIERTO */
    ok( ret, "PolyBezierTo failed\n" );

    hemf = CloseEnhMetaFile(hdcMetafile);
    ok(hemf != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    if(compare_emf_bits(hemf, EMF_BEZIER_BITS, sizeof(EMF_BEZIER_BITS),
        "emf_Bezier", FALSE) != 0)
    {
        dump_emf_bits(hemf, "emf_Bezier");
        dump_emf_records(hemf, "emf_Bezier");
    }

    DeleteEnhMetaFile(hemf);
}

static const unsigned char EMF_PATH_BITS[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x96, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00,
    0x08, 0x01, 0x00, 0x00, 0x09, 0x01, 0x00, 0x00,
    0x7e, 0x0f, 0x00, 0x00, 0x80, 0x0f, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x00, 0x03, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x97, 0x06, 0x00, 0x00, 0xb1, 0x03, 0x00, 0x00,
    0xbe, 0x01, 0x00, 0x00, 0xfa, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x90, 0xcf, 0x06, 0x00,
    0xaf, 0xd0, 0x03, 0x00, 0x3b, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00,
    0x32, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00,
    0x96, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00,
    0x96, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00,
    0x32, 0x00, 0x00, 0x00, 0x36, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00,
    0x32, 0x00, 0x00, 0x00, 0x2b, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
    0x13, 0x00, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00,
    0x15, 0x00, 0x00, 0x00, 0x26, 0x00, 0x00, 0x00,
    0x1c, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00,
    0x1d, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00,
    0x15, 0x00, 0x00, 0x00, 0x37, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00,
    0x17, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x1a, 0x00, 0x00, 0x00, 0x25, 0x00, 0x00, 0x00,
    0x1b, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00,
    0x17, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00,
    0x15, 0x00, 0x00, 0x00, 0x26, 0x00, 0x00, 0x00,
    0x1c, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00,
    0x1d, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00,
    0x15, 0x00, 0x00, 0x00, 0x2f, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00,
    0x15, 0x00, 0x00, 0x00, 0x26, 0x00, 0x00, 0x00,
    0x1c, 0x00, 0x00, 0x00, 0x27, 0x00, 0x00, 0x00,
    0x1d, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00,
    0x15, 0x00, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
    0x13, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
    0x13, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x57, 0x00, 0x00, 0x00,
    0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x0a, 0x00, 0x14, 0x00, 0x0a, 0x00,
    0x0a, 0x00, 0x14, 0x00, 0x14, 0x00, 0x14, 0x00,
    0x59, 0x00, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x04, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x0a, 0x00,
    0x14, 0x00, 0x0a, 0x00, 0x0a, 0x00, 0x14, 0x00,
    0x14, 0x00, 0x14, 0x00, 0x5a, 0x00, 0x00, 0x00,
    0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x0a, 0x00,
    0x14, 0x00, 0x0a, 0x00, 0x0a, 0x00, 0x14, 0x00,
    0x14, 0x00, 0x14, 0x00, 0x5c, 0x00, 0x00, 0x00,
    0x4c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x09, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x0a, 0x00, 0x14, 0x00, 0x0a, 0x00,
    0x0a, 0x00, 0x14, 0x00, 0x14, 0x00, 0x14, 0x00,
    0x1e, 0x00, 0x1e, 0x00, 0x28, 0x00, 0x14, 0x00,
    0x14, 0x00, 0x1e, 0x00, 0x14, 0x00, 0x14, 0x00,
    0x14, 0x00, 0x0a, 0x00, 0x06, 0x02, 0x04, 0x04,
    0x04, 0x02, 0x03, 0x06, 0x02, 0x00, 0x00, 0x00,
    0x29, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00,
    0x25, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x17, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb4, 0x42,
    0x00, 0x00, 0x34, 0x43, 0x3c, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x96, 0x00, 0x00, 0x00,
    0x96, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x00, 0x00,
    0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_EMPTY_PATH_BITS[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xd8, 0xff, 0xff, 0xff, 0xd8, 0xff, 0xff, 0xff,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xc8, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x20, 0x03, 0x00, 0x00, 0x58, 0x02, 0x00, 0x00,
    0x40, 0x01, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xe2, 0x04, 0x00,
    0x80, 0xa9, 0x03, 0x00, 0x3b, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x3b, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x3b, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00,
    0x08, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static void test_emf_paths(void)
{
    POINT pts[9] = {{10, 10}, {20, 10}, {10, 20}, {20, 20}, {30, 30}, {40, 20}, {20, 30}, {20, 20}, {20, 10}};
    DWORD counts[2] = {2, 2};
    BYTE types[9] = { PT_MOVETO, PT_LINETO, PT_BEZIERTO, PT_BEZIERTO, PT_BEZIERTO, PT_LINETO,
                      PT_LINETO | PT_CLOSEFIGURE, PT_MOVETO, PT_LINETO };
    HDC hdcMetafile;
    HENHMETAFILE hemf;
    BOOL ret;
    int size;

    SetLastError(0xdeadbeef);
    hdcMetafile = CreateEnhMetaFileA(GetDC(0), NULL, NULL, NULL);
    ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    ret = BeginPath(hdcMetafile);
    ok(ret, "BeginPath error %ld\n", GetLastError());
    ret = MoveToEx(hdcMetafile, 50, 50, NULL);
    ok( ret, "MoveToEx error %ld.\n", GetLastError());
    ret = LineTo(hdcMetafile, 50, 150);
    ok( ret, "LineTo error %ld.\n", GetLastError());
    ret = LineTo(hdcMetafile, 150, 150);
    ok( ret, "LineTo error %ld.\n", GetLastError());
    ret = LineTo(hdcMetafile, 150, 50);
    ok( ret, "LineTo error %ld.\n", GetLastError());
    ret = LineTo(hdcMetafile, 50, 50);
    ok( ret, "LineTo error %ld.\n", GetLastError());
    Rectangle(hdcMetafile, 10, 10, 20, 20);
    Arc(hdcMetafile, 21, 21, 39, 29, 39, 29, 21, 21);
    ArcTo(hdcMetafile, 23, 23, 37, 27, 37, 27, 23, 23);
    Chord(hdcMetafile, 21, 21, 39, 29, 39, 29, 21, 21);
    Pie(hdcMetafile, 21, 21, 39, 29, 39, 29, 21, 21);
    Ellipse(hdcMetafile, 10, 10, 20, 20);
    RoundRect(hdcMetafile, 10, 10, 20, 20, 3, 5);
    Polyline(hdcMetafile, pts, 4);
    PolylineTo(hdcMetafile, pts, 4);
    PolyPolyline(hdcMetafile, pts, counts, 2);
    PolyDraw(hdcMetafile, pts, types, 9);
    AngleArc(hdcMetafile, 37, 36, 23, 90, 180);
    ret = EndPath(hdcMetafile);
    ok(ret, "EndPath failed: %lu\n", GetLastError());

    size = GetPath(hdcMetafile, NULL, NULL, 0);
    ok( size == 112, "GetPath returned %d.\n", size);

    /* EndPath with no open path is recorded, but fails */
    ret = EndPath(hdcMetafile);
    ok(!ret, "EndPath succeeded\n");

    ret = StrokeAndFillPath( hdcMetafile );
    ok( ret, "StrokeAndFillPath failed err %ld\n", GetLastError() );
    ret = StrokeAndFillPath( hdcMetafile );
    ok( !ret, "StrokeAndFillPath succeeded\n" );

    hemf = CloseEnhMetaFile(hdcMetafile);
    ok(hemf != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    if (compare_emf_bits(hemf, EMF_PATH_BITS, sizeof(EMF_PATH_BITS), "test_emf_paths", FALSE) != 0)
    {
        dump_emf_bits(hemf, "test_emf_paths");
        dump_emf_records(hemf, "test_emf_paths");
    }

    DeleteEnhMetaFile(hemf);

    SetLastError(0xdeadbeef);
    hdcMetafile = CreateEnhMetaFileA(GetDC(0), NULL, NULL, NULL);
    ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    ret = BeginPath(hdcMetafile);
    ok( ret, "BeginPath failed error %ld\n", GetLastError() );
    ret = CloseFigure(hdcMetafile);
    ok( ret, "CloseFigure failed error %ld\n", GetLastError() );
    ret = BeginPath(hdcMetafile);
    ok( ret, "BeginPath failed error %ld\n", GetLastError() );
    ret = EndPath(hdcMetafile);
    ok( ret, "EndPath failed error %ld\n", GetLastError() );
    ret = EndPath(hdcMetafile);
    ok( !ret, "EndPath succeeded\n" );
    ret = CloseFigure(hdcMetafile);
    ok( !ret, "CloseFigure succeeded\n" );
    ret = BeginPath(hdcMetafile);
    ok( ret, "BeginPath failed error %ld\n", GetLastError() );
    ret = AbortPath(hdcMetafile);
    ok( ret, "AbortPath failed error %ld\n", GetLastError() );
    ret = AbortPath(hdcMetafile);
    ok( ret, "AbortPath failed error %ld\n", GetLastError() );

    hemf = CloseEnhMetaFile(hdcMetafile);
    ok(hemf != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    if (compare_emf_bits(hemf, EMF_EMPTY_PATH_BITS, sizeof(EMF_EMPTY_PATH_BITS), "empty path", FALSE) != 0)
    {
        dump_emf_bits(hemf, "empty path");
        dump_emf_records(hemf, "empty path");
    }

    DeleteEnhMetaFile(hemf);
}

static void test_emf_PolyPolyline(void)
{
    HDC hdcMetafile;
    HENHMETAFILE hemf;
    POINT pts[4] = {{10, 20}, {100, 200}, {0x9000,300}, {400, 500}};
    DWORD counts[2];
    BOOL ret;

    SetLastError(0xdeadbeef);
    hdcMetafile = CreateEnhMetaFileA(GetDC(0), NULL, NULL, NULL);
    ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

    ret = PolyPolyline(hdcMetafile, NULL, NULL, 0);
    ok( !ret, "PolyPolyline\n" );

    SetLastError( 0xdeadbeef );
    counts[0] = 0;
    counts[1] = 1;
    ret = PolyPolyline(hdcMetafile, pts, counts, 2);
    ok( !ret, "PolyPolyline\n" );
    ok( GetLastError() == ERROR_INVALID_PARAMETER, "gle %ld\n", GetLastError() );

    SetLastError( 0xdeadbeef );
    counts[0] = 1;
    counts[1] = 1;
    ret = PolyPolyline(hdcMetafile, pts, counts, 2);
    ok( !ret, "PolyPolyline\n" );
    ok( GetLastError() == ERROR_INVALID_PARAMETER, "gle %ld\n", GetLastError() );

    SetLastError( 0xdeadbeef );
    counts[0] = 2;
    counts[1] = 1;
    ret = PolyPolyline(hdcMetafile, pts, counts, 2);
    ok( !ret, "PolyPolyline\n" );
    ok( GetLastError() == ERROR_INVALID_PARAMETER, "gle %ld\n", GetLastError() );

    counts[0] = 2;
    counts[1] = 2;
    ret = PolyPolyline(hdcMetafile, pts, counts, 2);
    ok( ret, "PolyPolyline\n" );

    hemf = CloseEnhMetaFile(hdcMetafile);
    ok(hemf != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

    if(compare_emf_bits(hemf, EMF_POLYPOLYLINE_BITS, sizeof(EMF_POLYPOLYLINE_BITS),
        "emf_PolyPolyline", FALSE) != 0)
    {
        dump_emf_bits(hemf, "emf_PolyPolyline");
        dump_emf_records(hemf, "emf_PolyPolyline");
    }

    DeleteEnhMetaFile(hemf);
}

static void test_emf_GradientFill(void)
{
    HDC mf;
    HENHMETAFILE hemf;
    TRIVERTEX v[] =
    {
        {   1,  10, 0xff00, 0x8000, 0x0000, 0x8001 },
        { 200, 210, 0x0000, 0x0000, 0xff00, 0xff00 },
        { 180, 190, 0x1234, 0x5678, 0x9abc, 0xdef0 },
        { 300, 310, 0xff00, 0xff00, 0xff00, 0x0000 },
        { 400, 410, 0xff00, 0xff00, 0xff00, 0x0000 }
    };
    GRADIENT_TRIANGLE tri[] = { { 0, 1, 2 }, { 3, 1, 0 } };
    BOOL ret;

    mf = CreateEnhMetaFileA( GetDC( 0 ), NULL, NULL, NULL );
    ok( mf != 0, "CreateEnhMetaFileA error %ld\n", GetLastError() );

    /* Don't test the GRADIENT_FILL_RECT_ modes since a Windows bug
     * means it allocates three mesh indices rather than two per
     * rectangle.  This results in uninitialised values being written
     * to the EMF which is rather difficult to test against.
     *
     * Note also that the final vertex here is not required, yet it is
     * written to the EMF, but is not considered in the bounds
     * calculation.
     */
    ret = GdiGradientFill( mf, v, ARRAY_SIZE(v), tri, ARRAY_SIZE(tri), GRADIENT_FILL_TRIANGLE );
    ok( ret, "GradientFill\n" );

    hemf = CloseEnhMetaFile( mf );
    ok( hemf != 0, "CloseEnhMetaFile error %ld\n", GetLastError() );

    if (compare_emf_bits( hemf, EMF_GRADIENTFILL_BITS, sizeof(EMF_GRADIENTFILL_BITS),
                          "emf_GradientFill", FALSE ) != 0)
    {
        dump_emf_bits( hemf, "emf_GradientFill" );
        dump_emf_records( hemf, "emf_GradientFill" );
    }

    DeleteEnhMetaFile( hemf );
}

static void set_rotation_xform(XFORM *out, float rad, int dx, int dy)
{
    out->eM11 = cosf(rad);
    out->eM12 = -1.f * sinf(rad);
    out->eM21 = sinf(rad);
    out->eM22 = cosf(rad);
    out->eDx = dx;
    out->eDy = dy;
}

struct emf_WorldTransform_test_data
{
    const char *name;

    BOOL do_modify;
    BOOL do_playback;

    XFORM expected;
    XFORM scale;
    XFORM stored; /* this is the "hidden" world transform used in PlayEnhMetaFileRecord */
};

static BOOL xform_eq(const XFORM *a, const XFORM *b)
{
    return fabs(a->eM11 - b->eM11) < 0.001f &&
        fabs(a->eM12 - b->eM12) < 0.001f &&
        fabs(a->eM21 - b->eM21) < 0.001f &&
        fabs(a->eM22 - b->eM22) < 0.001f &&
        fabs(a->eDx - b->eDx) < 0.001f &&
        fabs(a->eDy - b->eDy) < 0.001f;
}

static INT CALLBACK enum_emf_WorldTransform(HDC hdc, HANDLETABLE *ht,
        const ENHMETARECORD *emr, INT nobj, LPARAM param)
{
    XFORM xform = {0};
    struct emf_WorldTransform_test_data *test_data = (struct emf_WorldTransform_test_data *)param;
    BOOL ret;

    switch(emr->iType)
    {
    case EMR_SETWORLDTRANSFORM:
        {
            const EMRSETWORLDTRANSFORM *lpXfrm = (const EMRSETWORLDTRANSFORM *)emr;

            /* get scale factors with an identity world transform */
            GetWorldTransform(hdc, &test_data->scale);

            /* play back record */
            ret = PlayEnhMetaFileRecord(hdc, ht, emr, nobj);
            ok(ret == TRUE, "%s: PlayEnhMetaFileRecord failed\n", test_data->name);

            test_data->stored = lpXfrm->xform;
            CombineTransform(&test_data->expected, &test_data->stored, &test_data->scale);

            /* verify it is updated immediately */
            ret = GetWorldTransform(hdc, &xform);
            ok(ret == TRUE, "%s: GetWorldTransform failed\n", test_data->name);
            ok(xform_eq(&xform, &test_data->expected),
                    "%s: After SWT playback, got wrong world transform: %f, %f; %f %f; %f %f; expected: %f, %f; %f %f; %f %f\n",
                    test_data->name,
                    xform.eM11, xform.eM12,
                    xform.eM21, xform.eM22,
                    xform.eDx, xform.eDy,
                    test_data->expected.eM11, test_data->expected.eM12,
                    test_data->expected.eM21, test_data->expected.eM22,
                    test_data->expected.eDx, test_data->expected.eDy);

            break;
        }

    case EMR_MODIFYWORLDTRANSFORM:
        {
            const EMRMODIFYWORLDTRANSFORM *lpXfrm = (const EMRMODIFYWORLDTRANSFORM *)emr;

            /* transform persists across calls */
            ret = GetWorldTransform(hdc, &xform);
            ok(ret == TRUE, "%s: GetWorldTransform failed\n", test_data->name);
            ok(xform_eq(&xform, &test_data->expected),
                    "%s: On MWT entry, got wrong world transform: %f, %f; %f %f; %f %f; expected: %f, %f; %f %f; %f %f\n",
                    test_data->name,
                    xform.eM11, xform.eM12,
                    xform.eM21, xform.eM22,
                    xform.eDx, xform.eDy,
                    test_data->expected.eM11, test_data->expected.eM12,
                    test_data->expected.eM21, test_data->expected.eM22,
                    test_data->expected.eDx, test_data->expected.eDy);

            if(test_data->do_playback)
            {
                /* play back record */
                ret = PlayEnhMetaFileRecord(hdc, ht, emr, nobj);
                ok(ret == TRUE, "%s: PlayEnhMetaFileRecord failed\n", test_data->name);

                if(lpXfrm->iMode == MWT_LEFTMULTIPLY)
                {
                    /* left multiply does not discard direct modifications */
                    CombineTransform(&test_data->expected, &lpXfrm->xform, &test_data->expected);

                    /* and updates the stored matrix separately */
                    CombineTransform(&test_data->stored, &lpXfrm->xform, &test_data->stored);

                }
                else if(lpXfrm->iMode == MWT_RIGHTMULTIPLY)
                {
                    /* but right multiply does discard */
                    CombineTransform(&test_data->stored, &test_data->stored, &lpXfrm->xform);

                    CombineTransform(&test_data->expected, &test_data->stored, &test_data->scale);
                }
                else if(lpXfrm->iMode == MWT_IDENTITY)
                {
                    /* reset to identity matrix also does discard */
                    test_data->stored.eM11 = 1.0f;
                    test_data->stored.eM12 = 0.0f;
                    test_data->stored.eM21 = 0.0f;
                    test_data->stored.eM22 = 1.0f;
                    test_data->stored.eDx = 0.0f;
                    test_data->stored.eDy = 0.0f;

                    CombineTransform(&test_data->expected, &test_data->stored, &test_data->scale);
                }

                /* verify it is updated immediately */
                ret = GetWorldTransform(hdc, &xform);
                ok(ret == TRUE, "%s: GetWorldTransform failed\n", test_data->name);
                ok(xform_eq(&xform, &test_data->expected),
                        "%s: After MWT playback, got wrong world transform: %f, %f; %f %f; %f %f; expected: %f, %f; %f %f; %f %f\n",
                        test_data->name,
                        xform.eM11, xform.eM12,
                        xform.eM21, xform.eM22,
                        xform.eDx, xform.eDy,
                        test_data->expected.eM11, test_data->expected.eM12,
                        test_data->expected.eM21, test_data->expected.eM22,
                        test_data->expected.eDx, test_data->expected.eDy);
            }

            if(test_data->do_modify)
            {
                /* modify directly */
                set_rotation_xform(&xform, M_PI / 4.f, 1, -1);
                ret = ModifyWorldTransform(hdc, &xform, MWT_LEFTMULTIPLY);
                ok(ret == TRUE, "%s: ModifyWorldTransform failed\n", test_data->name);

                /* the modified transform persists across callback calls */
                CombineTransform(&test_data->expected, &xform, &test_data->expected);

                ret = GetWorldTransform(hdc, &xform);
                ok(ret == TRUE, "%s: GetWorldTransform failed\n", test_data->name);
                ok(xform_eq(&xform, &test_data->expected),
                        "%s: After ModifyWT, got wrong world transform: %f, %f; %f %f; %f %f; expected: %f, %f; %f %f; %f %f\n",
                        test_data->name,
                        xform.eM11, xform.eM12,
                        xform.eM21, xform.eM22,
                        xform.eDx, xform.eDy,
                        test_data->expected.eM11, test_data->expected.eM12,
                        test_data->expected.eM21, test_data->expected.eM22,
                        test_data->expected.eDx, test_data->expected.eDy);
            }

            break;
        }

    case EMR_LINETO:
        ret = GetWorldTransform(hdc, &xform);
        ok(ret == TRUE, "%s: GetWorldTransform failed\n", test_data->name);
        ok(xform_eq(&xform, &test_data->expected),
                "%s: Before LINETO playback, got wrong world transform: %f, %f; %f %f; %f %f; expected: %f, %f; %f %f; %f %f\n",
                test_data->name,
                xform.eM11, xform.eM12,
                xform.eM21, xform.eM22,
                xform.eDx, xform.eDy,
                test_data->expected.eM11, test_data->expected.eM12,
                test_data->expected.eM21, test_data->expected.eM22,
                test_data->expected.eDx, test_data->expected.eDy);

        ret = PlayEnhMetaFileRecord(hdc, ht, emr, nobj);
        ok(ret == TRUE, "%s: PlayEnhMetaFileRecord failed\n", test_data->name);

        /* transform doesn't change during LINETO playback */
        ret = GetWorldTransform(hdc, &xform);
        ok(ret == TRUE, "%s: GetWorldTransform failed\n", test_data->name);
        ok(xform_eq(&xform, &test_data->expected),
                "%s: After LINETO playback, got wrong world transform: %f, %f; %f %f; %f %f; expected: %f, %f; %f %f; %f %f\n",
                test_data->name,
                xform.eM11, xform.eM12,
                xform.eM21, xform.eM22,
                xform.eDx, xform.eDy,
                test_data->expected.eM11, test_data->expected.eM12,
                test_data->expected.eM21, test_data->expected.eM22,
                test_data->expected.eDx, test_data->expected.eDy);

        break;

    default:
        PlayEnhMetaFileRecord(hdc, ht, emr, nobj);
        break;
    }

    return 1;
}

static void test_emf_WorldTransform(void)
{
    HDC hdcMetafile, hdc;
    HWND hwnd;
    HENHMETAFILE hemf;
    XFORM xform;
    BOOL ret;
    RECT rect = { 0, 0, 100, 100 };
    int i;
    struct emf_WorldTransform_test_data test_data[] = {
        { "normal", FALSE, TRUE },
        { "playback and modify", TRUE, TRUE },
        { "manual modify", TRUE, FALSE }
    };

    for(i = 0; i < ARRAY_SIZE(test_data); ++i)
    {
        hdcMetafile = CreateEnhMetaFileA(GetDC(0), NULL, NULL, NULL);
        ok(hdcMetafile != 0, "CreateEnhMetaFileA error %ld\n", GetLastError());

        ret = SetGraphicsMode(hdcMetafile, GM_ADVANCED);
        ok(ret == TRUE, "SetGraphicsMode failed\n");

        set_rotation_xform(&xform, M_PI / 4.f, 2, 3);
        ret = SetWorldTransform(hdcMetafile, &xform); /* EMR_SETWORLDTRANSFORM */
        ok(ret == TRUE, "SetWorldTransform failed\n");

        ret = ModifyWorldTransform(hdcMetafile, NULL, MWT_IDENTITY); /* EMR_MODIFYWORLDTRANSFORM */
        ok(ret == TRUE, "ModifyWorldTransform failed\n");

        set_rotation_xform(&xform, M_PI / 2.f, -2, -3);
        ret = ModifyWorldTransform(hdcMetafile, &xform, MWT_LEFTMULTIPLY); /* EMR_MODIFYWORLDTRANSFORM */
        ok(ret == TRUE, "ModifyWorldTransform failed\n");

        set_rotation_xform(&xform, M_PI / 3.f, -2, 3);
        ret = ModifyWorldTransform(hdcMetafile, &xform, MWT_LEFTMULTIPLY); /* EMR_MODIFYWORLDTRANSFORM */
        ok(ret == TRUE, "ModifyWorldTransform failed\n");

        set_rotation_xform(&xform, M_PI, 2, -3);
        ret = ModifyWorldTransform(hdcMetafile, &xform, MWT_RIGHTMULTIPLY); /* EMR_MODIFYWORLDTRANSFORM */
        ok(ret == TRUE, "ModifyWorldTransform failed\n");

        ret = LineTo(hdcMetafile, 1, 1);
        ok(ret == TRUE, "LineTo failed\n");

        hemf = CloseEnhMetaFile(hdcMetafile);
        ok(hemf != 0, "CloseEnhMetaFile error %ld\n", GetLastError());

        hwnd = CreateWindowExA(0, "static", NULL, WS_POPUP | WS_VISIBLE,
                               0, 0, 200, 200, 0, 0, 0, NULL);
        ok(hwnd != 0, "CreateWindowExA error %ld\n", GetLastError());

        hdc = GetDC(hwnd);
        ok(hdc != 0, "GetDC failed\n");

        ret = EnumEnhMetaFile(hdc, hemf, enum_emf_WorldTransform, &test_data[i], &rect);
        ok(ret == TRUE, "EnumEnhMetaFile failed: %lu\n", GetLastError());

        ReleaseDC(hwnd, hdc);
        DestroyWindow(hwnd);

        DeleteEnhMetaFile(hemf);
    }
}

static const unsigned char EMF_ALPHABLEND_1BIT[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x62, 0x00, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x72, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0xd7, 0xa3, 0x3b,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0xd7, 0x23, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x90, 0x01, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_ALPHABLEND_4BIT[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xc8, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x62, 0x00, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x72, 0x00, 0x00, 0x00, 0xe4, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0xd7, 0xa3, 0x3b,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0xd7, 0x23, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x68, 0x00, 0x00, 0x00, 0xd4, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x90, 0x01, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_ALPHABLEND_8BIT[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0x88, 0x05, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x62, 0x00, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x72, 0x00, 0x00, 0x00, 0xa4, 0x04, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0xd7, 0xa3, 0x3b,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0xd7, 0x23, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x28, 0x04, 0x00, 0x00, 0x94, 0x04, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x90, 0x01, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_ALPHABLEND_16BIT[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xa4, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xc0, 0x03, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfe, 0x00, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x30, 0xe0, 0x03, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x62, 0x00, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x72, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0xd7, 0xa3, 0x3b,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0xd7, 0x23, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x90, 0x01, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 0x00,
    0xe0, 0x07, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_ALPHABLEND_16BIT_555[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xa4, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x62, 0x00, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x72, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0xd7, 0xa3, 0x3b,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0xd7, 0x23, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x90, 0x01, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x10, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00,
    0xe0, 0x03, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_ALPHABLEND_24BIT[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xa8, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x62, 0x00, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x72, 0x00, 0x00, 0x00, 0xc4, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0xd7, 0xa3, 0x3b,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0xd7, 0x23, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x28, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x90, 0x01, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_ALPHABLEND_32BIT_888[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xc4, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x62, 0x00, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x72, 0x00, 0x00, 0x00, 0xe0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0xd7, 0xa3, 0x3b,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0xd7, 0x23, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x90, 0x01, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
    0x00, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static const unsigned char EMF_ALPHABLEND_32BIT[] =
{
    0x01, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4f, 0x00, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x00,
    0x20, 0x45, 0x4d, 0x46, 0x00, 0x00, 0x01, 0x00,
    0xc4, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x07, 0x00, 0x00, 0xd0, 0x03, 0x00, 0x00,
    0xfc, 0x01, 0x00, 0x00, 0x02, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x60, 0xc0, 0x07, 0x00,
    0xb9, 0xf0, 0x03, 0x00, 0x4c, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x62, 0x00, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x3f, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x72, 0x00, 0x00, 0x00, 0xe0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0a, 0xd7, 0xa3, 0x3b,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0a, 0xd7, 0x23, 0x3c, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00,
    0x34, 0x00, 0x00, 0x00, 0xa0, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x90, 0x01, 0x00, 0x00,
    0x90, 0x01, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x20, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00,
    0xe0, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x00
};

static void test_emf_AlphaBlend(void)
{
    static const XFORM xform = {0.5f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f};
    static const BLENDFUNCTION blend = {AC_SRC_OVER, 0, 128, 0};
    static const int bitmap_width = 4, bitmap_height = 4;
    unsigned char bmi_buffer[FIELD_OFFSET(BITMAPINFO, bmiColors[256])];
    BITMAPINFO *bmi = (BITMAPINFO *)bmi_buffer;
    HDC hdc, hdc_emf, hdc_emf2, hdc_bitmap;
    HBITMAP hbitmap, old_hbitmap;
    HENHMETAFILE hemf, hemf2;
    int ret, test_idx;
    char comment[64];
    void *bits;

    static const struct
    {
        WORD bpp;
        WORD compression;
        const void *bits;
        size_t bits_count;
        DWORD used_color_count;
        DWORD color_count;
        RGBQUAD colors[3];
    }
    tests[] =
    {
        {1, BI_RGB, EMF_ALPHABLEND_1BIT, sizeof(EMF_ALPHABLEND_1BIT), 1, 1, {{0xff, 0xff, 0xff}}},
        {4, BI_RGB, EMF_ALPHABLEND_4BIT, sizeof(EMF_ALPHABLEND_4BIT), 1, 1, {{0xff, 0xff, 0xff}}},
        {8, BI_RGB, EMF_ALPHABLEND_8BIT, sizeof(EMF_ALPHABLEND_8BIT), 1, 1, {{0xff, 0xff, 0xff}}},
        {16, BI_RGB, EMF_ALPHABLEND_16BIT_555, sizeof(EMF_ALPHABLEND_16BIT_555)},
        {24, BI_RGB, EMF_ALPHABLEND_24BIT, sizeof(EMF_ALPHABLEND_24BIT)},
        {32, BI_RGB, EMF_ALPHABLEND_32BIT_888, sizeof(EMF_ALPHABLEND_32BIT_888)},
        {16, BI_BITFIELDS, EMF_ALPHABLEND_16BIT, sizeof(EMF_ALPHABLEND_16BIT), 0, 3, {{0x00, 0xf8, 0x00}, {0xe0, 0x07, 0x00}, {0x1f, 0x00, 0x00}}},
        {32, BI_BITFIELDS, EMF_ALPHABLEND_32BIT, sizeof(EMF_ALPHABLEND_32BIT), 0, 3, {{0x00, 0x00, 0xff}, {0xe0, 0xff, 0x00}, {0xff, 0x00, 0x00}}},
    };

    hdc = GetDC(0);

    /* Test that source DC cannot be an enhanced metafile */
    hdc_emf = CreateEnhMetaFileW(hdc, NULL, NULL, NULL);
    ok(!!hdc_emf, "CreateEnhMetaFileW failed, error %ld\n", GetLastError());
    hdc_emf2 = CreateEnhMetaFileW(hdc, NULL, NULL, NULL);
    ok(!!hdc_emf2, "CreateEnhMetaFileW failed, error %ld\n", GetLastError());

    ret = GdiAlphaBlend(hdc_emf, 0, 0, 1, 1, hdc_emf2, 0, 0, 1, 1, blend);
    ok(!ret, "GdiAlphaBlend succeeded\n");

    hemf2 = CloseEnhMetaFile(hdc_emf2);
    ok(!!hemf2, "CloseEnhMetaFile failed, error %ld\n", GetLastError());
    hemf = CloseEnhMetaFile(hdc_emf);
    ok(!!hemf, "CloseEnhMetaFile failed, error %ld\n", GetLastError());
    DeleteEnhMetaFile(hemf2);
    DeleteEnhMetaFile(hemf);

    /* Test AlphaBlend with different format of bitmaps */
    for (test_idx = 0; test_idx < ARRAY_SIZE(tests); ++test_idx)
    {
        memset(bmi_buffer, 0, sizeof(bmi_buffer));
        bmi->bmiHeader.biSize = sizeof(bmi->bmiHeader);
        bmi->bmiHeader.biHeight = bitmap_width;
        bmi->bmiHeader.biWidth = bitmap_height;
        bmi->bmiHeader.biBitCount = tests[test_idx].bpp;
        bmi->bmiHeader.biPlanes = 1;
        bmi->bmiHeader.biCompression = tests[test_idx].compression;
        bmi->bmiHeader.biClrUsed = tests[test_idx].used_color_count;
        memcpy(bmi->bmiColors, tests[test_idx].colors, sizeof(RGBQUAD) * tests[test_idx].color_count);

        hbitmap = CreateDIBSection(hdc, bmi, DIB_RGB_COLORS, &bits, NULL, 0);
        ok(!!hbitmap, "Test %d: CreateDIBSection failed, error %ld\n", test_idx, GetLastError());
        hdc_bitmap = CreateCompatibleDC(hdc);
        ok(!!hdc_bitmap, "Test %d: CreateCompatibleDC failed, error %ld\n", test_idx, GetLastError());
        old_hbitmap = SelectObject(hdc_bitmap, hbitmap);

        SetBkColor(hdc_bitmap, RGB(0xff, 0xff, 0xff));
        ret = SetGraphicsMode(hdc_bitmap, GM_ADVANCED);
        ok(ret, "Test %d: SetGraphicsMode failed, error %ld\n", test_idx, GetLastError());
        ret = SetWorldTransform(hdc_bitmap, &xform);
        ok(ret, "Test %d: SetWorldTransform failed, error %ld\n", test_idx, GetLastError());
        ret = SetMapMode(hdc_bitmap, MM_ANISOTROPIC);
        ok(ret, "Test %d: SetMapMode failed, error %ld\n", test_idx, GetLastError());
        ret = SetWindowOrgEx(hdc_bitmap, 0, 0, NULL);
        ok(ret, "Test %d: SetWindowOrgEx failed, error %ld\n", test_idx, GetLastError());
        ret = SetWindowExtEx(hdc_bitmap, 400, 400, NULL);
        ok(ret, "Test %d: SetWindowExtEx failed, error %ld\n", test_idx, GetLastError());
        ret = SetViewportOrgEx(hdc_bitmap, 0, 0, NULL);
        ok(ret, "Test %d: SetViewportOrgEx failed, error %ld\n", test_idx, GetLastError());
        ret = SetViewportExtEx(hdc_bitmap, bitmap_width, bitmap_height, NULL);
        ok(ret, "Test %d: SetViewportExtEx failed, error %ld\n", test_idx, GetLastError());

        hdc_emf = CreateEnhMetaFileW(hdc, NULL, NULL, NULL);
        ok(!!hdc_emf, "Test %d: CreateEnhMetaFileW failed, error %ld\n", test_idx, GetLastError());

        ret = BitBlt(hdc_emf, 0, 0, bitmap_width, bitmap_height, 0, 0, 0, WHITENESS);
        ok(ret, "Test %d: BitBlt failed, error %ld\n", test_idx, GetLastError());
        ret = BitBlt(hdc_bitmap, 0, 0, bitmap_width, bitmap_height, 0, 0, 0, BLACKNESS);
        ok(ret, "Test %d: BitBlt failed, error %ld\n", test_idx, GetLastError());
        ret = GdiAlphaBlend(hdc_emf, 0, 0, bitmap_width, bitmap_height, hdc_bitmap, 0, 0, 400, 400, blend);
        ok(ret, "Test %d: GdiAlphaBlend failed, error %ld\n", test_idx, GetLastError());

        hemf = CloseEnhMetaFile(hdc_emf);
        ok(!!hemf, "Test %d: CloseEnhMetaFile failed, %ld\n", test_idx, GetLastError());

        sprintf(comment, "test_emf_AlphaBlend() test %d", test_idx);
        ret = compare_emf_bits(hemf, tests[test_idx].bits, tests[test_idx].bits_count, comment, FALSE);
        if (ret)
        {
            dump_emf_bits(hemf, comment);
            dump_emf_records(hemf, comment);
        }

        DeleteEnhMetaFile(hemf);
        SelectObject(hdc_bitmap, old_hbitmap);
        DeleteDC(hdc_bitmap);
        DeleteObject(hbitmap);
    }

    ReleaseDC(0, hdc);
}

static void test_emf_text_extents(void)
{
    static const XFORM xform = {0.5f, 0.0f, 0.0f, 0.5f, 0.0f, 0.0f};
    HFONT font, old_font, old_font2;
    LOGFONTW logfont = {0};
    HENHMETAFILE emf;
    SIZE size, size2;
    HDC dc, emf_dc;
    BOOL ret;

    dc = GetDC(0);
    emf_dc = CreateEnhMetaFileW(dc, NULL, NULL, NULL);
    ok(!!emf_dc, "CreateEnhMetaFileW failed, error %ld\n", GetLastError());

    logfont.lfWeight = FW_NORMAL;
    logfont.lfHeight = 20;
    lstrcpyW(logfont.lfFaceName, L"Tahoma");
    font = CreateFontIndirectW(&logfont);
    ok(!!font, "CreateFontIndirectW failed, error %ld\n", GetLastError());

    old_font = SelectObject(dc, font);
    old_font2 = SelectObject(emf_dc, font);

    ret = SetGraphicsMode(dc, GM_ADVANCED);
    ok(ret, "SetGraphicsMode failed, error %ld\n", GetLastError());
    ret = SetGraphicsMode(emf_dc, GM_ADVANCED);
    ok(ret, "SetGraphicsMode failed, error %ld\n", GetLastError());

    ret = ModifyWorldTransform(dc, &xform, MWT_RIGHTMULTIPLY);
    ok(ret, "ModifyWorldTransform failed, error %ld\n", GetLastError());
    ret = ModifyWorldTransform(emf_dc, &xform, MWT_RIGHTMULTIPLY);
    ok(ret, "ModifyWorldTransform failed, error %ld\n", GetLastError());

    ret = GetTextExtentPoint32W(dc, L"W", 1, &size);
    ok(ret, "GetTextExtentPoint32W failed, error %ld\n", GetLastError());
    ret = GetTextExtentPoint32W(emf_dc, L"W", 1, &size2);
    ok(ret, "GetTextExtentPoint32W failed, error %ld\n", GetLastError());
    ok(size2.cx == size.cx && size2.cy == size.cy, "Expected size %ldx%ld, got %ldx%ld\n",
       size.cx, size.cy, size2.cx, size2.cy);

    SelectObject(emf_dc, old_font2);
    SelectObject(dc, old_font);
    DeleteObject(font);
    emf = CloseEnhMetaFile(emf_dc);
    ok(!!emf, "CloseEnhMetaFile failed, error %ld\n", GetLastError());
    DeleteEnhMetaFile(emf);
    ReleaseDC(0, dc);
}

static const unsigned char MF_SETLAYOUT_BITS[] =
{
/*  Winedump output. Note that there is no META_SELECTOBJECT records after META_SETLAYOUT.
 *
 *  METAHEADER           00000012
 *  type 1 header_size 0x12 version 0x300 size 0x54 object_count 0 max_record_size 0xa parameter_count 0
 *     0001 0009 0300 002a 0000 0000 0005 0000
 *     0000
 *  META_SETLAYOUT       0000000a
 *     0000 0000
 *  META_SETLAYOUT       0000000a
 *     0001 0000
 *  META_SETLAYOUT       0000000a
 *     0008 0000
 *  META_SETLAYOUT       0000000a
 *     0000 8000
 *  META_SETLAYOUT       0000000a
 *     0009 0000
 *  META_SETLAYOUT       0000000a
 *     0001 8000
 *  META_EOF             00000006
 */
    0x01, 0x00, 0x09, 0x00, 0x00, 0x03, 0x2a, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x49, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x49, 0x01, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00,
    0x00, 0x00, 0x49, 0x01, 0x08, 0x00, 0x00, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x49, 0x01, 0x00, 0x00,
    0x00, 0x80, 0x05, 0x00, 0x00, 0x00, 0x49, 0x01,
    0x09, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x49, 0x01, 0x01, 0x00, 0x00, 0x80, 0x03, 0x00,
    0x00, 0x00, 0x00, 0x00
};

static void test_mf_SetLayout(void)
{
    static const DWORD tests[] =
    {
        LAYOUT_LTR,
        LAYOUT_RTL,
        LAYOUT_BITMAPORIENTATIONPRESERVED,
        NOMIRRORBITMAP,
        LAYOUT_RTL | LAYOUT_BITMAPORIENTATIONPRESERVED,
        LAYOUT_RTL | NOMIRRORBITMAP,
    };
    DWORD layout;
    HMETAFILE mf;
    int ret, i;
    HDC mf_dc;

    mf_dc = CreateMetaFileW(NULL);
    ok(!!mf_dc, "CreateMetaFileW failed, error %ld\n", GetLastError());

    for (i = 0; i < ARRAY_SIZE(tests); ++i)
    {
        winetest_push_context("Test %d", i);
        layout = SetLayout(mf_dc, tests[i]);
        ok(layout == LAYOUT_RTL, "Expected %#lx, got %#lx\n", tests[i], layout);
        layout = GetLayout(mf_dc);
        ok(layout == GDI_ERROR, "Expected %#x, got %#lx\n", GDI_ERROR, layout);
        winetest_pop_context();
    }

    mf = CloseMetaFile(mf_dc);
    ok(!!mf, "CloseMetaFile failed, error %ld\n", GetLastError());

    ret = compare_mf_bits(mf, MF_SETLAYOUT_BITS, sizeof(MF_SETLAYOUT_BITS), "mf_SetLayout");
    ok(!ret, "Bits mismatch\n");
    if (ret)
    {
        dump_mf_bits(mf, "mf_SetLayout");
        EnumMetaFile(0, mf, mf_enum_proc, 0);
    }

    DeleteMetaFile(mf);
}

START_TEST(metafile)
{
    init_function_pointers();

    /* For enhanced metafiles (enhmfdrv) */
    test_ExtTextOut();
    test_ExtTextOutScale();
    test_SaveDC();
    test_emf_AlphaBlend();
    test_emf_BitBlt();
    test_emf_DCBrush();
    test_emf_ExtTextOut_on_path();
    test_emf_clipping();
    test_emf_polybezier();
    test_emf_paths();
    test_emf_PolyPolyline();
    test_emf_GradientFill();
    test_emf_WorldTransform();
    test_emf_text_extents();
    test_enhmetafile_file();
    test_emf_SetPixel();
    test_emf_attrs();
    test_emf_select();
    test_emf_blit();
    test_emf_pattern_brush();
    test_emf_mask_blit();
    test_emf_StretchDIBits();
    test_emf_SetDIBitsToDevice();
    test_emf_palette();

    /* For win-format metafiles (mfdrv) */
    test_mf_SaveDC();
    test_mf_Blank();
    test_mf_Graphics();
    test_mf_PatternBrush();
    test_mf_palette_brush();
    test_mf_DCBrush();
    test_CopyMetaFile();
    test_SetMetaFileBits();
    test_mf_ExtTextOut_on_path();
    test_mf_clipping();
    test_mf_GetPath();
    test_mf_SetLayout();
    test_metafile_file();
    test_mf_SetPixel();
    test_mf_FloodFill();
    test_mf_attrs();
    test_mf_select();
    test_mf_palette();
    test_mf_blit();
    test_PlayMetaFile();

    /* For metafile conversions */
    test_mf_conversions();
    test_SetWinMetaFileBits();
    test_GetWinMetaFileBits();

    test_gdiis();
    test_SetEnhMetaFileBits();
}
