#!/bin/sh
#
# Script to automatically install all Wine Staging patches
#
# Copyright (C) 2015-2017 Sebastian Lackner
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#

# Show usage information
usage()
{
	echo ""
	echo "Usage: ./patchinstall.sh [DESTDIR=path] [--all] [-W patchset] [patchset ...]"
	echo ""
	echo "Autogenerated script to apply all Wine Staging patches on your Wine"
	echo "source tree."
	echo ""
	echo "Configuration:"
	echo "  DESTDIR=path         Specify the path to the wine source tree"
	echo "  --all                Select all patches"
	echo "  --force-autoconf     Run autoreconf and tools/make_requests after each patch"
	echo "  --help               Display this help and exit"
	echo "  --no-autoconf        Do not run autoreconf and tools/make_requests"
	echo "  --upstream-commit    Print the upstream Wine commit SHA1 and exit"
	echo "  --version            Show version information and exit"
	echo "  -W patchset          Exclude a specific patchset"
	echo ""
	echo "Backends:"
	echo "  --backend=patch      Use regular 'patch' utility to apply patches (default)"
	echo "  --backend=eapply     Use 'eapply' to apply patches (Gentoo only)"
	echo "  --backend=epatch     Use 'epatch' to apply patches (Gentoo only, deprecated)"
	echo "  --backend=git-am     Use 'git am' to apply patches"
	echo "  --backend=git-apply  Use 'git apply' to apply patches"
	echo "  --backend=stg        Import the patches using stacked git"
	echo ""
}

# Get the upstream commit sha
upstream_commit()
{
	echo "a8583acae9548e6340d8400ec7710136cd061ab2"
}

# Show version information
version()
{
	echo "Wine Staging 6.17"
	echo "Copyright (C) 2014-2019 the Wine Staging project authors."
	echo "Copyright (C) 2018-2020 Alistair Leslie-Hughes"
	echo ""
	echo "Patchset to be applied on upstream Wine:"
	echo "  commit $(upstream_commit)"
	echo ""
}

# Critical error, abort
abort()
{
	printf '%s\n' "ERROR: $1" >&2
	exit 1
}

# Show a warning
warning()
{
	printf '%s\n' "WARNING: $1" >&2
}

# Enable or disable all patchsets
patch_enable_all ()
{
	enable_Compiler_Warnings="$1"
	enable_Pipelight="$1"
	enable_Staging="$1"
	enable_advapi32_LsaLookupPrivilegeName="$1"
	enable_api_ms_win_Stub_DLLs="$1"
	enable_api_ms_win_core_psapi_K32GetModuleInformation="$1"
	enable_bcrypt_ECDHSecretAgreement="$1"
	enable_cmd_launch_association="$1"
	enable_comctl32_rebar_capture="$1"
	enable_comctl32_version_6="$1"
	enable_comdlg32_lpstrFileTitle="$1"
	enable_crypt32_CMS_Certificates="$1"
	enable_cryptext_CryptExtOpenCER="$1"
	enable_d3drm_IDirect3D3_support="$1"
	enable_d3dx11_43_D3DX11CreateTextureFromMemory="$1"
	enable_d3dx9_36_BumpLuminance="$1"
	enable_d3dx9_36_CloneEffect="$1"
	enable_d3dx9_36_D3DXDisassembleShader="$1"
	enable_d3dx9_36_D3DXOptimizeVertices="$1"
	enable_d3dx9_36_D3DXSHProjectCubeMap="$1"
	enable_d3dx9_36_D3DXStubs="$1"
	enable_d3dx9_36_DDS="$1"
	enable_d3dx9_36_Filter_Warnings="$1"
	enable_d3dx9_36_UpdateSkinnedMesh="$1"
	enable_dbghelp_Debug_Symbols="$1"
	enable_ddraw_Device_Caps="$1"
	enable_ddraw_IDirect3DTexture2_Load="$1"
	enable_ddraw_Silence_FIXMEs="$1"
	enable_ddraw_version_check="$1"
	enable_dinput_SetActionMap_genre="$1"
	enable_dinput_axis_recalc="$1"
	enable_dinput_joy_mappings="$1"
	enable_dinput_reconnect_joystick="$1"
	enable_dinput_remap_joystick="$1"
	enable_dsound_EAX="$1"
	enable_dsound_Fast_Mixer="$1"
	enable_dwrite_FontFallback="$1"
	enable_eventfd_synchronization="$1"
	enable_explorer_Video_Registry_Key="$1"
	enable_fonts_Missing_Fonts="$1"
	enable_gdi32_rotation="$1"
	enable_gdiplus_Performance_Improvements="$1"
	enable_imm32_com_initialization="$1"
	enable_imm32_message_on_focus="$1"
	enable_include_winsock="$1"
	enable_inseng_Implementation="$1"
	enable_iphlpapi_System_Ping="$1"
	enable_kernel32_CopyFileEx="$1"
	enable_kernel32_Debugger="$1"
	enable_kernel32_Job_Tests="$1"
	enable_kernel32_Processor_Group="$1"
	enable_krnl386_exe16_GDT_LDT_Emulation="$1"
	enable_krnl386_exe16_Invalid_Console_Handles="$1"
	enable_libs_Unicode_Collation="$1"
	enable_loader_KeyboardLayouts="$1"
	enable_mfplat_streaming_support="$1"
	enable_mmsystem_dll16_MIDIHDR_Refcount="$1"
	enable_mountmgr_DosDevices="$1"
	enable_mscoree_CorValidateImage="$1"
	enable_mshtml_HTMLLocation_put_hash="$1"
	enable_mshtml_TranslateAccelerator="$1"
	enable_msi_msi_vcl_get_cost="$1"
	enable_msxml3_FreeThreadedXMLHTTP60="$1"
	enable_ntdll_APC_Performance="$1"
	enable_ntdll_ApiSetMap="$1"
	enable_ntdll_Builtin_Prot="$1"
	enable_ntdll_CriticalSection="$1"
	enable_ntdll_DOS_Attributes="$1"
	enable_ntdll_Exception="$1"
	enable_ntdll_FileFsFullSizeInformation="$1"
	enable_ntdll_ForceBottomUpAlloc="$1"
	enable_ntdll_HashLinks="$1"
	enable_ntdll_Heap_Improvements="$1"
	enable_ntdll_Hide_Wine_Exports="$1"
	enable_ntdll_Junction_Points="$1"
	enable_ntdll_Manifest_Range="$1"
	enable_ntdll_NtAlertThreadByThreadId="$1"
	enable_ntdll_NtQueryEaFile="$1"
	enable_ntdll_NtQuerySection="$1"
	enable_ntdll_NtSetLdtEntries="$1"
	enable_ntdll_ProcessQuotaLimits="$1"
	enable_ntdll_RtlFirstFreeAce="$1"
	enable_ntdll_RtlQueryPackageIdentity="$1"
	enable_ntdll_Serial_Port_Detection="$1"
	enable_ntdll_Syscall_Emulation="$1"
	enable_ntdll_WRITECOPY="$1"
	enable_ntdll_Zero_mod_name="$1"
	enable_ntdll_ext4_case_folder="$1"
	enable_ntoskrnl_Stubs="$1"
	enable_nvapi_Stub_DLL="$1"
	enable_nvcuda_CUDA_Support="$1"
	enable_nvcuvid_CUDA_Video_Support="$1"
	enable_nvencodeapi_Video_Encoder="$1"
	enable_oleaut32_CreateTypeLib="$1"
	enable_oleaut32_Load_Save_EMF="$1"
	enable_oleaut32_OLEPictureImpl_SaveAsFile="$1"
	enable_oleaut32_OleLoadPicture="$1"
	enable_oleaut32_OleLoadPictureFile="$1"
	enable_packager_DllMain="$1"
	enable_programs_findstr="$1"
	enable_programs_systeminfo="$1"
	enable_riched20_Class_Tests="$1"
	enable_riched20_IText_Interface="$1"
	enable_secur32_InitializeSecurityContextW="$1"
	enable_server_File_Permissions="$1"
	enable_server_Key_State="$1"
	enable_server_PeekMessage="$1"
	enable_server_Realtime_Priority="$1"
	enable_server_Signal_Thread="$1"
	enable_server_Stored_ACLs="$1"
	enable_server_default_integrity="$1"
	enable_setupapi_DiskSpaceList="$1"
	enable_shdocvw_ParseURLFromOutsideSource_Tests="$1"
	enable_shell32_ACE_Viewer="$1"
	enable_shell32_Context_Menu="$1"
	enable_shell32_IconCache="$1"
	enable_shell32_NewMenu_Interface="$1"
	enable_shell32_Progress_Dialog="$1"
	enable_shell32_SFGAO_HASSUBFOLDER="$1"
	enable_shell32_SHFileOperation_Move="$1"
	enable_shell32_SHGetStockIconInfo="$1"
	enable_shell32_Toolbar_Bitmaps="$1"
	enable_shell32_UnixFS="$1"
	enable_shlwapi_AssocGetPerceivedType="$1"
	enable_shlwapi_UrlCanonicalize="$1"
	enable_shlwapi_UrlCombine="$1"
	enable_stdole32_idl_Typelib="$1"
	enable_stdole32_tlb_SLTG_Typelib="$1"
	enable_tasklist_basics="$1"
	enable_user32_DM_SETDEFID="$1"
	enable_user32_Dialog_Paint_Event="$1"
	enable_user32_DrawTextExW="$1"
	enable_user32_FlashWindowEx="$1"
	enable_user32_GetSystemMetrics="$1"
	enable_user32_Implement_CascadeWindows="$1"
	enable_user32_LR_LOADFROMFILE="$1"
	enable_user32_ListBox_Size="$1"
	enable_user32_LoadKeyboardLayoutEx="$1"
	enable_user32_MessageBox_WS_EX_TOPMOST="$1"
	enable_user32_Mouse_Message_Hwnd="$1"
	enable_user32_QueryDisplayConfig="$1"
	enable_user32_Refresh_MDI_Menus="$1"
	enable_user32_ScrollWindowEx="$1"
	enable_user32_message_order="$1"
	enable_user32_msgbox_Support_WM_COPY_mesg="$1"
	enable_user32_rawinput_mouse="$1"
	enable_user32_rawinput_mouse_experimental="$1"
	enable_user32_recursive_activation="$1"
	enable_uxtheme_CloseThemeClass="$1"
	enable_version_VerQueryValue="$1"
	enable_wbemdisp_ISWbemObject_Invoke="$1"
	enable_widl_SLTG_Typelib_Support="$1"
	enable_windows_networking_connectivity_new_dll="$1"
	enable_windowscodecs_GIF_Encoder="$1"
	enable_windowscodecs_TIFF_Support="$1"
	enable_wine_inf_Directory_ContextMenuHandlers="$1"
	enable_wine_inf_Dummy_CA_Certificate="$1"
	enable_wineboot_HKEY_DYN_DATA="$1"
	enable_wineboot_ProxySettings="$1"
	enable_winecfg_Libraries="$1"
	enable_winecfg_Staging="$1"
	enable_wined3d_Accounting="$1"
	enable_wined3d_Indexed_Vertex_Blending="$1"
	enable_wined3d_SWVP_shaders="$1"
	enable_wined3d_Silence_FIXMEs="$1"
	enable_wined3d_WINED3DFMT_B8G8R8X8_UNORM="$1"
	enable_wined3d_mesa_texture_download="$1"
	enable_wined3d_unset_flip_gdi="$1"
	enable_wined3d_wined3d_guess_gl_vendor="$1"
	enable_wined3d_zero_inf_shaders="$1"
	enable_winedbg_Process_Arguments="$1"
	enable_winedevice_Default_Drivers="$1"
	enable_winemac_drv_no_flicker_patch="$1"
	enable_winemapi_user_xdg_mail="$1"
	enable_winemenubuilder_Desktop_Icon_Path="$1"
	enable_winemenubuilder_integration="$1"
	enable_wineps_drv_PostScript_Fixes="$1"
	enable_winex11_CandidateWindowPos="$1"
	enable_winex11_MWM_Decorations="$1"
	enable_winex11_UpdateLayeredWindow="$1"
	enable_winex11_Vulkan_support="$1"
	enable_winex11_WM_WINDOWPOSCHANGING="$1"
	enable_winex11_Window_Style="$1"
	enable_winex11_XEMBED="$1"
	enable_winex11__NET_ACTIVE_WINDOW="$1"
	enable_winex11_ime_check_thread_data="$1"
	enable_winex11_key_translation="$1"
	enable_winex11_wglShareLists="$1"
	enable_winex11_drv_Query_server_position="$1"
	enable_wininet_Cleanup="$1"
	enable_winmm_mciSendCommandA="$1"
	enable_wintab32_improvements="$1"
	enable_wintrust_WTHelperGetProvCertFromChain="$1"
	enable_wscript_support_d_u_switches="$1"
	enable_xactengine_initial="$1"
	enable_xactengine3_7_Notification="$1"
	enable_xactengine3_7_PrepareWave="$1"
}

# Enable or disable a specific patchset
patch_enable ()
{
	case "$1" in
		Compiler_Warnings)
			enable_Compiler_Warnings="$2"
			;;
		Pipelight)
			enable_Pipelight="$2"
			;;
		Staging)
			enable_Staging="$2"
			;;
		advapi32-LsaLookupPrivilegeName)
			enable_advapi32_LsaLookupPrivilegeName="$2"
			;;
		api-ms-win-Stub_DLLs)
			enable_api_ms_win_Stub_DLLs="$2"
			;;
		api-ms-win-core-psapi-K32GetModuleInformation)
			enable_api_ms_win_core_psapi_K32GetModuleInformation="$2"
			;;
		bcrypt-ECDHSecretAgreement)
			enable_bcrypt_ECDHSecretAgreement="$2"
			;;
		cmd-launch-association)
			enable_cmd_launch_association="$2"
			;;
		comctl32-rebar-capture)
			enable_comctl32_rebar_capture="$2"
			;;
		comctl32-version_6)
			enable_comctl32_version_6="$2"
			;;
		comdlg32-lpstrFileTitle)
			enable_comdlg32_lpstrFileTitle="$2"
			;;
		crypt32-CMS_Certificates)
			enable_crypt32_CMS_Certificates="$2"
			;;
		cryptext-CryptExtOpenCER)
			enable_cryptext_CryptExtOpenCER="$2"
			;;
		d3drm-IDirect3D3-support)
			enable_d3drm_IDirect3D3_support="$2"
			;;
		d3dx11_43-D3DX11CreateTextureFromMemory)
			enable_d3dx11_43_D3DX11CreateTextureFromMemory="$2"
			;;
		d3dx9_36-BumpLuminance)
			enable_d3dx9_36_BumpLuminance="$2"
			;;
		d3dx9_36-CloneEffect)
			enable_d3dx9_36_CloneEffect="$2"
			;;
		d3dx9_36-D3DXDisassembleShader)
			enable_d3dx9_36_D3DXDisassembleShader="$2"
			;;
		d3dx9_36-D3DXOptimizeVertices)
			enable_d3dx9_36_D3DXOptimizeVertices="$2"
			;;
		d3dx9_36-D3DXSHProjectCubeMap)
			enable_d3dx9_36_D3DXSHProjectCubeMap="$2"
			;;
		d3dx9_36-D3DXStubs)
			enable_d3dx9_36_D3DXStubs="$2"
			;;
		d3dx9_36-DDS)
			enable_d3dx9_36_DDS="$2"
			;;
		d3dx9_36-Filter_Warnings)
			enable_d3dx9_36_Filter_Warnings="$2"
			;;
		d3dx9_36-UpdateSkinnedMesh)
			enable_d3dx9_36_UpdateSkinnedMesh="$2"
			;;
		dbghelp-Debug_Symbols)
			enable_dbghelp_Debug_Symbols="$2"
			;;
		ddraw-Device_Caps)
			enable_ddraw_Device_Caps="$2"
			;;
		ddraw-IDirect3DTexture2_Load)
			enable_ddraw_IDirect3DTexture2_Load="$2"
			;;
		ddraw-Silence_FIXMEs)
			enable_ddraw_Silence_FIXMEs="$2"
			;;
		ddraw-version-check)
			enable_ddraw_version_check="$2"
			;;
		dinput-SetActionMap-genre)
			enable_dinput_SetActionMap_genre="$2"
			;;
		dinput-axis-recalc)
			enable_dinput_axis_recalc="$2"
			;;
		dinput-joy-mappings)
			enable_dinput_joy_mappings="$2"
			;;
		dinput-reconnect-joystick)
			enable_dinput_reconnect_joystick="$2"
			;;
		dinput-remap-joystick)
			enable_dinput_remap_joystick="$2"
			;;
		dsound-EAX)
			enable_dsound_EAX="$2"
			;;
		dsound-Fast_Mixer)
			enable_dsound_Fast_Mixer="$2"
			;;
		dwrite-FontFallback)
			enable_dwrite_FontFallback="$2"
			;;
		eventfd_synchronization)
			enable_eventfd_synchronization="$2"
			;;
		explorer-Video_Registry_Key)
			enable_explorer_Video_Registry_Key="$2"
			;;
		fonts-Missing_Fonts)
			enable_fonts_Missing_Fonts="$2"
			;;
		gdi32-rotation)
			enable_gdi32_rotation="$2"
			;;
		gdiplus-Performance-Improvements)
			enable_gdiplus_Performance_Improvements="$2"
			;;
		imm32-com-initialization)
			enable_imm32_com_initialization="$2"
			;;
		imm32-message_on_focus)
			enable_imm32_message_on_focus="$2"
			;;
		include-winsock)
			enable_include_winsock="$2"
			;;
		inseng-Implementation)
			enable_inseng_Implementation="$2"
			;;
		iphlpapi-System_Ping)
			enable_iphlpapi_System_Ping="$2"
			;;
		kernel32-CopyFileEx)
			enable_kernel32_CopyFileEx="$2"
			;;
		kernel32-Debugger)
			enable_kernel32_Debugger="$2"
			;;
		kernel32-Job_Tests)
			enable_kernel32_Job_Tests="$2"
			;;
		kernel32-Processor_Group)
			enable_kernel32_Processor_Group="$2"
			;;
		krnl386.exe16-GDT_LDT_Emulation)
			enable_krnl386_exe16_GDT_LDT_Emulation="$2"
			;;
		krnl386.exe16-Invalid_Console_Handles)
			enable_krnl386_exe16_Invalid_Console_Handles="$2"
			;;
		libs-Unicode_Collation)
			enable_libs_Unicode_Collation="$2"
			;;
		loader-KeyboardLayouts)
			enable_loader_KeyboardLayouts="$2"
			;;
		mfplat-streaming-support)
			enable_mfplat_streaming_support="$2"
			;;
		mmsystem.dll16-MIDIHDR_Refcount)
			enable_mmsystem_dll16_MIDIHDR_Refcount="$2"
			;;
		mountmgr-DosDevices)
			enable_mountmgr_DosDevices="$2"
			;;
		mscoree-CorValidateImage)
			enable_mscoree_CorValidateImage="$2"
			;;
		mshtml-HTMLLocation_put_hash)
			enable_mshtml_HTMLLocation_put_hash="$2"
			;;
		mshtml-TranslateAccelerator)
			enable_mshtml_TranslateAccelerator="$2"
			;;
		msi-msi_vcl_get_cost)
			enable_msi_msi_vcl_get_cost="$2"
			;;
		msxml3-FreeThreadedXMLHTTP60)
			enable_msxml3_FreeThreadedXMLHTTP60="$2"
			;;
		ntdll-APC_Performance)
			enable_ntdll_APC_Performance="$2"
			;;
		ntdll-ApiSetMap)
			enable_ntdll_ApiSetMap="$2"
			;;
		ntdll-Builtin_Prot)
			enable_ntdll_Builtin_Prot="$2"
			;;
		ntdll-CriticalSection)
			enable_ntdll_CriticalSection="$2"
			;;
		ntdll-DOS_Attributes)
			enable_ntdll_DOS_Attributes="$2"
			;;
		ntdll-Exception)
			enable_ntdll_Exception="$2"
			;;
		ntdll-FileFsFullSizeInformation)
			enable_ntdll_FileFsFullSizeInformation="$2"
			;;
		ntdll-ForceBottomUpAlloc)
			enable_ntdll_ForceBottomUpAlloc="$2"
			;;
		ntdll-HashLinks)
			enable_ntdll_HashLinks="$2"
			;;
		ntdll-Heap_Improvements)
			enable_ntdll_Heap_Improvements="$2"
			;;
		ntdll-Hide_Wine_Exports)
			enable_ntdll_Hide_Wine_Exports="$2"
			;;
		ntdll-Junction_Points)
			enable_ntdll_Junction_Points="$2"
			;;
		ntdll-Manifest_Range)
			enable_ntdll_Manifest_Range="$2"
			;;
		ntdll-NtAlertThreadByThreadId)
			enable_ntdll_NtAlertThreadByThreadId="$2"
			;;
		ntdll-NtQueryEaFile)
			enable_ntdll_NtQueryEaFile="$2"
			;;
		ntdll-NtQuerySection)
			enable_ntdll_NtQuerySection="$2"
			;;
		ntdll-NtSetLdtEntries)
			enable_ntdll_NtSetLdtEntries="$2"
			;;
		ntdll-ProcessQuotaLimits)
			enable_ntdll_ProcessQuotaLimits="$2"
			;;
		ntdll-RtlFirstFreeAce)
			enable_ntdll_RtlFirstFreeAce="$2"
			;;
		ntdll-RtlQueryPackageIdentity)
			enable_ntdll_RtlQueryPackageIdentity="$2"
			;;
		ntdll-Serial_Port_Detection)
			enable_ntdll_Serial_Port_Detection="$2"
			;;
		ntdll-Syscall_Emulation)
			enable_ntdll_Syscall_Emulation="$2"
			;;
		ntdll-WRITECOPY)
			enable_ntdll_WRITECOPY="$2"
			;;
		ntdll-Zero_mod_name)
			enable_ntdll_Zero_mod_name="$2"
			;;
		ntdll-ext4-case-folder)
			enable_ntdll_ext4_case_folder="$2"
			;;
		ntoskrnl-Stubs)
			enable_ntoskrnl_Stubs="$2"
			;;
		nvapi-Stub_DLL)
			enable_nvapi_Stub_DLL="$2"
			;;
		nvcuda-CUDA_Support)
			enable_nvcuda_CUDA_Support="$2"
			;;
		nvcuvid-CUDA_Video_Support)
			enable_nvcuvid_CUDA_Video_Support="$2"
			;;
		nvencodeapi-Video_Encoder)
			enable_nvencodeapi_Video_Encoder="$2"
			;;
		oleaut32-CreateTypeLib)
			enable_oleaut32_CreateTypeLib="$2"
			;;
		oleaut32-Load_Save_EMF)
			enable_oleaut32_Load_Save_EMF="$2"
			;;
		oleaut32-OLEPictureImpl_SaveAsFile)
			enable_oleaut32_OLEPictureImpl_SaveAsFile="$2"
			;;
		oleaut32-OleLoadPicture)
			enable_oleaut32_OleLoadPicture="$2"
			;;
		oleaut32-OleLoadPictureFile)
			enable_oleaut32_OleLoadPictureFile="$2"
			;;
		packager-DllMain)
			enable_packager_DllMain="$2"
			;;
		programs-findstr)
			enable_programs_findstr="$2"
			;;
		programs-systeminfo)
			enable_programs_systeminfo="$2"
			;;
		riched20-Class_Tests)
			enable_riched20_Class_Tests="$2"
			;;
		riched20-IText_Interface)
			enable_riched20_IText_Interface="$2"
			;;
		secur32-InitializeSecurityContextW)
			enable_secur32_InitializeSecurityContextW="$2"
			;;
		server-File_Permissions)
			enable_server_File_Permissions="$2"
			;;
		server-Key_State)
			enable_server_Key_State="$2"
			;;
		server-PeekMessage)
			enable_server_PeekMessage="$2"
			;;
		server-Realtime_Priority)
			enable_server_Realtime_Priority="$2"
			;;
		server-Signal_Thread)
			enable_server_Signal_Thread="$2"
			;;
		server-Stored_ACLs)
			enable_server_Stored_ACLs="$2"
			;;
		server-default_integrity)
			enable_server_default_integrity="$2"
			;;
		setupapi-DiskSpaceList)
			enable_setupapi_DiskSpaceList="$2"
			;;
		shdocvw-ParseURLFromOutsideSource_Tests)
			enable_shdocvw_ParseURLFromOutsideSource_Tests="$2"
			;;
		shell32-ACE_Viewer)
			enable_shell32_ACE_Viewer="$2"
			;;
		shell32-Context_Menu)
			enable_shell32_Context_Menu="$2"
			;;
		shell32-IconCache)
			enable_shell32_IconCache="$2"
			;;
		shell32-NewMenu_Interface)
			enable_shell32_NewMenu_Interface="$2"
			;;
		shell32-Progress_Dialog)
			enable_shell32_Progress_Dialog="$2"
			;;
		shell32-SFGAO_HASSUBFOLDER)
			enable_shell32_SFGAO_HASSUBFOLDER="$2"
			;;
		shell32-SHFileOperation_Move)
			enable_shell32_SHFileOperation_Move="$2"
			;;
		shell32-SHGetStockIconInfo)
			enable_shell32_SHGetStockIconInfo="$2"
			;;
		shell32-Toolbar_Bitmaps)
			enable_shell32_Toolbar_Bitmaps="$2"
			;;
		shell32-UnixFS)
			enable_shell32_UnixFS="$2"
			;;
		shlwapi-AssocGetPerceivedType)
			enable_shlwapi_AssocGetPerceivedType="$2"
			;;
		shlwapi-UrlCanonicalize)
			enable_shlwapi_UrlCanonicalize="$2"
			;;
		shlwapi-UrlCombine)
			enable_shlwapi_UrlCombine="$2"
			;;
		stdole32.idl-Typelib)
			enable_stdole32_idl_Typelib="$2"
			;;
		stdole32.tlb-SLTG_Typelib)
			enable_stdole32_tlb_SLTG_Typelib="$2"
			;;
		tasklist-basics)
			enable_tasklist_basics="$2"
			;;
		user32-DM_SETDEFID)
			enable_user32_DM_SETDEFID="$2"
			;;
		user32-Dialog_Paint_Event)
			enable_user32_Dialog_Paint_Event="$2"
			;;
		user32-DrawTextExW)
			enable_user32_DrawTextExW="$2"
			;;
		user32-FlashWindowEx)
			enable_user32_FlashWindowEx="$2"
			;;
		user32-GetSystemMetrics)
			enable_user32_GetSystemMetrics="$2"
			;;
		user32-Implement-CascadeWindows)
			enable_user32_Implement_CascadeWindows="$2"
			;;
		user32-LR_LOADFROMFILE)
			enable_user32_LR_LOADFROMFILE="$2"
			;;
		user32-ListBox_Size)
			enable_user32_ListBox_Size="$2"
			;;
		user32-LoadKeyboardLayoutEx)
			enable_user32_LoadKeyboardLayoutEx="$2"
			;;
		user32-MessageBox_WS_EX_TOPMOST)
			enable_user32_MessageBox_WS_EX_TOPMOST="$2"
			;;
		user32-Mouse_Message_Hwnd)
			enable_user32_Mouse_Message_Hwnd="$2"
			;;
		user32-QueryDisplayConfig)
			enable_user32_QueryDisplayConfig="$2"
			;;
		user32-Refresh_MDI_Menus)
			enable_user32_Refresh_MDI_Menus="$2"
			;;
		user32-ScrollWindowEx)
			enable_user32_ScrollWindowEx="$2"
			;;
		user32-message-order)
			enable_user32_message_order="$2"
			;;
		user32-msgbox-Support-WM_COPY-mesg)
			enable_user32_msgbox_Support_WM_COPY_mesg="$2"
			;;
		user32-rawinput-mouse)
			enable_user32_rawinput_mouse="$2"
			;;
		user32-rawinput-mouse-experimental)
			enable_user32_rawinput_mouse_experimental="$2"
			;;
		user32-recursive-activation)
			enable_user32_recursive_activation="$2"
			;;
		uxtheme-CloseThemeClass)
			enable_uxtheme_CloseThemeClass="$2"
			;;
		version-VerQueryValue)
			enable_version_VerQueryValue="$2"
			;;
		wbemdisp-ISWbemObject-Invoke)
			enable_wbemdisp_ISWbemObject_Invoke="$2"
			;;
		widl-SLTG_Typelib_Support)
			enable_widl_SLTG_Typelib_Support="$2"
			;;
		windows.networking.connectivity-new-dll)
			enable_windows_networking_connectivity_new_dll="$2"
			;;
		windowscodecs-GIF_Encoder)
			enable_windowscodecs_GIF_Encoder="$2"
			;;
		windowscodecs-TIFF_Support)
			enable_windowscodecs_TIFF_Support="$2"
			;;
		wine.inf-Directory_ContextMenuHandlers)
			enable_wine_inf_Directory_ContextMenuHandlers="$2"
			;;
		wine.inf-Dummy_CA_Certificate)
			enable_wine_inf_Dummy_CA_Certificate="$2"
			;;
		wineboot-HKEY_DYN_DATA)
			enable_wineboot_HKEY_DYN_DATA="$2"
			;;
		wineboot-ProxySettings)
			enable_wineboot_ProxySettings="$2"
			;;
		winecfg-Libraries)
			enable_winecfg_Libraries="$2"
			;;
		winecfg-Staging)
			enable_winecfg_Staging="$2"
			;;
		wined3d-Accounting)
			enable_wined3d_Accounting="$2"
			;;
		wined3d-Indexed_Vertex_Blending)
			enable_wined3d_Indexed_Vertex_Blending="$2"
			;;
		wined3d-SWVP-shaders)
			enable_wined3d_SWVP_shaders="$2"
			;;
		wined3d-Silence_FIXMEs)
			enable_wined3d_Silence_FIXMEs="$2"
			;;
		wined3d-WINED3DFMT_B8G8R8X8_UNORM)
			enable_wined3d_WINED3DFMT_B8G8R8X8_UNORM="$2"
			;;
		wined3d-mesa_texture_download)
			enable_wined3d_mesa_texture_download="$2"
			;;
		wined3d-unset-flip-gdi)
			enable_wined3d_unset_flip_gdi="$2"
			;;
		wined3d-wined3d_guess_gl_vendor)
			enable_wined3d_wined3d_guess_gl_vendor="$2"
			;;
		wined3d-zero-inf-shaders)
			enable_wined3d_zero_inf_shaders="$2"
			;;
		winedbg-Process_Arguments)
			enable_winedbg_Process_Arguments="$2"
			;;
		winedevice-Default_Drivers)
			enable_winedevice_Default_Drivers="$2"
			;;
		winemac.drv-no-flicker-patch)
			enable_winemac_drv_no_flicker_patch="$2"
			;;
		winemapi-user-xdg-mail)
			enable_winemapi_user_xdg_mail="$2"
			;;
		winemenubuilder-Desktop_Icon_Path)
			enable_winemenubuilder_Desktop_Icon_Path="$2"
			;;
		winemenubuilder-integration)
			enable_winemenubuilder_integration="$2"
			;;
		wineps.drv-PostScript_Fixes)
			enable_wineps_drv_PostScript_Fixes="$2"
			;;
		winex11-CandidateWindowPos)
			enable_winex11_CandidateWindowPos="$2"
			;;
		winex11-MWM_Decorations)
			enable_winex11_MWM_Decorations="$2"
			;;
		winex11-UpdateLayeredWindow)
			enable_winex11_UpdateLayeredWindow="$2"
			;;
		winex11-Vulkan_support)
			enable_winex11_Vulkan_support="$2"
			;;
		winex11-WM_WINDOWPOSCHANGING)
			enable_winex11_WM_WINDOWPOSCHANGING="$2"
			;;
		winex11-Window_Style)
			enable_winex11_Window_Style="$2"
			;;
		winex11-XEMBED)
			enable_winex11_XEMBED="$2"
			;;
		winex11-_NET_ACTIVE_WINDOW)
			enable_winex11__NET_ACTIVE_WINDOW="$2"
			;;
		winex11-ime-check-thread-data)
			enable_winex11_ime_check_thread_data="$2"
			;;
		winex11-key_translation)
			enable_winex11_key_translation="$2"
			;;
		winex11-wglShareLists)
			enable_winex11_wglShareLists="$2"
			;;
		winex11.drv-Query_server_position)
			enable_winex11_drv_Query_server_position="$2"
			;;
		wininet-Cleanup)
			enable_wininet_Cleanup="$2"
			;;
		winmm-mciSendCommandA)
			enable_winmm_mciSendCommandA="$2"
			;;
		wintab32-improvements)
			enable_wintab32_improvements="$2"
			;;
		wintrust-WTHelperGetProvCertFromChain)
			enable_wintrust_WTHelperGetProvCertFromChain="$2"
			;;
		wscript-support-d-u-switches)
			enable_wscript_support_d_u_switches="$2"
			;;
		xactengine-initial)
			enable_xactengine_initial="$2"
			;;
		xactengine3_7-Notification)
			enable_xactengine3_7_Notification="$2"
			;;
		xactengine3_7-PrepareWave)
			enable_xactengine3_7_PrepareWave="$2"
			;;
		*)
			return 1
			;;
	esac
	return 0
}

# Default settings
patch_enable_all 0
enable_autoconf=1
backend="patch"

# Find location of patches
patchdir="$(cd "$(dirname "$0")" && pwd)"
if test ! -f "$patchdir/patchinstall.sh"; then
	if test -f ./patchinstall.sh; then
		patchdir="$(pwd)"
	else
		abort "Failed to find patch directory."
	fi
fi

# Parse commandline arguments
if test "$#" -eq 0; then
	abort "No commandline arguments given, don't know what to do."
fi

while test "$#" -gt 0; do
	case "$1" in
		DESTDIR=*)
			DESTDIR="${1#*=}"
			shift
			;;

		--all)
			patch_enable_all 1
			shift
			;;

		--backend=*)
			backend="${1#*=}"
			shift
			;;

		--force-autoconf)
			enable_autoconf=2
			shift
			;;

		--help)
			usage
			exit 0
			;;

		--no-autoconf)
			enable_autoconf=0
			shift
			;;

		--upstream-commit)
			upstream_commit
			exit 0
			;;

		--version)
			version
			exit 0
			;;

		-W)
			# Disable patchset
			if ! patch_enable "$2" 2; then
				abort "Wrong usage of -W commandline argument, expected patchname."
			fi
			shift
			shift
			;;

		*)
			# Enable patchset
			if ! patch_enable "$1" 1; then
				abort "Unknown commandline argument $1."
			fi
			shift
			;;
	esac
done

# Determine DESTDIR if not explicitly specified
if test -z "$DESTDIR" -a -f ./tools/make_requests; then
	DESTDIR="$(pwd)"

elif test ! -f "$DESTDIR/tools/make_requests"; then
	abort "DESTDIR does not point to the Wine source tree."
fi

# Change directory to DESTDIR, eapply/epatch depends on that
if ! cd "$DESTDIR"; then
	abort "Unable to change directory to $DESTDIR."
fi

# Helper to update configure / the wineserver protocol if required
if ! command -v diff >/dev/null 2>&1 ||
   ! command -v grep >/dev/null 2>&1 ||
   ! command -v cmp >/dev/null 2>&1; then

	update_configure()
	{
		autoreconf -f
	}

	update_protocol()
	{
		./tools/make_requests
	}

else

	update_configure()
	{
		_file="./configure"

		if ! cp -a "$_file" "$_file.old"; then
			abort "failed to create $_file.old"
		fi

		if ! autoreconf -f; then
			rm "$_file.old"
			unset _file
			return 1
		fi

		# Shifting by 62 bits is undefined behaviour when off_t is 32-bit, see also
		# https://launchpad.net/ubuntu/+source/autoconf/2.69-6 - the bug is still
		# present in some other distros (including Archlinux).
		_large_off_old="^#define LARGE_OFF_T (((off_t) 1 << 62) - 1 + ((off_t) 1 << 62))\$"
		_large_off_new="#define LARGE_OFF_T ((((off_t) 1 << 31) << 31) - 1 + (((off_t) 1 << 31) << 31))"
		sed -i'' -e "s|$_large_off_old|$_large_off_new|g" "$_file"
		unset _large_off_old _large_off_new

		# Restore original timestamp when nothing changed
		if ! cmp "$_file.old" "$_file" >/dev/null; then
			rm "$_file.old"
		else
			mv "$_file.old" "$_file"
		fi

		unset _file
		return 0
	}

	update_protocol()
	{
		_file="./include/wine/server_protocol.h"

		if ! cp -a "$_file" "$_file.old"; then
			abort "failed to create $_file.old"
		fi

		if ! ./tools/make_requests; then
			rm "$_file.old"
			unset _file
			return 1
		fi

		# Restore original timestamp when nothing changed
		if diff -u "$_file.old" "$_file" |
		   grep -v "^[+-]#define SERVER_PROTOCOL_VERSION" |
		   grep -v "^\(+++\|---\)" | grep -q "^[+-]"; then
			rm "$_file.old"
		else
			mv "$_file.old" "$_file"
		fi

		unset _file
		return 0
	}
fi


# Most backends will try to use git, either directly or indirectly.
# Unfortunately this does not work when "$DESTDIR" points to a
# subdirectory of a git tree, which has the effect that no patches
# are applied, but the exitcode is zero. To avoid broken builds we
# will workaround this issue or abort.
test ! -e ".git" && git rev-parse --git-dir >/dev/null 2>&1
workaround_git_bug="$?"

# Apply the patches using gitapply.sh, a small wrapper around 'patch'
if test "$backend" = "patch"; then

	if test "$workaround_git_bug" -eq 0; then
		gitapply_args="--nogit"
	else
		gitapply_args=""
	fi

	if test "$enable_autoconf" -gt 1; then
		warning "Ignoring commandline argument --force-autoconf."
		enable_autoconf=1
	fi

	patch_apply_file()
	{
		printf '%s\n' "Applying $1"
		if ! "$patchdir/gitapply.sh" $gitapply_args < "$1"; then
			abort "Failed to apply patch, aborting!"
		fi
	}

# 'eapply/epatch' backend - used on Gentoo
elif test "$backend" = "eapply" -o "$backend" = "epatch"; then

	if test "$workaround_git_bug" -eq 0; then
		gitapply_args="--nogit"
	else
		gitapply_args=""
	fi

	if ! command -v "$backend"  >/dev/null 2>&1 || \
	   ! command -v ebegin      >/dev/null 2>&1 || \
	   ! command -v eend        >/dev/null 2>&1 || \
	   ! command -v nonfatal    >/dev/null 2>&1; then
		abort "Shell functions $backend/ebegin/eend/nonfatal not found. You have to source this script from your ebuild."
	fi

	if test "$enable_autoconf" -gt 1; then
		warning "Ignoring commandline argument --force-autoconf."
		enable_autoconf=1
	fi

	patch_apply_file()
	{
		_shortname="$(basename "$1")"
		if grep -q "^GIT binary patch" "$1"; then
			ebegin "Applying $_shortname"
			"$patchdir/gitapply.sh" $gitapply_args < "$1"
			if ! eend $?; then
				exit 1
			fi
		else
			# we are run from a subshell, so we can't call die
			if ! nonfatal "$backend" "$1"; then
				exit 1
			fi
		fi
		unset _shortname
	}

# GIT backend - apply patches using 'git am'
elif test "$backend" = "git" -o "$backend" = "git-am"; then

	if test "$workaround_git_bug" -eq 0; then
		abort "Backend 'git-am' not possible when DESTDIR points to a git subdirectory."
	fi

	patch_apply_file()
	{
		printf '%s\n' "Applying $1"
		if ! git am "$1"; then
			abort "Failed to apply patch, aborting!"
		fi
		if test "$enable_autoconf" -gt 1; then
			_do_commit=0

			# Run 'autoreconf -f' if required
			if git show --pretty=format: --name-only | grep -q "^\(configure.ac\|aclocal.m4\)$"; then
				if ! update_configure; then
					abort "'autoreconf -f' failed."
				fi
				git add ./configure
				git add ./include/config.h.in
				_do_commit=1
			fi

			# Run './tools/make_requests' if required
			if git show --pretty=format: --name-only | grep -q "^server/"; then
				if ! update_protocol; then
					abort "'./tools/make_requests' failed."
				fi
				git add ./include/wine/server_protocol.h
				git add ./server/trace.c
				git add ./server/request.h
				_do_commit=1
			fi

			if test "$_do_commit" -ne 0; then
				if ! git commit --amend --reuse-message HEAD; then
					abort "Failed to include autogenerated changes in commit."
				fi
			fi

			unset _do_commit
		fi
	}

# Git apply backend
elif test "$backend" = "git-apply"; then

	if test "$workaround_git_bug" -eq 0; then
		abort "Backend 'git-apply' not possible when DESTDIR points to a git subdirectory."
	fi

	if test "$enable_autoconf" -gt 1; then
		warning "Ignoring commandline argument --force-autoconf."
		enable_autoconf=1
	fi

	patch_apply_file()
	{
		printf '%s\n' "Applying $1"
		if ! git apply "$1"; then
			abort "Failed to apply patch, aborting!"
		fi
	}

# Stacked GIT backend - import the patches (mainly for developers)
elif test "$backend" = "stg"; then

	if test "$workaround_git_bug" -eq 0; then
		abort "Backend 'stg' not possible when DESTDIR points to a git subdirectory."
	fi

	# Only import the regular patches, no autogenerated ones -
	# moreover, don't run autoreconf or ./tools/make_requests.
	enable_autoconf=0

	patch_apply_file()
	{
		printf '%s\n' "Applying $1"
		_shortname="$(basename "$1")"
		if ! printf '%s\n' "staging/$_shortname" | cat - "$1" | stg import; then
			abort "Failed to apply patch, aborting!"
		fi
		unset _shortname
	}

else
	abort "Selected backend $backend not supported."
fi

patch_apply()
{
	patch_apply_file "$patchdir/$1"
}


if test "$enable_winex11_WM_WINDOWPOSCHANGING" -eq 1; then
	if test "$enable_winex11__NET_ACTIVE_WINDOW" -gt 1; then
		abort "Patchset winex11-_NET_ACTIVE_WINDOW disabled, but winex11-WM_WINDOWPOSCHANGING depends on that."
	fi
	enable_winex11__NET_ACTIVE_WINDOW=1
fi

if test "$enable_winedevice_Default_Drivers" -eq 1; then
	if test "$enable_ntoskrnl_Stubs" -gt 1; then
		abort "Patchset ntoskrnl-Stubs disabled, but winedevice-Default_Drivers depends on that."
	fi
	enable_ntoskrnl_Stubs=1
fi

if test "$enable_wined3d_Indexed_Vertex_Blending" -eq 1; then
	if test "$enable_wined3d_SWVP_shaders" -gt 1; then
		abort "Patchset wined3d-SWVP-shaders disabled, but wined3d-Indexed_Vertex_Blending depends on that."
	fi
	enable_wined3d_SWVP_shaders=1
fi

if test "$enable_user32_rawinput_mouse_experimental" -eq 1; then
	if test "$enable_user32_rawinput_mouse" -gt 1; then
		abort "Patchset user32-rawinput-mouse disabled, but user32-rawinput-mouse-experimental depends on that."
	fi
	enable_user32_rawinput_mouse=1
fi

if test "$enable_stdole32_tlb_SLTG_Typelib" -eq 1; then
	if test "$enable_widl_SLTG_Typelib_Support" -gt 1; then
		abort "Patchset widl-SLTG_Typelib_Support disabled, but stdole32.tlb-SLTG_Typelib depends on that."
	fi
	enable_widl_SLTG_Typelib_Support=1
fi

if test "$enable_shlwapi_UrlCombine" -eq 1; then
	if test "$enable_shlwapi_UrlCanonicalize" -gt 1; then
		abort "Patchset shlwapi-UrlCanonicalize disabled, but shlwapi-UrlCombine depends on that."
	fi
	enable_shlwapi_UrlCanonicalize=1
fi

if test "$enable_shell32_ACE_Viewer" -eq 1; then
	if test "$enable_shell32_Progress_Dialog" -gt 1; then
		abort "Patchset shell32-Progress_Dialog disabled, but shell32-ACE_Viewer depends on that."
	fi
	enable_shell32_Progress_Dialog=1
fi

if test "$enable_shell32_Progress_Dialog" -eq 1; then
	if test "$enable_kernel32_CopyFileEx" -gt 1; then
		abort "Patchset kernel32-CopyFileEx disabled, but shell32-Progress_Dialog depends on that."
	fi
	if test "$enable_shell32_SHFileOperation_Move" -gt 1; then
		abort "Patchset shell32-SHFileOperation_Move disabled, but shell32-Progress_Dialog depends on that."
	fi
	enable_kernel32_CopyFileEx=1
	enable_shell32_SHFileOperation_Move=1
fi

if test "$enable_server_Stored_ACLs" -eq 1; then
	if test "$enable_ntdll_DOS_Attributes" -gt 1; then
		abort "Patchset ntdll-DOS_Attributes disabled, but server-Stored_ACLs depends on that."
	fi
	if test "$enable_server_File_Permissions" -gt 1; then
		abort "Patchset server-File_Permissions disabled, but server-Stored_ACLs depends on that."
	fi
	enable_ntdll_DOS_Attributes=1
	enable_server_File_Permissions=1
fi

if test "$enable_server_File_Permissions" -eq 1; then
	if test "$enable_ntdll_Junction_Points" -gt 1; then
		abort "Patchset ntdll-Junction_Points disabled, but server-File_Permissions depends on that."
	fi
	enable_ntdll_Junction_Points=1
fi

if test "$enable_oleaut32_OLEPictureImpl_SaveAsFile" -eq 1; then
	if test "$enable_oleaut32_Load_Save_EMF" -gt 1; then
		abort "Patchset oleaut32-Load_Save_EMF disabled, but oleaut32-OLEPictureImpl_SaveAsFile depends on that."
	fi
	enable_oleaut32_Load_Save_EMF=1
fi

if test "$enable_nvencodeapi_Video_Encoder" -eq 1; then
	if test "$enable_nvcuvid_CUDA_Video_Support" -gt 1; then
		abort "Patchset nvcuvid-CUDA_Video_Support disabled, but nvencodeapi-Video_Encoder depends on that."
	fi
	enable_nvcuvid_CUDA_Video_Support=1
fi

if test "$enable_nvcuvid_CUDA_Video_Support" -eq 1; then
	if test "$enable_nvapi_Stub_DLL" -gt 1; then
		abort "Patchset nvapi-Stub_DLL disabled, but nvcuvid-CUDA_Video_Support depends on that."
	fi
	enable_nvapi_Stub_DLL=1
fi

if test "$enable_nvapi_Stub_DLL" -eq 1; then
	if test "$enable_nvcuda_CUDA_Support" -gt 1; then
		abort "Patchset nvcuda-CUDA_Support disabled, but nvapi-Stub_DLL depends on that."
	fi
	enable_nvcuda_CUDA_Support=1
fi

if test "$enable_ntdll_Builtin_Prot" -eq 1; then
	if test "$enable_ntdll_WRITECOPY" -gt 1; then
		abort "Patchset ntdll-WRITECOPY disabled, but ntdll-Builtin_Prot depends on that."
	fi
	enable_ntdll_WRITECOPY=1
fi

if test "$enable_ntdll_WRITECOPY" -eq 1; then
	if test "$enable_ntdll_ForceBottomUpAlloc" -gt 1; then
		abort "Patchset ntdll-ForceBottomUpAlloc disabled, but ntdll-WRITECOPY depends on that."
	fi
	enable_ntdll_ForceBottomUpAlloc=1
fi

if test "$enable_imm32_com_initialization" -eq 1; then
	if test "$enable_winex11__NET_ACTIVE_WINDOW" -gt 1; then
		abort "Patchset winex11-_NET_ACTIVE_WINDOW disabled, but imm32-com-initialization depends on that."
	fi
	enable_winex11__NET_ACTIVE_WINDOW=1
fi

if test "$enable_eventfd_synchronization" -eq 1; then
	if test "$enable_ntdll_Junction_Points" -gt 1; then
		abort "Patchset ntdll-Junction_Points disabled, but eventfd_synchronization depends on that."
	fi
	if test "$enable_server_PeekMessage" -gt 1; then
		abort "Patchset server-PeekMessage disabled, but eventfd_synchronization depends on that."
	fi
	if test "$enable_server_Realtime_Priority" -gt 1; then
		abort "Patchset server-Realtime_Priority disabled, but eventfd_synchronization depends on that."
	fi
	if test "$enable_server_Signal_Thread" -gt 1; then
		abort "Patchset server-Signal_Thread disabled, but eventfd_synchronization depends on that."
	fi
	enable_ntdll_Junction_Points=1
	enable_server_PeekMessage=1
	enable_server_Realtime_Priority=1
	enable_server_Signal_Thread=1
fi

if test "$enable_server_PeekMessage" -eq 1; then
	if test "$enable_server_Key_State" -gt 1; then
		abort "Patchset server-Key_State disabled, but server-PeekMessage depends on that."
	fi
	enable_server_Key_State=1
fi

if test "$enable_ntdll_Junction_Points" -eq 1; then
	if test "$enable_ntdll_DOS_Attributes" -gt 1; then
		abort "Patchset ntdll-DOS_Attributes disabled, but ntdll-Junction_Points depends on that."
	fi
	if test "$enable_ntdll_NtQueryEaFile" -gt 1; then
		abort "Patchset ntdll-NtQueryEaFile disabled, but ntdll-Junction_Points depends on that."
	fi
	enable_ntdll_DOS_Attributes=1
	enable_ntdll_NtQueryEaFile=1
fi

if test "$enable_dsound_EAX" -eq 1; then
	if test "$enable_dsound_Fast_Mixer" -gt 1; then
		abort "Patchset dsound-Fast_Mixer disabled, but dsound-EAX depends on that."
	fi
	enable_dsound_Fast_Mixer=1
fi

if test "$enable_dinput_SetActionMap_genre" -eq 1; then
	if test "$enable_dinput_joy_mappings" -gt 1; then
		abort "Patchset dinput-joy-mappings disabled, but dinput-SetActionMap-genre depends on that."
	fi
	enable_dinput_joy_mappings=1
fi

if test "$enable_ddraw_version_check" -eq 1; then
	if test "$enable_ddraw_Device_Caps" -gt 1; then
		abort "Patchset ddraw-Device_Caps disabled, but ddraw-version-check depends on that."
	fi
	enable_ddraw_Device_Caps=1
fi


# Patchset Compiler_Warnings
# |
# | Modified files:
# |   *	dlls/bcrypt/gnutls.c, dlls/d2d1/bitmap.c, dlls/d2d1/brush.c, dlls/d2d1/dc_render_target.c, dlls/d2d1/geometry.c,
# | 	dlls/d2d1/hwnd_render_target.c, dlls/d2d1/state_block.c, dlls/d3d10/effect.c, dlls/d3d11/view.c, dlls/d3d8/texture.c,
# | 	dlls/d3d9/texture.c, dlls/ddraw/viewport.c, dlls/dwrite/font.c, dlls/dwrite/layout.c, dlls/dxgi/output.c,
# | 	dlls/msctf/range.c, dlls/msxml3/schema.c, dlls/ntdll/unix/virtual.c, dlls/oleaut32/oleaut.c, dlls/rpcrt4/cstub.c,
# | 	dlls/rpcrt4/ndr_marshall.c, dlls/rpcrt4/ndr_typelib.c, dlls/vbscript/vbdisp.c, dlls/wbemdisp/locator.c,
# | 	dlls/windowscodecs/info.c, dlls/wsdapi/msgparams.c, include/wine/list.h, include/wine/rbtree.h, include/winnt.h
# |
if test "$enable_Compiler_Warnings" -eq 1; then
	patch_apply Compiler_Warnings/0001-windowscodecs-Avoid-implicit-cast-of-interface-point.patch
	patch_apply Compiler_Warnings/0021-d2d1-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0022-d3d11-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0023-d3d8-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0024-d3d9-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0025-ddraw-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0026-dwrite-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0027-msxml3-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0028-oleaut32-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0029-rpcrt4-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0030-vbscript-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0031-include-Check-element-type-in-CONTAINING_RECORD-and-.patch
	patch_apply Compiler_Warnings/0032-wsdapi-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0033-ntdll-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0034-bcrypt-Stop-compile-error-when-HAVE_GNUTLS_CIPHER_IN.patch
	patch_apply Compiler_Warnings/0035-d3d10-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0036-d2d1-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0037-dxgi-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0038-msctf-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0039-rpcrt4-Avoid-implicit-cast-of-interface-pointer.patch
	patch_apply Compiler_Warnings/0040-wbemdisp-Avoid-implicit-cast-of-interface-pointer.patch
fi

# Patchset Pipelight
# |
# | Modified files:
# |   *	dlls/user32/message.c, dlls/user32/tests/msg.c, dlls/winex11.drv/init.c, dlls/winex11.drv/opengl.c,
# | 	dlls/winex11.drv/x11drv.h
# |
if test "$enable_Pipelight" -eq 1; then
	patch_apply Pipelight/0001-winex11-Implement-X11DRV_FLUSH_GDI_DISPLAY-ExtEscape-c.patch
	patch_apply Pipelight/0002-user32-Decrease-minimum-SetTimer-interval-to-5-ms.patch
	patch_apply Pipelight/0004-winex11.drv-Indicate-direct-rendering-through-OpenGL.patch
fi

# Patchset Staging
# |
# | Modified files:
# |   *	Makefile.in, dlls/ntdll/loader.c
# |
if test "$enable_Staging" -eq 1; then
	patch_apply Staging/0001-kernel32-Add-winediag-message-to-show-warning-that-t.patch
	patch_apply Staging/0002-winelib-Append-Staging-at-the-end-of-the-version-s.patch
fi

# Patchset advapi32-LsaLookupPrivilegeName
# |
# | Modified files:
# |   *	dlls/advapi32/lsa.c, dlls/advapi32/tests/lsa.c
# |
if test "$enable_advapi32_LsaLookupPrivilegeName" -eq 1; then
	patch_apply advapi32-LsaLookupPrivilegeName/0001-advapi32-Fix-error-code-when-calling-LsaOpenPolicy-f.patch
fi

# Patchset api-ms-win-Stub_DLLs
# |
# | Modified files:
# |   *	configure.ac, dlls/ext-ms-win-appmodel-usercontext-l1-1-0/Makefile.in, dlls/ext-ms-win-appmodel-usercontext-l1-1-0/ext-
# | 	ms-win-appmodel-usercontext-l1-1-0.spec, dlls/ext-ms-win-appmodel-usercontext-l1-1-0/main.c, dlls/ext-ms-win-xaml-
# | 	pal-l1-1-0/Makefile.in, dlls/ext-ms-win-xaml-pal-l1-1-0/ext-ms-win-xaml-pal-l1-1-0.spec, dlls/ext-ms-win-xaml-
# | 	pal-l1-1-0/main.c, dlls/iertutil/Makefile.in, dlls/iertutil/iertutil.spec, dlls/iertutil/main.c,
# | 	dlls/uiautomationcore/Makefile.in, dlls/uiautomationcore/uia_main.c
# |
if test "$enable_api_ms_win_Stub_DLLs" -eq 1; then
	patch_apply api-ms-win-Stub_DLLs/0006-iertutil-Add-dll-and-add-stub-for-ordinal-811.patch
	patch_apply api-ms-win-Stub_DLLs/0009-ext-ms-win-xaml-pal-l1-1-0-Add-dll-and-add-stub-for-.patch
	patch_apply api-ms-win-Stub_DLLs/0010-ext-ms-win-appmodel-usercontext-l1-1-0-Add-dll-and-a.patch
	patch_apply api-ms-win-Stub_DLLs/0012-ext-ms-win-xaml-pal-l1-1-0-Add-stub-for-GetThemeServ.patch
	patch_apply api-ms-win-Stub_DLLs/0027-uiautomationcore-Add-dll-and-stub-some-functions.patch
fi

# Patchset api-ms-win-core-psapi-K32GetModuleInformation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#51199] Add missing api-ms-win-core-psapi forwards
# |
# | Modified files:
# |   *	dlls/api-ms-win-core-psapi-ansi-l1-1-0/api-ms-win-core-psapi-ansi-l1-1-0.spec, dlls/api-ms-win-core-psapi-l1-1-0/api-ms-
# | 	win-core-psapi-l1-1-0.spec
# |
if test "$enable_api_ms_win_core_psapi_K32GetModuleInformation" -eq 1; then
	patch_apply api-ms-win-core-psapi-K32GetModuleInformation/0001-api-ms-win-core-psapi-l1-1-0-Add-K32GetModuleBaseNam.patch
	patch_apply api-ms-win-core-psapi-K32GetModuleInformation/0002-api-ms-win-core-psapi-ansi-l1-1-0-add-K32GetModuleBa.patch
fi

# Patchset bcrypt-ECDHSecretAgreement
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47699] Multiple games fail to connect to online services (missing BCryptSecretAgreement / BCryptDeriveKey
# | 	implementation)
# |
# | Modified files:
# |   *	configure.ac, dlls/bcrypt/Makefile.in, dlls/bcrypt/bcrypt_internal.h, dlls/bcrypt/bcrypt_main.c, dlls/bcrypt/gcrypt.c,
# | 	dlls/bcrypt/gnutls.c, dlls/bcrypt/macos.c, dlls/bcrypt/tests/bcrypt.c, dlls/bcrypt/unixlib.c
# |
if test "$enable_bcrypt_ECDHSecretAgreement" -eq 1; then
	patch_apply bcrypt-ECDHSecretAgreement/0001-bcrypt-Allow-multiple-backends-to-coexist.patch
	patch_apply bcrypt-ECDHSecretAgreement/0002-bcrypt-Implement-BCryptSecretAgreement-with-libgcryp.patch
	patch_apply bcrypt-ECDHSecretAgreement/0003-bcrypt-Implement-BCRYPT_KDF_HASH.patch
fi

# Patchset cmd-launch-association
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#18154] cmd: Support for launching programs based on file association
# |
# | Modified files:
# |   *	programs/cmd/builtins.c, programs/cmd/tests/test_builtins.cmd, programs/cmd/tests/test_builtins.cmd.exp,
# | 	programs/cmd/wcmdmain.c
# |
if test "$enable_cmd_launch_association" -eq 1; then
	patch_apply cmd-launch-association/0001-cmd-Support-for-launching-programs-based-on-file-ass.patch
	patch_apply cmd-launch-association/0002-cmd-ftype-failed-to-clear-file-associations.patch
fi

# Patchset comctl32-rebar-capture
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#14750] comctl32: Fixed rebar behaviour when there's capture and no drag.
# |
# | Modified files:
# |   *	dlls/comctl32/rebar.c
# |
if test "$enable_comctl32_rebar_capture" -eq 1; then
	patch_apply comctl32-rebar-capture/0001-comctl32-Fixed-rebar-behaviour-when-there-s-capture-.patch
fi

# Patchset comctl32-version_6
# |
# | Modified files:
# |   *	dlls/comctl32/comctl32.h, dlls/comctl32/comctl32.rc, include/commctrl.h
# |
if test "$enable_comctl32_version_6" -eq 1; then
	patch_apply comctl32-version_6/0001-comctl32-Bump-version-to-6.0.patch
fi

# Patchset comdlg32-lpstrFileTitle
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38400] Postpone setting lpstrFileTitle in GetSaveFileNameW
# |   *	[#35200] Postpone setting lpstrFileTitle in GetSaveFileNameA
# |
# | Modified files:
# |   *	dlls/comdlg32/filedlg.c
# |
if test "$enable_comdlg32_lpstrFileTitle" -eq 1; then
	patch_apply comdlg32-lpstrFileTitle/0001-comdlg32-Postpone-setting-ofn-lpstrFileTitle-to-work.patch
fi

# Patchset crypt32-CMS_Certificates
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34388] Skip unknown item when decoding a CMS certificate
# |
# | Modified files:
# |   *	dlls/crypt32/decode.c
# |
if test "$enable_crypt32_CMS_Certificates" -eq 1; then
	patch_apply crypt32-CMS_Certificates/0001-crypt32-Skip-unknown-item-when-decoding-a-CMS-certif.patch
fi

# Patchset cryptext-CryptExtOpenCER
# |
# | Modified files:
# |   *	configure, configure.ac, dlls/cryptext/Makefile.in, dlls/cryptext/cryptext.spec, dlls/cryptext/cryptext_main.c,
# | 	dlls/cryptext/tests/Makefile.in, dlls/cryptext/tests/cryptext.c
# |
if test "$enable_cryptext_CryptExtOpenCER" -eq 1; then
	patch_apply cryptext-CryptExtOpenCER/0001-cryptext-Implement-CryptExtOpenCER.patch
fi

# Patchset d3drm-IDirect3D3-support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39346] Support IDirect3D3 when creating device.
# |
# | Modified files:
# |   *	dlls/d3drm/device.c
# |
if test "$enable_d3drm_IDirect3D3_support" -eq 1; then
	patch_apply d3drm-IDirect3D3-support/0001-d3drm-Support-IDirect3D3-when-creating-device.patch
fi

# Patchset d3dx11_43-D3DX11CreateTextureFromMemory
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50210] - Implement D3DX11GetImageInfoFromMemory
# |   *	[#45533] - Implement D3DX11CreateTextureFromMemory
# |
# | Modified files:
# |   *	dlls/d3dx11_42/Makefile.in, dlls/d3dx11_43/Makefile.in, dlls/d3dx11_43/main.c, dlls/d3dx11_43/texture.c
# |
if test "$enable_d3dx11_43_D3DX11CreateTextureFromMemory" -eq 1; then
	patch_apply d3dx11_43-D3DX11CreateTextureFromMemory/0001-d3dx11_43-Implement-D3DX11GetImageInfoFromMemory.patch
	patch_apply d3dx11_43-D3DX11CreateTextureFromMemory/0002-d3dx11_42-Implement-D3DX11CreateTextureFromMemory.patch
fi

# Patchset d3dx9_36-BumpLuminance
# |
# | Modified files:
# |   *	dlls/d3dx9_36/util.c
# |
if test "$enable_d3dx9_36_BumpLuminance" -eq 1; then
	patch_apply d3dx9_36-BumpLuminance/0002-d3dx9_36-Add-format-description-for-X8L8V8U8-for-for.patch
fi

# Patchset d3dx9_36-CloneEffect
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44635] Improve stub for ID3DXEffectImpl_CloneEffect
# |
# | Modified files:
# |   *	dlls/d3dx9_36/effect.c, dlls/d3dx9_36/tests/effect.c
# |
if test "$enable_d3dx9_36_CloneEffect" -eq 1; then
	patch_apply d3dx9_36-CloneEffect/0001-d3dx9_36-Improve-stub-for-ID3DXEffectImpl_CloneEffec.patch
fi

# Patchset d3dx9_36-D3DXDisassembleShader
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46649] Multiple applications need D3DXDisassembleShader() implementation (Tom Clancy's Rainbow Six: Vegas 2, The Void)
# |
# | Modified files:
# |   *	dlls/d3dx9_36/shader.c, dlls/d3dx9_36/tests/shader.c
# |
if test "$enable_d3dx9_36_D3DXDisassembleShader" -eq 1; then
	patch_apply d3dx9_36-D3DXDisassembleShader/0004-d3dx9_36-Implement-D3DXDisassembleShader.patch
	patch_apply d3dx9_36-D3DXDisassembleShader/0005-d3dx9_36-tests-Add-initial-tests-for-D3DXDisassemble.patch
	patch_apply d3dx9_36-D3DXDisassembleShader/0006-d3dx9_36-tests-Add-additional-tests-for-special-case.patch
fi

# Patchset d3dx9_36-D3DXOptimizeVertices
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47776] Timeshift needs D3DXOptimizeVertices()
# |
# | Modified files:
# |   *	dlls/d3dx9_24/d3dx9_24.spec, dlls/d3dx9_25/d3dx9_25.spec, dlls/d3dx9_26/d3dx9_26.spec, dlls/d3dx9_27/d3dx9_27.spec,
# | 	dlls/d3dx9_28/d3dx9_28.spec, dlls/d3dx9_29/d3dx9_29.spec, dlls/d3dx9_30/d3dx9_30.spec, dlls/d3dx9_31/d3dx9_31.spec,
# | 	dlls/d3dx9_32/d3dx9_32.spec, dlls/d3dx9_33/d3dx9_33.spec, dlls/d3dx9_34/d3dx9_34.spec, dlls/d3dx9_35/d3dx9_35.spec,
# | 	dlls/d3dx9_36/d3dx9_36.spec, dlls/d3dx9_36/mesh.c, dlls/d3dx9_36/tests/mesh.c, dlls/d3dx9_37/d3dx9_37.spec,
# | 	dlls/d3dx9_38/d3dx9_38.spec, dlls/d3dx9_39/d3dx9_39.spec, dlls/d3dx9_40/d3dx9_40.spec, dlls/d3dx9_41/d3dx9_41.spec,
# | 	dlls/d3dx9_42/d3dx9_42.spec, dlls/d3dx9_43/d3dx9_43.spec
# |
if test "$enable_d3dx9_36_D3DXOptimizeVertices" -eq 1; then
	patch_apply d3dx9_36-D3DXOptimizeVertices/0002-d3dx9_36-Add-semi-stub-for-D3DXOptimizeVertices.patch
fi

# Patchset d3dx9_36-D3DXSHProjectCubeMap
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46284] Add D3DXSHProjectCubeMap stub
# |
# | Modified files:
# |   *	dlls/d3dx9_24/d3dx9_24.spec, dlls/d3dx9_25/d3dx9_25.spec, dlls/d3dx9_26/d3dx9_26.spec, dlls/d3dx9_27/d3dx9_27.spec,
# | 	dlls/d3dx9_28/d3dx9_28.spec, dlls/d3dx9_29/d3dx9_29.spec, dlls/d3dx9_30/d3dx9_30.spec, dlls/d3dx9_31/d3dx9_31.spec,
# | 	dlls/d3dx9_32/d3dx9_32.spec, dlls/d3dx9_33/d3dx9_33.spec, dlls/d3dx9_34/d3dx9_34.spec, dlls/d3dx9_35/d3dx9_35.spec,
# | 	dlls/d3dx9_36/math.c, dlls/d3dx9_37/d3dx9_37.spec, dlls/d3dx9_38/d3dx9_38.spec, dlls/d3dx9_39/d3dx9_39.spec,
# | 	dlls/d3dx9_40/d3dx9_40.spec, dlls/d3dx9_41/d3dx9_41.spec, dlls/d3dx9_42/d3dx9_42.spec, dlls/d3dx9_43/d3dx9_43.spec,
# | 	include/d3dx9math.h
# |
if test "$enable_d3dx9_36_D3DXSHProjectCubeMap" -eq 1; then
	patch_apply d3dx9_36-D3DXSHProjectCubeMap/0001-d3dx9_-Add-D3DXSHProjectCubeMap-stub.patch
fi

# Patchset d3dx9_36-D3DXStubs
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#41697] Add stub for D3DXComputeTangent
# |
# | Modified files:
# |   *	dlls/d3dx9_24/d3dx9_24.spec, dlls/d3dx9_25/d3dx9_25.spec, dlls/d3dx9_26/d3dx9_26.spec, dlls/d3dx9_27/d3dx9_27.spec,
# | 	dlls/d3dx9_28/d3dx9_28.spec, dlls/d3dx9_29/d3dx9_29.spec, dlls/d3dx9_30/d3dx9_30.spec, dlls/d3dx9_31/d3dx9_31.spec,
# | 	dlls/d3dx9_32/d3dx9_32.spec, dlls/d3dx9_33/d3dx9_33.spec, dlls/d3dx9_34/d3dx9_34.spec, dlls/d3dx9_35/d3dx9_35.spec,
# | 	dlls/d3dx9_36/d3dx9_36.spec, dlls/d3dx9_36/mesh.c, dlls/d3dx9_37/d3dx9_37.spec, dlls/d3dx9_38/d3dx9_38.spec,
# | 	dlls/d3dx9_39/d3dx9_39.spec, dlls/d3dx9_40/d3dx9_40.spec, dlls/d3dx9_41/d3dx9_41.spec, dlls/d3dx9_42/d3dx9_42.spec,
# | 	dlls/d3dx9_43/d3dx9_43.spec
# |
if test "$enable_d3dx9_36_D3DXStubs" -eq 1; then
	patch_apply d3dx9_36-D3DXStubs/0003-d3dx9-Implement-D3DXComputeTangent.patch
fi

# Patchset d3dx9_36-DDS
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#26898] Support for DDS file format in D3DXSaveTextureToFileInMemory
# |
# | Modified files:
# |   *	dlls/d3dx9_36/d3dx9_private.h, dlls/d3dx9_36/surface.c, dlls/d3dx9_36/texture.c
# |
if test "$enable_d3dx9_36_DDS" -eq 1; then
	patch_apply d3dx9_36-DDS/0001-d3dx9_36-Add-support-for-FOURCC-surface-to-save_dds_.patch
	patch_apply d3dx9_36-DDS/0002-d3dx9_36-Improve-D3DXSaveTextureToFile-to-save-simpl.patch
fi

# Patchset d3dx9_36-Filter_Warnings
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33770] D3DCompileShader should filter specific warning messages
# |
# | Modified files:
# |   *	dlls/d3dx9_36/shader.c
# |
if test "$enable_d3dx9_36_Filter_Warnings" -eq 1; then
	patch_apply d3dx9_36-Filter_Warnings/0001-d3dx9_36-Filter-out-D3DCompile-warning-messages-that.patch
fi

# Patchset d3dx9_36-UpdateSkinnedMesh
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32572] Support for ID3DXSkinInfoImpl_UpdateSkinnedMesh
# |
# | Modified files:
# |   *	dlls/d3dx9_36/skin.c, dlls/d3dx9_36/tests/mesh.c
# |
if test "$enable_d3dx9_36_UpdateSkinnedMesh" -eq 1; then
	patch_apply d3dx9_36-UpdateSkinnedMesh/0001-d3dx9_36-Implement-ID3DXSkinInfoImpl_UpdateSkinnedMe.patch
fi

# Patchset dbghelp-Debug_Symbols
# |
# | Modified files:
# |   *	dlls/dbghelp/Makefile.in, dlls/dbghelp/elf_module.c
# |
if test "$enable_dbghelp_Debug_Symbols" -eq 1; then
	patch_apply dbghelp-Debug_Symbols/0001-dbghelp-Always-check-for-debug-symbols-in-BINDIR.patch
fi

# Patchset ddraw-Device_Caps
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37019] Don't set HWTRANSFORMANDLIGHT flag on d3d7 RGB device
# |   *	[#27002] Properly initialize caps->dwZBufferBitDepths in ddraw7_GetCaps
# |
# | Modified files:
# |   *	dlls/ddraw/ddraw.c, dlls/ddraw/tests/ddraw7.c
# |
if test "$enable_ddraw_Device_Caps" -eq 1; then
	patch_apply ddraw-Device_Caps/0001-ddraw-Don-t-set-HWTRANSFORMANDLIGHT-flag-on-d3d7-RGB.patch
	patch_apply ddraw-Device_Caps/0002-ddraw-Set-dwZBufferBitDepth-in-ddraw7_GetCaps.patch
fi

# Patchset ddraw-IDirect3DTexture2_Load
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#48537] Prezzie Hunt fails to launch
# |
# | Modified files:
# |   *	dlls/ddraw/surface.c, dlls/ddraw/tests/d3d.c, dlls/ddraw/tests/ddraw2.c
# |
if test "$enable_ddraw_IDirect3DTexture2_Load" -eq 1; then
	patch_apply ddraw-IDirect3DTexture2_Load/0001-ddraw-Allow-size-and-format-conversions-in-IDirect3D.patch
	patch_apply ddraw-IDirect3DTexture2_Load/0002-ddraw-tests-Add-more-tests-for-IDirect3DTexture2-Loa.patch
fi

# Patchset ddraw-Silence_FIXMEs
# |
# | Modified files:
# |   *	dlls/ddraw/executebuffer.c
# |
if test "$enable_ddraw_Silence_FIXMEs" -eq 1; then
	patch_apply ddraw-Silence_FIXMEs/0001-ddraw-Silence-noisy-FIXME-about-unimplemented-D3DPRO.patch
fi

# Patchset ddraw-version-check
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ddraw-Device_Caps
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#19153] Resident Evil 1 fails to start (needs IDirect3D3::EnumDevices() to return a device named "RGB Emulation")
# |
# | Modified files:
# |   *	dlls/ddraw/ddraw.c
# |
if test "$enable_ddraw_version_check" -eq 1; then
	patch_apply ddraw-version-check/0001-ddraw-Return-correct-devices-based-off-requested-Dir.patch
fi

# Patchset dinput-joy-mappings
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34108] dinput: Improve support for user Joystick configuration.
# |
# | Modified files:
# |   *	dlls/dinput/ansi.c, dlls/dinput/config.c, dlls/dinput/device.c, dlls/dinput/device_private.h, dlls/dinput/dinput_main.c,
# | 	dlls/dinput/joystick.c, dlls/dinput8/tests/device.c
# |
if test "$enable_dinput_joy_mappings" -eq 1; then
	patch_apply dinput-joy-mappings/0001-dinput-Load-users-Joystick-mappings.patch
	patch_apply dinput-joy-mappings/0002-dinput-Allow-empty-Joystick-mappings.patch
	patch_apply dinput-joy-mappings/0003-dinput-Support-username-in-Config-dialog.patch
	patch_apply dinput-joy-mappings/0004-dinput-Dont-allow-Fixed-actions-to-be-changed.patch
fi

# Patchset dinput-SetActionMap-genre
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	dinput-joy-mappings
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47326] dinput: Allow mapping of controls based of genre type.
# |
# | Modified files:
# |   *	dlls/dinput/device.c
# |
if test "$enable_dinput_SetActionMap_genre" -eq 1; then
	patch_apply dinput-SetActionMap-genre/0001-dinput-Allow-mapping-of-controls-based-of-Genre-type.patch
fi

# Patchset dinput-axis-recalc
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#41317] dinput: Recalculated Axis after deadzone change.
# |
# | Modified files:
# |   *	dlls/dinput/joystick.c
# |
if test "$enable_dinput_axis_recalc" -eq 1; then
	patch_apply dinput-axis-recalc/0001-dinput-Recalculated-Axis-after-deadzone-change.patch
fi

# Patchset dinput-reconnect-joystick
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34297] dinput: Allow reconnecting to disconnected joysticks
# |
# | Modified files:
# |   *	dlls/dinput/joystick_linuxinput.c
# |
if test "$enable_dinput_reconnect_joystick" -eq 1; then
	patch_apply dinput-reconnect-joystick/0001-dinput-Allow-reconnecting-to-disconnected-joysticks.patch
fi

# Patchset dinput-remap-joystick
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35815] dinput: Allow remapping of joystick buttons.
# |
# | Modified files:
# |   *	dlls/dinput/joystick.c, dlls/dinput/joystick_linux.c, dlls/dinput/joystick_linuxinput.c, dlls/dinput/joystick_osx.c,
# | 	dlls/dinput/joystick_private.h
# |
if test "$enable_dinput_remap_joystick" -eq 1; then
	patch_apply dinput-remap-joystick/0001-dinput-Allow-remapping-of-joystick-buttons.patch
fi

# Patchset dsound-Fast_Mixer
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30639] Audio stuttering and performance drops in multiple applications
# |
# | Modified files:
# |   *	dlls/dsound/dsound_main.c, dlls/dsound/dsound_private.h, dlls/dsound/mixer.c
# |
if test "$enable_dsound_Fast_Mixer" -eq 1; then
	patch_apply dsound-Fast_Mixer/0001-dsound-Add-a-linear-resampler-for-use-with-a-large-n.patch
fi

# Patchset dsound-EAX
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	dsound-Fast_Mixer
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42886] Mushroom Wars - Has no sounds.
# |
# | Modified files:
# |   *	dlls/dsound/Makefile.in, dlls/dsound/buffer.c, dlls/dsound/dsound.c, dlls/dsound/dsound_eax.h,
# | 	dlls/dsound/dsound_main.c, dlls/dsound/dsound_private.h, dlls/dsound/eax.c, dlls/dsound/mixer.c
# |
if test "$enable_dsound_EAX" -eq 1; then
	patch_apply dsound-EAX/0001-dsound-Apply-filters-before-sound-is-multiplied-to-s.patch
	patch_apply dsound-EAX/0002-dsound-Add-EAX-v1-constants-and-structs.patch
	patch_apply dsound-EAX/0003-dsound-Report-that-we-support-EAX-v1.patch
	patch_apply dsound-EAX/0004-dsound-Add-EAX-propset-stubs.patch
	patch_apply dsound-EAX/0005-dsound-Add-EAX-presets.patch
	patch_apply dsound-EAX/0006-dsound-Support-getting-and-setting-EAX-properties.patch
	patch_apply dsound-EAX/0007-dsound-Support-getting-and-setting-EAX-buffer-proper.patch
	patch_apply dsound-EAX/0008-dsound-Add-EAX-init-and-free-stubs.patch
	patch_apply dsound-EAX/0009-dsound-Feed-data-through-EAX-function.patch
	patch_apply dsound-EAX/0010-dsound-Allocate-EAX-delay-lines.patch
	patch_apply dsound-EAX/0011-dsound-Add-EAX-VerbPass-stub.patch
	patch_apply dsound-EAX/0012-dsound-Implement-EAX-lowpass-filter.patch
	patch_apply dsound-EAX/0013-dsound-Add-delay-line-EAX-functions.patch
	patch_apply dsound-EAX/0014-dsound-Implement-EAX-early-reflections.patch
	patch_apply dsound-EAX/0015-dsound-Implement-EAX-decorrelator.patch
	patch_apply dsound-EAX/0016-dsound-Implement-EAX-late-reverb.patch
	patch_apply dsound-EAX/0017-dsound-Implement-EAX-late-all-pass-filter.patch
	patch_apply dsound-EAX/0018-dsound-Various-improvements-to-EAX-support.patch
	patch_apply dsound-EAX/0019-dsound-Allow-disabling-of-EAX-support-in-the-registr.patch
	patch_apply dsound-EAX/0020-dsound-Add-stub-support-for-DSPROPSETID_EAX20_Listen.patch
	patch_apply dsound-EAX/0021-dsound-Add-stub-support-for-DSPROPSETID_EAX20_Buffer.patch
	patch_apply dsound-EAX/0022-dsound-Enable-EAX-by-default.patch
	patch_apply dsound-EAX/0023-dsound-Fake-success-for-EAX-Set-Buffer-ListenerPrope.patch
fi

# Patchset dwrite-FontFallback
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44052] - Support for font fallback.
# |
# | Modified files:
# |   *	dlls/dwrite/analyzer.c, dlls/dwrite/layout.c, dlls/dwrite/tests/layout.c
# |
if test "$enable_dwrite_FontFallback" -eq 1; then
	patch_apply dwrite-FontFallback/0001-dwrite-Test-IDWriteTextFormat-with-nonexistent-font.patch
	patch_apply dwrite-FontFallback/0002-dwrite-Test-GetMetrics-with-custom-fontcollection.patch
	patch_apply dwrite-FontFallback/0004-dwrite-Use-font-fallback-when-mapping-characters.patch
	patch_apply dwrite-FontFallback/0005-dwrite-Use-MapCharacters-for-non-visual-characters.patch
	patch_apply dwrite-FontFallback/0006-dwrite-Use-MapCharacters-for-dummy-line-metrics.patch
fi

# Patchset ntdll-DOS_Attributes
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#9158] Support for DOS hidden/system file attributes
# |   *	[#15679] cygwin symlinks not working in wine
# |
# | Modified files:
# |   *	configure.ac, dlls/ntdll/tests/directory.c, dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_DOS_Attributes" -eq 1; then
	patch_apply ntdll-DOS_Attributes/0001-ntdll-Implement-retrieving-DOS-attributes-in-fd_-get.patch
	patch_apply ntdll-DOS_Attributes/0003-ntdll-Implement-storing-DOS-attributes-in-NtSetInfor.patch
	patch_apply ntdll-DOS_Attributes/0004-ntdll-Implement-storing-DOS-attributes-in-NtCreateFi.patch
	patch_apply ntdll-DOS_Attributes/0005-libport-Add-support-for-Mac-OS-X-style-extended-attr.patch
	patch_apply ntdll-DOS_Attributes/0006-libport-Add-support-for-FreeBSD-style-extended-attri.patch
	patch_apply ntdll-DOS_Attributes/0007-ntdll-Perform-the-Unix-style-hidden-file-check-withi.patch
	patch_apply ntdll-DOS_Attributes/0008-ntdll-Always-store-SAMBA_XATTR_DOS_ATTRIB-when-path-.patch
fi

# Patchset ntdll-NtQueryEaFile
# |
# | Modified files:
# |   *	dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_NtQueryEaFile" -eq 1; then
	patch_apply ntdll-NtQueryEaFile/0001-ntdll-Improve-stub-of-NtQueryEaFile.patch
fi

# Patchset ntdll-Junction_Points
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-DOS_Attributes, ntdll-NtQueryEaFile
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#12401] NET Framework 2.0, 3.0, 4.0 installers and other apps that make use of GAC API for managed assembly
# | 	installation on NTFS filesystems need reparse point/junction API support
# | 	(FSCTL_SET_REPARSE_POINT/FSCTL_GET_REPARSE_POINT)
# |   *	[#44948] Multiple apps (Spine (Mod starter for Gothic), MS Office 365 installer) need CreateSymbolicLinkW implementation
# |
# | Modified files:
# |   *	configure.ac, dlls/kernel32/path.c, dlls/kernel32/tests/path.c, dlls/kernelbase/file.c, dlls/mountmgr.sys/device.c,
# | 	dlls/msvcp120/tests/msvcp120.c, dlls/msvcp140/tests/msvcp140.c, dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c,
# | 	include/Makefile.in, include/ntifs.h, include/winnt.h, include/winternl.h, programs/cmd/builtins.c,
# | 	programs/cmd/directory.c, server/fd.c, server/file.c, server/protocol.def
# |
if test "$enable_ntdll_Junction_Points" -eq 1; then
	patch_apply ntdll-Junction_Points/0001-ntdll-Add-support-for-junction-point-creation.patch
	patch_apply ntdll-Junction_Points/0002-ntdll-Add-support-for-reading-junction-points.patch
	patch_apply ntdll-Junction_Points/0003-ntdll-Add-support-for-deleting-junction-points.patch
	patch_apply ntdll-Junction_Points/0004-ntdll-Add-a-test-for-junction-point-advertisement.patch
	patch_apply ntdll-Junction_Points/0005-server-Add-support-for-deleting-junction-points-with.patch
	patch_apply ntdll-Junction_Points/0006-kernel32-Advertise-junction-point-support.patch
	patch_apply ntdll-Junction_Points/0007-ntdll-Add-support-for-absolute-symlink-creation.patch
	patch_apply ntdll-Junction_Points/0008-ntdll-Add-support-for-reading-absolute-symlinks.patch
	patch_apply ntdll-Junction_Points/0009-ntdll-Add-support-for-deleting-symlinks.patch
	patch_apply ntdll-Junction_Points/0010-ntdll-Add-support-for-relative-symlink-creation.patch
	patch_apply ntdll-Junction_Points/0011-ntdll-Add-support-for-reading-relative-symlinks.patch
	patch_apply ntdll-Junction_Points/0012-ntdll-Add-support-for-file-symlinks.patch
	patch_apply ntdll-Junction_Points/0013-ntdll-Allow-creation-of-dangling-reparse-points-to-n.patch
	patch_apply ntdll-Junction_Points/0014-ntdll-Correctly-report-file-symbolic-links-as-files.patch
	patch_apply ntdll-Junction_Points/0015-kernelbase-Convert-FILE_FLAG_OPEN_REPARSE_POINT-for-.patch
	patch_apply ntdll-Junction_Points/0016-server-Implement-FILE_OPEN_REPARSE_POINT-option.patch
	patch_apply ntdll-Junction_Points/0017-ntdll-Allow-set_file_times_precise-to-work-on-repars.patch
	patch_apply ntdll-Junction_Points/0018-server-Properly-handle-file-symlink-deletion.patch
	patch_apply ntdll-Junction_Points/0019-server-Properly-handle-deleting-dangling-symlinks.patch
	patch_apply ntdll-Junction_Points/0020-kernelbase-Use-FILE_OPEN_REPARSE_POINT-in-RemoveDire.patch
	patch_apply ntdll-Junction_Points/0021-ntdll-Always-report-symbolic-links-as-containing-zer.patch
	patch_apply ntdll-Junction_Points/0022-ntdll-Find-dangling-symlinks-quickly.patch
	patch_apply ntdll-Junction_Points/0023-server-Fix-obtaining-information-about-a-symlink.patch
	patch_apply ntdll-Junction_Points/0024-ntdll-Succeed-with-no-data-for-NtReadFile-on-reparse.patch
	patch_apply ntdll-Junction_Points/0025-ntdll-Support-reparse-point-properties-in-fd_get_fil.patch
	patch_apply ntdll-Junction_Points/0026-ntdll-Add-support-for-FileAttributeTagInformation.patch
	patch_apply ntdll-Junction_Points/0027-kernel32-Implement-CreateSymbolicLink-A-W-with-ntdll.patch
	patch_apply ntdll-Junction_Points/0028-kernel32-Add-reparse-support-to-FindNextFile.patch
	patch_apply ntdll-Junction_Points/0029-wcmd-Display-reparse-point-type-in-directory-listing.patch
	patch_apply ntdll-Junction_Points/0030-wcmd-Show-reparse-point-target-in-directory-listing.patch
	patch_apply ntdll-Junction_Points/0031-wcmd-Add-junction-point-support-to-mklink.patch
	patch_apply ntdll-Junction_Points/0032-server-Properly-handle-renames-involving-symlinks.patch
	patch_apply ntdll-Junction_Points/0033-kernelbase-Use-FILE_OPEN_REPARSE_POINT-in-MoveFileWi.patch
	patch_apply ntdll-Junction_Points/0034-kernelbase-Use-FILE_OPEN_REPARSE_POINT-in-DeleteFile.patch
	patch_apply ntdll-Junction_Points/0035-ntdll-Treat-undecoded-unix-symlinks-as-WSL-Linux-Uni.patch
	patch_apply ntdll-Junction_Points/0036-ntdll-Add-support-for-creating-Unix-Linux-symlinks.patch
	patch_apply ntdll-Junction_Points/0037-ntdll-Strip-the-wine-prefix-from-reparse-point-paths.patch
	patch_apply ntdll-Junction_Points/0038-ntdll-Add-a-marker-to-reparse-point-paths-to-indicat.patch
	patch_apply ntdll-Junction_Points/0039-server-Rewrite-absolute-reparse-point-targets-if-the.patch
fi

# Patchset server-Key_State
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#26269] BioShock 2: Loss of keyboard input on loading screen
# |   *	[#31899] No keyboard input in La-Mulana remake (GetKeyState should behave similar to GetAsyncKeyState for specific
# | 	window messages / queue states)
# |   *	[#35907] Caps Lock state gets confused with multiple processes/threads
# |   *	[#45385] Wrong state of virtual keys after cycling windows. Usually VK_MENU, VK_SHIFT and VK_CONTROL, but every key can
# | 	be affected.
# |
# | Modified files:
# |   *	dlls/user32/tests/input.c, server/queue.c
# |
if test "$enable_server_Key_State" -eq 1; then
	patch_apply server-Key_State/0001-server-Create-message-queue-and-thread-input-in-set_.patch
	patch_apply server-Key_State/0002-server-Lock-thread-input-keystate-whenever-it-is-mod.patch
	patch_apply server-Key_State/0003-server-Create-message-queue-and-thread-input-in-get_.patch
fi

# Patchset server-PeekMessage
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	server-Key_State
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#28884] GetMessage should remove already seen messages with higher priority
# |
# | Modified files:
# |   *	dlls/user32/tests/msg.c, server/queue.c
# |
if test "$enable_server_PeekMessage" -eq 1; then
	patch_apply server-PeekMessage/0001-server-Fix-handling-of-GetMessage-after-previous-Pee.patch
fi

# Patchset server-Realtime_Priority
# |
# | Modified files:
# |   *	server/Makefile.in, server/main.c, server/scheduler.c, server/thread.c, server/thread.h
# |
if test "$enable_server_Realtime_Priority" -eq 1; then
	patch_apply server-Realtime_Priority/0001-wineserver-Draft-to-implement-priority-levels-throug.patch
fi

# Patchset server-Signal_Thread
# |
# | Modified files:
# |   *	server/thread.c, server/thread.h
# |
if test "$enable_server_Signal_Thread" -eq 1; then
	patch_apply server-Signal_Thread/0001-server-Do-not-signal-thread-until-it-is-really-gone.patch
fi

# Patchset eventfd_synchronization
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-DOS_Attributes, ntdll-NtQueryEaFile, ntdll-Junction_Points, server-Key_State, server-PeekMessage, server-
# | 	Realtime_Priority, server-Signal_Thread
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#36692] Many multi-threaded applications have poor performance due to heavy use of synchronization primitives
# |
# | Modified files:
# |   *	README.esync, configure, configure.ac, dlls/kernel32/tests/sync.c, dlls/ntdll/Makefile.in, dlls/ntdll/unix/esync.c,
# | 	dlls/ntdll/unix/esync.h, dlls/ntdll/unix/loader.c, dlls/ntdll/unix/server.c, dlls/ntdll/unix/sync.c,
# | 	dlls/ntdll/unix/unix_private.h, dlls/ntdll/unix/virtual.c, dlls/rpcrt4/rpc_server.c, include/config.h.in,
# | 	server/Makefile.in, server/async.c, server/atom.c, server/change.c, server/clipboard.c, server/completion.c,
# | 	server/console.c, server/debugger.c, server/device.c, server/directory.c, server/esync.c, server/esync.h,
# | 	server/event.c, server/fd.c, server/file.c, server/file.h, server/handle.c, server/hook.c, server/mailslot.c,
# | 	server/main.c, server/mapping.c, server/mutex.c, server/named_pipe.c, server/object.h, server/process.c,
# | 	server/process.h, server/protocol.def, server/queue.c, server/registry.c, server/request.c, server/semaphore.c,
# | 	server/serial.c, server/signal.c, server/sock.c, server/symlink.c, server/thread.c, server/thread.h, server/timer.c,
# | 	server/token.c, server/winstation.c
# |
if test "$enable_eventfd_synchronization" -eq 1; then
	patch_apply eventfd_synchronization/0001-configure-Check-for-sys-eventfd.h-ppoll-and-shm_open.patch
	patch_apply eventfd_synchronization/0002-server-Create-server-objects-for-eventfd-based-synch.patch
	patch_apply eventfd_synchronization/0003-ntdll-Create-eventfd-based-objects-for-semaphores.patch
	patch_apply eventfd_synchronization/0004-ntdll-Implement-NtReleaseSemaphore.patch
	patch_apply eventfd_synchronization/0005-ntdll-Implement-NtClose.patch
	patch_apply eventfd_synchronization/0006-ntdll-Implement-NtWaitForMultipleObjects.patch
	patch_apply eventfd_synchronization/0007-ntdll-server-Implement-NtCreateEvent.patch
	patch_apply eventfd_synchronization/0008-ntdll-Implement-NtSetEvent.patch
	patch_apply eventfd_synchronization/0009-ntdll-Implement-NtResetEvent.patch
	patch_apply eventfd_synchronization/0010-ntdll-Implement-waiting-on-manual-reset-events.patch
	patch_apply eventfd_synchronization/0011-server-Add-an-object-operation-to-grab-the-esync-fil.patch
	patch_apply eventfd_synchronization/0012-server-Add-a-request-to-get-the-eventfd-file-descrip.patch
	patch_apply eventfd_synchronization/0013-server-Create-eventfd-file-descriptors-for-process-o.patch
	patch_apply eventfd_synchronization/0014-ntdll-server-Implement-waiting-on-server-bound-objec.patch
	patch_apply eventfd_synchronization/0015-server-Create-eventfd-file-descriptors-for-event-obj.patch
	patch_apply eventfd_synchronization/0016-server-Allow-re-setting-esync-events-on-the-server-s.patch
	patch_apply eventfd_synchronization/0017-ntdll-Try-again-if-poll-returns-EINTR.patch
	patch_apply eventfd_synchronization/0018-server-Create-eventfd-file-descriptors-for-thread-ob.patch
	patch_apply eventfd_synchronization/0019-rpcrt4-Avoid-closing-the-server-thread-handle-while-.patch
	patch_apply eventfd_synchronization/0020-server-Create-eventfd-file-descriptors-for-message-q.patch
	patch_apply eventfd_synchronization/0021-server-ntdll-Implement-message-waits.patch
	patch_apply eventfd_synchronization/0022-server-Create-eventfd-descriptors-for-device-manager.patch
	patch_apply eventfd_synchronization/0023-ntdll-server-Implement-NtCreateMutant.patch
	patch_apply eventfd_synchronization/0024-ntdll-Implement-NtReleaseMutant.patch
	patch_apply eventfd_synchronization/0025-ntdll-Implement-waiting-on-mutexes.patch
	patch_apply eventfd_synchronization/0026-ntdll-Implement-wait-all.patch
	patch_apply eventfd_synchronization/0027-esync-Add-a-README.patch
	patch_apply eventfd_synchronization/0028-ntdll-Implement-NtSignalAndWaitForSingleObject.patch
	patch_apply eventfd_synchronization/0029-ntdll-Implement-NtOpenSemaphore.patch
	patch_apply eventfd_synchronization/0030-ntdll-Implement-NtOpenEvent.patch
	patch_apply eventfd_synchronization/0031-ntdll-Implement-NtOpenMutant.patch
	patch_apply eventfd_synchronization/0032-server-Implement-esync_map_access.patch
	patch_apply eventfd_synchronization/0033-server-Implement-NtDuplicateObject.patch
	patch_apply eventfd_synchronization/0034-server-Create-eventfd-descriptors-for-timers.patch
	patch_apply eventfd_synchronization/0035-ntdll-server-Implement-alertable-waits.patch
	patch_apply eventfd_synchronization/0036-esync-Update-README.patch
	patch_apply eventfd_synchronization/0037-kernel32-tests-Mark-some-existing-tests-as-failing-u.patch
	patch_apply eventfd_synchronization/0038-kernel32-tests-Add-some-semaphore-tests.patch
	patch_apply eventfd_synchronization/0039-kernel32-tests-Add-some-event-tests.patch
	patch_apply eventfd_synchronization/0040-kernel32-tests-Add-some-mutex-tests.patch
	patch_apply eventfd_synchronization/0041-kernel32-tests-Add-some-tests-for-wait-timeouts.patch
	patch_apply eventfd_synchronization/0042-kernel32-tests-Zigzag-test.patch
	patch_apply eventfd_synchronization/0043-ntdll-Implement-NtQuerySemaphore.patch
	patch_apply eventfd_synchronization/0044-ntdll-Implement-NtQueryEvent.patch
	patch_apply eventfd_synchronization/0045-ntdll-Implement-NtQueryMutant.patch
	patch_apply eventfd_synchronization/0046-server-Create-eventfd-descriptors-for-pseudo-fd-obje.patch
	patch_apply eventfd_synchronization/0047-esync-Update-README.patch
	patch_apply eventfd_synchronization/0048-esync-Add-note-about-file-limits-not-being-raised-wh.patch
	patch_apply eventfd_synchronization/0049-ntdll-Try-to-avoid-poll-for-uncontended-objects.patch
	patch_apply eventfd_synchronization/0050-ntdll-server-Try-to-avoid-poll-for-signaled-events.patch
	patch_apply eventfd_synchronization/0051-esync-Update-README.patch
	patch_apply eventfd_synchronization/0052-ntdll-Implement-NtPulseEvent.patch
	patch_apply eventfd_synchronization/0053-esync-Update-README.patch
	patch_apply eventfd_synchronization/0054-server-Create-esync-file-descriptors-for-true-file-o.patch
	patch_apply eventfd_synchronization/0055-ntdll-server-Abandon-esync-mutexes-on-thread-exit.patch
	patch_apply eventfd_synchronization/0056-server-Create-esync-file-descriptors-for-console-ser.patch
fi

# Patchset explorer-Video_Registry_Key
# |
# | Modified files:
# |   *	dlls/advapi32/tests/registry.c, programs/explorer/desktop.c
# |
if test "$enable_explorer_Video_Registry_Key" -eq 1; then
	patch_apply explorer-Video_Registry_Key/0001-explorer-Create-CurrentControlSet-Control-Video-regi.patch
fi

# Patchset fonts-Missing_Fonts
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32323] Implement an Arial replacement font
# |   *	[#32342] Implement a Times New Roman replacement font
# |   *	[#20456] Implement a Courier New replacement font
# |   *	[#13829] Implement a Microsoft Yahei replacement font
# |
# | Modified files:
# |   *	COPYING.arial, COPYING.cour, COPYING.msyh, COPYING.times, LICENSE, fonts/Makefile.in, fonts/arial.sfd, fonts/arial.ttf,
# | 	fonts/cour.sfd, fonts/cour.ttf, fonts/msyh.sfd, fonts/msyh.ttf, fonts/times.sfd, fonts/times.ttf
# |
if test "$enable_fonts_Missing_Fonts" -eq 1; then
	patch_apply fonts-Missing_Fonts/0001-fonts-Add-Liberation-Sans-as-an-Arial-replacement.patch
	patch_apply fonts-Missing_Fonts/0002-fonts-Add-Liberation-Serif-as-an-Times-New-Roman-rep.patch
	patch_apply fonts-Missing_Fonts/0003-fonts-Add-Liberation-Mono-as-an-Courier-New-replacem.patch
	patch_apply fonts-Missing_Fonts/0004-fonts-Add-WenQuanYi-Micro-Hei-as-a-Microsoft-Yahei-r.patch
	patch_apply fonts-Missing_Fonts/0005-Add-licenses-for-fonts-as-separate-files.patch
fi

# Patchset gdi32-rotation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34579] gdi32: fix for rotated Arc, ArcTo, Chord and Pie drawing problem
# |   *	[#35331] gdi32: fix for rotated ellipse
# |
# | Modified files:
# |   *	dlls/gdi32/dibdrv/graphics.c, dlls/gdi32/gdi_private.h
# |
if test "$enable_gdi32_rotation" -eq 1; then
	patch_apply gdi32-rotation/0001-gdi32-fix-for-rotated-Arc-ArcTo-Chord-and-Pie-drawin.patch
	patch_apply gdi32-rotation/0002-gdi32-fix-for-rotated-ellipse.patch
fi

# Patchset gdiplus-Performance-Improvements
# |
# | Modified files:
# |   *	dlls/gdiplus/graphics.c
# |
if test "$enable_gdiplus_Performance_Improvements" -eq 1; then
	patch_apply gdiplus-Performance-Improvements/0001-gdiplus-Change-the-order-of-x-y-loops-in-the-scaler.patch
	patch_apply gdiplus-Performance-Improvements/0002-gdiplus-Change-multiplications-by-additions-in-the-x.patch
	patch_apply gdiplus-Performance-Improvements/0003-gdiplus-Remove-ceilf-floorf-calls-from-bilinear-scal.patch
	patch_apply gdiplus-Performance-Improvements/0004-gdiplus-Prefer-using-pre-multiplied-ARGB-data-in-the.patch
fi

# Patchset winex11-_NET_ACTIVE_WINDOW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#2155] Forward activate window requests to WM using _NET_ACTIVE_WINDOW
# |
# | Modified files:
# |   *	dlls/user32/driver.c, dlls/user32/focus.c, dlls/user32/user_private.h, dlls/winex11.drv/event.c,
# | 	dlls/winex11.drv/window.c, dlls/winex11.drv/winex11.drv.spec, dlls/winex11.drv/x11drv.h, dlls/winex11.drv/x11drv_main.c
# |
if test "$enable_winex11__NET_ACTIVE_WINDOW" -eq 1; then
	patch_apply winex11-_NET_ACTIVE_WINDOW/0001-winex11.drv-Add-support-for-_NET_ACTIVE_WINDOW.patch
	patch_apply winex11-_NET_ACTIVE_WINDOW/0002-user32-Before-asking-a-WM-to-activate-a-window-make-.patch
fi

# Patchset imm32-com-initialization
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	winex11-_NET_ACTIVE_WINDOW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42695] Path of Exile fails - CoCreateInstance() called in uninitialized apartment
# |   *	[#47387] Victor Vran has no sound
# |
# | Modified files:
# |   *	dlls/imm32/Makefile.in, dlls/imm32/imm.c, dlls/imm32/imm32.spec, dlls/imm32/tests/imm32.c, dlls/user32/focus.c,
# | 	dlls/user32/misc.c, dlls/user32/user_private.h
# |
if test "$enable_imm32_com_initialization" -eq 1; then
	patch_apply imm32-com-initialization/0001-imm32-Automatically-initialize-COM-on-window-activat.patch
fi

# Patchset imm32-message_on_focus
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31157] imm32: Only generate 'WM_IME_SETCONTEXT' message if window has focus.
# |
# | Modified files:
# |   *	dlls/imm32/imm.c
# |
if test "$enable_imm32_message_on_focus" -eq 1; then
	patch_apply imm32-message_on_focus/0001-imm32-Only-generate-WM_IME_SETCONTEXT-message-if-win.patch
fi

# Patchset include-winsock
# |
# | Modified files:
# |   *	include/winsock.h
# |
if test "$enable_include_winsock" -eq 1; then
	patch_apply include-winsock/0001-include-Always-define-hton-ntoh-macros.patch
fi

# Patchset inseng-Implementation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39456] Implement CIF reader and download functionality in inseng.dll
# |
# | Modified files:
# |   *	dlls/inseng/Makefile.in, dlls/inseng/icif.c, dlls/inseng/inf.c, dlls/inseng/inseng.spec, dlls/inseng/inseng_main.c,
# | 	dlls/inseng/inseng_private.h, include/inseng.idl
# |
if test "$enable_inseng_Implementation" -eq 1; then
	patch_apply inseng-Implementation/0001-inseng-Implement-CIF-reader-and-download-functions.patch
fi

# Patchset iphlpapi-System_Ping
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8332] Fallback to system ping command when CAP_NET_RAW is not available
# |
# | Modified files:
# |   *	dlls/iphlpapi/icmp.c
# |
if test "$enable_iphlpapi_System_Ping" -eq 1; then
	patch_apply iphlpapi-System_Ping/0001-iphlpapi-Fallback-to-system-ping-when-ICMP-permissio.patch
fi

# Patchset kernel32-CopyFileEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22692] Add support for CopyFileEx progress callback
# |   *	[#22690] Allow to cancel a file operation via progress callback
# |
# | Modified files:
# |   *	dlls/kernel32/tests/file.c, dlls/kernelbase/file.c
# |
if test "$enable_kernel32_CopyFileEx" -eq 1; then
	patch_apply kernel32-CopyFileEx/0001-kernel32-Add-support-for-progress-callback-in-CopyFi.patch
fi

# Patchset kernel32-Debugger
# |
# | Modified files:
# |   *	dlls/kernelbase/debug.c
# |
if test "$enable_kernel32_Debugger" -eq 1; then
	patch_apply kernel32-Debugger/0001-kernel32-Always-start-debugger-on-WinSta0.patch
fi

# Patchset kernel32-Job_Tests
# |
# | Modified files:
# |   *	dlls/kernel32/tests/process.c
# |
if test "$enable_kernel32_Job_Tests" -eq 1; then
	patch_apply kernel32-Job_Tests/0001-kernel32-tests-Add-tests-for-job-object-accounting.patch
fi

# Patchset kernel32-Processor_Group
# |
# | Modified files:
# |   *	dlls/kernelbase/thread.c
# |
if test "$enable_kernel32_Processor_Group" -eq 1; then
	patch_apply kernel32-Processor_Group/0002-kernel32-Add-stub-for-SetThreadIdealProcessorEx.patch
fi

# Patchset krnl386.exe16-GDT_LDT_Emulation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30237] Implement emulation of GDT and LDT access in Win98 mode
# |
# | Modified files:
# |   *	dlls/krnl386.exe16/instr.c
# |
if test "$enable_krnl386_exe16_GDT_LDT_Emulation" -eq 1; then
	patch_apply krnl386.exe16-GDT_LDT_Emulation/0001-krnl386.exe16-Emulate-GDT-and-LDT-access.patch
fi

# Patchset krnl386.exe16-Invalid_Console_Handles
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#7106] Translate all invalid console handles into usable DOS handles
# |
# | Modified files:
# |   *	dlls/krnl386.exe16/file.c
# |
if test "$enable_krnl386_exe16_Invalid_Console_Handles" -eq 1; then
	patch_apply krnl386.exe16-Invalid_Console_Handles/0001-krnl386.exe16-Really-translate-all-invalid-console-h.patch
fi

# Patchset libs-Unicode_Collation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#5163] Microsoft Office XP 2002 installer reports error 25003 (installation source corrupted), custom action 'CADpc'
# | 	returns 1603
# |   *	[#10767] Fix comparison of punctuation characters in lstrcmp
# |   *	[#32490] Graphical issues in Inquisitor
# |
# | Modified files:
# |   *	dlls/kernel32/tests/locale.c, dlls/kernelbase/locale.c
# |
if test "$enable_libs_Unicode_Collation" -eq 1; then
	patch_apply libs-Unicode_Collation/0001-kernelbase-Implement-sortkey-generation-on-official-.patch
	patch_apply libs-Unicode_Collation/0002-kernelbase-Implement-sortkey-punctuation.patch
	patch_apply libs-Unicode_Collation/0003-kernelbase-Implement-sortkey-for-Japanese-characters.patch
	patch_apply libs-Unicode_Collation/0004-kernelbase-Implement-sortkey-expansion.patch
	patch_apply libs-Unicode_Collation/0005-kernelbase-Implement-sortkey-language-support.patch
	patch_apply libs-Unicode_Collation/0006-kernelbase-Implement-CompareString-functions.patch
fi

# Patchset loader-KeyboardLayouts
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47439] loader: Add Keyboard Layouts registry enteries.
# |
# | Modified files:
# |   *	dlls/user32/input.c, dlls/user32/tests/input.c, loader/wine.inf.in
# |
if test "$enable_loader_KeyboardLayouts" -eq 1; then
	patch_apply loader-KeyboardLayouts/0001-loader-Add-Keyboard-Layouts-registry-enteries.patch
	patch_apply loader-KeyboardLayouts/0002-user32-Improve-GetKeyboardLayoutList.patch
fi

# Patchset mfplat-streaming-support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49692] Multiple applications need a Media Foundation media source implementation
# |
# | Modified files:
# |   *	dlls/mf/topology.c, dlls/mfplat/main.c, dlls/winegstreamer/Makefile.in, dlls/winegstreamer/audioconvert.c,
# | 	dlls/winegstreamer/colorconvert.c, dlls/winegstreamer/decode_transform.c, dlls/winegstreamer/gst_private.h,
# | 	dlls/winegstreamer/media_source.c, dlls/winegstreamer/mfplat.c, dlls/winegstreamer/quartz_parser.c,
# | 	dlls/winegstreamer/wg_parser.c, dlls/winegstreamer/winegstreamer_classes.idl, include/mfidl.idl, include/wmcodecdsp.idl
# |
if test "$enable_mfplat_streaming_support" -eq 1; then
	patch_apply mfplat-streaming-support/0001-winegstreamer-Activate-source-pad-in-push-mode-if-it.patch
	patch_apply mfplat-streaming-support/0002-winegstreamer-Push-stream-start-and-segment-events-i.patch
	patch_apply mfplat-streaming-support/0003-winegstreamer-Introduce-H.264-decoder-transform.patch
	patch_apply mfplat-streaming-support/0004-winegstreamer-Implement-GetInputAvailableType-for-de.patch
	patch_apply mfplat-streaming-support/0005-winegstreamer-Implement-GetOutputAvailableType-for-d.patch
	patch_apply mfplat-streaming-support/0006-winegstreamer-Implement-SetInputType-for-decode-tran.patch
	patch_apply mfplat-streaming-support/0007-winegstreamer-Implement-SetOutputType-for-decode-tra.patch
	patch_apply mfplat-streaming-support/0008-winegstreamer-Implement-Get-Input-Output-StreamInfo-.patch
	patch_apply mfplat-streaming-support/0009-winegstreamer-Add-push-mode-path-for-wg_parser.patch
	patch_apply mfplat-streaming-support/0010-winegstreamer-Implement-Process-Input-Output-for-dec.patch
	patch_apply mfplat-streaming-support/0011-winestreamer-Implement-ProcessMessage-for-decoder-tr.patch
	patch_apply mfplat-streaming-support/0012-winegstreamer-Semi-stub-GetAttributes-for-decoder-tr.patch
	patch_apply mfplat-streaming-support/0013-winegstreamer-Register-the-H.264-decoder-transform.patch
	patch_apply mfplat-streaming-support/0014-winegstreamer-Introduce-AAC-decoder-transform.patch
	patch_apply mfplat-streaming-support/0015-winegstreamer-Register-the-AAC-decoder-transform.patch
	patch_apply mfplat-streaming-support/0016-winegstreamer-Rename-GStreamer-objects-to-be-more-ge.patch
	patch_apply mfplat-streaming-support/0017-winegstreamer-Report-streams-backwards-in-media-sour.patch
	patch_apply mfplat-streaming-support/0018-winegstreamer-Implement-Process-Input-Output-for-aud.patch
	patch_apply mfplat-streaming-support/0019-winegstreamer-Implement-Get-Input-Output-StreamInfo-.patch
	patch_apply mfplat-streaming-support/0020-winegstreamer-Semi-stub-Get-Attributes-functions-for.patch
	patch_apply mfplat-streaming-support/0021-winegstreamer-Introduce-color-conversion-transform.patch
	patch_apply mfplat-streaming-support/0022-winegstreamer-Register-the-color-conversion-transfor.patch
	patch_apply mfplat-streaming-support/0023-winegstreamer-Implement-GetInputAvailableType-for-co.patch
	patch_apply mfplat-streaming-support/0024-winegstreamer-Implement-SetInputType-for-color-conve.patch
	patch_apply mfplat-streaming-support/0025-winegstreamer-Implement-GetOutputAvailableType-for-c.patch
	patch_apply mfplat-streaming-support/0026-winegstreamer-Implement-SetOutputType-for-color-conv.patch
	patch_apply mfplat-streaming-support/0027-winegstreamer-Implement-Process-Input-Output-for-col.patch
	patch_apply mfplat-streaming-support/0028-winegstreamer-Implement-ProcessMessage-for-color-con.patch
	patch_apply mfplat-streaming-support/0029-winegstreamer-Implement-Get-Input-Output-StreamInfo-.patch
	patch_apply mfplat-streaming-support/0030-mf-topology-Forward-failure-from-SetOutputType-when-.patch
	patch_apply mfplat-streaming-support/0031-winegstreamer-Handle-flush-command-in-audio-converst.patch
	patch_apply mfplat-streaming-support/0032-winegstreamer-In-the-default-configuration-select-on.patch
	patch_apply mfplat-streaming-support/0033-winegstreamer-Implement-MF_SD_LANGUAGE.patch
	patch_apply mfplat-streaming-support/0034-winegstreamer-Only-require-videobox-element-for-pars.patch
	patch_apply mfplat-streaming-support/0035-winegstreamer-Don-t-rely-on-max_size-in-unseekable-p.patch
	patch_apply mfplat-streaming-support/0036-winegstreamer-Implement-MFT_MESSAGE_COMMAND_FLUSH-fo.patch
	patch_apply mfplat-streaming-support/0037-winegstreamer-Default-Frame-size-if-one-isn-t-availa.patch
	patch_apply mfplat-streaming-support/0038-mfplat-Stub-out-MFCreateDXGIDeviceManager-to-avoid-t.patch
fi

# Patchset mmsystem.dll16-MIDIHDR_Refcount
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#40024] Fix multiple issues in mmsystem.dll16 when translating MIDI messages
# |
# | Modified files:
# |   *	dlls/mmsystem.dll16/message16.c
# |
if test "$enable_mmsystem_dll16_MIDIHDR_Refcount" -eq 1; then
	patch_apply mmsystem.dll16-MIDIHDR_Refcount/0001-mmsystem.dll16-Refcount-midihdr-to-work-around-buggy.patch
	patch_apply mmsystem.dll16-MIDIHDR_Refcount/0002-mmsystem.dll16-Translate-MidiIn-messages.patch
fi

# Patchset mountmgr-DosDevices
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38235] Fix device paths in HKLM\SYSTEM\MountedDevices
# |
# | Modified files:
# |   *	dlls/mountmgr.sys/device.c, dlls/mountmgr.sys/mountmgr.c, dlls/mountmgr.sys/mountmgr.h
# |
if test "$enable_mountmgr_DosDevices" -eq 1; then
	patch_apply mountmgr-DosDevices/0001-mountmgr.sys-Write-usable-device-paths-into-HKLM-SYS.patch
fi

# Patchset mscoree-CorValidateImage
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38662] Implement mscoree._CorValidateImage for mono runtime
# |
# | Modified files:
# |   *	dlls/mscoree/mscoree_main.c
# |
if test "$enable_mscoree_CorValidateImage" -eq 1; then
	patch_apply mscoree-CorValidateImage/0001-mscoree-Implement-_CorValidateImage.patch
fi

# Patchset mshtml-HTMLLocation_put_hash
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#32967] Add IHTMLLocation::hash property's getter implementation
# |
# | Modified files:
# |   *	dlls/mshtml/htmllocation.c, dlls/mshtml/tests/htmldoc.c
# |
if test "$enable_mshtml_HTMLLocation_put_hash" -eq 1; then
	patch_apply mshtml-HTMLLocation_put_hash/0001-mshtml-Add-IHTMLLocation-hash-property-s-getter-impl.patch
fi

# Patchset mshtml-TranslateAccelerator
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37058] mshtml: Improve IOleInPlaceActiveObject TranslateAccelerator
# |
# | Modified files:
# |   *	dlls/mshtml/oleobj.c
# |
if test "$enable_mshtml_TranslateAccelerator" -eq 1; then
	patch_apply mshtml-TranslateAccelerator/0001-mshtml-Improve-IOleInPlaceActiveObject-TranslateAcce.patch
fi

# Patchset msi-msi_vcl_get_cost
# |
# | Modified files:
# |   *	dlls/msi/dialog.c
# |
if test "$enable_msi_msi_vcl_get_cost" -eq 1; then
	patch_apply msi-msi_vcl_get_cost/0001-msi-Do-not-sign-extend-after-multiplying.patch
fi

# Patchset msxml3-FreeThreadedXMLHTTP60
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50900] msxml3: Implement FreeThreadedXMLHTTP60.
# |
# | Modified files:
# |   *	dlls/msxml3/Makefile.in, dlls/msxml3/factory.c, dlls/msxml3/httprequest.c, dlls/msxml3/msxml_private.h,
# | 	dlls/msxml3/tests/httpreq.c, dlls/msxml3/tests/saxreader.c, dlls/msxml3/tests/schema.c, dlls/msxml3/uuid.c,
# | 	include/msxml2.idl, include/msxml6.idl
# |
if test "$enable_msxml3_FreeThreadedXMLHTTP60" -eq 1; then
	patch_apply msxml3-FreeThreadedXMLHTTP60/0001-include-Remove-interfaces-already-define-in-msxml6.i.patch
	patch_apply msxml3-FreeThreadedXMLHTTP60/0003-msxml3-Implement-FreeThreadedXMLHTTP60.patch
fi

# Patchset ntdll-APC_Performance
# |
# | Modified files:
# |   *	dlls/ntdll/unix/file.c, dlls/ntdll/unix/unix_private.h
# |
if test "$enable_ntdll_APC_Performance" -eq 1; then
	patch_apply ntdll-APC_Performance/0001-ntdll-Reuse-old-async-fileio-structures-if-possible.patch
fi

# Patchset ntdll-ApiSetMap
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44658] Add dummy apiset to PEB struct
# |
# | Modified files:
# |   *	dlls/ntdll/loader.c, include/Makefile.in, include/apiset.h, include/winternl.h
# |
if test "$enable_ntdll_ApiSetMap" -eq 1; then
	patch_apply ntdll-ApiSetMap/0001-ntdll-Add-dummy-apiset-to-PEB.patch
fi

# Patchset ntdll-ForceBottomUpAlloc
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#48175] AION (64 bit) - crashes in crysystem.dll.CryFree() due to high memory pointers allocated
# |   *	[#46568] 64-bit msxml6.dll from Microsoft Core XML Services 6.0 redist package fails to load (Wine doesn't respect
# | 	44-bit user-mode VA limitation from Windows < 8.1)
# |
# | Modified files:
# |   *	dlls/ntdll/unix/virtual.c
# |
if test "$enable_ntdll_ForceBottomUpAlloc" -eq 1; then
	patch_apply ntdll-ForceBottomUpAlloc/0001-ntdll-Increase-step-after-failed-map-attempt-in-try_.patch
	patch_apply ntdll-ForceBottomUpAlloc/0002-ntdll-Increase-free-ranges-view-block-size-on-64-bit.patch
	patch_apply ntdll-ForceBottomUpAlloc/0003-ntdll-Force-virtual-memory-allocation-order.patch
	patch_apply ntdll-ForceBottomUpAlloc/0004-ntdll-Exclude-natively-mapped-areas-from-free-areas-.patch
fi

# Patchset ntdll-WRITECOPY
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-ForceBottomUpAlloc
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29384] Multiple applications expect correct handling of WRITECOPY memory protection (Voobly fails to launch Age of
# | 	Empires II, MSYS2)
# |
# | Modified files:
# |   *	dlls/kernel32/tests/virtual.c, dlls/ntdll/unix/loader.c, dlls/ntdll/unix/server.c, dlls/ntdll/unix/signal_arm.c,
# | 	dlls/ntdll/unix/signal_arm64.c, dlls/ntdll/unix/signal_i386.c, dlls/ntdll/unix/signal_x86_64.c,
# | 	dlls/ntdll/unix/unix_private.h, dlls/ntdll/unix/virtual.c, dlls/psapi/tests/psapi_main.c
# |
if test "$enable_ntdll_WRITECOPY" -eq 1; then
	patch_apply ntdll-WRITECOPY/0001-ntdll-Trigger-write-watches-before-passing-userdata-.patch
	patch_apply ntdll-WRITECOPY/0003-ntdll-Setup-a-temporary-signal-handler-during-proces.patch
	patch_apply ntdll-WRITECOPY/0004-ntdll-Properly-handle-PAGE_WRITECOPY-protection.-try.patch
	patch_apply ntdll-WRITECOPY/0005-ntdll-Track-if-a-WRITECOPY-page-has-been-modified.patch
	patch_apply ntdll-WRITECOPY/0006-ntdll-Support-WRITECOPY-on-x64.patch
	patch_apply ntdll-WRITECOPY/0007-ntdll-Report-unmodified-WRITECOPY-pages-as-shared.patch
	patch_apply ntdll-WRITECOPY/0008-ntdll-Fallback-to-copy-pages-for-WRITECOPY.patch
	patch_apply ntdll-WRITECOPY/0009-kernel32-tests-psapi-tests-Update-tests.patch
fi

# Patchset ntdll-Builtin_Prot
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-ForceBottomUpAlloc, ntdll-WRITECOPY
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44650] Fix holes in ELF mappings
# |
# | Modified files:
# |   *	dlls/ntdll/unix/virtual.c, dlls/psapi/tests/psapi_main.c
# |
if test "$enable_ntdll_Builtin_Prot" -eq 1; then
	patch_apply ntdll-Builtin_Prot/0001-ntdll-Fix-holes-in-ELF-mappings.patch
fi

# Patchset ntdll-CriticalSection
# |
# | Modified files:
# |   *	dlls/ntdll/heap.c, dlls/ntdll/ntdll_misc.h, dlls/ntdll/threadpool.c
# |
if test "$enable_ntdll_CriticalSection" -eq 1; then
	patch_apply ntdll-CriticalSection/0002-ntdll-Add-inline-versions-of-RtlEnterCriticalSection.patch
	patch_apply ntdll-CriticalSection/0003-ntdll-Use-fast-CS-functions-for-heap-locking.patch
	patch_apply ntdll-CriticalSection/0004-ntdll-Use-fast-CS-functions-for-threadpool-locking.patch
fi

# Patchset ntdll-Exception
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44819] Throw second DBG_PRINTEXCEPTION_C when debugging.
# |
# | Modified files:
# |   *	dlls/kernelbase/debug.c, dlls/ntdll/tests/exception.c
# |
if test "$enable_ntdll_Exception" -eq 1; then
	patch_apply ntdll-Exception/0002-ntdll-OutputDebugString-should-throw-the-exception-a.patch
fi

# Patchset ntdll-FileFsFullSizeInformation
# |
# | Modified files:
# |   *	dlls/ntdll/tests/file.c, dlls/ntdll/unix/file.c
# |
if test "$enable_ntdll_FileFsFullSizeInformation" -eq 1; then
	patch_apply ntdll-FileFsFullSizeInformation/0001-ntdll-Add-support-for-FileFsFullSizeInformation-clas.patch
fi

# Patchset ntdll-HashLinks
# |
# | Modified files:
# |   *	dlls/kernel32/tests/loader.c, dlls/ntdll/loader.c, include/winternl.h
# |
if test "$enable_ntdll_HashLinks" -eq 1; then
	patch_apply ntdll-HashLinks/0001-ntdll-Implement-HashLinks-field-in-LDR-module-data.patch
	patch_apply ntdll-HashLinks/0002-ntdll-Use-HashLinks-when-searching-for-a-dll-using-t.patch
fi

# Patchset ntdll-Heap_Improvements
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#43224] Improvement for heap allocation performance
# |
# | Modified files:
# |   *	dlls/ntdll/heap.c
# |
if test "$enable_ntdll_Heap_Improvements" -eq 1; then
	patch_apply ntdll-Heap_Improvements/0001-ntdll-Add-helper-function-to-delete-free-blocks.patch
	patch_apply ntdll-Heap_Improvements/0002-ntdll-Improve-heap-allocation-performance.patch
fi

# Patchset ntdll-Hide_Wine_Exports
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38656] Add support for hiding wine version information from applications
# |
# | Modified files:
# |   *	dlls/ntdll/loader.c, dlls/ntdll/ntdll_misc.h
# |
if test "$enable_ntdll_Hide_Wine_Exports" -eq 1; then
	patch_apply ntdll-Hide_Wine_Exports/0001-ntdll-Add-support-for-hiding-wine-version-informatio.patch
fi

# Patchset ntdll-Manifest_Range
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#18889] ntdll: Support ISOLATIONAWARE_MANIFEST_RESOURCE_ID range
# |
# | Modified files:
# |   *	dlls/ntdll/actctx.c
# |
if test "$enable_ntdll_Manifest_Range" -eq 1; then
	patch_apply ntdll-Manifest_Range/0001-ntdll-Support-ISOLATIONAWARE_MANIFEST_RESOURCE_ID-ra.patch
fi

# Patchset ntdll-NtAlertThreadByThreadId
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50292] Process-local synchronization objects use private interfaces into the Unix library
# |
# | Modified files:
# |   *	dlls/ntdll/Makefile.in, dlls/ntdll/critsection.c, dlls/ntdll/ntdll.spec, dlls/ntdll/sync.c,
# | 	dlls/ntdll/tests/Makefile.in, dlls/ntdll/tests/om.c, dlls/ntdll/tests/sync.c, dlls/ntdll/unix/loader.c,
# | 	dlls/ntdll/unix/sync.c, dlls/ntdll/unix/unix_private.h, dlls/ntdll/unixlib.h, dlls/wow64/sync.c, dlls/wow64/syscall.h,
# | 	include/winternl.h
# |
if test "$enable_ntdll_NtAlertThreadByThreadId" -eq 1; then
	patch_apply ntdll-NtAlertThreadByThreadId/0001-ntdll-tests-Move-some-tests-to-a-new-sync.c-file.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0002-ntdll-tests-Add-some-tests-for-Rtl-resources.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0003-ntdll-Implement-NtAlertThreadByThreadId-and-NtWaitFo.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0004-ntdll-tests-Add-basic-tests-for-thread-id-alert-func.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0005-ntdll-Implement-thread-ID-alerts-using-futexes-if-po.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0006-ntdll-Implement-thread-ID-alerts-using-Mach-semaphor.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0007-ntdll-Reimplement-Win32-futexes-on-top-of-thread-ID-.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0008-ntdll-Merge-critsection.c-into-sync.c.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0009-ntdll-Reimplement-the-critical-section-fast-path-on-.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0010-ntdll-Get-rid-of-the-direct-futex-path-for-condition.patch
	patch_apply ntdll-NtAlertThreadByThreadId/0011-ntdll-Reimplement-SRW-locks-on-top-of-Win32-futexes.patch
fi

# Patchset ntdll-NtQuerySection
# |
# | Modified files:
# |   *	dlls/kernel32/tests/virtual.c
# |
if test "$enable_ntdll_NtQuerySection" -eq 1; then
	patch_apply ntdll-NtQuerySection/0002-kernel32-tests-Add-tests-for-NtQuerySection.patch
fi

# Patchset ntdll-NtSetLdtEntries
# |
# | Modified files:
# |   *	dlls/kernel32/tests/thread.c, dlls/ntdll/unix/signal_i386.c, libs/wine/ldt.c
# |
if test "$enable_ntdll_NtSetLdtEntries" -eq 1; then
	patch_apply ntdll-NtSetLdtEntries/0001-ntdll-Implement-NtSetLdtEntries.patch
	patch_apply ntdll-NtSetLdtEntries/0002-libs-wine-Allow-to-modify-reserved-LDT-entries.patch
fi

# Patchset ntdll-ProcessQuotaLimits
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44812] Multiple applications need NtQueryInformationProcess 'ProcessQuotaLimits' class support (MSYS2, ProcessHacker
# | 	2.x)
# |
# | Modified files:
# |   *	dlls/ntdll/unix/process.c
# |
if test "$enable_ntdll_ProcessQuotaLimits" -eq 1; then
	patch_apply ntdll-ProcessQuotaLimits/0001-ntdll-Add-fake-data-implementation-for-ProcessQuotaL.patch
fi

# Patchset ntdll-RtlFirstFreeAce
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50624] Waves Central 12.0.5 fails to start.
# |
# | Modified files:
# |   *	dlls/ntdll/sec.c
# |
if test "$enable_ntdll_RtlFirstFreeAce" -eq 1; then
	patch_apply ntdll-RtlFirstFreeAce/0001-ntdll-Check-return-parameter-before-use.patch
	patch_apply ntdll-RtlFirstFreeAce/0002-ntdll-RtlFirstFreeAce-only-return-FALSE-on-error.patch
fi

# Patchset ntdll-RtlQueryPackageIdentity
# |
# | Modified files:
# |   *	dlls/ntdll/tests/Makefile.in, dlls/ntdll/tests/rtl.c
# |
if test "$enable_ntdll_RtlQueryPackageIdentity" -eq 1; then
	patch_apply ntdll-RtlQueryPackageIdentity/0003-ntdll-tests-Add-basic-tests-for-RtlQueryPackageIdent.patch
fi

# Patchset ntdll-Serial_Port_Detection
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39793] Do a device check before returning a default serial port name
# |
# | Modified files:
# |   *	dlls/mountmgr.sys/device.c
# |
if test "$enable_ntdll_Serial_Port_Detection" -eq 1; then
	patch_apply ntdll-Serial_Port_Detection/0001-ntdll-Do-a-device-check-before-returning-a-default-s.patch
fi

# Patchset ntdll-Syscall_Emulation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#48291] Detroit: Become Human crashes on launch
# |
# | Modified files:
# |   *	configure.ac, dlls/ntdll/unix/signal_x86_64.c, tools/winebuild/import.c
# |
if test "$enable_ntdll_Syscall_Emulation" -eq 1; then
	patch_apply ntdll-Syscall_Emulation/0001-ntdll-Support-x86_64-syscall-emulation.patch
fi

# Patchset ntdll-Zero_mod_name
# |
# | Modified files:
# |   *	dlls/ntdll/loader.c
# |
if test "$enable_ntdll_Zero_mod_name" -eq 1; then
	patch_apply ntdll-Zero_mod_name/0001-ntdll-Initialize-mod_name-to-zero.patch
fi

# Patchset ntdll-ext4-case-folder
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47099] Support for EXT4 case folding per directory.
# |
# | Modified files:
# |   *	dlls/ntdll/unix/server.c
# |
if test "$enable_ntdll_ext4_case_folder" -eq 1; then
	patch_apply ntdll-ext4-case-folder/0002-ntdll-server-Mark-drive_c-as-case-insensitive-when-c.patch
fi

# Patchset ntoskrnl-Stubs
# |
# | Modified files:
# |   *	dlls/ntoskrnl.exe/ntoskrnl.c, dlls/ntoskrnl.exe/ntoskrnl.exe.spec
# |
if test "$enable_ntoskrnl_Stubs" -eq 1; then
	patch_apply ntoskrnl-Stubs/0009-ntoskrnl.exe-Implement-MmMapLockedPages-and-MmUnmapL.patch
	patch_apply ntoskrnl-Stubs/0011-ntoskrnl.exe-Add-IoGetDeviceAttachmentBaseRef-stub.patch
fi

# Patchset nvcuda-CUDA_Support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37664] MediaCoder needs CUDA for video encoding
# |   *	[#51523] Add cuD3D11GetDevice and cuGraphicsD3D11RegisterResource stubs
# |
# | Modified files:
# |   *	configure.ac, dlls/nvcuda/Makefile.in, dlls/nvcuda/internal.c, dlls/nvcuda/nvcuda.c, dlls/nvcuda/nvcuda.h,
# | 	dlls/nvcuda/nvcuda.rc, dlls/nvcuda/nvcuda.spec, dlls/nvcuda/tests/Makefile.in, dlls/nvcuda/tests/nvcuda.c,
# | 	include/Makefile.in, include/cuda.h
# |
if test "$enable_nvcuda_CUDA_Support" -eq 1; then
	patch_apply nvcuda-CUDA_Support/0001-include-Add-cuda.h.h.patch
	patch_apply nvcuda-CUDA_Support/0002-nvcuda-Add-stub-dll.patch
	patch_apply nvcuda-CUDA_Support/0003-nvcuda-First-implementation.patch
	patch_apply nvcuda-CUDA_Support/0004-nvcuda-Implement-new-functions-added-in-CUDA-6.5.patch
	patch_apply nvcuda-CUDA_Support/0005-nvcuda-Properly-wrap-undocumented-ContextStorage-int.patch
	patch_apply nvcuda-CUDA_Support/0006-nvcuda-Emulate-two-d3d9-initialization-functions.patch
	patch_apply nvcuda-CUDA_Support/0007-nvcuda-Properly-wrap-stream-callbacks-by-forwarding-.patch
	patch_apply nvcuda-CUDA_Support/0008-nvcuda-Add-support-for-CUDA-7.0.patch
	patch_apply nvcuda-CUDA_Support/0009-nvcuda-Implement-cuModuleLoad-wrapper-function.patch
	patch_apply nvcuda-CUDA_Support/0010-nvcuda-Search-for-dylib-library-on-Mac-OS-X.patch
	patch_apply nvcuda-CUDA_Support/0011-nvcuda-Add-semi-stub-for-cuD3D10GetDevice.patch
	patch_apply nvcuda-CUDA_Support/0012-nvcuda-Add-semi-stub-for-cuD3D11GetDevice-and-cuGrap.patch
fi

# Patchset nvapi-Stub_DLL
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	nvcuda-CUDA_Support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35062] Fix graphical corruption in FarCry 3 with NVIDIA drivers
# |   *	[#43862] CS:GO fails to start when nvapi cannot be initialized
# |
# | Modified files:
# |   *	configure.ac, dlls/d3d11/d3d11_private.h, dlls/d3d11/device.c, dlls/nvapi/Makefile.in, dlls/nvapi/nvapi.c,
# | 	dlls/nvapi/nvapi.spec, dlls/nvapi/tests/Makefile.in, dlls/nvapi/tests/nvapi.c, dlls/nvapi64/Makefile.in,
# | 	dlls/nvapi64/nvapi64.spec, dlls/wined3d/cs.c, dlls/wined3d/device.c, dlls/wined3d/state.c, dlls/wined3d/utils.c,
# | 	dlls/wined3d/wined3d.spec, dlls/wined3d/wined3d_private.h, include/Makefile.in, include/nvapi.h, include/wine/wined3d.h,
# | 	include/wine/winedxgi.idl
# |
if test "$enable_nvapi_Stub_DLL" -eq 1; then
	patch_apply nvapi-Stub_DLL/0001-nvapi-First-implementation.patch
	patch_apply nvapi-Stub_DLL/0002-nvapi-Add-stubs-for-NvAPI_EnumLogicalGPUs-and-undocu.patch
	patch_apply nvapi-Stub_DLL/0003-nvapi-Add-NvAPI_GetPhysicalGPUsFromLogicalGPU.patch
	patch_apply nvapi-Stub_DLL/0004-nvapi-Add-stub-for-NvAPI_EnumPhysicalGPUs.patch
	patch_apply nvapi-Stub_DLL/0005-nvapi-Add-stubs-for-NvAPI_GPU_GetFullName.patch
	patch_apply nvapi-Stub_DLL/0006-nvapi-Explicity-return-NULL-for-0x33c7358c-and-0x593.patch
	patch_apply nvapi-Stub_DLL/0007-nvapi-Add-stub-for-NvAPI_DISP_GetGDIPrimaryDisplayId.patch
	patch_apply nvapi-Stub_DLL/0008-nvapi-Add-stub-for-EnumNvidiaDisplayHandle.patch
	patch_apply nvapi-Stub_DLL/0009-nvapi-Add-stub-for-NvAPI_SYS_GetDriverAndBranchVersi.patch
	patch_apply nvapi-Stub_DLL/0010-nvapi-Add-stub-for-NvAPI_Unload.patch
	patch_apply nvapi-Stub_DLL/0011-nvapi-Add-stub-for-NvAPI_D3D_GetCurrentSLIState.patch
	patch_apply nvapi-Stub_DLL/0012-nvapi-tests-Use-structure-to-list-imports.patch
	patch_apply nvapi-Stub_DLL/0013-nvapi-Add-stub-for-NvAPI_GetLogicalGPUFromDisplay.patch
	patch_apply nvapi-Stub_DLL/0014-nvapi-Add-stub-for-NvAPI_D3D_GetObjectHandleForResou.patch
	patch_apply nvapi-Stub_DLL/0015-nvapi-Add-stub-for-NvAPI_D3D9_RegisterResource.patch
	patch_apply nvapi-Stub_DLL/0016-nvapi-Improve-NvAPI_D3D_GetCurrentSLIState.patch
	patch_apply nvapi-Stub_DLL/0017-nvapi-Implement-NvAPI_GPU_Get-Physical-Virtual-Frame.patch
	patch_apply nvapi-Stub_DLL/0018-nvapi-Add-stub-for-NvAPI_GPU_GetGpuCoreCount.patch
	patch_apply nvapi-Stub_DLL/0019-wined3d-Make-depth-bounds-test-into-a-proper-state.patch
	patch_apply nvapi-Stub_DLL/0020-d3d11-Introduce-a-COM-interface-to-retrieve-the-wine.patch
	patch_apply nvapi-Stub_DLL/0021-nvapi-Implement-NvAPI_D3D11_SetDepthBoundsTest.patch
	patch_apply nvapi-Stub_DLL/0022-nvapi-Implement-NvAPI_D3D11_CreateDevice-and-NvAPI_D.patch
fi

# Patchset nvcuvid-CUDA_Video_Support
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	nvcuda-CUDA_Support, nvapi-Stub_DLL
# |
# | Modified files:
# |   *	configure.ac, dlls/nvcuvid/Makefile.in, dlls/nvcuvid/nvcuvid.c, dlls/nvcuvid/nvcuvid.spec, include/Makefile.in,
# | 	include/cuviddec.h, include/nvcuvid.h
# |
if test "$enable_nvcuvid_CUDA_Video_Support" -eq 1; then
	patch_apply nvcuvid-CUDA_Video_Support/0001-nvcuvid-First-implementation.patch
fi

# Patchset nvencodeapi-Video_Encoder
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	nvcuda-CUDA_Support, nvapi-Stub_DLL, nvcuvid-CUDA_Video_Support
# |
# | Modified files:
# |   *	configure.ac, dlls/nvencodeapi/Makefile.in, dlls/nvencodeapi/nvencodeapi.c, dlls/nvencodeapi/nvencodeapi.spec,
# | 	dlls/nvencodeapi64/Makefile.in, dlls/nvencodeapi64/nvencodeapi64.spec, include/Makefile.in, include/nvencodeapi.h
# |
if test "$enable_nvencodeapi_Video_Encoder" -eq 1; then
	patch_apply nvencodeapi-Video_Encoder/0001-nvencodeapi-First-implementation.patch
	patch_apply nvencodeapi-Video_Encoder/0002-nvencodeapi-Add-debian-specific-paths-to-native-libr.patch
	patch_apply nvencodeapi-Video_Encoder/0003-nvencodeapi-Add-support-for-version-6.0.patch
fi

# Patchset oleaut32-CreateTypeLib
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8780] Forward CreateTypeLib to CreateTypeLib2
# |
# | Modified files:
# |   *	dlls/oleaut32/typelib.c
# |
if test "$enable_oleaut32_CreateTypeLib" -eq 1; then
	patch_apply oleaut32-CreateTypeLib/0001-oleaut32-Implement-semi-stub-for-CreateTypeLib.patch
fi

# Patchset oleaut32-Load_Save_EMF
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#40523] Implement support for loading and saving EMF to IPicture interface
# |
# | Modified files:
# |   *	dlls/oleaut32/olepicture.c, dlls/oleaut32/tests/olepicture.c
# |
if test "$enable_oleaut32_Load_Save_EMF" -eq 1; then
	patch_apply oleaut32-Load_Save_EMF/0001-oleaut32-tests-Add-some-tests-for-loading-and-saving.patch
	patch_apply oleaut32-Load_Save_EMF/0002-oleaut32-Add-support-for-loading-and-saving-EMF-to-I.patch
fi

# Patchset oleaut32-OLEPictureImpl_SaveAsFile
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	oleaut32-Load_Save_EMF
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8532] Implement a better stub for IPicture::SaveAsFile
# |
# | Modified files:
# |   *	dlls/oleaut32/olepicture.c, dlls/oleaut32/tests/olepicture.c
# |
if test "$enable_oleaut32_OLEPictureImpl_SaveAsFile" -eq 1; then
	patch_apply oleaut32-OLEPictureImpl_SaveAsFile/0002-oleaut32-Implement-a-better-stub-for-IPicture-SaveAs.patch
	patch_apply oleaut32-OLEPictureImpl_SaveAsFile/0003-oleaut32-Implement-SaveAsFile-for-PICTYPE_ENHMETAFIL.patch
fi

# Patchset oleaut32-OleLoadPicture
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39474] Create DIB section in OleLoadPicture
# |
# | Modified files:
# |   *	dlls/oleaut32/olepicture.c, dlls/oleaut32/tests/olepicture.c
# |
if test "$enable_oleaut32_OleLoadPicture" -eq 1; then
	patch_apply oleaut32-OleLoadPicture/0001-oleaut32-OleLoadPicture-should-create-a-DIB-section-.patch
	patch_apply oleaut32-OleLoadPicture/0002-oleaut32-Make-OleLoadPicture-load-DIBs-using-WIC-dec.patch
fi

# Patchset oleaut32-OleLoadPictureFile
# |
# | Modified files:
# |   *	dlls/oleaut32/olepicture.c, dlls/oleaut32/tests/olepicture.c
# |
if test "$enable_oleaut32_OleLoadPictureFile" -eq 1; then
	patch_apply oleaut32-OleLoadPictureFile/0001-oleaut32-Do-not-reimplement-OleLoadPicture-in-OleLoa.patch
	patch_apply oleaut32-OleLoadPictureFile/0002-oleaut32-Factor-out-stream-creation-from-OleLoadPict.patch
	patch_apply oleaut32-OleLoadPictureFile/0003-oleaut32-Implement-OleLoadPictureFile.patch
fi

# Patchset packager-DllMain
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#43472] Prefer native version of packager.dll
# |
# | Modified files:
# |   *	dlls/packager/Makefile.in
# |
if test "$enable_packager_DllMain" -eq 1; then
	patch_apply packager-DllMain/0001-packager-Prefer-native-version.patch
fi

# Patchset programs-findstr
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35254] findstr: Add minimal implementation
# |
# | Modified files:
# |   *	programs/findstr/Makefile.in, programs/findstr/findstr.rc, programs/findstr/main.c, programs/findstr/resources.h
# |
if test "$enable_programs_findstr" -eq 1; then
	patch_apply programs-findstr/0001-findstr-add-basic-functionality-also-support-literal.patch
fi

# Patchset programs-systeminfo
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42027] systeminfo: Add basic functionality.
# |
# | Modified files:
# |   *	programs/systeminfo/Makefile.in, programs/systeminfo/main.c
# |
if test "$enable_programs_systeminfo" -eq 1; then
	patch_apply programs-systeminfo/0001-systeminfo-add-basic-functionality.patch
fi

# Patchset riched20-Class_Tests
# |
# | Modified files:
# |   *	dlls/riched20/tests/editor.c
# |
if test "$enable_riched20_Class_Tests" -eq 1; then
	patch_apply riched20-Class_Tests/0001-riched20-tests-Add-a-test-to-see-what-richedit-class.patch
fi

# Patchset riched20-IText_Interface
# |
# | Modified files:
# |   *	dlls/riched20/richole.c, dlls/riched20/tests/richole.c
# |
if test "$enable_riched20_IText_Interface" -eq 1; then
	patch_apply riched20-IText_Interface/0003-riched20-Stub-for-ITextPara-interface-and-implement-.patch
	patch_apply riched20-IText_Interface/0010-riched20-Silence-repeated-FIXMEs-triggered-by-Adobe-.patch
fi

# Patchset secur32-InitializeSecurityContextW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#51049] Create a new Context when the input object is NULL.
# |
# | Modified files:
# |   *	dlls/secur32/schannel.c, dlls/secur32/tests/schannel.c
# |
if test "$enable_secur32_InitializeSecurityContextW" -eq 1; then
	patch_apply secur32-InitializeSecurityContextW/0001-secur32-Input-Parameter-should-be-NULL-on-first-call.patch
fi

# Patchset server-File_Permissions
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-DOS_Attributes, ntdll-NtQueryEaFile, ntdll-Junction_Points
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, dlls/ntdll/tests/file.c, server/fd.c
# |
if test "$enable_server_File_Permissions" -eq 1; then
	patch_apply server-File_Permissions/0001-server-Improve-STATUS_CANNOT_DELETE-checks-for-direc.patch
	patch_apply server-File_Permissions/0002-server-Allow-to-open-files-without-any-permission-bi.patch
	patch_apply server-File_Permissions/0003-server-When-creating-new-directories-temporarily-giv.patch
	patch_apply server-File_Permissions/0004-advapi32-tests-Add-tests-for-ACL-inheritance-in-Crea.patch
	patch_apply server-File_Permissions/0005-advapi32-tests-Add-ACL-inheritance-tests-for-creatin.patch
	patch_apply server-File_Permissions/0006-ntdll-tests-Added-tests-for-open-behaviour-on-readon.patch
	patch_apply server-File_Permissions/0007-server-FILE_WRITE_ATTRIBUTES-should-succeed-for-read.patch
fi

# Patchset server-Stored_ACLs
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntdll-DOS_Attributes, ntdll-NtQueryEaFile, ntdll-Junction_Points, server-File_Permissions
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33576] Support for stored file ACLs
# |
# | Modified files:
# |   *	dlls/advapi32/tests/security.c, server/change.c, server/file.c, server/file.h, server/object.c, server/object.h
# |
if test "$enable_server_Stored_ACLs" -eq 1; then
	patch_apply server-Stored_ACLs/0001-server-Unify-the-storage-of-security-attributes-for-.patch
	patch_apply server-Stored_ACLs/0002-server-Unify-the-retrieval-of-security-attributes-fo.patch
	patch_apply server-Stored_ACLs/0003-server-Add-a-helper-function-set_sd_from_token_inter.patch
	patch_apply server-Stored_ACLs/0004-server-Temporarily-store-the-full-security-descripto.patch
	patch_apply server-Stored_ACLs/0005-server-Store-file-security-attributes-with-extended-.patch
	patch_apply server-Stored_ACLs/0006-server-Convert-return-of-file-security-masks-with-ge.patch
	patch_apply server-Stored_ACLs/0007-server-Retrieve-file-security-attributes-with-extend.patch
fi

# Patchset server-default_integrity
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#40613] Multiple applications require UAC implementation to run installer/app as a normal user instead of administrator
# | 	(WhatsApp Desktop, Smartflix, Squirrel Installers, OneDrive)
# |   *	[#39262] DiscordSetup.exe (.NET 4.5.2 app): Squirrell installer requires being run as unelevated process ('explorer.exe'
# | 	should run unelevated by default with Vista+ setting)
# |
# | Modified files:
# |   *	dlls/kernelbase/process.c, dlls/msi/custom.c, dlls/ntdll/process.c, dlls/ntdll/unix/env.c, dlls/shell32/shlexec.c,
# | 	loader/wine.inf.in, server/process.c
# |
if test "$enable_server_default_integrity" -eq 1; then
	patch_apply server-default_integrity/0001-server-Create-processes-using-a-limited-administrato.patch
	patch_apply server-default_integrity/0002-shell32-Implement-the-runas-verb.patch
	patch_apply server-default_integrity/0003-wine.inf-Set-the-EnableLUA-value-to-1.patch
	patch_apply server-default_integrity/0004-msi-Create-the-custom-action-server-as-an-elevated-p.patch
	patch_apply server-default_integrity/0005-ntdll-Always-start-the-initial-process-through-start.patch
	patch_apply server-default_integrity/0006-kernelbase-Elevate-processes-if-requested-in-CreateP.patch
	patch_apply server-default_integrity/0007-ntdll-Elevate-processes-if-requested-in-RtlCreateUse.patch
fi

# Patchset setupapi-DiskSpaceList
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50337] Roland Zenology Pro (VST3 plugin) used with carla-bridge fails to save files
# |
# | Modified files:
# |   *	dlls/setupapi/diskspace.c, dlls/setupapi/queue.c, dlls/setupapi/setupapi.spec, dlls/setupapi/setupapi_private.h,
# | 	dlls/setupapi/stubs.c, dlls/setupapi/tests/diskspace.c, include/setupapi.h
# |
if test "$enable_setupapi_DiskSpaceList" -eq 1; then
	patch_apply setupapi-DiskSpaceList/0001-setupapi-Rewrite-DiskSpaceList-logic-using-lists.patch
	patch_apply setupapi-DiskSpaceList/0002-setupapi-Implement-SetupAddToDiskSpaceList.patch
	patch_apply setupapi-DiskSpaceList/0003-setupapi-Implement-SetupQueryDrivesInDiskSpaceList.patch
	patch_apply setupapi-DiskSpaceList/0004-setupapi-Ignore-deletion-of-added-files-in-SetupAddT.patch
	patch_apply setupapi-DiskSpaceList/0005-setupapi-ImplementSetupAddSectionToDiskSpaceList.patch
	patch_apply setupapi-DiskSpaceList/0006-setupapi-Implement-SetupAddInstallSectionToDiskSpace.patch
fi

# Patchset shdocvw-ParseURLFromOutsideSource_Tests
# |
# | Modified files:
# |   *	dlls/shdocvw/shdocvw_main.c, dlls/shdocvw/tests/shdocvw.c
# |
if test "$enable_shdocvw_ParseURLFromOutsideSource_Tests" -eq 1; then
	patch_apply shdocvw-ParseURLFromOutsideSource_Tests/0001-shdocvw-Check-precisely-ParseURLFromOutsideSourceX-r.patch
fi

# Patchset shell32-SHFileOperation_Move
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#25207] SHFileOperation with FO_MOVE should create new directory on Vista+
# |
# | Modified files:
# |   *	dlls/shell32/shlfileop.c, dlls/shell32/tests/shlfileop.c
# |
if test "$enable_shell32_SHFileOperation_Move" -eq 1; then
	patch_apply shell32-SHFileOperation_Move/0001-shell32-Fix-SHFileOperation-FO_MOVE-for-creating-sub.patch
fi

# Patchset shell32-Progress_Dialog
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	kernel32-CopyFileEx, shell32-SHFileOperation_Move
# |
# | Modified files:
# |   *	dlls/shell32/shell32.rc, dlls/shell32/shlfileop.c, dlls/shell32/shresdef.h
# |
if test "$enable_shell32_Progress_Dialog" -eq 1; then
	patch_apply shell32-Progress_Dialog/0001-shell32-Correct-indentation-in-shfileop.c.patch
	patch_apply shell32-Progress_Dialog/0002-shell32-Pass-FILE_INFORMATION-into-SHNotify-function.patch
	patch_apply shell32-Progress_Dialog/0003-shell32-Implement-file-operation-progress-dialog.patch
	patch_apply shell32-Progress_Dialog/0004-shell32-Show-animation-during-SHFileOperation.patch
fi

# Patchset shell32-ACE_Viewer
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	kernel32-CopyFileEx, shell32-SHFileOperation_Move, shell32-Progress_Dialog
# |
# | Modified files:
# |   *	dlls/shell32/Makefile.in, dlls/shell32/shell32.rc, dlls/shell32/shlview_cmenu.c, dlls/shell32/shresdef.h
# |
if test "$enable_shell32_ACE_Viewer" -eq 1; then
	patch_apply shell32-ACE_Viewer/0002-shell32-Add-security-property-tab.patch
fi

# Patchset shell32-Context_Menu
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34319] Add support for Paste in context menu
# |   *	[#34322] Fix implementation of Cut file operation
# |   *	[#34321] Fix Cut/Copy/Paste keyboard shortcuts in Total Commander
# |
# | Modified files:
# |   *	dlls/shell32/brsfolder.c, dlls/shell32/clipboard.c, dlls/shell32/dataobject.c, dlls/shell32/recyclebin.c,
# | 	dlls/shell32/shell32.rc, dlls/shell32/shell32_main.h, dlls/shell32/shellfolder.h, dlls/shell32/shfldr_fs.c,
# | 	dlls/shell32/shfldr_unixfs.c, dlls/shell32/shlview.c, dlls/shell32/shlview_cmenu.c
# |
if test "$enable_shell32_Context_Menu" -eq 1; then
	patch_apply shell32-Context_Menu/0003-shell32-Implement-insert-paste-for-item-context-menu.patch
	patch_apply shell32-Context_Menu/0005-shell32-Add-support-for-setting-getting-PREFERREDDRO.patch
	patch_apply shell32-Context_Menu/0006-shell32-Add-parameter-to-ISFHelper-DeleteItems-to-al.patch
	patch_apply shell32-Context_Menu/0007-shell32-Remove-source-files-when-using-cut-in-the-co.patch
	patch_apply shell32-Context_Menu/0008-shell32-Recognize-cut-copy-paste-string-verbs-in-ite.patch
fi

# Patchset shell32-IconCache
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45696] shell32: Generate icons from available icons if required.
# |
# | Modified files:
# |   *	dlls/shell32/iconcache.c
# |
if test "$enable_shell32_IconCache" -eq 1; then
	patch_apply shell32-IconCache/0001-shell32-iconcache-Generate-icons-from-available-icons-.patch
fi

# Patchset shell32-NewMenu_Interface
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#24812] Implement shell32 NewMenu class with new folder item
# |
# | Modified files:
# |   *	dlls/shell32/Makefile.in, dlls/shell32/shell32_classes.idl, dlls/shell32/shell32_main.h, dlls/shell32/shellnew.c,
# | 	dlls/shell32/shellole.c, dlls/shell32/tests/shlview.c
# |
if test "$enable_shell32_NewMenu_Interface" -eq 1; then
	patch_apply shell32-NewMenu_Interface/0001-shell32-Implement-NewMenu-with-new-folder-item.patch
fi

# Patchset shell32-SFGAO_HASSUBFOLDER
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#24851] Only set SFGAO_HASSUBFOLDER when there are really subfolders
# |
# | Modified files:
# |   *	dlls/shell32/shfldr_unixfs.c, dlls/shell32/shlfolder.c
# |
if test "$enable_shell32_SFGAO_HASSUBFOLDER" -eq 1; then
	patch_apply shell32-SFGAO_HASSUBFOLDER/0001-shell32-Set-SFGAO_HASSUBFOLDER-correctly-for-unixfs.patch
	patch_apply shell32-SFGAO_HASSUBFOLDER/0002-shell32-Set-SFGAO_HASSUBFOLDER-correctly-for-normal-.patch
fi

# Patchset shell32-SHGetStockIconInfo
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45868] Improve semi-stub SHGetStockIconInfo
# |
# | Modified files:
# |   *	dlls/shell32/iconcache.c, dlls/shell32/tests/shelllink.c
# |
if test "$enable_shell32_SHGetStockIconInfo" -eq 1; then
	patch_apply shell32-SHGetStockIconInfo/0001-shell32-Improve-semi-stub-SHGetStockIconInfo-try-fin.patch
fi

# Patchset shell32-Toolbar_Bitmaps
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#40236] Add missing resources for IE6 buttons
# |
# | Modified files:
# |   *	dlls/shell32/Makefile.in, dlls/shell32/resources/ietoolbar.bmp, dlls/shell32/resources/ietoolbar.svg,
# | 	dlls/shell32/resources/ietoolbar_small.bmp, dlls/shell32/resources/ietoolbar_small.svg, dlls/shell32/shell32.rc
# |
if test "$enable_shell32_Toolbar_Bitmaps" -eq 1; then
	patch_apply shell32-Toolbar_Bitmaps/0001-shell32-Add-toolbar-bitmaps-compatible-with-IE6.patch
	patch_apply shell32-Toolbar_Bitmaps/0002-shell32-Add-more-Tango-icons-to-the-IE-toolbar.patch
fi

# Patchset shell32-UnixFS
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#43109] Do not use unixfs for devices without mountpoint
# |
# | Modified files:
# |   *	dlls/shell32/shfldr_desktop.c, dlls/shell32/tests/shlfolder.c
# |
if test "$enable_shell32_UnixFS" -eq 1; then
	patch_apply shell32-UnixFS/0001-shell32-Do-not-use-unixfs-for-devices-without-mountp.patch
fi

# Patchset shlwapi-AssocGetPerceivedType
# |
# | Modified files:
# |   *	dlls/shlwapi/assoc.c, dlls/shlwapi/tests/assoc.c
# |
if test "$enable_shlwapi_AssocGetPerceivedType" -eq 1; then
	patch_apply shlwapi-AssocGetPerceivedType/0001-shlwapi-tests-Add-tests-for-AssocGetPerceivedType.patch
	patch_apply shlwapi-AssocGetPerceivedType/0002-shlwapi-Implement-AssocGetPerceivedType.patch
fi

# Patchset shlwapi-UrlCanonicalize
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#23166] shlwapi: Support ./ in UrlCanonicalize.
# |
# | Modified files:
# |   *	dlls/kernelbase/path.c, dlls/shlwapi/tests/url.c
# |
if test "$enable_shlwapi_UrlCanonicalize" -eq 1; then
	patch_apply shlwapi-UrlCanonicalize/0001-shlwapi-Support-.-in-UrlCanonicalize.patch
fi

# Patchset shlwapi-UrlCombine
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	shlwapi-UrlCanonicalize
# |
# | Modified files:
# |   *	dlls/kernelbase/path.c, dlls/shlwapi/tests/url.c
# |
if test "$enable_shlwapi_UrlCombine" -eq 1; then
	patch_apply shlwapi-UrlCombine/0001-shlwapi-tests-Add-additional-tests-for-UrlCombine-and-.patch
	patch_apply shlwapi-UrlCombine/0002-shlwapi-UrlCombineW-workaround-for-relative-paths.patch
fi

# Patchset stdole32.idl-Typelib
# |
# | Modified files:
# |   *	dlls/stdole32.tlb/std_ole_v1.idl, include/Makefile.in, include/stdole32.idl
# |
if test "$enable_stdole32_idl_Typelib" -eq 1; then
	patch_apply stdole32.idl-Typelib/0001-include-Make-stdole32.idl-a-public-component.patch
fi

# Patchset widl-SLTG_Typelib_Support
# |
# | Modified files:
# |   *	dlls/oleaut32/typelib.c, dlls/oleaut32/typelib.h, tools/widl/Makefile.in, tools/widl/register.c, tools/widl/typelib.h,
# | 	tools/widl/widl.c, tools/widl/widl.h, tools/widl/write_sltg.c
# |
if test "$enable_widl_SLTG_Typelib_Support" -eq 1; then
	patch_apply widl-SLTG_Typelib_Support/0001-widl-Add-initial-implementation-of-SLTG-typelib-gene.patch
	patch_apply widl-SLTG_Typelib_Support/0002-widl-Add-support-for-structures.patch
	patch_apply widl-SLTG_Typelib_Support/0003-widl-Properly-align-name-table-entries.patch
	patch_apply widl-SLTG_Typelib_Support/0004-widl-More-accurately-report-variable-descriptions-da.patch
	patch_apply widl-SLTG_Typelib_Support/0005-widl-Calculate-size-of-instance-for-structures.patch
	patch_apply widl-SLTG_Typelib_Support/0006-widl-Write-correct-typekind-to-the-SLTG-typeinfo-blo.patch
	patch_apply widl-SLTG_Typelib_Support/0007-widl-Write-SLTG-blocks-according-to-the-index-order.patch
	patch_apply widl-SLTG_Typelib_Support/0008-widl-Write-correct-syskind-by-SLTG-typelib-generator.patch
	patch_apply widl-SLTG_Typelib_Support/0009-widl-Add-support-for-VT_VOID-and-VT_VARIANT-to-SLTG-.patch
	patch_apply widl-SLTG_Typelib_Support/0010-widl-Add-support-for-VT_USERDEFINED-to-SLTG-typelib-.patch
	patch_apply widl-SLTG_Typelib_Support/0011-widl-Factor-out-SLTG-tail-initialization.patch
	patch_apply widl-SLTG_Typelib_Support/0012-widl-Add-support-for-recursive-type-references-to-SL.patch
	patch_apply widl-SLTG_Typelib_Support/0013-widl-Add-support-for-interfaces-to-SLTG-typelib-gene.patch
	patch_apply widl-SLTG_Typelib_Support/0014-widl-Add-support-for-inherited-interfaces-to-SLTG-ty.patch
	patch_apply widl-SLTG_Typelib_Support/0015-widl-Make-automatic-dispid-generation-scheme-better-.patch
	patch_apply widl-SLTG_Typelib_Support/0016-widl-Create-library-block-index-right-after-the-Comp.patch
	patch_apply widl-SLTG_Typelib_Support/0017-widl-Fix-generation-of-resources-containing-an-old-t.patch
	patch_apply widl-SLTG_Typelib_Support/0018-widl-Add-oldtlb-switch-in-usage-message.patch
	patch_apply widl-SLTG_Typelib_Support/0019-widl-Avoid-relying-on-side-effects-when-marking-func.patch
	patch_apply widl-SLTG_Typelib_Support/0020-widl-Set-the-lowest-bit-in-the-param-name-to-indicat.patch
	patch_apply widl-SLTG_Typelib_Support/0021-oleaut32-Fix-logic-for-deciding-whether-type-descrip.patch
	patch_apply widl-SLTG_Typelib_Support/0022-widl-Add-support-for-function-parameter-flags-to-SLT.patch
	patch_apply widl-SLTG_Typelib_Support/0023-oleaut32-Implement-decoding-of-SLTG-help-strings.patch
	patch_apply widl-SLTG_Typelib_Support/0024-oleaut32-Add-support-for-decoding-SLTG-function-help.patch
	patch_apply widl-SLTG_Typelib_Support/0025-oleaut32-Add-support-for-decoding-SLTG-variable-help.patch
	patch_apply widl-SLTG_Typelib_Support/0026-widl-Minor-cosmetic-clean-up.patch
fi

# Patchset stdole32.tlb-SLTG_Typelib
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	widl-SLTG_Typelib_Support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#3689] Compile stdole32.tlb in SLTG typelib format
# |
# | Modified files:
# |   *	dlls/stdole32.tlb/Makefile.in
# |
if test "$enable_stdole32_tlb_SLTG_Typelib" -eq 1; then
	patch_apply stdole32.tlb-SLTG_Typelib/0020-stdole32.tlb-Compile-typelib-with-oldtlb.patch
fi

# Patchset tasklist-basics
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#48596] Tasklist add basic functionality.
# |
# | Modified files:
# |   *	programs/tasklist/tasklist.c
# |
if test "$enable_tasklist_basics" -eq 1; then
	patch_apply tasklist-basics/0001-tasklist.exe-add-minimal-functionality.patch
fi

# Patchset user32-DM_SETDEFID
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22764] Use root dialog for DM_SETDEFID/DM_GETDEFID in DefDlgProc
# |
# | Modified files:
# |   *	dlls/user32/defdlg.c, dlls/user32/dialog.c, dlls/user32/tests/win.c
# |
if test "$enable_user32_DM_SETDEFID" -eq 1; then
	patch_apply user32-DM_SETDEFID/0001-user32-Do-not-initialize-dialog-info-for-every-windo.patch
	patch_apply user32-DM_SETDEFID/0002-user32-Use-root-dialog-for-DM_SETDEFID-DM_GETDEFID-i.patch
	patch_apply user32-DM_SETDEFID/0003-user32-tests-Add-a-bunch-of-tests-for-DM_SETDEFID-DM.patch
fi

# Patchset user32-Dialog_Paint_Event
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#35652] Send WM_PAINT event during dialog creation
# |
# | Modified files:
# |   *	dlls/user32/dialog.c
# |
if test "$enable_user32_Dialog_Paint_Event" -eq 1; then
	patch_apply user32-Dialog_Paint_Event/0001-user32-Call-UpdateWindow-during-DIALOG_CreateIndirec.patch
fi

# Patchset user32-DrawTextExW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22109] Fix handling of invert_y in DrawTextExW
# |
# | Modified files:
# |   *	dlls/user32/text.c
# |
if test "$enable_user32_DrawTextExW" -eq 1; then
	patch_apply user32-DrawTextExW/0001-user32-Fix-handling-of-invert_y-in-DrawTextExW.patch
fi

# Patchset user32-FlashWindowEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#43124] FlashWindowEx: WM_NCACTIVATE behavior is incorrect
# |
# | Modified files:
# |   *	dlls/user32/tests/win.c, dlls/user32/win.c
# |
if test "$enable_user32_FlashWindowEx" -eq 1; then
	patch_apply user32-FlashWindowEx/0001-user32-Improve-FlashWindowEx-message-and-return-valu.patch
fi

# Patchset user32-GetSystemMetrics
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#18732] Make it possible to change media center / tablet pc status
# |
# | Modified files:
# |   *	dlls/user32/sysparams.c
# |
if test "$enable_user32_GetSystemMetrics" -eq 1; then
	patch_apply user32-GetSystemMetrics/0001-user32-Allow-changing-the-tablet-media-center-status.patch
fi

# Patchset user32-Implement-CascadeWindows
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45968] user32: Implement CascadeWindows
# |   *	[#46197] user32: Implement TileWindows
# |
# | Modified files:
# |   *	dlls/user32/mdi.c
# |
if test "$enable_user32_Implement_CascadeWindows" -eq 1; then
	patch_apply user32-Implement-CascadeWindows/0001-user32-Implement-CascadeWindows.patch
	patch_apply user32-Implement-CascadeWindows/0002-user32-Implement-TileWindows.patch
fi

# Patchset user32-LR_LOADFROMFILE
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#24963] Workaround for Windows 3.1 apps which call LoadImage(LR_LOADFROMFILE) with a resource id
# |
# | Modified files:
# |   *	dlls/user32/cursoricon.c
# |
if test "$enable_user32_LR_LOADFROMFILE" -eq 1; then
	patch_apply user32-LR_LOADFROMFILE/0001-user32-Add-a-workaround-for-Windows-3.1-apps-which-c.patch
fi

# Patchset user32-ListBox_Size
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#38142] Fix calculation of listbox size when horizontal scrollbar is present
# |
# | Modified files:
# |   *	dlls/user32/listbox.c
# |
if test "$enable_user32_ListBox_Size" -eq 1; then
	patch_apply user32-ListBox_Size/0001-user32-Fix-calculation-of-listbox-size-when-horizont.patch
fi

# Patchset user32-LoadKeyboardLayoutEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#28170] user32: Added LoadKeyboardLayoutEx stub.
# |
# | Modified files:
# |   *	dlls/user32/input.c, dlls/user32/user32.spec
# |
if test "$enable_user32_LoadKeyboardLayoutEx" -eq 1; then
	patch_apply user32-LoadKeyboardLayoutEx/0001-user32-Added-LoadKeyboardLayoutEx-stub.patch
fi

# Patchset user32-MessageBox_WS_EX_TOPMOST
# |
# | Modified files:
# |   *	dlls/user32/msgbox.c, dlls/user32/tests/dialog.c
# |
if test "$enable_user32_MessageBox_WS_EX_TOPMOST" -eq 1; then
	patch_apply user32-MessageBox_WS_EX_TOPMOST/0001-user32-tests-Add-some-tests-to-see-when-MessageBox-g.patch
	patch_apply user32-MessageBox_WS_EX_TOPMOST/0002-user32-MessageBox-should-be-topmost-when-MB_SYSTEMMO.patch
fi

# Patchset user32-Mouse_Message_Hwnd
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#22458] Fix issues with inserting symbols by clicking on center in Word 2007 & 2010
# |   *	[#12007] Fix issues with dragging layers between images in Adobe Photoshop 7.0
# |   *	[#9512] Make sure popups don't block access to objects underneath in DVDPro
# |
# | Modified files:
# |   *	dlls/user32/message.c, dlls/user32/tests/input.c, dlls/winex11.drv/bitblt.c, server/protocol.def, server/window.c
# |
if test "$enable_user32_Mouse_Message_Hwnd" -eq 1; then
	patch_apply user32-Mouse_Message_Hwnd/0001-user32-Try-harder-to-find-a-target-for-mouse-message.patch
	patch_apply user32-Mouse_Message_Hwnd/0002-user32-tests-Add-tests-for-clicking-through-layered-.patch
	patch_apply user32-Mouse_Message_Hwnd/0003-user32-tests-Add-tests-for-window-region-of-layered-.patch
	patch_apply user32-Mouse_Message_Hwnd/0004-user32-tests-Add-tests-for-DC-region.patch
	patch_apply user32-Mouse_Message_Hwnd/0005-server-Add-support-for-a-layered-window-region.-v2.patch
fi

# Patchset user32-QueryDisplayConfig
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49583] Implement QueryDisplayConfig
# |
# | Modified files:
# |   *	dlls/user32/sysparams.c
# |
if test "$enable_user32_QueryDisplayConfig" -eq 1; then
	patch_apply user32-QueryDisplayConfig/0001-user32-Implement-QueryDisplayConfig.patch
fi

# Patchset user32-Refresh_MDI_Menus
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#21855] Refresh MDI menus when DefMDIChildProc(WM_SETTEXT) is called
# |
# | Modified files:
# |   *	dlls/user32/mdi.c
# |
if test "$enable_user32_Refresh_MDI_Menus" -eq 1; then
	patch_apply user32-Refresh_MDI_Menus/0001-user32-Refresh-MDI-menus-when-DefMDIChildProcW-WM_SE.patch
fi

# Patchset user32-ScrollWindowEx
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37706] Fix return value of ScrollWindowEx for invisible windows
# |
# | Modified files:
# |   *	dlls/user32/painting.c
# |
if test "$enable_user32_ScrollWindowEx" -eq 1; then
	patch_apply user32-ScrollWindowEx/0001-user32-Fix-return-value-of-ScrollWindowEx-for-invisi.patch
fi

# Patchset user32-message-order
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#40262] Correct order of windows messages.
# |
# | Modified files:
# |   *	dlls/user32/tests/msg.c, dlls/user32/winpos.c
# |
if test "$enable_user32_message_order" -eq 1; then
	patch_apply user32-message-order/0001-user32-Fix-messages-sent-on-a-window-without-WS_CHIL.patch
fi

# Patchset user32-msgbox-Support-WM_COPY-mesg
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#17205] Support WM_COPY in MessageBox dialogs.
# |
# | Modified files:
# |   *	dlls/user32/msgbox.c, dlls/user32/tests/dialog.c
# |
if test "$enable_user32_msgbox_Support_WM_COPY_mesg" -eq 1; then
	patch_apply user32-msgbox-Support-WM_COPY-mesg/0001-user32-msgbox-Support-WM_COPY-Message.patch
	patch_apply user32-msgbox-Support-WM_COPY-mesg/0002-user32-msgbox-Use-a-windows-hook-to-trap-Ctrl-C.patch
fi

# Patchset user32-rawinput-mouse
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42631] Mouse drift, jump or don't react to small slow movements in Unity-engine games and Fallout 4 (partly fixed in
# | 	Unity games, have walkaround in Fallout4 )
# |   *	[#42675] Overwatch: Phantom mouse input / view pulled up to ceiling
# |
# | Modified files:
# |   *	dlls/user32/input.c, dlls/user32/message.c, dlls/wineandroid.drv/keyboard.c, dlls/wineandroid.drv/window.c,
# | 	dlls/winemac.drv/ime.c, dlls/winemac.drv/keyboard.c, dlls/winemac.drv/mouse.c, dlls/winex11.drv/desktop.c,
# | 	dlls/winex11.drv/event.c, dlls/winex11.drv/keyboard.c, dlls/winex11.drv/mouse.c, dlls/winex11.drv/window.c,
# | 	dlls/winex11.drv/x11drv.h, dlls/winex11.drv/x11drv_main.c, server/protocol.def, server/queue.c
# |
if test "$enable_user32_rawinput_mouse" -eq 1; then
	patch_apply user32-rawinput-mouse/0001-winex11.drv-Split-XInput2-thread-initialization.patch
	patch_apply user32-rawinput-mouse/0002-winex11.drv-Support-XInput2-events-for-individual-wi.patch
	patch_apply user32-rawinput-mouse/0003-winex11.drv-Advertise-XInput2-version-2.1-support.patch
	patch_apply user32-rawinput-mouse/0004-winex11.drv-Keep-track-of-pointer-and-device-button-.patch
	patch_apply user32-rawinput-mouse/0005-server-Add-send_hardware_message-flags-for-rawinput-.patch
	patch_apply user32-rawinput-mouse/0006-user32-Set-SEND_HWMSG_RAWINPUT-flags-only-when-RAWIN.patch
	patch_apply user32-rawinput-mouse/0007-user32-Support-sending-RIM_TYPEMOUSE-through-__wine_.patch
	patch_apply user32-rawinput-mouse/0008-winex11.drv-Listen-to-RawMotion-and-RawButton-events.patch
fi

# Patchset user32-rawinput-mouse-experimental
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	user32-rawinput-mouse
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45882] - Raw Input should use untransformed mouse values (affects Overwatch, several Source games).
# |
# | Modified files:
# |   *	dlls/user32/rawinput.c, dlls/winex11.drv/mouse.c, dlls/winex11.drv/window.c, dlls/winex11.drv/x11drv.h,
# | 	dlls/winex11.drv/x11drv_main.c, server/queue.c
# |
if test "$enable_user32_rawinput_mouse_experimental" -eq 1; then
	patch_apply user32-rawinput-mouse-experimental/0001-winex11.drv-Add-support-for-absolute-RawMotion-event.patch
	patch_apply user32-rawinput-mouse-experimental/0002-winex11.drv-Send-relative-RawMotion-events-unprocess.patch
	patch_apply user32-rawinput-mouse-experimental/0003-winex11.drv-Accumulate-mouse-movement-to-avoid-round.patch
fi

# Patchset user32-recursive-activation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46274] user32: Prevent a recursive loop with the activation messages.
# |
# | Modified files:
# |   *	dlls/user32/focus.c, dlls/user32/tests/msg.c, dlls/user32/win.h
# |
if test "$enable_user32_recursive_activation" -eq 1; then
	patch_apply user32-recursive-activation/0001-user32-focus-Prevent-a-recursive-loop-with-the-activ.patch
	patch_apply user32-recursive-activation/0002-user32-tests-Test-a-recursive-activation-loop-on-WM_.patch
fi

# Patchset uxtheme-CloseThemeClass
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#41729] Protect CloseThemeData() from invalid input
# |
# | Modified files:
# |   *	dlls/uxtheme/msstyles.c, dlls/uxtheme/msstyles.h, dlls/uxtheme/tests/system.c
# |
if test "$enable_uxtheme_CloseThemeClass" -eq 1; then
	patch_apply uxtheme-CloseThemeClass/0001-uxtheme-Protect-CloseThemeData-from-invalid-input.patch
fi

# Patchset version-VerQueryValue
# |
# | Modified files:
# |   *	dlls/version/tests/info.c
# |
if test "$enable_version_VerQueryValue" -eq 1; then
	patch_apply version-VerQueryValue/0001-version-Test-for-VerQueryValueA-try-2.patch
fi

# Patchset wbemdisp-ISWbemObject-Invoke
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#51120] wbemdisp: Support DISPATCH_METHOD in ISWbemObject Invoke.
# |   *	[#39463] vbscript: Support looping of data from WMI
# |
# | Modified files:
# |   *	dlls/vbscript/interp.c, dlls/vbscript/tests/lang.vbs, dlls/vbscript/tests/run.c, dlls/vbscript/utils.c,
# | 	dlls/wbemdisp/locator.c, dlls/wbemprox/query.c
# |
if test "$enable_wbemdisp_ISWbemObject_Invoke" -eq 1; then
	patch_apply wbemdisp-ISWbemObject-Invoke/0001-wbemdisp-Support-DISPATCH_METHOD-in-ISWbemObject-Inv.patch
	patch_apply wbemdisp-ISWbemObject-Invoke/0002-vbscript-Support-VT_BSTR-VT_ARRAY-Iterator.patch
	patch_apply wbemdisp-ISWbemObject-Invoke/0003-wbemprox-Support-VT_BYREF-in-to_longlong.patch
	patch_apply wbemdisp-ISWbemObject-Invoke/0004-vbscript-Add-wmi-test.patch
fi

# Patchset windows.networking.connectivity-new-dll
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46534] Implement INetworkInformationStatics interface
# |
# | Modified files:
# |   *	configure.ac, dlls/windows.networking.connectivity/Makefile.in, dlls/windows.networking.connectivity/network.rgs,
# | 	dlls/windows.networking.connectivity/rsrc.rc, dlls/windows.networking.connectivity/windows.networking.connectivity.spec,
# | 	dlls/windows.networking.connectivity/windows.networking.connectivity_main.c, include/Makefile.in,
# | 	include/windows.networking.connectivity.idl, include/windows.networking.idl
# |
if test "$enable_windows_networking_connectivity_new_dll" -eq 1; then
	patch_apply windows.networking.connectivity-new-dll/0001-include-Add-windows.networking.connectivity.idl.patch
	patch_apply windows.networking.connectivity-new-dll/0002-include-Add-windows.networking.idl.patch
	patch_apply windows.networking.connectivity-new-dll/0003-windows.networking.connectivity-Add-stub-dll.patch
	patch_apply windows.networking.connectivity-new-dll/0004-windows.networking.connectivity-Implement-IActivatio.patch
	patch_apply windows.networking.connectivity-new-dll/0005-windows.networking.connectivity-Implement-INetworkIn.patch
	patch_apply windows.networking.connectivity-new-dll/0006-windows.networking.connectivity-Registry-DLL.patch
	patch_apply windows.networking.connectivity-new-dll/0007-windows.networking.connectivity-Implement-INetworkIn.patch
	patch_apply windows.networking.connectivity-new-dll/0008-windows.networking.connectivity-IConnectionProfile-G.patch
fi

# Patchset windowscodecs-GIF_Encoder
# |
# | Modified files:
# |   *	dlls/windowscodecs/tests/converter.c
# |
if test "$enable_windowscodecs_GIF_Encoder" -eq 1; then
	patch_apply windowscodecs-GIF_Encoder/0007-windowscodecs-tests-Add-IWICBitmapEncoderInfo-test.patch
fi

# Patchset windowscodecs-TIFF_Support
# |
# | Modified files:
# |   *	dlls/gdiplus/image.c, dlls/gdiplus/tests/image.c, dlls/windowscodecs/metadatahandler.c
# |
if test "$enable_windowscodecs_TIFF_Support" -eq 1; then
	patch_apply windowscodecs-TIFF_Support/0015-windowscodecs-Tolerate-partial-reads-in-the-IFD-meta.patch
	patch_apply windowscodecs-TIFF_Support/0016-gdiplus-Add-support-for-more-image-color-formats.patch
	patch_apply windowscodecs-TIFF_Support/0017-gdiplus-tests-Add-some-tests-for-loading-TIFF-images.patch
fi

# Patchset wine.inf-Directory_ContextMenuHandlers
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#29523] Add 'New' context menu handler entry for directories
# |
# | Modified files:
# |   *	loader/wine.inf.in
# |
if test "$enable_wine_inf_Directory_ContextMenuHandlers" -eq 1; then
	patch_apply wine.inf-Directory_ContextMenuHandlers/0001-wine.inf-Add-New-context-menu-handler-entry-for-dire.patch
fi

# Patchset wine.inf-Dummy_CA_Certificate
# |
# | Modified files:
# |   *	loader/wine.inf.in
# |
if test "$enable_wine_inf_Dummy_CA_Certificate" -eq 1; then
	patch_apply wine.inf-Dummy_CA_Certificate/0001-wine.inf.in-Add-invalid-dummy-certificate-to-CA-cert.patch
fi

# Patchset wineboot-HKEY_DYN_DATA
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#7115] Need for Speed 3 installer requires devices in HKEY_DYN_DATA
# |
# | Modified files:
# |   *	programs/wineboot/wineboot.c
# |
if test "$enable_wineboot_HKEY_DYN_DATA" -eq 1; then
	patch_apply wineboot-HKEY_DYN_DATA/0001-wineboot-Add-some-generic-hardware-in-HKEY_DYN_DATA-.patch
fi

# Patchset wineboot-ProxySettings
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42024] Create ProxyEnable key on wineprefix update
# |
# | Modified files:
# |   *	programs/wineboot/Makefile.in, programs/wineboot/wineboot.c
# |
if test "$enable_wineboot_ProxySettings" -eq 1; then
	patch_apply wineboot-ProxySettings/0001-wineboot-Initialize-proxy-settings-registry-key.patch
fi

# Patchset winecfg-Libraries
# |
# | Modified files:
# |   *	programs/winecfg/libraries.c
# |
if test "$enable_winecfg_Libraries" -eq 1; then
	patch_apply winecfg-Libraries/0001-winecfg-Double-click-in-dlls-list-to-edit-item-s-ove.patch
fi

# Patchset winecfg-Staging
# |
# | Modified files:
# |   *	programs/winecfg/Makefile.in, programs/winecfg/main.c, programs/winecfg/resource.h, programs/winecfg/staging.c,
# | 	programs/winecfg/winecfg.h, programs/winecfg/winecfg.rc
# |
if test "$enable_winecfg_Staging" -eq 1; then
	patch_apply winecfg-Staging/0001-winecfg-Add-staging-tab-for-CSMT.patch
	patch_apply winecfg-Staging/0002-winecfg-Add-checkbox-to-enable-disable-vaapi-GPU-dec.patch
	patch_apply winecfg-Staging/0003-winecfg-Add-checkbox-to-enable-disable-EAX-support.patch
	patch_apply winecfg-Staging/0004-winecfg-Add-checkbox-to-enable-disable-HideWineExpor.patch
	patch_apply winecfg-Staging/0005-winecfg-Add-option-to-enable-disable-GTK3-theming.patch
fi

# Patchset wined3d-Accounting
# |
# | Modified files:
# |   *	dlls/wined3d/adapter_gl.c, dlls/wined3d/device.c, dlls/wined3d/wined3d_gl.h
# |
if test "$enable_wined3d_Accounting" -eq 1; then
	patch_apply wined3d-Accounting/0001-wined3d-Use-real-values-for-memory-accounting-on-NVI.patch
fi

# Patchset wined3d-SWVP-shaders
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#8051] Sims 2 demo exits prematurely
# |
# | Modified files:
# |   *	dlls/d3d8/directx.c, dlls/d3d9/d3d9_private.h, dlls/d3d9/device.c, dlls/d3d9/directx.c, dlls/d3d9/tests/device.c,
# | 	dlls/d3d9/tests/visual.c, dlls/wined3d/adapter_gl.c, dlls/wined3d/device.c, dlls/wined3d/glsl_shader.c,
# | 	dlls/wined3d/shader.c, dlls/wined3d/shader_sm1.c, dlls/wined3d/state.c, dlls/wined3d/stateblock.c,
# | 	dlls/wined3d/wined3d_private.h, include/wine/wined3d.h
# |
if test "$enable_wined3d_SWVP_shaders" -eq 1; then
	patch_apply wined3d-SWVP-shaders/0001-wined3d-Use-UBO-for-vertex-shader-float-constants-if.patch
	patch_apply wined3d-SWVP-shaders/0002-d3d9-Support-SWVP-vertex-shader-float-constants-limi.patch
	patch_apply wined3d-SWVP-shaders/0003-wined3d-Report-actual-vertex-shader-float-constants-.patch
	patch_apply wined3d-SWVP-shaders/0004-wined3d-Support-SWVP-vertex-shader-constants-limit-i.patch
	patch_apply wined3d-SWVP-shaders/0005-wined3d-Support-SWVP-mode-vertex-shaders.patch
fi

# Patchset wined3d-Indexed_Vertex_Blending
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	wined3d-SWVP-shaders
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#39057] Support for Indexed Vertex Blending
# |
# | Modified files:
# |   *	dlls/d3d9/tests/visual.c, dlls/wined3d/adapter_gl.c, dlls/wined3d/cs.c, dlls/wined3d/device.c, dlls/wined3d/directx.c,
# | 	dlls/wined3d/glsl_shader.c, dlls/wined3d/state.c, dlls/wined3d/utils.c, dlls/wined3d/vertexdeclaration.c,
# | 	dlls/wined3d/wined3d_private.h
# |
if test "$enable_wined3d_Indexed_Vertex_Blending" -eq 1; then
	patch_apply wined3d-Indexed_Vertex_Blending/0001-d3d9-tests-Add-test-for-indexed-vertex-blending.patch
	patch_apply wined3d-Indexed_Vertex_Blending/0002-d3d9-tests-Test-normal-calculation-when-indexed-vert.patch
	patch_apply wined3d-Indexed_Vertex_Blending/0003-d3d9-tests-Check-MaxVertexBlendMatrixIndex-capabilit.patch
	patch_apply wined3d-Indexed_Vertex_Blending/0004-wined3d-Allow-higher-world-matrix-states.patch
	patch_apply wined3d-Indexed_Vertex_Blending/0005-wined3d-Support-indexed-vertex-blending.patch
fi

# Patchset wined3d-Silence_FIXMEs
# |
# | Modified files:
# |   *	dlls/wined3d/surface.c
# |
if test "$enable_wined3d_Silence_FIXMEs" -eq 1; then
	patch_apply wined3d-Silence_FIXMEs/0004-wined3d-Print-FIXME-only-once-in-surface_cpu_blt.patch
fi

# Patchset wined3d-WINED3DFMT_B8G8R8X8_UNORM
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44888] Implement WINED3DFMT_B8G8R8X8_UNORM to WINED3DFMT_L8_UNORM conversion
# |
# | Modified files:
# |   *	dlls/wined3d/surface.c
# |
if test "$enable_wined3d_WINED3DFMT_B8G8R8X8_UNORM" -eq 1; then
	patch_apply wined3d-WINED3DFMT_B8G8R8X8_UNORM/0001-wined3d-Implement-WINED3DFMT_B8G8R8X8_UNORM-to-WINED.patch
fi

# Patchset wined3d-mesa_texture_download
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#45901] Avoid GPU synchronization due to GPU-CPU transfer (Overwatch)
# |
# | Modified files:
# |   *	dlls/wined3d/texture.c
# |
if test "$enable_wined3d_mesa_texture_download" -eq 1; then
	patch_apply wined3d-mesa_texture_download/0001-wined3d-Use-glReadPixels-for-RT-texture-download.patch
fi

# Patchset wined3d-unset-flip-gdi
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#47419] wined3d: Dont set DDSCAPS_FLIP for gdi renderer.
# |
# | Modified files:
# |   *	dlls/wined3d/adapter_gl.c, dlls/wined3d/adapter_vk.c, dlls/wined3d/directx.c
# |
if test "$enable_wined3d_unset_flip_gdi" -eq 1; then
	patch_apply wined3d-unset-flip-gdi/0001-wined3d-Dont-set-DDSCAPS_FLIP-for-gdi-renderer.patch
fi

# Patchset wined3d-wined3d_guess_gl_vendor
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42538] Add check for GL_VENDOR = "Brian Paul" to detect Mesa
# |
# | Modified files:
# |   *	dlls/wined3d/adapter_gl.c
# |
if test "$enable_wined3d_wined3d_guess_gl_vendor" -eq 1; then
	patch_apply wined3d-wined3d_guess_gl_vendor/0001-wined3d-Also-check-for-Brian-Paul-to-detect-Mesa-gl_.patch
fi

# Patchset wined3d-zero-inf-shaders
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34266] wined3d: Add a setting to workaround 0 * inf problem in shader models 1-3.
# |   *	[#45375] Halo Online: Weird black display problems.
# |
# | Modified files:
# |   *	dlls/wined3d/glsl_shader.c, dlls/wined3d/wined3d_main.c, dlls/wined3d/wined3d_private.h
# |
if test "$enable_wined3d_zero_inf_shaders" -eq 1; then
	patch_apply wined3d-zero-inf-shaders/0001-wined3d-Add-a-setting-to-workaround-0-inf-problem-in.patch
fi

# Patchset winedbg-Process_Arguments
# |
# | Modified files:
# |   *	programs/winedbg/info.c
# |
if test "$enable_winedbg_Process_Arguments" -eq 1; then
	patch_apply winedbg-Process_Arguments/0001-programs-winedbg-Print-process-arguments-in-info-thr.patch
fi

# Patchset winedevice-Default_Drivers
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	ntoskrnl-Stubs
# |
# | Modified files:
# |   *	configure.ac, dlls/dxgkrnl.sys/Makefile.in, dlls/dxgkrnl.sys/dxgkrnl.sys.spec, dlls/dxgkrnl.sys/main.c,
# | 	dlls/dxgmms1.sys/Makefile.in, dlls/dxgmms1.sys/dxgmms1.sys.spec, dlls/dxgmms1.sys/main.c,
# | 	dlls/ntoskrnl.exe/tests/driver.c, dlls/win32k.sys/Makefile.in, dlls/win32k.sys/main.c, dlls/win32k.sys/win32k.sys.spec,
# | 	loader/wine.inf.in, programs/winedevice/device.c, tools/make_specfiles
# |
if test "$enable_winedevice_Default_Drivers" -eq 1; then
	patch_apply winedevice-Default_Drivers/0001-win32k.sys-Add-stub-driver.patch
	patch_apply winedevice-Default_Drivers/0002-dxgkrnl.sys-Add-stub-driver.patch
	patch_apply winedevice-Default_Drivers/0003-dxgmms1.sys-Add-stub-driver.patch
	patch_apply winedevice-Default_Drivers/0004-programs-winedevice-Load-some-common-drivers-and-fix.patch
fi

# Patchset winemac.drv-no-flicker-patch
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34166] Fullscreen flickering on Mac
# |
# | Modified files:
# |   *	dlls/winemac.drv/macdrv.h, dlls/winemac.drv/macdrv_main.c, dlls/winemac.drv/opengl.c
# |
if test "$enable_winemac_drv_no_flicker_patch" -eq 1; then
	patch_apply winemac.drv-no-flicker-patch/0001-winemac.drv-No-Flicker-patch.patch
fi

# Patchset winemapi-user-xdg-mail
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#11770] - use xdg-email if it's available.
# |
# | Modified files:
# |   *	dlls/winemapi/Makefile.in, dlls/winemapi/sendmail.c, dlls/winemapi/winemapi_private.h, dlls/winemapi/xdg-email.c
# |
if test "$enable_winemapi_user_xdg_mail" -eq 1; then
	patch_apply winemapi-user-xdg-mail/0001-winemapi-Directly-use-xdg-email-if-available-enablin.patch
fi

# Patchset winemenubuilder-Desktop_Icon_Path
# |
# | Modified files:
# |   *	programs/winemenubuilder/Makefile.in, programs/winemenubuilder/winemenubuilder.c
# |
if test "$enable_winemenubuilder_Desktop_Icon_Path" -eq 1; then
	patch_apply winemenubuilder-Desktop_Icon_Path/0001-winemenubuilder-Create-desktop-shortcuts-with-absolu.patch
fi

# Patchset winemenubuilder-integration
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#41275] Winemenubuilder should respect existing defaults for filetype associations
# |   *	[#22904] Register URL protocol handlers under Linux
# |
# | Modified files:
# |   *	dlls/mshtml/mshtml.inf, loader/wine.inf.in, programs/winemenubuilder/winemenubuilder.c
# |
if test "$enable_winemenubuilder_integration" -eq 1; then
	patch_apply winemenubuilder-integration/0001-winemenubuilder-Blacklist-desktop-integration-for-ce.patch
fi

# Patchset wineps.drv-PostScript_Fixes
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#4836] Various improvements for wineps.drv for Adobe PageMaker compatibility
# |
# | Modified files:
# |   *	dlls/wineps.drv/download.c, dlls/wineps.drv/escape.c, dlls/wineps.drv/psdrv.h
# |
if test "$enable_wineps_drv_PostScript_Fixes" -eq 1; then
	patch_apply wineps.drv-PostScript_Fixes/0004-wineps.drv-Add-support-for-GETFACENAME-and-DOWNLOADF.patch
fi

# Patchset winex11-CandidateWindowPos
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30938] Update a XIM candidate position when cursor location changes
# |
# | Modified files:
# |   *	dlls/user32/caret.c, dlls/user32/driver.c, dlls/user32/user_private.h, dlls/winex11.drv/winex11.drv.spec,
# | 	dlls/winex11.drv/xim.c
# |
if test "$enable_winex11_CandidateWindowPos" -eq 1; then
	patch_apply winex11-CandidateWindowPos/0001-winex11.drv-Update-a-candidate-window-s-position-wit.patch
fi

# Patchset winex11-MWM_Decorations
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#42117] Avoid double captions observed under some WMs
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_MWM_Decorations" -eq 1; then
	patch_apply winex11-MWM_Decorations/0001-winex11.drv-Don-t-use-MWM_DECOR_RESIZEH-window-resiz.patch
	patch_apply winex11-MWM_Decorations/0002-winex11.drv-Don-t-add-MWM_DECOR_BORDER-to-windows-wi.patch
fi

# Patchset winex11-UpdateLayeredWindow
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#33943] Fix alpha blending in X11DRV_UpdateLayeredWindow
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_UpdateLayeredWindow" -eq 1; then
	patch_apply winex11-UpdateLayeredWindow/0001-winex11-Fix-alpha-blending-in-X11DRV_UpdateLayeredWi.patch
fi

# Patchset winex11-Vulkan_support
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44775] Allow vulkan support to be detected at runtime.
# |
# | Modified files:
# |   *	dlls/winex11.drv/vulkan.c
# |
if test "$enable_winex11_Vulkan_support" -eq 1; then
	patch_apply winex11-Vulkan_support/0001-winex11-Specify-a-default-vulkan-driver-if-one-not-f.patch
fi

# Patchset winex11-WM_WINDOWPOSCHANGING
# |
# | This patchset has the following (direct or indirect) dependencies:
# |   *	winex11-_NET_ACTIVE_WINDOW
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#34594] Fix handling of WM_WINDOWPOS{CHANGING,CHANGED} for deactivated topmost window
# |
# | Modified files:
# |   *	dlls/winex11.drv/event.c
# |
if test "$enable_winex11_WM_WINDOWPOSCHANGING" -eq 1; then
	patch_apply winex11-WM_WINDOWPOSCHANGING/0001-winex11.drv-Send-WM_WINDOWPOSCHANGING-WM_WINDOWPOSCH.patch
fi

# Patchset winex11-Window_Style
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#37876] Fix handling of window attributes for WS_EX_LAYERED | WS_EX_COMPOSITED
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_Window_Style" -eq 1; then
	patch_apply winex11-Window_Style/0001-winex11-Fix-handling-of-window-attributes-for-WS_EX_.patch
fi

# Patchset winex11-XEMBED
# |
# | Modified files:
# |   *	dlls/winex11.drv/event.c
# |
if test "$enable_winex11_XEMBED" -eq 1; then
	patch_apply winex11-XEMBED/0001-winex11-Enable-disable-windows-when-they-are-un-mapped.patch
fi

# Patchset winex11-ime-check-thread-data
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#46263] Final Fantasy XI crashes after accepting EULA when using Ashita
# |   *	[#28861] Final Fantasy XI hangs after character selection
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_ime_check_thread_data" -eq 1; then
	patch_apply winex11-ime-check-thread-data/0001-winex11.drv-handle-missing-thread-data-in-X11DRV_get_ic.patch
fi

# Patchset winex11-key_translation
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#30984] Improve key translation.
# |   *	[#45605] Letter keys doesn't work in DirectX aplications
# |
# | Modified files:
# |   *	dlls/winex11.drv/keyboard.c
# |
if test "$enable_winex11_key_translation" -eq 1; then
	patch_apply winex11-key_translation/0001-winex11-Match-keyboard-in-Unicode.patch
	patch_apply winex11-key_translation/0002-winex11-Fix-more-key-translation.patch
	patch_apply winex11-key_translation/0003-winex11.drv-Fix-main-Russian-keyboard-layout.patch
fi

# Patchset winex11-wglShareLists
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#11436] Do not fail when a used context is passed to wglShareLists
# |   *	[#25419] Fix broken textures in XIII Century: Death or Glory
# |
# | Modified files:
# |   *	dlls/opengl32/tests/opengl.c, dlls/winex11.drv/opengl.c
# |
if test "$enable_winex11_wglShareLists" -eq 1; then
	patch_apply winex11-wglShareLists/0001-winex11.drv-Only-warn-about-used-contexts-in-wglShar.patch
fi

# Patchset winex11.drv-Query_server_position
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#15346] winex11.drv: Query the X server for the actual rect of the window before unmapping it
# |
# | Modified files:
# |   *	dlls/winex11.drv/window.c
# |
if test "$enable_winex11_drv_Query_server_position" -eq 1; then
	patch_apply winex11.drv-Query_server_position/0001-winex11.drv-window-Query-the-X-server-for-the-actual.patch
fi

# Patchset wininet-Cleanup
# |
# | Modified files:
# |   *	dlls/wininet/http.c, dlls/wininet/tests/http.c
# |
if test "$enable_wininet_Cleanup" -eq 1; then
	patch_apply wininet-Cleanup/0001-wininet-tests-Add-more-tests-for-cookies.patch
	patch_apply wininet-Cleanup/0002-wininet-tests-Test-auth-credential-reusage-with-host.patch
	patch_apply wininet-Cleanup/0003-wininet-tests-Check-cookie-behaviour-when-overriding.patch
	patch_apply wininet-Cleanup/0004-wininet-Strip-filename-if-no-path-is-set-in-cookie.patch
	patch_apply wininet-Cleanup/0005-wininet-Replacing-header-fields-should-fail-if-they-.patch
fi

# Patchset winmm-mciSendCommandA
# |
# | Modified files:
# |   *	dlls/winmm/mci.c
# |
if test "$enable_winmm_mciSendCommandA" -eq 1; then
	patch_apply winmm-mciSendCommandA/0001-winmm-Do-not-crash-in-Win-9X-mode-when-an-invalid-de.patch
fi

# Patchset wintab32-improvements
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#11846] Improve pressure sensitivity.
# |   *	[#15443] Improve Wacom Bambo drawing support
# |   *	[#18517] Improve eraser from working.
# |
# | Modified files:
# |   *	dlls/winex11.drv/wintab.c, dlls/wintab32/context.c
# |
if test "$enable_wintab32_improvements" -eq 1; then
	patch_apply wintab32-improvements/0002-wintab32-Set-lcSysExtX-Y-for-the-first-index-of-WTI_.patch
	patch_apply wintab32-improvements/0003-winex11-Handle-negative-orAltitude-values.patch
	patch_apply wintab32-improvements/0004-winex11.drv-Support-multiplex-categories-WTI_DSCTXS-.patch
	patch_apply wintab32-improvements/0005-winex11-Support-WTI_STATUS-in-WTInfo.patch
fi

# Patchset wintrust-WTHelperGetProvCertFromChain
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#44061] Check Parameter in WTHelperGetProvCertFromChain
# |
# | Modified files:
# |   *	dlls/wintrust/tests/softpub.c, dlls/wintrust/wintrust_main.c
# |
if test "$enable_wintrust_WTHelperGetProvCertFromChain" -eq 1; then
	patch_apply wintrust-WTHelperGetProvCertFromChain/0001-wintrust-Add-parameter-check-in-WTHelperGetProvCertF.patch
fi

# Patchset wscript-support-d-u-switches
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49905] wscript: return TRUE for /d and /u stub switches
# |
# | Modified files:
# |   *	programs/wscript/main.c
# |
if test "$enable_wscript_support_d_u_switches" -eq 1; then
	patch_apply wscript-support-d-u-switches/0001-wscript-return-TRUE-for-d-and-u-stub-switches.patch
fi

# Patchset xactengine-initial
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#31476] Support Bully Scholarship Edition xactengine3_1.dll.
# |   *	[#38615] DSA: Drakensang Demo fails on IXACTEngine::Initialize
# |   *	[#41030] Pac-Man Museum requires xactengine3_7
# |   *	[#41045] Captain Morgane requires xactengine3_4
# |   *	[#48684] BlazBlue: Calamity Trigger requires for xactengine 3.3 interface.
# |
# | Modified files:
# |   *	dlls/x3daudio1_7/Makefile.in, dlls/xactengine3_7/tests/Makefile.in, dlls/xactengine3_7/tests/globals.xgs,
# | 	dlls/xactengine3_7/tests/rsrc.rc, dlls/xactengine3_7/tests/xact3.c
# |
if test "$enable_xactengine_initial" -eq 1; then
	patch_apply xactengine-initial/0001-x3daudio1_7-Create-import-library.patch
	patch_apply xactengine-initial/0003-xactengine3_7-tests-Add-Global-settings-test.patch
fi

# Patchset xactengine3_7-Notification
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#50546] xactengine3_7: Send Notification after the Wavebank is created.
# |
# | Modified files:
# |   *	dlls/xactengine3_7/xact_dll.c
# |
if test "$enable_xactengine3_7_Notification" -eq 1; then
	patch_apply xactengine3_7-Notification/0001-xactengine3.7-Delay-Notication-for-WAVEBANKPREPARED.patch
	patch_apply xactengine3_7-Notification/0002-xactengine3_7-Record-context-for-each-notications.patch
fi

# Patchset xactengine3_7-PrepareWave
# |
# | This patchset fixes the following Wine bugs:
# |   *	[#49689] xactengine3_7: Implement IXACT3Engine PrepareWave
# |
# | Modified files:
# |   *	dlls/xactengine3_7/xact_dll.c
# |
if test "$enable_xactengine3_7_PrepareWave" -eq 1; then
	patch_apply xactengine3_7-PrepareWave/0002-xactengine3_7-Implement-IXACT3Engine-PrepareStreamin.patch
	patch_apply xactengine3_7-PrepareWave/0003-xactengine3_7-Implement-IXACT3Engine-PrepareInMemory.patch
fi

if test "$enable_autoconf" -eq 1; then
	if ! update_configure; then
		abort "'autoreconf -f' failed."
	fi
	if ! update_protocol; then
		abort "'./tools/make_requests' failed."
	fi
fi
# Success
exit 0
